/*
 * DebugFn.h
 *
 * Headers for the debug functions
 *
 * Stefan Haenssgen  10-Jul-92
 *
 * Updates:
 *
 *	10-Jul-92	First rough implementation:
 *			Definition of DebugNode
 *			Definition of types and modes
 *	11-Jul-92	Removed modes - work with different calls instead
 *			Added Depth of node in tree
 *			Defined ProfileNode for profiling information
 *	12-Jul-92	Introduced context (sequential, parallel, synchronous)
 *	17-Jul-92	Dropped DN_IF
 *	07-Aug-92	Added Show*Widget() and Tree <-> List stuff
 *			Introduced DebugFn_header to make double includes OK
 *	12-Aug-92	Added MColumn for the column in which the construct
 *			 starts, to be unique (line+type is not enough!)
 *			Added MToLine & MToColumn to denote the position of
 *			 the end of the Modula-2* construct in the source
 *			Also changed ProfileNode accordingly
 *	14-Aug-92	Added {profile,tree}_active to mark number of
 *			 profiles/trees being displayed
 *	24-Aug-92	Added GetLastNode()
 *	22-Oct-92	Created ADT ProfileTableType
 *	25-Oct-92	Added ProfileAsString()
 *	26-Oct-92	Split into DebugFn.h (basic stuff) + DebugFnV.h
 *			 (X11 / Visualizer specific stuff)
 *	27-Oct-92	Made ADT DebugTree
 *			Introduced NumC & NumK (instead of checksum)
 *			Updated Headers according to new source, including
 *			 parameters
 *	29-Oct-92	Updated AddListToTree() etc
 *	30-Oct-92	Added AdjustDepths()
 *	31-Oct-92	Made names more consistent ("List" -> "String" etc)
 *			Replaced DebugFnUpdate by macro (faster!)
 *	04-Nov-92	Added MergedTree(), BuildAndMergeNodes(), CopyNode()
 *	06-Nov-92	Added GroupTree(), GroupSubTree(), JoinNodes()
 *	08-Nov-92	Added NumEntries to Profile Table
 *			Introduced MakeProfileStat()
 *			Added ProfileStatAsString()
 *	25-Nov-92	Added NodeAsString() and BuildNodeFromString()
 *			Added PrintNodeAsString()
 *			Added GimmeData() and DumpToFile() for data transfer
 *	29-Nov-92	Introduced DebugNode->ProfCount to directly update
 *			 profiling counts
 *			Rewrote the macro accordingly (check for NULL)
 *	10-Jan-93	Added JoinInfoType to remember infos when joining nodes
 *	13-Jan-93	Added PE to DebugNode
 *			Call CreateDebugTree with PE
 *	21-Feb-93	Fixed spelling
 *	23-Feb-93	Added AllGroup{Tree,SubTree}() and EquivSubTree()
 *	24-Feb-93	Added AllJoinNodes(), SetPrintStyle() and print
 *			 styles TPS_*
 *
 */

#ifndef DebugFn_header
#define DebugFn_header

#include <stdio.h>


/*
 * Type of debug node, e.g. for WHILE, IF, THEN, LOOP, FORALL etc
 */

#define DN_ROOT		 1
#define	DN_WHILE	10
#define DN_REPEAT	11
#define DN_LOOP		12
#define DN_FOR		13
#define DN_SYN_FORALL	20
#define DN_PAR_FORALL	21
#define DN_THEN		31
#define DN_ELSE		32
#define DN_CASE		33
#define DN_OF		34
#define DN_FNCALL	40

/*
 * Context of node
 */

#define DN_SEQ_CONTEXT	1
#define DN_PAR_CONTEXT	2
#define DN_SYN_CONTEXT	3


/*
 * Styles for tree output in PrintNode()
 *
 * Normal / pointers shown as adresses / full output
 */

#define TPS_NORMAL	0
#define TPS_POINTER	1
#define	TPS_VERBOSE	2
#define	TPS_SHORT	3

/*
 * Abbreviations
 */

#ifndef NULL
#    define NULL (void*)0L
#endif
#ifndef ulong
#    define ulong unsigned long
#endif


/*
 * JoinInfo:
 *
 * Information which would otherwise be lost when joining nodes
 *
 */
typedef struct JoinI {
    long	 ID;			/* ID of joined node		*/
    long	 RunCount;		/* Counter for the node		*/
    int		 PE;			/* PE of this node		*/
    struct JoinI*Next;			/* Pointer to next info		*/
} JoinInfo;


/*
 * DebugNode:
 *
 * One node in the tree built at runtime
 *
 */
typedef struct DebN {

    int		Type;		/* Type, e.g. DN_WHILE, DN_IF, ...	*/
    int		Context;	/* Context, e.g. DN_PAR_CONTEXT, ...	*/
    long	ID;		/* Unique ID number			*/
    int		Depth;		/* Depth of node in tree		*/

    struct DebN	*Parent;	/* Parent of this node			*/
    struct DebN	*LBrother;	/* Next brother to left (ring list)	*/
    struct DebN	*RBrother;	/* Next brother to right		*/
    struct DebN	*Child;		/* First Child				*/

    JoinInfo	*Joined;	/* List of join infos (if any)		*/
    long	MLine;		/* Line number on Modula-2* Source	*/
    long	MColumn;	/* Dito Column				*/
    long	MToLine;	/* Corresponding ending line of..	*/
    long	MToColumn;	/* ..Modula-2* construct		*/
    long	ActValue;	/* Current activation value		*/
    void	*ActArray;	/* Array of activation values		*/

    void	*RangeVar;	/* Variable e.g. of FOR or FORALL	*/
    long	RangeLo;	/* Range of Variable e.g. for FORALL	*/
    long	RangeHi;	/* Dito					*/

    long	RunCount;	/* Profiling information		*/

    long	*ProfCountP;	/* Pointer to counter in Profile Table	*/
    
    int		PE;		/* PE we're on				*/

} DebugNode;


/*
 * DebugTree:
 *
 * Tree of Debug Nodes representing the dynamic call structure of
 * the program
 *
 */
typedef struct {
    DebugNode	*RootNode;	/* Root of the structure tree		*/
    DebugNode	*LastNode;	/* Most recently created node		*/
    long	 ID;		/* Unique ID for new nodes		*/
    long	 NumC, NumK;	/* Counters for create/kill (use as a	*/
    				/*  checksum: Change when tree changes)	*/
} DebugTree;



/* Size of hash table and hash function */

#define P_TABLE_SIZE	997
#define phash(x) (x % P_TABLE_SIZE)


/*
 * ProfileNode:
 *
 * One node in profile information list
 *
 */
typedef struct ProN {
    int		Type;		/* Type, e.g. DN_WHILE, DN_IF, ...	*/
    int		Context;	/* Most recently used context for this	*/
    struct ProN	*NextNode;	/* Chain of nodes in one Hash Entry	*/
    long	MLine;		/* Line number on Modula-2* Source	*/
    long	MColumn;	/* Dito Column				*/
    long	RunCount;	/* Profiling information		*/
    long	MToLine;	/* Corresponding ending line of..	*/
    long	MToColumn;	/* ..Modula-2* construct		*/
} ProfileNode;



/*
 * ProfileTable:
 *
 * One Profile Table with hashed node pointers and line range
 *
 */
typedef struct {
    ProfileNode	*ProfileTable[P_TABLE_SIZE];	/* Hash table		*/
    long	maxProfileLine;			/* Line range in Table	*/
    long	NumEntries;			/* Number of entries	*/
} ProfileTable;


extern DebugTree	*DTree;		/* Global Debug Structure Tree	*/
extern ProfileTable	*PTable;	/* Global Profile Info Table	*/

extern int tree_active;		/* Number of trees being displayed now	*/
extern int profile_active;	/* Dito: # of currently shown profiles	*/



/*
 * Debug functions visible to user
 */


/* DebugNode, DebugTree: Dynamic call structure tree of program */

extern DebugNode   *NewNode(/*tree,type,parent,		* Create new debug   *
                     mline,mcolumn,mtoline,mtocolumn,	*  information node  *
                     actvalue,actarray,			*  (internal use!)   *
                     rangevar,rangelo,rangehi*/);
extern DebugNode   *CopyNode(/*tree,par,node*/);/* Copy contents below parent*/
extern void	    SetPrintStyle(/*style*/);	/* Set print style for node  */
extern void	    PrintNode(/*tree,node*/);	/* Print one node	     */
extern void	    PrintSubTree(/*tree,node*/);/* Print subtree	     */
extern void	    PrintTree(/*tree*/);	/* Print whole tree	     */
extern DebugTree   *CreateDebugTree(/*PE*/);	/* Create a debug info tree  */
extern DebugNode   *GetLastNode(/*tree*/);	/* Return most recent node   */
extern DebugNode   *AdjustDepths(/*node,sdpt*/);/* Recompute depths w. start */
extern long	    GetChecksum(/*tree*/);	/* Changes when tree changes */
extern char	   *NodeAsString(/*node*/);	/* String for one node	     */
extern void	    PrintNodeAsString(/*node*/);/* Print this string	     */
extern char	   *TreeAsString(/*tree,node*/);/* String describing tree    */
extern void	    PrintTreeAsString(/*tree,node*/);/* Print this string    */
extern DebugNode   *BuildNodeFromString(/*s*/);	/* Build one node from data  */
extern DebugTree   *BuildTreeFromString(/*s,sID*/);/*Reconstruct tree from s.*/
extern DebugTree   *AddStringToTree(/*t,s,snod*/);/* Merge tree with tree-str*/
extern int	    KillNode(/*tree,node*/);	/* Destroy one node	     */
extern int	    KillSubTree(/*tree,node*/);	/* Destroy a subtree	     */
extern int	    KillTree(/*tree*/);		/* Destroy whole tree	     */
extern void	    BuildAndMergeNodes(/*tree,parent,
		     node1,node2*/);		/* Recurs.merge subtree nodes*/
extern DebugTree   *MergedTree(/*tree1,tree2*/);/* Merge trees (group nodes) */
extern void	    JoinNodes(/*n1,n2*/);	/* Merge infos recursively   */
extern char	    EquivSubTree(/*n1,n2*/);	/* Test if subtrees are equiv*/
extern void	    GroupSubTree(/*node*/);	/* Group nodes in subtree    */
extern void	    GroupTree(/*tree*/);	/* Group infos in tree	     */
extern void	    AllJoinNodes(/*n1,n2*/);	/* Merge infos, kill n2	     */
extern void	    AllGroupSubTree(/*node*/);	/* Dito, but only test equiv.*/
extern void	    AllGroupTree(/*tree*/);	/*  of nodes, not subtrees!  */


/* ProfileNode, ProfileTable: Profiling information table for program */

extern ProfileTable*CreateProfileTable();	/* Create empty profile table*/
extern void	    AddProfileNode(/*pt,dnode*/);/* Add inf of DbgNode to tab*/
extern void	    KillProfileNode(/*pt,line,column*/);/* Destroy one entry */
extern void	    ClearProfileTable(/*pt*/);	/* Destroy all entries	     */
extern void	    PrintProfileTable(/*pt*/);	/* Print profile Table	     */
extern char	    *ProfileAsString(/*pt*/);	/* String describing Table   */
extern void	    PrintProfileAsString(/*pt*/);/* Print this string	     */
extern void	    AddStringToProfile(/*pt,in,replace*/); /* Merge str & tab*/
extern ProfileNode *GetProfileNode(/*pt,line,column*/);  /* Find one node    */
extern ProfileNode**MakeProfileStat(/*pt*/);	/* Sorted array of nodes     */
extern char	   *ProfileStatAsString(/*pt,nlines*/);  /* This array as str*/
extern void	    KillProfileTable(/*pt*/);	/* Destroy whole table	     */


/* DebugFnXXX: Debug Functions, record dynamic structure of program in	*/
/*             DebugTree while also collection Profile Information	*/

extern void	    DebugInitialize();	/* Init debug function environment   */
extern void	    DebugEnd();		/* Close & free environment	     */
extern void	    DebugFnStart(/*type,		* Start a new	     *
		       mline,mcolumn,mtoline,		*  block	     *
		       mtocolumn,actvalue,actarray,
                       rangevar,rangelo,rangehi*/);

/* Update: Increment RunCount of node and also counter in */
/*         Profile Table entry, if defined		  */

#define DebugFnUpdate(type,mline,mcolumn,mtoline,mtocolumn,actvalue,actarray,rangevar,rangelo,rangehi)							\
                      {long*pc=DTree->LastNode->ProfCountP;(DTree->LastNode->RunCount)++;if(pc)(*pc)++;}
  
extern void	    DebugFnEnd(/*type,			* Close the block   *
		       mline,mcolumn,mtoline,		*  and remember	    *
		       mtocolumn,actvalue,actarray,	*  profile data etc *
                       rangevar,rangelo,rangehi*/);


extern int GimmeData(/*pid,buf,adr,len*/); /* Transfer data via ptrace()    */
extern int DumpToFile(/*adr,len,fname*/);  /* Transfer via seperate file    */



#endif /* DebugFn_header */
