------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               S N A M E S                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.55 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Types; use Types;

package Snames is

--  This package contains definitions of standard names (i.e. entries in the
--  Names table) that are used throughout the GNAT compiler). It also contains
--  the definitions of some enumeration types whose definitions are tied to
--  the order of these preset names.

   ------------------
   -- Preset Names --
   ------------------

   --  The following are preset entries in the names table, which are
   --  entered at the start of every compilation for easy access. Note
   --  that the order of initialization of these names in the body must
   --  be coordinated with the order of names in this table.

   --  Note: a name may not appear more than once in the following list.
   --  If additional pragmas or attributes are introduced which might
   --  otherwise cause a duplicate, then list it only once in this table,
   --  and adjust the definition of the functions for testing for pragma
   --  names and attribute names, and returning their ID values. Of course
   --  everything is simpler if no such duplications occur!

   --  First we have the one character names used to optimize the lookup
   --  process for one character identifiers (avoid the hashing in this case)
   --  There are a full 256 of these, but only the entries for lower case
   --  and upper case letters have identifiers

   --  The lower case letter entries are used for one character identifiers
   --  appearing in the source, for example in pragma Interface (C).

   Name_A         : constant Name_Id := First_Name_Id + Character'Pos ('a');
   Name_B         : constant Name_Id := First_Name_Id + Character'Pos ('b');
   Name_C         : constant Name_Id := First_Name_Id + Character'Pos ('c');
   Name_D         : constant Name_Id := First_Name_Id + Character'Pos ('d');
   Name_E         : constant Name_Id := First_Name_Id + Character'Pos ('e');
   Name_F         : constant Name_Id := First_Name_Id + Character'Pos ('f');
   Name_G         : constant Name_Id := First_Name_Id + Character'Pos ('g');
   Name_H         : constant Name_Id := First_Name_Id + Character'Pos ('h');
   Name_I         : constant Name_Id := First_Name_Id + Character'Pos ('i');
   Name_J         : constant Name_Id := First_Name_Id + Character'Pos ('j');
   Name_K         : constant Name_Id := First_Name_Id + Character'Pos ('k');
   Name_L         : constant Name_Id := First_Name_Id + Character'Pos ('l');
   Name_M         : constant Name_Id := First_Name_Id + Character'Pos ('m');
   Name_N         : constant Name_Id := First_Name_Id + Character'Pos ('n');
   Name_O         : constant Name_Id := First_Name_Id + Character'Pos ('o');
   Name_P         : constant Name_Id := First_Name_Id + Character'Pos ('p');
   Name_Q         : constant Name_Id := First_Name_Id + Character'Pos ('q');
   Name_R         : constant Name_Id := First_Name_Id + Character'Pos ('r');
   Name_S         : constant Name_Id := First_Name_Id + Character'Pos ('s');
   Name_T         : constant Name_Id := First_Name_Id + Character'Pos ('t');
   Name_U         : constant Name_Id := First_Name_Id + Character'Pos ('u');
   Name_V         : constant Name_Id := First_Name_Id + Character'Pos ('v');
   Name_W         : constant Name_Id := First_Name_Id + Character'Pos ('w');
   Name_X         : constant Name_Id := First_Name_Id + Character'Pos ('x');
   Name_Y         : constant Name_Id := First_Name_Id + Character'Pos ('y');
   Name_Z         : constant Name_Id := First_Name_Id + Character'Pos ('z');

   --  The upper case letter entries are used by expander code for local
   --  variables that do not require unique names (e.g. formal parameter
   --  names in constructed procedures)

   Name_uA        : constant Name_Id := First_Name_Id + Character'Pos ('A');
   Name_uB        : constant Name_Id := First_Name_Id + Character'Pos ('B');
   Name_uC        : constant Name_Id := First_Name_Id + Character'Pos ('C');
   Name_uD        : constant Name_Id := First_Name_Id + Character'Pos ('D');
   Name_uE        : constant Name_Id := First_Name_Id + Character'Pos ('E');
   Name_uF        : constant Name_Id := First_Name_Id + Character'Pos ('F');
   Name_uG        : constant Name_Id := First_Name_Id + Character'Pos ('G');
   Name_uH        : constant Name_Id := First_Name_Id + Character'Pos ('H');
   Name_uI        : constant Name_Id := First_Name_Id + Character'Pos ('I');
   Name_uJ        : constant Name_Id := First_Name_Id + Character'Pos ('J');
   Name_uK        : constant Name_Id := First_Name_Id + Character'Pos ('K');
   Name_uL        : constant Name_Id := First_Name_Id + Character'Pos ('L');
   Name_uM        : constant Name_Id := First_Name_Id + Character'Pos ('M');
   Name_uN        : constant Name_Id := First_Name_Id + Character'Pos ('N');
   Name_uO        : constant Name_Id := First_Name_Id + Character'Pos ('O');
   Name_uP        : constant Name_Id := First_Name_Id + Character'Pos ('P');
   Name_uQ        : constant Name_Id := First_Name_Id + Character'Pos ('Q');
   Name_uR        : constant Name_Id := First_Name_Id + Character'Pos ('R');
   Name_uS        : constant Name_Id := First_Name_Id + Character'Pos ('S');
   Name_uT        : constant Name_Id := First_Name_Id + Character'Pos ('T');
   Name_uU        : constant Name_Id := First_Name_Id + Character'Pos ('U');
   Name_uV        : constant Name_Id := First_Name_Id + Character'Pos ('V');
   Name_uW        : constant Name_Id := First_Name_Id + Character'Pos ('W');
   Name_uX        : constant Name_Id := First_Name_Id + Character'Pos ('X');
   Name_uY        : constant Name_Id := First_Name_Id + Character'Pos ('Y');
   Name_uZ        : constant Name_Id := First_Name_Id + Character'Pos ('Z');

   --  Note: the following table is read by the utility program XSNAMES and
   --  its format should not be changed without coordinating with this program.

   N : constant Name_Id := First_Name_Id + 256;
   --  Synonym used in standard name definitions

   --  Some special names used by the expander. Note that the lower case u's
   --  at the start of these names get translated to extra underscores. These
   --  names are only referenced internally by expander generated code. Note
   --  that we put these early in the list so that in particular Name_uParent
   --  and Name_uSize will be at fixed Name_Id values which will not change
   --  with the addition of other entries to Snames. This makes things easier
   --  because Gigi references these particular entries.

   Name_uParent                        : constant Name_Id := N + 000;
   Name_uSize                          : constant Name_Id := N + 001;

   --  Remaining names are not referenced by Gigi

   Name_uAbort_Signal                  : constant Name_Id := N + 002;
   Name_uChain                         : constant Name_Id := N + 003;
   Name_uClean                         : constant Name_Id := N + 004;
   Name_uEquality                      : constant Name_Id := N + 005;
   Name_uExpunge                       : constant Name_Id := N + 006;
   Name_uIdepth                        : constant Name_Id := N + 007;
   Name_uInit                          : constant Name_Id := N + 008;
   Name_uLocal_Final_List              : constant Name_Id := N + 009;
   Name_uMaster                        : constant Name_Id := N + 010;
   Name_uPriority                      : constant Name_Id := N + 011;
   Name_uTask                          : constant Name_Id := N + 012;
   Name_uTrace_Sp                      : constant Name_Id := N + 013;
   Name_uTag                           : constant Name_Id := N + 014;
   Name_uTags                          : constant Name_Id := N + 015;
   Name_uTask_Id                       : constant Name_Id := N + 016;

   --  Names of routines in Ada.Finalization, needed by expander

   Name_Initialize                     : constant Name_Id := N + 017;
   Name_Adjust                         : constant Name_Id := N + 018;
   Name_Finalize                       : constant Name_Id := N + 019;

   --  Some miscellaneous names used by the parser for error detection/recovery

   Name_Error                          : constant Name_Id := N + 020;
   Name_Go                             : constant Name_Id := N + 021;
   Name_To                             : constant Name_Id := N + 022;

   --  Names for packages that are treated specially by the compiler

   Name_Finalization                   : constant Name_Id := N + 023;
   Name_Interfaces                     : constant Name_Id := N + 024;
   Name_Standard                       : constant Name_Id := N + 025;
   Name_System                         : constant Name_Id := N + 026;
   Name_Unchecked_Conversion           : constant Name_Id := N + 027;

   --  Operator Symbol entries. The actual names have an upper case O at
   --  the start in place of the Op_ prefix (e.g. the actual name that
   --  corresponds to Name_Op_Abs is "Oabs".

   First_Operator_Name                 : constant Name_Id := N + 028;
   Name_Op_Abs                         : constant Name_Id := N + 028; -- "abs"
   Name_Op_And                         : constant Name_Id := N + 029; -- "and"
   Name_Op_Mod                         : constant Name_Id := N + 030; -- "mod"
   Name_Op_Not                         : constant Name_Id := N + 031; -- "not"
   Name_Op_Or                          : constant Name_Id := N + 032; -- "or"
   Name_Op_Rem                         : constant Name_Id := N + 033; -- "rem"
   Name_Op_Xor                         : constant Name_Id := N + 034; -- "xor"
   Name_Op_Eq                          : constant Name_Id := N + 035; -- "="
   Name_Op_Ne                          : constant Name_Id := N + 036; -- "/="
   Name_Op_Lt                          : constant Name_Id := N + 037; -- "<"
   Name_Op_Le                          : constant Name_Id := N + 038; -- "<="
   Name_Op_Gt                          : constant Name_Id := N + 039; -- ">"
   Name_Op_Ge                          : constant Name_Id := N + 040; -- ">="
   Name_Op_Add                         : constant Name_Id := N + 041; -- "+"
   Name_Op_Subtract                    : constant Name_Id := N + 042; -- "-"
   Name_Op_Concat                      : constant Name_Id := N + 043; -- "&"
   Name_Op_Multiply                    : constant Name_Id := N + 044; -- "*"
   Name_Op_Divide                      : constant Name_Id := N + 045; -- "/"
   Name_Op_Expon                       : constant Name_Id := N + 046; -- "**"
   Last_Operator_Name                  : constant Name_Id := N + 046;

   --  Additional names for operations which we syntatically consider to be
   --  operators but are not operator symbols, but we still want the names in
   --  the name list for use in error messages.

   First_Operator_Special_Name          : constant Name_Id := N + 047;
   Name_Op_In                           : constant Name_Id := N + 047;
   Name_Op_Not_In                       : constant Name_Id := N + 048;
   Name_Op_And_Then                     : constant Name_Id := N + 049;
   Name_Op_Or_Else                      : constant Name_Id := N + 050;
   Last_Operator_Special_Name           : constant Name_Id := N + 050;

   --  Names for all pragmas recognized by GNAT. The entries with the comment
   --  "Ada 83" are pragmas that are defined in Ada 83, but not in Ada 9X.
   --  These pragmas are fully implemented in both Ada 83 and Ada 9X modes
   --  in GNAT. The entries marked GNAT are pragmas that are defined by GNAT
   --  and implemented in both Ada 83 and Ada 9X modes.

   First_Pragma_Name                   : constant Name_Id := N + 051;
   Name_Abort_Defer                    : constant Name_Id := N + 051; -- GNAT
   Name_Ada_83                         : constant Name_Id := N + 052; -- GNAT
   Name_Ada_9X                         : constant Name_Id := N + 053; -- GNAT
   Name_All_Calls_Remote               : constant Name_Id := N + 054;
   Name_Assert                         : constant Name_Id := N + 055; -- GNAT
   Name_Asynchronous                   : constant Name_Id := N + 056;
   Name_Atomic                         : constant Name_Id := N + 057;
   Name_Atomic_Components              : constant Name_Id := N + 058;
   Name_Attach_Handler                 : constant Name_Id := N + 059;
   Name_Checks_On                      : constant Name_Id := N + 060; -- GNAT
   Name_Controlled                     : constant Name_Id := N + 061;
   Name_Convention                     : constant Name_Id := N + 062;
   Name_Debug                          : constant Name_Id := N + 063; -- GNAT
   Name_Elaborate                      : constant Name_Id := N + 064; -- Ada 83
   Name_Elaborate_All                  : constant Name_Id := N + 065;
   Name_Elaborate_Body                 : constant Name_Id := N + 066;
   Name_Export                         : constant Name_Id := N + 067;
   Name_Import                         : constant Name_Id := N + 068;
   Name_Improve                        : constant Name_Id := N + 069; -- GNAT
   Name_Inline                         : constant Name_Id := N + 070;
   Name_Inspection_Point               : constant Name_Id := N + 071;
   Name_Interface                      : constant Name_Id := N + 072; -- Ada 83
   Name_Interface_Name                 : constant Name_Id := N + 073; -- GNAT
   Name_Interrupt_Handler              : constant Name_Id := N + 074;
   Name_Interrupt_Priority             : constant Name_Id := N + 075;
   Name_List                           : constant Name_Id := N + 076;
   Name_Locking_Policy                 : constant Name_Id := N + 077;
   Name_Memory_Size                    : constant Name_Id := N + 078; -- Ada 83
   Name_Normalize_Scalars              : constant Name_Id := N + 079;
   Name_Optimize                       : constant Name_Id := N + 080;
   Name_Pack                           : constant Name_Id := N + 081;
   Name_Page                           : constant Name_Id := N + 082;
   Name_Preelaborate                   : constant Name_Id := N + 083;
   Name_Priority                       : constant Name_Id := N + 084;
   Name_Pure                           : constant Name_Id := N + 085;
   Name_Queuing_Policy                 : constant Name_Id := N + 086;
   Name_Remote_Call_Interface          : constant Name_Id := N + 087;
   Name_Remote_Types                   : constant Name_Id := N + 088;
   Name_Restrictions                   : constant Name_Id := N + 089;
   Name_Reviewable                     : constant Name_Id := N + 090;
   Name_Shared                         : constant Name_Id := N + 091; -- Ada 83
   Name_Shared_Passive                 : constant Name_Id := N + 092;

   --  Note: Storage_Size is not in this list because its name matches the
   --  name of the corresponding attribute. However, it is included in the
   --  definition of the type Attribute_Id, and the functions Get_Pragma_Id
   --  and Check_Pragma_Id correcly recognize and process Name_Storage_Size.

   --  Note: Storage_Unit is also omitted from the list because of a clash
   --  with an attribute name, and is treated similarly.

   Name_Suppress                       : constant Name_Id := N + 093;
   Name_System_Name                    : constant Name_Id := N + 094; -- Ada 83
   Name_Task_Dispatching_Policy        : constant Name_Id := N + 095;
   Name_Unimplemented_Unit             : constant Name_Id := N + 096; -- GNAT
   Name_Volatile                       : constant Name_Id := N + 097;
   Name_Volatile_Components            : constant Name_Id := N + 098;
   Last_Pragma_Name                    : constant Name_Id := N + 098;

   --  Language convention names for pragma Convention/Export/Import/Interface
   --  Note that Name_C is not included in this list, since it was already
   --  declared earlier in the context of one-character identifier names
   --  (where the order is critical to the fast look up process).

   First_Convention_Name               : constant Name_Id := N + 099;
   Name_Ada                            : constant Name_Id := N + 099;
   Name_Asm                            : constant Name_Id := N + 100;
   Name_Assembler                      : constant Name_Id := N + 101;
   Name_Intrinsic                      : constant Name_Id := N + 102;
   Last_Convention_Name                : constant Name_Id := N + 102;

   --  Other special names used in processing pragma arguments

   Name_Entity                         : constant Name_Id := N + 103;
   Name_External_Name                  : constant Name_Id := N + 104;
   Name_Gcc                            : constant Name_Id := N + 105;
   Name_Gnat                           : constant Name_Id := N + 106;
   Name_Link_Name                      : constant Name_Id := N + 107;
   Name_Off                            : constant Name_Id := N + 108;
   Name_On                             : constant Name_Id := N + 109;
   Name_Space                          : constant Name_Id := N + 110;
   Name_Time                           : constant Name_Id := N + 111;

   --  Names of recognized attributes. The entries with the comment "Ada 83"
   --  are attributes that are defined in Ada 83, but not in Ada 9X. These
   --  attributes are implemented in both Ada 83 and Ada 9X modes in GNAT.

   First_Attribute_Name                : constant Name_Id := N + 112;
   Name_Abort_Signal                   : constant Name_Id := N + 112;  -- GNAT
   Name_Access                         : constant Name_Id := N + 113;
   Name_Address                        : constant Name_Id := N + 114;
   Name_Address_Size                   : constant Name_Id := N + 115;  -- GNAT
   Name_Adjacent                       : constant Name_Id := N + 116;
   Name_Aft                            : constant Name_Id := N + 117;
   Name_Alignment                      : constant Name_Id := N + 118;
   Name_Bit_Order                      : constant Name_Id := N + 119;
   Name_Body_Version                   : constant Name_Id := N + 120;
   Name_Callable                       : constant Name_Id := N + 121;
   Name_Caller                         : constant Name_Id := N + 122;
   Name_Ceiling                        : constant Name_Id := N + 123;
   Name_Component_Size                 : constant Name_Id := N + 124;
   Name_Compose                        : constant Name_Id := N + 125;
   Name_Constrained                    : constant Name_Id := N + 126;
   Name_Copy_Sign                      : constant Name_Id := N + 127;
   Name_Count                          : constant Name_Id := N + 128;
   Name_Definite                       : constant Name_Id := N + 129;
   Name_Delta                          : constant Name_Id := N + 130;
   Name_Denorm                         : constant Name_Id := N + 131;
   Name_Digits                         : constant Name_Id := N + 132;
   Name_Emax                           : constant Name_Id := N + 133; -- Ada 83
   Name_Epsilon                        : constant Name_Id := N + 134; -- Ada 83
   Name_Exponent                       : constant Name_Id := N + 135;
   Name_External_Tag                   : constant Name_Id := N + 136;
   Name_First                          : constant Name_Id := N + 137;
   Name_First_Bit                      : constant Name_Id := N + 138;
   Name_Floor                          : constant Name_Id := N + 139;
   Name_Fore                           : constant Name_Id := N + 140;
   Name_Fraction                       : constant Name_Id := N + 141;
   Name_Identity                       : constant Name_Id := N + 142;
   Name_Image                          : constant Name_Id := N + 143;
   Name_Input                          : constant Name_Id := N + 144;
   Name_Large                          : constant Name_Id := N + 145; -- Ada 83
   Name_Last                           : constant Name_Id := N + 146;
   Name_Last_Bit                       : constant Name_Id := N + 147;
   Name_Leading_Part                   : constant Name_Id := N + 148;
   Name_Length                         : constant Name_Id := N + 149;
   Name_Machine                        : constant Name_Id := N + 150;
   Name_Machine_Emax                   : constant Name_Id := N + 151;
   Name_Machine_Emin                   : constant Name_Id := N + 152;
   Name_Machine_Mantissa               : constant Name_Id := N + 153;
   Name_Machine_Overflows              : constant Name_Id := N + 154;
   Name_Machine_Radix                  : constant Name_Id := N + 155;
   Name_Machine_Rounds                 : constant Name_Id := N + 156;
   Name_Mantissa                       : constant Name_Id := N + 157; -- Ada 83
   Name_Max                            : constant Name_Id := N + 158;
   Name_Max_Interrupt_Priority         : constant Name_Id := N + 159; -- GNAT
   Name_Max_Priority                   : constant Name_Id := N + 160; -- GNAT
   Name_Max_Size_In_Storage_Elements   : constant Name_Id := N + 161;
   Name_Min                            : constant Name_Id := N + 162;
   Name_Model                          : constant Name_Id := N + 163;
   Name_Model_Emin                     : constant Name_Id := N + 164;
   Name_Model_Epsilon                  : constant Name_Id := N + 165;
   Name_Model_Mantissa                 : constant Name_Id := N + 166;
   Name_Model_Small                    : constant Name_Id := N + 167;
   Name_Output                         : constant Name_Id := N + 168;
   Name_Pos                            : constant Name_Id := N + 169;
   Name_Position                       : constant Name_Id := N + 170;
   Name_Pred                           : constant Name_Id := N + 171;
   Name_Range                          : constant Name_Id := N + 172;
   Name_Read                           : constant Name_Id := N + 173;
   Name_Remainder                      : constant Name_Id := N + 174;
   Name_Round                          : constant Name_Id := N + 175;
   Name_Rounding                       : constant Name_Id := N + 176;
   Name_Safe_Emax                      : constant Name_Id := N + 177; -- Ada 83
   Name_Safe_First                     : constant Name_Id := N + 178;
   Name_Safe_Large                     : constant Name_Id := N + 179; -- Ada 83
   Name_Safe_Last                      : constant Name_Id := N + 180;
   Name_Safe_Small                     : constant Name_Id := N + 181; -- Ada 83
   Name_Scale                          : constant Name_Id := N + 182;
   Name_Scaling                        : constant Name_Id := N + 183;
   Name_Signed_Zeros                   : constant Name_Id := N + 184;
   Name_Size                           : constant Name_Id := N + 185;
   Name_Small                          : constant Name_Id := N + 186;
   Name_Storage_Pool                   : constant Name_Id := N + 187;
   Name_Storage_Size                   : constant Name_Id := N + 188;
   Name_Storage_Unit                   : constant Name_Id := N + 189; -- GNAT
   Name_Succ                           : constant Name_Id := N + 190;
   Name_Tag                            : constant Name_Id := N + 191;
   Name_Terminated                     : constant Name_Id := N + 192;
   Name_Tick                           : constant Name_Id := N + 193; -- GNAT
   Name_Truncation                     : constant Name_Id := N + 194;
   Name_Unbiased_Rounding              : constant Name_Id := N + 195;
   Name_Unchecked_Access               : constant Name_Id := N + 196;
   Name_Universal_Literal_String       : constant Name_Id := N + 197; -- GNAT
   Name_Unrestricted_Access            : constant Name_Id := N + 198; -- GNAT
   Name_Val                            : constant Name_Id := N + 199;
   Name_Valid                          : constant Name_Id := N + 200;
   Name_Value                          : constant Name_Id := N + 201;
   Name_Version                        : constant Name_Id := N + 202;
   Name_Wide_Image                     : constant Name_Id := N + 203;
   Name_Wide_Value                     : constant Name_Id := N + 204;
   Name_Width                          : constant Name_Id := N + 205;
   Name_Word_Size                      : constant Name_Id := N + 206; -- GNAT
   Name_Write                          : constant Name_Id := N + 207;

   --  These attributes are the ones that return types

   First_Type_Attribute_Name           : constant Name_Id := N + 208;
   Name_Base                           : constant Name_Id := N + 208;
   Name_Class                          : constant Name_Id := N + 209;
   Last_Type_Attribute_Name            : constant Name_Id := N + 209;

   Last_Attribute_Name                 : constant Name_Id := N + 209;

   --  Names of recognized locking policy identifiers

   First_Locking_Policy_Name           : constant Name_Id := N + 210;
   Name_Ceiling_Locking                : constant Name_Id := N + 210;
   Last_Locking_Policy_Name            : constant Name_Id := N + 210;

   --  Names of recognized locking policy identifiers

   First_Queuing_Policy_Name           : constant Name_Id := N + 211;
   Name_FIFO_Queuing                   : constant Name_Id := N + 211;
   Name_Priority_Queuing               : constant Name_Id := N + 212;
   Last_Queuing_Policy_Name            : constant Name_Id := N + 212;

   --  Names of recognized task dispatching policy identifiers

   First_Task_Dispatching_Policy_Name  : constant Name_Id := N + 213;
   Name_Fifo_Within_Priorities         : constant Name_Id := N + 213;
   Last_Task_Dispatching_Policy_Name   : constant Name_Id := N + 213;

   --  Names of recognized checks for pragma Suppress

   First_Check_Name                    : constant Name_Id := N + 214;
   Name_Access_Check                   : constant Name_Id := N + 214;
   Name_Accessibility_Check            : constant Name_Id := N + 215;
   Name_Discriminant_Check             : constant Name_Id := N + 216;
   Name_Division_Check                 : constant Name_Id := N + 217;
   Name_Elaboration_Check              : constant Name_Id := N + 218;
   Name_Index_Check                    : constant Name_Id := N + 219;
   Name_Length_Check                   : constant Name_Id := N + 220;
   Name_Overflow_Check                 : constant Name_Id := N + 221;
   Name_Range_Check                    : constant Name_Id := N + 222;
   Name_Storage_Check                  : constant Name_Id := N + 223;
   Name_Tag_Check                      : constant Name_Id := N + 224;
   Name_All_Checks                     : constant Name_Id := N + 225;
   Last_Check_Name                     : constant Name_Id := N + 225;

   --  Names corresponding to reserved keywords, excluding those already
   --  declared in the attribute list (Access, Delta, Digits, Range).

   Name_Abort                          : constant Name_Id := N + 226;
   Name_Abs                            : constant Name_Id := N + 227;
   Name_Abstract                       : constant Name_Id := N + 228;
   Name_Accept                         : constant Name_Id := N + 229;
   Name_And                            : constant Name_Id := N + 230;
   Name_All                            : constant Name_Id := N + 231;
   Name_Array                          : constant Name_Id := N + 232;
   Name_At                             : constant Name_Id := N + 233;
   Name_Begin                          : constant Name_Id := N + 234;
   Name_Body                           : constant Name_Id := N + 235;
   Name_Case                           : constant Name_Id := N + 236;
   Name_Constant                       : constant Name_Id := N + 237;
   Name_Declare                        : constant Name_Id := N + 238;
   Name_Delay                          : constant Name_Id := N + 239;
   Name_Do                             : constant Name_Id := N + 240;
   Name_Else                           : constant Name_Id := N + 241;
   Name_Elsif                          : constant Name_Id := N + 242;
   Name_End                            : constant Name_Id := N + 243;
   Name_Entry                          : constant Name_Id := N + 244;
   Name_Exception                      : constant Name_Id := N + 245;
   Name_Exit                           : constant Name_Id := N + 246;
   Name_For                            : constant Name_Id := N + 247;
   Name_Function                       : constant Name_Id := N + 248;
   Name_Generic                        : constant Name_Id := N + 249;
   Name_Goto                           : constant Name_Id := N + 250;
   Name_If                             : constant Name_Id := N + 251;
   Name_In                             : constant Name_Id := N + 252;
   Name_Is                             : constant Name_Id := N + 253;
   Name_Limited                        : constant Name_Id := N + 254;
   Name_Loop                           : constant Name_Id := N + 255;
   Name_Mod                            : constant Name_Id := N + 256;
   Name_New                            : constant Name_Id := N + 257;
   Name_Not                            : constant Name_Id := N + 258;
   Name_Null                           : constant Name_Id := N + 259;
   Name_Of                             : constant Name_Id := N + 260;
   Name_Or                             : constant Name_Id := N + 261;
   Name_Others                         : constant Name_Id := N + 262;
   Name_Out                            : constant Name_Id := N + 263;
   Name_Package                        : constant Name_Id := N + 264;
   Name_Pragma                         : constant Name_Id := N + 265;
   Name_Private                        : constant Name_Id := N + 266;
   Name_Procedure                      : constant Name_Id := N + 267;
   Name_Raise                          : constant Name_Id := N + 268;
   Name_Record                         : constant Name_Id := N + 269;
   Name_Rem                            : constant Name_Id := N + 270;
   Name_Renames                        : constant Name_Id := N + 271;
   Name_Return                         : constant Name_Id := N + 272;
   Name_Reverse                        : constant Name_Id := N + 273;
   Name_Select                         : constant Name_Id := N + 274;
   Name_Separate                       : constant Name_Id := N + 275;
   Name_Subtype                        : constant Name_Id := N + 276;
   Name_Task                           : constant Name_Id := N + 277;
   Name_Terminate                      : constant Name_Id := N + 278;
   Name_Then                           : constant Name_Id := N + 279;
   Name_Type                           : constant Name_Id := N + 280;
   Name_Use                            : constant Name_Id := N + 281;
   Name_When                           : constant Name_Id := N + 282;
   Name_While                          : constant Name_Id := N + 283;
   Name_With                           : constant Name_Id := N + 284;
   Name_Xor                            : constant Name_Id := N + 285;

   --  Names of intrinsic subprograms

   First_Intrinsic_Name                : constant Name_Id := N + 286;
   Name_Rotate_Left                    : constant Name_Id := N + 286;
   Name_Rotate_Right                   : constant Name_Id := N + 287;
   Name_Shift_Left                     : constant Name_Id := N + 288;
   Name_Shift_Right                    : constant Name_Id := N + 289;
   Name_Shift_Right_Arithmetic         : constant Name_Id := N + 290;
   Name_Unchecked_Deallocate           : constant Name_Id := N + 291;
   Last_Intrinsic_Name                 : constant Name_Id := N + 291;

   --  Reserved words used only in Ada 9X

   First_9X_Reserved_Word              : constant Name_Id := N + 292;
   Name_Aliased                        : constant Name_Id := N + 292;
   Name_Protected                      : constant Name_Id := N + 293;
   Name_Until                          : constant Name_Id := N + 294;
   Name_Requeue                        : constant Name_Id := N + 295;
   Name_Tagged                         : constant Name_Id := N + 296;
   Last_9X_Reserved_Word               : constant Name_Id := N + 296;

   subtype Ada_9X_Reserved_Words is
     Name_Id range First_9X_Reserved_Word .. Last_9X_Reserved_Word;

   --  Mark last defined name for consistency check in Snames body

   Last_Predefined_Name                : constant Name_Id := N + 296;

   subtype Any_Operator_Name is Name_Id range
     First_Operator_Name .. Last_Operator_Special_Name;
   --  Subtype covering all the encoded operator and special operator names,
   --  used for decoding names back to their original form.

   ------------------------------
   -- Attribute ID Definitions --
   ------------------------------

   type Attribute_Id is (
      Attribute_Abort_Signal,
      Attribute_Access,
      Attribute_Address,
      Attribute_Address_Size,
      Attribute_Adjacent,
      Attribute_Aft,
      Attribute_Alignment,
      Attribute_Bit_Order,
      Attribute_Body_Version,
      Attribute_Callable,
      Attribute_Caller,
      Attribute_Ceiling,
      Attribute_Component_Size,
      Attribute_Compose,
      Attribute_Constrained,
      Attribute_Copy_Sign,
      Attribute_Count,
      Attribute_Definite,
      Attribute_Delta,
      Attribute_Denorm,
      Attribute_Digits,
      Attribute_Emax,
      Attribute_Epsilon,
      Attribute_Exponent,
      Attribute_External_Tag,
      Attribute_First,
      Attribute_First_Bit,
      Attribute_Floor,
      Attribute_Fore,
      Attribute_Fraction,
      Attribute_Identity,
      Attribute_Image,
      Attribute_Input,
      Attribute_Large,
      Attribute_Last,
      Attribute_Last_Bit,
      Attribute_Leading_Part,
      Attribute_Length,
      Attribute_Machine,
      Attribute_Machine_Emax,
      Attribute_Machine_Emin,
      Attribute_Machine_Mantissa,
      Attribute_Machine_Overflows,
      Attribute_Machine_Radix,
      Attribute_Machine_Rounds,
      Attribute_Mantissa,
      Attribute_Max,
      Attribute_Max_Interrupt_Priority,
      Attribute_Max_Priority,
      Attribute_Max_Size_In_Storage_Elements,
      Attribute_Min,
      Attribute_Model,
      Attribute_Model_Emin,
      Attribute_Model_Epsilon,
      Attribute_Model_Mantissa,
      Attribute_Model_Small,
      Attribute_Output,
      Attribute_Pos,
      Attribute_Position,
      Attribute_Pred,
      Attribute_Range,
      Attribute_Read,
      Attribute_Remainder,
      Attribute_Round,
      Attribute_Rounding,
      Attribute_Safe_Emax,
      Attribute_Safe_First,
      Attribute_Safe_Large,
      Attribute_Safe_Last,
      Attribute_Safe_Small,
      Attribute_Scale,
      Attribute_Scaling,
      Attribute_Signed_Zeros,
      Attribute_Size,
      Attribute_Small,
      Attribute_Storage_Pool,
      Attribute_Storage_Size,
      Attribute_Storage_Unit,
      Attribute_Succ,
      Attribute_Tag,
      Attribute_Terminated,
      Attribute_Tick,
      Attribute_Truncation,
      Attribute_Unbiased_Rounding,
      Attribute_Unchecked_Access,
      Attribute_Universal_Literal_String,
      Attribute_Unrestricted_Access,
      Attribute_Val,
      Attribute_Valid,
      Attribute_Value,
      Attribute_Version,
      Attribute_Wide_Image,
      Attribute_Wide_Value,
      Attribute_Width,
      Attribute_Word_Size,
      Attribute_Write,

      --  Type attributes

      Attribute_Base,
      Attribute_Class);

   subtype Type_Attribute_Id is Attribute_Id range
     Attribute_Base .. Attribute_Class;

   -------------------------------
   -- Check Name ID Definitions --
   -------------------------------

   type Check_Id is (
      Access_Check,
      Accessibility_Check,
      Discriminant_Check,
      Division_Check,
      Elaboration_Check,
      Index_Check,
      Length_Check,
      Overflow_Check,
      Range_Check,
      Storage_Check,
      Tag_Check,
      All_Checks);

   ------------------------------------
   -- Convention Name ID Definitions --
   ------------------------------------

   type Convention_Id is (
      Convention_Ada,
      Convention_Asm,
      Convention_Assembler,
      Convention_C,
      Convention_Intrinsic);

   -----------------------------------
   -- Locking Policy ID Definitions --
   -----------------------------------

   type Locking_Policy_Id is (
      Locking_Policy_Ceiling_Locking);

   ---------------------------
   -- Pragma ID Definitions --
   ---------------------------

   type Pragma_Id is (
      Pragma_Abort_Defer,
      Pragma_Ada_83,
      Pragma_Ada_9X,
      Pragma_All_Calls_Remote,
      Pragma_Assert,
      Pragma_Asynchronous,
      Pragma_Atomic,
      Pragma_Atomic_Components,
      Pragma_Attach_Handler,
      Pragma_Checks_On,
      Pragma_Controlled,
      Pragma_Convention,
      Pragma_Debug,
      Pragma_Elaborate,
      Pragma_Elaborate_All,
      Pragma_Elaborate_Body,
      Pragma_Export,
      Pragma_Import,
      Pragma_Improve,
      Pragma_Inline,
      Pragma_Inspection_Point,
      Pragma_Interface,
      Pragma_Interface_Name,
      Pragma_Interrupt_Handler,
      Pragma_Interrupt_Priority,
      Pragma_List,
      Pragma_Locking_Policy,
      Pragma_Memory_Size,
      Pragma_Normalize_Scalars,
      Pragma_Optimize,
      Pragma_Pack,
      Pragma_Page,
      Pragma_Preelaborate,
      Pragma_Priority,
      Pragma_Pure,
      Pragma_Queuing_Policy,
      Pragma_Remote_Call_Interface,
      Pragma_Remote_Types,
      Pragma_Restrictions,
      Pragma_Reviewable,
      Pragma_Shared,
      Pragma_Shared_Passive,
      Pragma_Suppress,
      Pragma_System_Name,
      Pragma_Task_Dispatching_Policy,
      Pragma_Unimplemented_Unit,
      Pragma_Volatile,
      Pragma_Volatile_Components,

      --  Note: pragma Storage_Size is on its own, out of order, because of the
      --  special processing required to deal with the fact that its name is
      --  the same as an attribute name. If other such pragmas are introduced,
      --  they will have to be similarly treate.

      --  The following pragmas are on their own, out of order, because of
      --  the special processing required to deal with the fact that their
      --  names match existing attribute names.

      Pragma_Storage_Size,
      Pragma_Storage_Unit);

   ------------------------------------
   -- Queueing Policy ID definitions --
   ------------------------------------

   type Queuing_Policy_Id is (
      Queuing_Policy_FIFO_Queuing,
      Queuing_Policy_Priority_Queuing);

   --------------------------------------------
   -- Task Dispatching Policy ID definitions --
   --------------------------------------------

   type Task_Dispatching_Policy_Id is (
      Task_Dispatching_FIFO_Within_Priorities);
   --  Id values used to identify task dispatching policies

   -----------------
   -- Subprograms --
   -----------------

   procedure Initialize;
   --  Called to initialize the preset names in the names table.

   function Is_Attribute_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Attribute_Name);
   --  Test to see if the name N is the name of a recognized attribute

   function Is_Type_Attribute_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Type_Attribute_Name);
   --  Test to see if the name N is the name of a recognized type
   --  returning attribute

   function Is_Check_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Check_Name);
   --  Test to see if the name N is the name of a recognized suppress check
   --  as required by pragma Suppress.

   function Is_Convention_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Convention_Name);
   --  Test to see if the name N is the name of one of the recognized language
   --  conventions, as required by pragma Convention, Import, Export, Interface

   function Is_Locking_Policy_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Locking_Policy_Name);
   --  Test to see if the name N is the name of a recognized locking policy

   function Is_Operator_Symbol_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Operator_Symbol_Name);
   --  Test to see if the name N is the name of an operator symbol

   function Is_Pragma_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Pragma_Name);
   --  Test to see if the name N is the name of a recognized pragma. Note
   --  that pragma Storage_Size is recognized as a pragma by this function
   --  even though its name is separate from the other pragma names.

   function Is_Queuing_Policy_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Queuing_Policy_Name);
   --  Test to see if the name N is the name of a recognized queuing policy

   function Is_Task_Dispatching_Policy_Name (N : Name_Id) return Boolean;
   pragma Inline (Is_Task_Dispatching_Policy_Name);
   --  Test to see if the name N is the name of a recognized
   --  task dispatching policy

   function Get_Attribute_Id (N : Name_Id) return Attribute_Id;
   --  Returns Id of attribute corresponding to given name. It is an error to
   --  call this function with a name that is not the name of a attribute.

   function Get_Convention_Id (N : Name_Id) return Convention_Id;
   --  Returns Id of language convention corresponding to given name. It is an
   --  to call this function with a name that is not the name of a check.

   function Get_Check_Id (N : Name_Id) return Check_Id;
   --  Returns Id of suppress check corresponding to given name. It is an error
   --  to call this function with a name that is not the name of a check.

   function Get_Locking_Policy_Id (N : Name_Id) return Locking_Policy_Id;
   --  Returns Id of locking policy corresponding to given name. It is an error
   --  to call this function with a name that is not the name of a check.

   function Get_Pragma_Id (N : Name_Id) return Pragma_Id;
   --  Returns Id of pragma corresponding to given name. It is an error to
   --  call this function with a name that is not the name of a pragma. Note
   --  that Name_Storage_Size correctly returns Pragma_Storage_Size, even
   --  though its name is separated from the other pragma names.

   function Get_Queuing_Policy_Id (N : Name_Id) return Queuing_Policy_Id;
   --  Returns Id of queuing policy corresponding to given name. It is an error
   --  to call this function with a name that is not the name of a check.

   function Get_Task_Dispatching_Policy_Id (N : Name_Id)
     return Task_Dispatching_Policy_Id;
   --  Returns Id of task dispatching policy corresponding to given name. It
   --  is an error to call this function with a name that is not the name
   --  of a check.

end Snames;
