------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ D I S P                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.31 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Elists;   use Elists;
with Einfo;    use Einfo;
with Exp_Ch6;  use Exp_Ch6;
with Errout;   use Errout;
with Nlists;   use Nlists;
with Output;   use Output;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;

package body Sem_Disp is

   ---------------------------
   -- Find_Dispatching_Type --
   ---------------------------

   function Find_Dispatching_Type (Subp : Entity_Id) return Entity_Id is
      Formal      : Entity_Id := First_Formal (Subp);
      Tagged_Type : Entity_Id;
      Tagged_Seen : Entity_Id := Empty;

      procedure Check_Controlling_Type (T : Entity_Id) is
      begin
         if Is_Tagged_Type (T) then
            Tagged_Type := T;

         elsif Ekind (T) = E_Anonymous_Access_Type
           and then Is_Tagged_Type (Designated_Type (T))
         then
            Tagged_Type := Designated_Type (T);

         else
            Tagged_Type := Empty;
         end if;

         if Present (Tagged_Type)
           and then not Is_Class_Wide_Type (Tagged_Type)
         then
            if Present (Tagged_Seen) and then Tagged_Type /= Tagged_Seen then
               Error_Msg_N
                 ("operation can be dispatching in only one type", Subp);
            else
               Tagged_Seen := Tagged_Type;
            end if;
         end if;
      end Check_Controlling_Type;

   --  Start of processing Find_Dispatching_Type

   begin
      while Present (Formal) loop
         Check_Controlling_Type (Etype (Formal));
         Formal := Next_Formal (Formal);
      end loop;

      --  The subprogram may also be dispatching on result

      if Present (Etype (Subp)) then
         Check_Controlling_Type (Etype (Subp));
      end if;

      return Tagged_Seen;
   end Find_Dispatching_Type;

   ----------------------------------
   -- Check_Dispatching_Operation  --
   ----------------------------------

   procedure Check_Dispatching_Operation (Subp, Old_Subp : Entity_Id) is
      Tagged_Seen : constant Entity_Id := Find_Dispatching_Type (Subp);
      N           : Node_Id;

   begin
      Set_Is_Dispatching_Operation (Subp, False);

      if No (Tagged_Seen) then
         return;

      --  If the subprogram is not defined right after the tagged type
      --  it is not a primitive operation

      elsif Scope (Subp) /= Scope (Tagged_Seen) then
         return;

      --  The subprograms build internally after the freezing point (such as
      --  the Init procedure) are not primitives

      elsif Is_Frozen (Tagged_Seen) and then not Comes_From_Source (Subp) then
         return;

      --  If the type is not defined in a package spec, that's a potential
      --  error in most cases we, at least, need a warning unless we are just
      --  overriding the current primitive operations

      elsif (Ekind (Scope (Subp)) /= E_Package
               and then Ekind (Scope (Subp)) /= E_Generic_Package)
        or else Is_Package_Body (Scope (Subp))
      then
         if not Comes_From_Source (Subp)
            or else (Present (Old_Subp) and then not Is_Frozen (Tagged_Seen))
         then
            null;

         elsif Present (Old_Subp) then
            Error_Msg_N ("?not overriding (subprogram spec"
              & " should be right after the type)", Subp);

         else
            Error_Msg_N ("?not dispatching (primitive "
              & "operations must be defined in a package spec)", Subp);
            return;
         end if;

      --  Now, we are sure that the scope is a package spec. If the subprogram
      --  is declared after the freezing point ot the type that's an error

      elsif Is_Frozen (Tagged_Seen) then
         Error_Msg_N ("this primitive operation is declared too late", Subp);

         N := Parent (Tagged_Seen);
         while Present (N)
           and then (Nkind (N) /= N_Freeze_Entity
                       or else Entity (N) /= Tagged_Seen)
         loop
            N := Next (N);
         end loop;

         if Present (N) then
            Error_Msg_NE ("?no primitive operations for & after this line",
              N, Tagged_Seen);
         end if;

         return;
      end if;

      --  Now it should be a correct primitive operation

      Append_Elmt (Subp, Primitive_Operations (Tagged_Seen));
      Set_Is_Dispatching_Operation (Subp, True);
   end Check_Dispatching_Operation;

   ----------------------------
   -- Check_Dispatching_Call --
   ----------------------------

   procedure Check_Dispatching_Call (N : Node_Id) is
      Actual  : Node_Id;
      Control : Node_Id := Empty;

   begin
      --  Find a controlling argument, if any

      if Present (Parameter_Associations (N)) then
         Actual := First_Actual (N);

         while Present (Actual) loop
            Control := Find_Controlling_Arg (Actual);
            exit when Present (Control);
            Actual := Next_Actual (Actual);
         end loop;

         if Present (Control) then

            --  Verify that no controlling arguments are statically tagged

            if Debug_Flag_E then
               Write_Str ("Found Dispatching call");
               Write_Int (Int (N));
               Write_Eol;
            end if;

            Actual := First_Actual (N);

            while Present (Actual) loop
               if Actual /= Control then

                  if not Is_Tagged_Type (Etype (Actual)) then
                     null; -- can be anything

                  elsif (Is_Dynamically_Tagged (Actual)) then
                     null; --  valid parameter

                  elsif Is_Tag_Indeterminate (Actual) then

                     --  The tag is inherited from the enclosing call (the
                     --  node we are currently analyzing). Explicitly expand
                     --  the actual, since the previous call to Expand
                     --  (from Resolve_Call) had now way of knowing about
                     --  the required dispatching.

                     Propagate_Tag (Control, Actual);

                  else
                     Error_Msg_N
                         ("all dispatching call controlling arguments "
                     & "must be dynamically tagged", Actual);
                     return;
                  end if;
               end if;

               Actual := Next_Actual (Actual);
            end loop;

            --  Mark call as a dispatching call

            Set_Controlling_Argument (N, Control);
         end if;

      else
         --  If dispatching on result, the enclosing call, if any, will
         --  determine the controlling argument. Otherwise this is the
         --  primitive operation of the root type.

         null;
      end if;
   end Check_Dispatching_Call;

   ---------------------------
   -- Is_Dynamically_Tagged --
   ---------------------------

   function Is_Dynamically_Tagged (N : Node_Id) return Boolean is
   begin
      return Find_Controlling_Arg (N) /= Empty;
   end Is_Dynamically_Tagged;

   --------------------------
   -- Find_Controlling_Arg --
   --------------------------

   function Find_Controlling_Arg (N : Node_Id) return Node_Id is
      Orig_Node : constant Node_Id := Original_Node (N);
      Typ       : Entity_Id;

   begin
      Typ := Etype (N);

      if Is_Access_Type (Typ) then
         Typ := Designated_Type (Typ);
      end if;

      if not Is_Tagged_Type (Typ) then
         return Empty;

      elsif (Is_Entity_Name (N)
               or else Nkind (N) = N_Explicit_Dereference
               or else Nkind (N) = N_Selected_Component
               or else Nkind (N) = N_Indexed_Component
               or else Nkind (N) = N_Allocator
               or else Nkind (N) = N_Type_Conversion
               or else Nkind (N) = N_Unchecked_Type_Conversion)
        and then Is_Class_Wide_Type (Typ)
      then
         return N;

      elsif Nkind (Orig_Node) = N_Function_Call
        and then Present (Controlling_Argument (Orig_Node))
      then
         return Controlling_Argument (Orig_Node);

      elsif Nkind (N) = N_Qualified_Expression then
         return Find_Controlling_Arg (Expression (N));

      else
         return Empty;
      end if;
   end Find_Controlling_Arg;

   --------------------------
   -- Is_Tag_Indeterminate --
   --------------------------

   function Is_Tag_Indeterminate (N : Node_Id) return Boolean is
      Nam       : Entity_Id;
      Actual    : Node_Id;
      Orig_Node : constant Node_Id := Original_Node (N);

   begin
      if Nkind (Orig_Node) = N_Function_Call then
         Nam := Entity (Name (Orig_Node));

         if Present (Parameter_Associations (Orig_Node)) then
            Actual := First_Actual (Orig_Node);

            while Present (Actual) loop
               if Is_Tagged_Type (Etype (Actual))
                 and then Is_Dynamically_Tagged (Actual)
               then
                  return False; -- one operand is dispatching
               end if;

               Actual := Next_Actual (Actual);
            end loop;

            return True;

         --  If there are no actuals, the call is tag-indeterminate

         else
            return True;
         end if;

      elsif Nkind (Orig_Node) = N_Qualified_Expression then
         return Is_Tag_Indeterminate (Expression (Orig_Node));

      else
         return False;
      end if;
   end Is_Tag_Indeterminate;

   ------------------------------------
   -- Override_Dispatching_Operation --
   ------------------------------------

   procedure Override_Dispatching_Operation (Prev_Op, New_Op : Entity_Id) is
      Formal      : Entity_Id;
      Tagged_Type : Entity_Id;
      Op_Elmt     : Elmt_Id;

   begin

      --  The searched tagged type is the type of the first formal having a
      --  tagged non class wide type (controlling argument).

      Formal := First_Entity (Prev_Op);
      Tagged_Type := Empty;

      while Present (Formal) loop
         if Is_Tagged_Type (Etype (Formal))
           and then not Is_Class_Wide_Type (Etype (Formal))
         then
            Tagged_Type := Etype (Formal);
            exit;
         end if;

         Formal := Next_Entity (Formal);
      end loop;

      --  If we did not find tagged type among formals, then the subprogram
      --  must be a function that is dispatching on the result type.

      if No (Tagged_Type) then
         Tagged_Type := Etype (Prev_Op);
      end if;

      Op_Elmt := First_Elmt (Primitive_Operations (Tagged_Type));

      while Node (Op_Elmt) /= Prev_Op loop
         Op_Elmt := Next_Elmt (Op_Elmt);
      end loop;

      Replace_Elmt (Op_Elmt, New_Op);
      Remove_Last_Elmt (Primitive_Operations (Tagged_Type));
   end Override_Dispatching_Operation;

   -------------------
   -- Propagate_Tag --
   -------------------

   procedure Propagate_Tag (Control : Node_Id; Actual : Node_Id) is
      Call_Node : Node_Id;
      Arg       : Node_Id;

   begin
      if Nkind (Actual) = N_Function_Call then
         Call_Node := Actual;

      --  Only other possibility is parenthesized or qualified expression

      else
         Call_Node := Expression (Actual);
      end if;

      Set_Controlling_Argument (Call_Node, Control);
      Arg := First_Actual (Call_Node);

      while Present (Arg) loop
         if Is_Tag_Indeterminate (Arg) then
            Propagate_Tag (Control,  Arg);
         end if;

         Arg := Next_Actual (Arg);
      end loop;

      Expand_Dispatch_Call (Call_Node);
   end Propagate_Tag;

end Sem_Disp;
