/* Input line implementation
   Copyright (C) 1994 Miguel de Icaza, Mauricio Plaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <sys/types.h>
#include <malloc.h>
#include "input.h"
#include "util.h"
#include "global.h"
#include "main.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/input.c,v 1.4 1994/06/27 22:53:07 miguel Exp $";

void create_input (Input *in, int x, int y, WINDOW *win, int color, int field_len, char *def_text)
{
    in->start_x = x;
    in->start_y = y;
    in->current_max_len = field_len+1;
    in->window = win;
    in->buffer = xmalloc (field_len+1, "create_input");
    in->color = color;
    in->field_len = field_len;
    in->first = 1;
    in->first_shown = 0;
    leaveok (win, FALSE);

    strcpy (in->buffer, def_text);
    in->pos = strlen (in->buffer);
    update_input (in);
    in->first = 1;
}

void update_input (Input *in)
{
    int    x = in->start_x;
    int    y = in->start_y;
    WINDOW *w = in->window;
    int    i, j;

    if ((in->pos < in->first_shown) || (in->pos >= in->first_shown+in->field_len))
	in->first_shown = (in->pos / in->field_len) * in->field_len;
    wstandend (w);
    wattron (w, in->color);
    wmove (w, y, x);
    for (i = 0; i < in->field_len; i++)
	waddch (w, ' ');
    wmove (w, y, x);
    for (i = 0, j = in->first_shown; i < in->field_len && in->buffer [j]; i++)
	waddch (w, in->buffer [j++]);
    wmove (w, y, x + (in->pos - in->first_shown));
    wrefresh (w);
    in->first = 0;
}

static void insert_char (Input *in, int c_code)
{
    int i;

    if (strlen (in->buffer)+1 == in->current_max_len){
	/* Expand the buffer */
	char *narea = realloc(in->buffer, in->current_max_len + in->field_len);
	if (narea){
	    in->buffer = narea;
	    in->current_max_len += in->field_len;
	}
    }
    if (strlen (in->buffer)+1 < in->current_max_len){
	int l = strlen (&in->buffer [in->pos]);
	for (i = l+2; i > 0; i--)
	    in->buffer [in->pos+i] = in->buffer [in->pos+i-1];
	in->buffer [in->pos] = c_code;
	in->pos++;
    }
}

static void forward_word (Input *in)
{
    char *p = in->buffer+in->pos;

    while (*p && isspace (*p))
	p++;
    while (*p && isalpha (*p))
	p++;
    in->pos = p - in->buffer;
}

static void backward_word (Input *in)
{
    char *p = in->buffer+in->pos;

    while (p-1 > in->buffer-1 && isspace (*(p-1)))
	p--;
    while (p-1 > in->buffer-1 && isalpha (*(p-1)))
	p--;
    in->pos = p - in->buffer;
}

void handle_char (Input *in, int c_code)
{
    int    i;
    char   *p;
    static char memory [100] = "";
    static int x_buf = 0, y_buf = 0; 

    if (quote){
	insert_char (in, c_code);
	update_input (in);
	return;
    }

    switch (c_code){
    case 1:			/* C-a */
	in->pos = 0;
	break;

    case 5:			/* C-e */
	in->pos = strlen (in->buffer);
	break;
	
    case 2:			/* C-b */
    case KEY_LEFT:
	if (in->pos)
	    in->pos--;
	break;

    case KEY_BACKSPACE:
    case 0177:
    case 8:			/* C-h */
	if (!in->pos)
	    break;
	for (i = in->pos; in->buffer [i-1]; i++)
	    in->buffer [i-1] = in->buffer [i];
	in->pos--;
	break;

    case 0:			/* C-2 */
	x_buf = in->pos;
	if (y_buf < x_buf)
	    y_buf= x_buf;
	break;

    case XCTRL('w'):			/* C-w */
	y_buf = in->pos;
	
    case ALT('w'):
	if (y_buf < x_buf)
	    x_buf=0;
	strncpy (memory, &in->buffer [x_buf], y_buf - x_buf);
	memory [y_buf-x_buf] = 0;
	break;

    case ALT('b'):
	backward_word (in);
	break;

    case ALT('f'):
	forward_word (in);
	break;
	
    case 0x19:			/* C-y */
	for (p = memory; *p; p++)
	    insert_char (in, *p);
	break;

    case 0xb:			/* C-k */
	strcpy (memory, &in->buffer [in->pos]);
	in->buffer [in->pos]=0;
	break;
	
    case 6:			/* C-f */
    case KEY_RIGHT:
	if (in->buffer [in->pos])
	    in->pos++;
	break;

    default:
	if (c_code > 255 || c_code < ' ')
	    break;
	if (in->first){
	    *in->buffer = 0;
	    in->pos = 0;
	}
	insert_char (in, c_code);
    }
    update_input (in);
}
