------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                  R A T                                   --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.12 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package provides unlimited precision rational arithmetic capabilities.
--  The input values are either N_Real_Literal nodes, or N_Entity_Name nodes
--  where the identifier is a constant whose initial value is itself a real
--  literal, or recursively, another constant identifier. The output values
--  are N_Real_Literal nodes containing the result values which are always
--  normalized by reducing them to lowest terms (although the caller of this
--  package should not need to rely on this property for correct functioning).

with Types; use Types;

package Rat is

   subtype Real_Literal is Node_Id;
   --  Used to refer to a node for a real literal

   subtype Integer_Literal is Node_Id;
   --  Used to refer to a node for an integer literal.

   function Rlit_Half return Real_Literal;
   function Rlit_1 return Real_Literal;
   function Rlit_2 return Real_Literal;
   function Rlit_10 return Real_Literal;
   --  All these functions return constant N_Real_Literals with these
   --  corresponding values:
   --  Rlit_Half => 0.5, Rlit_1 => 1.0, Rlit_2 => 2.0, Rlit_10 => 10.0

   procedure Initialize;
   --  This procedure creates the constant N_Real_Literals that will be
   --  returned by the functions Rlit_Half, Rlit_1, etc.

   function Rat_From_Int (Int : Integer_Literal) return Real_Literal;
   --  Returns real corresponding to integer value

   function Rat_From_Uint (Int_Value : Uint) return Real_Literal;
   --  Returns real corresponding to Int_Value.

   function Int_From_Rat (Real : Real_Literal) return Integer_Literal;
   --  Return integer value obtained by accurate rounding of real value

   function Rat_From_Ints (Num, Den : Integer_Literal) return Real_Literal;
   --  Builds Real_Literal.

   function Rat_Trunc (Real : Real_Literal) return Uint;
   --  Return integer value obtained by a truncation of real value.

   function Rat_Round (Real : Real_Literal; N : Uint) return Real_Literal;
   --  Returns a value with an accuracy equivalent to N digits of the
   --  given real value.

   function Rat_Sum (Left, Right : Real_Literal) return Real_Literal;
   --  Returns sum of two reals.

   function Rat_Difference (Left, Right : Real_Literal) return Real_Literal;
   --  Returns difference of two reals.

   function Rat_Quotient (Left, Right : Real_Literal) return Real_Literal;
   --  Returns quotient of two reals. Fatal error if Right is zero

   function Rat_Product (Left, Right : Real_Literal) return Real_Literal;
   --  Returns product of two reals

   function Rat_Exponentiate
     (Left  : Real_Literal;
      Right : Uint)
      return  Real_Literal;
   --  Returns result of raising Real_Literal to Uint power.
   --  Fatal error if Left is 0 and Right is negative.

   function Rat_Abs (Right : Real_Literal) return Real_Literal;
   --  Returns abs function of real

   function Rat_Negate (Right : Real_Literal) return Real_Literal;
   --  Returns negative of real

   function Rat_Eq (Left, Right : Real_Literal) return Boolean;
   --  Compares reals for equality.

   function Rat_Max (Left, Right : Real_Literal) return Real_Literal;
   --  Returns real literal whose value is the maximum of two reals

   function Rat_Min (Left, Right : Real_Literal) return Real_Literal;
   --  Returns real literal whose value is the minimum of two reals

   function Rat_Ne (Left, Right : Real_Literal) return Boolean;
   --  Compares reals for inequality.

   function Rat_Lt (Left, Right : Real_Literal) return Boolean;
   --  Compares reals for less than.

   function Rat_Le (Left, Right : Real_Literal) return Boolean;
   --  Compares reals for less than or equal.

   function Rat_Gt (Left, Right : Real_Literal) return Boolean;
   --  Compares reals for greater than.

   function Rat_Ge (Left, Right : Real_Literal) return Boolean;
   --  Compares reals for greater than or equal.

   function Rat_Is_Zero (Arg : Real_Literal) return Boolean;
   --  Tests if real value is zero

   procedure Normal_Form
     (Real         : Real_Literal;
      Normal_Value : out Real_Literal;
      Scale_Factor : out Uint);
   --  Gives the parameters of the normal form of a real value (see Hilfinger
   --  algorithms for fixed types).
   --  Normal_Value is in the interval [1/2, 1[ and we have the following
   --  relationship:
   --         (2 ** Scale_Factor) * Normal_Value =  Real.

end Rat;
