/*  aKode Resampler (fast)

    Copyright (C) 2004 Allan Sandfeld Jensen <kde@carewolf.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "audioframe.h"
#include "fast_resampler.h"

namespace aKode {

FastResampler* FastResamplerPlugin::openResampler(float speed) {
    return new FastResampler(speed);
}

extern "C" { FastResamplerPlugin fast_resampler; }

FastResampler::FastResampler(float value)
{
    setSpeed(value);
}

// Sillyness to work around sillyness in C++ templates
template<typename S>
struct fast_Div_FP {
    static inline S div(S value, S divisor) {
        return value/divisor;
    }
    static inline S rem(S, S) {
        return 0.0;
    }
    static inline S max(int) {
        return 1.0;
    }
};

template<typename S>
struct fast_Div_Int {
    static inline S div(S value, S divisor) {
        return value / divisor;
    }
    static inline S rem(S value, S divisor) {
        return value % divisor;
    }
    static inline S max(int bits) {
        return (((S)1)<<(bits-1))-1;
    }
};

// A fast resampling by linear interpolation
// I assume you know binary arithmetics and convertions if you're reading this
// T is the input/output type, S is aritmetics type atleast 8 bits longer
template<typename T, typename S, class Div>
static bool _doBuffer(AudioFrame* in, AudioFrame* out, unsigned int speed)
{
    unsigned long vt_pos_start = 0;  // virtual positions of new sample
    unsigned long vt_pos_end = speed;
    unsigned long real_pos_start, real_pos_end;
    long start_fraction, end_fraction;

    unsigned long vt_end = in->length * 1024 -1;
    unsigned long out_length = (unsigned long)((in->length * 1024 + speed-1 ) / speed);
    out->reserveSpace(in, out_length);
    out->sample_rate = (uint32_t)((in->sample_rate*1024)/speed);

    unsigned char channels = in->channels; // take copy to reduce alias-induced reads
    unsigned long out_pos = 0;

    T** indata = (T**)in->data;    // templating ;)
    T** outdata = (T**)out->data;

    S sspeed = (S)speed;
    S smax = Div::max(in->sample_width);

    while(out_pos < out_length && vt_pos_start < vt_end) {
        real_pos_start = vt_pos_start / 1024;     start_fraction = vt_pos_start % 1024;
        real_pos_end   = vt_pos_end   / 1024;       end_fraction = vt_pos_end   % 1024;

        if (real_pos_start == real_pos_end) {
            for(int i=0; i<channels; i++) {
                outdata[i][out_pos] = indata[i][real_pos_start];
            }
        } else {
            for(int i=0; i<channels; i++) {
                S signal = 0;
                S remainder = 0;
                S temp;

                temp = (S)indata[i][real_pos_start];
                signal    += Div::div(temp,sspeed) * (1024L-start_fraction);
                remainder += Div::rem(temp,sspeed) * (1024L-start_fraction);

                temp= (S)indata[i][real_pos_end];
                signal    += Div::div(temp,sspeed) * (end_fraction);
                remainder += Div::rem(temp,sspeed) * (end_fraction);

                for(unsigned long j = real_pos_start+1; j<real_pos_end; j++) {
                    temp= (S)indata[i][j];
                    signal += Div::div(temp,sspeed) * 1024L;
                    remainder += Div::rem(temp,sspeed) * 1024L;
                }
                signal += Div::div(remainder,sspeed);

                if (signal > smax) signal = smax;
                else
                if (signal < -smax) signal = -smax;

                outdata[i][out_pos] = (T)signal;
            }
        }
        out_pos++;
        vt_pos_start = vt_pos_end;
        vt_pos_end += speed;
        if (vt_pos_end > vt_end) vt_pos_end = vt_end;
    }
    return true;
}

bool FastResampler::doFrame(AudioFrame* in, AudioFrame* out)
{
    if (in->sample_width < 0) {
        return _doBuffer<float, float, fast_Div_FP<float> >(in, out, speed);
    } else
    if (in->sample_width <= 8) {
        return _doBuffer<int8_t, int32_t, fast_Div_Int<int32_t> >(in, out, speed);
    } else
    if (in->sample_width <= 16) {
        return _doBuffer<int16_t, int32_t, fast_Div_Int<int32_t> >(in, out, speed);
    } else
    if (in->sample_width <= 24) {
        return _doBuffer<int32_t, int32_t, fast_Div_Int<int32_t> >(in, out, speed);
    } else
        return _doBuffer<int32_t, int64_t, fast_Div_Int<int64_t> >(in, out, speed);
}

void FastResampler::setSpeed(float value)
{
    speed = (unsigned int)((1.0/value)*1024.0+0.5);
}

} // namespace
