/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001 - 2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * main.c - start of gtk main loop and initialization
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <errno.h>
#include <time.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <string.h>
#if HAVE_DIRENT_H
#	include <dirent.h>
#	define NAMLEN(dirent) strlen((dirent)->d_name)
#else
#	define dirent direct
#	define NAMLEN(dirent) (dirent)->d_namlen
#	if HAVE_SYS_NDIR_H
#		include <sys/ndir.h>
#	endif
#	if HAVE_SYS_DIR_H
#		include <sys/dir.h>
#	endif
#	if HAVE_NDIR_H
#		include <ndir.h>
#	endif
#endif

#if WANT_HAMLIB
# include <hamlib/rig.h>
# include "hamlib-utils.h"
#endif

#include "gui_b4dialog.h"
#include "gui_mainwindow.h"
#include "gui_setupdialog.h"
#include "gui_warningdialog.h"
#include "gui_utils.h"
#include "support.h"
#include "callbacks_mainwindow_menu.h"
#include "callbacks_mainwindow_qsoframe.h"
#include "callbacks_mainwindow.h"
#include "types.h"
#include "preferences.h"
#include "log.h"
#include "utils.h"
#include "dxcc.h"
#include "remote.h"
#include "history.h"

gchar *xlogdir;
gint remotetimer = -1, clocktimer = -1, savetimer = -1;
gchar **bandsplit, **modesplit, **qso = NULL;
statetype state;
glong msgid;
GList *logwindowlist = NULL;
GdkColormap *colormap;

extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;
extern preferencestype preferences;
extern remotetype remote;
extern GtkWidget *b4dialog;

#if WANT_HAMLIB
GList *riglist = NULL;
#endif

/* command line options */
static void
parsecommandline (int argc, char *argv[])
{
	gint p;
	gchar *versionstr;
																																								
	while ((p = getopt (argc, argv, "hv")) != -1)
	{
		switch (p)
		{
			case ':':
			case '?':
			case 'h':
				g_print ("Usage: %s [option]...\n", PACKAGE);
				g_print ("	-h	Display this help and exit\n");
				g_print ("	-v	Output version information and exit\n");
				exit (0);
			case 'v':
				versionstr = g_strdup_printf (_("%s version %s\n"),
					g_path_get_basename (argv[0]), VERSION);
				g_print (versionstr);
				g_free (versionstr);
				exit (0);
		}
	}
}

/* see if we can create the ~/.xlog directory */
static gboolean
xlogdircheck (void)
{
	GString *packagedir = g_string_new ("");
	struct stat statdir;

	g_string_printf (packagedir, "%s.", G_DIR_SEPARATOR_S);
	g_string_append (packagedir, PACKAGE);
	xlogdir = g_strconcat (g_get_home_dir (), packagedir->str, NULL);
	if (stat (xlogdir, &statdir) == -1)
		{
			if (mkdir (xlogdir, S_IRUSR | S_IWUSR | S_IXUSR) == -1)
				g_error (_("Creating ~%s directory."), packagedir->str);
			else
				show_setupdialog();
		}
	else if (!S_ISDIR (statdir.st_mode))
		g_error (_("~%s is not a directory."), packagedir->str);
	g_string_free (packagedir, TRUE);
	return TRUE;
}

/* defaults for program state */
static void
setdefaultstate (void)
{
	state.qsos = 0;
	state.countries = 0;
	/* the default ham lives in the gulf of guinea */
	state.mylocation = g_strdup ("0.0N0.0E");
	state.controlkey = FALSE;
	state.rigfrequency = 0;
	state.rigmode = 0;
	state.rigrst = g_strdup ("0");
	state.rigpower = 0;
	state.scounter = 0;
	state.hlcounter = 0;
	state.tx = FALSE;
	state.statustimer = FALSE;
	state.shmid = -1;
	state.logwindows = 0;
	state.mylocation = setlocation (preferences.latitude, preferences.NS,
					preferences.longitude, preferences.EW);
	state.searchstr = g_strdup("");
	state.dupecheck = 0;
	state.notdupecheckmode = FALSE;
	state.notdupecheckband = FALSE;
}

/* the fun starts here */
gint
main (int argc, char *argv[])
{
	GtkWidget *bandoptionmenu, *modeoptionmenu, *dateentry, *qsoframe,
		*dxccframe, *bandentry, *modeentry, *clocklabel,
		*clockhandlebox, *mhzlabel, *mhzbutton,
		*frequencylabel, *frequencyhandlebox, *modelabel, *modebutton, *rstlabel,
		*rstbutton, *smeterhandlebox, *smeterdrawingarea,
		*hpaned, *powerlabel, *powerbutton,
		*locatorframe, *unknownlabel1, *unknownlabel2, *trace_hamlib,
		*handlebox, *view_toolbar, *worked;
	gchar *xlogfile, *xlogbackupfile, *temp, *logstoload, **loglist = NULL;
	gint i, ctyresult;
	gboolean showmainwindow = TRUE, logsfromprefs = FALSE;
	GString *logs;
	logtype *logwindow;
	LOGDB *lp = NULL;

#if WANT_HAMLIB
	gboolean hamlibresult = FALSE;
#endif

	parsecommandline (argc, argv);

#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);
#endif

	gtk_set_locale ();
	gtk_init (&argc, &argv);
	setlocale(LC_NUMERIC, "C");

	add_pixmap_directory (PACKAGE_DATA_DIR G_DIR_SEPARATOR_S "pixmaps");
	add_pixmap_directory (PACKAGE_SOURCE_DIR G_DIR_SEPARATOR_S "pixmaps");

	mainwindow = create_mainwindow ();
	showmainwindow = xlogdircheck();

	loadpreferences ();
	loadhistory ();
	setdefaultstate ();
	colormap = gdk_colormap_get_system ();

	/* defaults for remote data */
	remote.version = 0;
	remote.nr = 0;
	remote.program = g_strdup ("unknown");

	ctyresult = readctydata ();
	if (ctyresult)
		g_warning ("could not read cty.dat");
	ctyresult = readxlogdata ();

	qso = g_new0 (gchar *, QSO_FIELDS);
	for (i = 0; i < QSO_FIELDS; i++)
		qso[i] = g_new0 (gchar, 100);

	/* logging should always be in GMT, shouldn't it? */
	putenv ("TZ=GMT");
	tzset ();

	/* update the date field */
	temp = getdate ();
	dateentry = lookup_widget (mainwindow, "dateentry");
	gtk_entry_set_text (GTK_ENTRY (dateentry), temp);
	gtk_widget_grab_focus (GTK_WIDGET (dateentry));

	/* read the logs */
	mainnotebook = gtk_notebook_new ();
	gtk_widget_show (mainnotebook);
	hpaned = lookup_widget (mainwindow, "hpaned");
	gtk_paned_pack2 (GTK_PANED (hpaned), mainnotebook, TRUE, FALSE);

	/* check for multiple *.xlog */
	if (!opendir (preferences.savedir))
		preferences.savedir = g_strdup (xlogdir);
	if (g_strrstr (preferences.logstoload, "*"))
		/* read all logs of type xlog with a wildcard used */
		logs = getlogs (preferences.savedir, preferences.logstoload);
	else
		{ /* read logs from preferences.logstoload */
			logstoload = g_strdup (preferences.logstoload);
			g_strdelimit (logstoload, ",", '\n');
			logs = g_string_new (logstoload);
			g_free (logstoload);
			logsfromprefs = TRUE;
		}

	if ((logs->len) > 0)		/* flexible logs */
		loglist = g_strsplit (logs->str, "\n", -1);
	else				/* old log type */
		{
			loglist = g_malloc0 (2 * sizeof (gchar *));
			loglist[0] = g_strdup ("log");
		}

	state.utf8error = FALSE;
	for (i = 0;; i++)
		{
			if (!loglist || !loglist[i] || (strlen(loglist[i]) == 0))
				break;
			if ((logs->len) > 0)
			{
				if (logsfromprefs)
					xlogfile = g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, 
						loglist[i], ".xlog", NULL);
				else
					xlogfile = g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, 
						loglist[i], NULL);
				lp = log_file_open (xlogfile, TYPE_FLOG);
			}
			else
			{
				xlogfile = g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, 
					loglist[i],	NULL);
				lp = log_file_open (xlogfile, TYPE_OLD_LOG);
			}
			if (!lp) /* create a new log */
			{
				xlogfile = g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, 
					"log.xlog", NULL);
				lp = log_file_create (xlogfile, TYPE_FLOG, qso_field_nr, qso_fields,
					qso_widths);
			}
			if (lp)
			{
				logwindow = openlog (lp, loglist[i], i);
				log_file_qso_foreach (lp, fillin_list, logwindow);
				log_file_close (lp);
				logwindowlist = g_list_append (logwindowlist, logwindow);
				/* backup */
				xlogbackupfile = g_strconcat (xlogfile, ".backup", NULL);
				backuplog (xlogfile, xlogbackupfile);
				g_free (xlogbackupfile);
			}
			g_free (xlogfile);
		}
	state.logwindows = i;
	g_strfreev (loglist);
	g_string_free (logs, TRUE);

	g_signal_connect (G_OBJECT (mainnotebook), "switch_page",
					G_CALLBACK (on_mainnotebook_switch_page), NULL);

	/* first page has focus */
	logwindow = g_list_nth_data (logwindowlist, 0);
	set_qsoframe (logwindow);

	/* we have to set the labels of the unknown fields, wether they are visible or not */
	unknownlabel1 = lookup_widget (mainwindow, "unknownlabel1");
	gtk_label_set_text (GTK_LABEL (unknownlabel1), preferences.freefield1);
	unknownlabel2 = lookup_widget (mainwindow, "unknownlabel2");
	gtk_label_set_text (GTK_LABEL (unknownlabel2), preferences.freefield2);

	/* update the statusbar with some information */
	temp = g_strdup_printf (_("%d QSO's and %d countries loaded"),
				state.qsos, state.countries);
	update_statusbar (temp);

	bandsplit = g_strsplit (preferences.bands, ",", 0);
	modesplit = g_strsplit (preferences.modes, ",", 0);
	/* clean up modes and bands, don't want any spaces */
	deletespaces (bandsplit);
	deletespaces (modesplit);
	bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
	/* set the menu strings for the bandoption menu */
	updateoptionmenu (bandoptionmenu, bandsplit);
	gtk_option_menu_set_history (GTK_OPTION_MENU (bandoptionmenu),
		preferences.bandoptionmenu);
	modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
	/* set the menu strings for the modeoption menu */
	updateoptionmenu (modeoptionmenu, modesplit);
	gtk_option_menu_set_history (GTK_OPTION_MENU (modeoptionmenu),
		preferences.modeoptionmenu);

	
	/* either hide the editbox or the optionmenu */
	if (preferences.modeseditbox == 0)
	{ 
		modeentry = lookup_widget (mainwindow, "modeentry");
		gtk_widget_hide (modeentry);
	}
	else
		gtk_widget_hide (modeoptionmenu);
	if (preferences.bandseditbox == 0)
	{
		bandentry = lookup_widget (mainwindow, "bandentry");
		gtk_widget_hide (bandentry);
	}
	else
		gtk_widget_hide (bandoptionmenu);

	/* set clock appearance */
	clocklabel = lookup_widget (mainwindow, "clocklabel");
	clockhandlebox = lookup_widget (mainwindow, "clockhandlebox");
	setlabelstyle (clocklabel, preferences.themecolor, preferences.themefont);
	if (preferences.clock == 0)
		{				/* no clock */
			gtk_widget_hide (clockhandlebox);
		}
	else
		{				/* clock on statusbar */
			gtk_widget_show (clockhandlebox);
			clocktimer = gtk_timeout_add (1000, (GtkFunction) updateclock, NULL);
		}

	/* set appearance of the QSO, DXCC and locator frame */
	dxccframe = lookup_widget (mainwindow, "dxccframe");
	setframestyle (dxccframe, preferences.themecolor, preferences.themefont);
	qsoframe = lookup_widget (mainwindow, "qsoframe");
	setframestyle (qsoframe, preferences.themecolor, preferences.themefont);
	locatorframe = lookup_widget (mainwindow, "locatorframe");
	setframestyle (locatorframe, preferences.themecolor, preferences.themefont);

	/* set up the message queue for remote data */
	msgid = msgget ((key_t) 1238, 0666 | IPC_CREAT);
	if (msgid == -1)
		{
			temp = g_strdup_printf (_("Msgget failed: %s"), g_strerror (errno));
			update_statusbar (temp);
		}
	else				/* check for a message twice a second */
		remotetimer = gtk_timeout_add (500, (GtkFunction) remote_entry, NULL);

	/* hamlib stuff */
	mhzlabel = lookup_widget (mainwindow, "mhzlabel");
	mhzbutton = lookup_widget (mainwindow, "mhzbutton");
	modelabel = lookup_widget (mainwindow, "modelabel");
	modebutton = lookup_widget (mainwindow, "modebutton");
	rstlabel = lookup_widget (mainwindow, "rstlabel");
	rstbutton = lookup_widget (mainwindow, "rstbutton");
	frequencylabel = lookup_widget (mainwindow, "frequencylabel");
	frequencyhandlebox = lookup_widget (mainwindow, "frequencyhandlebox");
	smeterhandlebox = lookup_widget (mainwindow, "smeterhandlebox");
	smeterdrawingarea = lookup_widget (mainwindow, "smeterdrawingarea");
	powerlabel = lookup_widget (mainwindow, "powerlabel");
	powerbutton = lookup_widget (mainwindow, "powerbutton");
	trace_hamlib = lookup_widget (mainwindow, "trace_hamlib");
#if WANT_HAMLIB
	setlabelstyle (frequencylabel, preferences.themecolor,
		 preferences.themefont);

	/* clickable buttons and events for hamlib */
	g_signal_connect (G_OBJECT (mhzbutton), "clicked",
					G_CALLBACK (on_mhzbutton_clicked), NULL);
	g_signal_connect (G_OBJECT (modebutton), "clicked",
					G_CALLBACK (on_modebutton_clicked), NULL);
	g_signal_connect (G_OBJECT (rstbutton), "clicked",
					G_CALLBACK (on_rstbutton_clicked), NULL);
	g_signal_connect (G_OBJECT (powerbutton), "clicked",
					G_CALLBACK (on_powerbutton_clicked), NULL);

	/* need this to set popdown strings in prefs dialog */
	rig_set_debug (RIG_DEBUG_NONE);
	riglist = riglist_get_list ();

	/* init rig if hamlib is enabled and show/hide some widgets */
	if (preferences.hamlib > 0)
		hamlibresult = start_hamlib (preferences.radio, preferences.device, 
			RIG_DEBUG_NONE, preferences.polltime);


	g_signal_connect (G_OBJECT (smeterdrawingarea), "configure_event",
											G_CALLBACK (on_smeterdrawingarea_configure_event),
											NULL);
	g_signal_connect (G_OBJECT (smeterdrawingarea), "expose_event",
											G_CALLBACK (on_smeterdrawingarea_expose_event),
											NULL);
	g_signal_connect (G_OBJECT (mainwindow), "show",
											G_CALLBACK (on_mainwindow_show), NULL);
 
	sethamlibwidgets (preferences.hamlib, FALSE);

#else
	gtk_widget_hide (trace_hamlib);
	gtk_widget_show (mhzlabel);
	gtk_widget_show (modelabel);
	gtk_widget_show (rstlabel);
	gtk_widget_hide (mhzbutton);
	gtk_widget_hide (modebutton);
	gtk_widget_hide (rstbutton);
	gtk_widget_hide (frequencyhandlebox);
	gtk_widget_hide (smeterhandlebox);
#endif

	if ((preferences.autosave > 0) && (preferences.saving == 1))
		savetimer = gtk_timeout_add (preferences.autosave * 60 * 1000,
				 (GtkFunction) autosave, NULL);
	
	g_free (temp);

	/* create wkd B4 dialog in case we need it */
	b4dialog = create_b4dialog ();

	if (showmainwindow)
	{
		gtk_widget_show (mainwindow);
		/* position the main window */
		gdk_window_move_resize (mainwindow->window, preferences.x,
			preferences.y, preferences.width, preferences.height);
		handlebox = lookup_widget (mainwindow, "handlebox");
		view_toolbar = lookup_widget (mainwindow, "view_toolbar");
		worked = lookup_widget (mainwindow, "worked");
		if (preferences.viewtoolbar == 1)
		{
			gtk_widget_show (handlebox);
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_toolbar), TRUE);
		}
		else
		{
			gtk_widget_hide (handlebox);
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_toolbar), FALSE);
		}
		if (preferences.viewb4 == 1)
		{
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(worked), TRUE);
			gtk_widget_show (b4dialog);
			gdk_window_move_resize (b4dialog->window, preferences.b4x,
				preferences.b4y, preferences.b4width, preferences.b4height);
			gtk_window_present (GTK_WINDOW(mainwindow));
		}
		else
		{
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(worked), FALSE);
			gtk_widget_hide (b4dialog);
		}
		if (state.utf8error) 
			warningdialog (_("xlog - open log"), 
				_("There were some errors converting from your locale to UTF8, "
				"which is used by GTK+ internally. It is best if you start xlog "
				"from a terminal and see what the errors are. Please check your "
				"language settings and your log fields!"));
	}

	gtk_main ();
	return 0;
}
