//
//  kjots
//
//  Copyright (C) 1997 Christoph Neerfeld
//  Copyright (C) 2002, 2003 Aaron J. Seigo
//  Copyright (C) 2003 Stanislav Kljuhhin
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#include <qdir.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <kinputdialog.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kprogress.h>
#include <kstandarddirs.h>

#include <kio/job.h>

#include "kjotsentry.h"

//
// KJotsEntryBase
//

KJotsEntryBase::KJotsEntryBase(KListView* parent, QListViewItem* after=0)
    :KListViewItem(parent,after)
{

}

KJotsEntryBase::KJotsEntryBase(KListViewItem* parent, QListViewItem* after=0)
	:KListViewItem(parent,after)
{

}

void KJotsEntryBase::setSubject(const QString& subj)
{
    setText(0, subj);
}

void KJotsEntryBase::setText(int column, const QString& text)
{
    if (column == 0 && text.isEmpty())
        KListViewItem::setText(0, defaultSubject());
    else
        KListViewItem::setText(column, text);
}

//
// KJotsBook
//

KJotsBook::KJotsBook(KListView* parent, QString subj, QListViewItem* after)
    : KJotsEntryBase(parent, after),
      m_dirty(false),
      m_open(false),
      m_deleteFlag(false),
      m_pageID(0),
      m_saveInProgress(0),
      m_saveProgressDialog(0)
{
    KJotsEntryBase::setSubject(subj);
    m_beforeRename = subject();
    setExpandable(true);
    setPixmap(0, kapp->iconLoader()->loadIcon(QString("contents"),KIcon::Small));
}

KJotsBook::~KJotsBook()
{
    if (m_deleteFlag)
        annihilateBook();
    else
        closeBook();
}

bool KJotsBook::isBookFile(const QString& filename)
{
    QFile folder(filename);

    if ( !folder.exists() )
    {
        return false;
    }
    else if ( !folder.open(IO_ReadWrite) )
    {
        return false;
    }

    QTextStream st(static_cast<QIODevice*>(&folder));
    st.setEncoding( QTextStream::Locale );
    QString buf = st.readLine();

    if (buf.left(9) != "\\NewEntry")
    {
        return false;
    }

    return true;
}

bool KJotsBook::openBook()
{
    if (m_open)
    {
        return true;
    }

    listView()->setUpdatesEnabled(false);

    if (!isOpen())
    {
        setOpen(true);
    }

    QDir dir = QDir(locateLocal("appdata",""));
    QString file_name = dir.absPath();
    file_name += '/';
    file_name += QFile::encodeName(subject());

    QFile folder(file_name);

    if ( !folder.exists() )
    {
        addSubject();
        listView()->setUpdatesEnabled(true);
        m_open = true;
        return true;
    }
    else if ( !folder.open(IO_ReadWrite) )
    {
        // TODO: should check for IO_READ access and open read only if so
        addSubject();
        listView()->setUpdatesEnabled(true);
        m_open = true;
        return false;
    }

    QTextStream st(static_cast<QIODevice*>(&folder));
    st.setEncoding( QTextStream::Locale );
    QString buf = st.readLine();

    if (buf.left(9) != "\\NewEntry")
    {
        addSubject();
        listView()->setUpdatesEnabled(true);
        m_open = true;
        return false;
    }

    KJotsSubject *entry = 0;
    QString body;

    bool quit=false;

    while (1)
    {
        if (buf.left(9) == "\\NewEntry")
        {
            if (entry)
            {
                entry->setBody(body);
            }

            body = QString::null;
            QString title = buf.mid(10, buf.length());

            // now catch the page id
            buf = st.readLine();
            int pageID = -1;
            if (buf.left(3) == "\\ID")
            {
                pageID = buf.mid(3, buf.length()).toInt();
                if (pageID > m_pageID)
                {
                    m_pageID = pageID;
                }
                buf = st.readLine();
            }

            if (pageID < 0)
            {
                pageID = nextID();
            }
            entry = addSubject(title, entry, pageID);
        }

        int pos = 0;

        while ((pos = buf.find('\\',pos)) != -1)
            if (buf[++pos] == '\\')
                buf.remove(pos, 1 );

        body.append( buf + "\n");

        if (quit)
            break;

        buf = st.readLine();

        quit = st.eof();
    }

    entry->setBody(body);

    folder.close();

    listView()->setUpdatesEnabled(true);

    m_open = true;
    return true;
}

void KJotsBook::closeBook(bool saveOnly)
{
    if (!m_open)
        return;

    if (!saveOnly)
    {
        listView()->setUpdatesEnabled(false);
    }

    QDir dir = QDir(locateLocal("appdata",""));
    QString file_name = dir.absPath();
    file_name +=  '/' + subject();

    QFile folder(file_name);
    if (!firstChild() || !folder.open(IO_WriteOnly | IO_Truncate))
    {
        if (!saveOnly)
        {
            listView()->setUpdatesEnabled(true);
            m_open = false;
        }

        return;
    }

    KJotsSubject* entry = dynamic_cast<KJotsSubject*>(firstChild());
    QTextStream st((QIODevice *) &folder);
    st.setEncoding( QTextStream::Locale );

    while (entry)
    {
        st << "\\NewEntry " << entry->subject() << "\n";
        st << "\\ID " << entry->id() << "\n";
        QString buf = entry->body();
        st << buf.replace('\\', "\\\\");
        if( buf.right(1) != "\n" )
        {
            st << '\n';
        }

        if (!saveOnly)
        {
            KJotsSubject* newEntry = dynamic_cast<KJotsSubject*>(entry->nextSibling());
            delete entry;
            entry = newEntry;
        }
        else
            entry = dynamic_cast<KJotsSubject*>(entry->nextSibling());
    }

    folder.close();
    if (!saveOnly)
    {
        listView()->setUpdatesEnabled(true);
        m_open = false;
    }
}

void KJotsBook::annihilateBook()
{
    QDir dir = QDir(locateLocal("appdata",""));
    dir.remove(subject());
}

void KJotsBook::rename()
{
    bool ok;
    QString name = KInputDialog::getText(i18n( "Rename Book" ),
                                         i18n( "Book name:" ),
                                         subject(), &ok, listView());
    if (ok)
    {
        setSubject(name);
    }
}

void KJotsBook::setSubject(const QString& newSubject)
{
    QDir dir = QDir(locateLocal("appdata",""));
    QString fileName = dir.absPath();
    fileName += '/' + QFile::encodeName(newSubject);

    if (QFile::exists(fileName))
    {
        KJotsEntryBase::setSubject(m_beforeRename);
        KMessageBox::sorry(listView(), i18n("A book named %1 already exists.").arg(newSubject), i18n("Rename Book"));
        return;
    }

    bool wasOpen = m_open;
    if (wasOpen)
    {
        closeBook();
    }

    if (!dir.rename(m_beforeRename, newSubject))
    {
        KJotsEntryBase::setSubject(m_beforeRename);
        KMessageBox::sorry(listView(), i18n("The name change failed. Please check the file permissions.\n\nFrom: %1\nTo: %2").arg(m_beforeRename, newSubject), i18n("Rename Book"));
        return;
    }

    KJotsEntryBase::setSubject(newSubject);

    if (wasOpen)
    {
        openBook();
    }

    m_beforeRename = text(0);
}

void KJotsBook::saveToFile(KURL url)
{
    if (url.isEmpty() || m_saveInProgress != 0)
        return;

    KIO::TransferJob* job = KIO::put(url, -1, true, false, false);

    if (!job)
    {
        return;
    }

    m_saveProgressDialog = new KProgressDialog(listView(), "bookSaveInProgress",
                                               i18n("Saving %1").arg(subject()),
                                               i18n("Saving the contents of %1 to %2")
                                                    .arg(subject(), url.prettyURL()),
                                               true);
    m_saveProgressDialog->progressBar()->setTotalSteps(childCount());
    m_saveProgressDialog->showCancelButton(false);
    m_saveProgressDialog->setAutoClose(true);

    connect(job, SIGNAL(dataReq(KIO::Job*, QByteArray&)), SLOT(saveDataReq(KIO::Job*, QByteArray&)));
    connect(job, SIGNAL(result( KIO::Job *)), SLOT(slotSaveResult( KIO::Job *)));
    m_saveInProgress = firstChild();
}

void KJotsBook::saveDataReq(KIO::Job* /* job */, QByteArray& data)
{
    if (!m_saveInProgress)
    {
        return;
    }

    KJotsSubject* entry = 0;
    while (m_saveInProgress)
    {
        entry = dynamic_cast<KJotsSubject*>(m_saveInProgress);

        if (entry)
        {
            break;
        }

        m_saveInProgress = m_saveInProgress->nextSibling();
    }

    if (entry)
    {
        QTextStream stream(data, IO_WriteOnly);
        entry->write(stream);
        m_saveInProgress = m_saveInProgress->nextSibling();
    }
}

void KJotsBook::slotSaveResult(KIO::Job *)
{
    m_saveInProgress = 0;
    delete m_saveProgressDialog;
    m_saveProgressDialog = 0;
    /* if (job->error() != 0) {} */
}

KJotsSubject* KJotsBook::addSubject(QString subject, QListViewItem* after, int id)
{
    if (id < 0)
    {
        id = nextID();
    }

    // append new pages
    if (after == 0)
    {
        QListViewItem* tmp = firstChild();
        while (tmp)
        {
            after = tmp;
            tmp = tmp->nextSibling();
        }
    }

    return new KJotsSubject(this, subject, id, after);
}

void KJotsBook::print(QFont& defaultFont)
{
    KPrinter printer;
    printer.setDocName(subject());
    printer.setFullPage(false);
    printer.setCreator("KJots");

    if (!printer.setup(listView(), i18n("Print: %1").arg(subject())))
    {
        return;
    }

    QPainter painter(&printer);
    painter.setFont(defaultFont);
    QPaintDeviceMetrics metrics( &printer );
    int maxWidth = metrics.width();
    int maxHeight = metrics.height();
    int y = 0;
    int pageNo = 0;

    for (QListViewItem* page = firstChild(); page; page = page->nextSibling())
    {
        KJotsSubject* subject = dynamic_cast<KJotsSubject*>(page);

        if (!subject)
        {
            continue;
        }

        QString pageTitle = subject->subject().isEmpty() ?
                                     i18n("Page %1").arg(pageNo) :
                                     subject->subject();
        QRect r = painter.boundingRect(0, y,
                                       maxWidth, maxHeight,
                                       QPainter::ExpandTabs | QPainter::WordBreak | QPainter::AlignHCenter,
                                       pageTitle);
        y += 10;

        if ((y + r.height()) > maxHeight)
        {
            printer.newPage();
            y = 0;
        }
        else
        {
            r.moveBy(0, 10);
        }

        r.setLeft(0);
        r.setRight(maxWidth);
        painter.drawRect(r);
        painter.drawText(0, y, maxWidth, maxHeight - y,
                QPainter::ExpandTabs | QPainter::WordBreak | QPainter::AlignHCenter,
                pageTitle);
        y += r.height() + 15;
        y = subject->print(printer, painter, y);
        ++pageNo;
    }

    painter.end();
}

QString KJotsBook::defaultSubject()
{
    return i18n("Untitled Book");
}

//
// KJotsSubject
//

KJotsSubject::KJotsSubject(KJotsBook* parent, QString subj, int id, QListViewItem* after)
    : KJotsEntryBase(parent,after),
      m_id(id),
      m_saveInProgress(false)
{
    setSubject(subj);
    setPixmap(0, kapp->iconLoader()->loadIcon(QString("edit"), KIcon::Small));
}

QString KJotsSubject::body()
{
    return m_text;
}

void KJotsSubject::setBody(const QString& text)
{
    m_text = text;
}

void KJotsSubject::rename()
{
    bool ok;

    QString name = KInputDialog::getText(i18n( "Rename Page" ),
                                         i18n( "Page title:" ),
                                         subject(), &ok, listView() );

    if (ok)
    {
        setSubject(name);
    }
}

void KJotsSubject::write(QTextStream &stream)
{
    QString line, buf;
    line.fill('#', subject().length() + 2);
    buf = body();
    if (buf.right(1) != "\n")
    {
        buf.append('\n');
    }

    stream << line << '\n'
           << "# " << subject() << '\n'
           << line << '\n'
           << buf
           << '\n';
}

void KJotsSubject::saveToFile(KURL url)
{
    if (url.isEmpty() || m_saveInProgress)
        return;

    KIO::TransferJob* job = KIO::put(url, -1, true, false, false);
    if (!job)
    {
        return;
    }

    connect(job, SIGNAL(dataReq(KIO::Job*, QByteArray&)), SLOT(saveDataReq(KIO::Job*, QByteArray&)));
    connect(job, SIGNAL(result( KIO::Job *)), SLOT(slotSaveResult( KIO::Job *)));
    m_saveInProgress = true;
}

void KJotsSubject::saveDataReq(KIO::Job* /* job */, QByteArray& data)
{
    if (!m_saveInProgress)
    {
        return;
    }

    QTextStream stream(data, IO_WriteOnly);
    write(stream);

    m_saveInProgress = false;
}

void KJotsSubject::slotSaveResult(KIO::Job *)
{
    m_saveInProgress = false;
    /* if (job->error() != 0) {} */
}

void KJotsSubject::print(QFont& defaultFont)
{
    KJotsEntryBase* book = dynamic_cast<KJotsEntryBase*>(KListViewItem::parent());

    QString docName = book->subject();
    if (!subject().isNull())
    {
        docName += ": " + subject();
    }

    KPrinter printer;
    printer.setDocName(docName);
    printer.setFullPage(false);
    printer.setCreator("KJots");
    int y = 0;

    if (printer.setup(listView(), i18n("Print: %1").arg(docName)))
    {
        QPainter painter( &printer );
        painter.setFont(defaultFont);
        y = print(printer, painter, y);
        painter.end();
    }
}

int KJotsSubject::print(KPrinter& printer, QPainter& painter, int y)
{
    QPaintDeviceMetrics metrics( &printer );
    int maxWidth = metrics.width();
    int maxHeight = metrics.height();
    QStringList paragraphs = QStringList::split("\n", body());

    for (QStringList::iterator para = paragraphs.begin();
         para != paragraphs.end();
         ++para)
    {
        QRect r = painter.boundingRect(0, y,
                                       maxWidth, maxHeight,
                                       QPainter::ExpandTabs | QPainter::WordBreak,
                                       *para);

        if ((y + r.height()) > maxHeight)
        {
            printer.newPage();
            y = 0;
        }

        painter.drawText(0, y, maxWidth, maxHeight - y,
                QPainter::ExpandTabs | QPainter::WordBreak,
                *para);
        y += r.height();
    }

    return y;
}

QString KJotsSubject::defaultSubject()
{
    return i18n("Page %1").arg(m_id);
}

/* ex: set tabstop=4 softtabstop=4 shiftwidth=4 expandtab: */

#include "kjotsentry.moc"
