#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"

#include "main.h"
#include "confdialog.h"
#include "conf.h"

enum {
	CATEGORY_COLUMN = 0,
	PAGE_NUM_COLUMN,
	NUM_COLUMNS
};

static GtkWidget *ConfDialog = NULL;

static void confdialog_respose_callback(GtkDialog *dialog,
					gint id,
					gpointer data)
{
	GError *error = NULL;

	if (id == GTK_RESPONSE_HELP) {
		gnome_help_display("gmfsk.xml", "gmfsk-prefs", &error);

		if (error != NULL) {
			g_warning(error->message);
			g_error_free(error);
		}

		return;
	}

	gtk_widget_destroy(GTK_WIDGET(dialog));
	g_object_set_data(G_OBJECT(appwindow), "configdialog", NULL);
	ConfDialog = NULL;
}

static void confdialog_selection_changed_callback(GtkTreeSelection *selection,
						  gpointer data)
{
	GtkTreeModel *model = GTK_TREE_MODEL(data);
	GtkWidget *notebook;
	GtkTreeIter iter;
	GValue value = { 0 };
	gint page;

	if (!gtk_tree_selection_get_selected(selection, NULL, &iter))
		return;

	gtk_tree_model_get_value(model, &iter, PAGE_NUM_COLUMN, &value);
	page = g_value_get_int(&value);
	g_value_unset(&value);

	notebook = lookup_widget(ConfDialog, "confnotebook");
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
}

static GtkTreeStore *confdialog_build_tree_store(void)
{
	GtkTreeStore *store;
	GtkTreeIter iter, childiter;

	store = gtk_tree_store_new(NUM_COLUMNS,
				   G_TYPE_STRING,
				   G_TYPE_INT);

	gtk_tree_store_append(store, &iter, NULL);
	gtk_tree_store_set(store, &iter,
			   CATEGORY_COLUMN, "General",
			   PAGE_NUM_COLUMN, 0,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Station Info",
			   PAGE_NUM_COLUMN, 1,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Date & Time",
			   PAGE_NUM_COLUMN, 2,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Bands",
			   PAGE_NUM_COLUMN, 3,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Log",
			   PAGE_NUM_COLUMN, 4,
			   -1);

	gtk_tree_store_append(store, &iter, NULL);
	gtk_tree_store_set(store, &iter,
			   CATEGORY_COLUMN, "Appearance",
			   PAGE_NUM_COLUMN, 0,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Colors",
			   PAGE_NUM_COLUMN, 5,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Fonts",
			   PAGE_NUM_COLUMN, 6,
			   -1);

	gtk_tree_store_append(store, &iter, NULL);
	gtk_tree_store_set(store, &iter,
			   CATEGORY_COLUMN, "Modems",
			   PAGE_NUM_COLUMN, 0,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "All",
			   PAGE_NUM_COLUMN, 7,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "RTTY",
			   PAGE_NUM_COLUMN, 8,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "MT63",
			   PAGE_NUM_COLUMN, 9,
			   -1);

	gtk_tree_store_append(store, &iter, NULL);
	gtk_tree_store_set(store, &iter,
			   CATEGORY_COLUMN, "Devices",
			   PAGE_NUM_COLUMN, 0,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Sound",
			   PAGE_NUM_COLUMN, 10,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "PTT",
			   PAGE_NUM_COLUMN, 11,
			   -1);

	gtk_tree_store_append(store, &iter, NULL);
	gtk_tree_store_set(store, &iter,
			   CATEGORY_COLUMN, "Helpers",
			   PAGE_NUM_COLUMN, 0,
			   -1);
	gtk_tree_store_append(store, &childiter, &iter);
	gtk_tree_store_set(store, &childiter,
			   CATEGORY_COLUMN, "Apps",
			   PAGE_NUM_COLUMN, 12,
			   -1);

	return store;
}

enum {
	CONFDIALOG_CALL_ENTRY,
	CONFDIALOG_NAME_ENTRY,
	CONFDIALOG_QTH_ENTRY,
	CONFDIALOG_LOC_ENTRY,
	CONFDIALOG_EMAIL_ENTRY,
	CONFDIALOG_DATE_ENTRY,
	CONFDIALOG_TIME_ENTRY,
	CONFDIALOG_BANDS_ENTRY,
	CONFDIALOG_LOG_ENTRY,
	CONFDIALOG_RTTYSHIFT_ENTRY,
	CONFDIALOG_RTTYBAUD_ENTRY,
	CONFDIALOG_SOUNDDEV_ENTRY,
	CONFDIALOG_PTTDEV_ENTRY
};

static void confdialog_entry_changed(GtkEditable *editable, gpointer data)
{
	gint id = GPOINTER_TO_INT(data);
	gchar *s;

	s = gtk_editable_get_chars(editable, 0, -1);

	switch (id) {
	case CONFDIALOG_CALL_ENTRY:
		conf_set_string("info/mycall", s);
		break;
	case CONFDIALOG_NAME_ENTRY:
		conf_set_string("info/myname", s);
		break;
	case CONFDIALOG_QTH_ENTRY:
		conf_set_string("info/myqth", s);
		break;
	case CONFDIALOG_LOC_ENTRY:
		conf_set_string("info/myloc", s);
		break;
	case CONFDIALOG_EMAIL_ENTRY:
		conf_set_string("info/myemail", s);
		break;
	case CONFDIALOG_DATE_ENTRY:
		conf_set_string("misc/datefmt", s);
		break;
	case CONFDIALOG_TIME_ENTRY:
		conf_set_string("misc/timefmt", s);
		break;
	case CONFDIALOG_BANDS_ENTRY:
		conf_set_string("misc/bands", s);
		break;
	case CONFDIALOG_LOG_ENTRY:
		conf_set_string("misc/logfile", s);
		break;
	case CONFDIALOG_RTTYSHIFT_ENTRY:
		conf_set_float("rtty/shift", atof(s));
		break;
	case CONFDIALOG_RTTYBAUD_ENTRY:
		conf_set_float("rtty/baud", atof(s));
		break;
	case CONFDIALOG_SOUNDDEV_ENTRY:
		conf_set_string("sound/dev", s);
		break;
	case CONFDIALOG_PTTDEV_ENTRY:
		conf_set_string("ptt/dev", s);
		break;
	default:
		g_warning("confdialog_entry_changed: unknown id %d", id);
		break;
	}

	g_free(s);
}

enum {
	CONFDIALOG_OFFSET_SPINBUTTON,
	CONFDIALOG_SQUELCH_SPINBUTTON
};

static void confdialog_spinbutton_changed(GtkSpinButton *spin, gpointer data)
{
	gint id = GPOINTER_TO_INT(data);
	gdouble f;

	f = gtk_spin_button_get_value_as_float(spin);

	switch (id) {
	case CONFDIALOG_OFFSET_SPINBUTTON:
		conf_set_float("misc/txoffset", f);
		break;
	case CONFDIALOG_SQUELCH_SPINBUTTON:
		conf_set_float("misc/sqval", f);
		break;
	default:
		g_warning("confdialog_spinbutton_changed: unknown id %d", id);
		break;
	}
}

enum {
	CONFDIALOG_TX_COLORPICKER,
	CONFDIALOG_RX_COLORPICKER,
	CONFDIALOG_HL_COLORPICKER,
	CONFDIALOG_TXWIN_COLORPICKER,
	CONFDIALOG_RXWIN_COLORPICKER,
};

static void confdialog_colorpicker_set(GnomeColorPicker *colorpicker,
				       guint arg1,
				       guint arg2,
				       guint arg3,
				       guint arg4,
				       gpointer data)
{
	gint id = GPOINTER_TO_INT(data);
	gchar *s;

	s = g_strdup_printf("#%02X%02X%02X",
			    (arg1 & 0xff00) >> 8,
			    (arg2 & 0xff00) >> 8,
			    (arg3 & 0xff00) >> 8);

	switch (id) {
	case CONFDIALOG_TX_COLORPICKER:
		conf_set_string("colors/tx", s);
		break;
	case CONFDIALOG_RX_COLORPICKER:
		conf_set_string("colors/rx", s);
		break;
	case CONFDIALOG_HL_COLORPICKER:
		conf_set_string("colors/hl", s);
		break;
	case CONFDIALOG_TXWIN_COLORPICKER:
		conf_set_string("colors/txwin", s);
		break;
	case CONFDIALOG_RXWIN_COLORPICKER:
		conf_set_string("colors/rxwin", s);
		break;
	default:
		g_warning("confdialog_colorpicker_set: unknown id %d", id);
		break;
	}

	g_free(s);
}

static void color_picker_set(GnomeColorPicker *cp, const gchar *str)
{
	GdkColor clr;

	gdk_color_parse(str, &clr);
	gnome_color_picker_set_i16(cp, clr.red, clr.green, clr.blue, 0);
}

enum {
	CONFDIALOG_TX_FONTPICKER,
	CONFDIALOG_RX_FONTPICKER
};

static void confdialog_fontpicker_set(GnomeFontPicker *fontpicker,
				      gchar *arg,
				      gpointer data)
{
	gint id = GPOINTER_TO_INT(data);

	switch (id) {
	case CONFDIALOG_TX_FONTPICKER:
		conf_set_string("fonts/txfont", arg);
		break;
	case CONFDIALOG_RX_FONTPICKER:
		conf_set_string("fonts/rxfont", arg);
		break;
	default:
		g_warning("confdialog_fontpicker_set: unknown id %d", id);
		break;
	}
}

enum {
	CONFDIALOG_RTTYBITS_MENU,
	CONFDIALOG_RTTYPARITY_MENU,
	CONFDIALOG_RTTYSTOP_MENU,
	CONFDIALOG_MT63BW_MENU,
	CONFDIALOG_MT63ILEAVE_MENU
};

static void confdialog_menu_selected(GtkMenuShell *menushell, gpointer data)
{
	gint id = GPOINTER_TO_INT(data);
	GtkWidget *activeitem;
	gint index;

	activeitem = gtk_menu_get_active(GTK_MENU(menushell));
	index = g_list_index(menushell->children, activeitem);

	switch (id) {
	case CONFDIALOG_RTTYBITS_MENU:
		conf_set_int("rtty/bits", index);
		break;
	case CONFDIALOG_RTTYPARITY_MENU:
		conf_set_int("rtty/parity", index);
		break;
	case CONFDIALOG_RTTYSTOP_MENU:
		conf_set_int("rtty/stop", index);
		break;
	case CONFDIALOG_MT63BW_MENU:
		conf_set_int("mt63/bandwidth", index);
		break;
	case CONFDIALOG_MT63ILEAVE_MENU:
		conf_set_int("mt63/interleave", index);
		break;
	default:
		g_warning("confdialog_menu_selected: unknown id %d", id);
		break;
	}
}

enum {
	CONFDIALOG_RTTYREVERSE_BUTTON,
	CONFDIALOG_RTTYMSB_BUTTON,
	CONFDIALOG_MT63CWID_BUTTON,
	CONFDIALOG_8BIT_BUTTON,
	CONFDIALOG_STEREO_BUTTON,
	CONFDIALOG_FULLDUP_BUTTON,
	CONFDIALOG_TESTMODE_BUTTON,
	CONFDIALOG_PTTINV_BUTTON,
	CONFDIALOG_RTS_RADIOBUTTON,
	CONFDIALOG_DTR_RADIOBUTTON,
	CONFDIALOG_BOTH_RADIOBUTTON
};

static void confdialog_button_toggled(GtkToggleButton *button, gpointer data)
{
	gint id = GPOINTER_TO_INT(data);

	switch (id) {
	case CONFDIALOG_RTTYREVERSE_BUTTON:
		conf_set_bool("rtty/reverse", button->active);
		break;
	case CONFDIALOG_RTTYMSB_BUTTON:
		conf_set_bool("rtty/msbfirst", button->active);
		break;
	case CONFDIALOG_MT63CWID_BUTTON:
		conf_set_bool("mt63/cwid", button->active);
		break;
	case CONFDIALOG_8BIT_BUTTON:
		conf_set_bool("sound/8bit", button->active);
		break;
	case CONFDIALOG_STEREO_BUTTON:
		conf_set_bool("sound/stereo", button->active);
		break;
	case CONFDIALOG_FULLDUP_BUTTON:
		conf_set_bool("sound/fulldup", button->active);
		break;
	case CONFDIALOG_TESTMODE_BUTTON:
		conf_set_bool("sound/testmode", button->active);
		break;
	case CONFDIALOG_PTTINV_BUTTON:
		conf_set_bool("ptt/inverted", button->active);
		break;
	case CONFDIALOG_RTS_RADIOBUTTON:
		if (button->active)
			conf_set_int("ptt/mode", 0);
		break;
	case CONFDIALOG_DTR_RADIOBUTTON:
		if (button->active)
			conf_set_int("ptt/mode", 1);
		break;
	case CONFDIALOG_BOTH_RADIOBUTTON:
		if (button->active)
			conf_set_int("ptt/mode", 2);
		break;
	default:
		g_warning("confdialog_button_toggled: unknown id %d", id);
		break;
	}
}

static void confdialog_fill(GtkWidget *dialog)
{
	GtkWidget *w;
	gchar *s;

	w = lookup_widget(dialog, "callentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_mycall());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_CALL_ENTRY));

	w = lookup_widget(dialog, "nameentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_myname());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_NAME_ENTRY));

	w = lookup_widget(dialog, "qthentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_myqth());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_QTH_ENTRY));

	w = lookup_widget(dialog, "locentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_myloc());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_LOC_ENTRY));

	w = lookup_widget(dialog, "emailentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_myemail());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_EMAIL_ENTRY));

	w = lookup_widget(dialog, "dateentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_datefmt());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_DATE_ENTRY));

	w = lookup_widget(dialog, "timeentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_timefmt());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_TIME_ENTRY));

	w = lookup_widget(dialog, "qsobandsentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_qsobands());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_BANDS_ENTRY));

	w = lookup_widget(dialog, "logfileentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_logfile());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_LOG_ENTRY));

	w = lookup_widget(dialog, "txcolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(w),
			 conf_get_string("colors/tx", "red"));
	g_signal_connect((gpointer) w, "color_set",
			 G_CALLBACK(confdialog_colorpicker_set),
			 GINT_TO_POINTER(CONFDIALOG_TX_COLORPICKER));

	w = lookup_widget(dialog, "rxcolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(w),
			 conf_get_string("colors/rx", "black"));
	g_signal_connect((gpointer) w, "color_set",
			 G_CALLBACK(confdialog_colorpicker_set),
			 GINT_TO_POINTER(CONFDIALOG_RX_COLORPICKER));

	w = lookup_widget(dialog, "hlcolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(w),
			 conf_get_string("colors/hl", "blue"));
	g_signal_connect((gpointer) w, "color_set",
			 G_CALLBACK(confdialog_colorpicker_set),
			 GINT_TO_POINTER(CONFDIALOG_HL_COLORPICKER));

	w = lookup_widget(dialog, "txwincolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(w),
			 conf_get_string("colors/txwin", "white"));
	g_signal_connect((gpointer) w, "color_set",
			 G_CALLBACK(confdialog_colorpicker_set),
			 GINT_TO_POINTER(CONFDIALOG_TXWIN_COLORPICKER));

	w = lookup_widget(dialog, "rxwincolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(w),
			 conf_get_string("colors/rxwin", "white"));
	g_signal_connect((gpointer) w, "color_set",
			 G_CALLBACK(confdialog_colorpicker_set),
			 GINT_TO_POINTER(CONFDIALOG_RXWIN_COLORPICKER));

	w = lookup_widget(dialog, "rxfontpicker");
	gnome_font_picker_set_font_name(GNOME_FONT_PICKER(w),
					conf_get_string("fonts/rxfont",
							"sans 10"));
	gnome_font_picker_set_mode(GNOME_FONT_PICKER(w),
				   GNOME_FONT_PICKER_MODE_FONT_INFO);
	g_signal_connect((gpointer) w, "font_set",
			 G_CALLBACK(confdialog_fontpicker_set),
			 GINT_TO_POINTER(CONFDIALOG_RX_FONTPICKER));

	w = lookup_widget(dialog, "txfontpicker");
	gnome_font_picker_set_font_name(GNOME_FONT_PICKER(w),
					conf_get_string("fonts/txfont",
							"sans 10"));
	gnome_font_picker_set_mode(GNOME_FONT_PICKER(w),
				   GNOME_FONT_PICKER_MODE_FONT_INFO);
	g_signal_connect((gpointer) w, "font_set",
			 G_CALLBACK(confdialog_fontpicker_set),
			 GINT_TO_POINTER(CONFDIALOG_TX_FONTPICKER));

	w = lookup_widget(dialog, "txoffsetspinbutton");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(w),
				  conf_get_float("misc/txoffset", 0.0));
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_spinbutton_changed),
			 GINT_TO_POINTER(CONFDIALOG_OFFSET_SPINBUTTON));

	w = lookup_widget(dialog, "squelchspinbutton");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(w),
				  conf_get_float("misc/sqval", 15.0));
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_spinbutton_changed),
			 GINT_TO_POINTER(CONFDIALOG_SQUELCH_SPINBUTTON));

	w = lookup_widget(dialog, "rttyshiftentry");
	s = g_strdup_printf("%g", conf_get_float("rtty/shift", 170.0));
	gtk_entry_set_text(GTK_ENTRY(w), s);
	g_free(s);
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_RTTYSHIFT_ENTRY));

	w = lookup_widget(dialog, "rttybaudentry");
	s = g_strdup_printf("%g", conf_get_float("rtty/baud", 45.45));
	gtk_entry_set_text(GTK_ENTRY(w), s);
	g_free(s);
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_RTTYBAUD_ENTRY));

	w = lookup_widget(dialog, "bitsoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(w),
				    conf_get_int("rtty/bits", 0));
	g_signal_connect((gpointer) GTK_OPTION_MENU(w)->menu, "deactivate",
			 G_CALLBACK(confdialog_menu_selected),
			 GINT_TO_POINTER(CONFDIALOG_RTTYBITS_MENU));

	w = lookup_widget(dialog, "parityoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(w),
				    conf_get_int("rtty/parity", 0));
	g_signal_connect((gpointer) GTK_OPTION_MENU(w)->menu, "deactivate",
			 G_CALLBACK(confdialog_menu_selected),
			 GINT_TO_POINTER(CONFDIALOG_RTTYPARITY_MENU));

	w = lookup_widget(dialog, "stopoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(w),
				    conf_get_int("rtty/stop", 1));
	g_signal_connect((gpointer) GTK_OPTION_MENU(w)->menu, "deactivate",
			 G_CALLBACK(confdialog_menu_selected),
			 GINT_TO_POINTER(CONFDIALOG_RTTYSTOP_MENU));

	w = lookup_widget(dialog, "reversecheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w),
				     conf_get_bool("rtty/reverse", FALSE));
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_RTTYREVERSE_BUTTON));

	w = lookup_widget(dialog, "msbcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w),
				     conf_get_bool("rtty/msbfirst", FALSE));
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_RTTYMSB_BUTTON));

	w = lookup_widget(dialog, "bwoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(w),
				    conf_get_int("mt63/bandwidth", 1));
	g_signal_connect((gpointer) GTK_OPTION_MENU(w)->menu, "deactivate",
			 G_CALLBACK(confdialog_menu_selected),
			 GINT_TO_POINTER(CONFDIALOG_MT63BW_MENU));

	w = lookup_widget(dialog, "ileaveoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(w),
				    conf_get_int("mt63/interleave", 1));
	g_signal_connect((gpointer) GTK_OPTION_MENU(w)->menu, "deactivate",
			 G_CALLBACK(confdialog_menu_selected),
			 GINT_TO_POINTER(CONFDIALOG_MT63ILEAVE_MENU));

	w = lookup_widget(dialog, "cwidcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w),
				     conf_get_bool("mt63/cwid", FALSE));
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_MT63CWID_BUTTON));

	w = lookup_widget(dialog, "soundentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_sounddev());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_SOUNDDEV_ENTRY));

	w = lookup_widget(dialog, "eightbitcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), conf_get_eightbit());
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_8BIT_BUTTON));

	w = lookup_widget(dialog, "stereocheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), conf_get_stereo());
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_STEREO_BUTTON));

	w = lookup_widget(dialog, "rwcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), conf_get_fulldup());
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_FULLDUP_BUTTON));

	w = lookup_widget(dialog, "testcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), conf_get_testmode());
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_TESTMODE_BUTTON));

	w = lookup_widget(dialog, "pttentry");
	gtk_entry_set_text(GTK_ENTRY(w), conf_get_pttdev());
	g_signal_connect((gpointer) w, "changed",
			 G_CALLBACK(confdialog_entry_changed),
			 GINT_TO_POINTER(CONFDIALOG_PTTDEV_ENTRY));

	w = lookup_widget(dialog, "pttinvcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), conf_get_pttinv());
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_PTTINV_BUTTON));

	switch (conf_get_pttmode()) {
	case 0:
		w = lookup_widget(dialog, "pttrtsradiobutton");
		break;
	case 1:
		w = lookup_widget(dialog, "pttdtrradiobutton");
		break;
	case 2:
		w = lookup_widget(dialog, "pttbothradiobutton");
		break;
	}
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
	w = lookup_widget(dialog, "pttrtsradiobutton");
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_RTS_RADIOBUTTON));
	w = lookup_widget(dialog, "pttdtrradiobutton");
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_DTR_RADIOBUTTON));
	w = lookup_widget(dialog, "pttbothradiobutton");
	g_signal_connect((gpointer) w, "toggled",
			 G_CALLBACK(confdialog_button_toggled),
			 GINT_TO_POINTER(CONFDIALOG_BOTH_RADIOBUTTON));
}

GtkWidget *confdialog_init(void)
{
	GtkTreeView *treeview;
	GtkCellRenderer *renderer;
	GtkTreeSelection *selection;
	GtkTreeViewColumn *column;
	GtkTreeModel *model;
	gint offset;

	ConfDialog = create_confdialog();

	confdialog_fill(ConfDialog);

	g_signal_connect(G_OBJECT(ConfDialog), "response",
			 G_CALLBACK(confdialog_respose_callback),
			 NULL);

	model = GTK_TREE_MODEL(confdialog_build_tree_store());

	treeview = GTK_TREE_VIEW(lookup_widget(ConfDialog, "conftreeview"));
	gtk_tree_view_set_model(treeview, model);

	g_object_unref(G_OBJECT(model));

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

	renderer = gtk_cell_renderer_text_new();
	g_object_set(G_OBJECT(renderer), "xalign", 0.0, NULL);

	offset = gtk_tree_view_insert_column_with_attributes(treeview,
							     -1,
							     _("Categories"),
							     renderer,
							     "text",
							     CATEGORY_COLUMN,
							     NULL);

	column = gtk_tree_view_get_column(treeview, offset - 1);
	gtk_tree_view_column_set_clickable(column, FALSE);

	g_signal_connect(G_OBJECT(selection), "changed",
			 G_CALLBACK(confdialog_selection_changed_callback),
			 (gpointer) model);

	return ConfDialog;
}
