/*
 * bltGraph.h --
 *
 * Copyright 1991-1993 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 * Graph widget created by Sani Nassif and George Howlett.
 */

#ifndef _GRAPH_H
#define _GRAPH_H

#include "bltConfig.h"
#include <stdio.h>
#include <math.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_FLOAT_H
#include <float.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef _TK
#include <tk.h>
#endif

/* math.h */

#ifndef MIN
#define MIN(a,b)	(((a)<(b))?(a):(b))
#endif /* MIN */

#ifndef MAX
#define MAX(a,b)	(((a)>(b))?(a):(b))
#endif /* MAX */

#ifdef LOSE_RINT
#undef HAVE_RINT
#endif
#ifdef LOSE_SINCOS
#undef HAVE_SINCOS
#endif

#ifndef HAVE_RINT
#define rint(x)		((int)((x) + (((x)<0.0) ? -0.5 : 0.5)))
#endif /* HAVE_RINT */

#ifndef M_PI
#define M_PI    	3.14159265358979323846
#endif /* M_PI */

#ifndef M_SQRT2
#define M_SQRT2		1.41421356237309504880
#endif /* M_SQRT1_2 */

#ifndef M_SQRT1_2
#define M_SQRT1_2	0.70710678118654752440
#endif /* M_SQRT1_2 */

#ifndef SHRT_MAX
#define SHRT_MAX	0x7FFF
#endif /* SHRT_MAX */
#include "bltList.h"

#define TRUE 1
#define FALSE 0

/*
 * Mask values used to selectively enable entries in the
 * configuration specs:
 */

#define XYGRAPH_MASK	TK_CONFIG_USER_BIT
#define BARCHART_MASK	TK_CONFIG_USER_BIT << 1
#define CONTOUR_MASK	TK_CONFIG_USER_BIT << 2
#define PIECHART_MASK	TK_CONFIG_USER_BIT << 3
#define ALL_MASK	(XYGRAPH_MASK | BARCHART_MASK)

/*
 * NUMDIGITS specifies the number of digits accuracy used when
 * outputing data (e.g. axis tick labels, etc)
 */
#define NUMDIGITS	12

#define PADX		2	/* Padding between labels/titles */
#define PADY    	2	/* Padding between labels */
#define TEXTHEIGHT(f) 	((f)->ascent + (f)->descent)

#define MAJOR_TICK 	0.020	/* Length of a major tick */
#define MINOR_TICK 	0.015	/* Length of a minor (sub)tick */
#define LABEL_TICK 	0.030	/* Distance from graph to start of label */

#define AUTOSCALE_MIN(a) ((a)->reqPlotMin == Blt_negInfinity)
#define AUTOSCALE_MAX(a) ((a)->reqPlotMax == Blt_posInfinity)

#define DEF_POSITION -SHRT_MAX	/* Indicates that no position was specified */

/*
 * -------------------------------------------------------------------
 * Graph component structure definitions
 * -------------------------------------------------------------------
 */

typedef struct Graph Graph;
typedef struct GraphAxis GraphAxis;
typedef struct GraphLegend GraphLegend;
typedef struct GraphPostScript GraphPostScript;
typedef struct GraphCrosshairs GraphCrosshairs;

/*
 * -------------------------------------------------------------------
 *
 * GraphClassType --
 *
 *	Enumerates the different types of graphs this program
 *	produces. Currently, only one this of graph is allowed
 *	on a single axis.
 *
 * -------------------------------------------------------------------
 */
typedef enum {
    GRAPH_TYPE, BARCHART_TYPE, CONTOUR_TYPE, PIECHART_TYPE
} GraphClassType;

/*
 * -------------------------------------------------------------------
 *
 * AxisTypes --
 *
 *	Enumerated type representing the types of axes
 *
 * -------------------------------------------------------------------
 */
typedef enum AxisTypes {
    X_AXIS_TYPE, Y_AXIS_TYPE, Z_AXIS_TYPE
} AxisType;


/*
 * -------------------------------------------------------------------
 *
 * GraphLegend --
 *
 * 	Contains information specific to how the legend will be
 *	displayed.
 *
 * -------------------------------------------------------------------
 */

typedef void (LegendDisplayProc) _ANSI_ARGS_((Graph *));
typedef void (LegendPrintProc) _ANSI_ARGS_((Graph *));
typedef void (LegendDestroyProc) _ANSI_ARGS_((Graph *));
typedef int (LegendExtentsProc) _ANSI_ARGS_((Graph *));

struct GraphLegend {
    int mapped;			/* Requested state of the legend, If non-zero,
				 * legend is displayed */
    int width, height;		/* Dimensions of the legend */
    XPoint position;		/* Window coordinates of legend positioning
				 * point. Used in conjunction with the anchor
				 * to determine the location of the legend. If
				 * x or y are DEF_POSITION the legend is set
				 * in the right margin */

    LegendDisplayProc *displayProc;
    LegendPrintProc *printProc;
    LegendDestroyProc *destroyProc;
    LegendExtentsProc *extentsProc;
};

/*
 * -------------------------------------------------------------------
 *
 * GraphPostScript --
 *
 * 	Structure contains information specific to the outputing of
 *	PostScript commands to print the graph.
 *
 * -------------------------------------------------------------------
 */

typedef int (PostScriptConfigureProc) _ANSI_ARGS_((Graph *, int, char **));
typedef int (PostScriptPrintProc) _ANSI_ARGS_((Graph *, int, char **));
typedef void (PostScriptDestroyProc) _ANSI_ARGS_((Graph *));

struct GraphPostScript {
    PostScriptConfigureProc *configProc;
    PostScriptPrintProc *printProc;
    PostScriptDestroyProc *destroyProc;
};

/*
 * -------------------------------------------------------------------
 *
 * GraphAxis --
 *
 * 	Structure contains options controlling how the axis will be
 * 	displayed.
 *
 * -------------------------------------------------------------------
 */

typedef void (AxisDisplayProc) _ANSI_ARGS_((Graph *));
typedef void (AxisPrintProc) _ANSI_ARGS_((Graph *));
typedef void (AxisLayoutProc) _ANSI_ARGS_((Graph *));
typedef void (AxisDestroyProc) _ANSI_ARGS_((Graph *, GraphAxis *));

struct GraphAxis {
    AxisDisplayProc *displayProc;
    AxisPrintProc *printProc;
    AxisLayoutProc *layoutProc;
    AxisDestroyProc *destroyProc;
};

/*
 * -------------------------------------------------------------------
 *
 * GraphCrosshairs
 *
 *	Contains the line segments positions and graphics context used
 *	to simulate crosshairs (by XORing) on the graph.
 *
 * -------------------------------------------------------------------
 */
typedef void (CrosshairsToggleProc) _ANSI_ARGS_((Graph *));
typedef void (CrosshairsUpdateProc) _ANSI_ARGS_((Graph *));
typedef void (CrosshairsDestroyProc) _ANSI_ARGS_((Graph *));

struct GraphCrosshairs {

    CrosshairsToggleProc *toggleProc;
				/* Routine to toggle visiblity of crosshairs */
    CrosshairsUpdateProc *updateProc;
				/* Routine to update lengths of hairs */
    CrosshairsDestroyProc *destroyProc;
				/* Routine to release X resources allocated
				 * for crosshairs and free memory */


};

/*
 * -------------------------------------------------------------------
 *
 * Graph --
 *
 *	Top level structure containing everything pertaining to
 *	the graph.
 *
 * -------------------------------------------------------------------
 */
struct Graph {
    Tk_Window tkwin;		/* Window that embodies the graph.  NULL means
				 * that the window has been destroyed but the
				 * data structures haven't yet been cleaned
				 * up. */
    Pixmap canvas;		/* Pixmap for double buffering output */
    Display *display;		/* Display containing widget;  needed, among
				 * other things, to release resources after
				 * tkwin has already gone away. */
    char *pathName;		/* Pathname of the widget. Is saved here in
				 * case the widget is destroyed and tkwin
				 * become unavailable for querying */
    Tcl_Interp *interp;		/* Interpreter associated with graph widget */
    GraphClassType type;	/* Type of graph: GRAPH_TYPE, BARCHART_TYPE,
				 * or PIECHART_TYPE. */
    unsigned int flags;		/* Flags;  see below for definitions. */
    int reqWidth, reqHeight;	/* Requested size of graph window */
    int width, height;		/* Size of graph window or PostScript page */

    GraphPostScript *postscript;/* PostScript options */
    GraphLegend *legendPtr;	/* Legend information */
    GraphAxis *X, *Y;		/* X and Y Axis information */
    GraphCrosshairs *crosshairs;

    char *scratchPtr;		/* Utility space for building strings. Points
				 * to buffer of BUFSIZ bytes on the stack. 
				 * Currently used to create PostScript output 
				 * for the "postscript" command. */

    int leftMargin;		/* The following are the requested sizes for */
    int rightMargin;		/* the margins surrounding the plotting area */
    int topMargin;		/* If non-zero, the requested margin is used */
    int bottomMargin;		/* Otherwise the computed margin sizes are
				 * used. */
    XPoint origin;		/* Origin of the graph */
    XPoint extreme;		/* Outer limit of graph */

    Tk_3DBorder border;		/* 3-D border used to delineate the plot
				 * surface and outer edge of window */
    GC intGC;			/* Interior region (plotting surface) GC */
    int intBWidth;		/* Width of interior 3-D border. */
    int intRelief;		/* 3-d effect: TK_RELIEF_RAISED etc. */

    GC extGC;			/* Graphics context for margin. Includes
				 * margin background */
    GC extBgGC;			/* Background GC for margin areas */
    int extBWidth;		/* Width the exterior border */
    int extRelief;		/* Relief of the exterior border */

    XColor *intBgColorPtr;	/* Color of plot surface */
    double barWidth;		/* Scale factor for the width of bar */
    double avgSymSize;		/* Average size of a symbol */

    char *title;		/* Graph title */
    char *xTitle;		/* Axis titles */
    char *yTitle;
    XFontStruct *fontPtr;	/* Font for title and axis labels */
    XColor *fgColorPtr;		/* Foreground color of title and axis labels */
    Cursor cursor;

    Tcl_HashTable elemTable;	/* Hash table containing all elements created,
				 * not just those currently displayed. */
    Blt_LinkedList elemList;	/* Display list of elements */
    Tcl_HashTable tagTable;	/* Hash table of tags */
    Blt_LinkedList tagList;	/* Display list of tags */
    int nextTagId;		/* Tracks next tag identifier available */

};

/*
 * Flag bits for graphs:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler has
 *				already been queued to redraw this window.
 *
 * REDRAW_ALL:		        Non-zero means that exterior region of the
 *				graph, in addition to the interior (plotting
 *				surface) needs to be redrawn. The possible
 *				reasons are: 1) an axis configuration changed
 *				2) an axis limit changed 3) titles have changed
 *				4) window was resized.
 *
 * LAYOUT_PENDING:		Non-zero means that a graph configuration
 *				has changed (element, tag, axis, legend, etc)
 *				and the layout of the graph (position of the
 *				graph in the window) needs to be recalculated.
 *
 * LAYOUT_ALL:		        Non-zero indicates that the layout of the axes
 *				and all elements and tags need to be
 *				recalculated.  Otherwise, the layout routines
 *				will recompute the layout of only those tags
 *				and elements that have changed.
 *
 */
#define REDRAW_PENDING		(1<<0)
#define REDRAW_ALL		(1<<1)
#define LAYOUT_PENDING	        (1<<2)
#define LAYOUT_ALL		(1<<3)

/*
 * -------------------------------------------------------------------
 *
 * TextAttr --
 *
 * 	Represents a convenience structure to hold text attributes
 *	which determine how a text string is to be displayed on the
 *	window, or drawn with PostScript commands.  The alternative
 *	is to have drawing and printing routines with more parameters.
 * 	This seems like a more efficient and less cumbersome way.
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *text;			/* Text string to be displayed  */
    XFontStruct *fontPtr;	/* Font to use for string */
    XColor *bgColorPtr;		/* Background color of string */
    XColor *fgColorPtr;		/* Foreground color of string */
    Tk_Anchor anchor;		/* Anchor type: used to position the string */
    double theta;		/* Rotation of text in degrees. */
    GC gc;			/* Graphics context to use when displaying the
				 * string on the window */
} TextAttr;


/*
 * ---------------------- Forward declarations ------------------------
 */
extern Pixmap Blt_CreateTextBitmap _ANSI_ARGS_((Display *display,
	Drawable draw, XFontStruct *fontPtr, char *textStr, double theta,
	unsigned int *bmWidthPtr, unsigned int *bmHeightPtr));
extern void Blt_DrawText _ANSI_ARGS_((Display *display, Drawable draw,
	char *text, TextAttr *attrPtr, int x, int y));
extern void Blt_EventuallyRedraw _ANSI_ARGS_((Graph *graphPtr));
extern void Blt_GetBoundingBox _ANSI_ARGS_((int width, int height, double theta,
	int *widthPtr, int *heightPtr, XPoint *pointArr));
extern Pixmap Blt_RotateBitmap _ANSI_ARGS_((Display *display, Drawable draw,
	GC gc, Pixmap bitmap, int width, int height, double theta,
	unsigned int *rotWidthPtr, unsigned int *rotHeightPtr));
extern void Blt_ComputeAxes _ANSI_ARGS_((Graph *graphPtr));
extern XPoint Blt_WinPt _ANSI_ARGS_((Graph *graphPtr, double x, double y));
extern int Blt_WinX _ANSI_ARGS_((Graph *graphPtr, double x));
extern int Blt_WinY _ANSI_ARGS_((Graph *graphPtr, double y));
extern int Blt_WinXAbs _ANSI_ARGS_((Graph *graphPtr, double x));
extern int Blt_WinYAbs _ANSI_ARGS_((Graph *graphPtr, double y));
extern void Blt_StencilBitmap _ANSI_ARGS_((Display *display, Drawable draw,
	GC gc, Pixmap bitmap, int x, int y, unsigned int width,
	unsigned int height));
extern unsigned int Blt_TextStringWidth _ANSI_ARGS_((XFontStruct *fontPtr,
	char *text));
extern XPoint Blt_TranslateBoxCoords _ANSI_ARGS_((int x, int y, int width,
	int height, Tk_Anchor anchor));
extern XPoint Blt_TranslateTextCoords _ANSI_ARGS_((XFontStruct *fontPtr,
	char *text, int x, int y, Tk_Anchor anchor));
extern double Blt_WorldX _ANSI_ARGS_((Graph *graphPtr, int x));
extern double Blt_WorldY _ANSI_ARGS_((Graph *graphPtr, int y));

extern void Blt_PrintBitmap _ANSI_ARGS_((Graph *graphPtr,
	Pixmap bitmap, int x, int y, int width, int height));
extern void Blt_SetLineAttributes _ANSI_ARGS_((Graph *graphPtr,
	XColor *colorPtr, int lineWidth, int lineDashes));
extern void Blt_3DRectangleToPostScript _ANSI_ARGS_((Graph *graphPtr,
	Tk_3DBorder border, int x, int y, int width, int height,
	int borderWidth, int relief));
extern void Blt_BackgroundToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XColor *colorPtr));
extern void Blt_BitmapToPostScript _ANSI_ARGS_((Graph *graphPtr,
	Pixmap bitmap, int x, int y, int width, int height, double theta,
	XColor *bgColorPtr));
extern void Blt_FontToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XFontStruct *fontPtr));
extern void Blt_ForegroundToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XColor *colorPtr));
extern void Blt_LineDashesToPostScript _ANSI_ARGS_((Graph *graphPtr,
	int lineDashes));
extern void Blt_LineWidthToPostScript _ANSI_ARGS_((Graph *graphPtr,
	int lineWidth));
extern void Blt_LinesToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XPoint *pointArr, int numPoints));
extern void Blt_PolygonToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XPoint *pointArr, int numPoints));
extern void Blt_Print3DRectangle _ANSI_ARGS_((Graph *graphPtr,
	Tk_3DBorder border, int x, int y, int width, int height,
	int borderWidth, int relief));
extern void Blt_RectangleToPostScript _ANSI_ARGS_((Graph *graphPtr,
	int x, int y, unsigned int width, unsigned int height));
extern void Blt_RectanglesToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XRectangle *rectArr, int numRects));
extern void Blt_SegmentsToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XSegment *segArr, int numSegs));
extern void Blt_StippleToPostScript _ANSI_ARGS_((Graph *graphPtr,
	Pixmap bitmap, unsigned int width, unsigned int height, int fgOrBg));
extern void Blt_TextToPostScript _ANSI_ARGS_((Graph *graphPtr, char *text,
	TextAttr *attrPtr, int x, int y));

#ifdef HAVE_RINT
#ifndef __GNU_LIBRARY__
extern double rint _ANSI_ARGS_((double x));
#endif
#endif
extern double exp10 _ANSI_ARGS_((double x));
extern void sincos _ANSI_ARGS_((double x, double *sinX, double *cosX));
extern char *strdup _ANSI_ARGS_((CONST char *s));


/*
 * ---------------------- Global declarations ------------------------
 */
#ifndef __GNU_LIBRARY__
extern double HUGE_VAL;		/* this may be a function */
#endif
extern double Blt_negInfinity, Blt_posInfinity;

#endif /* _GRAPH_H */
