/*
 * bltBitmap.c --
 *
 *	This module implements Tcl bitmaps for the Tk toolkit.
 *
 *	Much of the code is taken from XRdBitF.c and XWrBitF.c
 *	from the MIT X11R5 distribution.
 *
 * Copyright, 1987, Massachusetts Institute of Technology
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 *
 * Copyright 1993 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 * "blt_bitmap" command developed by George Howlett.
 */

#include <tk.h>
#include <X11/Xutil.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#ifndef BITMAP_VERSION
#define BITMAP_VERSION "1.0"
#endif

#define MAX_SIZE 255
enum Formats {
    UNKNOWN, V10, V11
};

/* Shared data for the image read/parse logic */
static short hexTable[256];	/* conversion value */
static int initialized = 0;	/* easier to fill in at run time */

extern int Blt_FindCmd _ANSI_ARGS_((Tcl_Interp *, char *, ClientData *));

#define blt_width 40
#define blt_height 40
static char blt_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xff, 0xff, 0x03, 0x00, 0x04,
   0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x00, 0x02, 0x00, 0xe4, 0x33, 0x3f,
   0x01, 0x00, 0x64, 0x36, 0x0c, 0x01, 0x00, 0x64, 0x36, 0x8c, 0x00, 0x00,
   0xe4, 0x33, 0x8c, 0x00, 0x00, 0x64, 0x36, 0x8c, 0x00, 0x00, 0x64, 0x36,
   0x0c, 0x01, 0x00, 0xe4, 0xf3, 0x0d, 0x01, 0x00, 0x04, 0x00, 0x00, 0x02,
   0x00, 0x04, 0x00, 0x00, 0x02, 0x00, 0xfc, 0xff, 0xff, 0x03, 0x00, 0x0c,
   0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x0c, 0xf8, 0xff,
   0x03, 0x80, 0xed, 0x07, 0x00, 0x04, 0xe0, 0x0c, 0x00, 0x20, 0x09, 0x10,
   0x0c, 0x00, 0x00, 0x12, 0x10, 0x0c, 0x00, 0x00, 0x10, 0x30, 0x00, 0x00,
   0x00, 0x19, 0xd0, 0x03, 0x00, 0x00, 0x14, 0xb0, 0xfe, 0xff, 0xff, 0x1b,
   0x50, 0x55, 0x55, 0x55, 0x0d, 0xe8, 0xaa, 0xaa, 0xaa, 0x16, 0xe4, 0xff,
   0xff, 0xff, 0x2f, 0xf4, 0xff, 0xff, 0xff, 0x27, 0xd8, 0xae, 0xaa, 0xbd,
   0x2d, 0x6c, 0x5f, 0xd5, 0x67, 0x1b, 0xbc, 0xf3, 0x7f, 0xd0, 0x36, 0xf8,
   0x01, 0x10, 0xcc, 0x1f, 0xe0, 0x45, 0x8e, 0x92, 0x0f, 0xb0, 0x32, 0x41,
   0x43, 0x0b, 0xd0, 0xcf, 0x3c, 0x7c, 0x0d, 0xb0, 0xaa, 0xc2, 0xab, 0x0a,
   0x60, 0x55, 0x55, 0x55, 0x05, 0xc0, 0xff, 0xab, 0xaa, 0x03, 0x00, 0x00,
   0xfe, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

#define bigblt_width 64
#define bigblt_height 64
static char bigblt_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0x3f, 0x00,
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00,
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00,
   0x00, 0x00, 0xe2, 0x0f, 0xc7, 0xff, 0x10, 0x00, 0x00, 0x00, 0xe2, 0x1f,
   0xc7, 0xff, 0x10, 0x00, 0x00, 0x00, 0xe2, 0x38, 0x07, 0x1c, 0x08, 0x00,
   0x00, 0x00, 0xe2, 0x38, 0x07, 0x1c, 0x08, 0x00, 0x00, 0x00, 0xe2, 0x38,
   0x07, 0x1c, 0x08, 0x00, 0x00, 0x00, 0xe2, 0x1f, 0x07, 0x1c, 0x04, 0x00,
   0x00, 0x00, 0xe2, 0x1f, 0x07, 0x1c, 0x04, 0x00, 0x00, 0x00, 0xe2, 0x38,
   0x07, 0x1c, 0x08, 0x00, 0x00, 0x00, 0xe2, 0x38, 0x07, 0x1c, 0x08, 0x00,
   0x00, 0x00, 0xe2, 0x38, 0x07, 0x1c, 0x08, 0x00, 0x00, 0x00, 0xe2, 0x1f,
   0xff, 0x1c, 0x10, 0x00, 0x00, 0x00, 0xe2, 0x0f, 0xff, 0x1c, 0x10, 0x00,
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00,
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0x3f, 0x00, 0x00, 0x00, 0x06, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0xc0, 0xff, 0xff, 0x07, 0x00,
   0x00, 0xe0, 0xf6, 0x3f, 0x00, 0x00, 0x38, 0x00, 0x00, 0x1c, 0x06, 0x00,
   0x00, 0x00, 0xc0, 0x00, 0x80, 0x03, 0x06, 0x00, 0x00, 0xc0, 0x08, 0x03,
   0x40, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x04, 0x40, 0x00, 0x06, 0x00,
   0x00, 0x00, 0x40, 0x04, 0x40, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x04,
   0x40, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x04, 0xc0, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x0c, 0x06, 0x40, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
   0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x06, 0x40, 0x55, 0xff, 0xff,
   0xff, 0xff, 0x7f, 0x05, 0x80, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x06,
   0x80, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x03, 0x40, 0xab, 0xaa, 0xaa,
   0xaa, 0xaa, 0xaa, 0x01, 0x70, 0x57, 0x55, 0x55, 0x55, 0x55, 0xd5, 0x04,
   0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0b, 0xd8, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0x14, 0xd0, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0x13,
   0xf0, 0xda, 0xbf, 0xaa, 0xba, 0xfd, 0xd6, 0x0b, 0x70, 0xed, 0x77, 0x55,
   0x57, 0xe5, 0xad, 0x07, 0xb8, 0xf7, 0xab, 0xaa, 0xaa, 0xd2, 0x5b, 0x0f,
   0xf8, 0xfb, 0x54, 0x55, 0x75, 0x94, 0xf7, 0x1e, 0xf0, 0x7b, 0xfa, 0xff,
   0x9f, 0xa9, 0xef, 0x1f, 0xc0, 0xbf, 0x00, 0x20, 0x40, 0x54, 0xfe, 0x0f,
   0x00, 0x1f, 0x92, 0x00, 0x04, 0xa9, 0xfc, 0x01, 0xc0, 0x5f, 0x41, 0xf9,
   0x04, 0x21, 0xfd, 0x00, 0xc0, 0x9b, 0x28, 0x04, 0xd8, 0x0a, 0x9a, 0x03,
   0x40, 0x5d, 0x08, 0x40, 0x44, 0x44, 0x62, 0x03, 0xc0, 0xaa, 0x67, 0xe2,
   0x03, 0x64, 0xba, 0x02, 0x40, 0x55, 0xd5, 0x55, 0xfd, 0xdb, 0x55, 0x03,
   0x80, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x01, 0x00, 0x57, 0x55, 0x55,
   0x55, 0x55, 0xd5, 0x00, 0x00, 0xac, 0xaa, 0xaa, 0xaa, 0xaa, 0x2a, 0x00,
   0x00, 0xf0, 0xff, 0x57, 0x55, 0x55, 0x1d, 0x00, 0x00, 0x00, 0x00, 0xf8,
   0xff, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/*
 *--------------------------------------------------------------
 *
 * InitHexTable --
 *
 *	Table index for the hex values. Initialized once, first time.
 *	Used for translation value or delimiter significance lookup.
 *
 *	We build the table at run time for several reasons:
 *
 *     	  1.  portable to non-ASCII machines.
 *	  2.  still reentrant since we set the init flag after setting
 *            table.
 *        3.  easier to extend.
 *        4.  less prone to bugs.
 *
 * Results:
 *	None.
 *
 *--------------------------------------------------------------
 */
static void
InitHexTable()
{
    hexTable['0'] = 0;
    hexTable['1'] = 1;
    hexTable['2'] = 2;
    hexTable['3'] = 3;
    hexTable['4'] = 4;
    hexTable['5'] = 5;
    hexTable['6'] = 6;
    hexTable['7'] = 7;
    hexTable['8'] = 8;
    hexTable['9'] = 9;
    hexTable['A'] = 10;
    hexTable['B'] = 11;
    hexTable['C'] = 12;
    hexTable['D'] = 13;
    hexTable['E'] = 14;
    hexTable['F'] = 15;
    hexTable['a'] = 10;
    hexTable['b'] = 11;
    hexTable['c'] = 12;
    hexTable['d'] = 13;
    hexTable['e'] = 14;
    hexTable['f'] = 15;
}

/*
 *--------------------------------------------------------------
 *
 * GetHexValue --
 *
 *	Converts the hexadecimal string into an unsigned integer
 *	value.  The hexadecimal string need not have a leading "0x".
 *
 * Results:
 *	Returns a standard TCL result. If the conversion was
 *	successful, TCL_OK is returned, otherwise TCL_ERROR.
 *
 * Side Effects:
 * 	If the conversion fails, interp->result is filled with an
 *	error message.
 *
 *--------------------------------------------------------------
 */
static int
GetHexValue(interp, string, valuePtr)
    Tcl_Interp *interp;
    char *string;
    int *valuePtr;
{
    register int c;
    register char *s;
    register int value;

    s = string;
    if ((s[0] == '0') && ((s[1] == 'x') || (s[1] == 'X'))) {
	s += 2;
    }
    if (s[0] == '\0') {
	Tcl_AppendResult(interp, "expecting hex value: got \"", string, "\"",
	    (char *)NULL);
	return TCL_ERROR;	/* Only found "0x"  */
    }
    value = 0;
    for ( /*empty*/ ; *s != '\0'; s++) {
	/* trim high bits, check type and accumulate */
	c = *s & 0xff;
	if (!(isascii(c) && isxdigit(c))) {
	    Tcl_AppendResult(interp, "expecting hex value: got \"", string,
		"\"", (char *)NULL);
	    return TCL_ERROR;	/* Not a hexadecimal number */
	}
	value = (value << 4) + hexTable[c];
    }
    *valuePtr = value;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DefineBitmap --
 *
 *	Converts the dataList into an internal bitmap.
 *
 * Results:
 *	A standard TCL result.
 *
 * Side Effects:
 * 	If an error occurs while processing the data, interp->result
 *	is filled with a corresponding error message.
 *
 *--------------------------------------------------------------
 */
static int
DefineBitmap(interp, nameId, dataList)
    Tcl_Interp *interp;		/* Interpreter to report results to */
    Tk_Uid nameId;
    char *dataList;
{
    int arraySize;		/* Number of bytes of data */
    int value;			/* from an input line */
    int padding;		/* to handle alignment */
    int bytes_per_line;		/* per scanline of data */
    int width, height;		/* dimensions of bitmap */
    char **dimArr, **valueArr;
    char **elemArr;
    int numDim, numValues, numElem;
    unsigned char *sourceArr;	/* working variable */
    register unsigned char *sourcePtr;
    enum Formats format;
    register int i;		/*  */
    int result = TCL_ERROR;
    register char *p;

    elemArr = dimArr = valueArr = NULL;
    sourceArr = NULL;
    if (Tcl_SplitList(interp, dataList, &numElem, &elemArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (numElem != 2) {
	Tcl_AppendResult(interp, "wrong # of bitmap data components: ",
	    "should be \"dimList bitsList\"", (char *)NULL);
	goto cleanup;
    }
    if (Tcl_SplitList(interp, elemArr[0], &numDim, &dimArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (numDim != 2) {
	Tcl_AppendResult(interp, "wrong # of bitmap dimensions: ",
	    "should be \"width height\"", (char *)NULL);
	goto cleanup;
    }
    if ((Tcl_GetInt(interp, dimArr[0], &width) != TCL_OK) ||
	(Tcl_GetInt(interp, dimArr[1], &height) != TCL_OK)) {
	goto cleanup;
    }
    if ((width < 1) || (height < 1)) {
	Tcl_AppendResult(interp, "invalid bitmap dimensions", (char *)NULL);
	goto cleanup;
    }
    /* Convert commas to blank spaces */

    for (p = elemArr[1]; *p != '\0'; p++) {
	if (*p == ',') {
	    *p = ' ';
	}
    }
    if (Tcl_SplitList(interp, elemArr[1], &numValues, &valueArr) != TCL_OK) {
	goto cleanup;
    }
    bytes_per_line = (width + 7) / 8;
    arraySize = bytes_per_line * height;

    if (numValues == arraySize) {
	format = V11;
    } else if (numValues == (arraySize / 2)) {
	format = V10;
    } else {
	Tcl_AppendResult(interp, "wrong # data values in bitmap", (char *)NULL);
	goto cleanup;
    }
    padding = 0;
    if (format == V10) {
	padding = ((width % 16) && ((width % 16) < 9));
	if (padding) {
	    bytes_per_line = (width + 7) / 8 + padding;
	    arraySize = bytes_per_line * height;
	}
    }
    sourceArr = (unsigned char *)calloc(1, (unsigned int)arraySize);
    if (sourceArr == NULL) {
	Tcl_AppendResult(interp, "can't allocate memory for bitmap: ",
	    Tcl_PosixError(interp), (char *)NULL);
	goto cleanup;
    }
    sourcePtr = sourceArr;
    for (i = 0; i < numValues; i++) {
	if (GetHexValue(interp, valueArr[i], &value) != TCL_OK) {
	    goto cleanup;
	}
	*sourcePtr++ = value;
	if (format == V10) {
	    if ((!padding) || (((i * 2) + 2) % bytes_per_line)) {
		*sourcePtr++ = value >> 8;
	    }
	}
    }
    if (Tk_DefineBitmap(interp, nameId,
	    (char *)sourceArr, width, height) == TCL_OK) {
	result = TCL_OK;
    }
  cleanup:
    if (elemArr != NULL) {
	free((char *)elemArr);
    }
    if (dimArr != NULL) {
	free((char *)dimArr);
    }
    if (valueArr != NULL) {
	free((char *)valueArr);
    }
    if ((result != TCL_OK) && (sourceArr != NULL)) {
	free((char *)sourceArr);
    }
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * PutHexValue --
 *
 *	Converts the unsigned character value into a two character
 *	hexadecimal string.  A separator is also added, which may
 *	either a newline or space according the the number of bytes
 *	already output.
 *
 * Results:
 *	Always TCL_OK.
 *
 *--------------------------------------------------------------
 */
static int
PutHexValue(interp, value, numBytes)
    Tcl_Interp *interp;
    int value;
    int numBytes;
{
    char *separator;
    char string[10];

#define BYTES_PER_OUTPUT_LINE 24
    separator = (numBytes % BYTES_PER_OUTPUT_LINE) ? " " : "\n    ";
    sprintf(string, "%02x", value);
    Tcl_AppendResult(interp, separator, string, 0);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ShowBitmap --
 *
 *	Returns a list of hex values corresponding to the source
 *	bits of the bitmap given.
 *
 * Results:
 *	None.
 *
 *--------------------------------------------------------------
 */
static void
ShowBitmap(interp, tkwin, bitmap, width, height)
    Tcl_Interp *interp;		/* Interpreter to report results to */
    Tk_Window tkwin;		/* Main window of interpreter */
    Pixmap bitmap;		/* Bitmap to be queried */
    unsigned int width, height;	/* Dimensions of the bitmap */
{
    int value, bitMask;
    register int x, y;
    int count;
    XImage *image;

    /* Convert bitmap to an image */

    image = XGetImage(Tk_Display(tkwin), bitmap, 0, 0, width, height, 1L,
	XYPixmap);

    /*
     * The slow but robust brute force method of converting the image:
     */

    value = 0;
    bitMask = 1;
    count = 0;
    for (y = 0; y < height; y++) {
	for (x = 0; x < width; /*empty*/ ) {
	    if (XGetPixel(image, x, y)) {
		value |= bitMask;
	    }
	    bitMask <<= 1;
	    if (!(++x & 7)) {
		PutHexValue(interp, value, count);
		count++;
		value = 0, bitMask = 1;
	    }
	}
	if (x & 7) {
	    PutHexValue(interp, value, count);
	    count++;
	    value = 0, bitMask = 1;
	}
    }
    XDestroyImage(image);
}

/*
 *--------------------------------------------------------------
 *
 * BitmapCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to bitmaps managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
BitmapCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window of interpreter */
    Tcl_Interp *interp;		/* Interpreter to report results to */
    int argc;
    char **argv;
{
    Tk_Window tkwin = (Tk_Window)clientData;
    int length;
    unsigned int width, height;
    Pixmap bitmap;		/*  */
    Tk_Uid nameId;
    char string[200];
    char c;

    /* First time through initialize the ascii->hex translation table */
    if (!initialized) {
	InitHexTable();
	initialized = 1;
    }
    if (argc < 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " option name ?args?\"", (char *)NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    nameId = Tk_GetUid(argv[2]);
    length = strlen(argv[1]);
    bitmap = None;
    if ((c == 'd') && (length > 1) &&
	(strncmp(argv[1], "define", length) == 0)) {
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" define name data\"", (char *)NULL);
	    return TCL_ERROR;
	}
	return (DefineBitmap(interp, nameId, argv[3]));
    }
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " option name\"", (char *)NULL);
	return TCL_ERROR;
    }
    if ((c == 'w') && (strncmp(argv[1], "width", length) == 0)) {
	bitmap = Tk_GetBitmap(interp, tkwin, nameId);
	Tcl_ResetResult(interp);
	if (bitmap != None) {
	    Tk_SizeOfBitmap(Tk_Display(tkwin), bitmap, &width, &height);
	    sprintf(string, "%d", width);
	    Tcl_AppendResult(interp, string, (char *)NULL);
	}
    } else if ((c == 'h') && (strncmp(argv[1], "height", length) == 0)) {
	bitmap = Tk_GetBitmap(interp, tkwin, nameId);
	Tcl_ResetResult(interp);
	if (bitmap != None) {
	    Tk_SizeOfBitmap(Tk_Display(tkwin), bitmap, &width, &height);
	    sprintf(string, "%d", height);
	    Tcl_AppendResult(interp, string, (char *)NULL);
	}
    } else if ((c == 'b') && (strncmp(argv[1], "bits", length) == 0)) {
	bitmap = Tk_GetBitmap(interp, tkwin, nameId);
	Tcl_ResetResult(interp);
	if (bitmap != None)  {
	    Tk_SizeOfBitmap(Tk_Display(tkwin), bitmap, &width, &height);
	    ShowBitmap(interp, tkwin, bitmap, width, height);
	}
    } else if ((c == 'd') && (length > 1) &&
	(strncmp(argv[1], "data", length) == 0)) {
	bitmap = Tk_GetBitmap(interp, tkwin, nameId);
	Tcl_ResetResult(interp);
	if (bitmap != None) {
	    Tk_SizeOfBitmap(Tk_Display(tkwin), bitmap, &width, &height);
	    sprintf(string, "%d %d", width, height);
	    Tcl_AppendResult(interp, "  {", string, "} {", (char *)NULL);
	    ShowBitmap(interp, tkwin, bitmap, width, height);
	    Tcl_AppendResult(interp, "\n  }", (char *)NULL);
	}
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1], "\": should be ",
	    " define, width, height, or bits", (char *)NULL);
	return TCL_ERROR;
    }
    if (bitmap != None) {
	Tk_FreeBitmap(Tk_Display(tkwin), bitmap);
    }	
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Blt_BitmapInit --
 *
 *	This procedure is invoked to initialize the bitmap command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Adds the command to the interpreter and sets an array variable
 *	which its version number.
 *
 *--------------------------------------------------------------
 */

void
Blt_BitmapInit(interp)
    Tcl_Interp *interp;
{
    Tk_Window tkwin;

    if (Blt_FindCmd(interp, "blt_bitmap", (ClientData *)NULL) == TCL_OK) {
	return;			/* Command already exists */
    }
    tkwin = Tk_MainWindow(interp);
    if (tkwin == NULL) {
	return;			/* Not a Tk application */
    }
    Tcl_SetVar2(interp, "blt_versions", "blt_bitmap", BITMAP_VERSION,
	TCL_GLOBAL_ONLY);
    Tcl_CreateCommand(interp, "blt_bitmap", BitmapCmd, (ClientData)tkwin,
	(Tcl_CmdDeleteProc *)NULL);

    /* Define the BLT logo bitmaps */

    Tk_DefineBitmap(interp, Tk_GetUid("BLT"), blt_bits, blt_width, blt_height);
    Tk_DefineBitmap(interp, Tk_GetUid("bigBLT"), bigblt_bits, bigblt_width, 
	bigblt_height);
}
