/****************************************************************************
**
*W  string.c                    GAP source                     Frank Lübeck,
*W                                            Frank Celler & Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions which mainly deal with strings.
**
**  A *string* is a  list that  has no  holes, and  whose  elements  are  all
**  characters.  For the full definition of strings see chapter  "Strings" in
**  the {\GAP} manual.  Read also "More about Strings" about the  string flag
**  and the compact representation of strings.
**
**  A list  that  is  known to  be a  string is  represented by a bag of type
**  'T_STRING', which has the following format:
**
**      +--------+----+----+- - - -+----+----+
**      |length  |1st |2nd |       |last|null|
**      |as UInt |char|char|       |char|char|
**      +--------+----+----+- - - -+----+----+
**
**  Each entry is a  single character (of C type 'unsigned char').   The last
**  entry  in  the  bag is the  null  character  ('\0'),  which terminates  C
**  strings.  We add this null character although the length is stored in the
**  object. This allows to use C routines with  strings  directly  with  null 
**  character free strings (e.g., filenames). 
**
**  Note that a list represented by a bag of type  'T_PLIST' or 'T_SET' might
**  still be a string.  It is just that the kernel does not know this.
**
**  This package consists of three parts.
**  
**  The first part consists of the macros 'NEW_STRING', 'CHARS_STRING' (or
**  'CSTR_STRING'),  'GET_LEN_STRING', 'SET_LEN_STRING', 'GET_ELM_STRING',
**  'SET_ELM_STRING'  and  'C_NEW_STRING'.  These and  the functions below
**  use the detailed knowledge about the respresentation of strings.
**  
**  The second part  consists  of  the  functions  'LenString',  'ElmString',
**  'ElmsStrings', 'AssString',  'AsssString', PlainString', 'IsDenseString',
**  and 'IsPossString'.  They are the functions requried by the generic lists
**  package.  Using these functions the other  parts of the {\GAP} kernel can
**  access and  modify strings  without actually  being aware  that they  are
**  dealing with a string.
**
**  The third part consists  of the functions 'PrintString', which is  called
**  by 'FunPrint', and 'IsString', which test whether an arbitrary list  is a
** string, and if so converts it into the above format.  
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */
#include        "code.h"                /* coder                           */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */

#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */
#include        "tls.h"                 /* thread-local storage            */

#include        <assert.h>


/****************************************************************************
**

*F * * * * * * * * * * * * * * character functions  * * * * * * * * * * * * *
*/

/****************************************************************************
**


*V  ObjsChar[<chr>] . . . . . . . . . . . . . . . . table of character values
**
**  'ObjsChar' contains all the character values.  That way we do not need to
**  allocate new bags for new characters.
*/
libGAP_Obj libGAP_ObjsChar [256];


/****************************************************************************
**
*F  TypeChar( <chr> ) . . . . . . . . . . . . . . . type of a character value
**
**  'TypeChar' returns the type of the character <chr>.
**
**  'TypeChar' is the function in 'TypeObjFuncs' for character values.
*/
libGAP_Obj libGAP_TYPE_CHAR;

libGAP_Obj libGAP_TypeChar (
    libGAP_Obj                 chr )
{
    return libGAP_TYPE_CHAR;
}


/****************************************************************************
**
*F  EqChar( <charL>, <charR> )  . . . . . . . . . . .  compare two characters
**
**  'EqChar'  returns 'true'  if the two  characters <charL>  and <charR> are
**  equal, and 'false' otherwise.
*/
libGAP_Int libGAP_EqChar (
    libGAP_Obj                 charL,
    libGAP_Obj                 charR )
{
    return (*(libGAP_UChar*)libGAP_ADDR_OBJ(charL) == *(libGAP_UChar*)libGAP_ADDR_OBJ(charR));
}


/****************************************************************************
**
*F  LtChar( <charL>, <charR> )  . . . . . . . . . . .  compare two characters
**
**  'LtChar' returns  'true' if the    character <charL>  is less than    the
**  character <charR>, and 'false' otherwise.
*/
libGAP_Int libGAP_LtChar (
    libGAP_Obj                 charL,
    libGAP_Obj                 charR )
{
    return (*(libGAP_UChar*)libGAP_ADDR_OBJ(charL) < *(libGAP_UChar*)libGAP_ADDR_OBJ(charR));
}


/****************************************************************************
**
*F  PrintChar( <chr> )  . . . . . . . . . . . . . . . . . . print a character
**
**  'PrChar' prints the character <chr>.
*/
void libGAP_PrintChar (
    libGAP_Obj                 val )
{
    libGAP_UChar               chr;

    chr = *(libGAP_UChar*)libGAP_ADDR_OBJ(val);
    if      ( chr == '\n'  )  libGAP_Pr("'\\n'",0L,0L);
    else if ( chr == '\t'  )  libGAP_Pr("'\\t'",0L,0L);
    else if ( chr == '\r'  )  libGAP_Pr("'\\r'",0L,0L);
    else if ( chr == '\b'  )  libGAP_Pr("'\\b'",0L,0L);
    else if ( chr == '\01' )  libGAP_Pr("'\\>'",0L,0L);
    else if ( chr == '\02' )  libGAP_Pr("'\\<'",0L,0L);
    else if ( chr == '\03' )  libGAP_Pr("'\\c'",0L,0L);
    else if ( chr == '\''  )  libGAP_Pr("'\\''",0L,0L);
    else if ( chr == '\\'  )  libGAP_Pr("'\\\\'",0L,0L);
    /* print every non-printable on non-ASCII character in three digit
     * notation  */
    /*   old version (changed by FL)
    else if ( chr == '\0'  )  Pr("'\\0'",0L,0L);
    else if ( chr <  8     )  Pr("'\\0%d'",(Int)(chr&7),0L);
    else if ( chr <  32    )  Pr("'\\0%d%d'",(Int)(chr/8),(Int)(chr&7));*/
    else if ( chr < 32 || chr > 126 ) {
        libGAP_Pr("'\\%d%d", (libGAP_Int)((chr & 192) >> 6), (libGAP_Int)((chr & 56) >> 3));
        libGAP_Pr("%d'", (libGAP_Int)(chr&7), 0L);
    }
    else                      libGAP_Pr("'%c'",(libGAP_Int)chr,0L);
}


/****************************************************************************
**
*F  SaveChar( <char> )  . . . . . . . . . . . . . . . . . .  save a character
**
*/
void libGAP_SaveChar ( libGAP_Obj c )
{
    libGAP_SaveUInt1( *(libGAP_UChar *)libGAP_ADDR_OBJ(c));
}


/****************************************************************************
**
*F  LoadChar( <char> )  . . . . . . . . . . . . . . . . . .  load a character
**
*/
void libGAP_LoadChar( libGAP_Obj c )
{
    *(libGAP_UChar *)libGAP_ADDR_OBJ(c) = libGAP_LoadUInt1();
}



/****************************************************************************
**

*F * * * * * * * * * * * * * * GAP level functions  * * * * * * * * * * * * *
*/


/****************************************************************************
**
*F  FuncEmptyString( <self>, <len> ) . . . . . . . empty string with space
*
* Returns an empty string, but with space for len characters preallocated.
*
*/
libGAP_Obj    libGAP_FuncEmptyString( libGAP_Obj self, libGAP_Obj len )
{
    libGAP_Obj                 new;
    while ( ! libGAP_IS_INTOBJ(len) ) {
        len = libGAP_ErrorReturnObj(
            "<len> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(len), 0L,
            "you can replace <len> via 'return <len>;'" );
    }

    new = libGAP_NEW_STRING(libGAP_INT_INTOBJ(len));
    libGAP_SET_LEN_STRING(new, 0);
    return new;
}

/****************************************************************************
**
*F  FuncShrinkAllocationString( <self>, <str> )  . . give back unneeded memory
*
*  Shrinks the bag of <str> to minimal possible size (possibly converts to 
*  compact representation).
*
*/
libGAP_Obj   libGAP_FuncShrinkAllocationString( libGAP_Obj self, libGAP_Obj str )
{
    while (! libGAP_IsStringConv(str)) {
       str = libGAP_ErrorReturnObj(
           "<str> must be a string, not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(str), 0L,
           "you can replace <str> via 'return <str>;'" );
    }
    libGAP_SHRINK_STRING(str);
    return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  FuncCHAR_INT( <self>, <int> ) . . . . . . . . . . . . . . char by integer
*/
libGAP_Obj libGAP_FuncCHAR_INT (
    libGAP_Obj             self,
    libGAP_Obj             val )
{
    libGAP_Int             chr;

    /* get and check the integer value                                     */
again:
    while ( ! libGAP_IS_INTOBJ(val) ) {
        val = libGAP_ErrorReturnObj(
            "<val> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
            "you can replace <val> via 'return <val>;'" );
    }
    chr = libGAP_INT_INTOBJ(val);
    if ( 255 < chr || chr < 0 ) {
        val = libGAP_ErrorReturnObj(
            "<val> must be an integer between 0 and 255",
            0L, 0L, "you can replace <val> via 'return <val>;'" );
        goto again;
    }

    /* return the character                                                */
    return libGAP_ObjsChar[chr];
}


/****************************************************************************
**
*F  FuncINT_CHAR( <self>, <char> )  . . . . . . . . . . . . . integer by char
*/
libGAP_Obj libGAP_FuncINT_CHAR (
    libGAP_Obj             self,
    libGAP_Obj             val )
{
    /* get and check the character                                         */
    while ( libGAP_TNUM_OBJ(val) != libGAP_T_CHAR ) {
        val = libGAP_ErrorReturnObj(
            "<val> must be a character (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
            "you can replace <val> via 'return <val>;'" );
    }

    /* return the character                                                */
    return libGAP_INTOBJ_INT(*(libGAP_UChar*)libGAP_ADDR_OBJ(val));
}

/****************************************************************************
**
*F  FuncCHAR_SINT( <self>, <int> ) .. . . . . . . . . . char by signed integer
*/
libGAP_Obj libGAP_FuncCHAR_SINT (
    libGAP_Obj             self,
    libGAP_Obj             val )
{
  libGAP_Int             chr;

  /* get and check the integer value                                     */
agains:
  while ( ! libGAP_IS_INTOBJ(val) ) {
      val = libGAP_ErrorReturnObj(
	  "<val> must be an integer (not a %s)",
	  (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
	  "you can replace <val> via 'return <val>;'" );
  }
  chr = libGAP_INT_INTOBJ(val);
  if ( 127 < chr || chr < -128 ) {
      val = libGAP_ErrorReturnObj(
	  "<val> must be an integer between -128 and 127",
	  0L, 0L, "you can replace <val> via 'return <val>;'" );
      goto agains;
  }

    /* return the character                                                */
    return libGAP_ObjsChar[libGAP_CHAR_SINT(chr)];
}


/****************************************************************************
**
*F  FuncSINT_CHAR( <self>, <char> ) . . . . . . . . .  signed integer by char
*/
libGAP_Obj libGAP_FuncSINT_CHAR (
    libGAP_Obj             self,
    libGAP_Obj             val )
{
  /* get and check the character                                         */
  while ( libGAP_TNUM_OBJ(val) != libGAP_T_CHAR ) {
      val = libGAP_ErrorReturnObj(
	  "<val> must be a character (not a %s)",
	  (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
	  "you can replace <val> via 'return <val>;'" );
  }

  /* return the character                                                */
  return libGAP_INTOBJ_INT(libGAP_SINT_CHAR(*(libGAP_UChar*)libGAP_ADDR_OBJ(val)));
}

/****************************************************************************
**
*F  FuncSINTLIST_STRING( <self>, <string> ) signed integer list by string
*/
libGAP_Obj libGAP_SINTCHARS[256];
libGAP_Obj libGAP_INTCHARS[256];
libGAP_Obj libGAP_FuncINTLIST_STRING (
    libGAP_Obj             self,
    libGAP_Obj             val,
    libGAP_Obj             sign )
{
  libGAP_UInt l,i;
  libGAP_Obj n, *addr, *ints;
  libGAP_UInt1 *p;

  /* test whether val is a string, convert to compact rep if necessary */
  while (! libGAP_IsStringConv(val)) {
     val = libGAP_ErrorReturnObj(
         "<val> must be a string, not a %s)",
         (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
         "you can replace <val> via 'return <val>;'" );
  }

  /* initialize before first use */
  if ( libGAP_SINTCHARS[0] == (libGAP_Obj) 0 )
     for (i=0; i<256; i++) {
       libGAP_SINTCHARS[i] = libGAP_INTOBJ_INT(libGAP_SINT_CHAR(i));
       libGAP_INTCHARS[i] = libGAP_INTOBJ_INT((libGAP_UInt1)i);
     }
       

  l=libGAP_GET_LEN_STRING(val);
  n=libGAP_NEW_PLIST(libGAP_T_PLIST,l);
  libGAP_SET_LEN_PLIST(n,l);
  p=libGAP_CHARS_STRING(val);
  addr=libGAP_ADDR_OBJ(n);
  /* signed or unsigned ? */
  if (sign == libGAP_INTOBJ_INT(1L)) 
    ints = libGAP_INTCHARS;
  else
    ints = libGAP_SINTCHARS;
  for (i=1; i<=l; i++) {
    addr[i] = ints[p[i-1]];
  }

  libGAP_CHANGED_BAG(n);
  return n;
}

libGAP_Obj libGAP_FuncSINTLIST_STRING (
    libGAP_Obj             self,
    libGAP_Obj             val )
{
  return libGAP_FuncINTLIST_STRING ( self, val, libGAP_INTOBJ_INT(-1L) );
}

/****************************************************************************
**
*F  FuncSTRING_SINTLIST( <self>, <string> ) string by signed integer list
*/
libGAP_Obj libGAP_FuncSTRING_SINTLIST (
    libGAP_Obj             self,
    libGAP_Obj             val )
{
  libGAP_UInt l,i;
  libGAP_Int low, inc;
  libGAP_Obj n;
  libGAP_UInt1 *p;

  /* there should be a test here, but how do I check cheaply for list of
   * integers ? */

  /* general code */
  if (! libGAP_IS_RANGE(val) ) {
    if (! libGAP_IS_PLIST(val)) {
       val = libGAP_ErrorReturnObj(
           "<val> must be a plain list or range, not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
           "you can replace <val> via 'return <val>;'" );
    }
       
    l=libGAP_LEN_PLIST(val);
    n=libGAP_NEW_STRING(l);
    p=libGAP_CHARS_STRING(n);
    for (i=1;i<=l;i++) {
      *p++=libGAP_CHAR_SINT(libGAP_INT_INTOBJ(libGAP_ELM_PLIST(val,i)));
    }
  }
  else {
    l=libGAP_GET_LEN_RANGE(val);
    low=libGAP_GET_LOW_RANGE(val);
    inc=libGAP_GET_INC_RANGE(val);
    n=libGAP_NEW_STRING(l);
    p=libGAP_CHARS_STRING(n);
    for (i=1;i<=l;i++) {
      *p++=libGAP_CHAR_SINT(low);
      low=low+inc;
    }

  }

  libGAP_CHANGED_BAG(n);
  return n;
}

/****************************************************************************
**
*F  FuncREVNEG_STRING( <self>, <string> ) string by signed integer list
*/
libGAP_Obj libGAP_FuncREVNEG_STRING (
    libGAP_Obj             self,
    libGAP_Obj             val )
{
  libGAP_UInt l,i,j;
  libGAP_Obj n;
  libGAP_UInt1 *p,*q;

  /* test whether val is a string, convert to compact rep if necessary */
  while (! libGAP_IsStringConv(val)) {
     val = libGAP_ErrorReturnObj(
         "<val> must be a string, not a %s)",
         (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
         "you can replace <val> via 'return <val>;'" );
  }

  l=libGAP_GET_LEN_STRING(val);
  n=libGAP_NEW_STRING(l);
  p=libGAP_CHARS_STRING(val);
  q=libGAP_CHARS_STRING(n);
  j=l-1;
  for (i=1;i<=l;i++) {
    /* *q++=CHAR_SINT(-SINT_CHAR(p[j])); */
    *q++=-p[j];
    j--;
  }

  libGAP_CHANGED_BAG(n);
  return n;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * * * string functions * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  NEW_STRING( <len> )  . . . returns new string with length <len>, first
**  character and "first behind last" set to zero
**
*/
libGAP_Obj libGAP_NEW_STRING ( libGAP_Int len )
{
  libGAP_Obj res;
  if (len < 0)
       libGAP_ErrorQuit(
           "NEW_STRING: Cannot create string of negative length %d",
           (libGAP_Int)len, 0L);
  res = libGAP_NewBag( libGAP_T_STRING, libGAP_SIZEBAG_STRINGLEN(len)  ); 
  libGAP_SET_LEN_STRING(res, len);
  /* it may be sometimes useful to have trailing zero characters */
  libGAP_CHARS_STRING(res)[0] = '\0';
  libGAP_CHARS_STRING(res)[len] = '\0';
  return res;
}

/****************************************************************************
**
*F  GrowString(<list>,<len>) . . . . . .  make sure a string is large enough
**
**  returns the new length, but doesn't set SET_LEN_STRING.
*/
libGAP_Int             libGAP_GrowString (
    libGAP_Obj                 list,
    libGAP_UInt                need )
{
    libGAP_UInt                len;            /* new physical length             */
    libGAP_UInt                good;           /* good new physical length        */

    /* find out how large the data area  should become                     */
    good = 5 * (libGAP_GET_LEN_STRING(list)+3) / 4 + 1;

    /* but maybe we need more                                              */
    if ( need < good ) { len = good; }
    else               { len = need; }

    /* resize the bag                                                      */
    libGAP_ResizeBag( list, libGAP_SIZEBAG_STRINGLEN(len) );

    /* return the new maximal length                                       */
    return (libGAP_Int) len;
}

/****************************************************************************
**
*F  TypeString(<list>)  . . . . . . . . . . . . . . . . . .  type of a string
**
**  'TypeString' returns the type of the string <list>.
**
**  'TypeString' is the function in 'TypeObjFuncs' for strings.
*/
static libGAP_Obj libGAP_TYPES_STRING;


libGAP_Obj libGAP_TypeString (
    libGAP_Obj                 list )
{
    return libGAP_ELM_PLIST(libGAP_TYPES_STRING, libGAP_TNUM_OBJ(list) - libGAP_T_STRING + 1);
}



/****************************************************************************
**

*F * * * * * * * * * * * * * * copy functions * * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  CopyString( <list>, <mut> ) . . . . . . . . . . . . . . . . copy a string
**
**  'CopyString' returns a structural (deep) copy of the string <list>, i.e.,
**  a recursive copy that preserves the structure.
**
**  If <list> has not  yet  been copied, it makes   a copy, leaves  a forward
**  pointer to the copy in  the first entry of  the string, where the size of
**  the string usually resides,  and copies  all the  entries.  If  the plain
**  list  has already  been copied, it   returns the value of the  forwarding
**  pointer.
**
**  'CopyString' is the function in 'CopyObjFuncs' for strings.
**
**  'CleanString' removes the mark and the forwarding pointer from the string
**  <list>.
**
**  'CleanString' is the function in 'CleanObjFuncs' for strings.
*/
libGAP_Obj libGAP_CopyString (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* handle of the copy, result      */

    /* just return immutable objects                                       */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        return list;
    }

    /* make object for  copy                                               */
    if ( mut ) {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(list), libGAP_SIZE_OBJ(list) );
    }
    else {
        copy = libGAP_NewBag( libGAP_IMMUTABLE_TNUM( libGAP_TNUM_OBJ(list) ), libGAP_SIZE_OBJ(list) );
    }
    libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(list)[0];

    /* leave a forwarding pointer                                          */
    libGAP_ADDR_OBJ(list)[0] = copy;
    libGAP_CHANGED_BAG( list );

    /* now it is copied                                                    */
    libGAP_MARK_LIST( list, libGAP_COPYING );

    /* copy the subvalues                                                  */
    memcpy((void*)(libGAP_ADDR_OBJ(copy)+1), (void*)(libGAP_ADDR_OBJ(list)+1), 
           ((libGAP_SIZE_OBJ(copy)+sizeof(libGAP_Obj)-1)/sizeof(libGAP_Obj)-1) * sizeof(libGAP_Obj));

    /* return the copy                                                     */
    return copy;
}

/****************************************************************************
**
*F  CopyStringCopy( <list>, <mut> ) . . . . . . . . . .  copy a copied string
*/
libGAP_Obj libGAP_CopyStringCopy (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{
    return libGAP_ADDR_OBJ(list)[0];
}


/****************************************************************************
**
*F  CleanString( <list> ) . . . . . . . . . . . . . . . . . clean up a string
*/
void libGAP_CleanString (
    libGAP_Obj                 list )
{
}


/****************************************************************************
**
*F  CleanStringCopy( <list> ) . . . . . . . . . . .  clean up a copied string
*/
void libGAP_CleanStringCopy (
    libGAP_Obj                 list )
{
    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(list)[0] = libGAP_ADDR_OBJ( libGAP_ADDR_OBJ(list)[0] )[0];

    /* now it is cleaned                                                   */
    libGAP_UNMARK_LIST( list, libGAP_COPYING );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * list functions * * * * * * * * * * * * * * * *
*/

/****************************************************************************
**


*F  PrintString(<list>) . . . . . . . . . . . . . . . . . . .  print a string
**
**  'PrintString' prints the string with the handle <list>.
**
**  No linebreaks are  allowed, if one must be inserted  anyhow, it must
**  be escaped by a backslash '\', which is done in 'Pr'.
**
**  The kernel  buffer PrStrBuf  is used to  protect Pr  against garbage
**  collections caused by  printing to string streams,  which might move
**  the body of list.
**
**  The output uses octal number notation for non-ascii or non-printable
**  characters. The function can be used  to print *any* string in a way
**  which can be read in by GAP afterwards.
**
*/

void libGAP_PrintString (
    libGAP_Obj                 list )
{
  char PrStrBuf[10007];	/* 7 for a \c\123 at the end */
  libGAP_UInt scanout, n;
  libGAP_UInt1 c;
  libGAP_UInt len = libGAP_GET_LEN_STRING(list);
  libGAP_UInt off = 0;
  libGAP_Pr("\"", 0L, 0L);
  while (off < len)
    {
      scanout = 0;
      do 
	{
	  c = libGAP_CHARS_STRING(list)[off++];
	  switch (c)
	    {
            case '\\':
              PrStrBuf[scanout++] = '\\';
              PrStrBuf[scanout++] = '\\';
              break;
            case '\"':
              PrStrBuf[scanout++] = '\\';
              PrStrBuf[scanout++] = '\"';
              break;
	    case '\n':
	      PrStrBuf[scanout++] = '\\';
	      PrStrBuf[scanout++] = 'n';
	      break;
	    case '\t':
	      PrStrBuf[scanout++] = '\\';
	      PrStrBuf[scanout++] = 't';
	      break;
	    case '\r':
	      PrStrBuf[scanout++] = '\\';
	      PrStrBuf[scanout++] = 'r';
	      break;
	    case '\b':
	      PrStrBuf[scanout++] = '\\';
	      PrStrBuf[scanout++] = 'b';
	      break;
	    case '\01':
	      PrStrBuf[scanout++] = '\\';
	      PrStrBuf[scanout++] = '>';
	      break;
	    case '\02':
	      PrStrBuf[scanout++] = '\\';
	      PrStrBuf[scanout++] = '<';
	      break;
	    case '\03':
	      PrStrBuf[scanout++] = '\\';
	      PrStrBuf[scanout++] = 'c';
	      break;
	    default:
              if (c < 32 || c>126) {
                 PrStrBuf[scanout++] = '\\';
                 n = c / 64;
                 c = c - n*64;
                 PrStrBuf[scanout++] = n + '0';
                 n = c / 8;
                 c = c - n*8;
                 PrStrBuf[scanout++] = n + '0';
                 PrStrBuf[scanout++] = c + '0'; 
              }
              else
                 PrStrBuf[scanout++] = c;
	    }
	}
      while (off < len && scanout < 10000);
      PrStrBuf[scanout++] = '\0';
      libGAP_Pr( "%s", (libGAP_Int)PrStrBuf, 0L );
    }
  libGAP_Pr( "\"", 0L, 0L );
}
/****************************************************************************
**
*F  PrintString1(<list>)  . . . . . . . . . . . .  print a string for 'Print'
**
**  'PrintString1' prints the string  constant  in  the  format  used  by the
**  'Print' and 'PrintTo' function.
*/


void libGAP_PrintString1 (
    libGAP_Obj                 list )
{
  char PrStrBuf[10007];	/* 7 for a \c\123 at the end */
  libGAP_UInt len = libGAP_GET_LEN_STRING(list);
  libGAP_UInt scanout, off = 0;
  libGAP_UInt1  *p;

  while (off < len)    {
    for (p = libGAP_CHARS_STRING(list), scanout=0; 
         p[off] && off<len && scanout<10000; 
         off++, scanout++) {
      PrStrBuf[scanout] = p[off];
    }
    PrStrBuf[scanout] = '\0';
    libGAP_Pr( "%s", (libGAP_Int)PrStrBuf, 0L );
    for (; off<len && libGAP_CHARS_STRING(list)[off]==0; off++) {
      libGAP_Pr("%c", 0L, 0L);
    }
  }
}


/****************************************************************************
**
*F  EqString(<listL>,<listR>) . . . . . . . .  test whether strings are equal
**
**  'EqString'  returns  'true' if the  two  strings <listL>  and <listR> are
**  equal and 'false' otherwise.
*/
libGAP_Int libGAP_EqString (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
  libGAP_UInt lL, lR, i;
  libGAP_UInt1 *pL, *pR;
  lL = libGAP_GET_LEN_STRING(listL);
  lR = libGAP_GET_LEN_STRING(listR);
  if (lR != lL) return 0;
  pL = libGAP_CHARS_STRING(listL);
  pR = libGAP_CHARS_STRING(listR);
  for (i=0; i<lL && pL[i] == pR[i]; i++);
  return (i == lL);
}


/****************************************************************************
**
*F  LtString(<listL>,<listR>) .  test whether one string is less than another
**
**  'LtString' returns 'true' if  the string <listL> is  less than the string
**  <listR> and 'false' otherwise.
*/
libGAP_Int libGAP_LtString (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
  libGAP_UInt lL, lR, i;
  libGAP_UInt1 *pL, *pR;
  lL = libGAP_GET_LEN_STRING(listL);
  lR = libGAP_GET_LEN_STRING(listR);
  pL = libGAP_CHARS_STRING(listL);
  pR = libGAP_CHARS_STRING(listR);
  for (i=0; i<lL && i<lR && pL[i] == pR[i]; i++);
  if (i == lL) return (lR > lL);
  if (i == lR) return 0;
  return pL[i] < pR[i];
}


/****************************************************************************
**
*F  LenString(<list>) . . . . . . . . . . . . . . . . . .  length of a string
**
**  'LenString' returns the length of the string <list> as a C integer.
**
**  'LenString' is the function in 'LenListFuncs' for strings.
*/
libGAP_Int libGAP_LenString (
    libGAP_Obj                 list )
{
    return libGAP_GET_LEN_STRING( list );
}


/****************************************************************************
**
*F  IsbString(<list>,<pos>) . . . . . . . . . test for an element of a string
*F  IsbvString(<list>,<pos>)  . . . . . . . . test for an element of a string
**
**  'IsbString' returns 1 if the string <list> contains
**  a character at the position <pos> and 0 otherwise.
**  It can rely on <pos> being a positive integer.
**
**  'IsbvString' does the same thing as 'IsbString', but it can 
**  also rely on <pos> not being larger than the length of <list>.
**
**  'IsbString'  is the function in 'IsbListFuncs'  for strings.
**  'IsbvString' is the function in 'IsbvListFuncs' for strings.
*/
libGAP_Int libGAP_IsbString (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    /* since strings are dense, this must only test for the length         */
    return (pos <= libGAP_GET_LEN_STRING(list));
}

libGAP_Int libGAP_IsbvString (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    /* since strings are dense, this can only return 1                     */
    return 1L;
}


/****************************************************************************
**
*F  Elm0String(<list>,<pos>)  . . . . . . . . . select an element of a string
*F  Elm0vString(<list>,<pos>) . . . . . . . . . select an element of a string
**
**  'Elm0String' returns the element at the position <pos> of the string
**  <list>, or returns 0 if <list> has no assigned object at <pos>.
**  It can rely on <pos> being a positive integer.
**
**  'Elm0vString' does the same thing as 'Elm0String', but it can
**  also rely on <pos> not being larger than the length of <list>.
**
**  'Elm0String'  is the function on 'Elm0ListFuncs'  for strings.
**  'Elm0vString' is the function in 'Elm0vListFuncs' for strings.
*/
libGAP_Obj libGAP_Elm0String (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    if ( pos <= libGAP_GET_LEN_STRING( list ) ) {
        return libGAP_GET_ELM_STRING( list, pos );
    }
    else {
        return 0;
    }
}

libGAP_Obj libGAP_Elm0vString (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return libGAP_GET_ELM_STRING( list, pos );
}


/****************************************************************************
**
*F  ElmString(<list>,<pos>) . . . . . . . . . . select an element of a string
*F  ElmvString(<list>,<pos>)  . . . . . . . . . select an element of a string
**
**  'ElmString' returns the element at the position <pos> of the string
**  <list>, or signals an error if <list> has no assigned object at <pos>.
**  It can rely on <pos> being a positive integer.
**
**  'ElmvString' does the same thing as 'ElmString', but it can
**  also rely on <pos> not being larger than the length of <list>.
**
**  'ElmwString' does the same thing as 'ElmString', but it can
**  also rely on <list> having an assigned object at <pos>.
**
**  'ElmString'  is the function in 'ElmListFuncs'  for strings.
**  'ElmfString' is the function in 'ElmfListFuncs' for strings.
**  'ElmwString' is the function in 'ElmwListFuncs' for strings.
*/
libGAP_Obj libGAP_ElmString (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    /* check the position                                                  */
    if ( libGAP_GET_LEN_STRING( list ) < pos ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            (libGAP_Int)pos, 0L,
            "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, pos );
    }

    /* return the selected element                                         */
    return libGAP_GET_ELM_STRING( list, pos );
}

#define libGAP_ElmvString      libGAP_Elm0vString

#define libGAP_ElmwString      libGAP_Elm0vString


/****************************************************************************
**
*F  ElmsString(<list>,<poss>) . . . . . . . .  select a sublist from a string
**
**  'ElmsString' returns a new list containing the  elements at the positions
**  given   in  the  list   <poss> from   the  string   <list>.   It  is  the
**  responsibility of the called to ensure that  <poss> is dense and contains
**  only positive integers.  An error is signalled if an element of <poss> is
**  larger than the length of <list>.
**
**  'ElmsString' is the function in 'ElmsListFuncs' for strings.
*/
libGAP_Obj libGAP_ElmsString (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elms;         /* selected sublist, result        */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Char                elm;            /* one element from <list>         */
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt1               *p, *pn;        /* loop pointer                    */

    /* general code                                                        */
    if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <list>                                        */
        lenList = libGAP_GET_LEN_STRING( list );

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST( poss );

        /* make the result list                                            */
        elms = libGAP_NEW_STRING( lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
            pos = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, i ) );
            if ( lenList < pos ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* select the element                                          */
            elm = libGAP_CHARS_STRING(list)[pos-1];

            /* assign the element into <elms>                              */
            libGAP_CHARS_STRING(elms)[i-1] = elm;

        }

    }

    /* special code for ranges                                             */
    else {

        /* get the length of <list>                                        */
        lenList = libGAP_GET_LEN_STRING( list );

        /* get the length of <positions>, the first elements, and the inc. */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* check that no <position> is larger than 'LEN_LIST(<list>)'      */
        if ( lenList < pos ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }
        if ( lenList < pos + (lenPoss-1) * inc ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)(pos + (lenPoss-1) * inc), 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }

        /* make the result list                                            */
        elms = libGAP_NEW_STRING( lenPoss );

        /* loop over the entries of <positions> and select                 */
	p = libGAP_CHARS_STRING(list);
	pn = libGAP_CHARS_STRING(elms);
        for ( i = 1; i <= lenPoss; i++, pos += inc ) {
	  pn[i-1] = p[pos-1];
        }

    }

    /* return the result                                                   */
    return elms;
}


/****************************************************************************
**
*F  AssString(<list>,<pos>,<val>) . . . . . . . . . . . .  assign to a string
**
**  'AssString' assigns the value <val> to the  string <list> at the position
**  <pos>.   It is the responsibility  of the caller to  ensure that <pos> is
**  positive, and that <val> is not 0.
**
**  'AssString' is the function in 'AssListFuncs' for strings.
**
**  'AssString' keeps <list> in string representation if possible.
**  
*/
void libGAP_AssString (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
  libGAP_UInt len = libGAP_GET_LEN_STRING(list);

  if (libGAP_TNUM_OBJ(val) != libGAP_T_CHAR || pos > len+1) {
    /* convert the range into a plain list                                 */
    libGAP_PLAIN_LIST(list);
    libGAP_CLEAR_FILTS_LIST(list);

    /* resize the list if necessary                                        */
    if ( len < pos ) {
      libGAP_GROW_PLIST( list, pos );
      libGAP_SET_LEN_PLIST( list, pos );
    }

    /* now perform the assignment and return the assigned value            */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );
  }
  else {
    libGAP_CLEAR_FILTS_LIST(list);

    /* resize the list if necessary                                        */
    if ( len < pos ) {
      libGAP_GROW_STRING( list, pos );
      libGAP_SET_LEN_STRING( list, pos );
      libGAP_CHARS_STRING(list)[pos] = (libGAP_UInt1)0;
    }

    /* now perform the assignment and return the assigned value            */
    libGAP_SET_ELM_STRING( list, pos, val ); 
    /*    CHARS_STRING(list)[pos-1] = *((UInt1*)ADDR_OBJ(val)); */
    libGAP_CHANGED_BAG( list );
  }
}    

void libGAP_AssStringImm (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignment: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**
*F  AsssString(<list>,<poss>,<vals>)  . . assign several elements to a string
**
**  'AsssString' assignes the  values from the  list <vals> at the  positions
**  given in the list <poss> to the string  <list>.  It is the responsibility
**  of the caller to ensure that  <poss> is dense  and contains only positive
**  integers, that <poss> and <vals> have the same length, and that <vals> is
**  dense.
**
**  'AsssString' is the function in 'AsssListFuncs' for strings.
**
**  'AsssString' simply delegates to AssString. Note that the ordering of 
**  <poss> can be important if <list> should stay in string representation.
**   
*/
void libGAP_AsssString (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 vals )
{
  libGAP_Int i, len = libGAP_LEN_LIST(poss);
  for (i = 1; i <= len; i++) {
    libGAP_ASS_LIST(list, libGAP_INT_INTOBJ(libGAP_ELM_LIST(poss, i)), libGAP_ELM_LIST(vals, i));
  }
}

void libGAP_AsssStringImm (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignments: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**
*F  IsDenseString(<list>) . . . . . . .  dense list test function for strings
**
**  'IsDenseString' returns 1, since every string is dense.
**
**  'IsDenseString' is the function in 'IsDenseListFuncs' for strings.
*/
libGAP_Int libGAP_IsDenseString (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**
*F  IsHomogString(<list>) . . . .  homogeneous list test function for strings
**
**  'IsHomogString' returns  1 if  the string  <list>  is homogeneous.  Every
**  nonempty string is homogeneous.
**
**  'IsHomogString' is the function in 'IsHomogListFuncs' for strings.
*/
libGAP_Int libGAP_IsHomogString (
    libGAP_Obj                 list )
{
    return (0 < libGAP_GET_LEN_STRING(list));
}


/****************************************************************************
**
*F  IsSSortString(<list>) . . . . . . . strictly sorted list test for strings
**
**  'IsSSortString'  returns 1 if the string  <list> is strictly sorted and 0
**  otherwise.
**
**  'IsSSortString' is the function in 'IsSSortListFuncs' for strings.
*/
libGAP_Int libGAP_IsSSortString (
    libGAP_Obj                 list )
{
    libGAP_Int                 len;
    libGAP_Int                 i;
    libGAP_UInt1 *             ptr;

    /* test whether the string is strictly sorted                          */
    len = libGAP_GET_LEN_STRING( list );
    ptr = (libGAP_UInt1*) libGAP_CHARS_STRING(list);
    for ( i = 1; i < len; i++ ) {
        if ( ! (ptr[i-1] < ptr[i]) )
            break;
    }

    /* retype according to the outcome                                     */
    libGAP_SET_FILT_LIST( list, (len <= i) ? libGAP_FN_IS_SSORT : libGAP_FN_IS_NSORT );
    return (len <= i);
}

libGAP_Int libGAP_IsSSortStringNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int libGAP_IsSSortStringYes (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**
*F  IsPossString(<list>)  . . . . .  positions list test function for strings
**
**  'IsPossString' returns 0, since every string contains no integers.
**
**  'IsPossString' is the function in 'TabIsPossList' for strings.
*/
libGAP_Int libGAP_IsPossString (
    libGAP_Obj                 list )
{
    return libGAP_GET_LEN_STRING( list ) == 0;
}


/****************************************************************************
**
*F  PosString(<list>,<val>,<pos>) . . . .  position of an element in a string
**
**  'PosString' returns the position of the  value <val> in the string <list>
**  after the first position <start> as a C integer.   0 is returned if <val>
**  is not in the list.
**
**  'PosString' is the function in 'PosListFuncs' for strings.
*/ 
libGAP_Obj libGAP_PosString (
    libGAP_Obj                 list,
    libGAP_Obj                 val,
    libGAP_Obj                 start )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt1               valc;        /* C characters                    */
    libGAP_UInt1               *p;             /* pointer to chars of <list>      */
    libGAP_UInt                istart;

    /* if the starting position is too big to be a small int
       then there can't be anything to find */
    if (!libGAP_IS_INTOBJ(start))
      return libGAP_Fail;

    istart = libGAP_INT_INTOBJ(start);
    
    /* get the length of <list>                                            */
    lenList = libGAP_GET_LEN_STRING( list );

    /* a string contains only characters */
    if (libGAP_TNUM_OBJ(val) != libGAP_T_CHAR) return libGAP_Fail;
    
    /* val as C character   */
    valc = *(libGAP_UInt1*)libGAP_ADDR_OBJ(val);

    /* search entries in <list>                                     */
    p = libGAP_CHARS_STRING(list);
    for ( i = istart; i < lenList && p[i] != valc; i++ );

    /* return the position (0 if <val> was not found)                      */
    return (lenList <= i ? libGAP_Fail : libGAP_INTOBJ_INT(i+1));
}


/****************************************************************************
**
*F  PlainString(<list>) . . . . . . . . . .  convert a string to a plain list
**
**  'PlainString' converts the string <list> to a plain list.  Not much work.
**
**  'PlainString' is the function in 'PlainListFuncs' for strings.
*/
void libGAP_PlainString (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* logical length of the string    */
    libGAP_Obj                 tmp;            /* handle of the list              */
    libGAP_Int                 i;              /* loop variable                   */

    /* find the length and allocate a temporary copy                       */
    lenList = libGAP_GET_LEN_STRING( list );
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, lenList );
    libGAP_SET_LEN_PLIST( tmp, lenList );

    /* copy the characters                                                 */
    for ( i = 1; i <= lenList; i++ ) {
        libGAP_SET_ELM_PLIST( tmp, i, libGAP_GET_ELM_STRING( list, i ) );
    }

    /* change size and type of the string and copy back                    */
    libGAP_ResizeBag( list, libGAP_SIZE_OBJ(tmp) );
    libGAP_RetypeBag( list, libGAP_TNUM_OBJ(tmp) );

    /*    Why not just copying the data area ? (FL)
	  SET_LEN_PLIST( list, lenList );
	  for ( i = 1; i <= lenList; i++ ) {
	  SET_ELM_PLIST( list, i, ELM_PLIST( tmp, i ) );
	  CHANGED_BAG( list );
	  }
    */
    memcpy((void*)libGAP_ADDR_OBJ(list), (void*)libGAP_ADDR_OBJ(tmp), libGAP_SIZE_OBJ(tmp));
    libGAP_CHANGED_BAG(list);
}


/****************************************************************************
**

*F  IS_STRING( <obj> )  . . . . . . . . . . . . test if an object is a string
**
**  'IS_STRING' returns 1  if the object <obj>  is a string  and 0 otherwise.
**  It does not change the representation of <obj>.
*/
libGAP_Int (*libGAP_IsStringFuncs [libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj libGAP_IsStringFilt;

libGAP_Int libGAP_IsStringNot (
    libGAP_Obj                 obj )
{
    return 0;
}

libGAP_Int libGAP_IsStringYes (
    libGAP_Obj                 obj )
{
    return 1;
}

libGAP_Int libGAP_IsStringList (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;
    libGAP_Obj                 elm;
    libGAP_Int                 i;
    
    lenList = libGAP_LEN_LIST( list );
    for ( i = 1; i <= lenList; i++ ) {
        elm = libGAP_ELMV0_LIST( list, i );
        if ( elm == 0 || libGAP_TNUM_OBJ( elm ) != libGAP_T_CHAR )
            break;
    }

    return (lenList < i);
}

libGAP_Int libGAP_IsStringListHom (
    libGAP_Obj                 list )
{
    return (libGAP_TNUM_OBJ( libGAP_ELM_LIST(list,1) ) == libGAP_T_CHAR);
}

libGAP_Int libGAP_IsStringObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsStringFilt, obj ) != libGAP_False);
}


/****************************************************************************
**
*F  CopyToStringRep( <string> )  . . copy a string to the string representation
**
**  'CopyToStringRep' copies the string <string> to a new string in string
**  representation.
*/
libGAP_Obj libGAP_CopyToStringRep(
    libGAP_Obj                 string )
{
    libGAP_Int                 lenString;      /* length of the string            */
    libGAP_Obj                 elm;            /* one element of the string       */
    libGAP_Obj                 copy;           /* temporary string                */
    libGAP_Int                 i;              /* loop variable                   */

    lenString = libGAP_LEN_LIST(string);
    copy = libGAP_NEW_STRING(lenString);

    if ( libGAP_IS_STRING_REP(string) ) {
        memcpy(libGAP_ADDR_OBJ(copy), libGAP_ADDR_OBJ(string), libGAP_SIZE_OBJ(string));
        /* XXX no error checks? */
    } else {
        /* copy the string to the string representation                     */
        for ( i = 1; i <= lenString; i++ ) {
            elm = libGAP_ELMW_LIST( string, i );
            libGAP_CHARS_STRING(copy)[i-1] = *((libGAP_UChar*)libGAP_ADDR_OBJ(elm));
        } 
        libGAP_CHARS_STRING(copy)[lenString] = '\0';
    }
    libGAP_CHANGED_BAG(copy);
    return (copy);
}



/****************************************************************************
**
*F  ConvString( <string> )  . . convert a string to the string representation
**
**  'ConvString' converts the string <list> to the string representation.
*/
void libGAP_ConvString (
    libGAP_Obj                 string )
{
    libGAP_Int                 lenString;      /* length of the string            */
    libGAP_Obj                 elm;            /* one element of the string       */
    libGAP_Obj                 tmp;            /* temporary string                */
    libGAP_Int                 i;              /* loop variable                   */

    /* do nothing if the string is already in the string representation    */
    if ( libGAP_IS_STRING_REP(string) )
    {
        return;
    }


    lenString = libGAP_LEN_LIST(string);
    tmp = libGAP_NEW_STRING(lenString);

    /* copy the string to the string representation                     */
    for ( i = 1; i <= lenString; i++ ) {
        elm = libGAP_ELMW_LIST( string, i );
        libGAP_CHARS_STRING(tmp)[i-1] = *((libGAP_UChar*)libGAP_ADDR_OBJ(elm));
    }
    libGAP_CHARS_STRING(tmp)[lenString] = '\0';

    /* copy back to string  */
    libGAP_RetypeBag( string, libGAP_IS_MUTABLE_OBJ(string)?libGAP_T_STRING:libGAP_T_STRING+libGAP_IMMUTABLE );
    libGAP_ResizeBag( string, libGAP_SIZEBAG_STRINGLEN(lenString) );
    /* copy data area from tmp */
    memcpy((void*)libGAP_ADDR_OBJ(string), (void*)libGAP_ADDR_OBJ(tmp), libGAP_SIZE_OBJ(tmp));
    libGAP_CHANGED_BAG(string);
}



/****************************************************************************
**
*F  IsStringConv( <obj> ) . . . . . test if an object is a string and convert
**
**  'IsStringConv'   returns 1  if   the object <obj>  is   a  string,  and 0
**  otherwise.   If <obj> is a  string it  changes  its representation to the
**  string representation.
*/
libGAP_Obj libGAP_IsStringConvFilt;

libGAP_Int libGAP_IsStringConv (
    libGAP_Obj                 obj )
{
    libGAP_Int                 res;

    /* test whether the object is a string                                 */
    res = libGAP_IS_STRING( obj );

    /* if so, convert it to the string representation                      */
    if ( res ) {
        libGAP_ConvString( obj );
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
*F  MakeImmutableString(  <str> ) make a string immutable in place
**
*/

void libGAP_MakeImmutableString( libGAP_Obj str )
{
    libGAP_RetypeBag(str, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(str)));
}


libGAP_Obj libGAP_MakeString(libGAP_Char *cstr)
{
  libGAP_Obj result;
  libGAP_C_NEW_STRING(result, strlen(cstr), cstr);
  return result;
}

libGAP_Obj libGAP_MakeString2(libGAP_Char *cstr1, libGAP_Char *cstr2)
{
  libGAP_Obj result;
  size_t len1 = strlen(cstr1), len2 = strlen(cstr2);
  result = libGAP_NEW_STRING(len1 + len2);
  memcpy(libGAP_CSTR_STRING(result), cstr1, len1);
  memcpy(libGAP_CSTR_STRING(result)+len1, cstr2, len2);
  return result;
}

libGAP_Obj libGAP_MakeString3(libGAP_Char *cstr1, libGAP_Char *cstr2, libGAP_Char *cstr3)
{
  libGAP_Obj result;
  size_t len1 = strlen(cstr1), len2 = strlen(cstr2), len3 = strlen(cstr3);
  result = libGAP_NEW_STRING(len1 + len2 + len3);
  memcpy(libGAP_CSTR_STRING(result), cstr1, len1);
  memcpy(libGAP_CSTR_STRING(result)+len1, cstr2, len2);
  memcpy(libGAP_CSTR_STRING(result)+len1+len2, cstr3, len3);
  return result;
}

libGAP_Obj libGAP_MakeImmString(libGAP_Char *cstr)
{
  libGAP_Obj result = libGAP_MakeString(cstr);
  libGAP_MakeImmutableString(result);
  return result;
}

libGAP_Obj libGAP_MakeImmString2(libGAP_Char *cstr1, libGAP_Char *cstr2)
{
  libGAP_Obj result = libGAP_MakeString2(cstr1, cstr2);
  libGAP_MakeImmutableString(result);
  return result;
}

libGAP_Obj libGAP_MakeImmString3(libGAP_Char *cstr1, libGAP_Char *cstr2, libGAP_Char *cstr3)
{
  libGAP_Obj result = libGAP_MakeString3(cstr1, cstr2, cstr3);
  libGAP_MakeImmutableString(result);
  return result;
}

libGAP_Obj libGAP_ConvImmString(libGAP_Obj str)
{
  libGAP_Obj result;
  if (!str || !libGAP_IsStringConv(str))
    return (libGAP_Obj) 0;
  if (!libGAP_IS_MUTABLE_OBJ(str))
    return str;
  libGAP_C_NEW_STRING(result, libGAP_GET_LEN_STRING(str), libGAP_CSTR_STRING(str))
  libGAP_MakeImmutableString(result);
  return result;
}



/****************************************************************************
**

*F * * * * * * * * * * * * * * GAP level functions  * * * * * * * * * * * * *
*/

/****************************************************************************
**


*F  FuncIS_STRING( <self>, <obj> )  . . . . . . . . .  test value is a string
*/
libGAP_Obj libGAP_FuncIS_STRING (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_STRING( obj ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  FuncIS_STRING_CONV( <self>, <obj> ) . . . . . . . . . . check and convert
*/
libGAP_Obj libGAP_FuncIS_STRING_CONV (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    /* return 'true' if <obj> is a string and 'false' otherwise            */
    return (libGAP_IsStringConv(obj) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  FuncCONV_STRING( <self>, <string> ) . . . . . . . . convert to string rep
*/
libGAP_Obj libGAP_FuncCONV_STRING (
    libGAP_Obj                 self,
    libGAP_Obj                 string )
{
    /* check whether <string> is a string                                  */
    if ( ! libGAP_IS_STRING( string ) ) {
        string = libGAP_ErrorReturnObj(
            "ConvString: <string> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(string), 0L,
            "you can replace <string> via 'return <string>;'" );
        return libGAP_FuncCONV_STRING( self, string );
    }

    /* convert to the string representation                                */
    libGAP_ConvString( string );

    /* return nothing                                                      */
    return 0;
}


/****************************************************************************
**
*F  FuncIS_STRING_REP( <self>, <obj> )  . . . . test if value is a string rep
*/
libGAP_Obj libGAP_IsStringRepFilt;

libGAP_Obj libGAP_FuncIS_STRING_REP (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_STRING_REP( obj ) ? libGAP_True : libGAP_False);
}

/****************************************************************************
**
*F  FuncCOPY_TO_STRING_REP( <self>, <obj> ) . copy a string into string rep
*/
libGAP_Obj libGAP_FuncCOPY_TO_STRING_REP (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    /* check whether <obj> is a string                                  */
    if (!libGAP_IS_STRING(obj)) {
        obj = libGAP_ErrorReturnObj(
            "ConvString: <string> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L,
            "you can replace <string> via 'return <string>;'" );
        return libGAP_FuncCOPY_TO_STRING_REP( self, obj );
    }
    return libGAP_CopyToStringRep(obj);
}

/****************************************************************************
**
*F  FuncPOSITION_SUBSTRING( <self>,  <string>, <substr>, <off> ) .  position of
**  substring
**  
**  <str> and <substr> must be strings  and <off> an integer. The position
**  of  first  character of substring   in string,  search  starting  from
**  <off>+1, is  returned if such  a substring exists. Otherwise `fail' is
**  returned.
*/
libGAP_Obj libGAP_FuncPOSITION_SUBSTRING( 
			   libGAP_Obj                  self,
			   libGAP_Obj                  string,
			   libGAP_Obj                  substr,
			   libGAP_Obj                  off )
{
  libGAP_Int    ipos, i, j, lens, lenss, max;
  libGAP_UInt1  *s, *ss, c;

  /* check whether <string> is a string                                  */
  while ( ! libGAP_IsStringConv( string ) ) {
    string = libGAP_ErrorReturnObj(
	     "POSITION_SUBSTRING: <string> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(string), 0L,
	     "you can replace <string> via 'return <string>;'" );
  }
  
  /* check whether <substr> is a string                        */
  while ( ! libGAP_IsStringConv( substr ) ) {
    substr = libGAP_ErrorReturnObj(
	  "POSITION_SUBSTRING: <substr> must be a string (not a %s)",
	  (libGAP_Int)libGAP_TNAM_OBJ(substr), 0L,
	  "you can replace <substr> via 'return <substr>;'" );
  }

  /* check wether <off> is a non-negative integer  */
  while ( ! libGAP_IS_INTOBJ(off) || (ipos = libGAP_INT_INTOBJ(off)) < 0 ) {
    off = libGAP_ErrorReturnObj(
          "POSITION_SUBSTRING: <off> must be a non-negative integer (not a %s)",
          (libGAP_Int)libGAP_TNAM_OBJ(off), 0L,
          "you can replace <off> via 'return <off>;'");
  }

  /* special case for the empty string */
  lenss = libGAP_GET_LEN_STRING(substr);
  if ( lenss == 0 ) {
    return libGAP_INTOBJ_INT(ipos + 1);
  }

  lens = libGAP_GET_LEN_STRING(string);
  max = lens - lenss + 1;
  s = libGAP_CHARS_STRING(string);
  ss = libGAP_CHARS_STRING(substr);
  
  c = ss[0];
  for (i = ipos; i < max; i++) {
    if (c == s[i]) {
      for (j = 1; j < lenss; j++) {
        if (! (s[i+j] == ss[j]))
          break;
      }
      if (j == lenss) 
        return libGAP_INTOBJ_INT(i+1);
    }
  }
  return libGAP_Fail;
}

/****************************************************************************
**
*F  FuncNormalizeWhitespace( <self>, <string> ) . . . . . normalize white
**  space in place
**    
**  Whitespace  characters are  " \r\t\n".  Leading and  trailing whitespace  in
**  string  is  removed. Intermediate  sequences  of  whitespace characters  are
**  substituted by a single space.
**  
*/ 
libGAP_Obj libGAP_FuncNormalizeWhitespace (
			      libGAP_Obj     self,
			      libGAP_Obj     string )
{
  libGAP_UInt1  *s, c;
  libGAP_Int i, j, len, white;

  /* check whether <string> is a string                                  */
  if ( ! libGAP_IsStringConv( string ) ) {
    string = libGAP_ErrorReturnObj(
	     "NormalizeWhitespace: <string> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(string), 0L,
	     "you can replace <string> via 'return <string>;'" );
    return libGAP_FuncNormalizeWhitespace( self, string );
  }
  
  len = libGAP_GET_LEN_STRING(string);
  s = libGAP_CHARS_STRING(string);
  i = -1;
  white = 1;
  for (j = 0; j < len; j++) {
    c = s[j];
    if (c == ' ' || c == '\n' || c == '\t' || c == '\r') {
      if (! white) {
	i++;
	s[i] = ' ';
	white = 1;
      }
    }
    else {
      i++;
      s[i] = c;
      white = 0;
    }
  }
  if (white && i > -1) 
    i--;
  s[i+1] = '\0';
  libGAP_SET_LEN_STRING(string, i+1);
 
  /* to make it useful as C-string */
  libGAP_CHARS_STRING(string)[i+1] = (libGAP_UInt1)0;

  return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncRemoveCharacters( <self>, <string>, <rem> ) . . . . . delete characters
**  from <rem> in <string> in place 
**    
*/ 

libGAP_Obj libGAP_FuncRemoveCharacters (
			      libGAP_Obj     self,
			      libGAP_Obj     string,
                              libGAP_Obj     rem     )
{
  libGAP_UInt1  *s;
  libGAP_Int i, j, len;
  libGAP_UInt1 REMCHARLIST[257] = {0};

  /* check whether <string> is a string                                  */
  if ( ! libGAP_IsStringConv( string ) ) {
    string = libGAP_ErrorReturnObj(
	     "RemoveCharacters: first argument <string> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(string), 0L,
	     "you can replace <string> via 'return <string>;'" );
    return libGAP_FuncRemoveCharacters( self, string, rem );
  }
  
  /* check whether <rem> is a string                                  */
  if ( ! libGAP_IsStringConv( rem ) ) {
    rem = libGAP_ErrorReturnObj(
	     "RemoveCharacters: second argument <rem> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(rem), 0L,
	     "you can replace <rem> via 'return <rem>;'" );
    return libGAP_FuncRemoveCharacters( self, string, rem );
  }
  
  /* set REMCHARLIST by setting positions of characters in rem to 1 */
  len = libGAP_GET_LEN_STRING(rem);
  s = libGAP_CHARS_STRING(rem);
  REMCHARLIST[256] = 1;
  for(i=0; i<len; i++) REMCHARLIST[s[i]] = 1;
  
  /* now change string in place */
  len = libGAP_GET_LEN_STRING(string);
  s = libGAP_CHARS_STRING(string);
  i = -1;
  for (j = 0; j < len; j++) {
    if (REMCHARLIST[s[j]] == 0) {
      i++;
      s[i] = s[j];
    }
  }
  i++;
  s[i] = '\0';
  libGAP_SET_LEN_STRING(string, i);
  libGAP_SHRINK_STRING(string);

  return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncTranslateString( <self>, <string>, <trans> ) . . . translate characters
**  in <string> in place, <string>[i] = <trans>[<string>[i]] 
**    
*/ 
libGAP_Obj libGAP_FuncTranslateString (
			      libGAP_Obj     self,
			      libGAP_Obj     string,
                              libGAP_Obj     trans     )
{
  libGAP_UInt1  *s, *t;
  libGAP_Int j, len;

  /* check whether <string> is a string                                  */
  if ( ! libGAP_IsStringConv( string ) ) {
    string = libGAP_ErrorReturnObj(
	     "RemoveCharacters: first argument <string> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(string), 0L,
	     "you can replace <string> via 'return <string>;'" );
    return libGAP_FuncTranslateString( self, string, trans );
  }
  
  /* check whether <trans> is a string                                  */
  if ( ! libGAP_IsStringConv( trans ) ) {
    trans = libGAP_ErrorReturnObj(
	     "RemoveCharacters: second argument <trans> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(trans), 0L,
	     "you can replace <trans> via 'return <trans>;'" );
    return libGAP_FuncTranslateString( self, string, trans );
  }
 
  /* check if string has length at least 256 */
  if ( libGAP_GET_LEN_STRING( trans ) < 256 ) {
    trans = libGAP_ErrorReturnObj(
	     "RemoveCharacters: second argument <trans> must have length >= 256",
	     0L, 0L,
	     "you can replace <trans> via 'return <trans>;'" );
    return libGAP_FuncTranslateString( self, string, trans );
  }
  
  /* now change string in place */
  len = libGAP_GET_LEN_STRING(string);
  s = libGAP_CHARS_STRING(string);
  t = libGAP_CHARS_STRING(trans);
  for (j = 0; j < len; j++) {
    s[j] = t[s[j]];
  }
  
  return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncSplitString( <self>, <string>, <seps>, <wspace> ) . . . . split string
**  at characters in <seps> and <wspace>
**    
**  The difference of <seps> and <wspace> is that characters in <wspace> don't
**  separate empty strings.
*/ 
libGAP_UInt1 libGAP_SPLITSTRINGSEPS[257];
libGAP_UInt1 libGAP_SPLITSTRINGWSPACE[257];
libGAP_Obj libGAP_FuncSplitString (
			      libGAP_Obj     self,
			      libGAP_Obj     string,
                              libGAP_Obj     seps,
                              libGAP_Obj     wspace    )
{
  libGAP_UInt1  *s;
  libGAP_Int i, a, z, l, pos, len;
  libGAP_Obj res, part;

  /* check whether <string> is a string                                  */
  if ( ! libGAP_IsStringConv( string ) ) {
    string = libGAP_ErrorReturnObj(
	     "SplitString: first argument <string> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(string), 0L,
	     "you can replace <string> via 'return <string>;'" );
    return libGAP_FuncSplitString( self, string, seps, wspace );
  }
  
  /* check whether <seps> is a string                                  */
  if ( ! libGAP_IsStringConv( seps ) ) {
    seps = libGAP_ErrorReturnObj(
	     "SplitString: second argument <seps> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(seps), 0L,
	     "you can replace <seps> via 'return <seps>;'" );
    return libGAP_FuncSplitString( self, string, seps, wspace );
  }
  
  /* check whether <wspace> is a string                                  */
  if ( ! libGAP_IsStringConv( wspace ) ) {
    wspace = libGAP_ErrorReturnObj(
	     "SplitString: third argument <wspace> must be a string (not a %s)",
	     (libGAP_Int)libGAP_TNAM_OBJ(wspace), 0L,
	     "you can replace <wspace> via 'return <wspace>;'" );
    return libGAP_FuncSplitString( self, string, seps, wspace );
  }
  
  /* reset SPLITSTRINGSEPS (in case of previous error) */
  if (libGAP_SPLITSTRINGSEPS[256] != 0) {
    for(i=0; i<257; i++) libGAP_SPLITSTRINGSEPS[i] = 0;
  }
  
  /* set SPLITSTRINGSEPS by setting positions of characters in rem to 1 */
  len = libGAP_GET_LEN_STRING(seps);
  s = libGAP_CHARS_STRING(seps);
  libGAP_SPLITSTRINGSEPS[256] = 1;
  for(i=0; i<len; i++) libGAP_SPLITSTRINGSEPS[s[i]] = 1;
  
  /* reset SPLITSTRINGWSPACE (in case of previous error) */
  if (libGAP_SPLITSTRINGWSPACE[256] != 0) {
    for(i=0; i<257; i++) libGAP_SPLITSTRINGWSPACE[i] = 0;
  }
  
  /* set SPLITSTRINGWSPACE by setting positions of characters in rem to 1 */
  len = libGAP_GET_LEN_STRING(wspace);
  s = libGAP_CHARS_STRING(wspace);
  libGAP_SPLITSTRINGWSPACE[256] = 1;
  for(i=0; i<len; i++) libGAP_SPLITSTRINGWSPACE[s[i]] = 1;
 
  /* create the result (list of strings) */
  res = libGAP_NEW_PLIST(libGAP_T_PLIST, 2);
  libGAP_SET_LEN_PLIST(res, 0);
  pos = 0;

  /* now do the splitting */
  len = libGAP_GET_LEN_STRING(string);
  s = libGAP_CHARS_STRING(string);
  for (a=0, z=0; z<len; z++) {
    if (libGAP_SPLITSTRINGWSPACE[s[z]] == 1) {
      if (a<z) {
        l = z-a;
        part = libGAP_NEW_STRING(l);
        /* in case of garbage collection we need update */
        s = libGAP_CHARS_STRING(string);
        libGAP_COPY_CHARS(part, s+a, l);
        libGAP_CHARS_STRING(part)[l] = 0;
        pos++;
        libGAP_AssPlist(res, pos, part);
        s = libGAP_CHARS_STRING(string);
        a = z+1;
      }
      else {
        a = z+1;
      }
    }
    else {
      if (libGAP_SPLITSTRINGSEPS[s[z]] == 1) {
        l = z-a;
        part = libGAP_NEW_STRING(l);
        s = libGAP_CHARS_STRING(string);
        libGAP_COPY_CHARS(part, s+a, l);
        libGAP_CHARS_STRING(part)[l] = 0;
        pos++;
        libGAP_AssPlist(res, pos, part);
        s = libGAP_CHARS_STRING(string);
        a = z+1;
      }
    }
  }
  
  /* collect a trailing part */
  if (a<z) {
    /* copy until last position which is z-1 */
    l = z-a;
    part = libGAP_NEW_STRING(l);
    s = libGAP_CHARS_STRING(string);
    libGAP_COPY_CHARS(part, s+a, l);
    libGAP_CHARS_STRING(part)[l] = 0;
    pos++;
    libGAP_AssPlist(res, pos, part);
  }

  /* unset SPLITSTRINGSEPS  */
  len = libGAP_GET_LEN_STRING(seps);
  s = libGAP_CHARS_STRING(seps);
  for(i=0; i<len; i++) libGAP_SPLITSTRINGSEPS[s[i]] = 0;
  libGAP_SPLITSTRINGSEPS[256] = 0;

  /* unset SPLITSTRINGWSPACE  */
  len = libGAP_GET_LEN_STRING(wspace);
  s = libGAP_CHARS_STRING(wspace);
  for(i=0; i<len; i++) libGAP_SPLITSTRINGWSPACE[s[i]] = 0;
  libGAP_SPLITSTRINGWSPACE[256] = 0;

  return res;
}

/****************************************************************************
**
*F FuncSMALLINT_STR( <self>, <string> )
**
** Kernel function to extract parse small integers from strings. Needed before
** we can conveniently have Int working for things like parsing command line
** options
*/

libGAP_Obj libGAP_FuncSMALLINT_STR( libGAP_Obj self, libGAP_Obj string )
{
  return libGAP_INTOBJ_INT(libGAP_SyIntString(libGAP_CSTR_STRING(string)));
}

/****************************************************************************
**
*F  UnbString( <string>, <pos> ) . . . . . Unbind function for strings
**  
**  This is to avoid unpacking of string to plain list when <pos> is 
**  larger or equal to the length of <string>.
**  
*/
void libGAP_UnbString (
  libGAP_Obj     string,
  libGAP_Int     pos )
{
        libGAP_Int len;
        len = libGAP_GET_LEN_STRING(string);
	
        /* only do something special if last character is to be, and can be, 
         * unbound */
        if (len < pos) return;
        if (len != pos) {
                libGAP_UnbListDefault(string, pos);
                return;
        }
        if (! libGAP_IS_MUTABLE_OBJ(string)) {
                libGAP_UnbPlistImm(string, pos);
                return;
        }
        /* maybe the string becomes sorted */
        libGAP_CLEAR_FILTS_LIST(string);
        libGAP_CHARS_STRING(string)[pos-1] = (libGAP_UInt1)0;
        libGAP_SET_LEN_STRING(string, len-1);
} 
            

  
/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * * */

/****************************************************************************
**

*V  BagNames  . . . . . . . . . . . . . . . . . . . . . . . list of bag names
*/
static libGAP_StructBagNames libGAP_BagNames[] = {
  { libGAP_T_CHAR,                           "character"                      },
  { libGAP_T_STRING,                         "list (string)"                  },
  { libGAP_T_STRING              +libGAP_IMMUTABLE, "list (string,imm)"              },
  { libGAP_T_STRING      +libGAP_COPYING,           "list (string,copied)"           },
  { libGAP_T_STRING      +libGAP_COPYING+libGAP_IMMUTABLE, "list (string,imm,copied)"       },
  { libGAP_T_STRING_SSORT,                   "list (string,ssort)"            },
  { libGAP_T_STRING_SSORT        +libGAP_IMMUTABLE, "list (string,ssort,imm)"        },
  { libGAP_T_STRING_SSORT+libGAP_COPYING,           "list (string,ssort,copied)"     },
  { libGAP_T_STRING_SSORT+libGAP_COPYING+libGAP_IMMUTABLE, "list (string,ssort,imm,copied)" },
  { libGAP_T_STRING_NSORT,                   "list (string,nsort)"            },
  { libGAP_T_STRING_NSORT        +libGAP_IMMUTABLE, "list (string,nsort,imm)"        },
  { libGAP_T_STRING_NSORT+libGAP_COPYING,           "list (string,nsort,copied)"     },
  { libGAP_T_STRING_NSORT+libGAP_COPYING+libGAP_IMMUTABLE, "list (string,nsort,imm,copied)" },
  { -1,                               ""                               }
};


/****************************************************************************
**
*V  ClearFiltsTab . . . . . . . . . . . . . . . . . . . .  clear filter tnums
*/
static libGAP_Int libGAP_ClearFiltsTab [] = {
    libGAP_T_STRING,                 libGAP_T_STRING,
    libGAP_T_STRING      +libGAP_IMMUTABLE, libGAP_T_STRING+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT,           libGAP_T_STRING,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE, libGAP_T_STRING+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT,           libGAP_T_STRING,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE, libGAP_T_STRING+libGAP_IMMUTABLE,
    -1,                       -1
};


/****************************************************************************
**
*V  HasFiltTab  . . . . . . . . . . . . . . . . . . . . .  tester filter tnum
*/
static libGAP_Int libGAP_HasFiltTab [] = {

    /* mutable string                                                      */
    libGAP_T_STRING,                  libGAP_FN_IS_MUTABLE, 1,
    libGAP_T_STRING,                  libGAP_FN_IS_EMPTY,   0,
    libGAP_T_STRING,                  libGAP_FN_IS_DENSE,   1,
    libGAP_T_STRING,                  libGAP_FN_IS_NDENSE,  0,
    libGAP_T_STRING,                  libGAP_FN_IS_HOMOG,   1,
    libGAP_T_STRING,                  libGAP_FN_IS_NHOMOG,  0,
    libGAP_T_STRING,                  libGAP_FN_IS_TABLE,   0,
    libGAP_T_STRING,                  libGAP_FN_IS_RECT,    0,
    libGAP_T_STRING,                  libGAP_FN_IS_SSORT,   0,
    libGAP_T_STRING,                  libGAP_FN_IS_NSORT,   0,

    /* immutable string                                                    */
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, 0,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   0,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   1,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  0,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   1,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  0,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   0,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,    0,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   0,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   0,

    /* ssort mutable string                                                */
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_MUTABLE, 1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_EMPTY,   0,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_DENSE,   1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NDENSE,  0,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_HOMOG,   1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NHOMOG,  0,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_TABLE,   0,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_RECT,   0,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_SSORT,   1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NSORT,   0,

    /* ssort immutable string                                              */
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, 0,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   0,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   1,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  0,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   1,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  0,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   0,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,   0,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   1,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   0,

    /* nsort mutable string                                                */
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_MUTABLE, 1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_EMPTY,   0,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_DENSE,   1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NDENSE,  0,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_HOMOG,   1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NHOMOG,  0,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_TABLE,   0,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_RECT,   0,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_SSORT,   0,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NSORT,   1,

    /* nsort immutable string                                              */
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, 0,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   0,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  0,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  0,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   0,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,   0,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   0,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   1,

    -1,                        -1,            -1
};


/****************************************************************************
**
*V  SetFiltTab  . . . . . . . . . . . . . . . . . . . . .  setter filter tnum
*/
static libGAP_Int libGAP_SetFiltTab [] = {

    /* mutable string                                                      */
    libGAP_T_STRING,                  libGAP_FN_IS_MUTABLE, libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_EMPTY,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING,                  libGAP_FN_IS_DENSE,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_STRING,                  libGAP_FN_IS_HOMOG,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_STRING,                  libGAP_FN_IS_TABLE,   -1,
    libGAP_T_STRING,                  libGAP_FN_IS_RECT,   -1,
    libGAP_T_STRING,                  libGAP_FN_IS_SSORT,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING,                  libGAP_FN_IS_NSORT,   libGAP_T_STRING_NSORT,

    /* immutable string                                                    */
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, libGAP_T_STRING,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   -1,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,   -1,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,

    /* ssort mutable string                                                */
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_TABLE,   -1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_RECT,   -1,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NSORT,   -1,

    /* ssort immutable string                                              */
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   -1,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,   -1,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   -1,

    /* nsort mutable string                                                */
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_TABLE,   -1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_RECT,   -1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_SSORT,   -1,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_STRING_NSORT,

    /* nsort immutable string                                              */
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   -1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,   -1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   -1,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,

    -1,                        -1,            -1

};


/****************************************************************************
**
*V  ResetFiltTab  . . . . . . . . . . . . . . . . . . .  unsetter filter tnum
*/
static libGAP_Int libGAP_ResetFiltTab [] = {

    /* mutable string                                                      */
    libGAP_T_STRING,                  libGAP_FN_IS_MUTABLE, libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING,                  libGAP_FN_IS_EMPTY,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_DENSE,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_NDENSE,  libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_HOMOG,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_NHOMOG,  libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_TABLE,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_RECT,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_SSORT,   libGAP_T_STRING,
    libGAP_T_STRING,                  libGAP_FN_IS_NSORT,   libGAP_T_STRING,

    /* immutable string                                                    */
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,    libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING      +libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   libGAP_T_STRING      +libGAP_IMMUTABLE,

    /* ssort mutable string                                                */
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_STRING_SSORT,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_STRING,
    libGAP_T_STRING_SSORT,            libGAP_FN_IS_NSORT,   libGAP_T_STRING_SSORT,

    /* ssort immutable string                                              */
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   libGAP_T_STRING      +libGAP_IMMUTABLE,
    libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   libGAP_T_STRING_SSORT+libGAP_IMMUTABLE,

    /* nsort mutable string                                                */
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_RECT,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_SSORT,   libGAP_T_STRING_NSORT,
    libGAP_T_STRING_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_STRING,

    /* nsort immutable string                                              */
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_MUTABLE, libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_EMPTY,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_DENSE,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NDENSE,  libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_HOMOG,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NHOMOG,  libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_TABLE,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_RECT,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_SSORT,   libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,
    libGAP_T_STRING_NSORT+libGAP_IMMUTABLE,  libGAP_FN_IS_NSORT,   libGAP_T_STRING      +libGAP_IMMUTABLE,

    -1,                        -1,            -1

};


/****************************************************************************
**
*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_STRING", "obj", &libGAP_IsStringFilt,
      libGAP_FuncIS_STRING, "src/string.c:IS_STRING" },

    { "IS_STRING_REP", "obj", &libGAP_IsStringRepFilt,
      libGAP_FuncIS_STRING_REP, "src/lists.c:IS_STRING_REP" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "IS_STRING_CONV", 1, "string",
      libGAP_FuncIS_STRING_CONV, "src/string.c:IS_STRING_CONV" },

    { "CONV_STRING", 1, "string",
      libGAP_FuncCONV_STRING, "src/string.c:CONV_STRING" },

    { "COPY_TO_STRING_REP", 1, "string",
      libGAP_FuncCOPY_TO_STRING_REP, "src/string.c:COPY_TO_STRING_REP" },

    { "CHAR_INT", 1, "integer",
      libGAP_FuncCHAR_INT, "src/string.c:CHAR_INT" },

    { "INT_CHAR", 1, "char",
      libGAP_FuncINT_CHAR, "src/string.c:INT_CHAR" },

    { "CHAR_SINT", 1, "integer",
      libGAP_FuncCHAR_SINT, "src/string.c:CHAR_SINT" },

    { "SINT_CHAR", 1, "char",
      libGAP_FuncSINT_CHAR, "src/string.c:SINT_CHAR" },

    { "STRING_SINTLIST", 1, "list",
      libGAP_FuncSTRING_SINTLIST, "src/string.c:STRING_SINTLIST" },

    { "INTLIST_STRING", 2, "string, sign",
      libGAP_FuncINTLIST_STRING, "src/string.c:INTLIST_STRING" },

    { "SINTLIST_STRING", 1, "string",
      libGAP_FuncSINTLIST_STRING, "src/string.c:SINTLIST_STRING" },

    { "EmptyString", 1, "len",
      libGAP_FuncEmptyString, "src/string.c:FuncEmptyString" },
    
    { "ShrinkAllocationString", 1, "str",
      libGAP_FuncShrinkAllocationString, "src/string.c:FuncShrinkAllocationString" },
    
    { "REVNEG_STRING", 1, "string",
      libGAP_FuncREVNEG_STRING, "src/string.c:REVNEG_STRING" },

    { "POSITION_SUBSTRING", 3, "string, substr, off",
      libGAP_FuncPOSITION_SUBSTRING, "src/string.c:POSITION_SUBSTRING" },

    { "NormalizeWhitespace", 1, "string",
      libGAP_FuncNormalizeWhitespace, "src/string.c:NormalizeWhitespace" },

    { "REMOVE_CHARACTERS", 2, "string, rem",
      libGAP_FuncRemoveCharacters, "src/string.c:RemoveCharacters" },

    { "TranslateString", 2, "string, trans",
      libGAP_FuncTranslateString, "src/string.c:TranslateString" },

    { "SplitStringInternal", 3, "string, seps, wspace",
      libGAP_FuncSplitString, "src/string.c:SplitStringInternal" },

    { "SMALLINT_STR", 1, "string",
      libGAP_FuncSMALLINT_STR, "src/string.c:SMALLINT_STR" },

    { 0 }

};


/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Char libGAP_CharCookie[256][21];

static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                t1;
    libGAP_UInt                t2;
    libGAP_Int                 i, j;
    const libGAP_Char *        cookie_base = "src/string.c:Char";

    /* check dependencies                                                  */
    libGAP_RequireModule( libGAP_module, "lists", 403600000UL );

    /* GASMAN marking functions and GASMAN names                           */
    libGAP_InitBagNamesFromTable( libGAP_BagNames );

    libGAP_InitMarkFuncBags( libGAP_T_CHAR , libGAP_MarkNoSubBags );
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1 += 2 ) {
        libGAP_InitMarkFuncBags( t1                     , libGAP_MarkNoSubBags );
        libGAP_InitMarkFuncBags( t1          +libGAP_IMMUTABLE , libGAP_MarkNoSubBags );
        libGAP_InitMarkFuncBags( t1 +libGAP_COPYING            , libGAP_MarkNoSubBags );
        libGAP_InitMarkFuncBags( t1 +libGAP_COPYING +libGAP_IMMUTABLE , libGAP_MarkNoSubBags );
    }
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1 += 2 ) {
      libGAP_MakeBagTypePublic( t1 + libGAP_IMMUTABLE );
    }

    libGAP_MakeBagTypePublic(libGAP_T_CHAR);

    /* make all the character constants once and for all                   */
    for ( i = 0; i < 256; i++ ) {
        for (j = 0; j < 17; j++ ) {
            libGAP_CharCookie[i][j] = cookie_base[j];
        }
        libGAP_CharCookie[i][j++] = '0' + i/100;
        libGAP_CharCookie[i][j++] = '0' + (i % 100)/10;
        libGAP_CharCookie[i][j++] = '0' + i % 10;
        libGAP_CharCookie[i][j++] = '\0';
        libGAP_InitGlobalBag( &libGAP_ObjsChar[i], &(libGAP_CharCookie[i][0]) );
    }

    /* install the type method                                             */
    libGAP_ImportGVarFromLibrary( "TYPE_CHAR", &libGAP_TYPE_CHAR );
    libGAP_TypeObjFuncs[ libGAP_T_CHAR ] = libGAP_TypeChar;

    /* install the type method                                             */
    libGAP_ImportGVarFromLibrary( "TYPES_STRING", &libGAP_TYPES_STRING );
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1 += 2 ) {
        libGAP_TypeObjFuncs[ t1            ] = libGAP_TypeString;
        libGAP_TypeObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_TypeString;
    }

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* initialise list tables                                              */
    libGAP_InitClearFiltsTNumsFromTable   ( libGAP_ClearFiltsTab );
    libGAP_InitHasFiltListTNumsFromTable  ( libGAP_HasFiltTab    );
    libGAP_InitSetFiltListTNumsFromTable  ( libGAP_SetFiltTab    );
    libGAP_InitResetFiltListTNumsFromTable( libGAP_ResetFiltTab  );

    /* Install the saving function                                         */
    libGAP_SaveObjFuncs[ libGAP_T_CHAR ] = libGAP_SaveChar;
    libGAP_LoadObjFuncs[ libGAP_T_CHAR ] = libGAP_LoadChar;

    /* install the character functions                                     */
    libGAP_PrintObjFuncs[ libGAP_T_CHAR ] = libGAP_PrintChar;
    libGAP_EqFuncs[ libGAP_T_CHAR ][ libGAP_T_CHAR ] = libGAP_EqChar;
    libGAP_LtFuncs[ libGAP_T_CHAR ][ libGAP_T_CHAR ] = libGAP_LtChar;

    /* install the saving method                                             */
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1 += 2 ) {
        libGAP_SaveObjFuncs[ t1            ] = libGAP_SaveString;
        libGAP_SaveObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_SaveString;
        libGAP_LoadObjFuncs[ t1            ] = libGAP_LoadString;
        libGAP_LoadObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_LoadString;
    }

    /* install the copy method                                             */
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1++ ) {
        libGAP_CopyObjFuncs [ t1                     ] = libGAP_CopyString;
        libGAP_CopyObjFuncs [ t1          +libGAP_IMMUTABLE ] = libGAP_CopyString;
        libGAP_CleanObjFuncs[ t1                     ] = libGAP_CleanString;
        libGAP_CleanObjFuncs[ t1          +libGAP_IMMUTABLE ] = libGAP_CleanString;
        libGAP_CopyObjFuncs [ t1 +libGAP_COPYING            ] = libGAP_CopyStringCopy;
        libGAP_CopyObjFuncs [ t1 +libGAP_COPYING +libGAP_IMMUTABLE ] = libGAP_CopyStringCopy;
        libGAP_CleanObjFuncs[ t1 +libGAP_COPYING            ] = libGAP_CleanStringCopy;
        libGAP_CleanObjFuncs[ t1 +libGAP_COPYING +libGAP_IMMUTABLE ] = libGAP_CleanStringCopy;
    }

    /* install the print method                                            */
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1 += 2 ) {
        libGAP_PrintObjFuncs[ t1            ] = libGAP_PrintString;
        libGAP_PrintObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_PrintString;
    }

    /* install the comparison methods                                      */
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT+libGAP_IMMUTABLE; t1++ ) {
        for ( t2 = libGAP_T_STRING; t2 <= libGAP_T_STRING_SSORT+libGAP_IMMUTABLE; t2++ ) {
            libGAP_EqFuncs[ t1 ][ t2 ] = libGAP_EqString;
            libGAP_LtFuncs[ t1 ][ t2 ] = libGAP_LtString;
        }
    }

    /* install the list methods                                            */
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1 += 2 ) {
        libGAP_LenListFuncs    [ t1            ] = libGAP_LenString;
        libGAP_LenListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_LenString;
        libGAP_IsbListFuncs    [ t1            ] = libGAP_IsbString;
        libGAP_IsbListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbString;
        libGAP_IsbvListFuncs   [ t1            ] = libGAP_IsbvString;
        libGAP_IsbvListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbvString;
        libGAP_Elm0ListFuncs   [ t1            ] = libGAP_Elm0String;
        libGAP_Elm0ListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_Elm0String;
        libGAP_Elm0vListFuncs  [ t1            ] = libGAP_Elm0vString;
        libGAP_Elm0vListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_Elm0vString;
        libGAP_ElmListFuncs    [ t1            ] = libGAP_ElmString;
        libGAP_ElmListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmString;
        libGAP_ElmvListFuncs   [ t1            ] = libGAP_ElmvString;
        libGAP_ElmvListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmvString;
        libGAP_ElmwListFuncs   [ t1            ] = libGAP_ElmwString;
        libGAP_ElmwListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmwString;
        libGAP_ElmsListFuncs   [ t1            ] = libGAP_ElmsString;
        libGAP_ElmsListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmsString;
        libGAP_AssListFuncs    [ t1            ] = libGAP_AssString;
        libGAP_AssListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_AssStringImm;
        libGAP_AsssListFuncs   [ t1            ] = libGAP_AsssString;
        libGAP_AsssListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_AsssStringImm;
        libGAP_IsDenseListFuncs[ t1            ] = libGAP_IsDenseString;
        libGAP_IsDenseListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsDenseString;
        libGAP_IsHomogListFuncs[ t1            ] = libGAP_IsHomogString;
        libGAP_IsHomogListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsHomogString;
        libGAP_IsSSortListFuncs[ t1            ] = libGAP_IsSSortString;
        libGAP_IsSSortListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsSSortString;
        libGAP_IsPossListFuncs [ t1            ] = libGAP_IsPossString;
        libGAP_IsPossListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_IsPossString;
        libGAP_PosListFuncs    [ t1            ] = libGAP_PosString;
        libGAP_PosListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_PosString;
        libGAP_PlainListFuncs  [ t1            ] = libGAP_PlainString;
        libGAP_PlainListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_PlainString;
    }
    libGAP_IsSSortListFuncs[ libGAP_T_STRING_NSORT            ] = libGAP_IsSSortStringNot;
    libGAP_IsSSortListFuncs[ libGAP_T_STRING_NSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortStringNot;
    libGAP_IsSSortListFuncs[ libGAP_T_STRING_SSORT            ] = libGAP_IsSSortStringYes;
    libGAP_IsSSortListFuncs[ libGAP_T_STRING_SSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortStringYes;


    /* install the `IsString' functions                                    */
    for ( t1 = libGAP_FIRST_REAL_TNUM; t1 <= libGAP_LAST_REAL_TNUM; t1++ ) {
        libGAP_IsStringFuncs[ t1 ] = libGAP_IsStringNot;
    }

    for ( t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1++ ) {
        libGAP_IsStringFuncs[ t1 ] = libGAP_IsStringList;
    }

    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT; t1++ ) {
        libGAP_IsStringFuncs[ t1 ] = libGAP_IsStringYes;
    }

    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_IsStringFuncs[ t1 ] = libGAP_IsStringObject;
    }

    /* install the list unbind methods  */
    for ( t1 = libGAP_T_STRING; t1 <= libGAP_T_STRING_SSORT+libGAP_IMMUTABLE; t1++ ) {
           libGAP_UnbListFuncs    [ t1            ] = libGAP_UnbString;
    }

    libGAP_MakeImmutableObjFuncs[ libGAP_T_STRING       ] = libGAP_MakeImmutableString;
    libGAP_MakeImmutableObjFuncs[ libGAP_T_STRING_SSORT ] = libGAP_MakeImmutableString;
    libGAP_MakeImmutableObjFuncs[ libGAP_T_STRING_NSORT ] = libGAP_MakeImmutableString;
    

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int                 i;
    

    /* make all the character constants once and for all                   */
    for ( i = 0; i < 256; i++ ) {
        libGAP_ObjsChar[i] = libGAP_NewBag( libGAP_T_CHAR, 1L );
        *(libGAP_UChar*)libGAP_ADDR_OBJ(libGAP_ObjsChar[i]) = (libGAP_UChar)i;
    }

    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoString()  . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "string",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoString ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  string.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
