/****************************************************************************
**
*W  streams.c                   GAP source                       Frank Celler
*W                                                  & Burkhard Höfling (MAC)
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the  various read-eval-print loops and streams related
**  stuff.  The system depend part is in "sysfiles.c".
*/

#include        "system.h"              /* system dependent part           */

#include        <errno.h>
#include        <stdio.h>
#include        <string.h>              /* memcpy */

#include        <unistd.h>              /* fstat, write, read              */
#include        <sys/types.h>
#include        <dirent.h>              /* for reading a directory         */
#include        <sys/stat.h>
#if HAVE_SYS_TIME_H
#include        <sys/time.h>
#endif



#include        "sysfiles.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */
#include        "code.h"                /* coder                           */

#include        "gap.h"                 /* error handling, initialisation  */
#include        "read.h"                /* reader                          */
#include        "funcs.h"               /* functions                       */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */

#include        "streams.h"             /* streams package                 */

#include        "code.h"

#include	"tls.h"

#include        "vars.h"                /* TLS(BottomLVars) for execution contexts */


/****************************************************************************
**

*F * * * * * * * * * streams and files related functions  * * * * * * * * * *
*/

libGAP_Int libGAP_READ_COMMAND ( void )
{
    libGAP_ExecStatus    status;

    libGAP_ClearError();
    status = libGAP_ReadEvalCommand(libGAP_TLS(libGAP_BottomLVars), 0);
    if( status == libGAP_STATUS_EOF )
        return 0;

    if ( libGAP_TLS(libGAP_UserHasQuit) || libGAP_TLS(libGAP_UserHasQUIT) )
        return 0;
    
    /* handle return-value or return-void command                          */
    if ( status & (libGAP_STATUS_RETURN_VAL | libGAP_STATUS_RETURN_VOID) ) {
        libGAP_Pr( "'return' must not be used in file read-eval loop", 0L, 0L );
    }

    /* handle quit command                                 */
    else if (status == libGAP_STATUS_QUIT) {
        libGAP_TLS(libGAP_RecursionDepth) = 0;
        libGAP_TLS(libGAP_UserHasQuit) = 1;
    }
    else if (status == libGAP_STATUS_QQUIT) {
        libGAP_TLS(libGAP_UserHasQUIT) = 1;
    }
    libGAP_ClearError();

    return 1;
}

/*
 Returns a list with one or two entries. The first
 entry is set to "false" if there was any error
 executing the command, and "true" otherwise.
 The second entry, if present, is the return value of
 the command. If it not present, the command returned nothing.
*/
libGAP_Obj libGAP_FuncREAD_COMMAND_REAL ( libGAP_Obj self, libGAP_Obj stream, libGAP_Obj echo )
{
    libGAP_Int status;
    libGAP_Obj result;

    result = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST(result, 1);
    libGAP_SET_ELM_PLIST(result, 1, libGAP_False);

    /* try to open the file                                                */
    if ( ! libGAP_OpenInputStream(stream) ) {
        return result;
    }

    if (echo == libGAP_True)
      libGAP_TLS(libGAP_Input)->echo = 1;
    else
      libGAP_TLS(libGAP_Input)->echo = 0;

    status = libGAP_READ_COMMAND();
    
    libGAP_CloseInput();

    if( status == 0 ) return result;

    if (libGAP_TLS(libGAP_UserHasQUIT)) {
      libGAP_TLS(libGAP_UserHasQUIT) = 0;
      return result;
    }

    if (libGAP_TLS(libGAP_UserHasQuit)) {
      libGAP_TLS(libGAP_UserHasQuit) = 0;
    }
    
    libGAP_SET_ELM_PLIST(result, 1, libGAP_True);
    if (libGAP_TLS(libGAP_ReadEvalResult)) {
        libGAP_SET_LEN_PLIST(result, 2);
        libGAP_SET_ELM_PLIST(result, 2, libGAP_TLS(libGAP_ReadEvalResult));
    }
    return result;
}

/*
 Deprecated alternative to READ_COMMAND_REAL, kept for now to maintain
 compatibility with the few packages that use it.
 */
libGAP_Obj libGAP_FuncREAD_COMMAND ( libGAP_Obj self, libGAP_Obj stream, libGAP_Obj echo )
{
    libGAP_Obj result;
    result = libGAP_FuncREAD_COMMAND_REAL(self, stream, echo);
    return (libGAP_LEN_PLIST(result) == 2) ? libGAP_ELM_PLIST(result, 2) : libGAP_SuPeRfail;
}

/****************************************************************************
**

*F  READ()  . . . . . . . . . . . . . . . . . . . . . . .  read current input
**
**  Read the current input and close the input stream.
*/

static libGAP_UInt libGAP_LastReadValueGVar;

static libGAP_Int libGAP_READ_INNER ( libGAP_UInt UseUHQ )
{
    libGAP_ExecStatus                status;

    if (libGAP_TLS(libGAP_UserHasQuit))
      {
        libGAP_Pr("Warning: Entering READ with UserHasQuit set, this should never happen, resetting",0,0);
        libGAP_TLS(libGAP_UserHasQuit) = 0;
      }
    if (libGAP_TLS(libGAP_UserHasQUIT))
      {
        libGAP_Pr("Warning: Entering READ with UserHasQUIT set, this should never happen, resetting",0,0);
        libGAP_TLS(libGAP_UserHasQUIT) = 0;
      }
    libGAP_MakeReadWriteGVar(libGAP_LastReadValueGVar);
    libGAP_AssGVar( libGAP_LastReadValueGVar, 0);
    libGAP_MakeReadOnlyGVar(libGAP_LastReadValueGVar);
    /* now do the reading                                                  */
    while ( 1 ) {
        libGAP_ClearError();
        status = libGAP_ReadEvalCommand(libGAP_TLS(libGAP_BottomLVars), 0);
	if (libGAP_TLS(libGAP_UserHasQuit) || libGAP_TLS(libGAP_UserHasQUIT))
	  break;
        /* handle return-value or return-void command                      */
        if ( status & (libGAP_STATUS_RETURN_VAL | libGAP_STATUS_RETURN_VOID) ) {
            libGAP_Pr(
                "'return' must not be used in file read-eval loop",
                0L, 0L );
        }

        /* handle quit command or <end-of-file>                            */
        else if ( status  & (libGAP_STATUS_ERROR | libGAP_STATUS_EOF)) 
          break;
        else if (status == libGAP_STATUS_QUIT) {
          libGAP_TLS(libGAP_RecursionDepth) = 0;
          libGAP_TLS(libGAP_UserHasQuit) = 1;
          break;
        }
        else if (status == libGAP_STATUS_QQUIT) {
          libGAP_TLS(libGAP_UserHasQUIT) = 1;
          break;
        }
        if (libGAP_TLS(libGAP_ReadEvalResult))
          {
            libGAP_MakeReadWriteGVar(libGAP_LastReadValueGVar);
            libGAP_AssGVar( libGAP_LastReadValueGVar, libGAP_TLS(libGAP_ReadEvalResult));
            libGAP_MakeReadOnlyGVar(libGAP_LastReadValueGVar);
          }
        
    }


    /* close the input file again, and return 'true'                       */
    if ( ! libGAP_CloseInput() ) {
        libGAP_ErrorQuit(
            "Panic: READ cannot close input, this should not happen",
            0L, 0L );
    }
    libGAP_ClearError();

    if (!UseUHQ && libGAP_TLS(libGAP_UserHasQuit)) {
      libGAP_TLS(libGAP_UserHasQuit) = 0; /* stop recovery here */
      return 2;
    }

    return 1;
}


static libGAP_Int libGAP_READ( void ) {
  return libGAP_READ_INNER(1);
}

static libGAP_Int libGAP_READ_NORECOVERY( void ) {
  return libGAP_READ_INNER(0);
}

/****************************************************************************
**
*F  READ_AS_FUNC()  . . . . . . . . . . . . .  read current input as function
**
**  Read the current input as function and close the input stream.
*/
libGAP_Obj libGAP_READ_AS_FUNC ( void )
{
    libGAP_Obj                 func;
    libGAP_UInt                type;

    /* now do the reading                                                  */
    libGAP_ClearError();
    type = libGAP_ReadEvalFile();

    /* get the function                                                    */
    if ( type == 0 ) {
        func = libGAP_TLS(libGAP_ReadEvalResult);
    }
    else {
        func = libGAP_Fail;
    }

    /* close the input file again, and return 'true'                       */
    if ( ! libGAP_CloseInput() ) {
        libGAP_ErrorQuit(
            "Panic: READ_AS_FUNC cannot close input, this should not happen",
            0L, 0L );
    }
    libGAP_ClearError();

    /* return the function                                                 */
    return func;
}


static void libGAP_READ_TEST_OR_LOOP(void)
{
    libGAP_UInt                type;
    libGAP_UInt                oldtime;
    libGAP_UInt                dualSemicolon;

    /* get the starting time                                               */
    oldtime = libGAP_SyTime();

    /* now do the reading                                                  */
    while ( 1 ) {

        /* read and evaluate the command                                   */
        libGAP_ClearError();
        type = libGAP_ReadEvalCommand(libGAP_TLS(libGAP_BottomLVars), &dualSemicolon);

        /* stop the stopwatch                                              */
        libGAP_AssGVar( libGAP_Time, libGAP_INTOBJ_INT( libGAP_SyTime() - oldtime ) );

        /* handle ordinary command                                         */
        if ( type == 0 && libGAP_TLS(libGAP_ReadEvalResult) != 0 ) {

            /* remember the value in 'last' and the time in 'time'         */
            libGAP_AssGVar( libGAP_Last3, libGAP_VAL_GVAR( libGAP_Last2 ) );
            libGAP_AssGVar( libGAP_Last2, libGAP_VAL_GVAR( libGAP_Last  ) );
            libGAP_AssGVar( libGAP_Last,  libGAP_TLS(libGAP_ReadEvalResult)   );

            /* print the result                                            */
            if ( ! dualSemicolon ) {
                libGAP_Bag currLVars = libGAP_TLS(libGAP_CurrLVars); /* in case view runs into error */
                libGAP_ViewObjHandler( libGAP_TLS(libGAP_ReadEvalResult) );
                libGAP_SWITCH_TO_OLD_LVARS(currLVars);
            }
        }

        /* handle return-value or return-void command                      */
        else if ( type & (libGAP_STATUS_RETURN_VAL | libGAP_STATUS_RETURN_VOID) ) {
            libGAP_Pr( "'return' must not be used in file read-eval loop",
                0L, 0L );
        }

        /* handle quit command or <end-of-file>                            */
        else if ( type & (libGAP_STATUS_QUIT | libGAP_STATUS_EOF) ) {
            break;
        }
        // FIXME: what about other types? e.g. STATUS_ERROR and STATUS_QQUIT

    }
}


/****************************************************************************
**
*F  READ_LOOP() . . . . . . . . . .  read current input as read-eval-view loop
**
**  Read the current input as read-eval-view loop and close the input stream.
*/
static void libGAP_READ_LOOP ( void )
{
    libGAP_READ_TEST_OR_LOOP();

    /* close the input file again, and return 'true'                       */
    if ( ! libGAP_CloseInput() ) {
        libGAP_ErrorQuit(
            "Panic: ReadLoop cannot close input, this should not happen",
            0L, 0L );
    }
    libGAP_ClearError();
}


/****************************************************************************
**
*F  READ_GAP_ROOT( <filename> ) . . .  read from gap root, dyn-load or static
**
**  'READ_GAP_ROOT' tries to find  a file under  the root directory,  it will
**  search all   directories given   in 'SyGapRootPaths',  check  dynamically
**  loadable modules and statically linked modules.
*/


libGAP_Int libGAP_READ_GAP_ROOT ( libGAP_Char * filename )
{
    libGAP_TypGRF_Data         result;
    libGAP_Int                 res;
    libGAP_UInt                type;
    libGAP_StructInitInfo *    info;

    /* try to find the file                                                */
    res = libGAP_SyFindOrLinkGapRootFile( filename, 0L, &result );

    /* not found                                                           */
    if ( res == 0 ) {
        return 0;
    }

    /* dynamically linked                                                  */
    else if ( res == 1 ) {
        if ( libGAP_SyDebugLoading ) {
            libGAP_Pr( "#I  READ_GAP_ROOT: loading '%s' dynamically\n",
                (libGAP_Int)filename, 0L );
        }
        info = result.module_info;
        res  = info->initKernel(info);
        if (!libGAP_SyRestoring) {
          libGAP_UpdateCopyFopyInfo();
          res  = res || info->initLibrary(info);
        }
        if ( res ) {
            libGAP_Pr( "#W  init functions returned non-zero exit code\n", 0L, 0L );
        }
        
        info->isGapRootRelative = 1;
        libGAP_RecordLoadedModule(info, filename);
        return 1;
    }

    /* statically linked                                                   */
    else if ( res == 2 ) {
        if ( libGAP_SyDebugLoading ) {
            libGAP_Pr( "#I  READ_GAP_ROOT: loading '%s' statically\n",
                (libGAP_Int)filename, 0L );
        }
        info = result.module_info;
        res  = info->initKernel(info);
        if (!libGAP_SyRestoring) {
          libGAP_UpdateCopyFopyInfo();
          res  = res || info->initLibrary(info);
        }
        if ( res ) {
            libGAP_Pr( "#W  init functions returned non-zero exit code\n", 0L, 0L );
        }
        info->isGapRootRelative = 1;
        libGAP_RecordLoadedModule(info, filename);
        return 1;
    }

    /* special handling for the other cases, if we are trying to load compiled
       modules needed for a saved workspace ErrorQuit is not available */
    else if (libGAP_SyRestoring)
      {
        if (res == 3 || res == 4)
          {
            libGAP_Pr("Can't find compiled module '%s' needed by saved workspace\n",
               (libGAP_Int) filename, 0L);
            return 0;
          }
        else
          libGAP_Pr("unknown result code %d from 'SyFindGapRoot'", res, 0L );
        libGAP_SyExit(1);
      }
    
    /* ordinary gap file                                                   */
    else if ( res == 3 || res == 4  ) {
        if ( libGAP_SyDebugLoading ) {
            libGAP_Pr( "#I  READ_GAP_ROOT: loading '%s' as GAP file\n",
                (libGAP_Int)filename, 0L );
        }
        if ( libGAP_OpenInput(result.pathname) ) {
          libGAP_SySetBuffering(libGAP_TLS(libGAP_Input)->libGAP_file);
            while ( 1 ) {
                libGAP_ClearError();
                type = libGAP_ReadEvalCommand(libGAP_TLS(libGAP_BottomLVars), 0);
                if (libGAP_TLS(libGAP_UserHasQuit) || libGAP_TLS(libGAP_UserHasQUIT))
                  break;
                if ( type & (libGAP_STATUS_RETURN_VAL | libGAP_STATUS_RETURN_VOID) ) {
                    libGAP_Pr( "'return' must not be used in file", 0L, 0L );
                }
                else if ( type & (libGAP_STATUS_QUIT | libGAP_STATUS_EOF) ) {
                    break;
                }
            }
            libGAP_CloseInput();
            libGAP_ClearError();
            return 1;
        }
        else {
            return 0;
        }
    }

    /* don't know                                                          */
    else {
        libGAP_ErrorQuit( "unknown result code %d from 'SyFindGapRoot'", res, 0L );
        return 0;
    }
    return 0;
}


/****************************************************************************
**

*F  FuncCLOSE_LOG_TO()  . . . . . . . . . . . . . . . . . . . .  stop logging
**
**  'FuncCLOSE_LOG_TO' implements a method for 'LogTo'.
**
**  'LogTo()'
**
**  'LogTo' called with no argument closes the current logfile again, so that
**  input   from  '*stdin*'  and  '*errin*'  and  output  to  '*stdout*'  and
**  '*errout*' will no longer be echoed to a file.
*/
libGAP_Obj libGAP_FuncCLOSE_LOG_TO (
    libGAP_Obj                 self )
{
    if ( ! libGAP_CloseLog() ) {
        libGAP_ErrorQuit("LogTo: can not close the logfile",0L,0L);
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncLOG_TO( <filename> ) . . . . . . . . . . . .  start logging to a file
**
**  'FuncLOG_TO' implements a method for 'LogTo'
**
**  'LogTo( <filename> )'
**
**  'LogTo' instructs GAP to echo all input from the  standard  input  files,
**  '*stdin*' and '*errin*' and all output  to  the  standard  output  files,
**  '*stdout*'  and  '*errout*',  to  the  file  with  the  name  <filename>.
**  The file is created if it does not  exist,  otherwise  it  is  truncated.
*/
libGAP_Obj libGAP_FuncLOG_TO (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
    while ( ! libGAP_IsStringConv(filename) ) {
        filename = libGAP_ErrorReturnObj(
            "LogTo: <filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    if ( ! libGAP_OpenLog( libGAP_CSTR_STRING(filename) ) ) {
        libGAP_ErrorReturnVoid( "LogTo: cannot log to %s",
                         (libGAP_Int)libGAP_CSTR_STRING(filename), 0L,
                         "you can 'return;'" );
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncLOG_TO_STREAM( <stream> ) . . . . . . . . . start logging to a stream
*/
libGAP_Obj libGAP_FuncLOG_TO_STREAM (
    libGAP_Obj                 self,
    libGAP_Obj                 stream )
{
    if ( ! libGAP_OpenLogStream(stream) ) {
        libGAP_ErrorReturnVoid( "LogTo: cannot log to stream", 0L, 0L,
                         "you can 'return;'" );
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncCLOSE_INPUT_LOG_TO()  . . . . . . . . . . . . . . . . .  stop logging
**
**  'FuncCLOSE_INPUT_LOG_TO' implements a method for 'InputLogTo'.
**
**  'InputLogTo()'
**
**  'InputLogTo' called with no argument closes the current logfile again, so
**  that input from  '*stdin*' and '*errin*' will   no longer be  echoed to a
**  file.
*/
libGAP_Obj libGAP_FuncCLOSE_INPUT_LOG_TO (
    libGAP_Obj                 self )
{
    if ( ! libGAP_CloseInputLog() ) {
        libGAP_ErrorQuit("InputLogTo: can not close the logfile",0L,0L);
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncINPUT_LOG_TO( <filename> )  . . . . . . . . . start logging to a file
**
**  'FuncINPUT_LOG_TO' implements a method for 'InputLogTo'
**
**  'InputLogTo( <filename> )'
**
**  'InputLogTo'  instructs  GAP to echo   all input from  the standard input
**  files, '*stdin*' and '*errin*' to the file with the name <filename>.  The
**  file is created if it does not exist, otherwise it is truncated.
*/
libGAP_Obj libGAP_FuncINPUT_LOG_TO (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
    while ( ! libGAP_IsStringConv(filename) ) {
        filename = libGAP_ErrorReturnObj(
            "InputLogTo: <filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    if ( ! libGAP_OpenInputLog( libGAP_CSTR_STRING(filename) ) ) {
        libGAP_ErrorReturnVoid( "InputLogTo: cannot log to %s",
                         (libGAP_Int)libGAP_CSTR_STRING(filename), 0L,
                         "you can 'return;'" );
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncINPUT_LOG_TO_STREAM( <stream> ) . . . . . . start logging to a stream
*/
libGAP_Obj libGAP_FuncINPUT_LOG_TO_STREAM (
    libGAP_Obj                 self,
    libGAP_Obj                 stream )
{
    if ( ! libGAP_OpenInputLogStream(stream) ) {
        libGAP_ErrorReturnVoid( "InputLogTo: cannot log to stream", 0L, 0L,
                         "you can 'return;'" );
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncCLOSE_OUTPUT_LOG_TO()  . . . . . . . . . . . . . . . . . stop logging
**
**  'FuncCLOSE_OUTPUT_LOG_TO' implements a method for 'OutputLogTo'.
**
**  'OutputLogTo()'
**
**  'OutputLogTo'  called with no argument  closes the current logfile again,
**  so that output from '*stdin*' and '*errin*' will no longer be echoed to a
**  file.
*/
libGAP_Obj libGAP_FuncCLOSE_OUTPUT_LOG_TO (
    libGAP_Obj                 self )
{
    if ( ! libGAP_CloseOutputLog() ) {
        libGAP_ErrorQuit("OutputLogTo: can not close the logfile",0L,0L);
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncOUTPUT_LOG_TO( <filename> )  . . . . . . . .  start logging to a file
**
**  'FuncOUTPUT_LOG_TO' implements a method for 'OutputLogTo'
**
**  'OutputLogTo( <filename> )'
**
**  'OutputLogTo' instructs GAP  to echo all  output from the standard output
**  files, '*stdin*' and '*errin*' to the file with the name <filename>.  The
**  file is created if it does not exist, otherwise it is truncated.
*/
libGAP_Obj libGAP_FuncOUTPUT_LOG_TO (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
    while ( ! libGAP_IsStringConv(filename) ) {
        filename = libGAP_ErrorReturnObj(
            "OutputLogTo: <filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    if ( ! libGAP_OpenOutputLog( libGAP_CSTR_STRING(filename) ) ) {
        libGAP_ErrorReturnVoid( "OutputLogTo: cannot log to %s",
                         (libGAP_Int)libGAP_CSTR_STRING(filename), 0L,
                         "you can 'return;'" );
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncOUTPUT_LOG_TO_STREAM( <stream> ) . . . . .  start logging to a stream
*/
libGAP_Obj libGAP_FuncOUTPUT_LOG_TO_STREAM (
    libGAP_Obj                 self,
    libGAP_Obj                 stream )
{
    if ( ! libGAP_OpenOutputLogStream(stream) ) {
        libGAP_ErrorReturnVoid( "OutputLogTo: cannot log to stream", 0L, 0L,
                         "you can 'return;'" );
        return libGAP_False;
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncPrint( <self>, <args> ) . . . . . . . . . . . . . . . .  print <args>
*/
libGAP_Obj libGAP_FuncPrint (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    volatile libGAP_Obj        arg;
    volatile libGAP_UInt       i;
    libGAP_syJmp_buf           readJmpError;

    /* print all the arguments, take care of strings and functions         */
    for ( i = 1;  i <= libGAP_LEN_PLIST(args);  i++ ) {
        arg = libGAP_ELM_LIST(args,i);
        if ( libGAP_IS_PLIST(arg) && 0 < libGAP_LEN_PLIST(arg) && libGAP_IsStringConv(arg) ) {
            libGAP_PrintString1(arg);
        }
        else if ( libGAP_IS_STRING_REP(arg) ) {
            libGAP_PrintString1(arg);
        }
        else if ( libGAP_TNUM_OBJ( arg ) == libGAP_T_FUNCTION ) {
            libGAP_PrintFunction( arg );
        }
        else {
            memcpy( readJmpError, libGAP_TLS(libGAP_ReadJmpError), sizeof(libGAP_syJmp_buf) );

            /* if an error occurs stop printing                            */
            if ( ! libGAP_READ_ERROR() ) {
                libGAP_PrintObj( arg );
            }
            else {
                memcpy( libGAP_TLS(libGAP_ReadJmpError), readJmpError, sizeof(libGAP_syJmp_buf) );
                libGAP_ReadEvalError();
            }
            memcpy( libGAP_TLS(libGAP_ReadJmpError), readJmpError, sizeof(libGAP_syJmp_buf) );
        }
    }

    return 0;
}

static libGAP_Obj libGAP_PRINT_OR_APPEND_TO(libGAP_Obj args, int append)
{
    const char          *funcname = append ? "AppendTo" : "PrintTo";
    volatile libGAP_Obj        arg;
    volatile libGAP_Obj        filename;
    volatile libGAP_UInt       i;
    libGAP_syJmp_buf           readJmpError;

    /* first entry is the filename                                         */
    filename = libGAP_ELM_LIST(args,1);
    while ( ! libGAP_IsStringConv(filename) ) {
        filename = libGAP_ErrorReturnObj(
            "%s: <filename> must be a string (not a %s)",
            (libGAP_Int)funcname, (libGAP_Int)libGAP_TNAM_OBJ(filename),
            "you can replace <filename> via 'return <filename>;'" );
    }

    /* try to open the file for output                                     */
    i = append ? libGAP_OpenAppend( libGAP_CSTR_STRING(filename) )
               : libGAP_OpenOutput( libGAP_CSTR_STRING(filename) );
    if ( ! i ) {
        libGAP_ErrorQuit( "%s: cannot open '%s' for output",
                   (libGAP_Int)funcname, (libGAP_Int)libGAP_CSTR_STRING(filename) );
        return 0;
    }

    /* print all the arguments, take care of strings and functions         */
    for ( i = 2;  i <= libGAP_LEN_PLIST(args);  i++ ) {
        arg = libGAP_ELM_LIST(args,i);
        if ( libGAP_IS_PLIST(arg) && 0 < libGAP_LEN_PLIST(arg) && libGAP_IsStringConv(arg) ) {
            libGAP_PrintString1(arg);
        }
        else if ( libGAP_IS_STRING_REP(arg) ) {
            libGAP_PrintString1(arg);
        }
        else if ( libGAP_TNUM_OBJ(arg) == libGAP_T_FUNCTION ) {
            libGAP_TLS(libGAP_PrintObjFull) = 1;
            libGAP_PrintFunction( arg );
            libGAP_TLS(libGAP_PrintObjFull) = 0;
        }
        else {
            memcpy( readJmpError, libGAP_TLS(libGAP_ReadJmpError), sizeof(libGAP_syJmp_buf) );

            /* if an error occurs stop printing                            */
            if ( ! libGAP_READ_ERROR() ) {
                libGAP_PrintObj( arg );
            }
            else {
                libGAP_CloseOutput();
                memcpy( libGAP_TLS(libGAP_ReadJmpError), readJmpError, sizeof(libGAP_syJmp_buf) );
                libGAP_ReadEvalError();
            }
            memcpy( libGAP_TLS(libGAP_ReadJmpError), readJmpError, sizeof(libGAP_syJmp_buf) );
        }
    }

    /* close the output file again, and return nothing                     */
    if ( ! libGAP_CloseOutput() ) {
        libGAP_ErrorQuit( "%s: cannot close output", (libGAP_Int)funcname, 0L );
        return 0;
    }

    return 0;
}


static libGAP_Obj libGAP_PRINT_OR_APPEND_TO_STREAM(libGAP_Obj args, int append)
{
    const char          *funcname = append ? "AppendTo" : "PrintTo";
    volatile libGAP_Obj        arg;
    volatile libGAP_Obj        stream;
    volatile libGAP_UInt       i;
    libGAP_syJmp_buf           readJmpError;

    /* first entry is the stream                                           */
    stream = libGAP_ELM_LIST(args,1);

    /* try to open the file for output                                     */
    i = append ? libGAP_OpenAppendStream(stream)
               : libGAP_OpenOutputStream(stream);
    if ( ! i ) {
        libGAP_ErrorQuit( "%s: cannot open stream for output", (libGAP_Int)funcname, 0L );
        return 0;
    }

    /* print all the arguments, take care of strings and functions         */
    for ( i = 2;  i <= libGAP_LEN_PLIST(args);  i++ ) {
        arg = libGAP_ELM_LIST(args,i);

        /* if an error occurs stop printing                                */
        memcpy( readJmpError, libGAP_TLS(libGAP_ReadJmpError), sizeof(libGAP_syJmp_buf) );
        if ( ! libGAP_READ_ERROR() ) {
            if ( libGAP_IS_PLIST(arg) && 0 < libGAP_LEN_PLIST(arg) && libGAP_IsStringConv(arg) ) {
                libGAP_PrintString1(arg);
            }
            else if ( libGAP_IS_STRING_REP(arg) ) {
                libGAP_PrintString1(arg);
            }
            else if ( libGAP_TNUM_OBJ( arg ) == libGAP_T_FUNCTION ) {
                libGAP_TLS(libGAP_PrintObjFull) = 1;
                libGAP_PrintFunction( arg );
                libGAP_TLS(libGAP_PrintObjFull) = 0;
            }
            else {
                libGAP_PrintObj( arg );
            }
        }
        else {
            libGAP_CloseOutput();
            memcpy( libGAP_TLS(libGAP_ReadJmpError), readJmpError, sizeof(libGAP_syJmp_buf) );
            libGAP_ReadEvalError();
        }
        memcpy( libGAP_TLS(libGAP_ReadJmpError), readJmpError, sizeof(libGAP_syJmp_buf) );
    }

    /* close the output file again, and return nothing                     */
    if ( ! libGAP_CloseOutput() ) {
        libGAP_ErrorQuit( "%s: cannot close output", (libGAP_Int)funcname, 0L );
        return 0;
    }

    return 0;
}

/****************************************************************************
**
*F  FuncPRINT_TO( <self>, <args> )  . . . . . . . . . . . . . .  print <args>
*/
libGAP_Obj libGAP_FuncPRINT_TO (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    return libGAP_PRINT_OR_APPEND_TO(args, 0);
}


/****************************************************************************
**
*F  FuncPRINT_TO_STREAM( <self>, <args> ) . . . . . . . . . . .  print <args>
*/
libGAP_Obj libGAP_FuncPRINT_TO_STREAM (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    return libGAP_PRINT_OR_APPEND_TO_STREAM(args, 0);
}


/****************************************************************************
**
*F  FuncAPPEND_TO( <self>, <args> ) . . . . . . . . . . . . . . append <args>
*/
libGAP_Obj libGAP_FuncAPPEND_TO (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    return libGAP_PRINT_OR_APPEND_TO(args, 1);
}


/****************************************************************************
**
*F  FuncAPPEND_TO_STREAM( <self>, <args> )  . . . . . . . . . . append <args>
*/
libGAP_Obj libGAP_FuncAPPEND_TO_STREAM (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    return libGAP_PRINT_OR_APPEND_TO_STREAM(args, 1);
}

libGAP_Obj libGAP_FuncSET_OUTPUT (
    libGAP_Obj                 self,
    libGAP_Obj                 libGAP_file,
    libGAP_Obj                 append    )
{
    
    if ( libGAP_IsStringConv(libGAP_file) ) {
        if ( append != libGAP_False ) {
          if ( ! libGAP_OpenAppend( libGAP_CSTR_STRING(libGAP_file) ) ) {
             libGAP_ErrorQuit( "SET_OUTPUT: cannot open '%s' for appending",
                                  (libGAP_Int)libGAP_CSTR_STRING(libGAP_file), 0L );
          } else {
             return 0;
          }
        } else {
          if ( ! libGAP_OpenOutput( libGAP_CSTR_STRING(libGAP_file) ) ) {
             libGAP_ErrorQuit( "SET_OUTPUT: cannot open '%s' for output",
                                  (libGAP_Int)libGAP_CSTR_STRING(libGAP_file), 0L );
          } else {
            return 0;
          }
        }
    } else {  /* an open stream */
        if ( append != libGAP_False ) {
          if ( ! libGAP_OpenAppendStream( libGAP_file ) ) {
             libGAP_ErrorQuit( "SET_OUTPUT: cannot open stream for appending", 0L, 0L );
          } else {
             return 0;
          }
        } else {
          if ( ! libGAP_OpenOutputStream( libGAP_file ) ) {
             libGAP_ErrorQuit( "SET_OUTPUT: cannot open stream for output", 0L, 0L );
          } else {
            return 0;
          }
        }
    }
    return 0;
}

libGAP_Obj libGAP_FuncSET_PREVIOUS_OUTPUT( libGAP_Obj self ) {
    /* close the current output stream, and return nothing  */

    if ( ! libGAP_CloseOutput() ) {
        libGAP_ErrorQuit( "SET_PREVIOUS_OUTPUT: cannot close output", 0L, 0L );
        return 0;
    }
    return 0;
}
     
/****************************************************************************
**
*F  FuncREAD( <self>, <filename> )  . . . . . . . . . . . . . . . read a file
**
**  Read the current input and close the input stream.
*/
libGAP_Obj libGAP_FuncREAD (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
   /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "READ: <filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }

    /* try to open the file                                                */
    if ( ! libGAP_OpenInput( libGAP_CSTR_STRING(filename) ) ) {
        return libGAP_False;
    }

    libGAP_SySetBuffering(libGAP_TLS(libGAP_Input)->libGAP_file);
   
    /* read the test file                                                  */
    return libGAP_READ() ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  FuncREAD_NORECOVERY( <self>, <filename> )  . . .  . . . . . . read a file
**
** Read the current input and close the input stream. Disable the normal 
** mechanism which ensures that quitting from a break loop gets you back to a 
** live prompt. This is initially designed for the files read from the command 
** line
*/
libGAP_Obj libGAP_FuncREAD_NORECOVERY (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "READ: <filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }

    /* try to open the file                                                */
    if ( ! libGAP_OpenInput( libGAP_CSTR_STRING(filename) ) ) {
        return libGAP_False;
    }

    libGAP_SySetBuffering(libGAP_TLS(libGAP_Input)->libGAP_file);
   
    /* read the file */
    switch (libGAP_READ_NORECOVERY()) {
    case 0: return libGAP_False;
    case 1: return libGAP_True;
    case 2: return libGAP_Fail;
    default: return libGAP_Fail;
    }
}


/****************************************************************************
**
*F  FuncREAD_STREAM( <self>, <stream> )   . . . . . . . . . . . read a stream
*/
libGAP_Obj libGAP_FuncREAD_STREAM (
    libGAP_Obj                 self,
    libGAP_Obj                 stream )
{
    /* try to open the file                                                */
    if ( ! libGAP_OpenInputStream(stream) ) {
        return libGAP_False;
    }

    /* read the test file                                                  */
    return libGAP_READ() ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  FuncREAD_STREAM_LOOP( <self>, <stream>, <catcherrstdout> ) . read a stream
*/
libGAP_Obj libGAP_FuncREAD_STREAM_LOOP (
    libGAP_Obj                 self,
    libGAP_Obj                 stream,
    libGAP_Obj                 catcherrstdout )
{
    /* try to open the file                                                */
    if ( ! libGAP_OpenInputStream(stream) ) {
        return libGAP_False;
    }
    if ( catcherrstdout == libGAP_True )
      libGAP_TLS(libGAP_IgnoreStdoutErrout) = libGAP_TLS(libGAP_Output);
    else
      libGAP_TLS(libGAP_IgnoreStdoutErrout) = NULL;


    /* read the test file                                                  */
    libGAP_READ_LOOP();
    libGAP_TLS(libGAP_IgnoreStdoutErrout) = NULL;
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncREAD_AS_FUNC( <self>, <filename> )  . . . . . . . . . . . read a file
*/
libGAP_Obj libGAP_FuncREAD_AS_FUNC (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "READ_AS_FUNC: <filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }

    /* try to open the file                                                */
    if ( ! libGAP_OpenInput( libGAP_CSTR_STRING(filename) ) ) {
        return libGAP_Fail;
    }

    libGAP_SySetBuffering(libGAP_TLS(libGAP_Input)->libGAP_file);
    
    /* read the function                                                   */
    return libGAP_READ_AS_FUNC();
}


/****************************************************************************
**
*F  FuncREAD_AS_FUNC_STREAM( <self>, <filename> ) . . . . . . . . read a file
*/
libGAP_Obj libGAP_FuncREAD_AS_FUNC_STREAM (
    libGAP_Obj                 self,
    libGAP_Obj                 stream )
{
    /* try to open the file                                                */
    if ( ! libGAP_OpenInputStream(stream) ) {
        return libGAP_Fail;
    }

    /* read the function                                                   */
    return libGAP_READ_AS_FUNC();
}


/****************************************************************************
**
*F  FuncREAD_GAP_ROOT( <self>, <filename> ) . . . . . . . . . . . read a file
*/
libGAP_Obj libGAP_FuncREAD_GAP_ROOT (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
    libGAP_Char filenamecpy[4096];

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "READ: <filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }

    /* Copy to avoid garbage collection moving string                      */
    libGAP_strlcpy(filenamecpy, libGAP_CSTR_STRING(filename), 4096);
    /* try to open the file                                                */
    return libGAP_READ_GAP_ROOT(filenamecpy) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**

*F  FuncTmpName( <self> ) . . . . . . . . . . . . . . return a temporary name
*/
libGAP_Obj libGAP_FuncTmpName (
    libGAP_Obj                 self )
{
    libGAP_Char *              tmp;
    libGAP_Obj                 name;

    tmp = libGAP_SyTmpname();
    if ( tmp == 0 )
        return libGAP_Fail;
    libGAP_C_NEW_STRING_DYN( name, tmp );
    return name;
}


/****************************************************************************
**
*F  FuncTmpDirectory( <self> )  . . . . . . . .  return a temporary directory
*/
libGAP_Obj libGAP_FuncTmpDirectory (
    libGAP_Obj                 self )
{
    libGAP_Char *              tmp;
    libGAP_Obj                 name;

    tmp = libGAP_SyTmpdir("tm");
    if ( tmp == 0 )
        return libGAP_Fail;
    libGAP_C_NEW_STRING_DYN( name, tmp );
    return name;
}


/****************************************************************************
**
*F  FuncRemoveFile( <self>, <name> )  . . . . . . . . . .  remove file <name>
*/
libGAP_Obj libGAP_FuncRemoveFile (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    return libGAP_SyRemoveFile( libGAP_CSTR_STRING(filename) ) == -1 ? libGAP_Fail : libGAP_True;
}

/****************************************************************************
**
*F  FuncCreateDir( <self>, <name> )  . . . . . . . . . . . . create directory
*/
libGAP_Obj libGAP_FuncCreateDir (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    return libGAP_SyMkdir( libGAP_CSTR_STRING(filename) ) == -1 ? libGAP_Fail : libGAP_True;
}

/****************************************************************************
**
*F  FuncRemoveDir( <self>, <name> )  . . . . . . . . . . . . remove directory
*/
libGAP_Obj libGAP_FuncRemoveDir (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    return libGAP_SyRmdir( libGAP_CSTR_STRING(filename) ) == -1 ? libGAP_Fail : libGAP_True;
}

/****************************************************************************
**
*F  FuncIsDir( <self>, <name> )  . . . . . check whether something is a dir
*/
libGAP_Obj libGAP_FuncIsDir (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }

    /* call the system dependent function                                  */
    return libGAP_SyIsDir( libGAP_CSTR_STRING(filename) );
}




/****************************************************************************
**

*F * * * * * * * * * * * file access test functions * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  FuncLastSystemError( <self> ) .  . . . . . .  return the last system error
*/
libGAP_UInt libGAP_ErrorMessageRNam;
libGAP_UInt libGAP_ErrorNumberRNam;

libGAP_Obj libGAP_FuncLastSystemError (
    libGAP_Obj             self )
{
    libGAP_Obj             err;
    libGAP_Obj             msg;

    /* constructed an error record                                         */
    err = libGAP_NEW_PREC(0);

    /* check if an errors has occured                                      */
    if ( libGAP_SyLastErrorNo != 0 ) {
        libGAP_ASS_REC( err, libGAP_ErrorNumberRNam, libGAP_INTOBJ_INT(libGAP_SyLastErrorNo) );
        libGAP_C_NEW_STRING_DYN(msg, libGAP_SyLastErrorMessage);
        libGAP_ASS_REC( err, libGAP_ErrorMessageRNam, msg );
    }

    /* no error has occured                                                */
    else {
        libGAP_ASS_REC( err, libGAP_ErrorNumberRNam, libGAP_INTOBJ_INT(0) );
        libGAP_C_NEW_STRING_CONST( msg, "no error" );
        libGAP_ASS_REC( err, libGAP_ErrorMessageRNam, msg );
    }

    /* return the error record                                             */
    return err;
}


/****************************************************************************
**
*F  FuncIsExistingFile( <self>, <name> )  . . . . . . does file <name> exists
*/
libGAP_Obj libGAP_FuncIsExistingFile (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    libGAP_Int             res;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    res = libGAP_SyIsExistingFile( libGAP_CSTR_STRING(filename) );
    return res == -1 ? libGAP_False : libGAP_True;
}


/****************************************************************************
**
*F  FuncIsReadableFile( <self>, <name> )  . . . . . . is file <name> readable
*/
libGAP_Obj libGAP_FuncIsReadableFile (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    libGAP_Int             res;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    res = libGAP_SyIsReadableFile( libGAP_CSTR_STRING(filename) );
    return res == -1 ? libGAP_False : libGAP_True;
}


/****************************************************************************
**
*F  FuncIsWritableFile( <self>, <name> )  . . . . . . is file <name> writable
*/
libGAP_Obj libGAP_FuncIsWritableFile (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    libGAP_Int             res;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    res = libGAP_SyIsWritableFile( libGAP_CSTR_STRING(filename) );
    return res == -1 ? libGAP_False : libGAP_True;
}


/****************************************************************************
**
*F  FuncIsExecutableFile( <self>, <name> )  . . . . is file <name> executable
*/
libGAP_Obj libGAP_FuncIsExecutableFile (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    libGAP_Int             res;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    res = libGAP_SyIsExecutableFile( libGAP_CSTR_STRING(filename) );
    return res == -1 ? libGAP_False : libGAP_True;
}


/****************************************************************************
**
*F  FuncIsDirectoryPath( <self>, <name> ) . . . .  is file <name> a directory
*/
libGAP_Obj libGAP_FuncIsDirectoryPathString (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    libGAP_Int             res;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    res = libGAP_SyIsDirectoryPath( libGAP_CSTR_STRING(filename) );
    return res == -1 ? libGAP_False : libGAP_True;
}


/****************************************************************************
**
*F  FuncSTRING_LIST_DIR( <self>, <dirname> ) . . . read names of files in dir
**
**  This function returns a GAP string which contains the names of all files
**  contained in a directory <dirname>. The file names are separated by zero 
**  characters (which are not allowed in file names). 
**
**  If <dirname> could not be opened as a directory 'fail' is returned. The
**  reason for the error can be found with 'LastSystemError();' in GAP.
**
*/
libGAP_Obj libGAP_FuncSTRING_LIST_DIR (
    libGAP_Obj         self,
    libGAP_Obj         dirname  )
{
    DIR *dir;
    struct dirent *entry;
    libGAP_Obj res;
    libGAP_Int len, sl;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( dirname ) ) {
        dirname = libGAP_ErrorReturnObj(
            "<dirname> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(dirname), 0L,
            "you can replace <dirname> via 'return <dirname>;'" );
    }
    
    libGAP_SyClearErrorNo();
    dir = opendir(libGAP_CSTR_STRING(dirname));
    if (dir == NULL) {
      libGAP_SySetErrorNo();
      return libGAP_Fail;
    }
    res = libGAP_NEW_STRING(256);
    len = 0;
    entry = readdir(dir);
    while (entry != NULL) {
      sl = strlen(entry->d_name);
      libGAP_GROW_STRING(res, len + sl + 1);
      memcpy(libGAP_CHARS_STRING(res) + len, entry->d_name, sl + 1);
      len = len + sl + 1;
      entry = readdir(dir);
    }
    closedir(dir);
    /* tell the result string its length and terminate by 0 char */
    libGAP_SET_LEN_STRING(res, len);
    *(libGAP_CHARS_STRING(res) + len) = 0;
    return res;
}

/****************************************************************************
**

*F * * * * * * * * * * * * text stream functions  * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  FuncCLOSE_FILE( <self>, <fid> ) . . . . . . . . . . . . .  close a stream
*/
libGAP_Obj libGAP_FuncCLOSE_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid )
{
    libGAP_Int             ret;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    
    /* call the system dependent function                                  */
    ret = libGAP_SyFclose( libGAP_INT_INTOBJ(fid) );
    return ret == -1 ? libGAP_Fail : libGAP_True;
}


/****************************************************************************
**
*F  FuncINPUT_TEXT_FILE( <self>, <name> ) . . . . . . . . . . . open a stream
*/
libGAP_Obj libGAP_FuncINPUT_TEXT_FILE (
    libGAP_Obj             self,
    libGAP_Obj             filename )
{
    libGAP_Int             fid;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    
    /* call the system dependent function                                  */
    libGAP_SyClearErrorNo();
    fid = libGAP_SyFopen( libGAP_CSTR_STRING(filename), "r" );
    if ( fid == - 1)
        libGAP_SySetErrorNo();
    return fid == -1 ? libGAP_Fail : libGAP_INTOBJ_INT(fid);
}


/****************************************************************************
**
*F  FuncIS_END_OF_FILE( <self>, <fid> ) . . . . . . . . . . .  is end of file
*/
libGAP_Obj libGAP_FuncIS_END_OF_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid )
{
    libGAP_Int             ret;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    
    ret = libGAP_SyIsEndOfFile( libGAP_INT_INTOBJ(fid) );
    return ret == -1 ? libGAP_Fail : ( ret == 0 ? libGAP_False : libGAP_True );
}


/****************************************************************************
**
*F  FuncOUTPUT_TEXT_FILE( <self>, <name>, <append> )  . . . . . open a stream
*/
libGAP_Obj libGAP_FuncOUTPUT_TEXT_FILE (
    libGAP_Obj             self,
    libGAP_Obj             filename,
    libGAP_Obj             append )
{
    libGAP_Int             fid;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    while ( append != libGAP_True && append != libGAP_False ) {
        filename = libGAP_ErrorReturnObj(
            "<append> must be a boolean (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(append), 0L,
            "you can replace <append> via 'return <append>;'" );
    }
    
    /* call the system dependent function                                  */
    libGAP_SyClearErrorNo();
    if ( append == libGAP_True ) {
        fid = libGAP_SyFopen( libGAP_CSTR_STRING(filename), "a" );
    }
    else {
        fid = libGAP_SyFopen( libGAP_CSTR_STRING(filename), "w" );
    }
    if ( fid == - 1)
        libGAP_SySetErrorNo();
    return fid == -1 ? libGAP_Fail : libGAP_INTOBJ_INT(fid);
}


/****************************************************************************
**
*F  FuncPOSITION_FILE( <self>, <fid> )  . . . . . . . . .  position of stream
*/
libGAP_Obj libGAP_FuncPOSITION_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid )
{
    libGAP_Int             ret;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    
    ret = libGAP_SyFtell( libGAP_INT_INTOBJ(fid) );
    return ret == -1 ? libGAP_Fail : libGAP_INTOBJ_INT(ret);
}



/****************************************************************************
**
*F  FuncREAD_BYTE_FILE( <self>, <fid> ) . . . . . . . . . . . . . read a byte
*/
libGAP_Obj libGAP_FuncREAD_BYTE_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid )
{
    libGAP_Int             ret;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    
    /* call the system dependent function                                  */
    ret = libGAP_SyGetch( libGAP_INT_INTOBJ(fid) );

    return ret == EOF ? libGAP_Fail : libGAP_INTOBJ_INT(ret);
}


/****************************************************************************
**
*F  FuncREAD_LINE_FILE( <self>, <fid> ) . . . . . . . . . . . . . read a line
**  
**  This uses fgets and works only if there are no zero characters in <fid>.
*/

/*  this would be a proper function but it reads single chars and is slower

Now SyFputs uses read byte-by-byte, so probably OK

Obj FuncREAD_LINE_FILE (
    Obj             self,
    Obj             fid )
{
    Int             fidc, len, i;
    Obj             str;
    UInt1           *p;
    Int              c;

    while ( ! IS_INTOBJ(fid) ) {
        fid = ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (Int)TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    
    str = NEW_STRING(10);
    len = 10;
    i = 0;
    fidc = INT_INTOBJ(fid);
    p = CHARS_STRING(str); 
    while (1) {
      c = SyGetc(fidc);
      if (i == len) {
        len = GrowString(str, len+1);
        p = CHARS_STRING(str);
      }
      if (c == '\n') {
        p[i++] = (UInt1)c;
        break;
      }
      else if (c == EOF) 
        break;
      else {
        p[i++] = (UInt1)c;
      }
    }
    ResizeBag( str, SIZEBAG_STRINGLEN(i) );
    SET_LEN_STRING(str, i);
      
    return i == 0 ? Fail : str;
}
*/
libGAP_Obj libGAP_FuncREAD_LINE_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid )
{
    libGAP_Char            buf[256];
    libGAP_Char *          cstr;
    libGAP_Int             ifid, len, buflen;
    libGAP_UInt            lstr;
    libGAP_Obj             str;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    ifid = libGAP_INT_INTOBJ(fid);

    /* read <fid> until we see a newline or eof or we've read at least
       one byte and more are not immediately available */
    str = libGAP_NEW_STRING(0);
    len = 0;
    while (1) {
      if ( len > 0 && !libGAP_HasAvailableBytes(ifid))
        break;
      len += 255;
      libGAP_GROW_STRING( str, len );
      if ( libGAP_SyFgetsSemiBlock( buf, 256, ifid ) == 0 )
        break;
      buflen = strlen(buf);
      lstr = libGAP_GET_LEN_STRING(str);
      cstr = libGAP_CSTR_STRING(str) + lstr;
      memcpy( cstr, buf, buflen+1 );
      libGAP_SET_LEN_STRING(str, lstr+buflen);
      if ( buf[buflen-1] == '\n' )
        break;
    }

    /* fix the length of <str>                                             */
    len = libGAP_GET_LEN_STRING(str);
    libGAP_ResizeBag( str, libGAP_SIZEBAG_STRINGLEN(len) );

    /* and return                                                          */
    return len == 0 ? libGAP_Fail : str;
}

/****************************************************************************
**
*F  FuncREAD_ALL_FILE( <self>, <fid>, <limit> )  . . . . . . . read remainder
**  
** more precisely, read until either
**   (a) we have read at least one byte and no more are available
**   (b) we have evidence that it will never be possible to read a byte
**   (c) we have read <limit> bytes (-1 indicates no limit)
*/

libGAP_Obj libGAP_FuncREAD_ALL_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid,
    libGAP_Obj             limit)
{
    libGAP_Char            buf[20000];
    libGAP_Int             ifid, len;
    libGAP_UInt            lstr;
    libGAP_Obj             str;
    libGAP_Int             ilim;
    libGAP_UInt            csize;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    ifid = libGAP_INT_INTOBJ(fid);

    while ( ! libGAP_IS_INTOBJ(limit) ) {
      limit = libGAP_ErrorReturnObj(
			     "<limit> must be a small integer (not a %s)",
			     (libGAP_Int)libGAP_TNAM_OBJ(limit), 0L,
			     "you can replace limit via 'return <limit>;'" );
    }
    ilim = libGAP_INT_INTOBJ(limit);

    /* read <fid> until we see  eof or we've read at least
       one byte and more are not immediately available */
    str = libGAP_NEW_STRING(0);
    len = 0;
    lstr = 0;


    if (libGAP_syBuf[ifid].bufno >= 0)
      {
	libGAP_UInt bufno = libGAP_syBuf[ifid].bufno;

	/* first drain the buffer */
	lstr = libGAP_syBuffers[bufno].buflen - libGAP_syBuffers[bufno].bufstart;
	if (ilim != -1)
	  {
	    if (lstr > ilim)
	      lstr = ilim;
	    ilim -= lstr;
	  }
	libGAP_GROW_STRING(str, lstr);
	memcpy(libGAP_CHARS_STRING(str), libGAP_syBuffers[bufno].buf + libGAP_syBuffers[bufno].bufstart, lstr);
	len = lstr;
	libGAP_SET_LEN_STRING(str, len);
	libGAP_syBuffers[bufno].bufstart += lstr;
      }
#if SYS_IS_CYGWIN32
 getmore:
#endif
    while (ilim == -1 || len < ilim ) {
      if ( len > 0 && !libGAP_HasAvailableBytes(ifid))
	break;
      if (libGAP_syBuf[ifid].isTTY)
	{
	  if (ilim == -1)
	    {
	      libGAP_Pr("#W Warning -- reading to  end of input tty will never end\n",0,0);
	      csize = 20000;
	    }
	  else
	      csize = ((ilim- len) > 20000) ? 20000 : ilim - len;
	    
	  if (libGAP_SyFgetsSemiBlock(buf, csize, ifid))
	    lstr = strlen(buf);
	  else  
	    lstr = 0;
	}
      else
	{
	  do {
	    csize = (ilim == -1 || (ilim- len) > 20000) ? 20000 : ilim - len;
	    lstr = read(libGAP_syBuf[ifid].fp, buf, csize);
	  } while (lstr == -1 && errno == EAGAIN);
	}
      if (lstr <= 0)
	{
	  libGAP_syBuf[ifid].ateof = 1;
	  break;
	}
      libGAP_GROW_STRING( str, len+lstr );
      memcpy(libGAP_CHARS_STRING(str)+len, buf, lstr);
      len += lstr;
      libGAP_SET_LEN_STRING(str, len);
    }

    /* fix the length of <str>                                             */
    len = libGAP_GET_LEN_STRING(str);
#if SYS_IS_CYGWIN32
    /* line end hackery */
    {
      libGAP_UInt i = 0,j = 0;
      while ( i < len )
	{
	  if (libGAP_CHARS_STRING(str)[i] == '\r')
	    {
	      if (i < len -1 && libGAP_CHARS_STRING(str)[i+1] == '\n')
		{
		  i++;
		  continue;
		}
	      else
		libGAP_CHARS_STRING(str)[i] = '\n';
	    }
	  libGAP_CHARS_STRING(str)[j++] = libGAP_CHARS_STRING(str)[i++];
	}
      len = j;
      libGAP_SET_LEN_STRING(str, len);
      if (ilim != -1 && len < ilim)
	goto getmore;
      
    }
#endif
    libGAP_ResizeBag( str, libGAP_SIZEBAG_STRINGLEN(len) );

    /* and return                                                          */
    return len == 0 ? libGAP_Fail : str;
}

/****************************************************************************
**
*F  FuncSEEK_POSITION_FILE( <self>, <fid>, <pos> )  . seek position of stream
*/
libGAP_Obj libGAP_FuncSEEK_POSITION_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid,
    libGAP_Obj             pos )
{
    libGAP_Int             ret;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    while ( ! libGAP_IS_INTOBJ(pos) ) {
        pos = libGAP_ErrorReturnObj(
            "<pos> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L,
            "you can replace <pos> via 'return <pos>;'" );
    }
    
    if (libGAP_syBuf[libGAP_INT_INTOBJ(fid)].bufno >= 0)
    {
            libGAP_syBuffers[libGAP_syBuf[libGAP_INT_INTOBJ(fid)].bufno].buflen = 0;
            libGAP_syBuffers[libGAP_syBuf[libGAP_INT_INTOBJ(fid)].bufno].bufstart = 0;
    }
    ret = libGAP_SyFseek( libGAP_INT_INTOBJ(fid), libGAP_INT_INTOBJ(pos) );
    return ret == -1 ? libGAP_Fail : libGAP_True;
}


/****************************************************************************
**
*F  FuncWRITE_BYTE_FILE( <self>, <fid>, <byte> )  . . . . . . .  write a byte
*/
libGAP_Obj libGAP_FuncWRITE_BYTE_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid,
    libGAP_Obj             ch )
{
    libGAP_Int             ret;

    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    while ( ! libGAP_IS_INTOBJ(ch) ) {
        ch = libGAP_ErrorReturnObj(
            "<ch> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(ch), 0L,
            "you can replace <ch> via 'return <ch>;'" );
    }
    
    /* call the system dependent function                                  */
    ret = libGAP_SyEchoch( libGAP_INT_INTOBJ(ch), libGAP_INT_INTOBJ(fid) );
    return ret == -1 ? libGAP_Fail : libGAP_True;
}

/****************************************************************************
**
*F  FuncWRITE_STRING_FILE_NC( <self>, <fid>, <string> ) .write a whole string
*/
libGAP_Obj libGAP_FuncWRITE_STRING_FILE_NC (
    libGAP_Obj             self,
    libGAP_Obj             fid,
    libGAP_Obj             str )
{
    libGAP_Int             len = 0, l, ret;
    char            *ptr;

    /* don't check the argument                                            */
    
    len = libGAP_GET_LEN_STRING(str);
    ptr = libGAP_CSTR_STRING(str);
    while (len > 0) {
      l = (len > 1048576) ? 1048576 : len;
      ret = write( libGAP_syBuf[libGAP_INT_INTOBJ(fid)].echo, ptr, l);
      if (ret == -1) {
        libGAP_SySetErrorNo();
        return libGAP_Fail;
      }
      len -= ret;
      ptr += ret;
    }
    return libGAP_True;
}

libGAP_Obj libGAP_FuncREAD_STRING_FILE (
    libGAP_Obj             self,
    libGAP_Obj             fid )
{
    /* check the argument                                                  */
    while ( ! libGAP_IS_INTOBJ(fid) ) {
        fid = libGAP_ErrorReturnObj(
            "<fid> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(fid), 0L,
            "you can replace <fid> via 'return <fid>;'" );
    }
    return libGAP_SyReadStringFid(libGAP_INT_INTOBJ(fid));
}

/****************************************************************************
**
*F  FuncFD_OF_FILE( <fid> )
*/
libGAP_Obj libGAP_FuncFD_OF_FILE(libGAP_Obj self,libGAP_Obj fid)
{
  libGAP_Int fd;
  int fdi;
  while (fid == (libGAP_Obj) 0 || !(libGAP_IS_INTOBJ(fid))) {
    fid = libGAP_ErrorReturnObj(
           "<fid> must be a small integer (not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(fid),0L,
           "you can replace <fid> via 'return <fid>;'" );
  }

  fd = libGAP_INT_INTOBJ(fid);
  fdi = libGAP_syBuf[fd].fp;
  return libGAP_INTOBJ_INT(fdi);
}

#if HAVE_SELECT
libGAP_Obj libGAP_FuncUNIXSelect(libGAP_Obj self, libGAP_Obj inlist, libGAP_Obj outlist, libGAP_Obj exclist, 
                   libGAP_Obj timeoutsec, libGAP_Obj timeoutusec)
{
  fd_set infds,outfds,excfds;
  struct timeval tv;
  int n,maxfd;
  libGAP_Int i,j;
  libGAP_Obj o;

  while (inlist == (libGAP_Obj) 0 || !(libGAP_IS_PLIST(inlist)))
    inlist = libGAP_ErrorReturnObj(
           "<inlist> must be a list of small integers (not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(inlist),0L,
           "you can replace <inlist> via 'return <inlist>;'" );
  while (outlist == (libGAP_Obj) 0 || !(libGAP_IS_PLIST(outlist)))
    outlist = libGAP_ErrorReturnObj(
           "<outlist> must be a list of small integers (not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(outlist),0L,
           "you can replace <outlist> via 'return <outlist>;'" );
  while (exclist == (libGAP_Obj) 0 || !(libGAP_IS_PLIST(exclist)))
    exclist = libGAP_ErrorReturnObj(
           "<exclist> must be a list of small integers (not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(exclist),0L,
           "you can replace <exclist> via 'return <exclist>;'" );

  FD_ZERO(&infds);
  FD_ZERO(&outfds);
  FD_ZERO(&excfds);
  maxfd = 0;
  /* Handle input file descriptors: */
  for (i = 1;i <= libGAP_LEN_PLIST(inlist);i++) {
    o = libGAP_ELM_PLIST(inlist,i);
    if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
      j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
      FD_SET(j,&infds);
      if (j > maxfd) maxfd = j;
    }
  }
  /* Handle output file descriptors: */
  for (i = 1;i <= libGAP_LEN_PLIST(outlist);i++) {
    o = libGAP_ELM_PLIST(outlist,i);
    if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
      j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
      FD_SET(j,&outfds);
      if (j > maxfd) maxfd = j;
    }
  }
  /* Handle exception file descriptors: */
  for (i = 1;i <= libGAP_LEN_PLIST(exclist);i++) {
    o = libGAP_ELM_PLIST(exclist,i);
    if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
      j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
      FD_SET(j,&excfds);
      if (j > maxfd) maxfd = j;
    }
  }
  /* Handle the timeout: */
  if (timeoutsec != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(timeoutsec) &&
      timeoutusec != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(timeoutusec)) {
    tv.tv_sec = libGAP_INT_INTOBJ(timeoutsec);
    tv.tv_usec = libGAP_INT_INTOBJ(timeoutusec);
    n = select(maxfd+1,&infds,&outfds,&excfds,&tv);
  } else {
    n = select(maxfd+1,&infds,&outfds,&excfds,NULL);
  }
    
  if (n >= 0) {
    /* Now run through the lists and call functions if ready: */

    for (i = 1;i <= libGAP_LEN_PLIST(inlist);i++) {
      o = libGAP_ELM_PLIST(inlist,i);
      if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
        j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
        if (!(FD_ISSET(j,&infds))) {
          libGAP_SET_ELM_PLIST(inlist,i,libGAP_Fail);
          libGAP_CHANGED_BAG(inlist);
        }
      }
    }
    /* Handle output file descriptors: */
    for (i = 1;i <= libGAP_LEN_PLIST(outlist);i++) {
      o = libGAP_ELM_PLIST(outlist,i);
      if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
        j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
        if (!(FD_ISSET(j,&outfds))) {
          libGAP_SET_ELM_PLIST(outlist,i,libGAP_Fail);
          libGAP_CHANGED_BAG(outlist);
        }
      }
    }
    /* Handle exception file descriptors: */
    for (i = 1;i <= libGAP_LEN_PLIST(exclist);i++) {
      o = libGAP_ELM_PLIST(exclist,i);
      if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
        j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
        if (!(FD_ISSET(j,&excfds))) {
          libGAP_SET_ELM_PLIST(exclist,i,libGAP_Fail);
          libGAP_CHANGED_BAG(exclist);
        }
      }
    }
  }
  return libGAP_INTOBJ_INT(n);
}
#endif

/****************************************************************************
**

*F * * * * * * * * * * * * * execution functions  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  FuncExecuteProcess( <self>, <dir>, <prg>, <in>, <out>, <args> )   process
*/
static libGAP_Obj    libGAP_ExecArgs  [ 1024 ];
static libGAP_Char * libGAP_ExecCArgs [ 1024 ];

libGAP_Obj libGAP_FuncExecuteProcess (
    libGAP_Obj                 self,
    libGAP_Obj                 dir,
    libGAP_Obj                 prg,
    libGAP_Obj                 in,
    libGAP_Obj                 out,
    libGAP_Obj                 args )
{
    libGAP_Obj                 tmp;
    libGAP_Int                 res;
    libGAP_Int                 i;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv(dir) ) {
        dir = libGAP_ErrorReturnObj(
            "<dir> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(dir), 0L,
            "you can replace <dir> via 'return <dir>;'" );
    }
    while ( ! libGAP_IsStringConv(prg) ) {
        prg = libGAP_ErrorReturnObj(
            "<prg> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(prg), 0L,
            "you can replace <prg> via 'return <prg>;'" );
    }
    while ( ! libGAP_IS_INTOBJ(in) ) {
        in = libGAP_ErrorReturnObj(
            "<in> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(in), 0L,
            "you can replace <in> via 'return <in>;'" );
    }
    while ( ! libGAP_IS_INTOBJ(out) ) {
        out = libGAP_ErrorReturnObj(
            "<out> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(out), 0L,
            "you can replace <out> via 'return <out>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(args) ) {
        args = libGAP_ErrorReturnObj(
            "<args> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(args), 0L,
            "you can replace <args> via 'return <args>;'" );
    }

    /* create an argument array                                            */
    for ( i = 1;  i <= libGAP_LEN_LIST(args);  i++ ) {
        if ( i == 1023 )
            break;
        tmp = libGAP_ELM_LIST( args, i );
        while ( ! libGAP_IsStringConv(tmp) ) {
            tmp = libGAP_ErrorReturnObj(
                "<tmp> must be a string (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(tmp), 0L,
                "you can replace <tmp> via 'return <tmp>;'" );
        }
        libGAP_ExecArgs[i] = tmp;
    }
    libGAP_ExecCArgs[0]   = libGAP_CSTR_STRING(prg);
    libGAP_ExecCArgs[i] = 0;
    for ( i--;  0 < i;  i-- ) {
        libGAP_ExecCArgs[i] = libGAP_CSTR_STRING(libGAP_ExecArgs[i]);
    }
    if (libGAP_SyWindow && out == libGAP_INTOBJ_INT(1)) /* standard output */
      libGAP_syWinPut( libGAP_INT_INTOBJ(out), "@z","");

    /* execute the process                                                 */
    res = libGAP_SyExecuteProcess( libGAP_CSTR_STRING(dir),
                            libGAP_CSTR_STRING(prg),
                            libGAP_INT_INTOBJ(in),
                            libGAP_INT_INTOBJ(out),
                            libGAP_ExecCArgs );

    if (libGAP_SyWindow && out == libGAP_INTOBJ_INT(1)) /* standard output */
      libGAP_syWinPut( libGAP_INT_INTOBJ(out), "@mAgIc","");
    return res == 255 ? libGAP_Fail : libGAP_INTOBJ_INT(res);
}


/****************************************************************************
**


*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "READ", 1L, "filename",
      libGAP_FuncREAD, "src/streams.c:READ" },

    { "READ_NORECOVERY", 1L, "filename",
      libGAP_FuncREAD_NORECOVERY, "src/streams.c:READ_NORECOVERY" },

    { "READ_COMMAND_REAL", 2L, "stream, echo",
      libGAP_FuncREAD_COMMAND_REAL, "src/streams.c:READ_COMMAND_REAL" },

    { "READ_COMMAND", 2L, "stream, echo", 
      libGAP_FuncREAD_COMMAND, "src/streams.c:READ_COMMAND" },

    { "READ_STREAM", 1L, "stream",
      libGAP_FuncREAD_STREAM, "src/streams.c:READ_STREAM" },

    { "READ_STREAM_LOOP", 2L, "stream, catchstderrout",
      libGAP_FuncREAD_STREAM_LOOP, "src/streams.c:READ_STREAM_LOOP" },

    { "READ_AS_FUNC", 1L, "filename",
      libGAP_FuncREAD_AS_FUNC, "src/streams.c:READ_AS_FUNC" },

    { "READ_AS_FUNC_STREAM", 1L, "stream", 
      libGAP_FuncREAD_AS_FUNC_STREAM, "src/streams.c:READ_AS_FUNC_STREAM" },

    { "READ_GAP_ROOT", 1L, "filename",
      libGAP_FuncREAD_GAP_ROOT, "src/streams.c:READ_GAP_ROOT" },

    { "LOG_TO", 1L, "filename", 
      libGAP_FuncLOG_TO, "src/streams.c:LOG_TO" },

    { "LOG_TO_STREAM", 1L, "filename", 
      libGAP_FuncLOG_TO_STREAM, "src/streams.c:LOG_TO_STREAM" },

    { "CLOSE_LOG_TO", 0L, "", 
      libGAP_FuncCLOSE_LOG_TO, "src/streams.c:CLOSE_LOG_TO" },

    { "INPUT_LOG_TO", 1L, "filename", 
      libGAP_FuncINPUT_LOG_TO, "src/streams.c:INPUT_LOG_TO" },

    { "INPUT_LOG_TO_STREAM", 1L, "filename", 
      libGAP_FuncINPUT_LOG_TO_STREAM, "src/streams.c:INPUT_LOG_TO_STREAM" },

    { "CLOSE_INPUT_LOG_TO", 0L, "", 
      libGAP_FuncCLOSE_INPUT_LOG_TO, "src/streams.c:CLOSE_INPUT_LOG_TO" },

    { "OUTPUT_LOG_TO", 1L, "filename", 
      libGAP_FuncOUTPUT_LOG_TO, "src/streams.c:OUTPUT_LOG_TO" },

    { "OUTPUT_LOG_TO_STREAM", 1L, "filename", 
      libGAP_FuncOUTPUT_LOG_TO_STREAM, "src/streams.c:OUTPUT_LOG_TO_STREAM" },

    { "CLOSE_OUTPUT_LOG_TO", 0L, "", 
      libGAP_FuncCLOSE_OUTPUT_LOG_TO, "src/streams.c:CLOSE_OUTPUT_LOG_TO" },

    { "Print", -1L, "args",
      libGAP_FuncPrint, "src/streams.c:Print" },

    { "PRINT_TO", -1L, "args",
      libGAP_FuncPRINT_TO, "src/streams.c:PRINT_TO" },

    { "PRINT_TO_STREAM", -1L, "args",
      libGAP_FuncPRINT_TO_STREAM, "src/streams.c:PRINT_TO_STREAM" },

    { "APPEND_TO", -1L, "args",
      libGAP_FuncAPPEND_TO, "src/streams.c:APPEND_TO" },

    { "APPEND_TO_STREAM", -1L, "args",
      libGAP_FuncAPPEND_TO_STREAM, "src/streams.c:APPEND_TO_STREAM" },

    { "SET_OUTPUT", 2, "file, app",
      libGAP_FuncSET_OUTPUT, "src/streams.c:SET_OUTPUT" },

    { "SET_PREVIOUS_OUTPUT", 0, "",
      libGAP_FuncSET_PREVIOUS_OUTPUT, "src/streams.c:SET_PREVIOUS_OUTPUT" },

    { "TmpName", 0L, "",
      libGAP_FuncTmpName, "src/streams.c:TmpName" },

    { "TmpDirectory", 0L, "",
      libGAP_FuncTmpDirectory, "src/streams.c:TmpDirectory" },

    { "RemoveFile", 1L, "filename",
      libGAP_FuncRemoveFile, "src/streams.c:RemoveFile" },

    { "CreateDir", 1L, "filename",
      libGAP_FuncCreateDir, "src/streams.c:CreateDir" },

    { "RemoveDir", 1L, "filename",
      libGAP_FuncRemoveDir, "src/streams.c:RemoveDir" },

    { "IsDir", 1L, "filename",
      libGAP_FuncIsDir, "src/streams.c:IsDir" },

    { "LastSystemError", 0L, "", 
      libGAP_FuncLastSystemError, "src/streams.c:LastSystemError" },

    { "IsExistingFile", 1L, "filename", 
      libGAP_FuncIsExistingFile, "src/streams.c:IsExistingFile" },

    { "IsReadableFile", 1L, "filename",
      libGAP_FuncIsReadableFile, "src/streams.c:IsReadableFile" },

    { "IsWritableFile", 1L, "filename",
      libGAP_FuncIsWritableFile, "src/streams.c:IsWritableFile" },

    { "IsExecutableFile", 1L, "filename",
      libGAP_FuncIsExecutableFile, "src/streams.c:IsExecutableFile" },

    { "IsDirectoryPathString", 1L, "filename",
      libGAP_FuncIsDirectoryPathString, "src/streams.c:IsDirectoryPath" },

    { "STRING_LIST_DIR", 1L, "dirname",
      libGAP_FuncSTRING_LIST_DIR, "src/streams.c:STRING_LIST_DIR"},

    { "CLOSE_FILE", 1L, "fid",
      libGAP_FuncCLOSE_FILE, "src/streams.c:CLOSE_FILE" },

    { "INPUT_TEXT_FILE", 1L, "filename",
      libGAP_FuncINPUT_TEXT_FILE, "src/streams.c:INPUT_TEXT_FILE" },

    { "OUTPUT_TEXT_FILE", 2L, "filename, append",
      libGAP_FuncOUTPUT_TEXT_FILE, "src/streams.c:OUTPUT_TEXT_FILE" },

    { "IS_END_OF_FILE", 1L, "fid",
      libGAP_FuncIS_END_OF_FILE, "src/streams.c:IS_END_OF_FILE" },

    { "POSITION_FILE", 1L, "fid",
      libGAP_FuncPOSITION_FILE, "src/streams.c:POSITION_FILE" },

    { "READ_BYTE_FILE", 1L, "fid",
      libGAP_FuncREAD_BYTE_FILE, "src/streams.c:READ_BYTE_FILE" },

    { "READ_LINE_FILE", 1L, "fid",
      libGAP_FuncREAD_LINE_FILE, "src/streams.c:READ_LINE_FILE" },

    { "READ_ALL_FILE", 2L, "fid, limit",
      libGAP_FuncREAD_ALL_FILE, "src/streams.c:READ_ALL_FILE" },

    { "SEEK_POSITION_FILE", 2L, "fid, pos",
      libGAP_FuncSEEK_POSITION_FILE, "src/streams.c:SEEK_POSITION_FILE" },

    { "WRITE_BYTE_FILE", 2L, "fid, byte",
      libGAP_FuncWRITE_BYTE_FILE, "src/streams.c:WRITE_BYTE_FILE" },

    { "WRITE_STRING_FILE_NC", 2L, "fid, string",
      libGAP_FuncWRITE_STRING_FILE_NC, "src/streams.c:WRITE_STRING_FILE_NC" },

    { "READ_STRING_FILE", 1L, "fid",
      libGAP_FuncREAD_STRING_FILE, "src/streams.c:READ_STRING_FILE" },

    { "FD_OF_FILE", 1L, "fid",
      libGAP_FuncFD_OF_FILE, "src/streams.c:FD_OF_FILE" },

#ifdef HAVE_SELECT
    { "UNIXSelect", 5L, "inlist, outlist, exclist, timeoutsec, timeoutusec",
      libGAP_FuncUNIXSelect, "src/streams.c:UNIXSelect" },
#endif

    { "ExecuteProcess", 5L, "dir, prg, in, out, args",
      libGAP_FuncExecuteProcess, "src/streams.c:ExecuteProcess" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* file access test functions                                          */
    libGAP_ErrorNumberRNam  = libGAP_RNamName("number");
    libGAP_ErrorMessageRNam = libGAP_RNamName("message");

    /* pick up the number of this global */
    libGAP_LastReadValueGVar = libGAP_GVarName("LastReadValue");

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );


    /* return success                                                      */
    return libGAP_PostRestore( libGAP_module );
}


/****************************************************************************
**
*F  InitInfoStreams() . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "streams" ,                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoStreams ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  streams.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
