/****************************************************************************
**
*W  saveload.c                  GAP source                       Steve Linton
**
**
*Y  Copyright (C)  1997,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions concerned with saving and loading
**  the workspace. There are support functions in gasman.c and elsewhere
**  throughout the kernel
*/
#include        "system.h"              /* system dependent part           */


#include        <unistd.h>              /* write, read                     */
   
#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "bool.h"                /* booleans                        */
#include        "calls.h"               /* generic call mechanism          */
#include        "gap.h"                 /* error handling, initialisation  */
#include        "gvars.h"               /* global variables                */
#include        "streams.h"             /* streams                         */
#include        "string.h"              /* strings                         */
#include        "scanner.h"             /* scanner                         */
#include        "sysfiles.h"            /* file input/output               */
#include        "plist.h"               /* plain lists                     */
#include        "macfloat.h"            /* floating points */
#include        "compstat.h"            /* statically compiled modules     */
#include        "read.h"                /* to call function from library   */

#include        "saveload.h"            /* saving and loading              */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */


/***************************************************************************
**
** Temporary Stuff which will probably be revised to tie in with sysfiles
*/


static libGAP_Int libGAP_SaveFile;
static libGAP_UInt1 libGAP_LoadBuffer[100000];
static libGAP_UInt1* libGAP_LBPointer;
static libGAP_UInt1* libGAP_LBEnd;
static libGAP_Obj libGAP_userHomeExpand;

static libGAP_Int libGAP_OpenForSave( libGAP_Obj fname ) 
{
  if (libGAP_SaveFile != -1)
    {
      libGAP_Pr("Already saving",0L,0L);
      return 1;
    }
  libGAP_SaveFile = libGAP_SyFopen(libGAP_CSTR_STRING(fname), "wb");
  if (libGAP_SaveFile == -1)
    {
      libGAP_Pr("Couldn't open file %s to save workspace",
	 (libGAP_UInt)libGAP_CSTR_STRING(fname),0L);
      return 1;
    }
  libGAP_LBPointer = libGAP_LoadBuffer;
  libGAP_LBEnd = libGAP_LBPointer+sizeof(libGAP_LoadBuffer);
  return 0;
}

static void libGAP_CloseAfterSave( void )
{
  if (libGAP_SaveFile == -1)
    {
      libGAP_Pr("Internal error -- this should never happen",0L,0L);
      libGAP_SyExit(2);
    }

  if (write(libGAP_syBuf[libGAP_SaveFile].fp, libGAP_LoadBuffer, libGAP_LBPointer-libGAP_LoadBuffer) < 0)
    libGAP_ErrorQuit("Cannot write to file descriptor %d, see 'LastSystemError();'\n",
               libGAP_syBuf[libGAP_SaveFile].fp, 0L);
  libGAP_SyFclose(libGAP_SaveFile);
  libGAP_SaveFile = -1;
}

libGAP_Int libGAP_LoadFile = -1;


static void libGAP_OpenForLoad( libGAP_Char *fname ) 
{
  if (libGAP_LoadFile != -1)
    {
      libGAP_Pr("Internal error -- this should never happen\n",0L,0L);
      libGAP_SyExit(2);
    }
  libGAP_LoadFile = libGAP_SyFopen(fname, "rb");
  if (libGAP_LoadFile == -1)
    {
      libGAP_Pr("Couldn't open saved workspace %s\n",(libGAP_Int)fname,0L);
      libGAP_SyExit(1);
    }
}


static void libGAP_CloseAfterLoad( void )
{
  if (!libGAP_LoadFile)
    {
      libGAP_Pr("Internal error -- this should never happen\n",0L,0L);
      libGAP_SyExit(2);
    }
  libGAP_SyFclose(libGAP_LoadFile);
  libGAP_LoadFile = -1;
}

void libGAP_SAVE_BYTE_BUF( void )
{
  if (write(libGAP_syBuf[libGAP_SaveFile].fp, libGAP_LoadBuffer, libGAP_LBEnd-libGAP_LoadBuffer) < 0)
    libGAP_ErrorQuit("Cannot write to file descriptor %d, see 'LastSystemError();'\n",
               libGAP_syBuf[libGAP_SaveFile].fp, 0L);
  libGAP_LBPointer = libGAP_LoadBuffer;
  return;
}

#define libGAP_SAVE_BYTE(byte) {if (libGAP_LBPointer >= libGAP_LBEnd) {libGAP_SAVE_BYTE_BUF();} \
                          *libGAP_LBPointer++ = (libGAP_UInt1)(byte);}

const libGAP_Char * libGAP_LoadByteErrorMessage = "Unexpected End of File in Load\n";

libGAP_UInt1 libGAP_LOAD_BYTE_BUF( void )
{
  libGAP_Int ret;
  ret = read(libGAP_syBuf[libGAP_LoadFile].fp, libGAP_LoadBuffer, 100000);
  if (ret <= 0)
    {
      libGAP_Pr(libGAP_LoadByteErrorMessage, 0L, 0L );
      libGAP_SyExit(2);
    }
  libGAP_LBEnd = libGAP_LoadBuffer + ret;
  libGAP_LBPointer = libGAP_LoadBuffer;
  return *libGAP_LBPointer++;   
}

#define libGAP_LOAD_BYTE()    (libGAP_UInt1)((libGAP_LBPointer >= libGAP_LBEnd) ?\
                                  (libGAP_LOAD_BYTE_BUF()) : (*libGAP_LBPointer++))

/***************************************************************************
**
**  Low level saving routines
*/

void libGAP_SaveUInt1( libGAP_UInt1 data )
{
  libGAP_SAVE_BYTE( data );
}

libGAP_UInt1 libGAP_LoadUInt1( void )
{
  return libGAP_LOAD_BYTE( );
}

void libGAP_SaveUInt2( libGAP_UInt2 data )
{
  libGAP_SAVE_BYTE( (libGAP_UInt1) (data & 0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) (data >> 8) );
}

libGAP_UInt2 libGAP_LoadUInt2 ( void )
{
  libGAP_UInt2 res;
  res = (libGAP_UInt2)libGAP_LOAD_BYTE();
  res |= (libGAP_UInt2)libGAP_LOAD_BYTE()<<8;
  return res;
}

void libGAP_SaveUInt4( libGAP_UInt4 data )
{
  libGAP_SAVE_BYTE( (libGAP_UInt1) (data & 0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 8) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 16) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 24) &0xFF) );
}

libGAP_UInt4 libGAP_LoadUInt4 ( void )
{
  libGAP_UInt4 res;
  res = (libGAP_UInt)libGAP_LOAD_BYTE();
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 8;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 16;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 24;
  return res;
}


#ifdef libGAP_SYS_IS_64_BIT

void libGAP_SaveUInt8( libGAP_UInt8 data )
{
  libGAP_SAVE_BYTE( (libGAP_UInt1) (data & 0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 8) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 16) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 24) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 32) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 40) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 48) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 56) &0xFF) );
}

libGAP_UInt8 libGAP_LoadUInt8 ( void )
{
  libGAP_UInt8 res;
  res = (libGAP_UInt)libGAP_LOAD_BYTE();
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 8;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 16;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 24;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 32;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 40;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 48;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 56;

  return res;
}


void libGAP_SaveUInt( libGAP_UInt data )
{
  libGAP_SAVE_BYTE( (libGAP_UInt1) (data & 0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 8) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 16) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 24) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 32) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 40) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 48) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 56) &0xFF) );
}

libGAP_UInt8 libGAP_LoadUInt ( void )
{
  libGAP_UInt res;
  res = (libGAP_UInt)libGAP_LOAD_BYTE();
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 8;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 16;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 24;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 32;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 40;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 48;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 56;

  return res;
}

#else

void libGAP_SaveUInt( libGAP_UInt data )
{
  libGAP_SAVE_BYTE( (libGAP_UInt1) (data & 0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 8) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 16) &0xFF) );
  libGAP_SAVE_BYTE( (libGAP_UInt1) ((data >> 24) &0xFF) );
}

libGAP_UInt libGAP_LoadUInt ( void )
{
  libGAP_UInt res;
  res = (libGAP_UInt)libGAP_LOAD_BYTE();
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 8;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 16;
  res |= (libGAP_UInt)libGAP_LOAD_BYTE() << 24;
  return res;
}

#endif

void libGAP_SaveCStr( const libGAP_Char * str)
{
  do {
    libGAP_SAVE_BYTE( (libGAP_UInt1) *str);
  } while (*(str++));
}

#include <assert.h>

void libGAP_LoadCStr( libGAP_Char *buf, libGAP_UInt maxsize)
{
  libGAP_UInt nread = 0;
  libGAP_UInt1 c = 1;
  assert(maxsize > 0);
  while (c != '\0' && nread < maxsize )
    {
      c = libGAP_LOAD_BYTE();
      *buf++ = (libGAP_Char) c;
      nread++;
    }
  if (c != '\0')
    {
      libGAP_Pr("Buffer overflow reading workspace\n",0L,0L);
      libGAP_SyExit(1);
    }
}


/****************************************************************************
**
*F  SaveString( <string> )  . . . . . . . . . . . . . . . . . . save a string
**
*/
void libGAP_SaveString ( libGAP_Obj string )
{
  libGAP_UInt i, len = libGAP_GET_LEN_STRING(string);
  libGAP_UInt1 *p = (libGAP_UInt1*)libGAP_CHARS_STRING(string);
  libGAP_SaveUInt(len);
  for (i=0; i<len; i++)
    libGAP_SAVE_BYTE(p[i]);
}

/****************************************************************************
**
*F  LoadString( <string> )
**
*/
void libGAP_LoadString ( libGAP_Obj string )
{
  libGAP_UInt i, len;
  libGAP_UInt1 c;
  libGAP_UInt1 *p = (libGAP_UInt1*)libGAP_CHARS_STRING(string);
  len = libGAP_LoadUInt();
  libGAP_SET_LEN_STRING(string, len);
  for (i=0; i<len; i++) {
    c = libGAP_LOAD_BYTE();
    p[i] = c;
  }
}

void libGAP_SaveSubObj( libGAP_Obj subobj )
{
  if (!subobj)
    libGAP_SaveUInt(0);
  else if (libGAP_IS_INTOBJ(subobj))
    libGAP_SaveUInt((libGAP_UInt) subobj);
  else if (libGAP_IS_FFE(subobj))
    libGAP_SaveUInt((libGAP_UInt) subobj);
  else if ((((libGAP_UInt)subobj & 3) != 0) || 
           subobj < (libGAP_Bag)libGAP_MptrBags || 
           subobj > (libGAP_Bag)libGAP_OldBags ||
           (libGAP_Bag *)libGAP_PTR_BAG(subobj) < libGAP_OldBags)
    {
      libGAP_Pr("#W bad bag id %d found, 0 saved\n", (libGAP_Int)subobj, 0L);
      libGAP_SaveUInt(0);
    }
  else
    libGAP_SaveUInt(((libGAP_UInt)((libGAP_PTR_BAG(subobj))[-1])) << 2);

}

libGAP_Obj libGAP_LoadSubObj( void )
{
  libGAP_UInt word = libGAP_LoadUInt();
  if (word == 0)
    return (libGAP_Obj) 0;
  if ((word & 0x3) == 1 || (word & 0x3) == 2)
    return (libGAP_Obj) word;
  else
    return (libGAP_Obj)(libGAP_MptrBags + (word >> 2)-1);
}

void libGAP_SaveHandler( libGAP_ObjFunc hdlr )
{
  const libGAP_Char * cookie;
  if (hdlr == (libGAP_ObjFunc)0)
    libGAP_SaveCStr("");
  else
    {
      cookie = libGAP_CookieOfHandler(hdlr);
      if (!cookie)
	{
	  libGAP_Pr("No cookie for Handler -- workspace will be corrupt\n",0,0);
	  libGAP_SaveCStr("");
	}
      libGAP_SaveCStr(cookie);
    }
}


libGAP_ObjFunc libGAP_LoadHandler( void )
{
  libGAP_Char buf[256];
  libGAP_LoadCStr(buf, 256);
  if (buf[0] == '\0')
    return (libGAP_ObjFunc) 0;
  else
    return libGAP_HandlerOfCookie(buf);
}


void libGAP_SaveDouble( libGAP_Double d)
{
  libGAP_UInt i;
  libGAP_UInt1 buf[sizeof(libGAP_Double)];
  memcpy((void *) buf, (void *)&d, sizeof(libGAP_Double));
  for (i = 0; i < sizeof(libGAP_Double); i++)
    libGAP_SAVE_BYTE(buf[i]);
}

libGAP_Double libGAP_LoadDouble( void )
{
  libGAP_UInt i;
  libGAP_UInt1 buf[sizeof(libGAP_Double)];
  libGAP_Double d;
  for (i = 0; i < sizeof(libGAP_Double); i++)
    buf[i] = libGAP_LOAD_BYTE();
  memcpy((void *)&d, (void *)buf, sizeof(libGAP_Double));
  return d;
}

/***************************************************************************
**
**  Bag level saving routines
*/



static void libGAP_SaveBagData (libGAP_Bag bag )
{
#ifndef libGAP_USE_NEWSHAPE
  libGAP_SaveUInt((libGAP_UInt)libGAP_PTR_BAG(bag)[-3]);
#endif
  libGAP_SaveUInt((libGAP_UInt)libGAP_PTR_BAG(bag)[-2]);

  /* dispatch */
  (*(libGAP_SaveObjFuncs[ libGAP_TNUM_BAG( bag )]))(bag);

}



static void libGAP_LoadBagData ( void )
{
  libGAP_Bag bag;
  libGAP_UInt size;
  libGAP_UInt type;
  
  /* Recover the size & type */
#ifdef libGAP_USE_NEWSHAPE
  size = libGAP_LoadUInt();
  type = size & 0xFFL;
  size >>= 16;
#else
  type = libGAP_LoadUInt();
  size = libGAP_LoadUInt();
#endif

#ifdef DEBUG_LOADING
  {
    if (libGAP_InfoBags[type].name == NULL)
      {
        libGAP_Pr("Bad type %d, size %d\n",type,size);
        exit(1);
      }
  }
  
#endif  

  /* Get GASMAN to set up the bag for me */
  bag = libGAP_NextBagRestoring( size, type );
  
  /* despatch */
  (*(libGAP_LoadObjFuncs[ type ]))(bag);
  
  return;
}


/***************************************************************************
**

*F  WriteSaveHeader() . . . . .  and utility functions, and loading functions
**
*/

static void libGAP_WriteEndiannessMarker( void )
{
  libGAP_UInt x;
#ifdef libGAP_SYS_IS_64_BIT
  x = 0x0102030405060708L;
#else
  x = 0x01020304L;
#endif
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[0]);
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[1]);
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[2]);
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[3]);
#ifdef libGAP_SYS_IS_64_BIT
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[4]);
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[5]);
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[6]);
  libGAP_SAVE_BYTE(((libGAP_UInt1 *)&x)[7]);
#endif
}

static void libGAP_CheckEndiannessMarker( void )
{
  libGAP_UInt x;
  ((libGAP_UInt1 *)&x)[0] = libGAP_LOAD_BYTE();
  ((libGAP_UInt1 *)&x)[1] = libGAP_LOAD_BYTE();
  ((libGAP_UInt1 *)&x)[2] = libGAP_LOAD_BYTE();
  ((libGAP_UInt1 *)&x)[3] = libGAP_LOAD_BYTE();
#ifdef libGAP_SYS_IS_64_BIT
  ((libGAP_UInt1 *)&x)[4] = libGAP_LOAD_BYTE();
  ((libGAP_UInt1 *)&x)[5] = libGAP_LOAD_BYTE();
  ((libGAP_UInt1 *)&x)[6] = libGAP_LOAD_BYTE();
  ((libGAP_UInt1 *)&x)[7] = libGAP_LOAD_BYTE();
  if (x != 0x0102030405060708L)
#else
  if (x != 0x01020304L)
#endif  
    {
      libGAP_Pr("Saved workspace with incompatible byte order\n",0L,0L);
      libGAP_SyExit(1);
    }
}


/***************************************************************************
**
**  BagStats
*/

static FILE *libGAP_file;

static void libGAP_report( libGAP_Bag bag)
{
  fprintf(libGAP_file,"%li %li\n", (long) libGAP_TNUM_BAG(bag), (long) libGAP_SIZE_BAG(bag));
}

libGAP_Obj libGAP_BagStats(libGAP_Obj self, libGAP_Obj filename)
{
  libGAP_file = fopen((libGAP_Char *)libGAP_CHARS_STRING(filename),"w");
  libGAP_CallbackForAllBags(libGAP_report);
  fclose(libGAP_file);
  return (libGAP_Obj) 0;
}

/***************************************************************************
**
**  Find Bags -- a useful debugging tool -- scan for a bag of specified
**   type and size and return it to the GAP level. Could be a problem
**  if the bag is not a valid GAP object -- eg a local variables bag or
**  a functions body.
*/


static libGAP_UInt libGAP_fb_minsize, libGAP_fb_maxsize, libGAP_fb_tnum;
static libGAP_Bag libGAP_hit;

static void libGAP_ScanBag( libGAP_Bag bag)
{
  if (libGAP_hit == (libGAP_Bag)0 && 
      libGAP_SIZE_BAG(bag) >= libGAP_fb_minsize && 
      libGAP_SIZE_BAG(bag) <= libGAP_fb_maxsize && 
      libGAP_TNUM_BAG(bag) == libGAP_fb_tnum) 
    libGAP_hit = bag;
  return;
}

libGAP_Obj libGAP_FuncFindBag( libGAP_Obj self, libGAP_Obj minsize, libGAP_Obj maxsize, libGAP_Obj tnum )
{
  libGAP_hit = (libGAP_Bag) 0;
  libGAP_fb_minsize = libGAP_INT_INTOBJ(minsize);
  libGAP_fb_maxsize = libGAP_INT_INTOBJ(maxsize);
  libGAP_fb_tnum = libGAP_INT_INTOBJ(tnum);
  libGAP_CallbackForAllBags(libGAP_ScanBag);
  return (libGAP_hit != (libGAP_Bag) 0) ? libGAP_hit : libGAP_Fail;
}


/***************************************************************************
**
*F  SaveWorkspace( <fname> )  . . . . .  save the workspace to the named file
**
**  'SaveWorkspace' is the entry point to the workspace saving. It is not
**  installed as a GAP function, but instead as a keyword, so that we can be
**  sure it is only being called from the top-most prompt level
**  The file saveload.tex in the dev directory describes the saved format
**  in more detail. Most of the work will be done from inside GASMAN, because
**  we need to fiddle with Bag internals somewhat
**
**  The return value is either True or Fail
*/

static libGAP_UInt libGAP_NextSaveIndex = 1;

static void libGAP_AddSaveIndex( libGAP_Bag bag)
{
  libGAP_PTR_BAG(bag)[-1] = (libGAP_Obj)libGAP_NextSaveIndex++;
}

/*  is this obsolete ???    
static void CheckPlist( Bag bag)
{
  if (TNUM_BAG(bag) == 14 && sizeof(UInt)*((UInt)(PTR_BAG(bag)[0])) > SIZE_BAG(bag))
    Pr("Panic %d %d\n",sizeof(UInt)*((UInt)(PTR_BAG(bag)[0])), SIZE_BAG(bag));
  return;
}
*/

static void libGAP_RemoveSaveIndex( libGAP_Bag bag)
{
  libGAP_PTR_BAG(bag)[-1] = bag;
}
static void libGAP_WriteSaveHeader( void )
{
  libGAP_UInt i;
  libGAP_UInt globalcount = 0;
  
  libGAP_SaveCStr("GAP workspace");
  libGAP_SaveCStr(libGAP_SyKernelVersion);

#ifdef libGAP_SYS_IS_64_BIT             
  libGAP_SaveCStr("64 bit");
#else
  libGAP_SaveCStr("32 bit");
#endif

  libGAP_WriteEndiannessMarker();
  
  libGAP_SaveCStr("Counts and Sizes");
  libGAP_SaveUInt(libGAP_NrModules - libGAP_NrBuiltinModules);
  for (i = 0; i < libGAP_GlobalBags.nr; i++)
    if (libGAP_GlobalBags.cookie[i] != NULL)
      globalcount++;
  libGAP_SaveUInt(globalcount);
  libGAP_SaveUInt(libGAP_NextSaveIndex-1);
  libGAP_SaveUInt(libGAP_AllocBags - libGAP_OldBags);
  
  libGAP_SaveCStr("Loaded Modules");

  for ( i = libGAP_NrBuiltinModules; i < libGAP_NrModules; i++)
    {
      libGAP_SaveUInt(libGAP_Modules[i]->type);
      libGAP_SaveUInt(libGAP_Modules[i]->isGapRootRelative);
      libGAP_SaveCStr(libGAP_Modules[i]->filename);
    }

  libGAP_SaveCStr("Kernel to WS refs");
  for (i = 0; i < libGAP_GlobalBags.nr; i++)
    {
      if (libGAP_GlobalBags.cookie[i] != NULL)
	{
	  libGAP_SaveCStr((const libGAP_Char *)libGAP_GlobalBags.cookie[i]);
	  libGAP_SaveSubObj(*(libGAP_GlobalBags.addr[i]));
	}
    }
}


libGAP_Obj libGAP_SaveWorkspace( libGAP_Obj fname )
{

  libGAP_Int i;
  libGAP_Obj fullname;

  if (!libGAP_IsStringConv(fname))
    libGAP_ErrorQuit("usage: SaveWorkspace( <filename> )",0,0);
  /* maybe expand fname starting with ~/...   */
  fullname = libGAP_Call1ArgsInNewReader(libGAP_userHomeExpand, fname);
  
  for (i = 0; i < libGAP_NrModules; i++)
    if (libGAP_Modules[i]->preSave != NULL &&
        (*(libGAP_Modules[i]->preSave))(libGAP_Modules[i]))
      {
        libGAP_Pr("Failed to save workspace -- problem reported in %s\n",
           (libGAP_Int)libGAP_Modules[i]->name, 0L);
        for ( i--; i >= 0; i--)
          (*(libGAP_Modules[i]->postSave))(libGAP_Modules[i]);
        return libGAP_Fail;
      }

  /* Do a full garbage collection */
  libGAP_CollectBags( 0, 1);
  
  /* Add indices in link words of all bags, for saving inter-bag references */
  libGAP_NextSaveIndex = 1;
  libGAP_CallbackForAllBags( libGAP_AddSaveIndex );

  /* Now do the work */
  if (!libGAP_OpenForSave( fullname ))
    {
      libGAP_WriteSaveHeader();
      libGAP_SaveCStr("Bag data");
      libGAP_SortHandlers( 1 ); /* Sort by address to speed up CookieOfHandler */
      libGAP_CallbackForAllBags( libGAP_SaveBagData );
      libGAP_CloseAfterSave();
    }
      
  /* Finally, reset all the link words */
  libGAP_CallbackForAllBags( libGAP_RemoveSaveIndex );
  
  /* Restore situation by calling all post-save methods */
  for (i = 0; i < libGAP_NrModules; i++)
    if (libGAP_Modules[i]->postSave != NULL)
      (*(libGAP_Modules[i]->postSave))(libGAP_Modules[i]);

  return libGAP_True;
}


libGAP_Obj libGAP_FuncSaveWorkspace(libGAP_Obj self, libGAP_Obj filename )
{
  return libGAP_SaveWorkspace( filename );
}


/***************************************************************************
**
*F  LoadWorkspace( <fname> )  . . . . .  load the workspace to the named file
**
**  'LoadWorkspace' is the entry point to the workspace saving. It is not
**  installed as a GAP function, but instead called from InitGap when the
**  -L commad-line flag is given
**
**  The file saveload.tex in the dev directory describes the saved format
**  in more detail. Most of the work will be done from inside GASMAN, because
**  we need to fiddle with Bag internals somewhat
**
*/


void libGAP_LoadWorkspace( libGAP_Char * fname )
{
  libGAP_UInt nMods, nGlobs, nBags, i, maxSize, isGapRootRelative;
  libGAP_UInt globalcount = 0;
  libGAP_Char buf[256];
  libGAP_Obj * glob;

  /* Open saved workspace  */
  libGAP_OpenForLoad( fname );

  /* Check file header */

  libGAP_LoadCStr(buf,256);
  if (strncmp (buf, "GAP ", 4) != 0) {
     libGAP_Pr("File %s does not appear to be a GAP workspae.\n", (long) fname, 0L);
     libGAP_SyExit(1);
  }

  if (strcmp (buf, "GAP workspace") == 0) {

     libGAP_LoadCStr(buf,256);
     if (strcmp (buf, libGAP_SyKernelVersion) != 0) {
        libGAP_Pr("This workspace is not compatible with GAP kernel (%s, present: %s).\n", 
           (long)buf, (long)libGAP_SyKernelVersion);
        libGAP_SyExit(1);
     }

     libGAP_LoadCStr(buf,256);
#ifdef libGAP_SYS_IS_64_BIT             
     if (strcmp(buf,"64 bit") != 0)
#else
     if (strcmp(buf,"32 bit") != 0)
#endif
        {
           libGAP_Pr("This workspace was created by a %s version of GAP.\n", (long)buf, 0L);
           libGAP_SyExit(1);
        }
  } else {
     
     /* try if it is an old workspace */

#ifdef libGAP_SYS_IS_64_BIT             
     if (strcmp(buf,"GAP 4.0 beta 64 bit") != 0)
#else 
     if (strcmp(buf,"GAP 4.0 beta 32 bit") != 0)
#endif
        libGAP_Pr("File %s probably isn't a GAP workspace.\n", (long)fname, 0L);
     else 
        libGAP_Pr("This workspace was created by an old version of GAP.\n", 0L, 0L);
     libGAP_SyExit(1);
  } 
  
  libGAP_CheckEndiannessMarker();
  
  libGAP_LoadCStr(buf,256);
  if (strcmp(buf,"Counts and Sizes") != 0)
    {
      libGAP_Pr("Bad divider\n",0L,0L);
      libGAP_SyExit(1);
    }
  
  nMods = libGAP_LoadUInt();
  nGlobs = libGAP_LoadUInt();
  nBags = libGAP_LoadUInt();
  maxSize = libGAP_LoadUInt();

  /* Make sure there is enough room, and signal GASMAN that
     we are starting a restore */
  libGAP_StartRestoringBags(nBags, maxSize);

  /* The restoring kernel must have at least as many compiled modules
     as the saving one. */
  libGAP_LoadCStr(buf,256);
  if (strcmp(buf,"Loaded Modules") != 0)
    {
      libGAP_Pr("Bad divider\n",0L,0L);
      libGAP_SyExit(1);
    }

  for (i = 0; i < nMods; i++)
    {
      libGAP_UInt type = libGAP_LoadUInt();
      isGapRootRelative = libGAP_LoadUInt();
      libGAP_LoadCStr(buf,256);
      if (isGapRootRelative)
        libGAP_READ_GAP_ROOT( buf);
      else
	{
	  libGAP_StructInitInfo *info = NULL;
 	  /* Search for user module static case first */
	  if (type == libGAP_MODULE_STATIC) { 
	    libGAP_UInt k;
	    for ( k = 0;  libGAP_CompInitFuncs[k];  k++ ) {
	      info = (*(libGAP_CompInitFuncs[k]))();
	      if ( info == 0 ) {
		continue;
	      }
	      if ( ! strcmp( buf, info->name ) ) {
		break;
	      }
	    }
	    if ( libGAP_CompInitFuncs[k] == 0 ) {
	      libGAP_Pr( "Static module %s not found in loading kernel\n",
		  (libGAP_Int)buf, 0L );
	      libGAP_SyExit(1);
	    }
	
	  } else {
	    /* and dynamic case */
	    libGAP_InitInfoFunc init; 
	
	    init = libGAP_SyLoadModule(buf);
	    
	    if ((libGAP_Int)init == 1 || (libGAP_Int) init == 3 || (libGAP_Int) init == 5 || (libGAP_Int) init == 7)
	      {
		libGAP_Pr("Failed to load needed dynamic module %s, error code %d\n",
		   (libGAP_Int)buf, (libGAP_Int) init);
		libGAP_SyExit(1);
	      }
	    info = (*init)();
	     if (info == 0 )
	       {
		libGAP_Pr("Failed to init needed dynamic module %s, error code %d\n",
		   (libGAP_Int)buf, (libGAP_Int) info);
		libGAP_SyExit(1);
	       }
	  }
	/* link and init me                                                    */
	info->isGapRootRelative = 0;
	(info->initKernel)(info);
	libGAP_RecordLoadedModule(info, buf);
      }
      
    }

  /* Now the kernel variables that point into the workspace */
  libGAP_LoadCStr(buf,256);
  if (strcmp(buf,"Kernel to WS refs") != 0)
    {
      libGAP_Pr("Bad divider\n",0L,0L);
       libGAP_SyExit(1);
    }
  libGAP_SortGlobals(2);               /* globals by cookie for quick
                                 lookup */
  for (i = 0; i < libGAP_GlobalBags.nr; i++)
    {
      if (libGAP_GlobalBags.cookie[i] != NULL)
	globalcount++;
      else
	*(libGAP_GlobalBags.addr[i]) = (libGAP_Bag) 0;
    }
  if (nGlobs != globalcount)
    {
      libGAP_Pr("Wrong number of global bags in saved workspace %d %d\n",
         nGlobs, globalcount);
      libGAP_SyExit(1);
    }
  for (i = 0; i < globalcount; i++)
    {
      libGAP_LoadCStr(buf,256);
      glob = libGAP_GlobalByCookie(buf);
      if (glob == (libGAP_Obj *)0)
        {
          libGAP_Pr("Global object cookie from workspace not found in kernel %s\n",
             (libGAP_Int)buf,0L);
          libGAP_SyExit(1);
        }
      *glob = libGAP_LoadSubObj();
#ifdef DEBUG_LOADING
      libGAP_Pr("Restored global %s\n",(libGAP_Int)buf,0L);
#endif
    }

  libGAP_LoadCStr(buf,256);
  if (strcmp(buf,"Bag data") != 0)
    {
      libGAP_Pr("Bad divider\n",0L,0L);
      libGAP_SyExit(1);
    }
  
  libGAP_SortHandlers(2);
  for (i = 0; i < nBags; i++)
    libGAP_LoadBagData();

  libGAP_FinishedRestoringBags();

  /* Post restore methods are called elsewhere */
  
  libGAP_CloseAfterLoad();
}

#include        "finfield.h"            /* finite fields and ff elements   */

static void libGAP_PrSavedObj( libGAP_UInt x)
{
  if ((x & 3) == 1)
    libGAP_Pr("Immediate  integer %d\n", libGAP_INT_INTOBJ((libGAP_Obj)x),0L);
  else if ((x & 3) == 2)
    libGAP_Pr("Immediate FFE %d %d\n", libGAP_VAL_FFE(x), libGAP_SIZE_FF(libGAP_FLD_FFE(x)));
  else
    libGAP_Pr("Reference to bag number %d\n",x>>2,0L);
}

libGAP_Obj libGAP_FuncDumpWorkspace( libGAP_Obj self, libGAP_Obj fname )
{
  libGAP_UInt nMods, nGlobs, nBags, i, relative;
  libGAP_Char buf[256];
  libGAP_OpenForLoad( libGAP_CSTR_STRING(fname) );
  libGAP_LoadCStr(buf,256);
  libGAP_Pr("Header string: %s\n",(libGAP_Int) buf, 0L);
  libGAP_LoadCStr(buf,256);
  libGAP_Pr("GAP Version: %s\n",(libGAP_Int)buf, 0L);
  libGAP_LoadCStr(buf,256);
  libGAP_Pr("Word length: %s\n",(libGAP_Int)buf, 0L);
  libGAP_CheckEndiannessMarker();
  libGAP_LoadCStr(buf,256);
  libGAP_Pr("Divider string: %s\n",(libGAP_Int)buf,0L);
  if (strcmp(buf,"Counts and Sizes") != 0)
    libGAP_ErrorQuit("Bad divider",0L,0L);
  libGAP_Pr("Loaded modules: %d\n",nMods = libGAP_LoadUInt(), 0L);
  libGAP_Pr("Global Bags   : %d\n",nGlobs = libGAP_LoadUInt(), 0L);
  libGAP_Pr("Total Bags    : %d\n",nBags = libGAP_LoadUInt(), 0L);
  libGAP_Pr("Maximum Size  : %d\n",sizeof(libGAP_Bag)*libGAP_LoadUInt(), 0L);
  libGAP_LoadCStr(buf,256);
  libGAP_Pr("Divider string: %s\n",(libGAP_Int)buf, 0L);
  if (strcmp(buf,"Loaded Modules") != 0)
    libGAP_ErrorQuit("Bad divider",0L,0L);
  for (i = 0; i < nMods; i++)
    {
      libGAP_UInt type;
      type = libGAP_LoadUInt();
      libGAP_Pr("Type: %d ",type,0);
      relative = libGAP_LoadUInt();
      if (relative)
	libGAP_Pr("GAP root relative ", 0L, 0L);
      else
	libGAP_Pr("absolute ", 0L, 0L);
      libGAP_LoadCStr(buf,256);
      libGAP_Pr("  %s\n",(libGAP_Int)buf,0L);
    }
  libGAP_LoadCStr(buf,256);
  libGAP_Pr("Divider string: %s\n",(libGAP_Int)buf,0L);
  if (strcmp(buf,"Kernel to WS refs") != 0)
    libGAP_ErrorQuit("Bad divider",0L,0L);
  for (i = 0; i < nGlobs; i++)
    {
      libGAP_LoadCStr(buf,256);
      libGAP_Pr("  %s ",(libGAP_Int)buf,0L);
      libGAP_PrSavedObj(libGAP_LoadUInt());
    }
  libGAP_LoadCStr(buf,256);
  libGAP_Pr("Divider string: %s\n",(libGAP_Int)buf,0L);
  if (strcmp(buf,"Bag data") != 0)
    libGAP_ErrorQuit("Bad divider",0L,0L);
  libGAP_CloseAfterLoad();
  return (libGAP_Obj) 0;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "SaveWorkspace", 1, "fname", 
      libGAP_FuncSaveWorkspace, "src/saveload.c:SaveWorkspace" },

    { "DumpWorkspace", 1, "fname", 
      libGAP_FuncDumpWorkspace, "src/saveload.c:DumpWorkspace" },

    { "FindBag", 3, "minsize, maxsize, tnum", 
      libGAP_FuncFindBag, "src/saveload.c:FindBag" },

    { "BagStats", 1, "filename", 
      libGAP_BagStats, "src/saveload.c:BagStats" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_SaveFile = -1;
    libGAP_LBPointer = libGAP_LoadBuffer;
    libGAP_LBEnd = libGAP_LoadBuffer;
  
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );
    /* allow ~/... expansion in SaveWorkspace                              */ 
    libGAP_ImportFuncFromLibrary("USER_HOME_EXPAND", &libGAP_userHomeExpand);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* Create dummy variable, to support tab-completion */
    (void)libGAP_GVarName("SaveWorkspace");

    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoSaveLoad()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "saveload",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoSaveLoad ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  saveload.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
