/****************************************************************************
**
*W  records.c                   GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the generic record package.
**
**  This package  provides a uniform  interface to  the functions that access
**  records and the elements for the other packages in the GAP kernel.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "records.h"             /* generic records                 */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */


/****************************************************************************
**

*F  CountRnam . . . . . . . . . . . . . . . . . . . .  number of record names
**
**  'CountRnam' is the number of record names.
*/
libGAP_UInt            libGAP_CountRNam;


/****************************************************************************
**
*F  NAME_RNAM(<rnam>) . . . . . . . . . . . . . . . .  name for a record name
**
**  'NAME_RNAM' returns the name (as a C string) for the record name <rnam>.
**
**  Note that 'NAME_RNAM' is a  macro, so do not call  it with arguments that
**  have side effects.
**
**  'NAME_RNAM' is defined in the declaration part of this package as follows
**
#define NAME_RNAM(rnam) CSTR_STRING( ELM_PLIST( NamesRNam, rnam ) )
*/
libGAP_Obj             libGAP_NamesRNam;


/****************************************************************************
**
*F  RNamName(<name>)  . . . . . . . . . . . . convert a name to a record name
**
**  'RNamName' returns  the record name with the  name  <name> (which is  a C
**  string).
*/
libGAP_Obj             libGAP_HashRNam;

libGAP_UInt            libGAP_SizeRNam;

libGAP_UInt            libGAP_RNamName (
    const libGAP_Char *        name )
{
    libGAP_Obj                 rnam;           /* record name (as imm intobj)     */
    libGAP_UInt                pos;            /* hash position                   */
    libGAP_UInt                len;            /* length of name                  */
    libGAP_Char                namx [1024];    /* temporary copy of <name>        */
    libGAP_Obj                 string;         /* temporary string object <name>  */
    libGAP_Obj                 table;          /* temporary copy of <HashRNam>    */
    libGAP_Obj                 rnam2;          /* one element of <table>          */
    const libGAP_Char *        p;              /* loop variable                   */
    libGAP_UInt                i;              /* loop variable                   */

    /* start looking in the table at the following hash position           */
    pos = 0;
    len = 0;
    for ( p = name; *p != '\0'; p++ ) {
        pos = 65599 * pos + *p;
        len++;
    }
    pos = (pos % libGAP_SizeRNam) + 1;

    if(len >= 1023) {
        // Note: We can't pass 'name' here, as it might get moved by garbage collection
        libGAP_ErrorQuit("Record names must consist of less than 1023 characters", 0, 0);
    }
    /* look through the table until we find a free slot or the global      */
    while ( (rnam = libGAP_ELM_PLIST( libGAP_HashRNam, pos )) != 0
         && strncmp( libGAP_NAME_RNAM( libGAP_INT_INTOBJ(rnam) ), name, 1023 ) ) {
        pos = (pos % libGAP_SizeRNam) + 1;
    }

    /* if we did not find the global variable, make a new one and enter it */
    /* (copy the name first, to avoid a stale pointer in case of a GC)     */
    if ( rnam == 0 ) {
        libGAP_CountRNam++;
        rnam = libGAP_INTOBJ_INT(libGAP_CountRNam);
        libGAP_SET_ELM_PLIST( libGAP_HashRNam, pos, rnam );
        libGAP_strlcpy( namx, name, sizeof(namx) );
        libGAP_C_NEW_STRING_DYN(string, namx);
        libGAP_GROW_PLIST(    libGAP_NamesRNam,   libGAP_CountRNam );
        libGAP_SET_LEN_PLIST( libGAP_NamesRNam,   libGAP_CountRNam );
        libGAP_SET_ELM_PLIST( libGAP_NamesRNam,   libGAP_CountRNam, string );
        libGAP_CHANGED_BAG(   libGAP_NamesRNam );
    }

    /* if the table is too crowed, make a larger one, rehash the names     */
    if ( libGAP_SizeRNam < 3 * libGAP_CountRNam / 2 ) {
        table = libGAP_HashRNam;
        libGAP_SizeRNam = 2 * libGAP_SizeRNam + 1;
        libGAP_HashRNam = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_SizeRNam );
        libGAP_SET_LEN_PLIST( libGAP_HashRNam, libGAP_SizeRNam );
        for ( i = 1; i <= (libGAP_SizeRNam-1)/2; i++ ) {
            rnam2 = libGAP_ELM_PLIST( table, i );
            if ( rnam2 == 0 )  continue;
            pos = 0;
            for ( p = libGAP_NAME_RNAM( libGAP_INT_INTOBJ(rnam2) ); *p != '\0'; p++ ) {
                pos = 65599 * pos + *p;
            }
            pos = (pos % libGAP_SizeRNam) + 1;
            while ( libGAP_ELM_PLIST( libGAP_HashRNam, pos ) != 0 ) {
                pos = (pos % libGAP_SizeRNam) + 1;
            }
            libGAP_SET_ELM_PLIST( libGAP_HashRNam, pos, rnam2 );
        }
    }

    /* return the record name                                              */
    return libGAP_INT_INTOBJ(rnam);
}


/****************************************************************************
**
*F  RNamIntg(<intg>)  . . . . . . . . . . convert an integer to a record name
**
**  'RNamIntg' returns the record name corresponding to the integer <intg>.
*/
libGAP_UInt            libGAP_RNamIntg (
    libGAP_Int                 intg )
{
    libGAP_Char                name [32];      /* integer converted to a string   */
    libGAP_Char *              p;              /* loop variable                   */
    libGAP_UInt negative;

    /* convert the integer to a string                                     */
    p = name + sizeof(name);  *--p = '\0';
    negative = (intg < 0);
    if ( negative ) {
        intg = -intg;
    }
   
    do {
        *--p = '0' + intg % 10;
    } while ( (intg /= 10) != 0 );
    if( negative ) {
        *--p = '-';
    }

    /* return the name                                                     */
    return libGAP_RNamName( p );
}


/****************************************************************************
**
*F  RNamObj(<obj>)  . . . . . . . . . . .  convert an object to a record name
**
**  'RNamObj' returns the record name  corresponding  to  the  object  <obj>,
**  which currently must be a string or an integer.
*/
libGAP_UInt            libGAP_RNamObj (
    libGAP_Obj                 obj )
{
    /* convert integer object                                              */
    if ( libGAP_IS_INTOBJ(obj) ) {
        return libGAP_RNamIntg( libGAP_INT_INTOBJ(obj) );
    }

    /* convert string object (empty string may have type T_PLIST)          */
    else if ( libGAP_IsStringConv(obj) && libGAP_IS_STRING_REP(obj) ) {
        return libGAP_RNamName( libGAP_CSTR_STRING(obj) );
    }

    /* otherwise fail                                                      */
    else {
        obj = libGAP_ErrorReturnObj(
            "Record: '<rec>.(<obj>)' <obj> must be a string or an integer",
            0L, 0L,
            "you can replace <obj> via 'return <obj>;'" );
        return libGAP_RNamObj( obj );
    }
}


/****************************************************************************
**
*F  RNamObjHandler(<self>,<obj>)  . . . .  convert an object to a record name
**
**  'RNamObjHandler' implements the internal function 'RNamObj'.
**
**  'RNamObj( <obj> )'
**
**  'RNamObj' returns the record name  corresponding  to  the  object  <obj>,
**  which currently must be a string or an integer.
*/
libGAP_Obj             libGAP_RNamObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_INTOBJ_INT( libGAP_RNamObj( obj ) );
}


/****************************************************************************
**
*F  NameRNamHandler(<self>,<rnam>)  . . . . convert a record name to a string
**
**  'NameRNamHandler' implements the internal function 'NameRName'.
**
**  'NameRName( <rnam> )'
**
**  'NameRName' returns the string corresponding to the record name <rnam>.
*/
libGAP_Obj             libGAP_NameRNamFunc;

libGAP_Obj             libGAP_NameRNamHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 rnam )
{
    libGAP_Obj                 name;
    libGAP_Obj                 oname;
    while ( ! libGAP_IS_INTOBJ(rnam)
         || libGAP_INT_INTOBJ(rnam) <= 0
        || libGAP_CountRNam < libGAP_INT_INTOBJ(rnam) ) {
        rnam = libGAP_ErrorReturnObj(
            "NameRName: <rnam> must be a record name (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(rnam), 0L,
            "you can replace <rnam> via 'return <rnam>;'" );
    }
    oname = libGAP_NAME_OBJ_RNAM( libGAP_INT_INTOBJ(rnam) );
    name = libGAP_CopyToStringRep(oname);
    return name;
}


/****************************************************************************
**
*F  IS_REC(<obj>) . . . . . . . . . . . . . . . . . . . is an object a record
*V  IsRecFuncs[<type>]  . . . . . . . . . . . . . . . . table of record tests
**
**  'IS_REC' returns a nonzero value if the object <obj> is a  record  and  0
**  otherwise.
**
**  Note that 'IS_REC' is a macro, so do not call  it  with  arguments  that
**  have side effects.
**
**  'IS_REC' is defined in the declaration part of this package as follows
**
#define IS_REC(obj)     ((*IsRecFuncs[ TNUM_OBJ(obj) ])( obj ))
*/
libGAP_Int             (*libGAP_IsRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj             libGAP_IsRecFilt;

libGAP_Obj             libGAP_IsRecHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_REC(obj) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsRecNot (
    libGAP_Obj                 obj )
{
    return 0L;
}

libGAP_Int             libGAP_IsRecYes (
    libGAP_Obj                 obj )
{
    return 1L;
}

libGAP_Int             libGAP_IsRecObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsRecFilt, obj ) == libGAP_True);
}


/****************************************************************************
**
*F  ELM_REC(<rec>,<rnam>) . . . . . . . . . . select an element from a record
**
**  'ELM_REC' returns the element, i.e., the value of the component, with the
**  record name <rnam> in  the record <rec>.   An error is signalled if <rec>
**  is not a record or if <rec> has no component with the record name <rnam>.
**
**  Note that 'ELM_REC' is  a macro, so do   not call it with arguments  that
**  have side effects.
**
**  'ELM_REC' is defined in the declaration part of this package as follows
**
#define ELM_REC(rec,rnam) \
                        ((*ElmRecFuncs[ TNUM_OBJ(rec) ])( rec, rnam ))
*/
libGAP_Obj             (*libGAP_ElmRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam );

libGAP_Obj             libGAP_ElmRecOper;

libGAP_Obj             libGAP_ElmRecHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 rec,
    libGAP_Obj                 rnam )
{
    return libGAP_ELM_REC( rec, libGAP_INT_INTOBJ(rnam) );
}

libGAP_Obj             libGAP_ElmRecError (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    rec = libGAP_ErrorReturnObj(
        "Record Element: <rec> must be a record (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(rec), 0L,
        "you can replace <rec> via 'return <rec>;'" );
    return libGAP_ELM_REC( rec, rnam );
}

libGAP_Obj             libGAP_ElmRecObject (
    libGAP_Obj                 obj,
    libGAP_UInt                rnam )
{
  libGAP_Obj elm;
  elm = libGAP_DoOperation2Args( libGAP_ElmRecOper, obj, libGAP_INTOBJ_INT(rnam) );
  while (elm == 0)
    elm =  libGAP_ErrorReturnObj("Record access method must return a value",0L,0L,
                          "you can supply a value <val> via 'return <val>;'");
  return elm;

}


/****************************************************************************
**
*F  ISB_REC(<rec>,<rnam>) . . . . . . . . . test for an element from a record
**
**  'ISB_REC' returns 1 if the record <rec> has a component with  the  record
**  name <rnam> and 0 otherwise.  An error is signalled if  <rec>  is  not  a
**  record.
**
**  Note  that 'ISB_REC'  is a macro,  so do not call  it with arguments that
**  have side effects.
**
**  'ISB_REC' is defined in the declaration part of this package as follows
**
#define ISB_REC(rec,rnam) \
                        ((*IsbRecFuncs[ TNUM_OBJ(rec) ])( rec, rnam ))
*/
libGAP_Int             (*libGAP_IsbRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam );

libGAP_Obj             libGAP_IsbRecOper;

libGAP_Obj             libGAP_IsbRecHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 rec,
    libGAP_Obj                 rnam )
{
    return (libGAP_ISB_REC( rec, libGAP_INT_INTOBJ(rnam) ) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsbRecError (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    rec = libGAP_ErrorReturnObj(
        "IsBound: <rec> must be a record (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(rec), 0L,
        "you can replace <rec> via 'return <rec>;'" );
    return libGAP_ISB_REC( rec, rnam );
}

libGAP_Int             libGAP_IsbRecObject (
    libGAP_Obj                 obj,
    libGAP_UInt                rnam )
{
    return (libGAP_DoOperation2Args( libGAP_IsbRecOper, obj, libGAP_INTOBJ_INT(rnam) ) == libGAP_True);
}


/****************************************************************************
**
*F  ASS_REC(<rec>,<rnam>,<obj>) . . . . . . . . . . . . .  assign to a record
**
**  'ASS_REC' assigns the object <obj>  to  the  record  component  with  the
**  record name <rnam> in the record <rec>.  An error is signalled  if  <rec>
**  is not a record.
**
**  'ASS_REC' is defined in the declaration part of this package as follows
**
#define ASS_REC(rec,rnam,obj) \
                        ((*AssRecFuncs[ TNUM_OBJ(rec) ])( rec, rnam, obj ))
*/
void            (*libGAP_AssRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam, libGAP_Obj obj );

libGAP_Obj             libGAP_AssRecOper;

libGAP_Obj             libGAP_AssRecHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 rec,
    libGAP_Obj                 rnam,
    libGAP_Obj                 obj )
{
    libGAP_ASS_REC( rec, libGAP_INT_INTOBJ(rnam), obj );
    return 0;
}

void            libGAP_AssRecError (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam,
    libGAP_Obj                 obj )
{
    rec = libGAP_ErrorReturnObj(
        "Record Assignment: <rec> must be a record (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(rec), 0L,
        "you can replace <rec> via 'return <rec>;'" );
    libGAP_ASS_REC( rec, rnam, obj );
}

void            libGAP_AssRecObject (
    libGAP_Obj                 obj,
    libGAP_UInt                rnam,
    libGAP_Obj                 val )
{
    libGAP_DoOperation3Args( libGAP_AssRecOper, obj, libGAP_INTOBJ_INT(rnam), val );
}


/****************************************************************************
**
*F  UNB_REC(<rec>,<rnam>) . . . . . . unbind a record component from a record
**
**  'UNB_REC' removes the record component  with the record name <rnam>  from
**  the record <rec>.
**
**  Note that 'UNB_REC' is  a macro, so  do  not call it with  arguments that
**  have side effects.
**
**  'UNB_REC' is defined in the declaration part of this package as follows
**
#define UNB_REC(rec,rnam) \
                        ((*UnbRecFuncs[ TNUM_OBJ(rec) ])( rec, rnam ))
*/
void            (*libGAP_UnbRecFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj rec, libGAP_UInt rnam );

libGAP_Obj             libGAP_UnbRecOper;

libGAP_Obj             libGAP_UnbRecHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 rec,
    libGAP_Obj                 rnam )
{
    libGAP_UNB_REC( rec, libGAP_INT_INTOBJ(rnam) );
    return 0;
}

void            libGAP_UnbRecError (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    rec = libGAP_ErrorReturnObj(
        "Unbind: <rec> must be a record (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(rec), 0L,
        "you can replace <rec> via 'return <rec>;'" );
    libGAP_UNB_REC( rec, rnam );
}
        
void            libGAP_UnbRecObject (
    libGAP_Obj                 obj,
    libGAP_UInt                rnam )
{
    libGAP_DoOperation2Args( libGAP_UnbRecOper, obj, libGAP_INTOBJ_INT(rnam) );
}


/****************************************************************************
**
*F  iscomplete( <name>, <len> ) . . . . . . . .  find the completions of name
*F  completion( <name>, <len> ) . . . . . . . .  find the completions of name
*/
libGAP_UInt            libGAP_iscomplete_rnam (
    libGAP_Char *              name,
    libGAP_UInt                len )
{
    libGAP_Char *              curr;
    libGAP_UInt                i, k;

    for ( i = 1; i <= libGAP_CountRNam; i++ ) {
        curr = libGAP_NAME_RNAM( i );
        for ( k = 0; name[k] != 0 && curr[k] == name[k]; k++ ) ;
        if ( k == len && curr[k] == '\0' )  return 1;
    }
    return 0;
}

libGAP_UInt            libGAP_completion_rnam (
    libGAP_Char *              name,
    libGAP_UInt                len )
{
    libGAP_Char *              curr;
    libGAP_Char *              next;
    libGAP_UInt                i, k;

    next = 0;
    for ( i = 1; i <= libGAP_CountRNam; i++ ) {
        curr = libGAP_NAME_RNAM( i );
        for ( k = 0; name[k] != 0 && curr[k] == name[k]; k++ ) ;
        if ( k < len || curr[k] <= name[k] )  continue;
        if ( next != 0 ) {
            for ( k = 0; curr[k] != '\0' && curr[k] == next[k]; k++ ) ;
            if ( k < len || next[k] < curr[k] )  continue;
        }
        next = curr;
    }

    if ( next != 0 ) {
        for ( k = 0; next[k] != '\0'; k++ )
            name[k] = next[k];
        name[k] = '\0';
    }

    return next != 0;
}

libGAP_Obj libGAP_FuncALL_RNAMES (
    libGAP_Obj                 self )
{
    libGAP_Obj                 copy, s;
    libGAP_UInt                i;
    libGAP_Obj                 name;

    copy = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, libGAP_CountRNam );
    for ( i = 1;  i <= libGAP_CountRNam;  i++ ) {
        name = libGAP_NAME_OBJ_RNAM( i );
        s = libGAP_CopyToStringRep(name);
        libGAP_SET_ELM_PLIST( copy, i, s );
    }
    libGAP_SET_LEN_PLIST( copy, libGAP_CountRNam );
    return copy;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_REC", "obj", &libGAP_IsRecFilt,
      libGAP_IsRecHandler, "src/records.c:IS_REC" },

    { 0 }

};


/****************************************************************************
**
*V  GVarOpers . . . . . . . . . . . . . . . . .  list of operations to export
*/
static libGAP_StructGVarOper libGAP_GVarOpers [] = {

    { "ELM_REC",  2, "obj, rnam", &libGAP_ElmRecOper, 
      libGAP_ElmRecHandler, "src/records.c:ELM_REC" },

    { "ISB_REC",  2, "obj, rnam", &libGAP_IsbRecOper, 
      libGAP_IsbRecHandler, "src/records.c:ISB_REC" },

    { "ASS_REC",  3, "obj, rnam, val", &libGAP_AssRecOper, 
      libGAP_AssRecHandler, "src/records.c:ASS_REC" },

    { "UNB_REC",  2, "obj, rnam", &libGAP_UnbRecOper, 
      libGAP_UnbRecHandler, "src/records.c:UNB_REC" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "RNamObj", 1, "obj",
      libGAP_RNamObjHandler, "src/records.c:RNamObj" },

    { "NameRNam", 1, "rnam",
      libGAP_NameRNamHandler, "src/records.c:NameRNam" },

    { "ALL_RNAMES", 0, "",
      libGAP_FuncALL_RNAMES, "src/records.c:ALL_RNAMES" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                type;           /* loop variable                   */

    /* make the list of names of record names                              */
    libGAP_InitGlobalBag( &libGAP_NamesRNam, "src/records.c:NamesRNam" );

    /* make the hash list of record names                                  */
    libGAP_InitGlobalBag( &libGAP_HashRNam, "src/records.c:HashRNam" );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrOpersFromTable( libGAP_GVarOpers );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* make and install the 'IS_REC' filter                                */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsRecFuncs[ type ] = libGAP_IsRecNot;
    }
    for ( type = libGAP_FIRST_RECORD_TNUM; type <= libGAP_LAST_RECORD_TNUM; type++ ) {
        libGAP_IsRecFuncs[ type ] = libGAP_IsRecYes;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsRecFuncs[ type ] = libGAP_IsRecObject;
    }


    /* make and install the 'ELM_REC' operations                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_ElmRecFuncs[ type ] = libGAP_ElmRecError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_ElmRecFuncs[ type ] = libGAP_ElmRecObject;
    }


    /* make and install the 'ISB_REC' operation                            */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsbRecFuncs[ type ] = libGAP_IsbRecError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsbRecFuncs[ type ] = libGAP_IsbRecObject;
    }


    /* make and install the 'ASS_REC' operation                            */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_AssRecFuncs[ type ] = libGAP_AssRecError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_AssRecFuncs[ type ] = libGAP_AssRecObject;
    }


    /* make and install the 'UNB_REC' operation                            */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_UnbRecFuncs[ type ] = libGAP_UnbRecError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_UnbRecFuncs[ type ] = libGAP_UnbRecObject;
    }

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* make the list of names of record names                              */
    libGAP_CountRNam = libGAP_LEN_PLIST(libGAP_NamesRNam);

    /* make the hash list of record names                                  */
    libGAP_SizeRNam = libGAP_LEN_PLIST(libGAP_HashRNam);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* make the list of names of record names                              */
    libGAP_CountRNam = 0;
    libGAP_NamesRNam = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_MakeBagPublic(libGAP_NamesRNam);
    libGAP_SET_LEN_PLIST( libGAP_NamesRNam, 0 );

    /* make the hash list of record names                                  */
    libGAP_SizeRNam = 997;
    libGAP_HashRNam = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_SizeRNam );
    libGAP_MakeBagPublic(libGAP_HashRNam);
    libGAP_SET_LEN_PLIST( libGAP_HashRNam, libGAP_SizeRNam );

    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarOpersFromTable( libGAP_GVarOpers );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoRecords() . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "records",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoRecords ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  records.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
