/****************************************************************************
**
*W  profile.c                     GAP source              Chris Jefferson
**
**
*Y  Copyright (C) 2014 The GAP Group
**
**  This file contains profile related functionality.
**
*/
#include        "system.h"              /* system dependent part           */

#include        "tls.h"

#include        "sysfiles.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "bool.h"                /* booleans                        */

#include        "code.h"                /* coder                           */
#include        "vars.h"                /* variables                       */
#include        "exprs.h"               /* expressions                     */

#include        "intrprtr.h"            /* interpreter                     */

#include        "ariths.h"              /* basic arithmetic                */

#include        "stats.h"               /* statements                      */

#include        <assert.h>

#include        "profile.h"

#include        "thread.h"

#include        "calls.h"               /* function filename, line number  */

/****************************************************************************
**
** Overview of GAP profiling
**
** The basic idea behind profiling comes in two parts:
**
** 1) In each stat and expr, we store at creation time a filenameid and a
**    line number. This is always done (as it isn't very expensive)
**
** 2) When we want to profile, we wrap each of
**      - ExecStatFuncs
**      - ExecExprFuncs
**      - EvalBoolFuncs
**    and output the line number of filename we stored in the stat or expr.
**    We also use 1 bit to mark that we have executed this statement, so
**    we can (if we want to) only output each executed statement once.
**
**    We use this information in a few ways:
**
**  1) Output a straight list of every executed statement
**  2) Provide coloured printing (by wrapping PrintStatFuncs and PrintExprFuncs)
**     which show which parts of an expression have been executed
**
**  There are not that many tricky cases here. We have to be careful that
**  some Expr are integers or local variables, and not touch those.
**
**  Limitations (and how we overcome them):
**
**  There are three main limitations to this approach (2 of which we handle)
**
**  1) From our initial trace, we don't know if a line has a statement on
**     it to be executed! Therefore we also provide the ability to store the
**     line of each created statement, so we can chck which lines have code
**     on them.
**
**  2) If we wait until the user can run ProfileLineByLine, they have missed
**     the reading and execution of lots of the standard library. Therefore
**     we provide -P (profiling) and -c (code coverage) options to the GAP
**     executable so the user can start before code loading starts
**
**  3) Operating at just a line basis can sometimes be too course. We can
**     see this when we use ActivateProfileColour. However, without some
**     serious additional overheads, I can't see how to provide this
**     functionality in output (basically we would have to store
**     line and character positions for the start and end of every expression).
**
**
**  Achieving 100% code coverage is a little tricky. Here is a list of
**  the special cases which had to be considered (so far)
**
**  GAP special cases for-loops of the form 'for i in [a..b]', by digging
**  into the range to extract 'a' and 'b'. Therefore nothing on the line is
**  ever evaluated and it appears to never be executed. We special case this
**  if ForRange, by marking the range as evaluated.
**
**  We purposesfully ignore T_TRUE_EXPR and T_FALSE_EXPR, which represent the
**  constants 'true' and 'false', as they are often read but not 'executed'.
**  We already ignored all integer and float constants anyway.
**  However, the main reason this was added is that GAP represents 'else'
**  as 'elif true then', and this was leading to 'else' statements being
**  never marked as executed.
*/


/****************************************************************************
**
** Store the current state of the profiler
*/

#ifdef HAVE_GETRUSAGE
#if HAVE_SYS_TIME_H
# include       <sys/time.h>            /* definition of 'struct timeval'  */
#endif
#if HAVE_SYS_RESOURCE_H
# include       <sys/resource.h>        /* definition of 'struct rusage'   */
#endif
#endif
#ifdef HAVE_GETTIMEOFDAY
# include       <sys/time.h>            /* for gettimeofday                */
#endif

libGAP_Obj libGAP_OutputtedFilenameList;

struct libGAP_StatementLocation
{
    int fileID;
    int line;
};

struct libGAP_ProfileState
{
  // C steam we are writing to
  FILE* Stream;
  // Did we use 'popen' to open the stream (matters when closing)
  int StreamWasPopened;
  // Are we currently outputting repeats (false=code coverage)
  libGAP_Int OutputRepeats;
  // Are we colouring output (not related to profiling directly)
  libGAP_Int ColouringOutput;

  // Used to generate 'X' statements, to make sure we correctly
  // attach each function call to the line it was executed on
  struct libGAP_StatementLocation lastNotOutputted;


  // Record last executed statement, to avoid repeats
  struct libGAP_StatementLocation lastOutputted;
  int lastOutputtedExec;

#if defined(HAVE_GETRUSAGE) || defined(HAVE_GETTIMEOFDAY)
  struct timeval lastOutputtedTime;
#endif
  int useGetTimeOfDay;

  int minimumProfileTick;
#ifdef HPCGAP
  int profiledThread;
#endif

  /* Have we previously profiled this execution of GAP? We need this because
  ** code coverage doesn't work more than once, as we use a bit in each Stat
  ** to mark if we previously executed this statement, which we can't
  ** clear */
  libGAP_UInt profiledPreviously;

} libGAP_profileState;

/* We keep this seperate as it is exported for use in other files */
libGAP_UInt libGAP_profileState_Active;



void libGAP_ProfileLineByLineOutput(libGAP_Obj func, char type)
{
  libGAP_HashLock(&libGAP_profileState);
  if(libGAP_profileState_Active && libGAP_profileState.OutputRepeats)
  {
    int startline_i = 0, endline_i = 0;
    libGAP_Obj startline = libGAP_FuncSTARTLINE_FUNC(0, func);
    libGAP_Obj endline = libGAP_FuncENDLINE_FUNC(0, func);
    if(libGAP_IS_INTOBJ(startline)) {
      startline_i = libGAP_INT_INTOBJ(startline);
    }
    if(libGAP_IS_INTOBJ(endline)) {
      endline_i = libGAP_INT_INTOBJ(endline);
    }

    libGAP_Obj name = libGAP_NAME_FUNC(func);
    libGAP_Char *name_c = ((libGAP_UInt)name) ? (libGAP_Char *)libGAP_CHARS_STRING(name) : (libGAP_Char *)"nameless";

    libGAP_Obj filename = libGAP_FuncFILENAME_FUNC(0, func);
    libGAP_Char *filename_c = (libGAP_Char*)"<missing filename>";
    if(filename != libGAP_Fail && filename != NULL)
      filename_c = (libGAP_Char *)libGAP_CHARS_STRING(filename);

    if(type == 'I' && libGAP_profileState.lastNotOutputted.line != -1)
    {
      fprintf(libGAP_profileState.Stream, "{\"Type\":\"X\",\"Line\":%d,\"FileId\":%d}\n",
              (int)libGAP_profileState.lastNotOutputted.line,
              (int)libGAP_profileState.lastNotOutputted.fileID);
    }

    fprintf(libGAP_profileState.Stream,
            "{\"Type\":\"%c\",\"Fun\":\"%s\",\"Line\":%d,\"EndLine\":%d,\"File\":\"%s\"}\n",
            type, name_c, startline_i, endline_i, filename_c);
  }
  libGAP_HashUnlock(&libGAP_profileState);
}

/****************************************************************************
**
** Functionality to store streams compressed.
** If we could rely on the existence of the IO package, we would use that here.
** however, we want to be able to start compressing files right at the start
** of GAP's execution, before anything else is done.
*/

#if HAVE_POPEN
static int libGAP_endsWithgz(char* s)
{
  s = strrchr(s, '.');
  if(s)
    return strcmp(s, ".gz") == 0;
  else
    return 0;
}
#endif

static void libGAP_fopenMaybeCompressed(char* name, struct libGAP_ProfileState* ps)
{
#if HAVE_POPEN
  char popen_buf[4096];
  if(libGAP_endsWithgz(name) && strlen(name) < 3000)
  {
    strcpy(popen_buf, "gzip > ");
    strcat(popen_buf, name);
    ps->Stream = popen(popen_buf, "w");
    ps->StreamWasPopened = 1;
    return;
  }
#endif

  ps->Stream = fopen(name, "w");
  ps->StreamWasPopened = 0;
}

static void libGAP_fcloseMaybeCompressed(struct libGAP_ProfileState* ps)
{
#if HAVE_POPEN
  if(ps->StreamWasPopened)
  {
    pclose(ps->Stream);
    ps->Stream = 0;
    return;
  }
#endif
  fclose(ps->Stream);
  ps->Stream = 0;
}

/****************************************************************************
**
** Store the true values of each function we wrap for profiling. These always
** store the correct values and are never changed.
*/


libGAP_UInt (* libGAP_OriginalExecStatFuncsForProf[256]) ( libGAP_Stat stat );

libGAP_Obj  (* libGAP_OriginalEvalExprFuncsForProf[256]) ( libGAP_Expr expr );
libGAP_Obj  (* libGAP_OriginalEvalBoolFuncsForProf[256]) ( libGAP_Expr expr );

void (* libGAP_OriginalPrintStatFuncsForProf[256]) ( libGAP_Stat stat );
void (* libGAP_OriginalPrintExprFuncsForProf[256]) ( libGAP_Expr expr );

/****************************************************************************
**
** These functions are here because the library may want to install
** functions once profiling has started.
*/

void libGAP_InstallEvalBoolFunc( libGAP_Int pos, libGAP_Obj(*expr)(libGAP_Expr)) {
  libGAP_OriginalEvalBoolFuncsForProf[pos] = expr;
  libGAP_HashLock(&libGAP_profileState);
  if(!libGAP_profileState_Active) {
    libGAP_EvalBoolFuncs[pos] = expr;
  }
  libGAP_HashUnlock(&libGAP_profileState);
}

void libGAP_InstallEvalExprFunc( libGAP_Int pos, libGAP_Obj(*expr)(libGAP_Expr)) {
  libGAP_OriginalEvalExprFuncsForProf[pos] = expr;
  libGAP_HashLock(&libGAP_profileState);
  if(!libGAP_profileState_Active) {
    libGAP_EvalExprFuncs[pos] = expr;
  }
  libGAP_HashUnlock(&libGAP_profileState);
}

void libGAP_InstallExecStatFunc( libGAP_Int pos, libGAP_UInt(*stat)(libGAP_Stat)) {
  libGAP_OriginalExecStatFuncsForProf[pos] = stat;
  libGAP_HashLock(&libGAP_profileState);
  if(!libGAP_profileState_Active) {
    libGAP_ExecStatFuncs[pos] = stat;
  }
  libGAP_HashUnlock(&libGAP_profileState);
}

void libGAP_InstallPrintStatFunc(libGAP_Int pos, void(*stat)(libGAP_Stat)) {
  libGAP_OriginalPrintStatFuncsForProf[pos] = stat;
  libGAP_HashLock(&libGAP_profileState);
  if(!libGAP_profileState.ColouringOutput) {
    libGAP_PrintStatFuncs[pos] = stat;
  }
  libGAP_HashUnlock(&libGAP_profileState);
}

void libGAP_InstallPrintExprFunc(libGAP_Int pos, void(*expr)(libGAP_Expr)) {
  libGAP_OriginalPrintExprFuncsForProf[pos] = expr;
  libGAP_HashLock(&libGAP_profileState);
  if(!libGAP_profileState.ColouringOutput) {
    libGAP_PrintExprFuncs[pos] = expr;
  }
  libGAP_HashUnlock(&libGAP_profileState);
}

/****************************************************************************
**
** These functions are only used when profiling is enabled. They output
** as approriate, and then pass through to the true function
*/

// This function checks if we have ever printed out the id of stat
static inline libGAP_UInt libGAP_getFilenameId(libGAP_Stat stat)
{
  libGAP_UInt id = libGAP_FILENAMEID_STAT(stat);
  if(libGAP_LEN_PLIST(libGAP_OutputtedFilenameList) < id || libGAP_ELM_PLIST(libGAP_OutputtedFilenameList,id) != libGAP_True)
  {
    if(libGAP_LEN_PLIST(libGAP_OutputtedFilenameList) < id) {
      libGAP_GROW_PLIST(libGAP_OutputtedFilenameList, id);
      libGAP_SET_LEN_PLIST(libGAP_OutputtedFilenameList, id);
    }
    libGAP_SET_ELM_PLIST(libGAP_OutputtedFilenameList, id, libGAP_True);
    libGAP_CHANGED_BAG(libGAP_OutputtedFilenameList);
    fprintf(libGAP_profileState.Stream, "{\"Type\":\"S\",\"File\":\"%s\",\"FileId\":%d}\n",
                                  libGAP_CSTR_STRING(libGAP_FILENAME_STAT(stat)), (int)id);
  }
  return id;
}

// exec : are we executing this statement
// visit: Was this statement previously visited (that is, executed)
static inline void libGAP_outputStat(libGAP_Stat stat, int exec, int visited)
{
  libGAP_UInt line;
  int nameid;

  int ticks = 0;
#if defined(HAVE_GETTIMEOFDAY)
  struct timeval timebuf;
#else
#if defined(HAVE_GETRUSAGE)
  struct timeval timebuf;
  struct rusage buf;
#endif
#endif


  libGAP_HashLock(&libGAP_profileState);
  // Explicitly skip these two cases, as they are often specially handled
  // and also aren't really interesting statements (something else will
  // be executed whenever they are).
  if(libGAP_TNUM_STAT(stat) == libGAP_T_TRUE_EXPR || libGAP_TNUM_STAT(stat) == libGAP_T_FALSE_EXPR)
    return;

  // Catch the case we arrive here and profiling is already disabled
  if(!libGAP_profileState_Active) {
    libGAP_HashUnlock(&libGAP_profileState);
    return;
  }

  nameid = libGAP_getFilenameId(stat);
  line = libGAP_LINE_STAT(stat);
  if(libGAP_profileState.lastOutputted.line != line ||
     libGAP_profileState.lastOutputted.fileID != nameid ||
     libGAP_profileState.lastOutputtedExec != exec)
  {

    if(libGAP_profileState.OutputRepeats) {

      if(libGAP_profileState.useGetTimeOfDay) {
#if defined(HAVE_GETTIMEOFDAY)
        gettimeofday(&timebuf, 0);
#else
        abort(); // should never be reached
#endif
      }
      else {
#if defined(HAVE_GETRUSAGE)
        struct rusage buf;
        getrusage( RUSAGE_SELF, &buf );
        timebuf = buf.ru_utime;
#else
        abort(); // should never be reached
#endif
      }


#if defined(HAVE_GETTIMEOFDAY) || defined(HAVE_GETRUSAGE)
      ticks = (timebuf.tv_sec - libGAP_profileState.lastOutputtedTime.tv_sec) * 1000000 +
              (timebuf.tv_usec - libGAP_profileState.lastOutputtedTime.tv_usec);
#endif
      // Basic sanity check
      if(ticks < 0)
        ticks = 0;
      if((libGAP_profileState.minimumProfileTick == 0)
         || (ticks > libGAP_profileState.minimumProfileTick)
         || (!visited)) {
        int ticksDone;
        if(libGAP_profileState.minimumProfileTick == 0) {
          ticksDone = ticks;
        } else {
          ticksDone = (ticks/libGAP_profileState.minimumProfileTick) * libGAP_profileState.minimumProfileTick;
        }
        ticks -= ticksDone;
        fprintf(libGAP_profileState.Stream, "{\"Type\":\"%c\",\"Ticks\":%d,\"Line\":%d,\"FileId\":%d}\n",
                exec ? 'E' : 'R', ticksDone, (int)line, (int)nameid);
#if defined(HAVE_GETRUSAGE) || defined(HAVE_GETTIMEOFDAY)
        libGAP_profileState.lastOutputtedTime = timebuf;
#endif
        libGAP_profileState.lastNotOutputted.line = -1;
        libGAP_profileState.lastOutputted.line = line;
        libGAP_profileState.lastOutputted.fileID = nameid;
        libGAP_profileState.lastOutputtedExec = exec;
      }
      else {
        libGAP_profileState.lastNotOutputted.line = line;
        libGAP_profileState.lastNotOutputted.fileID = nameid;
      }
    }
    else {
      fprintf(libGAP_profileState.Stream, "{\"Type\":\"%c\",\"Line\":%d,\"FileId\":%d}\n",
              exec ? 'E' : 'R', (int)line, (int)nameid);
      libGAP_profileState.lastOutputted.line = line;
      libGAP_profileState.lastOutputted.fileID = nameid;
      libGAP_profileState.lastOutputtedExec = exec;
      libGAP_profileState.lastNotOutputted.line = -1;
    }
  }

  libGAP_HashUnlock(&libGAP_profileState);
}

void libGAP_visitStat(libGAP_Stat stat)
{
#ifdef HPCGAP
  if(libGAP_profileState.profiledThread != libGAP_TLS(threadID))
    return;
#endif

  int visited = libGAP_VISITED_STAT(stat);

  if(!visited) {
    libGAP_ADDR_STAT(stat)[-1] |= (libGAP_Stat)1 << 63;
  }

  if(libGAP_profileState.OutputRepeats || !visited) {
    libGAP_outputStat(stat, 1, visited);
  }
}

libGAP_UInt libGAP_ProfileStatPassthrough(libGAP_Stat stat)
{
  libGAP_visitStat(stat);
  return libGAP_OriginalExecStatFuncsForProf[libGAP_TNUM_STAT(stat)](stat);
}

libGAP_Obj libGAP_ProfileEvalExprPassthrough(libGAP_Expr stat)
{
  libGAP_visitStat(stat);
  return libGAP_OriginalEvalExprFuncsForProf[libGAP_TNUM_STAT(stat)](stat);
}

libGAP_Obj libGAP_ProfileEvalBoolPassthrough(libGAP_Expr stat)
{
  /* There are two cases we must pass through without touching */
  /* From TNUM_EXPR */
  if(libGAP_IS_REFLVAR(stat)) {
    return libGAP_OriginalEvalBoolFuncsForProf[libGAP_T_REFLVAR](stat);
  }
  if(libGAP_IS_INTEXPR(stat)) {
    return libGAP_OriginalEvalBoolFuncsForProf[libGAP_T_INTEXPR](stat);
  }
  libGAP_visitStat(stat);
  return libGAP_OriginalEvalBoolFuncsForProf[libGAP_TNUM_STAT(stat)](stat);
}


/****************************************************************************
**
** This functions check if we overflow either 2^16 lines, or files.
** In this case profiling will "give up". We print a warning to tell users
** that this happens.
**/

libGAP_Int libGAP_HaveReportedLineProfileOverflow;
libGAP_Int libGAP_ShouldReportLineProfileOverflow;

libGAP_Int libGAP_HaveReportedFileProfileOverflow;
libGAP_Int libGAP_ShouldReportFileProfileOverflow;

// This function only exists to allow testing of these overflow checks
libGAP_Obj libGAP_FuncCLEAR_PROFILE_OVERFLOW_CHECKS(libGAP_Obj self) {
  libGAP_HaveReportedLineProfileOverflow = 0;
  libGAP_ShouldReportLineProfileOverflow = 0;

  libGAP_HaveReportedFileProfileOverflow = 0;
  libGAP_ShouldReportFileProfileOverflow = 0;
  
  return 0;
}

void libGAP_CheckPrintOverflowWarnings() {
    if(!libGAP_HaveReportedLineProfileOverflow && libGAP_ShouldReportLineProfileOverflow)
    {
      libGAP_HaveReportedLineProfileOverflow = 1;
      libGAP_Pr("#I Profiling only works on the first 65,535 lines of each file\n"
         "#I (this warning will only appear once).\n",
          0L, 0L);
    }
    
    if(!libGAP_HaveReportedFileProfileOverflow && libGAP_ShouldReportFileProfileOverflow)
    {
      libGAP_HaveReportedFileProfileOverflow = 1;
      libGAP_Pr("#I Profiling only works for the first 65,535 read files\n"
         "#I (this warning will only appear once).\n",
          0L, 0L );
    }
}

void libGAP_RegisterProfilingLineOverflowOccured()
{
    libGAP_Int active;
    libGAP_HashLock(&libGAP_profileState);
    active = libGAP_profileState_Active;
    libGAP_HashUnlock(&libGAP_profileState);
    libGAP_ShouldReportLineProfileOverflow = 1;
    if(active)
    {
        libGAP_CheckPrintOverflowWarnings();
    }
}

void libGAP_RegisterProfilingFileOverflowOccured()
{
    libGAP_Int active;
    libGAP_HashLock(&libGAP_profileState);
    active = libGAP_profileState_Active;
    libGAP_HashUnlock(&libGAP_profileState);
    libGAP_ShouldReportFileProfileOverflow = 1;
    if(active)
    {
        libGAP_CheckPrintOverflowWarnings();
    }
}

/****************************************************************************
**
** Activating and deacivating profiling, either at startup or by user request
*/

void libGAP_outputVersionInfo()
{
    fprintf(libGAP_profileState.Stream, 
            "{ \"Type\": \"_\", \"Version\":1, \"IsCover\": %s, "
            "  \"TimeType\": \"%s\"}\n", 
            libGAP_profileState.OutputRepeats?"false":"true",
            libGAP_profileState.useGetTimeOfDay?"Wall":"CPU");
  
}

void libGAP_enableAtStartup(char* filename, libGAP_Int repeats)
{
    libGAP_Int i;

    if(libGAP_profileState_Active) {
        fprintf(stderr, "-P or -C can only be passed once\n");
        exit(1);
    }
    
    

    libGAP_profileState.OutputRepeats = repeats;

    libGAP_fopenMaybeCompressed(filename, &libGAP_profileState);
    if(!libGAP_profileState.Stream) {
        fprintf(stderr, "Failed to open '%s' for profiling output.\n", filename);
        fprintf(stderr, "Abandoning starting GAP.\n");
        exit(1);
    }

    for( i = 0; i < sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++) {
      libGAP_ExecStatFuncs[i] = libGAP_ProfileStatPassthrough;
      libGAP_EvalExprFuncs[i] = libGAP_ProfileEvalExprPassthrough;
      libGAP_EvalBoolFuncs[i] = libGAP_ProfileEvalBoolPassthrough;
    }

    libGAP_profileState_Active = 1;
    libGAP_profileState.profiledPreviously = 1;
#ifdef HPCGAP
    libGAP_profileState.profiledThread = libGAP_TLS(threadID);
#endif
    libGAP_profileState.lastNotOutputted.line = -1;
#ifdef HAVE_GETTIMEOFDAY
    libGAP_profileState.useGetTimeOfDay = 1;
    gettimeofday(&(libGAP_profileState.lastOutputtedTime), 0);
#else
#ifdef HAVE_GETRUSAGE
    libGAP_profileState.useGetTimeOfDay = 0;
    struct rusage buf;
    getrusage( RUSAGE_SELF, &buf );
    libGAP_profileState.lastOutputtedTime = buf.ru_utime;
#endif
#endif

    libGAP_outputVersionInfo();
}

// This function is for when GAP is started with -c, and
// enables profiling at startup. If anything goes wrong,
// we quit straight away.
libGAP_Int libGAP_enableCodeCoverageAtStartup( libGAP_Char **argv, void * dummy)
{
    libGAP_enableAtStartup(argv[0], 0);
    return 1;
}

// This function is for when GAP is started with -P, and
// enables profiling at startup. If anything goes wrong,
// we quit straight away.
libGAP_Int libGAP_enableProfilingAtStartup( libGAP_Char **argv, void * dummy)
{
    libGAP_enableAtStartup(argv[0], 1);
    return 1;
}

libGAP_Obj libGAP_FuncACTIVATE_PROFILING (
    libGAP_Obj                 self,
    libGAP_Obj                 filename, /* filename to write to */
    libGAP_Obj                 coverage,
    libGAP_Obj                 justStat,
    libGAP_Obj                 wallTime,
    libGAP_Obj                 resolution)
{
    libGAP_Int i;

    if(libGAP_profileState_Active) {
      return libGAP_Fail;
    }

    if(libGAP_profileState.profiledPreviously &&
       coverage == libGAP_True) {
        libGAP_ErrorMayQuit("Code coverage can only be started once per"
                     " GAP session. Please exit GAP and restart. Sorry.",0,0);
        return libGAP_Fail;
    }

    libGAP_CheckPrintOverflowWarnings();

    libGAP_OutputtedFilenameList = libGAP_NEW_PLIST(libGAP_T_PLIST, 0);

    if ( ! libGAP_IsStringConv( filename ) ) {
        libGAP_ErrorMayQuit("<filename> must be a string",0,0);
    }

    if(coverage != libGAP_True && coverage != libGAP_False) {
      libGAP_ErrorMayQuit("<coverage> must be a boolean",0,0);
    }

    if(justStat != libGAP_True && justStat != libGAP_False) {
      libGAP_ErrorMayQuit("<justStat> must be a boolean",0,0);
    }

    if(wallTime != libGAP_True && wallTime != libGAP_False) {
      libGAP_ErrorMayQuit("<wallTime> must be a boolean",0,0);
    }

#ifndef HAVE_GETTIMEOFDAY
    if(wallTime == libGAP_True) {
        libGAP_ErrorMayQuit("This OS does not support wall-clock based timing");
    }
#endif
#ifndef HAVE_GETRUSAGE
    if(wallTime == libGAP_False) {
        libGAP_ErrorMayQuit("This OS does not support CPU based timing");
    }
#endif

    libGAP_profileState.useGetTimeOfDay = (wallTime == libGAP_True);

    if( ! libGAP_IS_INTOBJ(resolution) ) {
      libGAP_ErrorMayQuit("<resolution> must be an integer",0,0);
    }

    libGAP_HashLock(&libGAP_profileState);

    // Recheck inside lock
    if(libGAP_profileState_Active) {
      libGAP_HashUnlock(&libGAP_profileState);
      return libGAP_Fail;
    }
    int tick = libGAP_INT_INTOBJ(resolution);
    if(tick < 0) {
        libGAP_ErrorMayQuit("<resolution> must be a non-negative integer",0,0);
    }
    libGAP_profileState.minimumProfileTick = tick;


    if(coverage == libGAP_True) {
      libGAP_profileState.OutputRepeats = 0;
    }
    else {
      libGAP_profileState.OutputRepeats = 1;
    }

    libGAP_fopenMaybeCompressed(libGAP_CSTR_STRING(filename), &libGAP_profileState);

    if(libGAP_profileState.Stream == 0) {
      libGAP_HashUnlock(&libGAP_profileState);
      return libGAP_Fail;
    }

    for( i = 0; i < sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++) {
      libGAP_ExecStatFuncs[i] = libGAP_ProfileStatPassthrough;
      if(justStat == libGAP_False) {
          libGAP_EvalExprFuncs[i] = libGAP_ProfileEvalExprPassthrough;
          libGAP_EvalBoolFuncs[i] = libGAP_ProfileEvalBoolPassthrough;
      }
    }

    libGAP_profileState_Active = 1;
    libGAP_profileState.profiledPreviously = 1;
#ifdef HPCGAP
    libGAP_profileState.profiledThread = libGAP_TLS(threadID);
#endif
    libGAP_profileState.lastNotOutputted.line = -1;

    if(wallTime == libGAP_True) {
#ifdef HAVE_GETTIMEOFDAY
        gettimeofday(&(libGAP_profileState.lastOutputtedTime), 0);
#else
        abort(); // this should never be reached
#endif
    }
    else {
#ifdef HAVE_GETRUSAGE
        struct rusage buf;
        getrusage( RUSAGE_SELF, &buf );
        libGAP_profileState.lastOutputtedTime = buf.ru_utime;
#else
        abort(); // this should never be reached
#endif
    }

    libGAP_outputVersionInfo();
    libGAP_HashUnlock(&libGAP_profileState);

    return libGAP_True;
}

libGAP_Obj libGAP_FuncDEACTIVATE_PROFILING (
    libGAP_Obj                 self)
{
  int i;

  libGAP_HashLock(&libGAP_profileState);

  if(!libGAP_profileState_Active) {
    libGAP_HashUnlock(&libGAP_profileState);
    return libGAP_Fail;
  }

  libGAP_fcloseMaybeCompressed(&libGAP_profileState);

  for( i = 0; i < sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++) {
    libGAP_ExecStatFuncs[i] = libGAP_OriginalExecStatFuncsForProf[i];
    libGAP_EvalExprFuncs[i] = libGAP_OriginalEvalExprFuncsForProf[i];
    libGAP_EvalBoolFuncs[i] = libGAP_OriginalEvalBoolFuncsForProf[i];
  }

  libGAP_profileState_Active = 0;

  libGAP_HashUnlock(&libGAP_profileState);

  return libGAP_True;
}

libGAP_Obj libGAP_FuncIS_PROFILE_ACTIVE (
    libGAP_Obj self)
{
  if(libGAP_profileState_Active) {
    return libGAP_True;
  } else {
    return libGAP_False;
  }
}

/****************************************************************************
**
** We are now into the functions which deal with colouring printing output.
** This code basically wraps all the existing print functions and will colour
** their output either green or red depending on if statements are marked
** as being executed.
*/



libGAP_Int libGAP_CurrentColour = 0;

static void libGAP_setColour()
{
  if(libGAP_CurrentColour == 0) {
    libGAP_Pr("\x1b[0m",0L,0L);
  }
  else if(libGAP_CurrentColour == 1) {
    libGAP_Pr("\x1b[32m",0L,0L);
  }
  else if(libGAP_CurrentColour == 2) {
    libGAP_Pr("\x1b[31m",0L,0L);
  }
}

void libGAP_ProfilePrintStatPassthrough(libGAP_Stat stat)
{
  libGAP_Int SavedColour = libGAP_CurrentColour;
  if(libGAP_VISITED_STAT(stat)) {
    libGAP_CurrentColour = 1;
  }
  else {
    libGAP_CurrentColour = 2;
  }
  libGAP_setColour();
  libGAP_OriginalPrintStatFuncsForProf[libGAP_TNUM_STAT(stat)](stat);
  libGAP_CurrentColour = SavedColour;
  libGAP_setColour();
}

void libGAP_ProfilePrintExprPassthrough(libGAP_Expr stat)
{
  libGAP_Int SavedColour = -1;
  /* There are two cases we must pass through without touching */
  /* From TNUM_EXPR */
  if(libGAP_IS_REFLVAR(stat)) {
    libGAP_OriginalPrintExprFuncsForProf[libGAP_T_REFLVAR](stat);
  } else if(libGAP_IS_INTEXPR(stat)) {
    libGAP_OriginalPrintExprFuncsForProf[libGAP_T_INTEXPR](stat);
  } else {
    SavedColour = libGAP_CurrentColour;
    if(libGAP_VISITED_STAT(stat)) {
      libGAP_CurrentColour = 1;
    }
    else {
      libGAP_CurrentColour = 2;
    }
    libGAP_setColour();
    libGAP_OriginalPrintExprFuncsForProf[libGAP_TNUM_STAT(stat)](stat);
    libGAP_CurrentColour = SavedColour;
    libGAP_setColour();
  }
}

libGAP_Obj libGAP_activate_colored_output_from_profile(void)
{
    int i;

    libGAP_HashLock(&libGAP_profileState);

    if(libGAP_profileState.ColouringOutput) {
      libGAP_HashUnlock(&libGAP_profileState);
      return libGAP_Fail;
    }

    for( i = 0; i < sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++) {
      libGAP_PrintStatFuncs[i] = libGAP_ProfilePrintStatPassthrough;
      libGAP_PrintExprFuncs[i] = libGAP_ProfilePrintExprPassthrough;
    }

    libGAP_profileState.ColouringOutput = 1;
    libGAP_CurrentColour = 0;
    libGAP_setColour();

    libGAP_HashUnlock(&libGAP_profileState);

    return libGAP_True;
}

libGAP_Obj libGAP_deactivate_colored_output_from_profile(void)
{
  int i;

  libGAP_HashLock(&libGAP_profileState);

  if(!libGAP_profileState.ColouringOutput) {
    libGAP_HashUnlock(&libGAP_profileState);
    return libGAP_Fail;
  }

  for( i = 0; i < sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++) {
    libGAP_PrintStatFuncs[i] = libGAP_OriginalPrintStatFuncsForProf[i];
    libGAP_PrintExprFuncs[i] = libGAP_OriginalPrintExprFuncsForProf[i];
  }

  libGAP_profileState.ColouringOutput = 0;
  libGAP_CurrentColour = 0;
  libGAP_setColour();

  libGAP_HashUnlock(&libGAP_profileState);

  return libGAP_True;
}

libGAP_Obj libGAP_FuncACTIVATE_COLOR_PROFILING(libGAP_Obj self, libGAP_Obj arg)
{
  if(arg == libGAP_True)
  {
    return libGAP_activate_colored_output_from_profile();
  }
  else if(arg == libGAP_False)
  {
    return libGAP_deactivate_colored_output_from_profile();
  }
  else
    return libGAP_Fail;
}

/****************************************************************************
**
** This function exists to help with code coverage -- this outputs which
** lines have statements on expressions on them, so later we can
** check we executed something on those lines!
**/

void libGAP_RegisterStatWithProfiling(libGAP_Stat stat)
{
    int active;
    libGAP_HashLock(&libGAP_profileState);
    active = libGAP_profileState_Active;
    libGAP_HashUnlock(&libGAP_profileState);
    if(active) {
      libGAP_outputStat(stat, 0, 0);
    }

}




/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "ACTIVATE_PROFILING", 5, "string,boolean,boolean,boolean,integer",
      libGAP_FuncACTIVATE_PROFILING, "src/profile.c:ACTIVATE_PROFILING" },
    { "DEACTIVATE_PROFILING", 0, "",
      libGAP_FuncDEACTIVATE_PROFILING, "src/profile.c:DEACTIVATE_PROFILING" },
    { "CLEAR_PROFILE_OVERFLOW_CHECKS", 0, "",
      libGAP_FuncCLEAR_PROFILE_OVERFLOW_CHECKS, "src/profile.c:CLEAR_PROFILE_OVERFLOW_CHECKS" },
    { "IsLineByLineProfileActive", 0, "",
      libGAP_FuncIS_PROFILE_ACTIVE, "src/profile.c:IsLineByLineProfileActive" },
    { "ACTIVATE_COLOR_PROFILING", 1, "bool",
        libGAP_FuncACTIVATE_COLOR_PROFILING, "src/profile.c:ACTIVATE_COLOR_PROFILING" },
    { 0 }
};


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    libGAP_OutputtedFilenameList = libGAP_NEW_PLIST(libGAP_T_PLIST, 0);
    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );
    libGAP_InitGlobalBag(&libGAP_OutputtedFilenameList, "src/profile.c:OutputtedFileList");
    return 0;
}


/****************************************************************************
**
*F  InitInfoStats() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "stats",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoProfile ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  stats.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
