/****************************************************************************
**
*W  listoper.c                  GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains  the functions of the  package with the operations for
**  generic lists.
*/
#include        "system.h"              /* Ints, UInts                     */


#include        "sysfiles.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "integer.h"             /* integers                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listoper.h"            /* operations for generic lists    */
#include        "listfunc.h"            /* functions for generic lists    */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */
#include        "opers.h"               /* TRY_NEXT_METHOD                 */
#include        "range.h"               /* Ranges                          */
#include        "code.h"		/* Coder                           */
#include        "thread.h"              /* threads                         */
#include        "tls.h"                 /* thread-local storage            */


/****************************************************************************
**

*F  EqListList(<listL>,<listR>) . . . . . . . . . test if two lists are equal
**
**  'EqListList' returns  'true' if  the  two lists <listL> and  <listR>  are
**  equal and 'false' otherwise.  This is a generic function, which works for
**  all types of lists.
*/
libGAP_Int             libGAP_EqListList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Int                 lenL;           /* length of the left operand      */
    libGAP_Int                 lenR;           /* length of the right operand     */
    libGAP_Obj                 elmL;           /* element of the left operand     */
    libGAP_Obj                 elmR;           /* element of the right operand    */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the lengths of the lists and compare them                       */
    lenL = libGAP_LEN_LIST( listL );
    lenR = libGAP_LEN_LIST( listR );
    if ( lenL != lenR ) {
        return 0L;
    }

    /* loop over the elements and compare them                             */
    for ( i = 1; i <= lenL; i++ ) {
        elmL = libGAP_ELMV0_LIST( listL, i );
        elmR = libGAP_ELMV0_LIST( listR, i );
        if ( elmL == 0 && elmR != 0 ) {
            return 0L;
        }
        else if ( elmR == 0 && elmL != 0 ) {
            return 0L;
        }
        else if ( ! libGAP_EQ( elmL, elmR ) ) {
            return 0L;
        }
    }

    /* no differences found, the lists are equal                           */
    return 1L;
}

libGAP_Obj             libGAP_EqListListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return (libGAP_EqListList( listL, listR ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  LtListList(<listL>,<listR>) . . . . . . . . . test if two lists are equal
**
**  'LtListList' returns 'true' if  the list <listL>   is less than the  list
**  <listR> and 'false'  otherwise.  This is a  generic function, which works
**  for all types of lists.
*/
libGAP_Int             libGAP_LtListList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Int                 lenL;           /* length of the left operand      */
    libGAP_Int                 lenR;           /* length of the right operand     */
    libGAP_Obj                 elmL;           /* element of the left operand     */
    libGAP_Obj                 elmR;           /* element of the right operand    */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the lengths of the lists and compare them                       */
    lenL = libGAP_LEN_LIST( listL );
    lenR = libGAP_LEN_LIST( listR );

    /* loop over the elements and compare them                             */
    for ( i = 1; i <= lenL && i <= lenR; i++ ) {
        elmL = libGAP_ELMV0_LIST( listL, i );
        elmR = libGAP_ELMV0_LIST( listR, i );
        if ( elmL == 0 && elmR != 0 ) {
            return 1L;
        }
        else if ( elmR == 0 && elmL != 0 ) {
            return 0L;
        }
        else if ( ! libGAP_EQ( elmL, elmR ) ) {
            return libGAP_LT( elmL, elmR );
        }
    }

    /* reached the end of at least one list                                */
    return (lenL < lenR);
}

libGAP_Obj             libGAP_LtListListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return (libGAP_LtListList( listL, listR ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  InList(<objL>,<listR>)  . . . . . . . . . . . . membership test for lists
**
**  'InList' returns a   nonzero value if  <objL>  is a  member of  the  list
**  <listR>, and zero otherwise.
*/
libGAP_Int             libGAP_InList (
    libGAP_Obj                 objL,
    libGAP_Obj                 listR )
{
  return libGAP_Fail != libGAP_POS_LIST( listR, objL, libGAP_INTOBJ_INT(0L) );
}

libGAP_Obj             libGAP_InListDefaultHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 list )
{
    return (libGAP_InList( obj, list ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  SumList(<listL>,<listR>)  . . . . . . . . . . . . . . . . .  sum of lists
*F  SumSclList(<listL>,<listR>) . . . . . . . . .  sum of a scalar and a list
*F  SumListScl(<listL>,<listR>) . . . . . . . . .  sum of a list and a scalar
*F  SumListList<listL>,<listR>)  . . . . . . . . . . . . .  sum of two lists
**
**  'SumList' is the extended dispatcher for the  sums involving lists.  That
**  is, whenever  two operands are  added and at  least one operand is a list
**  and 'SumFuncs'  does not point to  a special  function, then 'SumList' is
**  called.  'SumList' determines the extended  types of the operands  (e.g.,
**  'T_INT', 'T_VECTOR',  'T_MATRIX', 'T_LISTX') and then  dispatches through
**  'SumFuncs' again.
**
**  'SumSclList' is a generic function  for the first kind  of sum, that of a
**  scalar and a list.
**
**  'SumListScl' is a generic function for the second  kind of sum, that of a
**  list and a scalar.
**
**  'SumListList' is a generic  function for the third kind  of sum,  that of
**  two lists.
*/


libGAP_Obj             libGAP_SumSclList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listS;          /* sum, result                     */
    libGAP_Obj                 elmS;           /* one element of sum list         */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_Int                 len;            /* length                          */
    libGAP_Int                 i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_LIST( listR );
    listS = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(listL) ||  libGAP_IS_MUTABLE_OBJ(listR) ?
                       libGAP_T_PLIST : (libGAP_T_PLIST + libGAP_IMMUTABLE), len );
    libGAP_SET_LEN_PLIST( listS, len );

    /* loop over the entries and add                                       */
    for ( i = 1; i <= len; i++ ) {
        elmR = libGAP_ELMV0_LIST( listR, i );
        if (elmR)
          {
            elmS = libGAP_SUM( listL, elmR );
            libGAP_SET_ELM_PLIST( listS, i, elmS );
            libGAP_CHANGED_BAG( listS );
          }
    }

    /* return the result                                                   */
    return listS;
}

libGAP_Obj             libGAP_SumListScl (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listS;          /* sum, result                     */
    libGAP_Obj                 elmS;           /* one element of sum list         */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_Int                 len;            /* length                          */
    libGAP_Int                 i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_LIST( listL );
    listS = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(listR) || libGAP_IS_MUTABLE_OBJ(listL) ?
                       libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE, len );
    libGAP_SET_LEN_PLIST( listS, len );

    /* loop over the entries and add                                       */
    for ( i = 1; i <= len; i++ ) {
        elmL = libGAP_ELMV0_LIST( listL, i );
        if (elmL)
          {
            elmS = libGAP_SUM( elmL, listR );
            libGAP_SET_ELM_PLIST( listS, i, elmS );
            libGAP_CHANGED_BAG( listS );
          }
    }

    /* return the result                                                   */
    return listS;
}

libGAP_Obj             libGAP_SumListList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listS;          /* sum, result                     */
    libGAP_Obj                 elmS;           /* one element of the sum          */
    libGAP_Obj                 elmL;           /* one element of the left list    */
    libGAP_Obj                 elmR;           /* one element of the right list   */
    libGAP_Int                 lenL,lenR, lenS;/* lengths                         */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt                mutS;

    /* get and check the length                                            */
    lenL = libGAP_LEN_LIST( listL );
    lenR = libGAP_LEN_LIST( listR );
    lenS = (lenR > lenL) ? lenR : lenL;
    listS = libGAP_NEW_PLIST( (libGAP_IS_MUTABLE_OBJ(listL) || libGAP_IS_MUTABLE_OBJ(listR)) ?
                       libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE, lenS );
    libGAP_SET_LEN_PLIST( listS, lenS );

    /* Sort out mutability */
    mutS = 0;
    for (i = 1; i <= lenL; i++)
      if ((elmL = libGAP_ELM0_LIST( listL, i)))
        {
          mutS = libGAP_IS_MUTABLE_OBJ(elmL);
          break;
        }
    for (i = 1; i <= lenR; i++)
      if ((elmR = libGAP_ELM0_LIST( listR, i)))
        {
          mutS = mutS || libGAP_IS_MUTABLE_OBJ(elmR);
          break;
        }
    
    /* loop over the entries and add                                       */
    for ( i = 1; i <= lenS; i++ ) {
      elmL = libGAP_ELM0_LIST( listL, i ) ;
      elmR = libGAP_ELM0_LIST( listR, i ) ;
      elmS =  elmL ? (elmR ? libGAP_SUM( elmL, elmR ) : mutS ? libGAP_SHALLOW_COPY_OBJ(elmL): elmL) :
        elmR ? (mutS ? libGAP_SHALLOW_COPY_OBJ(elmR): elmR) : 0;
      if (elmS)
        {
          libGAP_SET_ELM_PLIST( listS, i, elmS );
          libGAP_CHANGED_BAG( listS );
        }
    }

    /* return the result                                                   */
    return listS;
}

libGAP_Obj             libGAP_SumSclListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_SumSclList( listL, listR );
}

libGAP_Obj             libGAP_SumListSclHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_SumListScl( listL, listR );
}

libGAP_Obj             libGAP_SumListListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_SumListList( listL, listR );
}


/****************************************************************************
**
*F  ZeroList(<list>)  . . . . . . . . . . . . . . . . . . . .  zero of a list
*F  ZeroListDefault(<list>) . . . . . . . . . . . . . . . . .  zero of a list
**
**  'ZeroList' is the extended dispatcher for the zero involving lists.  That
**  is, whenever zero for a list is called and  'ZeroFuncs' does not point to
**  a special function, then 'ZeroList' is called.  'ZeroList' determines the
**  extended   type of the  operand  and then  dispatches through 'ZeroFuncs'
**  again.
**
**  'ZeroListDefault' is a generic function for the zero.
*/



libGAP_Obj             libGAP_ZeroListDefault (
    libGAP_Obj                 list )
{
    libGAP_Obj                 res;
/*  Obj                 elm; */
    libGAP_Int                 len;
    libGAP_Int                 i;

    /* make the result list -- same mutability as argument */
    len = libGAP_LEN_LIST( list );
    res = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE, len );
    libGAP_SET_LEN_PLIST( res, len );

    /* enter zeroes everywhere                                             */
    /* For now, lets just do the simplest and safest thing */
    for (i = 1; i <= len; i++ )
      {
        libGAP_Obj tmp = libGAP_ELM0_LIST( list, i);
        if (tmp) {
          tmp = libGAP_ZERO(tmp);
          libGAP_SET_ELM_PLIST( res, i,tmp );
          libGAP_CHANGED_BAG( res);
        }
      }
    /* Now adjust the result TNUM info */

    if (len == 0)
      libGAP_SET_FILT_LIST( res, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( list ))
      {
        if (libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE ||
            libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE+libGAP_IMMUTABLE)
          libGAP_RetypeBag(res, libGAP_TNUM_OBJ(list));
        else if (libGAP_TNUM_OBJ(list) >= libGAP_T_PLIST_CYC &&
                 libGAP_TNUM_OBJ(list) < libGAP_T_PLIST_FFE)
          libGAP_RetypeBag(res, libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST_CYC : libGAP_T_PLIST_CYC+libGAP_IMMUTABLE);
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_DENSE))
          {
            libGAP_SET_FILT_LIST( res, libGAP_FN_IS_DENSE );
            if (libGAP_HAS_FILT_LIST (list, libGAP_FN_IS_HOMOG))
              {
                libGAP_SET_FILT_LIST( res, libGAP_FN_IS_HOMOG);
                if (libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_TABLE))
                  {
                    libGAP_SET_FILT_LIST( res, libGAP_FN_IS_TABLE);
                    if (libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_RECT))
                      libGAP_SET_FILT_LIST( res, libGAP_FN_IS_RECT);
                  }
              }
          }
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_NDENSE))
          libGAP_SET_FILT_LIST( res, libGAP_FN_IS_NDENSE );
      }

    /* return the result                                                   */
    return res;
}

libGAP_Obj             libGAP_ZeroListDefaultHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_ZeroListDefault( list );
}


libGAP_Obj             libGAP_ZeroListMutDefault (
    libGAP_Obj                 list )
{
    libGAP_Obj                 res;
/*  Obj                 elm; */
    libGAP_Int                 len;
    libGAP_Int                 i;

    /* make the result list -- always mutable */
    len = libGAP_LEN_LIST( list );
    res = libGAP_NEW_PLIST(  libGAP_T_PLIST ,len );
    libGAP_SET_LEN_PLIST( res, len );

    /* enter zeroes everywhere                                             */
    /* For now, lets just do the simplest and safest thing */
    for (i = 1; i <= len; i++ )
      {
        libGAP_Obj tmp = libGAP_ELM0_LIST( list, i);
        if (tmp) {
          tmp = libGAP_ZERO_MUT(tmp);
          libGAP_SET_ELM_PLIST( res, i,tmp );
          libGAP_CHANGED_BAG( res);
        }
      }
    /* Now adjust the result TNUM info */

    if (len == 0)
      libGAP_SET_FILT_LIST( res, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( list ))
      {
        if (libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE ||
            libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE+libGAP_IMMUTABLE)
          libGAP_RetypeBag(res, libGAP_T_PLIST_FFE);
        else if (libGAP_TNUM_OBJ(list) >= libGAP_T_PLIST_CYC &&
                 libGAP_TNUM_OBJ(list) < libGAP_T_PLIST_FFE)
          libGAP_RetypeBag(res, libGAP_T_PLIST_CYC);
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_DENSE))
          {
            libGAP_SET_FILT_LIST( res, libGAP_FN_IS_DENSE );
            if (libGAP_HAS_FILT_LIST (list, libGAP_FN_IS_HOMOG) &&
                !libGAP_IS_MUTABLE_OBJ(libGAP_ELM_PLIST(res,1)))
              {
                libGAP_SET_FILT_LIST( res, libGAP_FN_IS_HOMOG);
              }
          }
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_NDENSE))
          libGAP_SET_FILT_LIST( res, libGAP_FN_IS_NDENSE );
      }

    /* return the result                                                   */
    return res;
}

libGAP_Obj             libGAP_ZeroListMutDefaultHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_ZeroListMutDefault( list );
}



/* This is intended to be installed as a method for the Attribute Zero, for
   rectangular matrices
   rather than the Operation ZeroOp. This is useful because, knowing that
   we want an immutable result, we can (a) reuse a single row of zeros
   (b) record that the result is a rectangular table */

libGAP_Obj libGAP_ZeroAttrMat( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_Obj zrow;
  libGAP_UInt len;
  libGAP_UInt i;
  libGAP_Obj res;
  len = libGAP_LEN_LIST(mat);
  if (len == 0)
    return libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY + libGAP_IMMUTABLE, 0);
  zrow = libGAP_ZERO(libGAP_ELM_LIST(mat,1));
  libGAP_MakeImmutable(zrow);
  res = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT+libGAP_IMMUTABLE, len);
  libGAP_SET_LEN_PLIST(res,len);
  for (i = 1; i <= len; i++)
    libGAP_SET_ELM_PLIST(res,i,zrow);
  return res;
}

/****************************************************************************
**
*F  AInvListDefault(<list>) . . . . . . . . . . .  additive inverse of a list
*F  AInvMutListDefault
**
**  'AInvList' is the extended dispatcher for  the additive inverse involving
**  lists.  That is, whenever  the additive inverse for  lists is called  and
**  'AInvFuncs' does not   point to a   special function, then  'AInvList' is
**  called.  'AInvList' determines the extended  type of the operand and then
**  dispatches through 'AInvFuncs' again.
**
**  'AInvListDefault' is a generic function for the additive inverse.
*/

libGAP_Obj libGAP_AInvMutListDefault (
    libGAP_Obj                 list )
{
    libGAP_Obj                 res;
    libGAP_Obj                 elm;
    libGAP_Int                 len;
    libGAP_Int                 i;

    /* make the result list -- always mutable, since this might be a method for
       AdditiveInverseOp */
    len = libGAP_LEN_LIST( list );
    res = libGAP_NEW_PLIST( libGAP_T_PLIST , len );
    libGAP_SET_LEN_PLIST( res, len );

    /* enter the additive inverses everywhere                              */
    for ( i = 1; i <= len; i++ ) {
        elm = libGAP_ELM0_LIST( list, i );
        if (elm) {
          elm = libGAP_AINV_MUT( elm );
          libGAP_SET_ELM_PLIST( res, i, elm );
          libGAP_CHANGED_BAG( res );
        }
    }

    /* Now adjust the result TNUM info */

    if (len == 0)
      libGAP_SET_FILT_LIST( res, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( list ))
      {
        if (libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE ||
            libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE+libGAP_IMMUTABLE)
          libGAP_RetypeBag(res, libGAP_T_PLIST_FFE);
        else if (libGAP_TNUM_OBJ(list) >= libGAP_T_PLIST_CYC &&
                 libGAP_TNUM_OBJ(list) < libGAP_T_PLIST_FFE)
          libGAP_RetypeBag(res, libGAP_T_PLIST_CYC );
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_DENSE))
          {
            libGAP_SET_FILT_LIST( res, libGAP_FN_IS_DENSE );
            if (libGAP_HAS_FILT_LIST (list, libGAP_FN_IS_HOMOG) &&
                !libGAP_IS_MUTABLE_OBJ(libGAP_ELM_PLIST(res,1)))
              {
                libGAP_SET_FILT_LIST( res, libGAP_FN_IS_HOMOG);
                
              }
          }
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_NDENSE))
          libGAP_SET_FILT_LIST( res, libGAP_FN_IS_NDENSE );
      }
    /* return the result                                                   */
    return res;
}

libGAP_Obj libGAP_AInvMutListDefaultHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_AInvMutListDefault( list );
}

libGAP_Obj libGAP_AInvListDefault (
    libGAP_Obj                 list )
{
    libGAP_Obj                 res;
    libGAP_Obj                 elm;
    libGAP_Int                 len;
    libGAP_Int                 i;

    /* make the result list -- same mutability as input */
    len = libGAP_LEN_LIST( list );
    res = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE , len );
    libGAP_SET_LEN_PLIST( res, len );

    /* enter the additive inverses everywhere                              */
    for ( i = 1; i <= len; i++ ) {
        elm = libGAP_ELM0_LIST( list, i );
        if (elm) {
          elm = libGAP_AINV( elm );
          libGAP_SET_ELM_PLIST( res, i, elm );
          libGAP_CHANGED_BAG( res );
        }
    }

    /* Now adjust the result TNUM info */

    if (len == 0)
      libGAP_SET_FILT_LIST( res, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( list ))
      {
        if (libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE ||
            libGAP_TNUM_OBJ(list) == libGAP_T_PLIST_FFE+libGAP_IMMUTABLE)
          libGAP_RetypeBag(res, libGAP_TNUM_OBJ(list));
        else if (libGAP_TNUM_OBJ(list) >= libGAP_T_PLIST_CYC &&
                 libGAP_TNUM_OBJ(list) < libGAP_T_PLIST_FFE)
          libGAP_RetypeBag(res, libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST_CYC : libGAP_T_PLIST_CYC+libGAP_IMMUTABLE );
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_DENSE))
          {
            libGAP_SET_FILT_LIST( res, libGAP_FN_IS_DENSE );
            if (libGAP_HAS_FILT_LIST (list, libGAP_FN_IS_HOMOG) &&
                !libGAP_IS_MUTABLE_OBJ(libGAP_ELM_PLIST(res,1)))
              {
                libGAP_SET_FILT_LIST( res, libGAP_FN_IS_HOMOG);
                if (libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_TABLE))
                  {
                    libGAP_SET_FILT_LIST( res, libGAP_FN_IS_TABLE);
                    if (libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_RECT))
                      libGAP_SET_FILT_LIST( res, libGAP_FN_IS_RECT);
                  }
              }
          }
        else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_NDENSE))
          libGAP_SET_FILT_LIST( res, libGAP_FN_IS_NDENSE );
      }
    /* return the result                                                   */
    return res;
}

libGAP_Obj libGAP_AInvListDefaultHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_AInvListDefault( list );
}

/****************************************************************************
**
*F  DiffList(<listL>,<listR>) . . . . . . . . . . . . . . difference of lists
*F  DiffSclList(<listL>,<listR>)  . . . . . difference of a scalar and a list
*F  DiffListScl(<listL>,<listR>)  . . . . . difference of a list and a scalar
*F  DiffListList(<listL>,<listR>) . . . . . . . . . . difference of two lists
**
**  'DiffList' is  the   extended dispatcher for   the  differences involving
**  lists.  That  is, whenever two operands are  subtracted and at  least one
**  operand is a list and  'DiffFuncs' does not  point to a special function,
**  then 'DiffList' is called.   'DiffList' determines the extended  types of
**  the operands (e.g.,  'T_INT', 'T_VECTOR', 'T_MATRIX', 'T_LISTX') and then
**  dispatches through 'DiffFuncs' again.
**
**  'DiffSclList' is a  generic function  for  the first  kind of difference,
**  that of a scalar and a list.
**
**  'DiffListScl'  is a generic function for the  second kind of  difference,
**  that of a list and a scalar.
**
**  'DiffListList' is  a generic function for the  third  kind of difference,
**  that of two lists.
*/


libGAP_Obj             libGAP_DiffSclList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listD;          /* difference, result              */
    libGAP_Obj                 elmD;           /* one element of difference list  */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_Int                 len;            /* length                          */
    libGAP_Int                 i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_LIST( listR );
    listD = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(listL) || libGAP_IS_MUTABLE_OBJ(listR) ? libGAP_T_PLIST :
                      libGAP_T_PLIST+libGAP_IMMUTABLE, len );
    libGAP_SET_LEN_PLIST( listD, len );

    /* loop over the entries and subtract                                  */
    for ( i = 1; i <= len; i++ ) {
        elmR = libGAP_ELMV0_LIST( listR, i );
        if (elmR)
          {
            elmD = libGAP_DIFF( listL, elmR );
            libGAP_SET_ELM_PLIST( listD, i, elmD );
            libGAP_CHANGED_BAG( listD );
          }
    }

    /* Now adjust the result TNUM info */

    if (len == 0)
      libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( listR ))
      {
         if (libGAP_HAS_FILT_LIST(listR, libGAP_FN_IS_DENSE))
           libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_DENSE );
         else if (libGAP_HAS_FILT_LIST(listR, libGAP_FN_IS_NDENSE))
           libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_NDENSE );
      }
    /* return the result                                                   */
    return listD;
}

libGAP_Obj             libGAP_DiffListScl (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listD;          /* difference, result              */
    libGAP_Obj                 elmD;           /* one element of difference list  */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_Int                 len;            /* length                          */
    libGAP_Int                 i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_LIST( listL );
    listD = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(listL)|| libGAP_IS_MUTABLE_OBJ(listR) ? libGAP_T_PLIST :
                       libGAP_T_PLIST+libGAP_IMMUTABLE, len );
    libGAP_SET_LEN_PLIST( listD, len );

    /* loop over the entries and subtract                                  */
    for ( i = 1; i <= len; i++ ) {
        elmL = libGAP_ELMV0_LIST( listL, i );
        if (elmL)
          {
            elmD = libGAP_DIFF( elmL, listR );
            libGAP_SET_ELM_PLIST( listD, i, elmD );
            libGAP_CHANGED_BAG( listD );
          }
          
    }

    /* Now adjust the result TNUM info */

    if (len == 0)
      libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( listL ))
      {
         if (libGAP_HAS_FILT_LIST(listL, libGAP_FN_IS_DENSE))
           libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_DENSE );
         else if (libGAP_HAS_FILT_LIST(listL, libGAP_FN_IS_NDENSE))
           libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_NDENSE );
      }
    /* return the result                                                   */
    return listD;
}

libGAP_Obj             libGAP_DiffListList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listD;          /* difference, result              */
    libGAP_Obj                 elmD;           /* one element of the difference   */
    libGAP_Obj                 elmL;           /* one element of the left list    */
    libGAP_Obj                 elmR;           /* one element of the right list   */
    libGAP_Int                 lenL,lenR,lenD; /* length                          */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt                mutD;

    /* get and check the length                                            */
    lenL = libGAP_LEN_LIST( listL );
    lenR = libGAP_LEN_LIST( listR );
    lenD = (lenR > lenL) ? lenR : lenL;
    listD = libGAP_NEW_PLIST( (libGAP_IS_MUTABLE_OBJ(listL) || libGAP_IS_MUTABLE_OBJ(listR)) ?
                       libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE, lenD );
    libGAP_SET_LEN_PLIST( listD, lenD );

    /* Sort out mutability */
    mutD = 0;
    for (i = 1; i <= lenL; i++)
      if ((elmL = libGAP_ELM0_LIST( listL, i)))
        {
          mutD = libGAP_IS_MUTABLE_OBJ(elmL);
          break;
        }
    for (i = 1; i <= lenR; i++)
      if ((elmR = libGAP_ELM0_LIST( listR, i)))
        {
          mutD = mutD || libGAP_IS_MUTABLE_OBJ(elmR);
          break;
        }    
           
    
    /* loop over the entries and subtract                                  */
    for ( i = 1; i <= lenD; i++ ) {
        elmL = libGAP_ELM0_LIST( listL, i );
        elmR = libGAP_ELM0_LIST( listR, i );
        

        /* Now compute the result 6 different cases! */
        if (elmL)
          {
            if (elmR)
              elmD= libGAP_DIFF(elmL,elmR);
            else if ( mutD)
              elmD = libGAP_SHALLOW_COPY_OBJ(elmL);
            else
              elmD = elmL;
          }
        else if (elmR)
          {
            if (mutD)
              elmD = libGAP_AINV_MUT(elmR);
            else
              elmD = libGAP_AINV(elmR);
          }
        else
          elmD = 0;
          
        if (elmD)
          {
            libGAP_SET_ELM_PLIST( listD, i, elmD );
            libGAP_CHANGED_BAG( listD );
          }
    }
    /* Now adjust the result TNUM info. There's not so much we
       can say here with total reliability */

    if (lenD == 0)
      libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( listR ) && libGAP_IS_PLIST(listL) &&
             libGAP_HAS_FILT_LIST(listR, libGAP_FN_IS_DENSE) &&
             libGAP_HAS_FILT_LIST(listL, libGAP_FN_IS_DENSE))
      libGAP_SET_FILT_LIST( listD, libGAP_FN_IS_DENSE );
    
    /* return the result                                                   */
    return listD;
}

libGAP_Obj             libGAP_DiffSclListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_DiffSclList( listL, listR );
}

libGAP_Obj             libGAP_DiffListSclHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_DiffListScl( listL, listR );
}

libGAP_Obj             libGAP_DiffListListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_DiffListList( listL, listR );
}


/****************************************************************************
**
*F  ProdList(<listL>,<listR>) . . . . . . . . . . . . . . .  product of lists
*F  ProdSclList(<listL>,<listR>)  . . . . . .  product of a scalar and a list
*F  ProdListScl(<listL>,<listR>)  . . . . . .  product of a list and a scalar
*F  ProdListList(<listL>,<listR>) . . . . . . . . . . .  product of two lists
**
**  'ProdList' is the extended  dispatcher for the products  involving lists.
**  That is, whenever two operands are multiplied and at least one operand is
**  a list   and  'ProdFuncs' does not    point to a  special function,  then
**  'ProdList' is called.  'ProdList'   determines the extended types  of the
**  operands (e.g.,   'T_INT',  'T_VECTOR', 'T_MATRIX',  'T_LISTX')  and then
**  dispatches through 'ProdFuncs' again.
**
**  'ProdSclList' is a generic  function for the first  kind of product, that
**  of a scalar and a list.  Note that this  includes kind of product defines
**  the product of a matrix with a list of matrices.
**
**  'ProdListScl' is a generic function for the  second kind of product, that
**  of a  list  and a  scalar.  Note that   this kind of  product defines the
**  product of a  matrix with a vector, the  product of two matrices, and the
**  product of a list of matrices and a matrix.
**
**  'ProdListList' is a generic function for the third  kind of product, that
**  of two lists.  Note that this kind of product  defines the product of two
**  vectors, a vector and a matrix, and the product of a vector and a list of
**  matrices.
*/

libGAP_Obj             libGAP_ProdSclList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listP;          /* product, result                 */
    libGAP_Obj                 elmP;           /* one element of product list     */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_Int                 len;            /* length                          */
    libGAP_Int                 i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_LIST( listR );
    listP = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(listL) || libGAP_IS_MUTABLE_OBJ(listR) ?
                       libGAP_T_PLIST :libGAP_T_PLIST+libGAP_IMMUTABLE, len );
    libGAP_SET_LEN_PLIST( listP, len );

    /* loop over the entries and multiply                                  */
    for ( i = 1; i <= len; i++ ) {
        elmR = libGAP_ELMV0_LIST( listR, i );
        if (elmR)
          {
            elmP = libGAP_PROD( listL, elmR );
            libGAP_SET_ELM_PLIST( listP, i, elmP );
            libGAP_CHANGED_BAG( listP );
          }
    }
    if (len == 0)
      libGAP_SET_FILT_LIST( listP, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( listR ))
      {
         if (libGAP_HAS_FILT_LIST(listR, libGAP_FN_IS_DENSE))
           libGAP_SET_FILT_LIST( listP, libGAP_FN_IS_DENSE );
         else if (libGAP_HAS_FILT_LIST(listR, libGAP_FN_IS_NDENSE))
           libGAP_SET_FILT_LIST( listP, libGAP_FN_IS_NDENSE );
      }

    /* return the result                                                   */
    return listP;
}

libGAP_Obj             libGAP_ProdListScl (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listP;          /* product, result                 */
    libGAP_Obj                 elmP;           /* one element of product list     */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_Int                 len;            /* length                          */
    libGAP_Int                 i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_LIST( listL );
    listP = libGAP_NEW_PLIST( (libGAP_IS_MUTABLE_OBJ(listL) || libGAP_IS_MUTABLE_OBJ(listR))
                       ? libGAP_T_PLIST :libGAP_T_PLIST+libGAP_IMMUTABLE, len );
    libGAP_SET_LEN_PLIST( listP, len );

    /* loop over the entries and multiply                                  */
    for ( i = 1; i <= len; i++ ) {
        elmL = libGAP_ELMV0_LIST( listL, i );
        if (elmL) {
          elmP = libGAP_PROD( elmL, listR );
          libGAP_SET_ELM_PLIST( listP, i, elmP );
          libGAP_CHANGED_BAG( listP );
        }
    }

    if (len == 0)
      libGAP_SET_FILT_LIST( listP, libGAP_FN_IS_EMPTY );
    else if (libGAP_IS_PLIST( listL ))
      {
         if (libGAP_HAS_FILT_LIST(listL, libGAP_FN_IS_DENSE))
           libGAP_SET_FILT_LIST( listP, libGAP_FN_IS_DENSE );
         else if (libGAP_HAS_FILT_LIST(listL, libGAP_FN_IS_NDENSE))
           libGAP_SET_FILT_LIST( listP, libGAP_FN_IS_NDENSE );
      }
    /* return the result                                                   */
    return listP;
}

libGAP_Obj             libGAP_ProdListList (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Obj                 listP;          /* product, result                 */
    libGAP_Obj                 elmP;           /* one summand of the product      */
    libGAP_Obj                 elmL;           /* one element of the left list    */
    libGAP_Obj                 elmR;           /* one element of the right list   */
    libGAP_Int                 lenL,lenR,len; /* length                          */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Int                 imm;

    /* get and check the length                                            */
    lenL = libGAP_LEN_LIST( listL );
    lenR = libGAP_LEN_LIST( listR );
    len =  (lenL < lenR) ? lenL : lenR;
    /* loop over the entries and multiply and accumulate                   */
    listP = 0;
    imm = 0;
    for (i = 1; i <= len; i++)
      {
        elmL = libGAP_ELM0_LIST( listL, i );
        elmR = libGAP_ELM0_LIST( listR, i );
        if (elmL && elmR)
          {
            elmP = libGAP_PROD( elmL, elmR );
            if (listP)
              listP = libGAP_SUM( listP, elmP );
            else
              {
                listP = elmP;
                imm = !libGAP_IS_MUTABLE_OBJ(listP);
              }
          }
    }

    if (imm && libGAP_IS_MUTABLE_OBJ(listP))
      libGAP_MakeImmutable(listP);

    if (!listP)
      libGAP_ErrorMayQuit("Inner product multiplication of lists: no summands", 0, 0);
    
    /* return the result                                                   */
    return listP;
}

libGAP_Obj             libGAP_ProdSclListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_ProdSclList( listL, listR );
}

libGAP_Obj             libGAP_ProdListSclHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return libGAP_ProdListScl( listL, listR );
}

libGAP_Obj             libGAP_ProdListListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 listL,
    libGAP_Obj                 listR,
    libGAP_Obj                 depthdiff)
{
  libGAP_Obj prod;
  prod = libGAP_ProdListList( listL, listR );

  /* possibly adjust mutability */
  if (!libGAP_IS_MUTABLE_OBJ(prod))
    switch (libGAP_INT_INTOBJ(depthdiff)) {
    case -1:
      if (libGAP_IS_MUTABLE_OBJ(listL))
        prod = libGAP_SHALLOW_COPY_OBJ(prod);
      break;
    case 1:
      if (libGAP_IS_MUTABLE_OBJ(listR))
        prod = libGAP_SHALLOW_COPY_OBJ(prod);
      break;
    case 0:
      break;
    default:
      libGAP_ErrorReturnVoid("PROD_LIST_LIST_DEFAULT: depth difference should be -1, 0 or 1, not %i",
                      libGAP_INT_INTOBJ(depthdiff),0L,"you can return to carry on anyway");
    }
  return prod;
        
}


/****************************************************************************
**
*F  OneMatrix(<list>, <mut>) . . .  . . . . . . . . . . . . . . one of a list
**
**
**  'OneMatrix' is a generic function for the one. mut may be 0, for an
**  immutable result, 1 for a result of the same mutability level as <list>
**  and 2 for a fully mutable result.
*/

libGAP_Obj             libGAP_OneMatrix (
    libGAP_Obj                 mat,
    libGAP_UInt                mut)
{
    libGAP_Obj                 res = 0;        /* one, result                     */
    libGAP_Obj                 row;            /* one row of the result           */
    libGAP_Obj                 zero = 0;       /* zero element                    */
    libGAP_Obj                 one = 0;        /* one element                     */
    libGAP_UInt                len;            /* length (and width) of matrix    */
    libGAP_UInt                i, k;           /* loop variables                  */
    libGAP_UInt                rtype= 0;       /* tnum for rows of result        */
    libGAP_UInt                ctype = 0;      /* tnum for  result        */

    /* check that the operand is a *square* matrix                         */
    len = libGAP_LEN_LIST( mat );
    if ( len != libGAP_LEN_LIST( libGAP_ELM_LIST( mat, 1 ) ) ) {
        return libGAP_ErrorReturnObj(
            "Matrix ONE: <mat> must be square (not %d by %d)",
            (libGAP_Int)len, (libGAP_Int)libGAP_LEN_LIST( libGAP_ELM_LIST( mat, 1 ) ),
            "you can replace ONE matrix <mat> via 'return <mat>;'" );
    }

    /* get the zero and the one                                            */
    switch (mut) {
    case 0:
      zero = libGAP_ZERO_MUT( libGAP_ELM_LIST( libGAP_ELM_LIST( mat, 1 ), 1 ) );
      one  = libGAP_ONE_MUT( zero );
      libGAP_MakeImmutable(zero);
      libGAP_MakeImmutable(one);
      ctype = rtype = libGAP_T_PLIST+libGAP_IMMUTABLE;
      break;
      
    case 1:
      zero = libGAP_ZERO_MUT( libGAP_ELM_LIST( libGAP_ELM_LIST( mat, 1 ), 1 ) );
      one  = libGAP_ONE_MUT( zero );
      if (libGAP_IS_MUTABLE_OBJ(mat))
        {
          ctype = libGAP_T_PLIST;
          rtype = libGAP_IS_MUTABLE_OBJ(libGAP_ELM_LIST(mat, 1)) ? libGAP_T_PLIST : libGAP_T_PLIST + libGAP_IMMUTABLE;
        }
      else
        ctype = rtype = libGAP_T_PLIST + libGAP_IMMUTABLE;
      break;

    case 2:
      zero = libGAP_ZERO( libGAP_ELM_LIST( libGAP_ELM_LIST( mat, 1 ), 1 ) );
      one  = libGAP_ONE( zero );
      ctype = rtype = libGAP_T_PLIST;
      break;
    }

    

    /* make the identity matrix                                            */
    res = libGAP_NEW_PLIST(  ctype, len );
    libGAP_SET_LEN_PLIST( res, len );
    for ( i = 1; i <= len; i++ ) {
        row = libGAP_NEW_PLIST( rtype , len );
        libGAP_SET_LEN_PLIST( row, len );
        for ( k = 1; k <= len; k++ )
            libGAP_SET_ELM_PLIST( row, k, zero );
        libGAP_SET_ELM_PLIST( row, i, one );
        libGAP_SET_ELM_PLIST( res, i, row );
        libGAP_CHANGED_BAG( res );
    }

    /* return the identity matrix                                          */
    return res;
}

libGAP_Obj             libGAP_FuncOneMatrixImmutable (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_OneMatrix( list,0 );
}

libGAP_Obj             libGAP_FuncOneMatrixSameMutability (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_OneMatrix( list,1 );
}

libGAP_Obj             libGAP_FuncOneMatrixMutable (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_OneMatrix( list,2 );
}




/****************************************************************************
**
*F  InvList(<list>) . . . . . . . . . . . . . . . . . . . . inverse of a list
*F  InvMatrix(<list>) . . . . . . . . . . . . . . . . . . . inverse of a list
**
**  'InvList' is the extended dispatcher for  inverses involving lists.  That
**  is, whenever inverse for a list  is called and  'InvFuncs' does not point
**  to  a special function,  then 'InvList'  is called.  'InvList' determines
**  the  extended type of the  operand and then  dispatches through 'InvList'
**  again.
**
**  'InvMatrix' is a generic function for the inverse. In nearly all
**  circumstances, we should use a more efficient function based on
**  calls to AddRowVector, etc.
*/

libGAP_Obj             libGAP_InvMatrix (
    libGAP_Obj                 mat,
    libGAP_UInt                mut)
{
    libGAP_Obj                 res = 0;        /* power, result                   */
    libGAP_Obj                 row;            /* one row of the matrix           */
    libGAP_Obj                 row2;           /* another row of the matrix       */
    libGAP_Obj                 elm;            /* one element of the matrix       */
    libGAP_Obj                 elm2;           /* another element of the matrix   */
    libGAP_Obj                 zero = 0;       /* zero element                    */
    libGAP_Obj                 one = 0;        /* one element                     */
    libGAP_UInt                len;            /* length (and width) of matrix    */
    libGAP_UInt                i, k, l;        /* loop variables                  */
    libGAP_UInt                rtype = 0, ctype = 0;   /* types for lists to be created   */

    /* check that the operand is a *square* matrix                         */
    len = libGAP_LEN_LIST( mat );
    if ( len != libGAP_LEN_LIST( libGAP_ELM_LIST( mat, 1 ) ) ) {
        return libGAP_ErrorReturnObj(
            "Matrix INV: <mat> must be square (not %d by %d)",
            (libGAP_Int)len, (libGAP_Int)libGAP_LEN_LIST( libGAP_ELM_LIST( mat, 1 ) ),
            "you can replace <mat> via 'return <mat>;'" );
    }

    /* get the zero and the one                                            */
    switch(mut)
      {
      case 0:
        zero = libGAP_ZERO_MUT( libGAP_ELM_LIST( libGAP_ELM_LIST( mat, 1 ), 1 ) );
        one  = libGAP_ONE_MUT( zero );
        ctype = rtype = libGAP_T_PLIST+libGAP_IMMUTABLE;
        libGAP_MakeImmutable(zero);
        libGAP_MakeImmutable(one);
        break;
        
      case 1:
        zero = libGAP_ZERO_MUT( libGAP_ELM_LIST( libGAP_ELM_LIST( mat, 1 ), 1 ) );
        one  = libGAP_ONE_MUT( zero );
        if (libGAP_IS_MUTABLE_OBJ(mat))
          {
            ctype = libGAP_T_PLIST;
            rtype = libGAP_IS_MUTABLE_OBJ(libGAP_ELM_LIST(mat, 1)) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE;
          }
        else
          ctype = rtype = libGAP_T_PLIST+libGAP_IMMUTABLE;
        break;

      case 2:
        zero = libGAP_ZERO( libGAP_ELM_LIST( libGAP_ELM_LIST( mat, 1 ), 1 ) );
        one  = libGAP_ONE( zero );
        ctype = rtype = libGAP_T_PLIST;
        break;
      }

    /* make a matrix of the form $ ( Id_<len> | <mat> ) $                  */
    res = libGAP_NEW_PLIST( ctype, len );
    libGAP_SET_LEN_PLIST( res, len );
    for ( i = 1; i <= len; i++ ) {
        row = libGAP_NEW_PLIST( rtype, 2 * len );
        libGAP_SET_LEN_PLIST( row, 2 * len );
        libGAP_SET_ELM_PLIST( res, i, row );
        libGAP_CHANGED_BAG( res );
    }
    for ( i = 1; i <= len; i++ ) {
        row = libGAP_ELM_PLIST( res, i );
        for ( k = 1; k <= len; k++ )
            libGAP_SET_ELM_PLIST( row, k, zero );
        libGAP_SET_ELM_PLIST( row, i, one );
    }
    for ( i = 1; i <= len; i++ ) {
        row = libGAP_ELM_PLIST( res, i );
        row2 = libGAP_ELM_LIST( mat, i );
        for ( k = 1; k <= len; k++ ) {
            libGAP_SET_ELM_PLIST( row, k + len, libGAP_ELM_LIST( row2, k )  );
            libGAP_CHANGED_BAG( row );
        }
    }

    /* make row operations to reach form $ ( <inv> | Id_<len> ) $          */
    /* loop over the columns of <mat>                                      */
    for ( k = len+1; k <= 2*len; k++ ) {

        /* find a nonzero entry in this column                             */
        for ( i = k-len; i <= len; i++ ) {
            if ( ! libGAP_EQ( libGAP_ELM_PLIST( libGAP_ELM_PLIST(res,i), k ), zero ) )
               break;
        }
        if ( len < i ) {
            return libGAP_Fail;
        }

        /* make the row the <k>-th row and normalize it                    */
        row = libGAP_ELM_PLIST( res, i );
        libGAP_SET_ELM_PLIST( res, i, libGAP_ELM_PLIST( res, k-len ) );
        libGAP_SET_ELM_PLIST( res, k-len, row );
        if (mut < 2)
          elm2 = libGAP_INV_MUT( libGAP_ELM_PLIST( row, k ) );
        else
          elm2 = libGAP_INV( libGAP_ELM_PLIST( row, k ) );
        for ( l = 1; l <= 2*len; l++ ) {
            elm = libGAP_PROD( elm2, libGAP_ELM_PLIST( row, l ) );
            libGAP_SET_ELM_PLIST( row, l, elm );
            libGAP_CHANGED_BAG( row );
        }

        /* clear all entries in this column                                */
        for ( i = 1; i <= len; i++ ) {
            row2 = libGAP_ELM_PLIST( res, i );
            if (mut < 2)
              elm = libGAP_AINV_MUT( libGAP_ELM_PLIST( row2, k ) );
            else
              elm = libGAP_AINV( libGAP_ELM_PLIST( row2, k ) );
            if ( i != k-len && ! libGAP_EQ(elm,zero) ) {
                for ( l = 1; l <= 2*len; l++ ) {
                    elm2 = libGAP_PROD( elm, libGAP_ELM_PLIST( row, l ) );
                    elm2 = libGAP_SUM( libGAP_ELM_PLIST( row2, l ), elm2 );
                    libGAP_SET_ELM_PLIST( row2, l, elm2 );
                    libGAP_CHANGED_BAG( row2 );
                }
            }
        }

    }

    /* throw away the right halves of each row                             */
    for ( i = 1; i <= len; i++ ) {
        libGAP_SET_LEN_PLIST( libGAP_ELM_PLIST( res, i ), len );
        libGAP_SHRINK_PLIST(  libGAP_ELM_PLIST( res, i ), len );
    }

    /* return the result                                                   */
    return res;
}

libGAP_Obj libGAP_FuncINV_MATRIX_MUTABLE( libGAP_Obj self, libGAP_Obj mat)
{
  return libGAP_InvMatrix(mat, 2);
}

libGAP_Obj libGAP_FuncINV_MATRIX_SAME_MUTABILITY( libGAP_Obj self, libGAP_Obj mat)
{
  return libGAP_InvMatrix(mat, 1);
}

libGAP_Obj libGAP_FuncINV_MATRIX_IMMUTABLE( libGAP_Obj self, libGAP_Obj mat)
{
  return libGAP_InvMatrix(mat, 0);
}

/****************************************************************************
**
*F  FuncADD_ROW_VECTOR_5( <self>, <list1>, <list2>, <mult>, <from>, <to> )
**
**  This function adds <mult>*<list2>[i] destructively to <list1>[i] for
**  each i in the range <from>..<to>. It does very little checking
**
*/

/* We need these to redispatch when the user has supplied a replacement value. */

static libGAP_Obj libGAP_AddRowVectorOp;   /* BH changed to static */
static libGAP_Obj libGAP_MultRowVectorOp;  /* BH changed to static */

libGAP_Obj libGAP_FuncADD_ROW_VECTOR_5( libGAP_Obj self,
                          libGAP_Obj list1,
                          libGAP_Obj list2,
                          libGAP_Obj mult,
                          libGAP_Obj from,
                          libGAP_Obj to )
{
  libGAP_UInt i;
  libGAP_Obj el1,el2;
  while (!libGAP_IS_INTOBJ(to) ||
         libGAP_INT_INTOBJ(to) > libGAP_LEN_LIST(list1) ||
         libGAP_INT_INTOBJ(to) > libGAP_LEN_LIST(list2))
    to = libGAP_ErrorReturnObj("AddRowVector: Upper limit too large", 0L, 0L, 
                        "you can replace limit by <lim> via 'return <lim>;'");
  for (i = libGAP_INT_INTOBJ(from); i <= libGAP_INT_INTOBJ(to); i++)
    {
      el1 = libGAP_ELM_LIST(list1,i);
      el2 = libGAP_ELM_LIST(list2,i);
      el2 = libGAP_PROD(mult, el2);
      el1 = libGAP_SUM(el1,el2);
      libGAP_ASS_LIST(list1,i,el1);
      libGAP_CHANGED_BAG(list1);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncADD_ROW_VECTOR_5_FAST( <self>, <list1>, <list2>, <mult>, <from>, <to> )
**
**  This function adds <mult>*<list2>[i] destructively to <list1>[i] for
**  each i in the range <from>..<to>. It does very little checking
**
**  This version is specialised to the "fast" case where list1 and list2 are
**  plain lists of cyclotomics and mult is a small integers
*/
libGAP_Obj libGAP_FuncADD_ROW_VECTOR_5_FAST ( libGAP_Obj self,
                                libGAP_Obj list1,
                                libGAP_Obj list2,
                                libGAP_Obj mult,
                                libGAP_Obj from,
                                libGAP_Obj to )
{
  libGAP_UInt i;
  libGAP_Obj e1,e2, prd, sum;
  while (!libGAP_IS_INTOBJ(to) ||
         libGAP_INT_INTOBJ(to) > libGAP_LEN_LIST(list1) ||
         libGAP_INT_INTOBJ(to) > libGAP_LEN_LIST(list2))
    to = libGAP_ErrorReturnObj("AddRowVector: Upper limit too large", 0L, 0L, 
                        "you can replace limit by <lim> via 'return <lim>;'");
  for (i = libGAP_INT_INTOBJ(from); i <= libGAP_INT_INTOBJ(to); i++)
    {
      e1 = libGAP_ELM_PLIST(list1,i);
      e2 = libGAP_ELM_PLIST(list2,i);
      if ( !libGAP_ARE_INTOBJS( e2, mult ) || !libGAP_PROD_INTOBJS( prd, e2, mult ))
        {
          prd = libGAP_PROD(e2,mult);
        }
      if ( !libGAP_ARE_INTOBJS(e1, prd) || !libGAP_SUM_INTOBJS( sum, e1, prd) )
        {
          sum = libGAP_SUM(e1,prd);
          libGAP_SET_ELM_PLIST(list1,i,sum);
          libGAP_CHANGED_BAG(list1);
        }
      else
          libGAP_SET_ELM_PLIST(list1,i,sum);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncADD_ROW_VECTOR_3( <self>, <list1>, <list2>, <mult> )
**
**  This function adds <mult>*<list2>[i] destructively to <list1>[i] for
**  each i in the range 1..Length(<list1>). It does very little checking
**
*T  This could be speeded up still further by using special code for various
**  types of list -- this version just uses generic list ops
*/
libGAP_Obj libGAP_FuncADD_ROW_VECTOR_3( libGAP_Obj self,
                          libGAP_Obj list1,
                          libGAP_Obj list2,
                          libGAP_Obj mult)
{
  libGAP_UInt i;
  libGAP_UInt len = libGAP_LEN_LIST(list1);
  libGAP_Obj el1, el2;
  if (libGAP_LEN_LIST(list2) != len)
    {
      list2 = libGAP_ErrorReturnObj("AddRowVector: lists must be the same length",
                             0L, 0L, 
                             "you can replace second list <list2> via 'return <list2>;'");
      return libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, list1, list2,mult);
    }
  for (i = 1; i <= len; i++)
    {
      el1 = libGAP_ELMW_LIST(list1,i);
      el2 = libGAP_ELMW_LIST(list2,i);
      el2 = libGAP_PROD(mult, el2);
      el1 = libGAP_SUM(el1 , el2);
      libGAP_ASS_LIST(list1,i,el1);
      libGAP_CHANGED_BAG(list1);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncADD_ROW_VECTOR_3_FAST( <self>, <list1>, <list2>, <mult> )
**
**  This function adds <mult>*<list2>[i] destructively to <list1>[i] for
**  each i in the range 1..Length(<list1>). It does very little checking
**
**  This version is specialised to the "fast" case where list1 and list2 are
**  plain lists of cyclotomics and mult is a small integers
*/
libGAP_Obj libGAP_FuncADD_ROW_VECTOR_3_FAST ( libGAP_Obj self,
                                libGAP_Obj list1,
                                libGAP_Obj list2,
                                libGAP_Obj mult )
{
  libGAP_UInt i;
  libGAP_Obj e1,e2, prd, sum;
  libGAP_UInt len = libGAP_LEN_PLIST(list1);
  if (libGAP_LEN_PLIST(list2) != len)
    {
      list2 = libGAP_ErrorReturnObj("AddRowVector: lists must be the same length",
                             0L, 0L, 
                             "you can replace second list <list2> via 'return <list2>;'");
      return libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, list1, list2, mult);
    }
      
  for (i = 1; i <= len; i++)
    {
      e1 = libGAP_ELM_PLIST(list1,i);
      e2 = libGAP_ELM_PLIST(list2,i);
      if ( !libGAP_ARE_INTOBJS( e2, mult ) || !libGAP_PROD_INTOBJS( prd, e2, mult ))
        {
          prd = libGAP_PROD(e2,mult);
        }
      if ( !libGAP_ARE_INTOBJS(e1, prd) || !libGAP_SUM_INTOBJS( sum, e1, prd) )
        {
          sum = libGAP_SUM(e1,prd);
          libGAP_SET_ELM_PLIST(list1,i,sum);
          libGAP_CHANGED_BAG(list1);
        }
      else
          libGAP_SET_ELM_PLIST(list1,i,sum);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncADD_ROW_VECTOR_2( <self>, <list1>, <list2>)
**
**  This function adds <list2>[i] destructively to <list1>[i] for
**  each i in the range 1..Length(<list1>). It does very little checking
**
*T  This could be speeded up still further by using special code for various
**  types of list -- this version just uses generic list ops
*/
libGAP_Obj libGAP_FuncADD_ROW_VECTOR_2( libGAP_Obj self,
                          libGAP_Obj list1,
                          libGAP_Obj list2)
{
  libGAP_UInt i;
  libGAP_Obj el1,el2;
  libGAP_UInt len = libGAP_LEN_LIST(list1);
  if (libGAP_LEN_LIST(list2) != len)
    {
      list2 = libGAP_ErrorReturnObj("AddRowVector: lists must be the same length",
                             0L, 0L, 
                             "you can replace second list <list2> via 'return <list2>;'");
      return libGAP_CALL_2ARGS(libGAP_AddRowVectorOp, list1, list2);
    }
  for (i = 1; i <= len; i++)
    {
      el1 = libGAP_ELMW_LIST(list1,i);
      el2 = libGAP_ELMW_LIST(list2,i);
      el1 = libGAP_SUM(el1, el2 );
      libGAP_ASS_LIST(list1,i,el1);
      libGAP_CHANGED_BAG(list1);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncADD_ROW_VECTOR_2_FAST( <self>, <list1>, <list2> )
**
**  This function adds <list2>[i] destructively to <list1>[i] for
**  each i in the range 1..Length(<list1>). It does very little checking
**
**  This version is specialised to the "fast" case where list1 and list2 are
**  plain lists of cyclotomics 
*/
libGAP_Obj libGAP_FuncADD_ROW_VECTOR_2_FAST ( libGAP_Obj self,
                                libGAP_Obj list1,
                                libGAP_Obj list2 )
{
  libGAP_UInt i;
  libGAP_Obj e1,e2, sum;
  libGAP_UInt len = libGAP_LEN_PLIST(list1);
  if (libGAP_LEN_PLIST(list2) != len)
    {
      list2 = libGAP_ErrorReturnObj("AddRowVector: lists must be the same length",
                             0L, 0L, 
                             "you can replace second list <list2> via 'return <list2>;'");
      return libGAP_CALL_2ARGS(libGAP_AddRowVectorOp, list1, list2);
    }
  for (i = 1; i <= len; i++)
    {
      e1 = libGAP_ELM_PLIST(list1,i);
      e2 = libGAP_ELM_PLIST(list2,i);
      if ( !libGAP_ARE_INTOBJS(e1, e2) || !libGAP_SUM_INTOBJS( sum, e1, e2) )
        {
          sum = libGAP_SUM(e1,e2);
          libGAP_SET_ELM_PLIST(list1,i,sum);
          libGAP_CHANGED_BAG(list1);
        }
      else
          libGAP_SET_ELM_PLIST(list1,i,sum);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncMULT_ROW_VECTOR_2( <self>, <list>, <mult> )
**
**  This function destructively multiplies the entries of <list> by <mult>
**  It does very little checking
**
*/

libGAP_Obj libGAP_FuncMULT_ROW_VECTOR_2( libGAP_Obj self,
                           libGAP_Obj list,
                           libGAP_Obj mult )
{
  libGAP_UInt i;
  libGAP_Obj prd;
  libGAP_UInt len = libGAP_LEN_LIST(list);
  for (i = 1; i <= len; i++)
    {
      prd = libGAP_ELMW_LIST(list,i);
      prd = libGAP_PROD(prd,mult);
      libGAP_ASS_LIST(list,i,prd);
      libGAP_CHANGED_BAG(list);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncMULT_ROW_VECTOR_2_FAST( <self>, <list>, <mult> )
**
**  This function destructively multiplies the entries of <list> by <mult>
**  It does very little checking
**
**  This is the fast method for plain lists of cyclotomics and an integer
**  multiplier
*/

libGAP_Obj libGAP_FuncMULT_ROW_VECTOR_2_FAST( libGAP_Obj self,
                                libGAP_Obj list,
                                libGAP_Obj mult )
{
  libGAP_UInt i;
  libGAP_Obj el,prd;
  libGAP_UInt len = libGAP_LEN_PLIST(list);
  for (i = 1; i <= len; i++)
    {
      el = libGAP_ELM_PLIST(list,i);
      if (!libGAP_ARE_INTOBJS(el, mult) || !libGAP_PROD_INTOBJS(prd,el,mult))
        {
          prd = libGAP_PROD(el,mult);
          libGAP_SET_ELM_PLIST(list,i,prd);
          libGAP_CHANGED_BAG(list);
        }
      else
          libGAP_SET_ELM_PLIST(list,i,prd);
    }
  return 0;
}

/****************************************************************************
**
*F  FuncPROD_VEC_MAT_DEFAULT( <self>, <vec>, <mat> )
**
**  This is a specialized version of PROD_LIST_LIST_DEFAULT, that uses
**  AddRowVector rather than SUM and PROD.
*/


libGAP_Obj libGAP_FuncPROD_VEC_MAT_DEFAULT( libGAP_Obj self,
                              libGAP_Obj vec,
                              libGAP_Obj mat )
{
  libGAP_Obj res;
  libGAP_Obj elt;
  libGAP_Obj vecr;
  libGAP_UInt i,len;
  libGAP_Obj z;
/*Obj o;  */
  res = (libGAP_Obj) 0;
  len = libGAP_LEN_LIST(vec);
  if (len != libGAP_LEN_LIST(mat))
    {
      mat = libGAP_ErrorReturnObj("<vec> * <mat>: vector and matrix must have same length", 0L, 0L,
                           "you can replace <mat> via 'return <mat>;'");
      return libGAP_PROD(vec,mat);
    }
  elt = libGAP_ELMW_LIST(vec,1);
  z = libGAP_ZERO(elt);
  for (i = 1; i <= len; i++)
    {
      elt = libGAP_ELMW_LIST(vec,i);
      if (!libGAP_EQ(elt,z))
        {
          vecr = libGAP_ELMW_LIST(mat,i);
          if (res == (libGAP_Obj)0)
            {
              res = libGAP_SHALLOW_COPY_OBJ(vecr);
              libGAP_CALL_2ARGS(libGAP_MultRowVectorOp,res,elt);
            }
          else
            libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, res, vecr, elt);
        }
    }
  if (res == (libGAP_Obj)0)
    res = libGAP_ZERO(libGAP_ELMW_LIST(mat,1));
  if (!libGAP_IS_MUTABLE_OBJ(vec) && !libGAP_IS_MUTABLE_OBJ(mat))
    libGAP_MakeImmutable(res);
  return res;
}

/****************************************************************************
**
*F  FuncINV_MAT_DEFAULT
**
**  A faster version of InvMat for those matrices for whose rows AddRowVector
** and MultRowVector make sense (and might have fast kernel methods)
**
*/

libGAP_Obj libGAP_ConvertToMatrixRep;

libGAP_Obj libGAP_InvMatWithRowVecs( libGAP_Obj mat, libGAP_UInt mut)
{
  libGAP_Obj                 res;            /* result                          */
  libGAP_Obj                 matcopy;        /* copy of mat                     */
  libGAP_Obj                 row = 0;            /* one row of matcopy              */
  libGAP_Obj                 row2;           /* corresponding row of res        */
  libGAP_Obj                 row3;           /* another row of matcopy          */
  libGAP_Obj                 x = 0;              /* one element of the matrix       */
  libGAP_Obj                 xi;             /* 1/x                             */
  libGAP_Obj                 y;              /* another element of the matrix   */
  libGAP_Obj                 yi;             /* -y                              */
  libGAP_Obj                 zero;           /* zero element                    */
  libGAP_Obj                 zerov;          /* zero vector                     */
  libGAP_Obj                 one;            /* one element                     */
  libGAP_UInt                len;            /* length (and width) of matrix    */
  libGAP_UInt                i, k, j;        /* loop variables                  */

  /* check that the operand is a *square* matrix                         */
  len = libGAP_LEN_LIST( mat );
  if ( len != libGAP_LEN_LIST( libGAP_ELM_LIST( mat, 1 ) ) ) {
    mat = libGAP_ErrorReturnObj(
                         "Matrix INV: <mat> must be square (not %d by %d)",
                         (libGAP_Int)len, (libGAP_Int)libGAP_LEN_LIST( libGAP_ELM_LIST( mat, 1 ) ),
                         "you can replace <mat> via 'return <mat>;'" );
    switch( mut) {
    case 0:
      res = libGAP_INV(mat);
      libGAP_MakeImmutable(res);
      return res;
      break;
    case 1:
      return libGAP_INV_MUT(mat);
      break;
    case 2:
      return libGAP_INV(mat);
      break;
    }
  }

  /* get the zero and the one                                            */
  zerov = libGAP_ZERO( libGAP_ELMW_LIST(mat, 1));
  zero = libGAP_ZERO( libGAP_ELMW_LIST( libGAP_ELMW_LIST( mat, 1 ), 1 ) );
  one  = libGAP_ONE( zero );
    
  /* set up res (initially the identity) and matcopy */
  res = libGAP_NEW_PLIST(libGAP_T_PLIST,len);
  matcopy = libGAP_NEW_PLIST(libGAP_T_PLIST,len);
  libGAP_SET_LEN_PLIST(res,len);
  libGAP_SET_LEN_PLIST(matcopy,len);
  for (i = 1; i <= len; i++)
    {
      row = libGAP_SHALLOW_COPY_OBJ(zerov);
      libGAP_ASS_LIST(row,i,one);
      libGAP_SET_ELM_PLIST(res,i,row);
      libGAP_CHANGED_BAG(res);
      row = libGAP_SHALLOW_COPY_OBJ(libGAP_ELM_LIST(mat,i));
      libGAP_SET_ELM_PLIST(matcopy,i,row);
      libGAP_CHANGED_BAG(matcopy);
    }


  /* Now to work, make matcopy an identity by row operations and
     do the same row operations to res */

  /* outer loop over columns of matcopy */
  for (i = 1; i <= len; i++)
    {
      /* Find a non-zero leading entry that is in that column */
      for (j = i; j <= len; j++)
        {
          row = libGAP_ELM_PLIST(matcopy,j);
          x = libGAP_ELMW_LIST(row,i);
          if (!libGAP_EQ(x,zero))
            break;
        }

      /* if there isn't one then the matrix is not invertible */
      if (j > len)
        return libGAP_Fail;

      /* Maybe swap two rows */
      /* But I will want this value anyway */
      row2 = libGAP_ELM_PLIST(res,j);
      if (j != i)
        {
          libGAP_SET_ELM_PLIST(matcopy,j,libGAP_ELM_PLIST(matcopy,i));
          libGAP_SET_ELM_PLIST(res,j,libGAP_ELM_PLIST(res,i));
          libGAP_SET_ELM_PLIST(matcopy,i,row);
          libGAP_SET_ELM_PLIST(res,i,row2);
        }

      /*Maybe rescale the row */
      if (!libGAP_EQ(x, one))
        {
          xi = libGAP_INV(x);
          libGAP_CALL_2ARGS(libGAP_MultRowVectorOp, row, xi);
          libGAP_CALL_2ARGS(libGAP_MultRowVectorOp, row2, xi);
        }

      /* Clear the entries. We know that we can ignore the entries in rows i..j */
      for (k = 1; k < i; k++)
        {
          row3 = libGAP_ELM_PLIST(matcopy,k);
          y = libGAP_ELMW_LIST(row3,i);
          if (!libGAP_EQ(y,zero))
            {
              yi = libGAP_AINV(y);
              libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, row3, row, yi);
              libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, libGAP_ELM_PLIST(res,k), row2, yi);
            }
        }
      for (k = j+1; k <= len; k++)
        {
          row3 = libGAP_ELM_PLIST(matcopy,k);
          y = libGAP_ELMW_LIST(row3,i);
          if (!libGAP_EQ(y,zero))
            {
              yi = libGAP_AINV(y);
              libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, row3, row, yi);
              libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, libGAP_ELM_PLIST(res,k), row2, yi);
            }
        }
                                 
    }

  /* Now we adjust mutability. Couldn't do it earlier, because AddRowVector, etc.
     needs mutable target vectors */
  switch (mut)
    {
    case 0:
      libGAP_MakeImmutable(res);
      break;
      
    case 1:
      if (libGAP_IS_MUTABLE_OBJ(mat))
        {
          if (!libGAP_IS_MUTABLE_OBJ(libGAP_ELM_LIST(mat,1)))
            for (i = 1; i <= len; i++)
              libGAP_MakeImmutable(libGAP_ELM_LIST(res,i));
        }
      else
        libGAP_MakeImmutable(res);
      break;
    case 2:
      break;
    }
  
  return res;
}


libGAP_Obj libGAP_FuncINV_MAT_DEFAULT_MUTABLE ( libGAP_Obj self, libGAP_Obj mat)
{
  return libGAP_InvMatWithRowVecs(mat, 2);
}

libGAP_Obj libGAP_FuncINV_MAT_DEFAULT_SAME_MUTABILITY ( libGAP_Obj self, libGAP_Obj mat)
{
  return libGAP_InvMatWithRowVecs(mat, 1);
}

libGAP_Obj libGAP_FuncINV_MAT_DEFAULT_IMMUTABLE ( libGAP_Obj self, libGAP_Obj mat)
{
  return libGAP_InvMatWithRowVecs(mat, 0);
}

  
/****************************************************************************
**
*F  FuncADD_TO_LIST_ENTRIES_PLIST_RANGE( <list>, <range>, <x> )
**
**  This is a method for the operation AddToListEntries, used mainly in
** the MPQS of Stefan Kohl.
**
**  This method requires a plain list for the first argument, and a
**  range for the second and is optimised for the case where x and the list
** entries are small integers, as are their sums
*/

libGAP_Obj libGAP_FuncADD_TO_LIST_ENTRIES_PLIST_RANGE ( 
                              libGAP_Obj self,
                              libGAP_Obj list,
                              libGAP_Obj range,
                              libGAP_Obj x)
{
  libGAP_UInt low, high, incr;
  libGAP_Obj y,z;
  libGAP_UInt i;
  if (!libGAP_IS_INTOBJ(x))
    return libGAP_TRY_NEXT_METHOD;
  low = libGAP_GET_LOW_RANGE(range);
  incr = libGAP_GET_INC_RANGE(range);
  high = low + incr*(libGAP_GET_LEN_RANGE(range)-1);
  for (i = low; i <= high; i+= incr)
    {
      y = libGAP_ELM_PLIST(list,i);
      if (!libGAP_IS_INTOBJ(y) ||
          !libGAP_SUM_INTOBJS(z,x,y))
        {
          z = libGAP_SUM(x,y);
          libGAP_SET_ELM_PLIST(list,i,z);
          libGAP_CHANGED_BAG(list);
        }
      else
        libGAP_SET_ELM_PLIST(list,i,z);
        
    }
  return (libGAP_Obj) 0;
}


/****************************************************************************
**
*F  MONOM_TOT_DEG_LEX( u, v ) . . . . . total degree lexicographical ordering
**
**  This function  implements the total degree  plus lexicographical ordering
**  for monomials  of commuting indeterminates.   It is in this  file because
**  monomials  are presently implemented  as lists  of indeterminate-exponent
**  pairs.  Should there be  more functions supporting polynomial arithmetic,
**  then this function should go into a separate file.
**
**  Examples:      x^2y^3 < y^7,   x^4 y^5 < x^3 y^6
*/
static libGAP_Obj  libGAP_FuncMONOM_TOT_DEG_LEX ( libGAP_Obj self, libGAP_Obj u, libGAP_Obj  v ) {

  libGAP_Int4 i, lu, lv;

  libGAP_Obj  total;
  libGAP_Obj  lexico;
 
  while ( !(libGAP_T_PLIST<=libGAP_TNUM_OBJ(u) && libGAP_TNUM_OBJ(u)<=libGAP_LAST_PLIST_TNUM)
          || !libGAP_IS_DENSE_LIST(u)) {
      u = libGAP_ErrorReturnObj(
      "MONOM_TOT_DEG_LEX: first <list> must be a dense plain list (not a %s)", 
      (libGAP_Int)libGAP_TNAM_OBJ(u), 0L, "you can replace <list> via 'return <list>;'" );
  }
  while ( !(libGAP_T_PLIST<=libGAP_TNUM_OBJ(v) && libGAP_TNUM_OBJ(v)<=libGAP_LAST_PLIST_TNUM) ||
          !libGAP_IS_DENSE_LIST(v)) {
      v = libGAP_ErrorReturnObj(
      "MONOM_TOT_DEG_LEX: first <list> must be a dense plain list (not a %s)", 
      (libGAP_Int)libGAP_TNAM_OBJ(v), 0L, "you can replace <list> via 'return <list>;'" );
  }
    
  lu = libGAP_LEN_PLIST( u );
  lv = libGAP_LEN_PLIST( v );

  /* strip off common prefixes                                             */
  i = 1;
  while ( i <= lu && i <= lv && libGAP_EQ(libGAP_ELM_PLIST( u,i ), libGAP_ELM_PLIST( v,i )) ) 
      i++;
 
  /* Is u a prefix of v ? Return true if u is a proper prefix.             */
  if ( i > lu ) return (lu < lv) ? libGAP_True : libGAP_False;

  /* Is v a  prefix of u ?                                                 */
  if ( i > lv ) return libGAP_False;
 
  /* Now determine the lexicographic order.  The monomial is interpreted   */
  /* as a string of indeterminates.                                        */
  if ( i % 2 == 1 ) {
    /* The first difference between u and v is an indeterminate.           */
    lexico = libGAP_LT(libGAP_ELM_PLIST( u, i ), libGAP_ELM_PLIST( v, i )) ? libGAP_True : libGAP_False;
    i++;  
  }
  else {
    /* The first difference between u and v is an exponent.                */
    lexico = libGAP_LT(libGAP_ELM_PLIST( v, i ), libGAP_ELM_PLIST( u, i )) ? libGAP_True : libGAP_False;
  }
 
  /* Now add up the remaining exponents in order to compare the total      */
  /* degrees.                                                              */
  total = libGAP_INTOBJ_INT(0);
  while ( i <= lu && i <= lv )  {
    libGAP_C_SUM_FIA(  total, total, libGAP_ELM_PLIST( u, i ) );
    libGAP_C_DIFF_FIA( total, total, libGAP_ELM_PLIST( v, i ) );
    i += 2;
  }

  /* Only one of the following while loops is executed                     */
  while ( i <= lu ) {
    libGAP_C_SUM_FIA(  total, total, libGAP_ELM_PLIST( u, i ) );
    i += 2;
  }
  while ( i <= lv ) {
    libGAP_C_DIFF_FIA( total, total, libGAP_ELM_PLIST( v, i ) );
    i += 2;
  }
 
  if ( libGAP_EQ( total, libGAP_INTOBJ_INT(0)) ) return lexico;

  return libGAP_LT( total, libGAP_INTOBJ_INT(0)) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  MONOM_GRLLEX( u, v ) . . . . . ``grlex'' ordering for internal monomials
**
**  This function  implements the ``grlex'' (degree, then lexicographic) ordering
**  for monomials  of commuting indeterminates with x_1>x_2>x_3 etc. (this
**  is standard textbook usage). It is in this  file because
**  monomials  are presently implemented  as lists  of indeterminate-exponent
**  pairs.  Should there be  more functions supporting polynomial arithmetic,
**  then this function should go into a separate file.
**
**  Examples:      x^2y^3 < y^7,   x^4 y^5 < x^3 y^6
*/
static libGAP_Obj  libGAP_FuncMONOM_GRLEX( libGAP_Obj self, libGAP_Obj u, libGAP_Obj  v ) {

  libGAP_Int4 i, lu, lv;

  libGAP_Obj  total,ai,bi;
 
  while ( !(libGAP_T_PLIST<=libGAP_TNUM_OBJ(u) && libGAP_TNUM_OBJ(u)<=libGAP_LAST_PLIST_TNUM)
          || !libGAP_IS_DENSE_LIST(u)) {
      u = libGAP_ErrorReturnObj(
      "MONOM_TOT_DEG_LEX: first <list> must be a dense plain list (not a %s)", 
      (libGAP_Int)libGAP_TNAM_OBJ(u), 0L, "you can replace <list> via 'return <list>;'" );
  }
  while ( !(libGAP_T_PLIST<=libGAP_TNUM_OBJ(v) && libGAP_TNUM_OBJ(v)<=libGAP_LAST_PLIST_TNUM) ||
          !libGAP_IS_DENSE_LIST(v)) {
      v = libGAP_ErrorReturnObj(
      "MONOM_TOT_DEG_LEX: first <list> must be a dense plain list (not a %s)", 
      (libGAP_Int)libGAP_TNAM_OBJ(v), 0L, "you can replace <list> via 'return <list>;'" );
  }
    
  lu = libGAP_LEN_PLIST( u );
  lv = libGAP_LEN_PLIST( v );

  /* compare the total degrees */
  total = libGAP_INTOBJ_INT(0);
  for (i=2;i<=lu;i+=2) {
    libGAP_C_SUM_FIA(  total, total, libGAP_ELM_PLIST( u, i ) );
  }

  for (i=2;i<=lv;i+=2) {
    libGAP_C_DIFF_FIA(  total, total, libGAP_ELM_PLIST( v, i ) );
  }

  if ( ! (libGAP_EQ( total, libGAP_INTOBJ_INT(0))) ) {
    /* degrees differ, use these */
    return libGAP_LT( total, libGAP_INTOBJ_INT(0)) ? libGAP_True : libGAP_False;
  }

  /* now use lexicographic ordering */
  i=1;
  while (i<=lu && i<=lv) {
    ai=libGAP_ELM_PLIST(u,i);
    bi=libGAP_ELM_PLIST(v,i);
    if (libGAP_LT(bi,ai)) {
      return libGAP_True;
    }
    if (libGAP_LT(ai,bi)) {
      return libGAP_False;
    }
    ai=libGAP_ELM_PLIST(u,i+1);
    bi=libGAP_ELM_PLIST(v,i+1);
    if (libGAP_LT(ai,bi)) {
      return libGAP_True;
    }
    if (libGAP_LT(bi,ai)) {
      return libGAP_False;
    }
    i+=2;
  }
  if (i<lv) {
      return libGAP_True;
  }
  return libGAP_False;
}


/****************************************************************************
**
*F  ZIPPED_SUM_LISTS(z1,z2,zero,f)
**
**  implements the `ZippedSum' function to add polynomials in external
**  representation. This is time critical and thus in the kernel.
**  the function assumes that all lists are plists.
*/
static libGAP_Obj  libGAP_FuncZIPPED_SUM_LISTS( libGAP_Obj self, libGAP_Obj z1, libGAP_Obj  z2, libGAP_Obj zero, libGAP_Obj f ) {

  libGAP_Int l1,l2,i;
  libGAP_Int i1,i2;
  libGAP_Obj sum,x,y;
  libGAP_Obj cmpfun,sumfun,a,b,c;

  l1=libGAP_LEN_LIST(z1);
  l2=libGAP_LEN_LIST(z2);
  cmpfun=libGAP_ELM_LIST(f,1);
  sumfun=libGAP_ELM_LIST(f,2);
  sum=libGAP_NEW_PLIST(libGAP_T_PLIST,0);
  libGAP_SET_LEN_PLIST(sum,0);
  i1=1;
  i2=1;
  while ((i1<=l1) && (i2<=l2)) {
/* Pr("A= %d %d\n",i1,i2); */
    /* if z1[i1] = z2[i2] then */
    a=libGAP_ELM_PLIST(z1,i1);
    b=libGAP_ELM_PLIST(z2,i2);
    if (libGAP_EQ(a,b)) {
      /* the entries are equal */
      x=libGAP_ELM_PLIST(z1,i1+1);
      y=libGAP_ELM_PLIST(z2,i2+1);
/* Pr("EQ, %d %d\n",INT_INTOBJ(x),INT_INTOBJ(y)); */
      c=libGAP_CALL_2ARGS(sumfun,x,y);
      if (!(libGAP_EQ(c,zero))) {
/* Pr("Added %d\n",INT_INTOBJ(c),0L); */
        libGAP_AddList(sum,a);
        libGAP_AddList(sum,c);
      }
      i1=i1+2;
      i2=i2+2;
    }
    else {
      /* compare */
      a=libGAP_ELM_PLIST(z1,i1); /* in case the EQ triggered a GC */
      b=libGAP_ELM_PLIST(z2,i2);
/* Pr("B= %d %d\n",ELM_LIST(a,1),ELM_LIST(b,1)); */
      c=libGAP_CALL_2ARGS(cmpfun,a,b);
/* Pr("C= %d %d\n",c,0L); */

      if ( /* this construct is taken from the compiler */
          (libGAP_Obj)(libGAP_UInt)(c != libGAP_False) ) {
        a=libGAP_ELM_PLIST(z1,i1); 
        libGAP_AddList(sum,a);
        c=libGAP_ELM_PLIST(z1,i1+1);
        libGAP_AddList(sum,c);
        i1=i1+2;
      }
      else {
        b=libGAP_ELM_PLIST(z2,i2); 
        libGAP_AddList(sum,b);
        c=libGAP_ELM_PLIST(z2,i2+1);
        libGAP_AddList(sum,c);
        i2=i2+2;
      } /* else */
    } /*else (elif)*/
  } /* while */

  for (i=i1;i<l1;i+=2) {
    libGAP_AddList(sum,libGAP_ELM_PLIST(z1,i));
    libGAP_AddList(sum,libGAP_ELM_PLIST(z1,i+1));
  }

  for (i=i2;i<l2;i+=2) {
    libGAP_AddList(sum,libGAP_ELM_PLIST(z2,i));
    libGAP_AddList(sum,libGAP_ELM_PLIST(z2,i+1));
  }
  return sum;

}

/****************************************************************************
**
*F  FuncMONOM_PROD(m1,m2)
**
**  implements the multiplication of monomials. Both must be plain lists 
**  of integers.
*/
static libGAP_Obj  libGAP_FuncMONOM_PROD( libGAP_Obj self, libGAP_Obj m1, libGAP_Obj m2 ) {

   libGAP_UInt a,b,l1,l2,i1,i2,i;
   libGAP_Obj e,f,c,prod;

   prod=libGAP_NEW_PLIST(libGAP_T_PLIST,0);
   libGAP_SET_LEN_PLIST(prod,0);
   l1=libGAP_LEN_LIST(m1);
   l2=libGAP_LEN_LIST(m2);
   i1=1;
   i2=1;
   while ((i1<l1) && (i2<l2)) {
     /* assume <2^28 variables) */
     a=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(m1,i1));
     e=libGAP_ELM_PLIST(m1,i1+1);
     b=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(m2,i2));
     f=libGAP_ELM_PLIST(m2,i2+1);
     if (a==b) {
       libGAP_C_SUM_FIA(c,e,f); /* c=e+f, fast */
       libGAP_AddList(prod,libGAP_INTOBJ_INT(a));
       libGAP_AddList(prod,c);
       i1+=2;
       i2+=2;
     }
     else {
       if (a<b) {
         libGAP_AddList(prod,libGAP_INTOBJ_INT(a));
         libGAP_AddList(prod,e);
         i1+=2;
       }
       else {
         libGAP_AddList(prod,libGAP_INTOBJ_INT(b));
         libGAP_AddList(prod,f);
         i2+=2;
       }
     }

   }

  for (i=i1;i<l1;i+=2) {
    libGAP_AddList(prod,libGAP_ELM_PLIST(m1,i));
    libGAP_AddList(prod,libGAP_ELM_PLIST(m1,i+1));
  }

  for (i=i2;i<l2;i+=2) {
    libGAP_AddList(prod,libGAP_ELM_PLIST(m2,i));
    libGAP_AddList(prod,libGAP_ELM_PLIST(m2,i+1));
  }
  return prod;

}




/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * * */


/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "EQ_LIST_LIST_DEFAULT", 2, "listL, listR",
      libGAP_EqListListHandler, "src/listoper.c:EQ_LIST_LIST_DEFAULT" },

    { "LT_LIST_LIST_DEFAULT", 2, "listL, listR",
      libGAP_LtListListHandler, "src/listoper.c:LT_LIST_LIST_DEFAULT" },

    { "IN_LIST_DEFAULT", 2, "obj, list",
      libGAP_InListDefaultHandler, "src/listoper.c:IN_LIST_DEFAULT" },

    { "SUM_SCL_LIST_DEFAULT", 2, "listL, listR",
      libGAP_SumSclListHandler, "src/listoper.c:SUM_SCL_LIST_DEFAULT" },

    { "SUM_LIST_SCL_DEFAULT", 2, "listL, listR",
      libGAP_SumListSclHandler, "src/listoper.c:SUM_LIST_SCL_DEFAULT" },

    { "SUM_LIST_LIST_DEFAULT", 2, "listL, listR",
      libGAP_SumListListHandler, "src/listoper.c:SUM_LIST_LIST_DEFAULT" },

    { "ZERO_LIST_DEFAULT", 1, "list",
      libGAP_ZeroListDefaultHandler, "src/listoper.c:ZERO_LIST_DEFAULT" },

    { "ZERO_MUT_LIST_DEFAULT", 1, "list",
      libGAP_ZeroListMutDefaultHandler, "src/listoper.c:ZERO_MUT_LIST_DEFAULT" },

    { "ZERO_ATTR_MAT", 1, "mat",
      libGAP_ZeroAttrMat, "src/listoper.c:ZERO_ATTR_MAT" },

    { "AINV_LIST_DEFAULT", 1, "list",
      libGAP_AInvListDefaultHandler, "src/listoper.c:AINV_LIST_DEFAULT" },

    { "AINV_MUT_LIST_DEFAULT", 1, "list",
      libGAP_AInvMutListDefaultHandler, "src/listoper.c:AINV_MUT_LIST_DEFAULT" },

    { "DIFF_SCL_LIST_DEFAULT", 2, "listL, listR",
      libGAP_DiffSclListHandler, "src/listoper.c:DIFF_SCL_LIST_DEFAULT" },

    { "DIFF_LIST_SCL_DEFAULT", 2, "listL, listR",
      libGAP_DiffListSclHandler, "src/listoper.c:DIFF_LIST_SCL_DEFAULT" },

    { "DIFF_LIST_LIST_DEFAULT", 2, "listL, listR",
      libGAP_DiffListListHandler, "src/listoper.c:DIFF_LIST_LIST_DEFAULT" },

    { "PROD_SCL_LIST_DEFAULT", 2, "listL, listR",
      libGAP_ProdSclListHandler, "src/listoper.c:PROD_SCL_LIST_DEFAULT" },

    { "PROD_LIST_SCL_DEFAULT", 2, "listL, listR",
      libGAP_ProdListSclHandler, "src/listoper.c:PROD_LIST_SCL_DEFAULT" },

    { "PROD_LIST_LIST_DEFAULT", 3, "listL, listR, depthDiff",
      libGAP_ProdListListHandler, "src/listoper.c:PROD_LIST_LIST_DEFAULT" },

    { "ONE_MATRIX_MUTABLE", 1, "list",
      libGAP_FuncOneMatrixMutable, "src/listoper.c:ONE_MATRIX_MUTABLE" },

    { "ONE_MATRIX_SAME_MUTABILITY", 1, "list",
      libGAP_FuncOneMatrixSameMutability, "src/listoper.c:ONE_MATRIX_SAME_MUTABILITY" },

    { "ONE_MATRIX_IMMUTABLE", 1, "list",
      libGAP_FuncOneMatrixImmutable, "src/listoper.c:ONE_MATRIX_IMMUTABLE" },

    { "INV_MATRIX_MUTABLE", 1, "list",
      libGAP_FuncINV_MATRIX_MUTABLE, "src/listoper.c:INV_MATRIX_MUTABLE" },

    { "INV_MATRIX_SAME_MUTABILITY", 1, "list",
      libGAP_FuncINV_MATRIX_SAME_MUTABILITY, "src/listoper.c:INV_MATRIX_SAME_MUTABILITY" },

    { "INV_MATRIX_IMMUTABLE", 1, "list",
      libGAP_FuncINV_MATRIX_IMMUTABLE, "src/listoper.c:INV_MATRIX_IMMUTABLE" },

    { "ADD_ROW_VECTOR_5", 5, "list1, list2, mult, from, to",
      libGAP_FuncADD_ROW_VECTOR_5, "src/listoper.c:ADD_ROW_VECTOR_5" },

    { "ADD_ROW_VECTOR_5_FAST", 5, "list1, list2, mult, from, to",
      libGAP_FuncADD_ROW_VECTOR_5_FAST, "src/listoper.c:ADD_ROW_VECTOR_5_FAST" },

    { "ADD_ROW_VECTOR_3", 3, "list1, list2, mult",
      libGAP_FuncADD_ROW_VECTOR_3, "src/listoper.c:ADD_ROW_VECTOR_3" },

    { "ADD_ROW_VECTOR_3_FAST", 3, "list1, list2, mult",
      libGAP_FuncADD_ROW_VECTOR_3_FAST, "src/listoper.c:ADD_ROW_VECTOR_3_FAST" },

    { "ADD_ROW_VECTOR_2", 2, "list1, list2",
      libGAP_FuncADD_ROW_VECTOR_2, "src/listoper.c:ADD_ROW_VECTOR_2" },

    { "ADD_ROW_VECTOR_2_FAST", 2, "list1, list2",
      libGAP_FuncADD_ROW_VECTOR_2_FAST, "src/listoper.c:ADD_ROW_VECTOR_2_FAST" },

    { "MULT_ROW_VECTOR_2", 2, "list, mult",
      libGAP_FuncMULT_ROW_VECTOR_2, "src/listoper.c:MULT_ROW_VECTOR_2" },

    { "MULT_ROW_VECTOR_2_FAST", 2, "list, mult",
      libGAP_FuncMULT_ROW_VECTOR_2_FAST, "src/listoper.c:MULT_ROW_VECTOR_2_FAST" },

    { "PROD_VEC_MAT_DEFAULT", 2, "vec, mat",
      libGAP_FuncPROD_VEC_MAT_DEFAULT, "src/listoper.c:PROD_VEC_MAT_DEFAULT" },
    
    { "INV_MAT_DEFAULT_MUTABLE", 1, "mat",
      libGAP_FuncINV_MAT_DEFAULT_MUTABLE, "src/listoper.c:INV_MAT_DEFAULT_MUTABLE" },

    { "INV_MAT_DEFAULT_SAME_MUTABILITY", 1, "mat",
      libGAP_FuncINV_MAT_DEFAULT_SAME_MUTABILITY, "src/listoper.c:INV_MAT_DEFAULT_SAME_MUTABILITY" },

    { "INV_MAT_DEFAULT_IMMUTABLE", 1, "mat",
      libGAP_FuncINV_MAT_DEFAULT_IMMUTABLE, "src/listoper.c:INV_MAT_DEFAULT_IMMUTABLE" },

    { "ADD_TO_LIST_ENTRIES_PLIST_RANGE", 3, "list, range, x",
      libGAP_FuncADD_TO_LIST_ENTRIES_PLIST_RANGE, "src/listfunc.c:ADD_TO_LIST_ENTRIES_PLIST_RANGE" },

    { "MONOM_TOT_DEG_LEX", 2, "monomial, monomial",
      libGAP_FuncMONOM_TOT_DEG_LEX, "src/ratfun.c:FuncMONOM_TOT_DEG_LEX" },

    { "MONOM_GRLEX", 2, "monomial, monomial",
      libGAP_FuncMONOM_GRLEX, "src/ratfun.c:FuncMONOM_GRLEX" },

    { "ZIPPED_SUM_LISTS", 4, "list,list,zero,funclist",
      libGAP_FuncZIPPED_SUM_LISTS, "src/ratfun.c:FuncZIPPED_SUM_LISTS" },

    { "MONOM_PROD", 2, "monomial, monomial",
      libGAP_FuncMONOM_PROD, "src/ratfun.c:FuncMONOM_PROD" },

    { 0 }

};


/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
**
*/

static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    libGAP_InitFopyGVar( "AddRowVector", &libGAP_AddRowVectorOp );
    libGAP_InitFopyGVar( "MultRowVector", &libGAP_MultRowVectorOp );
    libGAP_InitFopyGVar( "ConvertToMatrixRep", &libGAP_ConvertToMatrixRep );


    /* install the generic comparisons                                     */
    for ( t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1++ ) {
        for ( t2 = libGAP_FIRST_LIST_TNUM; t2 <= libGAP_LAST_LIST_TNUM; t2++ ) {
            libGAP_EqFuncs[ t1 ][ t2 ] = libGAP_EqListList;
        }
    }
    for ( t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1++ ) {
        for ( t2 = libGAP_FIRST_LIST_TNUM; t2 <= libGAP_LAST_LIST_TNUM; t2++ ) {
            libGAP_LtFuncs[ t1 ][ t2 ] = libGAP_LtListList;
        }
    }
    for ( t1 = libGAP_FIRST_REAL_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1++ ) {
        for ( t2 = libGAP_FIRST_LIST_TNUM; t2 <= libGAP_LAST_LIST_TNUM; t2++ ) {
            libGAP_InFuncs[ t1 ][ t2 ] = libGAP_InList;
        }
    }

    for (t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1 ++ ) {
            libGAP_ZeroFuncs[t1] = libGAP_ZeroListDefault;
            libGAP_ZeroMutFuncs[t1] = libGAP_ZeroListMutDefault;
    }

    for (t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1 ++ ) {
            libGAP_AInvFuncs[t1] = libGAP_AInvListDefault;
            libGAP_AInvMutFuncs[t1] = libGAP_AInvMutListDefault;
    }

    /* No kernel installations for One or Inverse any more */

    /* Sum. Here we can do list + non-list and non-list + list,
       we have to careful about list+list, though, as it might
       really be list+matrix or matrix+list

       for the T_PLIST_CYC and T_PLIST_FFE cases, we know the nesting
       depth (1) and so we can do the cases of adding them to each
       other and to T_PLIST_TAB objects, which have at least nesting
       depth 2. Some of this will be overwritten in vector.c and
       vecffe.c


       everything else needs to wait until the library */
       
    for (t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1++ ) {
      for (t2 = libGAP_FIRST_REAL_TNUM; t2 < libGAP_FIRST_LIST_TNUM; t2++ ) {
        libGAP_SumFuncs[t1][t2] = libGAP_SumListScl;
        libGAP_SumFuncs[t2][t1] = libGAP_SumSclList;
      }
        
    }
    for (t1 = libGAP_T_PLIST_CYC; t1 <= libGAP_T_PLIST_FFE+libGAP_IMMUTABLE; t1++) {
      for (t2 = libGAP_T_PLIST_CYC; t2 <= libGAP_T_PLIST_FFE+libGAP_IMMUTABLE; t2++) {
        libGAP_SumFuncs[t1][t2] = libGAP_SumListList;
      }
      for (t2 = libGAP_T_PLIST_TAB; t2 <= libGAP_T_PLIST_TAB_RECT_SSORT+libGAP_IMMUTABLE; t2++) {
        libGAP_SumFuncs[t1][t2] = libGAP_SumSclList;
        libGAP_SumFuncs[t2][t1] = libGAP_SumListScl;
      }
    }
        
    /* Diff is just like Sum */
    
    for (t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1++ ) {
      for (t2 = libGAP_FIRST_REAL_TNUM; t2 < libGAP_FIRST_LIST_TNUM; t2++ ) {
        libGAP_DiffFuncs[t1][t2] = libGAP_DiffListScl;
        libGAP_DiffFuncs[t2][t1] = libGAP_DiffSclList;
      }
    }
    for (t1 = libGAP_T_PLIST_CYC; t1 <= libGAP_T_PLIST_FFE+libGAP_IMMUTABLE; t1++) {
      for (t2 = libGAP_T_PLIST_CYC; t2 <= libGAP_T_PLIST_FFE+libGAP_IMMUTABLE; t2++) {
        libGAP_DiffFuncs[t1][t2] = libGAP_DiffListList;
      }
      for (t2 = libGAP_T_PLIST_TAB; t2 <= libGAP_T_PLIST_TAB_RECT_SSORT+libGAP_IMMUTABLE; t2++) {
        libGAP_DiffFuncs[t1][t2] = libGAP_DiffSclList;
        libGAP_DiffFuncs[t2][t1] = libGAP_DiffListScl;
      }
    }

    /* Prod.

    Here we can't do the T_PLIST_TAB cases, in case they are nesting depth three or more
    in which case different rules apply.

    It's also less obvious what happens with the empty list
    */
    
    
    for (t1 = libGAP_FIRST_LIST_TNUM; t1 <= libGAP_LAST_LIST_TNUM; t1++ ) {
      for (t2 = libGAP_FIRST_REAL_TNUM; t2 < libGAP_FIRST_LIST_TNUM; t2++ ) {
        libGAP_ProdFuncs[t1][t2] = libGAP_ProdListScl;
        libGAP_ProdFuncs[t2][t1] = libGAP_ProdSclList;
      }
    }
    for (t1 = libGAP_T_PLIST_CYC; t1 <= libGAP_T_PLIST_FFE+libGAP_IMMUTABLE; t1++) {
      for (t2 = libGAP_T_PLIST_CYC; t2 <= libGAP_T_PLIST_FFE+libGAP_IMMUTABLE; t2++) {
        libGAP_ProdFuncs[t1][t2] = libGAP_ProdListList;
      }
    }
    

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{

  /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoListOper()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "listoper",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoListOper ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  listoper.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
