/* LibGAP - a shared library version of the GAP kernel
 * Copyright (C) 2013 Volker Braun <vbraun.name@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. 
 */


#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "system.h"
#include "scanner.h"
#include "gap.h"
#include "libgap.h"
#include "libgap_internal.h"

/* Pointers to input/output buffers. libGAP users must not access these buffers directly!
 */

#define libGAP_BUFFER_STEP 16*1024

static char* libGAP_stdin_buffer = NULL;

static char* libGAP_stdout_buffer = NULL;
static size_t libGAP_stdout_bufsize = 0;
static size_t libGAP_stdout_pos = 0;


/* stderr is captured in a static buffer to make it easier to pass it around in the error handler */

#define libGAP_STDERR_BUFSIZE 4096

static char libGAP_stderr_buffer[libGAP_STDERR_BUFSIZE];
static size_t libGAP_stderr_pos = 0;

int libgap_in_enter_exit_block = 0; /* false */

/*************************************************************************/
/*** Initialize / Finalize ***********************************************/
/*************************************************************************/

/* from gap.c */
extern char **libGAP_sysenviron;
extern libGAP_Int libGAP_NrImportedGVars;
extern libGAP_Int libGAP_NrImportedFuncs;



/*************************************************************************/
/*** Global Initialization ***********************************************/
/*************************************************************************/

void libgap_initialize(int argc, char** argv)
{
  libGAP_sysenviron = environ;
  libGAP_NrImportedGVars = 0;
  libGAP_NrImportedFuncs = 0;
  libGAP_UserHasQUIT = 0;
  libGAP_UserHasQuit = 0;
  libgap_mark_stack_bottom();
  libGAP_InitializeGap( &argc, argv );
}


void libgap_finalize()
{
  libGAP_FinishBags();
}


/*************************************************************************/
/*** Garbage collector callback ******************************************/
/*************************************************************************/

static libgap_gasman_callback_ptr libGAP_gasman_callback = NULL;

void libgap_set_gasman_callback(libgap_gasman_callback_ptr callback)
{
  libGAP_gasman_callback = callback;
}

void libgap_call_gasman_callback()
{
  if (libGAP_gasman_callback != NULL)
    (*libGAP_gasman_callback)();
}


/*************************************************************************/
/*** Input/Output interaction ********************************************/
/*************************************************************************/

void libgap_start_interaction(char* inputline)
{
  assert(libGAP_stdin_buffer == NULL);
  libGAP_stdin_buffer = inputline;
  
  libGAP_stdout_bufsize = libGAP_BUFFER_STEP;
  libGAP_stdout_buffer = (char*)malloc(libGAP_stdout_bufsize);
  libGAP_stdout_pos = 0;
  
  libGAP_stderr_pos = 0;
}

char* libgap_get_output() 
{
  libgap_append_stdout('\0');
  return libGAP_stdout_buffer;
}

char* libgap_get_error() 
{
  libgap_append_stderr('\0');
  return strdup(libGAP_stderr_buffer);
}

void libgap_finish_interaction()
{
  while (libGAP_Symbol != libGAP_S_EOF)
    libGAP_GetSymbol();
  libGAP_stdin_buffer = NULL;

  libGAP_stdout_bufsize = 0;
  libGAP_stdout_pos = 0;
  free(libGAP_stdout_buffer);
  libGAP_stdout_buffer = NULL;
  
  libGAP_stderr_pos = 0;
  libGAP_ClearError();
}




/*************************************************************************/
/*** Let GAP access the buffers ******************************************/
/*************************************************************************/

static libgap_error_func_ptr libGAP_error_func = NULL;

void libgap_set_error_handler(libgap_error_func_ptr callback)
{
  libGAP_error_func = callback;
}


void libgap_call_error_handler()
{
  if (libGAP_error_func == NULL) {
    printf("An error occurred, but libGAP has no handler set.\n");
    printf("Error message: %s\n", libgap_get_error());
    return;
  }
  libgap_append_stderr('\0');
  libGAP_stderr_pos = 0;
  libGAP_ClearError();
  (*libGAP_error_func) (libGAP_stderr_buffer);
}



/*************************************************************************/
/*** Let GAP access the buffers ******************************************/
/*************************************************************************/

char* libgap_get_input(char* line, int length)
{
  // TODO: copy in length chunks
  if (libGAP_stdin_buffer == NULL) {
    return NULL;
  }
  assert(strlen(libGAP_stdin_buffer) < length);
  strcpy(line, libGAP_stdin_buffer);
  libGAP_stdin_buffer = NULL;
  return line;
}

void libgap_append_stdout(char ch)
{
  if (libGAP_stdout_buffer == NULL)
    return;
  if (libGAP_stdout_pos == libGAP_stdout_bufsize) {
    char* old_stdout_buffer = libGAP_stdout_buffer;
    size_t old_stdout_bufsize = libGAP_stdout_bufsize;
    libGAP_stdout_bufsize += libGAP_BUFFER_STEP;
    libGAP_stdout_buffer = (char*)malloc(libGAP_stdout_bufsize);
    memcpy(libGAP_stdout_buffer, old_stdout_buffer, old_stdout_bufsize);
    free(old_stdout_buffer);
  }    
  libGAP_stdout_buffer[libGAP_stdout_pos++] = ch;
}


void libgap_append_stderr(char ch)
{
  libGAP_stderr_buffer[libGAP_stderr_pos++] = ch;
  if (libGAP_stderr_pos == libGAP_STDERR_BUFSIZE) 
    libGAP_stderr_pos--;
}


void libgap_set_error(char* msg)
{
  libGAP_stderr_pos = 0;
  int i;
  for (i=0; i<strlen(msg); i++)
    libgap_append_stderr(msg[i]);
}
