/****************************************************************************
**
*W  dt.c                        GAP source                  Wolfgang Merkwitz
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file implements the part of the deep thought package which deals
**  with computing the deep thought polynomials.
**
**  Deep Thought deals with trees.  A tree <tree> is a concatenation of 
**  several nodes where each node is a 5-tuple of immediate integers.  If
**  <tree> is an atom it contains only one node,  thus it is itself a
**  5-tuple. If <tree> is not an atom we obtain its list representation by
**
**  <tree>  :=  topnode(<tree>) concat left(<tree>) concat right(<tree>) .
**
**  Let us denote the i-th node of <tree> by (<tree>, i)  and the tree rooted 
**  at (<tree>, i) by tree(<tree>, i).  Let <a> be tree(<tree>, i)
**  The first entry of (<tree>, i) is pos(a),
**  and the second entry is num(a). The third entry of (<tree>, i) gives a 
**  mark.(<tree>, i)[3] = 1  means that (<tree>, i) is marked,  
**  (<tree>, i)[3] = 0 means that (<tree>, i) is not marked. The fourth entry
**  of (<tree>, i) contains the number of knodes of tree(<tree>, i).  The
**  fifth entry of (<tree>, i) finally contains a boundary for 
**  pos( tree(<tree>, i) ).  (<tree>, i)[5] <= 0 means that 
**  pos( tree(<tree>, i) ) is unbounded.  If tree(<tree>, i) is an atom we
**  already know that pos( tree(<tree>, i) ) is unbound.  Thus we then can
**  use the fifth component of (<tree>, i) to store the side.  In this case
**  (<tree>, i)[5] = -1 means  that tree(<tree>, i) is an atom from the
**  right hand word, and (<tree>, i)[5] = -2 means that tree(<tree>, i) is
**  an atom from the left hand word.
**
**  A second important data structure deep thought deals with is a deep
**  thought monomial. A deep thought monomial g_<tree> is a product of
**  binomial coefficients with a coefficient c. Deep thought monomials
**  are represented in this implementation by formula
**  vectors,  which are lists of integers.  The first entry of a formula
**  vector is 0,  to distinguish formula vectors from trees.  The second
**  entry is the coefficient c,  and the third and fourth entries are
**  num( left(tree) ) and num( right(tree) ).  The remaining part of the
**  formula vector is a concatenation of pairs of integers.  A pair (i, j)
**  with i > 0 represents binomial(x_i, j).  A pair (0, j) represents
**  binomial(y_gen, j) when word*gen^power is calculated.
**
**  Finally deep thought has to deal with pseudorepresentatives. A
**  pseudorepresentative <a> is stored in list of length 4. The first entry
**  stores left( <a> ),  the second entry contains right( <a> ),  the third
**  entry contains num( <a> ) and the last entry finally gives a boundary
**  for pos( <b> ) for all trees <b> which are represented by <a>.
*/
#include       "system.h"



#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */
#include        "bool.h"                /* booleans                        */
#include        "calls.h"               /* generic call mechanism          */
#include        "gap.h"                 /* error handling, initialisation  */
#include        "gvars.h"               /* global variables                */
#include        "integer.h"             /* integers                        */

#include        "dt.h"                  /* deep thought                    */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listfunc.h"            /* functions for generic lists     */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */


/****************************************************************************
**

*F  DT_POS(tree, index) . . . . . . . . . . . . . position of (<tree>, index)
**
**  'DT_POS' returns pos(<a>) where <a> is the subtree of <tree> rooted at
**  (<tree>, index).  <index> has to be a positive integer less or equal than
**  the number of nodes of <tree>.
*/
#define  libGAP_DT_POS(tree, index) \
              (libGAP_ELM_PLIST(tree, (index-1)*5 + 1 ) ) 


/***************************************************************************
**
*F  SET_DT_POS(tree, index, obj) . . . assign the position of(<tree>, index)
**
**  'SET_DT_POS sets pos(<a>) to the object <obj>, where <a> is the subtree
**  of <tree>,  rooted at (<tree>, index).  <index> has to be an positive
**  integer less or equal to the number of nodes of <tree>
*/
#define  libGAP_SET_DT_POS(tree, index, obj) \
              libGAP_SET_ELM_PLIST(tree, (index-1)*5 + 1, obj) 


/***************************************************************************
**
*F  DT_GEN(tree, index) . . . . . . . . . . . . . generator of (<tree>, index)
**
**  'DT_GEN' returns num(<a>) where <a> is the subtree of <tree> rooted at
**  (<tree>, index).  <index> has to be a positive integer less or equal than
**  the number of nodes of <tree>.
*/
#define  libGAP_DT_GEN(tree, index) \
              (libGAP_ELM_PLIST(tree, (index-1)*5 + 2) )


/**************************************************************************
**
*F  SET_DT_GEN(tree, index, obj) . . . assign the generator of(<tree>, index)
**
**  'SET_DT_GEN sets num(<a>) to the object <obj>, where <a> is the subtree
**  of <tree>,  rooted at (<tree>, index).  <index> has to be an positive
**  integer less or equal to the number of nodes of <tree>
*/
#define  libGAP_SET_DT_GEN(tree, index, obj) \
              (libGAP_SET_ELM_PLIST(tree, (index-1)*5 + 2, obj) )


/**************************************************************************
**
*F  DT_IS_MARKED(tree, index) . . . . . . tests if (<tree>, index) is marked
**
**  'DT_IS_MARKED' returns 1 (as C integer) if (<tree>, index) is marked, and
**  0 otherwise.  <index> has to be a positive integer less or equal to the
**  number of nodes of <tree>.
*/
#define  libGAP_DT_IS_MARKED(tree, index)  \
             (libGAP_INT_INTOBJ (libGAP_ELM_PLIST(tree, (index-1)*5 + 3) ) )


/**************************************************************************
**
*F  DT_MARK(tree, index) . . . . . . . . . . . . . . . . . . . . mark a node
**
**  'DT_MARK' marks the node (<tree>, index). <index> has to be a positive
**  integer less or equal to the number of nodes of <tree>.
*/
#define  libGAP_DT_MARK(tree, index) \
              libGAP_SET_ELM_PLIST(tree, (index-1)*5 + 3, libGAP_INTOBJ_INT(1) ) 


/**************************************************************************
**
*F  DT_UNMARK(tree, index) . . . . . . . . . . . remove the mark from a node
**
**  'DT_UNMARK' removes the mark from the node (<tree>, index). <index> has 
**  has to be a positive integer less or equal to the number of nodes of
**  <tree>.
*/
#define  libGAP_DT_UNMARK(tree, index) \
              libGAP_SET_ELM_PLIST(tree, (index-1)*5 + 3, libGAP_INTOBJ_INT(0) ) 


/****************************************************************************
**
*F  DT_RIGHT(tree, index) . . . .determine the right subnode of (<tree>, index)
*F  DT_LEFT(tree, index) . . . . determine the left subnode of (<tree>, index)
**
**  'DT_RIGHT' returns the right subnode of (<tree>, index).  That means if
**  DT_RIGHT(tree, index) = index2,  then (<tree>, index2) is the right
**  subnode of (<tree>, index).
**
**  'DT_LEFT' returns the left subnode of (<tree>, index).  That means if
**  DT_LEFT(tree, index) = index2,  then (<tree>, index2) is the left
**  subnode of (<tree>, index).
**
**  Before calling 'DT_RIGHT' or 'DT_LEFT' it should be ensured,  that 
**  (<tree>, index) is not an atom.  <index> has to be a positive integer
**  less or equal to the number of nodes of <tree>.
*/
#define  libGAP_DT_RIGHT(tree, index) \
              ( libGAP_INT_INTOBJ(libGAP_ELM_PLIST(tree, index*5 + 4) ) + index + 1)
#define  libGAP_DT_LEFT(tree, index) \
              ( index + 1 )


/****************************************************************************
**
*F  DT_SIDE(tree, index) . . . . . . . determine the side of (<tree>, index)
*V  RIGHT. . . . . . . . . . . . . . . integer describing "right"
*V  LEFT . . . . . . . . . . . . . . . integer describing "left"
**
**  'DT_SIDE' returns 'LEFT' if (<tree>, index) is an atom from the Left-hand
**  word,  and 'RIGHT'  if (<tree>, index) is an atom of the Right-hand word.
**  Otherwise 'DT_SIDE' returns an integer bigger than 1.  <index> has to be
**  a positive integer less or equal to the number of nodes of <tree>.
*/
#define  libGAP_RIGHT                  -1
#define  libGAP_LEFT                   -2
#define  libGAP_DT_SIDE(tree, index) \
              (libGAP_INT_INTOBJ( libGAP_ELM_PLIST(tree, (index-1)*5 + 5 ) )  )


/****************************************************************************
**
*F  DT_LENGTH(tree, index) . . . . . . . . number of nodes of (<tree>, index)
**
**  'DT_LENGTH' returns the number of nodes of (<tree>, index).  <index> has
**  to be a positive integer less or equal to the number of nodes of <tree>.
*/
#define  libGAP_DT_LENGTH(tree, index) \
              ( libGAP_INT_INTOBJ(libGAP_ELM_PLIST(tree, (index-1)*5 + 4) )  )


/***************************************************************************
**
*F  DT_MAX(tree, index) . . . . . . . . . . . . . . . . boundary of a node
**
**  'DT_MAX(tree, index)' returns a boundary for 'DT_POS(tree, index)'.
**  'DT_MAX(tree, index) = 0 ' means that 'DT_POS(tree, index)' is unbound.
**  <index> has to be a positive integer less or equal to the number of nodes
**  of tree.
*/
#define  libGAP_DT_MAX(tree, index) \
              (libGAP_ELM_PLIST(tree, (index-1)*5 + 5 ) )


/****************************************************************************
**
*F  CELM(list, pos) . . . . . . . . . . element of a plain list as C integer
**
**  'CELM' returns the <pos>-th element of the plain list <list>.  <pos> has
**  to be a positive integer less or equal to the physical length of <list>.
**  Before calling 'CELM' it should be ensured that the <pos>-th entry of
**  <list> is an immediate integer object.
*/
#define  libGAP_CELM(list, pos)         ( libGAP_INT_INTOBJ(libGAP_ELM_PLIST(list, pos) ) )


/****************************************************************************
**
*V  Dt_add
**
**  Dt_add is used to store the library function dt_add.
*/

libGAP_Obj                    libGAP_Dt_add;
extern libGAP_Obj             libGAP_ShallowCopyPlist( libGAP_Obj  list );

/****************************************************************************
**
*F  UnmarkTree( <tree> ) . . . . . . . remove the marks of all nodes of <tree>
**
**  'UnmarkTree' removes all marks of all nodes of the tree <tree>.
*/
void  libGAP_UnmarkTree(
                  libGAP_Obj   tree )
{
    libGAP_UInt     i, len; /*  loop variable                     */

    len = libGAP_DT_LENGTH(tree, 1);
    for (i=1; i <= len; i++ )
        libGAP_DT_UNMARK(tree, i);
}


/****************************************************************************
**
*F  FuncUnmarkTree(<self>, <tree>) . . remove the marks of all nodes of <tree>
**
**  'FuncUnmarkTree' implements the internal function 'UnmarkTree'.
**
**  'UnmarkTree( <tree> )'
**
**  'UnmarkTree' removes all marks of all nodes of the tree <tree>.
*/
libGAP_Obj  libGAP_FuncUnmarkTree(
                     libGAP_Obj  self,
                     libGAP_Obj  tree   )
{
    libGAP_UnmarkTree(tree);
    return  0;
}


/*****************************************************************************
**
*F  Mark(<tree>, <reftree>, <index>) . . . . . . . . find all nodes of <tree> 
**                                                   which are almost equal
**                                                   to (<reftree>, index)
**
**  'Mark' determines all nodes of the tree <tree>, rooting subtrees almost
**  equal to the tree rooted at (<reftree>, index).  'Mark' marks these nodes
**  and returns the number of different nodes among these nodes.  Since it
**  is assumed that the set {pos(a) | a almost equal to (<reftree>, index) }
**  is equal to {1,...,n} for a positive integer n,  'Mark' actually returns
**  the Maximum of {pos(a) | a almost equal to (<reftree>, index)}.
*/
libGAP_UInt   libGAP_Mark(
            libGAP_Obj   tree,
            libGAP_Obj   reftree,
            libGAP_Int   indexx  )
{
    libGAP_UInt  i, /*  loop variable                    */
          m, /*  integer to return                */
          len;
    libGAP_Obj   refgen;

    m = 0;
    i = 1;
    len = libGAP_DT_LENGTH(tree, 1);
    refgen = libGAP_DT_GEN(reftree, indexx);
    while ( i <= len )
    {
        /*  skip all nodes (<tree>, i) with 
        **  num(<tree>, i) > num(<reftree>, indexx)     */
        while( i < len && 
               libGAP_DT_GEN(tree, i)  >  refgen )
            i++;
        if ( libGAP_AlmostEqual(tree, i, reftree, indexx) )
        {
            libGAP_DT_MARK(tree, i);
            if ( m < libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) )  )
                m = libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) );
        }
        /*  Since num(a) < num(b) holds for all subtrees <a> of an arbitrary
        **  tree <b> we can now skip the whole tree rooted at (<tree>, i).
        **  If (<tree>, i) is the left subnode of another node we can even
        **  skip the tree rooted at that node,  because of 
        **  num( right(a) )  <  num( left(a) ) for all trees <a>.
        **  Note that (<tree>, i) is the left subnode of another node,  if and
        **  only if the previous node (<tree>, i-1) is not an atom. in this
        **  case (<tree>, i) is the left subnode of (<tree>, i-1).          */
        if ( libGAP_DT_LENGTH(tree, i-1) == 1 )
            /*   skip the tree rooted at (<tree>, i).                    */
            i = i + libGAP_DT_LENGTH(tree, i);
        else
            /*   skip the tree rooted at (<tree>, i-1)                   */
            i = i - 1 + libGAP_DT_LENGTH(tree, i-1);
    }
    return m;
}


/****************************************************************************
**
*F  AmostEqual(<tree1>,<index1>,<tree2>,<index2>) . . test of almost equality
**
**  'AlmostEqual' tests if tree(<tree1>, index1) is almost equal to
**  tree(<tree2>, index2).  'AlmostEqual' returns 1
**  if these trees are almost equal,  and 0 otherwise.  <index1> has to be
**  a positive integer less or equal to the number of nodes of <tree1>,
**  and <index2> has to be a positive integer less or equal to the number of
**  nodes of <tree2>.
*/
libGAP_Int     libGAP_AlmostEqual(
                     libGAP_Obj    tree1,
                     libGAP_Int    index1,
                     libGAP_Obj    tree2,
                     libGAP_Int    index2    )
{
    libGAP_UInt   k, schranke; /*   loop variable                                             */
    /*  First the two top nodes of tree(<tree1>, index1) and
    **  tree(<tree2>, index2) (that are (<tree1>, index1) and 
    **  (<tree2, index2) ) are compared by testing the equality of the 2-nd,
    **  5-th and 6-th entries the nodes.                                    */
    if ( libGAP_DT_GEN(tree1, index1) != libGAP_DT_GEN(tree2, index2) )
        return  0;
    if ( libGAP_DT_SIDE(tree1, index1) != libGAP_DT_SIDE(tree2, index2)  )
        return  0;
    if ( libGAP_DT_LENGTH(tree1, index1) != libGAP_DT_LENGTH(tree2, index2)  )
        return  0;
    /*  For the comparison of the remaining nodes of tree(<tree1>, index1)
    **  and tree(<tree2>, index2) it is also necessary to compare the first
    **  entries of the nodes.  Note that we know at this point,  that 
    **  tree(<tree1>, index1) and tree(<tree2>, index2) have the same number
    **  of nodes                                                             */
    schranke = index1 + libGAP_DT_LENGTH(tree1, index1);
    for (k = index1 + 1;  k < schranke;  k++ )
    {
        if ( libGAP_DT_GEN(tree1, k) != libGAP_DT_GEN(tree2, k + index2 - index1 ) )
            return  0;
        if ( libGAP_DT_POS(tree1, k) != libGAP_DT_POS(tree2, k + index2 - index1 ) )
            return  0;
        if ( libGAP_DT_SIDE(tree1, k)    !=
             libGAP_DT_SIDE(tree2, k + index2 - index1)  )
            return  0;
        if ( libGAP_DT_LENGTH(tree1, k) != libGAP_DT_LENGTH(tree2, k + index2 - index1) )
            return  0;
    }
    return  1;
}


/*****************************************************************************
**
*F  Equal(<tree1>,<index1>,<tree2>,<index2>) . . . . . . . . test of equality
**
**  'Equal' tests if tree(<tree1>, index1) is equal to
**  tree(<tree2>, index2).  'Equal' returns 1
**  if these trees are  equal,  and 0 otherwise.  <index1> has to be
**  a positive integer less or equal to the number of nodes of <tree1>,
**  and <index2> has to be a positive integer less or equal to the number of
**  nodes of <tree2>.
*/
libGAP_Int     libGAP_Equal(
               libGAP_Obj     tree1,
               libGAP_Int     index1,
               libGAP_Obj     tree2,
               libGAP_Int     index2   )
{
    libGAP_UInt   k, schranke; /*  loop variable                                   */

    /*  Each node of tree(<tree1>, index1) is compared to the corresponding
    **  node of tree(<tree2>, index2) by testing the equality of the 1-st,
    **  2-nd,  5-th and 6-th nodes.                                          */
    schranke = index1 + libGAP_DT_LENGTH(tree1, index1);
    for (k=index1; k < schranke;  k++)
    {
        if ( libGAP_DT_GEN(tree1, k) != libGAP_DT_GEN(tree2, k + index2 - index1 ) )
            return  0;
        if ( libGAP_DT_POS(tree1, k) != libGAP_DT_POS(tree2, k + index2 - index1 ) )
            return  0;
        if ( libGAP_DT_SIDE(tree1, k)   !=
             libGAP_DT_SIDE(tree2, k + index2 - index1)   )
            return  0;
        if ( libGAP_DT_LENGTH(tree1, k) != libGAP_DT_LENGTH(tree2, k + index2 - index1) )
            return  0;
    }
    return  1;
}


/****************************************************************************
**
*F  Mark2(<tree>,<index1>,<reftree>,<index2>) . . find all subtrees of
**                                                tree(<tree>, index1) which
**                                                are almost equal to
**                                                tree(<reftree>, index2)
**
**  'Mark2' determines all subtrees of tree(<tree>, index1) that are almost
**  equal to tree(<reftree>, index2).  'Mark2' marks the top nodes of these
**  trees and returns a list of lists <list> such that <list>[i]
**  for each subtree <a> of <tree> which is  almost equal to
**  tree(<reftree>, index2) and for which pos(<a>) = i holds contains an
**  integer describing the position of the top node of <a> in <tree>.
**  For example <list>[i] = [j, k] means that tree(<tree>, j) and
**  tree(<tree>, k) are almost equal to tree(<reftree>, index2) and
**  that pos(tree(<tree>, j) = pos(tree(<tree>, k) = i holds.
**
**  <index1> has to be a positive integer less or equal to the number of nodes
**  of <tree>,  and <index2> has to be a positive integer less or equal to
**  the number of nodes of <reftree>.
*/
libGAP_Obj    libGAP_Mark2(
              libGAP_Obj        tree,
              libGAP_Int        index1,
              libGAP_Obj        reftree,
              libGAP_Int        index2   )
{
    libGAP_UInt    i, /*  loop variable                                          */
            len;
    libGAP_Obj     new, 
            list, /*  list to return                                      */
            refgen;

    /*  initialize <list>                                                 */
    list = libGAP_NEW_PLIST(libGAP_T_PLIST, 0);
    libGAP_SET_LEN_PLIST(list, 0);
    i = index1;
    len = index1 + libGAP_DT_LENGTH(tree, index1) - 1;
    refgen = libGAP_DT_GEN(reftree, index2);
    while( i <= len )
    {
        /*  skip all nodes (<tree>, i) with 
        **  num(<tree>, i) > num(<reftree>, index)     */
        while( i < len     &&
               libGAP_DT_GEN(tree, i) > refgen   )
            i++;
        if ( libGAP_AlmostEqual(tree, i, reftree, index2) )
        {
            libGAP_DT_MARK(tree, i);
            /*  if <list> is too small grow it appropriately               */
            if ( libGAP_LEN_PLIST(list) < libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) )  )
            {
                libGAP_GROW_PLIST(list, libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) ) );
                libGAP_SET_LEN_PLIST(list, libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) )  );
            }
            /*  if <list> has no entry at position pos(tree(<tree>, i))
            **  create a new list <new>,  assign it to list at position
            **  pos(tree(<tree>, i)),  and add i to <new>                  */
            if ( libGAP_ELM_PLIST(list, libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) )  )  ==  0)
            {
                new = libGAP_NEW_PLIST( libGAP_T_PLIST, 1);
                libGAP_SET_LEN_PLIST(new, 1);
                libGAP_SET_ELM_PLIST(new, 1, libGAP_INTOBJ_INT(i) );
                libGAP_SET_ELM_PLIST(list, libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) ),  new);
                /*  tell gasman that list has changed                      */
                libGAP_CHANGED_BAG(list);
            }
            /*  add i to <list>[ pos(tree(<tree>, i)) ]                    */ 
            else
            {
                new = libGAP_ELM_PLIST(list, libGAP_INT_INTOBJ( libGAP_DT_POS(tree, i) )  );
                libGAP_GROW_PLIST(new, libGAP_LEN_PLIST(new) + 1);
                libGAP_SET_LEN_PLIST(new, libGAP_LEN_PLIST(new) + 1);
                libGAP_SET_ELM_PLIST(new, libGAP_LEN_PLIST(new), libGAP_INTOBJ_INT(i) );
                /*  tell gasman that new has changed                         */
                libGAP_CHANGED_BAG(new);
            }
        }
        /*  Since num(a) < num(b) holds for all subtrees <a> of an arbitrary
        **  tree <b> we can now skip the whole tree rooted at (<tree>, i).
        **  If (<tree>, i) is the left subnode of another node we can even
        **  skip the tree rooted at that node,  because of 
        **  num( right(a) )  <  num( left(a) ) for all trees <a>.
        **  Note that (<tree>, i) is the left subnode of another node,  if and
        **  only if the previous node (<tree>, i-1) is not an atom. In this
        **  case (<tree>, i) is the left subnode of (<tree>, i-1).          */
        if ( libGAP_DT_LENGTH(tree, i-1) == 1 )
            /*  skip tree(<tree>, i)                                        */
            i = i + libGAP_DT_LENGTH(tree, i);
        else
            /*  skip tree(<tree>, i-1)                                      */
            i = i - 1 + libGAP_DT_LENGTH(tree, i-1);
    }
    return  list;
}


/*****************************************************************************
**
*F  FindTree(<tree>, <index>)
**
**  'FindTree' looks for a subtree <a> of tree(<tree>, index) such that 
**  the top node of
**  <a> is not marked but all the other nodes of <a> are marked.  It is
**  assumed that if the top node of a subtree <b> of tree(<tree>, index) 
**  is marked,  all
**  nodes of of <b> are marked.  Hence it suffices to look for a subtree <a>
**  of <tree> such that the top node of <a> is unmarked and the left and the
**  right node of <a> are marked.  'FindTree' returns an integer <i> such 
**  that tree(<tree> ,i) has the properties mentioned above.  If such a tree
**  does not exist 'Findtree' returns 0 (as C integer).  Note that this holds
**  if and only if tree(<tree>, index) is marked.
*/
libGAP_UInt    libGAP_FindTree(
                 libGAP_Obj     tree,
                 libGAP_Int     indexx )
{
    libGAP_UInt   i; /*     loop variable                                    */

    /*  return 0 if (<tree>, indexx) is marked                           */
    if ( libGAP_DT_IS_MARKED(tree, indexx) )
        return  0;
    i = indexx;
    /*  loop over all nodes of tree(<tree>, indexx) to find a tree with the
    **  properties described above.                                       */
    while( i < indexx + libGAP_DT_LENGTH(tree, indexx)  )
    {
        /*  skip all nodes that are unmarked and rooting non-atoms        */
        while( !( libGAP_DT_IS_MARKED(tree, i) )  &&  libGAP_DT_LENGTH(tree, i) > 1  )
            i++;
        /*  if (<tree>, i) is unmarked we now know that tree(<tree>, i) is
        **  an atom and we can return i.  Note that an unmarked atom has the
        **  desired properties.                                             */
        if ( !( libGAP_DT_IS_MARKED(tree, i) )  )
            return  i;
        /*  go to the previous node                                          */
        i--;
        /*  If the right node of tree(<tree>, i) is marked return i.
        **  Else go to the right node of tree(<tree>, i).                    */
        if  ( libGAP_DT_IS_MARKED(tree, libGAP_DT_RIGHT(tree, i) )  )
            return   i;
        i = libGAP_DT_RIGHT(tree, i);
    }
    return 0;
}


/****************************************************************************
**
*F  MakeFormulaVector(<tree>, <pr>) . . . . . . . . . compute the polynomial 
**                                                    g_<tree> for <tree>
**
**  'MakeFormulaVector' returns the polynomial g_<tree> for a tree <tree>
**  and a pc-presentation <pr> of a nilpotent group.  This polynomial g_<tree>
**  is a product of binomial coefficients with a coefficient c ( see the
**  header of this file ).
**
**  For the calculation of the coefficient c the top node of <tree> is ignored
**  because it can happen that trees are equal except for the top node.
**  Hence it suffices to compute the formula vector for one of these trees.
**  Then we get the "correct" coefficient for the polynomial for each <tree'>
**  of those trees by multiplying the coefficient given by the formula vector
**  with c_( num(left(<tree'>)),  num(right(<tree'>));  num(<tree'>) ).  This
**  is also the reason for storing num(left(<tree>)) and num(right(<tree>))
**  in the formula vector.
**
**  'MakeFormulaVector' only returns correct results if all nodes of <tree>
**  are unmarked.
*/
libGAP_Obj    libGAP_MakeFormulaVector(
                          libGAP_Obj    tree,
                          libGAP_Obj    pr   )
{
    libGAP_UInt  i, /*    denominator of a binomial coefficient              */
          j, /*    loop variable                                      */
          u; /*    node index                                         */
    libGAP_Obj   rel, /*  stores relations of <pr>                           */
          vec, /*  stores formula vector to return                    */
          prod,/*  stores the product of two integers                 */
          gen;

    /*  initialize <vec> and set the first four elements              */
    vec = libGAP_NEW_PLIST(libGAP_T_PLIST, 4);
    libGAP_SET_LEN_PLIST(vec, 4);
    libGAP_SET_ELM_PLIST(vec, 1, libGAP_INTOBJ_INT(0) );
    libGAP_SET_ELM_PLIST(vec, 2, libGAP_INTOBJ_INT(1) );
    libGAP_SET_ELM_PLIST(vec, 3, libGAP_DT_GEN(tree, libGAP_DT_LEFT(tree, 1) )  );
    libGAP_SET_ELM_PLIST(vec, 4, libGAP_DT_GEN(tree, libGAP_DT_RIGHT(tree, 1) )  );
    /*  loop over all almost equal classes of subtrees of <tree> except for
    **  <tree> itself.                                                    */
    u = libGAP_FindTree(tree, 1);
    while( u > 1 )
    {
        /*  mark all subtrees of <tree> almost equal to tree(<tree>, u) and
        **  get the number of different trees in this almost equal class    */
        i = libGAP_Mark(tree, tree, u);
        /*  if tree(<tree>, u) is an atom from the Right-hand word append
        **  [ 0, i ] to <vec>                                               */
        if  ( libGAP_DT_SIDE(tree, u) == libGAP_RIGHT )
        {
            libGAP_GROW_PLIST(vec, libGAP_LEN_PLIST(vec)+2);
            libGAP_SET_LEN_PLIST(vec, libGAP_LEN_PLIST(vec)+2);
            libGAP_SET_ELM_PLIST(vec, libGAP_LEN_PLIST(vec)-1, libGAP_INTOBJ_INT(0) );
            libGAP_SET_ELM_PLIST(vec, libGAP_LEN_PLIST(vec), libGAP_INTOBJ_INT(i) );
        }
        /*  if tree(<tree>, u) is an atom from the Left-hand word append
        **  [ num(tree(<tree>, u)), i ] to <vec>                            */
        else if  ( libGAP_DT_SIDE(tree, u) == libGAP_LEFT)
        {
            libGAP_GROW_PLIST(vec, libGAP_LEN_PLIST(vec)+2);
            libGAP_SET_LEN_PLIST(vec, libGAP_LEN_PLIST(vec)+2);
            libGAP_SET_ELM_PLIST(vec, libGAP_LEN_PLIST(vec)-1, libGAP_DT_GEN(tree, u) );
            libGAP_SET_ELM_PLIST(vec, libGAP_LEN_PLIST(vec), libGAP_INTOBJ_INT(i) );
        }
        /*  if tree(<tree>, u) is not an atom multiply 
        **  <vec>[2] with binomial(d, i) where
        **  d = c_(num(left(<tree>,u)), num(right(<tree>,u)); num(<tree>,u)) */
        else
        {
            j = 3;
            rel = libGAP_ELM_PLIST( libGAP_ELM_PLIST(pr, libGAP_INT_INTOBJ( libGAP_DT_GEN(tree, 
                                                        libGAP_DT_LEFT(tree, u) ) ) ),
                             libGAP_INT_INTOBJ( libGAP_DT_GEN(tree, libGAP_DT_RIGHT(tree, u) ) )  );
            gen = libGAP_DT_GEN(tree, u);
            while ( 1  )
            {
                if ( libGAP_ELM_PLIST(rel, j) == gen  )
                {
                    prod = libGAP_ProdInt(libGAP_ELM_PLIST(vec, 2),
                                   libGAP_binomial(libGAP_ELM_PLIST(rel, j+1), 
                                            libGAP_INTOBJ_INT(i)        )        );
                    libGAP_SET_ELM_PLIST(vec,  2, prod);
                    /*  tell gasman that vec has changed                     */
                    libGAP_CHANGED_BAG(vec);
                    break;
                }
                j+=2;
            }
        }
        u = libGAP_FindTree(tree, 1);
    }
    return vec;
}


/**************************************************************************
**
*F  FuncMakeFormulaVector(<self>,<tree>,<pr>) . . . . . compute the formula 
**                                                      vector for <tree>
**
**  'FuncMakeFormulaVector' implements the internal function
**  'MakeFormulaVector(<tree>, <pr>)'.
**
**  'MakeFormulaVector(<tree>, <pr>)'
**
**  'MakeFormulaVector' returns the formula vector for the tree <tree> and
**  the pc-presentation <pr>.
*/
libGAP_Obj    libGAP_FuncMakeFormulaVector(
                              libGAP_Obj      self,
                              libGAP_Obj      tree,
                              libGAP_Obj      pr            )
{
    if  (libGAP_LEN_PLIST(tree) == 5)
        libGAP_ErrorReturnVoid("<tree> has to be a non-atom", 0L, 0L,
                        "you can 'return;'");
    return  libGAP_MakeFormulaVector(tree, pr);
}


/*****************************************************************************
**
*F  binomial(<n>, <k>) . . . . . . . . . binomial coefficient of <n> and <k>
**
**  'binomial' returns the binomial coefficient of the integers <n> and <k>.
*/
libGAP_Obj  libGAP_binomial( libGAP_Obj   n,
               libGAP_Obj   k  )
{
    libGAP_UInt    j, kc;
    libGAP_Obj     bin, help;

    if ( libGAP_LtInt( libGAP_INTOBJ_INT(0), n)  &&  libGAP_LtInt(n, k)  )
        return libGAP_INTOBJ_INT(0);
    if ( libGAP_IS_INTOBJ(n)  &&  n == k )
        return libGAP_INTOBJ_INT(1);
    kc = libGAP_INT_INTOBJ(k);
    bin = libGAP_INTOBJ_INT(1);
    help = libGAP_DiffInt(n, k);
    for (j=1; j<=kc; j++)
        bin = libGAP_ProdInt( bin, libGAP_SumInt(help, libGAP_INTOBJ_INT(j) )  );
    for (j=1; j<=kc; j++)
        bin = libGAP_QuoInt(bin, libGAP_INTOBJ_INT(j) );
    return  bin;
}
    


/****************************************************************************
**
*F  Leftof(<tree1>,<index1>,<tree2>,<index2>) . . . . test if one tree is left
**                                                    of another tree
**
**  'Leftof' returns 1 if tree(<tree1>, index1) is left of tree(<tree2>,index2)
**  in the word being collected at the first instance,  that 
**  tree(<tree1>, index1) and tree(<tree2>, index2) both occur. It is assumed
**  that tree(<tree1>, index1) is not equal to tree(<tree2>, index2). 
*/
libGAP_Int     libGAP_Leftof(
                libGAP_Obj     tree1,
                libGAP_Int     index1,
                libGAP_Obj     tree2,
                libGAP_Int     index2    )
{
    if  ( libGAP_DT_LENGTH(tree1, index1) ==  1  &&  libGAP_DT_LENGTH(tree2, index2) == 1 ) {
        if (libGAP_DT_SIDE(tree1, index1) == libGAP_LEFT && libGAP_DT_SIDE(tree2, index2) == libGAP_RIGHT)
            return  1;
        else if  (libGAP_DT_SIDE(tree1, index1) == libGAP_RIGHT  &&
                  libGAP_DT_SIDE(tree2, index2) == libGAP_LEFT         )
            return  0;
        else if (libGAP_DT_GEN(tree1, index1) == libGAP_DT_GEN(tree2, index2)  )
            return ( libGAP_DT_POS(tree1, index1) < libGAP_DT_POS(tree2, index2) );
        else
            return ( libGAP_DT_GEN(tree1, index1) < libGAP_DT_GEN(tree2, index2) );
    }
    if  ( libGAP_DT_LENGTH(tree1, index1) > 1                         &&  
          libGAP_DT_LENGTH(tree2, index2) > 1                         &&
          libGAP_Equal( tree1, libGAP_DT_RIGHT(tree1, index1) , 
                 tree2, libGAP_DT_RIGHT(tree2, index2)    )                    )
    {
        if  ( libGAP_Equal( tree1, libGAP_DT_LEFT(tree1, index1),
                     tree2, libGAP_DT_LEFT(tree2, index2)  )     ) {
            if  ( libGAP_DT_GEN(tree1, index1) == libGAP_DT_GEN(tree2, index2)  )
                return   ( libGAP_DT_POS(tree1, index1) < libGAP_DT_POS(tree2, index2) );
            else
                return   ( libGAP_DT_GEN(tree1, index1) < libGAP_DT_GEN(tree2, index2) );
        }
    }
    if( libGAP_Earlier(tree1, index1, tree2, index2)  )
        return  !libGAP_Leftof2( tree2, index2, tree1, index1);
    else
        return  libGAP_Leftof2( tree1, index1, tree2, index2);
}
     
  
/*****************************************************************************
**
*F  Leftof2(<tree1>,<index1>,<tree2>,<index2>) . . . . . test if one tree is
**                                                       left of another tree
**
**  'Leftof2' returns 1 if tree(<tree1>, index1) is left of 
**  tree(<tree2>,index2)in the word being collected at the first instance,  
**  that tree(<tree1>, index1) and tree(<tree2>, index2) both occur.  It is
**  assumed that tree(<tree2>, index2) occurs earlier than 
**  tree(<tree1>,index1).  Furthermore it is assumed that if both
**  tree(<tree1>, index1) and tree(<tree2>, index2) are non-atoms,  then their
**  right trees and their left trees are not equal. 
*/
libGAP_Int    libGAP_Leftof2(
                libGAP_Obj    tree1,
                libGAP_Int    index1,
                libGAP_Obj    tree2,
                libGAP_Int    index2     )
{
    if  ( libGAP_DT_GEN(tree2, index2) < libGAP_DT_GEN(tree1, libGAP_DT_RIGHT(tree1, index1) )  )
        return  0;
    else if  (libGAP_Equal(tree1, libGAP_DT_RIGHT(tree1, index1), tree2, index2 )  )
        return  0;
    else if  (libGAP_DT_GEN(tree2, index2) == libGAP_DT_GEN(tree1, libGAP_DT_RIGHT(tree1, index1)) )
        return  libGAP_Leftof(tree1, libGAP_DT_RIGHT(tree1, index1), tree2, index2 );
    else if  (libGAP_Equal(tree1, libGAP_DT_LEFT(tree1, index1), tree2, index2) )
        return  0;
    else
        return  libGAP_Leftof(tree1, libGAP_DT_LEFT(tree1, index1), tree2, index2);
}


/****************************************************************************
**
*F  Earlier(<tree1>,<index1>,<tree2>,<index2>) . . . test if one tree occurs
**                                                   earlier than another
**
**  'Earlier' returns 1 if tree(<tree1>, index1) occurs strictly earlier than
**  tree(<tree2>, index2).  It is assumed that at least one of these trees
**  is a non-atom. Furthermore it is assumed that if both of these trees are
**  non-atoms,  right(tree(<tree1>, index1) ) does not equal
**  right(tree(<tree2>, index2) ) or left(tree(<tree1>, index1) ) does not
**  equal left(tree(<tree2>, index2) ). 
*/
libGAP_Int    libGAP_Earlier(
                libGAP_Obj    tree1,
                libGAP_Int    index1,
                libGAP_Obj    tree2,
                libGAP_Int    index2         )
{
    if  ( libGAP_DT_LENGTH(tree1, index1) == 1 )
        return  1;
    if  ( libGAP_DT_LENGTH(tree2, index2) == 1 )
        return  0;
    if ( libGAP_Equal(tree1, libGAP_DT_RIGHT(tree1, index1), 
               tree2, libGAP_DT_RIGHT(tree2, index2)  ) )
        return libGAP_Leftof(tree1, libGAP_DT_LEFT(tree2, index2),
                      tree2, libGAP_DT_LEFT(tree1, index1)  );
    if  ( libGAP_DT_GEN(tree1, libGAP_DT_RIGHT(tree1, index1) )  ==
          libGAP_DT_GEN(tree2, libGAP_DT_RIGHT(tree2, index2) )            )
        return  libGAP_Leftof( tree1, libGAP_DT_RIGHT(tree1, index1) ,
                        tree2, libGAP_DT_RIGHT(tree2, index2)      );
    return  (libGAP_DT_GEN(tree1, libGAP_DT_RIGHT(tree1, index1) )   <
             libGAP_DT_GEN(tree2, libGAP_DT_RIGHT(tree2, index2) )      );
}


/****************************************************************************
**
**  GetPols( <list>, <pr>, <pols> )
**
**  GetPols computes all representatives which are represented by the
**  pseudorepresentative <list>,  converts them all into the corresponding
**  deep thought monomial and stores all these monomials in the list <pols>.
*/

/* See below: */
void libGAP_GetReps( libGAP_Obj list, libGAP_Obj reps );
void libGAP_FindNewReps2( libGAP_Obj tree, libGAP_Obj reps, libGAP_Obj pr);

void    libGAP_GetPols( 
                libGAP_Obj    list,
                libGAP_Obj    pr,
                libGAP_Obj    pols     )
{
    libGAP_Obj    lreps,
           rreps,
           tree,
           tree1;
    libGAP_UInt   i,j,k,l, lenr, lenl, len;

    lreps = libGAP_NEW_PLIST(libGAP_T_PLIST, 2);
    rreps = libGAP_NEW_PLIST(libGAP_T_PLIST, 2);
    libGAP_SET_LEN_PLIST(lreps, 0);
    libGAP_SET_LEN_PLIST(rreps, 0);
    /*  get the representatives that are represented by <list>[1] and those
    **  which are represented by <list>[2].                                 */
    libGAP_GetReps( libGAP_ELM_PLIST(list, 1), lreps );
    libGAP_GetReps( libGAP_ELM_PLIST(list, 2), rreps );
    lenr = libGAP_LEN_PLIST(rreps);
    lenl = libGAP_LEN_PLIST(lreps);
    for  (i=1; i<=lenl; i++)
        for  (j=1; j<=lenr; j++)
            {
                /* now get all representatives, which can be constructed from
                ** <lreps>[<i>] and <rreps>[<j>] and add the corresponding
                ** deep thought monomials to <pols>                         */
                k = libGAP_LEN_PLIST( libGAP_ELM_PLIST(lreps, i) )
                  + libGAP_LEN_PLIST( libGAP_ELM_PLIST(rreps, j) ) + 5;/* m"ogliche Inkom-*/
                tree = libGAP_NEW_PLIST(libGAP_T_PLIST, k);            /* patibilit"at nach*/
                libGAP_SET_LEN_PLIST(tree, k);        /*"Anderung der Datenstruktur */
                libGAP_SET_ELM_PLIST(tree, 1, libGAP_INTOBJ_INT(1) );
                libGAP_SET_ELM_PLIST(tree, 2, libGAP_ELM_PLIST( list, 3) );
                libGAP_SET_ELM_PLIST(tree, 3, libGAP_INTOBJ_INT(0) );
                libGAP_SET_ELM_PLIST(tree, 4, libGAP_INTOBJ_INT((int)(k/5)) );
                libGAP_SET_ELM_PLIST(tree, 5, libGAP_INTOBJ_INT(0) );
                tree1 = libGAP_ELM_PLIST(lreps, i);
                len = libGAP_LEN_PLIST( tree1 );
                for  (l=1; l<=len; l++)
                    libGAP_SET_ELM_PLIST(tree, l+5, libGAP_ELM_PLIST(tree1, l) );
                k = libGAP_LEN_PLIST(tree1) + 5;
                tree1 = libGAP_ELM_PLIST(rreps, j);
                len = libGAP_LEN_PLIST( tree1 );
                for  (l=1; l<=len; l++)
                    libGAP_SET_ELM_PLIST(tree, l+k, libGAP_ELM_PLIST(tree1, l) );
                libGAP_UnmarkTree(tree);
                libGAP_FindNewReps2(tree, pols, pr);
            }
}



/****************************************************************************
**
*F  FuncGetPols( <self>, <list>, <pr>, <pols> )
**
**  FuncGetPols implements the internal function GetPols.
*/

libGAP_Obj      libGAP_FuncGetPols(
                     libGAP_Obj      self,
                     libGAP_Obj      list,
                     libGAP_Obj      pr,
                     libGAP_Obj      pols      )
{
    if  (libGAP_LEN_PLIST(list) != 4)
        libGAP_ErrorReturnVoid("<list> must be a generalised representative not a tree"
                        ,0L, 0L, "you can 'return;'");
    libGAP_GetPols(list, pr, pols);
    return (libGAP_Obj) 0;
}



/****************************************************************************
**
*F  GetReps( <list>, <reps> )
**
**  GetReps computes all representatives which are represented by the
**  pseudorepresentative <list> and adds them to the list <reps>.
*/

/* See below: */
void libGAP_FindNewReps1( libGAP_Obj tree, libGAP_Obj reps);

void    libGAP_GetReps( 
                libGAP_Obj    list,
                libGAP_Obj    reps     )
{
    libGAP_Obj    lreps,
           rreps,
           tree,
           tree1;
    libGAP_UInt   i,j,k,l, lenr, lenl, len;;

    if  ( libGAP_LEN_PLIST(list) != 4 )
    {
        libGAP_SET_ELM_PLIST(reps, 1, list);
        libGAP_SET_LEN_PLIST(reps, 1);
        return;
    }
    lreps = libGAP_NEW_PLIST(libGAP_T_PLIST, 2);
    rreps = libGAP_NEW_PLIST(libGAP_T_PLIST, 2);
    libGAP_SET_LEN_PLIST(lreps, 0);
    libGAP_SET_LEN_PLIST(rreps, 0);
    /* now get all representatives which are represented by <list>[1] and
    ** all representatives which are represented by <list>[2].           */
    libGAP_GetReps( libGAP_ELM_PLIST(list, 1), lreps );
    libGAP_GetReps( libGAP_ELM_PLIST(list, 2), rreps );
    lenl = libGAP_LEN_PLIST( lreps );
    lenr = libGAP_LEN_PLIST( rreps );
    for  (i=1; i<=lenl; i++)
        for  (j=1; j<=lenr; j++)
        {
            /* compute all representatives which can be constructed from
            ** <lreps>[<i>] and <rreps>[<j>] and add them to <reps>.   */
            k = libGAP_LEN_PLIST( libGAP_ELM_PLIST(lreps, i) )
                + libGAP_LEN_PLIST( libGAP_ELM_PLIST(rreps, j) ) + 5;/* m"ogliche Inkom-*/
            tree = libGAP_NEW_PLIST(libGAP_T_PLIST, k);            /* patibilit"at nach*/
            libGAP_SET_LEN_PLIST(tree, k);        /*"Anderung der Datenstruktur */
            libGAP_SET_ELM_PLIST(tree, 1, libGAP_INTOBJ_INT(1) );
            libGAP_SET_ELM_PLIST(tree, 2, libGAP_ELM_PLIST( list, 3) );
            libGAP_SET_ELM_PLIST(tree, 3, libGAP_INTOBJ_INT(0) );
            libGAP_SET_ELM_PLIST(tree, 4, libGAP_INTOBJ_INT((int)(k/5)) );
            if  (  libGAP_TNUM_OBJ( libGAP_ELM_PLIST(list, 4) ) == libGAP_T_INT        &&
                   libGAP_CELM(list, 4) < 100                            &&
                   libGAP_CELM(list, 4) > 0                                 )
                libGAP_SET_ELM_PLIST(tree, 5, libGAP_ELM_PLIST(list, 4) );
            else
                libGAP_SET_ELM_PLIST(tree, 5, libGAP_INTOBJ_INT(0) );
            tree1 = libGAP_ELM_PLIST(lreps, i);
            len = libGAP_LEN_PLIST( tree1 );
            for  (l=1; l<=len; l++)
                libGAP_SET_ELM_PLIST(tree, l+5, libGAP_ELM_PLIST(tree1, l) );
            k = libGAP_LEN_PLIST(tree1) + 5;
            tree1 = libGAP_ELM_PLIST(rreps, j);
            len = libGAP_LEN_PLIST( tree1 );
            for  (l=1; l<=len; l++)
                libGAP_SET_ELM_PLIST(tree, l+k, libGAP_ELM_PLIST(tree1, l) );
            libGAP_UnmarkTree(tree);
            libGAP_FindNewReps1(tree, reps);
        }
}


/**************************************************************************
**
*F  FindNewReps(<tree>,<reps>,<pr>,<max>) . . construct new representatives
**
**  'FindNewReps' constructs all trees <tree'> with the following properties.
**  1) left(<tree'>) is equivalent to left(<tree>).
**     right(<tree'>) is equivalent to right(<tree>).
**     num(<tree'>) = num(<tree>)
**  2) <tree'> is the least tree in its equivalence class.
**  3) for each marked node of (<tree>, i) of <tree> tree(<tree>, i) is equal
**     to tree(<tree'>, i).
**  There are three versions of FindNewReps. FindNewReps1 adds all found
**  trees to the list <reps>.  This version is called by GetReps.
**  FindNewReps2 computes for each found tree the corresponding deep thought
**  monomial adds these deep thought monomials to <reps>.  This version
**  is called from GetPols.
**  The third version FindNewReps finally assumes that <reps> is the list of 
**  pseudorepresentatives. This Version adds all found trees to <reps> and
**  additionally all trees, that fulfill 1), 2) and 3) except for
**  num(<tree'>) = num(<tree>).  This version is called from the library
**  function calrepsn.
**  It is assumed that both left(<tree>) and right(<tree>) are the least
**  elements in their equivalence class.
*/

/* See below: */
void  libGAP_FindSubs1( libGAP_Obj tree, libGAP_Int x, libGAP_Obj list1, libGAP_Obj list2, libGAP_Obj a, libGAP_Obj b, 
                 libGAP_Int al, libGAP_Int ar, libGAP_Int bl, libGAP_Int br, libGAP_Obj reps );

void   libGAP_FindNewReps1(
                    libGAP_Obj     tree,
                    libGAP_Obj     reps
                                      )
{
    libGAP_Obj   y,           /*  stores a copy of <tree>                       */
          lsubs,       /*  stores pos(<subtree>) for all subtrees of
                       **  left(<tree>) in a given almost equal class    */

          rsubs,       /*  stores pos(<subtree>) for all subtrees of
                       **  right(<tree>) in the same almost equal class  */

          llist,       /*  stores all elements of an almost equal class
                       **  of subtrees of left(<tree>)                   */

          rlist;       /*  stores all elements of the same almost equal
                       **  class of subtrees of right(<tree>)            */
    libGAP_Int   a,           /*  stores a subtree of right((<tree>)            */
          n,           /*  Length of lsubs                               */
          m,           /*  Length of rsubs                               */
          i;           /*  loop variable                                 */

    /*  get a subtree of right(<tree>) which is unmarked but whose 
    **  subtrees are all marked                                          */
    a = libGAP_FindTree(tree, libGAP_DT_RIGHT(tree, 1) );
    /*  If we do not find such a tree we at the bottom of the recursion.
    **  If leftof(left(<tree>),  right(<tree>) ) holds we add all <tree>
    **  to <reps>.                                                       */
    if  ( a == 0 )
    {
        if ( libGAP_Leftof(tree, libGAP_DT_LEFT(tree, 1), tree, libGAP_DT_RIGHT(tree, 1) )  )
        {
            y = libGAP_ShallowCopyPlist(tree);
            libGAP_GROW_PLIST(reps, libGAP_LEN_PLIST(reps) + 1);
            libGAP_SET_LEN_PLIST(reps, libGAP_LEN_PLIST(reps) + 1);
            libGAP_SET_ELM_PLIST(reps, libGAP_LEN_PLIST(reps), y);
            /*  tell gasman that <reps> has changed           */
            libGAP_CHANGED_BAG(reps);              
        }
        return;
    }
    /*  get all subtrees of left(<tree>) which are almost equal to
    **  tree(<tree>, a) and mark them                                  */
    llist = libGAP_Mark2(tree, libGAP_DT_LEFT(tree, 1), tree, a);
    /*  get all subtrees of right(<tree>) which are almost equal to
    **  tree(<tree>, a) and mark them                                  */
    rlist = libGAP_Mark2(tree, libGAP_DT_RIGHT(tree, 1), tree, a);
    n = libGAP_LEN_PLIST(llist);
    m = libGAP_LEN_PLIST(rlist);
    /*  if no subtrees of left(<tree>) almost equal to
    **  tree(<tree>, a) have been found there is no possibility
    **  to change the pos-argument in the trees stored in llist and
    **  rlist,  so call FindNewReps without changing any pos-arguments.
    */
    if  ( n == 0 )
    {
        libGAP_FindNewReps1(tree, reps);
        /*  unmark all top nodes of the trees stored in rlist          */
        libGAP_UnmarkAEClass(tree, rlist);
        return;
    }
    /*  store all pos-arguments that occur in the trees of llist.
    **  Note that the set of the pos-arguments in llist actually
    **  equals {1,...,n}.                                              */
    lsubs = libGAP_NEW_PLIST( libGAP_T_PLIST, n );
    libGAP_SET_LEN_PLIST(lsubs, n);
    for (i=1; i<=n; i++)
        libGAP_SET_ELM_PLIST(lsubs, i, libGAP_INTOBJ_INT(i) );
    /*  store all pos-arguments that occur in the trees of rlist.
    **  Note that the set of the pos-arguments in rlist actually
    **  equals {1,...,m}.                                              */
    rsubs = libGAP_NEW_PLIST( libGAP_T_PLIST, m );
    libGAP_SET_LEN_PLIST(rsubs, m);
    for (i=1; i<=m; i++)
        libGAP_SET_ELM_PLIST(rsubs, i, libGAP_INTOBJ_INT(i) );
    /*  find all possibilities for lsubs and rsubs such that
    **  lsubs[1] < lsubs[2] <...<lsubs[n],
    **  rsubs[1] < rsubs[2] <...<rsubs[n],
    **  and set(lsubs concat rsubs) equals {1,...,k} for a positiv
    **  integer k.  For each found lsubs and rsubs 'FindSubs' changes
    **  pos-arguments of the subtrees in llist and rlist accordingly
    **  and  then calls 'FindNewReps' with the changed tree <tree>.
    */
    libGAP_FindSubs1(tree, a, llist, rlist, lsubs, rsubs, 1, n, 1, m, reps);
    /*  Unmark the subtrees of <tree> in llist and rlist and reset
    **  pos-arguments to the original state.                            */
    libGAP_UnmarkAEClass(tree, rlist);
    libGAP_UnmarkAEClass(tree, llist);
}

/* See below: */
void  libGAP_FindSubs2( libGAP_Obj tree, libGAP_Int x, libGAP_Obj list1, libGAP_Obj list2, libGAP_Obj a, libGAP_Obj b, 
                 libGAP_Int al, libGAP_Int ar, libGAP_Int bl, libGAP_Int br, libGAP_Obj reps, libGAP_Obj pr );

void   libGAP_FindNewReps2(
                    libGAP_Obj     tree,
                    libGAP_Obj     reps,
                    libGAP_Obj     pr  /*  pc-presentation for a 
                                 **  nilpotent group <G>                 */
                                      )
{
    libGAP_Obj   lsubs,       /*  stores pos(<subtree>) for all subtrees of
                       **  left(<tree>) in a given almost equal class    */

          rsubs,       /*  stores pos(<subtree>) for all subtrees of
                       **  right(<tree>) in the same almost equal class  */

          llist,       /*  stores all elements of an almost equal class
                       **  of subtrees of left(<tree>)                   */

          rlist;       /*  stores all elements of the same almost equal
                       **  class of subtrees of right(<tree>)            */
    libGAP_Int   a,           /*  stores a subtree of right((<tree>)            */
          n,           /*  Length of lsubs                               */
          m,           /*  Length of rsubs                               */
          i;           /*  loop variable                                 */

    /*  get a subtree of right(<tree>) which is unmarked but whose 
    **  subtrees are all marked                                          */
    a = libGAP_FindTree(tree, libGAP_DT_RIGHT(tree, 1) );
    /*  If we do not find such a tree we at the bottom of the recursion.
    **  If leftof(left(<tree>),  right(<tree>) ) holds we convert <tree>
    **  into the corresponding deep thought monomial and add that to
    **  <reps>.                                                          */
    if  ( a == 0 )
    {
        if ( libGAP_Leftof(tree, libGAP_DT_LEFT(tree, 1), tree, libGAP_DT_RIGHT(tree, 1) )  )
        {
                /*  get the formula vector of tree and add it to
                **  reps[ rel[1] ].                                */  
            libGAP_UnmarkTree(tree);
            tree = libGAP_MakeFormulaVector( tree, pr);
            libGAP_CALL_3ARGS(libGAP_Dt_add, tree, reps, pr);
        }
        return;
    }
    /*  get all subtrees of left(<tree>) which are almost equal to
    **  tree(<tree>, a) and mark them                                  */
    llist = libGAP_Mark2(tree, libGAP_DT_LEFT(tree, 1), tree, a);
    /*  get all subtrees of right(<tree>) which are almost equal to
    **  tree(<tree>, a) and mark them                                  */
    rlist = libGAP_Mark2(tree, libGAP_DT_RIGHT(tree, 1), tree, a);
    n = libGAP_LEN_PLIST(llist);
    m = libGAP_LEN_PLIST(rlist);
    /*  if no subtrees of left(<tree>) almost equal to
    **  tree(<tree>, a) have been found there is no possibility
    **  to change the pos-argument in the trees stored in llist and
    **  rlist,  so call FindNewReps without changing any pos-arguments.
    */
    if  ( n == 0 )
    {
        libGAP_FindNewReps2(tree, reps, pr);
        /*  unmark all top nodes of the trees stored in rlist          */
        libGAP_UnmarkAEClass(tree, rlist);
        return;
    }
    /*  store all pos-arguments that occur in the trees of llist.
    **  Note that the set of the pos-arguments in llist actually
    **  equals {1,...,n}.                                              */
    lsubs = libGAP_NEW_PLIST( libGAP_T_PLIST, n );
    libGAP_SET_LEN_PLIST(lsubs, n);
    for (i=1; i<=n; i++)
        libGAP_SET_ELM_PLIST(lsubs, i, libGAP_INTOBJ_INT(i) );
    /*  store all pos-arguments that occur in the trees of rlist.
    **  Note that the set of the pos-arguments in rlist actually
    **  equals {1,...,m}.                                              */
    rsubs = libGAP_NEW_PLIST( libGAP_T_PLIST, m );
    libGAP_SET_LEN_PLIST(rsubs, m);
    for (i=1; i<=m; i++)
        libGAP_SET_ELM_PLIST(rsubs, i, libGAP_INTOBJ_INT(i) );
    /*  find all possibilities for lsubs and rsubs such that
    **  lsubs[1] < lsubs[2] <...<lsubs[n],
    **  rsubs[1] < rsubs[2] <...<rsubs[n],
    **  and set(lsubs concat rsubs) equals {1,...,k} for a positiv
    **  integer k.  For each found lsubs and rsubs 'FindSubs' changes
    **  pos-arguments of the subtrees in llist and rlist accordingly
    **  and  then calls 'FindNewReps' with the changed tree <tree>.
    */
    libGAP_FindSubs2(tree, a, llist, rlist, lsubs, rsubs, 1, n, 1, m, reps, pr);
    /*  Unmark the subtrees of <tree> in llist and rlist and reset
    **  pos-arguments to the original state.                            */
    libGAP_UnmarkAEClass(tree, rlist);
    libGAP_UnmarkAEClass(tree, llist);
}


void   libGAP_FindNewReps(
                    libGAP_Obj     tree,
                    libGAP_Obj     reps,
                    libGAP_Obj     pr,  /*  pc-presentation for a 
                                 **  nilpotent group <G>                 */

                    libGAP_Obj     max  /*  every generator <g_i> of <G> with
                                 **  i > max lies in the center of <G>   */
                                      )
{
    libGAP_Obj   y,           /*  stores a copy of <tree>                       */
          lsubs,       /*  stores pos(<subtree>) for all subtrees of
                       **  left(<tree>) in a given almost equal class    */

          rsubs,       /*  stores pos(<subtree>) for all subtrees of
                       **  right(<tree>) in the same almost equal class  */

          llist,       /*  stores all elements of an almost equal class
                       **  of subtrees of left(<tree>)                   */

          rlist,       /*  stores all elements of the same almost equal
                       **  class of subtrees of right(<tree>)            */
          list1,       /*  stores a sublist of <reps>                    */
          rel;         /*  stores a commutator relation from <pr>        */
    libGAP_Int   a;           /*  stores a subtree of right((<tree>)            */
    libGAP_UInt  n,           /*  Length of lsubs                               */
          m,           /*  Length of rsubs                               */
          i, lenrel;   /*  loop variable                                 */

    /*  get a subtree of right(<tree>) which is unmarked but whose 
    **  subtrees are all marked                                          */
    a = libGAP_FindTree(tree, libGAP_DT_RIGHT(tree, 1) );
    /*  If we do not find such a tree we at the bottom of the recursion.
    **  If leftof(left(<tree>),  right(<tree>) ) holds we add all trees
    **  <tree'> with left(<tree'>) = left(<tree>), 
    **  right(<tree'>) = right(<tree>) to <reps>,  and <tree'> is the
    **  least element in its equivalence calss.  Note that for such a 
    **  tree we have pos(<tree'>) = 1 and num(<tree'>) = j where j is a
    **  positive integer for which
    **  c_( num(left(<tree>),  num(right(<tree>)), j ) does not equal
    **  0.  These integers are contained in the list
    **  pr[ num(left(<tree>)) ][ num(right(<tree>)) ].             */
    if  ( a == 0 )
    {
        if ( libGAP_Leftof(tree, libGAP_DT_LEFT(tree, 1), tree, libGAP_DT_RIGHT(tree, 1) )  )
        {
            /*  get  pr[ num(left(<tree>)) ][ num(right(<tree>)) ]      */
            rel = libGAP_ELM_PLIST( libGAP_ELM_PLIST(pr, libGAP_INT_INTOBJ( libGAP_DT_GEN(tree, 
                                                         libGAP_DT_LEFT(tree, 1)))) ,
                             libGAP_INT_INTOBJ( libGAP_DT_GEN(tree, libGAP_DT_RIGHT(tree, 1) ) )  );
            if  ( libGAP_ELM_PLIST(rel, 3) > max )
            {
              libGAP_UnmarkTree(tree);
              tree = libGAP_MakeFormulaVector(tree, pr);
              list1 = libGAP_ELM_PLIST(reps, libGAP_CELM(rel, 3) );
              libGAP_GROW_PLIST(list1, libGAP_LEN_PLIST(list1) + 1 );
              libGAP_SET_LEN_PLIST(list1, libGAP_LEN_PLIST(list1) + 1 );
              libGAP_SET_ELM_PLIST(list1, libGAP_LEN_PLIST(list1), tree);
              libGAP_CHANGED_BAG(list1);
            }
            else
            {
                y = libGAP_ShallowCopyPlist(tree);
                lenrel = libGAP_LEN_PLIST(rel);
                for  (  i=3;  
                        i < lenrel  &&
                        libGAP_ELM_PLIST(rel, i) <= max;  
                        i+=2                                        )
                {
                    list1 = libGAP_ELM_PLIST(reps, libGAP_CELM(rel, i)  );
                    libGAP_GROW_PLIST(list1, libGAP_LEN_PLIST(list1) + 1);
                    libGAP_SET_LEN_PLIST(list1, libGAP_LEN_PLIST(list1) + 1);
                    libGAP_SET_ELM_PLIST(list1, libGAP_LEN_PLIST(list1), y);
                    /*  tell gasman that <list1> has changed           */
                    libGAP_CHANGED_BAG(list1);             
                }
            }
        }
        return;
    }
    /*  get all subtrees of left(<tree>) which are almost equal to
    **  tree(<tree>, a) and mark them                                  */
    llist = libGAP_Mark2(tree, libGAP_DT_LEFT(tree, 1), tree, a);
    /*  get all subtrees of right(<tree>) which are almost equal to
    **  tree(<tree>, a) and mark them                                  */
    rlist = libGAP_Mark2(tree, libGAP_DT_RIGHT(tree, 1), tree, a);
    n = libGAP_LEN_PLIST(llist);
    m = libGAP_LEN_PLIST(rlist);
    /*  if no subtrees of left(<tree>) almost equal to
    **  tree(<tree>, a) have been found there is no possibility
    **  to change the pos-argument in the trees stored in llist and
    **  rlist,  so call FindNewReps without changing any pos-arguments.
    */
    if  ( n == 0 )
    {
        libGAP_FindNewReps(tree, reps, pr, max);
        /*  unmark all top nodes of the trees stored in rlist          */
        libGAP_UnmarkAEClass(tree, rlist);
        return;
    }
    /*  store all pos-arguments that occur in the trees of llist.
    **  Note that the set of the pos-arguments in llist actually
    **  equals {1,...,n}.                                              */
    lsubs = libGAP_NEW_PLIST( libGAP_T_PLIST, n );
    libGAP_SET_LEN_PLIST(lsubs, n);
    for (i=1; i<=n; i++)
        libGAP_SET_ELM_PLIST(lsubs, i, libGAP_INTOBJ_INT(i) );
    /*  store all pos-arguments that occur in the trees of rlist.
    **  Note that the set of the pos-arguments in rlist actually
    **  equals {1,...,m}.                                              */
    rsubs = libGAP_NEW_PLIST( libGAP_T_PLIST, m );
    libGAP_SET_LEN_PLIST(rsubs, m);
    for (i=1; i<=m; i++)
        libGAP_SET_ELM_PLIST(rsubs, i, libGAP_INTOBJ_INT(i) );
    /*  find all possibilities for lsubs and rsubs such that
    **  lsubs[1] < lsubs[2] <...<lsubs[n],
    **  rsubs[1] < rsubs[2] <...<rsubs[n],
    **  and set(lsubs concat rsubs) equals {1,...,k} for a positiv
    **  integer k.  For each found lsubs and rsubs 'FindSubs' changes
    **  pos-arguments of the subtrees in llist and rlist accordingly
    **  and  then calls 'FindNewReps' with the changed tree <tree>.
    */
    libGAP_FindSubs(tree, a, llist, rlist, lsubs, rsubs, 1, n, 1, m, reps, pr, max);
    /*  Unmark the subtrees of <tree> in llist and rlist and reset
    **  pos-arguments to the original state.                            */
    libGAP_UnmarkAEClass(tree, rlist);
    libGAP_UnmarkAEClass(tree, llist);
}


/***************************************************************************
**
*F  FuncFindNewReps(<self>,<args>) . . . . . . construct new representatives
**
**  'FuncFindNewReps' implements the internal function 'FindNewReps'.
*/

libGAP_Obj    libGAP_FuncFindNewReps(
                        libGAP_Obj     self,
                        libGAP_Obj     tree,
                        libGAP_Obj     reps,
                        libGAP_Obj     pr,
                        libGAP_Obj     max       )
{

    /*  test if <tree> is really a tree                                    */
    /* TestTree(tree);                                                     */
    if  ( libGAP_LEN_PLIST(tree) < 15 )  
        libGAP_ErrorReturnVoid("<tree> must be a tree not a plain list", 0L, 0L,
                        "you can 'return;'");
    libGAP_FindNewReps(tree, reps, pr, max);   
    return  0;
}


/***************************************************************************
**
*F  TestTree(<obj>) . . . . . . . . . . . . . . . . . . . . . . test a tree
**
**  'TestTree' tests if <tree> is a tree. If <tree> is not a tree 'TestTree'
**  signals an error.
*/
void  libGAP_TestTree(
               libGAP_Obj     tree)
{

    if ( libGAP_TNUM_OBJ(tree) != libGAP_T_PLIST || libGAP_LEN_PLIST(tree) % 7 != 0)
        libGAP_ErrorReturnVoid("<tree> must be a plain list,  whose length is a multiple of 7", 0L, 0L, "you can 'return;'");
    if ( libGAP_DT_LENGTH(tree, 1) != libGAP_LEN_PLIST(tree)/7 )
        libGAP_ErrorReturnVoid("<tree> must be a tree, not a plain list.", 0L, 0L,
                        "you can 'return;'");
    if ( libGAP_DT_SIDE(tree, 1) >= libGAP_DT_LENGTH(tree, 1) )
        libGAP_ErrorReturnVoid("<tree> must be a tree, not a plain list.", 0L, 0L,
                        "you can 'return;'");        
    if ( libGAP_DT_LENGTH(tree, 1) == 1)
    {
        if ( libGAP_DT_SIDE(tree, 1) != libGAP_LEFT && libGAP_DT_SIDE(tree, 1) != libGAP_RIGHT )
            libGAP_ErrorReturnVoid("<tree> must be a tree, not a plain list.", 0L, 0L,
                            "you can 'return;'");
        return;
    }
    if ( libGAP_DT_SIDE(tree, 1) <= 1 )
        libGAP_ErrorReturnVoid("<tree> must be a tree, not a plain list.", 0L, 0L,
                        "you can 'return;'");
    if (libGAP_DT_LENGTH(tree, 1) !=
          libGAP_DT_LENGTH(tree, libGAP_DT_LEFT(tree, 1)) + 
          libGAP_DT_LENGTH(tree, libGAP_DT_RIGHT(tree, 1)) +  
          1                                           )
        libGAP_ErrorReturnVoid("<tree> must be a tree, not a plain list.", 0L, 0L,
                        "you can 'return;'");
    if ( libGAP_DT_SIDE(tree, 1) != libGAP_DT_LENGTH(tree, libGAP_DT_LEFT(tree, 1) ) + 1 )
        libGAP_ErrorReturnVoid("<tree> must be a tree, not a plain list.", 0L, 0L,
                        "you can 'return;'");
    libGAP_TestTree( libGAP_Part(tree, (libGAP_DT_LEFT(tree, 1) - 1)*7, 
                         (libGAP_DT_RIGHT(tree, 1) - 1)*7                    )    );
    libGAP_TestTree( libGAP_Part(tree, (libGAP_DT_RIGHT(tree, 1) - 1)*7,  libGAP_LEN_PLIST(tree) ) );
}


/****************************************************************************
**
*F  Part(<list>, <pos1>, <pos2> . . . . . . . . . . . . return a part of list
**
**  'Part' returns <list>{ [<pos1>+1 .. <pos2>] }.
*/
libGAP_Obj    libGAP_Part(
             libGAP_Obj      list,
             libGAP_Int      pos1,
             libGAP_Int      pos2  )
{
    libGAP_Int      i, length;
    libGAP_Obj      part;

    length = pos2 - pos1;
    part = libGAP_NEW_PLIST(libGAP_T_PLIST, length);
    libGAP_SET_LEN_PLIST(part, length);
    for (i=1; i <= length; i++)
    {
        libGAP_SET_ELM_PLIST(part, i, libGAP_ELM_PLIST(list, pos1+i) );
    }
    return part;
}


/***************************************************************************
**
*F  FindSubs(<tree>,<x>,<list1>,<list2>,<a>,<b>,<al>,<ar>,<bl>,<br>,<reps>,
**           <pr>,<max>  ) . . . . . . . . . find possible pos-arguments for 
**                                           the trees in <list1> and <list2>
**
**  'FindSubs' finds all possibilities for a and b such that
**  1) a[1] < a[2] <..< a[ ar ]
**     b[1] < b[2] <..< b[ br ]
**  2) set( a concat b ) = {1,..,k} for a positiv integer k.
**  3) a[1],...,a[ al-1 ] and b[1],..,b[ bl-1 ] remain unchanged.
**  For each found possibility 'FindSubs' sets the pos-arguments in the
**  trees of <list1> and <list2> according to the entries of <a> and
**  <b>.  Then it calls 'FindNewReps' with the changed tree <tree> as
v**  argument.
**
**  It is assumed that the conditions 1) and 2) hold for a{ [1..al-1] } and
**  b{ [1..bl-1] }.  
**
**  There are three versions of FindSubs according to the three versions of
**  FindNewReps.  FindSubs1 is called from FindNewReps1 and calls
**  FindNewReps1.  FindSubs2 is called from FindNewReps2 and calls 
**  FindNewReps2.  FindSubs is called from FindNewReps and calls FindNewReps.
*/

void  libGAP_FindSubs1(
                libGAP_Obj        tree,
                libGAP_Int        x,     /*  subtree of <tree>                     */
                libGAP_Obj        list1, /*  list containing all subtrees of
                                  **  left(<tree>) almost equal to
                                  **  tree(<tree>, x)                       */
                                  
                libGAP_Obj        list2, /*  list containing all subtrees of
                                  **  right(<tree>) almost equal to
                                  **  tree(<tree>, x)                       */

                libGAP_Obj        a,     /*  list to change,  containing the
                                  **  pos-arguments of the trees in list1   */

                libGAP_Obj        b,     /*  list to change,  containing tthe
                                  **  pos-arguments of the trees in list2   */
                libGAP_Int        al,
                libGAP_Int        ar,
                libGAP_Int        bl,
                libGAP_Int        br,
                libGAP_Obj        reps  /*  list of representatives for all trees */
                                        )
{
   libGAP_Int    i;  /*  loop variable                                             */

   /*  if <al> > <ar> or <bl> > <br> nothing remains to change.             */
   if (  al > ar  ||  bl > br  )
   {
       /*  Set the pos-arguments of the trees in <list1> and <list2>
       **  according to the entries of <a> and <b>.                         */
       libGAP_SetSubs( list1, a, tree);
       libGAP_SetSubs( list2, b, tree);
       libGAP_FindNewReps1(tree, reps);
       return;
   }
   /*  If a[ ar] is bigger or equal to the boundary of pos(tree(<tree>, x)
   **  the execution of the statements in the body of this if-statement
   **  would have the consequence that some subtrees of <tree> in <list1>
   **  would get a pos-argument bigger than the boundary of
   **  pos(tree<tree>, x).  But since the trees in <list1> are almost
   **  equal to tree(<tree>, x) they have all the same boundary for their
   **  pos-argument as tree(<tree>, x).  So these statements are only
   **  executed when <a>[ar] is less than the boundary of 
   **  pos(tree(<tree>, x).
   */
   if ( libGAP_INT_INTOBJ( libGAP_DT_MAX(tree, x) ) <= 0  ||  
        libGAP_ELM_PLIST(a, ar) < libGAP_DT_MAX(tree, x)   )
   {
       for (i=al; i<=ar; i++)
           libGAP_SET_ELM_PLIST(a, i, libGAP_INTOBJ_INT( libGAP_CELM(a,i) + 1 ) );
       libGAP_FindSubs1(tree, x, list1, list2, a, b, al, ar, bl+1, br, reps);
       for  (i=al; i<=ar; i++)
           libGAP_SET_ELM_PLIST(a, i, libGAP_INTOBJ_INT( libGAP_CELM(a, i) - 1  ) );
   }
   libGAP_FindSubs1(tree, x, list1, list2, a, b, al+1, ar, bl+1, br, reps);
   /*  If b[ br] is bigger or equal to the boundary of pos(tree(<tree>, x)
   **  the execution of the statements in the body of this if-statement
   **  would have the consequence that some subtrees of <tree> in <list2>
   **  would get a pos-argument bigger than the boundary of
   **  pos(tree<tree>, x).  But since the trees in <list2> are almost
   **  equal to tree(<tree>, x) they have all the same boundary for their
   **  pos-argument as tree(<tree>, x).  So these statements are only
   **  executed when <b>[br] is less than the boundary of 
   **  pos(tree(<tree>, x).
   */
   if ( libGAP_INT_INTOBJ( libGAP_DT_MAX(tree, x) ) <= 0  ||
        libGAP_ELM_PLIST(b, br) < libGAP_DT_MAX(tree, x)        )
   {
       for  (i=bl; i<=br; i++)
           libGAP_SET_ELM_PLIST(b, i, libGAP_INTOBJ_INT( libGAP_CELM(b, i) + 1  ) );
       libGAP_FindSubs1(tree, x, list1, list2, a, b, al+1, ar, bl, br, reps);
       for  (i=bl; i<=br; i++)
           libGAP_SET_ELM_PLIST(b, i, libGAP_INTOBJ_INT( libGAP_CELM(b, i) - 1 ) );
   }
}


void  libGAP_FindSubs2(
                libGAP_Obj        tree,
                libGAP_Int        x,     /*  subtree of <tree>                     */
                libGAP_Obj        list1, /*  list containing all subtrees of
                                  **  left(<tree>) almost equal to
                                  **  tree(<tree>, x)                       */
                                  
                libGAP_Obj        list2, /*  list containing all subtrees of
                                  **  right(<tree>) almost equal to
                                  **  tree(<tree>, x)                       */

                libGAP_Obj        a,     /*  list to change,  containing the
                                  **  pos-arguments of the trees in list1   */

                libGAP_Obj        b,     /*  list to change,  containing the
                                  **  pos-arguments of the trees in list2   */
                libGAP_Int        al,
                libGAP_Int        ar,
                libGAP_Int        bl,
                libGAP_Int        br,
                libGAP_Obj        reps,  /*  list of representatives for all trees */
                libGAP_Obj        pr    /*  pc-presentation                       */
                                        )
{
   libGAP_Int    i;  /*  loop variable                                             */

   /*  if <al> > <ar> or <bl> > <br> nothing remains to change.             */
   if (  al > ar  ||  bl > br  )
   {
       /*  Set the pos-arguments of the trees in <list1> and <list2>
       **  according to the entries of <a> and <b>.                         */
       libGAP_SetSubs( list1, a, tree);
       libGAP_SetSubs( list2, b, tree);
       libGAP_FindNewReps2(tree, reps, pr);
       return;
   }
   /*  If a[ ar] is bigger or equal to the boundary of pos(tree(<tree>, x)
   **  the execution of the statements in the body of this if-statement
   **  would have the consequence that some subtrees of <tree> in <list1>
   **  would get a pos-argument bigger than the boundary of
   **  pos(tree<tree>, x).  But since the trees in <list1> are almost
   **  equal to tree(<tree>, x) they have all the same boundary for their
   **  pos-argument as tree(<tree>, x).  So these statements are only
   **  executed when <a>[ar] is less than the boundary of 
   **  pos(tree(<tree>, x).
   */
   if ( libGAP_INT_INTOBJ( libGAP_DT_MAX(tree, x) ) <= 0  ||  
        libGAP_ELM_PLIST(a, ar) < libGAP_DT_MAX(tree, x)   )
   {
       for (i=al; i<=ar; i++)
           libGAP_SET_ELM_PLIST(a, i, libGAP_INTOBJ_INT( libGAP_CELM(a,i) + 1 ) );
       libGAP_FindSubs2(tree, x, list1, list2, a, b, al, ar, bl+1, br, reps, pr);
       for  (i=al; i<=ar; i++)
           libGAP_SET_ELM_PLIST(a, i, libGAP_INTOBJ_INT( libGAP_CELM(a, i) - 1  ) );
   }
   libGAP_FindSubs2(tree, x, list1, list2, a, b, al+1, ar, bl+1, br, reps, pr);
   /*  If b[ br] is bigger or equal to the boundary of pos(tree(<tree>, x)
   **  the execution of the statements in the body of this if-statement
   **  would have the consequence that some subtrees of <tree> in <list2>
   **  would get a pos-argument bigger than the boundary of
   **  pos(tree<tree>, x).  But since the trees in <list2> are almost
   **  equal to tree(<tree>, x) they have all the same boundary for their
   **  pos-argument as tree(<tree>, x).  So these statements are only
   **  executed when <b>[br] is less than the boundary of 
   **  pos(tree(<tree>, x).
   */
   if ( libGAP_INT_INTOBJ( libGAP_DT_MAX(tree, x) ) <= 0  ||
        libGAP_ELM_PLIST(b, br) < libGAP_DT_MAX(tree, x)        )
   {
       for  (i=bl; i<=br; i++)
           libGAP_SET_ELM_PLIST(b, i, libGAP_INTOBJ_INT( libGAP_CELM(b, i) + 1  ) );
       libGAP_FindSubs2(tree, x, list1, list2, a, b, al+1, ar, bl, br, reps, pr);
       for  (i=bl; i<=br; i++)
           libGAP_SET_ELM_PLIST(b, i, libGAP_INTOBJ_INT( libGAP_CELM(b, i) - 1 ) );
   }
}


void  libGAP_FindSubs(
                libGAP_Obj        tree,
                libGAP_Int        x,     /*  subtree of <tree>                     */
                libGAP_Obj        list1, /*  list containing all subtrees of
                                  **  left(<tree>) almost equal to
                                  **  tree(<tree>, x)                       */
                                  
                libGAP_Obj        list2, /*  list containing all subtrees of
                                  **  right(<tree>) almost equal to
                                  **  tree(<tree>, x)                       */

                libGAP_Obj        a,     /*  list to change,  containing the
                                  **  pos-arguments of the trees in list1   */

                libGAP_Obj        b,     /*  list to change,  containing the
                                  **  pos-arguments of the trees in list2   */
                libGAP_Int        al,
                libGAP_Int        ar,
                libGAP_Int        bl,
                libGAP_Int        br,
                libGAP_Obj        reps,  /*  list of representatives for all trees */
                libGAP_Obj        pr,    /*  pc-presentation                       */
                libGAP_Obj        max    /*  needed to call 'FindNewReps'          */
                                        )
{
   libGAP_Int    i;  /*  loop variable                                             */

   /*  if <al> > <ar> or <bl> > <br> nothing remains to change.             */
   if (  al > ar  ||  bl > br  )
   {
       /*  Set the pos-arguments of the trees in <list1> and <list2>
       **  according to the entries of <a> and <b>.                         */
       libGAP_SetSubs( list1, a, tree);
       libGAP_SetSubs( list2, b, tree);
       libGAP_FindNewReps(tree, reps, pr, max);
       return;
   }
   /*  If a[ ar] is bigger or equal to the boundary of pos(tree(<tree>, x)
   **  the execution of the statements in the body of this if-statement
   **  would have the consequence that some subtrees of <tree> in <list1>
   **  would get a pos-argument bigger than the boundary of
   **  pos(tree<tree>, x).  But since the trees in <list1> are almost
   **  equal to tree(<tree>, x) they have all the same boundary for their
   **  pos-argument as tree(<tree>, x).  So these statements are only
   **  executed when <a>[ar] is less than the boundary of 
   **  pos(tree(<tree>, x).
   */
   if ( libGAP_INT_INTOBJ( libGAP_DT_MAX(tree, x) ) <= 0  ||  
        libGAP_ELM_PLIST(a, ar) < libGAP_DT_MAX(tree, x)   )
   {
       for (i=al; i<=ar; i++)
           libGAP_SET_ELM_PLIST(a, i, libGAP_INTOBJ_INT( libGAP_CELM(a,i) + 1 ) );
       libGAP_FindSubs(tree, x, list1, list2, a, b, al, ar, bl+1, br, reps, pr, max);
       for  (i=al; i<=ar; i++)
           libGAP_SET_ELM_PLIST(a, i, libGAP_INTOBJ_INT( libGAP_CELM(a, i) - 1  ) );
   }
   libGAP_FindSubs(tree, x, list1, list2, a, b, al+1, ar, bl+1, br, reps, pr, max);
   /*  If b[ br] is bigger or equal to the boundary of pos(tree(<tree>, x)
   **  the execution of the statements in the body of this if-statement
   **  would have the consequence that some subtrees of <tree> in <list2>
   **  would get a pos-argument bigger than the boundary of
   **  pos(tree<tree>, x).  But since the trees in <list2> are almost
   **  equal to tree(<tree>, x) they have all the same boundary for their
   **  pos-argument as tree(<tree>, x).  So these statements are only
   **  executed when <b>[br] is less than the boundary of 
   **  pos(tree(<tree>, x).
   */
   if ( libGAP_INT_INTOBJ( libGAP_DT_MAX(tree, x) ) <= 0  ||
        libGAP_ELM_PLIST(b, br) < libGAP_DT_MAX(tree, x)        )
   {
       for  (i=bl; i<=br; i++)
           libGAP_SET_ELM_PLIST(b, i, libGAP_INTOBJ_INT( libGAP_CELM(b, i) + 1  ) );
       libGAP_FindSubs(tree, x, list1, list2, a, b, al+1, ar, bl, br, reps, pr, max);
       for  (i=bl; i<=br; i++)
           libGAP_SET_ELM_PLIST(b, i, libGAP_INTOBJ_INT( libGAP_CELM(b, i) - 1 ) );
   }
}


/****************************************************************************
**
*F  SetSubs(<list>, <a>, <tree>) . . . . . . . . . . .. .  set pos-arguments
** 
**  'SetSubs' sets the pos-arguments of the subtrees of <tree>,  contained
**  in <list> according to the entries in the list <a>.
*/
void    libGAP_SetSubs(
                 libGAP_Obj       list,
                 libGAP_Obj       a,
                 libGAP_Obj       tree    )
{
    libGAP_UInt   i,j;  /*  loop variables                                         */
    libGAP_UInt   len, len2;
    
    len = libGAP_LEN_PLIST(list);
    for  (i=1; i <= len; i++)
    {
        len2 = libGAP_LEN_PLIST( libGAP_ELM_PLIST(list, i) );
        for  (j=1;  j <= len2;  j++)
            libGAP_SET_DT_POS(tree, libGAP_CELM( libGAP_ELM_PLIST(list, i), j), libGAP_ELM_PLIST(a, i) );
    }
}


/****************************************************************************
**
*F  UnmarkAEClass(<tree>, <list>) . . . . . . . . . . . . reset pos-arguments
**
**  'UnmarkAEClass' resets the pos arguments of the subtrees of <tree>,
**  contained in <list> to the original state.  Furthermore it unmarks the
**  top node of each of those trees.
*/

void    libGAP_UnmarkAEClass(
                       libGAP_Obj      tree,
                       libGAP_Obj      list  )
{
    libGAP_UInt  i,j, len, len2;

    len = libGAP_LEN_PLIST(list);
    for  (i=1; i <= len; i++)
    {
        len2 = libGAP_LEN_PLIST( libGAP_ELM_PLIST(list, i) );
        for (j=1;  j <= len2;  j++)
        {
            libGAP_DT_UNMARK(tree, libGAP_CELM( libGAP_ELM_PLIST(list, i), j)  );
            libGAP_SET_DT_POS(tree, libGAP_CELM( libGAP_ELM_PLIST(list, i), j), libGAP_INTOBJ_INT(i) );
        }
    }
}


/****************************************************************************
**
*F  FuncDT_evaluation( <self>, <vector> )
**
**  FuncDT_evaluation implements the internal function
**
**  DT_evaluation( <vector> ).
**
**  DT_evaluation returns a positive integer which is used to sort the deep
**  monomials.  DT_evaluation is called from the library function dt_add.
*/

libGAP_Obj    libGAP_FuncDT_evaluation(libGAP_Obj      self,
                    libGAP_Obj      vector)
{
    libGAP_UInt   res,i;

    res = libGAP_CELM(vector, 6)*libGAP_CELM(vector, 6);
    for  (i=7; i < libGAP_LEN_PLIST(vector); i+=2)
        res += libGAP_CELM(vector, i)*libGAP_CELM(vector, i+1)*libGAP_CELM(vector, i+1);
    return libGAP_INTOBJ_INT(res);
}



/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "MakeFormulaVector", 2, "tree, presentation",
      libGAP_FuncMakeFormulaVector, "src/dt.c:MakeFormulaVector" },

    { "FindNewReps", 4, "tree, representatives, presentation, maximum",
      libGAP_FuncFindNewReps, "src/dt.c:FindNewReps" },

    { "UnmarkTree", 1, "tree",
      libGAP_FuncUnmarkTree, "src/dt.c:UnmarkTree" },

    { "GetPols", 3, "list, presentation, polynomial",
      libGAP_FuncGetPols, "src/dt.c:GetPols" },
    
    { "DT_evaluation", 1, "vector",
      libGAP_FuncDT_evaluation, "src/dt.c:DT_evaluation" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_InitFopyGVar( "Dt_add" , &libGAP_Dt_add );

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoDeepThought() . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "dt",                               /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoDeepThought ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  dt.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
**
*/
