/****************************************************************************
**
*W  vars.c                      GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of variables package.
**
**  The variables  package is  the  part of   the interpreter  that  executes
**  assignments to variables and evaluates references to variables.
**
**  There are five  kinds of variables,  local variables (i.e., arguments and
**  locals), higher variables (i.e., local variables of enclosing functions),
**  global variables, list elements, and record elements.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "ariths.h"              /* equality */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */

#include        "records.h"             /* generic records                 */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */

#include        "exprs.h"               /* expressions                     */
#include        "stats.h"               /* statements                      */

#include        "tls.h"                 /* thread-local storage            */

#include        "vars.h"                /* variables                       */


#include        "aobjects.h"            /* atomic objects                  */
#include        "saveload.h"            /* saving and loading              */

#include	    "thread.h"		        /* threads                         */
#include        "profile.h"             /* installing methods              */


/****************************************************************************
**

*V  CurrLVars   . . . . . . . . . . . . . . . . . . . . . local variables bag
**
**  'CurrLVars'  is the bag containing the  values  of the local variables of
**  the currently executing interpreted function.
**
**  Assignments  to  the local variables change   this bag.  We  do  not call
**  'CHANGED_BAG' for  each of such change.  Instead we wait until  a garbage
**  collection begins  and then  call  'CHANGED_BAG'  in  'BeginCollectBags'.
*/
libGAP_Bag libGAP_CurrLVars;


/****************************************************************************
**
*V  BottomLVars . . . . . . . . . . . . . . . . .  bottom local variables bag
**
**  'BottomLVars' is the local variables bag at the bottom of the call stack.
**  Without   such a dummy  frame at  the bottom, 'SWITCH_TO_NEW_LVARS' would
**  have to check for the bottom, slowing it down.
**
*/
libGAP_Bag libGAP_BottomLVars;


/****************************************************************************
**
*V  PtrLVars  . . . . . . . . . . . . . . . .  pointer to local variables bag
**
**  'PtrLVars' is a pointer to the 'TLS(CurrLVars)' bag.  This  makes it faster to
**  access local variables.
**
**  Since   a   garbage collection may  move   this  bag  around, the pointer
**  'PtrLVars' must be recalculated afterwards in 'VarsAfterCollectBags'.
*/
libGAP_Obj * libGAP_PtrLVars;


/****************************************************************************
**

*F  ObjLVar(<lvar>) . . . . . . . . . . . . . . . . value of a local variable
**
**  'ObjLVar' returns the value of the local variable <lvar>.
*/
libGAP_Obj             libGAP_ObjLVar (
    libGAP_UInt                lvar )
{
    libGAP_Obj                 val;            /* value result                    */
    while ( (val = libGAP_OBJ_LVAR(lvar)) == 0 ) {
        libGAP_ErrorReturnVoid(
            "Variable: '%s' must have an assigned value",
            (libGAP_Int)libGAP_NAME_LVAR( lvar ), 0L,
            "you can 'return;' after assigning a value" );
    }
    return val;
}


/****************************************************************************
**
*F  ExecAssLVar(<stat>) . . . . . . . . . assign to            local variable
**
**  'ExecAssLVar' executes the local  variable assignment statement <stat> to
**  the local variable that is referenced in <stat>.
*/
libGAP_UInt            libGAP_ExecAssLVar (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]), rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}



/****************************************************************************
**
*F  ExecAssLVar01(<stat>) . . . . . . . . assign to first      local variable
*F  ExecAssLVar02(<stat>) . . . . . . . . assign to second     local variable
*F  ExecAssLVar03(<stat>) . . . . . . . . assign to third      local variable
*F  ExecAssLVar04(<stat>) . . . . . . . . assign to fourth     local variable
*F  ExecAssLVar05(<stat>) . . . . . . . . assign to fifth      local variable
*F  ExecAssLVar06(<stat>) . . . . . . . . assign to sixth      local variable
*F  ExecAssLVar07(<stat>) . . . . . . . . assign to seventh    local variable
*F  ExecAssLVar08(<stat>) . . . . . . . . assign to eigth      local variable
*F  ExecAssLVar09(<stat>) . . . . . . . . assign to ninth      local variable
*F  ExecAssLVar10(<stat>) . . . . . . . . assign to tenth      local variable
*F  ExecAssLVar11(<stat>) . . . . . . . . assign to eleventh   local variable
*F  ExecAssLVar12(<stat>) . . . . . . . . assign to twelveth   local variable
*F  ExecAssLVar13(<stat>) . . . . . . . . assign to thirteenth local variable
*F  ExecAssLVar14(<stat>) . . . . . . . . assign to fourteenth local variable
*F  ExecAssLVar15(<stat>) . . . . . . . . assign to fifteenth  local variable
*F  ExecAssLVar16(<stat>) . . . . . . . . assign to sixteenth  local variable
**
**  'ExecAssLVar<i>' executes  the local variable assignment statement <stat>
**  to the local variable <i>.
*/
libGAP_UInt            libGAP_ExecAssLVar01 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 1, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar02 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 2, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar03 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 3, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar04 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 4, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar05 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 5, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar06 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 6, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar07 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 7, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar08 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 8, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar09 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 9, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar10 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 10, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar11 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 11, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar12 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 12, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar13 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 13, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar14 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 14, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar15 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 15, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecAssLVar16 (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the local variable                    */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_LVAR( 16, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecUnbLVar (
    libGAP_Stat                stat )
{
    /* unbind the local variable                                           */
    libGAP_ASS_LVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]), (libGAP_Obj)0 );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  EvalRefLVar(<lvar>) . . . . . . . . .  value of            local variable
*F  EvalRefLVar01(<lvar>) . . . . . . . .  value of first      local variable
*F  EvalRefLVar02(<lvar>) . . . . . . . .  value of second     local variable
*F  EvalRefLVar03(<lvar>) . . . . . . . .  value of third      local variable
*F  EvalRefLVar04(<lvar>) . . . . . . . .  value of fourth     local variable
*F  EvalRefLVar05(<lvar>) . . . . . . . .  value of fifth      local variable
*F  EvalRefLVar06(<lvar>) . . . . . . . .  value of sixth      local variable
*F  EvalRefLVar07(<lvar>) . . . . . . . .  value of seventh    local variable
*F  EvalRefLVar08(<lvar>) . . . . . . . .  value of eigth      local variable
*F  EvalRefLVar09(<lvar>) . . . . . . . .  value of ninth      local variable
*F  EvalRefLVar10(<lvar>) . . . . . . . .  value of tenth      local variable
*F  EvalRefLVar11(<lvar>) . . . . . . . .  value of eleventh   local variable
*F  EvalRefLVar12(<lvar>) . . . . . . . .  value of twelth     local variable
*F  EvalRefLVar13(<lvar>) . . . . . . . .  value of thirteenth local variable
*F  EvalRefLVar14(<lvar>) . . . . . . . .  value of fourteenth local variable
*F  EvalRefLVar15(<lvar>) . . . . . . . .  value of fifteenth  local variable
*F  EvalRefLVar16(<lvar>) . . . . . . . .  value of sixteenth  local variable
**
**  'EvalRefLVar' evaluates the local variable reference expression <expr> to
**  the local variable that is referenced in <expr>
**
**  'EvalRefLVar<i>' evaluates the local variable reference expression <expr>
**  to the local variable <i>.
*/
libGAP_Obj             libGAP_EvalRefLVar (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar01 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 1 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 1 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 1 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar02 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 2 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 2 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 2 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar03 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 3 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 3 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 3 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar04 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 4 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 4 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 4 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar05 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 5 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 5 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 5 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar06 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 6 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 6 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 6 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar07 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 7 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 7 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 7 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar08 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 8 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 8 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 8 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar09 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 9 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 9 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 9 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar10 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 10 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 10 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 10 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar11 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 11 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 11 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 11 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar12 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 12 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 12 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 12 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar13 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 13 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 13 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 13 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar14 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 14 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 14 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 14 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar15 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 15 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 15 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 15 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalRefLVar16 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the local variable                       */
    if ( (val = libGAP_OBJ_LVAR( 16 )) == 0 ) {
        while ( (val = libGAP_OBJ_LVAR( 16 )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_LVAR( 16 ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalIsbLVar (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get the value of the local variable                                 */
    val = libGAP_OBJ_LVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) );

    /* return the value                                                    */
    return (val != (libGAP_Obj)0 ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  PrintAssLVar(<stat>)  . . . . . . print an assignment to a local variable
**
**  'PrintAssLVar' prints the local variable assignment statement <stat>.
*/
void            libGAP_PrintAssLVar (
    libGAP_Stat                stat )
{
    libGAP_Pr( "%2>", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_LVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]) ), 0L );
    libGAP_Pr( "%< %>:= ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr( "%2<;", 0L, 0L );
}

void            libGAP_PrintUnbLVar (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_LVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]) ), 0L );
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintRefLVar(<expr>)  . . . . . . . print a reference to a local variable
**
**  'PrintRefLVar' prints the local variable reference expression <expr>.
*/
void            libGAP_PrintRefLVar (
    libGAP_Expr                expr )
{
    if ( libGAP_IS_REFLVAR(expr) )
        libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_LVAR( libGAP_LVAR_REFLVAR(expr) ), 0L );
    else
        libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_LVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L );
}

void            libGAP_PrintIsbLVar (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_LVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L );
    libGAP_Pr( ")", 0L, 0L );
}


/****************************************************************************
**
*F  ASS_HVAR(<hvar>,<val>)  . . . . . . . . . . . assign to a higher variable
*F  OBJ_HVAR(<hvar>)  . . . . . . . . . . . . . .  value of a higher variable
*F  NAME_HVAR(<hvar>) . . . . . . . . . . . . . . . name of a higher variable
**
**  'ASS_HVAR' assigns the value <val> to the higher variable <hvar>.
**
**  'OBJ_HVAR' returns the value of the higher variable <hvar>.
**
**  'NAME_HVAR' returns the name of the higher variable <hvar> as a C string.
*/
void            libGAP_ASS_HVAR (
    libGAP_UInt                hvar,
    libGAP_Obj                 val )
{
    libGAP_Bag                 currLVars;      /* old current local variables     */
    libGAP_UInt                i;              /* loop variable                   */

    /* walk up the environment chain to the correct values bag             */
    currLVars = libGAP_TLS(libGAP_CurrLVars);
    for ( i = 1; i <= (hvar >> 16); i++ ) {
        libGAP_SWITCH_TO_OLD_LVARS( libGAP_ENVI_FUNC( libGAP_CURR_FUNC ) );
    }

    /* assign the value                                                    */
    libGAP_ASS_LVAR( hvar & 0xFFFF, val );
    /* CHANGED_BAG( TLS(CurrLVars) ); is done in the switch below               */

    /* switch back to current local variables bag                          */
    libGAP_SWITCH_TO_OLD_LVARS( currLVars );
}

libGAP_Obj             libGAP_OBJ_HVAR (
    libGAP_UInt                hvar )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Bag                 currLVars;      /* old current local variables     */
    libGAP_UInt                i;              /* loop variable                   */

    /* walk up the environment chain to the correct values bag             */
    currLVars = libGAP_TLS(libGAP_CurrLVars);
    for ( i = 1; i <= (hvar >> 16); i++ ) {
        libGAP_SWITCH_TO_OLD_LVARS( libGAP_ENVI_FUNC( libGAP_CURR_FUNC ) );
    }

    /* get the value                                                       */
    val = libGAP_OBJ_LVAR( hvar & 0xFFFF );

    /* switch back to current local variables bag                          */
    libGAP_SWITCH_TO_OLD_LVARS( currLVars );

    /* return the value                                                    */
    return val;
}

libGAP_Char *          libGAP_NAME_HVAR (
    libGAP_UInt                hvar )
{
    libGAP_Char *              name;           /* name, result                    */
    libGAP_Bag                 currLVars;      /* old current local variables     */
    libGAP_UInt                i;              /* loop variable                   */

    /* walk up the environment chain to the correct values bag             */
    currLVars = libGAP_TLS(libGAP_CurrLVars);
    for ( i = 1; i <= (hvar >> 16); i++ ) {
        libGAP_SWITCH_TO_OLD_LVARS( libGAP_ENVI_FUNC( libGAP_CURR_FUNC ) );
    }

    /* get the name                                                        */
    name = libGAP_NAME_LVAR( hvar & 0xFFFF );

    /* switch back to current local variables bag                          */
    libGAP_SWITCH_TO_OLD_LVARS( currLVars );

    /* return the name                                                     */
    return name;
}


/****************************************************************************
**
*F  ExecAssHVar(<stat>) . . . . . . . . . . . . . . assign to higher variable
**
**  'ExecAssHVar' executes the higher variable assignment statement <stat> to
**  the higher variable that is referenced in <stat>.
*/
libGAP_UInt            libGAP_ExecAssHVar (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the higher variable                   */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_ASS_HVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]), rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecUnbHVar (
    libGAP_Stat                stat )
{
    /* unbind the higher variable                                          */
    libGAP_ASS_HVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]), 0 );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  EvalRefHVar(<expr>) . . . . . . . . . . . . . .  value of higher variable
**
**  'EvalRefLVarXX' evaluates the higher variable reference expression <expr>
**  to the higher variable that is referenced in <expr>.
*/
libGAP_Obj             libGAP_EvalRefHVar (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the higher variable                      */
    if ( (val = libGAP_OBJ_HVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0 ) {
        while ( (val = libGAP_OBJ_HVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NAME_HVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalIsbHVar (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get the value of the higher variable                                */
    val = libGAP_OBJ_HVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) );

    /* return the value                                                    */
    return (val != (libGAP_Obj)0 ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  PrintAssHVar(<stat>)  . . . . . . . . print assignment to higher variable
**
**  'PrintAssHVar' prints the higher variable assignment statement <stat>.
*/
void            libGAP_PrintAssHVar (
    libGAP_Stat                stat )
{
    libGAP_Pr( "%2>", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_HVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]) ), 0L );
    libGAP_Pr( "%< %>:= ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr( "%2<;", 0L, 0L );
}

void            libGAP_PrintUnbHVar (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_HVAR( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]) ), 0L );
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintRefHVar(<expr>) . . . . . . . . . print reference to higher variable
**
**  'PrintRefHVar' prints the higher variable reference expression <expr>.
*/
void            libGAP_PrintRefHVar (
    libGAP_Expr                expr )
{
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_HVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L );
}

void            libGAP_PrintIsbHVar (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_HVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L );
    libGAP_Pr( ")", 0L, 0L );
}


/****************************************************************************
**
*F  ExecAssGVar(<stat>) . . . . . . . . . . . . . assign to a global variable
**
**  'ExecAssGVar' executes the global variable assignment statement <stat> to
**  the global variable that is referenced in <stat>.
*/
libGAP_UInt            libGAP_ExecAssGVar (
    libGAP_Stat                stat )
{
    libGAP_Obj                 rhs;            /* value of right hand side        */

    /* assign the right hand side to the global variable                   */
    libGAP_SET_BRK_CURR_STAT( stat );
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    libGAP_AssGVar( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]), rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecUnbGVar (
    libGAP_Stat                stat )
{
    /* unbind the global variable                                          */
    libGAP_AssGVar( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]), (libGAP_Obj)0 );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  EvalRefGVar(<expr>) . . . . . . . . . . . . . value of a globale variable
**
**  'EvalRefGVar' evaluates the  global variable reference expression  <expr>
**  to the global variable that is referenced in <expr>.
*/
libGAP_Obj             libGAP_EvalRefGVar (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get and check the value of the global variable                      */
    if ( (val = libGAP_VAL_GVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0
      && (val = libGAP_ValAutoGVar( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0 ) {
        while ( (val = libGAP_VAL_GVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0
             && (val = libGAP_ValAutoGVar( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) )) == 0 ) {
            libGAP_ErrorReturnVoid(
                "Variable: '%s' must have an assigned value",
                (libGAP_Int)libGAP_NameGVar( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* return the value                                                    */
    return val;
}

libGAP_Obj             libGAP_EvalIsbGVar (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* get the value of the global variable                                */
    val = libGAP_VAL_GVAR( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) );

    /* return the value                                                    */
    return (val != (libGAP_Obj)0 ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  PrintAssGVar(<stat>)  . . . . .  print an assignment to a global variable
**
**  'PrVarAss' prints the global variable assignment statement <stat>.
*/
void            libGAP_PrintAssGVar (
    libGAP_Stat                stat )
{
    libGAP_Pr( "%2>", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NameGVar( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]) ), 0L );
    libGAP_Pr( "%< %>:= ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr( "%2<;", 0L, 0L );
}

void            libGAP_PrintUnbGVar (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NameGVar( (libGAP_UInt)(libGAP_ADDR_STAT(stat)[0]) ), 0L );
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintRefGVar(<expr>)  . . . . . .  print a reference to a global variable
**
**  'PrintRefGVar' prints the global variable reference expression <expr>.
*/
void            libGAP_PrintRefGVar (
    libGAP_Expr                expr )
{
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NameGVar( (libGAP_UInt)(libGAP_ADDR_STAT(expr)[0]) ), 0L );
}

void            libGAP_PrintIsbGVar (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr( "%I", (libGAP_Int)libGAP_NameGVar( (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[0]) ), 0L );
    libGAP_Pr( ")", 0L, 0L );
}


/****************************************************************************
**
*F  ExecAssList(<ass>)  . . . . . . . . . . .  assign to an element of a list
**
**  'ExexAssList'  executes the list  assignment statement <stat> of the form
**  '<list>[<position>] := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAssList (
    libGAP_Expr                stat )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Int                 p;              /* position, as C integer          */
    libGAP_Obj                 rhs;            /* right hand side, right operand  */

    /* evaluate the list (checking is done by 'ASS_LIST')                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the position                                               */
    pos = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    if (libGAP_IS_POS_INTOBJ(pos)) {
        p = libGAP_INT_INTOBJ(pos);

        /* special case for plain list                                     */
        if ( libGAP_TNUM_OBJ(list) == libGAP_T_PLIST ) {
            if ( libGAP_LEN_PLIST(list) < p ) {
                libGAP_GROW_PLIST( list, p );
                libGAP_SET_LEN_PLIST( list, p );
            }
            libGAP_SET_ELM_PLIST( list, p, rhs );
            libGAP_CHANGED_BAG( list );
        }

        /* generic case                                                    */
        else {
            libGAP_ASS_LIST( list, p, rhs );
        }
    } else {
        libGAP_ASSB_LIST(list, pos, rhs);
    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}
/****************************************************************************
**
*F  ExecAss2List(<ass>)  . . . . . . . . . . .  assign to an element of a list
**
**  'ExexAss2List'  executes the list  assignment statement <stat> of the form
**  '<list>[<position>,<position>] := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAss2List (
    libGAP_Expr                stat )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos1;            /* position, left operand          */
    libGAP_Obj                 pos2;            /* position, left operand          */
    libGAP_Obj                 rhs;            /* right hand side, right operand  */

    /* evaluate the list (checking is done by 'ASS_LIST')                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the position                                               */
    pos1 = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    pos2 = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[3] );

    libGAP_ASS2_LIST( list, pos1, pos2, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}
/****************************************************************************
**
*F  ExecAssXList(<ass>)  . . . . . . . . . . .  assign to an element of a list
**
**  'ExexAssXList'  executes the list  assignment statement <stat> of the form
**  '<list>[<position>,<position>,<position>[,<position>]*] := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAssXList (
    libGAP_Expr                stat )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Obj                 rhs;            /* right hand side, right operand  */
    libGAP_Obj ixs;
    libGAP_Int i;
    libGAP_Int narg;

    /* evaluate the list (checking is done by 'ASS_LIST')                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    narg = libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat) - 2;
    ixs = libGAP_NEW_PLIST(libGAP_T_PLIST,narg);

    for (i = 1; i <= narg; i++) {
      /* evaluate the position                                               */
      pos = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[i] );
      libGAP_SET_ELM_PLIST(ixs,i,pos);
      libGAP_CHANGED_BAG(ixs);
    }
    libGAP_SET_LEN_PLIST(ixs,narg);

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    libGAP_ASSB_LIST(list, ixs, rhs);

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecAsssList(<stat>) . . . . . . . . assign to several elements of a list
**
**  'ExecAsssList' executes the list assignment statement  <stat> of the form
**  '<list>{<positions>} := <rhss>;'.
*/
libGAP_UInt            libGAP_ExecAsssList (
    libGAP_Expr                stat )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 poss;           /* positions, left operand         */
    libGAP_Obj                 rhss;           /* right hand sides, right operand */

    /* evaluate the list (checking is done by 'ASSS_LIST')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate and check the positions                                    */
    poss = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    while ( ! libGAP_IS_POSS_LIST( poss ) ) {
        poss = libGAP_ErrorReturnObj(
    "List Assignment: <positions> must be a dense list of positive integers",
            0L, 0L,
        "you can replace <positions> via 'return <positions>;'" );
    }

    /* evaluate and check right hand sides                                 */
    rhss = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );
    while ( ! libGAP_IS_DENSE_LIST( rhss )
         || libGAP_LEN_LIST( poss ) != libGAP_LEN_LIST( rhss ) ) {
        if ( ! libGAP_IS_DENSE_LIST( rhss ) ) {
            rhss = libGAP_ErrorReturnObj(
                "List Assignment: <rhss> must be a dense list (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(rhss), 0L,
                "you can replace <rhss> via 'return <rhss>;'" );
        }
        else /* if ( LEN_LIST( poss ) != LEN_LIST( rhss ) ) */ {
            rhss = libGAP_ErrorReturnObj(
     "List Assignment: <rhss> must be a list with the same length as <positions> (%d)",
                (libGAP_Int)libGAP_LEN_LIST( poss ), 0L,
                "you can replace <rhss> via 'return <rhss>;'" );
        }
    }

    /* assign the right hand sides to several elements of the list         */
    libGAP_ASSS_LIST( list, poss, rhss );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecAssListLevel(<stat>) . . . . . .  assign to elements of several lists
**
**  'ExecAssListLevel' executes the  list assignment statement  <stat> of the
**  form '<list>...{<positions>}...[<position>] :=  <rhss>;', where there may
**  actually be    several '{<positions>}'  selections  between  <list>   and
**  '[<position>]'.   The number of       those   is called    the     level.
**  'ExecAssListLevel' goes  that deep into  the left operand  and <rhss> and
**  assigns the  values from <rhss> to each  of those lists.  For example, if
**  the level is 1, the left operand must be a list  of lists, <rhss> must be
**  a  list, and 'ExecAssListLevel' assigns the  element '<rhss>[<i>]' to the
**  list '<list>[<i>]' at <position>.
*/
libGAP_UInt            libGAP_ExecAssListLevel (
    libGAP_Expr                stat )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Obj                 rhss;           /* right hand sides, right operand */
    libGAP_Int                 level;          /* level                           */
    libGAP_Int narg,i;
    libGAP_Obj ixs;

    /* evaluate lists (if this works, then <lists> is nested <level> deep, */
    /* checking it is nested <level>+1 deep is done by 'AssListLevel')     */
    libGAP_SET_BRK_CURR_STAT( stat );
    lists = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );
    narg = libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat) -3;
    ixs = libGAP_NEW_PLIST(libGAP_T_PLIST, narg);
    for (i = 1; i <= narg; i++) {
      pos = libGAP_EVAL_EXPR(libGAP_ADDR_STAT(stat)[i]);
      libGAP_SET_ELM_PLIST(ixs,i,pos);
      libGAP_CHANGED_BAG(ixs);
    }
    libGAP_SET_LEN_PLIST(ixs, narg);

    /* evaluate right hand sides (checking is done by 'AssListLevel')      */
    rhss = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[narg+1] );
      
    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_STAT(stat)[narg+2]);

    /* assign the right hand sides to the elements of several lists        */
    libGAP_AssListLevel( lists, ixs, rhss, level );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecAsssListLevel(<stat>) . . assign to several elements of several lists
**
**  'ExecAsssListLevel' executes the list  assignment statement <stat> of the
**  form '<list>...{<positions>}...{<positions>} := <rhss>;', where there may
**  actually be   several  '{<positions>}'  selections  between   <list>  and
**  '{<positions>}'.   The  number   of    those   is  called   the    level.
**  'ExecAsssListLevel' goes  that deep into the left  operand and <rhss> and
**  assigns the sublists from <rhss> to each of those lists.  For example, if
**  the level is 1, the left operand must be a  list of lists, <rhss> must be
**  a list, and 'ExecAsssListLevel' assigns the elements '<rhss>[<i>]' to the
**  list '<list>[<i>]' at the positions <positions>.
*/
libGAP_UInt            libGAP_ExecAsssListLevel (
    libGAP_Expr                stat )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 poss;           /* position, left operand          */
    libGAP_Obj                 rhss;           /* right hand sides, right operand */
    libGAP_Int                 level;          /* level                           */

    /* evaluate lists (if this works, then <lists> is nested <level> deep, */
    /* checking it is nested <level>+1 deep is done by 'AsssListLevel')    */
    libGAP_SET_BRK_CURR_STAT( stat );
    lists = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate and check the positions                                    */
    poss = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(stat)[1] );
    while ( ! libGAP_IS_POSS_LIST( poss ) ) {
        poss = libGAP_ErrorReturnObj(
    "List Assignment: <positions> must be a dense list of positive integers",
            0L, 0L,
        "you can replace <positions> via 'return <positions>;'" );
    }

    /* evaluate right hand sides (checking is done by 'AsssListLevel')     */
    rhss = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_STAT(stat)[3]);

    /* assign the right hand sides to several elements of several lists    */
    libGAP_AsssListLevel( lists, poss, rhss, level );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecUnbList(<ass>)  . . . . . . . . . . . . . unbind an element of a list
**
**  'ExexUnbList'  executes the list   unbind  statement <stat> of the   form
**  'Unbind( <list>[<position>] );'.
*/
libGAP_UInt            libGAP_ExecUnbList (
    libGAP_Expr                stat )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Obj ixs;
    libGAP_Int narg;
    libGAP_Int i;

    /* evaluate the list (checking is done by 'LEN_LIST')                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );
    narg = libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat) - 1;
    if (narg == 1) {
      pos = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
      /* unbind the element                                                  */
      if (libGAP_IS_POS_INTOBJ(pos)) {
        libGAP_UNB_LIST( list, libGAP_INT_INTOBJ(pos) );
      } else {
        libGAP_UNBB_LIST( list, pos );
      }
    } else {
      ixs = libGAP_NEW_PLIST(libGAP_T_PLIST, narg);
      for (i = 1; i <= narg; i++) {
	/* evaluate the position                                               */
	pos = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[i] );
	libGAP_SET_ELM_PLIST(ixs,i,pos);
	libGAP_CHANGED_BAG(ixs);
      }
      libGAP_SET_LEN_PLIST(ixs, narg);
      libGAP_UNBB_LIST(list, ixs);
    }
    

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  EvalElmList(<expr>) . . . . . . . . . . . . . select an element of a list
**
**  'EvalElmList' evaluates the list  element expression  <expr> of the  form
**  '<list>[<position>]'.
*/
libGAP_Obj             libGAP_EvalElmList (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Int                 p;              /* position, as C integer          */

    /* evaluate the list (checking is done by 'ELM_LIST')                  */
    list = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate and check the position                                     */
    pos = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] );

    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */

    if (libGAP_IS_POS_INTOBJ(pos)) {
        p = libGAP_INT_INTOBJ( pos );

        /* special case for plain lists (use generic code to signal errors) */
        if ( libGAP_IS_PLIST( list ) ) {
            if ( libGAP_LEN_PLIST(list) < p ) {
                return libGAP_ELM_LIST( list, p );
            }
            elm = libGAP_ELM_PLIST( list, p );
            if ( elm == 0 ) {
                return libGAP_ELM_LIST( list, p );
            }
        }
        /* generic case                                                    */
        else {
            elm = libGAP_ELM_LIST( list, p );
        }
    } else {
        elm = libGAP_ELMB_LIST(list, pos);
    }

    /* return the element                                                  */
    return elm;
}

/****************************************************************************
**
*F  EvalElm2List(<expr>) . . . . . . . . . . . . select an element of a list
**
**  'EvalElm2List' evaluates the list  element expression  <expr> of the  form
**  '<list>[<pos1>,<pos2>]'.
*/
libGAP_Obj             libGAP_EvalElm2List (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos1;            /* position, right operand         */
    libGAP_Obj                 pos2;            /* position, right operand         */

    /* evaluate the list (checking is done by 'ELM2_LIST')                  */
    list = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate and check the positions                                     */
    pos1 = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] ); 
    pos2 = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[2] ); 
   
    elm = libGAP_ELM2_LIST(list, pos1, pos2);


    /* return the element                                                  */
    return elm;
}

/****************************************************************************
**
*F  EvalElm2List(<expr>) . . . . . . . . . . . . select an element of a list
**
**  'EvalElm2List' evaluates the list  element expression  <expr> of the  form
**  '<list>[<pos1>,<pos2>,<pos3>,....]'.
*/
libGAP_Obj             libGAP_EvalElmXList (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Obj ixs;
    libGAP_Int narg;
    libGAP_Int i;
     

    /* evaluate the list (checking is done by 'ELM2_LIST')                  */
    list = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate and check the positions                                     */
    narg = libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr) -1;
    ixs = libGAP_NEW_PLIST(libGAP_T_PLIST,narg);
    for (i = 1; i <= narg; i++) {
      pos = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[i] );
      libGAP_SET_ELM_PLIST(ixs,i,pos);
      libGAP_CHANGED_BAG(ixs);
    }
    libGAP_SET_LEN_PLIST(ixs,narg);
   
    elm = libGAP_ELMB_LIST(list,ixs);

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  EvalElmsList(<expr>)  . . . . . . . . . select several elements of a list
**
**  'EvalElmsList' evaluates the  list element expression  <expr> of the form
**  '<list>{<positions>}'.
*/
libGAP_Obj             libGAP_EvalElmsList (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elms;           /* elements, result                */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 poss;           /* positions, right operand        */

    /* evaluate the list (checking is done by 'ELMS_LIST')                 */
    list = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate and check the positions                                    */
    poss = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] );
    while ( ! libGAP_IS_POSS_LIST( poss ) ) {
        poss = libGAP_ErrorReturnObj(
      "List Elements: <positions> must be a dense list of positive integers",
            0L, 0L,
        "you can replace <positions> via 'return <positions>;'" );
    }

    /* select several elements from the list                               */
    elms = libGAP_ELMS_LIST( list, poss );

    /* return the elements                                                 */
    return elms;
}


/****************************************************************************
**
*F  EvalElmListLevel(<expr>)  . . . . . . .  select elements of several lists
**
**  'EvalElmListLevel' evaluates the  list element  expression <expr> of  the
**  form '<list>...{<positions>}...[<position>]', where there may actually be
**  several '{<positions>}' selections   between <list> and   '[<position>]'.
**  The  number of those is called   the level.  'EvalElmListLevel' goes that
**  deep  into the left operand  and  selects the  element at <position> from
**  each of those  lists.  For example,  if the level  is 1, the left operand
**  must be a  list of lists  and 'EvalElmListLevel'  selects the element  at
**  <position> from each of the lists and returns the list of those values.
*/
libGAP_Obj             libGAP_EvalElmListLevel (
    libGAP_Expr                expr )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Obj                 ixs;
    libGAP_Int                 level;          /* level                           */
    libGAP_Int narg;
    libGAP_Int i;

    /* evaluate lists (if this works, then <lists> is nested <level> deep, */
    /* checking it is nested <level>+1 deep is done by 'ElmListLevel')     */
    lists = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );
    narg = libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr) -2;
    ixs = libGAP_NEW_PLIST(libGAP_T_PLIST, narg);
    for (i = 1; i <= narg; i++) {
      pos = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[i]);
      libGAP_SET_ELM_PLIST(ixs, i, pos);
      libGAP_CHANGED_BAG(ixs);
    }
    libGAP_SET_LEN_PLIST(ixs, narg);
    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_EXPR(expr)[narg+1]);
    
    /* select the elements from several lists (store them in <lists>)      */
    libGAP_ElmListLevel( lists, ixs, level );

    /* return the elements                                                 */
    return lists;
}


/****************************************************************************
**
*F  EvalElmsListLevel(<expr>) . . .  select several elements of several lists
**
**  'EvalElmsListLevel' evaluates the  list element expression <expr>  of the
**  form '<list>...{<positions>}...{<positions>}',   where there may actually
**  be several '{<positions>}' selections between <list> and '{<positions>}'.
**  The  number of those is called  the level.  'EvalElmsListLevel' goes that
**  deep into  the left operand and selects  the elements at <positions> from
**  each of those lists.   For example, if the  level is 1, the left  operand
**  must be  a list of lists  and 'EvalElmsListLevel' selects the elements at
**  <positions>  from each   of the lists  and  returns   the  list  of those
**  sublists.
*/
libGAP_Obj             libGAP_EvalElmsListLevel (
    libGAP_Expr                expr )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 poss;           /* positions, right operand        */
    libGAP_Int                 level;          /* level                           */

    /* evaluate lists (if this works, then <lists> is nested <level> deep, */
    /* checking it is nested <level>+1 deep is done by 'ElmsListLevel')    */
    lists = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate and check the positions                                    */
    poss = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] );
    while ( ! libGAP_IS_POSS_LIST( poss ) ) {
        poss = libGAP_ErrorReturnObj(
      "List Elements: <positions> must be a dense list of positive integers",
            0L, 0L,
        "you can replace <positions> via 'return <positions>;'" );
    }

    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_EXPR(expr)[2]);

    /* select several elements from several lists (store them in <lists>)  */
    libGAP_ElmsListLevel( lists, poss, level );

    /* return the elements                                                 */
    return lists;
}


/****************************************************************************
**
*F  EvalIsbList(<expr>) . . . . . . . . test if an element of a list is bound
**
**  'EvalElmList'  evaluates the list  isbound expression  <expr> of the form
**  'IsBound( <list>[<position>] )'.
*/
libGAP_Obj             libGAP_EvalIsbList (
    libGAP_Expr                expr )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Obj ixs;
    libGAP_Int narg, i;

    /* evaluate the list (checking is done by 'ISB_LIST')                  */
    list = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );
    narg = libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr) -1;
    if (narg == 1) {
      /* evaluate and check the position                                     */
      pos = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] );
      
      if (libGAP_IS_POS_INTOBJ(pos))
        return libGAP_ISB_LIST( list, libGAP_INT_INTOBJ(pos) ) ? libGAP_True : libGAP_False;
      else
        return libGAP_ISBB_LIST(list, pos) ? libGAP_True : libGAP_False;
    } else {
      ixs = libGAP_NEW_PLIST(libGAP_T_PLIST, narg);
      for (i = 1; i <= narg; i++) {
	pos = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[i] );
	libGAP_SET_ELM_PLIST(ixs,i,pos);
	libGAP_CHANGED_BAG(ixs);
      }
      libGAP_SET_LEN_PLIST(ixs, narg);
      return libGAP_ISBB_LIST(list, ixs) ? libGAP_True : libGAP_False;
    }
	
}


/****************************************************************************
**
*F  PrintAssList(<stat>)  . . . . print an assignment to an element of a list
**
**  'PrintAssList' prints the list  assignment statement  <stat> of the  form
**  '<list>[<position>] := <rhs>;'.
**
**  Linebreaks are preferred before the ':='.
*/
void            libGAP_PrintAssList (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintAss2List (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr("%<, %>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[3] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintAssXList (
    libGAP_Stat                stat )
{
  libGAP_Int narg = libGAP_SIZE_STAT(stat)/sizeof(stat) - 2;
  libGAP_Int i;
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    for (i = 2; i <= narg; i++) {
      libGAP_Pr("%<, %>",0L,0L);
      libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[i] );
    }
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[narg + 1] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintUnbList (
    libGAP_Stat                stat )
{
  libGAP_Int narg = libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat) -1;
  libGAP_Int i;
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    for (i = 2; i <= narg; i++) {
      libGAP_Pr("%<, %>",0L,0L);
      libGAP_PrintExpr(libGAP_ADDR_STAT(stat)[i]);
    }
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintAsssList(<stat>) . print an assignment to several elements of a list
**
**  'PrintAsssList'  prints the list assignment  statement <stat> of the form
**  '<list>{<positions>} := <rhss>;'.
**
**  Linebreaks are preferred before the ':='.
*/
void            libGAP_PrintAsssList (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<{",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr("%<}",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%2<;",0L,0L);
}


/****************************************************************************
**
*F  PrintElmList(<expr>)  . . . . . print a selection of an element of a list
**
**  'PrintElmList'   prints the list element   expression  <expr> of the form
**  '<list>[<position>]'.
**
**  Linebreaks are preferred after the '['.
*/
void            libGAP_PrintElmList (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr("%<]",0L,0L);
}

void libGAP_PrintElm2List (
		     libGAP_Expr expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr("%<, %<",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[2] );
    libGAP_Pr("%<]",0L,0L);
}

void libGAP_PrintElmXList (
		     libGAP_Expr expr )
{
  libGAP_Int i;
  libGAP_Int narg = libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr) -1 ;
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    for (i = 2; i <= narg; i++) {
      libGAP_Pr("%<, %<",0L,0L);
      libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[2] );
    }
    libGAP_Pr("%<]",0L,0L);
}

void libGAP_PrintElmListLevel (
		     libGAP_Expr expr )
{
  libGAP_Int i;
  libGAP_Int narg = libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr) -2 ;
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    for (i = 2; i <= narg; i++) {
      libGAP_Pr("%<, %<",0L,0L);
      libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[2] );
    }
    libGAP_Pr("%<]",0L,0L);
}


void            libGAP_PrintIsbList (
    libGAP_Expr                expr )
{
  libGAP_Int narg = libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr) - 1;
  libGAP_Int i;
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<[",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    for (i = 2; i <= narg; i++) {
      libGAP_Pr("%<, %>", 0L, 0L);
      libGAP_PrintExpr(libGAP_ADDR_EXPR(expr)[i] );
    }
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr( " )", 0L, 0L );
}


/****************************************************************************
**
*F  PrintElmsList(<expr>) . . print a selection of several elements of a list
**
**  'PrElmsList'  prints the list  elements  expression  <expr> of the   form
**  '<list>{<positions>}'.
**
**  Linebreaks are preferred after the '{'.
*/
void            libGAP_PrintElmsList (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<{",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr("%<}",0L,0L);
}


/****************************************************************************
**
*F  ExecAssRecName(<stat>)  . . . . . . . .  assign to an element of a record
**
**  'ExecAssRecName' executes the record  assignment statement <stat>  of the
**  form '<record>.<name> := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAssRecName (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* evaluate the record (checking is done by 'ASS_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    /* assign the right hand side to the element of the record             */
    libGAP_ASS_REC( record, rnam, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecAssRecExpr(<stat>)  . . . . . . . .  assign to an element of a record
**
**  'ExecAssRecExpr'  executes the record assignment  statement <stat> of the
**  form '<record>.(<name>) := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAssRecExpr (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* evaluate the record (checking is done by 'ASS_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] ) );

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    /* assign the right hand side to the element of the record             */
    libGAP_ASS_REC( record, rnam, rhs );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecUnbRecName(<stat>)  . . . . . . . . . . unbind an element of a record
**
**  'ExecAssRecName' executes the record  unbind statement <stat> of the form
**  'Unbind( <record>.<name> );'.
*/
libGAP_UInt            libGAP_ExecUnbRecName (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* evaluate the record (checking is done by 'UNB_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);

    /* unbind the element of the record                                    */
    libGAP_UNB_REC( record, rnam );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecUnbRecExpr(<stat>)  . . . . . . . . . . unbind an element of a record
**
**  'ExecAssRecExpr' executes the record  unbind statement <stat> of the form
**  'Unbind( <record>.(<name>) );'.
*/
libGAP_UInt            libGAP_ExecUnbRecExpr (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* evaluate the record (checking is done by 'UNB_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] ) );

    /* unbind the element of the record                                    */
    libGAP_UNB_REC( record, rnam );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  EvalElmRecName(<expr>)  . . . . . . . . . . . . . select a record element
**
**  'EvalElmRecName' evaluates the   record element expression  <expr> of the
**  form '<record>.<name>'.
*/
libGAP_Obj             libGAP_EvalElmRecName (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ELM_REC')                 */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);

    /* select the element of the record                                    */
    elm = libGAP_ELM_REC( record, rnam );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  EvalElmRecExpr(<expr>)  . . . . . . . . . . . . . select a record element
**
**  'EvalElmRecExpr'  evaluates the record   element expression <expr> of the
**  form '<record>.(<name>)'.
*/
libGAP_Obj             libGAP_EvalElmRecExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ELM_REC')                 */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] ) );

    /* select the element of the record                                    */
    elm = libGAP_ELM_REC( record, rnam );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  EvalIsbRecName(<expr>)  . . . . . . . . test if a record element is bound
**
**  'EvalElmRecName' evaluates the   record isbound expression  <expr> of the
**  form 'IsBound( <record>.<name> )'.
*/
libGAP_Obj             libGAP_EvalIsbRecName (
    libGAP_Expr                expr )
{
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ISB_REC')                 */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);

    /* return the result                                                   */
    return (libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  EvalIsbRecExpr(<expr>)  . . . . . . . . test if a record element is bound
**
**  'EvalIsbRecExpr' evaluates  the record isbound  expression  <expr> of the
**  form 'IsBound( <record>.(<name>) )'.
*/
libGAP_Obj             libGAP_EvalIsbRecExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ISB_REC')                 */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] ) );

    /* return the result                                                   */
    return (libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  PrintAssRecName(<stat>) . . print an assignment to an element of a record
**
**  'PrintAssRecName' prints the  record  assignment statement <stat>  of the
**  form '<record>.<name> := <rhs>;'.
*/
void            libGAP_PrintAssRecName (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_STAT(stat)[1])),0L);
    libGAP_Pr("%<",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintUnbRecName (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_STAT(stat)[1])),0L);
    libGAP_Pr("%<",0L,0L);
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintAssRecExpr(<stat>) . . print an assignment to an element of a record
**
**  'PrintAssRecExpr' prints the  record  assignment statement <stat>  of the
**  form '<record>.(<name>) := <rhs>;'.
*/
void            libGAP_PrintAssRecExpr (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr(")%<",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintUnbRecExpr (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr(")%<",0L,0L);
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintElmRecName(<expr>) . . . print a selection of an element of a record
**
**  'PrintElmRecName' prints the record element expression <expr> of the form
**  '<record>.<name>'.
*/
void            libGAP_PrintElmRecName (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1])),0L);
    libGAP_Pr("%<",0L,0L);
}

void            libGAP_PrintIsbRecName (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1])),0L);
    libGAP_Pr("%<",0L,0L);
    libGAP_Pr( " )", 0L, 0L );
}


/****************************************************************************
**
*F  PrintElmRecExpr(<expr>) . . . print a selection of an element of a record
**
**  'PrintElmRecExpr' prints the record element expression <expr> of the form
**  '<record>.(<name>)'.
*/
void            libGAP_PrintElmRecExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr(")%<",0L,0L);
}

void            libGAP_PrintIsbRecExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr(")%<",0L,0L);
    libGAP_Pr( " )", 0L, 0L );
}


/****************************************************************************
**
*F  ExecAssPosObj(<ass>)  . . . . . . . . . . .  assign to an element of a list
**
**  'ExexAssPosObj'  executes the list  assignment statement <stat> of the form
**  '<list>[<position>] := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAssPosObj (
    libGAP_Expr                stat )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Int                 p;              /* position, as a C integer        */
    libGAP_Obj                 rhs;            /* right hand side, right operand  */

    /* evaluate the list (checking is done by 'ASS_LIST')                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate and check the position                                     */
    pos = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    while ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        pos = libGAP_ErrorReturnObj(
         "PosObj Assignment: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L,
            "you can replace <position> via 'return <position>;'" );
    }
    p = libGAP_INT_INTOBJ(pos);

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    /* special case for plain list                                         */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        libGAP_WriteGuard(list);
        if ( libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1 < p ) {
            libGAP_ResizeBag( list, (p+1) * sizeof(libGAP_Obj) );
        }
        libGAP_SET_ELM_PLIST( list, p, rhs );
        libGAP_CHANGED_BAG( list );
    }
    /* generic case                                                        */
    else {
        libGAP_ASS_LIST( list, p, rhs );
    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecUnbPosObj(<ass>)  . . . . . . . . . . . . . unbind an element of a list
**
**  'ExexUnbPosObj'  executes the list   unbind  statement <stat> of the   form
**  'Unbind( <list>[<position>] );'.
*/
libGAP_UInt            libGAP_ExecUnbPosObj (
    libGAP_Expr                stat )
{
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Int                 p;              /* position, as a C integer        */

    /* evaluate the list (checking is done by 'LEN_LIST')                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate and check the position                                     */
    pos = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );
    while ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        pos = libGAP_ErrorReturnObj(
         "PosObj Assignment: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L,
            "you can replace <position> via 'return <position>;'" );
    }
    p = libGAP_INT_INTOBJ(pos);

    /* unbind the element                                                  */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        libGAP_WriteGuard(list);
        if ( p <= libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1 ) {
            libGAP_SET_ELM_PLIST( list, p, 0 );
        }
    }
    else {
        libGAP_UNB_LIST( list, p );
    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  EvalElmPosObj(<expr>) . . . . . . . . . . . . . select an element of a list
**
**  'EvalElmPosObj' evaluates the list  element expression  <expr> of the  form
**  '<list>[<position>]'.
*/
libGAP_Obj             libGAP_EvalElmPosObj (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Int                 p;              /* position, as C integer          */

    /* evaluate the list (checking is done by 'ELM_LIST')                  */
    list = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate and check the position                                     */
    pos = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] );
    while ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        pos = libGAP_ErrorReturnObj(
            "PosObj Element: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L,
            "you can replace <position> via 'return <position>;'" );
    }
    p = libGAP_INT_INTOBJ( pos );

    /* special case for plain lists (use generic code to signal errors)    */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        while ( libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1 < p ) {
            libGAP_ErrorReturnVoid(
                "PosObj Element: <PosObj>![%d] must have an assigned value",
                (libGAP_Int)p, 0L,
                "you can 'return;' after assigning a value" );
        }
        elm = libGAP_ELM_PLIST( list, p );
        while ( elm == 0 ) {
            libGAP_ErrorReturnVoid(
                "PosObj Element: <PosObj>![%d] must have an assigned value",
                (libGAP_Int)p, 0L,
                "you can 'return;' after assigning a value" );
        }
    }

    /* generic case                                                        */
    else {
        elm = libGAP_ELM_LIST( list, p );
    }

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  EvalIsbPosObj(<expr>) . . . . . . . . test if an element of a list is bound
**
**  'EvalElmPosObj'  evaluates the list  isbound expression  <expr> of the form
**  'IsBound( <list>[<position>] )'.
*/
libGAP_Obj             libGAP_EvalIsbPosObj (
    libGAP_Expr                expr )
{
    libGAP_Obj                 isb;            /* isbound, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Int                 p;              /* position, as C integer          */

    /* evaluate the list (checking is done by 'ISB_LIST')                  */
    list = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate and check the position                                     */
    pos = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] );
    while ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        pos = libGAP_ErrorReturnObj(
            "PosObj Element: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L,
            "you can replace <position> via 'return <position>;'" );
    }
    p = libGAP_INT_INTOBJ( pos );

    /* get the result                                                      */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        isb = (p <= libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1 && libGAP_ELM_PLIST(list,p) != 0 ?
               libGAP_True : libGAP_False);
    }
    else {
        isb = (libGAP_ISB_LIST( list, p ) ? libGAP_True : libGAP_False);
    }

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  PrintAssPosObj(<stat>)  . . . . print an assignment to an element of a list
**
**  'PrintAssPosObj' prints the list  assignment statement  <stat> of the  form
**  '<list>[<position>] := <rhs>;'.
**
**  Linebreaks are preferred before the ':='.
*/
void            libGAP_PrintAssPosObj (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<![",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintUnbPosObj (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<![",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintElmPosObj(<expr>)  . . . . . print a selection of an element of a list
**
**  'PrintElmPosObj'   prints the list element   expression  <expr> of the form
**  '<list>[<position>]'.
**
**  Linebreaks are preferred after the '['.
*/
void            libGAP_PrintElmPosObj (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<![",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr("%<]",0L,0L);
}

void            libGAP_PrintIsbPosObj (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<![",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr("%<]",0L,0L);
    libGAP_Pr( ")", 0L, 0L );
}


/****************************************************************************
**
*F  ExecAssComObjName(<stat>) . . . . . . . .  assign to an element of a record
**
**  'ExecAssComObjName' executes the  record assignment statement <stat> of the
**  form '<record>.<name> := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAssComObjName (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* evaluate the record (checking is done by 'ASS_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    /* assign the right hand side to the element of the record             */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        libGAP_AssPRec( record, rnam, rhs );
    }
    else {
        libGAP_ASS_REC( record, rnam, rhs );
    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecAssComObjExpr(<stat>) . . . . . . . .  assign to an element of a record
**
**  'ExecAssComObjExpr' executes the record assignment  statement <stat> of the
**  form '<record>.(<name>) := <rhs>;'.
*/
libGAP_UInt            libGAP_ExecAssComObjExpr (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* evaluate the record (checking is done by 'ASS_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] ) );

    /* evaluate the right hand side                                        */
    rhs = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[2] );

    /* assign the right hand side to the element of the record             */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        libGAP_AssPRec( record, rnam, rhs );
    }
    else {
        libGAP_ASS_REC( record, rnam, rhs );
    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecUnbComObjName(<stat>) . . . . . . . . . . unbind an element of a record
**
**  'ExecAssComObjName' executes the record unbind statement <stat> of the form
**  'Unbind( <record>.<name> );'.
*/
libGAP_UInt            libGAP_ExecUnbComObjName (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* evaluate the record (checking is done by 'UNB_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);

    /* unbind the element of the record                                    */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        libGAP_UnbPRec( record, rnam );
    }
    else {
        libGAP_UNB_REC( record, rnam );
    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecUnbComObjExpr(<stat>) . . . . . . . . . . unbind an element of a record
**
**  'ExecAssComObjExpr' executes the record unbind statement <stat> of the form
**  'Unbind( <record>.(<name>) );'.
*/
libGAP_UInt            libGAP_ExecUnbComObjExpr (
    libGAP_Stat                stat )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* evaluate the record (checking is done by 'UNB_REC')                 */
    libGAP_SET_BRK_CURR_STAT( stat );
    record = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] ) );

    /* unbind the element of the record                                    */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        libGAP_UnbPRec( record, rnam );
    }
    else {
        libGAP_UNB_REC( record, rnam );
    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  EvalElmComObjName(<expr>) . . . . . . . . . . . . . select a record element
**
**  'EvalElmComObjName' evaluates the  record element expression  <expr> of the
**  form '<record>.<name>'.
*/
libGAP_Obj             libGAP_EvalElmComObjName (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ELM_REC')                 */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);

    /* select the element of the record                                    */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        elm = libGAP_ElmPRec( record, rnam );
    }
    else {
        elm = libGAP_ELM_REC( record, rnam );
    }

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  EvalElmComObjExpr(<expr>) . . . . . . . . . . . . . select a record element
**
**  'EvalElmComObjExpr' evaluates the  record element expression  <expr> of the
**  form '<record>.(<name>)'.
*/
libGAP_Obj             libGAP_EvalElmComObjExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ELM_REC')                 */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] ) );

    /* select the element of the record                                    */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        elm = libGAP_ElmPRec( record, rnam );
    }
    else {
        elm = libGAP_ELM_REC( record, rnam );
    }

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  EvalIsbComObjName(<expr>) . . . . . . . . test if a record element is bound
**
**  'EvalElmComObjName' evaluates  the record isbound  expression <expr> of the
**  form 'IsBound( <record>.<name> )'.
*/
libGAP_Obj             libGAP_EvalIsbComObjName (
    libGAP_Expr                expr )
{
    libGAP_Obj                 isb;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ISB_REC')                 */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);

    /* select the element of the record                                    */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        isb = (libGAP_IsbPRec( record, rnam ) ? libGAP_True : libGAP_False);
    }
    else {
        isb = (libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False);
    }

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  EvalIsbComObjExpr(<expr>) . . . . . . . . test if a record element is bound
**
**  'EvalIsbComObjExpr'  evaluates the record isbound  expression <expr> of the
**  form 'IsBound( <record>.(<name>) )'.
*/
libGAP_Obj             libGAP_EvalIsbComObjExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 isb;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* evaluate the record (checking is done by 'ISB_REC')                */
    record = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );

    /* evaluate the name and convert it to a record name                   */
    rnam = libGAP_RNamObj( libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] ) );

    /* select the element of the record                                    */
    if ( libGAP_TNUM_OBJ(record) == libGAP_T_COMOBJ ) {
        isb = (libGAP_IsbPRec( record, rnam ) ? libGAP_True : libGAP_False);
    }
    else {
        isb = (libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False);
    }

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  PrintAssComObjName(<stat>)  . print an assignment to an element of a record
**
**  'PrintAssComObjName' prints the  record assignment statement <stat>  of the
**  form '<record>.<name> := <rhs>;'.
*/
void            libGAP_PrintAssComObjName (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<!.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_STAT(stat)[1])),0L);
    libGAP_Pr("%<",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintUnbComObjName (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<!.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_STAT(stat)[1])),0L);
    libGAP_Pr("%<",0L,0L);
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintAssComObjExpr(<stat>)  . print an assignment to an element of a record
**
**  'PrintAssComObjExpr' prints the  record assignment statement <stat>  of the
**  form '<record>.(<name>) := <rhs>;'.
*/
void            libGAP_PrintAssComObjExpr (
    libGAP_Stat                stat )
{
    libGAP_Pr("%4>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<!.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr(")%<",0L,0L);
    libGAP_Pr("%< %>:= ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Pr("%2<;",0L,0L);
}

void            libGAP_PrintUnbComObjExpr (
    libGAP_Stat                stat )
{
    libGAP_Pr( "Unbind( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<!.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr(")%<",0L,0L);
    libGAP_Pr( " );", 0L, 0L );
}


/****************************************************************************
**
*F  PrintElmComObjName(<expr>)  . . print a selection of an element of a record
**
**  'PrintElmComObjName' prints the  record  element expression <expr> of   the
**  form '<record>.<name>'.
*/
void            libGAP_PrintElmComObjName (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<!.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1])),0L);
    libGAP_Pr("%<",0L,0L);
}

void            libGAP_PrintIsbComObjName (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<!.",0L,0L);
    libGAP_Pr("%I",(libGAP_Int)libGAP_NAME_RNAM((libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1])),0L);
    libGAP_Pr("%<",0L,0L);
    libGAP_Pr( " )", 0L, 0L );
}


/****************************************************************************
**
*F  PrintElmComObjExpr(<expr>)  . . print a selection of an element of a record
**
**  'PrintElmComObjExpr' prints the record   element expression <expr>  of  the
**  form '<record>.(<name>)'.
*/
void            libGAP_PrintElmComObjExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<!.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr(")%<",0L,0L);
}

void            libGAP_PrintIsbComObjExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr( "IsBound( ", 0L, 0L );
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<!.(",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Pr(")%<",0L,0L);
    libGAP_Pr( " )", 0L, 0L );
}


/****************************************************************************
**
*F  FuncGetCurrentLVars
*F  FuncGetBottomLVars
*F  FuncParentLVars
*F  FuncContentsLVars
**
**  Provide access to local variable bags at GAP level. Mainly for use in
**  error handling.
**
*/


libGAP_Obj libGAP_FuncGetCurrentLVars( libGAP_Obj self )
{
  return libGAP_TLS(libGAP_CurrLVars);
}

libGAP_Obj libGAP_FuncGetBottomLVars( libGAP_Obj self )
{
  return libGAP_TLS(libGAP_BottomLVars);
}

libGAP_Obj libGAP_FuncParentLVars( libGAP_Obj self, libGAP_Obj lvars )
{
  if (lvars == libGAP_TLS(libGAP_BottomLVars))
    return libGAP_Fail;
  return libGAP_ADDR_OBJ(lvars)[2];
}

libGAP_Obj libGAP_FuncContentsLVars (libGAP_Obj self, libGAP_Obj lvars )
{
  libGAP_Obj contents = libGAP_NEW_PREC(0);
  libGAP_Obj func = libGAP_PTR_BAG(lvars)[0];
  libGAP_Obj nams = libGAP_NAMS_FUNC(func);
  libGAP_UInt len = (libGAP_SIZE_BAG(lvars) - 2*sizeof(libGAP_Obj) - sizeof(libGAP_UInt))/sizeof(libGAP_Obj);
  libGAP_Obj values = libGAP_NEW_PLIST(libGAP_T_PLIST+libGAP_IMMUTABLE, len);
  if (lvars == libGAP_TLS(libGAP_BottomLVars))
    return libGAP_False;
  libGAP_AssPRec(contents, libGAP_RNamName("func"), func);
  libGAP_AssPRec(contents,libGAP_RNamName("names"), nams);
  memcpy((void *)(1+libGAP_ADDR_OBJ(values)), (void *)(3+libGAP_ADDR_OBJ(lvars)), len*sizeof(libGAP_Obj));
  while (libGAP_ELM_PLIST(values, len) == 0)
    len--;
  libGAP_SET_LEN_PLIST(values, len);
  libGAP_AssPRec(contents, libGAP_RNamName("values"), values);
  if (libGAP_ENVI_FUNC(func) != libGAP_TLS(libGAP_BottomLVars))
    libGAP_AssPRec(contents, libGAP_RNamName("higher"), libGAP_ENVI_FUNC(func));
  return contents;
}

/****************************************************************************
**
*F  VarsBeforeCollectBags() . . . . . . . . actions before garbage collection
*F  VarsAfterCollectBags()  . . . . . . . .  actions after garbage collection
*/
void libGAP_VarsBeforeCollectBags ( void )
{
  if (libGAP_TLS(libGAP_CurrLVars))
    libGAP_CHANGED_BAG( libGAP_TLS(libGAP_CurrLVars) );
}

void libGAP_VarsAfterCollectBags ( void )
{
  if (libGAP_TLS(libGAP_CurrLVars))
    {
      libGAP_TLS(libGAP_PtrLVars) = libGAP_PTR_BAG( libGAP_TLS(libGAP_CurrLVars) );
      libGAP_TLS(libGAP_PtrBody)  = (libGAP_Stat*)libGAP_PTR_BAG( libGAP_BODY_FUNC( libGAP_CURR_FUNC ) );
    }
  if (libGAP_ValGVars)
    libGAP_PtrGVars = libGAP_PTR_BAG( libGAP_ValGVars );
}

/****************************************************************************
**
*F  SaveLVars ( <lvars> )
**
*/

void libGAP_SaveLVars( libGAP_Obj lvars )
{
  libGAP_UInt len,i;
  libGAP_Obj *ptr;
  libGAP_SaveSubObj(libGAP_ADDR_OBJ(lvars)[0]);
  libGAP_SaveUInt((libGAP_UInt)libGAP_ADDR_OBJ(lvars)[1]);
  libGAP_SaveSubObj(libGAP_ADDR_OBJ(lvars)[2]);
  len = (libGAP_SIZE_OBJ(lvars) - (2*sizeof(libGAP_Obj)+sizeof(libGAP_UInt)))/sizeof(libGAP_Obj);
  ptr = libGAP_ADDR_OBJ(lvars)+3;
  for (i = 0; i < len; i++)
    libGAP_SaveSubObj(*ptr++);
  return;
}

/****************************************************************************
**
*F  LoadLVars ( <lvars> )
**
*/

void libGAP_LoadLVars( libGAP_Obj lvars )
{
  libGAP_UInt len,i;
  libGAP_Obj *ptr;
  libGAP_ADDR_OBJ(lvars)[0] = libGAP_LoadSubObj();
  ((libGAP_UInt *)libGAP_ADDR_OBJ(lvars))[1] = libGAP_LoadUInt();
  libGAP_ADDR_OBJ(lvars)[2] = libGAP_LoadSubObj();
  len = (libGAP_SIZE_OBJ(lvars) - (2*sizeof(libGAP_Obj)+sizeof(libGAP_UInt)))/sizeof(libGAP_Obj);
  ptr = libGAP_ADDR_OBJ(lvars)+3;
  for (i = 0; i < len; i++)
    *ptr++ = libGAP_LoadSubObj();
  return;
}

libGAP_Obj libGAP_TYPE_LVARS;

libGAP_Obj libGAP_TypeLVars( libGAP_Obj lvars )
{
  return libGAP_TYPE_LVARS;
}

void libGAP_PrintLVars( libGAP_Obj lvars )
{
  libGAP_Pr("<lvars bag>", 0,0);
}

libGAP_Int libGAP_EqLVars (libGAP_Obj x, libGAP_Obj y)
{
  return (x == y);
}

libGAP_Int libGAP_EqLVarsX (libGAP_Obj x, libGAP_Obj y)
{
  return 0;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * Initialize Package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {
  { "GetCurrentLVars", 0, "",
    libGAP_FuncGetCurrentLVars, "src/vars.c: GetCurrentLVars"},

  { "GetBottomLVars", 0, "",
    libGAP_FuncGetBottomLVars, "src/vars.c: GetBottomLVars"},

  { "ParentLVars", 1, "lvars",
    libGAP_FuncParentLVars, "src/vars.c: ParentLVars"},

  { "ContentsLVars", 1, "lvars",
    libGAP_FuncContentsLVars, "src/vars.c: ContentsLVars"},

  { 0} };


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_TLS(libGAP_CurrLVars) = (libGAP_Bag) 0;

    /* make 'CurrLVars' known to Gasman                                    */
    libGAP_InitGlobalBag( &libGAP_CurrLVars,   "src/vars.c:CurrLVars"   );
    libGAP_InitGlobalBag( &libGAP_BottomLVars, "src/vars.c:BottomLVars" );

    /* install the marking functions for local variables bag               */
    libGAP_InfoBags[ libGAP_T_LVARS ].name = "values bag";
    libGAP_InitMarkFuncBags( libGAP_T_LVARS, libGAP_MarkAllSubBags );

    /* and the save restore functions */
    libGAP_SaveObjFuncs[ libGAP_T_LVARS ] = libGAP_SaveLVars;
    libGAP_LoadObjFuncs[ libGAP_T_LVARS ] = libGAP_LoadLVars;

    /* and a type */

    libGAP_TypeObjFuncs[ libGAP_T_LVARS ] = libGAP_TypeLVars;
    libGAP_PrintObjFuncs[ libGAP_T_LVARS ] = libGAP_PrintLVars;
    libGAP_EqFuncs[libGAP_T_LVARS][libGAP_T_LVARS] = libGAP_EqLVars;
    for (i = libGAP_FIRST_REAL_TNUM; i <= libGAP_LAST_REAL_TNUM; i++)
      {
        libGAP_EqFuncs[libGAP_T_LVARS][i] = libGAP_EqLVarsX;
        libGAP_EqFuncs[i][libGAP_T_LVARS] = libGAP_EqLVarsX;
      }


    /* install executors, evaluators, and printers for local variables     */
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR       , libGAP_ExecAssLVar);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_01    , libGAP_ExecAssLVar01);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_02    , libGAP_ExecAssLVar02);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_03    , libGAP_ExecAssLVar03);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_04    , libGAP_ExecAssLVar04);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_05    , libGAP_ExecAssLVar05);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_06    , libGAP_ExecAssLVar06);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_07    , libGAP_ExecAssLVar07);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_08    , libGAP_ExecAssLVar08);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_09    , libGAP_ExecAssLVar09);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_10    , libGAP_ExecAssLVar10);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_11    , libGAP_ExecAssLVar11);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_12    , libGAP_ExecAssLVar12);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_13    , libGAP_ExecAssLVar13);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_14    , libGAP_ExecAssLVar14);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_15    , libGAP_ExecAssLVar15);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LVAR_16    , libGAP_ExecAssLVar16);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_LVAR       , libGAP_ExecUnbLVar);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR       , libGAP_EvalRefLVar);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_01    , libGAP_EvalRefLVar01);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_02    , libGAP_EvalRefLVar02);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_03    , libGAP_EvalRefLVar03);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_04    , libGAP_EvalRefLVar04);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_05    , libGAP_EvalRefLVar05);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_06    , libGAP_EvalRefLVar06);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_07    , libGAP_EvalRefLVar07);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_08    , libGAP_EvalRefLVar08);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_09    , libGAP_EvalRefLVar09);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_10    , libGAP_EvalRefLVar10);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_11    , libGAP_EvalRefLVar11);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_12    , libGAP_EvalRefLVar12);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_13    , libGAP_EvalRefLVar13);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_14    , libGAP_EvalRefLVar14);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_15    , libGAP_EvalRefLVar15);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_LVAR_16    , libGAP_EvalRefLVar16);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_LVAR       , libGAP_EvalIsbLVar);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_LVAR       , libGAP_PrintAssLVar);

    for ( i = libGAP_T_ASS_LVAR_01; i <= libGAP_T_ASS_LVAR_16; i++ ) {
        libGAP_InstallPrintStatFunc( i , libGAP_PrintAssLVar);
    }

    libGAP_InstallPrintStatFunc( libGAP_T_UNB_LVAR       , libGAP_PrintUnbLVar);
    libGAP_InstallPrintExprFunc( libGAP_T_REFLVAR        , libGAP_PrintRefLVar);
    libGAP_InstallPrintExprFunc( libGAP_T_REF_LVAR       , libGAP_PrintRefLVar);

    for ( i = libGAP_T_REF_LVAR_01; i <= libGAP_T_REF_LVAR_16; i++ ) {
        libGAP_InstallPrintExprFunc( i , libGAP_PrintRefLVar);
    }

    libGAP_InstallPrintExprFunc( libGAP_T_ISB_LVAR       , libGAP_PrintIsbLVar);

    /* install executors, evaluators, and printers for higher variables    */
    libGAP_InstallExecStatFunc( libGAP_T_ASS_HVAR       , libGAP_ExecAssHVar);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_HVAR       , libGAP_ExecUnbHVar);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_HVAR       , libGAP_EvalRefHVar);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_HVAR       , libGAP_EvalIsbHVar);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_HVAR       , libGAP_PrintAssHVar);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_HVAR       , libGAP_PrintUnbHVar);
    libGAP_InstallPrintExprFunc( libGAP_T_REF_HVAR       , libGAP_PrintRefHVar);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_HVAR       , libGAP_PrintIsbHVar);

    /* install executors, evaluators, and printers for global variables    */
    libGAP_InstallExecStatFunc( libGAP_T_ASS_GVAR       , libGAP_ExecAssGVar);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_GVAR       , libGAP_ExecUnbGVar);
    libGAP_InstallEvalExprFunc( libGAP_T_REF_GVAR       , libGAP_EvalRefGVar);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_GVAR       , libGAP_EvalIsbGVar);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_GVAR       , libGAP_PrintAssGVar);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_GVAR       , libGAP_PrintUnbGVar);
    libGAP_InstallPrintExprFunc( libGAP_T_REF_GVAR       , libGAP_PrintRefGVar);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_GVAR       , libGAP_PrintIsbGVar);

    /* install executors, evaluators, and printers for list elements       */
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LIST       , libGAP_ExecAssList);
    libGAP_InstallExecStatFunc( libGAP_T_ASSS_LIST      , libGAP_ExecAsssList);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_LIST_LEV   , libGAP_ExecAssListLevel);
    libGAP_InstallExecStatFunc( libGAP_T_ASSS_LIST_LEV  , libGAP_ExecAsssListLevel);
    libGAP_InstallExecStatFunc( libGAP_T_ASS2_LIST  , libGAP_ExecAss2List);
    libGAP_InstallExecStatFunc( libGAP_T_ASSX_LIST  , libGAP_ExecAssXList);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS2_LIST  , libGAP_PrintAss2List);
    libGAP_InstallPrintStatFunc( libGAP_T_ASSX_LIST  , libGAP_PrintAssXList);
    
    libGAP_InstallExecStatFunc( libGAP_T_UNB_LIST       , libGAP_ExecUnbList);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM_LIST       , libGAP_EvalElmList);
    libGAP_InstallEvalExprFunc( libGAP_T_ELMS_LIST      , libGAP_EvalElmsList);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM_LIST_LEV   , libGAP_EvalElmListLevel);
    libGAP_InstallEvalExprFunc( libGAP_T_ELMS_LIST_LEV  , libGAP_EvalElmsListLevel);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_LIST       , libGAP_EvalIsbList);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM2_LIST      , libGAP_EvalElm2List);
    libGAP_InstallEvalExprFunc( libGAP_T_ELMX_LIST      , libGAP_EvalElmXList);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM2_LIST     , libGAP_PrintElm2List);
    libGAP_InstallPrintExprFunc( libGAP_T_ELMX_LIST     , libGAP_PrintElmXList);
    
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_LIST       , libGAP_PrintAssList);
    libGAP_InstallPrintStatFunc( libGAP_T_ASSS_LIST      , libGAP_PrintAsssList);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_LIST_LEV   , libGAP_PrintAssList);
    libGAP_InstallPrintStatFunc( libGAP_T_ASSS_LIST_LEV  , libGAP_PrintAsssList);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_LIST       , libGAP_PrintUnbList);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM_LIST       , libGAP_PrintElmList);
    libGAP_InstallPrintExprFunc( libGAP_T_ELMS_LIST      , libGAP_PrintElmsList);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM_LIST_LEV   , libGAP_PrintElmListLevel);
    libGAP_InstallPrintExprFunc( libGAP_T_ELMS_LIST_LEV  , libGAP_PrintElmsList);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_LIST       , libGAP_PrintIsbList);


    /* install executors, evaluators, and printers for record elements     */
    libGAP_InstallExecStatFunc( libGAP_T_ASS_REC_NAME   , libGAP_ExecAssRecName);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_REC_EXPR   , libGAP_ExecAssRecExpr);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_REC_NAME   , libGAP_ExecUnbRecName);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_REC_EXPR   , libGAP_ExecUnbRecExpr);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM_REC_NAME   , libGAP_EvalElmRecName);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM_REC_EXPR   , libGAP_EvalElmRecExpr);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_REC_NAME   , libGAP_EvalIsbRecName);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_REC_EXPR   , libGAP_EvalIsbRecExpr);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_REC_NAME   , libGAP_PrintAssRecName);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_REC_EXPR   , libGAP_PrintAssRecExpr);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_REC_NAME   , libGAP_PrintUnbRecName);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_REC_EXPR   , libGAP_PrintUnbRecExpr);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM_REC_NAME   , libGAP_PrintElmRecName);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM_REC_EXPR   , libGAP_PrintElmRecExpr);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_REC_NAME   , libGAP_PrintIsbRecName);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_REC_EXPR   , libGAP_PrintIsbRecExpr);

    /* install executors, evaluators, and printers for list elements       */
    libGAP_InstallExecStatFunc( libGAP_T_ASS_POSOBJ       , libGAP_ExecAssPosObj);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_POSOBJ       , libGAP_ExecUnbPosObj);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM_POSOBJ       , libGAP_EvalElmPosObj);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_POSOBJ       , libGAP_EvalIsbPosObj);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_POSOBJ       , libGAP_PrintAssPosObj);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_POSOBJ       , libGAP_PrintUnbPosObj);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM_POSOBJ       , libGAP_PrintElmPosObj);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_POSOBJ       , libGAP_PrintIsbPosObj);

    /* install executors, evaluators, and printers for record elements     */
    libGAP_InstallExecStatFunc( libGAP_T_ASS_COMOBJ_NAME  , libGAP_ExecAssComObjName);
    libGAP_InstallExecStatFunc( libGAP_T_ASS_COMOBJ_EXPR  , libGAP_ExecAssComObjExpr);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_COMOBJ_NAME  , libGAP_ExecUnbComObjName);
    libGAP_InstallExecStatFunc( libGAP_T_UNB_COMOBJ_EXPR  , libGAP_ExecUnbComObjExpr);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM_COMOBJ_NAME  , libGAP_EvalElmComObjName);
    libGAP_InstallEvalExprFunc( libGAP_T_ELM_COMOBJ_EXPR  , libGAP_EvalElmComObjExpr);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_COMOBJ_NAME  , libGAP_EvalIsbComObjName);
    libGAP_InstallEvalExprFunc( libGAP_T_ISB_COMOBJ_EXPR  , libGAP_EvalIsbComObjExpr);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_COMOBJ_NAME  , libGAP_PrintAssComObjName);
    libGAP_InstallPrintStatFunc( libGAP_T_ASS_COMOBJ_EXPR  , libGAP_PrintAssComObjExpr);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_COMOBJ_NAME  , libGAP_PrintUnbComObjName);
    libGAP_InstallPrintStatFunc( libGAP_T_UNB_COMOBJ_EXPR  , libGAP_PrintUnbComObjExpr);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM_COMOBJ_NAME  , libGAP_PrintElmComObjName);
    libGAP_InstallPrintExprFunc( libGAP_T_ELM_COMOBJ_EXPR  , libGAP_PrintElmComObjExpr);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_COMOBJ_NAME  , libGAP_PrintIsbComObjName);
    libGAP_InstallPrintExprFunc( libGAP_T_ISB_COMOBJ_EXPR  , libGAP_PrintIsbComObjExpr);

    /* install before and after actions for garbage collections            */
    libGAP_InitCollectFuncBags( libGAP_VarsBeforeCollectBags, libGAP_VarsAfterCollectBags );

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    libGAP_InitCopyGVar("TYPE_LVARS",&libGAP_TYPE_LVARS);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_TLS(libGAP_CurrLVars) = libGAP_TLS(libGAP_BottomLVars);
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_BottomLVars) );


    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Obj                 tmp;

    libGAP_TLS(libGAP_BottomLVars) = libGAP_NewBag( libGAP_T_LVARS, 3*sizeof(libGAP_Obj) );
    tmp = libGAP_NewFunctionC( "bottom", 0, "", 0 );
    libGAP_PTR_BAG(libGAP_TLS(libGAP_BottomLVars))[0] = tmp;
    tmp = libGAP_NewBag( libGAP_T_BODY, libGAP_NUMBER_HEADER_ITEMS_BODY*sizeof(libGAP_Obj) );
    libGAP_BODY_FUNC( libGAP_PTR_BAG(libGAP_TLS(libGAP_BottomLVars))[0] ) = tmp;

    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return libGAP_PostRestore( libGAP_module );
}


/****************************************************************************
**
*F  InitInfoVars()  . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "vars",                             /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoVars ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  vars.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
