/****************************************************************************
**
*W  objscoll.c                  GAP source                       Frank Celler
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains a single collector for finite polycyclic groups.
**
**  Unfortunately, there  are quite a  lot of stacks  required  in the single
**  collector. The collector functions will adjust the lists to have physical
**  length equal  to the maximum defined  in 'maxStackSize'.  Therefore it is
**  possible to initialise all stacks with an empty list.
**
**  There  are  also    two temporary   collector    vectors  'cwVector'  and
**  'cw2Vector',  the functions   'CXBits_VectorWord' will  adjust the string
**  length to  match the number of rws  generators.  Therefore it is possible
**  to initialise these  vectors with an  empty string.  WARNING: if  you use
**  such  a  vector, you *must* clear   it afterwards, because  all functions
**  assume that the vectors are cleared.
*/
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */
#include        "gap.h"                 /* error handling, initialisation  */

#include        "calls.h"               /* generic call mechanism          */

#include        "records.h"             /* generic records                 */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */

#include        "tls.h"                 /* thread-local storage            */
#include        "objfgelm.h"            /* objects of free groups          */

#include        "objscoll.h"            /* single collector                */

#include        "objccoll.h"            /* combinatorial collector         */
#include        "thread.h"

/****************************************************************************
**
*F * * * * * * * * * * * * local defines and typedefs * * * * * * * * * * * *
*/

libGAP_Obj libGAP_SC_NW_STACK;
libGAP_Obj libGAP_SC_LW_STACK;
libGAP_Obj libGAP_SC_PW_STACK;
libGAP_Obj libGAP_SC_EW_STACK;
libGAP_Obj libGAP_SC_GE_STACK;
libGAP_Obj libGAP_SC_CW_VECTOR;
libGAP_Obj libGAP_SC_CW2_VECTOR;
libGAP_UInt libGAP_SC_MAX_STACK_SIZE;

/****************************************************************************
**
*T  FinPowConjCol
**
**  'FinPowConjCol' is a structure containing  all the functions depending on
**  the number of bits used in the a finite power/conjugate collector.
*/
typedef libGAP_Int (*libGAP_FuncIOOO)  (libGAP_Obj,libGAP_Obj,libGAP_Obj);
typedef libGAP_Obj (*libGAP_FuncOOOI)  (libGAP_Obj,libGAP_Obj,libGAP_Int);
typedef libGAP_Int (*libGAP_FuncIOOI)  (libGAP_Obj,libGAP_Obj,libGAP_Int);
typedef libGAP_Obj (*libGAP_FuncOOOO)  (libGAP_Obj,libGAP_Obj,libGAP_Obj);
typedef libGAP_Int (*libGAP_FuncIOOOF) (libGAP_Obj,libGAP_Obj,libGAP_Obj,libGAP_FuncIOOO);

typedef struct {

    libGAP_FuncOOOI    wordVectorAndClear;
    libGAP_FuncIOOI    vectorWord;
    libGAP_FuncIOOO    collectWord;
    libGAP_FuncIOOOF   solution;

} libGAP_FinPowConjCol;


/****************************************************************************
**
*F * * * * * * * * * * * internal collector functions * * * * * * * * * * * *
*/

#define libGAP_WordVectorAndClear  C8Bits_WordVectorAndClear
#define libGAP_VectorWord          C8Bits_VectorWord
#define libGAP_SingleCollectWord   C8Bits_SingleCollectWord
#define libGAP_SAddWordIntoExpVec  C8Bits_SAddWordIntoExpVec
#define libGAP_SAddPartIntoExpVec  C8Bits_SAddPartIntoExpVec
#define libGAP_SingleCollectWord   C8Bits_SingleCollectWord
#define libGAP_Solution            C8Bits_Solution
#define libGAP_UIntN               libGAP_UInt1
#include "objscoll-impl.h"

/****************************************************************************
**
*V  C8Bits_SingleCollector
*/
libGAP_FinPowConjCol libGAP_C8Bits_SingleCollector = {
    C8Bits_WordVectorAndClear,
    C8Bits_VectorWord,
    C8Bits_SingleCollectWord,
    C8Bits_Solution
};


#define libGAP_WordVectorAndClear  C16Bits_WordVectorAndClear
#define libGAP_VectorWord          C16Bits_VectorWord
#define libGAP_SingleCollectWord   C16Bits_SingleCollectWord
#define libGAP_SAddWordIntoExpVec  C16Bits_SAddWordIntoExpVec
#define libGAP_SAddPartIntoExpVec  C16Bits_SAddPartIntoExpVec
#define libGAP_SingleCollectWord   C16Bits_SingleCollectWord
#define libGAP_Solution            C16Bits_Solution
#define libGAP_UIntN               libGAP_UInt2
#include "objscoll-impl.h"

/****************************************************************************
**
*V  C16Bits_SingleCollector
*/
libGAP_FinPowConjCol libGAP_C16Bits_SingleCollector = {
    C16Bits_WordVectorAndClear,
    C16Bits_VectorWord,
    C16Bits_SingleCollectWord,
    C16Bits_Solution
};


#define libGAP_WordVectorAndClear  C32Bits_WordVectorAndClear
#define libGAP_VectorWord          C32Bits_VectorWord
#define libGAP_SingleCollectWord   C32Bits_SingleCollectWord
#define libGAP_SAddWordIntoExpVec  C32Bits_SAddWordIntoExpVec
#define libGAP_SAddPartIntoExpVec  C32Bits_SAddPartIntoExpVec
#define libGAP_SingleCollectWord   C32Bits_SingleCollectWord
#define libGAP_Solution            C32Bits_Solution
#define libGAP_UIntN               libGAP_UInt4
#include "objscoll-impl.h"

/****************************************************************************
**
*V  C32Bits_SingleCollector
*/
libGAP_FinPowConjCol libGAP_C32Bits_SingleCollector = {
    C32Bits_WordVectorAndClear,
    C32Bits_VectorWord,
    C32Bits_SingleCollectWord,
    C32Bits_Solution
};

/****************************************************************************
**
*F * * * * * * * * * * *  combinatorial collectors  * * * * * * * * * * * * *
**
**  Here the combinatorial collectors are setup.  They behave like single
**  collectors and therefore can be used int the same way.
*/

/****************************************************************************
**

*V  C8Bits_CombiCollector
*/
libGAP_FinPowConjCol libGAP_C8Bits_CombiCollector = {
    C8Bits_WordVectorAndClear,
    C8Bits_VectorWord,
    libGAP_C8Bits_CombiCollectWord,
    C8Bits_Solution
};

/****************************************************************************
**

*V  C16Bits_CombiCollector
*/
libGAP_FinPowConjCol libGAP_C16Bits_CombiCollector = {
    C16Bits_WordVectorAndClear,
    C16Bits_VectorWord,
    libGAP_C16Bits_CombiCollectWord,
    C16Bits_Solution
};

/****************************************************************************
**

*V  C32Bits_CombiCollector
*/
libGAP_FinPowConjCol libGAP_C32Bits_CombiCollector = {
    C32Bits_WordVectorAndClear,
    C32Bits_VectorWord,
    libGAP_C32Bits_CombiCollectWord,
    C32Bits_Solution
};

/****************************************************************************
**

*V  FinPowConjCollectors
*/
libGAP_FinPowConjCol * libGAP_FinPowConjCollectors [6] =
{
#define libGAP_C8Bits_SingleCollectorNo        0
       &libGAP_C8Bits_SingleCollector,
#define libGAP_C16Bits_SingleCollectorNo       1
       &libGAP_C16Bits_SingleCollector,
#define libGAP_C32Bits_SingleCollectorNo       2
       &libGAP_C32Bits_SingleCollector,
#define libGAP_C8Bits_CombiCollectorNo         3
       &libGAP_C8Bits_CombiCollector,
#define libGAP_C16Bits_CombiCollectorNo        4
       &libGAP_C16Bits_CombiCollector,
#define libGAP_C32Bits_CombiCollectorNo        5
       &libGAP_C32Bits_CombiCollector
};

/****************************************************************************
**

*F * * * * * * * * * * * * reduce something functions * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  CollectWordOrFail( <fc>, <sc>, <vv>, <w> )
*/
libGAP_Obj libGAP_CollectWordOrFail ( 
    libGAP_FinPowConjCol *     fc, 
    libGAP_Obj                 sc,
    libGAP_Obj                 vv,
    libGAP_Obj                 w )
{
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Obj *               ptr;            /* pointer into the array <vv>     */

    /* convert <vv> into a list of C integers                              */
    ptr = libGAP_ADDR_OBJ(vv)+1;
    for ( i = libGAP_LEN_PLIST(vv);  0 < i;  i--, ptr++ )
        *ptr = (libGAP_Obj)libGAP_INT_INTOBJ(*ptr);

    /* now collect <w> into <vv>                                           */
    if ( fc->collectWord( sc, vv, w ) == -1 ) {
         /* If the collector fails, we return the vector clean.            */
        ptr = libGAP_ADDR_OBJ(vv)+1;
        for ( i = libGAP_LEN_PLIST(vv);  0 < i;  i--, ptr++ )
            *ptr = libGAP_INTOBJ_INT(0);

        return libGAP_Fail;
    }

    /* and convert back                                                    */
    ptr = libGAP_ADDR_OBJ(vv)+1;
    for ( i = libGAP_LEN_PLIST(vv);  0 < i;  i--, ptr++ )
        *ptr = libGAP_INTOBJ_INT((libGAP_Int)*ptr);

    return libGAP_True;
}


/****************************************************************************
**
*F  ReducedComm( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedComm (
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 type;       /* type of the returned object         */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <u>*<w> to                           */
    vcw = libGAP_TLS(libGAP_SC_CW_VECTOR);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length, unpack <u> into it            */
    if ( fc->vectorWord( vcw, u, num ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <w> into it                                                 */
    if ( fc->collectWord( sc, vcw, w ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedComm( fc, sc, w, u );
    }

    /* use 'cw2Vector' to collect word <w>*<u> to                          */
    vc2 = libGAP_TLS(libGAP_SC_CW2_VECTOR);

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vc2, w, num ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <u> into it                                                 */
    if ( fc->collectWord( sc, vc2, u ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedComm( fc, sc, w, u );
    }

    /* now use 'Solution' to solve the equation, will clear <vcw>          */
    if ( fc->solution( sc, vcw, vc2, fc->collectWord ) == -1 )
    {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedComm( fc, sc, w, u );
    }

    /* convert the vector <vc2> into a word and clear <vc2>                */
    type = libGAP_SC_DEFAULT_TYPE(sc);
    return fc->wordVectorAndClear( type, vc2, num );
}


/****************************************************************************
**
*F  ReducedForm( <fc>, <sc>, <w> )
*/
libGAP_Obj libGAP_ReducedForm (
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w )
{
    libGAP_Int                 num;    /* number of gen/exp pairs in <data>       */
    libGAP_Int                 i;      /* loop variable for gen/exp pairs         */
    libGAP_Obj                 vcw;    /* collect vector                          */
    libGAP_Obj                 type;   /* type of the return objue                */
    libGAP_Int *               qtr;    /* pointer into the collect vector         */

    /* use 'cwVector' to collect word <w> to                               */
    vcw = libGAP_TLS(libGAP_SC_CW_VECTOR);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length                                */
    if ( fc->vectorWord( vcw, 0, num ) == -1 )
        return libGAP_Fail;

    /* and collect <w> into it                                             */
    while ( (i = fc->collectWord( sc, vcw, w )) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
    }
    num = i;

    /* get the default type                                                */
    type = libGAP_SC_DEFAULT_TYPE(sc);

    /* convert the vector <cvw> into a word and clear <vcw>                */
    return fc->wordVectorAndClear( type, vcw, num );
}


/****************************************************************************
**
*F  ReducedLeftQuotient( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedLeftQuotient ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 type;       /* type of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <w> to                               */
    vcw = libGAP_TLS(libGAP_SC_CW_VECTOR);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vcw, w, num ) == -1 )  {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* use 'cw2Vector' to collect word <u> to                              */
    vc2 = libGAP_TLS(libGAP_SC_CW2_VECTOR);

    /* check that it has the correct length, unpack <u> into it            */
    if ( fc->vectorWord( vc2, u, num ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* now use 'Solution' to solve the equation, will clear <vcw>          */
    if ( fc->solution( sc, vcw, vc2, fc->collectWord ) == -1 )
    {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedLeftQuotient( fc, sc, w, u );
    }

    /* convert the vector <vc2> into a word and clear <vc2>                */
    type = libGAP_SC_DEFAULT_TYPE(sc);
    return fc->wordVectorAndClear( type, vc2, num );
}


/****************************************************************************
**
*F  ReducedProduct( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedProduct ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 type;       /* type of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <w> to                               */
    vcw = libGAP_TLS(libGAP_SC_CW_VECTOR);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vcw, w, num ) == -1 )  {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <w> into it                                                 */
    if ( fc->collectWord( sc, vcw, u ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedProduct( fc, sc, w, u );
    }

    /* convert the vector <vcw> into a word and clear <vcw>                */
    type = libGAP_SC_DEFAULT_TYPE(sc);
    return fc->wordVectorAndClear( type, vcw, num );
}


/****************************************************************************
**
*F  ReducedPowerSmallInt( <fc>, <sc>, <w>, <pow> )
*/
libGAP_Obj libGAP_ReducedPowerSmallInt ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 vpow )
{
    libGAP_Obj                 type;       /* type of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int                 pow;        /* power to raise <w> to               */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */
    libGAP_Obj                 res;        /* the result                          */

    /* get the integer of <vpow>                                           */
    pow = libGAP_INT_INTOBJ(vpow);

    /* use 'cwVector' and 'cw2Vector to collect words to                   */
    vcw  = libGAP_TLS(libGAP_SC_CW_VECTOR);
    vc2  = libGAP_TLS(libGAP_SC_CW2_VECTOR);
    num  = libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    type = libGAP_SC_DEFAULT_TYPE(sc);

    /* return the trivial word if <pow> is zero                            */
    if ( pow == 0 ) {
        libGAP_NEW_WORD( res, type, 0 );
        return res;
    }

    /* invert <w> if <pow> is negative                                     */
    if ( pow < 0 ) {
        
        /* check that it has the correct length, unpack <w> into it        */
        if ( fc->vectorWord( vcw, w, num ) == -1 )  {
            for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--,qtr++ )
                *qtr = 0;
            return libGAP_Fail;
        }

        /* use 'Solution' to invert it, this will clear <vcw>              */
        if (fc->solution(sc,vcw,vc2,fc->collectWord) == -1) {
                for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0<i;  i--,qtr++ )
                    *qtr = 0;
                for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0<i;  i--,qtr++ )
                    *qtr = 0;
                return libGAP_ReducedPowerSmallInt(fc,sc,w,vpow);
        }

        /* and replace <pow> and <w> by its inverse                        */
        pow  = -pow;
        vpow = libGAP_INTOBJ_INT(pow);
        w    = fc->wordVectorAndClear( type, vc2, num );

    }

    /* if <pow> is one, do nothing                                         */
    if ( pow == 1 ) {
        return w;
    }

    /* catch small cases                                                   */
    if ( pow < 6 ) {

        /* check that it has the correct length, unpack <w> into it        */
        if ( fc->vectorWord( vcw, w, num ) == -1 )  {
            for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--,qtr++ )
                *qtr = 0;
            return libGAP_Fail;
        }

        /* multiply <w> into <vcw>                                         */
        for ( i = pow;  1 < i;  i-- ) {
            if ( fc->collectWord( sc, vcw, w ) == -1 ) {
                for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0<i;  i--,qtr++ )
                    *qtr = 0;
                return libGAP_ReducedPowerSmallInt(fc,sc,w,vpow);
            }
        }

        /* convert it back, this will clear <vcw>                          */
        return fc->wordVectorAndClear( type, vcw, num );

    }

    /* use "divide et impera" instead of repeated squaring r2l             */
    if ( pow % 2 ) {
        res = libGAP_ReducedPowerSmallInt( fc, sc, w, libGAP_INTOBJ_INT((pow-1)/2) );
        return libGAP_ReducedProduct( fc, sc, w,
            libGAP_ReducedProduct( fc, sc, res, res ) );
    }
    else {
        res = libGAP_ReducedPowerSmallInt( fc, sc, w, libGAP_INTOBJ_INT(pow/2) );
        return libGAP_ReducedProduct( fc, sc, res, res );
    }

}


/****************************************************************************
**
*F  ReducedQuotient( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedQuotient ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 type;       /* type of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <w> to                               */
    vcw  = libGAP_TLS(libGAP_SC_CW_VECTOR);
    vc2  = libGAP_TLS(libGAP_SC_CW2_VECTOR);
    num  = libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    type = libGAP_SC_DEFAULT_TYPE(sc);

    /* check that it has the correct length, unpack <u> into it            */
    if ( fc->vectorWord( vcw, u, num ) == -1 )  {
        for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--,qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* use 'Solution' to invert it, this will clear <vcw>                  */
    if ( fc->solution( sc, vcw, vc2, fc->collectWord ) == -1 ) {
        for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0<i;  i--,qtr++ )
            *qtr = 0;
        for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0<i;  i--,qtr++ )
            *qtr = 0;
        return libGAP_ReducedQuotient( fc, sc, w, u );
    }

    /* and replace <u> by its inverse                                      */
    u = fc->wordVectorAndClear( type, vc2, num );

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vcw, w, num ) == -1 )  {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <w> into it                                                 */
    if ( fc->collectWord( sc, vcw, u ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedQuotient( fc, sc, w, u );
    }

    /* convert the vector <vcw> into a word and clear <vcw>                */
    return fc->wordVectorAndClear( type, vcw, num );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * exported GAP functions * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  FuncFinPowConjCol_CollectWordOrFail( <self>, <sc>, <vv>, <w> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_CollectWordOrFail ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    return libGAP_CollectWordOrFail( libGAP_SC_COLLECTOR(sc), sc, vv, w );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedComm( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedComm ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedComm( libGAP_SC_COLLECTOR(sc), sc, w, u );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedForm( <self>, <sc>, <w> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedForm ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w )
{
    return libGAP_ReducedForm( libGAP_SC_COLLECTOR(sc), sc, w );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedLeftQuotient( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedLeftQuotient ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedLeftQuotient( libGAP_SC_COLLECTOR(sc), sc, w, u );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedProduct( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedProduct ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedProduct( libGAP_SC_COLLECTOR(sc), sc, w, u );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedPowerSmallInt( <self>, <sc>, <w>, <pow> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedPowerSmallInt (libGAP_Obj self,libGAP_Obj sc,libGAP_Obj w,libGAP_Obj vpow)
{
    return libGAP_ReducedPowerSmallInt( libGAP_SC_COLLECTOR(sc), sc, w, vpow );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedQuotient( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedQuotient ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedQuotient( libGAP_SC_COLLECTOR(sc), sc, w, u );
}


/****************************************************************************
**
*F  SET_SCOBJ_MAX_STACK_SIZE( <self>, <size> )
*/
libGAP_Obj libGAP_FuncSET_SCOBJ_MAX_STACK_SIZE ( libGAP_Obj self, libGAP_Obj size )
{
    if (libGAP_IS_INTOBJ(size) && libGAP_INT_INTOBJ(size) > 0)
        libGAP_TLS(libGAP_SC_MAX_STACK_SIZE) = libGAP_INT_INTOBJ(size);
    else
        libGAP_ErrorQuit( "collect vector must be a positive small integer not a %s",
                   (libGAP_Int)libGAP_TNAM_OBJ(size), 0L );

    return 0;
}




/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "FinPowConjCol_CollectWordOrFail", 3, "sc, list, word",
      libGAP_FuncFinPowConjCol_CollectWordOrFail, 
      "src/objscoll.c:FinPowConjCol_CollectWordOrFail" },

    { "FinPowConjCol_ReducedComm", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedComm, 
      "src/objscoll.c:FinPowConjCol_ReducedComm" },

    { "FinPowConjCol_ReducedForm", 2, "sc, word",
      libGAP_FuncFinPowConjCol_ReducedForm, 
      "src/objscoll.c:FinPowConjCol_ReducedForm" },

    { "FinPowConjCol_ReducedLeftQuotient", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedLeftQuotient, 
      "src/objscoll.c:FinPowConjCol_ReducedLeftQuotient" },

    { "FinPowConjCol_ReducedPowerSmallInt", 3, "sc, word, int",
      libGAP_FuncFinPowConjCol_ReducedPowerSmallInt,
      "src/objscoll.c:FinPowConjCol_ReducedPowerSmallInt" },

    { "FinPowConjCol_ReducedProduct", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedProduct,
      "src/objscoll.c:FinPowConjCol_ReducedProduct" },

    { "FinPowConjCol_ReducedQuotient", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedQuotient,
      "src/objscoll.c:FinPowConjCol_ReducedQuotient" },

    { "SET_SCOBJ_MAX_STACK_SIZE", 1, "size",
      libGAP_FuncSET_SCOBJ_MAX_STACK_SIZE,
      "src/objscoll.c:SET_SCOBJ_MAX_STACK_SIZE" },

    { 0 }

};


/*
 * Allocate a Plist of the given length, pre-allocating
 * the number of entries given by 'reserved'.
 */
static inline libGAP_Obj libGAP_NewPlist( libGAP_UInt tnum, libGAP_UInt len, libGAP_UInt reserved )
{
    libGAP_Obj obj;
    obj = libGAP_NEW_PLIST( tnum, reserved );
    libGAP_SET_LEN_PLIST( obj, len );
    return obj;
}

/*
 * Setup the collector stacks etc.
 */
static void libGAP_SetupCollectorStacks()
{
    const libGAP_UInt maxStackSize = 256;
    libGAP_TLS(libGAP_SC_NW_STACK) = libGAP_NewPlist( libGAP_T_PLIST_EMPTY, 0, maxStackSize );
    libGAP_TLS(libGAP_SC_LW_STACK) = libGAP_NewPlist( libGAP_T_PLIST_EMPTY, 0, maxStackSize );
    libGAP_TLS(libGAP_SC_PW_STACK) = libGAP_NewPlist( libGAP_T_PLIST_EMPTY, 0, maxStackSize );
    libGAP_TLS(libGAP_SC_EW_STACK) = libGAP_NewPlist( libGAP_T_PLIST_EMPTY, 0, maxStackSize );
    libGAP_TLS(libGAP_SC_GE_STACK) = libGAP_NewPlist( libGAP_T_PLIST_EMPTY, 0, maxStackSize );
    libGAP_TLS(libGAP_SC_CW_VECTOR) = libGAP_NEW_STRING( 0 );
    libGAP_TLS(libGAP_SC_CW2_VECTOR) = libGAP_NEW_STRING( 0 );
    libGAP_TLS(libGAP_SC_MAX_STACK_SIZE) = maxStackSize;
}


/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    libGAP_InitGlobalBag( &libGAP_SC_NW_STACK, "SC_NW_STACK" );
    libGAP_InitGlobalBag( &libGAP_SC_LW_STACK, "SC_LW_STACK" );
    libGAP_InitGlobalBag( &libGAP_SC_PW_STACK, "SC_PW_STACK" );
    libGAP_InitGlobalBag( &libGAP_SC_EW_STACK, "SC_EW_STACK" );
    libGAP_InitGlobalBag( &libGAP_SC_GE_STACK, "SC_GE_STACK" );
    libGAP_InitGlobalBag( &libGAP_SC_CW_VECTOR, "SC_CW_VECTOR" );
    libGAP_InitGlobalBag( &libGAP_SC_CW2_VECTOR, "SC_CW2_VECTOR" );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* export position numbers 'SCP_SOMETHING'                             */
    libGAP_AssGVar( libGAP_GVarName( "SCP_UNDERLYING_FAMILY" ),
             libGAP_INTOBJ_INT(libGAP_SCP_UNDERLYING_FAMILY) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_RWS_GENERATORS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_RWS_GENERATORS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_NUMBER_RWS_GENERATORS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_NUMBER_RWS_GENERATORS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_DEFAULT_TYPE" ),
             libGAP_INTOBJ_INT(libGAP_SCP_DEFAULT_TYPE) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_IS_DEFAULT_TYPE" ),
             libGAP_INTOBJ_INT(libGAP_SCP_IS_DEFAULT_TYPE) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_RELATIVE_ORDERS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_RELATIVE_ORDERS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_POWERS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_POWERS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_CONJUGATES" ),
             libGAP_INTOBJ_INT(libGAP_SCP_CONJUGATES) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_INVERSES" ),
             libGAP_INTOBJ_INT(libGAP_SCP_INVERSES) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_COLLECTOR" ),
             libGAP_INTOBJ_INT(libGAP_SCP_COLLECTOR) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_AVECTOR" ),
             libGAP_INTOBJ_INT(libGAP_SCP_AVECTOR) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_WEIGHTS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_WEIGHTS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_CLASS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_CLASS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_AVECTOR2" ),
             libGAP_INTOBJ_INT(libGAP_SCP_AVECTOR2) );

    libGAP_SetupCollectorStacks();

    /* export collector number                                             */
    libGAP_AssGVar( libGAP_GVarName( "8Bits_SingleCollector" ),
             libGAP_INTOBJ_INT(libGAP_C8Bits_SingleCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "16Bits_SingleCollector" ),
             libGAP_INTOBJ_INT(libGAP_C16Bits_SingleCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "32Bits_SingleCollector" ),
             libGAP_INTOBJ_INT(libGAP_C32Bits_SingleCollectorNo) );

    libGAP_AssGVar( libGAP_GVarName( "8Bits_CombiCollector" ),
             libGAP_INTOBJ_INT(libGAP_C8Bits_CombiCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "16Bits_CombiCollector" ),
             libGAP_INTOBJ_INT(libGAP_C16Bits_CombiCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "32Bits_CombiCollector" ),
             libGAP_INTOBJ_INT(libGAP_C32Bits_CombiCollectorNo) );

    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoSingleCollector() . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "objscoll",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoSingleCollector ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  objscoll.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
