/****************************************************************************
**
*W  objects.c                   GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the objects package.
*/
#include	<stdlib.h>

#include        "system.h"              /* Ints, UInts, SyIsIntr           */


#include        "sysfiles.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */

#include        "objects.h"             /* objects                         */

#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "bool.h"                /* booleans                        */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */

#include        "aobjects.h"            /* atomic objects                  */
#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
//#include	"traverse.h"		/* object traversal		   */
#include	"tls.h"			/* thread-local storage		   */


static libGAP_Int libGAP_lastFreePackageTNUM = libGAP_FIRST_PACKAGE_TNUM;

/****************************************************************************
**
*F  RegisterPackageTNUM( <name>, <typeObjFunc> )
**
**  Allocates a TNUM for use by a package. The parameters <name> and
**  <typeObjFunc> are used to initialize the relevant entries in the
**  InfoBags and TypeObjFuncs arrays.
**
**  If allocation fails (e.g. because no more TNUMs are available),
**  a negative value is returned.
*/
libGAP_Int libGAP_RegisterPackageTNUM( const char *name, libGAP_Obj (*typeObjFunc)(libGAP_Obj obj) )
{
    if (libGAP_lastFreePackageTNUM > libGAP_LAST_PACKAGE_TNUM)
        return -1;

    libGAP_Int tnum = libGAP_lastFreePackageTNUM++;

    libGAP_InfoBags[tnum].name = name;
    libGAP_TypeObjFuncs[tnum] = typeObjFunc;

    return tnum;
}


/****************************************************************************
**

*F  FamilyTypeHandler( <self>, <type> ) . . . . . . handler for 'FAMILY_TYPE'
*/
libGAP_Obj libGAP_FamilyTypeHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 type )
{
    return libGAP_FAMILY_TYPE( type );
}


/****************************************************************************
**
*F  FamilyObjHandler( <self>, <obj> ) . . . . . . .  handler for 'FAMILY_OBJ'
*/
libGAP_Obj libGAP_FamilyObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_FAMILY_OBJ( obj );
}


/****************************************************************************
**

*F  TYPE_OBJ( <obj> ) . . . . . . . . . . . . . . . . . . . type of an object
**
**  'TYPE_OBJ' returns the type of the object <obj>.
**
**  'TYPE_OBJ' is defined in the declaration part of this package.
*/
libGAP_Obj (*libGAP_TypeObjFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj libGAP_TypeObjError (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit( "Panic: basic object of type '%s' is unkind",
               (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L );
    return 0;
}


/****************************************************************************
**
*F  TypeObjHandler( <self>, <obj> ) . . . . . . . . .  handler for 'TYPE_OBJ'
*/
libGAP_Obj libGAP_TypeObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_TYPE_OBJ( obj );
}


/****************************************************************************
**

*F  IS_MUTABLE_OBJ( <obj> ) . . . . . . . . . . . . . .  is an object mutable
**
**  'IS_MUTABLE_OBJ' returns   1 if the object  <obj> is mutable   (i.e., can
**  change due to assignments), and 0 otherwise.
**
**  'IS_MUTABLE_OBJ' is defined in the declaration part of this package.
*/
libGAP_Int (*libGAP_IsMutableObjFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj libGAP_IsMutableObjFilt;

libGAP_Int libGAP_IsMutableObjError (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "Panic: tried to test mutability of unknown type '%d'",
        (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
    return 0;
}

libGAP_Int libGAP_IsMutableObjNot (
    libGAP_Obj                 obj )
{
    return 0L;
}

libGAP_Int libGAP_IsMutableObjObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsMutableObjFilt, obj ) == libGAP_True);
}


/****************************************************************************
**
*F  IsMutableObjHandler( <self>, <obj> )  . . .  handler for 'IS_MUTABLE_OBJ'
*/
libGAP_Obj libGAP_IsMutableObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_MUTABLE_OBJ( obj ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  IS_COPYABLE_OBJ(<obj>)  . . . . . . . . . . . . . . is an object copyable
**
**  'IS_COPYABLE_OBJ' returns 1 if the object <obj> is copyable (i.e., can be
**  copied into a mutable object), and 0 otherwise.
**
**  'IS_COPYABLE_OBJ' is defined in the declaration part of this package.
*/
libGAP_Int (*libGAP_IsCopyableObjFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj libGAP_IsCopyableObjFilt;

libGAP_Int libGAP_IsCopyableObjError (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "Panic: tried to test copyability of unknown type '%d'",
        (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
    return 0L;
}

libGAP_Int libGAP_IsCopyableObjNot (
    libGAP_Obj                 obj )
{
    return 0L;
}

libGAP_Int libGAP_IsCopyableObjObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsCopyableObjFilt, obj ) == libGAP_True);
}


/****************************************************************************
**
*F  IsCopyableObjHandler( <self>, <obj> ) . . . handler for 'IS_COPYABLE_OBJ'
*/
libGAP_Obj libGAP_IsCopyableObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_COPYABLE_OBJ( obj ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**

*V  ShallowCopyObjFuncs[<type>] . . . . . . . . . .  shallow copier functions
*/
libGAP_Obj (*libGAP_ShallowCopyObjFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj libGAP_ShallowCopyObjOper;


/****************************************************************************
**
*F  ShallowCopyObjError( <obj> )  . . . . . . . . . . . . . . .  unknown type
*/
libGAP_Obj libGAP_ShallowCopyObjError (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "Panic: tried to shallow copy object of unknown type '%d'",
        (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  ShallowCopyObjConstant( <obj> ) . . . . . . . . . . . . . . .  do nothing
*/
libGAP_Obj libGAP_ShallowCopyObjConstant (
    libGAP_Obj                 obj )
{
    return obj;
}


/****************************************************************************
**
*F  ShallowCopyObjObject( <obj> ) . . . . . . . . . . . . . . . . call method
*/
libGAP_Obj libGAP_ShallowCopyObjObject (
    libGAP_Obj                 obj )
{
    return libGAP_DoOperation1Args( libGAP_ShallowCopyObjOper, obj );
}


/****************************************************************************
**
*F  ShallowCopyObjDefault( <obj> )  . . . . . . . . . .  default shallow copy
*/
libGAP_Obj libGAP_ShallowCopyObjDefault (
    libGAP_Obj                 obj )
{
    libGAP_Obj                 new;
    libGAP_Obj *               o;
    libGAP_Obj *               n;
    libGAP_UInt                len;
    libGAP_UInt                i;

    /* make the new object and copy the contents                           */
    len = (libGAP_SIZE_OBJ( obj ) + sizeof(libGAP_Obj)-1) / sizeof(libGAP_Obj);
    new = libGAP_NewBag( libGAP_MUTABLE_TNUM(libGAP_TNUM_OBJ(obj)), libGAP_SIZE_OBJ(obj) );
    o = libGAP_ADDR_OBJ( obj );
    n = libGAP_ADDR_OBJ( new );
    for ( i = 0; i < len; i++ ) {
        *n++ = *o++;
    }

    /* 'CHANGED_BAG(new);' not needed, <new> is newest object              */
    return new;
}


/****************************************************************************
**
*F  ShallowCopyObjHandler( <self>, <obj> )  .  handler for 'SHALLOW_COPY_OBJ'
*/
libGAP_Obj libGAP_ShallowCopyObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_SHALLOW_COPY_OBJ( obj );
}


/****************************************************************************
**
*F  CopyObj( <obj>, <mut> ) . . . . . . . make a structural copy of an object
**
**  'CopyObj' only calls 'COPY_OBJ' and then 'CLEAN_OBJ'.
*/
libGAP_Obj libGAP_CopyObj (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    libGAP_Obj                 new;            /* copy of <obj>                   */

    /* make a copy                                                         */
    new = libGAP_COPY_OBJ( obj, mut );

    /* clean up the marks                                                  */
    libGAP_CLEAN_OBJ( obj );

    /* return the copy                                                     */
    return new;
}


/****************************************************************************
**
*V  CopyObjFuncs[<type>]  . . . . . . . . . . . .  table of copying functions
*/
libGAP_Obj (*libGAP_CopyObjFuncs[ libGAP_LAST_REAL_TNUM+libGAP_COPYING+1 ]) ( libGAP_Obj obj, libGAP_Int mut );


/****************************************************************************
**
*V  CleanObjFuncs[<type>] . . . . . . . . . . . . table of cleaning functions
*/
void (*libGAP_CleanObjFuncs[ libGAP_LAST_REAL_TNUM+libGAP_COPYING+1 ]) ( libGAP_Obj obj );


/****************************************************************************
**
*F  CopyObjError( <obj> ) . . . . . . . . . . . . . . . . . . .  unknown type
*/
libGAP_Obj             libGAP_CopyObjError (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    libGAP_ErrorQuit(
        "Panic: tried to copy object of unknown type '%d'",
        (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  CleanObjError( <obj> )  . . . . . . . . . . . . . . . . . .  unknown type
*/
void libGAP_CleanObjError (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "Panic: tried to clean object of unknown type '%d'",
        (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
}


/****************************************************************************
**
*F  CopyObjConstant( <obj> )  . . . . . . . . . . . .  copy a constant object
*/
libGAP_Obj libGAP_CopyObjConstant (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    return obj;
}


/****************************************************************************
**
*F  CleanObjConstant(<obj>) . . . . . . . . . . . . . clean a constant object
*/
void libGAP_CleanObjConstant (
    libGAP_Obj                 obj )
{
}


/****************************************************************************
**
*F  CopyObjPosObj( <obj>, <mut> ) . . . . . . . . .  copy a positional object
*/
libGAP_Obj libGAP_CopyObjPosObj (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* copy, result                    */
    libGAP_Obj                 tmp;            /* temporary variable              */
    libGAP_UInt                i;              /* loop variable                   */

    /* don't change immutable objects                                      */
    if ( ! libGAP_IS_MUTABLE_OBJ(obj) ) {
        return obj;
    }

    /* if the object is not copyable return                                */
    if ( ! libGAP_IS_COPYABLE_OBJ(obj) ) {
        libGAP_ErrorQuit("Panic: encountered mutable, non-copyable object",0L,0L);
        return obj;
    }

    /* make a copy                                                         */
    if ( mut ) {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(obj), libGAP_SIZE_OBJ(obj) );
        libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(obj)[0];
    }
    else {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(obj), libGAP_SIZE_OBJ(obj) );
        libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(obj)[0];
        libGAP_CALL_2ARGS( libGAP_RESET_FILTER_OBJ, copy, libGAP_IsMutableObjFilt );
    }

    /* leave a forwarding pointer                                          */
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( tmp, 2 );
    libGAP_SET_ELM_PLIST( tmp, 1, libGAP_ADDR_OBJ(obj)[0] );
    libGAP_SET_ELM_PLIST( tmp, 2, copy );
    libGAP_ADDR_OBJ(obj)[0] = tmp;
    libGAP_CHANGED_BAG(obj);

    /* now it is copied                                                    */
    libGAP_RetypeBag( obj, libGAP_TNUM_OBJ(obj) + libGAP_COPYING );

    /* copy the subvalues                                                  */
    for ( i = 1; i < libGAP_SIZE_OBJ(obj)/sizeof(libGAP_Obj); i++ ) {
        if ( libGAP_ADDR_OBJ(obj)[i] != 0 ) {
            tmp = libGAP_COPY_OBJ( libGAP_ADDR_OBJ(obj)[i], mut );
            libGAP_ADDR_OBJ(copy)[i] = tmp;
            libGAP_CHANGED_BAG( copy );
        }
    }

    /* return the copy                                                     */
    return copy;
}


/****************************************************************************
**
*F  CleanObjPosObj( <obj> ) . . . . . . . . . . . . . . . . . .  clean posobj
*/
void libGAP_CleanObjPosObj (
    libGAP_Obj                 obj )
{
}


/****************************************************************************
**
*F  CopyObjPosObjCopy( <obj>, <mut> ) . . . . . . . . . .  copy a posobj copy
*/
libGAP_Obj libGAP_CopyObjPosObjCopy (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    return libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 2 );
}


/****************************************************************************
**
*F  CleanObjPosObjCopy( <obj> ) . . . . . . . . . . . . . . clean posobj copy
*/
void libGAP_CleanObjPosObjCopy (
    libGAP_Obj                 obj )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(obj)[0] = libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 1 );
    libGAP_CHANGED_BAG(obj);

    /* now it is cleaned                                                   */
    libGAP_RetypeBag( obj, libGAP_TNUM_OBJ(obj) - libGAP_COPYING );

    /* clean the subvalues                                                 */
    for ( i = 1; i < libGAP_SIZE_OBJ(obj)/sizeof(libGAP_Obj); i++ ) {
        if ( libGAP_ADDR_OBJ(obj)[i] != 0 )
            libGAP_CLEAN_OBJ( libGAP_ADDR_OBJ(obj)[i] );
    }

}


/****************************************************************************
**
*F  CopyObjComObj( <obj>, <mut> ) . . . . . . . . . . . . . . . copy a comobj
*/
libGAP_Obj libGAP_CopyObjComObj (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* copy, result                    */
    libGAP_Obj                 tmp;            /* temporary variable              */
    libGAP_UInt                i;              /* loop variable                   */

    /* don't change immutable objects                                      */
    if ( ! libGAP_IS_MUTABLE_OBJ(obj) ) {
        return obj;
    }

    /* if the object is not copyable return                                */
    if ( ! libGAP_IS_COPYABLE_OBJ(obj) ) {
        libGAP_ErrorQuit("Panic: encountered mutable, non-copyable object",0L,0L);
        return obj;
    }

    /* make a copy                                                         */
    if ( mut ) {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(obj), libGAP_SIZE_OBJ(obj) );
        libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(obj)[0];
        libGAP_SET_LEN_PREC(copy,libGAP_LEN_PREC(obj));
    }
    else {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(obj), libGAP_SIZE_OBJ(obj) );
        libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(obj)[0];
        libGAP_SET_LEN_PREC(copy,libGAP_LEN_PREC(obj));
        libGAP_CALL_2ARGS( libGAP_RESET_FILTER_OBJ, copy, libGAP_IsMutableObjFilt );
    }

    /* leave a forwarding pointer                                          */
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( tmp, 2 );
    libGAP_SET_ELM_PLIST( tmp, 1, libGAP_ADDR_OBJ(obj)[0] );
    libGAP_SET_ELM_PLIST( tmp, 2, copy );
    libGAP_ADDR_OBJ(obj)[0] = tmp;
    libGAP_CHANGED_BAG(obj);

    /* now it is copied                                                    */
    libGAP_RetypeBag( obj, libGAP_TNUM_OBJ(obj) + libGAP_COPYING );

    /* copy the subvalues                                                  */
    for ( i = 1; i <= libGAP_LEN_PREC(obj); i++) {
        libGAP_SET_RNAM_PREC(copy,i,libGAP_GET_RNAM_PREC(obj,i));
        tmp = libGAP_COPY_OBJ( libGAP_GET_ELM_PREC(obj,i), mut );
        libGAP_SET_ELM_PREC(copy,i,tmp);
        libGAP_CHANGED_BAG( copy );
    }

    /* return the copy                                                     */
    return copy;
}


/****************************************************************************
**
*F  CleanObjComObj( <obj> ) . . . . . . . . . . . . . . . . .  clean a comobj
*/
void libGAP_CleanObjComObj (
    libGAP_Obj                 obj )
{
}


/****************************************************************************
**
*F  CopyObjComObjCopy( <obj>, <mut> ) . . . . . . . . . .  copy a comobj copy
*/
libGAP_Obj libGAP_CopyObjComObjCopy (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    return libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 2 );
}


/****************************************************************************
**
*F  CleanObjComObjCopy( <obj> ) . . . . . . . . . . . . . clean a comobj copy
*/
void libGAP_CleanObjComObjCopy (
    libGAP_Obj                 obj )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(obj)[0] = libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 1 );
    libGAP_CHANGED_BAG(obj);

    /* now it is cleaned                                                   */
    libGAP_RetypeBag( obj, libGAP_TNUM_OBJ(obj) - libGAP_COPYING );

    /* clean the subvalues                                                 */
    for ( i = 1; i <= libGAP_LEN_PREC(obj); i++ ) {
        libGAP_CLEAN_OBJ( libGAP_GET_ELM_PREC(obj,i) );
    }

}


/****************************************************************************
**
*F  CopyObjDatObj( <obj>, <mut> ) . . . . . . . . . . . . . . . copy a datobj
*/
libGAP_Obj libGAP_CopyObjDatObj (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* copy, result                    */
    libGAP_Obj                 tmp;            /* temporary variable              */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Int               * src;
    libGAP_Int               * dst;

    /* don't change immutable objects                                      */
    if ( ! libGAP_IS_MUTABLE_OBJ(obj) ) {
        return obj;
    }

    /* if the object is not copyable return                                */
    if ( ! libGAP_IS_COPYABLE_OBJ(obj) ) {
        libGAP_ErrorQuit("Panic: encountered mutable, non-copyable object",0L,0L);
        return obj;
    }

    /* make a copy                                                         */
    if ( mut ) {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(obj), libGAP_SIZE_OBJ(obj) );
        libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(obj)[0];
    }
    else {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(obj), libGAP_SIZE_OBJ(obj) );
        libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(obj)[0];
        libGAP_CALL_2ARGS( libGAP_RESET_FILTER_OBJ, copy, libGAP_IsMutableObjFilt );
    }

    /* leave a forwarding pointer                                          */
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( tmp, 2 );
    libGAP_SET_ELM_PLIST( tmp, 1, libGAP_ADDR_OBJ(obj)[0] );
    libGAP_SET_ELM_PLIST( tmp, 2, copy );
    libGAP_ADDR_OBJ(obj)[0] = tmp;
    libGAP_CHANGED_BAG(obj);

    /* now it is copied                                                    */
    libGAP_RetypeBag( obj, libGAP_TNUM_OBJ(obj) + libGAP_COPYING );

    /* copy the subvalues                                                  */
    src = (libGAP_Int*)( libGAP_ADDR_OBJ(obj) + 1 );
    dst = (libGAP_Int*)( libGAP_ADDR_OBJ(copy) + 1 );
    i   = (libGAP_SIZE_OBJ(obj)-sizeof(libGAP_Obj)+sizeof(libGAP_Int)-1) / sizeof(libGAP_Int);
    for ( ;  0 < i;  i--, src++, dst++ ) {
        *dst = *src;
    }
    libGAP_CHANGED_BAG(copy);

    /* return the copy                                                     */
    return copy;
}


/****************************************************************************
**
*F  CleanObjDatObj( <obj> ) . . . . . . . . . . . . . . . . .  clean a datobj
*/
void libGAP_CleanObjDatObj (
    libGAP_Obj                 obj )
{
}


/****************************************************************************
**
*F  CopyObjDatObjCopy( <obj>, <mut> ) . . . . . . . . . .  copy a datobj copy
*/
libGAP_Obj libGAP_CopyObjDatObjCopy (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    return libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 2 );
}


/****************************************************************************
**
*F  CleanObjDatObjCopy( <obj> ) . . . . . . . . . . . . . clean a datobj copy
*/
void libGAP_CleanObjDatObjCopy (
    libGAP_Obj                 obj )
{
    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(obj)[0] = libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 1 );
    libGAP_CHANGED_BAG(obj);

    /* now it is cleaned                                                   */
    libGAP_RetypeBag( obj, libGAP_TNUM_OBJ(obj) - libGAP_COPYING );
}


/****************************************************************************
**

*F  ImmutableCopyObjHandler( <self>, <obj> )  . . . . immutable copy of <obj>
*/
libGAP_Obj libGAP_ImmutableCopyObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_CopyObj( obj, 0 );
}


/****************************************************************************
**
*F  MutableCopyObjHandler( <self>, <obj> )  . . . . . . mutable copy of <obj>
*/
libGAP_Obj libGAP_MutableCopyObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_CopyObj( obj, 1 );
}

/****************************************************************************
**
*F  MakeImmutable( <obj> . . . . . . . . . . make an object immutable inplace
**
**  Mark an object and all subobjects immutable in-place.
**  May cause confusion if there are shared subobjects
**
*/

libGAP_Obj libGAP_PostMakeImmutableOp = 0;

void (*libGAP_MakeImmutableObjFuncs[libGAP_LAST_REAL_TNUM+1])( libGAP_Obj );


void libGAP_MakeImmutable( libGAP_Obj obj )
{
  if (libGAP_IS_MUTABLE_OBJ( obj ))
    {
      (*(libGAP_MakeImmutableObjFuncs[libGAP_TNUM_OBJ(obj)]))(obj);
    }
}

void libGAP_MakeImmutableError( libGAP_Obj obj)
{
  libGAP_ErrorQuit("No make immutable function installed for a %s",
	    (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L);
}



void libGAP_MakeImmutableComObj( libGAP_Obj obj)
{
/*UInt i; */
  /* 
  for ( i = 1; i < SIZE_OBJ(obj)/sizeof(Obj); i += 2 ) {
       MakeImmutable(ADDR_OBJ(obj)[i]);
       CHANGED_BAG(obj);
    }
  */
  libGAP_CALL_2ARGS( libGAP_RESET_FILTER_OBJ, obj, libGAP_IsMutableObjFilt );
  libGAP_CALL_1ARGS( libGAP_PostMakeImmutableOp, obj);
}

void libGAP_MakeImmutablePosObj( libGAP_Obj obj)
{
/*UInt i; */
  /* 
  for ( i = 1; i < SIZE_OBJ(obj)/sizeof(Obj); i += 1 ) {
       MakeImmutable(ADDR_OBJ(obj)[i]);
       CHANGED_BAG(obj);
    }
  */
  libGAP_CALL_2ARGS( libGAP_RESET_FILTER_OBJ, obj, libGAP_IsMutableObjFilt );
  libGAP_CALL_1ARGS( libGAP_PostMakeImmutableOp, obj);
  
}

void libGAP_MakeImmutableDatObj( libGAP_Obj obj)
{
  libGAP_CALL_2ARGS( libGAP_RESET_FILTER_OBJ, obj, libGAP_IsMutableObjFilt );
}

libGAP_Obj libGAP_FuncMakeImmutable( libGAP_Obj self, libGAP_Obj obj)
{
  libGAP_MakeImmutable(obj);
  return obj;
}



/****************************************************************************
**
*F  PrintObj( <obj> ) . . . . . . . . . . . . . . . . . . . . print an object
**
**  'PrintObj' prints the object <obj>.
*/
libGAP_Obj libGAP_PrintObjThis;

libGAP_Int libGAP_PrintObjIndex;

libGAP_Int libGAP_PrintObjFull;

libGAP_Int libGAP_PrintObjDepth;

#define libGAP_MAXPRINTDEPTH 1024L
libGAP_Obj libGAP_PrintObjThiss [libGAP_MAXPRINTDEPTH];

libGAP_Int libGAP_PrintObjIndices [libGAP_MAXPRINTDEPTH];

/****************************************************************************
**
*F  ViewObj( <obj> ) . . . . . . . . . . . . . . . . . . . . view an object
**
**  'ViewObj' views the object <obj>.
**
**  ViewObj shares all the assocaited variables with PrintObj, so that
**  recursion works nicely.
*/

/* These macros and the in-line function are used to keep track of
which objects are already being prted or viewed to trigger the use of
~ when needed. The names (and the no-op macros) are hangovers from an
earlier version in which the bags themselves were marked, rather than
checking the list. This had to be changed to ensure that the method
selection process (for ViewObj) only sees "pristine" bags"   SL 9/4/98 */

#define libGAP_IS_MARKABLE(obj)    (libGAP_FIRST_RECORD_TNUM <= libGAP_TNUM_OBJ(obj) \
                            && libGAP_TNUM_OBJ(obj) <= libGAP_LAST_LIST_TNUM)

static inline libGAP_UInt libGAP_IS_MARKED( libGAP_Obj obj )
{
  libGAP_UInt i;
  if (!libGAP_IS_MARKABLE(obj))
    return 0;
  for (i = 0; i < libGAP_TLS(libGAP_PrintObjDepth)-1; i++)
    if (libGAP_TLS(libGAP_PrintObjThiss)[i] == obj)
      return 1;
  return 0;
}
     
#define libGAP_MARK(obj)     do {} while (0)
#define libGAP_UNMARK(obj)   do {} while (0)

/* This variable is used to allow a ViewObj method to call PrintObj on
   the same object without triggering use of ~ */

static libGAP_UInt libGAP_LastPV = 0; /* This variable contains one of the values
			   0, 1 and 2 according to whether (1) there
			   is no dynamically enc losing call to
			   PrintObj or ViewObj still open (0), or the
			   innermost such is Print (1) or View (2) */

void            libGAP_PrintObj (
    libGAP_Obj                 obj )
{
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt                lastPV;        /* save LastPV */
    libGAP_UInt                fromview;      /* non-zero when we were called
				        from viewObj of the SAME object */

    /* check for interrupts                                                */
    if ( libGAP_SyIsIntr() ) {
        i = libGAP_TLS(libGAP_PrintObjDepth);
        libGAP_Pr( "%c%c", (libGAP_Int)'\03', (libGAP_Int)'\04' );
        libGAP_ErrorReturnVoid(
            "user interrupt while printing",
            0L, 0L,
            "you can 'return;'" );
        libGAP_TLS(libGAP_PrintObjDepth) = i;
    }

    /* First check if <obj> is actually the current object being Viewed
       Since ViewObj(<obj>) may result in a call to PrintObj(<obj>) */

    lastPV = libGAP_LastPV;
    libGAP_LastPV = 1;
    fromview = (lastPV == 2) && (obj == libGAP_TLS(libGAP_PrintObjThis));

    /* if <obj> is a subobject, then mark and remember the superobject
       unless ViewObj has done that job already */
    
    if ( !fromview  && 0 < libGAP_TLS(libGAP_PrintObjDepth) ) {
        if ( libGAP_IS_MARKABLE(libGAP_TLS(libGAP_PrintObjThis)) )  libGAP_MARK( libGAP_TLS(libGAP_PrintObjThis) );
        libGAP_TLS(libGAP_PrintObjThiss)[libGAP_TLS(libGAP_PrintObjDepth)-1]   = libGAP_TLS(libGAP_PrintObjThis);
        libGAP_TLS(libGAP_PrintObjIndices)[libGAP_TLS(libGAP_PrintObjDepth)-1] = libGAP_TLS(libGAP_PrintObjIndex);
    }

    /* handle the <obj>                                                    */
    if (!fromview)
      {
	libGAP_TLS(libGAP_PrintObjDepth) += 1;
	libGAP_TLS(libGAP_PrintObjThis)   = obj;
	libGAP_TLS(libGAP_PrintObjIndex)  = 0;
      }

    /* dispatch to the appropriate printing function                       */
    if ( (! libGAP_IS_MARKED( libGAP_TLS(libGAP_PrintObjThis) )) ) {
      if (libGAP_TLS(libGAP_PrintObjDepth) < libGAP_MAXPRINTDEPTH) {
        (*libGAP_PrintObjFuncs[ libGAP_TNUM_OBJ(libGAP_TLS(libGAP_PrintObjThis)) ])( libGAP_TLS(libGAP_PrintObjThis) );
      }
      else {
        /* don't recurse if depth too high */
        libGAP_Pr("\nprinting stopped, too many recursion levels!\n", 0L, 0L);
      }
    }

    /* or print the path                                                   */
    else {
        libGAP_Pr( "~", 0L, 0L );
        for ( i = 0; libGAP_TLS(libGAP_PrintObjThis) != libGAP_TLS(libGAP_PrintObjThiss)[i]; i++ ) {
            (*libGAP_PrintPathFuncs[ libGAP_TNUM_OBJ(libGAP_TLS(libGAP_PrintObjThiss)[i])])
                ( libGAP_TLS(libGAP_PrintObjThiss)[i], libGAP_TLS(libGAP_PrintObjIndices)[i] );
        }
    }


    /* done with <obj>                                                     */
    if (!fromview)
      {
	libGAP_TLS(libGAP_PrintObjDepth) -= 1;
	
	/* if <obj> is a subobject, then restore and unmark the superobject    */
	if ( 0 < libGAP_TLS(libGAP_PrintObjDepth) ) {
	  libGAP_TLS(libGAP_PrintObjThis)  = libGAP_TLS(libGAP_PrintObjThiss)[libGAP_TLS(libGAP_PrintObjDepth)-1];
	  libGAP_TLS(libGAP_PrintObjIndex) = libGAP_TLS(libGAP_PrintObjIndices)[libGAP_TLS(libGAP_PrintObjDepth)-1];
	  if ( libGAP_IS_MARKED(libGAP_TLS(libGAP_PrintObjThis)) )  libGAP_UNMARK( libGAP_TLS(libGAP_PrintObjThis) );
	}
      }
    libGAP_LastPV = lastPV;

}



/****************************************************************************
**
*V  PrintObjFuncs[<type>] . . . . . . . .  printer for objects of type <type>
**
**  'PrintObjFuncs' is  the dispatch  table that  contains  for every type of
**  objects a pointer to the printer for objects of this  type.  The  printer
**  is the function '<func>(<obj>)' that should be called to print the object
**  <obj> of this type.
*/
void (* libGAP_PrintObjFuncs [ libGAP_LAST_REAL_TNUM  +1 ])( libGAP_Obj obj );


/****************************************************************************
**
*F  PrintObjObject( <obj> ) . . . . . . . . . . . . . . . . . print an object
*/
libGAP_Obj libGAP_PrintObjOper;

void libGAP_PrintObjObject (
    libGAP_Obj                 obj )
{
    libGAP_DoOperation1Args( libGAP_PrintObjOper, obj );
}


/****************************************************************************
**
*F  PrintObjHandler( <self>, <obj> )  . . . . . . . .  handler for 'PrintObj'
*/
libGAP_Obj libGAP_PrintObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_PrintObj( obj );
    return 0L;
}


libGAP_Obj libGAP_FuncSET_PRINT_OBJ_INDEX (libGAP_Obj self, libGAP_Obj ind)
{
  if (libGAP_IS_INTOBJ(ind))
    libGAP_TLS(libGAP_PrintObjIndex) = libGAP_INT_INTOBJ(ind);
  return 0;
}


/****************************************************************************
**
*F  ViewObj(<obj> )  . . .. . . . . . . . . . . . . . . . . .  view an object
**
**  This should really be merged with PrintObj
*/

libGAP_Obj libGAP_ViewObjOper;

void            libGAP_ViewObj (
    libGAP_Obj                 obj )
{
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt                lastPV;

    /* No check for interrupts here, viewing should not take so long that
       it is necessary */

    lastPV = libGAP_LastPV;
    libGAP_LastPV = 2;
    
    /* if <obj> is a subobject, then mark and remember the superobject     */
    if ( 0 < libGAP_TLS(libGAP_PrintObjDepth) ) {
        if ( libGAP_IS_MARKABLE(libGAP_TLS(libGAP_PrintObjThis)) )  libGAP_MARK( libGAP_TLS(libGAP_PrintObjThis) );
        libGAP_TLS(libGAP_PrintObjThiss)[libGAP_TLS(libGAP_PrintObjDepth)-1]   = libGAP_TLS(libGAP_PrintObjThis);
        libGAP_TLS(libGAP_PrintObjIndices)[libGAP_TLS(libGAP_PrintObjDepth)-1] =  libGAP_TLS(libGAP_PrintObjIndex);
    }

    /* handle the <obj>                                                    */
    libGAP_TLS(libGAP_PrintObjDepth) += 1;
    libGAP_TLS(libGAP_PrintObjThis)   = obj;
    libGAP_TLS(libGAP_PrintObjIndex)  = 0;

    /* dispatch to the appropriate viewing function                       */

    if ( ! libGAP_IS_MARKED( libGAP_TLS(libGAP_PrintObjThis) ) ) {
      if (libGAP_TLS(libGAP_PrintObjDepth) < libGAP_MAXPRINTDEPTH) {
        libGAP_DoOperation1Args( libGAP_ViewObjOper, obj );
      }
      else {
        /* don't recurse any more */
        libGAP_Pr("\nviewing stopped, too many recursion levels!\n", 0L, 0L);
      }
    }

    /* or view the path                                                   */
    else {
        libGAP_Pr( "~", 0L, 0L );
        for ( i = 0; libGAP_TLS(libGAP_PrintObjThis) != libGAP_TLS(libGAP_PrintObjThiss)[i]; i++ ) {
            (*libGAP_PrintPathFuncs[ libGAP_TNUM_OBJ(libGAP_TLS(libGAP_PrintObjThiss)[i]) ])
                ( libGAP_TLS(libGAP_PrintObjThiss)[i], libGAP_TLS(libGAP_PrintObjIndices)[i] );
        }
    }

    /* done with <obj>                                                     */
    libGAP_TLS(libGAP_PrintObjDepth) -= 1;

    /* if <obj> is a subobject, then restore and unmark the superobject    */
    if ( 0 < libGAP_TLS(libGAP_PrintObjDepth) ) {
        libGAP_TLS(libGAP_PrintObjThis)  = libGAP_TLS(libGAP_PrintObjThiss)[libGAP_TLS(libGAP_PrintObjDepth)-1];
        libGAP_TLS(libGAP_PrintObjIndex) = libGAP_TLS(libGAP_PrintObjIndices)[libGAP_TLS(libGAP_PrintObjDepth)-1];
        if ( libGAP_IS_MARKED(libGAP_TLS(libGAP_PrintObjThis)) )  libGAP_UNMARK( libGAP_TLS(libGAP_PrintObjThis) );
    }

    libGAP_LastPV = lastPV;
}


/****************************************************************************
**
*F  FuncViewObj( <self>, <obj> )  . . . . . . . .  handler for 'ViewObj'
*/
libGAP_Obj libGAP_FuncViewObj (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_ViewObj( obj );
    return 0L;
}


/****************************************************************************
**
*V  PrintPathFuncs[<type>]  . . . . . . printer for subobjects of type <type>
**
**  'PrintPathFuncs'  is   the   dispatch table  that     contains for  every
**  appropriate type of objects a pointer to  the path printer for objects of
**  that type.  The path  printer is the function '<func>(<obj>,<indx>)' that
**  should be  called  to print  the  selector   that selects  the  <indx>-th
**  subobject of the object <obj> of this type.
*/
void (* libGAP_PrintPathFuncs [ libGAP_LAST_REAL_TNUM /* +PRINTING */+1 ])( libGAP_Obj obj, libGAP_Int indx );

void libGAP_PrintPathError (
    libGAP_Obj                 obj,
    libGAP_Int                 indx )
{
    libGAP_ErrorQuit(
        "Panic: tried to print a path of unknown type '%d'",
        (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
}


/****************************************************************************
**

*F  TypeComObj( <obj> ) . . . . . . . . . . function version of 'TYPE_COMOBJ'
*/
libGAP_Obj             libGAP_TypeComObj (
    libGAP_Obj                 obj )
{
    return libGAP_TYPE_COMOBJ( obj );
}


/*****************************************************************************
**
*F  IS_COMOBJ_Hander( <self>, <obj> ) . . . . . . . . handler for 'IS_COMOBJ'
*/
libGAP_Obj             libGAP_IS_COMOBJ_Handler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_TNUM_OBJ(obj) == libGAP_T_COMOBJ ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  SET_TYPE_COMOBJ_Handler( <self>, <obj>, <type> ) . . .  'SET_TYPE_COMOBJ'
*/
libGAP_Obj libGAP_SET_TYPE_COMOBJ_Handler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 type )
{
    libGAP_TYPE_COMOBJ( obj ) = type;
    libGAP_RetypeBag( obj, libGAP_T_COMOBJ );
    libGAP_CHANGED_BAG( obj );
    return obj;
}


/****************************************************************************
**

*F  TypePosObj( <obj> ) . . . . . . . . . . function version of 'TYPE_POSOBJ'
*/
libGAP_Obj libGAP_TypePosObj (
    libGAP_Obj                 obj )
{
    return libGAP_TYPE_POSOBJ( obj );
}


/****************************************************************************
**
*F  IS_POSOBJ_Handler( <self>, <obj> )  . . . . . . . handler for 'IS_POSOBJ'
*/
libGAP_Obj libGAP_IS_POSOBJ_Handler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_TNUM_OBJ(obj) == libGAP_T_POSOBJ ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  SET_TYPE_POSOBJ_Handler( <self>, <obj>, <type> )  . . .  'SET_TYPE_POSOB'
*/
libGAP_Obj libGAP_SET_TYPE_POSOBJ_Handler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 type )
{
    libGAP_TYPE_POSOBJ( obj ) = type;
    libGAP_RetypeBag( obj, libGAP_T_POSOBJ );
    libGAP_CHANGED_BAG( obj );
    return obj;
}


/****************************************************************************
**
*F  LEN_POSOBJ_Handler( <self>, <obj> ) . . . . . .  handler for 'LEN_POSOBJ'
*/
libGAP_Obj libGAP_LEN_POSOBJ_Handler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_INTOBJ_INT( libGAP_SIZE_OBJ(obj) / sizeof(libGAP_Obj) - 1 );
}


/****************************************************************************
**

*F  TypeDatObj( <obj> ) . . . . . . . . . . function version of 'TYPE_DATOBJ'
*/
libGAP_Obj             libGAP_TypeDatObj (
    libGAP_Obj                 obj )
{
    return libGAP_TYPE_DATOBJ( obj );
}


/*****************************************************************************
**
*F  IS_DATOBJ_Hander( <self>, <obj> ) . . . . . . . . handler for 'IS_DATOBJ'
*/
libGAP_Obj             libGAP_IS_DATOBJ_Handler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_TNUM_OBJ(obj) == libGAP_T_DATOBJ ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  SET_TYPE_DATOBJ_Handler( <self>, <obj>, <type> ) . . .  'SET_TYPE_DATOBJ'
*/
libGAP_Obj libGAP_SET_TYPE_DATOBJ_Handler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 type )
{
    libGAP_TYPE_DATOBJ( obj ) = type;
    libGAP_RetypeBag( obj, libGAP_T_DATOBJ );
    libGAP_CHANGED_BAG( obj );
    return obj;
}


/****************************************************************************
**

*F  IsIdenticalHandler( <self>, <obj1>, <obj2> )  . . . . .  handler for '=='
**
**  'IsIdenticalHandler' implements 'IsIdentical'
*/
libGAP_Obj libGAP_IsIdenticalHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj1,
    libGAP_Obj                 obj2 )
{
    return (obj1 == obj2 ? libGAP_True : libGAP_False);
}

/****************************************************************************
**
*V  SaveObjFuncs (<type>) . . . . . . . . . . . . . functions to save objects
**
** 'SaveObjFuncs' is the dispatch table that  contains, for every type
**  of  objects, a pointer to the saving function for objects of that type
**  These should not handle the file directly, but should work via the
**  functions 'SaveSubObj', 'SaveUInt<n>' (<n> = 1,2,4 or 8), and others
**  to be determined. Their role is to identify the C types of the various
**  parts of the bag, and perhaps to leave out some information that does
**  not need to be saved. By the time this function is called, the bag
**  size and type have already been saved
**  No saving function may allocate any bag
*/

void (*libGAP_SaveObjFuncs[256]) ( libGAP_Obj obj );

void libGAP_SaveObjError( libGAP_Obj obj )
{
  libGAP_ErrorQuit(
            "Panic: tried to save an object of unknown type '%d'",
            (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
}
    
                   
     
     
/****************************************************************************
**
*V  LoadObjFuncs (<type>) . . . . . . . . . . . . . functions to load objects
**
** 'LoadObjFuncs' is the dispatch table that  contains, for every type
**  of  objects, a pointer to the loading function for objects of that type
**  These should not handle the file directly, but should work via the
**  functions 'LoadObjRef', 'LoadUInt<n>' (<n> = 1,2,4 or 8), and others
**  to be determined. Their role is to reinstall the information in the bag
**  and reconstruct anything that was left out. By the time this function is
**  called, the bag size and type have already been loaded and the bag argument
**  contains the bag in question
**  No loading function may allocate any bag
*/

void (*libGAP_LoadObjFuncs[256]) ( libGAP_Obj obj );

void libGAP_LoadObjError( libGAP_Obj obj )
{
  libGAP_ErrorQuit(
            "Panic: tried to load an object of unknown type '%d'",
            (libGAP_Int)libGAP_TNUM_OBJ(obj), 0L );
}

/****************************************************************************
**
*F  SaveComObj( Obj comobj)
**
*/

void libGAP_SaveComObj( libGAP_Obj comobj)
{
  libGAP_UInt len,i;
  libGAP_SaveSubObj(libGAP_TYPE_COMOBJ( comobj ));
  len = libGAP_LEN_PREC(comobj);
  libGAP_SaveUInt(len);
  for (i = 1; i <= len; i++)
    {
      libGAP_SaveUInt(libGAP_GET_RNAM_PREC(comobj, i));
      libGAP_SaveSubObj(libGAP_GET_ELM_PREC(comobj, i));
    }
  return;
}

/****************************************************************************
**
*F  SavePosObj( Obj posobj)
**
*/

void libGAP_SavePosObj( libGAP_Obj posobj)
{
  libGAP_UInt len,i;
  libGAP_SaveSubObj(libGAP_TYPE_POSOBJ( posobj ));
  len = (libGAP_SIZE_OBJ(posobj)/sizeof(libGAP_Obj) - 1);
  for (i = 1; i <= len; i++)
    {
      libGAP_SaveSubObj(libGAP_ADDR_OBJ(posobj)[i]);
    }
  return;
}

/****************************************************************************
**
*F  SaveDatObj( Obj datobj)
**
**  Here we lose endianness protection, because we don't know if this is really
**  UInts, or if it might be smaller data
*/

void libGAP_SaveDatObj( libGAP_Obj datobj)
{
  libGAP_UInt len,i;
  libGAP_UInt *ptr;
  libGAP_SaveSubObj(libGAP_TYPE_DATOBJ( datobj ));
  len = ((libGAP_SIZE_OBJ(datobj)+sizeof(libGAP_UInt)-1)/sizeof(libGAP_UInt) - 1);
  ptr = (libGAP_UInt *)libGAP_ADDR_OBJ(datobj)+1;
  for (i = 1; i <= len; i++)
    {
      libGAP_SaveUInt(*ptr++);
    }
  return;
}

/****************************************************************************
**
*F  LoadComObj( Obj comobj)
**
*/

void libGAP_LoadComObj( libGAP_Obj comobj)
{
  libGAP_UInt len,i;
  libGAP_TYPE_COMOBJ( comobj) = libGAP_LoadSubObj( );
  len = libGAP_LoadUInt();
  libGAP_SET_LEN_PREC(comobj,len);
  for (i = 1; i <= len; i++)
    {
      libGAP_SET_RNAM_PREC(comobj, i, libGAP_LoadUInt());
      libGAP_SET_ELM_PREC(comobj, i, libGAP_LoadSubObj());
    }
  return;
}

/****************************************************************************
**
*F  LoadPosObj( Obj posobj)
**
*/

void libGAP_LoadPosObj( libGAP_Obj posobj)
{
  libGAP_UInt len,i;
  libGAP_TYPE_POSOBJ( posobj ) = libGAP_LoadSubObj();
  len = (libGAP_SIZE_OBJ(posobj)/sizeof(libGAP_Obj) - 1);
  for (i = 1; i <= len; i++)
    {
      libGAP_ADDR_OBJ(posobj)[i] = libGAP_LoadSubObj();
    }
  return;
}

/****************************************************************************
**
*F  LoadDatObj( Obj datobj)
**
**  Here we lose endianness protection, because we don't know if this is really
**  UInts, or if it might be smaller data
*/

void libGAP_LoadDatObj( libGAP_Obj datobj)
{
  libGAP_UInt len,i;
  libGAP_UInt *ptr;
  libGAP_TYPE_DATOBJ( datobj ) = libGAP_LoadSubObj();
  len = ((libGAP_SIZE_OBJ(datobj)+sizeof(libGAP_UInt)-1)/sizeof(libGAP_UInt) - 1);
  ptr = (libGAP_UInt *)libGAP_ADDR_OBJ(datobj)+1;
  for (i = 1; i <= len; i++)
    {
      *ptr ++ = libGAP_LoadUInt();
    }
  return;
}


/****************************************************************************
**

*F * * * * * * * *  GAP functions for "to be defined" objects * * * * * * * *
*/


/****************************************************************************
**

*F  FuncCLONE_OBJ( <self>, <dst>, <src> ) . . . . . . .  clone <src> to <dst>
**
**  `CLONE_OBJ' clones  the source  <src> into  <dst>.  It  is not allowed to
**  clone small integers or finite field elements.
**
**  If <src> is a constant, than a "shallow" copy, that is to say, a bit-copy
**  of the bag of <src>  is created.  If <src>  is mutable than a "structural
**  copy is created, which is then in turn "shallow" cloned into <dst>.
**
**  WARNING: at the moment the functions breaks on cloning `[1,~]'.  This can
**  be fixed if necessary.
*/
libGAP_Obj libGAP_IsToBeDefinedObj;

static libGAP_Obj libGAP_REREADING;

libGAP_Obj libGAP_FuncCLONE_OBJ (
    libGAP_Obj             self,
    libGAP_Obj             dst,
    libGAP_Obj             src )
{
    libGAP_Obj *           psrc;
    libGAP_Obj *           pdst;
    libGAP_Int             i;

    /* check <src>                                                         */
    if ( libGAP_IS_INTOBJ(src) ) {
        libGAP_ErrorReturnVoid( "small integers cannot be cloned", 0, 0,
                         "you can 'return;' to skip the cloning" );
        return 0;
    }
    if ( libGAP_IS_FFE(src) ) {
        libGAP_ErrorReturnVoid( "finite field elements cannot be cloned", 0, 0,
                         "you can 'return;' to skip the cloning" );
        return 0;
    }

    /* check <dst>                                                         
    if ( (REREADING != True) &&
	 (CALL_1ARGS( IsToBeDefinedObj, dst ) != True) ) {
        ErrorReturnVoid( "the destination must be `to-be-defined' (not a %s)",
                         (Int)TNAM_OBJ(dst), 0,
                         "you can 'return;'" );
        return 0;
    }

    */
    
    /* if object is mutable, produce a structural copy                     */
    if ( libGAP_IS_MUTABLE_OBJ(src) ) {
        src = libGAP_CopyObj( src, 1 );
    }

    /* now shallow clone the object                                        */
    libGAP_ResizeBag( dst, libGAP_SIZE_OBJ(src) );
    libGAP_RetypeBag( dst, libGAP_TNUM_OBJ(src) );
    psrc = libGAP_ADDR_OBJ(src);
    pdst = libGAP_ADDR_OBJ(dst);
    for ( i = (libGAP_SIZE_OBJ(src)+sizeof(libGAP_Obj) - 1)/sizeof(libGAP_Obj);  0 < i;  i-- ) {
        *pdst++ = *psrc++;
    }
    libGAP_CHANGED_BAG(dst);

    return 0;
}

/****************************************************************************
**

*F  FuncSWITCH_OBJ( <self>, <obj1>, <obj2> ) . . .  switch <obj1> and <obj2>
**
**  `SWITCH_OBJ' exchanges the objects referenced by its two arguments.  It
**   is not allowed to switch clone small integers or finite field elements.
**
**   This is inspired by the Smalltalk 'become:' operation.
*/

libGAP_Obj libGAP_FuncSWITCH_OBJ(libGAP_Obj self, libGAP_Obj obj1, libGAP_Obj obj2) {
    libGAP_Obj *ptr1, *ptr2;

    if ( libGAP_IS_INTOBJ(obj1) || libGAP_IS_INTOBJ(obj2) ) {
        libGAP_ErrorReturnVoid( "small integer objects cannot be switched", 0, 0,
                         "you can 'return;' to leave them in place" );
        return 0;
    }
    if ( libGAP_IS_FFE(obj1) || libGAP_IS_FFE(obj2) ) {
        libGAP_ErrorReturnVoid( "finite field elements cannot be switched", 0, 0,
                         "you can 'return;' to leave them in place" );
        return 0;
    }
    ptr1 = libGAP_PTR_BAG(obj1);
    ptr2 = libGAP_PTR_BAG(obj2);
    libGAP_PTR_BAG(obj2) = ptr1;
    libGAP_PTR_BAG(obj1) = ptr2;
    libGAP_CHANGED_BAG(obj1);
    libGAP_CHANGED_BAG(obj2);
    return (libGAP_Obj) 0;
}


/****************************************************************************
**

*F  FuncFORCE_SWITCH_OBJ( <self>, <obj1>, <obj2> ) .  switch <obj1> and <obj2>
**
**  In GAP, FORCE_SWITCH_OBJ does the same thing as SWITCH_OBJ. In HPC_GAP
**  it allows public objects to be exchanged.
*/

libGAP_Obj libGAP_FuncFORCE_SWITCH_OBJ(libGAP_Obj self, libGAP_Obj obj1, libGAP_Obj obj2) {
    return libGAP_FuncSWITCH_OBJ(self, obj1, obj2);
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_MUTABLE_OBJ", "obj", &libGAP_IsMutableObjFilt,
      libGAP_IsMutableObjHandler, "src/objects.c:IS_MUTABLE_OBJ" },

    { "IS_COPYABLE_OBJ", "obj", &libGAP_IsCopyableObjFilt,
      libGAP_IsCopyableObjHandler, "src/objects.c:IS_COPYABLE_OBJ" },

    { 0 }

};


/****************************************************************************
**
*V  GVarOpers . . . . . . . . . . . . . . . . .  list of operations to export
*/
static libGAP_StructGVarOper libGAP_GVarOpers [] = {

    { "SHALLOW_COPY_OBJ", 1, "obj", &libGAP_ShallowCopyObjOper,
      libGAP_ShallowCopyObjHandler, "src/objects.c:SHALLOW_COPY_OBJ" },

    { "PRINT_OBJ", 1, "obj", &libGAP_PrintObjOper,
      libGAP_PrintObjHandler, "src/objects.c:PRINT_OBJ" },

    { "VIEW_OBJ", 1, "obj", &libGAP_ViewObjOper,
      libGAP_FuncViewObj, "src/objects.c:VIEW_OBJ" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "FAMILY_TYPE", 1, "type",
      libGAP_FamilyTypeHandler, "src/objects.c:FAMILY_TYPE" },

    { "TYPE_OBJ", 1, "obj",
      libGAP_TypeObjHandler, "src/objects.c:TYPE_OBJ" },

    { "FAMILY_OBJ", 1, "obj",
      libGAP_FamilyObjHandler, "src/objects.c:FAMILY_OBJ" },

    { "IMMUTABLE_COPY_OBJ", 1, "obj", 
      libGAP_ImmutableCopyObjHandler, "src/objects.c:IMMUTABLE_COPY_OBJ" },

    { "DEEP_COPY_OBJ", 1, "obj",
          libGAP_MutableCopyObjHandler, "src/objects.c:DEEP_COPY_OBJ" },

    { "IS_IDENTICAL_OBJ", 2, "obj1, obj2", 
      libGAP_IsIdenticalHandler, "src/objects.c:IS_IDENTICAL_OBJ" },

    { "IS_COMOBJ", 1, "obj",
      libGAP_IS_COMOBJ_Handler, "src/objects.c:IS_COMOBJ" },

    { "SET_TYPE_COMOBJ", 2, "obj, type",
      libGAP_SET_TYPE_COMOBJ_Handler, "src/objects.c:SET_TYPE_COMOBJ" },

    { "IS_POSOBJ", 1, "obj",
      libGAP_IS_POSOBJ_Handler, "src/objects.c:IS_POSOBJ" },
    
    { "SET_TYPE_POSOBJ", 2, "obj, type",
      libGAP_SET_TYPE_POSOBJ_Handler, "src/objects.c:SET_TYPE_POSOBJ" },
    
    { "LEN_POSOBJ", 1, "obj",
      libGAP_LEN_POSOBJ_Handler, "src/objects.c:LEN_POSOBJ" },
    
    { "IS_DATOBJ", 1, "obj",
      libGAP_IS_DATOBJ_Handler, "src/objects.c:IS_DATOBJ" },
    
    { "SET_TYPE_DATOBJ", 2, "obj, type",
      libGAP_SET_TYPE_DATOBJ_Handler, "src/objects.c:SET_TYPE_DATOBJ" },

    { "CLONE_OBJ", 2, "obj, dst, src",
      libGAP_FuncCLONE_OBJ, "src/objects.c:CLONE_OBJ" },

    { "SWITCH_OBJ", 2, "obj1, obj2",
      libGAP_FuncSWITCH_OBJ, "src/objects.c:SWITCH_OBJ" },

    { "FORCE_SWITCH_OBJ", 2, "obj1, obj2",
      libGAP_FuncFORCE_SWITCH_OBJ, "src/objects.c:FORCE_SWITCH_OBJ" },

    { "SET_PRINT_OBJ_INDEX", 1, "index",
      libGAP_FuncSET_PRINT_OBJ_INDEX, "src/objects.c:SET_PRINT_OBJ_INDEX" },

    { "MakeImmutable", 1, "obj",
      libGAP_FuncMakeImmutable, "src/objects.c:MakeImmutable" },
    
    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int                 t;              /* loop variable                   */

    /* install the marking methods                                         */
    libGAP_InfoBags[         libGAP_T_COMOBJ          ].name = "object (component)";
    libGAP_InitMarkFuncBags( libGAP_T_COMOBJ          , libGAP_MarkAllSubBags  );
    libGAP_InfoBags[         libGAP_T_COMOBJ +libGAP_COPYING ].name = "object (component,copied)";
    libGAP_InitMarkFuncBags( libGAP_T_COMOBJ +libGAP_COPYING , libGAP_MarkAllSubBags  );
    libGAP_InfoBags[         libGAP_T_POSOBJ          ].name = "object (positional)";
    libGAP_InitMarkFuncBags( libGAP_T_POSOBJ          , libGAP_MarkAllSubBags  );
    libGAP_InfoBags[         libGAP_T_POSOBJ +libGAP_COPYING ].name = "object (positional,copied)";
    libGAP_InitMarkFuncBags( libGAP_T_POSOBJ +libGAP_COPYING , libGAP_MarkAllSubBags  );
    libGAP_InfoBags[         libGAP_T_DATOBJ          ].name = "object (data)";
    libGAP_InitMarkFuncBags( libGAP_T_DATOBJ          , libGAP_MarkOneSubBags  );
    libGAP_InfoBags[         libGAP_T_DATOBJ +libGAP_COPYING ].name = "object (data,copied)";
    libGAP_InitMarkFuncBags( libGAP_T_DATOBJ +libGAP_COPYING , libGAP_MarkOneSubBags  );

    for ( t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ ) {
        libGAP_TypeObjFuncs[ t ] = libGAP_TypeObjError;
    }

    libGAP_TypeObjFuncs[ libGAP_T_COMOBJ ] = libGAP_TypeComObj;
    libGAP_TypeObjFuncs[ libGAP_T_POSOBJ ] = libGAP_TypePosObj;
    libGAP_TypeObjFuncs[ libGAP_T_DATOBJ ] = libGAP_TypeDatObj;

    /* SPARE TNUMs install placeholder entries for easier
       debugging. Packages that use these should overwrite the entries */
    libGAP_InfoBags[libGAP_T_SPARE1].name = "spare tnum 1 (overwrite this)";
    libGAP_InfoBags[libGAP_T_SPARE2].name = "spare tnum 2 (overwrite this)";
    libGAP_InfoBags[libGAP_T_SPARE3].name = "spare tnum 3 (overwrite this)";
    libGAP_InfoBags[libGAP_T_SPARE4].name = "spare tnum 4 (overwrite this)";

    /* functions for 'to-be-defined' objects                               */
    libGAP_ImportFuncFromLibrary( "IsToBeDefinedObj", &libGAP_IsToBeDefinedObj );
    libGAP_ImportFuncFromLibrary( "PostMakeImmutable", &libGAP_PostMakeImmutableOp );
    libGAP_ImportGVarFromLibrary( "REREADING", &libGAP_REREADING );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrOpersFromTable( libGAP_GVarOpers );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* make and install the 'IS_MUTABLE_OBJ' filter                        */
    for ( t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ )
        libGAP_IsMutableObjFuncs[ t ] = libGAP_IsMutableObjError;
    for ( t = libGAP_FIRST_CONSTANT_TNUM; t <= libGAP_LAST_CONSTANT_TNUM; t++ )
        libGAP_IsMutableObjFuncs[ t ] = libGAP_IsMutableObjNot;
    for ( t = libGAP_FIRST_EXTERNAL_TNUM; t <= libGAP_LAST_EXTERNAL_TNUM; t++ )
        libGAP_IsMutableObjFuncs[ t ] = libGAP_IsMutableObjObject;

    /* make and install the 'IS_COPYABLE_OBJ' filter                       */
    for ( t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ )
        libGAP_IsCopyableObjFuncs[ t ] = libGAP_IsCopyableObjError;
    for ( t = libGAP_FIRST_CONSTANT_TNUM; t <= libGAP_LAST_CONSTANT_TNUM; t++ )
        libGAP_IsCopyableObjFuncs[ t ] = libGAP_IsCopyableObjNot;
    for ( t = libGAP_FIRST_EXTERNAL_TNUM; t <= libGAP_LAST_EXTERNAL_TNUM; t++ )
        libGAP_IsCopyableObjFuncs[ t ] = libGAP_IsCopyableObjObject;

    /* make and install the 'SHALLOW_COPY_OBJ' operation                   */
    for ( t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ )
        libGAP_ShallowCopyObjFuncs[ t ] = libGAP_ShallowCopyObjError;
    for ( t = libGAP_FIRST_CONSTANT_TNUM; t <= libGAP_LAST_CONSTANT_TNUM; t++ )
        libGAP_ShallowCopyObjFuncs[ t ] = libGAP_ShallowCopyObjConstant;
    for ( t = libGAP_FIRST_RECORD_TNUM; t <= libGAP_LAST_RECORD_TNUM; t++ )
        libGAP_ShallowCopyObjFuncs[ t ] = libGAP_ShallowCopyObjDefault;
    for ( t = libGAP_FIRST_LIST_TNUM; t <= libGAP_LAST_LIST_TNUM; t++ )
        libGAP_ShallowCopyObjFuncs[ t ] = libGAP_ShallowCopyObjDefault;
    for ( t = libGAP_FIRST_EXTERNAL_TNUM; t <= libGAP_LAST_EXTERNAL_TNUM; t++ )
        libGAP_ShallowCopyObjFuncs[ t ] = libGAP_ShallowCopyObjObject;

    /* make and install the 'COPY_OBJ' function                            */
    for ( t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ ) {
        libGAP_CopyObjFuncs [ t ] = libGAP_CopyObjError;
        libGAP_CleanObjFuncs[ t ] = libGAP_CleanObjError;
    }
    for ( t = libGAP_FIRST_CONSTANT_TNUM; t <= libGAP_LAST_CONSTANT_TNUM; t++ ) {
        libGAP_CopyObjFuncs [ t ] = libGAP_CopyObjConstant;
        libGAP_CleanObjFuncs[ t ] = libGAP_CleanObjConstant;
    }
    libGAP_CopyObjFuncs[  libGAP_T_POSOBJ           ] = libGAP_CopyObjPosObj;
    libGAP_CopyObjFuncs[  libGAP_T_POSOBJ + libGAP_COPYING ] = libGAP_CopyObjPosObjCopy;
    libGAP_CleanObjFuncs[ libGAP_T_POSOBJ           ] = libGAP_CleanObjPosObj;
    libGAP_CleanObjFuncs[ libGAP_T_POSOBJ + libGAP_COPYING ] = libGAP_CleanObjPosObjCopy;
    libGAP_CopyObjFuncs[  libGAP_T_COMOBJ           ] = libGAP_CopyObjComObj;
    libGAP_CopyObjFuncs[  libGAP_T_COMOBJ + libGAP_COPYING ] = libGAP_CopyObjComObjCopy;
    libGAP_CleanObjFuncs[ libGAP_T_COMOBJ           ] = libGAP_CleanObjComObj;
    libGAP_CleanObjFuncs[ libGAP_T_COMOBJ + libGAP_COPYING ] = libGAP_CleanObjComObjCopy;
    libGAP_CopyObjFuncs[  libGAP_T_DATOBJ           ] = libGAP_CopyObjDatObj;
    libGAP_CopyObjFuncs[  libGAP_T_DATOBJ + libGAP_COPYING ] = libGAP_CopyObjDatObjCopy;
    libGAP_CleanObjFuncs[ libGAP_T_DATOBJ           ] = libGAP_CleanObjDatObj;
    libGAP_CleanObjFuncs[ libGAP_T_DATOBJ + libGAP_COPYING ] = libGAP_CleanObjDatObjCopy;

    /* make and install the 'PRINT_OBJ' operation                          */
    for ( t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ )
        libGAP_PrintObjFuncs[ t ] = libGAP_PrintObjObject;

    /* enter 'PrintUnknownObj' in the dispatching tables                   */
    for ( t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ )
        libGAP_PrintPathFuncs[ t ] = libGAP_PrintPathError;

    /* enter 'SaveObjError' and 'LoadObjError' for all types initially     */
    for ( t = libGAP_FIRST_REAL_TNUM;  t <= libGAP_LAST_REAL_TNUM;  t++ ) {
        libGAP_SaveObjFuncs[ t ] = libGAP_SaveObjError;
        libGAP_LoadObjFuncs[ t ] = libGAP_LoadObjError;
    }
  
    /* install the saving functions */
    libGAP_SaveObjFuncs[ libGAP_T_COMOBJ ] = libGAP_SaveComObj;
    libGAP_SaveObjFuncs[ libGAP_T_POSOBJ ] = libGAP_SavePosObj;
    libGAP_SaveObjFuncs[ libGAP_T_DATOBJ ] = libGAP_SaveDatObj;

    /* install the loading functions */
    libGAP_LoadObjFuncs[ libGAP_T_COMOBJ ] = libGAP_LoadComObj;
    libGAP_LoadObjFuncs[ libGAP_T_POSOBJ ] = libGAP_LoadPosObj;
    libGAP_LoadObjFuncs[ libGAP_T_DATOBJ ] = libGAP_LoadDatObj;

    for (t = libGAP_FIRST_REAL_TNUM; t <= libGAP_LAST_REAL_TNUM; t++ ) 
      libGAP_MakeImmutableObjFuncs[t] = libGAP_MakeImmutableError;
    
    /* install the makeimmutableing functions */
    libGAP_MakeImmutableObjFuncs[ libGAP_T_COMOBJ ] = libGAP_MakeImmutableComObj;
    libGAP_MakeImmutableObjFuncs[ libGAP_T_POSOBJ ] = libGAP_MakeImmutablePosObj;
    libGAP_MakeImmutableObjFuncs[ libGAP_T_DATOBJ ] = libGAP_MakeImmutableDatObj;
      

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarOpersFromTable( libGAP_GVarOpers );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* export certain TNUM values as global variable */
    libGAP_AssGVar(libGAP_GVarName("FIRST_CONSTANT_TNUM"), libGAP_INTOBJ_INT(libGAP_FIRST_CONSTANT_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("FIRST_CONSTANT_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("LAST_CONSTANT_TNUM"), libGAP_INTOBJ_INT(libGAP_LAST_CONSTANT_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("LAST_CONSTANT_TNUM"));
    
    libGAP_AssGVar(libGAP_GVarName("FIRST_RECORD_TNUM"), libGAP_INTOBJ_INT(libGAP_FIRST_RECORD_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("FIRST_RECORD_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("LAST_RECORD_TNUM"), libGAP_INTOBJ_INT(libGAP_LAST_RECORD_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("LAST_RECORD_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("FIRST_LIST_TNUM"), libGAP_INTOBJ_INT(libGAP_FIRST_LIST_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("FIRST_LIST_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("LAST_LIST_TNUM"), libGAP_INTOBJ_INT(libGAP_LAST_LIST_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("LAST_LIST_TNUM"));
    
    libGAP_AssGVar(libGAP_GVarName("FIRST_EXTERNAL_TNUM"), libGAP_INTOBJ_INT(libGAP_FIRST_EXTERNAL_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("FIRST_EXTERNAL_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("LAST_EXTERNAL_TNUM"), libGAP_INTOBJ_INT(libGAP_LAST_EXTERNAL_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("LAST_EXTERNAL_TNUM"));
    
    libGAP_AssGVar(libGAP_GVarName("FIRST_REAL_TNUM"), libGAP_INTOBJ_INT(libGAP_FIRST_REAL_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("FIRST_REAL_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("LAST_REAL_TNUM"), libGAP_INTOBJ_INT(libGAP_LAST_REAL_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("LAST_REAL_TNUM"));
    
    libGAP_AssGVar(libGAP_GVarName("FIRST_VIRTUAL_TNUM"), libGAP_Fail);
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("FIRST_VIRTUAL_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("LAST_VIRTUAL_TNUM"), libGAP_INTOBJ_INT(libGAP_LAST_VIRTUAL_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("LAST_VIRTUAL_TNUM"));
    
    libGAP_AssGVar(libGAP_GVarName("FIRST_IMM_MUT_TNUM"), libGAP_INTOBJ_INT(libGAP_FIRST_IMM_MUT_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("FIRST_IMM_MUT_TNUM"));

    libGAP_AssGVar(libGAP_GVarName("LAST_IMM_MUT_TNUM"), libGAP_INTOBJ_INT(libGAP_LAST_IMM_MUT_TNUM));
    libGAP_MakeReadOnlyGVar(libGAP_GVarName("LAST_IMM_MUT_TNUM"));
    

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoObjects() . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "objects",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoObjects ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  objects.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
