/****************************************************************************
**
*W  iostream.c                  GAP source                       Steve Linton
**
**
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**
**  This file will contains the functions for communicating with other
**  processes via ptys and sockets
**
**  The eventual intent is that there will be InputOutputStreams at the GAP level
**  with some API to be defined, and two ways of creating them. One is like Process
**  except that the external process is left running, the other connects as client
**  to a specified socket.
**
**  At this level, we provide the two interfaces separately. For each we have an integer
**  identifer for each open connection, and creation, read and write functions, and possibly
**  some sort of probe function
**
*/

#define _GNU_SOURCE  /* is used for getpt(), ptsname_r prototype etc. */

#include        "system.h"              /* system dependent part           */

#include        "iostream.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "lists.h"               /* generic lists                   */
#include        "listfunc.h"            /* functions for generic lists     */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "records.h"             /* generic records                 */
#include        "bool.h"                /* True and False                  */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */

#include        "libgap_internal.h"     /* GAP shared library              */

#include <stdio.h>                      /* standard input/output functions */
#include <stdlib.h>
#include <string.h>


#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_TIME_H
#include  <sys/time.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_TERMIOS_H
#include <termios.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#else
#ifdef NDEBUG
#define assert( a )
#else
#define assert( a ) do if (!(a)) {fprintf(stderr,"Assertion failed at line %d file %s\n",__LINE__,__FILE__); abort();} while (0)
#endif
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UTIL_H
#include <util.h> /* for openpty() on Mac OS X, OpenBSD and NetBSD */
#endif

#if HAVE_LIBUTIL_H
#include <libutil.h> /* for openpty() on FreeBSD */
#endif

#if HAVE_PTY_H
#include <pty.h> /* for openpty() on Cygwin, Interix, OSF/1 4 and 5 */
#endif


typedef struct {
  int childPID;    /* Also used as a link to make a linked free list */
  int ptyFD;       /* GAP reading from external prog */
  libGAP_UInt inuse;     /* we need to scan all the "live" structures when we have had SIGCHLD
                     so, for now, we just walk the array for the ones marked in use */
  libGAP_UInt changed;   /* set non-zero by the signal handler if our child has
                     done something -- stopped or exited */
  int status;     /* status from wait3 -- meaningful only if changed is 1 */
  libGAP_UInt blocked;   /* we have already reported a problem, which is still there */
  libGAP_UInt alive;     /* gets set after waiting for a child actually fails
                     implying that the child has vanished under our noses */
} libGAP_PtyIOStream;

#define libGAP_MAX_PTYS 64

static libGAP_PtyIOStream libGAP_PtyIOStreams[libGAP_MAX_PTYS];
static libGAP_Int libGAP_FreePtyIOStreams;

libGAP_Int libGAP_NewStream( void )
{
  libGAP_Int stream = -1;
  libGAP_HashLock(libGAP_PtyIOStreams);
  if ( libGAP_FreePtyIOStreams != -1 )
  {
      stream = libGAP_FreePtyIOStreams;
      libGAP_FreePtyIOStreams = libGAP_PtyIOStreams[stream].childPID;
  }
  libGAP_HashUnlock(libGAP_PtyIOStreams);
  return stream;
}

void libGAP_FreeStream( libGAP_UInt stream)
{
   libGAP_HashLock(libGAP_PtyIOStreams);
   libGAP_PtyIOStreams[stream].childPID = libGAP_FreePtyIOStreams;
   libGAP_FreePtyIOStreams = stream;
   libGAP_HashUnlock(libGAP_PtyIOStreams);
}

/****************************************************************************
**
*F  SignalChild(<stream>) . .. . . . . . . . . . .  interrupt the child process
*/
void libGAP_SignalChild (libGAP_UInt stream, libGAP_UInt sig)
{
    libGAP_HashLock(libGAP_PtyIOStreams);
    if ( libGAP_PtyIOStreams[stream].childPID != -1 )
    {
        kill( libGAP_PtyIOStreams[stream].childPID, sig );
    }
    libGAP_HashUnlock(libGAP_PtyIOStreams);
}

/****************************************************************************
**
*F  KillChild(<stream>) . . . . . . . . . . . . . . . .  kill the child process
*/
void libGAP_KillChild (libGAP_UInt stream)
{
    libGAP_HashLock(libGAP_PtyIOStreams);
    if ( libGAP_PtyIOStreams[stream].childPID != -1 )
    {
        close(libGAP_PtyIOStreams[stream].ptyFD);
        libGAP_SignalChild( stream, SIGKILL );
    }
    libGAP_HashUnlock(libGAP_PtyIOStreams);
}




/****************************************************************************
**
*F  GetMasterPty( <fid> ) . . . . . . . . .  open a master pty (from "xterm")
*/

#if !defined(HAVE_OPENPTY)

#ifndef libGAP_SYS_PTYDEV
#  ifdef hpux
#    define libGAP_SYS_PTYDEV          "/dev/ptym/ptyxx"
#  else
#    define libGAP_SYS_PTYDEV          "/dev/ptyxx"
#  endif
#endif

#ifndef libGAP_SYS_TTYDEV
#  ifdef hpux
#    define libGAP_SYS_TTYDEV          "/dev/pty/ttyxx"
#  else
#    define libGAP_SYS_TTYDEV          "/dev/ttyxx"
#  endif
#endif

#ifndef libGAP_SYS_PTYCHAR1
#  ifdef hpux
#    define libGAP_SYS_PTYCHAR1        "zyxwvutsrqp"
#  else
#    define libGAP_SYS_PTYCHAR1        "pqrstuvwxyz"
#  endif
#endif

#ifndef libGAP_SYS_PTYCHAR2
#  ifdef hpux
#    define libGAP_SYS_PTYCHAR2        "fedcba9876543210"
#  else
#    define libGAP_SYS_PTYCHAR2        "0123456789abcdef"
#  endif
#endif


static libGAP_UInt libGAP_GetMasterPty ( int *pty, libGAP_Char *nametty, libGAP_Char *namepty )
{
#if HAVE_POSIX_OPENPT && HAVE_PTSNAME
    /* Attempt to use POSIX 98 pseudo ttys. Opening a master tty is done
       via posix_openpt, which is available on virtually every current
       UNIX system; indeed, according to gnulib, it is available on at
       least the following systems:
         - glibc >= 2.2.1 (released January 2001; but is a stub on GNU/Hurd),
         - Mac OS X >= 10.4 (released April 2005),
         - FreeBSD >= 5.1 (released June 2003),
         - NetBSD >= 3.0 (released December 2005),
         - AIX >= 5.2 (released October 2002),
         - HP-UX >= 11.31 (released February 2007),
         - Solaris >= 10 (released January 2005),
         - Cygwin >= 1.7 (released December 2009).
       Systems lacking posix_openpt (in addition to older versions of
       the systems listed above) include:
         - OpenBSD
         - Minix 3.1.8
         - IRIX 6.5
         - OSF/1 5.1
         - mingw
         - MSVC 9
         - Interix 3.5
         - BeOS
       */
    *pty = posix_openpt( O_RDWR | O_NOCTTY );
    if (*pty > 0) {
        if (grantpt(*pty) || unlockpt(*pty)) {
            close(*pty);
            return 1;
        }
        libGAP_strxcpy(nametty, ptsname(*pty), 32);
        return 0;
    }
    return 1;

#elif HAVE_GETPT && HAVE_PTSNAME_R
    /* Attempt to use glibc specific APIs, for compatibility with older
       glibc versions (before 2.2.1, release January 2001). */
    *pty = getpt();
    if (*pty > 0) {
        if (grantpt(*pty) || unlockpt(*pty)) {
            close(*pty);
            return 1;
        }
        ptsname_r(*pty, nametty, 32); 
        return 0;
    }
    return 1;

#elif HAVE_PTSNAME
    /* Attempt to use Sys V pseudo ttys on systems that don't have posix_openpt,
       getpt or openpty, but do have ptsname.
       Platforms *missing* ptsname include:
       Mac OS X 10.3, OpenBSD 3.8, Minix 3.1.8, mingw, MSVC 9, BeOS. */
    *pty = open( "/dev/ptmx", O_RDWR );
    if ( *pty < 0 )
        return 1;
    libGAP_strxcpy(nametty, ptsname(*pty), 32);
    return 0;

#elif HAVE_GETPSEUDOTTY
    /* TODO: From which system(s) does getpseudotty originate? */
    *pty = getpseudotty( nametty, namepty );
    return *pty < 0 ? 1 : 0;

#elif HAVE__GETPTY
    /* Available on SGI IRIX >= 4.0 (released September 1991).
       See also http://techpubs.sgi.com/library/tpl/cgi-bin/getdoc.cgi?cmd=getdoc&coll=0530&db=man&fname=7%20pty */
    char  * line;
    line = _getpty(pty, O_RDWR|O_NDELAY, 0600, 0) ;
    if (0 == line)
        return 1;
    strcpy( nametty, line );
    return 0;

#else
    /* fallback to old-style BSD pseudoterminals, doing a brute force
       search over all pty device files. */
    int devindex;
    int letter;
    int ttylen, ptylen;

    ttylen = strlen(nametty);
    ptylen = strlen(namepty);

    for ( letter = 0; libGAP_SYS_PTYCHAR1[letter]; ++letter ) {
        nametty[ttylen-2] = libGAP_SYS_PTYCHAR1[letter];
        namepty[ptylen-2] = libGAP_SYS_PTYCHAR1[letter];

        for ( devindex = 0; libGAP_SYS_PTYCHAR2[devindex]; ++devindex ) {
            nametty[ttylen-1] = libGAP_SYS_PTYCHAR2[devindex];
            namepty[ptylen-1] = libGAP_SYS_PTYCHAR2[devindex];
                    
            *pty = open( namepty, O_RDWR );
            if ( *pty >= 0 ) {
                int slave = open( nametty, O_RDWR );
                if ( slave >= 0 ) {
                    close(slave);
                    return 0;
                }
                close(*pty);
            } 
        }
    }
    return 1;
#endif
}

#endif /* !defined(HAVE_OPENPTY) */


static libGAP_UInt libGAP_OpenPty( int *pty, int *tty )
{
#if HAVE_OPENPTY
    /* openpty is available on OpenBSD, NetBSD and FreeBSD, Mac OS X,
       Cygwin, Interix, OSF/1 4 and 5, and glibc (since 1998), and hence
       on most modern Linux systems. See also:
       http://www.gnu.org/software/gnulib/manual/html_node/openpty.html */
    return (openpty(pty, tty, NULL, NULL, NULL) < 0);
#else
    libGAP_Char ttyname[32];
    libGAP_Char ptyname[32];

    /* construct the name of the pseudo terminal */
    strcpy( ttyname, libGAP_SYS_TTYDEV );
    strcpy( ptyname, libGAP_SYS_PTYDEV );

    if ( libGAP_GetMasterPty(pty, ttyname, ptyname) ) {
        libGAP_Pr( "open master failed\n", 0L, 0L );
        return 1;
    }
    *tty = open( ttyname, O_RDWR, 0 );
    if ( *tty < 0 ) {
        libGAP_Pr( "open slave failed\n", 0L, 0L );
        close(*pty);
        return 1;
    }
    return 0;
#endif
}

/****************************************************************************
**
*F  StartChildProcess( <dir>, <name>, <args> ) . . . . start a subprocess using ptys
**  returns the stream number of the IOStream that is connected to the new processs
*/

void libGAP_ChildStatusChanged( int whichsig )
{
  libGAP_UInt i;
  int status;
  int retcode;
  assert(whichsig == SIGCHLD);
  libGAP_HashLock(libGAP_PtyIOStreams);
  for (i = 0; i < libGAP_MAX_PTYS; i++) {
      if (libGAP_PtyIOStreams[i].inuse) {
          retcode = waitpid( libGAP_PtyIOStreams[i].childPID, &status, WNOHANG | WUNTRACED );
          if (retcode != -1 && retcode != 0 && (WIFEXITED(status) || WIFSIGNALED(status)) ) {
              libGAP_PtyIOStreams[i].changed = 1;
              libGAP_PtyIOStreams[i].status = status;
              libGAP_PtyIOStreams[i].blocked = 0;
          }
      }
  }
  libGAP_HashUnlock(libGAP_PtyIOStreams);
  /* Collect up any other zombie children */
  do {
      retcode = waitpid( -1, &status, WNOHANG);
      if (retcode == -1 && errno != ECHILD)
          libGAP_Pr("#E Unexpected waitpid error %d\n",errno, 0);
  } while (retcode != 0 && retcode != -1);
  
  signal(SIGCHLD, libGAP_ChildStatusChanged);
}

libGAP_Int libGAP_StartChildProcess ( libGAP_Char *dir, libGAP_Char *prg, libGAP_Char *args[] )
{
/*  Int             j;       / loop variables                  */
/*  char            c[8];    / buffer for communication        */
/*  int             n;       / return value of 'select'        */
    int             slave;   /* pipe to child                   */
    libGAP_Int            stream;

#if HAVE_TERMIOS_H
    struct termios  tst;     /* old and new terminal state      */
#elif HAVE_TERMIO_H
    struct termio   tst;     /* old and new terminal state      */
#elif HAVE_SGTTY_H
    struct sgttyb   tst;     /* old and new terminal state      */
#elif !defined(USE_PRECOMPILED)
/* If no way to store and reset terminal states is known, and we are
   not currently re-making the dependency list (via cnf/Makefile),
   then trigger an error. */
    #error No supported way of (re)storing terminal state is available
#endif

    /* Get a stream record */
    stream = libGAP_NewStream();
    if (stream == -1)
      return -1;
    
    /* open pseudo terminal for communication with gap */
    if ( libGAP_OpenPty(&libGAP_PtyIOStreams[stream].ptyFD, &slave) )
    {
        libGAP_Pr( "open pseudo tty failed\n", 0L, 0L);
        libGAP_FreeStream(stream);
        return -1;
    }

    /* Now fiddle with the terminal sessions on the pty */
#if HAVE_TERMIOS_H
    if ( tcgetattr( slave, &tst ) == -1 )
    {
        libGAP_Pr( "tcgetattr on slave pty failed\n", 0L, 0L);
        goto cleanup;

    }
    tst.c_cc[VINTR] = 0377;
    tst.c_cc[VQUIT] = 0377;
    tst.c_iflag    &= ~(INLCR|ICRNL);
    tst.c_cc[VMIN]  = 1;
    tst.c_cc[VTIME] = 0;
    tst.c_lflag    &= ~(ECHO|ICANON);
    tst.c_oflag    &= ~(ONLCR);
    if ( tcsetattr( slave, TCSANOW, &tst ) == -1 )
    {
        libGAP_Pr("tcsetattr on slave pty failed\n", 0, 0 );
        goto cleanup;
    }
#elif HAVE_TERMIO_H
    if ( ioctl( slave, TCGETA, &tst ) == -1 )
    {
        libGAP_Pr( "ioctl TCGETA on slave pty failed\n");
        goto cleanup;
    }
    tst.c_cc[VINTR] = 0377;
    tst.c_cc[VQUIT] = 0377;
    tst.c_iflag    &= ~(INLCR|ICRNL);
    tst.c_cc[VMIN]  = 1;
    tst.c_cc[VTIME] = 0;   
    /* Note that this is at least on Linux dangerous! 
       Therefore, we now have the HAVE_TERMIOS_H section for POSIX
       Terminal control. */
    tst.c_lflag    &= ~(ECHO|ICANON);
    if ( ioctl( slave, TCSETAW, &tst ) == -1 )
    {
        libGAP_Pr( "ioctl TCSETAW on slave pty failed\n");
        goto cleanup;
    }
#elif HAVE_SGTTY_H
    if ( ioctl( slave, TIOCGETP, (char*)&tst ) == -1 )
    {
        libGAP_Pr( "ioctl TIOCGETP on slave pty failed\n");
        goto cleanup;
    }
    tst.sg_flags |= RAW;
    tst.sg_flags &= ~ECHO;
    if ( ioctl( slave, TIOCSETN, (char*)&tst ) == -1 )
    {
        libGAP_Pr( "ioctl on TIOCSETN slave pty failed\n");
        goto cleanup;
    }
#endif

    /* set input to non blocking operation */
    /* Not any more */

    libGAP_PtyIOStreams[stream].inuse = 1;
    libGAP_PtyIOStreams[stream].alive = 1;
    libGAP_PtyIOStreams[stream].blocked = 0;
    libGAP_PtyIOStreams[stream].changed = 0;
    /* fork */
    libGAP_PtyIOStreams[stream].childPID = fork();
    if ( libGAP_PtyIOStreams[stream].childPID == 0 )
    {
        /* Set up the child */
        close(libGAP_PtyIOStreams[stream].ptyFD);
        if ( dup2( slave, 0 ) == -1)
            _exit(-1);
        fcntl( 0, F_SETFD, 0 );
        
        if (dup2( slave, 1 ) == -1)
            _exit(-1);
        fcntl( 1, F_SETFD, 0 );
        
        if ( chdir(dir) == -1 ) {
            _exit(-1);
        }

#if HAVE_SETPGID
        setpgid(0,0);
#endif

        execv( prg, args );

        /* This should never happen */
        close(slave);
        _exit(1);
    }

    /* Now we're back in the master */
    /* check if the fork was successful */
    if ( libGAP_PtyIOStreams[stream].childPID == -1 )
    {
        libGAP_Pr( "Panic: cannot fork to subprocess.\n", 0, 0);
        goto cleanup;
    }
    close(slave);
    
    
    return stream;

 cleanup:
    close(slave);
    close(libGAP_PtyIOStreams[stream].ptyFD);
    libGAP_PtyIOStreams[stream].inuse = 0;
    libGAP_FreeStream(stream);
    return -1;
}


void libGAP_HandleChildStatusChanges( libGAP_UInt pty)
{
  /* common error handling, when we are asked to read or write to a stopped
     or dead child */
  libGAP_HashLock(libGAP_PtyIOStreams);
  if (libGAP_PtyIOStreams[pty].alive == 0)
  {
      libGAP_PtyIOStreams[pty].changed = 0;
      libGAP_PtyIOStreams[pty].blocked = 0;
      libGAP_HashUnlock(libGAP_PtyIOStreams);
      libGAP_ErrorQuit("Child Process is unexpectedly dead", (libGAP_Int) 0L, (libGAP_Int) 0L);
      return;
  }
  if (libGAP_PtyIOStreams[pty].blocked)
  {
      libGAP_HashUnlock(libGAP_PtyIOStreams);
      libGAP_ErrorQuit("Child Process is still dead", (libGAP_Int)0L,(libGAP_Int)0L);
      return;
  }
  if (libGAP_PtyIOStreams[pty].changed)
  {
      libGAP_PtyIOStreams[pty].blocked = 1;
      libGAP_PtyIOStreams[pty].changed = 0;
      libGAP_Int cPID = libGAP_PtyIOStreams[pty].childPID;
      libGAP_Int status = libGAP_PtyIOStreams[pty].status;
      libGAP_HashUnlock(libGAP_PtyIOStreams);
      libGAP_ErrorQuit("Child Process %d has stopped or died, status %d",
                cPID, status);
      return;
  }
  libGAP_HashUnlock(libGAP_PtyIOStreams);
}

#define libGAP_MAX_ARGS 1000

libGAP_Obj libGAP_FuncCREATE_PTY_IOSTREAM( libGAP_Obj self, libGAP_Obj dir, libGAP_Obj prog, libGAP_Obj args )
{
  libGAP_Obj  allargs[libGAP_MAX_ARGS+1];
  libGAP_Char *argv[libGAP_MAX_ARGS+2];
  libGAP_UInt i,len;
  libGAP_Int pty;
  len = libGAP_LEN_LIST(args);
  if (len > libGAP_MAX_ARGS)
    libGAP_ErrorQuit("Too many arguments",0,0);
  libGAP_ConvString(dir);
  libGAP_ConvString(prog);
  for (i = 1; i <=len; i++)
    {
      allargs[i] = libGAP_ELM_LIST(args,i);
      libGAP_ConvString(allargs[i]);
    }
  /* From here we cannot afford to have a garbage collection */
  argv[0] = libGAP_CSTR_STRING(prog);
  for (i = 1; i <=len; i++)
    {
      argv[i] = libGAP_CSTR_STRING(allargs[i]);
    }
  argv[i] = (libGAP_Char *)0;
  pty = libGAP_StartChildProcess( libGAP_CSTR_STRING(dir) , libGAP_CSTR_STRING(prog), argv );
  if (pty < 0)
    return libGAP_Fail;
  else
    return libGAP_INTOBJ_INT(pty);
}




libGAP_Int libGAP_ReadFromPty2( libGAP_UInt stream, libGAP_Char *buf, libGAP_Int maxlen, libGAP_UInt block)
{
  /* read at most maxlen bytes from stream, into buf.
    If block is non-zero then wait for at least one byte
    to be available. Otherwise don't. Return the number of
    bytes read, or -1 for error. A blocking return having read zero bytes
    definitely indicates an end of file */

  libGAP_Int nread = 0;
  int ret;
  
  while (maxlen > 0)
    {
#if HAVE_SELECT
      if (!block || nread > 0)
      {
        fd_set set;
        struct timeval tv;
        do {
          FD_ZERO( &set);
          FD_SET( libGAP_PtyIOStreams[stream].ptyFD, &set );
          tv.tv_sec = 0;
          tv.tv_usec = 0;
          ret =  select( libGAP_PtyIOStreams[stream].ptyFD + 1, &set, NULL, NULL, &tv);
        } while (ret == -1 && errno == EAGAIN);
        if (ret == -1 && nread == 0)
          return -1;
        if (ret < 1)
          return nread ? nread : -1;
      }
#endif
      do {
        ret = read(libGAP_PtyIOStreams[stream].ptyFD, buf, maxlen);
      } while (ret == -1 && errno == EAGAIN);
      if (ret == -1 && nread == 0)
        return -1;
      if (ret < 1)
        return nread;
      nread += ret;
      buf += ret;
      maxlen -= ret;
    }
  return nread;
}
  
  

extern int errno;

libGAP_UInt libGAP_WriteToPty ( libGAP_UInt stream, libGAP_Char *buf, libGAP_Int len )
{
    libGAP_Int         res;
    libGAP_Int         old;
/*  struct timeval tv; */
/*  fd_set      writefds; */
/*  int retval; */
    if (len < 0)
      return  write( libGAP_PtyIOStreams[stream].ptyFD, buf, -len );
    old = len;
    while ( 0 < len )
    {
        res = write( libGAP_PtyIOStreams[stream].ptyFD, buf, len );
        if ( res < 0 )
        {
          libGAP_HandleChildStatusChanges(stream);
            if ( errno == EAGAIN )
              {
                continue;
              }
            else
              return errno;
        }
        len  -= res;
        buf += res;
    }
    return old;
}




libGAP_Obj libGAP_FuncWRITE_IOSTREAM( libGAP_Obj self, libGAP_Obj stream, libGAP_Obj string, libGAP_Obj len )
{
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  libGAP_ConvString(string);
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));
  libGAP_HandleChildStatusChanges(pty);
  return libGAP_INTOBJ_INT(libGAP_WriteToPty(pty, libGAP_CSTR_STRING(string), libGAP_INT_INTOBJ(len)));
}

libGAP_Obj libGAP_FuncREAD_IOSTREAM( libGAP_Obj self, libGAP_Obj stream, libGAP_Obj len )
{
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  libGAP_Int ret;
  libGAP_Obj string;
  string = libGAP_NEW_STRING(libGAP_INT_INTOBJ(len));
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));
  /* HandleChildStatusChanges(pty);   Omit this to allow picking up "trailing" bytes*/
  ret = libGAP_ReadFromPty2(pty, libGAP_CSTR_STRING(string), libGAP_INT_INTOBJ(len), 1);
  if (ret == -1)
    return libGAP_Fail;
  libGAP_SET_LEN_STRING(string, ret);
  libGAP_ResizeBag(string, libGAP_SIZEBAG_STRINGLEN(ret));
  return string;
}

libGAP_Obj libGAP_FuncREAD_IOSTREAM_NOWAIT(libGAP_Obj self, libGAP_Obj stream, libGAP_Obj len)
{
  libGAP_Obj string;
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  libGAP_Int ret;
  string = libGAP_NEW_STRING(libGAP_INT_INTOBJ(len));
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));
  /* HandleChildStatusChanges(pty);   Omit this to allow picking up "trailing" bytes*/
  ret = libGAP_ReadFromPty2(pty, libGAP_CSTR_STRING(string), libGAP_INT_INTOBJ(len), 0);
  if (ret == -1)
    return libGAP_Fail;
  libGAP_SET_LEN_STRING(string, ret);
  libGAP_ResizeBag(string, libGAP_SIZEBAG_STRINGLEN(ret));
  return string;
}
     

libGAP_Obj libGAP_FuncKILL_CHILD_IOSTREAM( libGAP_Obj self, libGAP_Obj stream )
{
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));
  /* Don't check for child having changes status */
  libGAP_KillChild( pty );
  return 0;
}

libGAP_Obj libGAP_FuncSIGNAL_CHILD_IOSTREAM( libGAP_Obj self, libGAP_Obj stream , libGAP_Obj sig)
{
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));
  /* Don't check for child having changes status */
  libGAP_SignalChild( pty, libGAP_INT_INTOBJ(sig) );
  return 0;
}

libGAP_Obj libGAP_FuncCLOSE_PTY_IOSTREAM( libGAP_Obj self, libGAP_Obj stream )
{
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  int status;
  int retcode;
/*UInt count; */
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));

  libGAP_PtyIOStreams[pty].inuse = 0;
  
  /* Close down the child */
  retcode = close(libGAP_PtyIOStreams[pty].ptyFD);
  if (retcode)
    libGAP_Pr("Strange close return code %d\n",retcode, 0);
  kill(libGAP_PtyIOStreams[pty].childPID, SIGTERM);
  retcode = waitpid(libGAP_PtyIOStreams[pty].childPID, &status, 0);
  libGAP_FreeStream(pty);
  return 0;
}

libGAP_Obj libGAP_FuncIS_BLOCKED_IOSTREAM( libGAP_Obj self, libGAP_Obj stream )
{
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));
  return (libGAP_PtyIOStreams[pty].blocked || libGAP_PtyIOStreams[pty].changed || !libGAP_PtyIOStreams[pty].alive) ? libGAP_True : libGAP_False;
}

libGAP_Obj libGAP_FuncFD_OF_IOSTREAM( libGAP_Obj self, libGAP_Obj stream )
{
  libGAP_UInt pty = libGAP_INT_INTOBJ(stream);
  while (!libGAP_PtyIOStreams[pty].inuse)
    pty = libGAP_INT_INTOBJ(libGAP_ErrorReturnObj("IOSTREAM %d is not in use",pty,0L,
                                    "you can replace stream number <num> via 'return <num>;'"));
  return libGAP_INTOBJ_INT(libGAP_PtyIOStreams[pty].ptyFD);
}


/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "CREATE_PTY_IOSTREAM", 3, "dir, prog, args",
      libGAP_FuncCREATE_PTY_IOSTREAM, "src/iostream.c:CREATE_PTY_IOSTREAM" },
    
    { "WRITE_IOSTREAM", 3, "stream, string, len",
      libGAP_FuncWRITE_IOSTREAM, "src/iostream.c:WRITE_IOSTREAM" },
    
    { "READ_IOSTREAM", 2, "stream, len",
      libGAP_FuncREAD_IOSTREAM, "src/iostream.c:READ_IOSTREAM" },

    { "READ_IOSTREAM_NOWAIT", 2, "stream, len",
      libGAP_FuncREAD_IOSTREAM_NOWAIT, "src/iostream.c:READ_IOSTREAM_NOWAIT" },

    { "KILL_CHILD_IOSTREAM", 1, "stream",
      libGAP_FuncKILL_CHILD_IOSTREAM, "src/iostream.c:KILL_CHILD_IOSTREAM" },

    { "CLOSE_PTY_IOSTREAM", 1, "stream",
      libGAP_FuncCLOSE_PTY_IOSTREAM, "src/iostream.c:CLOSE_PTY_IOSTREAM" },

    { "SIGNAL_CHILD_IOSTREAM", 2, "stream, signal",
      libGAP_FuncSIGNAL_CHILD_IOSTREAM, "src/iostream.c:SIGNAL_CHILD_IOSTREAM" },
    
    { "IS_BLOCKED_IOSTREAM", 1, "stream",
      libGAP_FuncIS_BLOCKED_IOSTREAM, "src/iostream.c:IS_BLOCKED_IOSTREAM" },
    
    { "FD_OF_IOSTREAM", 1, "stream",
      libGAP_FuncFD_OF_IOSTREAM, "src/iostream.c:FD_OF_IOSTREAM" },

      {0} };
  
/* NB Should probably do some checks preSave for open files etc and refuse to save
   if any are found */

/****************************************************************************
**
*F  postResore( <module> ) . . . . . . .re-initialise library data structures
*/

static libGAP_Int libGAP_postRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  InitKernel( <module> ) . . . . . . .  initialise kernel data structures
*/

static libGAP_Int libGAP_InitKernel( 
      libGAP_StructInitInfo * libGAP_module )
{
  libGAP_UInt i;
  libGAP_PtyIOStreams[0].childPID = -1;
  for (i = 1; i < libGAP_MAX_PTYS; i++)
    {
      libGAP_PtyIOStreams[i].childPID = i-1;
      libGAP_PtyIOStreams[i].inuse = 0;
    }
  libGAP_FreePtyIOStreams = libGAP_MAX_PTYS-1;

  /* init filters and functions                                          */
  libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );
  
  /* Set up the trap to detect future dying children */
#ifdef LIBGAP_SIGNALS
  signal( SIGCHLD, libGAP_ChildStatusChanged );
#endif
  return 0;
}

/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/

static libGAP_Int libGAP_InitLibrary( 
      libGAP_StructInitInfo * libGAP_module )
{
      /* init filters and functions                                          */
  libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

  return libGAP_postRestore( libGAP_module );
}

/****************************************************************************
**
*F  InitInfoSysFiles()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "iostream",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_postRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoIOStream ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**
*E  iostream.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
