/****************************************************************************
**
*W  costab.c                    GAP source                       Frank Celler
*W                                                           & Volkmar Felsch
*W                                                         & Martin Schönert
*W                                                         & Alexander Hulpke
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of for coset tables.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "integer.h"             /* integers                        */
#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "costab.h"              /* coset table                     */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */


/****************************************************************************
**

*V  declaration of static variables
*/
static libGAP_Obj      libGAP_objRel;                 /* handle of a relator             */
static libGAP_Obj      libGAP_objNums;                /* handle of parallel numbers list */
static libGAP_Obj      libGAP_objTable;               /* handle of the coset table       */
static libGAP_Obj      libGAP_objTable2;              /* handle of coset factor table    */
static libGAP_Obj      libGAP_objNext;                /*                                 */
static libGAP_Obj      libGAP_objPrev;                /*                                 */
static libGAP_Obj      libGAP_objFactor;              /*                                 */
static libGAP_Obj      libGAP_objTree;                /* handle of subgroup gens tree    */

static libGAP_Obj      libGAP_objTree1;               /* first tree component            */
static libGAP_Obj      libGAP_objTree2;               /* second tree component           */

static libGAP_Obj      libGAP_objExponent;            /* handle of subgroup order        */
static libGAP_Obj      libGAP_objWordValue;           /* handle of word value            */

static libGAP_Int      libGAP_treeType;               /* tree type                       */
static libGAP_Int      libGAP_treeWordLength;         /* maximal tree word length        */
static libGAP_Int      libGAP_firstDef;               /*                                 */
static libGAP_Int      libGAP_lastDef;                /*                                 */
static libGAP_Int      libGAP_firstFree;              /*                                 */
static libGAP_Int      libGAP_lastFree;               /*                                 */

static libGAP_Int      libGAP_minGaps;                /* switch for marking mingaps      */
static libGAP_Int      libGAP_nrdel;                  /*                                 */

static libGAP_Int      libGAP_dedfst;                 /* position of first deduction     */
static libGAP_Int      libGAP_dedlst;                 /* position of last deduction      */
static libGAP_Int      libGAP_dedgen [40960];         /* deduction list keeping gens     */
static libGAP_Int      libGAP_dedcos [40960];         /* deduction list keeping cosets   */
static libGAP_Int      libGAP_dedSize = 40960;        /* size of deduction list buffers  */
static libGAP_Int      libGAP_dedprint;               /* print flag for warning          */

static libGAP_Int      libGAP_wordList [1024];        /* coset rep word buffer           */
static libGAP_Int      libGAP_wordSize = 1023;        /* maximal no. of coset rep words  */

/* clean out global Obj-type variables  to avoid hogging memory*/
static void libGAP_CleanOut( void )
{
  libGAP_objRel = (libGAP_Obj) 0;
  libGAP_objNums = (libGAP_Obj) 0;
  libGAP_objTable = (libGAP_Obj) 0;
  libGAP_objTable2 = (libGAP_Obj) 0;
  libGAP_objNext = (libGAP_Obj) 0;
  libGAP_objPrev = (libGAP_Obj) 0;
  libGAP_objFactor = (libGAP_Obj) 0;
  libGAP_objTree = (libGAP_Obj) 0;
  libGAP_objTree1 = (libGAP_Obj) 0;
  libGAP_objTree2 = (libGAP_Obj) 0;
  libGAP_objExponent = (libGAP_Obj) 0;
  libGAP_objWordValue = (libGAP_Obj) 0;
}

/****************************************************************************
**

*F  FuncApplyRel( <self>, <app>, <rel> )   apply a relator to a coset in a TC
**
**  'FuncApplyRel' implements the internal function 'ApplyRel'.
**
**  'ApplyRel( <app>, <rel> )'
**
**  'ApplyRel'  applies the relator  <rel>  to the  application  list  <app>.
**
**  ... more about ApplyRel ...
*/
libGAP_Obj libGAP_FuncApplyRel (
    libGAP_Obj                 self,
    libGAP_Obj                 app,            /* handle of the application list  */
    libGAP_Obj                 rel )           /* handle of the relator           */
{
    
    libGAP_Int                 lp;             /* left pointer into relator       */
    libGAP_Int                 lc;             /* left coset to apply to          */
    libGAP_Int                 rp;             /* right pointer into relator      */
    libGAP_Int                 rc;             /* right coset to apply to         */
    libGAP_Int                 tc;             /* temporary coset                 */

    /* check the application list                                          */
    /*T 1996/12/03 fceller this should be replaced by 'PlistConv'          */
    if ( ! libGAP_IS_PLIST(app) ) {
        libGAP_ErrorQuit( "<app> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(app), 0L );
        return 0;
    }
    if ( libGAP_LEN_PLIST(app) != 4 ) {
        libGAP_ErrorQuit( "<app> must be a list of length 4 not %d",
                   (libGAP_Int) libGAP_LEN_PLIST(app), 0L );
        return 0;
    }

    /* get the four entries                                                */
    lp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( app, 1 ) );
    lc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( app, 2 ) );
    rp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( app, 3 ) );
    rc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( app, 4 ) );

    /* get and check the relator (well, only a little bit)                 */
    /*T 1996/12/03 fceller this should be replaced by 'PlistConv'          */
    if ( ! libGAP_IS_PLIST(rel) ) {
        libGAP_ErrorQuit( "<rel> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(rel), 0L );
        return 0;
    }

    /* fix right pointer if requested                                      */
    if ( rp == -1 )
        rp = lp + libGAP_INT_INTOBJ( libGAP_ELM_PLIST( rel, 1 ) );

    /* scan as long as possible from the right to the left                 */
    while ( lp < rp
         && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(rel,rp),rc))) )
    {
        rc = tc;  rp = rp - 2;
    }

    /* scan as long as possible from the left to the right                 */
    while ( lp < rp
         && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(rel,lp),lc))) )
    {
        lc = tc;  lp = lp + 2;
    }

    /* copy the information back into the application list                 */
    libGAP_SET_ELM_PLIST( app, 1, libGAP_INTOBJ_INT( lp ) );
    libGAP_SET_ELM_PLIST( app, 2, libGAP_INTOBJ_INT( lc ) );
    libGAP_SET_ELM_PLIST( app, 3, libGAP_INTOBJ_INT( rp ) );
    libGAP_SET_ELM_PLIST( app, 4, libGAP_INTOBJ_INT( rc ) );

    /* return 'true' if a coincidence or deduction was found               */
    if ( lp == rp+1
         && libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(rel,lp),lc)) != rc )
    {
        return libGAP_True;
    }
    else
        return libGAP_False;
}


/****************************************************************************
**
*F  CompressDeductionList() . . . .  removes unused items from deduction list
**
**  'CompressDeductionList'  tries to find and delete  deduction list entries
**  which are not used any more.
**
**  'dedgen',  'dedcos',  'dedfst',  'dedlst',  'dedSize' and 'objTable'  are
**  assumed to be known as static variables.
*/
static void libGAP_CompressDeductionList ( void )
{
    libGAP_Obj               * ptTable;          /* pointer to the coset table    */
    libGAP_Int                 i;
    libGAP_Int                 j;

    /* check if the situation is as assumed                                */
    if ( libGAP_dedlst != libGAP_dedSize ) {
        libGAP_ErrorQuit( "invalid call of CompressDeductionList", 0L, 0L );
        return;
    }

    /* run through the lists and compress them                             */
    ptTable = &(libGAP_ELM_PLIST(libGAP_objTable,1)) - 1;
    j = 0;
    for ( i = libGAP_dedfst; i < libGAP_dedlst; i++ ) {
        if ( libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptTable[libGAP_dedgen[i]],libGAP_dedcos[i])) > 0
          && j < i )
        {
            libGAP_dedgen[j] = libGAP_dedgen[i];
            libGAP_dedcos[j] = libGAP_dedcos[i];
            j++;
        }
    }

    /* update the pointers                                                 */
    libGAP_dedfst = 0;
    libGAP_dedlst = j;

    /* check if we have at least one free position                         */
    if ( libGAP_dedlst == libGAP_dedSize ) {
        if ( libGAP_dedprint == 0 ) {
            libGAP_Pr( "#I  WARNING: deductions being discarded\n", 0L, 0L );
            libGAP_dedprint = 1;
        }
        libGAP_dedlst--;
    }
}


/****************************************************************************
**
*F  HandleCoinc( <cos1>, <cos2> ) . . . . . . . . handle coincidences in a TC
**
**  'HandleCoinc'  is a subroutine of 'FuncMakeConsequences'  and handles the
**  coincidence  cos2 = cos1.
*/
static void libGAP_HandleCoinc (
    libGAP_Int                 cos1,
    libGAP_Int                 cos2 )
{
    libGAP_Obj *               ptTable;          /* pointer to the coset table    */
    libGAP_Obj *               ptNext;
    libGAP_Obj *               ptPrev;
    libGAP_Int                 c1;
    libGAP_Int                 c2;
    libGAP_Int                 c3;
    libGAP_Int                 i;
    libGAP_Int                 firstCoinc;
    libGAP_Int                 lastCoinc;
    libGAP_Obj *               gen;
    libGAP_Obj *               inv;

    /* is this test necessary?                                             */
    if ( cos1 == cos2 )  return;

    /* get some pointers                                                   */
    ptTable = &(libGAP_ELM_PLIST(libGAP_objTable,1)) - 1;
    ptNext  = &(libGAP_ELM_PLIST(libGAP_objNext,1)) - 1;
    ptPrev  = &(libGAP_ELM_PLIST(libGAP_objPrev,1)) - 1;

    /* take the smaller one as new representative                          */
    if ( cos2 < cos1 ) { c3 = cos1;  cos1 = cos2;  cos2 = c3;  }

    /* if we are removing an important coset update it                     */
    if ( cos2 == libGAP_lastDef )
        libGAP_lastDef  = libGAP_INT_INTOBJ( ptPrev[libGAP_lastDef ] );
    if ( cos2 == libGAP_firstDef )
        libGAP_firstDef = libGAP_INT_INTOBJ( ptPrev[libGAP_firstDef] );

    /* remove <cos2> from the coset list                                   */
    ptNext[libGAP_INT_INTOBJ(ptPrev[cos2])] = ptNext[cos2];
    if ( ptNext[cos2] != libGAP_INTOBJ_INT( 0 ) )
        ptPrev[libGAP_INT_INTOBJ(ptNext[cos2])] = ptPrev[cos2];

    /* put the first coincidence into the list of coincidences             */
    firstCoinc        = cos2;
    lastCoinc         = cos2;
    ptNext[lastCoinc] = libGAP_INTOBJ_INT( 0 );

    /* <cos1> is the representative of <cos2> and its own representative   */
    ptPrev[cos2] = libGAP_INTOBJ_INT( cos1 );

    /* while there are coincidences to handle                              */
    while ( firstCoinc != 0 ) {

        /* replace <firstCoinc> by its representative in the table         */
        cos1 = libGAP_INT_INTOBJ( ptPrev[firstCoinc] );  cos2 = firstCoinc;
        for ( i = 1; i <= libGAP_LEN_PLIST(libGAP_objTable); i++ ) {
            gen = &(libGAP_ELM_PLIST(ptTable[i],1)) - 1;
            /* inv = ADDR_OBJ(ptTable[ ((i-1)^1)+1 ] ); */
            inv = &(libGAP_ELM_PLIST( ptTable[ i + 2*(i % 2) - 1 ], 1 ) ) - 1;

            /* replace <cos2> by <cos1> in the column of <gen>^-1          */
            c2 = libGAP_INT_INTOBJ( gen[cos2] );
            if ( c2 > 0 ) {
                c1 = libGAP_INT_INTOBJ( gen[cos1] );

                /* if the other entry is empty copy it                     */
                if ( c1 <= 0 )  {
                    gen[cos1] = libGAP_INTOBJ_INT( c2 );
                    gen[cos2] = libGAP_INTOBJ_INT( 0 );
                    inv[c2]   = libGAP_INTOBJ_INT( cos1 );
                    if ( libGAP_dedlst == libGAP_dedSize )
                        libGAP_CompressDeductionList( );
                    libGAP_dedgen[libGAP_dedlst] = i;
                    libGAP_dedcos[libGAP_dedlst] = cos1;
                    libGAP_dedlst++;
                }

                /* otherwise check for a coincidence                       */
                else {
                    inv[c2]   = libGAP_INTOBJ_INT( 0 );
                    gen[cos2] = libGAP_INTOBJ_INT( 0 );
                    if ( gen[cos1] <= libGAP_INTOBJ_INT( 0 ) ) {
                        gen[cos1] = libGAP_INTOBJ_INT( cos1 );
                        if ( libGAP_dedlst == libGAP_dedSize )
                            libGAP_CompressDeductionList( );
                        libGAP_dedgen[libGAP_dedlst] = i;
                        libGAP_dedcos[libGAP_dedlst] = cos1;
                        libGAP_dedlst++;
                    }

                    /* find the representative of <c1>                     */
                    while ( c1 != 1
                        && libGAP_INT_INTOBJ(ptNext[libGAP_INT_INTOBJ(ptPrev[c1])]) != c1 )
                    {
                        c1 = libGAP_INT_INTOBJ(ptPrev[c1]);
                    }

                    /* find the representative of <c2>                     */
                    while ( c2 != 1
                        && libGAP_INT_INTOBJ(ptNext[libGAP_INT_INTOBJ(ptPrev[c2])]) != c2 )
                    {
                        c2 = libGAP_INT_INTOBJ(ptPrev[c2]);
                    }

                    /* if the representatives differ we got a coincindence */
                    if ( c1 != c2 ) {

                        /* take the smaller one as new representative      */
                        if ( c2 < c1 ) { c3 = c1;  c1 = c2;  c2 = c3; }

                        /* if we are removing an important coset update it */
                        if ( c2 == libGAP_lastDef  )
                            libGAP_lastDef  = libGAP_INT_INTOBJ(ptPrev[libGAP_lastDef ]);
                        if ( c2 == libGAP_firstDef )
                            libGAP_firstDef = libGAP_INT_INTOBJ(ptPrev[libGAP_firstDef]);

                        /* remove <c2> from the coset list                 */
                        ptNext[libGAP_INT_INTOBJ(ptPrev[c2])] = ptNext[c2];
                        if ( ptNext[c2] != libGAP_INTOBJ_INT( 0 ) )
                            ptPrev[libGAP_INT_INTOBJ(ptNext[c2])] = ptPrev[c2];

                        /* append <c2> to the coincidence list             */
                        ptNext[lastCoinc] = libGAP_INTOBJ_INT( c2 );
                        lastCoinc         = c2;
                        ptNext[lastCoinc] = libGAP_INTOBJ_INT( 0 );

                        /* <c1> is the rep of <c2> and its own rep.        */
                        ptPrev[c2] = libGAP_INTOBJ_INT( c1 );
                    }
                }
            }

            /* save minimal gap flags                                      */
            else if ( libGAP_minGaps != 0 && c2 == -1 ) {
                if ( gen[cos1] <= libGAP_INTOBJ_INT( 0 ) ) {
                    gen[cos1] = libGAP_INTOBJ_INT( -1 );
                }
                gen[cos2] = libGAP_INTOBJ_INT( 0 );
            }
        }

        /* move the replaced coset to the free list                        */
        if ( libGAP_firstFree == 0 ) {
            libGAP_firstFree      = firstCoinc;
            libGAP_lastFree       = firstCoinc;
        }
        else {
            ptNext[libGAP_lastFree] = libGAP_INTOBJ_INT( firstCoinc );
            libGAP_lastFree         = firstCoinc;
        }
        firstCoinc = libGAP_INT_INTOBJ( ptNext[firstCoinc] );
        ptNext[libGAP_lastFree] = libGAP_INTOBJ_INT( 0 );

        libGAP_nrdel++;
    }
}


/****************************************************************************
**
*F  FuncMakeConsequences( <self>, <list> )  find consqs of a coset definition
*/
libGAP_Obj libGAP_FuncMakeConsequences (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    libGAP_Obj                 hdSubs;         /*                                 */
    libGAP_Obj                 objRels;        /*                                 */
    libGAP_Obj *               ptRel;          /* pointer to the relator bag      */
    libGAP_Obj *               ptNums;         /* pointer to this list            */
    libGAP_Int                 lp;             /* left pointer into relator       */
    libGAP_Int                 lc;             /* left coset to apply to          */
    libGAP_Int                 rp;             /* right pointer into relator      */
    libGAP_Int                 rc;             /* right coset to apply to         */
    libGAP_Int                 tc;             /* temporary coset                 */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Obj                 hdTmp;          /* temporary variable              */

    /*T 1996/12/03 fceller this should be replaced by 'PlistConv'          */
    if ( ! libGAP_IS_PLIST(list) ) {
        libGAP_ErrorQuit( "<list> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(list), 0L );
        return 0;
    }

    libGAP_objTable  = libGAP_ELM_PLIST( list, 1 );
    libGAP_objNext   = libGAP_ELM_PLIST( list, 2 );
    libGAP_objPrev   = libGAP_ELM_PLIST( list, 3 );

    libGAP_firstFree = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 6 ) );
    libGAP_lastFree  = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 7 ) );
    libGAP_firstDef  = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 8 ) );
    libGAP_lastDef   = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 9 ) );
    libGAP_minGaps   = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 12 ) );

    libGAP_nrdel     = 0;

    /* initialize the deduction queue                                      */
    libGAP_dedprint = 0;
    libGAP_dedfst = 0;
    libGAP_dedlst = 1;
    libGAP_dedgen[ 0 ] = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 10 ) );
    libGAP_dedcos[ 0 ] = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 11 ) );

    /* while the deduction queue is not empty                              */
    while ( libGAP_dedfst < libGAP_dedlst ) {

        /* skip the deduction, if it got irrelevant by a coincidence       */
        hdTmp = libGAP_ELM_PLIST( libGAP_objTable, libGAP_dedgen[libGAP_dedfst] );
        hdTmp = libGAP_ELM_PLIST( hdTmp, libGAP_dedcos[libGAP_dedfst] );
        if ( libGAP_INT_INTOBJ(hdTmp) <= 0 ) {
            libGAP_dedfst++;
            continue;
        }

        /* while there are still subgroup generators apply them            */
        hdSubs = libGAP_ELM_PLIST( list, 5 );
        for ( i = libGAP_LEN_LIST( hdSubs ); 1 <= i; i-- ) {
          if ( libGAP_ELM_PLIST( hdSubs, i ) != 0 ) {
            libGAP_objNums = libGAP_ELM_PLIST( libGAP_ELM_PLIST( hdSubs, i ), 1 );
            ptNums  = &(libGAP_ELM_PLIST(libGAP_objNums,1)) - 1;
            libGAP_objRel  = libGAP_ELM_PLIST( libGAP_ELM_PLIST( hdSubs, i ), 2 );
            ptRel   = &(libGAP_ELM_PLIST(libGAP_objRel,1)) - 1;

            lp = 2;
            lc = 1;
            rp = libGAP_LEN_LIST( libGAP_objRel ) - 1;
            rc = 1;

            /* scan as long as possible from the right to the left         */
            while ( lp<rp && 0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc))) ) {
                rc = tc;  rp = rp - 2;
            }

            /* scan as long as possible from the left to the right         */
            while ( lp<rp && 0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc))) ) {
                lc = tc;  lp = lp + 2;
            }

            /* if a coincidence or deduction has been found, handle it     */
            if ( lp == rp + 1 ) {
              if ( libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc)) != rc ) {
                if ( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[lp],lc) ) > 0 ) {
                    libGAP_HandleCoinc( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[lp],lc) ), rc );
                }
                else if ( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[rp],rc) ) > 0 ) {
                    libGAP_HandleCoinc( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[rp],rc) ), lc );
                }
                else {
                    libGAP_SET_ELM_PLIST( ptRel[lp], lc, libGAP_INTOBJ_INT( rc ) );
                    libGAP_SET_ELM_PLIST( ptRel[rp], rc, libGAP_INTOBJ_INT( lc ) );
                    if ( libGAP_dedlst == libGAP_dedSize )
                        libGAP_CompressDeductionList();
                    libGAP_dedgen[ libGAP_dedlst ] = libGAP_INT_INTOBJ( ptNums[lp] );
                    libGAP_dedcos[ libGAP_dedlst ] = lc;
                    libGAP_dedlst++;
                }
              }

              /* remove the completed subgroup generator                   */
              libGAP_SET_ELM_PLIST( hdSubs, i, 0 );
              if ( i == libGAP_LEN_PLIST(hdSubs) ) {
                  while ( 0 < i  && libGAP_ELM_PLIST(hdSubs,i) == 0 )
                      --i;
                  libGAP_SET_LEN_PLIST( hdSubs, i );
                  i++;
              }
            }

            /* if a minimal gap has been found, set a flag                 */
            else if ( libGAP_minGaps != 0 && lp == rp - 1 ) {
                libGAP_SET_ELM_PLIST( ptRel[lp], lc, libGAP_INTOBJ_INT( -1 ) );
                libGAP_SET_ELM_PLIST( ptRel[rp], rc, libGAP_INTOBJ_INT( -1 ) );
            }
          }
        }

        /* apply all relators that start with this generator               */
        objRels = libGAP_ELM_PLIST( libGAP_ELM_PLIST( list, 4 ), libGAP_dedgen[libGAP_dedfst] );
        for ( i = 1; i <= libGAP_LEN_LIST( objRels ); i++ ) {
            libGAP_objNums = libGAP_ELM_PLIST( libGAP_ELM_PLIST(objRels,i), 1 );
            ptNums  = &(libGAP_ELM_PLIST(libGAP_objNums,1)) - 1;
            libGAP_objRel  = libGAP_ELM_PLIST( libGAP_ELM_PLIST(objRels,i), 2 );
            ptRel   = &(libGAP_ELM_PLIST(libGAP_objRel,1)) - 1;

            lp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_ELM_PLIST(objRels,i), 3 ) );
            lc = libGAP_dedcos[ libGAP_dedfst ];
            rp = lp + libGAP_INT_INTOBJ( ptRel[1] );
            rc = lc;

            /* scan as long as possible from the right to the left         */
            while ( lp<rp && 0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc))) ) {
                rc = tc;  rp = rp - 2;
            }

            /* scan as long as possible from the left to the right         */
            while ( lp<rp && 0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc))) ) {
                lc = tc;  lp = lp + 2;
            }

            /* if a coincidence or deduction has been found, handle it     */
            if ( lp == rp+1 && libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc)) != rc ) {
                if ( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[lp],lc) ) > 0 ) {
                    libGAP_HandleCoinc( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[lp],lc) ), rc );
                }
                else if ( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[rp],rc) ) > 0 ) {
                    libGAP_HandleCoinc( libGAP_INT_INTOBJ( libGAP_ELM_PLIST(ptRel[rp],rc) ), lc );
                }
                else {
                    libGAP_SET_ELM_PLIST( ptRel[lp], lc, libGAP_INTOBJ_INT( rc ) );
                    libGAP_SET_ELM_PLIST( ptRel[rp], rc, libGAP_INTOBJ_INT( lc ) );
                    if ( libGAP_dedlst == libGAP_dedSize )
                        libGAP_CompressDeductionList();
                    libGAP_dedgen[ libGAP_dedlst ] = libGAP_INT_INTOBJ( ptNums[lp] );
                    libGAP_dedcos[ libGAP_dedlst ] = lc;
                    libGAP_dedlst++;
                }
            }

            /* if a minimal gap has been found, set a flag                 */
            else if ( libGAP_minGaps != 0 && lp == rp - 1 ) {
                libGAP_SET_ELM_PLIST( ptRel[lp], lc, libGAP_INTOBJ_INT( -1 ) );
                libGAP_SET_ELM_PLIST( ptRel[rp], rc, libGAP_INTOBJ_INT( -1 ) );
            }
        }

        libGAP_dedfst++;
    }

    libGAP_SET_ELM_PLIST( list, 6, libGAP_INTOBJ_INT( libGAP_firstFree ) );
    libGAP_SET_ELM_PLIST( list, 7, libGAP_INTOBJ_INT( libGAP_lastFree  ) );
    libGAP_SET_ELM_PLIST( list, 8, libGAP_INTOBJ_INT( libGAP_firstDef  ) );
    libGAP_SET_ELM_PLIST( list, 9, libGAP_INTOBJ_INT( libGAP_lastDef   ) );

    /* clean out  */
    libGAP_CleanOut();

    return libGAP_INTOBJ_INT( libGAP_nrdel );
}


/****************************************************************************
**
*F  FuncMakeConsequencesPres( <self>, <list> )  . . . . . . find consequences
**
**  This  is a  special version  of  `FuncMakeConsequences'  for the subgroup
**  presentation routines.
*/
libGAP_Obj libGAP_FuncMakeConsequencesPres (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    libGAP_Obj                 objDefs1;       /* handle of defs list part 1      */
    libGAP_Obj                 objDefs2;       /* handle of defs list part 2      */
    libGAP_Obj                 objRels;        /*                                 */
    libGAP_Obj *               ptRel;          /* pointer to the relator bag      */
    libGAP_Obj *               ptNums;         /* pointer to this list            */
    libGAP_Int                 ndefs;          /* number of defs done so far      */
    libGAP_Int                 undefined;      /* maximal of undefined entreis    */
    libGAP_Int                 apply;          /* num of next def to be applied   */
    libGAP_Int                 ndefsMax;       /* maximal number of definitons    */
    libGAP_Int                 coset;          /* coset involved in current def   */
    libGAP_Int                 gen;            /* gen involved in current def     */
    libGAP_Int                 lp;             /* left pointer into relator       */
    libGAP_Int                 lc;             /* left coset to apply to          */
    libGAP_Int                 rp;             /* right pointer into relator      */
    libGAP_Int                 rc;             /* right coset to apply to         */
    libGAP_Int                 tc;             /* temporary coset                 */
    libGAP_Int                 i;              /* loop variable                   */

    /*T 1996/12/03 fceller this should be replaced by 'PlistConv'          */
    if ( ! libGAP_IS_PLIST(list) ) {
        libGAP_ErrorQuit( "<list> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(list), 0L );
        return 0;
    }

    libGAP_objTable  = libGAP_ELM_PLIST( list, 1 );
    objDefs1  = libGAP_ELM_PLIST( list, 2 );
    objDefs2  = libGAP_ELM_PLIST( list, 3 );

    undefined = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 4 ) );
    ndefs     = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( list, 5 ) );

    /* check the definitions lists                                         */
    if ( ! ( libGAP_IS_PLIST(objDefs1) && libGAP_IS_PLIST(objDefs2) &&
        libGAP_LEN_PLIST(objDefs1) == libGAP_LEN_PLIST(objDefs2) ) ) {
        libGAP_ErrorQuit( "inconsistent definitions lists", 0L, 0L );
        return 0;
    }
    ndefsMax = libGAP_LEN_PLIST(objDefs1);
    apply = 1;

    /* while the deduction queue is not worked off                         */
    while ( apply <= ndefs ) {

        /* apply all relators that start with this generator               */
        coset = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( objDefs1, apply ) );
        gen = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( objDefs2, apply ) );
        objRels = libGAP_ELM_PLIST( libGAP_ELM_PLIST( list, 6 ), gen );
        for ( i = 1; i <= libGAP_LEN_LIST( objRels ); i++ ) {
            libGAP_objNums = libGAP_ELM_PLIST( libGAP_ELM_PLIST(objRels,i), 1 );
            ptNums  = &(libGAP_ELM_PLIST(libGAP_objNums,1)) - 1;
            libGAP_objRel  = libGAP_ELM_PLIST( libGAP_ELM_PLIST(objRels,i), 2 );
            ptRel   = &(libGAP_ELM_PLIST(libGAP_objRel,1)) - 1;

            lp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_ELM_PLIST(objRels,i), 3 ) );
            lc = coset;
            rp = lp + libGAP_INT_INTOBJ( ptRel[1] );
            rc = lc;

            /* scan as long as possible from the right to the left         */
            while ( lp<rp && 0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc))) ) {
                rc = tc;  rp = rp - 2;
            }

            /* scan as long as possible from the left to the right         */
            while ( lp<rp && 0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc))) ) {
                lc = tc;  lp = lp + 2;
            }

            /* if a deduction has been found, handle it     */
            if ( lp == rp+1 && libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc)) <= 0 ) {
                libGAP_SET_ELM_PLIST( ptRel[lp], lc, libGAP_INTOBJ_INT( rc ) );
                undefined--;
                if ( libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc)) <= 0 ) {
                    libGAP_SET_ELM_PLIST( ptRel[rp], rc, libGAP_INTOBJ_INT( lc ) );
                    undefined--;
                }
                ndefs++;
                if ( ndefs > ndefsMax ) {
                    libGAP_ErrorQuit( "inconsistent definitions lists", 0L, 0L );
                    return 0;
                }
                libGAP_SET_ELM_PLIST( objDefs1, ndefs, libGAP_INTOBJ_INT( lc ) );
                libGAP_SET_ELM_PLIST( objDefs2, ndefs, ptNums[lp] );
                if ( undefined == 0 ) {
                    return libGAP_INTOBJ_INT( 0 );
                }
            }
        }

        apply++;
    }

    /* clean out  */
    libGAP_CleanOut();

    return libGAP_INTOBJ_INT( undefined );
}


/****************************************************************************
**
*F  FuncStandardizeTableC(<self>,<table>,<stan>)  . . . . . .  standardize CT
**
**  This is the kernel routine for standardizing a coset table.  It is called
**  by the  GAP routine  'StandardizeTable'.  The user  should  not  call the
**  kernel routine but only the GAP routine.
**
**  If  <stan> = 1  the table  is standardized  using  the  (old)  semilenlex
**  standard.
**  If  not  <stan> = 1  the table  is standardized  using the  (new)  lenlex
**  standard (this is the default).
*/
libGAP_Obj libGAP_FuncStandardizeTableC (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 stan )
{
    libGAP_Obj *               ptTable;        /* pointer to table                */
    libGAP_UInt                nrgen;          /* number of rows of the table / 2 */
    libGAP_Obj *               g;              /* one generator list from table   */
    libGAP_Obj *               h;              /* generator list                  */
    libGAP_Obj *               i;              /*  and inverse                    */
    libGAP_UInt                acos;           /* actual coset                    */
    libGAP_UInt                lcos;           /* last seen coset                 */
    libGAP_UInt                mcos;           /*                                 */
    libGAP_UInt                c1, c2;         /* coset temporaries               */
    libGAP_Obj                 tmp;            /* temporary for swap              */
    libGAP_UInt                j, k, nloop;    /* loop variables                  */

    /* get the arguments                                                   */
    libGAP_objTable = list;
    if ( ! libGAP_IS_PLIST(libGAP_objTable) ) {
        libGAP_ErrorQuit( "<table> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(libGAP_objTable), 0L );
        return 0;
    }
    ptTable  = &(libGAP_ELM_PLIST(libGAP_objTable,1)) - 1;
    nrgen    = libGAP_LEN_PLIST(libGAP_objTable) / 2;
    for ( j = 1;  j <= nrgen*2;  j++ ) {
        if ( ! libGAP_IS_PLIST(ptTable[j]) ) {
            libGAP_ErrorQuit(
                "<table>[%d] must be a plain list (not a %s)",
                (libGAP_Int)j,
                (libGAP_Int)libGAP_TNAM_OBJ(ptTable[j]) );
            return 0;
        }
    }
    if ( libGAP_IS_INTOBJ(stan) && libGAP_INT_INTOBJ(stan) == 1 ) {
       /* use semilenlex standard                                          */
       nloop = nrgen;
    }
    else {
       /* use lenlex standard                                              */
       nloop = nrgen*2;
    }

    /* run over all cosets                                                 */
    acos = 1;
    lcos = 1;
    while ( acos <= lcos ) {

        /* scan through all columns of acos                                */
        for ( j = 1;  j <= nloop;  j++ ) {
            k = ( nloop == nrgen ) ? 2*j - 1 : j;
            g = &(libGAP_ELM_PLIST(ptTable[k],1)) - 1;

            /* if we haven't seen this coset yet                           */
            if ( lcos+1 < libGAP_INT_INTOBJ( g[acos] ) ) {

                /* swap rows lcos and g[acos]                              */
                lcos = lcos + 1;
                mcos = libGAP_INT_INTOBJ( g[acos] );
                for ( k = 1;  k <= nrgen;  k++ ) {
                    h = &(libGAP_ELM_PLIST(ptTable[2*k-1],1)) - 1;
                    i = &(libGAP_ELM_PLIST(ptTable[2*k],1)) - 1;
                    c1 = libGAP_INT_INTOBJ( h[lcos] );
                    c2 = libGAP_INT_INTOBJ( h[mcos] );
                    if ( c1 != 0 )  i[c1] = libGAP_INTOBJ_INT( mcos );
                    if ( c2 != 0 )  i[c2] = libGAP_INTOBJ_INT( lcos );
                    tmp     = h[lcos];
                    h[lcos] = h[mcos];
                    h[mcos] = tmp;
                    if ( i != h ) {
                        c1 = libGAP_INT_INTOBJ( i[lcos] );
                        c2 = libGAP_INT_INTOBJ( i[mcos] );
                        if ( c1 != 0 )  h[c1] = libGAP_INTOBJ_INT( mcos );
                        if ( c2 != 0 )  h[c2] = libGAP_INTOBJ_INT( lcos );
                        tmp     = i[lcos];
                        i[lcos] = i[mcos];
                        i[mcos] = tmp;
                    }
                }

            }

            /* if this is already the next only bump lcos                  */
            else if ( lcos < libGAP_INT_INTOBJ( g[acos] ) ) {
                lcos = lcos + 1;
            }

        }

        acos = acos + 1;
    }

    /* shrink the table                                                    */
    for ( j = 1; j <= nrgen; j++ ) {
        libGAP_SET_LEN_PLIST( ptTable[2*j-1], lcos );
        libGAP_SET_LEN_PLIST( ptTable[2*j  ], lcos );
    }

    /* clean out  */
    libGAP_CleanOut();

    /* return void                                                         */
    return 0;
}


/****************************************************************************
**
*F  InitializeCosetFactorWord() . . . . . . .  initialize a coset factor word
**
**  'InitializeCosetFactorWord'  initializes  a word  in  which  a new  coset
**  factor is to be built up.
**
**  'wordList', 'treeType', 'objTree2', and  'treeWordLength' are assumed  to
**  be known as static variables.
*/
static void libGAP_InitializeCosetFactorWord ( void )
{
    libGAP_Obj *               ptWord;         /* pointer to the word             */
    libGAP_Int                 i;              /* integer variable                */

    /* handle the one generator MTC case                                   */
    if ( libGAP_treeType == 1 ) {
        libGAP_objWordValue = libGAP_INTOBJ_INT(0);
    }

    /* handle the abelianized case                                         */
    else if ( libGAP_treeType == 0 ) {
        ptWord = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
        for ( i = 1;  i <= libGAP_treeWordLength;  i++ ) {
            ptWord[i] = libGAP_INTOBJ_INT(0);
        }
    }

    /* handle the general case                                             */
    else {
        libGAP_wordList[0] = 0;
    }
}


/****************************************************************************
**
*F  TreeEntryC()  . . . . . . . . . . . . returns a tree entry for a rep word
**
**  'TreeEntryC'  determines a tree entry  which represents the word given in
**  'wordList', if it finds any, or it defines a  new proper tree entry,  and
**  then returns it.
**
**  Warning:  It is assumed,  but not checked,  that the given word is freely
**  reduced  and that it does  not contain zeros,  and that the  tree type is
**  either 0 or 2.
**
**  'wordList'  is assumed to be known as static variable.
**
*/
static libGAP_Int libGAP_TreeEntryC ( void )
{
    libGAP_Obj *               ptTree1;        /* ptr to first tree component     */
    libGAP_Obj *               ptTree2;        /* ptr to second tree component    */
    libGAP_Obj *               ptWord;         /* ptr to given word               */
    libGAP_Obj *               ptFac;          /* ptr to old word                 */
    libGAP_Obj *               ptNew;          /* ptr to new word                 */
    libGAP_Obj                 objNew;         /* handle of new word              */
    libGAP_Int                 treesize;       /* tree size                       */
    libGAP_Int                 numgens;        /* tree length                     */
    libGAP_Int                 leng;           /* word length                     */
    libGAP_Int                 sign;           /* sign flag                       */
    libGAP_Int                 i, k;           /* integer variables               */
    libGAP_Int                 gen;            /* generator value                 */
    libGAP_Int                 u, u1, u2;      /* generator values                */
    libGAP_Int                 v, v1, v2;      /* generator values                */
    libGAP_Int                 t1, t2;         /* generator values                */
    libGAP_Int                 uabs, vabs;     /* generator values                */

    /*  Get the tree components                                            */
    ptTree1  = &(libGAP_ELM_PLIST(libGAP_objTree1,1)) - 1;
    ptTree2  = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
    treesize = libGAP_LEN_PLIST(libGAP_objTree1);
    numgens  = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objTree, 3 ) );

    /* handle the abelianized case                                         */
    if ( libGAP_treeType == 0 )
    {
        ptWord = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
        for ( leng = libGAP_treeWordLength;  leng >= 1;  leng-- ) {
            if ( ptWord[leng] != libGAP_INTOBJ_INT(0) )  {
                break;
            }
        }
        if ( leng == 0 )  {
            return 0;
        }
        for ( k = 1; k <= leng; k++ ) {
            if ( ptWord[k] != libGAP_INTOBJ_INT(0) )  { 
                break;
            }
        }
        sign = 1;
        if ( libGAP_INT_INTOBJ( ptWord[k] ) < 0 ) {

            /* invert the word                                             */
            sign = - 1;
            for ( i = k; i <= leng; i++ ) {
                ptWord[i] = libGAP_INTOBJ_INT( - libGAP_INT_INTOBJ( ptWord[i] ) );
            }
        }
        for ( k = 1; k <= numgens; k++ ) {
            ptFac = &(libGAP_ELM_PLIST(ptTree1[k],1)) - 1;
            if ( libGAP_LEN_PLIST(ptTree1[k]) == leng ) {
                for ( i = 1;  i <= leng;  i++ ) {
                    if ( ptFac[i] != ptWord[i] )  {
                        break;
                    }
                }
                if ( i > leng )  {
                    return sign * k;
                }
            }
        }

        /* extend the tree                                                 */
        numgens++;
        if ( treesize < numgens ) {
            treesize = 2 * treesize;
            libGAP_GROW_PLIST( libGAP_objTree1, treesize );
            libGAP_CHANGED_BAG(libGAP_objTree);
        }
        objNew = libGAP_NEW_PLIST( libGAP_T_PLIST, leng );
        libGAP_SET_LEN_PLIST( objNew, leng );

        libGAP_SET_ELM_PLIST( libGAP_objTree, 3, libGAP_INTOBJ_INT(numgens) );

        libGAP_SET_LEN_PLIST( libGAP_objTree1, treesize );
        libGAP_SET_ELM_PLIST( libGAP_objTree1, numgens, objNew );
        libGAP_CHANGED_BAG(libGAP_objTree1);

        /* copy the word to the new bag                                    */
        ptWord = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
        ptNew  = &(libGAP_ELM_PLIST(objNew,1)) - 1;
        while ( leng > 0 ) {
            ptNew[leng] = ptWord[leng];
            leng--;
        }

        return sign * numgens;
    }

    /* handle the general case                                             */

    /*  Get the length of the word                                         */
    leng = libGAP_wordList[0];

    gen = ( leng == 0 ) ? 0 : libGAP_wordList[1];
    u2  = 0; /* just to shut up gcc */
    for ( i = 2;  i <= leng;  i++ ) {
        u = gen;
        v = libGAP_wordList[i];
        while ( i ) {

            /*  First handle the trivial cases                             */
            if ( u == 0 || v == 0 || ( u + v ) == 0 ) {
                gen = u + v;
                break;
            }

            /*  Cancel out factors, if possible                            */
            u1 = libGAP_INT_INTOBJ( ptTree1[ (u > 0) ? u : -u ] );
            if ( u1 != 0 ) {
                if ( u > 0 ) {
                    u2 = libGAP_INT_INTOBJ( ptTree2[u] );
                }
                else {
                    u2 = - u1;
                    u1 = - libGAP_INT_INTOBJ( ptTree2[-u] );
                }
                if ( u2 == -v ) {
                    gen = u1;
                    break;
                }
            }
            v1 = libGAP_INT_INTOBJ( ptTree1[ (v > 0) ? v : -v ] );
            if ( v1 != 0 ) {
                if ( v > 0 ) {
                    v2 = libGAP_INT_INTOBJ( ptTree2[v] );
                }
                else {
                    v2 = - v1;
                    v1 = - libGAP_INT_INTOBJ( ptTree2[-v] );
                }
                if ( v1 == -u ) {
                    gen = v2;
                    break;
                }
                if ( u1 != 0 && v1 == - u2 ) {
                    u = u1;
                    v = v2;
                    continue;
                }
            }

            /*  Check if there is already a tree entry [u,v] or [-v,-u]    */
            if ( u < -v ) {
                t1 = u;
                t2 = v;
            }
            else {
                t1 = -v;
                t2 = -u;
            }
            uabs = ( u > 0 ) ? u : -u;
            vabs = ( v > 0 ) ? v : -v;
            k = ( uabs > vabs ) ? uabs : vabs;
            for ( k++; k <= numgens; k++ ) {
                if ( libGAP_INT_INTOBJ(ptTree1[k]) == t1 &&
                     libGAP_INT_INTOBJ(ptTree2[k]) == t2 )
                {
                    break;
                }
            }

            /*  Extend the tree, if necessary                              */
            if ( k > numgens ) {
                numgens++;
                if ( treesize < numgens ) {
                    treesize = 2 * treesize;
                    libGAP_GROW_PLIST( libGAP_objTree1, treesize );
                    libGAP_GROW_PLIST( libGAP_objTree2, treesize );
                    ptTree1 = &(libGAP_ELM_PLIST(libGAP_objTree1,1)) - 1;
                    ptTree2 = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
                    libGAP_SET_LEN_PLIST( libGAP_objTree1, treesize );
                    libGAP_SET_LEN_PLIST( libGAP_objTree2, treesize );
                    libGAP_CHANGED_BAG(libGAP_objTree);
                }
                ptTree1[numgens] = libGAP_INTOBJ_INT( t1 );
                ptTree2[numgens] = libGAP_INTOBJ_INT( t2 );
                libGAP_SET_ELM_PLIST( libGAP_objTree, 3, libGAP_INTOBJ_INT(numgens) );
            }
            gen = ( u > - v ) ? -k : k;
            break;
        }
    }

    return gen;
}


/****************************************************************************
**
*F  AddCosetFactor2( <factor> ) . add a factor to a coset representative word
**
**  'AddCosetFactor2'  adds  a  factor  to a  coset  representative word  and
**  extends the tree appropriately, if necessary.
**
**  'treeType', 'wordList', and 'wordSize'  are assumed to be known as static
**  variables, and 'treeType' is assumed to be either 0 or 2,
**
**  Warning: 'factor' is not checked for being zero.
*/
static void libGAP_AddCosetFactor2 (
    libGAP_Int                factor )
{
    libGAP_Obj *               ptFac;          /* pointer to the factor           */
    libGAP_Obj *               ptWord;         /* pointer to the word             */
    libGAP_Int                 leng;           /* length of the factor            */
    libGAP_Obj                 sum;            /* intermediate result             */
    libGAP_Int                 i;              /* integer variable                */
    libGAP_Obj                 tmp;

    /* handle the abelianized case                                         */
    if ( libGAP_treeType == 0 ) {
        ptWord = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
        if ( factor > 0 ) {
            tmp   = libGAP_ELM_PLIST( libGAP_objTree1, factor );
            ptFac = &(libGAP_ELM_PLIST(tmp,1)) - 1;
            leng  = libGAP_LEN_PLIST(tmp);
            for ( i = 1;  i <= leng;  i++ ) {
                if ( ! libGAP_SUM_INTOBJS( sum, ptWord[i], ptFac[i] ) ) {
                    libGAP_ErrorQuit(
                        "exponent too large, Modified Todd-Coxeter aborted",
                        0L, 0L );
                    return;
                }
                ptWord[i] = sum;
            }
        }
        else
        {
            tmp   = libGAP_ELM_PLIST( libGAP_objTree1, -factor );
            ptFac = &(libGAP_ELM_PLIST(tmp,1)) - 1;
            leng  = libGAP_LEN_PLIST(tmp);
            for ( i = 1;  i <= leng;  i++ ) {
                if ( ! libGAP_DIFF_INTOBJS( sum, ptWord[i], ptFac[i] ) ) {
                    libGAP_ErrorQuit(
                        "exponent too large, Modified Todd-Coxeter aborted",
                        0L, 0L );
                    return;
                }
                ptWord[i] = sum;
            }
        }
    }

    /* handle the general case                                             */
    else if ( libGAP_wordList[0] == 0 ) {
        libGAP_wordList[++libGAP_wordList[0]] = factor;
    }
    else if ( libGAP_wordList[libGAP_wordList[0]] == -factor ) {
        --libGAP_wordList[0];
    }
    else if ( libGAP_wordList[0] < libGAP_wordSize ) {
        libGAP_wordList[++libGAP_wordList[0]] = factor;
    }
    else {
        libGAP_wordList[0] = ( libGAP_wordList[1] = libGAP_TreeEntryC( ) == 0 ) ? 0 : 1;
        libGAP_AddCosetFactor2(factor);
    }
}


/****************************************************************************
**
*F  FuncApplyRel2( <self>, <app>, <rel>, <nums> ) . . . . . . apply a relator
**
**  'FunApplyRel2' implements the internal function 'ApplyRel2'.
**
**  'ApplyRel2( <app>, <rel>, <nums> )'
**
**  'ApplyRel2'  applies  the relator  <rel>  to a  coset representative  and
**  returns the corresponding factors in "word"
**
**  ...more about ApplyRel2...
*/
libGAP_Obj libGAP_FuncApplyRel2 (
    libGAP_Obj                 self,
    libGAP_Obj                 app,
    libGAP_Obj                 rel,
    libGAP_Obj                 nums )
{
    libGAP_Obj *               ptApp;          /* pointer to that list            */
    libGAP_Obj                 word;           /* handle of resulting word        */
    libGAP_Obj *               ptWord;         /* pointer to this word            */
    libGAP_Obj *               ptTree;         /* pointer to the tree             */
    libGAP_Obj *               ptTree2;        /* ptr to second tree component    */
    libGAP_Obj *               ptRel;          /* pointer to the relator bag      */
    libGAP_Obj *               ptNums;         /* pointer to this list            */
    libGAP_Obj *               ptTabl2;        /* pointer to coset factor table   */
    libGAP_Obj                 objRep;         /* handle of temporary factor      */
    libGAP_Int                 lp;             /* left pointer into relator       */
    libGAP_Int                 lc;             /* left coset to apply to          */
    libGAP_Int                 rp;             /* right pointer into relator      */
    libGAP_Int                 rc;             /* right coset to apply to         */
    libGAP_Int                 rep;            /* temporary factor                */
    libGAP_Int                 tc;             /* temporary coset                 */
    libGAP_Int                 bound;          /* maximal number of steps         */
    libGAP_Int                 last;           /* proper word length              */
    libGAP_Int                 size;           /* size of the word bag            */
    libGAP_Int                 i;              /* loop variables                  */
    libGAP_Int                 tmp;

    /* get and check the application list                                  */
    if ( ! libGAP_IS_PLIST(app) ) {
        libGAP_ErrorQuit( "<app> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(app), 0L );
        return 0;
    }
    if ( libGAP_LEN_PLIST(app) != 9 ) {
        libGAP_ErrorQuit( "<app> must be a list of length 9 not %d",
                   (libGAP_Int) libGAP_LEN_PLIST(app), 0L );
        return 0;
    }
    ptApp = &(libGAP_ELM_PLIST(app,1)) - 1;

    /* get the components of the proper application list                   */
    lp = libGAP_INT_INTOBJ( ptApp[1] );
    lc = libGAP_INT_INTOBJ( ptApp[2] );
    rp = libGAP_INT_INTOBJ( ptApp[3] );
    rc = libGAP_INT_INTOBJ( ptApp[4] );

    /* get and check the relator (well, only a little bit)                 */
    libGAP_objRel = rel;
    if ( ! libGAP_IS_PLIST(rel) ) {
        libGAP_ErrorQuit( "<rel> must be a plain list (not a %s)", 
                   (libGAP_Int)libGAP_TNAM_OBJ(rel), 0L );
        return 0;
    }

    /* fix right pointer if requested                                      */
    if ( rp == -1 )
        rp = lp + libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objRel,1) );

    /* get and check the numbers list parallel to the relator              */
    libGAP_objNums = nums;
    if ( ! libGAP_IS_PLIST(libGAP_objNums) ) {
        libGAP_ErrorQuit( "<nums> must be a plain list (not a %s)", 
                   (libGAP_Int)libGAP_TNAM_OBJ(libGAP_objNums), 0L );
        return 0;
    }

    /* get and check the corresponding factors list                        */
    libGAP_objTable2 = ptApp[6];
    if ( ! libGAP_IS_PLIST(libGAP_objTable2) ) {
        libGAP_ErrorQuit( "<nums> must be a plain list (not a %s)", 
                   (libGAP_Int)libGAP_TNAM_OBJ(libGAP_objTable2), 0L );
        return 0;
    }

    /* get the tree type                                                   */
    libGAP_treeType = libGAP_INT_INTOBJ( ptApp[5] );

    /* handle the one generator MTC case                                   */
    if ( libGAP_treeType == 1 ) {

        /* initialize the resulting exponent by zero                       */
        libGAP_objExponent = libGAP_INTOBJ_INT( 0 );

        /* scan as long as possible from the left to the right             */
        while ( lp < rp + 2 &&
                0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
        {
            tmp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objNums,lp) );
            objRep = libGAP_ELM_PLIST( libGAP_objTable2, tmp );
            objRep = libGAP_ELM_PLIST( objRep, lc );
            libGAP_objExponent = libGAP_DiffInt( libGAP_objExponent, objRep );
            lc = tc;
            lp = lp + 2;
        }

        /* scan as long as possible from the right to the left             */
        while ( lp < rp + 2 &&
                0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
        {
            tmp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objNums,rp) );
            objRep = libGAP_ELM_PLIST( libGAP_objTable2, tmp );
            objRep = libGAP_ELM_PLIST( objRep, rc );
            libGAP_objExponent = libGAP_SumInt( libGAP_objExponent, objRep );
            rc = tc;
            rp = rp - 2;
        }

        /* The functions DiffInt or SumInt may have caused a garbage       */
        /* collections. So restore the pointer.                            */

        /* save the resulting exponent                                     */
        libGAP_SET_ELM_PLIST( app, 9, libGAP_objExponent );
    }

    else {

        /* get and check the corresponding word                            */
        word = ptApp[7];
        if ( ! libGAP_IS_PLIST(word) ) {
            libGAP_ErrorQuit( "<word> must be a plain list (not a %s)", 
                       (libGAP_Int)libGAP_TNAM_OBJ(word), 0L );
            return 0;
        }

        /* handle the abelianized case                                     */
        if ( libGAP_treeType == 0 ) {
            libGAP_objTree  = ptApp[8];
            libGAP_objTree1 = libGAP_ELM_PLIST( libGAP_objTree, 1 );
            libGAP_objTree2 = libGAP_ELM_PLIST( libGAP_objTree, 2 );
            ptTree   = &(libGAP_ELM_PLIST(libGAP_objTree,1)) - 1;
            libGAP_treeWordLength = libGAP_INT_INTOBJ( ptTree[4] );
            if ( libGAP_LEN_PLIST(libGAP_objTree2) != libGAP_treeWordLength ) {
                libGAP_ErrorQuit( "ApplyRel2: illegal word length", 0L, 0L );
                return 0;
            }

            /* initialize the coset representative word                    */
            libGAP_InitializeCosetFactorWord();

            /* scan as long as possible from the left to the right         */
            while ( lp < rp + 2 &&
                    0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
            {
                tmp    = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objNums,lp) );
                objRep = libGAP_ELM_PLIST(libGAP_objTable2,tmp);
                objRep = libGAP_ELM_PLIST(objRep,lc);
                rep    = libGAP_INT_INTOBJ(objRep);
                if ( rep != 0 ) {
                    libGAP_AddCosetFactor2(-rep);
                }
                lc = tc;
                lp = lp + 2;
            }

            /* scan as long as possible from the right to the left         */
            while ( lp < rp + 2 &&
                    0 < (tc=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
            {
                tmp    = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objNums,rp) );
                objRep = libGAP_ELM_PLIST(libGAP_objTable2,tmp);
                objRep = libGAP_ELM_PLIST(objRep,rc);
                rep    = libGAP_INT_INTOBJ(objRep);
                if ( rep != 0 ) {
                    libGAP_AddCosetFactor2(rep);
                }
                rc = tc;
                rp = rp - 2;
            }

            /* initialize some local variables                             */
            ptWord  = &(libGAP_ELM_PLIST(word,1)) - 1;
            ptTree2 = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;

            /* copy the result to its destination, if necessary            */
            if ( ptWord != ptTree2 ) {
                if ( libGAP_LEN_PLIST(word) != libGAP_treeWordLength ) {
                    libGAP_ErrorQuit( "illegal word length", 0L, 0L );
                    return 0;
                }
                for ( i = 1;  i <= libGAP_treeWordLength;  i++ ) {
                    ptWord[i] = ptTree2[i];
                }
                libGAP_SET_LEN_PLIST( word, libGAP_LEN_PLIST(libGAP_objTree2) );
            }
        }

        /* handle the general case                                         */
        else {

            /* extend the word size, if necessary                          */
            bound = ( rp - lp + 3 ) / 2;
            size  = libGAP_SIZE_OBJ(word)/sizeof(libGAP_Obj) - 1;
            if ( size < bound ) {
                size = ( bound > 2 * size ) ? bound : 2 * size;
                libGAP_GROW_PLIST( word, size );
                libGAP_CHANGED_BAG(app);
            }

            /* initialize some local variables                             */
            ptRel   = &(libGAP_ELM_PLIST(libGAP_objRel,1)) - 1;
            ptNums  = &(libGAP_ELM_PLIST(libGAP_objNums,1)) - 1;
            ptTabl2 = &(libGAP_ELM_PLIST(libGAP_objTable2,1)) - 1;
            ptWord  = &(libGAP_ELM_PLIST(word,1)) - 1;
            last    = 0;

            /* scan as long as possible from the left to the right         */
            while ( lp < rp + 2
                  && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc))) )
            {
                objRep = libGAP_ELM_PLIST( ptTabl2[libGAP_INT_INTOBJ(ptNums[lp])], lc );
                rep    = libGAP_INT_INTOBJ(objRep);
                if ( rep != 0 ) {
                    if ( last > 0 && libGAP_INT_INTOBJ(ptWord[last]) == rep ) {
                        last--;
                    }
                    else {
                        ptWord[++last] = libGAP_INTOBJ_INT(-rep);
                    }
                }
                lc = tc;
                lp = lp + 2;
            }

            /* revert the ordering of the word constructed so far          */
            if ( last > 0 ) {
                last++;
                for ( i = last / 2;  i > 0;  i-- ) {
                    objRep = ptWord[i];
                    ptWord[i] = ptWord[last-i];
                    ptWord[last-i] = objRep;
                }
                last--;
            }

            /* scan as long as possible from the right to the left         */
            while ( lp < rp + 2
                 && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc))) )
            {
                objRep = libGAP_ELM_PLIST( ptTabl2[libGAP_INT_INTOBJ(ptNums[rp])], rc );
                rep    = libGAP_INT_INTOBJ(objRep);
                if ( rep != 0 ) {
                    if ( last > 0 && libGAP_INT_INTOBJ(ptWord[last]) == -rep ) {
                        last--;
                    }
                    else {
                        ptWord[++last] = libGAP_INTOBJ_INT(rep);
                    }
                }
                rc = tc;
                rp = rp - 2;
            }

            /* save the word length                                        */
            libGAP_SET_LEN_PLIST( word, last );
        }
    }

    /* copy the information back into the application list                 */
    libGAP_SET_ELM_PLIST( app, 1, libGAP_INTOBJ_INT( lp ) );
    libGAP_SET_ELM_PLIST( app, 2, libGAP_INTOBJ_INT( lc ) );
    libGAP_SET_ELM_PLIST( app, 3, libGAP_INTOBJ_INT( rp ) );
    libGAP_SET_ELM_PLIST( app, 4, libGAP_INTOBJ_INT( rc ) );

    /* return nothing                                                      */
    return 0;
}


/****************************************************************************
**
*F  FuncCopyRel( <self>, <rel> )   . . . . . . . . . . . .  copy of a relator
**
**  'FuncCopyRel' returns a copy  of the given RRS  relator such that the bag
**  of the copy does not exceed the minimal required size.
*/
libGAP_Obj libGAP_FuncCopyRel ( 
    libGAP_Obj                 self,
    libGAP_Obj                 rel )           /* the given relator               */
{
    libGAP_Obj *               ptRel;          /* pointer to the given relator    */
    libGAP_Obj                 copy;           /* the copy                        */
    libGAP_Obj *               ptCopy;         /* pointer to the copy             */
    libGAP_Int                 leng;           /* length of the given word        */

    /* Get and check argument                                              */
    if ( ! libGAP_IS_PLIST(rel) ) {
        libGAP_ErrorQuit( "<rel> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(rel), 0L );
        return 0;
    }
    leng = libGAP_LEN_PLIST(rel);

    /*  Allocate a bag for the copy                                        */
    copy   = libGAP_NEW_PLIST( libGAP_T_PLIST, leng );
    libGAP_SET_LEN_PLIST( copy, leng );
    ptRel  = &(libGAP_ELM_PLIST(rel,1));
    ptCopy = &(libGAP_ELM_PLIST(copy,1));

    /*  Copy the relator to the new bag                                    */
    while ( leng > 0 ) {
        *ptCopy++ = *ptRel++; 
        leng--;
    }

    /*  Return the copy                                                    */
    return copy;
}


/****************************************************************************
**
*F  FuncMakeCanonical( <self>, <rel> ) . . . . . . . make a relator canonical
**
**  'FuncMakeCanonical' is a subroutine  of the Reduced Reidemeister-Schreier
**  routines.  It replaces the given relator by its canonical representative.
**  It does not return anything.
*/
libGAP_Obj libGAP_FuncMakeCanonical (
    libGAP_Obj                 self,
    libGAP_Obj                 rel )           /* the given relator               */
{
    libGAP_Obj *               ptRel;          /* pointer to the relator          */
    libGAP_Obj                 obj1,  obj2;    /* handles 0f relator entries      */
    libGAP_Int                 leng, leng1;    /* length of the relator           */
    libGAP_Int                 max, min, next; /* relator entries                 */
    libGAP_Int                 i, j, k, l;     /* integer variables               */
    libGAP_Int                 ii, jj, kk;     /* integer variables               */

    /* Get and check the argument                                          */
    if ( ! libGAP_IS_PLIST(rel) ) {
        libGAP_ErrorQuit( "<rel> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(rel), 0L );
        return 0;
    }
    ptRel = &(libGAP_ELM_PLIST(rel,1));
    leng  = libGAP_LEN_PLIST(rel);
    leng1 = leng - 1;

    /*  cyclically reduce the relator, if necessary                        */
    i = 0;
    while ( i<leng1 && libGAP_INT_INTOBJ(ptRel[i]) == -libGAP_INT_INTOBJ(ptRel[leng1]) ) {
        i++;
        leng1--;
    }
    if ( i > 0 ) {
        for ( j = i;  j <= leng1;  j++ ) {
            ptRel[j-i] = ptRel[j];
        }
        leng1 = leng1 - i;
        leng  = leng1 + 1;
        libGAP_SET_LEN_PLIST( rel, leng );
    }

    /*  Loop over the relator and find the maximal postitve and negative   */
    /*  entries                                                            */
    max = min = libGAP_INT_INTOBJ(ptRel[0]);
    i = 0;  j = 0;
    for ( k = 1;  k < leng;  k++ ) {
        next = libGAP_INT_INTOBJ( ptRel[k] );
        if ( next > max ) {
            max = next; 
            i = k;
        }
        else if ( next <= min ) {
            min = next;
            j = k;
        }
    }

    /*  Find the lexicographically last cyclic permutation of the relator  */
    if ( max < -min ) {
        i = leng;
    }
    else {
        for ( k = i + 1;  k < leng;  k++ ) {
            for ( ii = i, kk = k, l = 0;
                  l < leng;
                  ii = (ii + 1) % leng, kk = (kk + 1) % leng, l++ )
            {
                if ( libGAP_INT_INTOBJ(ptRel[kk]) < libGAP_INT_INTOBJ(ptRel[ii]) ) {
                    break;
                }
                else if ( libGAP_INT_INTOBJ(ptRel[kk]) > libGAP_INT_INTOBJ(ptRel[ii]) ) {
                    i = k; 
                    break;
                }
            }
            if ( l == leng ) {
                break;
            }
        }
    }

    /*  Find the lexicographically last cyclic permutation of its inverse  */
    if ( -max < min ) {
        j = leng;
    }
    else {
        for ( k = j - 1;  k >= 0;  k-- ) {
            for ( jj = j, kk = k, l = 0;
                  l < leng;
                  jj = (jj + leng1) % leng, kk = (kk + leng1) % leng, l++ )
            {
                if ( libGAP_INT_INTOBJ(ptRel[kk]) > libGAP_INT_INTOBJ(ptRel[jj]) ) {
                    break;
                }
                else if ( libGAP_INT_INTOBJ(ptRel[kk]) < libGAP_INT_INTOBJ(ptRel[jj]) ) {
                    j = k;
                    break;
                }
            }
            if ( l == leng ) {
                break;
            }
        }
    }

    /*  Compare the two words and find the lexicographically last one      */
    if ( -min == max ) {
        for ( ii = i, jj = j, l = 0;
              l < leng;
              ii = (ii + 1) % leng, jj = (jj + leng1) % leng, l++ )
        {
            if ( - libGAP_INT_INTOBJ(ptRel[jj]) < libGAP_INT_INTOBJ(ptRel[ii]) ) {
                break;
            }
            else if ( - libGAP_INT_INTOBJ(ptRel[jj]) > libGAP_INT_INTOBJ(ptRel[ii]) ) {
                i = leng; 
                break;
            }
        }
    }

    /*  Invert the given relator, if necessary                             */
    if ( i == leng ) {
        for ( k = 0;  k < leng / 2;  k++ ) {
            next = libGAP_INT_INTOBJ( ptRel[k] );
            ptRel[k] = libGAP_INTOBJ_INT( - libGAP_INT_INTOBJ( ptRel[leng1-k] ) );
            ptRel[leng1-k] = libGAP_INTOBJ_INT( - next );
        }
        if ( leng % 2 ) {
            ptRel[leng1/2] = libGAP_INTOBJ_INT( - libGAP_INT_INTOBJ( ptRel[leng1/2] ) );
        }
        i = leng1 - j;
    }

    /*  Now replace the given relator by the resulting word                */
    if ( i > 0 ) {
        k = libGAP_INT_INTOBJ( libGAP_GcdInt( libGAP_INTOBJ_INT(i), libGAP_INTOBJ_INT(leng) ) );
        l = leng / k;
        leng1 = leng - i;
        for ( j = 0; j < k; j++ ) {
            jj = (j + i) % leng;
            obj1 = ptRel[jj];
            for ( ii = 0; ii < l; ii++ ) {
                jj = (jj + leng1) % leng;
                obj2 = ptRel[jj];  ptRel[jj] = obj1;  obj1 = obj2;
            }
        }
    }

    /* return nothing                                                      */
    return 0;
}


/****************************************************************************
**
*F  FuncTreeEntry( <self>, <tree>, <word> )  .  tree entry for the given word
**
**  'FuncTreeEntry' determines  a tree entry  which represents the given word
**  in the  current generators, if  it finds any, or it  defines a new proper
**  tree entry, and then returns it.
*/
libGAP_Obj libGAP_FuncTreeEntry(
    libGAP_Obj                 self,
    libGAP_Obj                 tree,
    libGAP_Obj                 word )
{
    libGAP_Obj *               ptTree1;        /* pointer to that component       */
    libGAP_Obj *               ptTree2;        /* pointer to that component       */
    libGAP_Obj *               ptWord;         /* pointer to that word            */
    libGAP_Obj                 new;            /* handle of new word              */
    libGAP_Obj *               ptNew;          /* pointer to new word             */
    libGAP_Obj *               ptFac;          /* pointer to old word             */
    libGAP_Int                 treesize;       /* tree size                       */
    libGAP_Int                 numgens;        /* tree length                     */
    libGAP_Int                 leng;           /* word length                     */
    libGAP_Int                 sign;           /* integer variable                */
    libGAP_Int                 i, j, k;        /* integer variables               */
    libGAP_Int                 gen;            /* generator value                 */
    libGAP_Int                 u, u1, u2;      /* generator values                */
    libGAP_Int                 v, v1, v2;      /* generator values                */
    libGAP_Int                 t1, t2;         /* generator values                */
    libGAP_Int                 uabs, vabs;     /* generator values                */

    /*  Get and check the first argument (tree)                            */
    libGAP_objTree = tree;
    if ( ! libGAP_IS_PLIST(tree) || libGAP_LEN_PLIST(tree) < 5 ) {
        libGAP_ErrorQuit( "invalid <tree>", 0L, 0L );
        return 0;
    }

    /*  Get and check the tree components                                  */
    libGAP_objTree1 = libGAP_ELM_PLIST(libGAP_objTree,1);
    if ( ! libGAP_IS_PLIST(libGAP_objTree1) ) {
        libGAP_ErrorQuit( "invalid <tree>[1]", 0L, 0L );
        return 0;
    }
    libGAP_objTree2 = libGAP_ELM_PLIST(libGAP_objTree,2);
    if ( ! libGAP_IS_PLIST(libGAP_objTree2) ) {
        libGAP_ErrorQuit( "invalid <tree>[2]", 0L, 0L );
        return 0;
    }
    ptTree1  = &(libGAP_ELM_PLIST(libGAP_objTree1,1)) - 1;
    ptTree2  = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
    treesize = libGAP_LEN_PLIST(libGAP_objTree1);
    numgens  = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objTree, 3 ) );
    libGAP_treeWordLength = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objTree, 4 ) );
    libGAP_treeType = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objTree, 5 ) );

    /*  Get the second argument (word)                                     */
    if ( ! libGAP_IS_PLIST(word) ) {
        libGAP_ErrorQuit( "invalid <word>", 0L, 0L );
        return 0;
    }

    /* handle the abelianized case                                         */
    ptWord = &(libGAP_ELM_PLIST(word,1)) - 1;
    if ( libGAP_treeType == 0 ) {
        if ( libGAP_LEN_PLIST(word) != libGAP_treeWordLength ) {
            libGAP_ErrorQuit( "inconsistent <word> length", 0L, 0L );
            return 0;
        }
        ptWord = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
        for ( leng = libGAP_treeWordLength;  leng >= 1;  leng-- ) {
            if ( ptWord[leng] != libGAP_INTOBJ_INT(0) ) {
                break;
            }
        }
        if ( leng == 0 ) {
            return libGAP_INTOBJ_INT( 0 );
        }

        for ( k = 1; k <= leng; k++ ) {
            if ( ptWord[k] != libGAP_INTOBJ_INT(0) ) {
                break;
            }
        }
        sign = 1;

        /* invert the word                                                 */
        if ( libGAP_INT_INTOBJ(ptWord[k]) < 0 ) {
            sign = -1;
            for ( i = k; i <= leng; i++ ) {
                ptWord[i] = libGAP_INTOBJ_INT( - libGAP_INT_INTOBJ( ptWord[i] ) );
            }
        }

        for ( k = 1;  k <= numgens;  k++ ) {
            ptFac = &(libGAP_ELM_PLIST(ptTree1[k],1)) - 1;
            if ( libGAP_LEN_PLIST(ptTree1[k]) == leng ) {
                for ( i = 1;  i <= leng;  i++ ) {
                    if ( ptFac[i] != ptWord[i] ) {
                        break;
                    }
                }
                if ( i > leng ) {
                    return libGAP_INTOBJ_INT( sign * k );
                }
            }
        }

        /* extend the tree                                                 */
        numgens++;
        if ( treesize < numgens ) {
            treesize = 2 * treesize;
            libGAP_GROW_PLIST( libGAP_objTree1, treesize );
            libGAP_SET_LEN_PLIST( libGAP_objTree1, treesize );
            libGAP_CHANGED_BAG(libGAP_objTree);
        }
        new = libGAP_NEW_PLIST( libGAP_T_PLIST, leng );
        libGAP_SET_LEN_PLIST( new, leng );

        libGAP_SET_ELM_PLIST( libGAP_objTree, 3, libGAP_INTOBJ_INT(numgens) );
        libGAP_SET_ELM_PLIST( libGAP_objTree1, numgens, new );
        libGAP_CHANGED_BAG(libGAP_objTree1);

        /* copy the word to the new bag                                    */
        ptWord = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
        ptNew  = &(libGAP_ELM_PLIST(new,1)) - 1;
        while ( leng > 0 ) {
            ptNew[leng] = ptWord[leng];
            leng--;
        }

        return libGAP_INTOBJ_INT( sign * numgens );
    }

    /* handle the general case                                             */
    if ( libGAP_LEN_PLIST(libGAP_objTree1) != libGAP_LEN_PLIST(libGAP_objTree2) ) {
        libGAP_ErrorQuit( "inconsistent <tree> components", 0L, 0L );
        return 0;
    }

    for ( i = 1;  i <= numgens;  i++ ) {
        if ( libGAP_INT_INTOBJ(ptTree1[i]) <= -i || libGAP_INT_INTOBJ(ptTree1[i]) >= i
          || libGAP_INT_INTOBJ(ptTree2[i]) <= -i || libGAP_INT_INTOBJ(ptTree2[i]) >= i )
        {
            libGAP_ErrorQuit( "invalid <tree> components", 0L, 0L );
            return 0;
        }
    }

    /*  Freely reduce the given word                                       */
    leng = libGAP_LEN_PLIST(word);
    for ( j = 0, i = 1;  i <= leng;  i++ ) {
        gen = libGAP_INT_INTOBJ(ptWord[i]);
        if ( gen == 0 ) {
            continue;
        }
        if ( gen > numgens || gen < -numgens ) {
            libGAP_ErrorQuit( "invalid <word> entry [%d]", i, 0L );
            return 0;
        }
        if ( j > 0 && gen == - libGAP_INT_INTOBJ(ptWord[j]) ) {
            j--;
        }
        else {
            ptWord[++j] = ptWord[i];
        }
    }
    for ( i = j + 1;  i <= leng;  i++ ) {
        ptWord[i] = libGAP_INTOBJ_INT( 0 );
    }
    leng = j;

    gen = ( leng == 0 ) ? 0 : libGAP_INT_INTOBJ( ptWord[1] );
    u2 = 0; /* just to shut up gcc */
    for ( i = 2;  i <= leng;  i++ ) {
        u = gen;
        v = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(word,i) );
        while ( i ) {

            /*  First handle the trivial cases                             */
            if ( u == 0 || v == 0 || ( u + v ) == 0 ) {
                gen = u + v;
                break;
            }

            /*  Cancel out factors, if possible                            */
            u1 = libGAP_INT_INTOBJ( ptTree1[ (u > 0) ? u : -u ] );
            if ( u1 != 0 ) {
                if ( u > 0 ) {
                    u2 = libGAP_INT_INTOBJ( ptTree2[u] );
                }
                else {
                    u2 = - u1;
                    u1 = - libGAP_INT_INTOBJ( ptTree2[-u] );
                }
                if ( u2 == -v ) {
                    gen = u1;
                    break;
                }
            }
            v1 = libGAP_INT_INTOBJ( ptTree1[ (v > 0) ? v : -v ] );
            if ( v1 != 0 ) {
                if ( v > 0 ) {
                    v2 = libGAP_INT_INTOBJ( ptTree2[v] );
                }
                else {
                    v2 = - v1;
                    v1 = - libGAP_INT_INTOBJ( ptTree2[-v] );
                }
                if ( v1 == -u ) {
                    gen = v2;
                    break;
                }
                if ( u1 != 0 && v1 == - u2 ) {
                    u = u1;
                    v = v2;
                    continue;
                }
            }

            /*  Check if there is already a tree entry [u,v] or [-v,-u]    */
            if ( u < -v ) {
                t1 = u; 
                t2 = v;
            }
            else {
                t1 = -v; 
                t2 = -u;
            }
            uabs = ( u > 0 ) ? u : -u;
            vabs = ( v > 0 ) ? v : -v;
            k = ( uabs > vabs ) ? uabs : vabs;
            for ( k++;  k <= numgens;  k++ ) {
                if ( libGAP_INT_INTOBJ(ptTree1[k]) == t1 &&
                     libGAP_INT_INTOBJ(ptTree2[k]) == t2 )
                {
                    break;
                }
            }

            /*  Extend the tree, if necessary                              */
            if ( k > numgens ) {
                numgens++;
                if ( treesize < numgens ) {
                    treesize = 2 * treesize;
                    libGAP_GROW_PLIST( libGAP_objTree1, treesize );
                    libGAP_GROW_PLIST( libGAP_objTree2, treesize );
                    libGAP_SET_LEN_PLIST( libGAP_objTree1, treesize );
                    libGAP_SET_LEN_PLIST( libGAP_objTree2, treesize );
                    ptTree1 = &(libGAP_ELM_PLIST(libGAP_objTree1,1)) - 1;
                    ptTree2 = &(libGAP_ELM_PLIST(libGAP_objTree2,1)) - 1;
                    libGAP_CHANGED_BAG(libGAP_objTree);
                }
                ptTree1[numgens] = libGAP_INTOBJ_INT( t1 );
                ptTree2[numgens] = libGAP_INTOBJ_INT( t2 );
                libGAP_SET_ELM_PLIST( libGAP_objTree, 3, libGAP_INTOBJ_INT( numgens ) );
            }
            gen = ( u > - v ) ? -k : k;
            break;
        }
    }

    return libGAP_INTOBJ_INT( gen );
}


/****************************************************************************
**
*F  AddCosetFactor( <factor> ) . . . . . . . . . . . . add a coset rep factor
**
**  'AddCosetFactor' adds a factor to a coset representative word by changing
**  its exponent appropriately.
**
**  'treeType', 'objWordValue',  and  'objExponent'  are assumed to be known as
**  static variables, and 'treeType' is assumed to be 1.
**
**  Warning: 'factor' is not checked for being zero.
*/
static void libGAP_AddCosetFactor (
    libGAP_Obj                 factor )
{
    /* handle the one generator MTC case                                   */
    libGAP_objWordValue = libGAP_SumInt( libGAP_objWordValue, factor );
    if ( libGAP_objExponent != libGAP_INTOBJ_INT(0) ) {
        libGAP_objWordValue = libGAP_RemInt( libGAP_objWordValue, libGAP_objExponent );
    }
}


/****************************************************************************
**
*F  SubtractCosetFactor( <factor> )  . . . . . .  subtract a coset rep factor
**
**  'SubtractCosetFactor' subtracts a factor from a coset representative word
**  by changing its exponent appropriately.
**
**  'treeType', 'objWordValue',  and  'objExponent'  are assumed to be known as
**  static variables, and 'treeType' is assumed to be 1.
**
**  Warning: 'factor' is not checked for being zero.
*/
static void libGAP_SubtractCosetFactor (
    libGAP_Obj                 factor )
{
    /* handle the one generator MTC case                                   */
    libGAP_objWordValue = libGAP_DiffInt( libGAP_objWordValue, factor );
    if ( libGAP_objExponent != libGAP_INTOBJ_INT(0) ) {
        libGAP_objWordValue = libGAP_RemInt( libGAP_objWordValue, libGAP_objExponent );
    }
}


/****************************************************************************
**
*F  HandleCoinc2( <cos1>, <cos2>, <factor> ) .  handle coincidences in an MTC
**
**  'HandleCoinc2' is a subroutine of 'FuncMakeConsequences2' and handles the
**  coincidence  cos2 = factor * cos1.
*/
static void libGAP_HandleCoinc2 (
    libGAP_Int                 cos1,
    libGAP_Int                 cos2,
    libGAP_Obj                 factor )
{
    libGAP_Obj                 f,  ff2;        /* handles of temporary factors    */
    libGAP_Obj                 f1, f2;         /* handles of temporary factors    */
    libGAP_Obj                 rem;            /* handle of remainder             */
    libGAP_Obj                 tmp;            /* temporary variable              */
    libGAP_Obj *               gen2;
    libGAP_Obj *               gen;
    libGAP_Obj *               inv2;
    libGAP_Obj *               inv;
    libGAP_Obj *               ptNext;
    libGAP_Obj *               ptPrev;
    libGAP_Int                 c1, c2;
    libGAP_Int                 firstCoinc;
    libGAP_Int                 i, j;           /* loop variables                  */
    libGAP_Int                 lastCoinc;
    libGAP_Int                 length;         /* length of coset rep word        */
    libGAP_Int                 save;           /* temporary factor                */

    /* return, if cos1 = cos2                                              */
    if ( cos1 == cos2 ) {

        /* but pick up a relator before in case treeType = 1               */
        if ( libGAP_treeType == 1 && factor != libGAP_INTOBJ_INT(0) ) {
            if ( libGAP_objExponent == libGAP_INTOBJ_INT(0) ) {
                libGAP_objExponent = factor;
            }
            else {
                rem = libGAP_RemInt( factor, libGAP_objExponent );
                while ( rem != libGAP_INTOBJ_INT(0) ) {
                    factor = libGAP_objExponent;
                    libGAP_objExponent = rem;
                    rem = libGAP_RemInt( factor, libGAP_objExponent );
                }
            }
        }
        return;
    }

    /* take the smaller one as new representative                          */
    if ( cos2 < cos1 ) {
        save = cos1;  cos1 = cos2;  cos2 = save;
        factor = ( libGAP_treeType == 1 ) ?
            libGAP_DiffInt( libGAP_INTOBJ_INT(0), factor ) :
            libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ(factor) );
    }

    /* get some pointers                                                   */
    ptNext = &(libGAP_ELM_PLIST(libGAP_objNext,1)) - 1;
    ptPrev = &(libGAP_ELM_PLIST(libGAP_objPrev,1)) - 1;

    /* if we are removing an important coset update it                     */
    if ( cos2 == libGAP_lastDef ) {
        libGAP_lastDef  = libGAP_INT_INTOBJ( ptPrev[libGAP_lastDef ] );
    }
    if ( cos2 == libGAP_firstDef ) {
        libGAP_firstDef = libGAP_INT_INTOBJ( ptPrev[libGAP_firstDef] );
    }

    /* remove <cos2> from the coset list                                   */
    ptNext[libGAP_INT_INTOBJ(ptPrev[cos2])] = ptNext[cos2];
    if ( ptNext[cos2] != libGAP_INTOBJ_INT(0) ) {
        ptPrev[libGAP_INT_INTOBJ(ptNext[cos2])] = ptPrev[cos2];
    }

    /* put the first coincidence into the list of coincidences             */
    firstCoinc        = cos2;
    lastCoinc         = cos2;
    ptNext[lastCoinc] = libGAP_INTOBJ_INT(0);

    /* <cos1> is the representative of <cos2> and its own representative   */
    ptPrev[cos2] = libGAP_INTOBJ_INT(cos1);
    libGAP_SET_ELM_PLIST( libGAP_objFactor, cos2, factor );

    /* while there are coincidences to handle                              */
    while ( firstCoinc != 0 ) {

        /* replace <firstCoinc> by its representative in the table         */
        cos2   = firstCoinc;
        cos1   = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objPrev, cos2 ) );
        factor = libGAP_ELM_PLIST( libGAP_objFactor, cos2 );
        for ( i = 1;  i <= libGAP_LEN_PLIST(libGAP_objTable);  i++ ) {
            j = i + 2*(i % 2) - 1;

            /* replace <cos2> by <cos1> in the column of <gen>^-1          */
            gen  = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable,i),1)) - 1;
            gen2 = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable2,i),1)) - 1;
            c2 = libGAP_INT_INTOBJ(gen[cos2]);
            if ( c2 != 0 ) {
                f2 = gen2[cos2];
                c1 = libGAP_INT_INTOBJ(gen[cos1]);

                /* if the other entry is empty copy it                     */
                if ( c1 == 0 )  {
                    if ( f2 == factor ) {
                        ff2 = libGAP_INTOBJ_INT(0);
                    }
                    else {
                        if ( libGAP_treeType == 1 ) {
                            libGAP_objWordValue = libGAP_INTOBJ_INT(0);
                            if ( factor != libGAP_INTOBJ_INT(0) ) {
                                libGAP_SubtractCosetFactor(factor);
                            }
                            if ( f2 != libGAP_INTOBJ_INT(0) ) {
                                libGAP_AddCosetFactor( f2 );
                            }
                            ff2 = libGAP_objWordValue;
                        }
                        else {
                            libGAP_InitializeCosetFactorWord();
                            if ( factor != libGAP_INTOBJ_INT(0) ) {
                                libGAP_AddCosetFactor2( -libGAP_INT_INTOBJ(factor) );
                            }
                            if ( f2 != libGAP_INTOBJ_INT(0) ) {
                                libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(f2) );
                            }
                            ff2 = libGAP_INTOBJ_INT(libGAP_TreeEntryC());
                        }
                    }
                    tmp =  ( libGAP_treeType == 1 ) ?
                        libGAP_DiffInt( libGAP_INTOBJ_INT(0), ff2 ) :
                        libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ(ff2) );
                    gen  = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable,i),1)) - 1;
                    gen2 = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable2,i),1)) - 1;
                    inv  = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable,j),1)) - 1;
                    inv2 = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable2,j),1)) - 1;
                    gen[cos1]  = libGAP_INTOBJ_INT(c2);
                    gen2[cos1] = ff2;
                    gen[cos2]  = libGAP_INTOBJ_INT(0);
                    gen2[cos2] = libGAP_INTOBJ_INT(0);
                    inv[c2]    = libGAP_INTOBJ_INT(cos1);
                    inv2[c2]   = tmp;
                    if ( libGAP_dedlst == libGAP_dedSize ) {
                        libGAP_CompressDeductionList();
                    }
                    libGAP_dedgen[libGAP_dedlst] = i;
                    libGAP_dedcos[libGAP_dedlst] = cos1;
                    libGAP_dedlst++;
                }

                /* otherwise check for a coincidence                       */
                else {
                    f1         = gen2[cos1];
                    inv        = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable,j),1)) - 1;
                    inv2       = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable2,j),1)) - 1;
                    inv[c2]    = libGAP_INTOBJ_INT(0);
                    inv2[c2]   = libGAP_INTOBJ_INT(0);
                    gen[cos2]  = libGAP_INTOBJ_INT(0);
                    gen2[cos2] = libGAP_INTOBJ_INT(0);

                    /* if gen = inv and c2 = cos1, reset the table entries */
                    if ( gen[cos1] == libGAP_INTOBJ_INT(0) ) {
                        if ( f2 == factor ) {
                            ff2 = libGAP_INTOBJ_INT(0);
                        }
                        else {
                            if ( libGAP_treeType == 1 ) {
                                libGAP_objWordValue = libGAP_INTOBJ_INT(0);
                                if ( factor != libGAP_INTOBJ_INT(0) ) {
                                    libGAP_SubtractCosetFactor(factor);
                                }
                                if ( f2 != libGAP_INTOBJ_INT(0) ) {
                                    libGAP_AddCosetFactor(f2);
                                }
                                ff2 = libGAP_objWordValue;
                            }
                            else {
                                libGAP_InitializeCosetFactorWord();
                                if ( factor != libGAP_INTOBJ_INT(0) ) {
                                    libGAP_AddCosetFactor2( -libGAP_INT_INTOBJ(factor) );
                                }
                                if ( f2 != libGAP_INTOBJ_INT(0) ) {
                                    libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(f2) );
                                }
                                ff2 = libGAP_INTOBJ_INT( libGAP_TreeEntryC() );
                            }
                            gen  = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable,i),1))-1;
                            gen2 = &(libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objTable2,i),1))-1;
                        }
                        gen[cos1]  = libGAP_INTOBJ_INT(cos1);
                        gen2[cos1] = ff2;
                        if ( libGAP_dedlst == libGAP_dedSize ) {
                            libGAP_CompressDeductionList();
                        }
                        libGAP_dedgen[libGAP_dedlst] = i;
                        libGAP_dedcos[libGAP_dedlst] = cos1;
                        libGAP_dedlst++;
                    }

                    /* initialize the factor for the new coincidence       */
                    libGAP_InitializeCosetFactorWord();

                    /* find the representative of <c2>                     */

                    /* handle the one generator MTC case                   */
                    if ( libGAP_treeType == 1 ) {

                        if ( f2 != libGAP_INTOBJ_INT(0) ) {
                           libGAP_SubtractCosetFactor(f2);
                        }
                        while ( c2 != 1 && libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objNext,
                                libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_objPrev,c2)))) != c2 )
                        {
                            f2 = libGAP_ELM_PLIST(libGAP_objFactor,c2);
                            c2 = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objPrev,c2) );
                            if ( f2 != libGAP_INTOBJ_INT(0) ) {
                                libGAP_SubtractCosetFactor(f2);
                            }
                        }
                        if ( factor != libGAP_INTOBJ_INT(0) ) {
                            libGAP_AddCosetFactor(factor);
                        }
                        if ( f1 != libGAP_INTOBJ_INT(0) ) {
                           libGAP_AddCosetFactor(f1);
                        }
                    }

                    /* handle the abelianized case                         */
                    else if ( libGAP_treeType == 0 ) {
                        if ( f2 != libGAP_INTOBJ_INT(0) ) {
                           libGAP_AddCosetFactor2( -libGAP_INT_INTOBJ(f2) );
                        }
                        while ( c2 != 1 && libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objNext,
                                libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_objPrev,c2)))) != c2 )
                        {
                            f2 = libGAP_ELM_PLIST(libGAP_objFactor,c2);
                            c2 = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objPrev,c2) );
                            if ( f2 != libGAP_INTOBJ_INT(0) ) {
                                libGAP_AddCosetFactor2( -libGAP_INT_INTOBJ(f2) );
                            }
                        }
                        if ( factor != libGAP_INTOBJ_INT(0) ) {
                            libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(factor) );
                        }
                        if ( f1 != libGAP_INTOBJ_INT(0) ) {
                            libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(f1) );
                        }
                    }

                    /* handle the general case                             */
                    else
                    {
                        if ( f2 != libGAP_INTOBJ_INT(0) ) {
                            libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(f2) );
                        }
                        while ( c2 != 1 && libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objNext,
                                libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_objPrev,c2)))) != c2 )
                        {
                            f2 = libGAP_ELM_PLIST(libGAP_objFactor,c2);
                            c2 = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objPrev,c2) );
                            if ( f2 != libGAP_INTOBJ_INT(0) ) {
                                libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(f2) );
                            }
                        }

                        /* invert the word constructed so far              */
                        if ( libGAP_wordList[0] > 0 ) {
                            length = libGAP_wordList[0] + 1;
                            for ( i = length / 2;  i > 0;  i-- ) {
                                save = libGAP_wordList[i];
                                libGAP_wordList[i] = - libGAP_wordList[length-i];
                                libGAP_wordList[length-i] = - save;
                            }
                        }
                        if ( factor != libGAP_INTOBJ_INT(0) ) {
                            libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(factor) );
                        }
                        if ( f1 != libGAP_INTOBJ_INT(0) ) {
                            libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(f1) );
                        }
                    }

                    /* find the representative of <c1>                     */
                    while ( c1 != 1 && libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_objNext,
                            libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_objPrev,c1)))) != c1 )
                    {
                        f1 = libGAP_ELM_PLIST(libGAP_objFactor,c1);
                        c1 = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objPrev,c1) );
                        if ( f1 != libGAP_INTOBJ_INT(0) ) {
                            if ( libGAP_treeType == 1 ) {
                                libGAP_AddCosetFactor( f1 );
                            }
                            else {
                                libGAP_AddCosetFactor2( libGAP_INT_INTOBJ(f1) );
                            }
                        }
                    }

                    /* if the representatives differ we got a coincidence  */
                    if ( c1 != c2 ) {

                        /* get the quotient of c2 by c1                    */
                        f = (libGAP_treeType == 1 ) ?
                            libGAP_objWordValue : libGAP_INTOBJ_INT(libGAP_TreeEntryC());

                        /* take the smaller one as new representative      */
                        if ( c2 < c1 ) {
                             save = c1;  c1 = c2;  c2 = save;
                             f = ( libGAP_treeType == 1 ) ?
                                 libGAP_DiffInt( libGAP_INTOBJ_INT(0), f ) :
                                 libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ(f) );
                        }

                        /* get some pointers                               */
                        ptNext = &(libGAP_ELM_PLIST(libGAP_objNext,1)) - 1;
                        ptPrev = &(libGAP_ELM_PLIST(libGAP_objPrev,1)) - 1;

                        /* if we are removing an important coset update it */
                        if ( c2 == libGAP_lastDef ) {
                            libGAP_lastDef  = libGAP_INT_INTOBJ(ptPrev[libGAP_lastDef]);
                        }
                        if ( c2 == libGAP_firstDef ) {
                            libGAP_firstDef = libGAP_INT_INTOBJ(ptPrev[libGAP_firstDef]);
                        }

                        /* remove <c2> from the coset list                 */
                        ptNext[libGAP_INT_INTOBJ(ptPrev[c2])] = ptNext[c2];
                        if ( ptNext[c2] != libGAP_INTOBJ_INT(0) ) {
                            ptPrev[libGAP_INT_INTOBJ(ptNext[c2])] = ptPrev[c2];
                        }

                        /* append <c2> to the coincidence list             */
                        ptNext[lastCoinc] = libGAP_INTOBJ_INT(c2);
                        lastCoinc         = c2;
                        ptNext[lastCoinc] = libGAP_INTOBJ_INT(0);

                        /* <c1> is the rep of <c2> and its own rep.        */
                        ptPrev[c2] = libGAP_INTOBJ_INT( c1 );
                        libGAP_SET_ELM_PLIST( libGAP_objFactor, c2, f );
                    }

                    /* pick up a relator in case treeType = 1              */
                    else if ( libGAP_treeType == 1 ) {
                        f = libGAP_objWordValue;
                        if ( f != libGAP_INTOBJ_INT(0) ) {
                            if ( libGAP_objExponent == libGAP_INTOBJ_INT(0) ) {
                                libGAP_objExponent = f;
                            }
                            else {
                                rem = libGAP_RemInt( f, libGAP_objExponent );
                                while ( rem != libGAP_INTOBJ_INT(0) ) {
                                    f = libGAP_objExponent;
                                    libGAP_objExponent = rem;
                                    rem = libGAP_RemInt( f, libGAP_objExponent );
                                }
                            }
                        }
                    }
                }
            }
        }

        /* move the replaced coset to the free list                        */
        ptNext = &(libGAP_ELM_PLIST(libGAP_objNext,1)) - 1;
        if ( libGAP_firstFree == 0 ) {
            libGAP_firstFree      = firstCoinc;
            libGAP_lastFree       = firstCoinc;
        }
        else {
            ptNext[libGAP_lastFree] = libGAP_INTOBJ_INT(firstCoinc);
            libGAP_lastFree         = firstCoinc;
        }
        firstCoinc = libGAP_INT_INTOBJ( ptNext[firstCoinc] );
        ptNext[libGAP_lastFree] = libGAP_INTOBJ_INT(0);

        libGAP_nrdel++;
    }
}


/****************************************************************************
**
*F  FuncMakeConsequences2( <self>, <list> )  . . . . . . .  find consequences
*/
libGAP_Obj libGAP_FuncMakeConsequences2 (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    libGAP_Obj                 subs;           /*                                 */
    libGAP_Obj                 rels;           /*                                 */
    libGAP_Obj *               ptRel;          /* pointer to the relator bag      */
    libGAP_Int                 lp;             /* left pointer into relator       */
    libGAP_Int                 lc;             /* left coset to apply to          */
    libGAP_Int                 rp;             /* right pointer into relator      */
    libGAP_Int                 rc;             /* right coset to apply to         */
    libGAP_Int                 tc;             /* temporary coset                 */
    libGAP_Int                 length;         /* length of coset rep word        */
    libGAP_Obj                 objNum;         /* handle of temporary factor      */
    libGAP_Obj                 objRep;         /* handle of temporary factor      */
    libGAP_Int                 rep;            /* temporary factor                */
    libGAP_Int                 i, j;           /* loop variables                  */
    libGAP_Obj                 tmp;            /* temporary variable              */

    /* get the list of arguments                                           */
    if ( ! libGAP_IS_PLIST(list) ) {
        libGAP_ErrorQuit( "<list> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(list), 0L );
        return 0;
    }
    if ( libGAP_LEN_PLIST(list) != 16 ) {
        libGAP_ErrorQuit( "<list> must be a list of length 16", 0L, 0L );
        return 0;
    }

    /* get the coset table, the corresponding factor table, the subgroup   */
    /* generators tree, and its components                                 */
    libGAP_objTable       = libGAP_ELM_PLIST(list,1);
    libGAP_objTable2      = libGAP_ELM_PLIST(list,12);
    libGAP_objTree        = libGAP_ELM_PLIST(list,14);
    libGAP_objTree1       = libGAP_ELM_PLIST(libGAP_objTree,1);
    libGAP_objTree2       = libGAP_ELM_PLIST(libGAP_objTree,2);
    libGAP_treeType       = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objTree,5) );
    libGAP_treeWordLength = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list,15) );
    libGAP_objExponent    = libGAP_ELM_PLIST(list,16);

    libGAP_objNext        = libGAP_ELM_PLIST(list,2);
    libGAP_objPrev        = libGAP_ELM_PLIST(list,3);
    libGAP_objFactor      = libGAP_ELM_PLIST(list,13);

    libGAP_firstFree      = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list,6) );
    libGAP_lastFree       = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list,7) );
    libGAP_firstDef       = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list,8) );
    libGAP_lastDef        = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list,9) );

    libGAP_nrdel          = 0;

    /* initialize the deduction queue                                      */
    libGAP_dedprint  = 0;
    libGAP_dedfst    = 0;
    libGAP_dedlst    = 1;
    libGAP_dedgen[0] = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list,10) );
    libGAP_dedcos[0] = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list,11) );

    /* while the deduction queue is not empty                              */
    while ( libGAP_dedfst < libGAP_dedlst ) {

        /* skip the deduction, if it got irrelevant by a coincidence       */
        tmp = libGAP_ELM_PLIST( libGAP_objTable, libGAP_dedgen[libGAP_dedfst] );
        tmp = libGAP_ELM_PLIST( tmp, libGAP_dedcos[libGAP_dedfst] );
        if ( libGAP_INT_INTOBJ(tmp) == 0 ) {
            libGAP_dedfst++;
            continue;
        }

        /* while there are still subgroup generators apply them            */
        subs = libGAP_ELM_PLIST(list,5);
        for ( i = libGAP_LEN_PLIST(subs);  1 <= i;  i-- ) {
          if ( libGAP_ELM_PLIST(subs,i) != 0 ) {
              tmp     = libGAP_ELM_PLIST(subs,i);
              libGAP_objNums = libGAP_ELM_PLIST(tmp,1);
              libGAP_objRel  = libGAP_ELM_PLIST(tmp,2);
              ptRel   = &(libGAP_ELM_PLIST(libGAP_objRel,1)) - 1;

              lp = 2;
              lc = 1;
              rp = libGAP_LEN_PLIST(libGAP_objRel) - 1;
              rc = 1;

              /* scan as long as possible from the left to the right       */
              while ( lp < rp 
                      && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc))) )
              {
                  lc = tc;
                  lp = lp + 2;
              }

              /* scan as long as possible from the right to the left       */
              while ( lp < rp
                      && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc))) )
              {
                  rc = tc;
                  rp = rp - 2;
              }

              /* scan once more, but now with factors, if a coincidence or */
              /* a deduction has been found                                */
              if (lp == rp+1 && libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc)) != rc) {
                  lp = 2;
                  lc = 1;
                  rp = libGAP_LEN_PLIST(libGAP_objRel) - 1;
                  rc = 1;

                  /* initialize the coset representative word              */
                  libGAP_InitializeCosetFactorWord();

                  /* scan as long as possible from the left to the right   */

                  /* handle the one generator MTC case                     */
                  if ( libGAP_treeType == 1 ) {
                      while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ(
                              libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
                      {
                          objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                          objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                          objRep = libGAP_ELM_PLIST(objRep,lc);
                          if ( objRep != libGAP_INTOBJ_INT(0) ) {
                              libGAP_SubtractCosetFactor(objRep);
                          }
                          lc = tc;
                          lp = lp + 2;
                      }

                      /* add the factor defined by the ith subgrp generator*/
                      if ( i != 0 ) {
                          libGAP_AddCosetFactor( libGAP_INTOBJ_INT(i) );
                      }

                      /* scan as long as poss from the right to the left   */
                      while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ(
                              libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
                      {
                          objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                          objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                          objRep = libGAP_ELM_PLIST(objRep,rc);
                          if ( objRep != libGAP_INTOBJ_INT(0) ) {
                              libGAP_AddCosetFactor(objRep);
                          }
                          rc = tc;
                          rp = rp - 2;
                      }
                  }

                  /* handle the abelianized case                           */
                  else if ( libGAP_treeType == 0 ) {
                      while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ(
                              libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
                      {
                          objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                          objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                          objRep = libGAP_ELM_PLIST(objRep,lc);
                          rep    = libGAP_INT_INTOBJ(objRep);
                          if ( rep != 0 ) {
                              libGAP_AddCosetFactor2(-rep);
                          }
                          lc = tc;
                          lp = lp + 2;
                      }

                      /* add the factor defined by the ith subgrp generator*/
                      if ( i != 0 ) {
                          libGAP_AddCosetFactor2(i);
                      }

                      /* scan as long as poss from the right to the left   */
                      while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ(
                              libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
                      {
                          objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                          objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                          objRep = libGAP_ELM_PLIST(objRep,rc);
                          rep    = libGAP_INT_INTOBJ(objRep);
                          if ( rep != 0 ) {
                              libGAP_AddCosetFactor2(rep);
                          }
                          rc = tc;
                          rp = rp - 2;
                      }
                  }

                  /* handle the general case                               */
                  else {
                      while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ(
                              libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
                      {
                          objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                          objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                          objRep = libGAP_ELM_PLIST(objRep,lc);
                          rep    = libGAP_INT_INTOBJ(objRep);
                          if ( rep != 0 ) {
                              libGAP_AddCosetFactor2(rep);
                          }
                          lc = tc;
                          lp = lp + 2;
                      }

                      /* invert the word constructed so far                */
                      if ( libGAP_wordList[0] > 0 ) {
                          length = libGAP_wordList[0] + 1;
                          for ( j = length / 2; j > 0; j-- ) {
                              rep = libGAP_wordList[j];
                              libGAP_wordList[j] = - libGAP_wordList[length-j];
                              libGAP_wordList[length-j] = - rep;
                          }
                      }

                      /* add the factor defined by the ith subgrp generator*/
                      if ( i != 0 ) {
                          libGAP_AddCosetFactor2(i);
                      }

                      /* scan as long as poss from the right to the left   */
                      while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ(
                              libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
                      {
                          objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                          objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                          objRep = libGAP_ELM_PLIST(objRep,rc);
                          rep    = libGAP_INT_INTOBJ(objRep);
                          if ( rep != 0 ) {
                              libGAP_AddCosetFactor2(rep);
                          }
                          rc = tc;
                          rp = rp - 2;
                      }
                  }

                  /* enter the word into the tree and return its number    */
                  objNum = ( libGAP_treeType == 1 ) ?
                      libGAP_objWordValue : libGAP_INTOBJ_INT(libGAP_TreeEntryC());

                  /* work off a coincidence                                */
                  if ( lp >= rp + 2 ) {
                      libGAP_HandleCoinc2( rc, lc, objNum );
                  }

                  /* enter a decuction to the tables                       */
                  else {
                      objRep = libGAP_ELM_PLIST(libGAP_objRel,lp);
                      libGAP_SET_ELM_PLIST( objRep, lc, libGAP_INTOBJ_INT(rc) );

                      objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                      objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                      libGAP_SET_ELM_PLIST( objRep, lc, objNum );

                      objRep = libGAP_ELM_PLIST(libGAP_objRel,rp);
                      libGAP_SET_ELM_PLIST( objRep, rc, libGAP_INTOBJ_INT(lc) );

                      tmp = ( libGAP_treeType == 1 ) ?
                          libGAP_DiffInt( libGAP_INTOBJ_INT(0), objNum ) :
                          libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ( objNum ) );
                      objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                      objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                      libGAP_SET_ELM_PLIST( objRep, rc, tmp );

                      if ( libGAP_dedlst == libGAP_dedSize ) {
                          libGAP_CompressDeductionList();
                      }
                      libGAP_dedgen[libGAP_dedlst] = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objNums,lp) );
                      libGAP_dedcos[libGAP_dedlst] = lc;
                      libGAP_dedlst++;
                  }

                  /* remove the completed subgroup generator               */
                  libGAP_SET_ELM_PLIST( subs, i, 0 );
                  if ( i == libGAP_LEN_PLIST(subs) ) {
                      while ( 0 < i  && libGAP_ELM_PLIST(subs,i) == 0 ) {
                          --i;
                      }
                      libGAP_SET_LEN_PLIST( subs, i );
                  }
              }
          }
        }

        /* apply all relators that start with this generator               */
        rels = libGAP_ELM_PLIST( libGAP_ELM_PLIST(list,4), libGAP_dedgen[libGAP_dedfst] );
        for ( i = 1;  i <= libGAP_LEN_PLIST(rels);  i++ ) {
            tmp     = libGAP_ELM_PLIST(rels,i);
            libGAP_objNums = libGAP_ELM_PLIST(tmp,1);
            libGAP_objRel  = libGAP_ELM_PLIST(tmp,2);
            ptRel   = &(libGAP_ELM_PLIST(libGAP_objRel,1)) - 1;

            lp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(tmp,3) );
            lc = libGAP_dedcos[libGAP_dedfst];
            rp = lp + libGAP_INT_INTOBJ(ptRel[1]);
            rc = lc;

            /* scan as long as possible from the left to the right         */
            while (lp < rp && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc))))
            {
                lc = tc;
                lp = lp + 2;
            }

            /* scan as long as possible from the right to the left         */
            while (lp < rp && 0 < (tc = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[rp],rc))))
            {
                rc = tc;
                rp = rp - 2;
            }

            /* scan once more, but now with factors, if a coincidence or a */
            /* deduction has been found                                    */
            if ( lp == rp+1 && ( libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ptRel[lp],lc)) != rc
                 || libGAP_treeType == 1 ) )
            {

                lp = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( libGAP_ELM_PLIST(rels,i), 3 ) );
                lc = libGAP_dedcos[libGAP_dedfst];
                rp = lp + libGAP_INT_INTOBJ(ptRel[1]);
                rc = lc;

                /* initialize the coset representative word                */
                libGAP_InitializeCosetFactorWord();

                /* scan as long as possible from the left to the right     */
                /* handle the one generator MTC case                       */

                if ( libGAP_treeType == 1 ) {

                    while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(
                            libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
                    {
                        objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                        objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                        objRep = libGAP_ELM_PLIST(objRep,lc);
                        if ( objRep != libGAP_INTOBJ_INT(0) ) {
                            libGAP_SubtractCosetFactor(objRep);
                        }
                        lc = tc;
                        lp = lp + 2;
                    }

                    /* scan as long as possible from the right to the left */
                    while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(
                            libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
                    {
                        objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                        objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                        objRep = libGAP_ELM_PLIST(objRep,rc);
                        if ( objRep != libGAP_INTOBJ_INT(0) ) {
                            libGAP_AddCosetFactor( objRep );
                        }
                        rc = tc;
                        rp = rp - 2;
                    }
                }

                /* handle the abelianized case                             */
                else if ( libGAP_treeType == 0 ) {
                    while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(
                            libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
                    {
                        objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                        objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                        objRep = libGAP_ELM_PLIST(objRep,lc);
                        rep    = libGAP_INT_INTOBJ(objRep);
                        if ( rep != 0 ) {
                            libGAP_AddCosetFactor2(-rep);
                        }
                        lc = tc;
                        lp = lp + 2;
                    }

                    /* scan as long as possible from the right to the left */
                    while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(
                            libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
                    {
                        objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                        objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                        objRep = libGAP_ELM_PLIST(objRep,rc);
                        rep    = libGAP_INT_INTOBJ(objRep);
                        if ( rep != 0 ) {
                            libGAP_AddCosetFactor2(rep);
                        }
                        rc = tc;
                        rp = rp - 2;
                    }
                }

                /* handle the general case                                 */
                else {
                    while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(
                            libGAP_ELM_PLIST(libGAP_objRel,lp),lc))) )
                    {
                        objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                        objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                        objRep = libGAP_ELM_PLIST(objRep,lc);
                        rep    = libGAP_INT_INTOBJ(objRep);
                        if ( rep != 0 ) {
                            libGAP_AddCosetFactor2(rep);
                        }
                        lc = tc;
                        lp = lp + 2;
                    }

                    /* invert the word constructed so far                  */
                    if ( libGAP_wordList[0] > 0 ) {
                        length = libGAP_wordList[0] + 1;
                        for ( j = length / 2;  j > 0;  j-- ) {
                            rep = libGAP_wordList[j];
                            libGAP_wordList[j] = - libGAP_wordList[length-j];
                            libGAP_wordList[length-j] = - rep;
                        }
                    }

                    /* scan as long as possible from the right to the left */
                    while ( lp < rp + 2 && 0 < (tc = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(
                            libGAP_ELM_PLIST(libGAP_objRel,rp),rc))) )
                    {
                        objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                        objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                        objRep = libGAP_ELM_PLIST(objRep,rc);
                        rep    = libGAP_INT_INTOBJ(objRep);
                        if ( rep != 0 ) {
                            libGAP_AddCosetFactor2( rep );
                        }
                        rc = tc;
                        rp = rp - 2;
                    }
                }

                /* enter the word into the tree and return its number      */
                objNum = ( libGAP_treeType == 1 ) ?
                    libGAP_objWordValue : libGAP_INTOBJ_INT(libGAP_TreeEntryC());

                /* work off a coincidence                                  */
                if ( lp >= rp + 2 ) {
                    libGAP_HandleCoinc2( rc, lc, objNum );
                }

                /* enter a decuction to the tables                         */
                else {
                    objRep = libGAP_ELM_PLIST(libGAP_objRel,lp);
                    libGAP_SET_ELM_PLIST( objRep, lc, libGAP_INTOBJ_INT(rc) );

                    objRep = libGAP_ELM_PLIST(libGAP_objNums,lp);
                    objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                    libGAP_SET_ELM_PLIST( objRep, lc, objNum );

                    objRep = libGAP_ELM_PLIST(libGAP_objRel,rp);
                    libGAP_SET_ELM_PLIST( objRep, rc, libGAP_INTOBJ_INT(lc) );

                    tmp = ( libGAP_treeType == 1 ) ?
                        libGAP_DiffInt( libGAP_INTOBJ_INT(0), objNum ) :
                        libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ(objNum) );
                    objRep = libGAP_ELM_PLIST(libGAP_objNums,rp);
                    objRep = libGAP_ELM_PLIST(libGAP_objTable2,libGAP_INT_INTOBJ(objRep));
                    libGAP_SET_ELM_PLIST( objRep, rc, tmp );

                    if ( libGAP_dedlst == libGAP_dedSize ) {
                        libGAP_CompressDeductionList();
                    }
                    libGAP_dedgen[libGAP_dedlst] = libGAP_INT_INTOBJ( libGAP_ELM_PLIST(libGAP_objNums,lp) );
                    libGAP_dedcos[libGAP_dedlst] = lc;
                    libGAP_dedlst++;
                }

            }

        }
        libGAP_dedfst++;
    }

    libGAP_SET_ELM_PLIST( list, 6, libGAP_INTOBJ_INT(libGAP_firstFree) );
    libGAP_SET_ELM_PLIST( list, 7, libGAP_INTOBJ_INT(libGAP_lastFree)  );
    libGAP_SET_ELM_PLIST( list, 8, libGAP_INTOBJ_INT(libGAP_firstDef)  );
    libGAP_SET_ELM_PLIST( list, 9, libGAP_INTOBJ_INT(libGAP_lastDef)   );
    if ( libGAP_treeType == 1 ) {
        libGAP_SET_ELM_PLIST( list, 16, libGAP_objExponent );
    }

    /* clean out  */
    libGAP_CleanOut();

    return libGAP_INTOBJ_INT(libGAP_nrdel);
}


/****************************************************************************
**
*F  FuncStandardizeTable2C(<self>,<table>,<table2>,<stan>)  . standardize ACT
**
**  This is the kernel routine for standardizing an augmented coset table. It
**  is called by the  GAP routine  'StandardizeTable2'.  The user should  not
**  call the kernel routine but only the GAP routine.
**
**  If  <stan> = 1  the table  is standardized  using  the  (old)  semilenlex
**  standard.
**  If  not  <stan> = 1  the table  is standardized  using the  (new)  lenlex
**  standard (this is the default).
*/
libGAP_Obj libGAP_FuncStandardizeTable2C (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 list2,
    libGAP_Obj                 stan )
{
    libGAP_Obj *               ptTable;        /* pointer to table                */
    libGAP_Obj *               ptTabl2;        /* pointer to coset factor table   */
    libGAP_UInt                nrgen;          /* number of rows of the table / 2 */
    libGAP_Obj *               g;              /* one generator list from table   */
    libGAP_Obj *               h;              /* generator list                  */
    libGAP_Obj *               i;              /*  and inverse                    */
    libGAP_Obj *               h2;             /* corresponding factor lists      */
    libGAP_Obj *               i2;             /*  and inverse                    */
    libGAP_UInt                acos;           /* actual coset                    */
    libGAP_UInt                lcos;           /* last seen coset                 */
    libGAP_UInt                mcos;           /*                                 */
    libGAP_UInt                c1, c2;         /* coset temporaries               */
    libGAP_Obj                 tmp;            /* temporary for swap              */
    libGAP_UInt                j, k, nloop;    /* loop variables                  */

    /* get the arguments                                                   */
    libGAP_objTable = list;
    if ( ! libGAP_IS_PLIST(libGAP_objTable) ) {
        libGAP_ErrorQuit( "<table> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(libGAP_objTable), 0L );
        return 0;
    }
    ptTable = &(libGAP_ELM_PLIST(libGAP_objTable,1)) - 1;
    nrgen   = libGAP_LEN_PLIST(libGAP_objTable) / 2;
    for ( j = 1;  j <= nrgen*2;  j++ ) {
        if ( ! libGAP_IS_PLIST(ptTable[j]) ) {
            libGAP_ErrorQuit(
                "<table>[%d] must be a plain list (not a %s)",
                (libGAP_Int)j,
                (libGAP_Int)libGAP_TNAM_OBJ(ptTable[j]) );
            return 0;
        }
    }
    libGAP_objTable2 = list2;
    if ( ! libGAP_IS_PLIST(libGAP_objTable2) ) {
        libGAP_ErrorQuit( "<table2> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(libGAP_objTable), 0L );
        return 0;
    }
    ptTabl2 = &(libGAP_ELM_PLIST(libGAP_objTable2,1)) - 1;
    if ( libGAP_IS_INTOBJ(stan) && libGAP_INT_INTOBJ(stan) == 1 ) {
       /* use semilenlex standard                                          */
       nloop = nrgen;
    }
    else {
       /* use lenlex standard                                              */
       nloop = nrgen*2;
    }

    /* run over all cosets                                                 */
    acos = 1;
    lcos = 1;
    while ( acos <= lcos ) {

        /* scan through all columns of acos                                */
        for ( j = 1;  j <= nloop;  j++ ) {
            k = ( nloop == nrgen ) ? 2*j - 1 : j;
            g = &(libGAP_ELM_PLIST(ptTable[k],1)) - 1;

            /* if we haven't seen this coset yet                           */
            if ( lcos+1 < libGAP_INT_INTOBJ( g[acos] ) ) {

                /* swap rows lcos and g[acos]                              */
                lcos = lcos + 1;
                mcos = libGAP_INT_INTOBJ( g[acos] );
                for ( k = 1;  k <= nrgen;  k++ ) {
                    h  = &(libGAP_ELM_PLIST(ptTable[2*k-1],1)) - 1;
                    i  = &(libGAP_ELM_PLIST(ptTable[2*k],1)) - 1;
                    h2 = &(libGAP_ELM_PLIST(ptTabl2[2*k-1],1)) - 1;
                    i2 = &(libGAP_ELM_PLIST(ptTabl2[2*k],1)) - 1;
                    c1 = libGAP_INT_INTOBJ( h[lcos] );
                    c2 = libGAP_INT_INTOBJ( h[mcos] );
                    if ( c1 != 0 )  i[c1] = libGAP_INTOBJ_INT( mcos );
                    if ( c2 != 0 )  i[c2] = libGAP_INTOBJ_INT( lcos );
                    tmp     = h[lcos];
                    h[lcos] = h[mcos];
                    h[mcos] = tmp;
                    tmp      = h2[lcos];
                    h2[lcos] = h2[mcos];
                    h2[mcos] = tmp;
                    if ( i != h ) {
                        c1 = libGAP_INT_INTOBJ( i[lcos] );
                        c2 = libGAP_INT_INTOBJ( i[mcos] );
                        if ( c1 != 0 )  h[c1] = libGAP_INTOBJ_INT( mcos );
                        if ( c2 != 0 )  h[c2] = libGAP_INTOBJ_INT( lcos );
                        tmp     = i[lcos];
                        i[lcos] = i[mcos];
                        i[mcos] = tmp;
                        tmp      = i2[lcos];
                        i2[lcos] = i2[mcos];
                        i2[mcos] = tmp;
                    }
                }

            }

            /* if this is already the next only bump lcos                  */
            else if ( lcos < libGAP_INT_INTOBJ( g[acos] ) ) {
                lcos = lcos + 1;
            }

        }

        acos = acos + 1;
    }

    /* shrink the tables                                                   */
    for ( j = 1; j <= nrgen; j++ ) {
        libGAP_SET_LEN_PLIST( ptTable[2*j-1], lcos );
        libGAP_SET_LEN_PLIST( ptTable[2*j  ], lcos );
        libGAP_SET_LEN_PLIST( ptTabl2[2*j-1], lcos );
        libGAP_SET_LEN_PLIST( ptTabl2[2*j  ], lcos );
    }

    /* return void                                                         */
    return 0;
}


/****************************************************************************
**
*F  FuncAddAbelianRelator( <hdCall> ) . . . . . . internal 'AddAbelianRelator'
**
**  'FuncAddAbelianRelator' implements 'AddAbelianRelator(<rels>,<number>)'
*/
libGAP_Obj libGAP_FuncAddAbelianRelator (
    libGAP_Obj                 self,
    libGAP_Obj                 rels,           /* relators list                   */
    libGAP_Obj                 number )
{
    libGAP_Obj *               ptRels;         /* pointer to relators list        */
    libGAP_Obj *               pt1;            /* pointer to a relator            */
    libGAP_Obj *               pt2;            /* pointer to another relator      */
    libGAP_Obj                 tmp;
    libGAP_Int                 numcols;        /* list length of the rel vectors  */
    libGAP_Int                 numrows;        /* number of relators              */
    libGAP_Int                 i, j;           /* loop variables                  */

    /* check the arguments                                                 */
    if ( ! libGAP_IS_PLIST(rels) ) {
        libGAP_ErrorQuit( "<rels> must be a plain list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(rels), 0L );
        return 0;
    }
    ptRels = &(libGAP_ELM_PLIST(rels,1)) - 1;
    if ( libGAP_TNUM_OBJ(number) != libGAP_T_INT ) {
        libGAP_ErrorQuit( "<number> must be a small integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(number), 0L );
        return 0;
    }

    /* get the length of the given relators list                           */
    numrows = libGAP_INT_INTOBJ(number);
    if ( numrows < 1 || libGAP_LEN_PLIST(rels) < numrows ) {
        libGAP_ErrorQuit( "inconsistent relator number", 0L, 0L );
        return 0;
    }
    tmp = libGAP_ELM_PLIST( rels, numrows );
    if ( tmp == 0 ) {
        libGAP_ErrorQuit( "inconsistent relator number", 0L, 0L );
        return 0;
    }
    pt2 = &(libGAP_ELM_PLIST(tmp,1)) - 1;

    /* get the length of the exponent vectors (the number of generators)   */
    numcols = libGAP_LEN_PLIST(tmp);

    /* remove the last relator if it has length zero                       */
    for ( i = 1;  i <= numcols;  i++ ) {
        if ( libGAP_INT_INTOBJ(pt2[i]) ) {
            break;
        }
    }
    if ( i > numcols ) {
        return libGAP_INTOBJ_INT(numrows-1);
    }

    /* invert the relator if its first non-zero exponent is negative       */
    if ( libGAP_INT_INTOBJ(pt2[i]) < 0 ) {
        for ( j = i;  j <= numcols;  j++ ) {
            pt2[j] = libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ( pt2[j] ) );
        }
    }

    /* if the last relator occurs twice, remove one of its occurrences     */
    for ( i = 1;  i < numrows;  i++ ) {
        pt1 = &(libGAP_ELM_PLIST(ptRels[i],1)) - 1;
        for ( j = 1;  j <= numcols;  j++ ) {
            if ( pt1[j] != pt2[j] ) {
                break;
            }
        }
        if ( j > numcols ) {
            break;
        }
    }
    if ( i < numrows ) {
        for ( i = 1;  i <= numcols;  i++ ) {
            pt2[i] = libGAP_INTOBJ_INT(0);
        }
        numrows = numrows - 1;
    }

    return libGAP_INTOBJ_INT( numrows );
}

/* new type functions that use different data structures */

libGAP_UInt libGAP_ret1,libGAP_ret2;

libGAP_UInt libGAP_RelatorScan (
  libGAP_Obj t,
  libGAP_UInt di,
  libGAP_Obj r )
{
    libGAP_UInt  m,i,p,a,j;
    libGAP_UInt  pa=0,pb=0;
    libGAP_UInt * rp;
    rp=(libGAP_UInt*)libGAP_ADDR_OBJ(r);
    m=rp[1]; /* length is in position 1 */
    i=2;
    p=di;
    while ((p!=0) && (i<=(m+1))){
      a=rp[i];
      pa=p;
      p=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(t,a),p));
      if (p!=0) i++;
    }

    if (i>(m+1)) {
      if (p==di) 
        return 1;
      else
        return 0;
    }

    /*  backwards scan */
    j=m+1;
    p=di;
    while ((p!=0) && (j>=i)) {
      /* a=INT_INTOBJ(ELM_PLIST(invtab,INT_INTOBJ(ELM_PLIST(r,j))));*/

      a=rp[j];
      if ((a%2)==1)
        a++;
      else
        a--;
      pb=p;
      p=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(t,a),p));
      if (p!=0) j--;
    }

    if (j<i) {
      if (p==pa) 
        return 1;
      else
        return 0;
    }
    else {
      if (j==i) {
        a=rp[i];
        if ((a%2)==0) {
          p=a-1;
          libGAP_ret1=pb;
          libGAP_ret2=p;
        }
        else {
          p=a+1;
          libGAP_ret1=pa;
          libGAP_ret2=a;
        }
        libGAP_SET_ELM_PLIST(libGAP_ELM_PLIST(t,a),pa,libGAP_INTOBJ_INT(pb));
        libGAP_SET_ELM_PLIST(libGAP_ELM_PLIST(t,p),pb,libGAP_INTOBJ_INT(pa));

        return 2;
      }
      else
        return 1;
    }

}

/* data object type for the mangled relators */
libGAP_Obj libGAP_TYPE_LOWINDEX_DATA;

/****************************************************************************
**
*F  FuncLOWINDEX_COSET_SCAN( <t>,<r>,<s1>,<s2>)
**
*/
libGAP_Obj libGAP_FuncLOWINDEX_COSET_SCAN (
    libGAP_Obj                 self,
    libGAP_Obj                 t,              /* table */
    libGAP_Obj                 r,              /* relators */
    libGAP_Obj                 s1,             /* stack */
    libGAP_Obj                 s2 )            /* stack */
{
  libGAP_UInt ok,i,j,d,e,x,y,l,sd;
  libGAP_Obj  rx;
  libGAP_UInt * s1a;
  libGAP_UInt * s2a;

  ok=1;
  j=1;
  /* we convert stack entries to c-integers to avoid conversion */
  sd=libGAP_LEN_PLIST(s1);
  s1a=(libGAP_UInt*)libGAP_ADDR_OBJ(s1);
  s2a=(libGAP_UInt*)libGAP_ADDR_OBJ(s2);
  s1a[1]=libGAP_INT_INTOBJ(s1a[1]);
  s2a[1]=libGAP_INT_INTOBJ(s2a[1]);
  while ((ok==1) && (j>0)) {
    d=s1a[j];
    x=s2a[j];
    j--;
    rx=libGAP_ELM_PLIST(r,x);
    l=libGAP_LEN_PLIST(rx);
    i=1;
    while ((ok==1)&&(i<=l)) {
      ok=libGAP_RelatorScan(t,d,libGAP_ELM_PLIST(rx,i));
      if (ok==2) {
        j++;
        if (j>sd) {
          sd=2*sd;
          libGAP_GROW_PLIST(s1,sd);
          libGAP_SET_LEN_PLIST(s1,sd);
          libGAP_CHANGED_BAG(s1);
          libGAP_GROW_PLIST(s2,sd);
          libGAP_SET_LEN_PLIST(s2,sd);
          libGAP_CHANGED_BAG(s2);
          s1a=(libGAP_UInt*)libGAP_ADDR_OBJ(s1);
          s2a=(libGAP_UInt*)libGAP_ADDR_OBJ(s2);
        }
        s1a[j]=libGAP_ret1;
        s2a[j]=libGAP_ret2;
        ok=1;
      }
      i++;
    }

    e=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(t,x),d));
    y=x+1;
    rx=libGAP_ELM_PLIST(r,y);
    i=1;
    while ((ok==1)&&(i<=l)) {
      ok=libGAP_RelatorScan(t,e,libGAP_ELM_PLIST(rx,i));
      if (ok==2) {
        j++;
        if (j>sd) {
          sd=2*sd;
          libGAP_GROW_PLIST(s1,sd);
          libGAP_GROW_PLIST(s2,sd);
          s1a=(libGAP_UInt*)libGAP_ADDR_OBJ(s1);
          s2a=(libGAP_UInt*)libGAP_ADDR_OBJ(s2);
        }
        s1a[j]=libGAP_ret1;
        s2a[j]=libGAP_ret2;
        ok=1;
      }
      i++;
    }
  }
  /* clean up the mess we made */
  for (i=1;i<=sd;i++) {
    s1a[i]=(libGAP_Int)libGAP_INTOBJ_INT(0);
    s2a[i]=(libGAP_Int)libGAP_INTOBJ_INT(0);
  }
  if (ok==1)
    return libGAP_True;
  else
    return libGAP_False;
}

/****************************************************************************
**
*F  FuncLOWINDEX_IS_FIRST( <t>,<n>,<mu>,<nu>)
**
*/
libGAP_Obj libGAP_FuncLOWINDEX_IS_FIRST (
    libGAP_Obj                 self,
    libGAP_Obj                 t,              /* table */
    libGAP_Obj                 nobj,              /* relators */
    libGAP_Obj                 muo,             /* stack */
    libGAP_Obj                 nuo )            /* stack */
{
  libGAP_UInt l,ok,b,g,ga,de,a,n,mm;
  libGAP_UInt * mu;
  libGAP_UInt * nu;

  mm=libGAP_LEN_PLIST(t)-1;
  n=libGAP_INT_INTOBJ(nobj);
  mu=(libGAP_UInt*)libGAP_ADDR_OBJ(muo);
  nu=(libGAP_UInt*)libGAP_ADDR_OBJ(nuo);
  for (b=1;b<=n;nu[b++]=0);
  l=0;
  for (a=2;a<=n;a++) {
    for (b=1;b<=l;nu[mu[b++]]=0);
    mu[1]=a;
    nu[a]=1;
    l=1;
    ok=1;
    b=1;
    while ((ok==1) && (b<=n)) {
      g=1;
      while ((ok==1)&&(g<=mm)) {
        ga=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(t,g),b));
        de=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_ELM_PLIST(t,g),mu[b]));
        if ((ga==0)||(de==0)) 
          ok=0;
        else {
          if (nu[de]==0) {
            l++;
            mu[l]=de;
            nu[de]=l;
          }
          if (nu[de]<ga) 
            return libGAP_False;
          else {
            if (nu[de]>ga) {
              ok=0;
            }
          }
        }
        g=g+2;
      }
      b=b+1;
    }
  }
  return libGAP_True;
}

/****************************************************************************
**
*F  FuncLOWINDEX_PREPARE_RELS( <rels> )
**
*/
libGAP_Obj libGAP_FuncLOWINDEX_PREPARE_RELS (
    libGAP_Obj                 self,
    libGAP_Obj                 r )             /* rels */
{
   libGAP_UInt i,j,k,l;
   libGAP_Obj ri, rel;
   libGAP_UInt * rp;

   for (i=1;i<=libGAP_LEN_PLIST(r);i++) {
    ri=libGAP_ELM_PLIST(r,i);
    for (j=1;j<=libGAP_LEN_PLIST(ri);j++) {
      rel=libGAP_ELM_PLIST(ri,j); /* single relator */
      l=libGAP_LEN_PLIST(rel);
      rp=(libGAP_UInt*)libGAP_ADDR_OBJ(rel);
      for (k=1;k<=l;k++) 
        rp[k]=libGAP_INT_INTOBJ(rp[k]); /* convert relator entries to C-integers */
      /* change type */
      libGAP_TYPE_DATOBJ(rel) = libGAP_TYPE_LOWINDEX_DATA;
      libGAP_RetypeBag(rel,libGAP_T_DATOBJ);

    }
   }
   return (libGAP_Obj) 0;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "ApplyRel", 2, "app, relator",
      libGAP_FuncApplyRel, "src/costab.c:ApplyRel" },

    { "MakeConsequences", 1, "list",
      libGAP_FuncMakeConsequences, "src/costab.c:MakeConsequences" },

    { "MakeConsequencesPres", 1, "list",
      libGAP_FuncMakeConsequencesPres, "src/costab.c:MakeConsequencesPres" },

    { "StandardizeTableC", 2, "table, standard",
      libGAP_FuncStandardizeTableC, "src/costab.c:StandardizeTableC" },

    { "ApplyRel2", 3, "app, relators, nums",
      libGAP_FuncApplyRel2, "src/costab.c:ApplyRel2" },

    { "CopyRel", 1, "relator",
      libGAP_FuncCopyRel, "src/costab.c:CopyRel" },

    { "MakeCanonical", 1, "relator",
      libGAP_FuncMakeCanonical, "src/costab.c:MakeCanonical" },

    { "TreeEntry", 2, "relator, word",
      libGAP_FuncTreeEntry, "src/costab.c:TreeEntry" },

    { "MakeConsequences2", 1, "list",
      libGAP_FuncMakeConsequences2, "src/costab.c:MakeConsequences2" },

    { "StandardizeTable2C", 3, "table, table, standard",
      libGAP_FuncStandardizeTable2C, "src/costab.c:StandardizeTable2C" },

    { "AddAbelianRelator", 2, "rels, number",
      libGAP_FuncAddAbelianRelator, "src/costab.c:AddAbelianRelator" },

    { "LOWINDEX_COSET_SCAN", 4, "table, relators, stack1,stack2",
      libGAP_FuncLOWINDEX_COSET_SCAN, "src/costab.c:LOWINDEX_COSET_SCAN" },

    { "LOWINDEX_IS_FIRST", 4, "table, n, mu, nu",
      libGAP_FuncLOWINDEX_IS_FIRST, "src/costab.c:LOWINDEX_IS_FIRST" },

    { "LOWINDEX_PREPARE_RELS", 1, "rels",
      libGAP_FuncLOWINDEX_PREPARE_RELS, "src/costab.c:LOWINDEX_PREPARE_RELS" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* import kind (and unkind) functions */
    libGAP_ImportGVarFromLibrary( "TYPE_LOWINDEX_DATA",&libGAP_TYPE_LOWINDEX_DATA     );

    /* static variables                                                    */
    libGAP_InitGlobalBag( &libGAP_objRel      , "src/costab.c:objRel"       );
    libGAP_InitGlobalBag( &libGAP_objNums     , "src/costab.c:objNums"      );
    libGAP_InitGlobalBag( &libGAP_objFactor   , "src/costab.c:objFactor"    );
    libGAP_InitGlobalBag( &libGAP_objTable    , "src/costab.c:objTable"     );
    libGAP_InitGlobalBag( &libGAP_objTable2   , "src/costab.c:objTable2"    );
    libGAP_InitGlobalBag( &libGAP_objNext     , "src/costab.c:objNext"      );
    libGAP_InitGlobalBag( &libGAP_objPrev     , "src/costab.c:objPrev"      );
    libGAP_InitGlobalBag( &libGAP_objTree     , "src/costab.c:objTree"      );
    libGAP_InitGlobalBag( &libGAP_objTree1    , "src/costab.c:objTree1"     );
    libGAP_InitGlobalBag( &libGAP_objTree2    , "src/costab.c:objTree2"     );
    libGAP_InitGlobalBag( &libGAP_objWordValue, "src/costab.c:objWordValue" );
    libGAP_InitGlobalBag( &libGAP_objExponent , "src/costab.c:objExponent"  );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoCosetTable()  . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "costab",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoCosetTable ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  costab.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
