/****************************************************************************
**
*W  bool.c                      GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions for the boolean package.
**
**  Note that boolean objects actually contain no data. The three of them
**  are distinguished by their addresses, kept in the C globals False, 
**  True and Fail.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "string.h"              /* strings                         */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */


/****************************************************************************
**

*V  True  . . . . . . . . . . . . . . . . . . . . . . . . . . . .  true value
**
**   'True' is the value 'true'.
*/
libGAP_Obj libGAP_True;


/****************************************************************************
**
*V  False . . . . . . . . . . . . . . . . . . . . . . . . . . . . false value
**
**  'False' is the value 'false'.
*/
libGAP_Obj libGAP_False;


/****************************************************************************
**
*V  Fail  . . . . . . . . . . . . . . . . . . . . . . . . . . . .  fail value
**
**  'Fail' is the value 'fail'.
*/
libGAP_Obj libGAP_Fail;

/****************************************************************************
**
*V  SuPeRfail  . . . . . . . . . . . . . . . . . . . . . . .  superfail value
**
**  'SuPeRfail' is an ``superfail'' object which is used to indicate failure if
**  `fail' itself is a sensible response. This is used when having GAP read
**  a file line-by-line via a library function (demo.g)
*/
libGAP_Obj libGAP_SuPeRfail;


/****************************************************************************
**

*F  TypeBool( <bool> )  . . . . . . . . . . . . . . . type of a boolean value
**
**  'TypeBool' returns the type of boolean values.
**
**  'TypeBool' is the function in 'TypeObjFuncs' for boolean values.
*/
libGAP_Obj libGAP_TYPE_BOOL;

libGAP_Obj libGAP_TypeBool (
    libGAP_Obj                 val )
{
    return libGAP_TYPE_BOOL;
}


/****************************************************************************
**
*F  PrintBool( <bool> ) . . . . . . . . . . . . . . . . print a boolean value
**
**  'PrintBool' prints the boolean value <bool>.
*/
void libGAP_PrintBool (
    libGAP_Obj                 bool )
{
    if ( bool == libGAP_True ) {
        libGAP_Pr( "true", 0L, 0L );
    }
    else if ( bool == libGAP_False ) {
        libGAP_Pr( "false", 0L, 0L );
    }
    else if ( bool == libGAP_Fail ) {
        libGAP_Pr( "fail", 0L, 0L );
    }
    else if ( bool == libGAP_SuPeRfail ) {
        libGAP_Pr( "SuPeRfail", 0L, 0L );
    }
    else {
        libGAP_Pr( "<<very strange boolean value>>", 0L, 0L );
    }
}


/****************************************************************************
**
*F  EqBool( <boolL>, <boolR> )  . . . . . . . . .  test if <boolL> =  <boolR>
**
**  'EqBool' returns 'True' if the two boolean values <boolL> and <boolR> are
**  equal, and 'False' otherwise.
*/
libGAP_Int libGAP_EqBool (
    libGAP_Obj                 boolL,
    libGAP_Obj                 boolR )
{
    if ( boolL == boolR ) {
        return 1L;
    }
    else {
        return 0L;
    }
}


/****************************************************************************
**
*F  LtBool( <boolL>, <boolR> )  . . . . . . . . .  test if <boolL> <  <boolR>
**
**  The ordering of Booleans is true < false <= fail (the <= comes from
**  the fact that Fail may be equal to False in some compatibility modes
*/
libGAP_Int libGAP_LtBool (
    libGAP_Obj                 boolL,
    libGAP_Obj                 boolR )
{
  return  ( boolL == libGAP_True && boolR != libGAP_True) ||
    ( boolL == libGAP_False && boolR == libGAP_Fail && boolL != boolR);
}


/****************************************************************************
**
*F  IsBoolFilt( <self>, <obj> ) . . . . . . . . . .  test for a boolean value
**
**  'IsBoolFilt' implements the internal filter 'IsBool'.
**
**  'IsBool( <obj> )'
**
**  'IsBool'  returns  'true'  if  <obj>  is   a boolean  value  and  'false'
**  otherwise.
*/
libGAP_Obj libGAP_IsBoolFilt;

libGAP_Obj libGAP_IsBoolHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    /* return 'true' if <obj> is a boolean and 'false' otherwise           */
    if ( libGAP_TNUM_OBJ(obj) == libGAP_T_BOOL ) {
        return libGAP_True;
    }
    else if ( libGAP_TNUM_OBJ(obj) < libGAP_FIRST_EXTERNAL_TNUM ) {
        return libGAP_False;
    }
    else {
        return libGAP_DoFilter( self, obj );
    }
}


/****************************************************************************
**

*F  ReturnTrue1( <val1> ) . . . . . . . . . . . . . . . . . .  return  'True'
**
**  'ReturnTrue?'  simply return  'True'  independent of  the values of   the
**  arguments.
**
**  Those  functions are  useful for  dispatcher  tables if the types already
**  determine the outcome.
*/
libGAP_Obj libGAP_ReturnTrue1 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1 )
{
    return libGAP_True;
}


/****************************************************************************
**
*F  ReturnTrue2( <val1>, <val2> ) . . . . . . . . . . . . . .  return  'True'
*/
libGAP_Obj libGAP_ReturnTrue2 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1,
    libGAP_Obj                 val2 )
{
    return libGAP_True;
}


/****************************************************************************
**
*F  ReturnTrue3( <val1>, <val2>, <val3> ) . . . . . . . . . .  return  'True'
*/
libGAP_Obj libGAP_ReturnTrue3 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1,
    libGAP_Obj                 val2,
    libGAP_Obj                 val3 )
{
    return libGAP_True;
}


/****************************************************************************
**
*F  ReturnFalse1( <val1> )  . . . . . . . . . . . . . . . . .  return 'False'
**
**  'ReturnFalse?' likewise return 'False'.
*/
libGAP_Obj libGAP_ReturnFalse1 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1 )
{
    return libGAP_False;
}


/****************************************************************************
**
*F  ReturnFalse2( <val1>, <val2> )  . . . . . . . . . . . . .  return 'False'
*/
libGAP_Obj libGAP_ReturnFalse2 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1,
    libGAP_Obj                 val2 )
{
    return libGAP_False;
}


/****************************************************************************
**
*F  ReturnFalse3( <val1>, <val2>, <val3> )  . . . . . . . . .  return 'False'
*/
libGAP_Obj libGAP_ReturnFalse3 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1,
    libGAP_Obj                 val2,
    libGAP_Obj                 val3 )
{
    return libGAP_False;
}


/****************************************************************************
**
*F  ReturnFail1( <val1> ) . . . . . . . . . . . . . . . . . .  return  'Fail'
**
**  'ReturnFail?' likewise return 'Fail'.
*/
libGAP_Obj libGAP_ReturnFail1 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1 )
{
    return libGAP_Fail;
}


/****************************************************************************
**
*F  ReturnFail2( <val1>, <val2> ) . . . . . . . . . . . . . .  return  'Fail'
*/
libGAP_Obj libGAP_ReturnFail2 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1,
    libGAP_Obj                 val2 )
{
    return libGAP_Fail;
}


/****************************************************************************
**
*F  ReturnFail3( <val1>, <val2>, <val3> ) . . . . . . . . . .  return  'Fail'
*/
libGAP_Obj libGAP_ReturnFail3 (
    libGAP_Obj                 self,
    libGAP_Obj                 val1,
    libGAP_Obj                 val2,
    libGAP_Obj                 val3 )
{
    return libGAP_Fail;
}


/****************************************************************************
**

*F  SaveBool( <bool> ) . . . . . . . . . . . . . . . . . . . . save a Boolean 
**
**  Actually, there is nothing to do
*/

void libGAP_SaveBool( libGAP_Obj obj )
{
  return;
}

/****************************************************************************
**
*F  LoadBool( <bool> ) . . . . . . . . . . . . . . . . . . . . save a Boolean 
**
**  Actually, there is nothing to do
*/

void libGAP_LoadBool( libGAP_Obj obj )
{
  return;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_BOOL", "obj", &libGAP_IsBoolFilt,
      libGAP_IsBoolHandler, "src/bool.c:IS_BOOL" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* install the marking functions for boolean values                    */
    libGAP_InfoBags[ libGAP_T_BOOL ].name = "boolean";
    libGAP_InitMarkFuncBags( libGAP_T_BOOL, libGAP_MarkNoSubBags );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );

    /* make and install the 'RETURN_TRUE' function                         */
    libGAP_InitHandlerFunc( libGAP_ReturnTrue1, "src/bool.c:ReturnTrue1" );
    libGAP_InitHandlerFunc( libGAP_ReturnTrue2, "src/bool.c:ReturnTrue2" );
    libGAP_InitHandlerFunc( libGAP_ReturnTrue3, "src/bool.c:ReturnTrue3" );

    /* make and install the 'RETURN_FALSE' function                        */
    libGAP_InitHandlerFunc( libGAP_ReturnFalse1, "src/bool.c:ReturnFalse1" );
    libGAP_InitHandlerFunc( libGAP_ReturnFalse2, "src/bool.c:ReturnFalse2" );
    libGAP_InitHandlerFunc( libGAP_ReturnFalse3, "src/bool.c:ReturnFalse3" );

    /* make and install the 'RETURN_FAIL' function                        */
    libGAP_InitHandlerFunc( libGAP_ReturnFail1, "src/bool.c:ReturnFail1" );
    libGAP_InitHandlerFunc( libGAP_ReturnFail2, "src/bool.c:ReturnFail2" );
    libGAP_InitHandlerFunc( libGAP_ReturnFail3, "src/bool.c:ReturnFail3" );

    /* install the type function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_BOOL", &libGAP_TYPE_BOOL );
    libGAP_TypeObjFuncs[ libGAP_T_BOOL ] = libGAP_TypeBool;

    /* make the boolean bags                                         */
    libGAP_InitGlobalBag( &libGAP_True,  "src/bool.c:TRUE"  );
    libGAP_InitGlobalBag( &libGAP_False, "src/bool.c:FALSE" );
    libGAP_InitGlobalBag( &libGAP_Fail,  "src/bool.c:FAIL"  );
    libGAP_InitGlobalBag( &libGAP_SuPeRfail,  "src/bool.c:SUPERFAIL"  );

    /* install the saving functions                                       */
    libGAP_SaveObjFuncs[ libGAP_T_BOOL ] = libGAP_SaveBool;

    /* install the loading functions                                       */
    libGAP_LoadObjFuncs[ libGAP_T_BOOL ] = libGAP_LoadBool;

    /* install the printer for boolean values                              */
    libGAP_PrintObjFuncs[ libGAP_T_BOOL ] = libGAP_PrintBool;

    /* install the comparison functions                                    */
    libGAP_EqFuncs[ libGAP_T_BOOL ][ libGAP_T_BOOL ] = libGAP_EqBool;
    libGAP_LtFuncs[ libGAP_T_BOOL ][ libGAP_T_BOOL ] = libGAP_LtBool;

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt            gvar;
    libGAP_Obj             tmp;

    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );

    /* bags are registered in 'InitKernel'                                 */
    libGAP_True  = libGAP_NewBag( libGAP_T_BOOL, 0L );
    libGAP_False = libGAP_NewBag( libGAP_T_BOOL, 0L );
    libGAP_Fail  = libGAP_NewBag( libGAP_T_BOOL, 0L );

    /* `fail' is a variable not a language construct                       */
    gvar = libGAP_GVarName( "fail" );
    libGAP_AssGVar( gvar, libGAP_Fail );
    libGAP_MakeReadOnlyGVar(gvar);

    /* `SuPeRfail' ditto                       */
    libGAP_SuPeRfail  = libGAP_NewBag( libGAP_T_BOOL, 0L );
    gvar = libGAP_GVarName( "SuPeRfail" );
    libGAP_AssGVar( gvar, libGAP_SuPeRfail );
    libGAP_MakeReadOnlyGVar(gvar);

    /* make and install the 'RETURN_TRUE' function                         */
    tmp = libGAP_NewFunctionC( "RETURN_TRUE", -1L, "arg", libGAP_ReturnTrue1 );
    libGAP_HDLR_FUNC( tmp, 1 ) = libGAP_ReturnTrue1;
    libGAP_HDLR_FUNC( tmp, 2 ) = libGAP_ReturnTrue2;
    libGAP_HDLR_FUNC( tmp, 3 ) = libGAP_ReturnTrue3;
    libGAP_AssGVar( libGAP_GVarName("RETURN_TRUE"), tmp );

    /* make and install the 'RETURN_FALSE' function                        */
    tmp = libGAP_NewFunctionC("RETURN_FALSE",-1L,"arg",libGAP_ReturnFalse1);
    libGAP_HDLR_FUNC( tmp, 1 ) = libGAP_ReturnFalse1;
    libGAP_HDLR_FUNC( tmp, 2 ) = libGAP_ReturnFalse2;
    libGAP_HDLR_FUNC( tmp, 3 ) = libGAP_ReturnFalse3;
    libGAP_AssGVar( libGAP_GVarName( "RETURN_FALSE" ), tmp );

    /* make and install the 'RETURN_FAIL' function                        */
    tmp = libGAP_NewFunctionC("RETURN_FAIL", -1L, "arg", libGAP_ReturnFail1);
    libGAP_HDLR_FUNC( tmp, 1 ) = libGAP_ReturnFail1;
    libGAP_HDLR_FUNC( tmp, 2 ) = libGAP_ReturnFail2;
    libGAP_HDLR_FUNC( tmp, 3 ) = libGAP_ReturnFail3;
    libGAP_AssGVar( libGAP_GVarName( "RETURN_FAIL" ), tmp );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoBool()  . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "bool",                             /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoBool ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  bool.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
