/****************************************************************************
**
*W  weakptr.c                   GAP source                       Steve Linton
**
**
*Y  Copyright (C)  1997,  School of Mathematical and Computational Sciences,
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
*Y                        University of St Andrews, Scotland
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions that deal with weak pointer objects
**  A weak pointer object looks like a plain list, except that its entries
**  are NOT kept alive through a garbage collection (unless they are contained
**  in some other kind of object). 
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "bool.h"                /* booleans                        */

#include        "weakptr.h"             /* weak pointers                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */

#include        "calls.h"               /* generic call mechanism          */
#include        "saveload.h"            /* saving and loading              */
#include        "opers.h"               /* generic operations              */

#include	"scanner.h"		/* scanner                         */
#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */


/****************************************************************************
**
*F  GROW_WPOBJ(<wp>,<plen>) . make sure a weak pointer object is large enough
**
**  'GROW_WPOBJ' grows the weak pointer   object <wp> if necessary  to
**  ensure that it has room for at least <plen> elements.
**
**  Note that 'GROW_WPOBJ' is a macro, so do not call it with arguments that
**  have side effects.  */

#define libGAP_GROW_WPOBJ(wp,plen)   ((plen) < libGAP_SIZE_OBJ(wp)/sizeof(libGAP_Obj) ? \
                                 0L : libGAP_GrowWPObj(wp,plen) )

libGAP_Int libGAP_GrowWPObj (
               libGAP_Obj                 wp,
               libGAP_UInt                need )
{
  libGAP_UInt                plen;           /* new physical length             */
  libGAP_UInt                good;           /* good new physical length        */

    /* find out how large the object should become                     */
    good = 5 * (libGAP_SIZE_OBJ(wp)/sizeof(libGAP_Obj)-1) / 4 + 4;

    /* but maybe we need more                                              */
    if ( need < good ) { plen = good; }
    else               { plen = need; }

    /* resize the plain list                                               */
    libGAP_ResizeBag( wp, ((plen)+1)*sizeof(libGAP_Obj) );

    /* return something (to please some C compilers)                       */
    return 0L;
}


/****************************************************************************
**
*F  STORE_LEN_WPOBJ(<wp>,<len>) . . . . . . .  set the length of a WP object
**
**  'STORE_LEN_WPOBJ' sets the length of  the WP object  <wp> to <len>.
**
**  Note  that 'STORE_LEN_WPOBJ'  is a macro, so do not call it with  arguments
**  that have side effects.
** 
**  Objects at the end of wp may evaporate, so the stored length can only
**  be regarded as an upper bound.
*/

#define libGAP_STORE_LEN_WPOBJ(wp,len)         (libGAP_ADDR_OBJ(wp)[0] = (libGAP_Obj)(len))


/****************************************************************************
**
*F  STORED_ LEN_WPOBJ(<wp>). . .. . . . . . .  stored length of a WP Object
**
**  'STORED_LEN_WPOBJ' returns the stored length of the WP object <wp> 
**  as a C integer.
**
**  Note that 'STORED_LEN_WPOBJ' is a  macro, so do  not call it 
**  with arguments that have side effects.
**
**  Note that as the list can mutate under your feet, the length may be
**  an overestimate
*/

#define libGAP_STORED_LEN_WPOBJ(wp)                 ((libGAP_Int)(libGAP_ADDR_OBJ(wp)[0]))

/****************************************************************************
**
*F  ELM_WPOBJ(<wp>,<pos>) . . . . . . . . . . . . . element of a WP object
**
**  'ELM_WPOBJ' return the <wp>-th element of the WP object <wp>.  <pos> must
**  be a positive integer less than or equal  to the physical length of <wp>.
**  If <wp> has no assigned element at position <pos>, 'ELM_WPOBJ' returns 0.
**
**  If the entry died at a recent garbage collection, it will return a Bag ID
**  for which IS_WEAK_DEAD_BAG will return 1
**
**  Note that  'ELM_WPOBJ' is a macro, so do  not call it with arguments that
**  have side effects.  
**
**  ELM_WPOBJ(<wp>,<pos>) is a valid lvalue and may be assigned to
*/

#define libGAP_ELM_WPOBJ(list,pos)             (libGAP_ADDR_OBJ(list)[pos])



/****************************************************************************
**
*F  FuncWeakPointerObj( <self>, <list> ) . . . . . .make a weak pointer object
**
** Handler  for the GAP function  WeakPointerObject(<list>), which makes a new
** WP object 
*/

libGAP_Obj libGAP_FuncWeakPointerObj( libGAP_Obj self, libGAP_Obj list ) { 
  libGAP_Obj wp; 
  libGAP_Int i;
  libGAP_Int len; 
  len = libGAP_LEN_LIST(list);
  wp = (libGAP_Obj) libGAP_NewBag(libGAP_T_WPOBJ, (len+1)*sizeof(libGAP_Obj));
  libGAP_STORE_LEN_WPOBJ(wp,len); 
  for (i = 1; i <= len ; i++) 
    { 
      libGAP_ELM_WPOBJ(wp,i) = libGAP_ELM0_LIST(list,i); 
      libGAP_CHANGED_BAG(wp);          /* this must be here in case list is 
                                 in fact an object and causes a GC in the 
                                 element access method */
    }

  return wp; 
} 


/****************************************************************************
**
*F  LengthWPObj(<wp>) . . . . . . . . . . . . . . current length of WP Object
**
**  'LengthWPObj(<wp>)' returns  the   current length  of WP  Object  as  a C
**  integer  the   value cannot be   trusted past  a   garbage collection, as
**  trailing items may evaporate.
**   
**  Any identifiers of trailing objects that have evaporated in a garbage
**  collection are cleaned up by this function
*/

libGAP_Int libGAP_LengthWPObj(libGAP_Obj wp)
{
  libGAP_Int len;
  libGAP_Obj elm;
  libGAP_Int changed = 0;
  for (len = libGAP_STORED_LEN_WPOBJ(wp); 
       len > 0 && 
         (!(elm = libGAP_ELM_WPOBJ(wp,len)) ||
          libGAP_IS_WEAK_DEAD_BAG(elm)); 
       len --)
    {
      changed = 1;
      if (elm)
        libGAP_ELM_WPOBJ(wp,len) = 0;
    }
  if (changed)
    libGAP_STORE_LEN_WPOBJ(wp,len);
  return len;
}

/****************************************************************************
**
*F  FuncLengthWPObj(<wp>) . . . . . . . . . . . . current length of WP Object
**
**  'FuncLengthWPObj(<wp>)' is a handler for a  GAP function that returns the
**  current length of WP  Object. The value  cannot be trusted past a garbage
**  collection, as trailing items may evaporate.
** 
*/

libGAP_Obj libGAP_FuncLengthWPObj(libGAP_Obj self, libGAP_Obj wp)
{
  return libGAP_INTOBJ_INT(libGAP_LengthWPObj(wp));
}


/****************************************************************************
**
*F  FuncSetElmWPObj(<self>, <wp>, <pos>, <obj> ) . set an entry in a WP Object
**
**  'FuncSetElmWPObj(<self>, <wp>,  <pos>, <obj>  )'  is a  handler for a GAP
**  function that sets an entry in a WP object.
** 
*/

libGAP_Obj libGAP_FuncSetElmWPObj(libGAP_Obj self, libGAP_Obj wp, libGAP_Obj pos, libGAP_Obj val)
{
  libGAP_UInt ipos = libGAP_INT_INTOBJ(pos);
  if (libGAP_LengthWPObj(wp)  < ipos)
    {
      libGAP_GROW_WPOBJ(wp, ipos);
      libGAP_STORE_LEN_WPOBJ(wp,ipos);
    }
  libGAP_ELM_WPOBJ(wp,ipos) = val;
  libGAP_CHANGED_BAG(wp);
  return 0;
}

/****************************************************************************
**
*F  IsBoundElmWPObj( <wp>, <pos> ) .  . . . . is an entry bound in a WP Object
**
**  'IsBoundElmWPObj( <wp>, <pos> )' returns 1 is there is (currently) a live
**  value at position pos or the WP object wp and  0 otherwise, cleaning up a
**  dead entry if there is one
** */


libGAP_Int libGAP_IsBoundElmWPObj( libGAP_Obj wp, libGAP_Obj pos)
{
  libGAP_UInt ipos = libGAP_INT_INTOBJ(pos);
  libGAP_Obj elm;
  if ( libGAP_LengthWPObj(wp) < ipos ) 
    {
      return 0;
    }
  elm = libGAP_ELM_WPOBJ(wp,ipos);
  if (libGAP_IS_WEAK_DEAD_BAG(elm))
    {
      libGAP_ELM_WPOBJ(wp,ipos) = 0;
      return 0;
    }
  if (elm == 0)
    {
      return 0;
    }
  return 1;
}

/****************************************************************************
**
*F  FuncIsBoundElmWPObj( <self>, <wp>, <pos> ) . . . . . . .IsBound WP Object
**
**  GAP  handler for IsBound  test on WP Object.   Remember that bindings can
**  evaporate in any garbage collection.
*/


libGAP_Obj libGAP_FuncIsBoundElmWPObj( libGAP_Obj self, libGAP_Obj wp, libGAP_Obj pos)
{
  return libGAP_IsBoundElmWPObj(wp, pos) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncUnbindElmWPObj( <self>, <wp>, <pos> ) . . . . . . . .Unbind WP Object
**
**  GAP  handler for Unbind on WP Object. 
*/

libGAP_Obj libGAP_FuncUnbindElmWPObj( libGAP_Obj self, libGAP_Obj wp, libGAP_Obj pos)
{
  libGAP_Int len = libGAP_LengthWPObj(wp);
  if ( libGAP_INT_INTOBJ(pos) <= len ) {
    libGAP_ELM_WPOBJ( wp, libGAP_INT_INTOBJ(pos)) =  0;
  }
  return 0;
}

/****************************************************************************
**
*F  FuncElmWPObj( <self>, <wp>, <pos> ) . . . . . . . . . . .Access WP Object
**
**  GAP handler for access to WP Object. If the entry is not bound, then fail
**  is  returned. It would not be  correct to return  an error, because there
**  would be no  way  to  safely access  an  element, which  might  evaporate
**  between a  call   to Isbound and the    access. This, of  course,  causes
**  possible  confusion  with a WP  object which  does have  a  value of fail
**  stored in  it. This, however  can be  checked  with a subsequent  call to
**  IsBound, relying on the fact  that fail can never  dissapear in a garbage
**  collection.
*/

libGAP_Obj libGAP_FuncElmWPObj( libGAP_Obj self, libGAP_Obj wp, libGAP_Obj pos)
{
  libGAP_Obj elm;
  libGAP_UInt ipos = libGAP_INT_INTOBJ(pos);
  if ( libGAP_STORED_LEN_WPOBJ(wp) < ipos ) 
    {
      return libGAP_Fail;
    }
  elm = libGAP_ELM_WPOBJ(wp,ipos);
  if (libGAP_IS_WEAK_DEAD_BAG(elm))
    {
      libGAP_ELM_WPOBJ(wp,ipos) = 0;
      return libGAP_Fail;
    }
  if (elm == 0)
    {
      return libGAP_Fail;
    }
  return elm;
}


/****************************************************************************
**
*F  TypeWPObj( <wp> ) . . . . . . . . . . . . . . . . . . . Type of WP Object
**
**  This is imported from the library variable  TYPE_WPOBJ. They all have the
**  same type
*/

libGAP_Obj libGAP_TYPE_WPOBJ;              

libGAP_Obj libGAP_TypeWPObj( libGAP_Obj wp )
{
  return libGAP_TYPE_WPOBJ;
}


/****************************************************************************
**
*F  FuncIsWPObj( <self>, <wp>) . . . . . . . Handler for GAP function IsWPObj
*/
static libGAP_Obj libGAP_IsWPObjFilt;

libGAP_Obj libGAP_FuncIsWPObj( libGAP_Obj self, libGAP_Obj wp)
{
  return (libGAP_TNUM_OBJ(wp) == libGAP_T_WPOBJ) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  MarkWeakPointerObj( <wp> ) . . . . . . . . . . . . . . . Marking function
*F  SweepWeakPointerObj( <src>, <dst>, <len> ) . . . . . . .Sweeping function
**
**  These functions are installed for GASMAN to use in garbage collection The
**  sweeping function must  clean up any  dead  weak pointers encountered  so
**  that, after a  full  GC, the  masterpointers  occupied by the  dead  weak
**  pointers can be reclaimed.  
*/

void libGAP_MarkWeakPointerObj( libGAP_Obj wp) 
{
  libGAP_Int i;
  /* can't use the stored length here, in case we
     are in the middle of copying */
  for (i = 1; i <= (libGAP_SIZE_BAG(wp)/sizeof(libGAP_Obj))-1; i++)
    libGAP_MarkBagWeakly(libGAP_ELM_WPOBJ(wp,i));
}

void libGAP_SweepWeakPointerObj( libGAP_Bag *src, libGAP_Bag *dst, libGAP_UInt len)
{
  libGAP_Bag elm;
  while (len --)
    {
      elm = *src++;
      *dst ++ = libGAP_IS_WEAK_DEAD_BAG(elm) ? (libGAP_Bag) 0 : elm;
    }
}


/****************************************************************************
**
*F  CopyObjWPObj( <obj>, <mut> ) . . . . . . . . .  copy a positional object
**
**  Note  that an  immutable   copy of  a  weak  pointer  object is a  normal
**  immutable plist. An Immutable WP object is a contradiction.
**
*N  I am far from clear that this is safe from a badly timed GC during copying.
**
*/

libGAP_Obj libGAP_CopyObjWPObj (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* copy, result                    */
    libGAP_Obj                 tmp;            /* temporary variable              */
    libGAP_Obj                 elm;
    libGAP_UInt                i;              /* loop variable                   */

    /* make a copy                                                         */
    if ( mut ) {
        copy = libGAP_NewBag( libGAP_T_WPOBJ, libGAP_SIZE_OBJ(obj) );
        libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(obj)[0];
    }
    else {
        copy = libGAP_NewBag( libGAP_T_PLIST+libGAP_IMMUTABLE, libGAP_SIZE_OBJ(obj) );
        libGAP_SET_LEN_PLIST(copy,libGAP_LengthWPObj(obj));
    }

    /* leave a forwarding pointer                                          */
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( tmp, 2 );
    libGAP_SET_ELM_PLIST( tmp, 1, libGAP_ADDR_OBJ(obj)[0] );
    libGAP_SET_ELM_PLIST( tmp, 2, copy );
    libGAP_ADDR_OBJ(obj)[0] = tmp;
    libGAP_CHANGED_BAG(obj);

    /* now it is copied                                                    */
    libGAP_RetypeBag( obj, libGAP_T_WPOBJ + libGAP_COPYING );

    /* copy the subvalues                                                  */
    for ( i =  libGAP_SIZE_OBJ(obj)/sizeof(libGAP_Obj)-1; i > 0; i-- ) {
        elm = libGAP_ADDR_OBJ(obj)[i];
        if ( elm != 0  && !libGAP_IS_WEAK_DEAD_BAG(elm)) {
            tmp = libGAP_COPY_OBJ( elm, mut );
            libGAP_ADDR_OBJ(copy)[i] = tmp;
            libGAP_CHANGED_BAG( copy );
        }
    }

    /* return the copy                                                     */
    return copy;
}

/****************************************************************************
**
*F  MakeImmutableWPObj( <obj> ) . . . . . . . . . . make immutable in place
**
*/

void libGAP_MakeImmutableWPObj( libGAP_Obj obj )
{
  libGAP_UInt i;
  libGAP_Obj elm;
  
  /* remove any weak dead bags */
  for (i = 1; i <= libGAP_STORED_LEN_WPOBJ(obj); i++)
    {
      elm = libGAP_ELM_WPOBJ(obj,i);
      if (elm != 0 && libGAP_IS_WEAK_DEAD_BAG(elm)) 
        libGAP_ELM_WPOBJ(obj,i) = 0;
    }
  /* Change the type */
  libGAP_RetypeBag( obj, libGAP_T_PLIST+libGAP_IMMUTABLE);
}

/****************************************************************************
**
*F  CleanObjWPObj( <obj> ) . . . . . . . . . . . . . . . . . . .  clean WPobj
*/
void libGAP_CleanObjWPObj (
    libGAP_Obj                 obj )
{
}


/****************************************************************************
**
*F  CopyObjWPObjCopy( <obj>, <mut> ) . . . . . . . . . .  . copy a WPobj copy
*/
libGAP_Obj libGAP_CopyObjWPObjCopy (
    libGAP_Obj                 obj,
    libGAP_Int                 mut )
{
    return libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 2 );
}


/****************************************************************************
**
*F  CleanObjWPObjCopy( <obj> ) . . . . . . . . . . . . . . clean WPobj copy
*/
void libGAP_CleanObjWPObjCopy (
    libGAP_Obj                 obj )
{
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 elm;            /* subobject                       */

    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(obj)[0] = libGAP_ELM_PLIST( libGAP_ADDR_OBJ(obj)[0], 1 );
    libGAP_CHANGED_BAG(obj);

    /* now it is cleaned                                                   */
    libGAP_RetypeBag( obj, libGAP_TNUM_OBJ(obj) - libGAP_COPYING );

    /* clean the subvalues                                                 */
    for ( i = 1; i < libGAP_SIZE_OBJ(obj)/sizeof(libGAP_Obj); i++ ) {
        elm = libGAP_ADDR_OBJ(obj)[i];
        if ( elm != 0  && !libGAP_IS_WEAK_DEAD_BAG(elm)) 
          libGAP_CLEAN_OBJ( elm );
    }

}

/****************************************************************************
**
*F  SaveWPObj( <wpobj> )
*/

void libGAP_SaveWPObj( libGAP_Obj wpobj )
{
  libGAP_UInt len, i;
  libGAP_Obj *ptr;
  libGAP_Obj x;
  ptr = libGAP_ADDR_OBJ(wpobj)+1;
  len = libGAP_STORED_LEN_WPOBJ(wpobj);
  libGAP_SaveUInt(len);
  for (i = 1; i <= len; i++)
    {
      x = *ptr;
      if (libGAP_IS_WEAK_DEAD_BAG(x))
        {
          libGAP_SaveSubObj(0);
          *ptr = 0;
        }
      else
        libGAP_SaveSubObj(x);
      ptr++;
    }
}

/****************************************************************************
**
*F  LoadWPObj( <wpobj> )
*/

void libGAP_LoadWPObj( libGAP_Obj wpobj )
{
  libGAP_UInt len, i;
  libGAP_Obj *ptr;
  ptr = libGAP_ADDR_OBJ(wpobj)+1;
  len =   libGAP_LoadUInt();
  libGAP_STORE_LEN_WPOBJ(wpobj, len);
  for (i = 1; i <= len; i++)
    {
      *ptr++ = libGAP_LoadSubObj();
    }
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IsWPObj", "obj", &libGAP_IsWPObjFilt,
      libGAP_FuncIsWPObj, "src/weakptr.c:IsWPObj" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "WeakPointerObj", 1, "list",
      libGAP_FuncWeakPointerObj, "src/weakptr.c:WeakPointerObj" },

    { "LengthWPObj", 1, "wp",
      libGAP_FuncLengthWPObj, "src/weakptr.c:LengthWPObj" },

    { "SetElmWPObj", 3, "wp, pos, val",
      libGAP_FuncSetElmWPObj, "src/weakptr.c:SetElmWPObj" },

    { "IsBoundElmWPObj", 2, "wp, pos",
      libGAP_FuncIsBoundElmWPObj, "src/weakptr.c:IsBoundElmWPObj" },

    { "UnbindElmWPObj", 2, "wp, pos",
      libGAP_FuncUnbindElmWPObj, "src/weakptr.c:UnbindElmWPObj" },

    { "ElmWPObj", 2, "wp, pos",
      libGAP_FuncElmWPObj, "src/weakptr.c:ElmWPObj" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* install the marking and sweeping methods                            */
    libGAP_InfoBags[ libGAP_T_WPOBJ          ].name = "object (weakptr)";
    libGAP_InfoBags[ libGAP_T_WPOBJ +libGAP_COPYING ].name = "object (weakptr, copied)";

    libGAP_InitMarkFuncBags ( libGAP_T_WPOBJ,          libGAP_MarkWeakPointerObj   );
    libGAP_InitSweepFuncBags( libGAP_T_WPOBJ,          libGAP_SweepWeakPointerObj  );
    libGAP_InitMarkFuncBags ( libGAP_T_WPOBJ +libGAP_COPYING, libGAP_MarkWeakPointerObj   );
    libGAP_InitSweepFuncBags( libGAP_T_WPOBJ +libGAP_COPYING, libGAP_SweepWeakPointerObj  );

    /* typing method                                                       */
    libGAP_TypeObjFuncs[ libGAP_T_WPOBJ ] = libGAP_TypeWPObj;
    libGAP_ImportGVarFromLibrary( "TYPE_WPOBJ", &libGAP_TYPE_WPOBJ );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* saving function                                                     */
    libGAP_SaveObjFuncs[ libGAP_T_WPOBJ ] = libGAP_SaveWPObj;
    libGAP_LoadObjFuncs[ libGAP_T_WPOBJ ] = libGAP_LoadWPObj;
    
    /* copying functions                                                   */
    libGAP_CopyObjFuncs[  libGAP_T_WPOBJ           ] = libGAP_CopyObjWPObj;
    libGAP_CopyObjFuncs[  libGAP_T_WPOBJ + libGAP_COPYING ] = libGAP_CopyObjWPObjCopy;
    libGAP_CleanObjFuncs[ libGAP_T_WPOBJ           ] = libGAP_CleanObjWPObj;
    libGAP_CleanObjFuncs[ libGAP_T_WPOBJ + libGAP_COPYING ] = libGAP_CleanObjWPObjCopy;

    libGAP_MakeImmutableObjFuncs[ libGAP_T_WPOBJ ] = libGAP_MakeImmutableWPObj;
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoWeakPtr() . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "weakptr",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoWeakPtr ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  weakptr.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
