
/****************************************************************************
**
*W  vecffe.c                    GAP source                      Werner Nickel
**
**
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "ariths.h"              /* basic arithmetic                */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "integer.h"             /* integers                        */
#include        "finfield.h"            /* finite fields                   */

#include        "gvars.h"               /* global variables                */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listoper.h"            /* operations for generic lists    */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "vecffe.h"              /* functions for fin field vectors */

#include        "range.h"               /* ranges                          */

#include        "calls.h"               /* needed for opers.h              */
#include        "opers.h"               /* for TRY_NEXT_METHOD             */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */

#include <assert.h>


/****************************************************************************
**
*F  SumFFEVecFFE(<elmL>,<vecR>) . . . .  sum of an fin field elm and a vector
**
**  'SumFFEVecFFE' returns the sum of the fin field elm <elmL> and the vector
**  <vecR>.  The sum is a  list, where each element is  the sum of <elmL> and
**  the corresponding element of <vecR>.
**
**  'SumFFEVecFFE' is an improved version  of  'SumSclList', which  does  not
**  call 'SUM'.
*/
libGAP_Obj             libGAP_SumFFEVecFFE (
    libGAP_Obj                 elmL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecS;           /* handle of the sum               */
    libGAP_Obj *               ptrS;           /* pointer into the sum            */
    libGAP_FFV                 valS;           /* the value of a sum              */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_FFV                 valR;           /* the value of an element in vecR */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */
    libGAP_FFV                 valL;           /* the value of elmL               */

    /* get the field and check that elmL and vecR have the same field      */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1));
    if (libGAP_FLD_FFE(elmL) != fld) {
        /* check the characteristic                                          */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(elmL)))
            return libGAP_SumSclList(elmL, vecR);

        elmL = libGAP_ErrorReturnObj(
            "<elm>+<vec>: <elm> and <vec> must belong to the same finite field",
            0L, 0L, "you can replace <elm> via 'return <elm>;'");
        return libGAP_SUM(elmL, vecR);
    }

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecR);
    vecS = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecR) ?
                libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecS, len);

    /* to add we need the successor table                                  */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and add                                      */
    valL = libGAP_VAL_FFE(elmL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrS = libGAP_ADDR_OBJ(vecS);
    for (i = 1; i <= len; i++) {
        valR = libGAP_VAL_FFE(ptrR[i]);
        valS = libGAP_SUM_FFV(valL, valR, succ);
        ptrS[i] = libGAP_NEW_FFE(fld, valS);
    }

    /* return the result                                                   */
    return vecS;
}


/****************************************************************************
**
*F  SumVecFFEFFE(<vecL>,<elmR>) . . . . . sum of a vector and a fin field elm
**
**  'SumVecFFEFFE' returns  the sum of   the  vector <vecL> and  the  finite
**  field element <elmR>.  The sum is a  list, where each element  is the sum
**  of <elmR> and the corresponding element of <vecL>.
**
**  'SumVecFFEFFE' is an improved version  of  'SumListScl', which  does  not
**  call 'SUM'.
*/
libGAP_Obj             libGAP_SumVecFFEFFE (
    libGAP_Obj                 vecL,
    libGAP_Obj                 elmR )
{
    libGAP_Obj                 vecS;           /* handle of the sum               */
    libGAP_Obj *               ptrS;           /* pointer into the sum            */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */
    libGAP_FFV                 valR;           /* the value of elmR               */
    libGAP_FFV                 valL;           /* the value of an element in vecL */
    libGAP_FFV                 valS;           /* the value of a sum              */

    /* get the field and check that vecL and elmR have the same field      */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(elmR) != fld) {
        /* check the characteristic                                          */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(elmR)))
            return libGAP_SumListScl(vecL, elmR);

        elmR = libGAP_ErrorReturnObj(
            "<vec>+<elm>: <elm> and <vec> must belong to the same finite field",
            0L, 0L, "you can replace <elm> via 'return <elm>;'");
        return libGAP_SUM(vecL, elmR);
    }

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecL);
    vecS = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecL) ?
                libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecS, len);

    /* to add we need the successor table                                  */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and add                                      */
    valR = libGAP_VAL_FFE(elmR);
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrS = libGAP_ADDR_OBJ(vecS);
    for (i = 1; i <= len; i++) {
        valL = libGAP_VAL_FFE(ptrL[i]);
        valS = libGAP_SUM_FFV(valL, valR, succ);
        ptrS[i] = libGAP_NEW_FFE(fld, valS);
    }

    /* return the result                                                   */
    return vecS;
}

/****************************************************************************
**
*F  SumVecFFEVecFFE(<vecL>,<vecR>)  . . . . . . . . . . .  sum of two vectors
**
**  'SumVecFFEVecFFE' returns the sum  of the two  vectors <vecL> and <vecR>.
**  The sum is a new list, where each element is the sum of the corresponding
**  elements of <vecL> and <vecR>.
**
**  'SumVecFFEVecFFE' is an improved version of 'SumListList', which does not
**  call 'SUM'.
*/
libGAP_Obj             libGAP_SumVecFFEVecFFE (
    libGAP_Obj                 vecL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecS;           /* handle of the sum               */
    libGAP_Obj *               ptrS;           /* pointer into the sum            */
    libGAP_FFV                 valS;           /* one element of sum list         */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_FFV                 valL;           /* one element of left operand     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_FFV                 valR;           /* one element of right operand    */
    libGAP_UInt                lenL, lenR, len; /* length                          */
    libGAP_UInt                lenmin;
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */

    /* check the lengths                                                   */
    lenL = libGAP_LEN_PLIST(vecL);
    lenR = libGAP_LEN_PLIST(vecR);
    if (lenR > lenL) {
        len = lenR;
        lenmin = lenL;
    } else {
        len = lenL;
        lenmin = lenR;
    }

    /* check the fields                                                    */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1)) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1))))
            return libGAP_SumListList(vecL, vecR);

        vecR = libGAP_ErrorReturnObj(
            "Vector +: vectors have different fields",
            0L, 0L, "you can replace vector <right> via 'return <right>;'");
        return libGAP_SUM(vecL, vecR);
    }

    /* make the result list                                                */
    vecS = libGAP_NEW_PLIST((libGAP_IS_MUTABLE_OBJ(vecL) || libGAP_IS_MUTABLE_OBJ(vecR)) ?
                libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecS, len);

    /* to add we need the successor table                                  */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and add                                      */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrS = libGAP_ADDR_OBJ(vecS);
    for (i = 1; i <= lenmin; i++) {
        valL = libGAP_VAL_FFE(ptrL[i]);
        valR = libGAP_VAL_FFE(ptrR[i]);
        valS = libGAP_SUM_FFV(valL, valR, succ);
        ptrS[i] = libGAP_NEW_FFE(fld, valS);
    }
    if (lenL < lenR)
        for (; i <= len; i++)
            ptrS[i] = ptrR[i];
    else
        for (; i <= len; i++)
            ptrS[i] = ptrL[i];

    /* return the result                                                   */
    return vecS;
}

/****************************************************************************
**
*F  DiffFFEVecFFE(<elmL>,<vecR>)   difference of a fin field elm and a vector
**
**  'DiffFFEVecFFE' returns  the difference  of  the finite field element
**  <elmL> and  the vector <vecR>.   The difference  is  a list,  where  each
**  element is the difference of <elmL> and the corresponding element of
**  <vecR>. 
**
**  'DiffFFEVecFFE'  is an improved  version of 'DiffSclList', which does not
**  call 'DIFF'.
*/
libGAP_Obj             libGAP_DiffFFEVecFFE (
    libGAP_Obj                 elmL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecD;           /* handle of the difference        */
    libGAP_Obj *               ptrD;           /* pointer into the difference     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */
    libGAP_FFV                 valR;           /* the value of elmL               */
    libGAP_FFV                 valL;           /* the value of an element in vecR */
    libGAP_FFV                 valD;           /* the value of a difference       */

    /* check the fields                                                    */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1));
    if (libGAP_FLD_FFE(elmL) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(elmL)))
            return libGAP_DiffSclList(elmL, vecR);

        elmL = libGAP_ErrorReturnObj(
            "<elm>-<vec>: <elm> and <vec> must belong to the same finite field",
            0L, 0L, "you can replace <elm> via 'return <elm>;'");
        return libGAP_DIFF(elmL, vecR);
    }

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecR);
    vecD = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecR) ?
                libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecD, len);

    /* to subtract we need the successor table                             */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and subtract                                 */
    valL = libGAP_VAL_FFE(elmL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrD = libGAP_ADDR_OBJ(vecD);
    for (i = 1; i <= len; i++) {
        valR = libGAP_VAL_FFE(ptrR[i]);
        valR = libGAP_NEG_FFV(valR, succ);
        valD = libGAP_SUM_FFV(valL, valR, succ);
        ptrD[i] = libGAP_NEW_FFE(fld, valD);
    }
    /* return the result                                                   */
    return vecD;
}


/****************************************************************************
**
*F  DiffVecFFEFFE(<vecL>,<elmR>)   difference of a vector and a fin field elm
**
**  'DiffVecFFEFFE' returns   the  difference of the  vector  <vecL>  and the
**  finite field element <elmR>.  The difference   is a list,   where each
**  element  is the difference of <elmR> and the corresponding element of
**  <vecL>. 
**
**  'DiffVecFFEFFE' is an improved  version of 'DiffListScl', which  does not
**  call 'DIFF'.
*/
libGAP_Obj             libGAP_DiffVecFFEFFE (
    libGAP_Obj                 vecL,
    libGAP_Obj                 elmR )
{
    libGAP_Obj                 vecD;           /* handle of the difference        */
    libGAP_Obj *               ptrD;           /* pointer into the difference     */
    libGAP_FFV                 valD;           /* the value of a difference       */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_FFV                 valL;           /* the value of an element in vecL */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */
    libGAP_FFV                 valR;           /* the value of elmR               */

    /* get the field and check that vecL and elmR have the same field      */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(elmR) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(elmR)))
            return libGAP_DiffListScl(vecL, elmR);

        elmR = libGAP_ErrorReturnObj(
            "<vec>-<elm>: <elm> and <vec> must belong to the same finite field",
            0L, 0L, "you can replace <elm> via 'return <elm>;'");
        return libGAP_DIFF(vecL, elmR);
    }

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecL);
    vecD = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecL) ?
                libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecD, len);

    /* to subtract we need the successor table                             */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and subtract                                 */
    valR = libGAP_VAL_FFE(elmR);
    valR = libGAP_NEG_FFV(valR, succ);
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrD = libGAP_ADDR_OBJ(vecD);
    for (i = 1; i <= len; i++) {
        valL = libGAP_VAL_FFE(ptrL[i]);
        valD = libGAP_SUM_FFV(valL, valR, succ);
        ptrD[i] = libGAP_NEW_FFE(fld, valD);
    }

    /* return the result                                                   */
    return vecD;
}


/****************************************************************************
**
*F  DiffVecFFEVecFFE(<vecL>,<vecR>) . . . . . . . . difference of two vectors
**
**  'DiffVecFFEVecFFE'  returns the difference of the  two vectors <vecL> and
**  <vecR>.   The  difference is   a new   list, where  each  element  is the
**  difference of the corresponding elements of <vecL> and <vecR>.
**
**  'DiffVecFFEVecFFE' is an improved  version of  'DiffListList', which does
**  not call 'DIFF'.
*/
libGAP_Obj             libGAP_DiffVecFFEVecFFE (
    libGAP_Obj                 vecL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecD;           /* handle of the difference        */
    libGAP_Obj *               ptrD;           /* pointer into the difference     */
    libGAP_FFV                 valD;           /* one element of difference list  */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_FFV                 valL;           /* one element of left operand     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_FFV                 valR;           /* one element of right operand    */
    libGAP_UInt                len, lenL, lenR; /* length                          */
    libGAP_UInt                lenmin;
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */

    /* check the lengths                                                   */
    lenL = libGAP_LEN_PLIST(vecL);
    lenR = libGAP_LEN_PLIST(vecR);
    if (lenR > lenL) {
        len = lenR;
        lenmin = lenL;
    } else {
        len = lenL;
        lenmin = lenR;
    }

    /* check the fields                                                    */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1)) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1))))
            return libGAP_DiffListList(vecL, vecR);

        vecR = libGAP_ErrorReturnObj(
            "Vector -: vectors have different fields",
            0L, 0L, "you can replace vector <right> via 'return <right>;'");
        return libGAP_DIFF(vecL, vecR);
    }

    /* make the result list                                                */
    vecD = libGAP_NEW_PLIST((libGAP_IS_MUTABLE_OBJ(vecL) || libGAP_IS_MUTABLE_OBJ(vecR)) ?
                libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecD, len);

    /* to subtract we need the successor table                             */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and subtract                                 */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrD = libGAP_ADDR_OBJ(vecD);
    for (i = 1; i <= lenmin; i++) {
        valL = libGAP_VAL_FFE(ptrL[i]);
        valR = libGAP_VAL_FFE(ptrR[i]);
        valR = libGAP_NEG_FFV(valR, succ);
        valD = libGAP_SUM_FFV(valL, valR, succ);
        ptrD[i] = libGAP_NEW_FFE(fld, valD);
    }

    if (lenL < lenR)
        for (; i <= len; i++) {
            valR = libGAP_VAL_FFE(ptrR[i]);
            valD = libGAP_NEG_FFV(valR, succ);
            ptrD[i] = libGAP_NEW_FFE(fld, valD);
        }
    else
        for (; i <= len; i++)
            ptrD[i] = ptrL[i];

    /* return the result                                                   */
    return vecD;
}


/****************************************************************************
**
*F  ProdFFEVecFFE(<elmL>,<vecR>)  . . product of a fin field elm and a vector
**
**  'ProdFFEVecFFE' returns the product of the finite field element  <elmL>
**  and the vector <vecR>.  The product is  the list, where  each element is
**  the product  of <elmL> and the corresponding entry of <vecR>.
**
**  'ProdFFEVecFFE'  is an  improved version of 'ProdSclList', which does not
**  call 'PROD'.
*/
libGAP_Obj             libGAP_ProdFFEVecFFE (
    libGAP_Obj                 elmL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecP;           /* handle of the product           */
    libGAP_Obj *               ptrP;           /* pointer into the product        */
    libGAP_FFV                 valP;           /* the value of a product          */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_FFV                 valR;           /* the value of an element in vecR */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */
    libGAP_FFV                 valL;           /* the value of elmL               */

    /* get the field and check that elmL and vecR have the same field      */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1));
    if (libGAP_FLD_FFE(elmL) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(elmL)))
            return libGAP_ProdSclList(elmL, vecR);

        elmL = libGAP_ErrorReturnObj(
            "<elm>*<vec>: <elm> and <vec> must belong to the same finite field",
            0L, 0L, "you can replace <elm> via 'return <elm>;'");
        return libGAP_PROD(elmL, vecR);
    }

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecR);
    vecP = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecR) ?
                libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecP, len);

    /* to multiply we need the successor table                             */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and multiply                                 */
    valL = libGAP_VAL_FFE(elmL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrP = libGAP_ADDR_OBJ(vecP);
    for (i = 1; i <= len; i++) {
        valR = libGAP_VAL_FFE(ptrR[i]);
        valP = libGAP_PROD_FFV(valL, valR, succ);
        ptrP[i] = libGAP_NEW_FFE(fld, valP);
    }

    /* return the result                                                   */
    return vecP;
}

/****************************************************************************
**
*F  ProdVecFFEFFE(<vecL>,<elmR>)  .  product of a vector and a fin field elm
**
**  'ProdVecFFEFFE' returns the product of the finite field element  <elmR>
**  and the vector <vecL>.  The  product is the  list, where each element  is
**  the product of <elmR> and the corresponding element of <vecL>.
**
**  'ProdVecFFEFFE'  is an  improved version of 'ProdSclList', which does not
**  call 'PROD'.
*/
libGAP_Obj             libGAP_ProdVecFFEFFE (
    libGAP_Obj                 vecL,
    libGAP_Obj                 elmR )
{
    libGAP_Obj                 vecP;           /* handle of the product           */
    libGAP_Obj *               ptrP;           /* pointer into the product        */
    libGAP_FFV                 valP;           /* the value of a product          */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_FFV                 valL;           /* the value of an element in vecL */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */
    libGAP_FFV                 valR;           /* the value of elmR               */

    /* get the field and check that vecL and elmR have the same field      */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(elmR) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(elmR)))
            return libGAP_ProdListScl(vecL, elmR);

        elmR = libGAP_ErrorReturnObj(
            "<vec>*<elm>: <elm> and <vec> must belong to the same finite field",
            0L, 0L, "you can replace <elm> via 'return <elm>;'");
        return libGAP_PROD(vecL, elmR);
    }

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecL);
    vecP = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecL) ?
                    libGAP_T_PLIST_FFE : libGAP_T_PLIST_FFE + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecP, len);

    /* to multiply we need the successor table                             */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and multiply                                 */
    valR = libGAP_VAL_FFE(elmR);
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrP = libGAP_ADDR_OBJ(vecP);
    for (i = 1; i <= len; i++) {
        valL = libGAP_VAL_FFE(ptrL[i]);
        valP = libGAP_PROD_FFV(valL, valR, succ);
        ptrP[i] = libGAP_NEW_FFE(fld, valP);
    }

    /* return the result                                                   */
    return vecP;
}


/****************************************************************************
**
*F  ProdVecFFEVecFFE(<vecL>,<vecR>) . . . . . . . . .  product of two vectors
**
**  'ProdVecFFEVecFFE'  returns the product  of   the two vectors <vecL>  and
**  <vecR>.  The  product  is the  sum of the   products of the corresponding
**  elements of the two lists.
**
**  'ProdVecFFEVecFFE' is an improved version  of 'ProdListList',  which does
**  not call 'PROD'.
*/
libGAP_Obj             libGAP_ProdVecFFEVecFFE (
    libGAP_Obj                 vecL,
    libGAP_Obj                 vecR )
{
    libGAP_FFV                 valP;           /* one product                     */
    libGAP_FFV                 valS;           /* sum of the products             */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_FFV                 valL;           /* one element of left operand     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_FFV                 valR;           /* one element of right operand    */
    libGAP_UInt                lenL, lenR, len; /* length                          */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_FF                  fld;            /* finite field                    */
    libGAP_FF *                succ;           /* successor table                 */

    /* check the lengths                                                   */
    lenL = libGAP_LEN_PLIST(vecL);
    lenR = libGAP_LEN_PLIST(vecR);
    len = (lenL < lenR) ? lenL : lenR;

    /* check the fields                                                    */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1)) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1))))
            return libGAP_ProdListList(vecL, vecR);

        vecR = libGAP_ErrorReturnObj(
            "Vector *: vectors have different fields",
            0L, 0L,
            "you can replace vector <right> via 'return <right>;'");
        return libGAP_PROD(vecL, vecR);
    }

    /* to add we need the successor table                                  */
    succ = libGAP_SUCC_FF(fld);

    /* loop over the elements and add                                      */
    valS = (libGAP_FFV)0;
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    for (i = 1; i <= len; i++) {
        valL = libGAP_VAL_FFE(ptrL[i]);
        valR = libGAP_VAL_FFE(ptrR[i]);
        valP = libGAP_PROD_FFV(valL, valR, succ);
        valS = libGAP_SUM_FFV(valS, valP, succ);
    }

    /* return the result                                                   */
    return libGAP_NEW_FFE(fld, valS);
}

/****************************************************************************
**
*F  FuncAddRowVectorVecFFEsMult( <self>, <vecL>, <vecR>, <mult> )
**
*/

static libGAP_Obj libGAP_AddRowVectorOp;   /* BH changed to static */

libGAP_Obj libGAP_FuncAddRowVectorVecFFEsMult( libGAP_Obj self, libGAP_Obj vecL, libGAP_Obj vecR, libGAP_Obj mult )
{
    libGAP_Obj *ptrL;
    libGAP_Obj *ptrR;
    libGAP_FFV  valM;
    libGAP_FFV  valS;
    libGAP_FFV  valL;
    libGAP_FFV  valR;
    libGAP_FF  fld;
    libGAP_FFV *succ;
    libGAP_UInt len;
    libGAP_UInt xtype;
    libGAP_UInt i;

    if (libGAP_TNUM_OBJ(mult) != libGAP_T_FFE)
        return libGAP_TRY_NEXT_METHOD;

    if (libGAP_VAL_FFE(mult) == 0)
        return (libGAP_Obj) 0;

    xtype = libGAP_KTNumPlist(vecL, (libGAP_Obj *) 0);
    if (xtype != libGAP_T_PLIST_FFE && xtype != libGAP_T_PLIST_FFE + libGAP_IMMUTABLE)
        return libGAP_TRY_NEXT_METHOD;

    xtype = libGAP_KTNumPlist(vecR, (libGAP_Obj *) 0);
    if (xtype != libGAP_T_PLIST_FFE && xtype != libGAP_T_PLIST_FFE + libGAP_IMMUTABLE)
        return libGAP_TRY_NEXT_METHOD;


    /* check the lengths                                                   */
    len = libGAP_LEN_PLIST(vecL);
    if (len != libGAP_LEN_PLIST(vecR)) {
        vecR = libGAP_ErrorReturnObj(
            "AddRowVector: vector lengths differ <left> %d,  <right> %d",
            (libGAP_Int)len, (libGAP_Int)libGAP_LEN_PLIST(vecR),
            "you can replace vector <right> via 'return <right>;'");
        return libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, vecL, vecR, mult);
    }

    /* check the fields                                                    */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1)) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1))))
            return libGAP_TRY_NEXT_METHOD;

        vecR = libGAP_ErrorReturnObj(
            "AddRowVector: vectors have different fields",
            0L, 0L,
            "you can replace vector <right> via 'return <right>;'");
        return libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, vecL, vecR, mult);
    }

    /* Now check the multiplier field */
    if (libGAP_FLD_FFE(mult) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) != libGAP_CHAR_FF(libGAP_FLD_FFE(mult))) {
            mult = libGAP_ErrorReturnObj(
                "AddRowVector: <multiplier> has different field",
                0L, 0L,
                "you can replace <multiplier> via 'return <multiplier>;'");
            return libGAP_CALL_3ARGS(libGAP_AddRowVectorOp, vecL, vecR, mult);
        }

        /* if the multiplier is over a non subfield then redispatch */
        if ((libGAP_DEGR_FF(fld) % libGAP_DegreeFFE(mult)) != 0)
            return libGAP_TRY_NEXT_METHOD;

        /* otherwise it's a subfield, so promote it */
        valM = libGAP_VAL_FFE(mult);
        if (valM != 0)
            valM = 1 + (valM - 1) * (libGAP_SIZE_FF(fld) - 1) / (libGAP_SIZE_FF(libGAP_FLD_FFE(mult)) - 1);
    } else
        valM = libGAP_VAL_FFE(mult);


    succ = libGAP_SUCC_FF(fld);
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);

    /* two versions of the loop to avoid multipling by 1 */
    if (valM == 1)
        for (i = 1; i <= len; i++) {
            valL = libGAP_VAL_FFE(ptrL[i]);
            valR = libGAP_VAL_FFE(ptrR[i]);
            valS = libGAP_SUM_FFV(valL, valR, succ);
            ptrL[i] = libGAP_NEW_FFE(fld, valS);
        }
    else
        for (i = 1; i <= len; i++) {
            valL = libGAP_VAL_FFE(ptrL[i]);
            valR = libGAP_VAL_FFE(ptrR[i]);
            valS = libGAP_PROD_FFV(valR, valM, succ);
            valS = libGAP_SUM_FFV(valL, valS, succ);
            ptrL[i] = libGAP_NEW_FFE(fld, valS);
        }
    return (libGAP_Obj) 0;
}
/****************************************************************************
**
*F  FuncMultRowVectorVecFFEs( <self>, <vec>, <mult> )
**
*/

static libGAP_Obj libGAP_MultRowVectorOp;   /* BH changed to static */

libGAP_Obj libGAP_FuncMultRowVectorVecFFEs( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj mult )
{
    libGAP_Obj *ptr;
    libGAP_FFV  valM;
    libGAP_FFV  valS;
    libGAP_FFV  val;
    libGAP_FF  fld;
    libGAP_FFV *succ;
    libGAP_UInt len;
    libGAP_UInt xtype;
    libGAP_UInt i;

    if (libGAP_TNUM_OBJ(mult) != libGAP_T_FFE)
        return libGAP_TRY_NEXT_METHOD;

    if (libGAP_VAL_FFE(mult) == 1)
        return (libGAP_Obj) 0;

    xtype = libGAP_KTNumPlist(vec, (libGAP_Obj *) 0);
    if (xtype != libGAP_T_PLIST_FFE &&
    xtype != libGAP_T_PLIST_FFE + libGAP_IMMUTABLE)
        return libGAP_TRY_NEXT_METHOD;

    /* check the lengths                                                   */
    len = libGAP_LEN_PLIST(vec);

    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vec, 1));
    /* Now check the multiplier field */
    if (libGAP_FLD_FFE(mult) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) != libGAP_CHAR_FF(libGAP_FLD_FFE(mult))) {
            mult = libGAP_ErrorReturnObj(
                "MultRowVector: <multiplier> has different field",
                0L, 0L,
                "you can replace <multiplier> via 'return <multiplier>;'");
            return libGAP_CALL_2ARGS(libGAP_MultRowVectorOp, vec, mult);
        }

        /* if the multiplier is over a non subfield then redispatch */
        if ((libGAP_DEGR_FF(fld) % libGAP_DegreeFFE(mult)) != 0)
            return libGAP_TRY_NEXT_METHOD;

        /* otherwise it's a subfield, so promote it */
        valM = libGAP_VAL_FFE(mult);
        if (valM != 0)
            valM = 1 + (valM - 1) * (libGAP_SIZE_FF(fld) - 1) / (libGAP_SIZE_FF(libGAP_FLD_FFE(mult)) - 1);
    } else
        valM = libGAP_VAL_FFE(mult);


    succ = libGAP_SUCC_FF(fld);
    ptr = libGAP_ADDR_OBJ(vec);

    /* two versions of the loop to avoid multipling by 0 */
    if (valM == 0) {
        libGAP_Obj z;
        z = libGAP_NEW_FFE(fld, 0);
        for (i = 1; i <= len; i++) {
            ptr[i] = z;
        }
    } else
        for (i = 1; i <= len; i++) {
            val = libGAP_VAL_FFE(ptr[i]);
            valS = libGAP_PROD_FFV(val, valM, succ);
            ptr[i] = libGAP_NEW_FFE(fld, valS);
        }
    return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  FuncAddRowVectorVecFFEs( <self>, <vecL>, <vecR> )
**
*/
libGAP_Obj libGAP_FuncAddRowVectorVecFFEs( libGAP_Obj self, libGAP_Obj vecL, libGAP_Obj vecR )
{
    libGAP_Obj *ptrL;
    libGAP_Obj *ptrR;
    libGAP_FFV  valS;
    libGAP_FFV  valL;
    libGAP_FFV  valR;
    libGAP_FF  fld;
    libGAP_FFV *succ;
    libGAP_UInt len;
    libGAP_UInt xtype;
    libGAP_UInt i;

    xtype = libGAP_KTNumPlist(vecL, (libGAP_Obj *) 0);
    if (xtype != libGAP_T_PLIST_FFE && xtype != libGAP_T_PLIST_FFE + libGAP_IMMUTABLE)
        return libGAP_TRY_NEXT_METHOD;

    xtype = libGAP_KTNumPlist(vecR, (libGAP_Obj *) 0);
    if (xtype != libGAP_T_PLIST_FFE && xtype != libGAP_T_PLIST_FFE + libGAP_IMMUTABLE)
        return libGAP_TRY_NEXT_METHOD;

    /* check the lengths                                                   */
    len = libGAP_LEN_PLIST(vecL);
    if (len != libGAP_LEN_PLIST(vecR)) {
        vecR = libGAP_ErrorReturnObj(
            "Vector *: vector lengths differ <left> %d,  <right> %d",
            (libGAP_Int)len, (libGAP_Int)libGAP_LEN_PLIST(vecR),
            "you can replace vector <right> via 'return <right>;'");
        return libGAP_CALL_2ARGS(libGAP_AddRowVectorOp, vecL, vecR);
    }

    /* check the fields                                                    */
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1)) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1))))
            return libGAP_TRY_NEXT_METHOD;

        vecR = libGAP_ErrorReturnObj(
            "AddRowVector: vectors have different fields",
            0L, 0L,
            "you can replace vector <right> via 'return <right>;'");
        return libGAP_CALL_2ARGS(libGAP_AddRowVectorOp, vecL, vecR);
    }

    succ = libGAP_SUCC_FF(fld);
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);

    for (i = 1; i <= len; i++) {
        valL = libGAP_VAL_FFE(ptrL[i]);
        valR = libGAP_VAL_FFE(ptrR[i]);
        valS = libGAP_SUM_FFV(valL, valR, succ);
        ptrL[i] = libGAP_NEW_FFE(fld, valS);
    }
    return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  ProdVectorMatrix(<vecL>,<vecR>) . . . .  product of a vector and a matrix
**
**  'ProdVectorMatrix' returns the product of the vector <vecL> and the matrix
**  <vecR>.  The product is the sum of the  rows  of <vecR>, each multiplied by
**  the corresponding entry of <vecL>.
**
**  'ProdVectorMatrix'  is an improved version of 'ProdListList',  which does
**  not  call 'PROD' and  also accumulates  the sum into  one  fixed  vector
**  instead of allocating a new for each product and sum.
*/
libGAP_Obj             libGAP_ProdVecFFEMatFFE (
				  libGAP_Obj                 vecL,
				  libGAP_Obj                 matR )
{
    libGAP_Obj                 vecP;           /* handle of the product           */
    libGAP_Obj *               ptrP;           /* pointer into the product        */
    libGAP_FFV *               ptrV;           /* value pointer into the product  */
    libGAP_FFV                 valP;           /* one value of the product        */
    libGAP_FFV                 valL;           /* one value of the left operand   */
    libGAP_Obj                 vecR;           /* one vector of the right operand */
    libGAP_Obj *               ptrR;           /* pointer into the right vector   */
    libGAP_FFV                 valR;           /* one value from the right vector */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                col;            /* length of the rows in matR      */
    libGAP_UInt                i, k;           /* loop variables                  */
    libGAP_FF                  fld;            /* the common finite field         */
    libGAP_FF *                succ;           /* the successor table             */

    /* check the lengths                                                   */
    len = libGAP_LEN_PLIST(vecL);
    col = libGAP_LEN_PLIST(libGAP_ELM_PLIST(matR, 1));
    if (len != libGAP_LEN_PLIST(matR)) {
        matR = libGAP_ErrorReturnObj(
            "<vec>*<mat>: <vec> (%d) must have the same length as <mat> (%d)",
            (libGAP_Int)len, (libGAP_Int)col,
            "you can replace matrix <mat> via 'return <mat>;'");
        return libGAP_PROD(vecL, matR);
    }

    /* check the fields                                                    */
    vecR = libGAP_ELM_PLIST(matR, 1);
    fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(vecL, 1));
    if (libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1)) != fld) {
        /* check the characteristic                                        */
        if (libGAP_CHAR_FF(fld) == libGAP_CHAR_FF(libGAP_FLD_FFE(libGAP_ELM_PLIST(vecR, 1))))
            return libGAP_ProdListList(vecL, matR);

        matR = libGAP_ErrorReturnObj(
            "<vec>*<mat>: <vec> and <mat> have different fields",
            0L, 0L,
            "you can replace matrix <mat> via 'return <mat>;'");
        return libGAP_PROD(vecL, matR);
    }

    /* make the result list by multiplying the first entries               */
    vecP = libGAP_ProdFFEVecFFE(libGAP_ELM_PLIST(vecL, 1), vecR);

    /* to add we need the successor table                                  */
    succ = libGAP_SUCC_FF(fld);

    /* convert vecP into a list of values                                  */
    /*N 5Jul1998 werner: This only works if sizeof(FFV) <= sizeof(Obj)     */
    /*N We have to be careful not to overwrite the length info             */
    ptrP = libGAP_ADDR_OBJ(vecP);
    ptrV = ((libGAP_FFV*)(ptrP + 1)) - 1;
    for (k = 1; k <= col; k++)
        ptrV[k] = libGAP_VAL_FFE(ptrP[k]);

    /* loop over the other entries and multiply                            */
    for (i = 2; i <= len; i++) {
        valL = libGAP_VAL_FFE(libGAP_ELM_PLIST(vecL, i));
        vecR = libGAP_ELM_PLIST(matR, i);
        ptrR = libGAP_ADDR_OBJ(vecR);
        if (valL == (libGAP_FFV)1) {
            for (k = 1; k <= col; k++) {
                valR = libGAP_VAL_FFE(ptrR[k]);
                valP = ptrV[k];
                ptrV[k] = libGAP_SUM_FFV(valP, valR, succ);
            }
        } else if (valL != (libGAP_FFV)0) {
            for (k = 1; k <= col; k++) {
                valR = libGAP_VAL_FFE(ptrR[k]);
                valR = libGAP_PROD_FFV(valL, valR, succ);
                valP = ptrV[k];
                ptrV[k] = libGAP_SUM_FFV(valP, valR, succ);
            }
        }
    }

    /* convert vecP back into a list of finite field elements              */
    /*N 5Jul1998 werner: This only works if sizeof(FFV) <= sizeof(Obj)     */
    /*N We have to be careful not to overwrite the length info             */
    for (k = col; k >= 1; k--)
        ptrP[k] = libGAP_NEW_FFE(fld, ptrV[k]);

    /* return the result                                                   */
    return vecP;
}


/****************************************************************************
**
*F  ZeroVecFFE(<vec>) . . . .  zero of an FFE Vector
**
**  'ZeroVecFEE' returns the zero of the vector <vec>.
**
**  It is a better version of ZeroListDefault for the case of vecffes
**  becuase it knows tha the zero is common and the result a vecffe
*/

libGAP_Obj libGAP_ZeroMutVecFFE( libGAP_Obj vec )
{
    libGAP_UInt i, len;
    libGAP_Obj res;
    libGAP_Obj z;
    assert(libGAP_TNUM_OBJ(vec) >= libGAP_T_PLIST_FFE && \
    libGAP_TNUM_OBJ(vec) <= libGAP_T_PLIST_FFE + libGAP_IMMUTABLE);
    len = libGAP_LEN_PLIST(vec);
    assert(len);
    res  = libGAP_NEW_PLIST(libGAP_T_PLIST_FFE, len);
    libGAP_SET_LEN_PLIST(res, len);
    z = libGAP_ZERO(libGAP_ELM_PLIST(vec, 1));
    for (i = 1; i <= len; i++)
        libGAP_SET_ELM_PLIST(res, i, z);
    return res;
}

libGAP_Obj libGAP_ZeroVecFFE( libGAP_Obj vec )
{
    libGAP_UInt i, len;
    libGAP_Obj res;
    libGAP_Obj z;
    assert(libGAP_TNUM_OBJ(vec) >= libGAP_T_PLIST_FFE && \
    libGAP_TNUM_OBJ(vec) <= libGAP_T_PLIST_FFE + libGAP_IMMUTABLE);
    len = libGAP_LEN_PLIST(vec);
    assert(len);
    res  = libGAP_NEW_PLIST(libGAP_TNUM_OBJ(vec), len);
    libGAP_SET_LEN_PLIST(res, len);
    z = libGAP_ZERO(libGAP_ELM_PLIST(vec, 1));
    for (i = 1; i <= len; i++)
        libGAP_SET_ELM_PLIST(res, i, z);
    return res;
}

libGAP_UInt libGAP_IsVecFFE(libGAP_Obj vec)
{
    libGAP_UInt tn;
    tn = libGAP_TNUM_OBJ(vec);
    if (tn >= libGAP_T_PLIST_FFE && tn <= libGAP_T_PLIST_FFE + libGAP_IMMUTABLE)
        return 1;
    if (!libGAP_IS_PLIST(vec))
        return 0;
    libGAP_TYPE_OBJ(vec); /* force a full inspection of the list */
    tn = libGAP_TNUM_OBJ(vec);
    return tn >= libGAP_T_PLIST_FFE && tn <= libGAP_T_PLIST_FFE + libGAP_IMMUTABLE;
}


libGAP_Obj libGAP_FuncIS_VECFFE( libGAP_Obj self, libGAP_Obj vec)
{
    return libGAP_IsVecFFE(vec) ? libGAP_True : libGAP_False;
}

libGAP_Obj libGAP_FuncCOMMON_FIELD_VECFFE( libGAP_Obj self, libGAP_Obj vec)
{
    libGAP_Obj elm;
    if (!libGAP_IsVecFFE(vec))
        return libGAP_Fail;
    elm = libGAP_ELM_PLIST(vec, 1);
    return libGAP_INTOBJ_INT(libGAP_SIZE_FF(libGAP_FLD_FFE(elm)));
}

libGAP_Obj libGAP_FuncSMALLEST_FIELD_VECFFE( libGAP_Obj self, libGAP_Obj vec)
{
    libGAP_Obj elm;
    libGAP_UInt deg, deg1, deg2, i, len, p, q;
    libGAP_UInt isVecFFE = libGAP_IsVecFFE(vec);
    len  = libGAP_LEN_PLIST(vec);
    if (len == 0)
        return libGAP_Fail;
    elm = libGAP_ELM_PLIST(vec, 1);
    if (!isVecFFE && !libGAP_IS_FFE(elm))
        return libGAP_Fail;
    deg = libGAP_DegreeFFE(elm);
    p = libGAP_CharFFE(elm);
    for (i = 2; i <= len; i++) {
        elm = libGAP_ELM_PLIST(vec, i);
        if (!isVecFFE && (!libGAP_IS_FFE(elm) || libGAP_CharFFE(elm) != p))
            return libGAP_Fail;
        deg2 =  libGAP_DegreeFFE(elm);
        deg1 = deg;
        while (deg % deg2 != 0)
            deg += deg1;
    }
    q = p;
    for (i = 2; i <= deg; i++)
        q *= p;
    return libGAP_INTOBJ_INT(q);
}

/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

  { "ADD_ROWVECTOR_VECFFES_3", 3, "vecl, vecr, mult",
    libGAP_FuncAddRowVectorVecFFEsMult, "src/vecffe.c: ADD_ROWVECTOR_VECFFES_3" },

  { "ADD_ROWVECTOR_VECFFES_2", 2, "vecl, vecr",
    libGAP_FuncAddRowVectorVecFFEs, "src/vecffe.c: ADD_ROWVECTOR_VECFFES_2" },

  { "MULT_ROWVECTOR_VECFFES", 2, "vec, mult",
    libGAP_FuncMultRowVectorVecFFEs, "src/vecffe.c: MULT_ROWVECTOR_VECFFES" },
  
  { "IS_VECFFE", 1, "vec",
    libGAP_FuncIS_VECFFE, "src/vecffe.c: IS_VECFFE" },

  { "COMMON_FIELD_VECFFE", 1, "vec",
    libGAP_FuncCOMMON_FIELD_VECFFE, "src/vecffe.c: COMMON_FIELD_VECFFE" },

  { "SMALLEST_FIELD_VECFFE", 1, "vec",
    libGAP_FuncSMALLEST_FIELD_VECFFE, "src/vecffe.c: SMALLEST_FIELD_VECFFE" },
  
    { 0 }

};


/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int                 t1;
    libGAP_Int                 t2;

    /* install the arithmetic operation methods                            */
    for (t1 = libGAP_T_PLIST_FFE; t1 <= libGAP_T_PLIST_FFE + libGAP_IMMUTABLE; t1++) {
        libGAP_SumFuncs[  libGAP_T_FFE ][  t1   ] = libGAP_SumFFEVecFFE;
        libGAP_SumFuncs[   t1   ][ libGAP_T_FFE ] = libGAP_SumVecFFEFFE;
        libGAP_DiffFuncs[ libGAP_T_FFE ][  t1   ] = libGAP_DiffFFEVecFFE;
        libGAP_DiffFuncs[  t1   ][ libGAP_T_FFE ] = libGAP_DiffVecFFEFFE;
        libGAP_ProdFuncs[ libGAP_T_FFE ][  t1   ] = libGAP_ProdFFEVecFFE;
        libGAP_ProdFuncs[  t1   ][ libGAP_T_FFE ] = libGAP_ProdVecFFEFFE;
        libGAP_ZeroFuncs[  t1   ] = libGAP_ZeroVecFFE;
        libGAP_ZeroMutFuncs[  t1   ] = libGAP_ZeroMutVecFFE;
    }

    for (t1 = libGAP_T_PLIST_FFE; t1 <= libGAP_T_PLIST_FFE + libGAP_IMMUTABLE; t1++) {
        for (t2 = libGAP_T_PLIST_FFE; t2 <= libGAP_T_PLIST_FFE + libGAP_IMMUTABLE; t2++) {
            libGAP_SumFuncs[  t1 ][ t2 ] =  libGAP_SumVecFFEVecFFE;
            libGAP_DiffFuncs[ t1 ][ t2 ] = libGAP_DiffVecFFEVecFFE;
            libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdVecFFEVecFFE;
        }
    }


    libGAP_InitHdlrFuncsFromTable(libGAP_GVarFuncs);

    libGAP_InitFopyGVar("AddRowVector", &libGAP_AddRowVectorOp);
    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable(libGAP_GVarFuncs);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoVecFFE()  . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "vecffe",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoVecFFE ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  vecffe.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
