/****************************************************************************
**
*W  sysfiles.c                  GAP source                       Frank Celler
*W                                                         & Martin Schönert
*W                                                  & Burkhard Höfling (MAC)
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  The  files  "system.c" and  "sysfiles.c"   contain  all operating  system
**  dependent functions.  File and  stream operations are implemented in this
**  files, all the other system dependent functions in "system.c".  There are
**  various  labels determine which operating  system is  actually used, they
**  are described in "system.c".
*/
#include        "system.h"              /* system dependent part           */


#include        "sysfiles.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */

#include        "lists.h"               /* generic lists                   */
#include        "listfunc.h"            /* functions for generic lists     */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "records.h"             /* generic records                 */
#include        "bool.h"                /* Global True and False           */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */

#include        "libgap_internal.h"     /* GAP shared library              */

#include        <assert.h>
#include        <fcntl.h>

#if HAVE_LIBREADLINE
#include        <readline/readline.h>   /* readline for interactive input  */
#endif

#include        "read.h"                /* reader                          */


#if HAVE_SELECT
/* Only for the Hook handler calls: */

#include        <sys/time.h>
#include        <sys/types.h>
#endif

#include        <stdio.h>               /* standard input/output functions */
#include        <stdlib.h>              /* ANSI standard functions         */
#include        <string.h>              /* string functions                */
#include        <time.h>                /* time functions                  */

#if HAVE_UNISTD_H                       /* definition of 'R_OK'            */
# include <unistd.h>
#endif


#if HAVE_SIGNAL_H                       /* signal handling functions       */
# include       <signal.h>
typedef void       libGAP_sig_handler_t ( int );
#endif


#if HAVE_VFORK_H
# include       <vfork.h>
#endif

#if HAVE_ERRNO_H
# include       <errno.h>
#else
extern int errno;
#endif

#if SYS_IS_CYGWIN32
# include       <process.h>
#endif


/* utility to check return value of 'write'  */
ssize_t libGAP_writeandcheck(int fd, const char *buf, size_t count) {
  int ret;
  ret = write(fd, buf, count);
  if (ret < 0)
    libGAP_ErrorQuit("Cannot write to file descriptor %d, see 'LastSystemError();'\n",
               fd, 0L);
  return ret;
}


/****************************************************************************
**


*F * * * * * * * * * * * * * * dynamic loading  * * * * * * * * * * * * * * *
*/


/****************************************************************************
**
*F  SyFindOrLinkGapRootFile( <filename>, <crc>, <res> ) . . . .  load or link
**
**  'SyFindOrLinkGapRootFile'  tries to find a GAP  file in the root area and
**  check  if   there is a corresponding    statically  or dynamically linked
**  module.  If the CRC matches this module  is loaded otherwise the filename
**  is returned.
**
**  The function returns:
**
**  0: no file or module was found
**  1: if a dynamically linked module was found
**  2: if a statically linked module was found
**  3: a GAP file was found
**  4: a GAP file was found and the CRC value didn't match
*/
#include        "compstat.h"            /* statically linked modules       */


libGAP_Int libGAP_SyFindOrLinkGapRootFile (
    const libGAP_Char *        filename,
    libGAP_Int4                crc_gap,
    libGAP_TypGRF_Data *       result )
{
    libGAP_UInt4               crc_dyn = 0;
    libGAP_UInt4               crc_sta = 0;
    libGAP_Int                 found_gap = 0;
    libGAP_Int                 found_dyn = 0;
    libGAP_Int                 found_sta = 0;
    libGAP_Char                tmpbuffer[256];
    libGAP_Char *              tmp;
    libGAP_Char                libGAP_module[256];
    libGAP_Char                name[256];
    libGAP_StructInitInfo *    info_dyn = 0;
    libGAP_StructInitInfo *    info_sta = 0;
    libGAP_Int                 k;

#if HAVE_DLOPEN
    const libGAP_Char *        p;
    const libGAP_Char *        dot;
    libGAP_Int                 pos;
    libGAP_Int                 pot = 0;
    libGAP_InitInfoFunc        init;
#endif

    /* find the GAP file                                                   */
    result->pathname[0] = '\0';
    tmp = libGAP_SyFindGapRootFile(filename, tmpbuffer);
    if ( tmp ) {
        libGAP_strxcpy( result->pathname, tmp, sizeof(result->pathname) );
        libGAP_strxcpy( name, tmp, sizeof(name) );
    }
    if ( result->pathname[0] ) {
        if ( libGAP_SyIsReadableFile(result->pathname) == 0 ) {
            found_gap = 1;
        }
        else {
            result->pathname[0] = '\0';
        }
    }
    if ( ! libGAP_SyUseModule ) {
        return ( found_gap ? 3 : 0 );
    }

    /* try to find any statically link module                              */
    libGAP_strxcpy( libGAP_module, "GAPROOT/", sizeof(libGAP_module) );

    libGAP_strxcat( libGAP_module, filename, sizeof(libGAP_module) );
    for ( k = 0;  libGAP_CompInitFuncs[k];  k++ ) {
        info_sta = (*(libGAP_CompInitFuncs[k]))();
        if ( info_sta == 0 ) {
            continue;
        }
        if ( ! strcmp( libGAP_module, info_sta->name ) ) {
            crc_sta   = info_sta->crc;
            found_sta = 1;
            break;
        }
    }


    /* try to find any dynamically loadable module for filename            */
#if HAVE_DLOPEN
    pos = strlen(filename);
    p   = filename + pos;
    dot = 0;
    while ( filename <= p && *p != '/' ) {
        if ( *p == '.' ) {
            dot = p;
            pot = pos;
        }
        p--;
        pos--;
    }
    libGAP_strxcpy( libGAP_module, "bin/", sizeof(libGAP_module) );
    libGAP_strxcat( libGAP_module, libGAP_SyArchitecture, sizeof(libGAP_module) );
    libGAP_strxcat( libGAP_module, "/compiled/", sizeof(libGAP_module) );
    if ( dot ) {
        if ( p < filename ) {
            libGAP_strxcat( libGAP_module, dot+1, sizeof(libGAP_module) );
            libGAP_strxcat( libGAP_module, "/", sizeof(libGAP_module) );
            libGAP_strxncat( libGAP_module, filename, sizeof(libGAP_module), pot );
        }
        else {
            libGAP_strxncat( libGAP_module, filename, sizeof(libGAP_module), pos );
            libGAP_strxcat( libGAP_module, "/", sizeof(libGAP_module) );
            libGAP_strxcat( libGAP_module, dot+1, sizeof(libGAP_module) );
            libGAP_strxncat( libGAP_module, filename+pos, sizeof(libGAP_module), pot-pos );
        }
    }
    else {
        libGAP_strxcat( libGAP_module, filename, sizeof(libGAP_module) );
    }
    libGAP_strxcat( libGAP_module, ".so", sizeof(libGAP_module) );
    tmp = libGAP_SyFindGapRootFile(libGAP_module, tmpbuffer);

    /* special handling for the case of package files */
    if (!tmp && !strncmp(filename, "pkg", 3)) {
        libGAP_Char pkgname[16];
        const libGAP_Char *p2;
        libGAP_Char *p1;
        p2 = filename + 4; /* after the pkg/ */
        p1 = pkgname;
        while (*p2 != '\0' && *p2 != '/')
          *p1++ = *p2++;
        *p1 = '\0';

        libGAP_module[0] = '\0';
        libGAP_strxcat( libGAP_module, "pkg/", sizeof(libGAP_module) );
        libGAP_strxncat( libGAP_module, pkgname, sizeof(libGAP_module), p1 - pkgname + 1 );
        libGAP_strxcat( libGAP_module, "/bin/", sizeof(libGAP_module) );
        libGAP_strxcat( libGAP_module, libGAP_SyArchitecture, sizeof(libGAP_module) );
        libGAP_strxcat( libGAP_module, "/compiled/", sizeof(libGAP_module) );
        if ( dot ) {
          if ( p <= p2 ) {
            libGAP_strxncat( libGAP_module, dot+1, sizeof(libGAP_module), strlen(dot+1) );
            libGAP_strxcat( libGAP_module, "/", sizeof(libGAP_module) );
            libGAP_strxncat( libGAP_module, p2+1, sizeof(libGAP_module), pot - (p2 + 1 - filename) );
          }
          else {
            libGAP_strxncat( libGAP_module, p2+1, sizeof(libGAP_module), pos - (p2 +1 - filename) );
            libGAP_strxcat( libGAP_module, "/", sizeof(libGAP_module) );
            libGAP_strxncat( libGAP_module, dot+1, sizeof(libGAP_module), strlen(dot+1) );
            libGAP_strxncat( libGAP_module, filename+pos, sizeof(libGAP_module), pot-pos );
          }
        }
        else {
          libGAP_strxcat( libGAP_module, p2, sizeof(libGAP_module) );
        }
        libGAP_strxcat( libGAP_module, ".so", sizeof(libGAP_module) );
        tmp = libGAP_SyFindGapRootFile(libGAP_module, tmpbuffer);

     }
    if ( tmp ) {
        init = libGAP_SyLoadModule(tmp);
        if ( ( (libGAP_Int)init & 1 ) == 0 ) {
            info_dyn  = (*init)();
            crc_dyn   = info_dyn->crc;
            found_dyn = 1;
        }
    }
#endif

    /* check if we have to compute the crc                                 */
    if ( found_gap && ( found_dyn || found_sta ) ) {
        if ( crc_gap == 0 ) {
            crc_gap = libGAP_SyGAPCRC(name);
        } else if ( libGAP_SyCheckCRCCompiledModule ) {
            if ( crc_gap != libGAP_SyGAPCRC(name) ) {
                return 4;
            }
        }
    }


    /* now decide what to do                                               */
    if ( found_gap && found_dyn && crc_gap != crc_dyn ) {
        libGAP_Pr("#W Dynamic module %s has CRC mismatch, ignoring\n", (libGAP_Int) filename, 0);
        found_dyn = 0;
    }
    if ( found_gap && found_sta && crc_gap != crc_sta ) {
        libGAP_Pr("#W Static module %s has CRC mismatch, ignoring\n", (libGAP_Int) filename, 0);
        found_sta = 0;
    }
    if ( found_gap && found_sta ) {
        result->module_info = info_sta;
        return 2;
    }
    if ( found_gap && found_dyn ) {
        *(libGAP_StructInitInfo**)result = info_dyn;
        return 1;
    }
    if ( found_gap ) {
        return 3;
    }
    if ( found_sta ) {
        result->module_info = info_sta;
        return 2;
    }
    if ( found_dyn ) {
        result->module_info = info_dyn;
        return 1;
    }
    return 0;
}


/****************************************************************************
**
*F  SyGAPCRC( <name> )  . . . . . . . . . . . . . . . . . . crc of a GAP file
**
**  This function should  be clever and handle  white spaces and comments but
**  one has to make certain that such characters are not ignored in strings.
**
**  This function *never* returns a 0 unless an error occurred.
*/
static libGAP_UInt4 libGAP_syCcitt32[ 256 ] =
{
0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L, 0xe0d5e91eL,
0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L, 0x90bf1d91L, 0x1db71064L,
0x6ab020f2L, 0xf3b97148L, 0x84be41deL, 0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L,
0x83d385c7L, 0x136c9856L, 0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL,
0x63066cd9L, 0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL, 0x35b5a8faL,
0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L, 0x45df5c75L, 0xdcd60dcfL,
0xabd13d59L, 0x26d930acL, 0x51de003aL, 0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L,
0x56b3c423L, 0xcfba9599L, 0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L,
0xb10be924L, 0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL, 0x9fbfe4a5L,
0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL, 0xe10e9818L, 0x7f6a0dbbL,
0x086d3d2dL, 0x91646c97L, 0xe6635c01L, 0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L,
0xf262004eL, 0x6c0695edL, 0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L,
0x12b7e950L, 0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L, 0x4adfa541L,
0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL, 0x346ed9fcL, 0xad678846L,
0xda60b8d0L, 0x44042d73L, 0x33031de5L, 0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL,
0x270241aaL, 0xbe0b1010L, 0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L,
0xce61e49fL, 0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L, 0x03b6e20cL,
0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L, 0x73dc1683L, 0xe3630b12L,
0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L, 0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L,
0x7d079eb1L, 0xf00f9344L, 0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL,
0x806567cbL, 0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L, 0xd6d6a3e8L,
0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L, 0xa6bc5767L, 0x3fb506ddL,
0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL, 0x36034af6L, 0x41047a60L, 0xdf60efc3L,
0xa867df55L, 0x316e8eefL, 0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L,
0x5268e236L, 0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L, 0x2cd99e8bL,
0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL, 0x026d930aL, 0x9c0906a9L,
0xeb0e363fL, 0x72076785L, 0x05005713L, 0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL,
0x0cb61b38L, 0x92d28e9bL, 0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L,
0xf1d4e242L, 0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL, 0x8f659effL,
0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L, 0xd70dd2eeL, 0x4e048354L,
0x3903b3c2L, 0xa7672661L, 0xd06016f7L, 0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL,
0xd9d65adcL, 0x40df0b66L, 0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL,
0x30b5ffe9L, 0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L, 0x5d681b02L,
0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL, 0x2d02ef8dL
};

libGAP_Int4 libGAP_SyGAPCRC( const libGAP_Char * name )
{
    libGAP_UInt4       crc;
    libGAP_UInt4       old;
    libGAP_UInt4       new;
    libGAP_Int4        ch;
    libGAP_Int         fid;
    libGAP_Int         seen_nl;
    libGAP_Char        buf[BUFSIZ];
    FILE        *f;

    /* the CRC of a non existing file is 0                                 */
    fid = libGAP_SyFopen( name, "r" );
    if ( fid == -1 ) {
        return 0;
    }

    /* read in the file byte by byte and compute the CRC                   */
    crc = 0x12345678L;
    seen_nl = 0;

    /* Here it is both safe and sensible to use buffered IO */
    f = fdopen(libGAP_syBuf[fid].fp, "r");
    setbuf(f, buf);

    while ( (ch =  fgetc(f) )!= EOF ) {
        if ( ch == '\377' || ch == '\n' || ch == '\r' )
            ch = '\n';
        if ( ch == '\n' ) {
            if ( seen_nl )
                continue;
            else
                seen_nl = 1;
        }
        else
            seen_nl = 0;
        old = (crc >> 8) & 0x00FFFFFFL;
        new = libGAP_syCcitt32[ ( (libGAP_UInt4)( crc ^ ch ) ) & 0xff ];
        crc = old ^ new;
    }
    if ( crc == 0 ) {
        crc = 1;
    }

    /* and close it again                                                  */
    libGAP_SyFclose( fid );
    fclose(f);
    /* Emulate a signed shift: */
    if (crc & 0x80000000L)
        return (libGAP_Int4) ((crc >> 4) | 0xF0000000L);
    else
        return (libGAP_Int4) (crc >> 4);
}


/*
<#GAPDoc Label="CrcString">
<ManSection>
<Func Name="CrcString" Arg='str'/>
<Returns>an integer</Returns>

<Description>
This function computes a cyclic redundancy check number from a string
<A>str</A>. See also <Ref Func="CrcFile"/>.
<Example>
gap> CrcString("GAP example string");
-50451670
</Example>
</Description>
</ManSection>

<#/GAPDoc>
*/

/* And here we include a variant working on a GAP string */
libGAP_Obj libGAP_FuncCrcString( libGAP_Obj self, libGAP_Obj str ) {
    libGAP_UInt4       crc;
    libGAP_UInt4       old;
    libGAP_UInt4       new;
    libGAP_UInt4       i, len;
    libGAP_Char        *ptr;
    libGAP_Int4        ch;
    libGAP_Int         seen_nl;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( str ) ) {
        str = libGAP_ErrorReturnObj(
            "<str> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(str), 0L,
            "you can replace <filename> via 'return <str>;'" );
    }

    ptr = libGAP_CSTR_STRING(str);
    len = libGAP_GET_LEN_STRING(str);
    crc = 0x12345678L;
    seen_nl = 0;
    for (i = 0; i < len; i++) {
        ch = (libGAP_Int4)(ptr[i]);
        if ( ch == '\377' || ch == '\n' || ch == '\r' )
            ch = '\n';
        if ( ch == '\n' ) {
            if ( seen_nl )
                continue;
            else
                seen_nl = 1;
        }
        else
            seen_nl = 0;
        old = (crc >> 8) & 0x00FFFFFFL;
        new = libGAP_syCcitt32[ ( (libGAP_UInt4)( crc ^ ch ) ) & 0xff ];
        crc = old ^ new;
    }
    if ( crc == 0 ) {
        crc = 1;
    }
    return libGAP_INTOBJ_INT(((libGAP_Int4) crc) >> 4);
}


/****************************************************************************
**
*F  SyLoadModule( <name> )  . . . . . . . . . . . . link a module dynamically
*/

/* some compiles define symbols beginning with an underscore               */
/* but Mac OSX's dlopen adds one in for free!                              */
#if C_UNDERSCORE_SYMBOLS
#if defined(SYS_IS_DARWIN) && SYS_IS_DARWIN
# define libGAP_SYS_INIT_DYNAMIC       "Init__Dynamic"
#else
#if defined(SYS_IS_CYGWIN32) && SYS_IS_CYGWIN32
# define libGAP_SYS_INIT_DYNAMIC       "Init__Dynamic"
#else
# define libGAP_SYS_INIT_DYNAMIC       "_Init__Dynamic"
#endif
#endif
#else
# define libGAP_SYS_INIT_DYNAMIC       "Init__Dynamic"
#endif

/****************************************************************************
**
*f  SyLoadModule( <name> )  . . . . . . . . . . . . . . . . . . . . .  dlopen
*/
#if HAVE_DLOPEN

#include <dlfcn.h>

#ifndef RTLD_LAZY
#define RTLD_LAZY               1
#endif

libGAP_InitInfoFunc libGAP_SyLoadModule ( const libGAP_Char * name )
{
    void *          init;
    void *          handle;

    handle = dlopen( name, RTLD_LAZY | RTLD_GLOBAL);
#if 0
    if ( handle == 0 )  return (libGAP_InitInfoFunc) 1;
#else
    if ( handle == 0 ) {
      libGAP_Pr("#W dlopen() error: %s\n", (long) dlerror(), 0L);
      return (libGAP_InitInfoFunc) 1;
    }
#endif

    init = dlsym( handle, libGAP_SYS_INIT_DYNAMIC );
    if ( init == 0 )  return (libGAP_InitInfoFunc) 3;

    return (libGAP_InitInfoFunc) init;
}

#endif


/****************************************************************************
**
*f  SyLoadModule( <name> )  . . . . . . . . . . . . . . . . . . . .  rld_load
*/
#if HAVE_RLD_LOAD

#include <mach-o/rld.h>

libGAP_InitInfoFunc libGAP_SyLoadModule ( const libGAP_Char * name )
{
    const libGAP_Char *    names[2];
    unsigned long   init;

    names[0] = name;
    names[1] = 0;
    if ( rld_load( 0, 0,  names, 0 ) == 0 ) {
        return (libGAP_InitInfoFunc) 1;
    }
    if ( rld_lookup( 0, libGAP_SYS_INIT_DYNAMIC, &init ) == 0 ) {
        return (libGAP_InitInfoFunc) 3;
    }
    if ( rld_forget_symbol( 0, libGAP_SYS_INIT_DYNAMIC ) == 0 ) {
        return (libGAP_InitInfoFunc) 5;
    }
    return (libGAP_InitInfoFunc) init;
}

#endif


/****************************************************************************
**
*f  SyLoadModule( <name> )  . . . . . . . . . . . . . . . . . . .  no support
*/
#if !HAVE_DLOPEN && !HAVE_RLD_LOAD

libGAP_InitInfoFunc libGAP_SyLoadModule ( const libGAP_Char * name )
{
    return (libGAP_InitInfoFunc) 7;
}

#endif


/****************************************************************************
**

*F * * * * * * * * * * * * * * * window handler * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  IS_SEP( <C> ) . . . . . . . . . . . . . . . . . . . .  is <C> a separator
*/
#define libGAP_IS_SEP(C)       (!libGAP_IsAlpha(C) && !libGAP_IsDigit(C) && (C)!='_')


/****************************************************************************
**
*F  CTR( <V> )  . . . . . . . . . . . . . . . .  convert <V> into control-<V>
*/
#define libGAP_CTR(C)          ((C) & 0x1F)    /* <ctr> character                 */


/****************************************************************************
**
*F  Esc( <V> )  . . . . . . . . . . . . . . . . . convert <V> into escape-<V>
*/
#define libGAP_Esc(C)          ((C) | 0x100)   /* <esc> character                 */


/****************************************************************************
**
*F  CTV( <V> )  . . . . . . . . . . . . . . . . .  convert <V> into quote <V>
*/
#define libGAP_CTV(C)          ((C) | 0x200)   /* <ctr>V quotes characters        */


/****************************************************************************
**

*F  syWinPut( <fid>, <cmd>, <str> ) . . . . send a line to the window handler
**
**  'syWinPut'  send the command   <cmd> and the  string  <str> to the window
**  handler associated with the  file identifier <fid>.   In the string <str>
**  '@'  characters are duplicated, and   control characters are converted to
**  '@<chr>', e.g., <newline> is converted to '@J'.
*/
void libGAP_syWinPut (
    libGAP_Int                 fid,
    const libGAP_Char *        cmd,
    const libGAP_Char *        str )
{
    libGAP_Int                 fd;             /* file descriptor                 */
    libGAP_Char                tmp [130];      /* temporary buffer                */
    const libGAP_Char *        s;              /* pointer into the string         */
    libGAP_Char *              t;              /* pointer into the temporary      */

    /* if not running under a window handler, don't do anything            */
    if ( ! libGAP_SyWindow || 4 <= fid )
        return;

    /* get the file descriptor                                             */
    if ( fid == 0 || fid == 2 )  fd = libGAP_syBuf[fid].echo;
    else                         fd = libGAP_syBuf[fid].fp;

    /* print the cmd                                                       */
    libGAP_writeandcheck( fd, cmd, strlen(cmd) );

    /* print the output line, duplicate '@' and handle <ctr>-<chr>         */
    s = str;  t = tmp;
    while ( *s != '\0' ) {
        if ( *s == '@' ) {
            *t++ = '@';  *t++ = *s++;
        }
        else if ( libGAP_CTR('A') <= *s && *s <= libGAP_CTR('Z') ) {
            *t++ = '@';  *t++ = *s++ - libGAP_CTR('A') + 'A';
        }
        else {
            *t++ = *s++;
        }
        if ( 128 <= t-tmp ) {
            libGAP_writeandcheck( fd, tmp, t-tmp );
            t = tmp;
        }
    }
    if ( 0 < t-tmp ) {
        libGAP_writeandcheck( fd, tmp, t-tmp );
    }
}


/****************************************************************************
**
*F  SyWinCmd( <str>, <len> )  . . . . . . . . . . . . .  execute a window cmd
**
**  'SyWinCmd' send   the  command <str> to  the   window  handler (<len>  is
**  ignored).  In the string <str> '@' characters are duplicated, and control
**  characters  are converted to  '@<chr>', e.g.,  <newline> is converted  to
**  '@J'.  Then  'SyWinCmd' waits for  the window handlers answer and returns
**  that string.
*/
libGAP_Char libGAP_WinCmdBuffer [8000];

libGAP_Char * libGAP_SyWinCmd (
    const libGAP_Char *        str,
    libGAP_UInt                len )
{
    libGAP_Char                buf [130];      /* temporary buffer                */
    const libGAP_Char *        s;              /* pointer into the string         */
    const libGAP_Char *        bb;             /* pointer into the temporary      */
    libGAP_Char *              b;              /* pointer into the temporary      */
    libGAP_UInt                i;              /* loop variable                   */
#if SYS_IS_CYGWIN32
    libGAP_UInt                len1;           /* temporary storage for len       */
#endif

    /* if not running under a window handler, don't do nothing             */
    if ( ! libGAP_SyWindow )
        return "I1+S52+No Window Handler Present";

    /* compute the length of the (expanded) string (and ignore argument)   */
    len = 0;
    for ( s = str; *s != '\0'; s++ )
        len += 1 + (*s == '@' || (libGAP_CTR('A') <= *s && *s <= libGAP_CTR('Z')));

    /* send the length to the window handler                               */
    b = buf;
    for ( ; 0 < len;  len /= 10 ) {
        *b++ = (len % 10) + '0';
    }
    *b++ = '+';
    *b++ = '\0';
    libGAP_syWinPut( 1, "@w", buf );

    /* send the string to the window handler                               */
    libGAP_syWinPut( 1, "", str );

    /* read the length of the answer                                       */
    b = libGAP_WinCmdBuffer;
    i = 3;
    while ( 0 < i ) {
        len = read( 0, b, i );
        i  -= len;
        b  += len;
    }
    if ( libGAP_WinCmdBuffer[0] != '@' || libGAP_WinCmdBuffer[1] != 'a' )
        return "I1+S41+Illegal Answer";
    b = libGAP_WinCmdBuffer+2;
    for ( i=1,len=0; '0' <= *b && *b <= '9';  i *= 10 ) {
        len += (*b-'0')*i;
        while ( read( 0, b, 1 ) != 1 )  ;
    }

    /* read the arguments of the answer                                    */
    b = libGAP_WinCmdBuffer;
    i = len;
#if SYS_IS_CYGWIN32
    len1 = len;
    while ( 0 < i ) {
        len = read( 0, b, i );
        b += len;
        i  -= len;
        s  += len;
    }
    len = len1;
#else
    while ( 0 < i ) {
        len = read( 0, b, i );
        i  -= len;
        s  += len;
    }
#endif

    /* shrink '@@' into '@'                                                */
    for ( bb = b = libGAP_WinCmdBuffer;  0 < len;  len-- ) {
        if ( *bb == '@' ) {
            bb++;
            if ( *bb == '@' )
                *b++ = '@';
            else if ( 'A' <= *bb && *bb <= 'Z' )
                *b++ = libGAP_CTR(*bb);
            bb++;
        }
        else {
            *b++ = *bb++;
        }
    }
    *b = 0;

    /* return the string                                                   */
    return libGAP_WinCmdBuffer;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * * * open/close * * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  syBuf . . . . . . . . . . . . . .  buffer and other info for files, local
**
**  'syBuf' is  a array used as  buffers for  file I/O to   prevent the C I/O
**  routines  from   allocating their  buffers  using  'malloc',  which would
**  otherwise confuse Gasman.
**
**
**  Actually these days SyBuf just stores various file info. SyBuffers
**  stores buffers for the relatively few files that need them.
*/

libGAP_SYS_SY_BUF libGAP_syBuf [256];

libGAP_SYS_SY_BUFFER libGAP_syBuffers [ 32];


/****************************************************************************
**
*F  SyFopen( <name>, <mode> ) . . . . . . . .  open the file with name <name>
**
**  The function 'SyFopen'  is called to open the file with the name  <name>.
**  If <mode> is "r" it is opened for reading, in this case  it  must  exist.
**  If <mode> is "w" it is opened for writing, it is created  if  neccessary.
**  If <mode> is "a" it is opened for appending, i.e., it is  not  truncated.
**
**  'SyFopen' returns an integer used by the scanner to  identify  the  file.
**  'SyFopen' returns -1 if it cannot open the file.
**
**  The following standard files names and file identifiers  are  guaranteed:
**  'SyFopen( "*stdin*", "r")' returns 0 identifying the standard input file.
**  'SyFopen( "*stdout*","w")' returns 1 identifying the standard outpt file.
**  'SyFopen( "*errin*", "r")' returns 2 identifying the brk loop input file.
**  'SyFopen( "*errout*","w")' returns 3 identifying the error messages file.
**
**  If it is necessary  to adjust the filename  this should be done here, the
**  filename convention used in GAP is that '/' is the directory separator.
**
**  Right now GAP does not read nonascii files, but if this changes sometimes
**  'SyFopen' must adjust the mode argument to open the file in binary mode.
*/

libGAP_Int libGAP_SyFopen (
    const libGAP_Char *        name,
    const libGAP_Char *        mode )
{
    libGAP_Int                 fid;
    libGAP_Char                namegz [1024];
    libGAP_Char                cmd [1024];
    int                 flags = 0;

    /* handle standard files                                               */
    if ( strcmp( name, "*stdin*" ) == 0 ) {
        if ( strcmp( mode, "r" ) != 0 )
          return -1;
        else
          return 0;
    }
    else if ( strcmp( name, "*stdout*" ) == 0 ) {
        if ( strcmp( mode, "w" ) != 0 && strcmp( mode, "a" ) != 0 )
          return -1;
        else
          return 1;
    }
    else if ( strcmp( name, "*errin*" ) == 0 ) {
        if ( strcmp( mode, "r" ) != 0 )
          return -1;
        else if ( libGAP_syBuf[2].fp == -1 )
          return -1;
        else
          return 2;
    }
    else if ( strcmp( name, "*errout*" ) == 0 ) {
        if ( strcmp( mode, "w" ) != 0 && strcmp( mode, "a" ) != 0 )
          return -1;
        else
          return 3;
    }

    libGAP_HashLock(&libGAP_syBuf);
    /* try to find an unused file identifier                               */
    for ( fid = 4; fid < sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]); ++fid )
        if ( libGAP_syBuf[fid].fp == -1 )
          break;
    
    if ( fid == sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) ) {
        libGAP_HashUnlock(&libGAP_syBuf);
        return (libGAP_Int)-1;
    }

    /* set up <namegz> and <cmd> for pipe command                          */
    namegz[0] = '\0';
    if (strlen(name) <= 1018) {
      libGAP_strxcpy( namegz, name, sizeof(namegz) );
      libGAP_strxcat( namegz, ".gz", sizeof(namegz) );

      libGAP_strxcpy( cmd, "gunzip <", sizeof(cmd) );
      libGAP_strxcat( cmd, namegz, sizeof(cmd) );
    }
    if (strncmp( mode, "r", 1 ) == 0)
      flags = O_RDONLY;
    else if (strncmp( mode, "w",1 ) == 0)
      flags = O_WRONLY | O_CREAT | O_TRUNC;
    else if (strncmp( mode, "a",1) == 0)
      flags = O_WRONLY | O_APPEND | O_CREAT;
    else
      {
        libGAP_Pr("Panic: Unknown mode %s\n",(libGAP_Int) mode, 0);
        libGAP_SyExit(2);
      }

#if SYS_IS_CYGWIN32
    if(strlen(mode) >= 2 && mode[1] == 'b')
       flags |= O_BINARY;
#endif
    /* try to open the file                                                */
    if ( 0 <= (libGAP_syBuf[fid].fp = open(name,flags, 0644)) ) {
        libGAP_syBuf[fid].pipe = 0;
        libGAP_syBuf[fid].echo = libGAP_syBuf[fid].fp;
        libGAP_syBuf[fid].ateof = 0;
        libGAP_syBuf[fid].crlast = 0;
        libGAP_syBuf[fid].bufno = -1;
        libGAP_syBuf[fid].isTTY = 0;
    }
#if HAVE_POPEN
   else if ( strncmp(mode,"r",1) == 0
           && libGAP_SyIsReadableFile(namegz) == 0
             && ( (libGAP_syBuf[fid].pipehandle = popen(cmd,"r"))
               ) ) {
        libGAP_syBuf[fid].pipe = 1;
        libGAP_syBuf[fid].fp = fileno(libGAP_syBuf[fid].pipehandle);
        libGAP_syBuf[fid].ateof = 0;
        libGAP_syBuf[fid].crlast = 0;
        libGAP_syBuf[fid].bufno = -1;
        libGAP_syBuf[fid].isTTY = 0;
    }
#endif
    else {
        libGAP_HashUnlock(&libGAP_syBuf);
        return (libGAP_Int)-1;
    }

    libGAP_HashUnlock(&libGAP_syBuf);

    if(strncmp(mode, "r", 1) == 0)
        libGAP_SySetBuffering(fid);

    /* return file identifier                                              */
    return fid;
}

// Lock on SyBuf for both SyBuf and SyBuffers

libGAP_UInt libGAP_SySetBuffering( libGAP_UInt fid )
{
  libGAP_UInt bufno;

  if (libGAP_syBuf[fid].fp == -1)
    libGAP_ErrorQuit("Can't set buffering for a closed stream", 0, 0);
  if (libGAP_syBuf[fid].bufno >= 0)
    return 1;

  bufno = 0;
  libGAP_HashLock(&libGAP_syBuf);
  while (bufno < sizeof(libGAP_syBuffers)/sizeof(libGAP_syBuffers[0]) &&
         libGAP_syBuffers[bufno].inuse != 0)
    bufno++;
  if (bufno >= sizeof(libGAP_syBuffers)/sizeof(libGAP_syBuffers[0])) {
      libGAP_HashUnlock(&libGAP_syBuf);
      return 0;
  }
  libGAP_syBuf[fid].bufno = bufno;
  libGAP_syBuffers[bufno].inuse = 1;
  libGAP_syBuffers[bufno].bufstart = 0;
  libGAP_syBuffers[bufno].buflen = 0;
  libGAP_HashUnlock(&libGAP_syBuf);
  return 1;
}

/****************************************************************************
**
*F  SyFclose( <fid> ) . . . . . . . . . . . . . . . . .  close the file <fid>
**
**  'SyFclose' closes the file with the identifier <fid>  which  is  obtained
**  from 'SyFopen'.
*/
libGAP_Int libGAP_SyFclose (
    libGAP_Int                 fid )
{
    /* check file identifier                                               */
    if ( sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) <= fid || fid < 0 ) {
        fputs("gap: panic 'SyFclose' asked to close illegal fid!\n",stderr);
        return -1;
    }
    if ( libGAP_syBuf[fid].fp == -1 ) {
        fputs("gap: panic 'SyFclose' asked to close closed file!\n",stderr);
        return -1;
    }

    /* refuse to close the standard files                                  */
    if ( fid == 0 || fid == 1 || fid == 2 || fid == 3 ) {
        return -1;
    }
    libGAP_HashLock(&libGAP_syBuf);
    /* try to close the file                                               */
    if ( (libGAP_syBuf[fid].pipe == 0 && close( libGAP_syBuf[fid].fp ) == EOF)
      || (libGAP_syBuf[fid].pipe == 1 && pclose( libGAP_syBuf[fid].pipehandle ) == -1
#ifdef ECHILD
          && errno != ECHILD
#endif
          ) )
    {
        fputs("gap: 'SyFclose' cannot close file, ",stderr);
        fputs("maybe your file system is full?\n",stderr);
        libGAP_syBuf[fid].fp = -1;
        libGAP_HashUnlock(&libGAP_syBuf);
        return -1;
    }

    /* mark the buffer as unused                                           */
    if (libGAP_syBuf[fid].bufno >= 0)
      libGAP_syBuffers[libGAP_syBuf[fid].bufno].inuse = 0;
    libGAP_syBuf[fid].fp = -1;
    libGAP_HashUnlock(&libGAP_syBuf);
    return 0;
}


/****************************************************************************
**
*F  SyIsEndOfFile( <fid> )  . . . . . . . . . . . . . . . end of file reached
*/
libGAP_Int libGAP_SyIsEndOfFile (
    libGAP_Int                 fid )
{
    /* check file identifier                                               */
    if ( sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) <= fid || fid < 0 ) {
        return -1;
    }
    if ( libGAP_syBuf[fid].fp == -1 ) {
        return -1;
    }

    /* *stdin* and *errin* are never at end of file                        */
    if ( fid < 4 )
        return 0;

    /* How to detect end of file ?? */

    return libGAP_syBuf[fid].ateof;
    /* return feof(syBuf[fid].fp);*/
}


/****************************************************************************
**
*F  syStartraw( <fid> ) . . . . . . start raw mode on input file <fid>, local
**
**  The  following four  functions are  the  actual system  dependent part of
**  'SyFgets'.
**
**  'syStartraw' tries to put the file with the file  identifier  <fid>  into
**  raw mode.  I.e.,  disabling  echo  and  any  buffering.  It also finds  a
**  place to put the echoing  for  'syEchoch'.  If  'syStartraw'  succedes it
**  returns 1, otherwise, e.g., if the <fid> is not a terminal, it returns 0.
**
**  'syStopraw' stops the raw mode for the file  <fid>  again,  switching  it
**  back into whatever mode the terminal had before 'syStartraw'.
**
**  'syGetch' reads one character from the file <fid>, which must  have  been
**  turned into raw mode before, and returns it.
**
**  'syEchoch' puts the character <ch> to the file opened by 'syStartraw' for
**  echoing.  Note that if the user redirected 'stdout' but not 'stdin',  the
**  echo for 'stdin' must go to 'ttyname(fileno(stdin))' instead of 'stdout'.
*/

extern libGAP_UInt libGAP_syStartraw (
            libGAP_Int                 fid );

extern void libGAP_syStopraw (
            libGAP_Int                 fid );

#if HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H


#if HAVE_TERMIOS_H

  /****************************************************************************
  **  For UNIX System V, input/output redirection and typeahead are  supported.
  **  We  turn off input buffering  and canonical input editing and  also echo.
  **  Because we leave the signals enabled  we  have  to disable the characters
  **  for interrupt and quit, which are usually set to '<ctr>-C' and '<ctr>-B'.
  **  We   also turn off the  xon/xoff  start  and  stop  characters, which are
  **  usually set to  '<ctr>-S'  and '<ctr>-Q' so we  can get those characters.
  **  We do  not turn of  signals  'ISIG' because  we want   to catch  stop and
  **  continue signals if this particular version  of UNIX supports them, so we
  **  can turn the terminal line back to cooked mode before stopping GAP.
  */
  #include <termios.h>
  struct termios   libGAP_syOld, libGAP_syNew;           /* old and new terminal state      */

#elif HAVE_TERMIO_H

  #include       <termio.h>
  struct termio   libGAP_syOld, libGAP_syNew;           /* old and new terminal state      */

  #ifndef TCSETAW
    /* cygwin32 provides no SETAW. Try using SETA instead in that case */
    #define TCSETAW TCSETA
  #endif

#elif HAVE_SGTTY_H

  /****************************************************************************
  **  For Berkeley UNIX, input/output redirection and typeahead are  supported.
  **  We switch the terminal line into 'CBREAK' mode and also disable the echo.
  **  We do not switch to 'RAW'  mode because  this would flush  all typeahead.
  **  Because 'CBREAK' leaves signals enabled we have to disable the characters
  **  for interrupt and quit, which are usually set to '<ctr>-C' and '<ctr>-B'.
  **  We also turn  off  the  xon/xoff  start and  stop characters,  which  are
  **  usually set  to '<ctr>-S' and '<ctr>-Q' so  we can get  those characters.
  **  We  do not  change the  suspend  character, which  is usually  '<ctr>-Z',
  **  instead we catch the signal, so that we  can turn  the terminal line back
  **  to cooked mode before stopping GAP and back to raw mode when continuing.
  */

  #include       <sgtty.h>
  struct sgttyb   libGAP_syOld, libGAP_syNew;           /* old and new terminal state      */
  struct libGAP_tchars   libGAP_syOldT, libGAP_syNewT;         /* old and new special characters  */

#endif


#ifdef SIGTSTP

libGAP_Int libGAP_syFid;

void libGAP_syAnswerCont ( int signr )
{
#ifdef LIBGAP_SIGNALS
    libGAP_syStartraw( libGAP_syFid );
    signal( SIGCONT, SIG_DFL );
    kill( getpid(), SIGCONT );
#endif
}

void libGAP_syAnswerTstp ( int signr )
{
#ifdef LIBGAP_SIGNALS
    libGAP_syStopraw( libGAP_syFid );
    signal( SIGCONT, libGAP_syAnswerCont );
    kill( getpid(), SIGTSTP );
#endif
}

#endif

libGAP_UInt libGAP_syStartraw ( libGAP_Int fid )
{
    /* if running under a window handler, tell it that we want to read     */
    if ( libGAP_SyWindow ) {
        if      ( fid == 0 ) { libGAP_syWinPut( fid, "@i", "" );  return 1; }
        else if ( fid == 2 ) { libGAP_syWinPut( fid, "@e", "" );  return 1; }
        else {                                             return 0; }
    }

#if HAVE_TERMIOS_H || HAVE_TERMIO_H

    /* try to get the terminal attributes, will fail if not terminal       */
#if HAVE_TERMIOS_H
    if ( tcgetattr( libGAP_syBuf[fid].fp, &libGAP_syOld) == -1 )
        return 0;
#elif HAVE_TERMIO_H
    if ( ioctl( libGAP_syBuf[fid].fp, TCGETA, &libGAP_syOld ) == -1 )
        return 0;
#endif

    /* disable interrupt, quit, start and stop output characters           */
    libGAP_syNew = libGAP_syOld;
    libGAP_syNew.c_cc[VINTR] = 0377;
    libGAP_syNew.c_cc[VQUIT] = 0377;
    /*C 27-Nov-90 martin changing '<ctr>S' and '<ctr>Q' does not work      */
    /*C syNew.c_iflag    &= ~(IXON|INLCR|ICRNL);                           */
    libGAP_syNew.c_iflag    &= ~(INLCR|ICRNL);

    /* disable input buffering, line editing and echo                      */
    libGAP_syNew.c_cc[VMIN]  = 1;
    libGAP_syNew.c_cc[VTIME] = 0;
    libGAP_syNew.c_lflag    &= ~(ECHO|ICANON);

#if HAVE_TERMIOS_H
    if ( tcsetattr( libGAP_syBuf[fid].fp, TCSANOW, &libGAP_syNew) == -1 )
        return 0;
#elif HAVE_TERMIO_H
    if ( ioctl( libGAP_syBuf[fid].fp, TCSETAW, &libGAP_syNew ) == -1 )
        return 0;
#endif

#elif HAVE_SGTTY_H

    /* try to get the terminal attributes, will fail if not terminal       */
    if ( ioctl( libGAP_syBuf[fid].fp, TIOCGETP, (char*)&libGAP_syOld ) == -1 )
        return 0;

    /* disable interrupt, quit, start and stop output characters           */
    if ( ioctl( libGAP_syBuf[fid].fp, TIOCGETC, (char*)&libGAP_syOldT ) == -1 )
        return 0;
    libGAP_syNewT = libGAP_syOldT;
    libGAP_syNewT.t_intrc  = -1;
    libGAP_syNewT.t_quitc  = -1;
    /*C 27-Nov-90 martin changing '<ctr>S' and '<ctr>Q' does not work      */
    /*C syNewT.t_startc = -1;                                              */
    /*C syNewT.t_stopc  = -1;                                              */
    if ( ioctl( libGAP_syBuf[fid].fp, TIOCSETC, (char*)&libGAP_syNewT ) == -1 )
        return 0;

    /* disable input buffering, line editing and echo                      */
    libGAP_syNew = libGAP_syOld;
    libGAP_syNew.sg_flags |= CBREAK;
    libGAP_syNew.sg_flags &= ~ECHO;
    if ( ioctl( libGAP_syBuf[fid].fp, TIOCSETN, (char*)&libGAP_syNew ) == -1 )
        return 0;

#endif

#ifdef LIBGAP_SIGNALS
#ifdef SIGTSTP
    /* install signal handler for stop                                     */
    libGAP_syFid = fid;
    signal( SIGTSTP, libGAP_syAnswerTstp );
#endif
#endif

    /* indicate success                                                    */
    return 1;
}


#endif /* HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H */


/****************************************************************************
**
*F  syStopraw( <fid> )  . . . . . .  stop raw mode on input file <fid>, local
*/
#if HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H

void libGAP_syStopraw (
    libGAP_Int                 fid )
{
    /* if running under a window handler, don't do nothing                 */
    if ( libGAP_SyWindow )
        return;

#ifdef LIBGAP_SIGNALS
#ifdef SIGTSTP
    /* remove signal handler for stop                                      */
    signal( SIGTSTP, SIG_DFL );
#endif
#endif

#if HAVE_TERMIOS_H

    /* enable input buffering, line editing and echo again                 */
    if (tcsetattr(libGAP_syBuf[fid].fp, TCSANOW, &libGAP_syOld) == -1)
        fputs("gap: 'ioctl' could not turn off raw mode!\n",stderr);

#elif HAVE_TERMIO_H

    /* enable input buffering, line editing and echo again                 */
    if ( ioctl( libGAP_syBuf[fid].fp, TCSETAW, &libGAP_syOld ) == -1 )
        fputs("gap: 'ioctl' could not turn off raw mode!\n",stderr);

#elif HAVE_SGTTY_H

    /* enable input buffering, line editing and echo again                 */
    if ( ioctl( libGAP_syBuf[fid].fp, TIOCSETN, (char*)&libGAP_syOld ) == -1 )
        fputs("gap: 'ioctl' could not turn off raw mode!\n",stderr);

    /* enable interrupt, quit, start and stop output characters again      */
    if ( ioctl( libGAP_syBuf[fid].fp, TIOCSETC, (char*)&libGAP_syOldT ) == -1 )
        fputs("gap: 'ioctl' could not turn off raw mode!\n",stderr);

#endif
}

#endif /* HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H */



/****************************************************************************
**

*F  SyIsIntr()  . . . . . . . . . . . . . . . . check wether user hit <ctr>-C
**
**  'SyIsIntr' is called from the evaluator at  regular  intervals  to  check
**  wether the user hit '<ctr>-C' to interrupt a computation.
**
**  'SyIsIntr' returns 1 if the user typed '<ctr>-C' and 0 otherwise.
*/


/****************************************************************************
**
*f  SyIsIntr()
**
**  For  UNIX  we  install 'syAnswerIntr' to  answer interrupt 'SIGINT'. If
**  two interrupts  occur within 1 second 'syAnswerIntr' exits GAP.
*/
#if HAVE_SIGNAL 


libGAP_UInt            libGAP_syLastIntr;             /* time of the last interrupt      */


#if HAVE_LIBREADLINE
libGAP_Int libGAP_doingReadline;
#endif

void libGAP_syAnswerIntr ( int signr )
{
    libGAP_UInt                nowIntr;

#if HAVE_LIBREADLINE
    /* ignore during readline */
    if (libGAP_doingReadline) return;
#endif

    /* get the current wall clock time                                     */
    nowIntr = time(0);

    /* if the last '<ctr>-C' was less than a second ago, exit GAP          */
    if ( libGAP_syLastIntr && nowIntr-libGAP_syLastIntr < 1 ) {
        fputs("gap: you hit '<ctr>-C' twice in a second, goodbye.\n",stderr);
        libGAP_SyExit( 1 );
    }

    /* reinstall 'syAnswerIntr' as signal handler                          */
    signal( SIGINT, libGAP_syAnswerIntr );
    siginterrupt( SIGINT, 0 );

    /* remember time of this interrupt                                     */
    libGAP_syLastIntr = nowIntr;

#if HAVE_SIGNAL
    /* interrupt the executor                                              */
    libGAP_InterruptExecStat();
#endif
}


void libGAP_SyInstallAnswerIntr ( void )
{
#ifdef LIBGAP_SIGNALS
    if ( signal( SIGINT, SIG_IGN ) != SIG_IGN )
    {
        signal( SIGINT, libGAP_syAnswerIntr );
        siginterrupt( SIGINT, 0 );
    }
#endif
}


libGAP_UInt libGAP_SyIsIntr ( void )
{
    libGAP_UInt                isIntr;

    isIntr = (libGAP_syLastIntr != 0);
    libGAP_syLastIntr = 0;
    return isIntr;
}


/* Code for Timeouts */

volatile int libGAP_SyAlarmRunning = 0;
volatile int libGAP_SyAlarmHasGoneOff = 0;

#endif

#if HAVE_TIMER_CREATE && HAVE_SIGACTION

/* Could live without sigaction but it seems to be pretty universal */

/* This uses the POSIX 2001 API
   which allows per-thread timing and minimises risk of
   interference with other code using timers.

   Sadly it's not always available, so we have an alternative implementation
   below using the odler setitimer interface */

/* Handler for the Alarm signal */

int libGAP_SyHaveAlarms = 1;

/* This API lets us pick wich signal to use */
#define libGAP_TIMER_SIGNAL SIGVTALRM


/* For now anyway we create one timer at initialisation and use it */
static libGAP_timer_t libGAP_syTimer = 0;

#if SYS_IS_CYGWIN32
#define libGAP_MY_CLOCK CLOCK_REALTIME
#else
#define libGAP_MY_CLOCK CLOCK_THREAD_CPUTIME_ID
#endif

static void libGAP_SyInitAlarm( void ) {
/* Create the CPU timer used for timeouts */
  struct sigevent se;
  se.sigev_notify = SIGEV_SIGNAL;
  se.sigev_signo = libGAP_TIMER_SIGNAL;
  se.sigev_value.sival_int = 0x12345678;
  if (timer_create( libGAP_MY_CLOCK, &se, &libGAP_syTimer)) {
    libGAP_Pr("#E  Could not create interval timer. Timeouts will not be supported\n",0L,0L);
    libGAP_SyHaveAlarms = 0;
  }
}

static void libGAP_syAnswerAlarm ( int signr, siginfo_t * si, void *context)
{
    /* interrupt the executor                                             
       Later we might want to do something cleverer with throwing an 
       exception or dealing better if this isn't our timer     */
  assert( signr == libGAP_TIMER_SIGNAL);
  assert( si->si_signo == libGAP_TIMER_SIGNAL);
  assert( si->si_code == SI_TIMER);
  assert( si->si_value.sival_int == 0x12345678 );
  libGAP_SyAlarmRunning = 0;
  libGAP_SyAlarmHasGoneOff = 1;
  libGAP_InterruptExecStat();
}

 
void libGAP_SyInstallAlarm ( libGAP_UInt seconds, libGAP_UInt nanoseconds )
{
  struct sigaction sa;
  
  sa.sa_handler = NULL;
  sa.sa_sigaction = libGAP_syAnswerAlarm;
  sigemptyset(&(sa.sa_mask));
  sa.sa_flags = SA_RESETHAND | SA_SIGINFO | SA_RESTART;
  
  /* First install the handler */
  if (sigaction( libGAP_TIMER_SIGNAL, &sa, NULL ))
    {
      libGAP_ErrorReturnVoid("Could not set handler for alarm signal",0L,0L,"you can return to ignore");
      return;
    }

  
  struct itimerspec tv;
  tv.it_value.tv_sec = (time_t)seconds;
  tv.it_value.tv_nsec = (long)nanoseconds;
  tv.it_interval.tv_sec = (time_t)0;
  tv.it_interval.tv_nsec = 0L;
  
  libGAP_SyAlarmRunning = 1;
  libGAP_SyAlarmHasGoneOff = 0;
  if (timer_settime(libGAP_syTimer, 0, &tv, NULL)) {
    signal(libGAP_TIMER_SIGNAL, SIG_DFL);
    libGAP_ErrorReturnVoid("Could not set interval timer", 0L, 0L, "you can return to ignore");
  }
  return;
}

void libGAP_SyStopAlarm(libGAP_UInt *seconds, libGAP_UInt *nanoseconds) {
  struct itimerspec tv, buf;
  tv.it_value.tv_sec = (time_t)0;
  tv.it_value.tv_nsec = 0L;
  tv.it_interval.tv_sec = (time_t)0;
  tv.it_interval.tv_nsec = 0L;

  timer_settime(libGAP_syTimer, 0, &tv, &buf);
  libGAP_SyAlarmRunning = 0;
  signal(libGAP_TIMER_SIGNAL, SIG_IGN);

  if (seconds)
    *seconds = (libGAP_UInt)buf.it_value.tv_sec;
  if (nanoseconds)
    *nanoseconds = (libGAP_UInt)buf.it_value.tv_nsec;
  return;
}

#else
#if HAVE_SETITIMER && HAVE_SIGACTION

/* Using setitimer and getitimer from sys/time.h */
/* again sigaction could be replaced by signal if that was useful
 sigaction is just a bit more robust */

/* Handler for the Alarm signal */

int libGAP_SyHaveAlarms = 1;


static void libGAP_SyInitAlarm( void ) {
  /* No initialisation in this case */
  return; 
}

static void libGAP_syAnswerAlarm ( int signr, siginfo_t * si, void *context)
{
    /* interrupt the executor                                             
       Later we might want to do something cleverer with throwing an 
       exception or dealing better if this isn't our timer     */
  assert( signr == SIGVTALRM);
  assert( si->si_signo == SIGVTALRM);
  libGAP_SyAlarmRunning = 0;
  libGAP_SyAlarmHasGoneOff = 1;
  libGAP_InterruptExecStat();
}

 
void libGAP_SyInstallAlarm ( libGAP_UInt seconds, libGAP_UInt nanoseconds )
{
  struct sigaction sa;
  
  sa.sa_handler = NULL;
  sa.sa_sigaction = libGAP_syAnswerAlarm;
  sigemptyset(&(sa.sa_mask));
  sa.sa_flags = SA_RESETHAND | SA_SIGINFO | SA_RESTART;

  
  /* First install the handler */
  if (sigaction( SIGVTALRM, &sa, NULL ))
    {
      libGAP_ErrorReturnVoid("Could not set handler for alarm signal",0L,0L,"you can return to ignore");
      return;
    }

  
  struct itimerval tv;
  tv.it_value.tv_sec = (time_t)seconds;
  tv.it_value.tv_usec = (suseconds_t)(nanoseconds/1000);
  tv.it_interval.tv_sec = (time_t)0;
  tv.it_interval.tv_usec = (suseconds_t)0L;
  
  libGAP_SyAlarmRunning = 1;
  libGAP_SyAlarmHasGoneOff = 0;
  if (setitimer(ITIMER_VIRTUAL, &tv, NULL)) {
    signal(SIGVTALRM, SIG_IGN);
    libGAP_ErrorReturnVoid("Could not set interval timer", 0L, 0L, "you can return to ignore");
  }
  return;
}

void libGAP_SyStopAlarm(libGAP_UInt *seconds, libGAP_UInt *nanoseconds) {
  struct itimerval tv, buf;
  tv.it_value.tv_sec = (time_t)0;
  tv.it_value.tv_usec = (suseconds_t)0L;
  tv.it_interval.tv_sec = (time_t)0;
  tv.it_interval.tv_usec = (suseconds_t)0L;

  setitimer(ITIMER_VIRTUAL, &tv, &buf);
  libGAP_SyAlarmRunning = 0;
  signal(SIGVTALRM, SIG_IGN);

  if (seconds)
    *seconds = (libGAP_UInt)buf.it_value.tv_sec;
  if (nanoseconds)
    *nanoseconds = 1000*(libGAP_UInt)buf.it_value.tv_usec;
  return;
}

#else
int libGAP_SyHaveAlarms = 0;

/* stub implementations */

static void libGAP_SyInitAlarm( void ) {
  /* No initialisation in this case */
  return; 
}

 
void libGAP_SyInstallAlarm ( libGAP_UInt seconds, libGAP_UInt nanoseconds )
{
  assert(0);
  return;
}

void libGAP_SyStopAlarm(libGAP_UInt *seconds, libGAP_UInt *nanoseconds) {
  assert(0);
  return;
}

#endif
#endif


/****************************************************************************
 **
 *F  getwindowsize() . . . . . . . get screen size from termcap or TIOCGWINSZ
 **
 **  For UNIX  we  install 'syWindowChangeIntr' to answer 'SIGWINCH'.
 */

#if HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>             /* for TIOCGWINSZ */
#endif

#define libGAP_CO libGAP_SyNrCols
#define libGAP_LI libGAP_SyNrRows
/* SyNrCols can be at most MAXLENOUTPUTLINE */
#define libGAP_ML libGAP_MAXLENOUTPUTLINE

#ifdef TIOCGWINSZ
/* signal routine: window size changed */
void libGAP_syWindowChangeIntr ( int signr )
{
    struct winsize win;
    if(ioctl(0, TIOCGWINSZ, (char *) &win) >= 0) {
        if(!libGAP_SyNrRowsLocked && win.ws_row > 0)
            libGAP_LI = win.ws_row;
        if(!libGAP_SyNrColsLocked && win.ws_col > 0)
          libGAP_CO = win.ws_col - 1;        /* never trust last column */
        if (libGAP_CO < 20) libGAP_CO = 20;
        if (libGAP_CO > libGAP_ML) libGAP_CO = libGAP_ML;
    }
}

#endif /* TIOCGWINSZ */

void libGAP_getwindowsize( void )
{
/* it might be that LI, CO have been set by the user with -x, -y */
/* otherwise they are zero */

/* first strategy: try to ask the operating system */
#ifdef TIOCGWINSZ
    if (libGAP_LI <= 0 || libGAP_CO <= 0) {
        struct winsize win;

        if(ioctl(0, TIOCGWINSZ, (char *) &win) >= 0) {
            if (libGAP_LI <= 0)
                libGAP_LI = win.ws_row;
            if (libGAP_CO <= 0)
                libGAP_CO = win.ws_col;
        }
#ifdef LIBGAP_SIGNALS
        (void) signal(SIGWINCH, libGAP_syWindowChangeIntr);
#endif
    }
#endif /* TIOCGWINSZ */

#ifdef USE_TERMCAP
/* note that if we define TERMCAP, this has to be linked with -ltermcap */
/* maybe that is -ltermlib on some SYSV machines */
    if (libGAP_LI <= 0 || libGAP_CO <= 0) {
              /* this failed - next attempt: try to find info in TERMCAP */
        char *sp;
        char bp[1024];

        if ((sp = getenv("TERM")) != NULL && tgetent(bp,sp) == 1) {
            if(libGAP_LI <= 0)
                libGAP_LI = tgetnum("li");
            if(libGAP_CO <= 0)
                libGAP_CO = tgetnum("co");
        }
    }
#endif

    /* if nothing worked, use 80x24 */
    if (libGAP_CO <= 0)
        libGAP_CO = 80;
    if (libGAP_LI <= 0)
        libGAP_LI = 24;

    /* reset CO if value is strange */
    if (libGAP_CO < 20) libGAP_CO = 20;
    if (libGAP_CO > libGAP_ML) libGAP_CO = libGAP_ML;
}

#undef libGAP_CO
#undef libGAP_LI
#undef libGAP_ML


/****************************************************************************
**

*F * * * * * * * * * * * * * * * * * output * * * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  syEchoch( <ch>, <fid> ) . . . . . . . . . . . echo a char to <fid>, local
*/


/****************************************************************************
**
*f  syEchoch( <ch>, <fid> )
*/
#if HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H

void libGAP_syEchoch (
    libGAP_Int                 ch,
    libGAP_Int                 fid )
{
    libGAP_Char                ch2;

    /* write the character to the associate echo output device             */
    ch2 = ch;
    libGAP_writeandcheck( libGAP_syBuf[fid].echo, (char*)&ch2, 1 );

    /* if running under a window handler, duplicate '@'                    */
    if ( libGAP_SyWindow && ch == '@' ) {
        ch2 = ch;
        libGAP_writeandcheck( libGAP_syBuf[fid].echo, (char*)&ch2, 1 );
    }
}

#endif /* HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H */

/****************************************************************************
**
*F  SyEchoch( <ch>, <fid> ) . . . . . . . . . . . . .  echo a char from <fid>
*/
libGAP_Int libGAP_SyEchoch (
    libGAP_Int                 ch,
    libGAP_Int                 fid )
{
    /* check file identifier                                               */
    if ( sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) <= fid || fid < 0 ) {
        return -1;
    }
    if ( libGAP_syBuf[fid].fp == -1 ) {
        return -1;
    }
    libGAP_syEchoch(ch,fid);
    return 0;
}



/****************************************************************************
**
*F  syEchos( <ch>, <fid> )  . . . . . . . . . . . echo a char to <fid>, local
*/


/****************************************************************************
**
*f  syEchos( <ch>, <fid> )
*/
#if HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H

void libGAP_syEchos (
    const libGAP_Char *        str,
    libGAP_Int                 fid )
{
    /* if running under a window handler, send the line to it              */
    if ( libGAP_SyWindow && fid < 4 )
        libGAP_syWinPut( fid, (fid == 1 ? "@n" : "@f"), str );

    /* otherwise, write it to the associate echo output device             */
    else
        libGAP_writeandcheck( libGAP_syBuf[fid].echo, str, strlen(str) );
}

#endif /* HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H */


/****************************************************************************
**
*F  SyFputs( <line>, <fid> )  . . . . . . . .  write a line to the file <fid>
**
**  'SyFputs' is called to put the  <line>  to the file identified  by <fid>.
*/
libGAP_UInt   libGAP_syNrchar;                        /* nr of chars already on the line */
libGAP_Char   libGAP_syPrompt [256];                  /* characters already on the line   */



/****************************************************************************
**
*f  SyFputs( <line>, <fid> )
*/
#if HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H

void libGAP_SyFputs (
    const libGAP_Char *        line,
    libGAP_Int                 fid )
{
    libGAP_UInt                i;

    /* if outputing to the terminal compute the cursor position and length */
    if ( fid == 1 || fid == 3 ) {
        libGAP_syNrchar = 0;
        for ( i = 0; line[i] != '\0'; i++ ) {
            if ( line[i] == '\n' )  libGAP_syNrchar = 0;
            else                    libGAP_syPrompt[libGAP_syNrchar++] = line[i];
        }
        libGAP_syPrompt[libGAP_syNrchar] = '\0';
    }

    /* otherwise compute only the length                                   */
    else {
        for ( i = 0; line[i] != '\0'; i++ )
            ;
    }

    /* if running under a window handler, send the line to it              */
    if ( libGAP_SyWindow && fid < 4 )
        libGAP_syWinPut( fid, (fid == 1 ? "@n" : "@f"), line );

    /* otherwise, write it to the output file                              */
    else
        libGAP_writeandcheck( libGAP_syBuf[fid].fp, line, i );
}

#endif /* HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H */


/****************************************************************************
**

*F * * * * * * * * * * * * * * * * * input  * * * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  SyFtell( <fid> )  . . . . . . . . . . . . . . . . . .  position of stream
*/
libGAP_Int libGAP_SyFtell (
    libGAP_Int                 fid )
{
    /* check file identifier                                               */
    if ( sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) <= fid || fid < 0 ) {
        return -1;
    }
    if ( libGAP_syBuf[fid].fp == -1 ) {
        return -1;
    }

    /* cannot seek in a pipe                                               */
    if ( libGAP_syBuf[fid].pipe ) {
        return -1;
    }

    /* get the position
     */

    return (libGAP_Int) lseek(libGAP_syBuf[fid].fp, 0, SEEK_CUR);
}


/****************************************************************************
**
*F  SyFseek( <fid>, <pos> )   . . . . . . . . . . . seek a position of stream
*/
libGAP_Int libGAP_SyFseek (
    libGAP_Int                 fid,
    libGAP_Int                 pos )
{
    /* check file identifier                                               */
    if ( sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) <= fid || fid < 0 ) {
        return -1;
    }
    if ( libGAP_syBuf[fid].fp == -1 ) {
        return -1;
    }

    /* cannot seek in a pipe                                               */
    if ( libGAP_syBuf[fid].pipe ) {
        return -1;
    }

    /* get the position                                                    */
    lseek( libGAP_syBuf[fid].fp, pos, SEEK_SET );
    return 0;
}


/****************************************************************************
**
*F  syGetchTerm( <fid> )  . . . . . . . . . . . . . . . . . get a char from <fid>
**
**  'SyGetchTerm' reads a character from <fid>, which is already switched
**  to raw mode if it is *stdin* or *errin*.

*/



/****************************************************************************
**
*f  syGetchTerm( <fid> )  . . . . . . . . . . . . . . . . . . . . . UNIX
**
**  This version should be called if the input is stdin and command-line editing
**  etc. is switched on. It handles possible messages from xgap and systems
**  that return odd things rather than waiting for a key
**
*/
#if HAVE_SGTTY_H || HAVE_TERMIO_H || HAVE_TERMIOS_H

/* In the cygwin environment it is not predictable if text files get the
 * '\r' in their line ends filtered out *before* GAP sees them. This leads
 * to problem with continuation of strings or integers over several lines in
 * GAP input. Therefore we introduce a hack which removes such '\r's
 * before '\n's on such a system. Add here if there are other systems with
 * a similar problem.
 */

#if SYS_IS_CYGWIN32
#  define libGAP_LINE_END_HACK 1
#endif


libGAP_Int libGAP_syGetchTerm (
    libGAP_Int                 fid )
{
    libGAP_UChar                ch;
    libGAP_Char                str[2];
    libGAP_Int ret;

    /* retry on errors or end-of-file. Ignore 0 bytes */

#if libGAP_LINE_END_HACK
 tryagain:
#endif
    while ( (ret = read( libGAP_syBuf[fid].fp, &ch, 1 )) == -1 && errno == EAGAIN )
        ;
    if (ret <= 0) return EOF;

    /* if running under a window handler, handle special characters        */
    if ( libGAP_SyWindow && ch == '@' ) {
        do {
            while ( (ret = read(libGAP_syBuf[fid].fp, &ch, 1)) == -1 &&
                    errno == EAGAIN ) ;
            if (ret <= 0) return EOF;
        } while ( ch < '@' || 'z' < ch );
        if ( ch == 'y' ) {
            do {
                while ( (ret = read(libGAP_syBuf[fid].fp, &ch, 1)) == -1 &&
                        errno == EAGAIN );
                if (ret <= 0) return EOF;
            } while ( ch < '@' || 'z' < ch );
            str[0] = ch;
            str[1] = 0;
            libGAP_syWinPut( libGAP_syBuf[fid].echo, "@s", str );
            ch = libGAP_syGetchTerm(fid);
        }
        else if ( 'A' <= ch && ch <= 'Z' )
            ch = libGAP_CTR(ch);
    }

#if libGAP_LINE_END_HACK
    /* A hack for non ANSI-C confirming systems which deliver \r or \r\n
     * line ends. These are translated to \n here.
     */
    if (ch == '\n') {
        if (libGAP_syBuf[fid].crlast) {
            libGAP_syBuf[fid].crlast = 0;
            goto tryagain;
        } else
            return (libGAP_UChar)'\n';
    }
    if (ch == '\r') {
        libGAP_syBuf[fid].crlast = 1;
        return (libGAP_Int)'\n';
    }
#endif  /* line end hack */

    /* return the character                                                */
    return (libGAP_Int)ch;
}

libGAP_Int libGAP_syGetchNonTerm (
    libGAP_Int                 fid )
{
    libGAP_UChar                ch;
    libGAP_UInt                bufno;
    int                 ret;


    /* we jump back here if the byte we just read was the \n of \r\n, in which
       case it doesn't count */

#if libGAP_LINE_END_HACK
 tryagain:
#endif
    if (libGAP_syBuf[fid].bufno < 0)
        while ( (ret = read( libGAP_syBuf[fid].fp, &ch, 1 )) == -1 && errno == EAGAIN)
           ;
    else {
        bufno = libGAP_syBuf[fid].bufno;
        if (libGAP_syBuffers[bufno].bufstart < libGAP_syBuffers[bufno].buflen) {
            ch = libGAP_syBuffers[bufno].buf[libGAP_syBuffers[bufno].bufstart++];
            ret = 1;
        } else {
            while ( (ret = read( libGAP_syBuf[fid].fp,
                                 libGAP_syBuffers[bufno].buf,
                                 libGAP_SYS_FILE_BUF_SIZE )) == -1 && errno == EAGAIN)
              ;
            if (ret > 0) {
                ch = libGAP_syBuffers[bufno].buf[0];
                libGAP_syBuffers[bufno].bufstart = 1;
                libGAP_syBuffers[bufno].buflen = ret;
            }
        }
    }

    if (ret < 1) {
        libGAP_syBuf[fid].ateof = 1;
        return EOF;
    }

#if libGAP_LINE_END_HACK
    /* A hack for non ANSI-C confirming systems which deliver \r or \r\n
     * line ends. These are translated to \n here.
     */
    if (ch == '\n') {
        if (libGAP_syBuf[fid].crlast) {
            libGAP_syBuf[fid].crlast = 0;
            goto tryagain;
        } else
            return (libGAP_UChar)'\n';
    }
    if (ch == '\r') {
        libGAP_syBuf[fid].crlast = 1;
        return (libGAP_Int)'\n';
    }
#endif  /* line end hack */

    /* return the character                                                */
    return (libGAP_Int)ch;
}






/****************************************************************************
**
*f  syGetch( <fid> )
*/

libGAP_Int libGAP_syGetch (
    libGAP_Int                 fid )
{
    if (libGAP_syBuf[fid].isTTY)
      return libGAP_syGetchTerm(fid);
    else
      return libGAP_syGetchNonTerm(fid);
}

#endif


/****************************************************************************
**
*F  SyGetch( <fid> )  . . . . . . . . . . . . . . . . . get a char from <fid>
**
**  'SyGetch' reads a character from <fid>, which is switch to raw mode if it
**  is *stdin* or *errin*.
*/
libGAP_Int libGAP_SyGetch (
    libGAP_Int                 fid )
{
    libGAP_Int                 ch;

    /* check file identifier                                               */
    if ( sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) <= fid || fid < 0 ) {
        return -1;
    }
        /* on the Mac, syBuf[fid].fp is -1 for stdin, stdout, errin, errout
           the other cases are handled by syGetch */
    if ( libGAP_syBuf[fid].fp == -1 ) {
        return -1;
    }

    /* if we are reading stdin or errin use raw mode                       */
    if ( fid == 0 || fid == 2 ) {
        libGAP_syStartraw(fid);
    }
    ch = libGAP_syGetch(fid);
    if ( fid == 0 || fid == 2 ) {
        libGAP_syStopraw(fid);
    }
    return ch;
}


/****************************************************************************
**
*F  SyGetc( <fid> ).  . . . . . . . . . . . . . . . . . get a char from <fid>
**
**  'SyGetc' reads a character from <fid>, without any translation or
**   interference
*/

libGAP_Int libGAP_SyGetc
(
    libGAP_Int                 fid )
{
    unsigned char ch;
    int ret = read(libGAP_syBuf[fid].fp, &ch, 1);
    if (ret < 1)
      return EOF;
    else
      return (libGAP_Int)ch;
}

/****************************************************************************
**
*F  SyPutc( <fid>, <char> ).. . . . . . . . . . . . . . . put a char to <fid>
**
**  'SyPutc' writes a character to <fid>, without any translation or
**   interference
*/

extern libGAP_Int libGAP_SyPutc
(
    libGAP_Int                 fid,
    libGAP_Char                c )
{
    libGAP_writeandcheck(libGAP_syBuf[fid].fp,&c,1);
    return 0;
}


/****************************************************************************
**
*F  SyFgets( <line>, <length>, <fid> )  . . . . .  get a line from file <fid>
**
**  'SyFgets' is called to read a line from the file  with  identifier <fid>.
**  'SyFgets' (like 'fgets') reads characters until either  <length>-1  chars
**  have been read or until a <newline> or an  <eof> character is encoutered.
**  It retains the '\n' (unlike 'gets'), if any, and appends '\0' to  <line>.
**  'SyFgets' returns <line> if any char has been read, otherwise '(char*)0'.
**
**  'SyFgets'  allows to edit  the input line if the  file  <fid> refers to a
**  terminal with the following commands:
**
**      <ctr>-A move the cursor to the beginning of the line.
**      <esc>-B move the cursor to the beginning of the previous word.
**      <ctr>-B move the cursor backward one character.
**      <ctr>-F move the cursor forward  one character.
**      <esc>-F move the cursor to the end of the next word.
**      <ctr>-E move the cursor to the end of the line.
**
**      <ctr>-H, <del> delete the character left of the cursor.
**      <ctr>-D delete the character under the cursor.
**      <ctr>-K delete up to the end of the line.
**      <esc>-D delete forward to the end of the next word.
**      <esc>-<del> delete backward to the beginning of the last word.
**      <ctr>-X delete entire input line, and discard all pending input.
**      <ctr>-Y insert (yank) a just killed text.
**
**      <ctr>-T exchange (twiddle) current and previous character.
**      <esc>-U uppercase next word.
**      <esc>-L lowercase next word.
**      <esc>-C capitalize next word.
**
**      <tab>   complete the identifier before the cursor.
**      <ctr>-L insert last input line before current character.
**      <ctr>-P redisplay the last input line, another <ctr>-P will redisplay
**              the line before that, etc.  If the cursor is not in the first
**              column only the lines starting with the string to the left of
**              the cursor are taken. The history is limitied to ~8000 chars.
**      <ctr>-N Like <ctr>-P but goes the other way round through the history
**      <esc>-< goes to the beginning of the history.
**      <esc>-> goes to the end of the history.
**      <ctr>-O accept this line and perform a <ctr>-N.
**
**      <ctr>-V enter next character literally.
**      <ctr>-U execute the next command 4 times.
**      <esc>-<num> execute the next command <num> times.
**      <esc>-<ctr>-L repaint input line.
**
**  Not yet implemented commands:
**
**      <ctr>-S search interactive for a string forward.
**      <ctr>-R search interactive for a string backward.
**      <esc>-Y replace yanked string with previously killed text.
**      <ctr>-_ undo a command.
**      <esc>-T exchange two words.
*/

libGAP_UInt   libGAP_syCTRO;                          /* number of '<ctr>-O' pending     */
libGAP_UInt   libGAP_syESCN;                          /* number of '<Esc>-N' pending     */

#if HAVE_SELECT
libGAP_Obj libGAP_OnCharReadHookActive = 0;  /* if bound the hook is active */
libGAP_Obj libGAP_OnCharReadHookInFds = 0;   /* a list of UNIX file descriptors for reading */
libGAP_Obj libGAP_OnCharReadHookInFuncs = 0; /* a list of GAP functions with 0 args */
libGAP_Obj libGAP_OnCharReadHookOutFds = 0;  /* a list of UNIX file descriptors for writing */
libGAP_Obj libGAP_OnCharReadHookOutFuncs = 0;/* a list of GAP functions with 0 args */
libGAP_Obj libGAP_OnCharReadHookExcFds = 0;  /* a list of UNIX file descriptors */
libGAP_Obj libGAP_OnCharReadHookExcFuncs = 0;/* a list of GAP functions with 0 args */


void libGAP_HandleCharReadHook(int stdinfd)
/* This is called directly before a character is read from stdin in the case
 * of an interactive session with command line editing. We have to return
 * as soon as stdin is ready to read! We just use `select' and care for
 * handlers for streams. */
{
  fd_set infds,outfds,excfds;
  int n,maxfd;
  libGAP_Int i,j;
  libGAP_Obj o;
  static int WeAreAlreadyInHere = 0;

  /* Just to make sure: */
  if (WeAreAlreadyInHere) return;
  WeAreAlreadyInHere = 1;

  while (1) {  /* breaks when fd becomes ready */
    FD_ZERO(&infds);
    FD_ZERO(&outfds);
    FD_ZERO(&excfds);
    FD_SET(stdinfd,&infds);
    maxfd = stdinfd;
    /* Handle input file descriptors: */
    if (libGAP_OnCharReadHookInFds != (libGAP_Obj) 0 &&
        libGAP_IS_PLIST(libGAP_OnCharReadHookInFds) &&
        libGAP_OnCharReadHookInFuncs != (libGAP_Obj) 0 &&
        libGAP_IS_PLIST(libGAP_OnCharReadHookInFuncs)) {
      for (i = 1;i <= libGAP_LEN_PLIST(libGAP_OnCharReadHookInFds);i++) {
        o = libGAP_ELM_PLIST(libGAP_OnCharReadHookInFds,i);
        if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
          j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
          FD_SET(j,&infds);
          if (j > maxfd) maxfd = j;
        }
      }
    }
    /* Handle output file descriptors: */
    if (libGAP_OnCharReadHookOutFds != (libGAP_Obj) 0 &&
        libGAP_IS_PLIST(libGAP_OnCharReadHookOutFds) &&
        libGAP_OnCharReadHookOutFuncs != (libGAP_Obj) 0 &&
        libGAP_IS_PLIST(libGAP_OnCharReadHookOutFuncs)) {
      for (i = 1;i <= libGAP_LEN_PLIST(libGAP_OnCharReadHookOutFds);i++) {
        o = libGAP_ELM_PLIST(libGAP_OnCharReadHookOutFds,i);
        if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
          j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
          FD_SET(j,&outfds);
          if (j > maxfd) maxfd = j;
        }
      }
    }
    /* Handle exception file descriptors: */
    if (libGAP_OnCharReadHookExcFds != (libGAP_Obj) 0 &&
        libGAP_IS_PLIST(libGAP_OnCharReadHookExcFds) &&
        libGAP_OnCharReadHookExcFuncs != (libGAP_Obj) 0 &&
        libGAP_IS_PLIST(libGAP_OnCharReadHookExcFuncs)) {
      for (i = 1;i <= libGAP_LEN_PLIST(libGAP_OnCharReadHookExcFds);i++) {
        o = libGAP_ELM_PLIST(libGAP_OnCharReadHookExcFds,i);
        if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
          j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
          FD_SET(j,&excfds);
          if (j > maxfd) maxfd = j;
        }
      }
    }

    n = select(maxfd+1,&infds,&outfds,&excfds,NULL);
    if (n >= 0) {
      /* Now run through the lists and call functions if ready: */

      if (libGAP_OnCharReadHookInFds != (libGAP_Obj) 0 &&
          libGAP_IS_PLIST(libGAP_OnCharReadHookInFds) &&
          libGAP_OnCharReadHookInFuncs != (libGAP_Obj) 0 &&
          libGAP_IS_PLIST(libGAP_OnCharReadHookInFuncs)) {
        for (i = 1;i <= libGAP_LEN_PLIST(libGAP_OnCharReadHookInFds);i++) {
          o = libGAP_ELM_PLIST(libGAP_OnCharReadHookInFds,i);
          if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
            j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
            if (FD_ISSET(j,&infds)) {
              o = libGAP_ELM_PLIST(libGAP_OnCharReadHookInFuncs,i);
              if (o != (libGAP_Obj) 0 && libGAP_IS_FUNC(o))
                libGAP_Call1ArgsInNewReader(o,libGAP_INTOBJ_INT(i));
            }
          }
        }
      }
      /* Handle output file descriptors: */
      if (libGAP_OnCharReadHookOutFds != (libGAP_Obj) 0 &&
          libGAP_IS_PLIST(libGAP_OnCharReadHookOutFds) &&
          libGAP_OnCharReadHookOutFuncs != (libGAP_Obj) 0 &&
          libGAP_IS_PLIST(libGAP_OnCharReadHookOutFuncs)) {
        for (i = 1;i <= libGAP_LEN_PLIST(libGAP_OnCharReadHookOutFds);i++) {
          o = libGAP_ELM_PLIST(libGAP_OnCharReadHookOutFds,i);
          if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
            j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
            if (FD_ISSET(j,&outfds)) {
              o = libGAP_ELM_PLIST(libGAP_OnCharReadHookOutFuncs,i);
              if (o != (libGAP_Obj) 0 && libGAP_IS_FUNC(o))
                libGAP_Call1ArgsInNewReader(o,libGAP_INTOBJ_INT(i));
            }
          }
        }
      }
      /* Handle exception file descriptors: */
      if (libGAP_OnCharReadHookExcFds != (libGAP_Obj) 0 &&
          libGAP_IS_PLIST(libGAP_OnCharReadHookExcFds) &&
          libGAP_OnCharReadHookExcFuncs != (libGAP_Obj) 0 &&
          libGAP_IS_PLIST(libGAP_OnCharReadHookExcFuncs)) {
        for (i = 1;i <= libGAP_LEN_PLIST(libGAP_OnCharReadHookExcFds);i++) {
          o = libGAP_ELM_PLIST(libGAP_OnCharReadHookExcFds,i);
          if (o != (libGAP_Obj) 0 && libGAP_IS_INTOBJ(o)) {
            j = libGAP_INT_INTOBJ(o);  /* a UNIX file descriptor */
            if (FD_ISSET(j,&excfds)) {
              o = libGAP_ELM_PLIST(libGAP_OnCharReadHookExcFuncs,i);
              if (o != (libGAP_Obj) 0 && libGAP_IS_FUNC(o))
                libGAP_Call1ArgsInNewReader(o,libGAP_INTOBJ_INT(i));
            }
          }
        }
      }

      if (FD_ISSET(stdinfd,&infds)) {
        WeAreAlreadyInHere = 0;
        break;
      }
    } else
      break;
  } /* while (1) */
}
#endif   /* HAVE_SELECT */



/***************************************************************************
**
*F HasAvailableBytes( <fid> ) returns positive if  a subsequent read to <fid>
**                            will read at least one byte without blocking
**
*/

libGAP_Int libGAP_HasAvailableBytes( libGAP_UInt fid )
{
#if ! (HAVE_SELECT)
  libGAP_Int ret;
#endif
  libGAP_UInt bufno;
  if (fid > sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) ||
      libGAP_syBuf[fid].fp == -1)
    return -1;

  if (libGAP_syBuf[fid].bufno > 0)
    {
      bufno = libGAP_syBuf[fid].bufno;
      if (libGAP_syBuffers[bufno].bufstart < libGAP_syBuffers[bufno].buflen)
        return 1;
    }

#if HAVE_SELECT
  {
    fd_set set;
    struct timeval tv;
    FD_ZERO( &set);
    FD_SET( libGAP_syBuf[fid].fp, &set );
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    return select( libGAP_syBuf[fid].fp + 1, &set, NULL, NULL, &tv);
  }
#else
    /* best guess */
  ret =  libGAP_SyIsEndOfFile( fid);
  return (ret != -1 && ret != 1);
#endif
}




libGAP_Char * libGAP_syFgetsNoEdit (
    libGAP_Char *              line,
    libGAP_UInt                length,
    libGAP_Int                 fid,
    libGAP_UInt                block)
{
  libGAP_UInt x = 0;
  int ret = 0;

  /* if stream is buffered, and the buffer has a full line,
   * grab it -- we could make more use of the buffer, but
   * this covers the majority of cases simply. */
#ifndef libGAP_LINE_END_HACK
  libGAP_UInt bufno;
  libGAP_Char* newlinepos;
  libGAP_Char* bufstart;
  int buflen;
  if(!libGAP_syBuf[fid].isTTY && libGAP_syBuf[fid].bufno >= 0) {
    bufno = libGAP_syBuf[fid].bufno;
    if (libGAP_syBuffers[bufno].bufstart < libGAP_syBuffers[bufno].buflen) {
      bufstart = libGAP_syBuffers[bufno].buf + libGAP_syBuffers[bufno].bufstart;
      buflen = libGAP_syBuffers[bufno].buflen - libGAP_syBuffers[bufno].bufstart;
      newlinepos = memchr(bufstart, '\n', buflen);
      if(newlinepos && (newlinepos - bufstart) < length - 2) {
          newlinepos++;
          memcpy(line, bufstart, newlinepos - bufstart);
          line[newlinepos - bufstart] = '\0';
          libGAP_syBuffers[bufno].bufstart += (newlinepos - bufstart);
          return line;
      }
    }
  }
#endif

  while (x < length -1) {
    if (!block && x && !libGAP_HasAvailableBytes( fid ))
      {
        break;
      }
    ret = libGAP_syGetch(fid);
    if (ret == EOF)
      break;
    if ((line[x++] = ret) == '\n')
      break;
  }
  line[x] = '\0';
  libGAP_syBuf[fid].ateof = (ret == EOF);
  if (x)
    return line;
  else
    return NULL;
}

/* will be imported from library, first is generic function which does some
   checks before returning result to kernel, the second is the list of handler
   functions which do the actual work. */
libGAP_Obj libGAP_LineEditKeyHandler;
libGAP_Obj libGAP_LineEditKeyHandlers;
libGAP_Obj libGAP_GAPInfo;

#if HAVE_LIBREADLINE

/* we import GAP level functions from GAPInfo components */
libGAP_Obj libGAP_CLEFuncs;
libGAP_Obj libGAP_KeyHandler;

static int libGAP_GAPMacroNumber = 0;

int libGAP_GAP_set_macro(int count, int key)
{
 libGAP_GAPMacroNumber = count;
 return 0;
}
/* a generic rl_command_func_t that delegates to GAP level */
int libGAP_GAP_rl_func(int count, int key)
{
   libGAP_Obj   rldata, linestr, okey, res, obj, data, beginchange, endchange, m;
   libGAP_Int   len, n, hook, dlen, max, i;

   /* we shift indices 0-based on C-level and 1-based on GAP level */
   libGAP_C_NEW_STRING_DYN(linestr, rl_line_buffer);
   okey = libGAP_INTOBJ_INT(key + 1000*libGAP_GAPMacroNumber);
   libGAP_GAPMacroNumber = 0;
   rldata = libGAP_NEW_PLIST(libGAP_T_PLIST, 6);
   if (libGAP_GAP_rl_func == rl_last_func) {
     libGAP_SET_LEN_PLIST(rldata, 6);
     libGAP_SET_ELM_PLIST(rldata, 6, libGAP_True);
   }
   else
     libGAP_SET_LEN_PLIST(rldata, 5);
   libGAP_SET_ELM_PLIST(rldata, 1, libGAP_INTOBJ_INT(count));
   libGAP_SET_ELM_PLIST(rldata, 2, okey);
   libGAP_SET_ELM_PLIST(rldata, 3, linestr);
   libGAP_SET_ELM_PLIST(rldata, 4, libGAP_INTOBJ_INT(rl_point+1));
   libGAP_SET_ELM_PLIST(rldata, 5, libGAP_INTOBJ_INT(rl_mark+1));
   res = libGAP_Call1ArgsInNewReader(libGAP_KeyHandler, rldata);
   if (!res) return 0;
   if (!libGAP_IS_LIST(res)) return 0;
   len = libGAP_LEN_LIST(res);
   if (len == 0) return 0;
   obj = libGAP_ELM_LIST(res, 1);
   if (libGAP_IsStringConv(obj)) {
      /* insert txt */
      rl_insert_text(libGAP_CSTR_STRING(obj));
      n = 1;
   } else if ((obj == libGAP_True || obj == libGAP_False) && len > 2) {
      /* kill or delete text */
      beginchange = libGAP_ELM_LIST(res, 2);
      if (!libGAP_IS_INTOBJ(beginchange)) return 0;
      endchange = libGAP_ELM_LIST(res, 3);
      if (!libGAP_IS_INTOBJ(endchange)) return 0;
      if (obj == libGAP_True)
         rl_kill_text(libGAP_INT_INTOBJ(beginchange)-1, libGAP_INT_INTOBJ(endchange)-1);
      else
         rl_delete_text(libGAP_INT_INTOBJ(beginchange)-1, libGAP_INT_INTOBJ(endchange)-1);
      n = 3;
   }  else if (libGAP_IS_INTOBJ(obj) && len > 2) {
      /* delete some text and insert */
      beginchange = obj;
      endchange = libGAP_ELM_LIST(res, 2);
      if (!libGAP_IS_INTOBJ(endchange)) return 0;
      obj = libGAP_ELM_LIST(res, 3);
      if (!libGAP_IsStringConv(obj)) return 0;
      rl_begin_undo_group();
      rl_delete_text(libGAP_INT_INTOBJ(beginchange)-1, libGAP_INT_INTOBJ(endchange)-1);
      rl_point = libGAP_INT_INTOBJ(beginchange)-1;
      rl_insert_text(libGAP_CSTR_STRING(obj));
      rl_end_undo_group();
      n = 3;
   } else if (libGAP_IS_INTOBJ(obj) && len == 2) {
      /* several hooks to particular rl_ functions with data */
      hook = libGAP_INT_INTOBJ(obj);
      data = libGAP_ELM_LIST(res, 2);
      if (hook == 1) {
         /* display matches */
         if (!libGAP_IS_LIST(data)) return 0;
         dlen = libGAP_LEN_LIST(data);
         char **strs = (char**)malloc((dlen+1) * sizeof(char*));
         max = 0;
         for (i=1; i <= dlen; i++) {
            if (!libGAP_IsStringConv(libGAP_ELM_LIST(data, i))) {
               free(strs);
               return 0;
            }
            strs[i] = libGAP_CSTR_STRING(libGAP_ELM_LIST(data, i));
            if (max < strlen(strs[i])) max = strlen(strs[i]);
         }
         rl_display_match_list(strs, dlen, max);
         free(strs);
         rl_on_new_line();
      }
      else if (hook == 2) {
         /* put these characters into sequence of input keys */
         if (!libGAP_IsStringConv(data)) return 0;
         dlen = strlen(libGAP_CSTR_STRING(data));
         for (i=0; i < dlen; i++)
             rl_stuff_char(libGAP_CSTR_STRING(data)[i]);
      }
      n = 2;
   } else if (libGAP_IS_INTOBJ(obj) && len == 1) {
      /* several hooks to particular rl_ functions with no data */
      hook = libGAP_INT_INTOBJ(obj);
      /* ring bell */
      if (hook == 100) rl_ding();
      /* return line (execute Ctrl-m) */
      else if (hook == 101) rl_execute_next(13);
      n = 1;
   } else
      n = 0;

   /* optionally we can return the new point, or new point and mark */
   if (len > n) {
      n++;
      m = libGAP_ELM_LIST(res, n);
      if libGAP_IS_INTOBJ(m) rl_point = libGAP_INT_INTOBJ(m)-1;
   }
   if (len > n) {
      n++;
      m = libGAP_ELM_LIST(res, n);
      if libGAP_IS_INTOBJ(m) rl_mark = libGAP_INT_INTOBJ(m)-1;
   }
   return 0;
}

libGAP_Obj libGAP_FuncBINDKEYSTOGAPHANDLER (libGAP_Obj self, libGAP_Obj keys)
{
  libGAP_Char*  seq;

  if (!libGAP_IsStringConv(keys)) return libGAP_False;
  seq = libGAP_CSTR_STRING(keys);
  rl_bind_keyseq(seq, libGAP_GAP_rl_func);

  return libGAP_True;
}

libGAP_Obj libGAP_FuncBINDKEYSTOMACRO (libGAP_Obj self, libGAP_Obj keys, libGAP_Obj macro)
{
  libGAP_Char   *seq, *macr;

  if (!libGAP_IsStringConv(keys)) return libGAP_False;
  if (!libGAP_IsStringConv(macro)) return libGAP_False;
  seq = libGAP_CSTR_STRING(keys);
  macr = libGAP_CSTR_STRING(macro);
  rl_generic_bind(ISMACR, seq, macr, rl_get_keymap());
  return libGAP_True;
}

libGAP_Obj libGAP_FuncREADLINEINITLINE (libGAP_Obj self, libGAP_Obj line)
{
  libGAP_Char   *cline;

  if (!libGAP_IsStringConv(line)) return libGAP_False;
  cline = libGAP_CSTR_STRING(line);
  rl_parse_and_bind(cline);
  return libGAP_True;
}

/* init is needed once */
libGAP_Int libGAP_ISINITREADLINE = 0;
/* a hook function called regularly while waiting on input */
libGAP_Int libGAP_current_rl_fid;
int libGAP_charreadhook_rl ( void )
{
  if (libGAP_OnCharReadHookActive != (libGAP_Obj) 0)
    libGAP_HandleCharReadHook(libGAP_syBuf[libGAP_current_rl_fid].fp);
  return 0;
}

void libGAP_initreadline ( void )
{

  /* allows users to configure GAP specific settings in their ~/.inputrc like:
       $if GAP
          ....
       $endif                                                             */
  rl_readline_name = "GAP";
  /* this should pipe signals through to GAP  */
  rl_already_prompted = 1 ;

  rl_catch_signals = 0;
  rl_catch_sigwinch = 1;
  /* hook to read from other channels */
  rl_event_hook = 0;
  /* give GAP_rl_func a name that can be used in .inputrc */
  rl_add_defun( "handled-by-GAP", libGAP_GAP_rl_func, -1 );

  rl_bind_keyseq("\\C-x\\C-g", libGAP_GAP_set_macro);

  libGAP_CLEFuncs = libGAP_ELM_REC(libGAP_GAPInfo, libGAP_RNamName("CommandLineEditFunctions"));
  libGAP_KeyHandler = libGAP_ELM_REC(libGAP_CLEFuncs, libGAP_RNamName("KeyHandler"));
  libGAP_ISINITREADLINE = 1;
}

libGAP_Char * libGAP_readlineFgets (
    libGAP_Char *              line,
    libGAP_UInt                length,
    libGAP_Int                 fid,
    libGAP_UInt                block)
{
  char *                 rlres = (char*)NULL;
  libGAP_UInt                   len;

  libGAP_current_rl_fid = fid;
  if (!libGAP_ISINITREADLINE) libGAP_initreadline();

  /* read at most as much as we can buffer */
  rl_num_chars_to_read = length-2;
  /* hook to read from other channels */
  rl_event_hook = (libGAP_OnCharReadHookActive != (libGAP_Obj) 0) ? libGAP_charreadhook_rl : 0;
  /* now do the real work */
  libGAP_doingReadline = 1;
  rlres = readline(libGAP_TLS(libGAP_Prompt));
  libGAP_doingReadline = 0;
  /* we get a NULL pointer on EOF, say by pressing Ctr-d  */
  if (!rlres) {
    if (!libGAP_SyCTRD) {
      while (!rlres)
        rlres = readline(libGAP_TLS(libGAP_Prompt));
    }
    else {
      printf("\n");fflush(stdout);
      line[0] = '\0';
      return (libGAP_Char*)0;
    }
  }
  /* maybe add to history, we use key 0 for this function */
  libGAP_GAP_rl_func(0, 0);
  len = strlen(rlres);
  strncpy(line, rlres, len);
  free(rlres);
  line[len] = '\n';
  line[len+1] = '\0';

  /* send the whole line (unclipped) to the window handler               */
  libGAP_syWinPut( fid, (*line != '\0' ? "@r" : "@x"), line );

  return line;
}

#endif

libGAP_Char * libGAP_syFgets (
    libGAP_Char *              line,
    libGAP_UInt                length,
    libGAP_Int                 fid,
    libGAP_UInt                block)
{
    libGAP_Int                 ch,  ch2,  ch3, last;
    libGAP_Char                * p,  * q,  * r,  * s,  * t;
    static libGAP_Char         yank [32768];
    libGAP_Char                old [512],  new [512];
    libGAP_Int                 oldc,  newc;
    libGAP_Int                 rep, len;
    libGAP_Char                buffer [512];
    libGAP_Int                 rn;
    libGAP_Int                 rubdel;
    libGAP_Obj                 linestr, yankstr, args, res;

    /* check file identifier                                               */
    if ( sizeof(libGAP_syBuf)/sizeof(libGAP_syBuf[0]) <= fid || fid < 0 ) {
        return (libGAP_Char*)0;
    }
    if ( libGAP_syBuf[fid].fp == -1 ) {
        return (libGAP_Char*)0;
    }

    /* no line editing if the file is not '*stdin*' or '*errin*'           */
    if ( fid != 0 && fid != 2 ) {
      p = libGAP_syFgetsNoEdit(line, length, fid, block);

        return p;
    }

    /* no line editing if the user disabled it
       or we can't make it into raw mode */
    if ( libGAP_SyLineEdit == 0 || ! libGAP_syStartraw(fid) ) {
        libGAP_SyStopTime = libGAP_SyTime();
        p = libGAP_syFgetsNoEdit(line, length, fid, block );
        libGAP_SyStartTime += libGAP_SyTime() - libGAP_SyStopTime;
        return p;
    }

#if HAVE_LIBREADLINE
    if (libGAP_SyUseReadline) {
      /* switch back to cooked mode                                          */
      if ( libGAP_SyLineEdit )
          libGAP_syStopraw(fid);

      /* stop the clock, reading should take no time                         */
      libGAP_SyStopTime = libGAP_SyTime();

      p = libGAP_readlineFgets(line, length, fid, block);
      /* start the clock again                                               */
      libGAP_SyStartTime += libGAP_SyTime() - libGAP_SyStopTime;
      if ( libGAP_EndLineHook ) libGAP_Call0ArgsInNewReader( libGAP_EndLineHook );
      if (!p)
        return p;
      else
        return line;
    } else {
#endif

/*  EXPERIMENT    */
    if ( libGAP_LEN_PLIST(libGAP_LineEditKeyHandlers) > 999 &&
                         libGAP_ELM_PLIST(libGAP_LineEditKeyHandlers, 1000) != 0) {
      linestr = libGAP_Call0ArgsInNewReader(libGAP_ELM_PLIST(libGAP_LineEditKeyHandlers, 1000));
      len = libGAP_GET_LEN_STRING(linestr);
      memcpy(line,libGAP_CHARS_STRING(linestr),len);
      line[len] = '\0';
      /* switch back to cooked mode                                          */
      if ( libGAP_SyLineEdit == 1 )
          libGAP_syStopraw(fid);

      /* return the line (or '0' at end-of-file)                             */
      if ( *line == '\0' )
          return (libGAP_Char*)0;
      return line;
    }

/*  END EXPERIMENT    */
    /* In line editing mode 'length' is not allowed bigger than the
      yank buffer (= length of line buffer for input files).*/
    if (length > 32768)
       libGAP_ErrorQuit("Cannot handle lines with more than 32768 characters in line edit mode.",0,0);
    /* stop the clock, reading should take no time                         */
    libGAP_SyStopTime = libGAP_SyTime();

    /* the line starts out blank                                           */
    line[0] = '\0';  p = line;
    for ( q = old; q < old+sizeof(old); ++q )  *q = ' ';
    oldc = 0;
    last = 0;
    ch = 0;
    rubdel=0; /* do we want to east a `del' character? */

    while ( 1 ) {

        /* get a character, handle <ctr>V<chr>, <esc><num> and <ctr>U<num> */
        rep = 1; ch2 = 0;
        do {
            if ( libGAP_syESCN > 0 ) { if (ch == libGAP_Esc('N')) {ch = '\n'; libGAP_syESCN--; }
                                else {ch = libGAP_Esc('N'); } }
            else if ( libGAP_syCTRO % 2 == 1 ) { ch = libGAP_CTR('N'); libGAP_syCTRO = libGAP_syCTRO - 1; }
            else if ( libGAP_syCTRO != 0 ) { ch = libGAP_CTR('O'); rep = libGAP_syCTRO / 2; }
            else {
#if HAVE_SELECT
              if (libGAP_OnCharReadHookActive != (libGAP_Obj) 0)
                libGAP_HandleCharReadHook(libGAP_syBuf[fid].fp);
#endif
              ch = libGAP_syGetch(fid);
            }
            if ( ch2==0        && ch==libGAP_CTR('V') ) {             ch2=ch; ch=0;}
            if ( ch2==0        && ch==libGAP_CTR('[') ) {             ch2=ch; ch=0;}
            if ( ch2==0        && ch==libGAP_CTR('U') ) {             ch2=ch; ch=0;}
            if ( ch2==libGAP_CTR('[') && ch==libGAP_CTR('V') ) { ch2=libGAP_Esc(libGAP_CTR('V'));  ch=0;}
            if ( ch2==libGAP_CTR('[') && libGAP_IsDigit(ch)  ) { rep=ch-'0'; ch2=ch; ch=0;}
            if ( ch2==libGAP_CTR('[') && ch=='['      ) {             ch2=ch; ch=0;}
            if ( ch2==libGAP_CTR('U') && ch==libGAP_CTR('V') ) { rep=4*rep;  ch2=ch; ch=0;}
            if ( ch2==libGAP_CTR('U') && ch==libGAP_CTR('[') ) { rep=4*rep;  ch2=ch; ch=0;}
            if ( ch2==libGAP_CTR('U') && ch==libGAP_CTR('U') ) { rep=4*rep;  ch2=ch; ch=0;}
            if ( ch2==libGAP_CTR('U') && libGAP_IsDigit(ch)  ) { rep=ch-'0'; ch2=ch; ch=0;}
            if ( libGAP_IsDigit(ch2)  && ch==libGAP_CTR('V') ) {             ch2=ch; ch=0;}
            if ( libGAP_IsDigit(ch2)  && ch==libGAP_CTR('[') ) {             ch2=ch; ch=0;}
            if ( libGAP_IsDigit(ch2)  && ch==libGAP_CTR('U') ) {             ch2=ch; ch=0;}
            if ( libGAP_IsDigit(ch2)  && libGAP_IsDigit(ch)  ) { rep=10*rep+ch-'0';  ch=0;}
            /* get rid of tilde in windows commands */
            if (rubdel==1) {
              if ( ch==126 ) {ch2=0;ch=0;};
              rubdel=0;
            }
        } while ( ch == 0 );
        if ( ch2==libGAP_CTR('V') )       ch  = libGAP_CTV(ch);
        if ( ch2==libGAP_Esc(libGAP_CTR('V')) )  ch  = libGAP_CTV(ch | 0x80);
        if ( ch2==libGAP_CTR('[') )       ch  = libGAP_Esc(ch);
        if ( ch2==libGAP_CTR('U') )       rep = 4*rep;
        /* windows keys */
        if ( ch2=='[' && ch=='A')  ch  = libGAP_CTR('P');
        if ( ch2=='[' && ch=='B')  ch  = libGAP_CTR('N');
        if ( ch2=='[' && ch=='C')  ch  = libGAP_CTR('F');
        if ( ch2=='[' && ch=='D')  ch  = libGAP_CTR('B');
        if ( ch2=='[' && ch=='1') { ch  = libGAP_CTR('A');rubdel=1;} /* home */
        if ( ch2=='[' && ch=='3') { ch  = libGAP_CTR('D');rubdel=1;} /* del */
        if ( ch2=='[' && ch=='4') { ch  = libGAP_CTR('E');rubdel=1;} /* end */
        if ( ch2=='[' && ch=='5') { ch  = libGAP_CTR('P');rubdel=1;} /* pgup */
        if ( ch2=='[' && ch=='6') { ch  = libGAP_CTR('N');rubdel=1;} /* pgdwn */

        /* now perform the requested action <rep> times in the input line  */
        while ( rep-- > 0 ) {
          /* check for key handler on GAP level */
          if (ch >= 0 && ch < libGAP_LEN_PLIST(libGAP_LineEditKeyHandlers) &&
                         libGAP_ELM_PLIST(libGAP_LineEditKeyHandlers, ch+1) != 0) {
            /* prepare data for GAP handler:
                   [linestr, ch, ppos, length, yankstr]
               GAP handler must return new
                   [linestr, ppos, yankstr]
               or an integer, interpreted as number of Esc('N')
               calls for the next lines.                                  */
            libGAP_C_NEW_STRING_DYN(linestr,line);
            libGAP_C_NEW_STRING_DYN(yankstr,yank);
            args = libGAP_NEW_PLIST(libGAP_T_PLIST, 5);
            libGAP_SET_LEN_PLIST(args, 5);
            libGAP_SET_ELM_PLIST(args,1,linestr);
            libGAP_SET_ELM_PLIST(args,2,libGAP_INTOBJ_INT(ch));
            libGAP_SET_ELM_PLIST(args,3,libGAP_INTOBJ_INT((p-line)+1));
            libGAP_SET_ELM_PLIST(args,4,libGAP_INTOBJ_INT(length));
            libGAP_SET_ELM_PLIST(args,5,yankstr);
            res = libGAP_Call1ArgsInNewReader(libGAP_LineEditKeyHandler, args);
            if (libGAP_IS_INTOBJ(res)){
               libGAP_syESCN = libGAP_INT_INTOBJ(res);
               ch = libGAP_Esc('N');
               libGAP_SET_ELM_PLIST(args,2,libGAP_INTOBJ_INT(ch));
               res = libGAP_Call1ArgsInNewReader(libGAP_LineEditKeyHandler, args);
            }
            linestr = libGAP_ELM_PLIST(res,1);
            len = libGAP_GET_LEN_STRING(linestr);
            memcpy(line,libGAP_CHARS_STRING(linestr),len);
            line[len] = '\0';
            p = line + (libGAP_INT_INTOBJ(libGAP_ELM_PLIST(res,2)) - 1);
            yankstr = libGAP_ELM_PLIST(res,3);
            len = libGAP_GET_LEN_STRING(yankstr);
            memcpy(yank,libGAP_CHARS_STRING(yankstr),len);
            yank[len] = '\0';
          }
          else {
            switch ( ch ) {

            case libGAP_CTR('A'): /* move cursor to the start of the line         */
                while ( p > line )  --p;
                break;

            case libGAP_Esc('B'): /* move cursor one word to the left             */
            case libGAP_Esc('b'):
                if ( p > line ) do {
                    --p;
                } while ( p>line && (!libGAP_IS_SEP(*(p-1)) || libGAP_IS_SEP(*p)));
                break;

            case libGAP_CTR('B'): /* move cursor one character to the left        */
                if ( p > line )  --p;
                break;

            case libGAP_CTR('F'): /* move cursor one character to the right       */
                if ( *p != '\0' )  ++p;
                break;

            case libGAP_Esc('F'): /* move cursor one word to the right            */
            case libGAP_Esc('f'):
                if ( *p != '\0' ) do {
                    ++p;
                } while ( *p!='\0' && (libGAP_IS_SEP(*(p-1)) || !libGAP_IS_SEP(*p)));
                break;

            case libGAP_CTR('E'): /* move cursor to the end of the line           */
                while ( *p != '\0' )  ++p;
                break;

            case libGAP_CTR('H'): /* delete the character left of the cursor      */
            case 127:
                if ( p == line ) break;
                --p;
                /* let '<ctr>-D' do the work                               */

            case libGAP_CTR('D'): /* delete the character at the cursor           */
                           /* on an empty line '<ctr>-D' is <eof>          */
                if ( p == line && *p == '\0' && libGAP_SyCTRD && !rubdel ) {
                    ch = EOF; rep = 0; break;
                }
                if ( *p != '\0' ) {
                    for ( q = p; *(q+1) != '\0'; ++q )
                        *q = *(q+1);
                    *q = '\0';
                }
                break;

            case libGAP_CTR('X'): /* delete the line                              */
                p = line;
                /* let '<ctr>-K' do the work                               */

            case libGAP_CTR('K'): /* delete to end of line                        */
                if ( last!=libGAP_CTR('X') && last!=libGAP_CTR('K') && last!=libGAP_Esc(127)
                  && last!=libGAP_Esc('D') && last!=libGAP_Esc('d') )  yank[0] = '\0';
                for ( r = yank; *r != '\0'; ++r ) ;
                for ( s = p; *s != '\0'; ++s )  r[s-p] = *s;
                r[s-p] = '\0';
                *p = '\0';
                break;

            case libGAP_Esc(127): /* delete the word left of the cursor           */
                q = p;
                if ( p > line ) do {
                    --p;
                } while ( p>line && (!libGAP_IS_SEP(*(p-1)) || libGAP_IS_SEP(*p)));
                if ( last!=libGAP_CTR('X') && last!=libGAP_CTR('K') && last!=libGAP_Esc(127)
                  && last!=libGAP_Esc('D') && last!=libGAP_Esc('d') )  yank[0] = '\0';
                for ( r = yank; *r != '\0'; ++r ) ;
                for ( ; yank <= r; --r )  r[q-p] = *r;
                for ( s = p; s < q; ++s )  yank[s-p] = *s;
                for ( r = p; *q != '\0'; ++q, ++r )
                    *r = *q;
                *r = '\0';
                break;

            case libGAP_Esc('D'): /* delete the word right of the cursor          */
            case libGAP_Esc('d'):
                q = p;
                if ( *q != '\0' ) do {
                    ++q;
                } while ( *q!='\0' && (libGAP_IS_SEP(*(q-1)) || !libGAP_IS_SEP(*q)));
                if ( last!=libGAP_CTR('X') && last!=libGAP_CTR('K') && last!=libGAP_Esc(127)
                  && last!=libGAP_Esc('D') && last!=libGAP_Esc('d') )  yank[0] = '\0';
                for ( r = yank; *r != '\0'; ++r ) ;
                for ( s = p; s < q; ++s )  r[s-p] = *s;
                r[s-p] = '\0';
                for ( r = p; *q != '\0'; ++q, ++r )
                    *r = *q;
                *r = '\0';
                break;

            case libGAP_CTR('T'): /* twiddle characters                           */
                if ( p == line )  break;
                if ( *p == '\0' )  --p;
                if ( p == line )  break;
                ch2 = *(p-1);  *(p-1) = *p;  *p = ch2;
                ++p;
                break;

            case libGAP_CTR('Y'): /* insert (yank) deleted text                   */
                if (strlen(yank) + strlen(line) - 2 > length) {
                    libGAP_syEchoch(libGAP_CTR('G'), fid);
                    break;
                }
                for ( r = yank; *r != '\0' && *r != '\n'; ++r ) {
                    ch2 = *r;
                    for ( q = p; ch2; ++q ) {
                        ch3 = *q; *q = ch2; ch2 = ch3;
                    }
                    *q = '\0'; ++p;
                }
                break;

            case libGAP_Esc('U'): /* uppercase word                               */
            case libGAP_Esc('u'):
                if ( *p != '\0' ) do {
                    if ('a' <= *p && *p <= 'z')  *p = *p + 'A' - 'a';
                    ++p;
                } while ( *p!='\0' && (libGAP_IS_SEP(*(p-1)) || !libGAP_IS_SEP(*p)));
                break;

            case libGAP_Esc('C'): /* capitalize word                              */
            case libGAP_Esc('c'):
                while ( *p!='\0' && libGAP_IS_SEP(*p) )  ++p;
                if ( 'a' <= *p && *p <= 'z' )  *p = *p + 'A'-'a';
                if ( *p != '\0' ) ++p;
                /* lowercase rest of the word                              */

            case libGAP_Esc('L'): /* lowercase word                               */
            case libGAP_Esc('l'):
                if ( *p != '\0' ) do {
                    if ('A' <= *p && *p <= 'Z')  *p = *p + 'a' - 'A';
                    ++p;
                } while ( *p!='\0' && (libGAP_IS_SEP(*(p-1)) || !libGAP_IS_SEP(*p)));
                break;

            case libGAP_Esc(libGAP_CTR('L')): /* repaint input line                      */
                libGAP_syEchoch('\n',fid);
                for ( q = libGAP_syPrompt; q < libGAP_syPrompt+libGAP_syNrchar; ++q )
                    libGAP_syEchoch( *q, fid );
                for ( q = old; q < old+sizeof(old); ++q )  *q = ' ';
                oldc = 0;
                break;

            case EOF:     /* end of file on input                          */
                break;

            case libGAP_CTR('M'): /* (same as '\r', '\n') append \n and exit      */
            case libGAP_CTR('J'):
                while ( *p != '\0' )  ++p;
                *p++ = '\n'; *p = '\0';
                rep = 0;
                break;

            case libGAP_CTR('O'): /* accept line, perform '<ctr>-N' next time     */
                while ( *p != '\0' )  ++p;
                *p++ = '\n'; *p = '\0';
                libGAP_syCTRO = 2 * rep + 1;
                rep = 0;
                break;

            case libGAP_CTR('I'): /* try to complete the identifier before dot    */
                if ( p == line || libGAP_IS_SEP(p[-1]) ) {
                  /* If we don't have an identifier to complete, insert a tab */
                    ch2 = ch & 0xff;
                    for ( q = p; ch2; ++q ) {
                        ch3 = *q; *q = ch2; ch2 = ch3;
                    }
                    *q = '\0'; ++p;
                }
                else {
  /* Here is actually a bug, because it is not checked if the results
     leaves 'line' shorter than 'length'. But we ignore this problem
     assuming that interactive input lines are much shorter than
     32768 characters.                                                       */

                  /* Locate in q the current identifier */
                    if ( (q = p) > line ) do {
                        --q;
                    } while ( q>line && (!libGAP_IS_SEP(*(q-1)) || libGAP_IS_SEP(*q)));

                    /* determine if the thing immediately before the
                       current identifier is a . */
                    rn = (line < q && *(q-1) == '.'
                                   && (line == q-1 || *(q-2) != '.'));

                    /* Copy the current identifier into buffer */
                    r = buffer;  s = q;
                    while ( s < p )  *r++ = *s++;
                    *r = '\0';

                    if ( (rn ? libGAP_iscomplete_rnam( buffer, p-q )
                          : libGAP_iscomplete_gvar( buffer, p-q )) ) {
                      /* Complete already, just beep for single tab */
                      if ( last != libGAP_CTR('I') )
                        libGAP_syEchoch( libGAP_CTR('G'), fid );
                      else {

                        /* Double tab after a complete identifier
                           print list of completions */
                        libGAP_syWinPut( fid, "@c", "" );
                        libGAP_syEchos( "\n    ", fid );
                        libGAP_syEchos( buffer, fid );
                        while ( (rn ? libGAP_completion_rnam( buffer, p-q )
                                 : libGAP_completion_gvar( buffer, p-q )) ) {
                          libGAP_syEchos( "\n    ", fid );
                          libGAP_syEchos( buffer, fid );
                        }
                        libGAP_syEchos( "\n", fid );

                        /* Reprint the prompt and input line so far */
                        for ( q=libGAP_syPrompt; q<libGAP_syPrompt+libGAP_syNrchar; ++q )
                          libGAP_syEchoch( *q, fid );
                        for ( q = old; q < old+sizeof(old); ++q )
                          *q = ' ';
                        oldc = 0;
                        libGAP_syWinPut( fid, (fid == 0 ? "@i" : "@e"), "" );
                      }
                    }
                    else if ( (rn ? ! libGAP_completion_rnam( buffer, p-q )
                                  : ! libGAP_completion_gvar( buffer, p-q )) ) {

                      /* Not complete, and there are no completions */
                        if ( last != libGAP_CTR('I') )

                          /* beep after 1 tab */
                            libGAP_syEchoch( libGAP_CTR('G'), fid );
                        else {

                          /* print a message otherwise */
                          libGAP_syWinPut( fid, "@c", "" );
                          libGAP_syEchos("\n    identifier has no completions\n",
                                  fid);
                          for ( q=libGAP_syPrompt; q<libGAP_syPrompt+libGAP_syNrchar; ++q )
                            libGAP_syEchoch( *q, fid );
                          for ( q = old; q < old+sizeof(old); ++q )
                            *q = ' ';
                          oldc = 0;
                          libGAP_syWinPut( fid, (fid == 0 ? "@i" : "@e"), "" );
                        }
                    }
                    else {

                      /*not complete and we have a completion. Now we have to
                        find the longest common prefix of all the completions*/

                        t = p;

                      /* Insert the necessary part of the current completion */
                        for ( s = buffer+(p-q); *s != '\0'; s++ ) {

                      /* Insert character from buffer into the line, I think */
                            ch2 = *s;
                            for ( r = p; ch2; r++ ) {
                                ch3 = *r; *r = ch2; ch2 = ch3;
                            }
                            *r = '\0'; p++;
                        }

                        /* Now we work through the alternative
                           completions reducing p, each time to point
                           just after the longest common stem t
                           meanwhile still points to the place where
                           we started this batch of completion, so if
                           p gets down to t, we have nothing
                           unambiguous to add */

                        while ( t < p
                             && (rn ? libGAP_completion_rnam( buffer, t-q )
                                    : libGAP_completion_gvar( buffer, t-q )) ) {

                          /* check the length of common prefix */
                            r = t;  s = buffer+(t-q);
                            while ( r < p && *r == *s ) {
                                r++; s++;
                            }
                            s = p;  p = r;

                            /* Now close up over the part of the
                               completion which turned out to be
                               ambiguous */
                            while ( *s != '\0' )  *r++ = *s++;
                            *r = '\0';
                        }

                        /* OK, now we have done the largest possible completion.
                           If it was nothing then we can't complete. Deal appropriately */
                        if ( t == p ) {
                            if ( last != libGAP_CTR('I') )
                                libGAP_syEchoch( libGAP_CTR('G'), fid );
                            else {
                                libGAP_syWinPut( fid, "@c", "" );
                                buffer[t-q] = '\0';
                                while (
                                  (rn ? libGAP_completion_rnam( buffer, t-q )
                                      : libGAP_completion_gvar( buffer, t-q )) ) {
                                    libGAP_syEchos( "\n    ", fid );
                                    libGAP_syEchos( buffer, fid );
                                }
                                libGAP_syEchos( "\n", fid );
                                for ( q=libGAP_syPrompt; q<libGAP_syPrompt+libGAP_syNrchar; ++q )
                                    libGAP_syEchoch( *q, fid );
                                for ( q = old; q < old+sizeof(old); ++q )
                                    *q = ' ';
                                oldc = 0;
                                libGAP_syWinPut( fid, (fid == 0 ? "@i" : "@e"), "");
                            }
                        }

                      /* If we managed to do some completion then we're happy */
                    }
                }
                break;

            default:      /* default, insert normal character              */
                ch2 = ch & 0xff;
                for ( q = p; ch2; ++q ) {
                    ch3 = *q; *q = ch2; ch2 = ch3;
                }
                *q = '\0'; ++p;
                break;

            } /* switch ( ch ) */
          } /* key handler hook */
          last = ch;
        }

        if ( ch==EOF || ch=='\n' || ch=='\r' || ch==libGAP_CTR('O') ) {
            /* if there is a hook for line ends, call it before echoing */
            if ( libGAP_EndLineHook ) libGAP_Call0ArgsInNewReader( libGAP_EndLineHook );
            libGAP_syEchoch('\r',fid);  libGAP_syEchoch('\n',fid);  break;
        }

        /* now update the screen line according to the differences         */
        for ( q = line, r = new, newc = 0; *q != '\0'; ++q ) {
            if ( q == p )  newc = r-new;
            if ( *q==libGAP_CTR('I') )  { do *r++=' '; while ((r-new+libGAP_syNrchar)%8); }
            else if ( *q==0x7F ) { *r++ = '^'; *r++ = '?'; }
            else if ( /* '\0'<=*q  && */*q<' '  ) { *r++ = '^'; *r++ = *q+'@'; }
            else if ( ' ' <=*q && *q<0x7F ) { *r++ = *q; }
            else {
                *r++ = '\\';                 *r++ = '0'+*(libGAP_UChar*)q/64%4;
                *r++ = '0'+*(libGAP_UChar*)q/8 %8;  *r++ = '0'+*(libGAP_UChar*)q   %8;
            }
            if ( r >= new+libGAP_SyNrCols-libGAP_syNrchar-2 ) {
                if ( q >= p ) { q++; break; }
                new[0] = '$';   new[1] = r[-5]; new[2] = r[-4];
                new[3] = r[-3]; new[4] = r[-2]; new[5] = r[-1];
                r = new+6;
            }
        }
        if ( q == p )  newc = r-new;
        for (      ; r < new+sizeof(new); ++r )  *r = ' ';
        if ( q[0] != '\0' && q[1] != '\0' )
            new[libGAP_SyNrCols-libGAP_syNrchar-2] = '$';
        else if ( q[1] == '\0' && ' ' <= *q && *q < 0x7F )
            new[libGAP_SyNrCols-libGAP_syNrchar-2] = *q;
        else if ( q[1] == '\0' && q[0] != '\0' )
            new[libGAP_SyNrCols-libGAP_syNrchar-2] = '$';
        for ( q = old, r = new; r < new+sizeof(new); ++r, ++q ) {
            if ( *q == *r )  continue;
            while (oldc<(q-old)) { libGAP_syEchoch(old[oldc],fid);  ++oldc; }
            while (oldc>(q-old)) { libGAP_syEchoch('\b',fid);       --oldc; }
            *q = *r;  libGAP_syEchoch( *q, fid ); ++oldc;
        }
        while ( oldc < newc ) { libGAP_syEchoch(old[oldc],fid);  ++oldc; }
        while ( oldc > newc ) { libGAP_syEchoch('\b',fid);       --oldc; }


    }

    if (line[1] != '\0') {
      /* Now we put the new string into the history,
         we use key handler with key 0 to update the command line history */
        libGAP_C_NEW_STRING_DYN(linestr,line);
        args = libGAP_NEW_PLIST(libGAP_T_PLIST, 5);
        libGAP_SET_LEN_PLIST(args, 5);
        libGAP_SET_ELM_PLIST(args,1,linestr);
        libGAP_SET_ELM_PLIST(args,2,libGAP_INTOBJ_INT(0));
        libGAP_SET_ELM_PLIST(args,3,libGAP_INTOBJ_INT(1));
        libGAP_SET_ELM_PLIST(args,4,libGAP_INTOBJ_INT(length));
        libGAP_SET_ELM_PLIST(args,5,linestr);
        libGAP_Call1ArgsInNewReader(libGAP_LineEditKeyHandler, args);
    }

    /* send the whole line (unclipped) to the window handler               */
    libGAP_syWinPut( fid, (*line != '\0' ? "@r" : "@x"), line );

    /* switch back to cooked mode                                          */
    if ( libGAP_SyLineEdit == 1 )
        libGAP_syStopraw(fid);

    /* start the clock again                                               */
    libGAP_SyStartTime += libGAP_SyTime() - libGAP_SyStopTime;

    /* return the line (or '0' at end-of-file)                             */
    if ( *line == '\0' )
        return (libGAP_Char*)0;
    return line;

#if HAVE_LIBREADLINE
}
#endif
}

libGAP_Char * libGAP_SyFgets (
    libGAP_Char *              line,
    libGAP_UInt                length,
    libGAP_Int                 fid)
{
  if(fid!=0 && fid!=2) {
    // not stdin/stderr; probably file IO. Do the standard thing.
    // printf("SyFgets fid=%i\n", fid);
    return libGAP_syFgets( line, length, fid, 1);
  }
  return libgap_get_input(line, length);
  // return syFgets( line, length, fid, 1);
}


libGAP_Char *libGAP_SyFgetsSemiBlock (
    libGAP_Char *              line,
    libGAP_UInt                length,
    libGAP_Int                 fid)
{
  return libGAP_syFgets( line, length, fid, 0);
}

libGAP_Obj libGAP_FuncEchoLine(libGAP_Obj self, libGAP_Obj line, libGAP_Obj len, libGAP_Obj off, libGAP_Obj pos, libGAP_Obj gfid) {
  libGAP_Int i, clen, coff, cpos, fid;
  libGAP_Char *ptr;
  clen = libGAP_INT_INTOBJ(len);
  coff = libGAP_INT_INTOBJ(off);
  ptr = libGAP_CSTR_STRING(line);
  cpos = libGAP_INT_INTOBJ(pos);
  fid = libGAP_INT_INTOBJ(gfid);
  for (i=0; i < coff; libGAP_syEchoch('\b', fid), i++);
  for (i=0; i < clen; i++) {
    libGAP_syEchoch(ptr[i], fid);
  }
  for (; cpos<0; libGAP_syEchoch('\b', fid), cpos++);
  for (; cpos>0; libGAP_syEchoch('\6', fid), cpos--);
  return (libGAP_Obj)0;
}



/****************************************************************************
**

*F * * * * * * * * * * * * system error messages  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  SyLastErrorNo . . . . . . . . . . . . . . . . . . . . . last error number
*/
libGAP_Int libGAP_SyLastErrorNo;


/****************************************************************************
**
*V  SyLastErrorMessage  . . . . . . . . . . . . . . . . .  last error message
*/
libGAP_Char libGAP_SyLastErrorMessage [ 1024 ];


/****************************************************************************
**
*F  SyClearErrorNo()  . . . . . . . . . . . . . . . . .  clear error messages
*/

void libGAP_SyClearErrorNo ( void )
{
    errno = 0;
    libGAP_SyLastErrorNo = 0;
    libGAP_strxcpy( libGAP_SyLastErrorMessage, "no error", sizeof(libGAP_SyLastErrorMessage) );
}


/****************************************************************************
**
*F  SySetErrorNo()  . . . . . . . . . . . . . . . . . . . . set error message
*/

void libGAP_SySetErrorNo ( void )
{
    const libGAP_Char *        err;

    if ( errno != 0 ) {
        libGAP_SyLastErrorNo = errno;
        err = strerror(errno);
        libGAP_strxcpy( libGAP_SyLastErrorMessage, err, sizeof(libGAP_SyLastErrorMessage) );
    }
    else {
        libGAP_SyClearErrorNo();
    }
}

/****************************************************************************
**


*F * * * * * * * * * * * * * file and execution * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  SyExecuteProcess( <dir>, <prg>, <in>, <out>, <args> ) . . . . new process
**
**  Start  <prg> in  directory <dir>  with  standard input connected to <in>,
**  standard  output  connected to <out>   and arguments.  No  path search is
**  performed, the return  value of the process  is returned if the operation
**  system supports such a concept.
*/


/****************************************************************************
**
*f  SyExecuteProcess( <dir>, <prg>, <in>, <out>, <args> )
*/
#if HAVE_FORK || HAVE_VFORK

#if HAVE_UNION_WAIT
# include <sys/wait.h>
#else
# include <sys/types.h>
# if HAVE_SYS_WAIT_H
#  include <sys/wait.h>
# endif
# ifndef WEXITSTATUS
#  define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
# endif
# ifndef WIFEXITED
#  define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
# endif
#endif

#include "libgap_internal.h"

void libGAP_NullSignalHandler(int scratch) {}

#if SYS_IS_CYGWIN32

libGAP_UInt libGAP_SyExecuteProcess (
    libGAP_Char *                  dir,
    libGAP_Char *                  prg,
    libGAP_Int                     in,
    libGAP_Int                     out,
    libGAP_Char *                  args[] )
{
    int savestdin, savestdout;
    libGAP_Int tin, tout;
    int res;

    /* change the working directory                                    */
    if ( chdir(dir) == -1 ) return -1;

    /* if <in> is -1 open "/dev/null"                                  */
    if ( in == -1 ) {
        tin = open( "/dev/null", O_RDONLY );
        if ( tin == -1 ) return -1;
    } else tin = libGAP_SyFileno(in);

    /* if <out> is -1 open "/dev/null"                                 */
    if ( out == -1 ) {
        tout = open( "/dev/null", O_WRONLY );
        if ( tout == -1 ) {
            if (in == -1) close(tin);
            return -1;
        }
    } else tout = libGAP_SyFileno(out);

    /* set standard input to <in>, standard output to <out>            */
    savestdin = -1;   /* Just to please the compiler */
    if ( tin != 0 ) {
        savestdin = dup(0);
        if (savestdin == -1 || dup2(tin,0) == -1) {
            if (out == -1) close(tout);
            if (in == -1) close(tin);
            return -1;
        }
        fcntl( 0, F_SETFD, 0 );
    }

    if ( tout != 1 ) {
        savestdout = dup(1);
        if (savestdout == -1 || dup2( tout, 1 ) == -1) {
            if (tin != 0) {
                close(0);
                dup2(savestdin,0);
                close(savestdin);
            }
            if (out == -1) close(tout);
            if (in == -1) close(tin);
            return -1;
        }
        fcntl( 1, F_SETFD, 0 );
    }

    /* now try to execute the program                                  */
    res = spawnve( _P_WAIT, prg, (const char * const *) args, 
                                 (const char * const *) environ );

    /* Now repair the open file descriptors: */
    if (tout != 1) {
        close(1);
        dup2(savestdout,1);
        close(savestdout);
    }
    if (tin != 0) {
        close(0);
        dup2(savestdin,0);
        close(savestdin);
    }
    if (out == -1) close(tout);
    if (in == -1) close(tin);

    /* Report result: */
    if (res < 0) return -1;
    return WEXITSTATUS(res);
}
    
#else

libGAP_UInt libGAP_SyExecuteProcess (
    libGAP_Char *                  dir,
    libGAP_Char *                  prg,
    libGAP_Int                     in,
    libGAP_Int                     out,
    libGAP_Char *                  args[] )
{
    pid_t                   pid;                    /* process id          */
    pid_t                   wait_pid;
#if HAVE_UNION_WAIT
    union wait              status;                 /* non POSIX           */
#else
    int                     status;                 /* do not use `Int'    */
#endif
    libGAP_Int                     tin;                    /* temp in             */
    libGAP_Int                     tout;                   /* temp out            */
    libGAP_sig_handler_t           *func;
    libGAP_sig_handler_t           *func2;

#if !HAVE_WAITPID
    struct rusage           usage;
#endif


    /* turn off the SIGCHLD handling, so that we can be sure to collect this child
       `After that, we call the old signal handler, in case any other children have died in the
       meantime. This resets the handler */

#ifdef LIBGAP_SIGNALS
    func2 = signal( SIGCHLD, SIG_DFL );

    /* This may return SIG_DFL (0x0) or SIG_IGN (0x1) if the previous handler
     * was set to the default or 'ignore'. In these cases (or if SIG_ERR is
     * returned), just use a null signal hander - the default on most systems
     * is to do nothing */
    if(func2 == SIG_ERR || func2 == SIG_DFL || func2 == SIG_IGN)
      func2 = &libGAP_NullSignalHandler;
#endif

    /* clone the process                                                   */
    pid = vfork();
    if ( pid == -1 ) {
        return -1;
    }

    /* we are the parent                                                   */
    if ( pid != 0 ) {

        /* ignore a CTRL-C                                                 */
#ifdef LIBGAP_SIGNALS
        func = signal( SIGINT, SIG_IGN );
#endif
        /* wait for some action                                            */
#if HAVE_WAITPID
        wait_pid = waitpid( pid, &status, 0 );
#else
        wait_pid = wait4( pid, &status, 0, &usage );
#endif
        if ( wait_pid == -1 ) {
#ifdef LIBGAP_SIGNALS
            signal( SIGINT, func );
            (*func2)(SIGCHLD);
#endif
            return -1;
        }

        if ( WIFSIGNALED(status) ) {
#ifdef LIBGAP_SIGNALS
            signal( SIGINT, func );
            (*func2)(SIGCHLD);
#endif
            return -1;
        }
#ifdef LIBGAP_SIGNALS
        signal( SIGINT, func );
        (*func2)(SIGCHLD);
#endif
        return WEXITSTATUS(status);
    }

    /* we are the child                                                    */
    else {

        /* change the working directory                                    */
        if ( chdir(dir) == -1 ) {
            _exit(-1);
        }

        /* if <in> is -1 open "/dev/null"                                  */
        if ( in == -1 ) {
            tin = open( "/dev/null", O_RDONLY );
            if ( tin == -1 ) {
                _exit(-1);
            }
        }
        else {
            tin = libGAP_SyFileno(in);
        }

        /* if <out> is -1 open "/dev/null"                                 */
        if ( out == -1 ) {
            tout = open( "/dev/null", O_WRONLY );
            if ( tout == -1 ) {
                _exit(-1);
            }
        }
        else {
            tout = libGAP_SyFileno(out);
        }

        /* set standard input to <in>, standard output to <out>            */
        if ( tin != 0 ) {
            if ( dup2( tin, 0 ) == -1 ) {
                _exit(-1);
            }
        }
        fcntl( 0, F_SETFD, 0 );

        if ( tout != 1 ) {
            if ( dup2( tout, 1 ) == -1 ) {
                _exit(-1);
            }
        }
        fcntl( 1, F_SETFD, 0 );

        /* now try to execute the program                                  */
        execve( prg, args, environ );
        _exit(-1);
    }

    /* this should not happen                                              */
    return -1;
}
#endif

#endif


/****************************************************************************
**

*F  SyIsExistingFile( <name> )  . . . . . . . . . . . does file <name> exists
**
**  'SyIsExistingFile' returns 1 if the  file <name> exists and 0  otherwise.
**  It does not check if the file is readable, writable or excuteable. <name>
**  is a system dependent description of the file.
*/


/****************************************************************************
**
*f  SyIsExistingFile( <name> )  . . . . . . . . . . . . . . .  using `access'
*/
#if HAVE_ACCESS

libGAP_Int libGAP_SyIsExistingFile ( const libGAP_Char * name )
{
    libGAP_Int         res;

    libGAP_SyClearErrorNo();
    res = access( name, F_OK );
    if ( res == -1 ) {
        libGAP_SySetErrorNo();
    }
    return res;
}

#endif

/****************************************************************************
**
*F  SyIsReadableFile( <name> )  . . . . . . . . . . . is file <name> readable
**
**  'SyIsReadableFile'   returns 1  if the   file  <name> is   readable and 0
**  otherwise. <name> is a system dependent description of the file.
*/


/****************************************************************************
**
*f  SyIsReadableFile( <name> )  . . . . . . . . . . . . . . .  using `access'
*/
#if HAVE_ACCESS

libGAP_Int libGAP_SyIsReadableFile ( const libGAP_Char * name )
{
    libGAP_Int         res;
#ifdef HAVE_POPEN 
    libGAP_Char        xname[1024];
#endif

    libGAP_SyClearErrorNo();
    res = access( name, R_OK );
    if ( res == -1 ) {
      /* if there is popen then we might be able to read the file via gunzip */

#ifdef HAVE_POPEN 
      /* beware of buffer overflows */
      if ( libGAP_strlcpy(xname, name, sizeof(xname)) < sizeof(xname) &&
            libGAP_strlcat(xname, ".gz", sizeof(xname))  < sizeof(xname) ) {
        res = access(xname, R_OK);
      } 

      if (res == -1)
#endif
        libGAP_SySetErrorNo();
    }
    return res;
}

#endif


/****************************************************************************
**
*F  SyIsWritableFile( <name> )  . . . . . . . . . is the file <name> writable
**
**  'SyIsWritableFile'   returns  1  if  the  file  <name> is  writable and 0
**  otherwise. <name> is a system dependent description of the file.
*/


/****************************************************************************
**
*f  SyIsWritableFile( <name> )  . . . . . . . . . . . . . . .  using `access'
*/
#if HAVE_ACCESS

libGAP_Int libGAP_SyIsWritableFile ( const libGAP_Char * name )
{
    libGAP_Int         res;

    libGAP_SyClearErrorNo();
    res = access( name, W_OK );
    if ( res == -1 ) {
        libGAP_SySetErrorNo();
    }
    return res;
}

#endif


/****************************************************************************
**
*F  SyIsExecutableFile( <name> )  . . . . . . . . . is file <name> executable
**
**  'SyIsExecutableFile' returns 1 if the  file <name>  is  executable and  0
**  otherwise. <name> is a system dependent description of the file.
*/


/****************************************************************************
**
*f  SyIsExecutableFile( <name> )  . . . . . . . . . . . . . .  using `access'
*/
#if HAVE_ACCESS

libGAP_Int libGAP_SyIsExecutableFile ( const libGAP_Char * name )
{
    libGAP_Int         res;

    libGAP_SyClearErrorNo();
    res = access( name, X_OK );
    if ( res == -1 ) {
        libGAP_SySetErrorNo();
    }
    return res;
}

#endif


/****************************************************************************
**
*F  SyIsDirectoryPath( <name> ) . . . . . . . . .  is file <name> a directory
**
**  'SyIsDirectoryPath' returns 1 if the  file <name>  is a directory  and  0
**  otherwise. <name> is a system dependent description of the file.
*/


/****************************************************************************
**
*f  SyIsDirectoryPath( <name> ) . . . . . . . . . . . . . . . .  using `stat'
*/
#if HAVE_STAT

#include <sys/stat.h>

libGAP_Int libGAP_SyIsDirectoryPath ( const libGAP_Char * name )
{
    struct stat     buf;                /* buffer for `stat'               */

    libGAP_SyClearErrorNo();
    if ( stat( name, &buf ) == -1 ) {
        libGAP_SySetErrorNo();
        return -1;
    }
    return S_ISDIR(buf.st_mode) ? 0 : -1;
}

#endif


/****************************************************************************
**
*F  SyRemoveFile( <name> )  . . . . . . . . . . . . . . .  remove file <name>
*/


/****************************************************************************
**
*f  SyRemoveFile( <name> )  . . . . . . . . . . . . . . . . .  using `unlink'
*/
#if HAVE_UNLINK

libGAP_Int libGAP_SyRemoveFile ( const libGAP_Char * name )
{
    libGAP_Int res;
    libGAP_SyClearErrorNo();
    res = unlink(name);
    if (res == -1)
       libGAP_SySetErrorNo();
    return res;
}

#endif

/****************************************************************************
**
*f  SyMkdir( <name> )  . . . . . . . . . . . . . . . . create directory
**  with users umask permissions.
*/
#if HAVE_MKDIR

libGAP_Int libGAP_SyMkdir ( const libGAP_Char * name )
{
    libGAP_Int res;
    libGAP_SyClearErrorNo();
    res = mkdir(name, 0777);
    if (res == -1)
       libGAP_SySetErrorNo();
    return res;
}

#endif

/****************************************************************************
**
*f  SyRemoveDir( <name> )  . . . . . . . . . . . . . . . . .  using `rmdir'
*/
#if HAVE_RMDIR

libGAP_Int libGAP_SyRmdir ( const libGAP_Char * name )
{
    libGAP_Int res;
    libGAP_SyClearErrorNo();
    res = rmdir(name);
    if (res == -1)
       libGAP_SySetErrorNo();
    return res;
}

#endif

/****************************************************************************
**
*F  SyIsDir( <name> )  . . . . . . . . . . . . .  test if something is a dir
**
**  Returns 'F' for a regular file, 'L' for a symbolic link and 'D'
**  for a real directory, 'C' for a character device, 'B' for a block
**  device 'P' for a FIFO (named pipe) and 'S' for a socket.
*/
#ifdef HAVE_LSTAT

#include <sys/types.h>
#include <sys/stat.h>

libGAP_Obj libGAP_SyIsDir ( const libGAP_Char * name )
{
  libGAP_Int res;
  struct stat ourlstatbuf;

  res = lstat(name,&ourlstatbuf);
  if (res < 0) {
    libGAP_SySetErrorNo();
    return libGAP_Fail;
  }
  if      (S_ISREG(ourlstatbuf.st_mode)) return libGAP_ObjsChar['F'];
  else if (S_ISDIR(ourlstatbuf.st_mode)) return libGAP_ObjsChar['D'];
  else if (S_ISLNK(ourlstatbuf.st_mode)) return libGAP_ObjsChar['L'];
#ifdef S_ISCHR
  else if (S_ISCHR(ourlstatbuf.st_mode)) return libGAP_ObjsChar['C'];
#endif
#ifdef S_ISBLK
  else if (S_ISBLK(ourlstatbuf.st_mode)) return libGAP_ObjsChar['B'];
#endif
#ifdef S_ISFIFO
  else if (S_ISFIFO(ourlstatbuf.st_mode)) return libGAP_ObjsChar['P'];
#endif
#ifdef S_ISSOCK
  else if (S_ISSOCK(ourlstatbuf.st_mode)) return libGAP_ObjsChar['S'];
#endif
  else return libGAP_ObjsChar['?'];
}
#else
libGAP_Obj libGAP_SyIsDir ( const libGAP_Char * name )
{
  return libGAP_ObjsChar['?'];
}
#endif

/****************************************************************************
**
*F  SyFindGapRootFile( <filename>,<buffer> ) . .  find file in system area
*/
libGAP_Char * libGAP_SyFindGapRootFile ( const libGAP_Char * filename, libGAP_Char * result )
{
    libGAP_Int             k;

    for ( k=0;  k<sizeof(libGAP_SyGapRootPaths)/sizeof(libGAP_SyGapRootPaths[0]);  k++ ) {
        if ( libGAP_SyGapRootPaths[k][0] ) {
            result[0] = '\0';
            if (libGAP_strlcpy( result, libGAP_SyGapRootPaths[k], 256 ) >= 256)
                continue;
            if (libGAP_strlcat( result, filename, 256 ) >= 256)
            	continue;
            if ( libGAP_SyIsReadableFile(result) == 0 ) {
                return result;
            }
        }
    }
    return 0;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * * directories  * * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  SyTmpname() . . . . . . . . . . . . . . . . . return a temporary filename
**
**  'SyTmpname' creates and returns  a new temporary name.  Subsequent  calls
**  to 'SyTmpname'  should  produce different  file names  *even* if no files
**  were created.
*/

#if HAVE_MKSTEMP
libGAP_Char *libGAP_SyTmpname ( void )
{
  static char name[1024];
  libGAP_strxcpy(name, "/tmp/gaptempfile.XXXXXX", sizeof(name));
  close(mkstemp(name));
  return name;
}

#else
libGAP_Char * libGAP_SyTmpname ( void )
{
    static libGAP_Char   * base = 0;
    static libGAP_Char     name[1024];
    static libGAP_UInt     count = 0;
    libGAP_Char          * s;
    libGAP_UInt            c;

    libGAP_SyClearErrorNo();
    count++;
    if ( base == 0 ) {
        base = tmpnam( (char*)0 );
    }
    if ( base == 0 ) {
        libGAP_SySetErrorNo();
        return 0;
    }
    if (count == 0) { /* one time in 2^32 */
        strcat( base, "x" ); /* FIXME: BUG, we must not modify the pointer returned by tmpnam */
        count++;
    }
    libGAP_strxcpy( name, base, sizeof(name) );
    libGAP_strxcat( name, ".", sizeof(name) );
    s = name + strlen(name);
    c = count;
    while (c != 0) {
        *s++ = '0' + c % 10;
        c /= 10;
    }
    *s = (libGAP_Char)0;
    return name;
}
#endif


/****************************************************************************
**
*F  SyTmpdir( <hint> )  . . . . . . . . . . . .  return a temporary directory
**
**  'SyTmpdir'  returns the directory   for  a temporary directory.  This  is
**  guaranteed  to be newly  created and empty  immediately after the call to
**  'SyTmpdir'. <hint> should be used by 'SyTmpdir' to  construct the name of
**  the directory (but 'SyTmpdir' is free to use only  a part of <hint>), and
**  must be a string of at most 8 alphanumerical characters.  Under UNIX this
**  would   usually   represent   '/usr/tmp/<hint>_<proc_id>_<cnt>/',   e.g.,
**  '/usr/tmp/guava_17188_1/'.
*/


/****************************************************************************
**
*f  SyTmpdir( <hint> )  . . . . . . . . . . . . . . . . . . . . using `mkdir'
*/

#if SYS_IS_CYGWIN32
#define libGAP_TMPDIR_BASE "/cygdrive/c/WINDOWS/Temp/"
#else
#define libGAP_TMPDIR_BASE "/tmp/"
#endif


#if HAVE_MKDTEMP
libGAP_Char * libGAP_SyTmpdir( const libGAP_Char * hint )
{
  static char name[1024];
  static const char *base = libGAP_TMPDIR_BASE;
  libGAP_strxcpy(name, base, sizeof(name));
  if (hint)
    libGAP_strxcat(name, hint, sizeof(name));
  else
    libGAP_strxcat(name, "gaptempdir", sizeof(name));
  libGAP_strxcat(name, "XXXXXX", sizeof(name));
  return mkdtemp(name);
}
#else
#if HAVE_MKDIR

libGAP_Char * libGAP_SyTmpdir ( const libGAP_Char * hint )
{
    libGAP_Char *      tmp;
    int         res;                    /* result of `mkdir'               */

    /* for the moment ignore <hint>                                        */
    tmp = libGAP_SyTmpname();
    if ( tmp == 0 )
        return 0;

    /* create a new directory                                              */
    unlink( tmp );
    res = mkdir( tmp, 0777 );
    if ( res == -1 ) {
        libGAP_SySetErrorNo();
        return 0;
    }

    return tmp;
}


#endif
#endif



/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "CrcString", 1, "string",
      libGAP_FuncCrcString, "src/sysfiles.c:FuncCrcString" },

    { "EchoLine", 5, "line, len, off, pos, fid",
      libGAP_FuncEchoLine, "src/sysfiles.c:FuncEchoLine" },

#if HAVE_LIBREADLINE
    { "BINDKEYSTOGAPHANDLER", 1, "keyseq",
       libGAP_FuncBINDKEYSTOGAPHANDLER, "src/sysfiles.c:FuncBINDKEYSTOGAPHANDLER" },

    { "BINDKEYSTOMACRO", 2, "keyseq, macro",
       libGAP_FuncBINDKEYSTOMACRO, "src/sysfiles.c:FuncBINDKEYSTOMACRO" },

    { "READLINEINITLINE", 1, "line",
       libGAP_FuncREADLINEINITLINE, "src/sysfiles.c:FuncREADLINEINITLINE" },
#endif


    { 0 } };


/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/* NB Should probably do some checks preSave for open files etc and refuse to save
   if any are found */

/****************************************************************************
**
*F  postResore( <module> ) . . . . . . .re-initialise library data structures
*/

static libGAP_Int libGAP_postRestore (
    libGAP_StructInitInfo *    libGAP_module )
{

    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  InitKernel( <module> ) . . . . . . .  initialise kernel data structures
*/

static libGAP_Int libGAP_InitKernel(
      libGAP_StructInitInfo * libGAP_module )
{
  libGAP_SyInitAlarm();
  
  /* init filters and functions                                          */
  libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

  /* line edit key handler from library                                  */
  libGAP_ImportGVarFromLibrary("GAPInfo", &libGAP_GAPInfo);
  libGAP_ImportFuncFromLibrary("LineEditKeyHandler", &libGAP_LineEditKeyHandler);
  libGAP_ImportGVarFromLibrary("LineEditKeyHandlers", &libGAP_LineEditKeyHandlers);
  /* return success                                                      */
  return 0;

}

/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/

static libGAP_Int libGAP_InitLibrary(
      libGAP_StructInitInfo * libGAP_module )
{
  /* init filters and functions                                          */
  libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

  return libGAP_postRestore( libGAP_module );
}

/****************************************************************************
**
*F  InitInfoSysFiles()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "sysfiles",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_postRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoSysFiles ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  sysfiles.h  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
