/****************************************************************************
**
*W  stats.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the statements package.
**
**  The  statements package  is the  part  of  the interpreter that  executes
**  statements for their effects and prints statements.
*/
#include        "system.h"              /* system dependent part           */


#include        "sysfiles.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "bool.h"                /* booleans                        */

#include        "code.h"                /* coder                           */
#include        "exprs.h"               /* expressions                     */

#include        "intrprtr.h"            /* interpreter                     */

#include        "ariths.h"              /* basic arithmetic                */

#include        "stats.h"               /* statements                      */

#include        "profile.h"             /* installing methods              */

#include        <assert.h>

#include	"tls.h"
#include	"thread.h"

#include        "vars.h"                /* variables                       */

#include        "profile.h"             /* visit statements for profiling  */

/****************************************************************************
**

*F  EXEC_STAT(<stat>) . . . . . . . . . . . . . . . . . . execute a statement
**
**  'EXEC_STAT' executes the statement <stat>.
**
**  If   this  causes   the  execution  of   a  return-value-statement,  then
**  'EXEC_STAT' returns 1, and the return value is stored in 'ReturnObjStat'.
**  If this causes the execution of a return-void-statement, then 'EXEC_STAT'
**  returns 2.  If  this causes execution  of a break-statement (which cannot
**  happen if <stat> is the body of a  function), then 'EXEC_STAT' returns 4.
**  Similarly, for a continue-statement, EXEC_STAT returns 8
**  Otherwise 'EXEC_STAT' returns 0.
**
**  'EXEC_STAT'  causes  the  execution  of  <stat>  by dispatching   to  the
**  executor, i.e., to the  function that executes statements  of the type of
**  <stat>.
**
**  'EXEC_STAT' is defined in the declaration part of this package as follows:
**
#define EXEC_STAT(stat) ( (*TLS(CurrExecStatFuncs)[ TNUM_STAT(stat) ]) ( stat ) )
*/


/****************************************************************************
**
*V  ExecStatFuncs[<type>] . . . . . .  executor for statements of type <type>
**
**  'ExecStatFuncs' is   the dispatch table  that contains  for every type of
**  statements a pointer to the executor  for statements of  this type, i.e.,
**  the function  that should  be  called  if a  statement   of that type  is
**  executed.
*/
libGAP_UInt            (* libGAP_ExecStatFuncs[256]) ( libGAP_Stat stat );


/****************************************************************************
**
*V  CurrStat  . . . . . . . . . . . . . . . . .  currently executed statement
**
**  'CurrStat'  is the statement that  is currently being executed.  The sole
**  purpose of 'CurrStat' is to make it possible to  point to the location in
**  case an error is signalled.
*/
libGAP_Stat            libGAP_CurrStat;


/****************************************************************************
**
*V  ReturnObjStat . . . . . . . . . . . . . . . .  result of return-statement
**
**  'ReturnObjStat'  is   the result of the   return-statement  that was last
**  executed.  It is set  in  'ExecReturnObj' and  used in the  handlers that
**  interpret functions.
*/
libGAP_Obj             libGAP_ReturnObjStat;


/****************************************************************************
**

*F  ExecUnknownStat(<stat>) . . . . . executor for statements of unknown type
**
**  'ExecUnknownStat' is the executor that is called if an attempt is made to
**  execute a statement <stat> of an unknown type.  It  signals an error.  If
**  this  is  ever  called, then   GAP is   in  serious  trouble, such as  an
**  overwritten type field of a statement.
*/
libGAP_UInt            libGAP_ExecUnknownStat (
    libGAP_Stat                stat )
{
    libGAP_Pr(
        "Panic: tried to execute a statement of unknown type '%d'\n",
        (libGAP_Int)libGAP_TNUM_STAT(stat), 0L );
    return 0;
}

/****************************************************************************
**
*F  UInt HaveInterrupt() . . . . . . . . check for user interrupts
**
*/

#define libGAP_HaveInterrupt()   libGAP_SyIsIntr()


/****************************************************************************
**
*F  ExecSeqStat(<stat>) . . . . . . . . . . . .  execute a statement sequence
**
**  'ExecSeqStat' executes the statement sequence <stat>.
**
**  This is done  by  executing  the  statements one  after  another.  If   a
**  leave-statement  ('break' or  'return')  is executed  inside  one  of the
**  statements, then the execution of  the  statement sequence is  terminated
**  and the non-zero leave-value  is returned (to  tell the calling  executor
**  that a leave-statement was executed).  If no leave-statement is executed,
**  then 0 is returned.
**
**  A statement sequence with <n> statements is represented by  a bag of type
**  'T_SEQ_STAT' with  <n> subbags.  The first  is  the  first statement, the
**  second is the second statement, and so on.
*/
libGAP_UInt            libGAP_ExecSeqStat (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_UInt                nr;             /* number of statements            */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the number of statements                                        */
    nr = libGAP_SIZE_STAT( stat ) / sizeof(libGAP_Stat);

    /* loop over the statements                                            */
    for ( i = 1; i <= nr; i++ ) {

        /* execute the <i>-th statement                                    */
        if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[i-1] )) != 0 ) {
            return leave;
        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecSeqStat2 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */

    /* execute the statements                                              */
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[0] )) != 0 ) { return leave; }

    /* execute the last statement                                          */
    return libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[1] );
}

libGAP_UInt            libGAP_ExecSeqStat3 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */

    /* execute the statements                                              */
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[0] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[1] )) != 0 ) { return leave; }

    /* execute the last statement                                          */
    return libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[2] );
}

libGAP_UInt            libGAP_ExecSeqStat4 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */

    /* execute the statements                                              */
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[0] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[1] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[2] )) != 0 ) { return leave; }

    /* execute the last statement                                          */
    return libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[3] );
}

libGAP_UInt            libGAP_ExecSeqStat5 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */

    /* execute the statements                                              */
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[0] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[1] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[2] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[3] )) != 0 ) { return leave; }

    /* execute the last statement                                          */
    return libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[4] );
}

libGAP_UInt            libGAP_ExecSeqStat6 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */

    /* execute the statements                                              */
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[0] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[1] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[2] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[3] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[4] )) != 0 ) { return leave; }

    /* execute the last statement                                          */
    return libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[5] );
}

libGAP_UInt            libGAP_ExecSeqStat7 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */

    /* execute the statements                                              */
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[0] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[1] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[2] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[3] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[4] )) != 0 ) { return leave; }
    if ( (leave = libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[5] )) != 0 ) { return leave; }

    /* execute the last statement                                          */
    return libGAP_EXEC_STAT( libGAP_ADDR_STAT(stat)[6] );
}


/****************************************************************************
**
*F  ExecIf(<stat>)  . . . . . . . . . . . . . . . . . execute an if-statement
**
**  'ExecIf' executes the if-statement <stat>.
**
**  This is done by evaluating the conditions  until one evaluates to 'true',
**  and then executing the corresponding body.  If a leave-statement ('break'
**  or  'return') is executed  inside the  body, then   the execution of  the
**  if-statement is  terminated and the  non-zero leave-value is returned (to
**  tell the  calling executor that a  leave-statement was executed).   If no
**  leave-statement is executed, then 0 is returned.
**
**  An if-statement with <n> branches is represented by  a bag of type 'T_IF'
**  with 2*<n> subbags.  The first subbag is  the first condition, the second
**  subbag is the  first body, the third subbag  is the second condition, the
**  fourth subbag is the second body, and so  on.  If the if-statement has an
**  else-branch, this is represented by a branch without a condition.
*/
libGAP_UInt            libGAP_ExecIf (
    libGAP_Stat                stat )
{
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body;           /* body                            */

    /* if the condition evaluates to 'true', execute the if-branch body    */
    libGAP_SET_BRK_CURR_STAT( stat );
    cond = libGAP_ADDR_STAT(stat)[0];
    if ( libGAP_EVAL_BOOL_EXPR( cond ) != libGAP_False ) {

        /* execute the if-branch body and leave                            */
        body = libGAP_ADDR_STAT(stat)[1];
        return libGAP_EXEC_STAT( body );

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecIfElse (
    libGAP_Stat                stat )
{
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body;           /* body                            */

    /* if the condition evaluates to 'true', execute the if-branch body    */
    libGAP_SET_BRK_CURR_STAT( stat );
    cond = libGAP_ADDR_STAT(stat)[0];
    if ( libGAP_EVAL_BOOL_EXPR( cond ) != libGAP_False ) {

        /* execute the if-branch body and leave                            */
        body = libGAP_ADDR_STAT(stat)[1];
        return libGAP_EXEC_STAT( body );

    }

    /* otherwise execute the else-branch body and leave                    */
    body = libGAP_ADDR_STAT(stat)[3];
    return libGAP_EXEC_STAT( body );
}

libGAP_UInt            libGAP_ExecIfElif (
    libGAP_Stat                stat )
{
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body;           /* body                            */
    libGAP_UInt                nr;             /* number of branches              */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the number of branches                                          */
    nr = libGAP_SIZE_STAT( stat ) / (2*sizeof(libGAP_Stat));

    /* loop over all branches                                              */
    for ( i = 1; i <= nr; i++ ) {

        /* if the condition evaluates to 'true', execute the branch body   */
        libGAP_SET_BRK_CURR_STAT( stat );
        cond = libGAP_ADDR_STAT(stat)[2*(i-1)];
        if ( libGAP_EVAL_BOOL_EXPR( cond ) != libGAP_False ) {

            /* execute the branch body and leave                           */
            body = libGAP_ADDR_STAT(stat)[2*(i-1)+1];
            return libGAP_EXEC_STAT( body );

        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecIfElifElse (
    libGAP_Stat                stat )
{
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body;           /* body                            */
    libGAP_UInt                nr;             /* number of branches              */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the number of branches                                          */
    nr = libGAP_SIZE_STAT( stat ) / (2*sizeof(libGAP_Stat)) - 1;

    /* loop over all branches                                              */
    for ( i = 1; i <= nr; i++ ) {

        /* if the condition evaluates to 'true', execute the branch body   */
        libGAP_SET_BRK_CURR_STAT( stat );
        cond = libGAP_ADDR_STAT(stat)[2*(i-1)];
        if ( libGAP_EVAL_BOOL_EXPR( cond ) != libGAP_False ) {

            /* execute the branch body and leave                           */
            body = libGAP_ADDR_STAT(stat)[2*(i-1)+1];
            return libGAP_EXEC_STAT( body );

        }

    }

    /* otherwise execute the else-branch body and leave                    */
    body = libGAP_ADDR_STAT(stat)[2*(i-1)+1];
    return libGAP_EXEC_STAT( body );
}


/****************************************************************************
**
*F  ExecFor(<stat>) . . . . . . . . . . . . . . . . . . .  execute a for-loop
**
**  'ExecFor' executes the for-loop <stat>.
**
**  This  is   done by   evaluating  the  list-expression, checking  that  it
**  evaluates  to  a list, and   then looping over the   entries in the list,
**  executing the  body for each element  of the list.   If a leave-statement
**  ('break' or 'return') is executed inside the  body, then the execution of
**  the for-loop is terminated and 0 is returned if the leave-statement was a
**  break-statement   or  the   non-zero leave-value   is   returned  if  the
**  leave-statement was a return-statement (to tell the calling executor that
**  a return-statement was  executed).  If  no leave-statement was  executed,
**  then 0 is returned.
**
**  A for-loop with <n> statements  in its body   is represented by a bag  of
**  type 'T_FOR' with <n>+2  subbags.  The first  subbag is an assignment bag
**  for the loop variable, the second subbag  is the list-expression, and the
**  remaining subbags are the statements.
*/
libGAP_Obj             libGAP_ITERATOR;

libGAP_Obj             libGAP_IS_DONE_ITER;

libGAP_Obj             libGAP_NEXT_ITER;

libGAP_Obj             libGAP_STD_ITER;

libGAP_UInt            libGAP_ExecFor (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_UInt                var;            /* variable                        */
    libGAP_UInt                vart;           /* variable type                   */
    libGAP_Obj                 list;           /* list to loop over               */
    libGAP_Obj                 elm;            /* one element of the list         */
    libGAP_Stat                body;           /* body of loop                    */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 nfun, dfun;     /* functions for NextIterator and
                                           IsDoneIterator                  */  

    /* get the variable (initialize them first to please 'lint')           */
    if ( libGAP_IS_REFLVAR( libGAP_ADDR_STAT(stat)[0] ) ) {
        var = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );
        vart = 'l';
    }
    else if ( libGAP_T_REF_LVAR <= libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] )
           && libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) <= libGAP_T_REF_LVAR_16 ) {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'l';
    }
    else if ( libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) == libGAP_T_REF_HVAR ) {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'h';
    }
    else /* if ( TNUM_EXPR( ADDR_STAT(stat)[0] ) == T_REF_GVAR ) */ {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'g';
    }

    /* evaluate the list                                                   */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );

    /* get the body                                                        */
    body = libGAP_ADDR_STAT(stat)[2];

    /* special case for lists                                              */
    if ( libGAP_IS_SMALL_LIST( list ) ) {

        /* loop over the list, skipping unbound entries                    */
        i = 1;
        while ( i <= libGAP_LEN_LIST(list) ) {

            /* get the element and assign it to the variable               */
            elm = libGAP_ELMV0_LIST( list, i );
            i++;
            if ( elm == 0 )  continue;
            if      ( vart == 'l' )  libGAP_ASS_LVAR( var, elm );
            else if ( vart == 'h' )  libGAP_ASS_HVAR( var, elm );
            else if ( vart == 'g' )  libGAP_AssGVar(  var, elm );

#if ! HAVE_SIGNAL
            /* test for an interrupt                                       */
            if ( libGAP_HaveInterrupt() ) {
                libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
            }
#endif

            /* execute the statements in the body                          */
            if ( (leave = libGAP_EXEC_STAT( body )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }

        }

    }

    /* general case                                                        */
    else {

        /* get the iterator                                                */
        list = libGAP_CALL_1ARGS( libGAP_ITERATOR, list );

        if ( libGAP_CALL_1ARGS( libGAP_STD_ITER, list ) == libGAP_True ) {
            /* this can avoid method selection overhead on iterator        */
            dfun = libGAP_ElmPRec( list, libGAP_RNamName("IsDoneIterator") );
            nfun = libGAP_ElmPRec( list, libGAP_RNamName("NextIterator") );
        } else {
            dfun = libGAP_IS_DONE_ITER;
            nfun = libGAP_NEXT_ITER;
        }

        /* loop over the iterator                                          */
        while ( libGAP_CALL_1ARGS( dfun, list ) == libGAP_False ) {

            /* get the element and assign it to the variable               */
            elm = libGAP_CALL_1ARGS( nfun, list );
            if      ( vart == 'l' )  libGAP_ASS_LVAR( var, elm );
            else if ( vart == 'h' )  libGAP_ASS_HVAR( var, elm );
            else if ( vart == 'g' )  libGAP_AssGVar(  var, elm );

#if ! HAVE_SIGNAL
            /* test for an interrupt                                       */
            if ( libGAP_HaveInterrupt() ) {
                libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
            }
#endif

            /* execute the statements in the body                          */
            if ( (leave = libGAP_EXEC_STAT( body )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }

        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecFor2 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_UInt                var;            /* variable                        */
    libGAP_UInt                vart;           /* variable type                   */
    libGAP_Obj                 list;           /* list to loop over               */
    libGAP_Obj                 elm;            /* one element of the list         */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 nfun, dfun;     /* functions for NextIterator and
                                           IsDoneIterator                  */  

    /* get the variable (initialize them first to please 'lint')           */
    if ( libGAP_IS_REFLVAR( libGAP_ADDR_STAT(stat)[0] ) ) {
        var = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );
        vart = 'l';
    }
    else if ( libGAP_T_REF_LVAR <= libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] )
           && libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) <= libGAP_T_REF_LVAR_16 ) {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'l';
    }
    else if ( libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) == libGAP_T_REF_HVAR ) {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'h';
    }
    else /* if ( TNUM_EXPR( ADDR_STAT(stat)[0] ) == T_REF_GVAR ) */ {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'g';
    }

    /* evaluate the list                                                   */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );

    /* get the body                                                        */
    body1 = libGAP_ADDR_STAT(stat)[2];
    body2 = libGAP_ADDR_STAT(stat)[3];

    /* special case for lists                                              */
    if ( libGAP_IS_SMALL_LIST( list ) ) {

        /* loop over the list, skipping unbound entries                    */
        i = 1;
        while ( i <= libGAP_LEN_LIST(list) ) {

            /* get the element and assign it to the variable               */
            elm = libGAP_ELMV0_LIST( list, i );
            i++;
            if ( elm == 0 )  continue;
            if      ( vart == 'l' )  libGAP_ASS_LVAR( var, elm );
            else if ( vart == 'h' )  libGAP_ASS_HVAR( var, elm );
            else if ( vart == 'g' )  libGAP_AssGVar(  var, elm );

#if ! HAVE_SIGNAL
            /* test for an interrupt                                       */
            if ( libGAP_HaveInterrupt() ) {
                libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
            }
#endif

            /* execute the statements in the body                          */
            if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }
            if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }

        }

    }

    /* general case                                                        */
    else {

        /* get the iterator                                                */
        list = libGAP_CALL_1ARGS( libGAP_ITERATOR, list );

        if ( libGAP_CALL_1ARGS( libGAP_STD_ITER, list ) == libGAP_True ) {
            /* this can avoid method selection overhead on iterator        */
            dfun = libGAP_ElmPRec( list, libGAP_RNamName("IsDoneIterator") );
            nfun = libGAP_ElmPRec( list, libGAP_RNamName("NextIterator") );
        } else {
            dfun = libGAP_IS_DONE_ITER;
            nfun = libGAP_NEXT_ITER;
        }

        /* loop over the iterator                                          */
        while ( libGAP_CALL_1ARGS( dfun, list ) == libGAP_False ) {

            /* get the element and assign it to the variable               */
            elm = libGAP_CALL_1ARGS( nfun, list );
            if      ( vart == 'l' )  libGAP_ASS_LVAR( var, elm );
            else if ( vart == 'h' )  libGAP_ASS_HVAR( var, elm );
            else if ( vart == 'g' )  libGAP_AssGVar(  var, elm );

#if ! HAVE_SIGNAL
            /* test for an interrupt                                       */
            if ( libGAP_HaveInterrupt() ) {
                libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
            }
#endif

            /* execute the statements in the body                          */
            if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }
            if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }

        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecFor3 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_UInt                var;            /* variable                        */
    libGAP_UInt                vart;           /* variable type                   */
    libGAP_Obj                 list;           /* list to loop over               */
    libGAP_Obj                 elm;            /* one element of the list         */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */
    libGAP_Stat                body3;          /* third  stat. of body of loop    */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 nfun, dfun;     /* functions for NextIterator and
                                           IsDoneIterator                  */  

    /* get the variable (initialize them first to please 'lint')           */
    if ( libGAP_IS_REFLVAR( libGAP_ADDR_STAT(stat)[0] ) ) {
        var = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );
        vart = 'l';
    }
    else if ( libGAP_T_REF_LVAR <= libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] )
           && libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) <= libGAP_T_REF_LVAR_16 ) {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'l';
    }
    else if ( libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) == libGAP_T_REF_HVAR ) {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'h';
    }
    else /* if ( TNUM_EXPR( ADDR_STAT(stat)[0] ) == T_REF_GVAR ) */ {
        var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
        vart = 'g';
    }

    /* evaluate the list                                                   */
    libGAP_SET_BRK_CURR_STAT( stat );
    list = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[1] );

    /* get the body                                                        */
    body1 = libGAP_ADDR_STAT(stat)[2];
    body2 = libGAP_ADDR_STAT(stat)[3];
    body3 = libGAP_ADDR_STAT(stat)[4];

    /* special case for lists                                              */
    if ( libGAP_IS_SMALL_LIST( list ) ) {

        /* loop over the list, skipping unbound entries                    */
        i = 1;
        while ( i <= libGAP_LEN_LIST(list) ) {

            /* get the element and assign it to the variable               */
            elm = libGAP_ELMV0_LIST( list, i );
            i++;
            if ( elm == 0 )  continue;
            if      ( vart == 'l' )  libGAP_ASS_LVAR( var, elm );
            else if ( vart == 'h' )  libGAP_ASS_HVAR( var, elm );
            else if ( vart == 'g' )  libGAP_AssGVar(  var, elm );

#if ! HAVE_SIGNAL
            /* test for an interrupt                                       */
            if ( libGAP_HaveInterrupt() ) {
                libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
            }
#endif

            /* execute the statements in the body                          */
            if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }
            if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }
            if ( (leave = libGAP_EXEC_STAT( body3 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }


        }

    }

    /* general case                                                        */
    else {

        /* get the iterator                                                */
        list = libGAP_CALL_1ARGS( libGAP_ITERATOR, list );

        if ( libGAP_CALL_1ARGS( libGAP_STD_ITER, list ) == libGAP_True ) {
            /* this can avoid method selection overhead on iterator        */
            dfun = libGAP_ElmPRec( list, libGAP_RNamName("IsDoneIterator") );
            nfun = libGAP_ElmPRec( list, libGAP_RNamName("NextIterator") );
        } else {
            dfun = libGAP_IS_DONE_ITER;
            nfun = libGAP_NEXT_ITER;
        }

        /* loop over the iterator                                          */
        while ( libGAP_CALL_1ARGS( dfun, list ) == libGAP_False ) {

            /* get the element and assign it to the variable               */
            elm = libGAP_CALL_1ARGS( nfun, list );
            if      ( vart == 'l' )  libGAP_ASS_LVAR( var, elm );
            else if ( vart == 'h' )  libGAP_ASS_HVAR( var, elm );
            else if ( vart == 'g' )  libGAP_AssGVar(  var, elm );

#if ! HAVE_SIGNAL
            /* test for an interrupt                                       */
            if ( libGAP_HaveInterrupt() ) {
                libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
            }
#endif

            /* execute the statements in the body                          */
            if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }
            if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }
            if ( (leave = libGAP_EXEC_STAT( body3 )) != 0 ) {
                if (leave == 8) 
                    continue;
                return (leave & 3);
            }


        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecForRange(<stat>)  . . . . . . . . . . . . . . . .  execute a for-loop
**
**  'ExecForRange' executes the  for-loop  <stat>, which is a  for-loop whose
**  loop variable is  a  local variable and  whose  list is a  literal  range
**  expression.
**
**  This  is   done by   evaluating  the  list-expression, checking  that  it
**  evaluates  to  a list, and   then looping over the   entries in the list,
**  executing the  body for each element  of the list.   If a leave-statement
**  ('break' or 'return') is executed inside the  body, then the execution of
**  the for-loop is terminated and 0 is returned if the leave-statement was a
**  break-statement   or  the   non-zero leave-value   is   returned  if  the
**  leave-statement was a return-statement (to tell the calling executor that
**  a return-statement was  executed).  If  no leave-statement was  executed,
**  then 0 is returned.
**
**  A short for-loop with <n> statements in its body is  represented by a bag
**  of   type 'T_FOR_RANGE'  with <n>+2 subbags.     The  first subbag is  an
**  assignment   bag  for  the  loop  variable,   the second    subbag is the
**  list-expression, and the remaining subbags are the statements.
*/
libGAP_UInt            libGAP_ExecForRange (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_UInt                lvar;           /* local variable                  */
    libGAP_Int                 first;          /* first value of range            */
    libGAP_Int                 last;           /* last value of range             */
    libGAP_Obj                 elm;            /* one element of the list         */
    libGAP_Stat                body;           /* body of the loop                */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the variable (initialize them first to please 'lint')           */
    lvar = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the range                                                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    libGAP_VisitStatIfProfiling(libGAP_ADDR_STAT(stat)[1]);
    elm = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[0] );
    while ( ! libGAP_IS_INTOBJ(elm) ) {
        elm = libGAP_ErrorReturnObj(
            "Range: <first> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(elm), 0L,
            "you can replace <first> via 'return <first>;'" );
    }
    first = libGAP_INT_INTOBJ(elm);
    elm = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[1] );
    while ( ! libGAP_IS_INTOBJ(elm) ) {
        elm = libGAP_ErrorReturnObj(
            "Range: <last> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(elm), 0L,
            "you can replace <last> via 'return <last>;'" );
    }
    last  = libGAP_INT_INTOBJ(elm);

    /* get the body                                                        */
    body = libGAP_ADDR_STAT(stat)[2];

    /* loop over the range                                                 */
    for ( i = first; i <= last; i++ ) {

        /* get the element and assign it to the variable                   */
        elm = libGAP_INTOBJ_INT( i );
        libGAP_ASS_LVAR( lvar, elm );

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the statements in the body                              */
        if ( (leave = libGAP_EXEC_STAT( body )) != 0 ) {
          if (leave == 8) 
            continue;
          return (leave & 3);
        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecForRange2 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_UInt                lvar;           /* local variable                  */
    libGAP_Int                 first;          /* first value of range            */
    libGAP_Int                 last;           /* last value of range             */
    libGAP_Obj                 elm;            /* one element of the list         */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the variable (initialize them first to please 'lint')           */
    lvar = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the range                                                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    libGAP_VisitStatIfProfiling(libGAP_ADDR_STAT(stat)[1]);
    elm = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[0] );
    while ( ! libGAP_IS_INTOBJ(elm) ) {
        elm = libGAP_ErrorReturnObj(
            "Range: <first> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(elm), 0L,
            "you can replace <first> via 'return <first>;'" );
    }
    first = libGAP_INT_INTOBJ(elm);
    elm = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[1] );
    while ( ! libGAP_IS_INTOBJ(elm) ) {
        elm = libGAP_ErrorReturnObj(
            "Range: <last> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(elm), 0L,
            "you can replace <last> via 'return <last>;'" );
    }
    last  = libGAP_INT_INTOBJ(elm);

    /* get the body                                                        */
    body1 = libGAP_ADDR_STAT(stat)[2];
    body2 = libGAP_ADDR_STAT(stat)[3];

    /* loop over the range                                                 */
    for ( i = first; i <= last; i++ ) {

        /* get the element and assign it to the variable                   */
        elm = libGAP_INTOBJ_INT( i );
        libGAP_ASS_LVAR( lvar, elm );

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the statements in the body                              */
        if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecForRange3 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_UInt                lvar;           /* local variable                  */
    libGAP_Int                 first;          /* first value of range            */
    libGAP_Int                 last;           /* last value of range             */
    libGAP_Obj                 elm;            /* one element of the list         */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */
    libGAP_Stat                body3;          /* third  stat. of body of loop    */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the variable (initialize them first to please 'lint')           */
    lvar = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );

    /* evaluate the range                                                  */
    libGAP_SET_BRK_CURR_STAT( stat );
    libGAP_VisitStatIfProfiling(libGAP_ADDR_STAT(stat)[1]);
    elm = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[0] );
    while ( ! libGAP_IS_INTOBJ(elm) ) {
        elm = libGAP_ErrorReturnObj(
            "Range: <first> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(elm), 0L,
            "you can replace <first> via 'return <first>;'" );
    }
    first = libGAP_INT_INTOBJ(elm);
    elm = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[1] );
    while ( ! libGAP_IS_INTOBJ(elm) ) {
        elm = libGAP_ErrorReturnObj(
            "Range: <last> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(elm), 0L,
            "you can replace <last> via 'return <last>;'" );
    }
    last  = libGAP_INT_INTOBJ(elm);

    /* get the body                                                        */
    body1 = libGAP_ADDR_STAT(stat)[2];
    body2 = libGAP_ADDR_STAT(stat)[3];
    body3 = libGAP_ADDR_STAT(stat)[4];

    /* loop over the range                                                 */
    for ( i = first; i <= last; i++ ) {

        /* get the element and assign it to the variable                   */
        elm = libGAP_INTOBJ_INT( i );
        libGAP_ASS_LVAR( lvar, elm );

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the statements in the body                              */
        if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body3 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

/****************************************************************************
**
*F  ExecAtomic(<stat>)
*/

libGAP_UInt libGAP_ExecAtomic(
		libGAP_Stat stat)
{
    // In non-HPC GAP, we completely ignore all the 'atomic' terms
    return libGAP_EXEC_STAT(libGAP_ADDR_STAT(stat)[0]);
}


/****************************************************************************
**
*F  ExecWhile(<stat>) . . . . . . . . . . . . . . . . .  execute a while-loop
**
**  'ExecWhile' executes the while-loop <stat>.
**
**  This is done  by  executing the  body while  the condition   evaluates to
**  'true'.  If a leave-statement   ('break' or 'return') is executed  inside
**  the  body, then the execution of  the while-loop  is  terminated and 0 is
**  returned if the  leave-statement was  a  break-statement or the  non-zero
**  leave-value is returned if the leave-statement was a return-statement (to
**  tell the calling executor  that a return-statement  was executed).  If no
**  leave-statement was executed, then 0 is returned.
**
**  A while-loop with <n> statements  in its body  is represented by a bag of
**  type  'T_WHILE' with <n>+1 subbags.   The first  subbag is the condition,
**  the second subbag is the first statement,  the third subbag is the second
**  statement, and so on.
*/
libGAP_UInt libGAP_ExecWhile (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body;           /* body of loop                    */

    /* get the condition and the body                                      */
    cond = libGAP_ADDR_STAT(stat)[0];
    body = libGAP_ADDR_STAT(stat)[1];

    /* while the condition evaluates to 'true', execute the body           */
    libGAP_SET_BRK_CURR_STAT( stat );
    while ( libGAP_EVAL_BOOL_EXPR( cond ) != libGAP_False ) {

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the body                                                */
        if ( (leave = libGAP_EXEC_STAT( body )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        libGAP_SET_BRK_CURR_STAT( stat );

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt libGAP_ExecWhile2 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */

    /* get the condition and the body                                      */
    cond = libGAP_ADDR_STAT(stat)[0];
    body1 = libGAP_ADDR_STAT(stat)[1];
    body2 = libGAP_ADDR_STAT(stat)[2];

    /* while the condition evaluates to 'true', execute the body           */
    libGAP_SET_BRK_CURR_STAT( stat );
    while ( libGAP_EVAL_BOOL_EXPR( cond ) != libGAP_False ) {

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the body                                                */
        if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        libGAP_SET_BRK_CURR_STAT( stat );

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt libGAP_ExecWhile3 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */
    libGAP_Stat                body3;          /* third  stat. of body of loop    */

    /* get the condition and the body                                      */
    cond = libGAP_ADDR_STAT(stat)[0];
    body1 = libGAP_ADDR_STAT(stat)[1];
    body2 = libGAP_ADDR_STAT(stat)[2];
    body3 = libGAP_ADDR_STAT(stat)[3];

    /* while the condition evaluates to 'true', execute the body           */
    libGAP_SET_BRK_CURR_STAT( stat );
    while ( libGAP_EVAL_BOOL_EXPR( cond ) != libGAP_False ) {

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the body                                                */
        if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body3 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        libGAP_SET_BRK_CURR_STAT( stat );

    }

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecRepeat(<stat>)  . . . . . . . . . . . . . . . . execute a repeat-loop
**
**  'ExecRepeat' executes the repeat-loop <stat>.
**
**  This is  done by  executing  the body until   the condition evaluates  to
**  'true'.  If  a leave-statement ('break'  or 'return')  is executed inside
**  the  body, then the  execution of the repeat-loop  is terminated and 0 is
**  returned  if the leave-statement   was a break-statement  or the non-zero
**  leave-value is returned if the leave-statement was a return-statement (to
**  tell the  calling executor that a  return-statement was executed).  If no
**  leave-statement was executed, then 0 is returned.
**
**  A repeat-loop with <n> statements in its body is  represented by a bag of
**  type 'T_REPEAT'  with <n>+1 subbags.  The  first subbag is the condition,
**  the second subbag is the first statement, the  third subbag is the second
**  statement, and so on.
*/
libGAP_UInt libGAP_ExecRepeat (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body;           /* body of loop                    */

    /* get the condition and the body                                      */
    cond = libGAP_ADDR_STAT(stat)[0];
    body = libGAP_ADDR_STAT(stat)[1];

    /* execute the body until the condition evaluates to 'true'            */
    libGAP_SET_BRK_CURR_STAT( stat );
    do {

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the body                                                */
        if ( (leave = libGAP_EXEC_STAT( body )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        libGAP_SET_BRK_CURR_STAT( stat );

    } while ( libGAP_EVAL_BOOL_EXPR( cond ) == libGAP_False );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt libGAP_ExecRepeat2 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */

    /* get the condition and the body                                      */
    cond = libGAP_ADDR_STAT(stat)[0];
    body1 = libGAP_ADDR_STAT(stat)[1];
    body2 = libGAP_ADDR_STAT(stat)[2];

    /* execute the body until the condition evaluates to 'true'            */
    libGAP_SET_BRK_CURR_STAT( stat );
    do {

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the body                                                */
        if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        libGAP_SET_BRK_CURR_STAT( stat );

    } while ( libGAP_EVAL_BOOL_EXPR( cond ) == libGAP_False );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt libGAP_ExecRepeat3 (
    libGAP_Stat                stat )
{
    libGAP_UInt                leave;          /* a leave-statement was executed  */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                body1;          /* first  stat. of body of loop    */
    libGAP_Stat                body2;          /* second stat. of body of loop    */
    libGAP_Stat                body3;          /* third  stat. of body of loop    */

    /* get the condition and the body                                      */
    cond = libGAP_ADDR_STAT(stat)[0];
    body1 = libGAP_ADDR_STAT(stat)[1];
    body2 = libGAP_ADDR_STAT(stat)[2];
    body3 = libGAP_ADDR_STAT(stat)[3];

    /* execute the body until the condition evaluates to 'true'            */
    libGAP_SET_BRK_CURR_STAT( stat );
    do {

#if ! HAVE_SIGNAL
        /* test for an interrupt                                           */
        if ( libGAP_HaveInterrupt() ) {
            libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
        }
#endif

        /* execute the body                                                */
        if ( (leave = libGAP_EXEC_STAT( body1 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body2 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        if ( (leave = libGAP_EXEC_STAT( body3 )) != 0 ) {
            if (leave == 8) 
                continue;
            return (leave & 3);
        }
        libGAP_SET_BRK_CURR_STAT( stat );

    } while ( libGAP_EVAL_BOOL_EXPR( cond ) == libGAP_False );

    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}


/****************************************************************************
**
*F  ExecBreak(<stat>) . . . . . . . . . . . . . . . execute a break-statement
**
**  'ExecBreak' executes the break-statement <stat>.
**
**  This  is done   by  returning 4  (to tell  the   calling executor that  a
**  break-statement was executed).
**
**  A break-statement is  represented  by a bag of   type 'T_BREAK' with   no
**  subbags.
*/
libGAP_UInt            libGAP_ExecBreak (
    libGAP_Stat                stat )
{
    /* return to the next loop                                             */
    return 4;
}

/****************************************************************************
**
*F  ExecContinue(<stat>) . . . . . . . . . . . . . . . execute a continue-statement
**
**  'ExecContinue' executes the continue-statement <stat>.
**
**  This  is done   by  returning 8 (to tell  the   calling executor that  a
**  continue-statement was executed).
**
**  A continue-statement is  represented  by a bag of   type 'T_CONTINUE' with   no
**  subbags.
*/
libGAP_UInt            libGAP_ExecContinue (
    libGAP_Stat                stat )
{
    /* return to the next loop                                             */
    return 8;
}

/****************************************************************************
**
*F  ExecEmpty( <stat> ) . . . . . execute an empty statement
**
**  Does nothing
*/
libGAP_UInt libGAP_ExecEmpty( libGAP_Stat stat )
{
  return 0;
}


/****************************************************************************
**
*F  ExecInfo( <stat> )  . . . . . . . . . . . . . . execute an info-statement
**
**  'ExecInfo' executes the info-statement <stat>.
**
**  This is  done by evaluating the first  two arguments, using the GAP level
**  function InfoDecision to decide whether the message has to be printed. If
**  it has, the other arguments are evaluated and passed to InfoDoPrint
**
**  An  info-statement is represented by a  bag of type 'T_INFO' with subbags
**  for the arguments
*/
libGAP_UInt libGAP_ExecInfo (
    libGAP_Stat            stat )
{
    libGAP_Obj             selectors;
    libGAP_Obj             level;
    libGAP_Obj             selected;
    libGAP_UInt            narg;
    libGAP_UInt            i;
    libGAP_Obj             args;
    libGAP_Obj             arg;

    selectors = libGAP_EVAL_EXPR( libGAP_ARGI_INFO( stat, 1 ) );
    level = libGAP_EVAL_EXPR( libGAP_ARGI_INFO( stat, 2) );

    libGAP_SET_BRK_CALL_TO( stat );
    libGAP_SET_BRK_CURR_STAT( stat );

    selected = libGAP_CALL_2ARGS(libGAP_InfoDecision, selectors, level);
    if (selected == libGAP_True) {

        /* Get the number of arguments to be printed                       */
        narg = libGAP_NARG_SIZE_INFO(libGAP_SIZE_STAT(stat)) - 2;

        /* set up a list                                                   */
        args = libGAP_NEW_PLIST( libGAP_T_PLIST, narg );
        libGAP_SET_LEN_PLIST( args, narg );

        /* evaluate the objects to be printed into the list                */
        for (i = 1; i <= narg; i++) {

            /* These two statements must not be combined into one because of
               the risk of a garbage collection during the evaluation
               of arg, which may happen after the pointer to args has been
               extracted
            */
            arg = libGAP_EVAL_EXPR(libGAP_ARGI_INFO(stat, i+2));
            libGAP_SET_ELM_PLIST(args, i, arg);
            libGAP_CHANGED_BAG(args);
        }

        /* and print them                                                  */
        libGAP_CALL_1ARGS(libGAP_InfoDoPrint, args);
    }
    return 0;
}

/****************************************************************************
**
*F  ExecAssert2Args(<stat>) . . . . . . . . . . . execute an assert-statement
**
**  'ExecAssert2Args' executes the 2 argument assert-statement <stat>.
**
**  A 2 argument assert-statement is  represented  by a bag of   type
**  'T_ASSERT_2ARGS' with subbags for the 2 arguments
*/
libGAP_UInt libGAP_ExecAssert2Args (
    libGAP_Stat            stat )
{
    libGAP_Obj             level;
    libGAP_Obj             decision;

    libGAP_SET_BRK_CURR_STAT( stat );
    libGAP_SET_BRK_CALL_TO( stat );

    level = libGAP_EVAL_EXPR( libGAP_ADDR_STAT( stat )[0] );
    if ( ! libGAP_LT(libGAP_CurrentAssertionLevel, level) )  {
        decision = libGAP_EVAL_EXPR( libGAP_ADDR_STAT( stat )[1]);
        while ( decision != libGAP_True && decision != libGAP_False ) {
         decision = libGAP_ErrorReturnObj(
          "Assertion condition must evaluate to 'true' or 'false', not a %s",
          (libGAP_Int)libGAP_TNAM_OBJ(decision), 0L,
          "you may 'return true;' or 'return false;'");
        }
        if ( decision == libGAP_False ) {
            libGAP_SET_BRK_CURR_STAT( stat );
            libGAP_ErrorReturnVoid( "Assertion failure", 0L, 0L, "you may 'return;'");
        }

        /* decision must be 'True' here                                    */
        else {
            return 0;
        }
    }
  return 0;
}

/****************************************************************************
**
*F  ExecAssert3Args(<stat>) . . . . . . . . . . . execute an assert-statement
**
**  'ExecAssert3Args' executes the 3 argument assert-statement <stat>.
**
**  A 3 argument assert-statement is  represented  by a bag of   type
**  'T_ASSERT_3ARGS' with subbags for the 3 arguments
*/
libGAP_UInt libGAP_ExecAssert3Args (
    libGAP_Stat            stat )
{
    libGAP_Obj             level;
    libGAP_Obj             decision;
    libGAP_Obj             message;

    libGAP_SET_BRK_CURR_STAT( stat );
    libGAP_SET_BRK_CALL_TO( stat );
    
    level = libGAP_EVAL_EXPR( libGAP_ADDR_STAT( stat )[0] );
    if ( ! libGAP_LT(libGAP_CurrentAssertionLevel, level) ) {
        decision = libGAP_EVAL_EXPR( libGAP_ADDR_STAT( stat )[1]);
        while ( decision != libGAP_True && decision != libGAP_False ) {
            decision = libGAP_ErrorReturnObj(
            "Assertion condition must evaluate to 'true' or 'false', not a %s",
            (libGAP_Int)libGAP_TNAM_OBJ(decision), 0L,
            "you may 'return true;' or 'return false;'");
        }
        if ( decision == libGAP_False ) {
            message = libGAP_EVAL_EXPR(libGAP_ADDR_STAT( stat )[2]);
            if ( message != (libGAP_Obj) 0 ) {
                if (libGAP_IS_STRING_REP( message ))
                    libGAP_PrintString1( message );
                else
                    libGAP_PrintObj(message);
            }
        }
        return 0;
    }
    return 0;
}


/****************************************************************************
**
*F  ExecReturnObj(<stat>) . . . . . . . . .  execute a return-value-statement
**
**  'ExecRetval' executes the return-value-statement <stat>.
**
**  This    is  done  by  setting  'ReturnObjStat'    to   the  value of  the
**  return-value-statement, and returning   1 (to tell   the calling executor
**  that a return-value-statement was executed).
**
**  A return-value-statement  is represented by a  bag of type 'T_RETURN_OBJ'
**  with      one  subbag.    This  subbag     is   the    expression  of the
**  return-value-statement.
*/
libGAP_UInt            libGAP_ExecReturnObj (
    libGAP_Stat                stat )
{
#if ! HAVE_SIGNAL
    /* test for an interrupt                                               */
    if ( libGAP_HaveInterrupt() ) {
        libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
    }
#endif

    /* evaluate the expression                                             */
    libGAP_SET_BRK_CURR_STAT( stat );
    libGAP_TLS(libGAP_ReturnObjStat) = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(stat)[0] );

    /* return up to function interpreter                                   */
    return 1;
}


/****************************************************************************
**
*F  ExecReturnVoid(<stat>)  . . . . . . . . . execute a return-void-statement
**
**  'ExecReturnVoid'   executes  the return-void-statement <stat>,  i.e., the
**  return-statement that returns not value.
**
**  This  is done by   returning 2  (to tell    the calling executor  that  a
**  return-void-statement was executed).
**
**  A return-void-statement  is represented by  a bag of type 'T_RETURN_VOID'
**  with no subbags.
*/
libGAP_UInt            libGAP_ExecReturnVoid (
    libGAP_Stat                stat )
{
#if ! HAVE_SIGNAL
    /* test for an interrupt                                               */
    if ( libGAP_HaveInterrupt() ) {
        libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
    }
#endif

    /* set 'TLS(ReturnObjStat)' to void                                         */
    libGAP_TLS(libGAP_ReturnObjStat) = 0;

    /* return up to function interpreter                                   */
    return 2;
}

libGAP_UInt (* libGAP_RealExecStatFuncs[256]) ( libGAP_Stat stat );

#ifdef HAVE_SIG_ATOMIC_T
libGAP_sig_atomic_t volatile libGAP_RealExecStatCopied;
#else
int volatile libGAP_RealExecStatCopied;
#endif

/****************************************************************************
**
*F  void CheckAndRespondToAlarm()
**
*/

static void libGAP_CheckAndRespondToAlarm(void) {
  if ( libGAP_SyAlarmHasGoneOff ) {
    assert(libGAP_NumAlarmJumpBuffers);
    libGAP_syLongjmp(libGAP_AlarmJumpBuffers[--libGAP_NumAlarmJumpBuffers],1);
  }
}

/****************************************************************************
**
*F  UInt TakeInterrupt() . . . . . . . . allow user interrupts
**
**  When you call this you promise that the heap is in a normal state, 
**  allowing GAP execution in the usual way
**
**  This will do nothing (pretty quickly) if Ctrl-C has not been pressed and 
**  return 0. Otherwise it
**   will respond appropriately.  This may result in a longjmp
**  or in returning to the caller after arbitrary execution of GAP code
** including possible garbage collection. In this case 1 is returned.
*/

libGAP_UInt libGAP_TakeInterrupt( void ) {
  if (libGAP_HaveInterrupt()) {
    libGAP_UnInterruptExecStat();
    libGAP_CheckAndRespondToAlarm();
    
    libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
    return 1;
  }
  return 0;
}


/****************************************************************************
**
*F  UnInterruptExecStat()  . . . . .revert the Statement execution jump table 
**                                   to normal 
*/

void libGAP_UnInterruptExecStat() {
  libGAP_UInt i;
  assert(libGAP_RealExecStatCopied);
  for ( i=0; i<sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++ ) {
    libGAP_ExecStatFuncs[i] = libGAP_RealExecStatFuncs[i];
  }
  libGAP_RealExecStatCopied = 0;
  return;
}


/****************************************************************************
**
*F  ExecIntrStat(<stat>)  . . . . . . . . . . . . . . interrupt a computation
**
**  'ExecIntrStat' is called when a computation was interrupted (by a call to
**  'InterruptExecStat').  It  changes   the entries in    the dispatch table
**  'ExecStatFuncs' back   to   their original   value,   calls 'Error',  and
**  redispatches after a return from the break-loop.
*/

libGAP_UInt libGAP_ExecIntrStat (
    libGAP_Stat                stat )
{

    /* change the entries in 'ExecStatFuncs' back to the original          */
    if ( libGAP_RealExecStatCopied ) {
      libGAP_UnInterruptExecStat();
    }
    libGAP_HaveInterrupt();


    /* One reason we might be here is a timeout. If so longjump out to the 
       CallWithTimeLimit where we started */
    libGAP_CheckAndRespondToAlarm();
      
    /* and now for something completely different                          */
    libGAP_SET_BRK_CURR_STAT( stat );
    if ( libGAP_SyStorOverrun != 0 ) {
      libGAP_SyStorOverrun = 0; /* reset */
      libGAP_ErrorReturnVoid(
  "reached the pre-set memory limit\n(change it with the -o command line option)",
        0L, 0L, "you can 'return;'" );
    }
    else {
      libGAP_ErrorReturnVoid( "user interrupt", 0L, 0L, "you can 'return;'" );
    }

    /* continue at the interrupted statement                               */
    return libGAP_EXEC_STAT( stat );
}


/****************************************************************************
**

*F  InterruptExecStat() . . . . . . . . interrupt the execution of statements
**
**  'InterruptExecStat'  interrupts the execution of   statements at the next
**  possible moment.  It is called from 'SyAnsIntr' if an interrupt signal is
**  received.  It is never called on systems that do not support signals.  On
**  those systems the executors test 'SyIsIntr' at regular intervals.
**
**  'InterruptExecStat' changes all entries   in the executor  dispatch table
**  'ExecStatFuncs'  to point to  'ExecIntrStat',  which changes  the entries
**  back, calls 'Error', and redispatches after a return from the break-loop.
*/
void libGAP_InterruptExecStat ( void )
{
    libGAP_UInt                i;              /* loop variable                   */
    /*    assert(reason > 0) */

    /* remember the original entries from the table 'ExecStatFuncs'        */
    if ( ! libGAP_RealExecStatCopied ) {
        for ( i=0; i<sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++ ) {
            libGAP_RealExecStatFuncs[i] = libGAP_ExecStatFuncs[i];
        }
        libGAP_RealExecStatCopied = 1;
    }

    /* change the entries in the table 'ExecStatFuncs' to 'ExecIntrStat'   */
    for ( i = 0;
          i < libGAP_T_SEQ_STAT;
          i++ ) {
        libGAP_ExecStatFuncs[i] = libGAP_ExecIntrStat;
    }
    for ( i = libGAP_T_RETURN_VOID;
          i < sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]);
          i++ ) {
        libGAP_ExecStatFuncs[i] = libGAP_ExecIntrStat;
    }
}

/****************************************************************************
**
*F  ClearError()  . . . . . . . . . . . . . .  reset execution and error flag
*/

libGAP_Int libGAP_BreakLoopPending( void ) {
     return libGAP_RealExecStatCopied;
}

void libGAP_ClearError ( void )
{

    /* change the entries in 'ExecStatFuncs' back to the original          */
    
    if ( libGAP_RealExecStatCopied ) {
      libGAP_UnInterruptExecStat();
        /* check for user interrupt */
        if ( libGAP_HaveInterrupt() ) {
          libGAP_Pr("Noticed user interrupt, but you are back in main loop anyway.\n",
              0L, 0L);
        }
        /* and check if maximal memory was overrun */
        if ( libGAP_SyStorOverrun != 0 ) {
          libGAP_SyStorOverrun = 0; /* reset */
          libGAP_Pr("GAP has exceeded the permitted memory (-o option),\n", 0L, 0L);
          libGAP_Pr("the maximum is now enlarged to %d kB.\n", (libGAP_Int)libGAP_SyStorMax, 0L);
        }
    }

    /* reset <TLS(NrError)>                                                     */
    libGAP_TLS(libGAP_NrError) = 0;
}

/****************************************************************************
**
*F  PrintStat(<stat>) . . . . . . . . . . . . . . . . . . . print a statement
**
**  'PrintStat' prints the statements <stat>.
**
**  'PrintStat' simply dispatches  through the table  'PrintStatFuncs' to the
**  appropriate printer.
*/
void            libGAP_PrintStat (
    libGAP_Stat                stat )
{
    (*libGAP_PrintStatFuncs[libGAP_TNUM_STAT(stat)])( stat );
}


/****************************************************************************
**
*V  PrintStatFuncs[<type>]  . .  print function for statements of type <type>
**
**  'PrintStatFuncs' is the dispatching table that contains for every type of
**  statements a pointer to the  printer for statements  of this type,  i.e.,
**  the function that should be called to print statements of this type.
*/
void            (* libGAP_PrintStatFuncs[256] ) ( libGAP_Stat stat );


/****************************************************************************
**
*F  PrintUnknownStat(<stat>)  . . . . . . . . print statement of unknown type
**
**  'PrintUnknownStat' is the printer  that is called if  an attempt  is made
**  print a statement <stat>  of an unknown type.   It signals an error.   If
**  this  is  ever called,   then GAP  is in  serious   trouble, such  as  an
**  overwritten type field of a statement.
*/
void            libGAP_PrintUnknownStat (
    libGAP_Stat                stat )
{
    libGAP_ErrorQuit(
        "Panic: cannot print statement of type '%d'",
        (libGAP_Int)libGAP_TNUM_STAT(stat), 0L );
}


/****************************************************************************
**
*F  PrintSeqStat(<stat>)  . . . . . . . . . . . .  print a statement sequence
**
**  'PrintSeqStat' prints the statement sequence <stat>.
*/
void            libGAP_PrintSeqStat (
    libGAP_Stat                stat )
{
    libGAP_UInt                nr;             /* number of statements            */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the number of statements                                        */
    nr = libGAP_SIZE_STAT( stat ) / sizeof(libGAP_Stat);

    /* loop over the statements                                            */
    for ( i = 1; i <= nr; i++ ) {

        /* print the <i>-th statement                                      */
        libGAP_PrintStat( libGAP_ADDR_STAT(stat)[i-1] );

        /* print a line break after all but the last statement             */
        if ( i < nr )  libGAP_Pr( "\n", 0L, 0L );

    }

}


/****************************************************************************
**
*F  PrintIf(<stat>) . . . . . . . . . . . . . . . . . . print an if-statement
**
**  'PrIf' prints the if-statement <stat>.
**
**  Linebreaks are printed after the 'then' and the statements in the bodies.
**  If necessary one is preferred immediately before the 'then'.
*/
void            libGAP_PrintIf (
    libGAP_Stat                stat )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* print the 'if' branch                                               */
    libGAP_Pr( "if%4> ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr( "%2<  then%2>\n", 0L, 0L );
    libGAP_PrintStat( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr( "%4<\n", 0L, 0L );

    /* print the 'elif' branch                                             */
    for ( i = 2; i <= libGAP_SIZE_STAT(stat)/(2*sizeof(libGAP_Stat)); i++ ) {
        if ( libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[2*(i-1)] ) == libGAP_T_TRUE_EXPR ) {
            libGAP_Pr( "else%4>\n", 0L, 0L );
        }
        else {
            libGAP_Pr( "elif%4> ", 0L, 0L );
            libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2*(i-1)] );
            libGAP_Pr( "%2<  then%2>\n", 0L, 0L );
        }
        libGAP_PrintStat( libGAP_ADDR_STAT(stat)[2*(i-1)+1] );
        libGAP_Pr( "%4<\n", 0L, 0L );
    }

    /* print the 'fi'                                                      */
    libGAP_Pr( "fi;", 0L, 0L );
}


/****************************************************************************
**
*F  PrintFor(<stat>)  . . . . . . . . . . . . . . . . . . .  print a for-loop
**
**  'PrintFor' prints the for-loop <stat>.
**
**  Linebreaks are printed after the 'do' and the statements in the body.  If
**  necesarry it is preferred immediately before the 'in'.
*/
void            libGAP_PrintFor (
    libGAP_Stat                stat )
{
    libGAP_UInt                i;              /* loop variable                   */

    libGAP_Pr( "for%4> ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr( "%2<  in%2> ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr( "%2<  do%2>\n", 0L, 0L );
    for ( i = 2; i <= libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat)-1; i++ ) {
        libGAP_PrintStat( libGAP_ADDR_STAT(stat)[i] );
        if ( i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat)-1 )  libGAP_Pr( "\n", 0L, 0L );
    }
    libGAP_Pr( "%4<\nod;", 0L, 0L );
}


/****************************************************************************
**
*F  PrintWhile(<stat>)  . . . . . . . . . . . . . . . . .  print a while loop
**
**  'PrintWhile' prints the while-loop <stat>.
**
**  Linebreaks are printed after the 'do' and the statments  in the body.  If
**  necessary one is preferred immediately before the 'do'.
*/
void            libGAP_PrintWhile (
    libGAP_Stat                stat )
{
    libGAP_UInt                i;              /* loop variable                   */

    libGAP_Pr( "while%4> ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr( "%2<  do%2>\n", 0L, 0L );
    for ( i = 1; i <= libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat)-1; i++ ) {
        libGAP_PrintStat( libGAP_ADDR_STAT(stat)[i] );
        if ( i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat)-1 )  libGAP_Pr( "\n", 0L, 0L );
    }
    libGAP_Pr( "%4<\nod;", 0L, 0L );
}

/****************************************************************************
**
*F  PrintAtomic(<stat>)  . . . . . . . . . . . . . . . . .  print a atomic loop
**
**  'PrintAtomic' prints the atomic-loop <stat>.
**
**  Linebreaks are printed after the 'do' and the statments  in the body.  If
**  necessary one is preferred immediately before the 'do'.
*/
void            libGAP_PrintAtomic (
    libGAP_Stat                stat )
{
  libGAP_UInt nrexprs;
    libGAP_UInt                i;              /* loop variable                   */

    libGAP_Pr( "atomic%4> ", 0L, 0L );
    nrexprs = ((libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat))-1)/2;
    for (i = 1; i <=  nrexprs; i++) {
      if (i != 1)
	libGAP_Pr(", ",0L,0L);
      switch (libGAP_INT_INTEXPR(libGAP_ADDR_STAT(stat)[2*i-1])) {
      case 0: break;
      case 1: libGAP_Pr("readonly ",0L,0L);
	break;
      case 2: libGAP_Pr("readwrite ",0L,0L);
	break;
      }
      libGAP_PrintExpr(libGAP_ADDR_STAT(stat)[2*i]);
    }
    libGAP_Pr( "%2<  do%2>\n", 0L, 0L );
    libGAP_PrintStat( libGAP_ADDR_STAT(stat)[0]);
    libGAP_Pr( "%4<\nod;", 0L, 0L );
}


/****************************************************************************
**
*F  PrintRepeat(<stat>) . . . . . . . . . . . . . . . . . print a repeat-loop
**
**  'PrintRepeat' prints the repeat-loop <stat>.
**
**  Linebreaks are printed after the 'repeat' and the statements in the body.
**  If necessary one is preferred after the 'until'.
*/
void            libGAP_PrintRepeat (
    libGAP_Stat                stat )
{
    libGAP_UInt                i;              /* loop variable                   */

    libGAP_Pr( "repeat%4>\n", 0L, 0L );
    for ( i = 1; i <= libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat)-1; i++ ) {
        libGAP_PrintStat( libGAP_ADDR_STAT(stat)[i] );
        if ( i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat)-1 )  libGAP_Pr( "\n", 0L, 0L );
    }
    libGAP_Pr( "%4<\nuntil%2> ", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr( "%2<;", 0L, 0L );
}


/****************************************************************************
**
*F  PrintBreak(<stat>)  . . . . . . . . . . . . . . . print a break-statement
**
**  'PrintBreak' prints the break-statement <stat>.
*/
void            libGAP_PrintBreak (
    libGAP_Stat                stat )
{
    libGAP_Pr( "break;", 0L, 0L );
}

/****************************************************************************
**
*F  PrintContinue(<stat>)  . . . . . . . . . . . . . . . print a continue-statement
**
**  'PrintContinue' prints the continue-statement <stat>.
*/
void            libGAP_PrintContinue (
    libGAP_Stat                stat )
{
    libGAP_Pr( "continue;", 0L, 0L );
}

/****************************************************************************
**
*F  PrintEmpty(<stat>)
**
*/
void             libGAP_PrintEmpty( libGAP_Stat stat )
{
  libGAP_Pr( ";", 0L, 0L);
}

/****************************************************************************
**
*F  PrintInfo(<stat>)  . . . . . . . . . . . . . . . print an info-statement
**
**  'PrintInfo' prints the info-statement <stat>.
*/

void            libGAP_PrintInfo (
    libGAP_Stat               stat )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* print the keyword                                                   */
    libGAP_Pr("%2>Info",0L,0L);

    /* print the opening parenthesis                                       */
    libGAP_Pr("%<( %>",0L,0L);

    /* print the expressions that evaluate to the actual arguments         */
    for ( i = 1; i <= libGAP_NARG_SIZE_INFO( libGAP_SIZE_STAT(stat) ); i++ ) {
        libGAP_PrintExpr( libGAP_ARGI_INFO(stat,i) );
        if ( i != libGAP_NARG_SIZE_INFO( libGAP_SIZE_STAT(stat) ) ) {
            libGAP_Pr("%<, %>",0L,0L);
        }
    }

    /* print the closing parenthesis                                       */
    libGAP_Pr(" %2<);",0L,0L);
}

/****************************************************************************
**
*F  PrintAssert2Args(<stat>)  . . . . . . . . . . . . print an info-statement
**
**  'PrintAssert2Args' prints the 2 argument assert-statement <stat>.
*/

void            libGAP_PrintAssert2Args (
    libGAP_Stat               stat )
{

    /* print the keyword                                                   */
    libGAP_Pr("%2>Assert",0L,0L);

    /* print the opening parenthesis                                       */
    libGAP_Pr("%<( %>",0L,0L);

    /* Print the arguments, separated by a comma                           */
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<, %>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );

    /* print the closing parenthesis                                       */
    libGAP_Pr(" %2<);",0L,0L);
}
  
/****************************************************************************
**
*F  PrintAssert3Args(<stat>)  . . . . . . . . . . . . print an info-statement
**
**  'PrintAssert3Args' prints the 3 argument assert-statement <stat>.
*/

void            libGAP_PrintAssert3Args (
    libGAP_Stat               stat )
{

    /* print the keyword                                                   */
    libGAP_Pr("%2>Assert",0L,0L);

    /* print the opening parenthesis                                       */
    libGAP_Pr("%<( %>",0L,0L);

    /* Print the arguments, separated by commas                            */
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr("%<, %>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Pr("%<, %>",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2] );

    /* print the closing parenthesis                                       */
    libGAP_Pr(" %2<);",0L,0L);
}
  


/****************************************************************************
**
*F  PrintReturnObj(<stat>)  . . . . . . . . .  print a return-value-statement
**
**  'PrintReturnObj' prints the return-value-statement <stat>.
*/
void            libGAP_PrintReturnObj (
    libGAP_Stat                stat )
{
    libGAP_Pr( "%2>return%< %>", 0L, 0L );
    libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Pr( "%2<;", 0L, 0L );
}


/****************************************************************************
**
*F  PrintReturnVoid(<stat>) . . . . . . . . . . print a return-void-statement
**
**  'PrintReturnVoid' prints the return-void-statement <stat>.
*/
void            libGAP_PrintReturnVoid (
    libGAP_Stat                stat )
{
    libGAP_Pr( "return;", 0L, 0L );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/



/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{

    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                i;              /* loop variable                   */

    libGAP_RealExecStatCopied = 0;
    
    /* make the global bags known to Gasman                                */
    /* 'InitGlobalBag( &CurrStat );' is not really needed, since we are in */
    /* for a lot of trouble if 'CurrStat' ever becomes the last reference. */
    /* furthermore, statements are no longer bags                          */
    /* InitGlobalBag( &CurrStat );                                         */

    libGAP_InitGlobalBag( &libGAP_ReturnObjStat, "src/stats.c:ReturnObjStat" );

    /* connect to external functions                                       */
    libGAP_ImportFuncFromLibrary( "Iterator",       &libGAP_ITERATOR );
    libGAP_ImportFuncFromLibrary( "IsDoneIterator", &libGAP_IS_DONE_ITER );
    libGAP_ImportFuncFromLibrary( "NextIterator",   &libGAP_NEXT_ITER );
    libGAP_ImportFuncFromLibrary( "IsStandardIterator",   &libGAP_STD_ITER );

    /* install executors for non-statements                                */
    for ( i = 0; i < sizeof(libGAP_ExecStatFuncs)/sizeof(libGAP_ExecStatFuncs[0]); i++ ) {
        libGAP_InstallExecStatFunc(i, libGAP_ExecUnknownStat);
    }

    /* install executors for compound statements                           */
    libGAP_InstallExecStatFunc( libGAP_T_SEQ_STAT       , libGAP_ExecSeqStat);
    libGAP_InstallExecStatFunc( libGAP_T_SEQ_STAT2      , libGAP_ExecSeqStat2);
    libGAP_InstallExecStatFunc( libGAP_T_SEQ_STAT3      , libGAP_ExecSeqStat3);
    libGAP_InstallExecStatFunc( libGAP_T_SEQ_STAT4      , libGAP_ExecSeqStat4);
    libGAP_InstallExecStatFunc( libGAP_T_SEQ_STAT5      , libGAP_ExecSeqStat5);
    libGAP_InstallExecStatFunc( libGAP_T_SEQ_STAT6      , libGAP_ExecSeqStat6);
    libGAP_InstallExecStatFunc( libGAP_T_SEQ_STAT7      , libGAP_ExecSeqStat7);
    libGAP_InstallExecStatFunc( libGAP_T_IF             , libGAP_ExecIf);
    libGAP_InstallExecStatFunc( libGAP_T_IF_ELSE        , libGAP_ExecIfElse);
    libGAP_InstallExecStatFunc( libGAP_T_IF_ELIF        , libGAP_ExecIfElif);
    libGAP_InstallExecStatFunc( libGAP_T_IF_ELIF_ELSE   , libGAP_ExecIfElifElse);
    libGAP_InstallExecStatFunc( libGAP_T_FOR            , libGAP_ExecFor);
    libGAP_InstallExecStatFunc( libGAP_T_FOR2           , libGAP_ExecFor2);
    libGAP_InstallExecStatFunc( libGAP_T_FOR3           , libGAP_ExecFor3);
    libGAP_InstallExecStatFunc( libGAP_T_FOR_RANGE      , libGAP_ExecForRange);
    libGAP_InstallExecStatFunc( libGAP_T_FOR_RANGE2     , libGAP_ExecForRange2);
    libGAP_InstallExecStatFunc( libGAP_T_FOR_RANGE3     , libGAP_ExecForRange3);
    libGAP_InstallExecStatFunc( libGAP_T_WHILE          , libGAP_ExecWhile);
    libGAP_InstallExecStatFunc( libGAP_T_WHILE2         , libGAP_ExecWhile2);
    libGAP_InstallExecStatFunc( libGAP_T_WHILE3         , libGAP_ExecWhile3);
    libGAP_InstallExecStatFunc( libGAP_T_REPEAT         , libGAP_ExecRepeat);
    libGAP_InstallExecStatFunc( libGAP_T_REPEAT2        , libGAP_ExecRepeat2);
    libGAP_InstallExecStatFunc( libGAP_T_REPEAT3        , libGAP_ExecRepeat3);
    libGAP_InstallExecStatFunc( libGAP_T_BREAK          , libGAP_ExecBreak);
    libGAP_InstallExecStatFunc( libGAP_T_CONTINUE       , libGAP_ExecContinue);
    libGAP_InstallExecStatFunc( libGAP_T_INFO           , libGAP_ExecInfo);
    libGAP_InstallExecStatFunc( libGAP_T_ASSERT_2ARGS   , libGAP_ExecAssert2Args);
    libGAP_InstallExecStatFunc( libGAP_T_ASSERT_3ARGS   , libGAP_ExecAssert3Args);
    libGAP_InstallExecStatFunc( libGAP_T_RETURN_OBJ     , libGAP_ExecReturnObj);
    libGAP_InstallExecStatFunc( libGAP_T_RETURN_VOID    , libGAP_ExecReturnVoid);
    libGAP_InstallExecStatFunc( libGAP_T_EMPTY          , libGAP_ExecEmpty);
    libGAP_InstallExecStatFunc( libGAP_T_ATOMIC         , libGAP_ExecAtomic);

    /* install printers for non-statements                                */
    for ( i = 0; i < sizeof(libGAP_PrintStatFuncs)/sizeof(libGAP_PrintStatFuncs[0]); i++ ) {
        libGAP_InstallPrintStatFunc(i, libGAP_PrintUnknownStat);
    }
    /* install printing functions for compound statements                  */
    libGAP_InstallPrintStatFunc( libGAP_T_SEQ_STAT       , libGAP_PrintSeqStat);
    libGAP_InstallPrintStatFunc( libGAP_T_SEQ_STAT2      , libGAP_PrintSeqStat);
    libGAP_InstallPrintStatFunc( libGAP_T_SEQ_STAT3      , libGAP_PrintSeqStat);
    libGAP_InstallPrintStatFunc( libGAP_T_SEQ_STAT4      , libGAP_PrintSeqStat);
    libGAP_InstallPrintStatFunc( libGAP_T_SEQ_STAT5      , libGAP_PrintSeqStat);
    libGAP_InstallPrintStatFunc( libGAP_T_SEQ_STAT6      , libGAP_PrintSeqStat);
    libGAP_InstallPrintStatFunc( libGAP_T_SEQ_STAT7      , libGAP_PrintSeqStat);
    libGAP_InstallPrintStatFunc( libGAP_T_IF             , libGAP_PrintIf);
    libGAP_InstallPrintStatFunc( libGAP_T_IF_ELSE        , libGAP_PrintIf);
    libGAP_InstallPrintStatFunc( libGAP_T_IF_ELIF        , libGAP_PrintIf);
    libGAP_InstallPrintStatFunc( libGAP_T_IF_ELIF_ELSE   , libGAP_PrintIf);
    libGAP_InstallPrintStatFunc( libGAP_T_FOR            , libGAP_PrintFor);
    libGAP_InstallPrintStatFunc( libGAP_T_FOR2           , libGAP_PrintFor);
    libGAP_InstallPrintStatFunc( libGAP_T_FOR3           , libGAP_PrintFor);
    libGAP_InstallPrintStatFunc( libGAP_T_FOR_RANGE      , libGAP_PrintFor);
    libGAP_InstallPrintStatFunc( libGAP_T_FOR_RANGE2     , libGAP_PrintFor);
    libGAP_InstallPrintStatFunc( libGAP_T_FOR_RANGE3     , libGAP_PrintFor);
    libGAP_InstallPrintStatFunc( libGAP_T_WHILE          , libGAP_PrintWhile);
    libGAP_InstallPrintStatFunc( libGAP_T_WHILE2         , libGAP_PrintWhile);
    libGAP_InstallPrintStatFunc( libGAP_T_WHILE3         , libGAP_PrintWhile);
    libGAP_InstallPrintStatFunc( libGAP_T_REPEAT         , libGAP_PrintRepeat);
    libGAP_InstallPrintStatFunc( libGAP_T_REPEAT2        , libGAP_PrintRepeat);
    libGAP_InstallPrintStatFunc( libGAP_T_REPEAT3        , libGAP_PrintRepeat);
    libGAP_InstallPrintStatFunc( libGAP_T_BREAK          , libGAP_PrintBreak);
    libGAP_InstallPrintStatFunc( libGAP_T_CONTINUE       , libGAP_PrintContinue);
    libGAP_InstallPrintStatFunc( libGAP_T_INFO           , libGAP_PrintInfo);
    libGAP_InstallPrintStatFunc( libGAP_T_ASSERT_2ARGS   , libGAP_PrintAssert2Args);
    libGAP_InstallPrintStatFunc( libGAP_T_ASSERT_3ARGS   , libGAP_PrintAssert3Args);
    libGAP_InstallPrintStatFunc( libGAP_T_RETURN_OBJ     , libGAP_PrintReturnObj);
    libGAP_InstallPrintStatFunc( libGAP_T_RETURN_VOID    , libGAP_PrintReturnVoid);
    libGAP_InstallPrintStatFunc( libGAP_T_EMPTY          , libGAP_PrintEmpty);
    libGAP_InstallPrintStatFunc( libGAP_T_ATOMIC         , libGAP_PrintAtomic);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoStats() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "stats",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoStats ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  stats.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
