/****************************************************************************
**
*W  scanner.c                   GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl  für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the scanner, which is responsible for
**  all input and output processing.
**
**  The scanner  exports two very  important abstractions.  The  first is the
**  concept that an input file is  a stream of symbols,  such nasty things as
**  <space>,  <tab>,  <newline> characters or  comments (they are worst  :-),
**  characters making  up identifiers  or  digits that  make  up integers are
**  hidden from the rest of GAP.
**
**  The second is  the concept of  a current input  and output file.   In the
**  main   module   they are opened  and   closed  with the  'OpenInput'  and
**  'CloseInput' respectively  'OpenOutput' and 'CloseOutput' calls.  All the
**  other modules just read from the  current input  and write to the current
**  output file.
**
**  SL 5/99 I now plan to break the second abstraction in regard of output
**  streams. Instead of all Print/View/etc output going via Pr to PutLine, etc.
**  they will go via PrTo and PutLineTo. The extra argument of these will be
**  of type KOutputStream, a pointer to a C structure (using a GAP object would
**  be difficult in the early bootstrap, and because writing to a string stream
**  may cause a garbage collection, which can be a pain).
**
**  The scanner relies on the functions  provided  by  the  operating  system
**  dependent module 'system.c' for the low level input/output.
*/
#include        "system.h"              /* system dependent part           */


#include        "sysfiles.h"            /* file input/output               */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */

#include        "scanner.h"             /* scanner                         */

#include        "code.h"                /* coder                           */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"              /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "opers.h"               /* DoFilter...                     */
#include        "read.h"                /* Call0ArgsInNewReader            */

#include	"tls.h"
#include	"thread.h"

#include        "libgap_internal.h"     /* GAP shared library              */

#include <assert.h>
#include <limits.h>
#include <stdlib.h>

/* the following global variables are documented in scanner.h */

libGAP_UInt            libGAP_Symbol;

libGAP_Char            libGAP_Value [1030];
libGAP_UInt            libGAP_ValueLen;

libGAP_UInt            libGAP_NrError;
libGAP_UInt            libGAP_NrErrLine;

const libGAP_Char *    libGAP_Prompt;

libGAP_Obj             libGAP_PrintPromptHook = 0;
libGAP_Obj             libGAP_EndLineHook = 0;

libGAP_TypInputFile    libGAP_InputFiles [16];
libGAP_TypInputFile *  libGAP_Input;
libGAP_Char *          libGAP_In;

libGAP_TypOutputFile   libGAP_OutputFiles [16];
libGAP_TypOutputFile * libGAP_Output;

libGAP_TypOutputFile * libGAP_InputLog;

libGAP_TypOutputFile * libGAP_OutputLog;


/****************************************************************************
**
*V  TestInput . . . . . . . . . . . . .  file identifier of test input, local
*V  TestOutput  . . . . . . . . . . . . file identifier of test output, local
*V  TestLine  . . . . . . . . . . . . . . . . one line from test input, local
**
**  'TestInput' is the file identifier  of the file for  test input.  If this
**  is  not -1  and  'GetLine' reads  a line  from  'TestInput' that does not
**  begins with  'gap>'  'GetLine' assumes  that this was  expected as output
**  that did not appear and echoes this input line to 'TestOutput'.
**
**  'TestOutput' is the current output file  for test output.  If 'TestInput'
**  is not -1 then 'PutLine' compares every line that is  about to be printed
**  to 'TestOutput' with the next  line from 'TestInput'.   If this line does
**  not starts with 'gap>'  and the rest of  it  matches the output line  the
**  output  line  is not printed  and the  input   comment line is discarded.
**  Otherwise 'PutLine' prints the output line and does not discard the input
**  line.
**
**  'TestLine' holds the one line that is read from 'TestInput' to compare it
**  with a line that is about to be printed to 'TestOutput'.
*/
static libGAP_TypInputFile  *libGAP_TestInput  = 0;
static libGAP_TypOutputFile *libGAP_TestOutput = 0;
static libGAP_Char           libGAP_TestLine[256];


/****************************************************************************
**
*F  SyntaxError( <msg> )  . . . . . . . . . . . . . . .  raise a syntax error
**
*/
void            libGAP_SyntaxError (
    const libGAP_Char *        msg )
{
    libGAP_Int                 i;

    /* open error output                                                   */
    libGAP_OpenOutput( "*errout*" );
    assert(libGAP_TLS(libGAP_Output));

    /* one more error                                                      */
    libGAP_TLS(libGAP_NrError)++;
    libGAP_TLS(libGAP_NrErrLine)++;

    /* do not print a message if we found one already on the current line  */
    if ( libGAP_TLS(libGAP_NrErrLine) == 1 )

      {
        /* print the message and the filename, unless it is '*stdin*'          */
        libGAP_Pr( "Syntax error: %s", (libGAP_Int)msg, 0L );
        if ( strcmp( "*stdin*", libGAP_TLS(libGAP_Input)->name ) != 0 )
          libGAP_Pr( " in %s line %d", (libGAP_Int)libGAP_TLS(libGAP_Input)->name, (libGAP_Int)libGAP_TLS(libGAP_Input)->number );
        libGAP_Pr( "\n", 0L, 0L );

        /* print the current line                                              */
        libGAP_Pr( "%s", (libGAP_Int)libGAP_TLS(libGAP_Input)->line, 0L );

        /* print a '^' pointing to the current position                        */
        for ( i = 0; i < libGAP_TLS(libGAP_In) - libGAP_TLS(libGAP_Input)->line - 1; i++ ) {
          if ( libGAP_TLS(libGAP_Input)->line[i] == '\t' )  libGAP_Pr("\t",0L,0L);
          else  libGAP_Pr(" ",0L,0L);
        }
        libGAP_Pr( "^\n", 0L, 0L );
      }
    /* close error output                                                  */
    assert(libGAP_TLS(libGAP_Output));
    libGAP_CloseOutput();
    assert(libGAP_TLS(libGAP_Output));
}

/****************************************************************************
**
*F  SyntaxWarning( <msg> )  . . . . . . . . . . . . . . .  raise a syntax error
**
*/
void            libGAP_SyntaxWarning (
    const libGAP_Char *        msg )
{
    libGAP_Int                 i;

    /* open error output                                                   */
    libGAP_OpenOutput( "*errout*" );
    assert(libGAP_TLS(libGAP_Output));


    /* do not print a message if we found one already on the current line  */
    if ( libGAP_TLS(libGAP_NrErrLine) == 0 )

      {
        /* print the message and the filename, unless it is '*stdin*'          */
        libGAP_Pr( "Syntax warning: %s", (libGAP_Int)msg, 0L );
        if ( strcmp( "*stdin*", libGAP_TLS(libGAP_Input)->name ) != 0 )
          libGAP_Pr( " in %s line %d", (libGAP_Int)libGAP_TLS(libGAP_Input)->name, (libGAP_Int)libGAP_TLS(libGAP_Input)->number );
        libGAP_Pr( "\n", 0L, 0L );

        /* print the current line                                              */
        libGAP_Pr( "%s", (libGAP_Int)libGAP_TLS(libGAP_Input)->line, 0L );

        /* print a '^' pointing to the current position                        */
        for ( i = 0; i < libGAP_TLS(libGAP_In) - libGAP_TLS(libGAP_Input)->line - 1; i++ ) {
          if ( libGAP_TLS(libGAP_Input)->line[i] == '\t' )  libGAP_Pr("\t",0L,0L);
          else  libGAP_Pr(" ",0L,0L);
        }
        libGAP_Pr( "^\n", 0L, 0L );
      }
    /* close error output                                                  */
    assert(libGAP_TLS(libGAP_Output));
    libGAP_CloseOutput();
    assert(libGAP_TLS(libGAP_Output));
}


/****************************************************************************
**
*F  Match( <symbol>, <msg>, <skipto> )  . match current symbol and fetch next
**
**  'Match' is the main  interface between the  scanner and the  parser.   It
**  performs the  4 most common actions in  the scanner  with  just one call.
**  First it checks that  the current symbol stored  in the variable 'Symbol'
**  is the expected symbol  as passed in the  argument <symbol>.  If  it  is,
**  'Match' reads the next symbol from input  and returns.  Otherwise 'Match'
**  first prints the current input line followed by the syntax error message:
**  '^ syntax error, <msg> expected' with '^' pointing to the current symbol.
**  It then  skips symbols up to one  in the resynchronisation  set <skipto>.
**  Actually 'Match' calls 'SyntaxError' so its comments apply here too.
**
**  One kind of typical 'Match' call has the form
**
**      'Match( Symbol, "", 0L );'.
**
**  This is used if the parser knows that the current  symbol is correct, for
**  example in 'RdReturn'  the   first symbol must be 'S_RETURN',   otherwise
**  'RdReturn' would not have been  called.  Called this  way 'Match' will of
**  course never raise an syntax error,  therefore <msg>  and <skipto> are of
**  no concern, they are passed nevertheless  to please  lint.  The effect of
**  this call is merely to read the next symbol from input.
**
**  Another typical 'Match' call is in 'RdIf' after we read the if symbol and
**  the condition following, and now expect to see the 'then' symbol:
**
**      Match( S_THEN, "then", STATBEGIN|S_ELIF|S_ELSE|S_FI|follow );
**
**  If the current symbol  is 'S_THEN' it is  matched  and the next symbol is
**  read.  Otherwise 'Match'  prints the  current line followed by the  error
**  message: '^ syntax error, then expected'.  Then 'Match' skips all symbols
**  until finding either  a symbol  that can begin  a statment,  an 'elif' or
**  'else' or 'fi' symbol, or a symbol that is  contained in the set <follow>
**  which is passed to  'RdIf' and contains  all symbols allowing  one of the
**  calling functions to resynchronize, for example 'S_OD' if 'RdIf' has been
**  called from 'RdFor'.  <follow>  always contain 'S_EOF', which 'Read' uses
**  to resynchronise.
**
**  If 'Match' needs to  read a  new line from  '*stdin*' or '*errin*' to get
**  the next symbol it prints the string pointed to by 'Prompt'.
*/
void libGAP_Match (
    libGAP_UInt                symbol,
    const libGAP_Char *        msg,
    libGAP_TypSymbolSet        skipto )
{
    libGAP_Char                errmsg [256];

    /* if 'TLS(Symbol)' is the expected symbol match it away                    */
    if ( symbol == libGAP_TLS(libGAP_Symbol) ) {
        libGAP_GetSymbol();
    }

    /* else generate an error message and skip to a symbol in <skipto>     */
    else {
        libGAP_strlcpy( errmsg, msg, sizeof(errmsg) );
        libGAP_strlcat( errmsg, " expected", sizeof(errmsg) );
        libGAP_SyntaxError( errmsg );
        while ( ! libGAP_IS_IN( libGAP_TLS(libGAP_Symbol), skipto ) )
            libGAP_GetSymbol();
    }
}


/****************************************************************************
**

*F * * * * * * * * * * * open input/output functions  * * * * * * * * * * * *
*/


/****************************************************************************
**
*F  OpenInput( <filename> ) . . . . . . . . . .  open a file as current input
**
**  'OpenInput' opens  the file with  the name <filename>  as  current input.
**  All  subsequent input will  be taken from that  file, until it is  closed
**  again  with 'CloseInput'  or  another file  is opened  with  'OpenInput'.
**  'OpenInput'  will not  close the  current  file, i.e., if  <filename>  is
**  closed again, input will again be taken from the current input file.
**
**  'OpenInput'  returns 1 if  it   could  successfully open  <filename>  for
**  reading and 0  to indicate  failure.   'OpenInput' will fail if  the file
**  does not exist or if you do not have permissions to read it.  'OpenInput'
**  may  also fail if  you have too  many files open at once.   It  is system
**  dependent how many are  too many, but  16  files should  work everywhere.
**
**  Directely after the 'OpenInput' call the variable  'Symbol' has the value
**  'S_ILLEGAL' to indicate that no symbol has yet been  read from this file.
**  The first symbol is read by 'Read' in the first call to 'Match' call.
**
**  You can open  '*stdin*' to  read  from the standard  input file, which is
**  usually the terminal, or '*errin*' to  read from the standard error file,
**  which  is  the  terminal  even if '*stdin*'  is  redirected from  a file.
**  'OpenInput' passes those  file names  to  'SyFopen' like any other  name,
**  they are  just  a  convention between the  main  and the system  package.
**  'SyFopen' and thus 'OpenInput' will  fail to open  '*errin*' if the  file
**  'stderr'  (Unix file  descriptor  2)  is  not a  terminal,  because  of a
**  redirection say, to avoid that break loops take their input from a file.
**
**  It is not neccessary to open the initial input  file, 'InitScanner' opens
**  '*stdin*' for  that purpose.  This  file on   the other   hand  cannot be
**  closed by 'CloseInput'.
*/
libGAP_UInt libGAP_OpenInput (
    const libGAP_Char *        filename )
{
    libGAP_Int                 libGAP_file;

    /* fail if we can not handle another open input file                   */
    if ( libGAP_TLS(libGAP_Input)+1 == libGAP_TLS(libGAP_InputFiles)+(sizeof(libGAP_TLS(libGAP_InputFiles))/sizeof(libGAP_TLS(libGAP_InputFiles)[0])) )
        return 0;

    /* in test mode keep reading from test input file for break loop input */
    if ( libGAP_TLS(libGAP_TestInput) != 0 && ! strcmp( filename, "*errin*" ) )
        return 1;

    /* try to open the input file                                          */
    libGAP_file = libGAP_SyFopen( filename, "r" );
    if ( libGAP_file == -1 )
        return 0;

    /* remember the current position in the current file                   */
    if ( libGAP_TLS(libGAP_Input)+1 != libGAP_TLS(libGAP_InputFiles) ) {
        libGAP_TLS(libGAP_Input)->ptr    = libGAP_TLS(libGAP_In);
        libGAP_TLS(libGAP_Input)->symbol = libGAP_TLS(libGAP_Symbol);
    }

    /* enter the file identifier and the file name                         */
    libGAP_TLS(libGAP_Input)++;
    libGAP_TLS(libGAP_Input)->isstream = 0;
    libGAP_TLS(libGAP_Input)->libGAP_file = libGAP_file;
    if (strcmp("*errin*", filename) && strcmp("*stdin*", filename))
      libGAP_TLS(libGAP_Input)->echo = 0;
    else
      libGAP_TLS(libGAP_Input)->echo = 1;
    libGAP_strlcpy( libGAP_TLS(libGAP_Input)->name, filename, sizeof(libGAP_TLS(libGAP_Input)->name) );
    libGAP_TLS(libGAP_Input)->gapname = (libGAP_Obj) 0;

    /* start with an empty line and no symbol                              */
    libGAP_TLS(libGAP_In) = libGAP_TLS(libGAP_Input)->line;
    libGAP_TLS(libGAP_In)[0] = libGAP_TLS(libGAP_In)[1] = '\0';
    libGAP_TLS(libGAP_Symbol) = libGAP_S_ILLEGAL;
    libGAP_TLS(libGAP_Input)->number = 1;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  OpenInputStream( <stream> ) . . . . . . .  open a stream as current input
**
**  The same as 'OpenInput' but for streams.
*/
libGAP_Obj libGAP_IsStringStream;

libGAP_UInt libGAP_OpenInputStream (
    libGAP_Obj                 stream )
{
    /* fail if we can not handle another open input file                   */
    if ( libGAP_TLS(libGAP_Input)+1 == libGAP_TLS(libGAP_InputFiles)+(sizeof(libGAP_TLS(libGAP_InputFiles))/sizeof(libGAP_TLS(libGAP_InputFiles)[0])) )
        return 0;

    /* remember the current position in the current file                   */
    if ( libGAP_TLS(libGAP_Input)+1 != libGAP_TLS(libGAP_InputFiles) ) {
        libGAP_TLS(libGAP_Input)->ptr    = libGAP_TLS(libGAP_In);
        libGAP_TLS(libGAP_Input)->symbol = libGAP_TLS(libGAP_Symbol);
    }

    /* enter the file identifier and the file name                         */
    libGAP_TLS(libGAP_Input)++;
    libGAP_TLS(libGAP_Input)->isstream = 1;
    libGAP_TLS(libGAP_Input)->stream = stream;
    libGAP_TLS(libGAP_Input)->isstringstream = (libGAP_CALL_1ARGS(libGAP_IsStringStream, stream) == libGAP_True);
    if (libGAP_TLS(libGAP_Input)->isstringstream) {
        libGAP_TLS(libGAP_Input)->sline = libGAP_ADDR_OBJ(stream)[2];
        libGAP_TLS(libGAP_Input)->spos = libGAP_INT_INTOBJ(libGAP_ADDR_OBJ(stream)[1]);
    }
    else {
        libGAP_TLS(libGAP_Input)->sline = 0;
    }
    libGAP_TLS(libGAP_Input)->libGAP_file = -1;
    libGAP_TLS(libGAP_Input)->echo = 0;
    libGAP_strlcpy( libGAP_TLS(libGAP_Input)->name, "stream", sizeof(libGAP_TLS(libGAP_Input)->name) );

    /* start with an empty line and no symbol                              */
    libGAP_TLS(libGAP_In) = libGAP_TLS(libGAP_Input)->line;
    libGAP_TLS(libGAP_In)[0] = libGAP_TLS(libGAP_In)[1] = '\0';
    libGAP_TLS(libGAP_Symbol) = libGAP_S_ILLEGAL;
    libGAP_TLS(libGAP_Input)->number = 1;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  CloseInput()  . . . . . . . . . . . . . . . . .  close current input file
**
**  'CloseInput'  will close the  current input file.   Subsequent input will
**  again be taken from the previous input file.   'CloseInput' will return 1
**  to indicate success.
**
**  'CloseInput' will not close the initial input file '*stdin*', and returns
**  0  if such  an  attempt is made.   This is  used in  'Error'  which calls
**  'CloseInput' until it returns 0, therebye closing all open input files.
**
**  Calling 'CloseInput' if the  corresponding  'OpenInput' call failed  will
**  close the current output file, which will lead to very strange behaviour.
*/
libGAP_UInt libGAP_CloseInput ( void )
{
    /* refuse to close the initial input file                              */
    if ( libGAP_TLS(libGAP_Input) == libGAP_TLS(libGAP_InputFiles) )
        return 0;

    /* refuse to close the test input file                                 */
    if ( libGAP_TLS(libGAP_Input) == libGAP_TLS(libGAP_TestInput) )
        return 0;

    /* close the input file                                                */
    if ( ! libGAP_TLS(libGAP_Input)->isstream ) {
        libGAP_SyFclose( libGAP_TLS(libGAP_Input)->libGAP_file );
    }

    /* don't keep GAP objects alive unnecessarily */
    libGAP_TLS(libGAP_Input)->gapname = 0;
    libGAP_TLS(libGAP_Input)->sline = 0;

    /* revert to last file                                                 */
    libGAP_TLS(libGAP_Input)--;
    libGAP_TLS(libGAP_In)     = libGAP_TLS(libGAP_Input)->ptr;
    libGAP_TLS(libGAP_Symbol) = libGAP_TLS(libGAP_Input)->symbol;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  FlushRestOfInputLine()  . . . . . . . . . . . . discard remainder of line
*/

void libGAP_FlushRestOfInputLine( void )
{
  libGAP_TLS(libGAP_In)[0] = libGAP_TLS(libGAP_In)[1] = '\0';
  /* TLS(Input)->number = 1; */
  libGAP_TLS(libGAP_Symbol) = libGAP_S_ILLEGAL;
}



/****************************************************************************
**
*F  OpenTest( <filename> )  . . . . . . . .  open an input file for test mode
**
*/
libGAP_UInt libGAP_OpenTest (
    const libGAP_Char *        filename )
{
    /* do not allow to nest test files                                     */
    if ( libGAP_TLS(libGAP_TestInput) != 0 )
        return 0;

    /* try to open the file as input file                                  */
    if ( ! libGAP_OpenInput( filename ) )
        return 0;

    /* remember this is a test input                                       */
    libGAP_TLS(libGAP_TestInput)   = libGAP_TLS(libGAP_Input);
    libGAP_TLS(libGAP_TestOutput)  = libGAP_TLS(libGAP_Output);
    libGAP_TLS(libGAP_TestLine)[0] = '\0';

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  OpenTestStream( <stream> )  . . . . .  open an input stream for test mode
**
*/
libGAP_UInt libGAP_OpenTestStream (
    libGAP_Obj                 stream )
{
    /* do not allow to nest test files                                     */
    if ( libGAP_TLS(libGAP_TestInput) != 0 )
        return 0;

    /* try to open the file as input file                                  */
    if ( ! libGAP_OpenInputStream( stream ) )
        return 0;

    /* remember this is a test input                                       */
    libGAP_TLS(libGAP_TestInput)   = libGAP_TLS(libGAP_Input);
    libGAP_TLS(libGAP_TestOutput)  = libGAP_TLS(libGAP_Output);
    libGAP_TLS(libGAP_TestLine)[0] = '\0';

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  CloseTest() . . . . . . . . . . . . . . . . . . close the test input file
**
*/
libGAP_UInt libGAP_CloseTest ( void )
{
    /* refuse to a non test file                                           */
    if ( libGAP_TLS(libGAP_TestInput) != libGAP_TLS(libGAP_Input) )
        return 0;

    /* close the input file                                                */
    if ( ! libGAP_TLS(libGAP_Input)->isstream ) {
        libGAP_SyFclose( libGAP_TLS(libGAP_Input)->libGAP_file );
    }

    /* don't keep GAP objects alive unnecessarily */
    libGAP_TLS(libGAP_Input)->gapname = 0;
    libGAP_TLS(libGAP_Input)->sline = 0;

    /* revert to last file                                                 */
    libGAP_TLS(libGAP_Input)--;
    libGAP_TLS(libGAP_In)     = libGAP_TLS(libGAP_Input)->ptr;
    libGAP_TLS(libGAP_Symbol) = libGAP_TLS(libGAP_Input)->symbol;

    /* we are no longer in test mode                                       */
    libGAP_TLS(libGAP_TestInput)   = 0;
    libGAP_TLS(libGAP_TestOutput)  = 0;
    libGAP_TLS(libGAP_TestLine)[0] = '\0';

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  OpenLog( <filename> ) . . . . . . . . . . . . . log interaction to a file
**
**  'OpenLog'  instructs  the scanner to   echo  all  input   from  the files
**  '*stdin*' and  '*errin*'  and  all  output to  the  files '*stdout*'  and
**  '*errout*' to the file with  name <filename>.  The  file is truncated  to
**  size 0 if it existed, otherwise it is created.
**
**  'OpenLog' returns 1 if it could  successfully open <filename> for writing
**  and 0  to indicate failure.   'OpenLog' will  fail if  you do  not   have
**  permissions  to create the file or   write to  it.  'OpenOutput' may also
**  fail if you have too many files open at once.  It is system dependent how
**  many   are too   many, but  16   files should  work everywhere.   Finally
**  'OpenLog' will fail if there is already a current logfile.
*/
static libGAP_TypOutputFile libGAP_LogFile;

libGAP_UInt libGAP_OpenLog (
    const libGAP_Char *        filename )
{

    /* refuse to open a logfile if we already log to one                   */
    if ( libGAP_TLS(libGAP_InputLog) != 0 || libGAP_TLS(libGAP_OutputLog) != 0 )
        return 0;

    /* try to open the file                                                */
    libGAP_TLS(libGAP_LogFile).libGAP_file = libGAP_SyFopen( filename, "w" );
    libGAP_TLS(libGAP_LogFile).isstream = 0;
    if ( libGAP_TLS(libGAP_LogFile).libGAP_file == -1 )
        return 0;

    libGAP_TLS(libGAP_InputLog)  = &libGAP_TLS(libGAP_LogFile);
    libGAP_TLS(libGAP_OutputLog) = &libGAP_TLS(libGAP_LogFile);

    /* otherwise indicate success                                          */
    return 1;
}


/****************************************************************************
**
*F  OpenLogStream( <stream> ) . . . . . . . . . . log interaction to a stream
**
**  The same as 'OpenLog' but for streams.
*/
static libGAP_TypOutputFile libGAP_LogStream;

libGAP_UInt libGAP_OpenLogStream (
    libGAP_Obj             stream )
{

    /* refuse to open a logfile if we already log to one                   */
    if ( libGAP_TLS(libGAP_InputLog) != 0 || libGAP_TLS(libGAP_OutputLog) != 0 )
        return 0;

    /* try to open the file                                                */
    libGAP_TLS(libGAP_LogStream).isstream = 1;
    libGAP_TLS(libGAP_LogStream).stream = stream;
    libGAP_TLS(libGAP_LogStream).libGAP_file = -1;

    libGAP_TLS(libGAP_InputLog)  = &libGAP_TLS(libGAP_LogStream);
    libGAP_TLS(libGAP_OutputLog) = &libGAP_TLS(libGAP_LogStream);

    /* otherwise indicate success                                          */
    return 1;
}


/****************************************************************************
**
*F  CloseLog()  . . . . . . . . . . . . . . . . . . close the current logfile
**
**  'CloseLog' closes the current logfile again, so that input from '*stdin*'
**  and '*errin*' and output to '*stdout*' and '*errout*' will no  longer  be
**  echoed to a file.  'CloseLog' will return 1 to indicate success.
**
**  'CloseLog' will fail if there is no logfile active and will return  0  in
**  this case.
*/
libGAP_UInt libGAP_CloseLog ( void )
{
    /* refuse to close a non existent logfile                              */
    if ( libGAP_TLS(libGAP_InputLog) == 0 || libGAP_TLS(libGAP_OutputLog) == 0 || libGAP_TLS(libGAP_InputLog) != libGAP_TLS(libGAP_OutputLog) )
        return 0;

    /* close the logfile                                                   */
    if ( ! libGAP_TLS(libGAP_InputLog)->isstream ) {
        libGAP_SyFclose( libGAP_TLS(libGAP_InputLog)->libGAP_file );
    }
    libGAP_TLS(libGAP_InputLog)  = 0;
    libGAP_TLS(libGAP_OutputLog) = 0;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  OpenInputLog( <filename> )  . . . . . . . . . . . . . log input to a file
**
**  'OpenInputLog'  instructs the  scanner  to echo  all input from the files
**  '*stdin*' and  '*errin*' to the file  with  name <filename>.  The file is
**  truncated to size 0 if it existed, otherwise it is created.
**
**  'OpenInputLog' returns 1  if it  could successfully open  <filename>  for
**  writing  and  0 to indicate failure.  'OpenInputLog' will fail  if you do
**  not have  permissions to create the file  or write to it.  'OpenInputLog'
**  may also fail  if you  have  too many  files open  at once.  It is system
**  dependent  how many are too many,  but 16 files  should work  everywhere.
**  Finally 'OpenInputLog' will fail if there is already a current logfile.
*/
static libGAP_TypOutputFile libGAP_InputLogFile;

libGAP_UInt libGAP_OpenInputLog (
    const libGAP_Char *        filename )
{

    /* refuse to open a logfile if we already log to one                   */
    if ( libGAP_TLS(libGAP_InputLog) != 0 )
        return 0;

    /* try to open the file                                                */
    libGAP_TLS(libGAP_InputLogFile).libGAP_file = libGAP_SyFopen( filename, "w" );
    libGAP_TLS(libGAP_InputLogFile).isstream = 0;
    if ( libGAP_TLS(libGAP_InputLogFile).libGAP_file == -1 )
        return 0;

    libGAP_TLS(libGAP_InputLog) = &libGAP_TLS(libGAP_InputLogFile);

    /* otherwise indicate success                                          */
    return 1;
}


/****************************************************************************
**
*F  OpenInputLogStream( <stream> )  . . . . . . . . . . log input to a stream
**
**  The same as 'OpenInputLog' but for streams.
*/
static libGAP_TypOutputFile libGAP_InputLogStream;

libGAP_UInt libGAP_OpenInputLogStream (
    libGAP_Obj                 stream )
{

    /* refuse to open a logfile if we already log to one                   */
    if ( libGAP_TLS(libGAP_InputLog) != 0 )
        return 0;

    /* try to open the file                                                */
    libGAP_TLS(libGAP_InputLogStream).isstream = 1;
    libGAP_TLS(libGAP_InputLogStream).stream = stream;
    libGAP_TLS(libGAP_InputLogStream).libGAP_file = -1;

    libGAP_TLS(libGAP_InputLog) = &libGAP_TLS(libGAP_InputLogStream);

    /* otherwise indicate success                                          */
    return 1;
}


/****************************************************************************
**
*F  CloseInputLog() . . . . . . . . . . . . . . . . close the current logfile
**
**  'CloseInputLog'  closes  the current  logfile again,  so  that input from
**  '*stdin*'  and   '*errin*'  will  no  longer   be  echoed   to  a   file.
**  'CloseInputLog' will return 1 to indicate success.
**
**  'CloseInputLog' will fail if there is no logfile active and will return 0
**  in this case.
*/
libGAP_UInt libGAP_CloseInputLog ( void )
{
    /* refuse to close a non existent logfile                              */
    if ( libGAP_TLS(libGAP_InputLog) == 0 )
        return 0;

    /* refuse to close a log opened with LogTo */
    if (libGAP_TLS(libGAP_InputLog) == libGAP_TLS(libGAP_OutputLog))
      return 0;
    
    /* close the logfile                                                   */
    if ( ! libGAP_TLS(libGAP_InputLog)->isstream ) {
        libGAP_SyFclose( libGAP_TLS(libGAP_InputLog)->libGAP_file );
    }

    libGAP_TLS(libGAP_InputLog) = 0;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  OpenOutputLog( <filename> )  . . . . . . . . . . .  log output to a file
**
**  'OpenInputLog'  instructs the  scanner to echo   all output to  the files
**  '*stdout*' and '*errout*' to the file with name  <filename>.  The file is
**  truncated to size 0 if it existed, otherwise it is created.
**
**  'OpenOutputLog'  returns 1 if it  could  successfully open <filename> for
**  writing and 0 to  indicate failure.  'OpenOutputLog'  will fail if you do
**  not have permissions to create the file  or write to it.  'OpenOutputLog'
**  may also  fail if you have  too many  files  open at  once.  It is system
**  dependent how many are  too many,  but  16 files should  work everywhere.
**  Finally 'OpenOutputLog' will fail if there is already a current logfile.
*/
static libGAP_TypOutputFile libGAP_OutputLogFile;

libGAP_UInt libGAP_OpenOutputLog (
    const libGAP_Char *        filename )
{

    /* refuse to open a logfile if we already log to one                   */
    if ( libGAP_TLS(libGAP_OutputLog) != 0 )
        return 0;

    /* try to open the file                                                */
    libGAP_TLS(libGAP_OutputLogFile).libGAP_file = libGAP_SyFopen( filename, "w" );
    libGAP_TLS(libGAP_OutputLogFile).isstream = 0;
    if ( libGAP_TLS(libGAP_OutputLogFile).libGAP_file == -1 )
        return 0;

    libGAP_TLS(libGAP_OutputLog) = &libGAP_TLS(libGAP_OutputLogFile);

    /* otherwise indicate success                                          */
    return 1;
}


/****************************************************************************
**
*F  OpenOutputLogStream( <stream> )  . . . . . . . .  log output to a stream
**
**  The same as 'OpenOutputLog' but for streams.
*/
static libGAP_TypOutputFile libGAP_OutputLogStream;

libGAP_UInt libGAP_OpenOutputLogStream (
    libGAP_Obj                 stream )
{

    /* refuse to open a logfile if we already log to one                   */
    if ( libGAP_TLS(libGAP_OutputLog) != 0 )
        return 0;

    /* try to open the file                                                */
    libGAP_TLS(libGAP_OutputLogStream).isstream = 1;
    libGAP_TLS(libGAP_OutputLogStream).stream = stream;
    libGAP_TLS(libGAP_OutputLogStream).libGAP_file = -1;

    libGAP_TLS(libGAP_OutputLog) = &libGAP_TLS(libGAP_OutputLogStream);

    /* otherwise indicate success                                          */
    return 1;
}


/****************************************************************************
**
*F  CloseOutputLog()  . . . . . . . . . . . . . . . close the current logfile
**
**  'CloseInputLog' closes   the current logfile   again, so  that output  to
**  '*stdout*'  and    '*errout*'  will no   longer  be   echoed to  a  file.
**  'CloseOutputLog' will return 1 to indicate success.
**
**  'CloseOutputLog' will fail if there is  no logfile active and will return
**  0 in this case.
*/
libGAP_UInt libGAP_CloseOutputLog ( void )
{
    /* refuse to close a non existent logfile                              */
    if ( libGAP_TLS(libGAP_OutputLog) == 0 )
        return 0;

    /* refuse to close a log opened with LogTo */
    if (libGAP_TLS(libGAP_OutputLog) == libGAP_TLS(libGAP_InputLog))
      return 0;

    /* close the logfile                                                   */
    if ( ! libGAP_TLS(libGAP_OutputLog)->isstream ) {
        libGAP_SyFclose( libGAP_TLS(libGAP_OutputLog)->libGAP_file );
    }

    libGAP_TLS(libGAP_OutputLog) = 0;

    /* indicate success                                                    */
    return 1;
}

libGAP_TypOutputFile*  libGAP_IgnoreStdoutErrout = NULL;

/****************************************************************************
**
*F  OpenOutput( <filename> )  . . . . . . . . . open a file as current output
**
**  'OpenOutput' opens the file  with the name  <filename> as current output.
**  All subsequent output will go  to that file, until either   it is  closed
**  again  with 'CloseOutput' or  another  file is  opened with 'OpenOutput'.
**  The file is truncated to size 0 if it existed, otherwise it  is  created.
**  'OpenOutput' does not  close  the  current file, i.e., if  <filename>  is
**  closed again, output will go again to the current output file.
**
**  'OpenOutput'  returns  1 if it  could  successfully  open  <filename> for
**  writing and 0 to indicate failure.  'OpenOutput' will fail if  you do not
**  have  permissions to create the  file or write   to it.  'OpenOutput' may
**  also   fail if you   have  too many files   open  at once.   It is system
**  dependent how many are too many, but 16 files should work everywhere.
**
**  You can open '*stdout*'  to write  to the standard output  file, which is
**  usually the terminal, or '*errout*' to write  to the standard error file,
**  which is the terminal  even   if '*stdout*'  is  redirected to   a  file.
**  'OpenOutput' passes  those  file names to 'SyFopen'  like any other name,
**  they are just a convention between the main and the system package.
**
**  It is not neccessary to open the initial output file, 'InitScanner' opens
**  '*stdout*' for that purpose.  This  file  on the other hand   can not  be
**  closed by 'CloseOutput'.
*/
libGAP_UInt libGAP_OpenOutput (
    const libGAP_Char *        filename )
{
    libGAP_Int                 libGAP_file;

    /* do nothing for stdout and errout if catched */
    if ( libGAP_TLS(libGAP_Output) != NULL && libGAP_TLS(libGAP_IgnoreStdoutErrout) == libGAP_TLS(libGAP_Output) &&
          ( strcmp( filename, "*errout*" ) == 0
           || strcmp( filename, "*stdout*" ) == 0 ) ) {
        return 1;
    }

    /* fail if we can not handle another open output file                  */
    if ( libGAP_TLS(libGAP_Output)+1==libGAP_TLS(libGAP_OutputFiles)+(sizeof(libGAP_TLS(libGAP_OutputFiles))/sizeof(libGAP_TLS(libGAP_OutputFiles)[0])) )
        return 0;

    /* in test mode keep printing to test output file for breakloop output */
    if ( libGAP_TLS(libGAP_TestInput) != 0 && ! strcmp( filename, "*errout*" ) )
        return 1;

    /* try to open the file                                                */
    libGAP_file = libGAP_SyFopen( filename, "w" );
    if ( libGAP_file == -1 )
        return 0;

    /* put the file on the stack, start at position 0 on an empty line     */
    if (libGAP_TLS(libGAP_Output) == 0L)
      libGAP_TLS(libGAP_Output) = libGAP_TLS(libGAP_OutputFiles);
    else
      libGAP_TLS(libGAP_Output)++;
    libGAP_TLS(libGAP_Output)->libGAP_file     = libGAP_file;
    libGAP_TLS(libGAP_Output)->line[0]  = '\0';
    libGAP_TLS(libGAP_Output)->pos      = 0;
    libGAP_TLS(libGAP_Output)->indent   = 0;
    libGAP_TLS(libGAP_Output)->isstream = 0;
    libGAP_TLS(libGAP_Output)->format   = 1;

    /* variables related to line splitting, very bad place to split        */
    libGAP_TLS(libGAP_Output)->hints[0] = -1;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  OpenOutputStream( <stream> )  . . . . . . open a stream as current output
**
**  The same as 'OpenOutput' but for streams.
*/

libGAP_Obj libGAP_PrintFormattingStatus;

libGAP_UInt libGAP_OpenOutputStream (
    libGAP_Obj                 stream )
{
    /* fail if we can not handle another open output file                  */
    if ( libGAP_TLS(libGAP_Output)+1==libGAP_TLS(libGAP_OutputFiles)+(sizeof(libGAP_TLS(libGAP_OutputFiles))/sizeof(libGAP_TLS(libGAP_OutputFiles)[0])) )
        return 0;

    /* put the file on the stack, start at position 0 on an empty line     */
    libGAP_TLS(libGAP_Output)++;
    libGAP_TLS(libGAP_Output)->stream   = stream;
    libGAP_TLS(libGAP_Output)->isstringstream = (libGAP_CALL_1ARGS(libGAP_IsStringStream, stream) == libGAP_True);
    libGAP_TLS(libGAP_Output)->format   = (libGAP_CALL_1ARGS(libGAP_PrintFormattingStatus, stream) == libGAP_True);
    libGAP_TLS(libGAP_Output)->line[0]  = '\0';
    libGAP_TLS(libGAP_Output)->pos      = 0;
    libGAP_TLS(libGAP_Output)->indent   = 0;
    libGAP_TLS(libGAP_Output)->isstream = 1;

    /* variables related to line splitting, very bad place to split        */
    libGAP_TLS(libGAP_Output)->hints[0] = -1;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  CloseOutput() . . . . . . . . . . . . . . . . . close current output file
**
**  'CloseOutput' will  first flush all   pending output and  then  close the
**  current  output  file.   Subsequent output will  again go to the previous
**  output file.  'CloseOutput' returns 1 to indicate success.
**
**  'CloseOutput' will  not  close the  initial output file   '*stdout*', and
**  returns 0 if such attempt is made.  This  is  used in 'Error' which calls
**  'CloseOutput' until it returns 0, thereby closing all open output files.
**
**  Calling 'CloseOutput' if the corresponding 'OpenOutput' call failed  will
**  close the current output file, which will lead to very strange behaviour.
**  On the other  hand if you  forget  to call  'CloseOutput' at the end of a
**  'PrintTo' call or an error will not yield much better results.
*/
libGAP_UInt libGAP_CloseOutput ( void )
{
    /* silently refuse to close the test output file this is probably
         an attempt to close *errout* which is silently not opened, so
         lets silently not close it  */
    if ( libGAP_TLS(libGAP_Output) == libGAP_TLS(libGAP_TestOutput) )
        return 1;
    /* and similarly */
    if ( libGAP_TLS(libGAP_IgnoreStdoutErrout) == libGAP_TLS(libGAP_Output) )
        return 1;

    /* refuse to close the initial output file '*stdout*'                  */
    if ( libGAP_TLS(libGAP_Output) == libGAP_TLS(libGAP_OutputFiles) )
        return 0;

    /* flush output and close the file                                     */
    libGAP_Pr( "%c", (libGAP_Int)'\03', 0L );
    if ( ! libGAP_TLS(libGAP_Output)->isstream ) {
        libGAP_SyFclose( libGAP_TLS(libGAP_Output)->libGAP_file );
    }

    /* revert to previous output file and indicate success                 */
    libGAP_TLS(libGAP_Output)--;
    return 1;
}


/****************************************************************************
**
*F  OpenAppend( <filename> )  . . open a file as current output for appending
**
**  'OpenAppend' opens the file  with the name  <filename> as current output.
**  All subsequent output will go  to that file, until either   it is  closed
**  again  with 'CloseOutput' or  another  file is  opened with 'OpenOutput'.
**  Unlike 'OpenOutput' 'OpenAppend' does not truncate the file to size 0  if
**  it exists.  Appart from that 'OpenAppend' is equal to 'OpenOutput' so its
**  description applies to 'OpenAppend' too.
*/
libGAP_UInt libGAP_OpenAppend (
    const libGAP_Char *        filename )
{
    libGAP_Int                 libGAP_file;

    /* fail if we can not handle another open output file                  */
    if ( libGAP_TLS(libGAP_Output)+1==libGAP_TLS(libGAP_OutputFiles)+(sizeof(libGAP_TLS(libGAP_OutputFiles))/sizeof(libGAP_TLS(libGAP_OutputFiles)[0])) )
        return 0;

    /* in test mode keep printing to test output file for breakloop output */
    if ( libGAP_TLS(libGAP_TestInput) != 0 && ! strcmp( filename, "*errout*" ) )
        return 1;

    /* try to open the file                                                */
    libGAP_file = libGAP_SyFopen( filename, "a" );
    if ( libGAP_file == -1 )
        return 0;

    /* put the file on the stack, start at position 0 on an empty line     */
    libGAP_TLS(libGAP_Output)++;
    libGAP_TLS(libGAP_Output)->libGAP_file     = libGAP_file;
    libGAP_TLS(libGAP_Output)->line[0]  = '\0';
    libGAP_TLS(libGAP_Output)->pos      = 0;
    libGAP_TLS(libGAP_Output)->indent   = 0;
    libGAP_TLS(libGAP_Output)->isstream = 0;

    /* variables related to line splitting, very bad place to split        */
    libGAP_TLS(libGAP_Output)->hints[0] = -1;

    /* indicate success                                                    */
    return 1;
}


/****************************************************************************
**
*F  OpenAppendStream( <stream> )  . . . . . . open a stream as current output
**
**  The same as 'OpenAppend' but for streams.
*/
libGAP_UInt libGAP_OpenAppendStream (
    libGAP_Obj                 stream )
{
    return libGAP_OpenOutputStream(stream);
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * input functions  * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  ReadLineFunc  . . . . . . . . . . . . . . . . . . . . . . . .  'ReadLine'
*/
libGAP_Obj libGAP_ReadLineFunc;


/****************************************************************************
**
*F  GetLine2( <input>, <buffer>, <length> ) . . . . . . . . get a line, local
*/
static libGAP_Int libGAP_GetLine2 (
    libGAP_TypInputFile *          input,
    libGAP_Char *                  buffer,
    libGAP_UInt                    length )
{

    if ( input->isstream ) {
        if ( input->sline == 0
          || libGAP_GET_LEN_STRING(input->sline) <= input->spos )
        {
            input->sline = libGAP_CALL_1ARGS( libGAP_ReadLineFunc, input->stream );
            input->spos  = 0;
        }
        if ( input->sline == libGAP_Fail || ! libGAP_IS_STRING(input->sline) ) {
            return 0;
        }
        else {
            libGAP_ConvString(input->sline);
            /* we now allow that input->sline actually contains several lines,
               e.g., it can be a  string from a string stream  */
            {
                /***  probably this can be a bit more optimized  ***/
                register libGAP_Char * ptr, * bptr;
                register libGAP_UInt count, len, max, cbuf;
                /* start position in buffer */
                for(cbuf = 0; buffer[cbuf]; cbuf++);
                /* copy piece of input->sline into buffer and adjust counters */
                for(count = input->spos,
                    ptr = (libGAP_Char *)libGAP_CHARS_STRING(input->sline) + count,
                    len = libGAP_GET_LEN_STRING(input->sline),
                    max = length-2,
                    bptr = buffer + cbuf;
                    cbuf < max && count < len
                                  && *ptr != '\n' && *ptr != '\r';
                    *bptr = *ptr, cbuf++, ptr++, bptr++, count++);
                /* we also copy an end of line if there is one */
                if (*ptr == '\n' || *ptr == '\r') {
                    buffer[cbuf] = *ptr;
                    cbuf++;
                    count++;
                }
                buffer[cbuf] = '\0';
                input->spos = count;
                /* if input->stream is a string stream, we have to adjust the
                   position counter in the stream object as well */
                if (input->isstringstream) {
                    libGAP_ADDR_OBJ(input->stream)[1] = libGAP_INTOBJ_INT(count);
                }
            }
        }
    }
    else {
        if ( ! libGAP_SyFgets( buffer, length, input->libGAP_file ) ) {
            return 0;
        }
    }
    return 1;
}


/****************************************************************************
**
*F  GetLine() . . . . . . . . . . . . . . . . . . . . . . . get a line, local
**
**  'GetLine'  fetches another  line from  the  input 'Input' into the buffer
**  'Input->line', sets the pointer 'In' to  the beginning of this buffer and
**  returns the first character from the line.
**
**  If   the input file is  '*stdin*'   or '*errin*' 'GetLine'  first  prints
**  'Prompt', unless it is '*stdin*' and GAP was called with option '-q'.
**
**  If there is an  input logfile in use  and the input  file is '*stdin*' or
**  '*errin*' 'GetLine' echoes the new line to the logfile.
*/
extern void libGAP_PutLine2(
    libGAP_TypOutputFile *         output,
    const libGAP_Char *            line,
    libGAP_UInt                    len   );

libGAP_Int libGAP_HELPSubsOn = 1;

libGAP_Char libGAP_GetLine ( void )
{
    libGAP_Char            buf[200];
    libGAP_Char *          p;
    libGAP_Char *          q;

    /* if file is '*stdin*' or '*errin*' print the prompt and flush it     */
    /* if the GAP function `PrintPromptHook' is defined then it is called  */
    /* for printing the prompt, see also `EndLineHook'                     */
    if ( ! libGAP_TLS(libGAP_Input)->isstream ) {
       if ( libGAP_TLS(libGAP_Input)->libGAP_file == 0 ) {
            if ( ! libGAP_SyQuiet ) {
                if (libGAP_TLS(libGAP_Output)->pos > 0)
                    libGAP_Pr("\n", 0L, 0L);
                if ( libGAP_PrintPromptHook )
                     libGAP_Call0ArgsInNewReader( libGAP_PrintPromptHook );
                else
                     libGAP_Pr( "%s%c", (libGAP_Int)libGAP_TLS(libGAP_Prompt), (libGAP_Int)'\03' );
            } else
                libGAP_Pr( "%c", (libGAP_Int)'\03', 0L );
        }
        else if ( libGAP_TLS(libGAP_Input)->libGAP_file == 2 ) {
            if (libGAP_TLS(libGAP_Output)->pos > 0)
                libGAP_Pr("\n", 0L, 0L);
            if ( libGAP_PrintPromptHook )
                 libGAP_Call0ArgsInNewReader( libGAP_PrintPromptHook );
            else
                 libGAP_Pr( "%s%c", (libGAP_Int)libGAP_TLS(libGAP_Prompt), (libGAP_Int)'\03' );
        }
    }

    /* bump the line number                                                */
    if ( libGAP_TLS(libGAP_Input)->line < libGAP_TLS(libGAP_In) && (*(libGAP_TLS(libGAP_In)-1) == '\n' || *(libGAP_TLS(libGAP_In)-1) == '\r') ) {
        libGAP_TLS(libGAP_Input)->number++;
    }

    /* initialize 'TLS(In)', no errors on this line so far                      */
    libGAP_TLS(libGAP_In) = libGAP_TLS(libGAP_Input)->line;  libGAP_TLS(libGAP_In)[0] = '\0';
    libGAP_TLS(libGAP_NrErrLine) = 0;

    /* read a line from an ordinary input file                             */
    if ( libGAP_TLS(libGAP_TestInput) != libGAP_TLS(libGAP_Input) ) {

        /* try to read a line                                              */
        if ( ! libGAP_GetLine2( libGAP_TLS(libGAP_Input), libGAP_TLS(libGAP_Input)->line, sizeof(libGAP_TLS(libGAP_Input)->line) ) ) {
            libGAP_TLS(libGAP_In)[0] = '\377';  libGAP_TLS(libGAP_In)[1] = '\0';
        }


        /* convert '?' at the beginning into 'HELP'
           (if not inside reading long string which may have line
           or chunk from GetLine starting with '?')                        */

        if ( libGAP_TLS(libGAP_In)[0] == '?' && libGAP_TLS(libGAP_HELPSubsOn) == 1) {
            libGAP_strlcpy( buf, libGAP_TLS(libGAP_In)+1, sizeof(buf) );
            strcpy( libGAP_TLS(libGAP_In), "HELP(\"" );
            for ( p = libGAP_TLS(libGAP_In)+6,  q = buf;  *q;  q++ ) {
                if ( *q != '"' && *q != '\n' ) {
                    *p++ = *q;
                }
                else if ( *q == '"' ) {
                    *p++ = '\\';
                    *p++ = *q;
                }
            }
            *p = '\0';
            /* FIXME: We should do bounds checking, but don't know what 'In' points to */
            strcat( libGAP_TLS(libGAP_In), "\");\n" );
        }

        /* if necessary echo the line to the logfile                      */
        if( libGAP_TLS(libGAP_InputLog) != 0 && libGAP_TLS(libGAP_Input)->echo == 1)
            if ( !(libGAP_TLS(libGAP_In)[0] == '\377' && libGAP_TLS(libGAP_In)[1] == '\0') )
            libGAP_PutLine2( libGAP_TLS(libGAP_InputLog), libGAP_TLS(libGAP_In), strlen(libGAP_TLS(libGAP_In)) );

                /*      if ( ! TLS(Input)->isstream ) {
          if ( TLS(InputLog) != 0 && ! TLS(Input)->isstream ) {
            if ( TLS(Input)->file == 0 || TLS(Input)->file == 2 ) {
              PutLine2( TLS(InputLog), TLS(In) );
            }
            }
            } */

    }

    /* read a line for test input file                                     */
    else {

        /* continue until we got an input line                             */
        while ( libGAP_TLS(libGAP_In)[0] == '\0' ) {

            /* there may be one line waiting                               */
            if ( libGAP_TLS(libGAP_TestLine)[0] != '\0' ) {
                strncat( libGAP_TLS(libGAP_In), libGAP_TLS(libGAP_TestLine), sizeof(libGAP_TLS(libGAP_Input)->line) );
                libGAP_TLS(libGAP_TestLine)[0] = '\0';
            }

            /* otherwise try to read a line                                */
            else {
                if ( ! libGAP_GetLine2(libGAP_TLS(libGAP_Input), libGAP_TLS(libGAP_Input)->line, sizeof(libGAP_TLS(libGAP_Input)->line)) ) {
                    libGAP_TLS(libGAP_In)[0] = '\377';  libGAP_TLS(libGAP_In)[1] = '\0';
                }
            }

            /* if the line starts with a prompt its an input line          */
            if      ( libGAP_TLS(libGAP_In)[0] == 'g' && libGAP_TLS(libGAP_In)[1] == 'a' && libGAP_TLS(libGAP_In)[2] == 'p'
                   && libGAP_TLS(libGAP_In)[3] == '>' && libGAP_TLS(libGAP_In)[4] == ' ' ) {
                libGAP_TLS(libGAP_In) = libGAP_TLS(libGAP_In) + 5;
            }
            else if ( libGAP_TLS(libGAP_In)[0] == '>' && libGAP_TLS(libGAP_In)[1] == ' ' ) {
                libGAP_TLS(libGAP_In) = libGAP_TLS(libGAP_In) + 2;
            }

            /* if the line is not empty or a comment, print it             */
            else if ( libGAP_TLS(libGAP_In)[0] != '\n' && libGAP_TLS(libGAP_In)[0] != '#' && libGAP_TLS(libGAP_In)[0] != '\377' ) {
                /* Commented out by AK
                char obuf[8];
                snprintf(obuf, sizeof(obuf), "-%5i:\n- ", (int)TLS(TestInput)->number++);
                PutLine2( TLS(TestOutput), obuf, 7 );
                */
                libGAP_PutLine2( libGAP_TLS(libGAP_TestOutput), "- ", 2 );
                libGAP_PutLine2( libGAP_TLS(libGAP_TestOutput), libGAP_TLS(libGAP_In), strlen(libGAP_TLS(libGAP_In)) );
                libGAP_TLS(libGAP_In)[0] = '\0';
            }

        }

    }

    /* return the current character                                        */
    return *libGAP_TLS(libGAP_In);
}


/****************************************************************************
**
*F  GET_CHAR()  . . . . . . . . . . . . . . . . get the next character, local
**
**  'GET_CHAR' returns the next character from  the current input file.  This
**  character is afterwords also available as '*In'.
**
**  For efficiency  reasons 'GET_CHAR' is a  macro that  just  increments the
**  pointer 'In'  and checks that  there is another  character.  If  not, for
**  example at the end a line, 'GET_CHAR' calls 'GetLine' to fetch a new line
**  from the input file.
*/

static libGAP_Char libGAP_Pushback = '\0';
static libGAP_Char *libGAP_RealIn;

static inline void libGAP_GET_CHAR( void ) {
  if (libGAP_TLS(libGAP_In) == &libGAP_Pushback) {
      libGAP_TLS(libGAP_In) = libGAP_RealIn;
  } else
    libGAP_TLS(libGAP_In)++;
  if (!*libGAP_TLS(libGAP_In))
    libGAP_GetLine();
}

static inline void libGAP_UNGET_CHAR( libGAP_Char c ) {
  assert(libGAP_TLS(libGAP_In) != &libGAP_Pushback);
  libGAP_Pushback = c;
  libGAP_RealIn = libGAP_TLS(libGAP_In);
  libGAP_TLS(libGAP_In) = &libGAP_Pushback;
}


/****************************************************************************
**
*F  GetIdent()  . . . . . . . . . . . . . get an identifier or keyword, local
**
**  'GetIdent' reads   an identifier from  the current  input  file  into the
**  variable 'TLS(Value)' and sets 'Symbol' to 'S_IDENT'.   The first character of
**  the   identifier  is  the current character  pointed to  by 'In'.  If the
**  characters make  up   a  keyword 'GetIdent'  will  set   'Symbol'  to the
**  corresponding value.  The parser will ignore 'TLS(Value)' in this case.
**
**  An  identifier consists of a letter  followed by more letters, digits and
**  underscores '_'.  An identifier is terminated by the first  character not
**  in this  class.  The escape sequence '\<newline>'  is ignored,  making it
**  possible to split  long identifiers  over multiple lines.  The  backslash
**  '\' can be used  to include special characters like  '('  in identifiers.
**  For example 'G\(2\,5\)' is an identifier not a call to a function 'G'.
**
**  The size  of 'TLS(Value)' limits the  number  of significant characters  in an
**  identifier.   If  an  identifier   has more characters    'GetIdent' will
**  silently truncate it.
**
**  After reading the identifier 'GetIdent'  looks at the  first and the last
**  character  of  'TLS(Value)' to see if  it  could possibly  be  a keyword.  For
**  example 'test'  could  not be  a  keyword  because there  is  no  keyword
**  starting and ending with a 't'.  After that  test either 'GetIdent' knows
**  that 'TLS(Value)' is not a keyword, or there is a unique possible keyword that
**  could match, because   no two  keywords  have  identical  first and  last
**  characters.  For example if 'TLS(Value)' starts with 'f' and ends with 'n' the
**  only possible keyword  is 'function'.   Thus in this case  'GetIdent' can
**  decide with one string comparison if 'TLS(Value)' holds a keyword or not.
*/
extern void libGAP_GetSymbol ( void );

typedef struct {const libGAP_Char *name; libGAP_UInt sym;} libGAP_s_keyword;

static const libGAP_s_keyword libGAP_AllKeywords[] = {
  {"and",       libGAP_S_AND},
  {"atomic",    libGAP_S_ATOMIC},
  {"break",     libGAP_S_BREAK},
  {"continue",  libGAP_S_CONTINUE},
  {"do",        libGAP_S_DO},
  {"elif",      libGAP_S_ELIF},
  {"else",      libGAP_S_ELSE},
  {"end",       libGAP_S_END},
  {"false",     libGAP_S_FALSE},
  {"fi",        libGAP_S_FI},
  {"for",       libGAP_S_FOR},
  {"function",  libGAP_S_FUNCTION},
  {"if",        libGAP_S_IF},
  {"in",        libGAP_S_IN},
  {"local",     libGAP_S_LOCAL},
  {"mod",       libGAP_S_MOD},
  {"not",       libGAP_S_NOT},
  {"od",        libGAP_S_OD},
  {"or",        libGAP_S_OR},
  {"readonly",  libGAP_S_READONLY},
  {"readwrite", libGAP_S_READWRITE},
  {"rec",       libGAP_S_REC},
  {"repeat",    libGAP_S_REPEAT},
  {"return",    libGAP_S_RETURN},
  {"then",      libGAP_S_THEN},
  {"true",      libGAP_S_TRUE},
  {"until",     libGAP_S_UNTIL},
  {"while",     libGAP_S_WHILE},
  {"quit",      libGAP_S_QUIT},
  {"QUIT",      libGAP_S_QQUIT},
  {"IsBound",   libGAP_S_ISBOUND},
  {"Unbind",    libGAP_S_UNBIND},
  {"TryNextMethod", libGAP_S_TRYNEXT},
  {"Info",      libGAP_S_INFO},
  {"Assert",    libGAP_S_ASSERT}};


static int libGAP_IsIdent(char c) {
    return libGAP_IsAlpha(c) || c == '_' || c == '@';
}

void libGAP_GetIdent ( void )
{
    libGAP_Int                 i, fetch;
    libGAP_Int                 isQuoted;

    /* initially it could be a keyword                                     */
    isQuoted = 0;

    /* read all characters into 'TLS(Value)'                                    */
    for ( i=0; libGAP_IsIdent(*libGAP_TLS(libGAP_In)) || libGAP_IsDigit(*libGAP_TLS(libGAP_In)) || *libGAP_TLS(libGAP_In)=='\\'; i++ ) {

        fetch = 1;
        /* handle escape sequences                                         */
        /* we ignore '\ newline' by decrementing i, except at the
           very start of the identifier, when we cannot do that
           so we recurse instead                                           */
        if ( *libGAP_TLS(libGAP_In) == '\\' ) {
            libGAP_GET_CHAR();
            if      ( *libGAP_TLS(libGAP_In) == '\n' && i == 0 )  { libGAP_GetSymbol();  return; }
            else if ( *libGAP_TLS(libGAP_In) == '\r' )  {
                libGAP_GET_CHAR();
                if  ( *libGAP_TLS(libGAP_In) == '\n' )  {
                     if (i == 0) { libGAP_GetSymbol();  return; }
                     else i--;
                }
                else  {libGAP_TLS(libGAP_Value)[i] = '\r'; fetch = 0;}
            }
            else if ( *libGAP_TLS(libGAP_In) == '\n' && i < libGAP_SAFE_VALUE_SIZE-1 )  i--;
            else if ( *libGAP_TLS(libGAP_In) == 'n'  && i < libGAP_SAFE_VALUE_SIZE-1 )  libGAP_TLS(libGAP_Value)[i] = '\n';
            else if ( *libGAP_TLS(libGAP_In) == 't'  && i < libGAP_SAFE_VALUE_SIZE-1 )  libGAP_TLS(libGAP_Value)[i] = '\t';
            else if ( *libGAP_TLS(libGAP_In) == 'r'  && i < libGAP_SAFE_VALUE_SIZE-1 )  libGAP_TLS(libGAP_Value)[i] = '\r';
            else if ( *libGAP_TLS(libGAP_In) == 'b'  && i < libGAP_SAFE_VALUE_SIZE-1 )  libGAP_TLS(libGAP_Value)[i] = '\b';
            else if ( i < libGAP_SAFE_VALUE_SIZE-1 )  {
                libGAP_TLS(libGAP_Value)[i] = *libGAP_TLS(libGAP_In);
                isQuoted = 1;
            }
        }

        /* put normal chars into 'TLS(Value)' but only if there is room         */
        else {
            if ( i < libGAP_SAFE_VALUE_SIZE-1 )  libGAP_TLS(libGAP_Value)[i] = *libGAP_TLS(libGAP_In);
        }

        /* read the next character                                         */
        if (fetch) libGAP_GET_CHAR();

    }

    /* terminate the identifier and lets assume that it is not a keyword   */
    if ( i < libGAP_SAFE_VALUE_SIZE-1 )
        libGAP_TLS(libGAP_Value)[i] = '\0';
    else {
        libGAP_SyntaxError("Identifiers in GAP must consist of less than 1023 characters.");
        i =  libGAP_SAFE_VALUE_SIZE-1;
        libGAP_TLS(libGAP_Value)[i] = '\0';
    }
    libGAP_TLS(libGAP_Symbol) = libGAP_S_IDENT;

    /* now check if 'TLS(Value)' holds a keyword                                */
    switch ( 256*libGAP_TLS(libGAP_Value)[0]+libGAP_TLS(libGAP_Value)[i-1] ) {
    case 256*'a'+'d': if(!strcmp(libGAP_TLS(libGAP_Value),"and"))     libGAP_TLS(libGAP_Symbol)=libGAP_S_AND;     break;
    case 256*'a'+'c': if(!strcmp(libGAP_TLS(libGAP_Value),"atomic"))  libGAP_TLS(libGAP_Symbol)=libGAP_S_ATOMIC;  break;
    case 256*'b'+'k': if(!strcmp(libGAP_TLS(libGAP_Value),"break"))   libGAP_TLS(libGAP_Symbol)=libGAP_S_BREAK;   break;
    case 256*'c'+'e': if(!strcmp(libGAP_TLS(libGAP_Value),"continue"))   libGAP_TLS(libGAP_Symbol)=libGAP_S_CONTINUE;   break;
    case 256*'d'+'o': if(!strcmp(libGAP_TLS(libGAP_Value),"do"))      libGAP_TLS(libGAP_Symbol)=libGAP_S_DO;      break;
    case 256*'e'+'f': if(!strcmp(libGAP_TLS(libGAP_Value),"elif"))    libGAP_TLS(libGAP_Symbol)=libGAP_S_ELIF;    break;
    case 256*'e'+'e': if(!strcmp(libGAP_TLS(libGAP_Value),"else"))    libGAP_TLS(libGAP_Symbol)=libGAP_S_ELSE;    break;
    case 256*'e'+'d': if(!strcmp(libGAP_TLS(libGAP_Value),"end"))     libGAP_TLS(libGAP_Symbol)=libGAP_S_END;     break;
    case 256*'f'+'e': if(!strcmp(libGAP_TLS(libGAP_Value),"false"))   libGAP_TLS(libGAP_Symbol)=libGAP_S_FALSE;   break;
    case 256*'f'+'i': if(!strcmp(libGAP_TLS(libGAP_Value),"fi"))      libGAP_TLS(libGAP_Symbol)=libGAP_S_FI;      break;
    case 256*'f'+'r': if(!strcmp(libGAP_TLS(libGAP_Value),"for"))     libGAP_TLS(libGAP_Symbol)=libGAP_S_FOR;     break;
    case 256*'f'+'n': if(!strcmp(libGAP_TLS(libGAP_Value),"function"))libGAP_TLS(libGAP_Symbol)=libGAP_S_FUNCTION;break;
    case 256*'i'+'f': if(!strcmp(libGAP_TLS(libGAP_Value),"if"))      libGAP_TLS(libGAP_Symbol)=libGAP_S_IF;      break;
    case 256*'i'+'n': if(!strcmp(libGAP_TLS(libGAP_Value),"in"))      libGAP_TLS(libGAP_Symbol)=libGAP_S_IN;      break;
    case 256*'l'+'l': if(!strcmp(libGAP_TLS(libGAP_Value),"local"))   libGAP_TLS(libGAP_Symbol)=libGAP_S_LOCAL;   break;
    case 256*'m'+'d': if(!strcmp(libGAP_TLS(libGAP_Value),"mod"))     libGAP_TLS(libGAP_Symbol)=libGAP_S_MOD;     break;
    case 256*'n'+'t': if(!strcmp(libGAP_TLS(libGAP_Value),"not"))     libGAP_TLS(libGAP_Symbol)=libGAP_S_NOT;     break;
    case 256*'o'+'d': if(!strcmp(libGAP_TLS(libGAP_Value),"od"))      libGAP_TLS(libGAP_Symbol)=libGAP_S_OD;      break;
    case 256*'o'+'r': if(!strcmp(libGAP_TLS(libGAP_Value),"or"))      libGAP_TLS(libGAP_Symbol)=libGAP_S_OR;      break;
    case 256*'r'+'e': if(!strcmp(libGAP_TLS(libGAP_Value),"readwrite")) libGAP_TLS(libGAP_Symbol)=libGAP_S_READWRITE;     break;
    case 256*'r'+'y': if(!strcmp(libGAP_TLS(libGAP_Value),"readonly"))  libGAP_TLS(libGAP_Symbol)=libGAP_S_READONLY;     break;
    case 256*'r'+'c': if(!strcmp(libGAP_TLS(libGAP_Value),"rec"))     libGAP_TLS(libGAP_Symbol)=libGAP_S_REC;     break;
    case 256*'r'+'t': if(!strcmp(libGAP_TLS(libGAP_Value),"repeat"))  libGAP_TLS(libGAP_Symbol)=libGAP_S_REPEAT;  break;
    case 256*'r'+'n': if(!strcmp(libGAP_TLS(libGAP_Value),"return"))  libGAP_TLS(libGAP_Symbol)=libGAP_S_RETURN;  break;
    case 256*'t'+'n': if(!strcmp(libGAP_TLS(libGAP_Value),"then"))    libGAP_TLS(libGAP_Symbol)=libGAP_S_THEN;    break;
    case 256*'t'+'e': if(!strcmp(libGAP_TLS(libGAP_Value),"true"))    libGAP_TLS(libGAP_Symbol)=libGAP_S_TRUE;    break;
    case 256*'u'+'l': if(!strcmp(libGAP_TLS(libGAP_Value),"until"))   libGAP_TLS(libGAP_Symbol)=libGAP_S_UNTIL;   break;
    case 256*'w'+'e': if(!strcmp(libGAP_TLS(libGAP_Value),"while"))   libGAP_TLS(libGAP_Symbol)=libGAP_S_WHILE;   break;
    case 256*'q'+'t': if(!strcmp(libGAP_TLS(libGAP_Value),"quit"))    libGAP_TLS(libGAP_Symbol)=libGAP_S_QUIT;    break;
    case 256*'Q'+'T': if(!strcmp(libGAP_TLS(libGAP_Value),"QUIT"))    libGAP_TLS(libGAP_Symbol)=libGAP_S_QQUIT;   break;

    case 256*'I'+'d': if(!strcmp(libGAP_TLS(libGAP_Value),"IsBound")) libGAP_TLS(libGAP_Symbol)=libGAP_S_ISBOUND; break;
    case 256*'U'+'d': if(!strcmp(libGAP_TLS(libGAP_Value),"Unbind"))  libGAP_TLS(libGAP_Symbol)=libGAP_S_UNBIND;  break;
    case 256*'T'+'d': if(!strcmp(libGAP_TLS(libGAP_Value),"TryNextMethod"))
                                                     libGAP_TLS(libGAP_Symbol)=libGAP_S_TRYNEXT; break;
    case 256*'I'+'o': if(!strcmp(libGAP_TLS(libGAP_Value),"Info"))    libGAP_TLS(libGAP_Symbol)=libGAP_S_INFO;    break;
    case 256*'A'+'t': if(!strcmp(libGAP_TLS(libGAP_Value),"Assert"))  libGAP_TLS(libGAP_Symbol)=libGAP_S_ASSERT;  break;

    default: ;
    }

    /* if it is quoted it is an identifier                                 */
    if ( isQuoted )  libGAP_TLS(libGAP_Symbol) = libGAP_S_IDENT;


}


/******************************************************************************
*F  GetNumber()  . . . . . . . . . . . . . .  get an integer or float literal
**
**  'GetNumber' reads  a number from  the  current  input file into the
**  variable  'TLS(Value)' and sets  'Symbol' to 'S_INT', 'S_PARTIALINT',
**  'S_FLOAT' or 'S_PARTIALFLOAT'.   The first character of
**  the number is the current character pointed to by 'In'.
**
**  If the sequence contains characters which do not match the regular expression
**  [0-9]+.?[0-9]*([edqEDQ][+-]?[0-9]+)? 'GetNumber'  will
**  interpret the sequence as an identifier and set 'Symbol' to 'S_IDENT'.
**
**  As we read, we keep track of whether we have seen a . or exponent notation
**  and so whether we will return S_[PARTIAL]INT or S_[PARTIAL]FLOAT.
**
**  When TLS(Value) is  completely filled we have to check  if the reading of
**  the number  is complete  or not to  decide whether to return a PARTIAL type.
**
**  The argument reflects how far we are through reading a possibly very long number
**  literal. 0 indicates that nothing has been read. 1 that at least one digit has been
**  read, but no decimal point. 2 that a decimal point has been read with no digits before
**  or after it. 3 a decimal point and at least one digit, but no exponential indicator
**  4 an exponential indicator  but no exponent digits and 5 an exponential indicator and
**  at least one exponent digit.
**
*/
static libGAP_Char libGAP_GetCleanedChar( libGAP_UInt *wasEscaped ) {
  libGAP_GET_CHAR();
  *wasEscaped = 0;
  if (*libGAP_TLS(libGAP_In) == '\\') {
    libGAP_GET_CHAR();
    if      ( *libGAP_TLS(libGAP_In) == '\n')
      return libGAP_GetCleanedChar(wasEscaped);
    else if ( *libGAP_TLS(libGAP_In) == '\r' )  {
      libGAP_GET_CHAR();
      if  ( *libGAP_TLS(libGAP_In) == '\n' )
        return libGAP_GetCleanedChar(wasEscaped);
      else {
        libGAP_UNGET_CHAR(*libGAP_TLS(libGAP_In));
        *wasEscaped = 1;
        return '\r';
      }
    }
    else {
      *wasEscaped = 1;
      if ( *libGAP_TLS(libGAP_In) == 'n')  return '\n';
      else if ( *libGAP_TLS(libGAP_In) == 't')  return '\t';
      else if ( *libGAP_TLS(libGAP_In) == 'r')  return '\r';
      else if ( *libGAP_TLS(libGAP_In) == 'b')  return '\b';
      else if ( *libGAP_TLS(libGAP_In) == '>')  return '\01';
      else if ( *libGAP_TLS(libGAP_In) == '<')  return '\02';
      else if ( *libGAP_TLS(libGAP_In) == 'c')  return '\03';
    }
  }
  return *libGAP_TLS(libGAP_In);
}


void libGAP_GetNumber ( libGAP_UInt StartingStatus )
{
  libGAP_Int                 i=0;
  libGAP_Char                c;
  libGAP_UInt seenExp = 0;
  libGAP_UInt wasEscaped = 0;
  libGAP_UInt seenADigit = (StartingStatus != 0 && StartingStatus != 2);
  libGAP_UInt seenExpDigit = (StartingStatus ==5);

  c = *libGAP_TLS(libGAP_In);
  if (StartingStatus  <  2) {
    /* read initial sequence of digits into 'Value'             */
    for (i = 0; !wasEscaped && libGAP_IsDigit(c) && i < libGAP_SAFE_VALUE_SIZE-1; i++) {
      libGAP_TLS(libGAP_Value)[i] = c;
      seenADigit = 1;
      c = libGAP_GetCleanedChar(&wasEscaped);
    }

    /* So why did we run off the end of that loop */
    /* maybe we saw an identifier character and realised that this is an identifier we are reading */
    if (wasEscaped || libGAP_IsIdent(c)) {
      /* Now we know we have an identifier read the rest of it */
      libGAP_TLS(libGAP_Value)[i++] = c;
      c = libGAP_GetCleanedChar(&wasEscaped);
      for (; wasEscaped || libGAP_IsIdent(c) || libGAP_IsDigit(c); i++) {
        if (i < libGAP_SAFE_VALUE_SIZE -1)
          libGAP_TLS(libGAP_Value)[i] = c;
        c = libGAP_GetCleanedChar(&wasEscaped);
      }
      if (i < libGAP_SAFE_VALUE_SIZE -1)
        libGAP_TLS(libGAP_Value)[i] = '\0';
      else
        libGAP_TLS(libGAP_Value)[libGAP_SAFE_VALUE_SIZE-1] = '\0';
      libGAP_TLS(libGAP_Symbol) = libGAP_S_IDENT;
      return;
    }

    /* Or maybe we just ran out of space */
    if (libGAP_IsDigit(c)) {
      assert(i >= libGAP_SAFE_VALUE_SIZE-1);
      libGAP_TLS(libGAP_Symbol) = libGAP_S_PARTIALINT;
      libGAP_TLS(libGAP_Value)[libGAP_SAFE_VALUE_SIZE-1] = '\0';
      return;
    }

    /* Or maybe we saw a . which could indicate one of two things:
       a float literal or .. */
    if (c == '.'){
      /* If the symbol before this integer was S_DOT then 
         we must be in a nested record element expression, so don't 
         look for a float.

      This is a bit fragile  */
      if (libGAP_TLS(libGAP_Symbol) == libGAP_S_DOT || libGAP_TLS(libGAP_Symbol) == libGAP_S_BDOT) {
        libGAP_TLS(libGAP_Value)[i]  = '\0';
        libGAP_TLS(libGAP_Symbol) = libGAP_S_INT;
        return;
      }
      
      /* peek ahead to decide which */
      libGAP_GET_CHAR();
      if (*libGAP_TLS(libGAP_In) == '.') {
        /* It was .. */
        libGAP_UNGET_CHAR(*libGAP_TLS(libGAP_In));
        libGAP_TLS(libGAP_Symbol) = libGAP_S_INT;
        libGAP_TLS(libGAP_Value)[i] = '\0';
        return;
      }


      /* Not .. Put back the character we peeked at */
      libGAP_UNGET_CHAR(*libGAP_TLS(libGAP_In));
      /* Now the . must be part of our number
         store it and move on */
      libGAP_TLS(libGAP_Value)[i++] = c;
      c = libGAP_GetCleanedChar(&wasEscaped);
    }

    else {
      /* Anything else we see tells us that the token is done */
      libGAP_TLS(libGAP_Value)[i]  = '\0';
      libGAP_TLS(libGAP_Symbol) = libGAP_S_INT;
      return;
    }
  }



  /* The only case in which we fall through to here is when
     we have read zero or more digits, followed by . which is not part of a .. token
     or we were called with StartingStatus >= 2 so we read at least that much in
     a previous token */


  if (StartingStatus< 4) {
    /* When we get here we have read (either in this token or a previous S_PARTIALFLOAT*)
       possibly some digits, a . and possibly some more digits, but not an e,E,d,D,q or Q */

    /* read digits */
    for (; !wasEscaped && libGAP_IsDigit(c) && i < libGAP_SAFE_VALUE_SIZE-1; i++) {
      libGAP_TLS(libGAP_Value)[i] = c;
      seenADigit = 1;
      c = libGAP_GetCleanedChar(&wasEscaped);
    }
    /* If we found an identifier type character in this context could be an error
      or the start of one of the allowed trailing marker sequences */
    if (wasEscaped || (libGAP_IsIdent(c)  && c != 'e' && c != 'E' && c != 'D' && c != 'q' &&
                       c != 'd' && c != 'Q')) {

      if (!seenADigit)
        libGAP_SyntaxError("Badly formed number, need a digit before or after the decimal point");
      /* We allow one letter on the end of the numbers -- could be an i,
       C99 style */
      if (!wasEscaped) {
        if (libGAP_IsAlpha(c)) {
          libGAP_TLS(libGAP_Value)[i++] = c;
          c = libGAP_GetCleanedChar(&wasEscaped);
        }
        /* independently of that, we allow an _ signalling immediate conversion */
        if (c == '_') {
          libGAP_TLS(libGAP_Value)[i++] = c;
          c = libGAP_GetCleanedChar(&wasEscaped);
          /* After which there may be one character signifying the conversion style */
          if (libGAP_IsAlpha(c)) {
            libGAP_TLS(libGAP_Value)[i++] = c;
            c = libGAP_GetCleanedChar(&wasEscaped);
          }
        }
        /* Now if the next character is alphanumerical, or an identifier type symbol then we
           really do have an error, otherwise we return a result */
        if (!libGAP_IsIdent(c) && !libGAP_IsDigit(c)) {
          libGAP_TLS(libGAP_Value)[i] = '\0';
          libGAP_TLS(libGAP_Symbol) = libGAP_S_FLOAT;
          return;
        }
      }
      libGAP_SyntaxError("Badly formed number");
    }
    /* If the next thing is the start of the exponential notation,
       read it now -- we have left enough space at the end of the buffer even if we
       left the previous loop because of overflow */
    if (libGAP_IsAlpha(c)) {
        if (!seenADigit)
          libGAP_SyntaxError("Badly formed number, need a digit before or after the decimal point");
        seenExp = 1;
        libGAP_TLS(libGAP_Value)[i++] = c;
        c = libGAP_GetCleanedChar(&wasEscaped);
        if (!wasEscaped && (c == '+' || c == '-'))
          {
            libGAP_TLS(libGAP_Value)[i++] = c;
            c = libGAP_GetCleanedChar(&wasEscaped);
          }
      }

    /* Now deal with full buffer case */
    if (i >= libGAP_SAFE_VALUE_SIZE -1) {
      libGAP_TLS(libGAP_Symbol) = seenExp ? libGAP_S_PARTIALFLOAT3 : libGAP_S_PARTIALFLOAT2;
      libGAP_TLS(libGAP_Value)[i] = '\0';
      return;
    }

    /* Either we saw an exponent indicator, or we hit end of token
       deal with the end of token case */
    if (!seenExp) {
      if (!seenADigit)
        libGAP_SyntaxError("Badly formed number, need a digit before or after the decimal point");
      /* Might be a conversion marker */
      if (!wasEscaped) {
        if (libGAP_IsAlpha(c) && c != 'e' && c != 'E' && c != 'd' && c != 'D' && c != 'q' && c != 'Q') {
          libGAP_TLS(libGAP_Value)[i++] = c;
          c = libGAP_GetCleanedChar(&wasEscaped);
        }
        /* independently of that, we allow an _ signalling immediate conversion */
        if (c == '_') {
          libGAP_TLS(libGAP_Value)[i++] = c;
          c = libGAP_GetCleanedChar(&wasEscaped);
          /* After which there may be one character signifying the conversion style */
          if (libGAP_IsAlpha(c))
            libGAP_TLS(libGAP_Value)[i++] = c;
          c = libGAP_GetCleanedChar(&wasEscaped);
        }
        /* Now if the next character is alphanumerical, or an identifier type symbol then we
           really do have an error, otherwise we return a result */
        if (!libGAP_IsIdent(c) && !libGAP_IsDigit(c)) {
          libGAP_TLS(libGAP_Value)[i] = '\0';
          libGAP_TLS(libGAP_Symbol) = libGAP_S_FLOAT;
          return;
        }
      }
      libGAP_SyntaxError("Badly Formed Number");
    }

  }

  /* Here we are into the unsigned exponent of a number
     in scientific notation, so we just read digits */
  for (; !wasEscaped && libGAP_IsDigit(c) && i < libGAP_SAFE_VALUE_SIZE-1; i++) {
    libGAP_TLS(libGAP_Value)[i] = c;
    seenExpDigit = 1;
    c = libGAP_GetCleanedChar(&wasEscaped);
  }

  /* Look out for a single alphabetic character on the end
     which could be a conversion marker */
  if (seenExpDigit) {
    if (libGAP_IsAlpha(c)) {
      libGAP_TLS(libGAP_Value)[i] = c;
      c = libGAP_GetCleanedChar(&wasEscaped);
      libGAP_TLS(libGAP_Value)[i+1] = '\0';
      libGAP_TLS(libGAP_Symbol) = libGAP_S_FLOAT;
      return;
    }
    if (c == '_') {
      libGAP_TLS(libGAP_Value)[i++] = c;
      c = libGAP_GetCleanedChar(&wasEscaped);
      /* After which there may be one character signifying the conversion style */
      if (libGAP_IsAlpha(c)) {
        libGAP_TLS(libGAP_Value)[i++] = c;
        c = libGAP_GetCleanedChar(&wasEscaped);
      }
      libGAP_TLS(libGAP_Value)[i] = '\0';
      libGAP_TLS(libGAP_Symbol) = libGAP_S_FLOAT;
      return;
    }
  }

  /* If we ran off the end */
  if (i >= libGAP_SAFE_VALUE_SIZE -1) {
    libGAP_TLS(libGAP_Symbol) = seenExpDigit ? libGAP_S_PARTIALFLOAT4 : libGAP_S_PARTIALFLOAT3;
    libGAP_TLS(libGAP_Value)[i] = '\0';
    return;
  }

  /* Otherwise this is the end of the token */
  if (!seenExpDigit)
    libGAP_SyntaxError("Badly Formed Number, need at least one digit in the exponent");
  libGAP_TLS(libGAP_Symbol) = libGAP_S_FLOAT;
  libGAP_TLS(libGAP_Value)[i] = '\0';
  return;
}


/****************************************************************************
 **
 *F  GetStr()  . . . . . . . . . . . . . . . . . . . . . . get a string, local
 **
 **  'GetStr' reads  a  string from the  current input file into  the variable
 **  'TLS(Value)' and sets 'Symbol'   to  'S_STRING'.  The opening double quote '"'
 **  of the string is the current character pointed to by 'In'.
 **
 **  A string is a sequence of characters delimited  by double quotes '"'.  It
 **  must not include  '"' or <newline>  characters, but the  escape sequences
 **  '\"' or '\n' can  be used instead.  The  escape sequence  '\<newline>' is
 **  ignored, making it possible to split long strings over multiple lines.
 **
 **  An error is raised if the string includes a <newline> character or if the
 **  file ends before the closing '"'.
 **
 **  When TLS(Value) is  completely filled we have to check  if the reading of
 **  the string is  complete or not to decide  between Symbol=S_STRING or
 **  S_PARTIALSTRING.
 */
void libGAP_GetStr ( void )
{
  libGAP_Int                 i = 0, fetch;
  libGAP_Char                a, b, c;

  /* Avoid substitution of '?' in beginning of GetLine chunks */
  libGAP_TLS(libGAP_HELPSubsOn) = 0;

  /* read all characters into 'Value'                                    */
  for ( i = 0; i < libGAP_SAFE_VALUE_SIZE-1 && *libGAP_TLS(libGAP_In) != '"'
           && *libGAP_TLS(libGAP_In) != '\n' && *libGAP_TLS(libGAP_In) != '\377'; i++ ) {

    fetch = 1;
    /* handle escape sequences                                         */
    if ( *libGAP_TLS(libGAP_In) == '\\' ) {
      libGAP_GET_CHAR();
      /* if next is another '\\' followed by '\n' it must be ignored */
      while ( *libGAP_TLS(libGAP_In) == '\\' ) {
        libGAP_GET_CHAR();
        if ( *libGAP_TLS(libGAP_In) == '\n' )
          libGAP_GET_CHAR();
        else {
          libGAP_UNGET_CHAR( '\\' );
          break;
        }
      }
      if      ( *libGAP_TLS(libGAP_In) == '\n' )  i--;
      else if ( *libGAP_TLS(libGAP_In) == '\r' )  {
        libGAP_GET_CHAR();
        if  ( *libGAP_TLS(libGAP_In) == '\n' )  i--;
        else  {libGAP_TLS(libGAP_Value)[i] = '\r'; fetch = 0;}
      }
      else if ( *libGAP_TLS(libGAP_In) == 'n'  )  libGAP_TLS(libGAP_Value)[i] = '\n';
      else if ( *libGAP_TLS(libGAP_In) == 't'  )  libGAP_TLS(libGAP_Value)[i] = '\t';
      else if ( *libGAP_TLS(libGAP_In) == 'r'  )  libGAP_TLS(libGAP_Value)[i] = '\r';
      else if ( *libGAP_TLS(libGAP_In) == 'b'  )  libGAP_TLS(libGAP_Value)[i] = '\b';
      else if ( *libGAP_TLS(libGAP_In) == '>'  )  libGAP_TLS(libGAP_Value)[i] = '\01';
      else if ( *libGAP_TLS(libGAP_In) == '<'  )  libGAP_TLS(libGAP_Value)[i] = '\02';
      else if ( *libGAP_TLS(libGAP_In) == 'c'  )  libGAP_TLS(libGAP_Value)[i] = '\03';
      else if ( libGAP_IsDigit( *libGAP_TLS(libGAP_In) ) ) {
        a = *libGAP_TLS(libGAP_In); libGAP_GET_CHAR(); b = *libGAP_TLS(libGAP_In); libGAP_GET_CHAR(); c = *libGAP_TLS(libGAP_In);
        if (!( libGAP_IsDigit(b) && libGAP_IsDigit(c) )){
          libGAP_SyntaxError("expecting three octal digits after \\ in string");
        }
        libGAP_TLS(libGAP_Value)[i] = (a-'0') * 64 + (b-'0') * 8 + c-'0';
      }
      else  libGAP_TLS(libGAP_Value)[i] = *libGAP_TLS(libGAP_In);
    }

    /* put normal chars into 'Value' but only if there is room         */
    else {
      libGAP_TLS(libGAP_Value)[i] = *libGAP_TLS(libGAP_In);
    }

    /* read the next character                                         */
    if (fetch) libGAP_GET_CHAR();

  }

  /* XXX although we have ValueLen we need trailing \000 here,
     in gap.c, function FuncMAKE_INIT this is still used as C-string
     and long integers and strings are not yet supported!    */
  libGAP_TLS(libGAP_Value)[i] = '\0';

  /* check for error conditions                                          */
  if ( *libGAP_TLS(libGAP_In) == '\n'  )
    libGAP_SyntaxError("string must not include <newline>");
  if ( *libGAP_TLS(libGAP_In) == '\377' )
    libGAP_SyntaxError("string must end with \" before end of file");

  /* set length of string, set 'Symbol' and skip trailing '"'            */
  libGAP_TLS(libGAP_ValueLen) = i;
  if ( i < libGAP_SAFE_VALUE_SIZE-1 )  {
    libGAP_TLS(libGAP_Symbol) = libGAP_S_STRING;
    if ( *libGAP_TLS(libGAP_In) == '"' )  libGAP_GET_CHAR();
  }
  else
    libGAP_TLS(libGAP_Symbol) = libGAP_S_PARTIALSTRING;

  /* switching on substitution of '?' */
  libGAP_TLS(libGAP_HELPSubsOn) = 1;
}

/****************************************************************************
 **
 *F  GetTripStr()  . . . . . . . . . . . . .get a triple quoted string, local
 **
 **  'GetTripStr' reads a triple-quoted string from the  current input file
 **  into  the variable 'Value' and sets 'Symbol'   to  'S_STRING'.
 **  The last member of the opening triple quote '"'
 **  of the string is the current character pointed to by 'In'.
 **
 **  A triple quoted string is any sequence of characters which is terminated
 **  by """. No escaping is performed.
 **
 **  An error is raised if the file ends before the closing """.
 **
 **  When Value is  completely filled we have to check  if the reading of
 **  the string is  complete or not to decide  between Symbol=S_STRING or
 **  S_PARTIALTRIPLESTRING.
 */
void libGAP_GetTripStr ( void )
{
  libGAP_Int                 i = 0;

  /* Avoid substitution of '?' in beginning of GetLine chunks */
  libGAP_TLS(libGAP_HELPSubsOn) = 0;
  
  /* print only a partial prompt while reading a triple string           */
  if ( !libGAP_SyQuiet )
    libGAP_TLS(libGAP_Prompt) = "> ";
  else
    libGAP_TLS(libGAP_Prompt) = "";
  
  /* read all characters into 'Value'                                    */
  for ( i = 0; i < libGAP_SAFE_VALUE_SIZE-1 && *libGAP_TLS(libGAP_In) != '\377'; i++ ) {
    // Only thing to check for is a triple quote.
    
    if ( *libGAP_TLS(libGAP_In) == '"') {
        libGAP_GET_CHAR();
        if (*libGAP_TLS(libGAP_In) == '"') {
            libGAP_GET_CHAR();
            if(*libGAP_TLS(libGAP_In) == '"' ) {
                break;
            }
            libGAP_TLS(libGAP_Value)[i] = '"';
            i++;
        }
        libGAP_TLS(libGAP_Value)[i] = '"';
        i++;
    }
    libGAP_TLS(libGAP_Value)[i] = *libGAP_TLS(libGAP_In);


    /* read the next character                                         */
    libGAP_GET_CHAR();
  }

  /* XXX although we have ValueLen we need trailing \000 here,
     in gap.c, function FuncMAKE_INIT this is still used as C-string
     and long integers and strings are not yet supported!    */
  libGAP_TLS(libGAP_Value)[i] = '\0';

  /* check for error conditions                                          */
  if ( *libGAP_TLS(libGAP_In) == '\377' )
    libGAP_SyntaxError("string must end with \" before end of file");

  /* set length of string, set 'Symbol' and skip trailing '"'            */
  libGAP_TLS(libGAP_ValueLen) = i;
  if ( i < libGAP_SAFE_VALUE_SIZE-1 )  {
    libGAP_TLS(libGAP_Symbol) = libGAP_S_STRING;
    if ( *libGAP_TLS(libGAP_In) == '"' )  libGAP_GET_CHAR();
  }
  else
    libGAP_TLS(libGAP_Symbol) = libGAP_S_PARTIALTRIPSTRING;

  /* switching on substitution of '?' */
  libGAP_TLS(libGAP_HELPSubsOn) = 1;
}

/****************************************************************************
 **
 *F  GetMaybeTripStr()  . . . . . . . . . . . . . . . . . get a string, local
 **
 **  'GetMaybeTripStr' decides if we are reading a single quoted string,
 **  or a triple quoted string.
 */

void libGAP_GetMaybeTripStr ( void )
{
    /* Avoid substitution of '?' in beginning of GetLine chunks */
    libGAP_TLS(libGAP_HELPSubsOn) = 0;
    
    /* This is just a normal string! */
    if ( *libGAP_TLS(libGAP_In) != '"' ) {
        libGAP_GetStr();
        return;
    }
    
    libGAP_GET_CHAR();
    /* This was just an empty string! */
    if ( *libGAP_TLS(libGAP_In) != '"' ) {
        libGAP_TLS(libGAP_Value)[0] = '\0';
        libGAP_TLS(libGAP_ValueLen) = 0;
        libGAP_TLS(libGAP_Symbol) = libGAP_S_STRING;
        libGAP_TLS(libGAP_HELPSubsOn) = 1;
        return;
    }
    
    libGAP_GET_CHAR();
    /* Now we know we are reading a triple string */
    libGAP_GetTripStr();
}


/****************************************************************************
 **
 *F  GetChar() . . . . . . . . . . . . . . . . . get a single character, local
 **
 **  'GetChar' reads the next  character from the current input file  into the
 **  variable 'TLS(Value)' and sets 'Symbol' to 'S_CHAR'.  The opening single quote
 **  '\'' of the character is the current character pointed to by 'In'.
 **
 **  A  character is  a  single character delimited by single quotes '\''.  It
 **  must not  be '\'' or <newline>, but  the escape  sequences '\\\'' or '\n'
 **  can be used instead.
 */
void libGAP_GetChar ( void )
{
  libGAP_Char c;

  /* skip '\''                                                           */
  libGAP_GET_CHAR();

  /* handle escape equences                                              */
  if ( *libGAP_TLS(libGAP_In) == '\\' ) {
    libGAP_GET_CHAR();
    if ( *libGAP_TLS(libGAP_In) == 'n'  )       libGAP_TLS(libGAP_Value)[0] = '\n';
    else if ( *libGAP_TLS(libGAP_In) == 't'  )  libGAP_TLS(libGAP_Value)[0] = '\t';
    else if ( *libGAP_TLS(libGAP_In) == 'r'  )  libGAP_TLS(libGAP_Value)[0] = '\r';
    else if ( *libGAP_TLS(libGAP_In) == 'b'  )  libGAP_TLS(libGAP_Value)[0] = '\b';
    else if ( *libGAP_TLS(libGAP_In) == '>'  )  libGAP_TLS(libGAP_Value)[0] = '\01';
    else if ( *libGAP_TLS(libGAP_In) == '<'  )  libGAP_TLS(libGAP_Value)[0] = '\02';
    else if ( *libGAP_TLS(libGAP_In) == 'c'  )  libGAP_TLS(libGAP_Value)[0] = '\03';
    else if ( *libGAP_TLS(libGAP_In) >= '0' && *libGAP_TLS(libGAP_In) <= '7' ) {
      /* escaped three digit octal numbers are allowed in input */
      c = 64 * (*libGAP_TLS(libGAP_In) - '0');
      libGAP_GET_CHAR();
      if ( *libGAP_TLS(libGAP_In) < '0' || *libGAP_TLS(libGAP_In) > '7' )
        libGAP_SyntaxError("expecting octal digit in character constant");
      c = c + 8 * (*libGAP_TLS(libGAP_In) - '0');
      libGAP_GET_CHAR();
      if ( *libGAP_TLS(libGAP_In) < '0' || *libGAP_TLS(libGAP_In) > '7' )
        libGAP_SyntaxError("expecting 3 octal digits in character constant");
      c = c + (*libGAP_TLS(libGAP_In) - '0');
      libGAP_TLS(libGAP_Value)[0] = c;
    }
    else                     libGAP_TLS(libGAP_Value)[0] = *libGAP_TLS(libGAP_In);
  }
  else if ( *libGAP_TLS(libGAP_In) == '\n' ) {
    libGAP_SyntaxError("newline not allowed in character literal");
  }
  /* put normal chars into 'TLS(Value)'                                       */
  else {
    libGAP_TLS(libGAP_Value)[0] = *libGAP_TLS(libGAP_In);
  }

  /* read the next character                                             */
  libGAP_GET_CHAR();

  
  /* check for terminating single quote                                  */
  if ( *libGAP_TLS(libGAP_In) != '\'' )
    libGAP_SyntaxError("missing single quote in character constant");

  /* skip the closing quote                                              */
  libGAP_TLS(libGAP_Symbol) = libGAP_S_CHAR;
  if ( *libGAP_TLS(libGAP_In) == '\'' )  libGAP_GET_CHAR();

}


/****************************************************************************
 **
 *F  GetSymbol() . . . . . . . . . . . . . . . . .  get the next symbol, local
 **
 **  'GetSymbol' reads  the  next symbol from   the  input,  storing it in the
 **  variable 'Symbol'.  If 'Symbol' is  'S_IDENT', 'S_INT' or 'S_STRING'  the
 **  value of the symbol is stored in the variable 'TLS(Value)'.  'GetSymbol' first
 **  skips all <space>, <tab> and <newline> characters and comments.
 **
 **  After reading  a  symbol the current  character   is the first  character
 **  beyond that symbol.
 */
void libGAP_GetSymbol ( void )
{
  /* special case if reading of a long token is not finished */
  if (libGAP_TLS(libGAP_Symbol) == libGAP_S_PARTIALSTRING) {
    libGAP_GetStr();
    return;
  }
  
  if (libGAP_TLS(libGAP_Symbol) == libGAP_S_PARTIALTRIPSTRING) {
      libGAP_GetTripStr();
      return;
  }
  
  if (libGAP_TLS(libGAP_Symbol) == libGAP_S_PARTIALINT) {
    if (libGAP_TLS(libGAP_Value)[0] == '\0')
      libGAP_GetNumber(0);
    else
      libGAP_GetNumber(1);
    return;
  }
  if (libGAP_TLS(libGAP_Symbol) == libGAP_S_PARTIALFLOAT1) {
    libGAP_GetNumber(2);
    return;
  }

  if (libGAP_TLS(libGAP_Symbol) == libGAP_S_PARTIALFLOAT2) {
    libGAP_GetNumber(3);
    return;
  }
  if (libGAP_TLS(libGAP_Symbol) == libGAP_S_PARTIALFLOAT3) {
    libGAP_GetNumber(4);
    return;
  }

  if (libGAP_TLS(libGAP_Symbol) == libGAP_S_PARTIALFLOAT4) {
    libGAP_GetNumber(5);
    return;
  }


  /* if no character is available then get one                           */
  if ( *libGAP_TLS(libGAP_In) == '\0' )
    { libGAP_TLS(libGAP_In)--;
      libGAP_GET_CHAR();
    }

  /* skip over <spaces>, <tabs>, <newlines> and comments                 */
  while (*libGAP_TLS(libGAP_In)==' '||*libGAP_TLS(libGAP_In)=='\t'||*libGAP_TLS(libGAP_In)=='\n'||*libGAP_TLS(libGAP_In)=='\r'||*libGAP_TLS(libGAP_In)=='\f'||*libGAP_TLS(libGAP_In)=='#') {
    if ( *libGAP_TLS(libGAP_In) == '#' ) {
      while ( *libGAP_TLS(libGAP_In) != '\n' && *libGAP_TLS(libGAP_In) != '\r' && *libGAP_TLS(libGAP_In) != '\377' )
        libGAP_GET_CHAR();
    }
    libGAP_GET_CHAR();
  }

  /* switch according to the character                                   */
  switch ( *libGAP_TLS(libGAP_In) ) {

  case '.':   libGAP_TLS(libGAP_Symbol) = libGAP_S_DOT;                         libGAP_GET_CHAR();
    /*            if ( *TLS(In) == '\\' ) { GET_CHAR();
            if ( *TLS(In) == '\n' ) { GET_CHAR(); } }   */
    if ( *libGAP_TLS(libGAP_In) == '.' ) { 
            libGAP_TLS(libGAP_Symbol) = libGAP_S_DOTDOT; libGAP_GET_CHAR();
            if ( *libGAP_TLS(libGAP_In) == '.') {
                    libGAP_TLS(libGAP_Symbol) = libGAP_S_DOTDOTDOT; libGAP_GET_CHAR();
            }
    }
    break;

  case '!':   libGAP_TLS(libGAP_Symbol) = libGAP_S_ILLEGAL;                     libGAP_GET_CHAR();
    if ( *libGAP_TLS(libGAP_In) == '\\' ) { libGAP_GET_CHAR();
      if ( *libGAP_TLS(libGAP_In) == '\n' ) { libGAP_GET_CHAR(); } }
    if ( *libGAP_TLS(libGAP_In) == '.' ) { libGAP_TLS(libGAP_Symbol) = libGAP_S_BDOT;    libGAP_GET_CHAR();  break; }
    if ( *libGAP_TLS(libGAP_In) == '[' ) { libGAP_TLS(libGAP_Symbol) = libGAP_S_BLBRACK; libGAP_GET_CHAR();  break; }
    if ( *libGAP_TLS(libGAP_In) == '{' ) { libGAP_TLS(libGAP_Symbol) = libGAP_S_BLBRACE; libGAP_GET_CHAR();  break; }
    break;
  case '[':   libGAP_TLS(libGAP_Symbol) = libGAP_S_LBRACK;                      libGAP_GET_CHAR();  break;
  case ']':   libGAP_TLS(libGAP_Symbol) = libGAP_S_RBRACK;                      libGAP_GET_CHAR();  break;
  case '{':   libGAP_TLS(libGAP_Symbol) = libGAP_S_LBRACE;                      libGAP_GET_CHAR();  break;
  case '}':   libGAP_TLS(libGAP_Symbol) = libGAP_S_RBRACE;                      libGAP_GET_CHAR();  break;
  case '(':   libGAP_TLS(libGAP_Symbol) = libGAP_S_LPAREN;                      libGAP_GET_CHAR();  break;
  case ')':   libGAP_TLS(libGAP_Symbol) = libGAP_S_RPAREN;                      libGAP_GET_CHAR();  break;
  case ',':   libGAP_TLS(libGAP_Symbol) = libGAP_S_COMMA;                       libGAP_GET_CHAR();  break;

  case ':':   libGAP_TLS(libGAP_Symbol) = libGAP_S_COLON;                       libGAP_GET_CHAR();
    if ( *libGAP_TLS(libGAP_In) == '\\' ) {
      libGAP_GET_CHAR();
      if ( *libGAP_TLS(libGAP_In) == '\n' )
        { libGAP_GET_CHAR(); }
    }
    if ( *libGAP_TLS(libGAP_In) == '=' ) { libGAP_TLS(libGAP_Symbol) = libGAP_S_ASSIGN;  libGAP_GET_CHAR(); break; }
    if ( libGAP_TLS(libGAP_In)[0] == ':' && libGAP_TLS(libGAP_In)[1] == '=') {
      libGAP_TLS(libGAP_Symbol) = libGAP_S_INCORPORATE; libGAP_GET_CHAR(); libGAP_GET_CHAR(); break;
    }
    break;

  case ';':   libGAP_TLS(libGAP_Symbol) = libGAP_S_SEMICOLON;                   libGAP_GET_CHAR();  break;

  case '=':   libGAP_TLS(libGAP_Symbol) = libGAP_S_EQ;                          libGAP_GET_CHAR();  break;
  case '<':   libGAP_TLS(libGAP_Symbol) = libGAP_S_LT;                          libGAP_GET_CHAR();
    if ( *libGAP_TLS(libGAP_In) == '\\' ) { libGAP_GET_CHAR();
      if ( *libGAP_TLS(libGAP_In) == '\n' ) { libGAP_GET_CHAR(); } }
    if ( *libGAP_TLS(libGAP_In) == '=' ) { libGAP_TLS(libGAP_Symbol) = libGAP_S_LE;      libGAP_GET_CHAR();  break; }
    if ( *libGAP_TLS(libGAP_In) == '>' ) { libGAP_TLS(libGAP_Symbol) = libGAP_S_NE;      libGAP_GET_CHAR();  break; }
    break;
  case '>':   libGAP_TLS(libGAP_Symbol) = libGAP_S_GT;                          libGAP_GET_CHAR();
    if ( *libGAP_TLS(libGAP_In) == '\\' ) { libGAP_GET_CHAR();
      if ( *libGAP_TLS(libGAP_In) == '\n' ) { libGAP_GET_CHAR(); } }
    if ( *libGAP_TLS(libGAP_In) == '=' ) { libGAP_TLS(libGAP_Symbol) = libGAP_S_GE;      libGAP_GET_CHAR();  break; }
    break;

  case '+':   libGAP_TLS(libGAP_Symbol) = libGAP_S_PLUS;                        libGAP_GET_CHAR();  break;
  case '-':   libGAP_TLS(libGAP_Symbol) = libGAP_S_MINUS;                       libGAP_GET_CHAR();
    if ( *libGAP_TLS(libGAP_In) == '\\' ) { libGAP_GET_CHAR();
      if ( *libGAP_TLS(libGAP_In) == '\n' ) { libGAP_GET_CHAR(); } }
    if ( *libGAP_TLS(libGAP_In) == '>' ) { libGAP_TLS(libGAP_Symbol)=libGAP_S_MAPTO;     libGAP_GET_CHAR();  break; }
    break;
  case '*':   libGAP_TLS(libGAP_Symbol) = libGAP_S_MULT;                        libGAP_GET_CHAR();  break;
  case '/':   libGAP_TLS(libGAP_Symbol) = libGAP_S_DIV;                         libGAP_GET_CHAR();  break;
  case '^':   libGAP_TLS(libGAP_Symbol) = libGAP_S_POW;                         libGAP_GET_CHAR();  break;
  case '`':   libGAP_TLS(libGAP_Symbol) = libGAP_S_BACKQUOTE;                   libGAP_GET_CHAR();  break;

  case '"':                        libGAP_GET_CHAR(); libGAP_GetMaybeTripStr();  break;
  case '\'':                                          libGAP_GetChar();   break;
  case '\\':                                          libGAP_GetIdent();  break;
  case '_':                                           libGAP_GetIdent();  break;
  case '@':                                           libGAP_GetIdent();  break;
  case '~':   libGAP_TLS(libGAP_Value)[0] = '~';  libGAP_TLS(libGAP_Value)[1] = '\0';
    libGAP_TLS(libGAP_Symbol) = libGAP_S_IDENT;                       libGAP_GET_CHAR();  break;

  case '0': case '1': case '2': case '3': case '4':
  case '5': case '6': case '7': case '8': case '9':
    libGAP_GetNumber(0);    break;

  case '\377': libGAP_TLS(libGAP_Symbol) = libGAP_S_EOF;                        *libGAP_TLS(libGAP_In) = '\0';  break;

  default :   if ( libGAP_IsAlpha(*libGAP_TLS(libGAP_In)) )                   { libGAP_GetIdent();  break; }
    libGAP_TLS(libGAP_Symbol) = libGAP_S_ILLEGAL;                     libGAP_GET_CHAR();  break;
  }
}


/****************************************************************************
 **

 *F * * * * * * * * * * * * *  output functions  * * * * * * * * * * * * * * *
 */


/****************************************************************************
 **

 *V  WriteAllFunc  . . . . . . . . . . . . . . . . . . . . . . . .  'WriteAll'
 */
libGAP_Obj libGAP_WriteAllFunc;


/****************************************************************************
 **
 *F  PutLine2( <output>, <line>, <len> )  . . . . . . . . . print a line, local
 **
 **  Introduced  <len> argument. Actually in all cases where this is called one
 **  knows the length of <line>, so it is not necessary to compute it again
 **  with the inefficient C- strlen.  (FL)
 */


void libGAP_PutLine2(
        libGAP_TypOutputFile *         output,
        const libGAP_Char *            line,
        libGAP_UInt                    len )
{
  libGAP_Obj                     str;
  libGAP_UInt                    lstr;
  if ( output->isstream ) {
    /* special handling of string streams, where we can copy directly */
    if (output->isstringstream) {
      str = libGAP_ADDR_OBJ(output->stream)[1];
      lstr = libGAP_GET_LEN_STRING(str);
      libGAP_GROW_STRING(str, lstr+len);
      memcpy((void *) (libGAP_CHARS_STRING(str) + lstr), line, len);
      libGAP_SET_LEN_STRING(str, lstr + len);
      *(libGAP_CHARS_STRING(str) + lstr + len) = '\0';
      libGAP_CHANGED_BAG(str);
      return;
    }

    /* Space for the null is allowed for in GAP strings */
    libGAP_C_NEW_STRING( str, len, line );

    /* now delegate to library level */
    libGAP_CALL_2ARGS( libGAP_WriteAllFunc, output->stream, str );
  }
  else {
    libGAP_SyFputs( line, output->libGAP_file );
  }
}


/****************************************************************************
 **
 *F  PutLineTo ( stream, len ) . . . . . . . . . . . . . . print a line, local
 **
 **  'PutLineTo'  prints the first len characters of the current output
 **  line   'stream->line' to <stream>
 **  It  is  called from 'PutChrTo'.
 **
 **  'PutLineTo' also compares the output line with the  next line from the test
 **  input file 'TestInput' if 'TestInput' is not 0.  If  this input line does
 **  not starts with 'gap>' and the rest  of the line  matches the output line
 **  then the output line is not printed and the input line is discarded.
 **
 **  'PutLineTo'  also echoes the  output  line  to the  logfile 'OutputLog' if
 **  'OutputLog' is not 0 and the output file is '*stdout*' or '*errout*'.
 **
 **  Finally 'PutLineTo' checks whether the user has hit '<ctr>-C' to  interrupt
 **  the printing.
 */
void libGAP_PutLineTo ( libGAP_KOutputStream stream, libGAP_UInt len )
{
  libGAP_Char *          p;
  libGAP_UInt lt,ls;     /* These are supposed to hold string lengths */

  /* if in test mode and the next input line matches print nothing       */
  if ( libGAP_TLS(libGAP_TestInput) != 0 && libGAP_TLS(libGAP_TestOutput) == stream ) {
    if ( libGAP_TLS(libGAP_TestLine)[0] == '\0' ) {
      if ( ! libGAP_GetLine2( libGAP_TLS(libGAP_TestInput), libGAP_TLS(libGAP_TestLine), sizeof(libGAP_TLS(libGAP_TestLine)) ) ) {
        libGAP_TLS(libGAP_TestLine)[0] = '\0';
      }
      libGAP_TLS(libGAP_TestInput)->number++;
    }

    /* Note that TLS(TestLine) is ended by a \n, but stream->line need not! */

    lt = strlen(libGAP_TLS(libGAP_TestLine));   /* this counts including the newline! */
    p = libGAP_TLS(libGAP_TestLine) + (lt-2);
    /* this now points to the last char before \n in the line! */
    while ( libGAP_TLS(libGAP_TestLine) <= p && ( *p == ' ' || *p == '\t' ) ) {
      p[1] = '\0';  p[0] = '\n';  p--; lt--;
    }
    /* lt is still the correct string length including \n */
    ls = strlen(stream->line);
    p = stream->line + (ls-1);
    /* this now points to the last char of the string, could be a \n */
    if (*p == '\n') {
      p--;   /* now we point before that newline character */
      while ( stream->line <= p && ( *p == ' ' || *p == '\t' ) ) {
        p[1] = '\0';  p[0] = '\n';  p--; ls--;
      }
    }
    /* ls is still the correct string length including a possible \n */
    if ( ! strncmp( libGAP_TLS(libGAP_TestLine), stream->line, ls ) ) {
      if (ls < lt)
        memmove(libGAP_TLS(libGAP_TestLine),libGAP_TLS(libGAP_TestLine) + ls,lt-ls+1);
      else
        libGAP_TLS(libGAP_TestLine)[0] = '\0';
    }
    else {
      char obuf[80];
      /* snprintf(obuf, sizeof(obuf), "+ 5%i bad example:\n+ ", (int)TLS(TestInput)->number); */
      snprintf(obuf, sizeof(obuf), "Line %i : \n+ ", (int)libGAP_TLS(libGAP_TestInput)->number);
      libGAP_PutLine2( stream, obuf, strlen(obuf) );
      libGAP_PutLine2( stream, libGAP_TLS(libGAP_Output)->line, strlen(libGAP_TLS(libGAP_Output)->line) );
    }
  }

  /* otherwise output this line                                          */
  else {
    libGAP_PutLine2( stream, stream->line, len );
  }

  /* if neccessary echo it to the logfile                                */
  if ( libGAP_TLS(libGAP_OutputLog) != 0 && ! stream->isstream ) {
    if ( stream->libGAP_file == 1 || stream->libGAP_file == 3 ) {
      libGAP_PutLine2( libGAP_TLS(libGAP_OutputLog), stream->line, len );
    }
  }
}


/****************************************************************************
 **
 *F  PutChrTo( <stream>, <ch> )  . . . . . . . . . print character <ch>, local
 **
 **  'PutChrTo' prints the single character <ch> to the stream <stream>
 **
 **  'PutChrTo' buffers the  output characters until  either <ch> is  <newline>,
 **  <ch> is '\03' (<flush>) or the buffer fills up.
 **
 **  In the later case 'PutChrTo' has to decide where to  split the output line.
 **  It takes the point at which $linelength - pos + 8 * indent$ is minimal.
 */
libGAP_Int libGAP_NoSplitLine = 0;

/* helper function to add a hint about a possible line break;
   a triple (pos, value, indent), such that the minimal (value-pos) wins */
void libGAP_addLineBreakHint( libGAP_KOutputStream stream, libGAP_Int pos, libGAP_Int val, libGAP_Int indentdiff )
{
  libGAP_Int nr, i;
  /* find next free slot */
  for (nr = 0; nr < libGAP_MAXHINTS && stream->hints[3*nr] != -1; nr++);
  if (nr == libGAP_MAXHINTS) {
    /* forget the first stored hint */
    for (i = 0; i < 3*libGAP_MAXHINTS - 3; i++)
       stream->hints[i] =  stream->hints[i+3];
    nr--;
  }
  /* if pos is same as before only relevant if new entry has higher
     priority */
  if ( nr > 0 && stream->hints[3*(nr-1)] == pos )
    nr--;

  if ( stream->indent < pos &&
       (stream->hints[3*nr] == -1 || val < stream->hints[3*(nr)+1]) ) {
    stream->hints[3*nr] = pos;
    stream->hints[3*nr+1] = val;
    stream->hints[3*nr+2] = stream->indent;
    stream->hints[3*nr+3] = -1;
  }
  stream->indent += indentdiff;
}
/* helper function to find line break position,
   returns position nr in stream[hints] or -1 if none found */
libGAP_Int libGAP_nrLineBreak( libGAP_KOutputStream stream )
{
  libGAP_Int nr=-1, min, i;
  for (i = 0, min = INT_MAX; stream->hints[3*i] != -1; i++)
  {
    if (stream->hints[3*i] > 0 &&
        stream->hints[3*i+1] - stream->hints[3*i] <= min)
    {
      nr = i;
      min = stream->hints[3*i+1] - stream->hints[3*i];
    }
  }
  if (min < INT_MAX)
    return nr;
  else
    return -1;
}



void libGAP_PutChrTo (
         libGAP_KOutputStream stream,
         libGAP_Char                ch )
{
    // printf("PutChrTo %i %i %c %hhx\n", stream->file, stream->isstream, ch, ch);
    if (ch <= 3)  // GAP control characters
    return;
  // Magic constants are defined in SyFopen
    if (stream->libGAP_file == 0) {        // negative number indicates an error
      assert(libGAP_False);
    } else if (stream->libGAP_file == 0) { // 0 identifies the standard input file "*stdin*"
      assert(libGAP_False);
    } else if (stream->libGAP_file == 1) { // 1 identifies the standard outpt file "*stdout*"
      libgap_append_stdout(ch);
    } else if (stream->libGAP_file == 2) { // 2 identifies the brk loop input file "*errin*"
      assert(libGAP_False);
    } else if (stream->libGAP_file == 3) { // 3 identifies the error messages file "*errout*"
      libgap_append_stderr(ch);
    } else {                        // anything else is a real file descriptor
      stream->line[stream->pos++] = ch;
      stream->line[stream->pos++] = '\0';
      libGAP_PutLineTo(stream, stream->pos);
      stream->pos = 0;
    }
}

/****************************************************************************
 **
 *F  FuncToggleEcho( )
 **
*/

libGAP_Obj libGAP_FuncToggleEcho( libGAP_Obj self)
{
  libGAP_TLS(libGAP_Input)->echo = 1 - libGAP_TLS(libGAP_Input)->echo;
  return (libGAP_Obj)0;
}

/****************************************************************************
 **
 *F  FuncCPROMPT( )
 **
 **  returns the current `Prompt' as GAP string.
 */
libGAP_Obj libGAP_FuncCPROMPT( libGAP_Obj self)
{
  libGAP_Obj p;
  libGAP_C_NEW_STRING_DYN( p, libGAP_TLS(libGAP_Prompt) );
  return p;
}

/****************************************************************************
 **
 *F  FuncPRINT_CPROMPT( <prompt> )
 **
 **  prints current `Prompt' if argument <prompt> is not in StringRep, otherwise
 **  uses the content of <prompt> as `Prompt' (at most 80 characters).
 **  (important is the flush character without resetting the cursor column)
 */
libGAP_Char libGAP_promptBuf[81];

libGAP_Obj libGAP_FuncPRINT_CPROMPT( libGAP_Obj self, libGAP_Obj prompt )
{
  if (libGAP_IS_STRING_REP(prompt)) {
    /* by assigning to Prompt we also tell readline (if used) what the
       current prompt is  */
    libGAP_strlcpy(libGAP_promptBuf, libGAP_CSTR_STRING(prompt), sizeof(libGAP_promptBuf));
    libGAP_TLS(libGAP_Prompt) = libGAP_promptBuf;
  }
  libGAP_Pr("%s%c", (libGAP_Int)libGAP_TLS(libGAP_Prompt), (libGAP_Int)'\03' );
  return (libGAP_Obj) 0;
}

/****************************************************************************
 **
 *F  Pr( <format>, <arg1>, <arg2> )  . . . . . . . . .  print formatted output
 *F  PrTo( <stream>, <format>, <arg1>, <arg2> )  . . .  print formatted output
 **
 **  'Pr' is the output function. The first argument is a 'printf' like format
 **  string containing   up   to 2  '%'  format   fields,   specifing  how the
 **  corresponding arguments are to be  printed.  The two arguments are passed
 **  as  'Int'   integers.   This  is possible  since every  C object  ('int',
 **  'char', pointers) except 'float' or 'double', which are not used  in GAP,
 **  can be converted to a 'Int' without loss of information.
 **
 **  The function 'Pr' currently support the following '%' format  fields:
 **  '%c'    the corresponding argument represents a character,  usually it is
 **          its ASCII or EBCDIC code, and this character is printed.
 **  '%s'    the corresponding argument is the address of  a  null  terminated
 **          character string which is printed.
 **  '%S'    the corresponding argument is the address of  a  null  terminated
 **          character string which is printed with escapes.
 **  '%C'    the corresponding argument is the address of  a  null  terminated
 **          character string which is printed with C escapes.
 **  '%d'    the corresponding argument is a signed integer, which is printed.
 **          Between the '%' and the 'd' an integer might be used  to  specify
 **          the width of a field in which the integer is right justified.  If
 **          the first character is '0' 'Pr' pads with '0' instead of <space>.
 **  '%i'    is a synonym of %d, in line with recent C library developements
 **  '%I'    print an identifier
 **  '%>'    increment the indentation level.
 **  '%<'    decrement the indentation level.
 **  '%%'    can be used to print a single '%' character. No argument is used.
 **
 **  You must always  cast the arguments to  '(Int)'  to avoid  problems  with
 **  those compilers with a default integer size of 16 instead of 32 bit.  You
 **  must pass 0L if you don't make use of an argument to please lint.
 */

void libGAP_FormatOutput(void (*put_a_char)(libGAP_Char c), const libGAP_Char *format, libGAP_Int arg1, libGAP_Int arg2 ) {
  const libGAP_Char *        p;
  libGAP_Char *              q;
  libGAP_Int                 prec,  n;
  libGAP_Char                fill;

  /* loop over the characters of the <format> string                     */
  for ( p = format; *p != '\0'; p++ ) {

    /* not a '%' character, simply print it                            */
    if ( *p != '%' ) {
      put_a_char( *p );
      continue;
    }

    /* if the character is '%' do something special                    */

    /* first look for a precision field                            */
    p++;
    prec = 0;
    fill = (*p == '0' ? '0' : ' ');
    while ( libGAP_IsDigit(*p) ) {
      prec = 10 * prec + *p - '0';
      p++;
    }

    /* handle the case of a missing argument                     */
    if (arg1 == 0 && (*p == 's' || *p == 'S' || *p == 'C' || *p == 'I')) {
      put_a_char('<');
      put_a_char('n');
      put_a_char('u');
      put_a_char('l');
      put_a_char('l');
      put_a_char('>');

      /* on to the next argument                                 */
      arg1 = arg2;
    }

    /* '%d' print an integer                                       */
    else if ( *p == 'd'|| *p == 'i' ) {
      int is_neg = (arg1 < 0);
      if ( is_neg ) {
        arg1 = -arg1;
        prec--; /* we loose one digit of output precision for the minus sign */
      }

      /* compute how many characters this number requires    */
      for ( n = 1; n <= arg1/10; n*=10 ) {
        prec--;
      }
      while ( --prec > 0 )  put_a_char(fill);

      if ( is_neg ) {
        put_a_char('-');
      }

      for ( ; n > 0; n /= 10 )
        put_a_char( (libGAP_Char)(((arg1/n)%10) + '0') );

      /* on to the next argument                                 */
      arg1 = arg2;
    }

    /* '%s' print a string                                         */
    else if ( *p == 's' ) {

      /* compute how many characters this identifier requires    */
      for ( q = (libGAP_Char*)arg1; *q != '\0' && prec > 0; q++ ) {
        prec--;
      }

      /* if wanted push an appropriate number of <space>-s       */
      while ( prec-- > 0 )  put_a_char(' ');

      /* print the string                                        */
      /* must be careful that line breaks don't go inside
         escaped sequences \n or \123 or similar */
      for ( q = (libGAP_Char*)arg1; *q != '\0'; q++ ) {
        if (*q == '\\' && libGAP_TLS(libGAP_NoSplitLine) == 0) {
          if (*(q+1) < '8' && *(q+1) >= '0')
            libGAP_TLS(libGAP_NoSplitLine) = 3;
          else
            libGAP_TLS(libGAP_NoSplitLine) = 1;
        }
        else if (libGAP_TLS(libGAP_NoSplitLine) > 0)
          libGAP_TLS(libGAP_NoSplitLine)--;
        put_a_char( *q );
      }

      /* on to the next argument                                 */
      arg1 = arg2;
    }

    /* '%S' print a string with the necessary escapes              */
    else if ( *p == 'S' ) {

      /* compute how many characters this identifier requires    */
      for ( q = (libGAP_Char*)arg1; *q != '\0' && prec > 0; q++ ) {
        if      ( *q == '\n'  ) { prec -= 2; }
        else if ( *q == '\t'  ) { prec -= 2; }
        else if ( *q == '\r'  ) { prec -= 2; }
        else if ( *q == '\b'  ) { prec -= 2; }
        else if ( *q == '\01' ) { prec -= 2; }
        else if ( *q == '\02' ) { prec -= 2; }
        else if ( *q == '\03' ) { prec -= 2; }
        else if ( *q == '"'   ) { prec -= 2; }
        else if ( *q == '\\'  ) { prec -= 2; }
        else                    { prec -= 1; }
      }

      /* if wanted push an appropriate number of <space>-s       */
      while ( prec-- > 0 )  put_a_char(' ');

      /* print the string                                        */
      for ( q = (libGAP_Char*)arg1; *q != '\0'; q++ ) {
        if      ( *q == '\n'  ) { put_a_char('\\'); put_a_char('n');  }
        else if ( *q == '\t'  ) { put_a_char('\\'); put_a_char('t');  }
        else if ( *q == '\r'  ) { put_a_char('\\'); put_a_char('r');  }
        else if ( *q == '\b'  ) { put_a_char('\\'); put_a_char('b');  }
        else if ( *q == '\01' ) { put_a_char('\\'); put_a_char('>');  }
        else if ( *q == '\02' ) { put_a_char('\\'); put_a_char('<');  }
        else if ( *q == '\03' ) { put_a_char('\\'); put_a_char('c');  }
        else if ( *q == '"'   ) { put_a_char('\\'); put_a_char('"');  }
        else if ( *q == '\\'  ) { put_a_char('\\'); put_a_char('\\'); }
        else                    { put_a_char( *q );               }
      }

      /* on to the next argument                                 */
      arg1 = arg2;
    }

    /* '%C' print a string with the necessary C escapes            */
    else if ( *p == 'C' ) {

      /* compute how many characters this identifier requires    */
      for ( q = (libGAP_Char*)arg1; *q != '\0' && prec > 0; q++ ) {
        if      ( *q == '\n'  ) { prec -= 2; }
        else if ( *q == '\t'  ) { prec -= 2; }
        else if ( *q == '\r'  ) { prec -= 2; }
        else if ( *q == '\b'  ) { prec -= 2; }
        else if ( *q == '\01' ) { prec -= 3; }
        else if ( *q == '\02' ) { prec -= 3; }
        else if ( *q == '\03' ) { prec -= 3; }
        else if ( *q == '"'   ) { prec -= 2; }
        else if ( *q == '\\'  ) { prec -= 2; }
        else                    { prec -= 1; }
      }

      /* if wanted push an appropriate number of <space>-s       */
      while ( prec-- > 0 )  put_a_char(' ');

      /* print the string                                        */
      for ( q = (libGAP_Char*)arg1; *q != '\0'; q++ ) {
        if      ( *q == '\n'  ) { put_a_char('\\'); put_a_char('n');  }
        else if ( *q == '\t'  ) { put_a_char('\\'); put_a_char('t');  }
        else if ( *q == '\r'  ) { put_a_char('\\'); put_a_char('r');  }
        else if ( *q == '\b'  ) { put_a_char('\\'); put_a_char('b');  }
        else if ( *q == '\01' ) { put_a_char('\\'); put_a_char('0');
          put_a_char('1');                }
        else if ( *q == '\02' ) { put_a_char('\\'); put_a_char('0');
          put_a_char('2');                }
        else if ( *q == '\03' ) { put_a_char('\\'); put_a_char('0');
          put_a_char('3');                }
        else if ( *q == '"'   ) { put_a_char('\\'); put_a_char('"');  }
        else if ( *q == '\\'  ) { put_a_char('\\'); put_a_char('\\'); }
        else                    { put_a_char( *q );               }
      }

      /* on to the next argument                                 */
      arg1 = arg2;
    }

    /* '%I' print an identifier                                    */
    else if ( *p == 'I' ) {
      int found_keyword = 0;
      int i;

      /* check if q matches a keyword    */
      q = (libGAP_Char*)arg1;
      for ( i = 0; i < sizeof(libGAP_AllKeywords)/sizeof(libGAP_AllKeywords[0]); i++ ) {
        if ( strcmp(q, libGAP_AllKeywords[i].name) == 0 ) {
          found_keyword = 1;
          break;
        }
      }

      /* compute how many characters this identifier requires    */
      if (found_keyword) {
        prec--;
      }
      for ( q = (libGAP_Char*)arg1; *q != '\0'; q++ ) {
        if ( !libGAP_IsIdent(*q) && !libGAP_IsDigit(*q) ) {
          prec--;
        }
        prec--;
      }

      /* if wanted push an appropriate number of <space>-s       */
      while ( prec-- > 0 ) { put_a_char(' '); }

      /* print the identifier                                    */
      if ( found_keyword ) {
        put_a_char( '\\' );
      }
      for ( q = (libGAP_Char*)arg1; *q != '\0'; q++ ) {
        if ( !libGAP_IsIdent(*q) && !libGAP_IsDigit(*q) ) {
          put_a_char( '\\' );
        }
        put_a_char( *q );
      }

      /* on to the next argument                                 */
      arg1 = arg2;
    }

    /* '%c' print a character                                      */
    else if ( *p == 'c' ) {
      put_a_char( (libGAP_Char)arg1 );
      arg1 = arg2;
    }

    /* '%%' print a '%' character                                  */
    else if ( *p == '%' ) {
      put_a_char( '%' );
    }

    /* '%>' increment the indentation level                        */
    else if ( *p == '>' ) {
      put_a_char( '\01' );
      while ( --prec > 0 )
        put_a_char( '\01' );
    }

    /* '%<' decrement the indentation level                        */
    else if ( *p == '<' ) {
      put_a_char( '\02' );
      while ( --prec > 0 )
        put_a_char( '\02' );
    }

    /* else raise an error                                         */
    else {
      for ( p = "%format error"; *p != '\0'; p++ )
        put_a_char( *p );
    }

  }

}


static libGAP_KOutputStream libGAP_TheStream;

static void libGAP_putToTheStream( libGAP_Char c) {
  libGAP_PutChrTo(libGAP_TLS(libGAP_TheStream), c);
}

void libGAP_PrTo (
           libGAP_KOutputStream     stream,
           const libGAP_Char *      format,
           libGAP_Int                 arg1,
           libGAP_Int                 arg2 )
{
  libGAP_KOutputStream savedStream = libGAP_TLS(libGAP_TheStream);
  libGAP_TLS(libGAP_TheStream) = stream;
  libGAP_FormatOutput( libGAP_putToTheStream, format, arg1, arg2);
  libGAP_TLS(libGAP_TheStream) = savedStream;
}

void libGAP_Pr (
         const libGAP_Char *      format,
         libGAP_Int                 arg1,
         libGAP_Int                 arg2 )
{
  libGAP_PrTo(libGAP_TLS(libGAP_Output), format, arg1, arg2);
}

static libGAP_Char *libGAP_TheBuffer;
static libGAP_UInt libGAP_TheCount;
static libGAP_UInt libGAP_TheLimit;

static void libGAP_putToTheBuffer( libGAP_Char c)
{
  if (libGAP_TLS(libGAP_TheCount) < libGAP_TLS(libGAP_TheLimit))
    libGAP_TLS(libGAP_TheBuffer)[libGAP_TLS(libGAP_TheCount)++] = c;
}

void libGAP_SPrTo(libGAP_Char *buffer, libGAP_UInt maxlen, const libGAP_Char *format, libGAP_Int arg1, libGAP_Int arg2)
{
  libGAP_Char *savedBuffer = libGAP_TLS(libGAP_TheBuffer);
  libGAP_UInt savedCount = libGAP_TLS(libGAP_TheCount);
  libGAP_UInt savedLimit = libGAP_TLS(libGAP_TheLimit);
  libGAP_TLS(libGAP_TheBuffer) = buffer;
  libGAP_TLS(libGAP_TheCount) = 0;
  libGAP_TLS(libGAP_TheLimit) = maxlen;
  libGAP_FormatOutput(libGAP_putToTheBuffer, format, arg1, arg2);
  libGAP_putToTheBuffer('\0');
  libGAP_TLS(libGAP_TheBuffer) = savedBuffer;
  libGAP_TLS(libGAP_TheCount) = savedCount;
  libGAP_TLS(libGAP_TheLimit) = savedLimit;
}


libGAP_Obj libGAP_FuncINPUT_FILENAME( libGAP_Obj self) {
  libGAP_Obj s;
  if (libGAP_TLS(libGAP_Input)) {
    libGAP_C_NEW_STRING_DYN( s, libGAP_TLS(libGAP_Input)->name );
  } else {
    libGAP_C_NEW_STRING_CONST( s, "*defin*" );
  }
  return s;
}

libGAP_Obj libGAP_FuncINPUT_LINENUMBER( libGAP_Obj self) {
  return libGAP_INTOBJ_INT(libGAP_TLS(libGAP_Input) ? libGAP_TLS(libGAP_Input)->number : 0);
}

libGAP_Obj libGAP_FuncALL_KEYWORDS(libGAP_Obj self) {
  libGAP_Obj l;

  libGAP_Obj s;
  libGAP_UInt i;
  l = libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
  libGAP_SET_LEN_PLIST(l,0);
  for (i = 0; i < sizeof(libGAP_AllKeywords)/sizeof(libGAP_AllKeywords[0]); i++) {
    libGAP_C_NEW_STRING_DYN(s,libGAP_AllKeywords[i].name);
    libGAP_ASS_LIST(l, i+1, s);
  }
  libGAP_MakeImmutable(l);
  return l;
}

libGAP_Obj libGAP_FuncSET_PRINT_FORMATTING_STDOUT(libGAP_Obj self, libGAP_Obj val) {
  if (val == libGAP_False)
    (libGAP_TLS(libGAP_OutputFiles)+1)->format = 0;
  else
    (libGAP_TLS(libGAP_OutputFiles)+1)->format = 1;
  return val;
}



/****************************************************************************
 **
 *F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
 */

/****************************************************************************
 **
 *V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
 */
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

  { "ToggleEcho", 0, "",
    libGAP_FuncToggleEcho, "src/scanner.c:ToggleEcho" },

  { "CPROMPT", 0, "",
    libGAP_FuncCPROMPT, "src/scanner.c:CPROMPT" },

  { "PRINT_CPROMPT", 1, "prompt",
    libGAP_FuncPRINT_CPROMPT, "src/scanner.c:PRINT_CPROMPT" },

  { "INPUT_FILENAME", 0 , "",
    libGAP_FuncINPUT_FILENAME, "src/scanner.c:INPUT_FILENAME" },

  { "INPUT_LINENUMBER", 0 , "",
    libGAP_FuncINPUT_LINENUMBER, "src/scanner.c:INPUT_LINENUMBER" },

  { "ALL_KEYWORDS", 0 , "",
    libGAP_FuncALL_KEYWORDS, "src/scanner.c:ALL_KEYWORDS"},

  { "SET_PRINT_FORMATTING_STDOUT", 1 , "format",
    libGAP_FuncSET_PRINT_FORMATTING_STDOUT,
    "src/scanner.c:SET_PRINT_FORMATTING_STDOUT"},

  { 0 }

};

/****************************************************************************
 **
 *F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
 */
static libGAP_Int libGAP_InitLibrary (
                        libGAP_StructInitInfo *    libGAP_module )
{
  /* init filters and functions                                          */
  libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

  /* return success                                                      */
  return 0;
}

/****************************************************************************
 **
 *F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
 */
static libGAP_Char libGAP_Cookie[sizeof(libGAP_TLS(libGAP_InputFiles))/sizeof(libGAP_TLS(libGAP_InputFiles)[0])][9];
static libGAP_Char libGAP_MoreCookie[sizeof(libGAP_TLS(libGAP_InputFiles))/sizeof(libGAP_TLS(libGAP_InputFiles)[0])][9];
static libGAP_Char libGAP_StillMoreCookie[sizeof(libGAP_TLS(libGAP_InputFiles))/sizeof(libGAP_TLS(libGAP_InputFiles)[0])][9];

static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int                 i;

    libGAP_TLS(libGAP_Input) = libGAP_TLS(libGAP_InputFiles);
    libGAP_TLS(libGAP_Input)--;
    (void)libGAP_OpenInput(  "*stdin*"  );
    libGAP_TLS(libGAP_Input)->echo = 1; /* echo stdin */
    libGAP_TLS(libGAP_Output) = 0L;
    (void)libGAP_OpenOutput( "*stdout*" );

    libGAP_TLS(libGAP_InputLog)  = 0;  libGAP_TLS(libGAP_OutputLog)  = 0;
    libGAP_TLS(libGAP_TestInput) = 0;  libGAP_TLS(libGAP_TestOutput) = 0;

    /* initialize cookies for streams                                      */
    /* also initialize the cookies for the GAP strings which hold the
       latest lines read from the streams  and the name of the current input file*/
    for ( i = 0;  i < sizeof(libGAP_TLS(libGAP_InputFiles))/sizeof(libGAP_TLS(libGAP_InputFiles)[0]);  i++ ) {
      libGAP_Cookie[i][0] = 's';  libGAP_Cookie[i][1] = 't';  libGAP_Cookie[i][2] = 'r';
      libGAP_Cookie[i][3] = 'e';  libGAP_Cookie[i][4] = 'a';  libGAP_Cookie[i][5] = 'm';
      libGAP_Cookie[i][6] = ' ';  libGAP_Cookie[i][7] = '0'+i;
      libGAP_Cookie[i][8] = '\0';
      libGAP_InitGlobalBag(&(libGAP_TLS(libGAP_InputFiles)[i].stream), &(libGAP_Cookie[i][0]));

      libGAP_MoreCookie[i][0] = 's';  libGAP_MoreCookie[i][1] = 'l';  libGAP_MoreCookie[i][2] = 'i';
      libGAP_MoreCookie[i][3] = 'n';  libGAP_MoreCookie[i][4] = 'e';  libGAP_MoreCookie[i][5] = ' ';
      libGAP_MoreCookie[i][6] = ' ';  libGAP_MoreCookie[i][7] = '0'+i;
      libGAP_MoreCookie[i][8] = '\0';
      libGAP_InitGlobalBag(&(libGAP_TLS(libGAP_InputFiles)[i].sline), &(libGAP_MoreCookie[i][0]));

      libGAP_StillMoreCookie[i][0] = 'g';  libGAP_StillMoreCookie[i][1] = 'a';  libGAP_StillMoreCookie[i][2] = 'p';
      libGAP_StillMoreCookie[i][3] = 'n';  libGAP_StillMoreCookie[i][4] = 'a';  libGAP_StillMoreCookie[i][5] = 'm';
      libGAP_StillMoreCookie[i][6] = 'e';  libGAP_StillMoreCookie[i][7] = '0'+i;
      libGAP_StillMoreCookie[i][8] = '\0';
      libGAP_InitGlobalBag(&(libGAP_TLS(libGAP_InputFiles)[i].gapname), &(libGAP_StillMoreCookie[i][0]));
    }

    /* tell GASMAN about the global bags                                   */
    libGAP_InitGlobalBag(&(libGAP_LogFile.stream),        "src/scanner.c:LogFile"        );
    libGAP_InitGlobalBag(&(libGAP_LogStream.stream),      "src/scanner.c:LogStream"      );
    libGAP_InitGlobalBag(&(libGAP_InputLogStream.stream), "src/scanner.c:InputLogStream" );
    libGAP_InitGlobalBag(&(libGAP_OutputLogStream.stream),"src/scanner.c:OutputLogStream");


    /* import functions from the library                                   */
    libGAP_ImportFuncFromLibrary( "ReadLine", &libGAP_ReadLineFunc );
    libGAP_ImportFuncFromLibrary( "WriteAll", &libGAP_WriteAllFunc );
    libGAP_ImportFuncFromLibrary( "IsInputTextStringRep", &libGAP_IsStringStream );
    libGAP_InitCopyGVar( "PrintPromptHook", &libGAP_PrintPromptHook );
    libGAP_InitCopyGVar( "EndLineHook", &libGAP_EndLineHook );
    libGAP_InitFopyGVar( "PrintFormattingStatus", &libGAP_PrintFormattingStatus);

    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );
    /* return success                                                      */
    return 0;
}


/****************************************************************************
 **
 *F  InitInfoScanner() . . . . . . . . . . . . . . . . table of init functions
 */
static libGAP_StructInitInfo libGAP_module = {
  libGAP_MODULE_BUILTIN,                     /* type                           */
  "scanner",                          /* name                           */
  0,                                  /* revision entry of c file       */
  0,                                  /* revision entry of h file       */
  0,                                  /* version                        */
  0,                                  /* crc                            */
  libGAP_InitKernel,                         /* initKernel                     */
  libGAP_InitLibrary,                        /* initLibrary                    */
  0,                                  /* checkInit                      */
  0,                                  /* preSave                        */
  0,                                  /* postSave                       */
  0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoScanner ( void )
{
  return &libGAP_module;
}


/****************************************************************************
 **
 *E  scanner.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
 */
