/****************************************************************************
**
*W  intrprtr.c                  GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the immediate interpreter package.
**
**  The immediate interpreter package  is  the part  of the interpreter  that
**  interprets code immediately (while it is read).  Its functions are called
**  from the reader.  When it encounters  constructs that it cannot interpret
**  immediately, it switches into coding mode, and  delegates the work to the
**  coder.
*/
#include        <assert.h>              /* assert                          */
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */
#include        "read.h"                /* reader                          */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */
#include        "records.h"             /* generic records                 */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */
#include        "integer.h"             /* integers                        */

#include        "permutat.h"            /* permutations                    */
#include        "trans.h"               /* transformations                 */
#include        "pperm.h"               /* partial perms                   */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */
#include        "funcs.h"               /* functions                       */
#include        "read.h"

#include        "intrprtr.h"            /* interpreter                     */

#include	"tls.h"
#include	"thread.h"
#include	"aobjects.h"		/* atomic objects		   */

#include        "vars.h"                /* variables                       */

#include        "saveload.h"            /* saving and loading              */

/****************************************************************************
**

*V  IntrResult  . . . . . . . . . . . . . . . . . result value of interpreter
**
**  'IntrResult'  is the result value of  the interpreter, i.e., the value of
**  the  statement  that  was  last  interpreted (which   might  have been  a
**  return-value-statement).
*/
libGAP_Obj libGAP_IntrResult;


/****************************************************************************
**
*V  IntrReturning   . . . . . . . . . . .  interpreter is currently returning
**
**  If 'IntrReturning' is  non-zero, the interpreter is currently  returning.
**  The interpreter switches  to this mode when  it finds a return-statement.
**  If it interpretes a return-value-statement, it sets 'IntrReturning' to 1.
**  If it interpretes a return-void-statement,  it sets 'IntrReturning' to 2.
**  If it interpretes a quit-statement, it sets 'IntrReturning' to 8.
*/
libGAP_UInt libGAP_IntrReturning;


/****************************************************************************
**
*V  IntrIgnoring  . . . . . . . . . interpreter is currently ignoring actions
**
**  If 'IntrIgnoring'  is  non-zero,  the interpreter  is  currently ignoring
**  actions.  The interpreter switches to this mode for  the right operand of
**  'or' and 'and'  constructs where the  left operand already determines the
**  outcome.
**
**  This mode is also used in Info and Assert, when arguments are not printed.
*/
libGAP_UInt libGAP_IntrIgnoring;


/****************************************************************************
**
*V  IntrCoding  . . . . . . . . . . . interpreter is currently coding actions
**
**  If 'IntrCoding' is non-zero, the interpreter is currently coding actions.
**  The interpreter  switches  to this  mode for  constructs  that it  cannot
**  directly interpret, such as loops or function bodies.
*/
libGAP_UInt libGAP_IntrCoding;


/****************************************************************************
**

*F  StackObj  . . . . . . . . . . . . . . . . . . . . . . . . .  values stack
*F  CountObj  . . . . . . . . . . . . . . . . . number of values on the stack
*F  PushObj(<val>)  . . . . . . . . . . . . . . . . push value onto the stack
*F  PushVoidObj() . . . . . . . . . . . . . .  push void value onto the stack
*F  PopObj()  . . . . . . . . . . . . . . . . . . .  pop value from the stack
*F  PopVoidObj()  . . . . . . . . . . . . . . . . .  pop value from the stack
**
**  'StackObj' is the stack of values.
**
**  'CountObj' is the number of values currently on the values stack.
**
**  'PushObj' pushes the value <val>  onto the values stack.   It is an error
**  to push the void value.  The stack is automatically resized if necessary.
**
**  'PushVoidObj' pushes the void value onto the values stack.  This value is
**  the value of if-statements and loops and procedure calls.
**
**  'PopObj' returns the top element from  the values stack  and pops it.  It
**  is an error if the stack is empty or if the top element is void.
**
**  'PopVoidObj' returns the  top element from the values  stack and pops it.
**  It is an error if the stack is empty but not if the top element is void.
**
**  Since interpreters  can nest, there can   be more than one  values stack.
**  The bottom  two  elements of each values  stack  are the  'StackObj'  and
**  'CountObj' there were active when the current interpreter was started and
**  which will be made active again when the current interpreter will stop.
*/
static libGAP_Obj             libGAP_IntrState;

static libGAP_Obj             libGAP_StackObj;

static libGAP_Int             libGAP_CountObj;

void            libGAP_PushObj (
    libGAP_Obj                 val )
{
    /* there must be a stack, it must not be underfull or overfull         */
    assert( libGAP_TLS(libGAP_StackObj) != 0 );
    assert( 0 <= libGAP_TLS(libGAP_CountObj) && libGAP_TLS(libGAP_CountObj) == libGAP_LEN_PLIST(libGAP_TLS(libGAP_StackObj)) );
    assert( val != 0 );

    /* count up and put the value onto the stack                           */
    libGAP_TLS(libGAP_CountObj)++;
    libGAP_GROW_PLIST(    libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) );
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) );
    libGAP_SET_ELM_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj), val );
    libGAP_CHANGED_BAG(   libGAP_TLS(libGAP_StackObj) );
}

void            libGAP_PushVoidObj ( void )
{
    /* there must be a stack, it must not be underfull or overfull         */
    assert( libGAP_TLS(libGAP_StackObj) != 0 );
    assert( 0 <= libGAP_TLS(libGAP_CountObj) && libGAP_TLS(libGAP_CountObj) == libGAP_LEN_PLIST(libGAP_TLS(libGAP_StackObj)) );

    /* count up and put the void value onto the stack                      */
    libGAP_TLS(libGAP_CountObj)++;
    libGAP_GROW_PLIST(    libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) );
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) );
    libGAP_SET_ELM_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj), (libGAP_Obj)0 );
}

libGAP_Obj             libGAP_PopObj ( void )
{
    libGAP_Obj                 val;

    /* there must be a stack, it must not be underfull/empty or overfull   */
    assert( libGAP_TLS(libGAP_StackObj) != 0 );
    assert( 1 <= libGAP_TLS(libGAP_CountObj) && libGAP_TLS(libGAP_CountObj) == libGAP_LEN_LIST(libGAP_TLS(libGAP_StackObj)) );

    /* get the top element from the stack and count down                   */
    val = libGAP_ELM_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) );
    libGAP_SET_ELM_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj), 0 );
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj)-1  );
    libGAP_TLS(libGAP_CountObj)--;

    /* return the popped value (which must be non-void)                    */
    assert( val != 0 );
    return val;
}

libGAP_Obj             libGAP_PopVoidObj ( void )
{
    libGAP_Obj                 val;

    /* there must be a stack, it must not be underfull/empty or overfull   */
    assert( libGAP_TLS(libGAP_StackObj) != 0 );
    assert( 1 <= libGAP_TLS(libGAP_CountObj) && libGAP_TLS(libGAP_CountObj) == libGAP_LEN_LIST(libGAP_TLS(libGAP_StackObj)) );

    /* get the top element from the stack and count down                   */
    val = libGAP_ELM_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) );
    libGAP_SET_ELM_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj), 0 );
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj)-1  );
    libGAP_TLS(libGAP_CountObj)--;

    /* return the popped value (which may be void)                         */
    return val;
}


/****************************************************************************
**

*F  IntrBegin() . . . . . . . . . . . . . . . . . . . .  start an interpreter
*F  IntrEnd( <error> )  . . . . . . . . . . . . . . . . . stop an interpreter
**
**  'IntrBegin' starts a new interpreter.
**
**  'IntrEnd(<error>)' stops the current interpreter.
**
**  If <error>  is non-zero a  syntax error was found by  the reader, and the
**  interpreter only clears up the mess.
**
**  If 'IntrEnd' returns  0, then no  return-statement or quit-statement  was
**  interpreted.  If  'IntrEnd' returns 1,  then a return-value-statement was
**  interpreted and in this case the  return value is stored in 'TLS(IntrResult)'.
**  If  'IntrEnd' returns 2, then a  return-void-statement  was  interpreted.
**  If 'IntrEnd' returns 8, then a quit-statement was interpreted.
*/
void libGAP_IntrBegin ( libGAP_Obj frame )
{
    libGAP_Obj                 intrState;      /* old interpreter state           */

    /* remember old interpreter state                                      */
    /* This bag cannot exist at the top-most loop, which is the only
       place from which we might call SaveWorkspace */
    intrState = libGAP_NewBag( libGAP_T_PLIST, 4*sizeof(libGAP_Obj) );
    libGAP_ADDR_OBJ(intrState)[0] = (libGAP_Obj)3;
    libGAP_ADDR_OBJ(intrState)[1] = libGAP_TLS(libGAP_IntrState);
    libGAP_ADDR_OBJ(intrState)[2] = libGAP_TLS(libGAP_StackObj);
    libGAP_ADDR_OBJ(intrState)[3] = libGAP_INTOBJ_INT(libGAP_TLS(libGAP_CountObj));
    libGAP_TLS(libGAP_IntrState) = intrState;

    /* allocate a new values stack                                         */
    libGAP_TLS(libGAP_StackObj) = libGAP_NEW_PLIST( libGAP_T_PLIST, 64 );
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), 0 );
    libGAP_TLS(libGAP_CountObj) = 0;

    /* must be in immediate (non-ignoring, non-coding) mode                */
    assert( libGAP_TLS(libGAP_IntrIgnoring) == 0 );
    assert( libGAP_TLS(libGAP_IntrCoding)   == 0 );

    /* no return-statement was yet interpreted                             */
    libGAP_TLS(libGAP_IntrReturning) = 0;

    /* start an execution environment                                      */
    libGAP_ExecBegin(frame);
}

libGAP_ExecStatus libGAP_IntrEnd (
    libGAP_UInt                error )
{
    libGAP_UInt                intrReturning;  /* interpreted return-statement?   */

    /* if everything went fine                                             */
    if ( ! error ) {

        /* leave the execution environment                                 */
        libGAP_ExecEnd( 0UL );

        /* remember whether the interpreter interpreted a return-statement */
        intrReturning = libGAP_TLS(libGAP_IntrReturning);
        libGAP_TLS(libGAP_IntrReturning) = 0;

        /* must be back in immediate (non-ignoring, non-coding) mode       */
        assert( libGAP_TLS(libGAP_IntrIgnoring) == 0 );
        assert( libGAP_TLS(libGAP_IntrCoding)   == 0 );

        /* and the stack must contain the result value (which may be void) */
        assert( libGAP_TLS(libGAP_CountObj) == 1 );
        libGAP_TLS(libGAP_IntrResult) = libGAP_PopVoidObj();

        /* switch back to the old state                                    */
        libGAP_TLS(libGAP_CountObj)  = libGAP_INT_INTOBJ( libGAP_ADDR_OBJ(libGAP_TLS(libGAP_IntrState))[3] );
        libGAP_TLS(libGAP_StackObj)  = libGAP_ADDR_OBJ(libGAP_TLS(libGAP_IntrState))[2];
        libGAP_TLS(libGAP_IntrState) = libGAP_ADDR_OBJ(libGAP_TLS(libGAP_IntrState))[1];

    }

    /* otherwise clean up the mess                                         */
    else {

        /* leave the execution environment                                 */
        libGAP_ExecEnd( 1UL );

        /* clean up the coder too                                          */
        if ( libGAP_TLS(libGAP_IntrCoding) > 0 ) { libGAP_CodeEnd( 1UL ); }

        /* remember that we had an error                                   */
        intrReturning = libGAP_STATUS_ERROR;
        libGAP_TLS(libGAP_IntrReturning) = 0;

        /* must be back in immediate (non-ignoring, non-coding) mode       */
        libGAP_TLS(libGAP_IntrIgnoring) = 0;
        libGAP_TLS(libGAP_IntrCoding)   = 0;

        /* dummy result value (probably ignored)                           */
        libGAP_TLS(libGAP_IntrResult) = (libGAP_Obj)0;

        /* switch back to the old state                                    */
        libGAP_TLS(libGAP_CountObj)  = libGAP_INT_INTOBJ( libGAP_ADDR_OBJ(libGAP_TLS(libGAP_IntrState))[3] );
        libGAP_TLS(libGAP_StackObj)  = libGAP_ADDR_OBJ(libGAP_TLS(libGAP_IntrState))[2];
        libGAP_TLS(libGAP_IntrState) = libGAP_ADDR_OBJ(libGAP_TLS(libGAP_IntrState))[1];

    }

    /* indicate whether a return-statement was interpreted                 */
    return intrReturning;
}


/****************************************************************************
**
*F  IntrFuncCallBegin() . . . . . . . . . . .  interpret function call, begin
*F  IntrFuncCallEnd(<funccall>,<options>, <nr>)  interpret function call, end
**
**  'IntrFuncCallBegin' is an action  to  interpret a  function call.  It  is
**  called by  the reader  when  it  encounters  the parenthesis  '(',  i.e.,
**  *after* the function expression is read.
**
**  'IntrFuncCallEnd'  is an  action to  interpret  a  function call.   It is
**  called by    the reader when it encounters     the parenthesis ')', i.e.,
**  *after* the argument expressions are read.  <funccall>  is 1 if this is a
**  function call, and 0 if this is a procedure call.  <nr>  is the number of
**  arguments. <options> is 1 if options were present after the ':' in which
**  case the options have been read already.
*/
void            libGAP_IntrFuncCallBegin ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFuncCallBegin(); return; }

}

static libGAP_Obj libGAP_PushOptions;
static libGAP_Obj libGAP_PopOptions;

void            libGAP_IntrFuncCallEnd (
    libGAP_UInt                funccall,
    libGAP_UInt                libGAP_options,
    libGAP_UInt                nr )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 a1;             /* first argument                  */
    libGAP_Obj                 a2;             /* second argument                 */
    libGAP_Obj                 a3;             /* third argument                  */
    libGAP_Obj                 a4;             /* fourth argument                 */
    libGAP_Obj                 a5;             /* fifth  argument                 */
    libGAP_Obj                 a6;             /* sixth  argument                 */
    libGAP_Obj                 args;           /* argument list                   */
    libGAP_Obj                 argi;           /* <i>-th argument                 */
    libGAP_Obj                 val;            /* return value of function        */
    libGAP_Obj                 opts;           /* record of options               */
    libGAP_UInt                i;              /* loop variable                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) {
      libGAP_CodeFuncCallEnd( funccall, libGAP_options, nr );
      return; }


    if (libGAP_options) {
        opts = libGAP_PopObj();
        libGAP_CALL_1ARGS(libGAP_PushOptions, opts);
    }

    /* get the arguments from the stack                                    */
    a1 = a2 = a3 = a4 = a5 = a6 = args = 0;
    if ( nr <= 6 ) {
        if ( 6 <= nr ) { a6 = libGAP_PopObj(); }
        if ( 5 <= nr ) { a5 = libGAP_PopObj(); }
        if ( 4 <= nr ) { a4 = libGAP_PopObj(); }
        if ( 3 <= nr ) { a3 = libGAP_PopObj(); }
        if ( 2 <= nr ) { a2 = libGAP_PopObj(); }
        if ( 1 <= nr ) { a1 = libGAP_PopObj(); }
    } else {
        args = libGAP_NEW_PLIST( libGAP_T_PLIST, nr );
        libGAP_SET_LEN_PLIST( args, nr );
        for ( i = nr; 1 <= i; i-- ) {
            argi = libGAP_PopObj();
            libGAP_SET_ELM_PLIST( args, i, argi );
        }
    }

    /* get and check the function from the stack                           */
    func = libGAP_PopObj();
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
      args = libGAP_NEW_PLIST( libGAP_T_PLIST_DENSE, nr );
      libGAP_SET_LEN_PLIST( args, nr );
      switch(nr) {
      case 6: libGAP_SET_ELM_PLIST(args,6,a6);
      case 5: libGAP_SET_ELM_PLIST(args,5,a5);
      case 4: libGAP_SET_ELM_PLIST(args,4,a4);
      case 3: libGAP_SET_ELM_PLIST(args,3,a3);
      case 2: libGAP_SET_ELM_PLIST(args,2,a2);
      case 1: libGAP_SET_ELM_PLIST(args,1,a1);
      }
      val = libGAP_DoOperation2Args(libGAP_CallFuncListOper, func, args);
    } else {
      /* call the function                                                   */
      if      ( 0 == nr ) { val = libGAP_CALL_0ARGS( func ); }
      else if ( 1 == nr ) { val = libGAP_CALL_1ARGS( func, a1 ); }
      else if ( 2 == nr ) { val = libGAP_CALL_2ARGS( func, a1, a2 ); }
      else if ( 3 == nr ) { val = libGAP_CALL_3ARGS( func, a1, a2, a3 ); }
      else if ( 4 == nr ) { val = libGAP_CALL_4ARGS( func, a1, a2, a3, a4 ); }
      else if ( 5 == nr ) { val = libGAP_CALL_5ARGS( func, a1, a2, a3, a4, a5 ); }
      else if ( 6 == nr ) { val = libGAP_CALL_6ARGS( func, a1, a2, a3, a4, a5, a6 ); }
      else                { val = libGAP_CALL_XARGS( func, args ); }

      if (libGAP_TLS(libGAP_UserHasQuit) || libGAP_TLS(libGAP_UserHasQUIT)) {
        /* the procedure must have called READ() and the user quit
           from a break loop inside it */
        libGAP_ReadEvalError();
      }
    }

    /* check the return value                                              */
    if ( funccall && val == 0 ) {
        libGAP_ErrorQuit(
            "Function call: <func> must return a value",
            0L, 0L );
    }

    if (libGAP_options)
      libGAP_CALL_0ARGS(libGAP_PopOptions);

    /* push the value onto the stack                                       */
    if ( val == 0 )
        libGAP_PushVoidObj();
    else
        libGAP_PushObj( val );
}


/****************************************************************************
**
*F  IntrFuncExprBegin(<narg>,<nloc>,<nams>) .  interpret function expr, begin
*F  IntrFuncExprEnd(<nr>) . . . . . . . . . . .  interpret function expr, end
**
**  'IntrFuncExprBegin' is an action to interpret  a function expression.  It
**  is  called when   the reader  encounters  the  beginning  of  a  function
**  expression.  <narg> is the number of  arguments (-1 if the function takes
**  a variable number of arguments),  <nloc> is the  number of locals, <nams>
**  is a list of local variable names.
**
**  'IntrFuncExprEnd' is an action to interpret a function expression.  It is
**  called when the reader encounters the end  of a function expression. <nr>
**  is the number of statements in the body of the function.
*/
void            libGAP_IntrFuncExprBegin (
    libGAP_Int                 narg,
    libGAP_Int                 nloc,
    libGAP_Obj                 nams,
    libGAP_Int                 startLine)
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) {
        libGAP_TLS(libGAP_IntrCoding)++;
        libGAP_CodeFuncExprBegin( narg, nloc, nams, startLine );
        return;
    }

    /* switch to coding mode now                                           */
    libGAP_CodeBegin();
    libGAP_TLS(libGAP_IntrCoding) = 1;

    /* code a function expression                                          */
    libGAP_CodeFuncExprBegin( narg, nloc, nams, startLine );
}

void            libGAP_IntrFuncExprEnd (
    libGAP_UInt                nr,
    libGAP_UInt                mapsto )
{
    libGAP_Obj                 func;           /* the function, result            */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 1 ) {
        libGAP_TLS(libGAP_IntrCoding)--;
        libGAP_CodeFuncExprEnd( nr, mapsto );
        return;
    }

    /* must be coding                                                      */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );

    /* code a function expression                                          */
    libGAP_CodeFuncExprEnd( nr, mapsto );

    /* switch back to immediate mode, get the function                     */
    libGAP_CodeEnd(0);
    libGAP_TLS(libGAP_IntrCoding) = 0;
    func = libGAP_TLS(libGAP_CodeResult);

    /* push the function                                                   */
    libGAP_PushObj(func);
}


/****************************************************************************
**
*F  IntrIfBegin() . . . . . . . .  interpret if-statement, begin of statement
*F  IntrIfElif()  . . . . . . .  interpret if-statement, begin of elif-branch
*F  IntrIfElse()  . . . . . . .  interpret if-statement, begin of else-branch
*F  IntrIfBeginBody() . . . . . . . . . interpret if-statement, begin of body
*F  IntrIfEndBody(<nr>) . . . . . . . . . interpret if-statement, end of body
*F  IntrIfEnd(<nr>) . . . . . . . .  interpret if-statement, end of statement
**
**  'IntrIfBegin' is an  action to interpret  an if-statement.   It is called
**  when the reader encounters the   'if',  i.e., *before* the condition   is
**  read.
**
**  'IntrIfElif' is  an action to   interpret an if-statement.  It is  called
**  when the  reader encounters an  'elif', i.e.,  *before* the condition  is
**  read.
**
**  'IntrIfElse' is  an  action to interpret an   if-statement.  It is called
**  when the reader encounters an 'else'.
**
**  'IntrIfBeginBody'  is  an action to   interpret  an if-statement.  It  is
**  called when the reader encounters the beginning  of the statement body of
**  an 'if', 'elif', or 'else' branch, i.e., *after* the condition is read.
**
**  'IntrIfEndBody' is an action to interpret  an if-statement.  It is called
**  when the reader  encounters the end of  the  statements body of an  'if',
**  'elif', or 'else' branch.  <nr> is the number of statements in the body.
**
**  'IntrIfEnd' is an action to interpret an if-statement.  It is called when
**  the reader  encounters the end of the  statement.  <nr>  is the number of
**  'if', 'elif', or 'else' branches.
*/
void            libGAP_IntrIfBegin ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIfBegin(); return; }

}

void            libGAP_IntrIfElif ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIfElif(); return; }

}

void            libGAP_IntrIfElse ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIfElse(); return; }


    /* push 'true' (to execute body of else-branch)                        */
    libGAP_PushObj( libGAP_True );
}

void            libGAP_IntrIfBeginBody ( void )
{
    libGAP_Obj                 cond;           /* value of condition              */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIfBeginBody(); return; }


    /* get and check the condition                                         */
    cond = libGAP_PopObj();
    if ( cond != libGAP_True && cond != libGAP_False ) {
        libGAP_ErrorQuit(
            "<expr> must be 'true' or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(cond), 0L );
    }

    /* if the condition is 'false', ignore the body                        */
    if ( cond == libGAP_False ) {
        libGAP_TLS(libGAP_IntrIgnoring) = 1;
    }
}

void            libGAP_IntrIfEndBody (
    libGAP_UInt                nr )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 1 ) { libGAP_TLS(libGAP_IntrIgnoring)--; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIfEndBody( nr ); return; }


    /* if the condition was 'false', the body was ignored                  */
    if ( libGAP_TLS(libGAP_IntrIgnoring) == 1 ) {
        libGAP_TLS(libGAP_IntrIgnoring) = 0;
        return;
    }

    /* otherwise drop the values for the statements executed in the body   */
    for ( i = nr; 1 <= i; i-- ) {
        libGAP_PopVoidObj();
    }

    /* one branch of the if-statement was executed, ignore the others      */
    libGAP_TLS(libGAP_IntrIgnoring) = 1;
}

void            libGAP_IntrIfEnd (
    libGAP_UInt                nr )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 1 ) { libGAP_TLS(libGAP_IntrIgnoring)--; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIfEnd( nr ); return; }


    /* if one branch was executed (ignoring the others)                    */
    if ( libGAP_TLS(libGAP_IntrIgnoring) == 1 ) {
        libGAP_TLS(libGAP_IntrIgnoring) = 0;
        libGAP_PushVoidObj();
    }

    /* if no branch was executed                                           */
    else {
        libGAP_PushVoidObj();
    }
}


/****************************************************************************
**
*F  IntrForBegin()  . . . . . . . interpret for-statement, begin of statement
*F  IntrForIn() . . . . . . . . . . . . .  interpret for-statement, 'in'-read
*F  IntrForBeginBody()  . . . . . . .  interpret for-statement, begin of body
*F  IntrForEndBody(<nr>)  . . . . . . .  interpret for-statement, end of body
*F  IntrForEnd()  . . . . . . . . . interpret for-statement, end of statement
**
**  'IntrForBegin' is  an action to interpret  a for-statement.  It is called
**  when the   reader encounters the  'for', i.e.,  *before*  the variable is
**  read.
**
**  'IntrForIn' is an action to interpret a for-statement.  It is called when
**  the  reader encounters the 'in', i.e.,  *after* the variable is read, but
**  *before* the list expression is read.
**
**  'IntrForBeginBody'  is  an action to interpret   a for-statement.   It is
**  called when  the reader encounters  the beginning  of the statement body,
**  i.e., *after* the list expression is read.
**
**  'IntrForEndBody' is an action to interpret a for-statement.  It is called
**  when the  reader encounters the  end of the  statement body.  <nr> is the
**  number of statements in the body.
**
**  'IntrForEnd' is an  action  to interpret a  for-statement.   It is called
**  when the  reader encounters the end of  the statement,  i.e., immediately
**  after 'IntrForEndBody'.
**
**  Since loops cannot be interpreted immediately,  the interpreter calls the
**  coder  to create a  procedure (with no arguments) and  calls that.
*/
void libGAP_IntrForBegin ( void )
{
    libGAP_Obj                 nams;           /* (empty) list of names           */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_TLS(libGAP_IntrCoding)++; libGAP_CodeForBegin(); return; }


    /* switch to coding mode now                                           */
    libGAP_CodeBegin();
    libGAP_TLS(libGAP_IntrCoding) = 1;

    /* code a function expression (with no arguments and locals)           */
    nams = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( nams, 0 );

    /* If we are in the break loop, then a local variable context may well exist,
       and we have to create an empty local variable names list to match the
       function expression that we are creating.

       If we are not in a break loop, then this would be a waste of time and effort */

    if (libGAP_TLS(libGAP_CountNams) > 0) {
        libGAP_GROW_PLIST(libGAP_TLS(libGAP_StackNams), ++libGAP_TLS(libGAP_CountNams));
        libGAP_SET_ELM_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams), nams);
        libGAP_SET_LEN_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams));
    }

    libGAP_CodeFuncExprBegin( 0, 0, nams, 0 );

    /* code a for loop                                                     */
    libGAP_CodeForBegin();
}

void libGAP_IntrForIn ( void )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }


    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeForIn();
}

void libGAP_IntrForBeginBody ( void )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }


    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeForBeginBody();
}

void libGAP_IntrForEndBody (
    libGAP_UInt                nr )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if ( libGAP_TLS(libGAP_IntrCoding) != 0 ) {
        libGAP_CodeForEndBody( nr );
    }
}

void libGAP_IntrForEnd ( void )
{
    libGAP_Obj                 func;           /* the function, result            */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 1 ) { libGAP_TLS(libGAP_IntrCoding)--; libGAP_CodeForEnd(); return; }


    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );

    /* code a function expression (with one statement in the body)         */
    libGAP_CodeFuncExprEnd( 1UL, 0UL );

    /* switch back to immediate mode, get the function                     */
    libGAP_TLS(libGAP_IntrCoding) = 0;
    libGAP_CodeEnd( 0 );
    /* If we are in a break loop, then we will have created a "dummy" local
       variable names list to get the counts right. Remove it */
    if (libGAP_TLS(libGAP_CountNams) > 0)
      libGAP_TLS(libGAP_CountNams)--;

    func = libGAP_TLS(libGAP_CodeResult);

    /* call the function                                                   */
    libGAP_CALL_0ARGS( func );

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrWhileBegin()  . . . . . interpret while-statement, begin of statement
*F  IntrWhileBeginBody()  . . . . .  interpret while-statement, begin of body
*F  IntrWhileEndBody(<nr>)  . . . . .  interpret while-statement, end of body
*F  IntrWhileEnd()  . . . . . . . interpret while-statement, end of statement
**
**  'IntrWhileBegin' is   an action to  interpret   a while-statement.  It is
**  called when the    reader encounters the    'while', i.e., *before*   the
**  condition is read.
**
**  'IntrWhileBeginBody' is an action  to interpret a while-statement.  It is
**  called when the reader encounters  the  beginning of the statement  body,
**  i.e., *after* the condition is read.
**
**  'IntrWhileEndBody' is  an action to interpret   a while-statement.  It is
**  called when the reader encounters the end of the statement body.  <nr> is
**  the number of statements in the body.
**
**  'IntrWhileEnd' is an action to interpret a while-statement.  It is called
**  when  the reader encounters  the  end of  the  statement, i.e., immediate
**  after 'IntrWhileEndBody'.
**
**  Since loops cannot be interpreted immediately,  the interpreter calls the
**  coder  to create a  procedure (with no arguments) and  calls that.
*/
void            libGAP_IntrWhileBegin ( void )
{
    libGAP_Obj                 nams;           /* (empty) list of names           */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_TLS(libGAP_IntrCoding)++; libGAP_CodeWhileBegin(); return; }


    /* switch to coding mode now                                           */
    libGAP_CodeBegin();
    libGAP_TLS(libGAP_IntrCoding) = 1;

    /* code a function expression (with no arguments and locals)           */
    nams = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( nams, 0 );

    /* If we are in the break loop, then a local variable context may well exist,
       and we have to create an empty local variable names list to match the
       function expression that we are creating.

       If we are not in a break loop, then this would be a waste of time and effort */

    if (libGAP_TLS(libGAP_CountNams) > 0) {
        libGAP_GROW_PLIST(libGAP_TLS(libGAP_StackNams), ++libGAP_TLS(libGAP_CountNams));
        libGAP_SET_ELM_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams), nams);
        libGAP_SET_LEN_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams));
    }

    libGAP_CodeFuncExprBegin( 0, 0, nams, 0 );

    /* code a while loop                                                   */
    libGAP_CodeWhileBegin();
}

void            libGAP_IntrWhileBeginBody ( void )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }


    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeWhileBeginBody();
}

void            libGAP_IntrWhileEndBody (
    libGAP_UInt                nr )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeWhileEndBody( nr );
}

void            libGAP_IntrWhileEnd ( void )
{
    libGAP_Obj                 func;           /* the function, result            */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 1 ) { libGAP_TLS(libGAP_IntrCoding)--; libGAP_CodeWhileEnd(); return; }


    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeWhileEnd();

    /* code a function expression (with one statement in the body)         */
    libGAP_CodeFuncExprEnd( 1UL, 0UL );


    /* switch back to immediate mode, get the function                     */
    libGAP_TLS(libGAP_IntrCoding) = 0;
    libGAP_CodeEnd( 0 );

    /* If we are in a break loop, then we will have created a "dummy" local
       variable names list to get the counts right. Remove it */
    if (libGAP_TLS(libGAP_CountNams) > 0)
      libGAP_TLS(libGAP_CountNams)--;

    func = libGAP_TLS(libGAP_CodeResult);


    /* call the function                                                   */
    libGAP_CALL_0ARGS( func );

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrQualifiedExprBegin( UInt qual ) . . . . . . interpret expression guarded
**                                       by readwrite or readonlu
*F  IntrQualifiedExprEnd( ) 
**                                       by readwrite or readonlu
**
*/

void libGAP_IntrQualifiedExprBegin(libGAP_UInt qual) 
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeQualifiedExprBegin(qual); return; }
    libGAP_PushObj(libGAP_INTOBJ_INT(qual));
    return;
}

void libGAP_IntrQualifiedExprEnd( void ) 
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)--; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeQualifiedExprEnd(); return; }
    return;
}

/****************************************************************************
**
*F  IntrAtomicBegin()  . . . . . interpret atomic-statement, begin of statement
*F  IntrAtomicBeginBody(<nrexprs>)  . . . . .  interpret atomic-statement, begin of body
*F  IntrAtomicEndBody(<nrstats>)  . . . . .  interpret atomic-statement, end of body
*F  IntrAtomicEnd()  . . . . . . . interpret atomic-statement, end of statement
**
**  'IntrAtomicBegin' is   an action to  interpret   a atomic-statement.  It is
**  called when the    reader encounters the    'atomic', i.e., *before*   the
**  expressions to be locked are read.
**
**  'IntrAtomicBeginBody' is an action  to interpret a atomic-statement.  It is
**  called when the reader encounters  the  beginning of the statement  body,
**  i.e., *after* the expressions to be locked are read. <nrexprs> is the number
** of expressions to be locked
**
**  'IntrAtomicEndBody' is  an action to interpret   a atomic-statement.  It is
**  called when the reader encounters the end of the statement body.  <nrstats> is
**  the number of statements in the body.
**
**  'IntrAtomicEnd' is an action to interpret a atomic-statement.  It is called
**  when  the reader encounters  the  end of  the  statement, i.e., immediate
**  after 'IntrAtomicEndBody'.
**
**  These functions are just placeholders for the future HPC-GAP code
**
*/
void            libGAP_IntrAtomicBegin ( void )
{

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAtomicBegin(); return; }
    /* nothing to do here */
    return;
  
}

void            libGAP_IntrAtomicBeginBody ( libGAP_UInt nrexprs )
{
  libGAP_Obj nams;

    /* ignore    or code                                                          */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_TLS(libGAP_IntrCoding)++; libGAP_CodeAtomicBeginBody(nrexprs); return; }


    /* leave the expressions and qualifiers on the stack, switch to coding to process the body
       of the Atomic expression */
    libGAP_PushObj(libGAP_INTOBJ_INT(nrexprs));
    libGAP_CodeBegin();
    libGAP_TLS(libGAP_IntrCoding) = 1;
    
    
    /* code a function expression (with no arguments and locals)           */
    
    nams = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( nams, 0 );
    
    /* If we are in the break loop, then a local variable context may well exist,
       and we have to create an empty local variable names list to match the
       function expression that we are creating.
       
       If we are not in a break loop, then this would be a waste of time and effort */
    
    if (libGAP_TLS(libGAP_CountNams) > 0)
      {
	libGAP_GROW_PLIST(libGAP_TLS(libGAP_StackNams), ++libGAP_TLS(libGAP_CountNams));
	libGAP_SET_ELM_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams), nams);
	libGAP_SET_LEN_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams));
      }
    
    libGAP_CodeFuncExprBegin( 0, 0, nams, libGAP_TLS(libGAP_Input)->number );
}

void            libGAP_IntrAtomicEndBody (
    libGAP_Int                nrstats )
{
  libGAP_Obj body;

    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    if (libGAP_TLS(libGAP_IntrCoding) == 1) {
      /* This is the case where we are really immediately interpreting an atomic
	 statement, but we switched to coding to store the body until we have it all */

    /* If we are in a break loop, then we will have created a "dummy" local
       variable names list to get the counts right. Remove it */
      if (libGAP_TLS(libGAP_CountNams) > 0)
	libGAP_TLS(libGAP_CountNams)--;

      /* Code the body as a function expression */
      libGAP_CodeFuncExprEnd( nrstats, 0UL );
    

      /* switch back to immediate mode, get the function                     */
      libGAP_TLS(libGAP_IntrCoding) = 0;
      libGAP_CodeEnd( 0 );
      body = libGAP_TLS(libGAP_CodeResult);
      libGAP_PushObj(body);
      return;
    } else {
      
        assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
        libGAP_CodeAtomicEndBody( nrstats );
    }
}


void            libGAP_IntrAtomicEnd ( void )
{
    libGAP_Obj                 body;           /* the function, result            */
    libGAP_UInt                nrexprs;
    libGAP_UInt                i;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_TLS(libGAP_IntrCoding)--; libGAP_CodeAtomicEnd(); return; }
    /* Now we need to recover the objects to lock, and go to work */

    body = libGAP_PopObj();

    nrexprs = libGAP_INT_INTOBJ(libGAP_PopObj());

    for (i = 0; i < nrexprs; i++) {
      libGAP_PopObj(); /* pop object */
      libGAP_PopObj(); /* pop mode */
    }
    
      libGAP_CALL_0ARGS( body );

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrRepeatBegin() . . . .  interpret repeat-statement, begin of statement
*F  IntrRepeatBeginBody() . . . . . interpret repeat-statement, begin of body
*F  IntrRepeatEndBody(<nr>) . . . . . interpret repeat-statement, end of body
*F  IntrRepeatEnd() . . . . . .  interpret repeat-statement, end of statement
**
**  'IntrRepeatBegin"  is an action to interpret  a  repeat-statement.  It is
**  called when the read encounters the 'repeat'.
**
**  'IntrRepeatBeginBody' is an action  to interpret a  repeat-statement.  It
**  is called when the reader encounters the beginning of the statement body,
**  i.e., immediately after 'IntrRepeatBegin'.
**
**  'IntrRepeatEndBody' is an action  to interpret a repeat-statement.  It is
**  called when the reader  encounters the end of  the statement  body, i.e.,
**  *before* the condition is read.  <nr> is the  number of statements in the
**  body.
**
**  'IntrRepeatEnd' is  an  action to interpret  a repeat-statement.    It is
**  called when the reader encounters the end of the statement, i.e., *after*
**  the condition is read.
**
**  Since loops cannot be interpreted immediately,  the interpreter calls the
**  coder  to create a  procedure (with no arguments) and  calls that.
*/
void            libGAP_IntrRepeatBegin ( void )
{
    libGAP_Obj                 nams;           /* (empty) list of names           */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_TLS(libGAP_IntrCoding)++; libGAP_CodeRepeatBegin(); return; }


    /* switch to coding mode now                                           */
    libGAP_CodeBegin();
    libGAP_TLS(libGAP_IntrCoding) = 1;

    /* code a function expression (with no arguments and locals)           */
    nams = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( nams, 0 );

    /* If we are in the break loop, then a local variable context may well exist,
       and we have to create an empty local variable names list to match the
       function expression that we are creating.

       If we are not in a break loop, then this would be a waste of time and effort */

    if (libGAP_TLS(libGAP_CountNams) > 0) {
        libGAP_GROW_PLIST(libGAP_TLS(libGAP_StackNams), ++libGAP_TLS(libGAP_CountNams));
        libGAP_SET_ELM_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams), nams);
        libGAP_SET_LEN_PLIST(libGAP_TLS(libGAP_StackNams), libGAP_TLS(libGAP_CountNams));
    }

    libGAP_CodeFuncExprBegin( 0, 0, nams, libGAP_TLS(libGAP_Input)->number );

    /* code a repeat loop                                                  */
    libGAP_CodeRepeatBegin();
}

void            libGAP_IntrRepeatBeginBody ( void )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }


    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeRepeatBeginBody();
}

void            libGAP_IntrRepeatEndBody (
    libGAP_UInt                nr )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeRepeatEndBody( nr );
}

void            libGAP_IntrRepeatEnd ( void )
{
    libGAP_Obj                 func;           /* the function, result            */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 1 ) { libGAP_TLS(libGAP_IntrCoding)--; libGAP_CodeRepeatEnd(); return; }


    /* otherwise must be coding                                            */
    assert( libGAP_TLS(libGAP_IntrCoding) > 0 );
    libGAP_CodeRepeatEnd();

    /* code a function expression (with one statement in the body)         */
    libGAP_CodeFuncExprEnd( 1UL, 0UL );

    /* switch back to immediate mode, get the function                     */
    libGAP_TLS(libGAP_IntrCoding) = 0;
    libGAP_CodeEnd( 0 );
    /* If we are in a break loop, then we will have created a "dummy" local
       variable names list to get the counts right. Remove it */
    if (libGAP_TLS(libGAP_CountNams) > 0)
      libGAP_TLS(libGAP_CountNams)--;
    func = libGAP_TLS(libGAP_CodeResult);

    /* call the function                                                   */
    libGAP_CALL_0ARGS( func );

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrBreak() . . . . . . . . . . . . . . . . . . interpret break-statement
**
**  'IntrBreak'  is the action to interpret  a break-statement.  It is called
**  when the reader encounters a 'break;'.
**
**  Break-statements are  always coded (if  they are not ignored), since they
**  can only appear in loops.
*/
void            libGAP_IntrBreak ( void )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if ( libGAP_TLS(libGAP_IntrCoding) == 0 )
      libGAP_ErrorQuit("A break statement can only appear inside a loop",0L,0L);
    else
      libGAP_CodeBreak();
    return;
}


/****************************************************************************
**
*F  IntrContinue() . . . . . . . . . . . . . . . . . . interpret continue-statement
**
**  'IntrContinue'  is the action to interpret  a continue-statement.  It is called
**  when the reader encounters a 'continue;'.
**
**  Continue-statements are  always coded (if  they are not ignored), since they
**  can only appear in loops.
*/
void            libGAP_IntrContinue ( void )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if ( libGAP_TLS(libGAP_IntrCoding) == 0 )
      libGAP_ErrorQuit("A continue statement can only appear inside a loop",0L,0L);
    else
      libGAP_CodeContinue();
    return;
}


/****************************************************************************
**
*F  IntrReturnObj() . . . . . . . . . . . .  interpret return-value-statement
**
**  'IntrReturnObj' is the action  to interpret a return-value-statement.  It
**  is  called when  the reader encounters  a  'return  <expr>;', but *after*
**  reading the expression <expr>.
*/
void            libGAP_IntrReturnObj ( void )
{
    libGAP_Obj                 val;            /* return value                    */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeReturnObj(); return; }


    /* empty the values stack and push the return value                    */
    val = libGAP_PopObj();
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), 0 );
    libGAP_TLS(libGAP_CountObj) = 0;
    libGAP_PushObj( val );

    /* indicate that a return-value-statement was interpreted              */
    libGAP_TLS(libGAP_IntrReturning) = 1;
}


/****************************************************************************
**
*F  IntrReturnVoid()  . . . . . . . . . . . . interpret return-void-statement
**
**  'IntrReturnVoid' is the action to interpret  a return-void-statement.  It
**  is called when the reader encounters a 'return;'.
*/
void            libGAP_IntrReturnVoid ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeReturnVoid(); return; }


    /* empty the values stack and push the void value                      */
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), 0 );
    libGAP_TLS(libGAP_CountObj) = 0;
    libGAP_PushVoidObj();

    /* indicate that a return-void-statement was interpreted               */
    libGAP_TLS(libGAP_IntrReturning) = 2;
}


/****************************************************************************
**
*F  IntrQuit()  . . . . . . . . . . . . . . . . . .  interpret quit-statement
**
**  'IntrQuit' is the  action to interpret   a quit-statement.  It  is called
**  when the reader encounters a 'quit;'.
*/
void            libGAP_IntrQuit ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* 'quit' is not allowed in functions (by the reader)                  */
    /* assert( TLS(IntrCoding) == 0 ); */
    if ( libGAP_TLS(libGAP_IntrCoding) > 0 ) {
      libGAP_SyntaxError("'quit;' cannot be used in this context");
    }

    /* empty the values stack and push the void value                      */
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), 0 );
    libGAP_TLS(libGAP_CountObj) = 0;
    libGAP_PushVoidObj();


    /* indicate that a quit-statement was interpreted                      */
    libGAP_TLS(libGAP_IntrReturning) = libGAP_STATUS_QUIT;
}

/****************************************************************************
**
*F  IntrQUIT()  . . . . . . . . . . . . . . . . . .  interpret quit-statement
**
**  'IntrQUIT' is the  action to interpret   a quit-statement.  It  is called
**  when the reader encounters a 'QUIT;'.
*/
void            libGAP_IntrQUIT ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* 'quit' is not allowed in functions (by the reader)                  */
    assert( libGAP_TLS(libGAP_IntrCoding) == 0 );

    /* empty the values stack and push the void value                      */
    libGAP_SET_LEN_PLIST( libGAP_TLS(libGAP_StackObj), 0 );
    libGAP_TLS(libGAP_CountObj) = 0;
    libGAP_PushVoidObj();


    /* indicate that a quit-statement was interpreted                      */
    libGAP_TLS(libGAP_IntrReturning) = libGAP_STATUS_QQUIT;
}


/****************************************************************************
**
*F  IntrOrL() . . . . . . . . . .  interpret or-expression, left operand read
*F  IntrOr()  . . . . . . . . . . interpret or-expression, right operand read
**
**  'IntrOrL' is an action to interpret an or-expression.   It is called when
**  the reader encounters the 'or' keyword, i.e., *after* the left operand is
**  read by *before* the right operand is read.
**
**  'IntrOr' is an action to  interpret an or-expression.   It is called when
**  the reader encountered  the  end of  the  expression, i.e., *after*  both
**  operands are read.
*/
void            libGAP_IntrOrL ( void )
{
    libGAP_Obj                 opL;            /* value of left operand           */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeOrL(); return; }


    /* if the left operand is 'true', ignore the right operand             */
    opL = libGAP_PopObj();
    libGAP_PushObj( opL );
    if ( opL == libGAP_True ) {
        libGAP_PushObj( opL );
        libGAP_TLS(libGAP_IntrIgnoring) = 1;
    }
}

void            libGAP_IntrOr ( void )
{
    libGAP_Obj                 opL;            /* value of left  operand          */
    libGAP_Obj                 opR;            /* value of right operand          */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 1 ) { libGAP_TLS(libGAP_IntrIgnoring)--; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeOr(); return; }


    /* stop ignoring things now                                            */
    libGAP_TLS(libGAP_IntrIgnoring) = 0;

    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* if the left operand is 'true', this is the result                   */
    if      ( opL == libGAP_True ) {
        libGAP_PushObj( opL );
    }

    /* if the left operand is 'false', the result is the right operand     */
    else if ( opL == libGAP_False  ) {
        if ( opR == libGAP_True || opR == libGAP_False  ) {
            libGAP_PushObj( opR );
        }
        else {
            libGAP_ErrorQuit( "<expr> must be 'true' or 'false' (not a %s)",
                       (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L );
        }
    }

    /* signal an error                                                     */
    else {
        libGAP_ErrorQuit( "<expr> must be 'true' or 'false' (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L );
    }
}


/****************************************************************************
**
*F  IntrAndL()  . . . . . . . . . interpret and-expression, left operand read
*F  IntrAnd() . . . . . . . . .  interpret and-expression, right operand read
**
**  'IntrAndL' is  an action  to interpret an   and-expression.  It is called
**  when the reader  encounters the  'and'  keyword, i.e., *after*  the  left
**  operand is read by *before* the right operand is read.
**
**  'IntrAnd' is an action to interpret an and-expression.  It is called when
**  the reader encountered   the end of   the expression, i.e., *after*  both
**  operands are read.
*/
void            libGAP_IntrAndL ( void )
{
    libGAP_Obj                 opL;            /* value of left operand           */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAndL(); return; }


    /* if the left operand is 'false', ignore the right operand            */
    opL = libGAP_PopObj();
    libGAP_PushObj( opL );
    if ( opL == libGAP_False ) {
        libGAP_PushObj( opL );
        libGAP_TLS(libGAP_IntrIgnoring) = 1;
    }
}

extern  libGAP_Obj             libGAP_NewAndFilter (
            libGAP_Obj                     oper1,
            libGAP_Obj                     oper2 );

void            libGAP_IntrAnd ( void )
{
    libGAP_Obj                 opL;            /* value of left  operand          */
    libGAP_Obj                 opR;            /* value of right operand          */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 1 ) { libGAP_TLS(libGAP_IntrIgnoring)--; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAnd(); return; }


    /* stop ignoring things now                                            */
    libGAP_TLS(libGAP_IntrIgnoring) = 0;

    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* if the left operand is 'false', this is the result                  */
    if      ( opL == libGAP_False ) {
        libGAP_PushObj( opL );
    }

    /* if the left operand is 'true', the result is the right operand      */
    else if ( opL == libGAP_True  ) {
        if ( opR == libGAP_False || opR == libGAP_True  ) {
            libGAP_PushObj( opR );
        }
        else {
            libGAP_ErrorQuit(
                "<expr> must be 'true' or 'false' (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L );
        }
    }

    /* handle the 'and' of two filters                                    */
    else if ( libGAP_TNUM_OBJ(opL) == libGAP_T_FUNCTION ) {
        if ( libGAP_TNUM_OBJ(opR) == libGAP_T_FUNCTION ) {
            libGAP_PushObj( libGAP_NewAndFilter( opL, opR ) );
        }
        else {
            libGAP_ErrorQuit(
                "<expr> must be 'true' or 'false' (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L );
        }
    }

    /* signal an error                                                     */
    else {
        libGAP_ErrorQuit(
            "<expr> must be 'true' or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L );
    }
}


/****************************************************************************
**
*F  IntrNot() . . . . . . . . . . . . . . . . . . .  interpret not-expression
**
**  'IntrNot' is the action to interpret a not-expression.  It is called when
**  the reader encounters a not-expression, *after* the operand is read.
*/
void            libGAP_IntrNot ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 op;             /* operand                         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrIgnoring) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)   > 0 ) { libGAP_CodeNot(); return; }


    /* get and check the operand                                           */
    op = libGAP_PopObj();
    if ( op != libGAP_True && op != libGAP_False ) {
        libGAP_ErrorQuit(
            "<expr> must be 'true' or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(op), 0L );
    }

    /* negate the operand                                                  */
    val = (op == libGAP_False ? libGAP_True : libGAP_False);

    /* push the result                                                     */
    libGAP_PushObj( val );
}


/****************************************************************************
**
*F  IntrEq()  . . . . . . . . . . . . . . . . . . . .  interpret =-expression
*F  IntrNe()  . . . . . . . . . . . . . . . . . . . . interpret <>-expression
*F  IntrLt()  . . . . . . . . . . . . . . . . . . . . interpret  <-expression
*F  IntrGe()  . . . . . . . . . . . . . . . . . . . . interpret >=-expression
*F  IntrGt()  . . . . . . . . . . . . . . . . . . . .  interpret >-expression
*F  IntrLe()  . . . . . . . . . . . . . . . . . . . . interpret <=-expression
**
**  'IntrEq', 'IntrNe', 'IntrLt', 'IntrGe', 'IntrGt',   and 'IntrLe' are  the
**  actions to interpret the respective operator expression.  They are called
**  by the reader *after* *both* operands are read.
*/
void            libGAP_IntrXX ( void )
{
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* push the operands in reverse order                                  */
    libGAP_PushObj( opR );
    libGAP_PushObj( opL );
}

void            libGAP_IntrEq ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeEq(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compare them                                                        */
    val = (libGAP_EQ( opL, opR ) ? libGAP_True : libGAP_False);

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrNe ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeNe(); return; }


    /* '<left> <> <right>' is 'not <left> = <right>'                       */
    libGAP_IntrEq();
    libGAP_IntrNot();
}

void            libGAP_IntrLt ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeLt(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compare them                                                        */
    val = (libGAP_LT( opL, opR ) ? libGAP_True : libGAP_False);

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrGe ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeGe(); return; }


    /* '<left> >= <right>' is 'not <left> < <right>'                       */
    libGAP_IntrLt();
    libGAP_IntrNot();
}

void            libGAP_IntrGt ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeGt(); return; }


    /* '<left> > <right>' is '<right> < <left>'                            */
    libGAP_IntrXX();
    libGAP_IntrLt();
}

void            libGAP_IntrLe ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeLe(); return; }


    /* '<left> <= <right>' is 'not <right> < <left>'                       */
    libGAP_IntrXX();
    libGAP_IntrLt();
    libGAP_IntrNot();
}


/****************************************************************************
**
*F  IntrIn()  . . . . . . . . . . . . . . . . . . . . interpret in-expression
**
**  'IntrIn'  is the action  to interpret an  in-expression.  It is called by
**  the reader *after* *both* operands are read.
*/
void            libGAP_IntrIn ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIn(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* perform the test                                                    */
    val = (libGAP_IN( opL, opR ) ? libGAP_True : libGAP_False);

    /* push the result                                                     */
    libGAP_PushObj( val );
}


/****************************************************************************
**
*F  IntrSum() . . . . . . . . . . . . . . . . . . . .  interpret +-expression
*F  IntrAInv()  . . . . . . . . . . . . . . . .  interpret unary --expression
*F  IntrDiff()  . . . . . . . . . . . . . . . . . . .  interpret --expression
*F  IntrProd()  . . . . . . . . . . . . . . . . . . .  interpret *-expression
*F  IntrInv() . . . . . . . . . . . . . . . . . . .  interpret ^-1-expression
*F  IntrQuo() . . . . . . . . . . . . . . . . . . . .  interpret /-expression
*F  IntrMod()   . . . . . . . . . . . . . . . . . .  interpret mod-expression
*F  IntrPow() . . . . . . . . . . . . . . . . . . . .  interpret ^-expression
**
**  'IntrSum', 'IntrDiff',  'IntrProd',  'IntrQuo',  'IntrMod', and 'IntrPow'
**  are  the actions to interpret  the  respective operator expression.  They
**  are called by the reader *after* *both* operands are read.
*/
void            libGAP_IntrSum ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeSum(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compute the sum                                                     */
    val = libGAP_SUM( opL, opR );

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrAInv ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAInv(); return; }


    /* get the operand                                                     */
    opL = libGAP_PopObj();

    /* compute the additive inverse                                        */
    val = libGAP_AINV( opL );

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrDiff ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeDiff(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compute the difference                                              */
    val = libGAP_DIFF( opL, opR );

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrProd ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeProd(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compute the product                                                 */
    val = libGAP_PROD( opL, opR );

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrInv ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeInv(); return; }


    /* get the operand                                                     */
    opL = libGAP_PopObj();

    /* compute the multiplicative inverse                                  */
    val = libGAP_INV_MUT( opL );

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrQuo ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeQuo(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compute the quotient                                                */
    val = libGAP_QUO( opL, opR );

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrMod ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeMod(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compute the remainder                                               */
    val = libGAP_MOD( opL, opR );

    /* push the result                                                     */
    libGAP_PushObj( val );
}

void            libGAP_IntrPow ( void )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* left operand                    */
    libGAP_Obj                 opR;            /* right operand                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodePow(); return; }


    /* get the operands                                                    */
    opR = libGAP_PopObj();
    opL = libGAP_PopObj();

    /* compute the power                                                   */
    val = libGAP_POW( opL, opR );

    /* push the result                                                     */
    libGAP_PushObj( val );
}


/****************************************************************************
**
*F  IntrIntExpr(<str>)  . . . . . . . .  interpret literal integer expression
**
**  'IntrIntExpr' is the action  to  interpret a literal  integer expression.
**  <str> is the integer as a (null terminated) C character string.
*/
void            libGAP_IntrIntExpr (
    libGAP_Char *              str )
{
    libGAP_Obj                 val;            /* value = <upp> * <pow> + <low>   */
    libGAP_Obj                 upp;            /* upper part                      */
    libGAP_Int                 pow;            /* power                           */
    libGAP_Int                 low;            /* lower part                      */
    libGAP_Int                 sign;           /* is the integer negative         */
    libGAP_UInt                i;              /* loop variable                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIntExpr( str ); return; }


    /* get the signs, if any                                                */
    sign = 1;
    i = 0;
    while ( str[i] == '-' ) {
        sign = - sign;
        i++;
    }

    /* collect the digits in groups of 8                                   */
    low = 0;
    pow = 1;
    upp = libGAP_INTOBJ_INT(0);
    while ( str[i] != '\0' ) {
        low = 10 * low + str[i] - '0';
        pow = 10 * pow;
        if ( pow == 100000000L ) {
            upp = libGAP_PROD(upp,libGAP_INTOBJ_INT(pow) );
            upp = libGAP_SUM(upp  , libGAP_INTOBJ_INT(sign*low) );
            pow = 1;
            low = 0;
        }
        i++;
    }

    /* compose the integer value                                           */
    val = 0;
    if ( upp == libGAP_INTOBJ_INT(0) ) {
        val = libGAP_INTOBJ_INT(sign*low);
    }
    else if ( pow == 1 ) {
        val = upp;
    }
    else {
        upp =  libGAP_PROD( upp, libGAP_INTOBJ_INT(pow) );
        val = libGAP_SUM( upp , libGAP_INTOBJ_INT(sign*low) );
    }

    /* push the integer value                                              */
    libGAP_PushObj( val );
}


/****************************************************************************
**
*F  IntrLongIntExpr(<str>)   .  .  interpret literal long integer expression
**
**  'IntrLongIntExpr' is the action to  interpret a long literal integer
**  expression whose digits are stored in a string GAP object.
*/
void            libGAP_IntrLongIntExpr (
    libGAP_Obj               string )
{
    libGAP_Obj                 val;            /* value = <upp> * <pow> + <low>   */
    libGAP_Obj                 upp;            /* upper part                      */
    libGAP_Int                 pow;            /* power                           */
    libGAP_Int                 low;            /* lower part                      */
    libGAP_Int                 sign;           /* is the integer negative         */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_UChar *              str;            /* temp pointer                    */
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeLongIntExpr( string ); return; }


    /* get the signs, if any                                                */
    str = libGAP_CHARS_STRING(string);
    sign = 1;
    i = 0;
    while ( str[i] == '-' ) {
        sign = - sign;
        i++;
    }

    /* collect the digits in groups of 8                                   */
    low = 0;
    pow = 1;
    upp = libGAP_INTOBJ_INT(0);
    while ( str[i] != '\0' ) {
        low = 10 * low + str[i] - '0';
        pow = 10 * pow;
        if ( pow == 100000000L ) {
            upp = libGAP_PROD(upp,libGAP_INTOBJ_INT(pow) );
            upp = libGAP_SUM(upp  , libGAP_INTOBJ_INT(sign*low) );
            str = libGAP_CHARS_STRING(string);
            pow = 1;
            low = 0;
        }
        i++;
    }

    /* compose the integer value                                           */
    val = 0;
    if ( upp == libGAP_INTOBJ_INT(0) ) {
        val = libGAP_INTOBJ_INT(sign*low);
    }
    else if ( pow == 1 ) {
        val = upp;
    }
    else {
        upp =  libGAP_PROD( upp, libGAP_INTOBJ_INT(pow) );
        val = libGAP_SUM( upp , libGAP_INTOBJ_INT(sign*low) );
    }

    /* push the integer value                                              */
    libGAP_PushObj( val );
}

/****************************************************************************
**
*F  IntrFloatExpr(<str>)  . . . . . . . .  interpret literal float expression
**
**  'IntrFloatExpr' is the action  to  interpret a literal  float expression.
**  <str> is the float as a (null terminated) C character string.
*/

static libGAP_Obj libGAP_CONVERT_FLOAT_LITERAL_EAGER;

static libGAP_Obj libGAP_ConvertFloatLiteralEager(libGAP_Obj str) {
  libGAP_Char *chars = (libGAP_Char *)libGAP_CHARS_STRING(str);
  libGAP_UInt len = libGAP_GET_LEN_STRING(str);
  libGAP_Char mark = '\0';
  if (chars[len-1] == '_') {
    libGAP_SET_LEN_STRING(str, len-1);
    chars[len-1] = '\0';
  } else if (chars[len-2] == '_') {
    mark = chars[len-1];
    libGAP_SET_LEN_STRING(str, len-2);
    chars[len-2] = '\0';
  }
  return libGAP_CALL_2ARGS(libGAP_CONVERT_FLOAT_LITERAL_EAGER, str, libGAP_ObjsChar[(libGAP_UInt)mark]);
}

void            libGAP_IntrFloatExpr (
    libGAP_Char *              str )
{
    libGAP_Obj                 val;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) {  libGAP_CodeFloatExpr( str );   return; }

    libGAP_C_NEW_STRING_DYN(val, str);
    libGAP_PushObj(libGAP_ConvertFloatLiteralEager(val));
}


/****************************************************************************
**
*F  IntrLongFloatExpr(<str>)   .  .  interpret literal long float expression
**
**  'IntrLongFloatExpr' is the action to  interpret a long literal float
**  expression whose digits are stored in a string GAP object.
*/
void            libGAP_IntrLongFloatExpr (
    libGAP_Obj               string )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeLongFloatExpr( string );  return; }

    libGAP_PushObj(libGAP_ConvertFloatLiteralEager(string));
}

/****************************************************************************
**
*F  IntrTrueExpr()  . . . . . . . . . . . . interpret literal true expression
**
**  'IntrTrueExpr' is the action to interpret a literal true expression.
*/
void            libGAP_IntrTrueExpr ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeTrueExpr(); return; }


    /* push the value                                                      */
    libGAP_PushObj( libGAP_True );
}


/****************************************************************************
**
*F  IntrFalseExpr() . . . . . . . . . . .  interpret literal false expression
**
**  'IntrFalseExpr' is the action to interpret a literal false expression.
*/
void            libGAP_IntrFalseExpr ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFalseExpr(); return; }


    /* push the value                                                      */
    libGAP_PushObj( libGAP_False );
}


/****************************************************************************
**
*F  IntrCharExpr(<chr>) . . . . . . .  interpret literal character expression
**
**  'IntrCharExpr' is the action to interpret a literal character expression.
**  <chr> is the C character.
*/
void            libGAP_IntrCharExpr (
    libGAP_Char                chr )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeCharExpr( chr ); return; }


    /* push the value                                                      */
    libGAP_PushObj( libGAP_ObjsChar[ (libGAP_UChar)chr ] );
}


/****************************************************************************
**
*F  IntrPermCycle(<nr>) . . . . . .  interpret literal permutation expression
*F  IntrPerm(<nr>)  . . . . . . . .  interpret literal permutation expression
*/
void            libGAP_IntrPermCycle (
    libGAP_UInt                nrx,
    libGAP_UInt                nrc )
{
    libGAP_Obj                 perm;           /* permutation                     */
    libGAP_UInt4 *             ptr4;           /* pointer into perm               */
    libGAP_Obj                 val;            /* one entry as value              */
    libGAP_UInt                c, p, l;        /* entries in permutation          */
    libGAP_UInt                m;              /* maximal entry in permutation    */
    libGAP_UInt                j, k;           /* loop variable                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodePermCycle(nrx,nrc); return; }


    /* get the permutation (allocate for the first cycle)                  */
    if ( nrc == 1 ) {
        m = 0;
        perm = libGAP_NEW_PERM4( 0 );
        ptr4 = libGAP_ADDR_PERM4( perm );
    }
    else {
        m = libGAP_INT_INTOBJ( libGAP_ELM_LIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) - nrx ) );
        perm = libGAP_ELM_LIST( libGAP_TLS(libGAP_StackObj), libGAP_TLS(libGAP_CountObj) - nrx - 1 );
        ptr4 = libGAP_ADDR_PERM4( perm );
    }

    /* multiply the permutation with the cycle                             */
    c = p = l = 0;
    for ( j = nrx; 1 <= j; j-- ) {

        /* get and check current entry for the cycle                       */
        val = libGAP_PopObj();
        if ( ! libGAP_IS_INTOBJ(val) || libGAP_INT_INTOBJ(val) <= 0 ) {
            libGAP_ErrorQuit(
                "Permutation: <expr> must be a positive integer (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(val), 0L );
        }
        c = libGAP_INT_INTOBJ(val);

        /* if necessary resize the permutation                             */
        if ( libGAP_SIZE_OBJ(perm)/sizeof(libGAP_UInt4) < c ) {
            libGAP_ResizeBag( perm, (c + 1023) / 1024 * 1024 * sizeof(libGAP_UInt4) );
            ptr4 = libGAP_ADDR_PERM4( perm );
            for ( k = m+1; k <= libGAP_SIZE_OBJ(perm)/sizeof(libGAP_UInt4); k++ ) {
                ptr4[k-1] = k-1;
            }
        }
        if ( m < c ) {
            m = c;
        }

        /* check that the cycles are disjoint                              */
        if ( (p != 0 && p == c) || (ptr4[c-1] != c-1) ) {
            libGAP_ErrorQuit(
                "Permutation: cycles must be disjoint and duplicate-free",
                0L, 0L );
        }

        /* enter the previous entry at current location                    */
        if ( p != 0 ) { ptr4[c-1] = p-1; }
        else          { l = c;          }

        /* remember current entry for next round                           */
        p = c;
    }

    /* enter first (last popped) entry at last (first popped) location     */
    if (ptr4[l-1] != l-1) {
        libGAP_ErrorQuit("Permutation: cycles must be disjoint and duplicate-free", 0L, 0L );
    }
    ptr4[l-1] = p-1;

    /* push the permutation (if necessary, drop permutation first)         */
    if ( nrc != 1 ) { libGAP_PopObj(); libGAP_PopObj(); }
    libGAP_PushObj( perm );
    libGAP_PushObj( libGAP_INTOBJ_INT(m) );
}

void            libGAP_IntrPerm (
    libGAP_UInt                nrc )
{
    libGAP_Obj                 perm;           /* permutation, result             */
    libGAP_UInt4 *             ptr4;           /* pointer into permutation        */
    libGAP_UInt2 *             ptr2;           /* pointer into permutation        */
    libGAP_UInt                m;              /* maximal entry in permutation    */
    libGAP_UInt                k;              /* loop variable                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodePerm(nrc); return; }


    /* special case for identity permutation                               */
    if ( nrc == 0 ) {
        perm = libGAP_NEW_PERM2( 0 );
    }

    /* otherwise                                                           */
    else {

        /* get the permutation and its maximal entry                       */
        m  = libGAP_INT_INTOBJ( libGAP_PopObj() );
        perm = libGAP_PopObj();

        /* if possible represent the permutation with short entries        */
        if ( m <= 65536UL ) {
            ptr2 = libGAP_ADDR_PERM2( perm );
            ptr4 = libGAP_ADDR_PERM4( perm );
            for ( k = 1; k <= m; k++ ) {
                ptr2[k-1] = ptr4[k-1];
            };
            libGAP_RetypeBag( perm, libGAP_T_PERM2 );
            libGAP_ResizeBag( perm, m * sizeof(libGAP_UInt2) );
        }

        /* otherwise just shorten the permutation                          */
        else {
            libGAP_ResizeBag( perm, m * sizeof(libGAP_UInt4) );
        }

    }

    /* push the result                                                     */
    libGAP_PushObj( perm );
}


/****************************************************************************
**
*F  IntrListExprBegin(<top>)  . . . . . . . . . .  interpret list expr, begin
*F  IntrListExprBeginElm(<pos>) . . . . .  interpret list expr, begin element
*F  IntrListExprEndElm()  . . . . . . . . .  interpret list expr, end element
*F  IntrListExprEnd(<nr>,<range>,<top>,<tilde>) . .  interpret list expr, end
*/
void            libGAP_IntrListExprBegin (
    libGAP_UInt                top )
{
    libGAP_Obj                 list;           /* new list                        */
    libGAP_Obj                 old;            /* old value of '~'                */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeListExprBegin( top ); return; }


    /* allocate the new list                                               */
    list = libGAP_NEW_PLIST( libGAP_T_PLIST_EMPTY, 0 );
    libGAP_SET_LEN_PLIST( list, 0 );

    /* if this is an outmost list, save it for reference in '~'            */
    /* (and save the old value of '~' on the values stack)                 */
    if ( top ) {
        old = libGAP_VAL_GVAR( libGAP_Tilde );
        if ( old != 0 ) { libGAP_PushObj( old ); }
        else            { libGAP_PushVoidObj();  }
        libGAP_AssGVar( libGAP_Tilde, list );
    }

    /* push the list                                                       */
    libGAP_PushObj( list );
}

void            libGAP_IntrListExprBeginElm (
    libGAP_UInt                pos )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeListExprBeginElm( pos ); return; }


    /* remember this position on the values stack                          */
    libGAP_PushObj( libGAP_INTOBJ_INT(pos) );
}

void            libGAP_IntrListExprEndElm ( void )
{
    libGAP_Obj                 list;           /* list that is currently made     */
    libGAP_Obj                 pos;            /* position                        */
    libGAP_UInt                p;              /* position, as a C integer        */
    libGAP_Obj                 val;            /* value to assign into list       */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeListExprEndElm(); return; }


    /* get the value                                                       */
    val = libGAP_PopObj();

    /* get the position                                                    */
    pos = libGAP_PopObj();
    p = libGAP_INT_INTOBJ( pos );

    /* get the list                                                        */
    list = libGAP_PopObj();

    /* assign the element into the list                                    */
    libGAP_ASS_LIST( list, p, val );

    /* push the list again                                                 */
    libGAP_PushObj( list );
}

void            libGAP_IntrListExprEnd (
    libGAP_UInt                nr,
    libGAP_UInt                range,
    libGAP_UInt                top,
    libGAP_UInt                tilde )
{
    libGAP_Obj                 list;           /* the list, result                */
    libGAP_Obj                 old;            /* old value of '~'                */
    libGAP_Int                 low;            /* low value of range              */
    libGAP_Int                 inc;            /* increment of range              */
    libGAP_Int                 high;           /* high value of range             */
    libGAP_Obj                 val;            /* temporary value                 */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeListExprEnd(nr,range,top,tilde); return; }


    /* if this was a top level expression, restore the value of '~'        */
    if ( top ) {
        list = libGAP_PopObj();
        old = libGAP_PopVoidObj();
        libGAP_AssGVar( libGAP_Tilde, old );
        libGAP_PushObj( list );
    }

    /* if this was a range, convert the list to a range                    */
    if ( range ) {

        /* get the list                                                    */
        list = libGAP_PopObj();

        /* get the low value                                               */
        val = libGAP_ELM_LIST( list, 1 );
        if ( ! libGAP_IS_INTOBJ(val) ) {
            libGAP_ErrorQuit(
                "Range: <first> must be an integer less than 2^%d (not a %s)",
                libGAP_NR_SMALL_INT_BITS, (libGAP_Int)libGAP_TNAM_OBJ(val) );
        }
        low = libGAP_INT_INTOBJ( val );

        /* get the increment                                               */
        if ( nr == 3 ) {
            val = libGAP_ELM_LIST( list, 2 );
            if ( ! libGAP_IS_INTOBJ(val) ) {
                libGAP_ErrorQuit(
                    "Range: <second> must be an integer less than 2^%d (not a %s)",
                    libGAP_NR_SMALL_INT_BITS, (libGAP_Int)libGAP_TNAM_OBJ(val) );
            }
            if ( libGAP_INT_INTOBJ(val) == low ) {
                libGAP_ErrorQuit(
                      "Range: <second> must not be equal to <first> (%d)",
                      (libGAP_Int)low, 0L );
            }
            inc = libGAP_INT_INTOBJ(val) - low;
        }
        else {
            inc = 1;
        }

        /* get and check the high value                                    */
        val = libGAP_ELM_LIST( list, libGAP_LEN_LIST(list) );
        if ( ! libGAP_IS_INTOBJ(val) ) {
            libGAP_ErrorQuit(
                "Range: <last> must be an integer less than 2^%d (not a %s)",
                libGAP_NR_SMALL_INT_BITS, (libGAP_Int)libGAP_TNAM_OBJ(val) );
        }
        if ( (libGAP_INT_INTOBJ(val) - low) % inc != 0 ) {
            libGAP_ErrorQuit(
                "Range: <last>-<first> (%d) must be divisible by <inc> (%d)",
                (libGAP_Int)(libGAP_INT_INTOBJ(val)-low), (libGAP_Int)inc );
        }
        high = libGAP_INT_INTOBJ(val);

        /* if <low> is larger than <high> the range is empty               */
        if ( (0 < inc && high < low) || (inc < 0 && low < high) ) {
            list = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
            libGAP_SET_LEN_PLIST( list, 0 );
        }

        /* if <low> is equal to <high> the range is a singleton list       */
        else if ( low == high ) {
            list = libGAP_NEW_PLIST( libGAP_T_PLIST, 1 );
            libGAP_SET_LEN_PLIST( list, 1 );
            libGAP_SET_ELM_PLIST( list, 1, libGAP_INTOBJ_INT(low) );
        }

        /* else make the range                                             */
        else {
            /* length must be a small integer as well */
            if ((high-low) / inc + 1 >= (1L<<libGAP_NR_SMALL_INT_BITS)) {
                libGAP_ErrorQuit("Range: the length of a range must be less than 2^%d",
                           libGAP_NR_SMALL_INT_BITS, 0L);
            }

            if ( 0 < inc )
                list = libGAP_NEW_RANGE_SSORT();
            else
                list = libGAP_NEW_RANGE_NSORT();
            libGAP_SET_LEN_RANGE( list, (high-low) / inc + 1 );
            libGAP_SET_LOW_RANGE( list, low );
            libGAP_SET_INC_RANGE( list, inc );
        }

        /* push the list again                                             */
        libGAP_PushObj( list );
    }
    else {
        /* give back unneeded memory */
        list = libGAP_PopObj( );
        libGAP_SHRINK_PLIST( list, libGAP_LEN_PLIST(list) );
        libGAP_PushObj( list );
    }
}


/****************************************************************************
**
*F  IntrStringExpr(<str>) . . . . . . . . interpret literal string expression
*/
void           libGAP_IntrStringExpr (
    libGAP_Obj               string )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeStringExpr( string ); return; }


    /* push the string, already newly created                              */
    libGAP_PushObj( string );
}

/****************************************************************************
**
*F  IntrRecExprBegin(<top>) . . . . . . . . . .  interpret record expr, begin
*F  IntrRecExprBeginElmName(<rnam>) . .  interpret record expr, begin element
*F  IntrRecExprBeginElmExpr() . . . . .  interpret record expr, begin element
*F  IntrRecExprEndElmExpr() . . . . . . .  interpret record expr, end element
*F  IntrRecExprEnd(<nr>,<top>,<tilde>)  . . . . .  interpret record expr, end
*/
void            libGAP_IntrRecExprBegin (
    libGAP_UInt                top )
{
    libGAP_Obj                 record;         /* new record                      */
    libGAP_Obj                 old;            /* old value of '~'                */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeRecExprBegin( top ); return; }


    /* allocate the new record                                             */
    record = libGAP_NEW_PREC( 0 );

    /* if this is an outmost record, save it for reference in '~'          */
    /* (and save the old value of '~' on the values stack)                 */
    if ( top ) {
        old = libGAP_VAL_GVAR( libGAP_Tilde );
        if ( old != 0 ) { libGAP_PushObj( old ); }
        else            { libGAP_PushVoidObj();  }
        libGAP_AssGVar( libGAP_Tilde, record );
    }

    /* push the record                                                     */
    libGAP_PushObj( record );
}

void            libGAP_IntrRecExprBeginElmName (
    libGAP_UInt                rnam )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeRecExprBeginElmName( rnam ); return; }


    /* remember the name on the values stack                               */
    libGAP_PushObj( (libGAP_Obj)rnam );
}

void            libGAP_IntrRecExprBeginElmExpr ( void )
{
    libGAP_UInt                rnam;           /* record name                     */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeRecExprBeginElmExpr(); return; }


    /* convert the expression to a record name                             */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* remember the name on the values stack                               */
    libGAP_PushObj( (libGAP_Obj)rnam );
}

void            libGAP_IntrRecExprEndElm ( void )
{
    libGAP_Obj                 record;         /* record that is currently made   */
    libGAP_UInt                rnam;           /* name of record element          */
    libGAP_Obj                 val;            /* value of record element         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeRecExprEndElm(); return; }


    /* get the value                                                       */
    val = libGAP_PopObj();

    /* get the record name                                                 */
    rnam = (libGAP_UInt)libGAP_PopObj();

    /* get the record                                                      */
    record = libGAP_PopObj();

    /* assign the value into the record                                    */
    libGAP_ASS_REC( record, rnam, val );

    /* push the record again                                               */
    libGAP_PushObj( record );
}

void            libGAP_IntrRecExprEnd (
    libGAP_UInt                nr,
    libGAP_UInt                top,
    libGAP_UInt                tilde )
{
    libGAP_Obj                 record;         /* record that is currently made   */
    libGAP_Obj                 old;            /* old value of '~'                */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeRecExprEnd(nr,top,tilde); return; }


    /* if this was a top level expression, restore the value of '~'        */
    if ( top ) {
        record = libGAP_PopObj();
        old = libGAP_PopVoidObj();
        libGAP_AssGVar( libGAP_Tilde, old );
        libGAP_PushObj( record );
    }
}

/****************************************************************************
**
*F  IntrFuncCallOptionsBegin() . . . .. . . . . .  interpret options, begin
*F  IntrFuncCallOptionsBeginElmName(<rnam>).  interpret options, begin element
*F  IntrFuncCallOptionsBeginElmExpr() . .. .  interpret options, begin element
*F  IntrFuncCallOptionsEndElm() . . .. .  . .  interpret options, end element
*F  IntrFuncCallOptionsEndElmEmpty() .. .  . .  interpret options, end element
*F  IntrFuncCallOptionsEnd(<nr>)  . . . . . . . .  interpret options, end
**
**  The net effect of all of these is to leave a record object on the stack
**  where IntrFuncCallEnd can use it
*/
void            libGAP_IntrFuncCallOptionsBegin ( void )
{
    libGAP_Obj                 record;         /* new record                      */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFuncCallOptionsBegin( ); return; }


    /* allocate the new record                                             */
    record = libGAP_NEW_PREC( 0 );
    /* push the record                                                     */
    libGAP_PushObj( record );
}

void            libGAP_IntrFuncCallOptionsBeginElmName (
    libGAP_UInt                rnam )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFuncCallOptionsBeginElmName( rnam ); return; }


    /* remember the name on the values stack                               */
    libGAP_PushObj( (libGAP_Obj)rnam );
}

void            libGAP_IntrFuncCallOptionsBeginElmExpr ( void )
{
    libGAP_UInt                rnam;           /* record name                     */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFuncCallOptionsBeginElmExpr(); return; }


    /* convert the expression to a record name                             */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* remember the name on the values stack                               */
    libGAP_PushObj( (libGAP_Obj)rnam );
}

void            libGAP_IntrFuncCallOptionsEndElm ( void )
{
    libGAP_Obj                 record;         /* record that is currently made   */
    libGAP_UInt                rnam;           /* name of record element          */
    libGAP_Obj                 val;            /* value of record element         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFuncCallOptionsEndElm(); return; }


    /* get the value                                                       */
    val = libGAP_PopObj();

    /* get the record name                                                 */
    rnam = (libGAP_UInt)libGAP_PopObj();

    /* get the record                                                      */
    record = libGAP_PopObj();

    /* assign the value into the record                                    */
    libGAP_ASS_REC( record, rnam, val );

    /* push the record again                                               */
    libGAP_PushObj( record );
}

void            libGAP_IntrFuncCallOptionsEndElmEmpty ( void )
{
    libGAP_Obj                 record;         /* record that is currently made   */
    libGAP_UInt                rnam;           /* name of record element          */
    libGAP_Obj                 val;            /* value of record element         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFuncCallOptionsEndElmEmpty(); return; }


    /* get the value                                                       */
    val = libGAP_True;

    /* get the record name                                                 */
    rnam = (libGAP_UInt)libGAP_PopObj();

    /* get the record                                                      */
    record = libGAP_PopObj();

    /* assign the value into the record                                    */
    libGAP_ASS_REC( record, rnam, val );

    /* push the record again                                               */
    libGAP_PushObj( record );
}

void            libGAP_IntrFuncCallOptionsEnd ( libGAP_UInt nr )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeFuncCallOptionsEnd(nr); return; }


}


/****************************************************************************
**
*F  IntrAssLVar(<lvar>) . . . . . . . . . . . . interpret assignment to local
*/
void            libGAP_IntrAssLVar (
    libGAP_UInt                lvar )
{
  libGAP_Obj val;
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if ( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeAssLVar( lvar );

    /* Or in the break loop */
    else {
        val = libGAP_PopObj();
        libGAP_ASS_LVAR(lvar, val);
        libGAP_PushObj(val);
    }
}

void            libGAP_IntrUnbLVar (
    libGAP_UInt                lvar )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if ( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeUnbLVar( lvar );

    /* or in the break loop */
    else {
        libGAP_ASS_LVAR(lvar,0);
        libGAP_PushVoidObj();
    }
}


/****************************************************************************
**
*F  IntrRefLVar(<lvar>) . . . . . . . . . . . .  interpret reference to local
*/
void            libGAP_IntrRefLVar (
    libGAP_UInt                lvar )
{
  libGAP_Obj val;
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if ( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeRefLVar( lvar );

    /* or in the break loop */

    else {
        while ((val = libGAP_OBJ_LVAR(lvar))==0) {
            libGAP_ErrorReturnVoid(
                            "Variable: '%s' must have an assigned value",
                            (libGAP_Int)libGAP_NAME_LVAR( (libGAP_UInt)( lvar )), 0L,
                            "you can 'return;' after assigning a value" );

        }
        libGAP_PushObj(val);
    }
}

void            libGAP_IntrIsbLVar (
    libGAP_UInt                lvar )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeIsbLVar( lvar );

    /* or debugging */
    else {
        libGAP_PushObj(libGAP_OBJ_LVAR(lvar) != (libGAP_Obj)0 ? libGAP_True : libGAP_False);
    }
}


/****************************************************************************
**
*F  IntrAssHVar(<hvar>) . . . . . . . . . . .  interpret assignment to higher
*/
void            libGAP_IntrAssHVar (
    libGAP_UInt                hvar )
{
  libGAP_Obj val;
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeAssHVar( hvar );
    /* Or in the break loop */
    else {
        val = libGAP_PopObj();
        libGAP_ASS_HVAR(hvar, val);
        libGAP_PushObj(val);
    }
}

void            libGAP_IntrUnbHVar (
    libGAP_UInt                hvar )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if ( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeUnbHVar( hvar );
    /* or debugging */
    else {
        libGAP_ASS_HVAR(hvar, 0);
        libGAP_PushVoidObj();
    }
}


/****************************************************************************
**
*F  IntrRefHVar(<hvar>) . . . . . . . . . . . . interpret reference to higher
*/
void            libGAP_IntrRefHVar (
    libGAP_UInt                hvar )
{
  libGAP_Obj val;
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeRefHVar( hvar );
    /* or debugging */
    else {
        while ((val = libGAP_OBJ_HVAR(hvar))==0) {
            libGAP_ErrorReturnVoid(
                            "Variable: '%s' must have an assigned value",
                            (libGAP_Int)libGAP_NAME_HVAR( (libGAP_UInt)( hvar )), 0L,
                            "you can 'return;' after assigning a value" );

        }
        libGAP_PushObj(val);
    }
}

void            libGAP_IntrIsbHVar (
    libGAP_UInt                hvar )
{
    /* ignore                                                              */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }

    /* otherwise must be coding                                            */
    if( libGAP_TLS(libGAP_IntrCoding) > 0 )
      libGAP_CodeIsbHVar( hvar );
    /* or debugging */
    else
      libGAP_PushObj((libGAP_OBJ_HVAR(hvar) != (libGAP_Obj) 0) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  IntrAssDVar(<dvar>) . . . . . . . . . . . . interpret assignment to debug
*/
extern  libGAP_Obj             libGAP_ErrorLVars;

void            libGAP_IntrAssDVar (
    libGAP_UInt                dvar,
    libGAP_UInt                depth )
{
    libGAP_Obj                 rhs;            /* right hand side                 */
    libGAP_Obj                 currLVars;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    /* if ( TLS(IntrCoding)    > 0 ) { CodeAssDVar( gvar ); return; } */


    if ( libGAP_TLS(libGAP_IntrCoding) > 0 ) {
        libGAP_ErrorQuit( "Variable: <debug-variable-%d-%d> cannot be used here",
                   dvar >> 10, dvar & 0x3FF );
    }


    /* get the right hand side                                             */
    rhs = libGAP_PopObj();

    /* assign the right hand side                                          */
    currLVars = libGAP_TLS(libGAP_CurrLVars);
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_ErrorLVars) );
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_ErrorLVars) );
    while (depth--)
      libGAP_SWITCH_TO_OLD_LVARS( libGAP_PTR_BAG(libGAP_TLS(libGAP_CurrLVars)) [2] );
    libGAP_ASS_HVAR( dvar, rhs );
    libGAP_SWITCH_TO_OLD_LVARS( currLVars  );

    /* push the right hand side again                                      */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrUnbDVar (
    libGAP_UInt                dvar,
    libGAP_UInt                depth )
{
    libGAP_Obj                 currLVars;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    /* if ( TLS(IntrCoding)    > 0 ) { CodeUnbGVar( gvar ); return; } */


    if ( libGAP_TLS(libGAP_IntrCoding) > 0 ) {
        libGAP_ErrorQuit( "Variable: <debug-variable-%d-%d> cannot be used here",
                   dvar >> 10, dvar & 0x3FF );
    }

    /* assign the right hand side                                          */
    currLVars = libGAP_TLS(libGAP_CurrLVars);
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_ErrorLVars) );
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_ErrorLVars) );
    while (depth--)
      libGAP_SWITCH_TO_OLD_LVARS( libGAP_PTR_BAG(libGAP_TLS(libGAP_CurrLVars)) [2] );
    libGAP_ASS_HVAR( dvar, (libGAP_Obj)0 );
    libGAP_SWITCH_TO_OLD_LVARS( currLVars  );

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrRefDVar(<dvar>) . . . . . . . . . . . .  interpret reference to debug
*/
void            libGAP_IntrRefDVar (
    libGAP_UInt                dvar,
    libGAP_UInt                depth )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 currLVars;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    /* if ( TLS(IntrCoding)    > 0 ) { CodeRefGVar( gvar ); return; } */


    if ( libGAP_TLS(libGAP_IntrCoding) > 0 ) {
        libGAP_ErrorQuit( "Variable: <debug-variable-%d-%d> cannot be used here",
                   dvar >> 10, dvar & 0x3FF );
    }

    /* get and check the value                                             */
    currLVars = libGAP_TLS(libGAP_CurrLVars);
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_ErrorLVars) );
    while (depth--)
      libGAP_SWITCH_TO_OLD_LVARS( libGAP_PTR_BAG(libGAP_TLS(libGAP_CurrLVars)) [2] );
    val = libGAP_OBJ_HVAR( dvar );
    libGAP_SWITCH_TO_OLD_LVARS( currLVars  );
    if ( val == 0 ) {
        libGAP_ErrorQuit( "Variable: <debug-variable-%d-%d> must have a value",
                   dvar >> 10, dvar & 0xFFFF );
    }

    /* push the value                                                      */
    libGAP_PushObj( val );
}

void            libGAP_IntrIsbDVar (
    libGAP_UInt                dvar,
    libGAP_UInt                depth )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 currLVars;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    /* if ( TLS(IntrCoding)    > 0 ) { CodeIsbGVar( gvar ); return; } */


    /* get the value                                                       */
    currLVars = libGAP_TLS(libGAP_CurrLVars);
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_ErrorLVars) );
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_ErrorLVars) );
    while (depth--)
      libGAP_SWITCH_TO_OLD_LVARS( libGAP_PTR_BAG(libGAP_TLS(libGAP_CurrLVars)) [2] );
    val = libGAP_OBJ_HVAR( dvar );
    libGAP_SWITCH_TO_OLD_LVARS( currLVars  );

    /* push the value                                                      */
    libGAP_PushObj( (val != 0 ? libGAP_True : libGAP_False) );
}


/****************************************************************************
**
*F  IntrAssGVar(<gvar>) . . . . . . . . . . .  interpret assignment to global
*/
void            libGAP_IntrAssGVar (
    libGAP_UInt                gvar )
{
    libGAP_Obj                 rhs;            /* right hand side                 */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssGVar( gvar ); return; }


    /* get the right hand side                                             */
    rhs = libGAP_PopObj();

    /* assign the right hand side                                          */
    libGAP_AssGVar( gvar, rhs );

    /* push the right hand side again                                      */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrUnbGVar (
    libGAP_UInt                gvar )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeUnbGVar( gvar ); return; }


    /* assign the right hand side                                          */
    libGAP_AssGVar( gvar, (libGAP_Obj)0 );

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrRefGVar(<gvar>) . . . . . . . . . . . . interpret reference to global
*/
void            libGAP_IntrRefGVar (
    libGAP_UInt                gvar )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeRefGVar( gvar ); return; }


    /* get and check the value                                             */
    if ( (val = libGAP_ValAutoGVar( gvar )) == 0 ) {
        libGAP_ErrorQuit(
            "Variable: '%s' must have a value",
            (libGAP_Int)libGAP_NameGVar(gvar), 0L );
    }

    /* push the value                                                      */
    libGAP_PushObj( val );
}

void            libGAP_IntrIsbGVar (
    libGAP_UInt                gvar )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIsbGVar( gvar ); return; }


    /* get the value                                                       */
    val = libGAP_ValAutoGVar( gvar );

    /* push the value                                                      */
    libGAP_PushObj( (val != 0 ? libGAP_True : libGAP_False) );
}


/****************************************************************************
**
*F  IntrAssList() . . . . . . . . . . . . . .  interpret assignment to a list
*F  IntrAsssList()  . . . . . . . . . interpret multiple assignment to a list
*F  IntrAssListLevel(<level>) . . . . . interpret assignment to several lists
*F  IntrAsssListLevel(<level>)  . . intr multiple assignment to several lists
*/
void            libGAP_IntrAssList ( libGAP_Int narg )
{
    libGAP_Obj                 list;           /* list                            */
    libGAP_Obj                 pos;            /* position                        */
    libGAP_Obj                 rhs;            /* right hand side                 */
    libGAP_Obj pos1,pos2;
    libGAP_Obj ixs;
    libGAP_Int i;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssList( narg); return; }

    /* get the right hand side                                             */
    rhs = libGAP_PopObj();
    
    switch (narg) {
    case 1:

      /* get the position                                                    */
      pos = libGAP_PopObj();

      /* get the list (checking is done by 'ASS_LIST' or 'ASSB_LIST')        */
      list = libGAP_PopObj();

      /* assign to the element of the list                                   */
      if (libGAP_IS_POS_INTOBJ(pos)) {
        libGAP_ASS_LIST( list, libGAP_INT_INTOBJ(pos), rhs );
      } else {
        libGAP_ASSB_LIST(list, pos, rhs);
      }
      break;

    case 2:
      pos2 = libGAP_PopObj();
      pos1 = libGAP_PopObj();
      list = libGAP_PopObj();

      libGAP_ASS2_LIST(list, pos1, pos2, rhs);
      break;

    default:
      ixs = libGAP_NEW_PLIST(libGAP_T_PLIST, narg);
      for (i = narg; i > 0; i--) {
	pos = libGAP_PopObj();
	libGAP_SET_ELM_PLIST(ixs, i, pos);
	libGAP_CHANGED_BAG(ixs);
      }
      libGAP_SET_LEN_PLIST(ixs, narg);
      list = libGAP_PopObj();
      libGAP_ASSB_LIST(list, ixs, rhs);
    }
      
    /* push the right hand side again                                      */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrAsssList ( void )
{
    libGAP_Obj                 list;           /* list                            */
    libGAP_Obj                 poss;           /* positions                       */
    libGAP_Obj                 rhss;           /* right hand sides                */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAsssList(); return; }


    /* get the right hand sides                                            */
    rhss = libGAP_PopObj();
    if ( ! libGAP_IS_DENSE_LIST( rhss ) ) {
        libGAP_ErrorQuit(
            "List Assignment: <rhss> must be a dense list",
            0L, 0L );
    }

    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
    "List Assignment: <positions> must be a dense list of positive integers",
               0L, 0L );
    }
    if ( libGAP_LEN_LIST( poss ) != libGAP_LEN_LIST( rhss ) ) {
        libGAP_ErrorQuit(
     "List Assignment: <rhss> must have the same length as <positions> (%d)",
            (libGAP_Int)libGAP_LEN_LIST(poss), 0L );
    }

    /* get the list (checking is done by 'ASSS_LIST')                      */
    list = libGAP_PopObj();

    /* assign to several elements of the list                              */
    libGAP_ASSS_LIST( list, poss, rhss );

    /* push the right hand sides again                                     */
    libGAP_PushObj( rhss );
}

void            libGAP_IntrAssListLevel (
				  libGAP_Int narg,
				  libGAP_UInt                level )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Obj                 rhss;           /* right hand sides, right operand */
    libGAP_Obj ixs;
    libGAP_Int i;
    
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssListLevel( narg, level ); return; }

    /* get right hand sides (checking is done by 'AssListLevel')           */
    rhss = libGAP_PopObj();

    ixs = libGAP_NEW_PLIST(libGAP_T_PLIST, narg);
    for (i = narg; i > 0; i--) {
      /* get and check the position                                          */
      pos = libGAP_PopObj();
      libGAP_SET_ELM_PLIST(ixs, i, pos);
      libGAP_CHANGED_BAG(ixs);
    }
    libGAP_SET_LEN_PLIST(ixs, narg);

    /* get lists (if this works, then <lists> is nested <level> deep,      */
    /* checking it is nested <level>+1 deep is done by 'AssListLevel')     */
    lists = libGAP_PopObj();

    /* assign the right hand sides to the elements of several lists        */
    libGAP_AssListLevel( lists, ixs, rhss, level );

    /* push the assigned values again                                      */
    libGAP_PushObj( rhss );
}

void            libGAP_IntrAsssListLevel (
    libGAP_UInt                level )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 poss;           /* position, left operand          */
    libGAP_Obj                 rhss;           /* right hand sides, right operand */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAsssListLevel( level ); return; }


    /* get right hand sides (checking is done by 'AsssListLevel')          */
    rhss = libGAP_PopObj();

    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
    "List Assignment: <positions> must be a dense list of positive integers",
            0L, 0L );
    }

    /* get lists (if this works, then <lists> is nested <level> deep,      */
    /* checking it is nested <level>+1 deep is done by 'AsssListLevel')    */
    lists = libGAP_PopObj();

    /* assign the right hand sides to several elements of several lists    */
    libGAP_AsssListLevel( lists, poss, rhss, level );

    /* push the assigned values again                                      */
    libGAP_PushObj( rhss );
}

void            libGAP_IntrUnbList ( libGAP_Int narg )
{
    libGAP_Obj                 list;           /* list                            */
    libGAP_Obj                 pos;            /* position                        */
    libGAP_Obj                 ixs;
    libGAP_Int                 i;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeUnbList( narg); return; }

    if (narg == 1) {
      /* get and check the position                                          */
      pos = libGAP_PopObj();
      
      /* get the list (checking is done by 'UNB_LIST' or 'UNBB_LIST')        */
      list = libGAP_PopObj();

      /* unbind the element                                                  */
      if (libGAP_IS_POS_INTOBJ(pos)) {
        libGAP_UNB_LIST( list, libGAP_INT_INTOBJ(pos) );
      } else {
        libGAP_UNBB_LIST(list, pos);
      }
    } else {
      ixs = libGAP_NEW_PLIST(libGAP_T_PLIST,narg);
      for (i = narg; i > 0; i--) {
	pos = libGAP_PopObj();
	libGAP_SET_ELM_PLIST(ixs, i, pos);
	libGAP_CHANGED_BAG(ixs);
      }
      libGAP_SET_LEN_PLIST(ixs, narg);
      list = libGAP_PopObj();
      libGAP_UNBB_LIST(list, ixs);
    }

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrElmList() . . . . . . . . . . . . . . . interpret selection of a list
*F  IntrElmsList()  . . . . . . . . .  interpret multiple selection of a list
*F  IntrElmListLevel(<level>) . . . . .  interpret selection of several lists
*F  IntrElmsListLevel(<level>)  . .  intr multiple selection of several lists
*/
void            libGAP_IntrElmList ( libGAP_Int narg )
{
  libGAP_Obj                 elm = (libGAP_Obj) 0;            /* element, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Int                 p;              /* position, as C integer          */
    libGAP_Int                 i;
    libGAP_Obj                 ixs;
    libGAP_Obj                 pos1;
    libGAP_Obj                 pos2;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmList( narg ); return; }

    if (narg <= 0)
      libGAP_SyntaxError("This should never happen");

    if (narg == 1) {
      /* get  the position                                                   */
      pos = libGAP_PopObj();
      /* get the list (checking is done by 'ELM_LIST')                       */
      list = libGAP_PopObj();
      
      
      if ( ! libGAP_IS_INTOBJ(pos)  || (p = libGAP_INT_INTOBJ(pos)) <= 0) {
        /* This mostly dispatches to the library */
        elm = libGAP_ELMB_LIST( list, pos);
      } else {
        /* get the element of the list                                         */
        elm = libGAP_ELM_LIST( list, p );
      }
    }
    if (narg == 2) {
      pos2 = libGAP_PopObj();
      pos1 = libGAP_PopObj();
      list = libGAP_PopObj();
      /* leave open space for a fastpath for 2 */
      elm = libGAP_ELM2_LIST(list, pos1, pos2);
    }
    
    if (narg > 2) {
      ixs = libGAP_NEW_PLIST(libGAP_T_PLIST,narg);
      for (i = narg; i > 0; i--) {
	libGAP_SET_ELM_PLIST(ixs,i,libGAP_PopObj());
	libGAP_CHANGED_BAG(ixs);
      }
      libGAP_SET_LEN_PLIST(ixs, narg);
      list = libGAP_PopObj();
      elm = libGAP_ELMB_LIST(list, ixs);
    }
      
    /* push the element                                                    */
    libGAP_PushObj( elm );
}

void            libGAP_IntrElmsList ( void )
{
    libGAP_Obj                 elms;           /* elements, result                */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 poss;           /* positions, right operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmsList(); return; }


    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
      "List Elements: <positions> must be a dense list of positive integers",
            0L, 0L );
    }

    /* get the list (checking is done by 'ELMS_LIST')                      */
    list = libGAP_PopObj();

    /* select several elements from the list                               */
    elms = libGAP_ELMS_LIST( list, poss );

    /* push the elements                                                   */
    libGAP_PushObj( elms );
}

void            libGAP_IntrElmListLevel ( libGAP_Int narg,
    libGAP_UInt                level )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Obj ixs;
    libGAP_Int i;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmListLevel( narg, level ); return; }

    /* get the positions */
    ixs = libGAP_NEW_PLIST(libGAP_T_PLIST, narg);
    for (i = narg; i > 0; i--) {
      pos = libGAP_PopObj();
      libGAP_SET_ELM_PLIST(ixs,i,pos);
      libGAP_CHANGED_BAG(ixs);
    }
    libGAP_SET_LEN_PLIST(ixs, narg);
      
    /* /\* get and check the position                                          *\/ */
    /* pos = PopObj(); */
    /* if ( TNUM_OBJ(pos) != T_INTPOS && (! IS_POS_INTOBJ(pos) )) { */
    /*     ErrorQuit( */
    /*         "List Element: <position> must be a positive integer (not a %s)", */
    /*         (Int)TNAM_OBJ(pos), 0L ); */
    /* } */

    /* get lists (if this works, then <lists> is nested <level> deep,      */
    /* checking it is nested <level>+1 deep is done by 'ElmListLevel')     */
    lists = libGAP_PopObj();

    /* select the elements from several lists (store them in <lists>)      */
    libGAP_ElmListLevel( lists, ixs, level );

    /* push the elements                                                   */
    libGAP_PushObj( lists );
}

void            libGAP_IntrElmsListLevel (
    libGAP_UInt                level )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 poss;           /* positions, right operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmsListLevel( level ); return; }


    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
      "List Elements: <positions> must be a dense list of positive integers",
            0L, 0L );
    }

    /* get lists (if this works, then <lists> is nested <level> deep,      */
    /* checking it is nested <level>+1 deep is done by 'ElmsListLevel')    */
    lists = libGAP_PopObj();

    /* select several elements from several lists (store them in <lists>)  */
    libGAP_ElmsListLevel( lists, poss, level );

    /* push the elements                                                   */
    libGAP_PushObj( lists );
}

void            libGAP_IntrIsbList ( libGAP_Int narg )
{
    libGAP_Obj                 isb;            /* isbound, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Obj ixs;
    libGAP_Int i;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIsbList(narg); return; }

    if (narg == 1) {
      /* get and check the position                                          */
      pos = libGAP_PopObj();
      
      /* get the list (checking is done by 'ISB_LIST' or 'ISBB_LIST')        */
      list = libGAP_PopObj();
      
      /* get the result                                                      */
      if (libGAP_IS_POS_INTOBJ(pos)) {
        isb = libGAP_ISB_LIST( list, libGAP_INT_INTOBJ(pos) ) ? libGAP_True : libGAP_False;
      } else {
        isb = libGAP_ISBB_LIST( list, pos) ? libGAP_True : libGAP_False;
      }
    } else {
      ixs = libGAP_NEW_PLIST(libGAP_T_PLIST,narg);
      for (i = narg; i > 0; i--) {
	pos = libGAP_PopObj();
	libGAP_SET_ELM_PLIST(ixs, i, pos);
	libGAP_CHANGED_BAG(ixs);
      }
      libGAP_SET_LEN_PLIST(ixs, narg);
      list = libGAP_PopObj();
      isb = libGAP_ISBB_LIST(list, ixs) ? libGAP_True: libGAP_False;
    }
      
      
    /* push the result                                                     */
    libGAP_PushObj( isb );
}


/****************************************************************************
**
*F  IntrAssRecName(<rnam>)  . . . . . . . .  interpret assignment to a record
*F  IntrAssRecExpr()  . . . . . . . . . . .  interpret assignment to a record
*/
void            libGAP_IntrAssRecName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssRecName( rnam ); return; }


    /* get the right hand side                                             */
    rhs = libGAP_PopObj();

    /* get the record (checking is done by 'ASS_REC')                      */
    record = libGAP_PopObj();

    /* assign the right hand side to the element of the record             */
    libGAP_ASS_REC( record, rnam, rhs );

    /* push the assigned value                                             */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrAssRecExpr ( void )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssRecExpr(); return; }


    /* get the right hand side                                             */
    rhs = libGAP_PopObj();

    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'ASS_REC')                      */
    record = libGAP_PopObj();

    /* assign the right hand side to the element of the record             */
    libGAP_ASS_REC( record, rnam, rhs );

    /* push the assigned value                                             */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrUnbRecName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 record;         /* record, left operand            */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeUnbRecName( rnam ); return; }


    /* get the record (checking is done by 'UNB_REC')                      */
    record = libGAP_PopObj();

    /* assign the right hand side to the element of the record             */
    libGAP_UNB_REC( record, rnam );

    /* push void                                                           */
    libGAP_PushVoidObj();
}

void            libGAP_IntrUnbRecExpr ( void )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeUnbRecExpr(); return; }


    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'UNB_REC')                      */
    record = libGAP_PopObj();

    /* assign the right hand side to the element of the record             */
    libGAP_UNB_REC( record, rnam );

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrElmRecName(<rnam>)  . . . . . . . . . interpret selection of a record
*F  IntrElmRecExpr()  . . . . . . . . . . . . interpret selection of a record
*/
void            libGAP_IntrElmRecName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmRecName( rnam ); return; }


    /* get the record (checking is done by 'ELM_REC')                      */
    record = libGAP_PopObj();

    /* select the element of the record                                    */
    elm = libGAP_ELM_REC( record, rnam );

    /* push the element                                                    */
    libGAP_PushObj( elm );
}

void            libGAP_IntrElmRecExpr ( void )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmRecExpr(); return; }


    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'ELM_REC')                      */
    record = libGAP_PopObj();

    /* select the element of the record                                    */
    elm = libGAP_ELM_REC( record, rnam );

    /* push the element                                                    */
    libGAP_PushObj( elm );
}

void            libGAP_IntrIsbRecName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 isb;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIsbRecName( rnam ); return; }


    /* get the record (checking is done by 'ISB_REC')                      */
    record = libGAP_PopObj();

    /* get the result                                                      */
    isb = (libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False);

    /* push the result                                                     */
    libGAP_PushObj( isb );
}

void            libGAP_IntrIsbRecExpr ( void )
{
    libGAP_Obj                 isb;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIsbRecExpr(); return; }


    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'ISB_REC')                      */
    record = libGAP_PopObj();

    /* get the result                                                      */
    isb = (libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False);

    /* push the result                                                     */
    libGAP_PushObj( isb );
}


/****************************************************************************
**
*F  IntrAssPosObj() . . . . . . . . . . . . .  interpret assignment to a list
*F  IntrAsssPosObj()  . . . . . . . . interpret multiple assignment to a list
*F  IntrAssPosObjLevel(<level>) . . . . interpret assignment to several lists
*F  IntrAsssPosObjLevel(<level>)  . intr multiple assignment to several lists
*/
void            libGAP_IntrAssPosObj ( void )
{
    libGAP_Obj                 list;           /* list                            */
    libGAP_Obj                 pos;            /* position                        */
    libGAP_Int                 p;              /* position, as a C integer        */
    libGAP_Obj                 rhs;            /* right hand side                 */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssPosObj(); return; }


    /* get the right hand side                                             */
    rhs = libGAP_PopObj();

    /* get and check the position                                          */
    pos = libGAP_PopObj();
    if ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        libGAP_ErrorQuit(
         "PosObj Assignment: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L );
    }
    p = libGAP_INT_INTOBJ(pos);

    /* get the list (checking is done by 'ASS_LIST')                       */
    list = libGAP_PopObj();

    /* assign to the element of the list                                   */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        if ( libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj) - 1 < p ) {
            libGAP_ResizeBag( list, (p+1) * sizeof(libGAP_Obj) );
        }
        libGAP_SET_ELM_PLIST( list, p, rhs );
        libGAP_CHANGED_BAG( list );
    }
    else {
        libGAP_ASS_LIST( list, p, rhs );
    }

    /* push the right hand side again                                      */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrAsssPosObj ( void )
{
    libGAP_Obj                 list;           /* list                            */
    libGAP_Obj                 poss;           /* positions                       */
    libGAP_Obj                 rhss;           /* right hand sides                */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAsssPosObj(); return; }


    /* get the right hand sides                                            */
    rhss = libGAP_PopObj();
    if ( ! libGAP_IS_DENSE_LIST( rhss ) ) {
        libGAP_ErrorQuit(
            "PosObj Assignment: <rhss> must be a dense list",
            0L, 0L );
    }

    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
    "PosObj Assignment: <positions> must be a dense list of positive integers",
               0L, 0L );
    }
    if ( libGAP_LEN_LIST( poss ) != libGAP_LEN_LIST( rhss ) ) {
        libGAP_ErrorQuit(
     "PosObj Assignment: <rhss> must have the same length as <positions> (%d)",
            (libGAP_Int)libGAP_LEN_LIST(poss), 0L );
    }

    /* get the list (checking is done by 'ASSS_LIST')                      */
    list = libGAP_PopObj();

    /* assign to several elements of the list                              */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        libGAP_ErrorQuit( "sorry: <posobj>!{<poss>} not yet implemented", 0L, 0L );
    }
    else {
        libGAP_ASSS_LIST( list, poss, rhss );
    }

    /* push the right hand sides again                                     */
    libGAP_PushObj( rhss );
}

void            libGAP_IntrAssPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Obj                 pos;            /* position, left operand          */
    libGAP_Obj                 rhss;           /* right hand sides, right operand */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssPosObjLevel( level ); return; }


    /* get right hand sides (checking is done by 'AssPosObjLevel')           */
    rhss = libGAP_PopObj();

    /* get and check the position                                          */
    pos = libGAP_PopObj();
    if ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        libGAP_ErrorQuit(
         "PosObj Assignment: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L );
    }

    /* assign the right hand sides to the elements of several lists        */
    libGAP_ErrorQuit(
        "sorry: <lists>{<poss>}![<pos>] not yet implemented",
        0L, 0L );

    /* push the assigned values again                                      */
    libGAP_PushObj( rhss );
}

void            libGAP_IntrAsssPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Obj                 poss;           /* position, left operand          */
    libGAP_Obj                 rhss;           /* right hand sides, right operand */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAsssPosObjLevel( level ); return; }


    /* get right hand sides (checking is done by 'AsssPosObjLevel')          */
    rhss = libGAP_PopObj();

    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
    "PosObj Assignment: <positions> must be a dense list of positive integers",
            0L, 0L );
    }

    /* assign the right hand sides to several elements of several lists    */
    libGAP_ErrorQuit(
        "sorry: <lists>{<poss>}!{<poss>} not yet implemented",
        0L, 0L );

    /* push the assigned values again                                      */
    libGAP_PushObj( rhss );
}

void            libGAP_IntrUnbPosObj ( void )
{
    libGAP_Obj                 list;           /* list                            */
    libGAP_Obj                 pos;            /* position                        */
    libGAP_Int                 p;              /* position, as a C integer        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeUnbPosObj(); return; }


    /* get and check the position                                          */
    pos = libGAP_PopObj();
    if ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        libGAP_ErrorQuit(
         "PosObj Assignment: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L );
    }
    p = libGAP_INT_INTOBJ(pos);

    /* get the list (checking is done by 'UNB_LIST')                       */
    list = libGAP_PopObj();

    /* unbind the element                                                  */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        if ( p <= libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1 ) {
            libGAP_SET_ELM_PLIST( list, p, 0 );
        }
    }
    else {
        libGAP_UNB_LIST( list, p );
    }

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrElmPosObj() . . . . . . . . . . . . . . interpret selection of a list
*F  IntrElmsPosObj()  . . . . . . . .  interpret multiple selection of a list
*F  IntrElmPosObjLevel(<level>) . . . .  interpret selection of several lists
*F  IntrElmsPosObjLevel(<level>)  .  intr multiple selection of several lists
*/
void            libGAP_IntrElmPosObj ( void )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Int                 p;              /* position, as C integer          */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmPosObj(); return; }


    /* get and check the position                                          */
    pos = libGAP_PopObj();
    if ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        libGAP_ErrorQuit(
            "PosObj Element: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L );
    }
    p = libGAP_INT_INTOBJ( pos );

    /* get the list (checking is done by 'ELM_LIST')                       */
    list = libGAP_PopObj();

    /* get the element of the list                                         */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        if ( libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1 < p ) {
            libGAP_ErrorQuit(
                "PosObj Element: <posobj>![%d] must have an assigned value",
                (libGAP_Int)p, 0L );
        }
        elm = libGAP_ELM_PLIST( list, p );
        if ( elm == 0 ) {
            libGAP_ErrorQuit(
                "PosObj Element: <posobj>![%d] must have an assigned value",
                (libGAP_Int)p, 0L );
        }
    }
    else {
        elm = libGAP_ELM_LIST( list, p );
    }

    /* push the element                                                    */
    libGAP_PushObj( elm );
}

void            libGAP_IntrElmsPosObj ( void )
{
    libGAP_Obj                 elms;           /* elements, result                */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 poss;           /* positions, right operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmsPosObj(); return; }


    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
      "PosObj Elements: <positions> must be a dense list of positive integers",
            0L, 0L );
    }

    /* get the list (checking is done by 'ELMS_LIST')                      */
    list = libGAP_PopObj();

    /* select several elements from the list                               */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        elms = 0;
        libGAP_ErrorQuit( "sorry: <posobj>!{<poss>} not yet implemented", 0L, 0L );
    }
    else {
        elms = libGAP_ELMS_LIST( list, poss );
    }

    /* push the elements                                                   */
    libGAP_PushObj( elms );
}

void            libGAP_IntrElmPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 pos;            /* position, right operand         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmPosObjLevel( level ); return; }


    /* get and check the position                                          */
    pos = libGAP_PopObj();
    if ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        libGAP_ErrorQuit(
            "PosObj Element: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L );
    }

    /* get lists (if this works, then <lists> is nested <level> deep,      */
    /* checking it is nested <level>+1 deep is done by 'ElmPosObjLevel')     */
    lists = libGAP_PopObj();

    /* select the elements from several lists (store them in <lists>)      */
    libGAP_ErrorQuit(
        "sorry: <lists>{<poss>}![<pos>] not yet implemented",
        0L, 0L );

    /* push the elements                                                   */
    libGAP_PushObj( lists );
}

void            libGAP_IntrElmsPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Obj                 lists;          /* lists, left operand             */
    libGAP_Obj                 poss;           /* positions, right operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmsPosObjLevel( level ); return; }


    /* get and check the positions                                         */
    poss = libGAP_PopObj();
    if ( ! libGAP_IS_POSS_LIST( poss ) ) {
        libGAP_ErrorQuit(
      "PosObj Elements: <positions> must be a dense list of positive integers",
            0L, 0L );
    }

    /* get lists (if this works, then <lists> is nested <level> deep,      */
    /* checking it is nested <level>+1 deep is done by 'ElmsPosObjLevel')    */
    lists = libGAP_PopObj();

    /* select several elements from several lists (store them in <lists>)  */
    libGAP_ErrorQuit(
        "sorry: <lists>{<poss>}!{<poss>} not yet implemented",
        0L, 0L );

    /* push the elements                                                   */
    libGAP_PushObj( lists );
}

void            libGAP_IntrIsbPosObj ( void )
{
    libGAP_Obj                 isb;            /* isbound, result                 */
    libGAP_Obj                 list;           /* list, left operand              */
    libGAP_Obj                 pos;            /* position, right operand         */
    libGAP_Int                 p;              /* position, as C integer          */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIsbPosObj(); return; }


    /* get and check the position                                          */
    pos = libGAP_PopObj();
    if ( ! libGAP_IS_POS_INTOBJ(pos) ) {
        libGAP_ErrorQuit(
            "PosObj Element: <position> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L );
    }
    p = libGAP_INT_INTOBJ( pos );

    /* get the list (checking is done by 'ISB_LIST')                       */
    list = libGAP_PopObj();

    /* get the result                                                      */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        isb = (p <= libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1 && libGAP_ELM_PLIST(list,p) != 0 ?
               libGAP_True : libGAP_False);
    }
    else {
        isb = (libGAP_ISB_LIST( list, p ) ? libGAP_True : libGAP_False);
    }

    /* push the result                                                     */
    libGAP_PushObj( isb );
}


/****************************************************************************
**
*F  IntrAssComObjName(<rnam>) . . . . . . .  interpret assignment to a record
*F  IntrAssComObjExpr() . . . . . . . . . .  interpret assignment to a record
*/
void            libGAP_IntrAssComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssComObjName( rnam ); return; }


    /* get the right hand side                                             */
    rhs = libGAP_PopObj();

    /* get the record (checking is done by 'ASS_REC')                      */
    record = libGAP_PopObj();

    /* assign the right hand side to the element of the record             */
    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        libGAP_AssPRec( record, rnam, rhs );
        break;
      default:
        libGAP_ASS_REC( record, rnam, rhs );
        break;
    }

    /* push the assigned value                                             */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrAssComObjExpr ( void )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_Obj                 rhs;            /* rhs, right operand              */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssComObjExpr(); return; }


    /* get the right hand side                                             */
    rhs = libGAP_PopObj();

    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'ASS_REC')                      */
    record = libGAP_PopObj();

    /* assign the right hand side to the element of the record             */
    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        libGAP_AssPRec( record, rnam, rhs );
        break;
      default:
        libGAP_ASS_REC( record, rnam, rhs );
        break;
    }

    /* push the assigned value                                             */
    libGAP_PushObj( rhs );
}

void            libGAP_IntrUnbComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 record;         /* record, left operand            */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeUnbComObjName( rnam ); return; }


    /* get the record (checking is done by 'UNB_REC')                      */
    record = libGAP_PopObj();

    /* unbind the element of the record             			   */
    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        libGAP_UnbPRec( record, rnam );
        break;
      default:
        libGAP_UNB_REC( record, rnam );
        break;
    }

    /* push void                                                           */
    libGAP_PushVoidObj();
}

void            libGAP_IntrUnbComObjExpr ( void )
{
    libGAP_Obj                 record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeUnbComObjExpr(); return; }


    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'UNB_REC')                      */
    record = libGAP_PopObj();

    /* unbind the element of the record             			   */
    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        libGAP_UnbPRec( record, rnam );
        break;
      default:
        libGAP_UNB_REC( record, rnam );
        break;
    }

    /* push void                                                           */
    libGAP_PushVoidObj();
}


/****************************************************************************
**
*F  IntrElmComObjName(<rnam>) . . . . . . . . interpret selection of a record
*F  IntrElmComObjExpr() . . . . . . . . . . . interpret selection of a record
*/
void            libGAP_IntrElmComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmComObjName( rnam ); return; }


    /* get the record (checking is done by 'ELM_REC')                      */
    record = libGAP_PopObj();

    /* select the element of the record                                    */

    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        elm = libGAP_ElmPRec( record, rnam );
        break;
      default:
        elm = libGAP_ELM_REC( record, rnam );
        break;
    }

    /* push the element                                                    */
    libGAP_PushObj( elm );
}

void            libGAP_IntrElmComObjExpr ( void )
{
    libGAP_Obj                 elm;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeElmComObjExpr(); return; }


    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'ELM_REC')                      */
    record = libGAP_PopObj();

    /* select the element of the record                                    */
    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        elm = libGAP_ElmPRec( record, rnam );
        break;
      default:
        elm = libGAP_ELM_REC( record, rnam );
        break;
    }

    /* push the element                                                    */
    libGAP_PushObj( elm );
}

void            libGAP_IntrIsbComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Obj                 isb;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIsbComObjName( rnam ); return; }


    /* get the record (checking is done by 'ISB_REC')                      */
    record = libGAP_PopObj();

    /* get the result                                                      */
    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        isb = libGAP_IsbPRec( record, rnam ) ? libGAP_True : libGAP_False;
        break;
      default:
        isb = libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False;
        break;
    }

    /* push the result                                                     */
    libGAP_PushObj( isb );
}

void            libGAP_IntrIsbComObjExpr ( void )
{
    libGAP_Obj                 isb;            /* element, result                 */
    libGAP_Obj                 record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeIsbComObjExpr(); return; }


    /* get the name and convert it to a record name                        */
    rnam = libGAP_RNamObj( libGAP_PopObj() );

    /* get the record (checking is done by 'ISB_REC')                      */
    record = libGAP_PopObj();

    /* get the result                                                      */
    switch (libGAP_TNUM_OBJ(record)) {
      case libGAP_T_COMOBJ:
        isb = libGAP_IsbPRec( record, rnam ) ? libGAP_True : libGAP_False;
        break;
      default:
        isb = libGAP_ISB_REC( record, rnam ) ? libGAP_True : libGAP_False;
        break;
    }

    /* push the result                                                     */
    libGAP_PushObj( isb );
}

/****************************************************************************
**
*F  IntrEmpty() . . . . . . . . . . . . .  Interpret an empty statement body
**
*/

void             libGAP_IntrEmpty ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeEmpty(); return; }


    /* interpret */
    libGAP_PushVoidObj();
    return;

}


/****************************************************************************
**
*F  IntrInfoBegin() . . . . . . . . .  start interpretation of Info statement
*F  IntrInfoMiddle()  . . . . . .  shift to interpreting printable statements
*F  IntrInfoEnd( <narg> ) . . Info statement complete, <narg> things to print
*V  InfoDecision . . . . . . . . . . .  fopy of the InfoDecision GAP function
*V  InfoDoPrint  . . . . . . . . . . . . fopy of the InfoDoPrint GAP function
**
**  These are the actions which are used to interpret an Info statement:
**
**  IntrInfoBegin is called after the Info is read
**
**  IntrInfoMiddle is called after reading two arguments, because we can
**  now decide whether we should evaluate or ignore the remaining arguments
**
**  IntrInfoEnd is called when the closing ')' is detected and should
**  trigger the actual printing, if needed. The argument is the number of
**  things to print
*/


void            libGAP_IntrInfoBegin( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeInfoBegin(); return; }

}

libGAP_Obj             libGAP_InfoDecision;

void            libGAP_IntrInfoMiddle( void )
{

    libGAP_Obj selectors;   /* first argument of Info */
    libGAP_Obj level;       /* second argument of Info */
    libGAP_Obj selected;    /* GAP Boolean answer to whether this message
                        gets printed or not */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeInfoMiddle(); return; }


    level = libGAP_PopObj();
    selectors = libGAP_PopObj();
    selected = libGAP_CALL_2ARGS( libGAP_InfoDecision, selectors, level);
    if (selected == libGAP_False)
      libGAP_TLS(libGAP_IntrIgnoring) = 1;
    return;
}

libGAP_Obj             libGAP_InfoDoPrint;

void            libGAP_IntrInfoEnd( libGAP_UInt narg )
{

     libGAP_Obj args;    /* gathers up the arguments to be printed */

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeInfoEnd( narg ); return; }


    /* print if necessary                                                  */
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 )
      libGAP_TLS(libGAP_IntrIgnoring)--;
    else
      {
        args = libGAP_NEW_PLIST( libGAP_T_PLIST, narg);
        libGAP_SET_LEN_PLIST(args, narg);

        while (narg > 0)
          libGAP_SET_ELM_PLIST(args, narg--, libGAP_PopObj());

        libGAP_CALL_1ARGS(libGAP_InfoDoPrint, args);
      }

    /* If we actually executed this statement at all
       (even if we printed nothing) then return a Void */
    if (libGAP_TLS(libGAP_IntrIgnoring) == 0)
      libGAP_PushVoidObj();
    return;
}


/****************************************************************************
**
*F  IntrAssertBegin()  . . . . . . . start interpretation of Assert statement
*F  IntrAssertAfterLevel() . .  called after the first argument has been read
**
**  At this stage, we can decide whether to evaluate the second argument --
**   the check in question
**
*F  IntrAssertAfterCondition() called after the second argument has been read
**
**  At this point we know whether there is an assertion failure. We still need
**  to read the third argument if any, to decide what to do about it One of:
**
*F  IntrAssertEnd2Args() . . . . called after reading the closing parenthesis
*F  IntrAssertEnd3Args() . . . . called after reading the closing parenthesis
**
*V  CurrentAssertionLevel  . .  . . . . . . . . . . . .  copy of GAP variable
**
**
**  TLS(IntrIgnoring) is increased by (a total of) 2 if an assertion either is not
**  tested (because we were Ignoring when we got to it, or due to level)
**  or is tested and passes
*/

libGAP_Obj              libGAP_CurrentAssertionLevel;

void              libGAP_IntrAssertBegin ( void )
{
    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssertBegin(); return; }

}


void             libGAP_IntrAssertAfterLevel ( void )
{
  libGAP_Obj level;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssertAfterLevel(); return; }


    level = libGAP_PopObj();

    if (libGAP_LT( libGAP_CurrentAssertionLevel, level))
           libGAP_TLS(libGAP_IntrIgnoring) = 1;
}

void             libGAP_IntrAssertAfterCondition ( void )
{
  libGAP_Obj condition;

    /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrIgnoring)  > 0 ) { libGAP_TLS(libGAP_IntrIgnoring)++; return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssertAfterCondition(); return; }


    condition = libGAP_PopObj();

    if (condition == libGAP_True)
      libGAP_TLS(libGAP_IntrIgnoring)= 2;
    else if (condition != libGAP_False)
        libGAP_ErrorQuit(
            "<condition> in Assert must yield 'true' or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(condition), 0L );
}

void             libGAP_IntrAssertEnd2Args ( void )
{
      /* ignore or code                                                      */
    if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
    if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssertEnd2Args(); return; }


    if ( libGAP_TLS(libGAP_IntrIgnoring)  == 0 )
      libGAP_ErrorQuit("Assertion Failure", 0, 0);
    else
      libGAP_TLS(libGAP_IntrIgnoring) -= 2;

    if (libGAP_TLS(libGAP_IntrIgnoring) == 0)
      libGAP_PushVoidObj();
    return;
}


void             libGAP_IntrAssertEnd3Args ( void )
{
  libGAP_Obj message;
  /* ignore or code                                                      */
  if ( libGAP_TLS(libGAP_IntrReturning) > 0 ) { return; }
  if ( libGAP_TLS(libGAP_IntrCoding)    > 0 ) { libGAP_CodeAssertEnd3Args(); return; }


  if ( libGAP_TLS(libGAP_IntrIgnoring)  == 0 ) {
      message = libGAP_PopVoidObj();
      if (message != (libGAP_Obj) 0 ) {
          if (libGAP_IS_STRING_REP( message ))
            libGAP_PrintString1(message);
          else
            libGAP_PrintObj(message);
      }
  } else
      libGAP_TLS(libGAP_IntrIgnoring) -= 2;

  if (libGAP_TLS(libGAP_IntrIgnoring) == 0)
      libGAP_PushVoidObj();
  return;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_InitGlobalBag( &libGAP_IntrResult, "src/intrprtr.c:IntrResult" );
    libGAP_InitGlobalBag( &libGAP_IntrState,  "src/intrprtr.c:IntrState"  );
    libGAP_InitGlobalBag( &libGAP_StackObj,   "src/intrprtr.c:StackObj"   );
    libGAP_InitCopyGVar( "CurrentAssertionLevel", &libGAP_CurrentAssertionLevel );
    libGAP_InitFopyGVar( "CONVERT_FLOAT_LITERAL_EAGER", &libGAP_CONVERT_FLOAT_LITERAL_EAGER);

    /* The work of handling Info messages is delegated to the GAP level */
    libGAP_ImportFuncFromLibrary( "InfoDecision", &libGAP_InfoDecision );
    libGAP_ImportFuncFromLibrary( "InfoDoPrint",  &libGAP_InfoDoPrint  );

    /* The work of handling Options is also delegated*/
    libGAP_ImportFuncFromLibrary( "PushOptions", &libGAP_PushOptions );
    libGAP_ImportFuncFromLibrary( "PopOptions",  &libGAP_PopOptions  );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt            lev;

    /* The Assertion level is also controlled at GAP level                 */
    lev = libGAP_GVarName("CurrentAssertionLevel");
    libGAP_AssGVar( lev, libGAP_INTOBJ_INT(0) );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoIntrprtr()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "intrprtr",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoIntrprtr ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  intrprtr.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
