/****************************************************************************
**
*W  gvars.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the global variables package.
**
**  The global variables  package  is the   part of the  kernel that  manages
**  global variables, i.e., the global namespace.  A global variable binds an
**  identifier to a value.
**
**  A global variable can be automatic.   That means that the global variable
**  binds the  identifier to a function and  an argument.   When the value of
**  the global variable is needed, the  function is called with the argument.
**  This function call  should, as a side-effect, execute  an assignment of a
**  value to the global variable, otherwise an error is signalled.
**
**  A global variable can have a number of internal copies, i.e., C variables
**  that always reference the same value as the global variable.
**  It can also have a special type of internal copy (a fopy) only used for
**  functions,  where  the internal copies
**  only reference the same value as the global variable if it is a function.
**  Otherwise the internal copies reference functions that signal an error.
*/
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */

#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "code.h"                /* coder                           */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "bool.h"                /* booleans                        */

#include        "tls.h"                 /* thread-local storage            */
#include        "thread.h"              /* threads                         */
#include        "aobjects.h"            /* atomic objects                  */

/****************************************************************************
**
*V  ValGVars  . . . . . . . . . . . . . . . . . .  values of global variables
*V  PtrGVars  . . . . . . . . . . . . . pointer to values of global variables
**
**  'ValGVars' is the bag containing the values of the global variables.
**
**  'PtrGVars' is a pointer  to the 'ValGVars'  bag.  This makes it faster to
**  access global variables.
**
**  Since a   garbage  collection may move   this  bag around,    the pointer
**  'PtrGVars' must be  revalculated afterwards.   This  should be done by  a
**  function in this package, but is still done in 'VarsAfterCollectBags'.
*/
libGAP_Obj   libGAP_ValGVars;

libGAP_Obj * libGAP_PtrGVars;


/****************************************************************************
**
*F  VAL_GVAR(<gvar>)  . . . . . . . . . . . . . . .  value of global variable
**
**  'VAL_GVAR' returns the  value of the global  variable  <gvar>.  If <gvar>
**  has no  assigned value, 'VAL_GVAR' returns 0.   In this case <gvar> might
**  be an automatic global variable, and one should call 'ValAutoGVar', which
**  will return the value of <gvar>  after evaluating <gvar>-s expression, or
**  0 if <gvar> was not an automatic variable.
**
**  'VAL_GVAR' is defined in the declaration part of this package as follows
**
#define VAL_GVAR(gvar)          PtrGVars[ (gvar) ]
*/


/****************************************************************************
**
*V  NameGVars . . . . . . . . . . . . . . . . . . . names of global variables
*V  WriteGVars  . . . . . . . . . . . . .  writable flags of global variables
*V  ExprGVars . . . . . . . . . .  expressions for automatic global variables
*V  CopiesGVars . . . . . . . . . . . . . internal copies of global variables
*V  FopiesGVars . . . . . . . .  internal function copies of global variables
*V  CountGVars  . . . . . . . . . . . . . . . . .  number of global variables
*/
libGAP_Obj             libGAP_NameGVars;
libGAP_Obj             libGAP_WriteGVars;
libGAP_Obj             libGAP_ExprGVars;
libGAP_Obj             libGAP_CopiesGVars;
libGAP_Obj             libGAP_FopiesGVars;
libGAP_UInt            libGAP_CountGVars;


/****************************************************************************
**
*V  TableGVars  . . . . . . . . . . . . . .  hashed table of global variables
*V  SizeGVars . . . . . . .  current size of hashed table of global variables
*/
libGAP_Obj             libGAP_TableGVars;
libGAP_UInt            libGAP_SizeGVars;


/****************************************************************************
**
*V  ErrorMustEvalToFuncFunc . . . . . . . . .  function that signals an error
*F  ErrorMustEvalToFuncHandler(<self>,<args>) . handler that signals an error
**
**  'ErrorMustEvalToFuncFunc' is a (variable number of  args)  function  that
**  signals the error ``Function: <func> be a function''.
**
**  'ErrorMustEvalToFuncHandler'  is  the  handler  that  signals  the  error
**  ``Function: <func> must be a function''.
*/
libGAP_Obj             libGAP_ErrorMustEvalToFuncFunc;

libGAP_Obj             libGAP_ErrorMustEvalToFuncHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit(
        "Function Calls: <func> must be a function",
        0L, 0L );
    return 0;
}


/****************************************************************************
**
*V  ErrorMustHaveAssObjFunc . . . . . . . . .  function that signals an error
*F  ErrorMustHaveAssObjHandler(<self>,<args>) . handler that signals an error
**
**  'ErrorMustHaveAssObjFunc' is a (variable number of  args)  function  that
**  signals the error ``Variable: <<unknown>> must have an assigned value''.
**
**  'ErrorMustHaveAssObjHandler'  is  the  handler  that  signals  the  error
**  ``Variable: <<unknown>> must have an assigned value''.
*/
libGAP_Obj             libGAP_ErrorMustHaveAssObjFunc;

libGAP_Obj             libGAP_ErrorMustHaveAssObjHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit(
        "Variable: <<unknown>> must have an assigned value",
        0L, 0L );
    return 0;
}


/****************************************************************************
**
*F  AssGVar(<gvar>,<val>) . . . . . . . . . . . . assign to a global variable
**
**  'AssGVar' assigns the value <val> to the global variable <gvar>.
*/

static libGAP_Obj libGAP_REREADING;                   /* Copy of GAP global variable REREADING */

void            libGAP_AssGVar (
    libGAP_UInt                gvar,
    libGAP_Obj                 val )
{
    libGAP_Obj                 cops;           /* list of internal copies         */
    libGAP_Obj *               copy;           /* one copy                        */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Char *              name;           /* name of a function              */
    libGAP_Obj                 onam;           /* object of <name>                */

    /* make certain that the variable is not read only                     */
    while ( (libGAP_REREADING != libGAP_True) &&
            (libGAP_ELM_PLIST( libGAP_WriteGVars, gvar ) == libGAP_INTOBJ_INT(0)) ) {
        libGAP_ErrorReturnVoid(
            "Variable: '%s' is read only",
            (libGAP_Int)libGAP_CSTR_STRING( libGAP_ELM_PLIST(libGAP_NameGVars,gvar) ), 0L,
            "you can 'return;' after making it writable" );
    }

    /* assign the value to the global variable                             */
    libGAP_VAL_GVAR(gvar) = val;
    libGAP_CHANGED_BAG( libGAP_ValGVars );

    /* if the global variable was automatic, convert it to normal          */
    libGAP_SET_ELM_PLIST( libGAP_ExprGVars, gvar, 0 );

    /* assign the value to all the internal copies                         */
    cops = libGAP_ELM_PLIST( libGAP_CopiesGVars, gvar );
    if ( cops != 0 ) {
        for ( i = 1; i <= libGAP_LEN_PLIST(cops); i++ ) {
            copy  = (libGAP_Obj*) libGAP_ELM_PLIST(cops,i);
            *copy = val;
        }
    }

    /* if the value is a function, assign it to all the internal fopies    */
    cops = libGAP_ELM_PLIST( libGAP_FopiesGVars, gvar );
    if ( cops != 0 && val != 0 && libGAP_TNUM_OBJ(val) == libGAP_T_FUNCTION ) {
        for ( i = 1; i <= libGAP_LEN_PLIST(cops); i++ ) {
            copy  = (libGAP_Obj*) libGAP_ELM_PLIST(cops,i);
            *copy = val;
        }
    }

    /* if the values is not a function, assign the error function          */
    else if ( cops != 0 && val != 0 /* && TNUM_OBJ(val) != T_FUNCTION */ ) {
        for ( i = 1; i <= libGAP_LEN_PLIST(cops); i++ ) {
            copy  = (libGAP_Obj*) libGAP_ELM_PLIST(cops,i);
            *copy = libGAP_ErrorMustEvalToFuncFunc;
        }
    }

    /* if this was an unbind, assign the other error function              */
    else if ( cops != 0 /* && val == 0 */ ) {
        for ( i = 1; i <= libGAP_LEN_PLIST(cops); i++ ) {
            copy  = (libGAP_Obj*) libGAP_ELM_PLIST(cops,i);
            *copy = libGAP_ErrorMustHaveAssObjFunc;
        }
    }

    /* assign name to a function                                           */
    if ( val != 0 && libGAP_TNUM_OBJ(val) == libGAP_T_FUNCTION && libGAP_NAME_FUNC(val) == 0 ) {
        name = libGAP_NameGVar(gvar);
        libGAP_C_NEW_STRING_DYN(onam, name);
        libGAP_RESET_FILT_LIST( onam, libGAP_FN_IS_MUTABLE );
        libGAP_NAME_FUNC(val) = onam;
        libGAP_CHANGED_BAG(val);
    }
}


/****************************************************************************
**
*F  ValAutoGVar(<gvar>) . . . . . . . .  value of a automatic global variable
**
**  'ValAutoGVar' returns the value of the global variable <gvar>.  This will
**  be 0 if  <gvar> has  no assigned value.    It will also cause a  function
**  call, if <gvar> is automatic.
*/
libGAP_Obj             libGAP_ValAutoGVar (
    libGAP_UInt                gvar )
{
    libGAP_Obj                 func;           /* function to call for automatic  */
    libGAP_Obj                 arg;            /* argument to pass for automatic  */

    /* if this is an automatic variable, make the function call            */
    if ( libGAP_VAL_GVAR(gvar) == 0 && libGAP_ELM_PLIST( libGAP_ExprGVars, gvar ) != 0 ) {

        /* make the function call                                          */
        func = libGAP_ELM_PLIST( libGAP_ELM_PLIST( libGAP_ExprGVars, gvar ), 1 );
        arg  = libGAP_ELM_PLIST( libGAP_ELM_PLIST( libGAP_ExprGVars, gvar ), 2 );
        libGAP_CALL_1ARGS( func, arg );

        /* if this is still an automatic variable, this is an error        */
        while ( libGAP_VAL_GVAR(gvar) == 0 ) {
            libGAP_ErrorReturnVoid(
       "Variable: automatic variable '%s' must get a value by function call",
                (libGAP_Int)libGAP_CSTR_STRING( libGAP_ELM_PLIST(libGAP_NameGVars,gvar) ), 0L,
                "you can 'return;' after assigning a value" );
        }

    }

    /* return the value                                                    */
    return libGAP_VAL_GVAR(gvar);
}


/****************************************************************************
**
*F  NameGVar(<gvar>)  . . . . . . . . . . . . . . . name of a global variable
**
**  'NameGVar' returns the name of the global variable <gvar> as a C string.
*/
libGAP_Char *          libGAP_NameGVar (
    libGAP_UInt                gvar )
{
    return libGAP_CSTR_STRING( libGAP_ELM_PLIST( libGAP_NameGVars, gvar ) );
}

libGAP_Obj libGAP_NameGVarObj ( libGAP_UInt gvar )
{
    return libGAP_ELM_PLIST( libGAP_NameGVars, gvar );
}

#define libGAP_NSCHAR '@'

libGAP_Obj libGAP_CurrNamespace = 0;

libGAP_Obj libGAP_FuncSET_NAMESPACE(libGAP_Obj self, libGAP_Obj str)
{
    libGAP_TLS(libGAP_CurrNamespace) = str;
    return 0;
}

libGAP_Obj libGAP_FuncGET_NAMESPACE(libGAP_Obj self)
{
    return libGAP_TLS(libGAP_CurrNamespace);
}

/****************************************************************************
**
*F  GVarName(<name>)  . . . . . . . . . . . . . .  global variable for a name
**
**  'GVarName' returns the global variable with the name <name>.
*/
libGAP_UInt libGAP_GVarName ( 
    const libGAP_Char *        name )
{
    libGAP_Obj                 gvar;           /* global variable (as imm intval) */
    libGAP_Char                gvarbuf[1024];  /* temporary copy for namespace    */
    libGAP_Char *              cns;            /* Pointer to current namespace    */
    libGAP_UInt                pos;            /* hash position                   */
    libGAP_Char                namx [1024];    /* temporary copy of <name>        */
    libGAP_Obj                 string;         /* temporary string value <name>   */
    libGAP_Obj                 table;          /* temporary copy of <TableGVars>  */
    libGAP_Obj                 gvar2;          /* one element of <table>          */
    const libGAP_Char *        p;              /* loop variable                   */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Int                 len;            /* length of name                  */

    /* First see whether it could be namespace-local: */
    cns = libGAP_CSTR_STRING(libGAP_TLS(libGAP_CurrNamespace));
    if (*cns) {   /* only if a namespace is set */
        len = strlen(name);
        if (name[len-1] == libGAP_NSCHAR) {
            libGAP_strlcpy(gvarbuf, name, 512);
            libGAP_strlcat(gvarbuf, cns, sizeof(gvarbuf));
            name = gvarbuf;
        }
    }

    /* start looking in the table at the following hash position           */
    pos = 0;
    for ( p = name; *p != '\0'; p++ ) {
        pos = 65599 * pos + *p;
    }
    pos = (pos % libGAP_SizeGVars) + 1;

    /* look through the table until we find a free slot or the global      */
    while ( (gvar = libGAP_ELM_PLIST( libGAP_TableGVars, pos )) != 0
         && strncmp( libGAP_NameGVar( libGAP_INT_INTOBJ(gvar) ), name, 1023 ) ) {
        pos = (pos % libGAP_SizeGVars) + 1;
    }

    /* if we did not find the global variable, make a new one and enter it */
    /* (copy the name first, to avoid a stale pointer in case of a GC)     */
    if ( gvar == 0 ) {
        libGAP_CountGVars++;
        gvar = libGAP_INTOBJ_INT(libGAP_CountGVars);
        libGAP_SET_ELM_PLIST( libGAP_TableGVars, pos, gvar );
        libGAP_strlcpy(namx, name, sizeof(namx));
        libGAP_C_NEW_STRING_DYN(string, namx);

        libGAP_RESET_FILT_LIST( string, libGAP_FN_IS_MUTABLE );
        libGAP_GROW_PLIST(    libGAP_ValGVars,    libGAP_CountGVars );
        libGAP_SET_LEN_PLIST( libGAP_ValGVars,    libGAP_CountGVars );
        libGAP_SET_ELM_PLIST( libGAP_ValGVars,    libGAP_CountGVars, 0 );
        libGAP_GROW_PLIST(    libGAP_NameGVars,   libGAP_CountGVars );
        libGAP_SET_LEN_PLIST( libGAP_NameGVars,   libGAP_CountGVars );
        libGAP_SET_ELM_PLIST( libGAP_NameGVars,   libGAP_CountGVars, string );
        libGAP_CHANGED_BAG(   libGAP_NameGVars );
        libGAP_GROW_PLIST(    libGAP_WriteGVars,  libGAP_CountGVars );
        libGAP_SET_LEN_PLIST( libGAP_WriteGVars,  libGAP_CountGVars );
        libGAP_SET_ELM_PLIST( libGAP_WriteGVars,  libGAP_CountGVars, libGAP_INTOBJ_INT(1) );
        libGAP_GROW_PLIST(    libGAP_ExprGVars,   libGAP_CountGVars );
        libGAP_SET_LEN_PLIST( libGAP_ExprGVars,   libGAP_CountGVars );
        libGAP_SET_ELM_PLIST( libGAP_ExprGVars,   libGAP_CountGVars, 0 );
        libGAP_GROW_PLIST(    libGAP_CopiesGVars, libGAP_CountGVars );
        libGAP_SET_LEN_PLIST( libGAP_CopiesGVars, libGAP_CountGVars );
        libGAP_SET_ELM_PLIST( libGAP_CopiesGVars, libGAP_CountGVars, 0 );
        libGAP_GROW_PLIST(    libGAP_FopiesGVars, libGAP_CountGVars );
        libGAP_SET_LEN_PLIST( libGAP_FopiesGVars, libGAP_CountGVars );
        libGAP_SET_ELM_PLIST( libGAP_FopiesGVars, libGAP_CountGVars, 0 );
        libGAP_PtrGVars = libGAP_ADDR_OBJ( libGAP_ValGVars );
    }

    /* if the table is too crowed, make a larger one, rehash the names     */
    if ( libGAP_SizeGVars < 3 * libGAP_CountGVars / 2 ) {
        table = libGAP_TableGVars;
        libGAP_SizeGVars = 2 * libGAP_SizeGVars + 1;
        libGAP_TableGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_SizeGVars );
        libGAP_SET_LEN_PLIST( libGAP_TableGVars, libGAP_SizeGVars );
        for ( i = 1; i <= (libGAP_SizeGVars-1)/2; i++ ) {
            gvar2 = libGAP_ELM_PLIST( table, i );
            if ( gvar2 == 0 )  continue;
            pos = 0;
            for ( p = libGAP_NameGVar( libGAP_INT_INTOBJ(gvar2) ); *p != '\0'; p++ ) {
                pos = 65599 * pos + *p;
            }
            pos = (pos % libGAP_SizeGVars) + 1;
            while ( libGAP_ELM_PLIST( libGAP_TableGVars, pos ) != 0 ) {
                pos = (pos % libGAP_SizeGVars) + 1;
            }
            libGAP_SET_ELM_PLIST( libGAP_TableGVars, pos, gvar2 );
        }
    }

    /* return the global variable                                          */
    return libGAP_INT_INTOBJ(gvar);
}

/****************************************************************************
**

*V  Tilde . . . . . . . . . . . . . . . . . . . . . . . . global variable '~'
**
**  'Tilde' is  the global variable '~', the  one used in expressions such as
**  '[ [ 1, 2 ], ~[1] ]'.
**
**  Actually  when such expressions  appear in functions, one should probably
**  use a local variable.  But for now this is good enough.
*/
libGAP_UInt libGAP_Tilde;


/****************************************************************************
**
*F  MakeReadOnlyGVar( <gvar> )  . . . . . .  make a global variable read only
*/
void libGAP_MakeReadOnlyGVar (
    libGAP_UInt                gvar )
{       
    libGAP_SET_ELM_PLIST( libGAP_WriteGVars, gvar, libGAP_INTOBJ_INT(0) );
    libGAP_CHANGED_BAG(libGAP_WriteGVars)
}


/****************************************************************************
**
*F  MakeReadOnlyGVarHandler(<self>,<name>)   make a global variable read only
**
**  'MakeReadOnlyGVarHandler' implements the function 'MakeReadOnlyGVar'.
**
**  'MakeReadOnlyGVar( <name> )'
**
**  'MakeReadOnlyGVar' make the global  variable with the name <name>  (which
**  must be a GAP string) read only.
*/
libGAP_Obj libGAP_MakeReadOnlyGVarHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{       
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( name ) ) {
        name = libGAP_ErrorReturnObj(
            "MakeReadOnlyGVar: <name> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(name), 0L,
            "you can return a string for <name>" );
    }

    /* get the variable and make it read only                              */
    libGAP_MakeReadOnlyGVar(libGAP_GVarName(libGAP_CSTR_STRING(name)));

    /* return void                                                         */
    return 0;
}


/****************************************************************************
**
*F  MakeReadWriteGVar( <gvar> ) . . . . . . make a global variable read write
*/
void libGAP_MakeReadWriteGVar (
    libGAP_UInt                gvar )
{
    libGAP_SET_ELM_PLIST( libGAP_WriteGVars, gvar, libGAP_INTOBJ_INT(1) );
    libGAP_CHANGED_BAG(libGAP_WriteGVars)
}


/****************************************************************************
**
*F  MakeReadWriteGVarHandler(<self>,<name>) make a global variable read write
**
**  'MakeReadWriteGVarHandler' implements the function 'MakeReadWriteGVar'.
**
**  'MakeReadWriteGVar( <name> )'
**
**  'MakeReadWriteGVar' make the global  variable with the name <name>  (which
**  must be a GAP string) read and writable.
*/
libGAP_Obj libGAP_MakeReadWriteGVarHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( name ) ) {
        name = libGAP_ErrorReturnObj(
            "MakeReadWriteGVar: <name> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(name), 0L,
            "you can return a string for <name>" );
    }

    /* get the variable and make it read write                             */
    libGAP_MakeReadWriteGVar(libGAP_GVarName(libGAP_CSTR_STRING(name)));

    /* return void                                                         */
    return 0;
}

/****************************************************************************
**
*F  IsReadOnlyGVar( <gvar> ) . . . . . . return status of a global variable
*/
libGAP_Int libGAP_IsReadOnlyGVar (
    libGAP_UInt                gvar )
{
  return !libGAP_INT_INTOBJ(libGAP_ELM_PLIST(libGAP_WriteGVars, gvar));
}


/****************************************************************************
**
*F  FuncIsReadOnlyGVar( <name> ) . . .handler for GAP function
**
*/

static libGAP_Obj libGAP_FuncIsReadOnlyGVar (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( name ) ) {
        name = libGAP_ErrorReturnObj(
            "IsReadOnlyGVar: <name> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(name), 0L,
            "you can return a string for <name>" );
    }

    /* get the answer                             */
    return libGAP_IsReadOnlyGVar(libGAP_GVarName(libGAP_CSTR_STRING(name))) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  AUTOHandler() . . . . . . . . . . . . .   make automatic global variables
**
**  'AUTOHandler' implements the internal function 'AUTO'.
**
**  'AUTO( <func>, <arg>, <name1>, ... )'
**
**  'AUTO' makes   the global variables,  whose  names are given  the strings
**  <name1>, <name2>, ..., automatic.  That means  that when the value of one
**  of  those global  variables  is requested,  then  the function  <func> is
**  called and the  argument <arg>  is passed.   This function  call  should,
**  cause the execution  of an assignment to  that global variable, otherwise
**  an error is signalled.
*/
libGAP_Obj             libGAP_AUTOFunc;

libGAP_Obj             libGAP_AUTOHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 func;           /* the function to call            */
    libGAP_Obj                 arg;            /* the argument to pass            */
    libGAP_Obj                 list;           /* function and argument list      */
    libGAP_Obj                 name;           /* one name (as a GAP string)      */
    libGAP_UInt                gvar;           /* one global variable             */
    libGAP_UInt                i;              /* loop variable                   */

    /* check that there are enough arguments                               */
    if ( libGAP_LEN_LIST(args) < 2 ) {
        libGAP_ErrorQuit(
            "usage: AUTO( <func>, <arg>, <name1>... )",
            0L, 0L );
        return 0;
    }

    /* get and check the function                                          */
    func = libGAP_ELM_LIST( args, 1 );
    while ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        func = libGAP_ErrorReturnObj(
            "AUTO: <func> must be a function (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(func), 0L,
            "you can return a function for <func>" );
    }

    /* get the argument                                                    */
    arg = libGAP_ELM_LIST( args, 2 );

    /* make the list of function and argument                              */
    list = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( list, 2 );
    libGAP_SET_ELM_PLIST( list, 1, func );
    libGAP_SET_ELM_PLIST( list, 2, arg );

    /* make the global variables automatic                                 */
    for ( i = 3; i <= libGAP_LEN_LIST(args); i++ ) {
        name = libGAP_ELM_LIST( args, i );
        while ( ! libGAP_IsStringConv(name) ) {
            name = libGAP_ErrorReturnObj(
                "AUTO: <name> must be a string (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(name), 0L,
                "you can return a string for <name>" );
        }
        gvar = libGAP_GVarName( libGAP_CSTR_STRING(name) );
        libGAP_SET_ELM_PLIST( libGAP_ValGVars,   gvar, 0    );
        libGAP_SET_ELM_PLIST( libGAP_ExprGVars, gvar, list );
        libGAP_CHANGED_BAG(   libGAP_ExprGVars );
    }

    /* return void                                                         */
    return 0;
}


/****************************************************************************
**
*F  iscomplete( <name>, <len> ) . . . . . . . .  find the completions of name
*F  completion( <name>, <len> ) . . . . . . . .  find the completions of name
*/
libGAP_UInt            libGAP_iscomplete_gvar (
    libGAP_Char *              name,
    libGAP_UInt                len )
{
    libGAP_Char *              curr;
    libGAP_UInt                i, k;

    for ( i = 1; i <= libGAP_CountGVars; i++ ) {
        curr = libGAP_NameGVar( i );
        for ( k = 0; name[k] != 0 && curr[k] == name[k]; k++ ) ;
        if ( k == len && curr[k] == '\0' )  return 1;
    }
    return 0;
}

libGAP_UInt            libGAP_completion_gvar (
    libGAP_Char *              name,
    libGAP_UInt                len )
{
    libGAP_Char *              curr;
    libGAP_Char *              next;
    libGAP_UInt                i, k;

    next = 0;
    for ( i = 1; i <= libGAP_CountGVars; i++ ) {
        /* consider only variables which are currently bound for completion */
        if ( libGAP_VAL_GVAR( i ) || libGAP_ELM_PLIST( libGAP_ExprGVars, i )) {
            curr = libGAP_NameGVar( i );
            for ( k = 0; name[k] != 0 && curr[k] == name[k]; k++ ) ;
            if ( k < len || curr[k] <= name[k] )  continue;
            if ( next != 0 ) {
                for ( k = 0; curr[k] != '\0' && curr[k] == next[k]; k++ ) ;
                if ( k < len || next[k] < curr[k] )  continue;
            }
            next = curr;
        }
    }

    if ( next != 0 ) {
        for ( k = 0; next[k] != '\0'; k++ )
            name[k] = next[k];
        name[k] = '\0';
    }

    return next != 0;
}


/****************************************************************************
**
*F  FuncIDENTS_GVAR( <self> ) . . . . . . . . . .  idents of global variables
*/
libGAP_Obj libGAP_FuncIDENTS_GVAR (
    libGAP_Obj                 self )
{
    /*QQ extern Obj          NameGVars;   */
    libGAP_Obj                 copy;
    libGAP_UInt                i;

    copy = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, libGAP_LEN_PLIST(libGAP_NameGVars) );
    for ( i = 1;  i <= libGAP_LEN_PLIST(libGAP_NameGVars);  i++ ) {
        libGAP_SET_ELM_PLIST( copy, i, libGAP_ELM_PLIST( libGAP_NameGVars, i ) );
    }
    libGAP_SET_LEN_PLIST( copy, libGAP_LEN_PLIST(libGAP_NameGVars) );
    return copy;
}

libGAP_Obj libGAP_FuncIDENTS_BOUND_GVARS (
    libGAP_Obj                 self )
{
    /*QQ extern Obj          NameGVars;   */
    libGAP_Obj                 copy;
    libGAP_UInt                i, j;

    copy = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, libGAP_LEN_PLIST(libGAP_NameGVars) );
    for ( i = 1, j = 1;  i <= libGAP_LEN_PLIST(libGAP_NameGVars);  i++ ) {
        if ( libGAP_VAL_GVAR( i ) || libGAP_ELM_PLIST( libGAP_ExprGVars, i )) {
           libGAP_SET_ELM_PLIST( copy, j, libGAP_ELM_PLIST( libGAP_NameGVars, i ) );
           j++;
        }
    }
    libGAP_SET_LEN_PLIST( copy, j - 1 );
    return copy;
}

/****************************************************************************
**
*F  FuncASS_GVAR( <self>, <gvar>, <val> ) . . . . assign to a global variable
*/
libGAP_Obj libGAP_FuncASS_GVAR (
    libGAP_Obj                 self,
    libGAP_Obj                 gvar,
    libGAP_Obj                 val )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( gvar ) ) {
        gvar = libGAP_ErrorReturnObj(
            "READ: <gvar> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(gvar), 0L,
            "you can return a string for <gvar>" );
    }

    libGAP_AssGVar( libGAP_GVarName( libGAP_CSTR_STRING(gvar) ), val );
    return 0L;
}


/****************************************************************************
**
*F  FuncISB_GVAR( <self>, <gvar> )  . . check assignment of a global variable
*/
libGAP_Obj libGAP_FuncISB_GVAR (
    libGAP_Obj                 self,
    libGAP_Obj                 gvar )
{
  libGAP_UInt gv;
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( gvar ) ) {
        gvar = libGAP_ErrorReturnObj(
            "ISB_GVAR: <gvar> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(gvar), 0L,
            "you can return a string for <gvar>" );
    }

    gv = libGAP_GVarName( libGAP_CSTR_STRING(gvar) );
    return ( libGAP_VAL_GVAR( gv ) ||
             libGAP_ELM_PLIST( libGAP_ExprGVars, gv )) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncVAL_GVAR( <self>, <gvar> )  . . contents of a global variable
*/

libGAP_Obj libGAP_FuncVAL_GVAR (
    libGAP_Obj                 self,
   libGAP_Obj                 gvar )
{
  libGAP_Obj val;
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( gvar ) ) {
        gvar = libGAP_ErrorReturnObj(
            "VAL_GVAR: <gvar> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(gvar), 0L,
            "you can return a string for <gvar>" );
    }

    /* get the value */
    val = libGAP_ValAutoGVar( libGAP_GVarName( libGAP_CSTR_STRING(gvar) ) );

    while (val == (libGAP_Obj) 0)
      val = libGAP_ErrorReturnObj("VAL_GVAR: No value bound to %s",
                           (libGAP_Int)libGAP_CSTR_STRING(gvar), (libGAP_Int) 0,
                           "you can return a value" );
    return val;
}

/****************************************************************************
**
*F  FuncUNB_GVAR( <self>, <gvar> )  . . unbind a global variable
*/

libGAP_Obj libGAP_FuncUNB_GVAR (
    libGAP_Obj                 self,
    libGAP_Obj                 gvar )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( gvar ) ) {
        gvar = libGAP_ErrorReturnObj(
            "UNB_GVAR: <gvar> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(gvar), 0L,
            "you can return a string for <gvar>" );
    }

    /*  */
    libGAP_AssGVar( libGAP_GVarName( libGAP_CSTR_STRING(gvar) ), (libGAP_Obj)0 );
    return (libGAP_Obj) 0;
}



/****************************************************************************
**

*F * * * * * * * * * * * * * copies and fopies  * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  CopyAndFopyGVars  . . . . . .  kernel table of kernel copies and "fopies"
**
**  This needs to be kept inside the kernel so that the copies can be updated
**  after loading a workspace.
*/  
typedef struct  { 
    libGAP_Obj *               copy;
    libGAP_UInt                isFopy;
    const libGAP_Char *        name;
} libGAP_StructCopyGVar;

#ifndef libGAP_MAX_COPY_AND_FOPY_GVARS
#define libGAP_MAX_COPY_AND_FOPY_GVARS         30000
#endif

static libGAP_StructCopyGVar libGAP_CopyAndFopyGVars[libGAP_MAX_COPY_AND_FOPY_GVARS];
static libGAP_Int libGAP_NCopyAndFopyGVars;


/****************************************************************************
**
*F  InitCopyGVar( <name>, <copy> )  . .  declare C variable as copy of global
**
**  'InitCopyGVar' makes  the C variable <cvar>  at address  <copy> a copy of
**  the global variable named <name> (which must be a kernel string).
**
**  The function only registers the  information in <CopyAndFopyGVars>.  At a
**  latter stage one  has to call  'UpdateCopyFopyInfo' to actually enter the
**  information stored in <CopyAndFopyGVars> into a plain list.
**
**  This is OK for garbage collection, but  a real problem  for saving in any
**  event, this information  does not really want to  be saved  because it is
**  kernel centred rather than workspace centred.
**
**  Accordingly we     provide  two    functions    `RemoveCopyFopyInfo'  and
**  `RestoreCopyFopyInfo' to  remove  or restore   the  information from  the
**  workspace.  The  Restore  function is  also   intended to  be used  after
**  loading a saved workspace
*/
void libGAP_InitCopyGVar (
    const libGAP_Char *        name ,
    libGAP_Obj *               copy )
{
    /* make a record in the kernel for saving and loading                  */
    if ( libGAP_NCopyAndFopyGVars >= libGAP_MAX_COPY_AND_FOPY_GVARS ) {
        libGAP_Pr( "Panic, no room to record CopyGVar\n", 0L, 0L );
        libGAP_SyExit(1);
    }
    libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyGVars].copy = copy;
    libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyGVars].isFopy = 0;
    libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyGVars].name = name;
    libGAP_NCopyAndFopyGVars++;
}


/****************************************************************************
**
*F  InitFopyGVar( <name>, <copy> )  . .  declare C variable as copy of global
**
**  'InitFopyGVar' makes the C variable <cvar> at address <copy> a (function)
**  copy  of the  global variable <gvar>,  whose name  is <name>.  That means
**  that whenever   the value  of   <gvar> is a    function, then <cvar> will
**  reference the same value (i.e., will hold the same bag identifier).  When
**  the value  of <gvar>  is not a   function, then  <cvar> will  reference a
**  function  that signals  the error ``<func>  must be  a function''.   When
**  <gvar> has no assigned value, then <cvar> will  reference a function that
**  signals the error ``<gvar> must have an assigned value''.
*/
void libGAP_InitFopyGVar (
    const libGAP_Char *        name,
    libGAP_Obj *               copy )
{
    /* make a record in the kernel for saving and loading                  */
    if ( libGAP_NCopyAndFopyGVars >= libGAP_MAX_COPY_AND_FOPY_GVARS ) {
        libGAP_Pr( "Panic, no room to record FopyGVar\n", 0L, 0L );
        libGAP_SyExit(1);
    }
    libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyGVars].copy = copy;
    libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyGVars].isFopy = 1;
    libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyGVars].name = name;
    libGAP_NCopyAndFopyGVars++;
}


/****************************************************************************
**
*F  UpdateCopyFopyInfo()  . . . . . . . . . .  convert kernel info into plist
*/
static libGAP_Int libGAP_NCopyAndFopyDone;

void libGAP_UpdateCopyFopyInfo ( void )
{
    libGAP_Obj                 cops;           /* copies list                     */
    libGAP_UInt                ncop;           /* number of copies                */
    libGAP_UInt                gvar;
    const libGAP_Char *        name;           /* name of the variable            */
    libGAP_Obj *               copy;           /* address of the copy             */

    /* loop over new copies and fopies                                     */
    for ( ; libGAP_NCopyAndFopyDone < libGAP_NCopyAndFopyGVars; libGAP_NCopyAndFopyDone++ ) {
        name = libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyDone].name;
        copy = libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyDone].copy;
        gvar = libGAP_GVarName(name);

        /* get the copies list and its length                              */
        if ( libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyDone].isFopy ) {
            if ( libGAP_ELM_PLIST( libGAP_FopiesGVars, gvar ) != 0 ) {
                cops = libGAP_ELM_PLIST( libGAP_FopiesGVars, gvar );
            }
            else {
                cops = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
                libGAP_SET_ELM_PLIST( libGAP_FopiesGVars, gvar, cops );
                libGAP_CHANGED_BAG(libGAP_FopiesGVars);
            }
        }
        else {
            if ( libGAP_ELM_PLIST( libGAP_CopiesGVars, gvar ) != 0 ) {
                cops = libGAP_ELM_PLIST( libGAP_CopiesGVars, gvar );
            }
            else {
                cops = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
                libGAP_SET_ELM_PLIST( libGAP_CopiesGVars, gvar, cops );
                libGAP_CHANGED_BAG(libGAP_CopiesGVars);
            }
        }
        ncop = libGAP_LEN_PLIST(cops);

        /* append the copy to the copies list                              */
        libGAP_GROW_PLIST( cops, ncop+1 );
        libGAP_SET_LEN_PLIST( cops, ncop+1 );
        libGAP_SET_ELM_PLIST( cops, ncop+1, (libGAP_Obj)copy );
        libGAP_CHANGED_BAG(cops);

        /* now copy the value of <gvar> to <cvar>                          */
        if ( libGAP_CopyAndFopyGVars[libGAP_NCopyAndFopyDone].isFopy ) {
            if ( libGAP_VAL_GVAR(gvar) != 0 && libGAP_IS_FUNC(libGAP_VAL_GVAR(gvar)) ) {
                *copy = libGAP_VAL_GVAR(gvar);
            }
            else if ( libGAP_VAL_GVAR(gvar) != 0 ) {
                *copy = libGAP_ErrorMustEvalToFuncFunc;
            }
            else {
                *copy = libGAP_ErrorMustHaveAssObjFunc;
            }
        }
        else {
            *copy = libGAP_VAL_GVAR(gvar);
        }
    }
}


/****************************************************************************
**
*F  RemoveCopyFopyInfo()  . . . remove the info about copies of gvars from ws
*/
void libGAP_RemoveCopyFopyInfo( void )
{
    libGAP_UInt        i, l;

    l = libGAP_LEN_PLIST(libGAP_CopiesGVars);
    for ( i = 1; i <= l; i++ )
        libGAP_SET_ELM_PLIST( libGAP_CopiesGVars, i, 0 );
    l = libGAP_LEN_PLIST(libGAP_FopiesGVars);
    for ( i = 1; i <= l; i++ )
        libGAP_SET_ELM_PLIST( libGAP_FopiesGVars, i, 0 );
    libGAP_NCopyAndFopyDone = 0;
    return;
}


/****************************************************************************
**
*F  RestoreCopyFopyInfo() . . .  restore the info from the copy in the kernel
*/
void libGAP_RestoreCopyFopyInfo( void )
{
    libGAP_NCopyAndFopyDone = 0;
    libGAP_UpdateCopyFopyInfo();
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "MakeReadOnlyGVar", 1, "name",
      libGAP_MakeReadOnlyGVarHandler, "src/gap.c:MakeReadOnlyGVar" },

    { "MakeReadWriteGVar", 1, "name",
      libGAP_MakeReadWriteGVarHandler, "src/gap.c:MakeReadWriteGVar" },

    { "IsReadOnlyGVar", 1, "name",
      libGAP_FuncIsReadOnlyGVar, "src/gap.c:IsReadOnlyGVar" },

    { "AUTO", -1, "args",
      libGAP_AUTOHandler, "src/gap.c:AUTO" },
               
    { "IDENTS_GVAR", 0L, "",
      libGAP_FuncIDENTS_GVAR, "src/gap.c:IDENTS_GVAR" },

    { "IDENTS_BOUND_GVARS", 0L, "",
      libGAP_FuncIDENTS_BOUND_GVARS, "src/gap.c:IDENTS_BOUND_GVARS" },

    { "ISB_GVAR", 1L, "gvar",
      libGAP_FuncISB_GVAR, "src/gap.c:ISB_GVAR" },

    { "ASS_GVAR", 2L, "gvar, value",
      libGAP_FuncASS_GVAR, "src/gap.c:ASS_GVAR" },

    { "VAL_GVAR", 1L, "gvar",
      libGAP_FuncVAL_GVAR, "src/gap.c:VAL_GVAR" },

    { "UNB_GVAR", 1L, "gvar",
      libGAP_FuncUNB_GVAR, "src/gap.c:UNB_GVAR" },

    { "SET_NAMESPACE", 1L, "str",
      libGAP_FuncSET_NAMESPACE, "src/gvars.c:SET_NAMESPACE" },

    { "GET_NAMESPACE", 0L, "",
      libGAP_FuncGET_NAMESPACE, "src/gvars.c:GET_NAMESPACE" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_ValGVars = (libGAP_Obj) 0;
  libGAP_NCopyAndFopyGVars = 0;
  libGAP_NCopyAndFopyDone = 0;
  libGAP_InitHandlerRegistration();
  
    /* init global bags and handler                                        */
    libGAP_InitGlobalBag( &libGAP_ErrorMustEvalToFuncFunc,
                   "src/gvars.c:ErrorMustEvalToFuncFunc" );
    libGAP_InitGlobalBag( &libGAP_ErrorMustHaveAssObjFunc,
                   "src/gvars.c:ErrorMustHaveAssObjFunc" );
    libGAP_InitGlobalBag( &libGAP_ValGVars,
                   "src/gvars.c:ValGVars" );
    libGAP_InitGlobalBag( &libGAP_NameGVars,
                   "src/gvars.c:NameGVars" );
    libGAP_InitGlobalBag( &libGAP_WriteGVars,
                   "src/gvars.c:WriteGVars" );
    libGAP_InitGlobalBag( &libGAP_ExprGVars,
                   "src/gvars.c:ExprGVars" );
    libGAP_InitGlobalBag( &libGAP_CopiesGVars,
                   "src/gvars.c:CopiesGVars" );
    libGAP_InitGlobalBag( &libGAP_FopiesGVars,
                   "src/gvars.c:FopiesGVars"  );
    libGAP_InitGlobalBag( &libGAP_TableGVars,
                   "src/gvars.c:TableGVars" );
    libGAP_InitGlobalBag( &libGAP_CurrNamespace,
                   "src/gvars.c:CurrNamespace" );

    libGAP_InitHandlerFunc( libGAP_ErrorMustEvalToFuncHandler,
                     "src/gvars.c:ErrorMustEvalToFuncHandler" );
    libGAP_InitHandlerFunc( libGAP_ErrorMustHaveAssObjHandler,
                     "src/gvars.c:ErrorMustHaveAssObjHandler" );

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* Get a copy of REREADING                                             */
    libGAP_ImportGVarFromLibrary("REREADING", &libGAP_REREADING);
    
    
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/

static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* make the lists for global variables                                 */
    libGAP_CountGVars = libGAP_LEN_PLIST( libGAP_ValGVars );
    libGAP_PtrGVars   = libGAP_ADDR_OBJ( libGAP_ValGVars );
    libGAP_SizeGVars  = libGAP_LEN_PLIST( libGAP_TableGVars );

    /* create the global variable '~'                                      */
    libGAP_Tilde = libGAP_GVarName( "~" );


    /* update fopies and copies                                            */
    libGAP_UpdateCopyFopyInfo();

    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  PreSave( <module> ) . . . . . . . . . . . . . before save workspace
*/
static libGAP_Int libGAP_PreSave (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_RemoveCopyFopyInfo();
  return 0;
}

/****************************************************************************
**
*F  PostSave( <module> ) . . . . . . . . . . . . . aftersave workspace
*/
static libGAP_Int libGAP_PostSave (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_UpdateCopyFopyInfo();
  return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* make the error functions for 'AssGVar'                              */
    libGAP_ErrorMustEvalToFuncFunc = libGAP_NewFunctionC(
        "ErrorMustEvalToFunc", -1,"args", libGAP_ErrorMustEvalToFuncHandler );
    
    libGAP_ErrorMustHaveAssObjFunc = libGAP_NewFunctionC(
        "ErrorMustHaveAssObj", -1L,"args", libGAP_ErrorMustHaveAssObjHandler );

    /* make the lists for global variables                                 */
    libGAP_ValGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( libGAP_ValGVars, 0 );

    libGAP_NameGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( libGAP_NameGVars, 0 );

    libGAP_WriteGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( libGAP_WriteGVars, 0 );

    libGAP_ExprGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( libGAP_ExprGVars, 0 );

    libGAP_CopiesGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( libGAP_CopiesGVars, 0 );

    libGAP_FopiesGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( libGAP_FopiesGVars, 0 );

    /* make the list of global variables                                   */
    libGAP_SizeGVars  = 997;
    libGAP_TableGVars = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_SizeGVars );
    libGAP_SET_LEN_PLIST( libGAP_TableGVars, libGAP_SizeGVars );

    /* Create the current namespace: */
    libGAP_TLS(libGAP_CurrNamespace) = libGAP_NEW_STRING(0);
    libGAP_SET_LEN_STRING(libGAP_TLS(libGAP_CurrNamespace),0);
    
    /* fix C vars                                                          */
    libGAP_PostRestore( libGAP_module );

    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  CheckInit( <module> ) . . . . . . . . . . . . . . .  check initialisation
*/
static libGAP_Int libGAP_CheckInit (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int                 success = 1;

    if ( libGAP_NCopyAndFopyGVars != libGAP_NCopyAndFopyDone ) {
        success = 0;
        libGAP_Pr( "#W  failed to updated copies and fopies\n", 0L, 0L );
    }

    /* return success                                                      */
    return ! success;
}


/****************************************************************************
**
*F  InitInfoGVars() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "gvars",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    libGAP_CheckInit,                          /* checkInit                      */
    libGAP_PreSave,                            /* preSave                        */
    libGAP_PostSave,                            /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoGVars ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  gvars.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
