/****************************************************************************
**
*W  code.c                      GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the coder package.
**
**  The  coder package  is   the part of   the interpreter  that creates  the
**  expressions.  Its functions are called from the reader.
*/
#include        <stdio.h>               /* on SunOS, assert.h uses stderr
                                           but does not include stdio.h    */
#include        <assert.h>              /* assert                          */
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "calls.h"               /* generic call mechanism          */
/*N 1996/06/16 mschoene func expressions should be different from funcs    */

#include        "records.h"             /* generic records                 */

#include        "integer.h"             /* integers                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "funcs.h"               /* functions                       */

#include        "code.h"                /* coder                           */

#include        "saveload.h"            /* saving and loading              */
#include        "read.h"                /* to access stack of for loop globals */
#include        "gvars.h"
#include        "thread.h"              /* threads                         */
#include        "tls.h"                 /* thread-local storage            */
#include	"aobjects.h"		/* atomic objects		   */

#include        "vars.h"                /* variables                       */


#include        "profile.h"             /* access to stat register function*/

/****************************************************************************
**

*V  PtrBody . . . . . . . . . . . . . . . . . . . . . pointer to current body
**
**  'PtrBody' is a pointer to the current body.
*/
libGAP_Stat * libGAP_PtrBody;

/****************************************************************************
**

*V  FilenameCache . . . . . . . . . . . . . . . . . . list of filenames
**
**  'FilenameCache' is a list of previously opened filenames.
*/
libGAP_Obj libGAP_FilenameCache;

/****************************************************************************
**
*V  OffsBody  . . . . . . . . . . . . . . . . . . . .  offset in current body
**
**  'OffsBody' is the  offset in the current   body.  It is  only valid while
**  coding.
*/
#define libGAP_MAX_FUNC_EXPR_NESTING 1024


libGAP_Stat libGAP_OffsBody;

libGAP_Stat libGAP_OffsBodyStack[libGAP_MAX_FUNC_EXPR_NESTING];
libGAP_UInt libGAP_OffsBodyCount = 0;

libGAP_UInt libGAP_LoopNesting = 0;
libGAP_UInt libGAP_LoopStack[libGAP_MAX_FUNC_EXPR_NESTING];
libGAP_UInt libGAP_LoopStackCount = 0;

static inline void libGAP_PushOffsBody( void ) {
  assert(libGAP_TLS(libGAP_OffsBodyCount) <= libGAP_MAX_FUNC_EXPR_NESTING-1);
  libGAP_TLS(libGAP_OffsBodyStack)[libGAP_TLS(libGAP_OffsBodyCount)++] = libGAP_TLS(libGAP_OffsBody);
}

static inline void libGAP_PopOffsBody( void ) {
  assert(libGAP_TLS(libGAP_OffsBodyCount));
  libGAP_TLS(libGAP_OffsBody) = libGAP_TLS(libGAP_OffsBodyStack)[--libGAP_TLS(libGAP_OffsBodyCount)];
}

static inline void libGAP_PushLoopNesting( void ) {
  assert(libGAP_TLS(libGAP_LoopStackCount) <= libGAP_MAX_FUNC_EXPR_NESTING-1);
  libGAP_TLS(libGAP_LoopStack)[libGAP_TLS(libGAP_LoopStackCount)++] = libGAP_TLS(libGAP_LoopNesting);
}

static inline void libGAP_PopLoopNesting( void ) {
  assert(libGAP_TLS(libGAP_LoopStackCount));
  libGAP_TLS(libGAP_LoopNesting) = libGAP_TLS(libGAP_LoopStack)[--libGAP_TLS(libGAP_LoopStackCount)];
}

static inline void libGAP_setup_gapname(libGAP_TypInputFile* i)
{
  libGAP_UInt len;
  if(!i->gapname) {
    libGAP_C_NEW_STRING_DYN(i->gapname, i->name);
    len = libGAP_LEN_PLIST( libGAP_FilenameCache );
    libGAP_GROW_PLIST(      libGAP_FilenameCache, len+1 );
    libGAP_SET_LEN_PLIST(   libGAP_FilenameCache, len+1 );
    libGAP_SET_ELM_PLIST(   libGAP_FilenameCache, len+1, i->gapname );
    libGAP_CHANGED_BAG(     libGAP_FilenameCache );
    i->gapnameid = len+1;
  }
}

libGAP_Obj libGAP_FILENAME_STAT(libGAP_Stat stat)
{
  libGAP_Obj filename;
  libGAP_UInt filenameid = libGAP_FILENAMEID_STAT(stat);
  if (filenameid == 0)
      filename = libGAP_NEW_STRING(0);
  else
      filename = libGAP_ELM_PLIST(libGAP_FilenameCache, filenameid);
  return filename;
}
    
    
/****************************************************************************
**
*F  NewStat( <type>, <size> ) . . . . . . . . . . .  allocate a new statement
**
**  'NewStat'   allocates a new   statement memory block  of  type <type> and
**  <size> bytes.  'NewStat' returns the identifier of the new statement.
**
**  NewStat( <type>, <size>, <line> ) allows the line number of the statement
**  to also be specified (else the current line when NewStat is called is
**  used).
*/
libGAP_Stat libGAP_NewStatWithLine (
    libGAP_UInt                type,
    libGAP_UInt                size,
    libGAP_UInt                line)
{
    libGAP_Stat                stat;           /* result                          */

    /* this is where the new statement goes                                */
    stat = libGAP_TLS(libGAP_OffsBody) + libGAP_FIRST_STAT_CURR_FUNC;

    /* increase the offset                                                 */
    libGAP_TLS(libGAP_OffsBody) = stat + ((size+sizeof(libGAP_Stat)-1) / sizeof(libGAP_Stat)) * sizeof(libGAP_Stat);

    /* make certain that the current body bag is large enough              */
    if ( libGAP_SIZE_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC)) == 0 ) {
      libGAP_ResizeBag( libGAP_BODY_FUNC(libGAP_CURR_FUNC), libGAP_TLS(libGAP_OffsBody) + libGAP_NUMBER_HEADER_ITEMS_BODY*sizeof(libGAP_Obj) );
        libGAP_TLS(libGAP_PtrBody) = (libGAP_Stat*)libGAP_PTR_BAG( libGAP_BODY_FUNC(libGAP_CURR_FUNC) );
    }
    while ( libGAP_SIZE_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC)) < libGAP_TLS(libGAP_OffsBody) + libGAP_NUMBER_HEADER_ITEMS_BODY*sizeof(libGAP_Obj)  ) {
        libGAP_ResizeBag( libGAP_BODY_FUNC(libGAP_CURR_FUNC), 2*libGAP_SIZE_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC)) );
        libGAP_TLS(libGAP_PtrBody) = (libGAP_Stat*)libGAP_PTR_BAG( libGAP_BODY_FUNC(libGAP_CURR_FUNC) );
    }
    libGAP_setup_gapname(libGAP_TLS(libGAP_Input));
    
    /* enter type and size                                                 */
    libGAP_ADDR_STAT(stat)[-1] = ((libGAP_Stat)libGAP_TLS(libGAP_Input)->gapnameid << 48) + ((libGAP_Stat)line << 32) +
                          ((libGAP_Stat)size << 8) + (libGAP_Stat)type;
    libGAP_RegisterStatWithProfiling(stat);
    /* return the new statement                                            */
    return stat;
}

libGAP_Stat libGAP_NewStat (
    libGAP_UInt                type,
    libGAP_UInt                size)
{
    return libGAP_NewStatWithLine(type, size, libGAP_TLS(libGAP_Input)->number);
}


/****************************************************************************
**
*F  NewExpr( <type>, <size> ) . . . . . . . . . . . allocate a new expression
**
**  'NewExpr' allocates a new expression memory block of  the type <type> and
**  <size> bytes.  'NewExpr' returns the identifier of the new expression.
*/
libGAP_Expr            libGAP_NewExpr (
    libGAP_UInt                type,
    libGAP_UInt                size )
{
    libGAP_Expr                expr;           /* result                          */

    /* this is where the new expression goes                               */
    expr = libGAP_TLS(libGAP_OffsBody) + libGAP_FIRST_STAT_CURR_FUNC;

    /* increase the offset                                                 */
    libGAP_TLS(libGAP_OffsBody) = expr + ((size+sizeof(libGAP_Expr)-1) / sizeof(libGAP_Expr)) * sizeof(libGAP_Expr);

    /* make certain that the current body bag is large enough              */
    if ( libGAP_SIZE_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC)) == 0 ) {
        libGAP_ResizeBag( libGAP_BODY_FUNC(libGAP_CURR_FUNC), libGAP_TLS(libGAP_OffsBody) );
        libGAP_TLS(libGAP_PtrBody) = (libGAP_Stat*)libGAP_PTR_BAG( libGAP_BODY_FUNC(libGAP_CURR_FUNC) );
    }
    while ( libGAP_SIZE_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC)) < libGAP_TLS(libGAP_OffsBody) ) {
        libGAP_ResizeBag( libGAP_BODY_FUNC(libGAP_CURR_FUNC), 2*libGAP_SIZE_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC)) );
        libGAP_TLS(libGAP_PtrBody) = (libGAP_Stat*)libGAP_PTR_BAG( libGAP_BODY_FUNC(libGAP_CURR_FUNC) );
    }

    /* enter type and size                                                 */
    libGAP_ADDR_EXPR(expr)[-1] = ((libGAP_Stat)libGAP_TLS(libGAP_Input)->gapnameid << 48) +
                          ((libGAP_Stat)libGAP_TLS(libGAP_Input)->number << 32) +
                          ((libGAP_Stat)size << 8) + type;
    libGAP_RegisterStatWithProfiling(expr);
    /* return the new expression                                           */
    return expr;
}


/****************************************************************************
**
*V  CodeResult  . . . . . . . . . . . . . . . . . . . . . .  result of coding
**
**  'CodeResult'  is the result  of the coding, i.e.,   the function that was
**  coded.
*/
libGAP_Obj libGAP_CodeResult;


/****************************************************************************
**
*V  StackStat . . . . . . . . . . . . . . . . . . . . . . .  statements stack
*V  CountStat . . . . . . . . . . . . . . . number of statements on the stack
*F  PushStat( <stat> )  . . . . . . . . . . . . push statement onto the stack
*F  PopStat() . . . . . . . . . . . . . . . . .  pop statement from the stack
**
**  'StackStat' is the stack of statements that have been coded.
**
**  'CountStat'   is the number   of statements  currently on  the statements
**  stack.
**
**  'PushStat'  pushes the statement  <stat> onto the  statements stack.  The
**  stack is automatically resized if necessary.
**
**  'PopStat' returns the  top statement from the  statements  stack and pops
**  it.  It is an error if the stack is empty.
*/
libGAP_Bag libGAP_StackStat;

libGAP_Int libGAP_CountStat;

void libGAP_PushStat (
    libGAP_Stat                stat )
{
    /* there must be a stack, it must not be underfull or overfull         */
    assert( libGAP_TLS(libGAP_StackStat) != 0 );
    assert( 0 <= libGAP_TLS(libGAP_CountStat) );
    assert( libGAP_TLS(libGAP_CountStat) <= libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackStat))/sizeof(libGAP_Stat) );
    assert( stat != 0 );

    /* count up and put the statement onto the stack                       */
    if ( libGAP_TLS(libGAP_CountStat) == libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackStat))/sizeof(libGAP_Stat) ) {
        libGAP_ResizeBag( libGAP_TLS(libGAP_StackStat), 2*libGAP_TLS(libGAP_CountStat)*sizeof(libGAP_Stat) );
    }
    ((libGAP_Stat*)libGAP_PTR_BAG(libGAP_TLS(libGAP_StackStat)))[libGAP_TLS(libGAP_CountStat)] = stat;
    libGAP_TLS(libGAP_CountStat)++;
}

libGAP_Stat libGAP_PopStat ( void )
{
    libGAP_Stat                stat;

    /* there must be a stack, it must not be underfull/empty or overfull   */
    assert( libGAP_TLS(libGAP_StackStat) != 0 );
    assert( 1 <= libGAP_TLS(libGAP_CountStat) );
    assert( libGAP_TLS(libGAP_CountStat) <= libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackStat))/sizeof(libGAP_Stat) );

    /* get the top statement from the stack, and count down                */
    libGAP_TLS(libGAP_CountStat)--;
    stat = ((libGAP_Stat*)libGAP_PTR_BAG(libGAP_TLS(libGAP_StackStat)))[libGAP_TLS(libGAP_CountStat)];

    /* return the popped statement                                         */
    return stat;
}

libGAP_Stat libGAP_PopSeqStat (
    libGAP_UInt                nr )
{
    libGAP_Stat                body;           /* sequence, result                */
    libGAP_Stat                stat;           /* single statement                */
    libGAP_UInt                i;              /* loop variable                   */

    if (nr == 0 ) {
      body = libGAP_NewStat(libGAP_T_EMPTY, 0);
    }
    /* special case for a single statement                                 */
    else if ( nr == 1 ) {
        body = libGAP_PopStat();
    }

    /* general case                                                        */
    else {

        /* allocate the sequence                                           */
        if ( 2 <= nr && nr <= 7 ) {
            body = libGAP_NewStat( libGAP_T_SEQ_STAT+(nr-1), nr * sizeof(libGAP_Stat) );
        }
        else {
            body = libGAP_NewStat( libGAP_T_SEQ_STAT,        nr * sizeof(libGAP_Stat) );
        }

        /* enter the statements into the sequence                          */
        for ( i = nr; 1 <= i; i-- ) {
            stat = libGAP_PopStat();
            libGAP_ADDR_STAT(body)[i-1] = stat;
        }
    }

    /* return the sequence                                                 */
    return body;
}


/****************************************************************************
**
*V  StackExpr . . . . . . . . . . . . . . . . . . . . . . . expressions stack
*V  CountExpr . . . . . . . . . . . . . .  number of expressions on the stack
*F  PushExpr( <expr> )  . . . . . . . . . . .  push expression onto the stack
*F  PopExpr() . . . . . . . . . . . . . . . .   pop expression from the stack
**
**  'StackExpr' is the stack of expressions that have been coded.
**
**  'CountExpr'  is the number   of expressions currently  on the expressions
**  stack.
**
**  'PushExpr' pushes the expression <expr> onto the  expressions stack.  The
**  stack is automatically resized if necessary.
**
**  'PopExpr' returns the top expressions from the expressions stack and pops
**  it.  It is an error if the stack is empty.
*/
libGAP_Bag libGAP_StackExpr;

libGAP_Int libGAP_CountExpr;

void libGAP_PushExpr (
    libGAP_Expr                expr )
{
    /* there must be a stack, it must not be underfull or overfull         */
    assert( libGAP_TLS(libGAP_StackExpr) != 0 );
    assert( 0 <= libGAP_TLS(libGAP_CountExpr) );
    assert( libGAP_TLS(libGAP_CountExpr) <= libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackExpr))/sizeof(libGAP_Expr) );
    assert( expr != 0 );

    /* count up and put the expression onto the stack                      */
    if ( libGAP_TLS(libGAP_CountExpr) == libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackExpr))/sizeof(libGAP_Expr) ) {
        libGAP_ResizeBag( libGAP_TLS(libGAP_StackExpr), 2*libGAP_TLS(libGAP_CountExpr)*sizeof(libGAP_Expr) );
    }
    ((libGAP_Expr*)libGAP_PTR_BAG(libGAP_TLS(libGAP_StackExpr)))[libGAP_TLS(libGAP_CountExpr)] = expr;
    libGAP_TLS(libGAP_CountExpr)++;
}

libGAP_Expr libGAP_PopExpr ( void )
{
    libGAP_Expr                expr;

    /* there must be a stack, it must not be underfull/empty or overfull   */
    assert( libGAP_TLS(libGAP_StackExpr) != 0 );
    assert( 1 <= libGAP_TLS(libGAP_CountExpr) );
    assert( libGAP_TLS(libGAP_CountExpr) <= libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackExpr))/sizeof(libGAP_Expr) );

    /* get the top expression from the stack, and count down               */
    libGAP_TLS(libGAP_CountExpr)--;
    expr = ((libGAP_Expr*)libGAP_PTR_BAG(libGAP_TLS(libGAP_StackExpr)))[libGAP_TLS(libGAP_CountExpr)];

    /* return the popped expression                                        */
    return expr;
}


/****************************************************************************
**
*F  PushUnaryOp( <type> ) . . . . . . . . . . . . . . . . push unary operator
**
**  'PushUnaryOp' pushes a   unary  operator expression onto the   expression
**  stack.  <type> is the type of the operator (currently only 'T_NOT').
*/
void libGAP_PushUnaryOp (
    libGAP_UInt                type )
{
    libGAP_Expr                unop;           /* unary operator, result          */
    libGAP_Expr                op;             /* operand                         */

    /* allocate the unary operator                                         */
    unop = libGAP_NewExpr( type, sizeof(libGAP_Expr) );

    /* enter the operand                                                   */
    op = libGAP_PopExpr();
    libGAP_ADDR_EXPR(unop)[0] = op;

    /* push the unary operator                                             */
    libGAP_PushExpr( unop );
}


/****************************************************************************
**
*F  PushBinaryOp( <type> )  . . . . . . . . . . . . . .  push binary operator
**
**  'PushBinaryOp' pushes a binary   operator expression onto  the expression
**  stack.  <type> is the type of the operator.
*/
void libGAP_PushBinaryOp (
    libGAP_UInt                type )
{
    libGAP_Expr                binop;          /* binary operator, result         */
    libGAP_Expr                opL;            /* left operand                    */
    libGAP_Expr                opR;            /* right operand                   */

    /* allocate the binary operator                                        */
    binop = libGAP_NewExpr( type, 2*sizeof(libGAP_Expr) );

    /* enter the right operand                                             */
    opR = libGAP_PopExpr();
    libGAP_ADDR_EXPR(binop)[1] = opR;

    /* enter the left operand                                              */
    opL = libGAP_PopExpr();
    libGAP_ADDR_EXPR(binop)[0] = opL;

    /* push the binary operator                                            */
    libGAP_PushExpr( binop );
}


/****************************************************************************
**

*F * * * * * * * * * * * * *  coder functions * * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  CodeFuncCallOptionsBegin() . . . . . . . . . . . . .  code options, begin
*F  CodeFuncCallOptionsBeginElmName(<rnam>). . .  code options, begin element
*F  CodeFuncCallOptionsBeginElmExpr() . .. . . . .code options, begin element
*F  CodeFuncCallOptionsEndElm() . . .. .  . . . . . code options, end element
*F  CodeFuncCallOptionsEndElmEmpty() .. .  . . . . .code options, end element
*F  CodeFuncCallOptionsEnd(<nr>)  . . . . . . . . . . . . . code options, end
**
**  The net effect of all of these is to leave a record expression on the stack
**  containing the options record. It will be picked up by
**  CodeFuncCallEnd()
**
*/
void            libGAP_CodeFuncCallOptionsBegin ( void )
{
}

void            libGAP_CodeFuncCallOptionsBeginElmName (
    libGAP_UInt                rnam )
{
    /* push the record name as integer expressions                         */
    libGAP_PushExpr( libGAP_INTEXPR_INT( rnam ) );
}

void            libGAP_CodeFuncCallOptionsBeginElmExpr ( void )
{
  /* The expression is on the stack where we want it */
}

void            libGAP_CodeFuncCallOptionsEndElm ( void )
{
}

void            libGAP_CodeFuncCallOptionsEndElmEmpty ( void )
{
  /* The default value is true */
      libGAP_PushExpr( libGAP_NewExpr( libGAP_T_TRUE_EXPR, 0L ) );
}

void            libGAP_CodeFuncCallOptionsEnd ( libGAP_UInt nr )
{
    libGAP_Expr                record;         /* record, result                  */
    libGAP_Expr                entry;          /* entry                           */
    libGAP_Expr                rnam;           /* position of an entry            */
    libGAP_UInt                i;              /* loop variable                   */

    /* allocate the record expression                                      */
    record = libGAP_NewExpr( libGAP_T_REC_EXPR,      nr * 2 * sizeof(libGAP_Expr) );
    

    /* enter the entries                                                   */
    for ( i = nr; 1 <= i; i-- ) {
        entry = libGAP_PopExpr();
        rnam  = libGAP_PopExpr();
        libGAP_ADDR_EXPR(record)[2*(i-1)]   = rnam;
        libGAP_ADDR_EXPR(record)[2*(i-1)+1] = entry;
    }

    /* push the record                                                     */
    libGAP_PushExpr( record );

}


/****************************************************************************
**

*F  CodeBegin() . . . . . . . . . . . . . . . . . . . . . . . start the coder
*F  CodeEnd( <error> )  . . . . . . . . . . . . . . . . . . .  stop the coder
**
**  'CodeBegin'  starts  the  coder.    It is   called  from  the   immediate
**  interpreter   when he encounters  a construct  that it cannot immediately
**  interpret.
**
**  'CodeEnd' stops the coder.  It  is called from the immediate  interpreter
**  when he is done with the construct  that it cannot immediately interpret.
**  If <error> is  non-zero, a syntax error  was detected by the  reader, and
**  the coder should only clean up.
**
**  ...only function expressions inbetween...
*/
libGAP_Bag libGAP_CodeLVars;

void libGAP_CodeBegin ( void )
{
    /* the stacks must be empty                                            */
    assert( libGAP_TLS(libGAP_CountStat) == 0 );
    assert( libGAP_TLS(libGAP_CountExpr) == 0 );

    /* remember the current frame                                          */
    libGAP_TLS(libGAP_CodeLVars) = libGAP_TLS(libGAP_CurrLVars);

    /* clear the code result bag                                           */
    libGAP_TLS(libGAP_CodeResult) = 0;
}

libGAP_UInt libGAP_CodeEnd (
    libGAP_UInt                error )
{
    /* if everything went fine                                             */
    if ( ! error ) {

        /* the stacks must be empty                                        */
        assert( libGAP_TLS(libGAP_CountStat) == 0 );
        assert( libGAP_TLS(libGAP_CountExpr) == 0 );

        /* we must be back to 'TLS(CurrLVars)'                                  */
        assert( libGAP_TLS(libGAP_CurrLVars) == libGAP_TLS(libGAP_CodeLVars) );

        /* 'CodeFuncExprEnd' left the function already in 'TLS(CodeResult)'     */
    }

    /* otherwise clean up the mess                                         */
    else {

        /* empty the stacks                                                */
        libGAP_TLS(libGAP_CountStat) = 0;
        libGAP_TLS(libGAP_CountExpr) = 0;

        /* go back to the correct frame                                    */
        libGAP_SWITCH_TO_OLD_LVARS( libGAP_TLS(libGAP_CodeLVars) );
    }

    /* return value is ignored                                             */
    return 0;
}


/****************************************************************************
**
*F  CodeFuncCallBegin() . . . . . . . . . . . . . . code function call, begin
*F  CodeFuncCallEnd( <funccall>, <options>, <nr> )  code function call, end
**
**  'CodeFuncCallBegin'  is an action to code  a function call.  It is called
**  by the reader  when it encounters the parenthesis  '(', i.e., *after* the
**  function expression is read.
**
**  'CodeFuncCallEnd' is an action to code a  function call.  It is called by
**  the reader when  it  encounters the parenthesis  ')',  i.e.,  *after* the
**  argument expressions are read.   <funccall> is 1  if  this is a  function
**  call,  and 0  if  this  is  a procedure  call.    <nr> is the   number of
**  arguments. <options> is 1 if options were present after the ':' in which
**  case the options have been read already.
*/
void libGAP_CodeFuncCallBegin ( void )
{
}

void libGAP_CodeFuncCallEnd (
    libGAP_UInt                funccall,
    libGAP_UInt                libGAP_options,
    libGAP_UInt                nr )
{
    libGAP_Expr                call;           /* function call, result           */
    libGAP_Expr                func;           /* function expression             */
    libGAP_Expr                arg;            /* one argument expression         */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Expr                opts = 0;       /* record literal for the options  */
    libGAP_Expr                wrapper;        /* wrapper for calls with options  */

    /* allocate the function call                                          */
    if ( funccall && nr <= 6 ) {
        call = libGAP_NewExpr( libGAP_T_FUNCCALL_0ARGS+nr, libGAP_SIZE_NARG_CALL(nr) );
    }
    else if ( funccall /* && 6 < nr */ ) {
        call = libGAP_NewExpr( libGAP_T_FUNCCALL_XARGS,    libGAP_SIZE_NARG_CALL(nr) );
    }
    else if ( /* ! funccall && */ nr <=6 ) {
        call = libGAP_NewExpr( libGAP_T_PROCCALL_0ARGS+nr, libGAP_SIZE_NARG_CALL(nr) );
    }
    else /* if ( ! funccall && 6 < nr ) */ {
        call = libGAP_NewExpr( libGAP_T_PROCCALL_XARGS,    libGAP_SIZE_NARG_CALL(nr) );
    }

    /* get the options record if any */
    if (libGAP_options)
      opts = libGAP_PopExpr();
    
    /* enter the argument expressions                                      */
    for ( i = nr; 1 <= i; i-- ) {
        arg = libGAP_PopExpr();
        libGAP_ARGI_CALL(call,i) = arg;
    }

    /* enter the function expression                                       */
    func = libGAP_PopExpr();
    libGAP_FUNC_CALL(call) = func;

    /* wrap up the call with the options */
    if (libGAP_options)
      {
        wrapper = libGAP_NewExpr( funccall ? libGAP_T_FUNCCALL_OPTS : libGAP_T_PROCCALL_OPTS, 
                           2*sizeof(libGAP_Expr));
        libGAP_ADDR_EXPR(wrapper)[0] = opts;
        libGAP_ADDR_EXPR(wrapper)[1] = call;
        call = wrapper;
      }

    /* push the function call                                              */
    if ( funccall ) {
        libGAP_PushExpr( call );
    }
    else {
        libGAP_PushStat( call );
    }
}


/****************************************************************************
**
*F  CodeFuncExprBegin( <narg>, <nloc>, <nams> ) . . code function expr, begin
*F  CodeFuncExprEnd( <nr>, <mapsto> )   . . . . code function expression, end
**
**  'CodeFuncExprBegin'  is an action to code  a  function expression.  It is
**  called when the reader encounters the beginning of a function expression.
**  <narg> is the number of  arguments (-1 if the  function takes a  variable
**  number of arguments), <nloc> is the number of locals, <nams> is a list of
**  local variable names.
**
**  'CodeFuncExprEnd'  is an action to  code  a function  expression.  It  is
**  called when the reader encounters the end of a function expression.  <nr>
**  is the number of statements in the body of the function.
*/
void libGAP_CodeFuncExprBegin (
    libGAP_Int                 narg,
    libGAP_Int                 nloc,
    libGAP_Obj                 nams,
    libGAP_Int                 startLine)
{
    libGAP_Obj                 fexp;           /* function expression bag         */
    libGAP_Obj                 fexs;           /* function expressions list       */
    libGAP_Bag                 body;           /* function body                   */
    libGAP_Bag                 old;            /* old frame                       */
    libGAP_Stat                stat1;          /* first statement in body         */

    /* remember the current offset                                         */
    libGAP_PushOffsBody();

    /* and the loop nesting depth */
    libGAP_PushLoopNesting();
    
    /* create a function expression                                        */
    fexp = libGAP_NewBag( libGAP_T_FUNCTION, libGAP_SIZE_FUNC );
    libGAP_NARG_FUNC( fexp ) = narg;
    libGAP_NLOC_FUNC( fexp ) = nloc;
    libGAP_NAMS_FUNC( fexp ) = nams;
    libGAP_CHANGED_BAG( fexp );

    /* give it a functions expressions list                                */
    fexs = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( fexs, 0 );
    libGAP_FEXS_FUNC( fexp ) = fexs;
    libGAP_CHANGED_BAG( fexp );

    /* give it a body                                                      */
    body = libGAP_NewBag( libGAP_T_BODY, 1024*sizeof(libGAP_Stat) );
    libGAP_BODY_FUNC( fexp ) = body;
    libGAP_CHANGED_BAG( fexp );

    /* record where we are reading from */
    libGAP_setup_gapname(libGAP_TLS(libGAP_Input));
    libGAP_FILENAME_BODY(body) = libGAP_TLS(libGAP_Input)->gapname;
    libGAP_STARTLINE_BODY(body) = libGAP_INTOBJ_INT(startLine);
    /*    Pr("Coding begin at %s:%d ",(Int)(TLS(Input)->name),TLS(Input)->number);
          Pr(" Body id %d\n",(Int)(body),0L); */
    libGAP_TLS(libGAP_OffsBody) = 0;
    libGAP_TLS(libGAP_LoopNesting) = 0;

    /* give it an environment                                              */
    libGAP_ENVI_FUNC( fexp ) = libGAP_TLS(libGAP_CurrLVars);
    libGAP_CHANGED_BAG( fexp );

    /* switch to this function                                             */
    libGAP_SWITCH_TO_NEW_LVARS( fexp, (narg >0 ? narg : -narg), nloc, old );
    (void) old; /* please picky compilers. */

    /* allocate the top level statement sequence                           */
    stat1 = libGAP_NewStat( libGAP_T_SEQ_STAT, 8*sizeof(libGAP_Stat) );
    assert( stat1 == libGAP_FIRST_STAT_CURR_FUNC );
}

void libGAP_CodeFuncExprEnd (
    libGAP_UInt                nr,
    libGAP_UInt                mapsto )
{
    libGAP_Expr                expr;           /* function expression, result     */
    libGAP_Stat                stat1;          /* single statement of body        */
    libGAP_Obj                 fexp;           /* function expression bag         */
    libGAP_Obj                 fexs;           /* funct. expr. list of outer func */
    libGAP_UInt                len;            /* length of func. expr. list      */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the function expression                                         */
    fexp = libGAP_CURR_FUNC;
    assert(!libGAP_LoopNesting);
    
    /* get the body of the function                                        */
    /* push an addition return-void-statement if neccessary                */
    /* the function interpreters depend on each function ``returning''     */
    if ( nr == 0 ) {
        libGAP_CodeReturnVoid();
        nr++;
    }
    else {
        stat1 = libGAP_PopStat();
        libGAP_PushStat( stat1 );
        if ( libGAP_TNUM_STAT(stat1) != libGAP_T_RETURN_VOID
          && libGAP_TNUM_STAT(stat1) != libGAP_T_RETURN_OBJ )
        {
            libGAP_CodeReturnVoid();
            nr++;
        }
    }

    /* if the body is a long sequence, pack the other statements           */
    if ( 7 < nr ) {
        stat1 = libGAP_PopSeqStat( nr-6 );
        libGAP_PushStat( stat1 );
        nr = 7;
    }

    /* stuff the first statements into the first statement sequence       */
    /* Making sure to preserve the line number and file name              */
    libGAP_ADDR_STAT(libGAP_FIRST_STAT_CURR_FUNC)[-1]
        = ((libGAP_Stat)libGAP_FILENAMEID_STAT(libGAP_FIRST_STAT_CURR_FUNC) << 48) +
          ((libGAP_Stat)libGAP_LINE_STAT(libGAP_FIRST_STAT_CURR_FUNC) << 32) +
          ((nr*sizeof(libGAP_Stat)) << 8) + libGAP_T_SEQ_STAT+nr-1;
    for ( i = 1; i <= nr; i++ ) {
        stat1 = libGAP_PopStat();
        libGAP_ADDR_STAT(libGAP_FIRST_STAT_CURR_FUNC)[nr-i] = stat1;
    }

    /* make the body smaller                                               */
    libGAP_ResizeBag( libGAP_BODY_FUNC(fexp), libGAP_TLS(libGAP_OffsBody)+libGAP_NUMBER_HEADER_ITEMS_BODY*sizeof(libGAP_Obj) );
    libGAP_ENDLINE_BODY(libGAP_BODY_FUNC(fexp)) = libGAP_INTOBJ_INT(libGAP_TLS(libGAP_Input)->number);
    /*    Pr("  finished coding %d at line %d\n",(Int)(BODY_FUNC(fexp)), TLS(Input)->number); */

    /* switch back to the previous function                                */
    libGAP_SWITCH_TO_OLD_LVARS( libGAP_ENVI_FUNC(fexp) );

    /* restore loop nesting info */
    libGAP_PopLoopNesting();
    
    /* restore the remembered offset                                       */
    libGAP_PopOffsBody();

    /* if this was inside another function definition, make the expression */
    /* and store it in the function expression list of the outer function  */
    if ( libGAP_TLS(libGAP_CurrLVars) != libGAP_TLS(libGAP_CodeLVars) ) {
        fexs = libGAP_FEXS_FUNC( libGAP_CURR_FUNC );
        len = libGAP_LEN_PLIST( fexs );
        libGAP_GROW_PLIST(      fexs, len+1 );
        libGAP_SET_LEN_PLIST(   fexs, len+1 );
        libGAP_SET_ELM_PLIST(   fexs, len+1, fexp );
        libGAP_CHANGED_BAG(     fexs );
        expr = libGAP_NewExpr( libGAP_T_FUNC_EXPR, sizeof(libGAP_Expr) );
        libGAP_ADDR_EXPR(expr)[0] = (libGAP_Expr)(len+1);
        libGAP_PushExpr( expr );
    }

    /* otherwise, make the function and store it in 'TLS(CodeResult)'           */
    else {
        libGAP_TLS(libGAP_CodeResult) = libGAP_MakeFunction( fexp );
    }

}


/****************************************************************************
**
*F  CodeIfBegin() . . . . . . . . . . . code if-statement, begin of statement
*F  CodeIfElif()  . . . . . . . . . . code if-statement, begin of elif-branch
*F  CodeIfElse()  . . . . . . . . . . code if-statement, begin of else-branch
*F  CodeIfBeginBody() . . . . . . . . . . .  code if-statement, begin of body
*F  CodeIfEndBody( <nr> ) . . . . . . . . . .  code if-statement, end of body
*F  CodeIfEnd( <nr> ) . . . . . . . . . . code if-statement, end of statement
**
**  'CodeIfBegin' is an  action to code an  if-statement.  It is called  when
**  the reader encounters the 'if', i.e., *before* the condition is read.
**
**  'CodeIfElif' is an action to code an if-statement.  It is called when the
**  reader encounters an 'elif', i.e., *before* the condition is read.
**
**  'CodeIfElse' is an action to code an if-statement.  It is called when the
**  reader encounters an 'else'.
**
**  'CodeIfBeginBody' is  an action to   code an if-statement.  It  is called
**  when  the  reader encounters the beginning   of the statement  body of an
**  'if', 'elif', or 'else' branch, i.e., *after* the condition is read.
**
**  'CodeIfEndBody' is an action to code an if-statement.   It is called when
**  the reader encounters the end of the  statements body of an 'if', 'elif',
**  or 'else' branch.  <nr> is the number of statements in the body.
**
**  'CodeIfEnd' is an action to code an if-statement.  It  is called when the
**  reader encounters the end of the statement.   <nr> is the number of 'if',
**  'elif', or 'else' branches.
*/
void libGAP_CodeIfBegin ( void )
{
}

void libGAP_CodeIfElif ( void )
{
}

void libGAP_CodeIfElse ( void )
{
    libGAP_CodeTrueExpr();
}

void libGAP_CodeIfBeginBody ( void )
{
}

void libGAP_CodeIfEndBody (
    libGAP_UInt                nr )
{
    /* collect the statements in a statement sequence if necessary         */
    libGAP_PushStat( libGAP_PopSeqStat( nr ) );
}

void libGAP_CodeIfEnd (
    libGAP_UInt                nr )
{
    libGAP_Stat                stat;           /* if-statement, result            */
    libGAP_Expr                cond;           /* condition of a branch           */
    libGAP_Stat                body;           /* body of a branch                */
    libGAP_UInt                hase;           /* has else branch                 */
    libGAP_UInt                i;              /* loop variable                   */

    /* peek at the last condition                                          */
    body = libGAP_PopStat();
    cond = libGAP_PopExpr();
    hase = (libGAP_TNUM_EXPR(cond) == libGAP_T_TRUE_EXPR);
    libGAP_PushExpr( cond );
    libGAP_PushStat( body );

    /* allocate the if-statement                                           */
    if      ( nr == 1 ) {
        stat = libGAP_NewStat( libGAP_T_IF,            nr * (sizeof(libGAP_Expr)+sizeof(libGAP_Stat)) );
    }
    else if ( nr == 2 && hase ) {
        stat = libGAP_NewStat( libGAP_T_IF_ELSE,       nr * (sizeof(libGAP_Expr)+sizeof(libGAP_Stat)) );
    }
    else if ( ! hase ) {
        stat = libGAP_NewStat( libGAP_T_IF_ELIF,       nr * (sizeof(libGAP_Expr)+sizeof(libGAP_Stat)) );
    }
    else {
        stat = libGAP_NewStat( libGAP_T_IF_ELIF_ELSE,  nr * (sizeof(libGAP_Expr)+sizeof(libGAP_Stat)) );
    }

    /* enter the branches                                                  */
    for ( i = nr; 1 <= i; i-- ) {
        body = libGAP_PopStat();
        cond = libGAP_PopExpr();
        libGAP_ADDR_STAT(stat)[2*(i-1)] = cond;
        libGAP_ADDR_STAT(stat)[2*(i-1)+1] = body;
    }

    /* push the if-statement                                               */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeForBegin()  . . . . . . . . .  code for-statement, begin of statement
*F  CodeForIn() . . . . . . . . . . . . . . . . code for-statement, 'in' read
*F  CodeForBeginBody()  . . . . . . . . . . code for-statement, begin of body
*F  CodeForEndBody( <nr> )  . . . . . . . . . code for-statement, end of body
*F  CodeForEnd()  . . . . . . . . . . .  code for-statement, end of statement
**
**  'CodeForBegin' is  an action to code  a for-statement.  It is called when
**  the reader encounters the 'for', i.e., *before* the variable is read.
**
**  'CodeForIn' is an action to code a for-statement.  It  is called when the
**  reader encounters  the 'in',  i.e., *after*  the  variable  is  read, but
**  *before* the list expression is read.
**
**  'CodeForBeginBody'  is an action to  code a for-statement.   It is called
**  when   the reader encounters the beginning   of the statement body, i.e.,
**  *after* the list expression is read.
**
**  'CodeForEndBody' is an action to code a for-statement.  It is called when
**  the reader encounters the end of the statement  body.  <nr> is the number
**  of statements in the body.
**
**  'CodeForEnd' is an action to code a for-statement.  It is called when the
**  reader encounters  the end of   the  statement, i.e., immediately   after
**  'CodeForEndBody'.
*/
void libGAP_CodeForBegin ( void )
{
}

void libGAP_CodeForIn ( void )
{
  libGAP_Expr var = libGAP_PopExpr();
  if (libGAP_TNUM_EXPR(var) == libGAP_T_REF_GVAR)
    {
      libGAP_PushGlobalForLoopVariable((libGAP_UInt)libGAP_ADDR_EXPR(var)[0]);
    }
  libGAP_PushExpr(var);
}

void libGAP_CodeForBeginBody ( void )
{
  libGAP_TLS(libGAP_LoopNesting)++;
}

void libGAP_CodeForEndBody (
    libGAP_UInt                nr )
{
    libGAP_Stat                stat;           /* for-statement, result           */
    libGAP_UInt                type;           /* type of for-statement           */
    libGAP_Expr                var;            /* variable                        */
    libGAP_Expr                list;           /* list                            */
    libGAP_Stat                stat1;          /* single statement of body        */
    libGAP_UInt                i;              /* loop variable                   */

    /* fix up the case of no statements */
    if ( 0 == nr ) {
      libGAP_PushStat( libGAP_NewStat( libGAP_T_EMPTY, 0) );
      nr = 1;
    }

    /* collect the statements into a statement sequence if necessary       */
    if ( 3 < nr ) {
        libGAP_PushStat( libGAP_PopSeqStat( nr ) );
        nr = 1;
    }

    /* get the list expression                                             */
    list = libGAP_PopExpr();

    /* get the variable reference                                          */
    var = libGAP_PopExpr();

    if (libGAP_TNUM_EXPR(var) == libGAP_T_REF_GVAR)
      libGAP_PopGlobalForLoopVariable();
    
    /* select the type of the for-statement                                */
    if ( libGAP_TNUM_EXPR(list) == libGAP_T_RANGE_EXPR && libGAP_SIZE_EXPR(list) == 2*sizeof(libGAP_Expr)
      && libGAP_TNUM_EXPR(var)  == libGAP_T_REFLVAR ) {
        type = libGAP_T_FOR_RANGE + (nr-1);
    }
    else {
        type = libGAP_T_FOR + (nr-1);
    }

    /* allocate the for-statement                                          */
    stat = libGAP_NewStat( type, 2*sizeof(libGAP_Expr) + nr * sizeof(libGAP_Stat) );

    /* enter the body statements                                           */
    for ( i = nr; 1 <= i; i-- ) {
        stat1 = libGAP_PopStat();
        libGAP_ADDR_STAT(stat)[i+1] = stat1;
    }

    /* enter the list expression                                           */
    libGAP_ADDR_STAT(stat)[1] = list;

    /* enter the variable reference                                        */
    libGAP_ADDR_STAT(stat)[0] = var;

    /* push the for-statement                                              */
    libGAP_PushStat( stat );

    /* decrement loop nesting count */
    libGAP_TLS(libGAP_LoopNesting)--;
}

void libGAP_CodeForEnd ( void )
{
}


/****************************************************************************
**
*F  CodeAtomicBegin()  . . . . . . .  code atomic-statement, begin of statement
*F  CodeAtomicBeginBody()  . . . . . . . . code atomic-statement, begin of body
*F  CodeAtomicEndBody( <nr> )  . . . . . . . code atomic-statement, end of body
*F  CodeAtomicEnd()  . . . . . . . . .  code atomic-statement, end of statement
**
**  'CodeAtomicBegin'  is an action to  code a atomic-statement.   It is called
**  when the  reader encounters the 'atomic',  i.e., *before* the condition is
**  read.
**
**  'CodeAtomicBeginBody'  is  an action   to code a  atomic-statement.   It is
**  called when  the reader encounters  the beginning  of the statement body,
**  i.e., *after* the condition is read.
**
**  'CodeAtomicEndBody' is an action to  code a atomic-statement.  It is called
**  when the reader encounters  the end of  the statement body.  <nr> is  the
**  number of statements in the body.
**
**  'CodeAtomicEnd' is an action to code a atomic-statement.  It is called when
**  the reader encounters  the end  of the  statement, i.e., immediate  after
**  'CodeAtomicEndBody'.
**
**  These functions are just placeholders for the future HPC-GAP code.
*/

void libGAP_CodeAtomicBegin ( void )
{
}

void libGAP_CodeAtomicBeginBody ( libGAP_UInt nrexprs )
{
  libGAP_PushExpr(libGAP_INTEXPR_INT(nrexprs)); 
  return;
}

void libGAP_CodeAtomicEndBody (
    libGAP_UInt                nrstats )
{
    libGAP_Stat                stat;           /* atomic-statement, result         */
    libGAP_Stat                stat1;          /* single statement of body        */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_UInt nrexprs;
    libGAP_Expr  e,qual;


    /* fix up the case of no statements */
    if ( 0 == nrstats ) {
       libGAP_PushStat( libGAP_NewStat( libGAP_T_EMPTY, 0) );
       nrstats = 1;
    }
    
    /* collect the statements into a statement sequence   */
    if ( 1 < nrstats ) {
      stat1 = libGAP_PopSeqStat( nrstats );
    } else {
      stat1 = libGAP_PopStat();
    }
    nrexprs = libGAP_INT_INTEXPR(libGAP_PopExpr());
    
    /* allocate the atomic-statement                                        */
    stat = libGAP_NewStat( libGAP_T_ATOMIC, sizeof(libGAP_Stat) + nrexprs*2*sizeof(libGAP_Stat) );
    

    /* enter the statement sequence */
    libGAP_ADDR_STAT(stat)[0] = stat1;

    
    /* enter the expressions                                                */
    for ( i = 2*nrexprs; 1 <= i; i -= 2 ) {
        e = libGAP_PopExpr();
        qual = libGAP_PopExpr();
        libGAP_ADDR_STAT(stat)[i] = e;
        libGAP_ADDR_STAT(stat)[i-1] = qual;
    }


    /* push the atomic-statement                                            */
    libGAP_PushStat( stat );
}

void libGAP_CodeAtomicEnd ( void )
{
}

/****************************************************************************
**
*F  CodeQualifiedExprBegin()  . . . code readonly/readwrite expression start
*F  CodeQualifiedExprEnd()  . . . . . code readonly/readwrite expression end
**
**  These functions code the beginning and end of the readonly/readwrite
**  qualified expressions of an atomic statement.
*/

void libGAP_CodeQualifiedExprBegin(libGAP_UInt qual) 
{
  libGAP_PushExpr(libGAP_INTEXPR_INT(qual));
}

void libGAP_CodeQualifiedExprEnd() 
{
}




/****************************************************************************
**
*F  CodeWhileBegin()  . . . . . . .  code while-statement, begin of statement
*F  CodeWhileBeginBody()  . . . . . . . . code while-statement, begin of body
*F  CodeWhileEndBody( <nr> )  . . . . . . . code while-statement, end of body
*F  CodeWhileEnd()  . . . . . . . . .  code while-statement, end of statement
**
**  'CodeWhileBegin'  is an action to  code a while-statement.   It is called
**  when the  reader encounters the 'while',  i.e., *before* the condition is
**  read.
**
**  'CodeWhileBeginBody'  is  an action   to code a  while-statement.   It is
**  called when  the reader encounters  the beginning  of the statement body,
**  i.e., *after* the condition is read.
**
**  'CodeWhileEndBody' is an action to  code a while-statement.  It is called
**  when the reader encounters  the end of  the statement body.  <nr> is  the
**  number of statements in the body.
**
**  'CodeWhileEnd' is an action to code a while-statement.  It is called when
**  the reader encounters  the end  of the  statement, i.e., immediate  after
**  'CodeWhileEndBody'.
*/
void libGAP_CodeWhileBegin ( void )
{
}

void libGAP_CodeWhileBeginBody ( void )
{
  libGAP_TLS(libGAP_LoopNesting)++;
}

void libGAP_CodeWhileEndBody (
    libGAP_UInt                nr )
{
    libGAP_Stat                stat;           /* while-statement, result         */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                stat1;          /* single statement of body        */
    libGAP_UInt                i;              /* loop variable                   */


    /* fix up the case of no statements */
    if ( 0 == nr ) {
      libGAP_PushStat( libGAP_NewStat( libGAP_T_EMPTY, 0) );
      nr = 1;
    }
    
    /* collect the statements into a statement sequence if necessary       */
    if ( 3 < nr ) {
        libGAP_PushStat( libGAP_PopSeqStat( nr ) );
        nr = 1;
    }

    /* allocate the while-statement                                        */
    stat = libGAP_NewStat( libGAP_T_WHILE + (nr-1), sizeof(libGAP_Expr) + nr * sizeof(libGAP_Stat) );

    /* enter the statements                                                */
    for ( i = nr; 1 <= i; i-- ) {
        stat1 = libGAP_PopStat();
        libGAP_ADDR_STAT(stat)[i] = stat1;
    }

    /* enter the condition                                                 */
    cond = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = cond;

    /* decrmement loop nesting */
    libGAP_TLS(libGAP_LoopNesting)--;
    
    /* push the while-statement                                            */
    libGAP_PushStat( stat );
}

void libGAP_CodeWhileEnd ( void )
{
}


/****************************************************************************
**
*F  CodeRepeatBegin() . . . . . . . code repeat-statement, begin of statement
*F  CodeRepeatBeginBody() . . . . . . .  code repeat-statement, begin of body
*F  CodeRepeatEndBody( <nr> ) . . . . . .  code repeat-statement, end of body
*F  CodeRepeatEnd() . . . . . . . . . code repeat-statement, end of statement
**
**  'CodeRepeatBegin' is an action to code a  repeat-statement.  It is called
**  when the reader encounters the 'repeat'.
**
**  'CodeRepeatBeginBody' is an  action  to code  a  repeat-statement.  It is
**  called when the reader encounters  the  beginning of the statement  body,
**  i.e., immediately after 'CodeRepeatBegin'.
**
**  'CodeRepeatEndBody'   is an action  to code   a repeat-statement.  It  is
**  called when  the reader encounters the end  of  the statement body, i.e.,
**  *before* the condition is read.  <nr> is the  number of statements in the
**  body.
**
**  'CodeRepeatEnd' is an action to   code a repeat-statement.  It is  called
**  when  the reader encounters the end  of the statement,  i.e., *after* the
**  condition is read.
*/
void libGAP_CodeRepeatBegin ( void )
{
}

void libGAP_CodeRepeatBeginBody ( void )
{
  libGAP_TLS(libGAP_LoopNesting)++;
}

void libGAP_CodeRepeatEndBody (
    libGAP_UInt                nr )
{
    /* leave the number of statements in the body on the expression stack  */
    libGAP_PushExpr( libGAP_INTEXPR_INT(nr) );
    libGAP_TLS(libGAP_LoopNesting)--;
}

void libGAP_CodeRepeatEnd ( void )
{
    libGAP_Stat                stat;           /* repeat-statement, result        */
    libGAP_UInt                nr;             /* number of statements in body    */
    libGAP_Expr                cond;           /* condition                       */
    libGAP_Stat                stat1;          /* single statement of body        */
    libGAP_Expr                tmp;            /* temporary                       */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the condition                                                   */
    cond = libGAP_PopExpr();

    /* get the number of statements in the body                            */
    /* 'CodeUntil' left this number on the expression stack (hack)         */
    tmp = libGAP_PopExpr();
    nr = libGAP_INT_INTEXPR( tmp );

    /* fix up the case of no statements */
    if ( 0 == nr ) {
      libGAP_PushStat( libGAP_NewStat( libGAP_T_EMPTY, 0) );
      nr = 1;
    }
    /* collect the statements into a statement sequence if necessary       */
    if ( 3 < nr ) {
        libGAP_PushStat( libGAP_PopSeqStat( nr ) );
        nr = 1;
    }

    /* allocate the repeat-statement                                       */
    stat = libGAP_NewStat( libGAP_T_REPEAT + (nr-1), sizeof(libGAP_Expr) + nr * sizeof(libGAP_Stat) );

    /* enter the condition                                                 */
    libGAP_ADDR_STAT(stat)[0] = cond;

    /* enter the statements                                                */
    for ( i = nr; 1 <= i; i-- ) {
        stat1 = libGAP_PopStat();
        libGAP_ADDR_STAT(stat)[i] = stat1;
    }

    /* push the repeat-statement                                           */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeBreak() . . . . . . . . . . . . . . . . . . . .  code break-statement
**
**  'CodeBreak' is the  action to code a  break-statement.  It is called when
**  the reader encounters a 'break;'.
*/
void            libGAP_CodeBreak ( void )
{
    libGAP_Stat                stat;           /* break-statement, result         */

    if (!libGAP_TLS(libGAP_LoopNesting))
      libGAP_SyntaxError("break statement not enclosed in a loop");
    
    /* allocate the break-statement                                        */
    stat = libGAP_NewStat( libGAP_T_BREAK, 0 * sizeof(libGAP_Expr) );

    /* push the break-statement                                            */
    libGAP_PushStat( stat );
}

/****************************************************************************
**
*F  CodeContinue() . . . . . . . . . . . . . . . . . . . .  code continue-statement
**
**  'CodeContinue' is the  action to code a  continue-statement.  It is called when
**  the reader encounters a 'continue;'.
*/
void            libGAP_CodeContinue ( void )
{
    libGAP_Stat                stat;           /* continue-statement, result         */

    if (!libGAP_TLS(libGAP_LoopNesting))
      libGAP_SyntaxError("continue statement not enclosed in a loop");

    /* allocate the continue-statement                                        */
    stat = libGAP_NewStat( libGAP_T_CONTINUE, 0 * sizeof(libGAP_Expr) );

    /* push the continue-statement                                            */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeReturnObj() . . . . . . . . . . . . . . . code return-value-statement
**
**  'CodeReturnObj' is the  action to code  a return-value-statement.  It  is
**  called when the reader encounters a 'return <expr>;', but *after* reading
**  the expression <expr>.
*/
void libGAP_CodeReturnObj ( void )
{
    libGAP_Stat                stat;           /* return-statement, result        */
    libGAP_Expr                expr;           /* expression                      */

    /* allocate the return-statement                                       */
    stat = libGAP_NewStat( libGAP_T_RETURN_OBJ, sizeof(libGAP_Expr) );

    /* enter the expression                                                */
    expr = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = expr;

    /* push the return-statement                                           */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeReturnVoid()  . . . . . . . . . . . . . .  code return-void-statement
**
**  'CodeReturnVoid' is the action  to  code a return-void-statement.   It is
**  called when the reader encounters a 'return;'.
*/
void libGAP_CodeReturnVoid ( void )
{
    libGAP_Stat                stat;           /* return-statement, result        */

    /* allocate the return-statement                                       */
    stat = libGAP_NewStat( libGAP_T_RETURN_VOID, 0 * sizeof(libGAP_Expr) );

    /* push the return-statement                                           */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeOr()  . . . . . . . . . . . . . . . . . . . . . .  code or-expression
*F  CodeAnd() . . . . . . . . . . . . . . . . . . . . . . code and-expression
*F  CodeNot() . . . . . . . . . . . . . . . . . . . . . . code not-expression
*F  CodeEq()  . . . . . . . . . . . . . . . . . . . . . . . code =-expression
*F  CodeNe()  . . . . . . . . . . . . . . . . . . . . . .  code <>-expression
*F  CodeLt()  . . . . . . . . . . . . . . . . . . . . . . . code <-expression
*F  CodeGe()  . . . . . . . . . . . . . . . . . . . . . .  code >=-expression
*F  CodeGt()  . . . . . . . . . . . . . . . . . . . . . . . code >-expression
*F  CodeLe()  . . . . . . . . . . . . . . . . . . . . . .  code <=-expression
*F  CodeIn()  . . . . . . . . . . . . . . . . . . . . . .  code in-expression
*F  CodeSum() . . . . . . . . . . . . . . . . . . . . . . . code +-expression
*F  CodeAInv()  . . . . . . . . . . . . . . . . . . . code unary --expression
*F  CodeDiff()  . . . . . . . . . . . . . . . . . . . . . . code --expression
*F  CodeProd()  . . . . . . . . . . . . . . . . . . . . . . code *-expression
*F  CodeInv() . . . . . . . . . . . . . . . . . . . . . . code ^-1-expression
*F  CodeQuo() . . . . . . . . . . . . . . . . . . . . . . . code /-expression
*F  CodeMod() . . . . . . . . . . . . . . . . . . . . . . code mod-expression
*F  CodePow() . . . . . . . . . . . . . . . . . . . . . . . code ^-expression
**
**  'CodeOr', 'CodeAnd', 'CodeNot',  'CodeEq', 'CodeNe',  'CodeGt', 'CodeGe',
**  'CodeIn',  'CodeSum',  'CodeDiff', 'CodeProd', 'CodeQuo',  'CodeMod', and
**  'CodePow' are the actions to   code the respective operator  expressions.
**  They are called by the reader *after* *both* operands are read.
*/
void libGAP_CodeOrL ( void )
{
}

void libGAP_CodeOr ( void )
{
    libGAP_PushBinaryOp( libGAP_T_OR );
}

void libGAP_CodeAndL ( void )
{
}

void libGAP_CodeAnd ( void )
{
    libGAP_PushBinaryOp( libGAP_T_AND );
}

void libGAP_CodeNot ( void )
{
    libGAP_PushUnaryOp( libGAP_T_NOT );
}

void libGAP_CodeEq ( void )
{
    libGAP_PushBinaryOp( libGAP_T_EQ );
}

void libGAP_CodeNe ( void )
{
    libGAP_PushBinaryOp( libGAP_T_NE );
}

void libGAP_CodeLt ( void )
{
    libGAP_PushBinaryOp( libGAP_T_LT );
}

void libGAP_CodeGe ( void )
{
    libGAP_PushBinaryOp( libGAP_T_GE );
}

void libGAP_CodeGt ( void )
{
    libGAP_PushBinaryOp( libGAP_T_GT );
}

void libGAP_CodeLe ( void )
{
    libGAP_PushBinaryOp( libGAP_T_LE );
}

void libGAP_CodeIn ( void )
{
    libGAP_PushBinaryOp( libGAP_T_IN );
}

void libGAP_CodeSum ( void )
{
    libGAP_PushBinaryOp( libGAP_T_SUM );
}

void libGAP_CodeAInv ( void )
{
    libGAP_Expr                expr;
    libGAP_Int                 i;

    expr = libGAP_PopExpr();
    if ( libGAP_IS_INTEXPR(expr) && libGAP_INT_INTEXPR(expr) != -(1L<<libGAP_NR_SMALL_INT_BITS) ) {
        i = libGAP_INT_INTEXPR(expr);
        libGAP_PushExpr( libGAP_INTEXPR_INT( -i ) );
    }
    else {
        libGAP_PushExpr( expr );
        libGAP_PushUnaryOp( libGAP_T_AINV );
    }
}

void libGAP_CodeDiff ( void )
{
    libGAP_PushBinaryOp( libGAP_T_DIFF );
}

void libGAP_CodeProd ( void )
{
    libGAP_PushBinaryOp( libGAP_T_PROD );
}

void libGAP_CodeInv ( void )
{
    libGAP_PushUnaryOp( libGAP_T_INV );
}

void libGAP_CodeQuo ( void )
{
    libGAP_PushBinaryOp( libGAP_T_QUO );
}

void libGAP_CodeMod ( void )
{
    libGAP_PushBinaryOp( libGAP_T_MOD );
}

void libGAP_CodePow ( void )
{
    libGAP_PushBinaryOp( libGAP_T_POW );
}


/****************************************************************************
**
*F  CodeIntExpr( <str> )  . . . . . . . . . . code literal integer expression
**
**  'CodeIntExpr' is the action to code a literal integer expression.  <str>
**  is the integer as a (null terminated) C character string.
*/
void libGAP_CodeIntExpr (
    libGAP_Char *              str )
{
    libGAP_Expr                expr;           /* expression, result              */
    libGAP_Obj                 val;            /* value = <upp> * <pow> + <low>    */
    libGAP_Obj                 upp;            /* upper part                       */
    libGAP_Int                 pow;            /* power                            */
    libGAP_Int                 low;            /* lower part                       */
    libGAP_Int                 sign;           /* is the integer negative          */
    libGAP_UInt                i;              /* loop variable                    */

    /* get the signs, if any                                                */
    sign = 1;
    i = 0;
    while ( str[i] == '-' ) {
        sign = - sign;
        i++;
    }

    /* collect the digits in groups of 8                                    */
    low = 0;
    pow = 1;
    upp = libGAP_INTOBJ_INT(0);
    while ( str[i] != '\0' ) {
        low = 10 * low + str[i] - '0';
        pow = 10 * pow;
        if ( pow == 100000000L ) {
            upp = libGAP_SumInt( libGAP_ProdInt( upp, libGAP_INTOBJ_INT(pow) ),
                          libGAP_INTOBJ_INT(sign*low) );
            pow = 1;
            low = 0;
        }
        i++;
    }

    /* compose the integer value (set <val> first to silence 'lint')       */
    val = 0;
    if ( upp == libGAP_INTOBJ_INT(0) ) {
        val = libGAP_INTOBJ_INT(sign*low);
    }
    else if ( pow == 1 ) {
        val = upp;
    }
    else {
        val = libGAP_SumInt( libGAP_ProdInt( upp, libGAP_INTOBJ_INT(pow) ),
                      libGAP_INTOBJ_INT(sign*low) );
    }

    /* if it is small enough code it immediately                           */
    if ( libGAP_IS_INTOBJ(val) ) {
        expr = libGAP_INTEXPR_INT( libGAP_INT_INTOBJ(val) );
    }

    /* otherwise stuff the value into the values list                      */
    else {
        expr = libGAP_NewExpr( libGAP_T_INT_EXPR, sizeof(libGAP_UInt) + libGAP_SIZE_OBJ(val) );
        ((libGAP_UInt *)libGAP_ADDR_EXPR(expr))[0] = (libGAP_UInt)libGAP_TNUM_OBJ(val);
        memcpy((void *)((libGAP_UInt *)libGAP_ADDR_EXPR(expr)+1), (void *)libGAP_ADDR_OBJ(val), (size_t)libGAP_SIZE_OBJ(val));
    }

    /* push the expression                                                 */
    libGAP_PushExpr( expr );
}

/****************************************************************************
**
*F  CodeLongIntExpr( <str> )   . . . code literal long integer expression
**
**  'CodeIntExpr'  is  the  action  to   code  a  long  literal  integer
**  expression whose digits are stored in a string GAP object.
*/
void libGAP_CodeLongIntExpr (
    libGAP_Obj              string )
{
    libGAP_Expr                expr;           /* expression, result              */
    libGAP_Obj                 val;            /* value = <upp> * <pow> + <low>    */
    libGAP_Obj                 upp;            /* upper part                       */
    libGAP_Int                 pow;            /* power                            */
    libGAP_Int                 low;            /* lower part                       */
    libGAP_Int                 sign;           /* is the integer negative          */
    libGAP_UInt                i;              /* loop variable                    */
    libGAP_UChar *              str;

    /* get the signs, if any                                                */
    str = libGAP_CHARS_STRING(string);
    sign = 1;
    i = 0;
    while ( str[i] == '-' ) {
        sign = - sign;
        i++;
    }

    /* collect the digits in groups of 8                                    */
    low = 0;
    pow = 1;
    upp = libGAP_INTOBJ_INT(0);
    while ( str[i] != '\0' ) {
        low = 10 * low + str[i] - '0';
        pow = 10 * pow;
        if ( pow == 100000000L ) {
            upp = libGAP_SumInt( libGAP_ProdInt( upp, libGAP_INTOBJ_INT(pow) ),
                          libGAP_INTOBJ_INT(sign*low) );
            str = libGAP_CHARS_STRING(string);
            pow = 1;
            low = 0;
        }
        i++;
    }

    /* compose the integer value (set <val> first to silence 'lint')       */
    val = 0;
    if ( upp == libGAP_INTOBJ_INT(0) ) {
        val = libGAP_INTOBJ_INT(sign*low);
    }
    else if ( pow == 1 ) {
        val = upp;
    }
    else {
        val = libGAP_SumInt( libGAP_ProdInt( upp, libGAP_INTOBJ_INT(pow) ),
                      libGAP_INTOBJ_INT(sign*low) );
    }

    /* if it is small enough code it immediately                           */
    if ( libGAP_IS_INTOBJ(val) ) {
        expr = libGAP_INTEXPR_INT( libGAP_INT_INTOBJ(val) );
    }

    /* otherwise stuff the value into the values list                      */
    /* Need to fix this up for GMP integers */
    else {
        expr = libGAP_NewExpr( libGAP_T_INT_EXPR, sizeof(libGAP_UInt) + libGAP_SIZE_OBJ(val) );
        ((libGAP_UInt *)libGAP_ADDR_EXPR(expr))[0] = (libGAP_UInt)libGAP_TNUM_OBJ(val);
        memcpy((void *)((libGAP_UInt *)libGAP_ADDR_EXPR(expr)+1), (void *)libGAP_ADDR_OBJ(val), (size_t)libGAP_SIZE_OBJ(val));
    }

    /* push the expression                                                 */
    libGAP_PushExpr( expr );
}

/****************************************************************************
**
*F  CodeTrueExpr()  . . . . . . . . . . . . . .  code literal true expression
**
**  'CodeTrueExpr' is the action to code a literal true expression.
*/
void libGAP_CodeTrueExpr ( void )
{
    libGAP_PushExpr( libGAP_NewExpr( libGAP_T_TRUE_EXPR, 0L ) );
}


/****************************************************************************
**
*F  CodeFalseExpr() . . . . . . . . . . . . . . code literal false expression
**
**  'CodeFalseExpr' is the action to code a literal false expression.
*/
void libGAP_CodeFalseExpr ( void )
{
    libGAP_PushExpr( libGAP_NewExpr( libGAP_T_FALSE_EXPR, 0L ) );
}


/****************************************************************************
**
*F  CodeCharExpr( <chr> ) . . . . . . . . code a literal character expression
**
**  'CodeCharExpr'  is the action  to  code a  literal  character expression.
**  <chr> is the C character.
*/
void libGAP_CodeCharExpr (
    libGAP_Char                chr )
{
    libGAP_Expr                litr;           /* literal expression, result      */

    /* allocate the character expression                                   */
    litr = libGAP_NewExpr( libGAP_T_CHAR_EXPR, sizeof(libGAP_UChar) );
    ((libGAP_UChar*)libGAP_ADDR_EXPR(litr))[0] = chr;

    /* push the literal expression                                         */
    libGAP_PushExpr( litr );
}


/****************************************************************************
**
*F  CodePermCycle( <nrx>, <nrc> ) . . . . code literal permutation expression
*F  CodePerm( <nrc> ) . . . . . . . . . . code literal permutation expression
**
**  'CodePermCycle'  is an action to code  a  literal permutation expression.
**  It is called when one cycles is read completely.  <nrc>  is the number of
**  elements in that cycle.  <nrx> is the number of that  cycles (i.e., 1 for
**  the first cycle, 2 for the second, and so on).
**
**  'CodePerm' is an action to code a  literal permutation expression.  It is
**  called when  the permutation is read completely.   <nrc> is the number of
**  cycles.
*/
void libGAP_CodePermCycle (
    libGAP_UInt                nrx,
    libGAP_UInt                nrc )
{
    libGAP_Expr                cycle;          /* cycle, result                   */
    libGAP_Expr                entry;          /* entry of cycle                  */
    libGAP_UInt                j;              /* loop variable                   */

    /* allocate the new cycle                                              */
    cycle = libGAP_NewExpr( libGAP_T_PERM_CYCLE, nrx * sizeof(libGAP_Expr) );

    /* enter the entries                                                   */
    for ( j = nrx; 1 <= j; j-- ) {
        entry = libGAP_PopExpr();
        libGAP_ADDR_EXPR(cycle)[j-1] = entry;
    }

    /* push the cycle                                                      */
    libGAP_PushExpr( cycle );
}

void libGAP_CodePerm (
    libGAP_UInt                nrc )
{
    libGAP_Expr                perm;           /* permutation, result             */
    libGAP_Expr                cycle;          /* cycle of permutation            */
    libGAP_UInt                i;              /* loop variable                   */

    /* allocate the new permutation                                        */
    perm = libGAP_NewExpr( libGAP_T_PERM_EXPR, nrc * sizeof(libGAP_Expr) );

    /* enter the cycles                                                    */
    for ( i = nrc; 1 <= i; i-- ) {
        cycle = libGAP_PopExpr();
        libGAP_ADDR_EXPR(perm)[i-1] = cycle;
    }

    /* push the permutation                                                */
    libGAP_PushExpr( perm );

}


/****************************************************************************
**
*F  CodeListExprBegin( <top> )  . . . . . . . . . code list expression, begin
*F  CodeListExprBeginElm( <pos> ) . . . . code list expression, begin element
*F  CodeListExprEndElm()  . . . . . . .  .. code list expression, end element
*F  CodeListExprEnd( <nr>, <range>, <top>, <tilde> )  . . code list expr, end
*/
void libGAP_CodeListExprBegin (
    libGAP_UInt                top )
{
}

void libGAP_CodeListExprBeginElm (
    libGAP_UInt                pos )
{
    /* push the literal integer value                                      */
    libGAP_PushExpr( libGAP_INTEXPR_INT(pos) );
}

void libGAP_CodeListExprEndElm ( void )
{
}

void libGAP_CodeListExprEnd (
    libGAP_UInt                nr,
    libGAP_UInt                range,
    libGAP_UInt                top,
    libGAP_UInt                tilde )
{
    libGAP_Expr                list;           /* list, result                    */
    libGAP_Expr                entry;          /* entry                           */
    libGAP_Expr                pos;            /* position of an entry            */
    libGAP_UInt                i;              /* loop variable                   */

    /* peek at the last position (which is the largest)                    */
    if ( nr != 0 ) {
        entry = libGAP_PopExpr();
        pos   = libGAP_PopExpr();
        libGAP_PushExpr( pos );
        libGAP_PushExpr( entry );
    }
    else {
        pos = libGAP_INTEXPR_INT(0);
    }

    /* allocate the list expression                                        */
    if ( ! range && ! (top && tilde) ) {
        list = libGAP_NewExpr( libGAP_T_LIST_EXPR,      libGAP_INT_INTEXPR(pos) * sizeof(libGAP_Expr) );
    }
    else if ( ! range && (top && tilde) ) {
        list = libGAP_NewExpr( libGAP_T_LIST_TILD_EXPR, libGAP_INT_INTEXPR(pos) * sizeof(libGAP_Expr) );
    }
    else /* if ( range && ! (top && tilde) ) */ {
        list = libGAP_NewExpr( libGAP_T_RANGE_EXPR,     libGAP_INT_INTEXPR(pos) * sizeof(libGAP_Expr) );
    }

    /* enter the entries                                                   */
    for ( i = nr; 1 <= i; i-- ) {
        entry = libGAP_PopExpr();
        pos   = libGAP_PopExpr();
        libGAP_ADDR_EXPR(list)[ libGAP_INT_INTEXPR(pos)-1 ] = entry;
    }

    /* push the list                                                       */
    libGAP_PushExpr( list );
}


/****************************************************************************
**
*F  CodeStringExpr( <str> ) . . . . . . . .  code literal string expression
*/
void libGAP_CodeStringExpr (
    libGAP_Obj              str )
{
    libGAP_Expr                string;         /* string, result                  */

    /* allocate the string expression                                      */
    string = libGAP_NewExpr( libGAP_T_STRING_EXPR, libGAP_SIZEBAG_STRINGLEN(libGAP_GET_LEN_STRING(str)) );

    /* copy the string                                                     */
    memcpy( (void *)libGAP_ADDR_EXPR(string), libGAP_ADDR_OBJ(str), 
                        libGAP_SIZEBAG_STRINGLEN(libGAP_GET_LEN_STRING(str)) );

    /* push the string                                                     */
    libGAP_PushExpr( string );
}

/****************************************************************************
**
*F  CodeFloatExpr( <str> ) . . . . . . . .  code literal float expression
*/
#define libGAP_FLOAT_0_INDEX 1
#define libGAP_FLOAT_1_INDEX 2
#define libGAP_MAX_FLOAT_INDEX ((1L<<libGAP_NR_SMALL_INT_BITS)-2)

static libGAP_UInt libGAP_GVAR_SAVED_FLOAT_INDEX;
static libGAP_UInt libGAP_NextFloatExprNumber = 3;

static libGAP_UInt libGAP_NextEagerFloatLiteralNumber = 1;

static libGAP_Obj libGAP_EAGER_FLOAT_LITERAL_CACHE = 0;
static libGAP_Obj libGAP_CONVERT_FLOAT_LITERAL_EAGER;


static libGAP_UInt libGAP_getNextFloatExprNumber( void ) {
  libGAP_UInt next;
  libGAP_HashLock(&libGAP_NextFloatExprNumber);
  if (libGAP_NextFloatExprNumber > libGAP_MAX_FLOAT_INDEX)
    next = 0;
  else {
    next = libGAP_NextFloatExprNumber++;
  }
  libGAP_HashUnlock(&libGAP_NextFloatExprNumber);
  return next;
}

static libGAP_UInt libGAP_CheckForCommonFloat(libGAP_Char *str) {
  /* skip leading zeros */
  while (*str == '0')
    str++;
  if (*str == '.')
    /* might be zero literal */
    {
      /* skip point */
      str++;
      /* skip more zeroes */
      while (*str == '0')
        str++;
      /* if we've got to end of string we've got zero. */
      if (!libGAP_IsDigit(*str))
        return libGAP_FLOAT_0_INDEX;
    }
  if (*str++ !='1')
    return 0;
  /* might be one literal */
  if (*str++ != '.')
    return 0;
  /* skip zeros */
  while (*str == '0')
    str++;
  if (*str == '\0')
    return libGAP_FLOAT_1_INDEX;
  if (libGAP_IsDigit(*str))
    return 0;
  /* must now be an exponent character */
  assert(libGAP_IsAlpha(*str));
  /* skip it */
  str++;
  /*skip + and - in exponent */
  if (*str == '+' || *str == '-')
    str++;
  /* skip leading zeros in the exponent */
  while (*str == '0')
    str++;
  /* if there's anything but leading zeros this isn't 
     a one literal */
  if (*str == '\0')
    return libGAP_FLOAT_1_INDEX;
  else
    return 0;
}

static void libGAP_CodeLazyFloatExpr( libGAP_Char *str, libGAP_UInt len) {
    libGAP_UInt ix;

    /* Lazy case, store the string for conversion at run time */
    libGAP_Expr fl = libGAP_NewExpr( libGAP_T_FLOAT_EXPR_LAZY, 2*sizeof(libGAP_UInt) +len+1  );
    /* copy the string                                                     */
    memcpy( (void *)((char *)libGAP_ADDR_EXPR(fl)+2*sizeof(libGAP_UInt)), (void *)str, 
            len+1 );
      
    *(libGAP_UInt *)libGAP_ADDR_EXPR(fl) = len;
    ix = libGAP_CheckForCommonFloat(str);
    if (!ix) 
      ix = libGAP_getNextFloatExprNumber();
    ((libGAP_UInt *)libGAP_ADDR_EXPR(fl))[1] = ix;
    
    /* push the expression                                                     */
    libGAP_PushExpr( fl );
}

static void libGAP_CodeEagerFloatExpr( libGAP_Obj str, libGAP_Char mark ) {
  /* Eager case, do the conversion now */
  libGAP_UInt l = libGAP_GET_LEN_STRING(str);
  libGAP_Expr fl = libGAP_NewExpr( libGAP_T_FLOAT_EXPR_EAGER, sizeof(libGAP_UInt)* 3 + l + 1);
  libGAP_Obj v = libGAP_CALL_2ARGS(libGAP_CONVERT_FLOAT_LITERAL_EAGER, str, libGAP_ObjsChar[(libGAP_Int)mark]);
  assert(libGAP_EAGER_FLOAT_LITERAL_CACHE);
  assert(libGAP_IS_PLIST(libGAP_EAGER_FLOAT_LITERAL_CACHE));
  libGAP_GROW_PLIST(libGAP_EAGER_FLOAT_LITERAL_CACHE, libGAP_NextEagerFloatLiteralNumber);
  libGAP_SET_ELM_PLIST(libGAP_EAGER_FLOAT_LITERAL_CACHE, libGAP_NextEagerFloatLiteralNumber, v);
  libGAP_CHANGED_BAG(libGAP_EAGER_FLOAT_LITERAL_CACHE);
  libGAP_SET_LEN_PLIST(libGAP_EAGER_FLOAT_LITERAL_CACHE, libGAP_NextEagerFloatLiteralNumber);
  libGAP_ADDR_EXPR(fl)[0] = libGAP_NextEagerFloatLiteralNumber;
  libGAP_ADDR_EXPR(fl)[1] = l;
  libGAP_ADDR_EXPR(fl)[2] = (libGAP_UInt)mark;
  memcpy((void*)(libGAP_ADDR_EXPR(fl)+3), (void *)libGAP_CHARS_STRING(str), l+1);
  libGAP_NextEagerFloatLiteralNumber++;
  libGAP_PushExpr(fl);
}

void libGAP_CodeFloatExpr (
    libGAP_Char *              str )
{
  
  libGAP_UInt l = strlen(str);
  libGAP_UInt l1 = l;
  libGAP_Char mark = '\0'; /* initialize to please compilers */
  if (str[l-1] == '_' )
    {
      l1 = l-1;
      mark = '\0';
    }
  else if (str[l-2] == '_')
    {
      l1 = l-2;
      mark = str[l-1];
    }
  if (l1 < l)
    {
      libGAP_Obj s;
      libGAP_C_NEW_STRING(s, l1, str);
      libGAP_CodeEagerFloatExpr(s,mark);
    } else {
    libGAP_CodeLazyFloatExpr(str, l);
  }
}

/****************************************************************************
**
*F  CodeLongFloatExpr( <str> ) . . . . . . .code long literal float expression
*/

void libGAP_CodeLongFloatExpr (
    libGAP_Obj              str )
{
  libGAP_Char mark = '\0'; /* initialize to please compilers */

    /* allocate the float expression                                      */
    libGAP_UInt l = libGAP_GET_LEN_STRING(str);
    libGAP_UInt l1 = l;
    if (libGAP_CHARS_STRING(str)[l-1] == '_') {
      l1 = l-1;
      mark = '\0';
    } else if (libGAP_CHARS_STRING(str)[l-2] == '_') {
      l1 = l-2;
      mark = libGAP_CHARS_STRING(str)[l-1];
    }
    if (l1 < l) {
      libGAP_CHARS_STRING(str)[l1] = '\0';
      libGAP_SET_LEN_STRING(str,l1);
      libGAP_CodeEagerFloatExpr(str, mark);
    } else {
      libGAP_CodeLazyFloatExpr((libGAP_Char *)libGAP_CHARS_STRING(str), l);
    }
    
}


/****************************************************************************
**
*F  CodeRecExprBegin( <top> ) . . . . . . . . . . . . code record expr, begin
*F  CodeRecExprBeginElmName( <rnam> ) . . . . code record expr, begin element
*F  CodeRecExprBeginElmExpr() . . . . . . . . code record expr, begin element
*F  CodeRecExprEndElmExpr() . . . . . . . . . . code record expr, end element
*F  CodeRecExprEnd( <nr>, <top>, <tilde> )  . . . . . . code record expr, end
*/
void libGAP_CodeRecExprBegin (
    libGAP_UInt                top )
{
}

void libGAP_CodeRecExprBeginElmName (
    libGAP_UInt                rnam )
{
    /* push the record name as integer expressions                         */
    libGAP_PushExpr( libGAP_INTEXPR_INT( rnam ) );
}

void libGAP_CodeRecExprBeginElmExpr ( void )
{
    libGAP_Expr                expr;

    /* convert an integer expression to a record name                      */
    expr = libGAP_PopExpr();
    if ( libGAP_IS_INTEXPR(expr) ) {
        libGAP_PushExpr( libGAP_INTEXPR_INT( libGAP_RNamIntg( libGAP_INT_INTEXPR(expr) ) ) );
    }
    else {
        libGAP_PushExpr( expr );
    }
}

void libGAP_CodeRecExprEndElm ( void )
{
}

void libGAP_CodeRecExprEnd (
    libGAP_UInt                nr,
    libGAP_UInt                top,
    libGAP_UInt                tilde )
{
    libGAP_Expr                record;         /* record, result                  */
    libGAP_Expr                entry;          /* entry                           */
    libGAP_Expr                rnam;           /* position of an entry            */
    libGAP_UInt                i;              /* loop variable                   */

    /* allocate the record expression                                      */
    if ( ! (top && tilde) ) {
        record = libGAP_NewExpr( libGAP_T_REC_EXPR,      nr * 2 * sizeof(libGAP_Expr) );
    }
    else /* if ( (top && tilde) ) */ {
        record = libGAP_NewExpr( libGAP_T_REC_TILD_EXPR, nr * 2 * sizeof(libGAP_Expr) );
    }

    /* enter the entries                                                   */
    for ( i = nr; 1 <= i; i-- ) {
        entry = libGAP_PopExpr();
        rnam  = libGAP_PopExpr();
        libGAP_ADDR_EXPR(record)[2*(i-1)]   = rnam;
        libGAP_ADDR_EXPR(record)[2*(i-1)+1] = entry;
    }

    /* push the record                                                     */
    libGAP_PushExpr( record );
}


/****************************************************************************
**
*F  CodeAssLVar( <lvar> ) . . . . . . . . . . . . .  code assignment to local
**
**  'CodeAssLVar' is the action  to code an  assignment to the local variable
**  <lvar> (given  by its  index).  It is   called by the  reader *after* the
**  right hand side expression is read.
**
**  An assignment  to a  local variable  is   represented by a  bag with  two
**  subexpressions.  The  *first* is the local variable,  the *second* is the
**  right hand side expression.
*/
void libGAP_CodeAssLVar (
    libGAP_UInt                lvar )
{
    libGAP_Stat                ass;            /* assignment, result              */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /* allocate the assignment                                             */
    if ( lvar <= 16 ) {
        ass = libGAP_NewStat( libGAP_T_ASS_LVAR + lvar, 2 * sizeof(libGAP_Stat) );
    }
    else {
        ass = libGAP_NewStat( libGAP_T_ASS_LVAR,        2 * sizeof(libGAP_Stat) );
    }

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[1] = (libGAP_Stat)rhsx;

    /* enter the local variable                                            */
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)lvar;

    /* push the assignment                                                 */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeUnbLVar( <lvar> ) . . . . . . . . . . .  code unbind a local variable
*/
void libGAP_CodeUnbLVar (
    libGAP_UInt                lvar )
{
    libGAP_Stat                ass;            /* unbind, result                  */

    /* allocate the unbind                                                 */
    ass = libGAP_NewStat( libGAP_T_UNB_LVAR, sizeof(libGAP_Stat) );

    /* enter the local variable                                            */
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)lvar;

    /* push the unbind                                                     */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeRefLVar( <lvar> ) . . . . . . . . . . . . . . code reference to local
**
**  'CodeRefLVar' is  the action  to code a  reference  to the local variable
**  <lvar> (given  by its   index).  It is   called by  the  reader  when  it
**  encounters a local variable.
**
**  A   reference to   a local  variable    is represented immediately   (see
**  'REFLVAR_LVAR').
*/
void libGAP_CodeRefLVar (
    libGAP_UInt                lvar )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* make the reference                                                  */
    ref = libGAP_REFLVAR_LVAR(lvar);

    /* push the reference                                                  */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeIsbLVar( <lvar> ) . . . . . . . . . . code bound local variable check
*/
void libGAP_CodeIsbLVar (
    libGAP_UInt                lvar )
{
    libGAP_Expr                ref;            /* isbound, result                 */

    /* allocate the isbound                                                */
    ref = libGAP_NewExpr( libGAP_T_ISB_LVAR, sizeof(libGAP_Expr) );

    /* enter the local variable                                            */
    libGAP_ADDR_EXPR(ref)[0] = (libGAP_Expr)lvar;

    /* push the isbound                                                    */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeAssHVar( <hvar> ) . . . . . . . . . . . . . code assignment to higher
**
**  'CodeAssHVar' is the action to code an  assignment to the higher variable
**  <hvar> (given by its  level  and  index).  It  is  called by  the  reader
**  *after* the right hand side expression is read.
**
**  An assignment to a higher variable is represented by a statement bag with
**  two subexpressions.  The *first* is the higher  variable, the *second* is
**  the right hand side expression.
*/
void libGAP_CodeAssHVar (
    libGAP_UInt                hvar )
{
    libGAP_Stat                ass;            /* assignment, result              */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /* allocate the assignment                                             */
    ass = libGAP_NewStat( libGAP_T_ASS_HVAR, 2 * sizeof(libGAP_Stat) );

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[1] = (libGAP_Stat)rhsx;

    /* enter the higher variable                                           */
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)hvar;

    /* push the assignment                                                 */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeUnbHVar( <hvar> ) . . . . . . . . . . . . . . . code unbind of higher
*/
void libGAP_CodeUnbHVar (
    libGAP_UInt                hvar )
{
    libGAP_Stat                ass;            /* unbind, result                  */

    /* allocate the unbind                                                 */
    ass = libGAP_NewStat( libGAP_T_UNB_HVAR, sizeof(libGAP_Stat) );

    /* enter the higher variable                                           */
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)hvar;

    /* push the unbind                                                     */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeRefHVar( <hvar> ) . . . . . . . . . . . . .  code reference to higher
**
**  'CodeRefHVar' is the  action to code  a reference to the higher  variable
**  <hvar> (given by its level  and index).  It is  called by the reader when
**  it encounters a higher variable.
**
**  A reference to a higher variable is represented by an expression bag with
**  one subexpression.  This is the higher variable.
*/
void libGAP_CodeRefHVar (
    libGAP_UInt                hvar )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference                                              */
    ref = libGAP_NewExpr( libGAP_T_REF_HVAR, sizeof(libGAP_Expr) );

    /* enter the higher variable                                           */
    libGAP_ADDR_EXPR(ref)[0] = (libGAP_Expr)hvar;

    /* push the reference                                                  */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeIsbHVar( <hvar> ) . . . . . . . . . . . . . . code bound higher check
*/
void libGAP_CodeIsbHVar (
    libGAP_UInt                hvar )
{
    libGAP_Expr                ref;            /* isbound, result                 */

    /* allocate the isbound                                                */
    ref = libGAP_NewExpr( libGAP_T_ISB_HVAR, sizeof(libGAP_Expr) );

    /* enter the higher variable                                           */
    libGAP_ADDR_EXPR(ref)[0] = (libGAP_Expr)hvar;

    /* push the isbound                                                    */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeAssGVar( <gvar> ) . . . . . . . . . . . . . code assignment to global
**
**  'CodeAssGVar' is the action to code  an assignment to the global variable
**  <gvar>.  It is  called   by  the reader    *after* the right   hand  side
**  expression is read.
**
**  An assignment to a global variable is represented by a statement bag with
**  two subexpressions.  The *first* is the  global variable, the *second* is
**  the right hand side expression.
*/
void libGAP_CodeAssGVar (
    libGAP_UInt                gvar )
{
    libGAP_Stat                ass;            /* assignment, result              */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /*  allocate the assignment                                            */
    ass = libGAP_NewStat( libGAP_T_ASS_GVAR, 2 * sizeof(libGAP_Stat) );

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[1] = (libGAP_Stat)rhsx;

    /* enter the global variable                                           */
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)gvar;

    /* push the assignment                                                 */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeUnbGVar( <gvar> ) . . . . . . . . . . . . . . . code unbind of global
*/
void libGAP_CodeUnbGVar (
    libGAP_UInt                gvar )
{
    libGAP_Stat                ass;            /* unbind, result                  */

    /* allocate the unbind                                                 */
    ass = libGAP_NewStat( libGAP_T_UNB_GVAR, sizeof(libGAP_Stat) );

    /* enter the global variable                                           */
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)gvar;

    /* push the unbind                                                     */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeRefGVar( <gvar> ) . . . . . . . . . . . . .  code reference to global
**
**  'CodeRefGVar' is the  action to code a  reference to  the global variable
**  <gvar>.  It is called by the reader when it encounters a global variable.
**
**  A reference to a global variable is represented by an expression bag with
**  one subexpression.  This is the global variable.
*/
void libGAP_CodeRefGVar (
    libGAP_UInt                gvar )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference                                              */
    ref = libGAP_NewExpr( libGAP_T_REF_GVAR, sizeof(libGAP_Expr) );

    /* enter the global variable                                           */
    libGAP_ADDR_EXPR(ref)[0] = (libGAP_Expr)gvar;

    /* push the reference                                                  */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeIsbGVar( <gvar> ) . . . . . . . . . . . . . . code bound global check
*/
void libGAP_CodeIsbGVar (
    libGAP_UInt                gvar )
{
    libGAP_Expr                ref;            /* isbound, result                 */

    /* allocate the isbound                                                */
    ref = libGAP_NewExpr( libGAP_T_ISB_GVAR, sizeof(libGAP_Expr) );

    /* enter the global variable                                           */
    libGAP_ADDR_EXPR(ref)[0] = (libGAP_Expr)gvar;

    /* push the isbound                                                    */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeAssList() . . . . . . . . . . . . . . . . . code assignment to a list
*F  CodeAsssList()  . . . . . . . . . . .  code multiple assignment to a list
*F  CodeAssListLevel( <level> ) . . . . . .  code assignment to several lists
*F  CodeAsssListLevel( <level> )  . code multiple assignment to several lists
*/
void libGAP_CodeAssListUniv ( 
		      libGAP_Stat                ass,
		       libGAP_Int narg)
{
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */
    libGAP_Expr                rhsx;           /* right hand side expression      */
    libGAP_Int i;

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[narg+1] = (libGAP_Stat)rhsx;

    /* enter the position expression                                       */
    for (i = narg; i > 0; i--) {
      pos = libGAP_PopExpr();
      libGAP_ADDR_STAT(ass)[i] = (libGAP_Stat)pos;
    }

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)list;

    /* push the assignment                                                 */
    libGAP_PushStat( ass );
}

void libGAP_CodeAssList ( libGAP_Int narg )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment                                             */
    switch (narg) {
    case 1:
      ass = libGAP_NewStat( libGAP_T_ASS_LIST, 3 * sizeof(libGAP_Stat) );
      break;

    case 2:
      ass = libGAP_NewStat(libGAP_T_ASS2_LIST, 4* sizeof(libGAP_Stat));
      break;
    default:
      ass  = libGAP_NewStat(libGAP_T_ASSX_LIST, (narg + 2)*sizeof(libGAP_Stat));
    }

    /* let 'CodeAssListUniv' do the rest                                   */
    libGAP_CodeAssListUniv( ass, narg );
}

void libGAP_CodeAsssList ( void )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment                                             */
    ass = libGAP_NewStat( libGAP_T_ASSS_LIST, 3 * sizeof(libGAP_Stat) );

    /* let 'CodeAssListUniv' do the rest                                   */
    libGAP_CodeAssListUniv( ass, 1 );
}

void libGAP_CodeAssListLevel ( libGAP_Int narg,
    libGAP_UInt                level )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment and enter the level                         */
    ass = libGAP_NewStat( libGAP_T_ASS_LIST_LEV, (narg +3) * sizeof(libGAP_Stat) );
    libGAP_ADDR_STAT(ass)[narg+2] = (libGAP_Stat)level;

    /* let 'CodeAssListUniv' do the rest                                   */
    libGAP_CodeAssListUniv( ass, narg );
}

void libGAP_CodeAsssListLevel (
    libGAP_UInt                level )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment and enter the level                         */
    ass = libGAP_NewStat( libGAP_T_ASSS_LIST_LEV, 4 * sizeof(libGAP_Stat) );
    libGAP_ADDR_STAT(ass)[3] = (libGAP_Stat)level;

    /* let 'CodeAssListUniv' do the rest                                   */
    libGAP_CodeAssListUniv( ass, 1 );
}


/****************************************************************************
**
*F  CodeUnbList() . . . . . . . . . . . . . . .  code unbind of list position
*/
void libGAP_CodeUnbList ( libGAP_Int narg )
{
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */
    libGAP_Stat                ass;            /* unbind, result                  */
    libGAP_Int i;

    /* allocate the unbind                                                 */
    ass = libGAP_NewStat( libGAP_T_UNB_LIST, (narg+1) * sizeof(libGAP_Stat) );

    /* enter the position expressions                                       */
    for (i = narg; i > 0; i--) {
      pos = libGAP_PopExpr();
      libGAP_ADDR_STAT(ass)[i] = (libGAP_Stat)pos;
    }

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)list;

    /* push the unbind                                                     */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeElmList() . . . . . . . . . . . . . . . . .  code selection of a list
*F  CodeElmsList()  . . . . . . . . . . . . code multiple selection of a list
*F  CodeElmListLevel( <level> ) . . . . . . . code selection of several lists
*F  CodeElmsListLevel( <level> )  .  code multiple selection of several lists
*/
void libGAP_CodeElmListUniv (
		      libGAP_Expr                ref,
		      libGAP_Int narg)
{
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */
    libGAP_Int                i;

    /* enter the position expression                                       */

    for (i = narg; i > 0; i--) {
      pos = libGAP_PopExpr();
      libGAP_ADDR_EXPR(ref)[i] = pos;
    }

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_EXPR(ref)[0] = list;

    /* push the reference                                                  */
    libGAP_PushExpr( ref );
}

void libGAP_CodeElmList ( libGAP_Int narg )
{
    libGAP_Expr                ref;            /* reference, result               */

      /* allocate the reference                                              */
    if (narg == 1)
      ref = libGAP_NewExpr( libGAP_T_ELM_LIST, 2 * sizeof(libGAP_Expr) );
    else if (narg == 2)
      ref = libGAP_NewExpr( libGAP_T_ELM2_LIST, 3 * sizeof(libGAP_Expr) );
    else
      ref = libGAP_NewExpr( libGAP_T_ELMX_LIST, (narg + 1) *sizeof(libGAP_Expr));
      
      /* let 'CodeElmListUniv' to the rest                                   */
    libGAP_CodeElmListUniv( ref, narg );
      
}

void libGAP_CodeElmsList ( void )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference                                              */
    ref = libGAP_NewExpr( libGAP_T_ELMS_LIST, 2 * sizeof(libGAP_Expr) );

    /* let 'CodeElmListUniv' to the rest                                   */
    libGAP_CodeElmListUniv( ref, 1 );
}

void libGAP_CodeElmListLevel ( libGAP_Int narg,
    libGAP_UInt                level )
{
    libGAP_Expr                ref;            /* reference, result               */

    ref = libGAP_NewExpr( libGAP_T_ELM_LIST_LEV, (narg+2)*sizeof(libGAP_Expr));
    libGAP_ADDR_EXPR(ref)[narg+1] = (libGAP_Stat)level;
      

    /* let 'CodeElmListUniv' do the rest                                   */
    libGAP_CodeElmListUniv( ref, narg );
}

void libGAP_CodeElmsListLevel (
    libGAP_UInt                level )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference and enter the level                          */
    ref = libGAP_NewExpr( libGAP_T_ELMS_LIST_LEV, 3 * sizeof(libGAP_Expr) );
    libGAP_ADDR_EXPR(ref)[2] = (libGAP_Stat)level;

    /* let 'CodeElmListUniv' do the rest                                   */
    libGAP_CodeElmListUniv( ref, 1 );
}


/****************************************************************************
**
*F  CodeIsbList() . . . . . . . . . . . . . .  code bound list position check
*/
void libGAP_CodeIsbList ( libGAP_Int narg )
{
    libGAP_Expr                ref;            /* isbound, result                 */
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */
    libGAP_Int i;

    /* allocate the isbound                                                */
    ref = libGAP_NewExpr( libGAP_T_ISB_LIST, (narg + 1) * sizeof(libGAP_Expr) );

    /* enter the position expression                                       */
    for (i = narg; i > 0; i--) {
      pos = libGAP_PopExpr();
      libGAP_ADDR_EXPR(ref)[i] = pos;
    }

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_EXPR(ref)[0] = list;

    /* push the isbound                                                    */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeAssRecName( <rnam> )  . . . . . . . . . . code assignment to a record
*F  CodeAssRecExpr()  . . . . . . . . . . . . . . code assignment to a record
*/
void            libGAP_CodeAssRecName (
    libGAP_UInt                rnam )
{
    libGAP_Stat                stat;           /* assignment, result              */
    libGAP_Expr                rec;            /* record expression               */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /* allocate the assignment                                             */
    stat = libGAP_NewStat( libGAP_T_ASS_REC_NAME, 3 * sizeof(libGAP_Stat) );

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[2] = (libGAP_Stat)rhsx;

    /* enter the name                                                      */
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the assignment                                                 */
    libGAP_PushStat( stat );
}

void            libGAP_CodeAssRecExpr ( void )
{
    libGAP_Stat                stat;           /* assignment, result              */
    libGAP_Expr                rec;            /* record expression               */
    libGAP_Expr                rnam;           /* name expression                 */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /* allocate the assignment                                             */
    stat = libGAP_NewStat( libGAP_T_ASS_REC_EXPR, 3 * sizeof(libGAP_Stat) );

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[2] = (libGAP_Stat)rhsx;

    /* enter the name expression                                           */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the assignment                                                 */
    libGAP_PushStat( stat );
}

void            libGAP_CodeUnbRecName (
    libGAP_UInt                rnam )
{
    libGAP_Stat                stat;           /* unbind, result                  */
    libGAP_Expr                rec;            /* record expression               */

    /* allocate the unbind                                                 */
    stat = libGAP_NewStat( libGAP_T_UNB_REC_NAME, 2 * sizeof(libGAP_Stat) );

    /* enter the name                                                      */
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the unbind                                                     */
    libGAP_PushStat( stat );
}

void            libGAP_CodeUnbRecExpr ( void )
{
    libGAP_Stat                stat;           /* unbind, result                  */
    libGAP_Expr                rec;            /* record expression               */
    libGAP_Expr                rnam;           /* name expression                 */

    /* allocate the unbind                                                 */
    stat = libGAP_NewStat( libGAP_T_UNB_REC_EXPR, 2 * sizeof(libGAP_Stat) );

    /* enter the name expression                                           */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the unbind                                                     */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeElmRecName( <rnam> )  . . . . . . . . . .  code selection of a record
*F  CodeElmRecExpr()  . . . . . . . . . . . . . .  code selection of a record
*/
void libGAP_CodeElmRecName (
    libGAP_UInt                rnam )
{
    libGAP_Expr                expr;           /* reference, result               */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the reference                                              */
    expr = libGAP_NewExpr( libGAP_T_ELM_REC_NAME, 2 * sizeof(libGAP_Expr) );

    /* enter the name                                                      */
    libGAP_ADDR_EXPR(expr)[1] = (libGAP_Expr)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the reference                                                  */
    libGAP_PushExpr( expr );
}

void libGAP_CodeElmRecExpr ( void )
{
    libGAP_Expr                expr;           /* reference, result               */
    libGAP_Expr                rnam;           /* name expression                 */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the reference                                              */
    expr = libGAP_NewExpr( libGAP_T_ELM_REC_EXPR, 2 * sizeof(libGAP_Expr) );

    /* enter the expression                                                */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[1] = rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the reference                                                  */
    libGAP_PushExpr( expr );
}


/****************************************************************************
**
*F  CodeIsbRecName( <rnam> )  . . . . . . . . . . . code bound rec name check
*/
void libGAP_CodeIsbRecName (
    libGAP_UInt                rnam )
{
    libGAP_Expr                expr;           /* isbound, result                 */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the isbound                                                */
    expr = libGAP_NewExpr( libGAP_T_ISB_REC_NAME, 2 * sizeof(libGAP_Expr) );

    /* enter the name                                                      */
    libGAP_ADDR_EXPR(expr)[1] = (libGAP_Expr)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the isbound                                                    */
    libGAP_PushExpr( expr );
}


/****************************************************************************
**
*F  CodeIsbRecExpr()  . . . . . . . . . . . . . . . code bound rec expr check
*/
void libGAP_CodeIsbRecExpr ( void )
{
    libGAP_Expr                expr;           /* reference, result               */
    libGAP_Expr                rnam;           /* name expression                 */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the isbound                                                */
    expr = libGAP_NewExpr( libGAP_T_ISB_REC_EXPR, 2 * sizeof(libGAP_Expr) );

    /* enter the expression                                                */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[1] = rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the isbound                                                    */
    libGAP_PushExpr( expr );
}


/****************************************************************************
**
*F  CodeAssPosObj() . . . . . . . . . . . . . . . . code assignment to a list
*F  CodeAsssPosObj()  . . . . . . . . . .  code multiple assignment to a list
*F  CodeAssPosObjLevel( <level> ) . . . . .  code assignment to several lists
*F  CodeAsssPosObjLevel( <level> )  code multiple assignment to several lists
*/
void libGAP_CodeAssPosObjUniv (
    libGAP_Stat                ass )
{
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[2] = (libGAP_Stat)rhsx;

    /* enter the position expression                                       */
    pos = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[1] = (libGAP_Stat)pos;

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)list;

    /* push the assignment                                                 */
    libGAP_PushStat( ass );
}

void libGAP_CodeAssPosObj ( void )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment                                             */
    ass = libGAP_NewStat( libGAP_T_ASS_POSOBJ, 3 * sizeof(libGAP_Stat) );

    /* let 'CodeAssPosObjUniv' do the rest                                 */
    libGAP_CodeAssPosObjUniv( ass );
}

void libGAP_CodeAsssPosObj ( void )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment                                             */
    ass = libGAP_NewStat( libGAP_T_ASSS_POSOBJ, 3 * sizeof(libGAP_Stat) );

    /* let 'CodeAssPosObjUniv' do the rest                                 */
    libGAP_CodeAssPosObjUniv( ass );
}

void libGAP_CodeAssPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment and enter the level                         */
    ass = libGAP_NewStat( libGAP_T_ASS_POSOBJ_LEV, 4 * sizeof(libGAP_Stat) );
    libGAP_ADDR_STAT(ass)[3] = (libGAP_Stat)level;

    /* let 'CodeAssPosObjUniv' do the rest                                 */
    libGAP_CodeAssPosObjUniv( ass );
}

void libGAP_CodeAsssPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Stat                ass;            /* assignment, result              */

    /* allocate the assignment and enter the level                         */
    ass = libGAP_NewStat( libGAP_T_ASSS_POSOBJ_LEV, 4 * sizeof(libGAP_Stat) );
    libGAP_ADDR_STAT(ass)[3] = (libGAP_Stat)level;

    /* let 'CodeAssPosObjUniv' do the rest                                 */
    libGAP_CodeAssPosObjUniv( ass );
}


/****************************************************************************
**
*F  CodeUnbPosObj() . . . . . . . . . . . . . . . . .  code unbind pos object
*/
void libGAP_CodeUnbPosObj ( void )
{
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */
    libGAP_Stat                ass;            /* unbind, result                  */

    /* allocate the unbind                                                 */
    ass = libGAP_NewStat( libGAP_T_UNB_POSOBJ, 2 * sizeof(libGAP_Stat) );

    /* enter the position expression                                       */
    pos = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[1] = (libGAP_Stat)pos;

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_STAT(ass)[0] = (libGAP_Stat)list;

    /* push the unbind                                                     */
    libGAP_PushStat( ass );
}


/****************************************************************************
**
*F  CodeElmPosObj() . . . . . . . . . . . . . . . .  code selection of a list
*F  CodeElmsPosObj()  . . . . . . . . . . . code multiple selection of a list
*F  CodeElmPosObjLevel( <level> ) . . . . . . code selection of several lists
*F  CodeElmsPosObjLevel( <level> )   code multiple selection of several lists
*/
void libGAP_CodeElmPosObjUniv (
    libGAP_Expr                ref )
{
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */

    /* enter the position expression                                       */
    pos = libGAP_PopExpr();
    libGAP_ADDR_EXPR(ref)[1] = pos;

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_EXPR(ref)[0] = list;

    /* push the reference                                                  */
    libGAP_PushExpr( ref );
}

void libGAP_CodeElmPosObj ( void )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference                                              */
    ref = libGAP_NewExpr( libGAP_T_ELM_POSOBJ, 2 * sizeof(libGAP_Expr) );

    /* let 'CodeElmPosObjUniv' to the rest                                   */
    libGAP_CodeElmPosObjUniv( ref );
}

void libGAP_CodeElmsPosObj ( void )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference                                              */
    ref = libGAP_NewExpr( libGAP_T_ELMS_POSOBJ, 2 * sizeof(libGAP_Expr) );

    /* let 'CodeElmPosObjUniv' to the rest                                   */
    libGAP_CodeElmPosObjUniv( ref );
}

void libGAP_CodeElmPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference and enter the level                          */
    ref = libGAP_NewExpr( libGAP_T_ELM_POSOBJ_LEV, 3 * sizeof(libGAP_Expr) );
    libGAP_ADDR_EXPR(ref)[2] = (libGAP_Stat)level;

    /* let 'CodeElmPosObjUniv' do the rest                                 */
    libGAP_CodeElmPosObjUniv( ref );
}

void libGAP_CodeElmsPosObjLevel (
    libGAP_UInt                level )
{
    libGAP_Expr                ref;            /* reference, result               */

    /* allocate the reference and enter the level                          */
    ref = libGAP_NewExpr( libGAP_T_ELMS_POSOBJ_LEV, 3 * sizeof(libGAP_Expr) );
    libGAP_ADDR_EXPR(ref)[2] = (libGAP_Stat)level;

    /* let 'CodeElmPosObjUniv' do the rest                                 */
    libGAP_CodeElmPosObjUniv( ref );
}


/****************************************************************************
**
*F  CodeIsbPosObj() . . . . . . . . . . . . . . . code bound pos object check
*/
void libGAP_CodeIsbPosObj ( void )
{
    libGAP_Expr                ref;            /* isbound, result                 */
    libGAP_Expr                list;           /* list expression                 */
    libGAP_Expr                pos;            /* position expression             */

    /* allocate the isbound                                                */
    ref = libGAP_NewExpr( libGAP_T_ISB_POSOBJ, 2 * sizeof(libGAP_Expr) );

    /* enter the position expression                                       */
    pos = libGAP_PopExpr();
    libGAP_ADDR_EXPR(ref)[1] = pos;

    /* enter the list expression                                           */
    list = libGAP_PopExpr();
    libGAP_ADDR_EXPR(ref)[0] = list;

    /* push the isbound                                                    */
    libGAP_PushExpr( ref );
}


/****************************************************************************
**
*F  CodeAssComObjName( <rnam> ) . . . . . . . . . code assignment to a record
*F  CodeAssComObjExpr() . . . . . . . . . . . . . code assignment to a record
*/
void            libGAP_CodeAssComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Stat                stat;           /* assignment, result              */
    libGAP_Expr                rec;            /* record expression               */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /* allocate the assignment                                             */
    stat = libGAP_NewStat( libGAP_T_ASS_COMOBJ_NAME, 3 * sizeof(libGAP_Stat) );

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[2] = (libGAP_Stat)rhsx;

    /* enter the name                                                      */
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the assignment                                                 */
    libGAP_PushStat( stat );
}

void            libGAP_CodeAssComObjExpr ( void )
{
    libGAP_Stat                stat;           /* assignment, result              */
    libGAP_Expr                rec;            /* record expression               */
    libGAP_Expr                rnam;           /* name expression                 */
    libGAP_Expr                rhsx;           /* right hand side expression      */

    /* allocate the assignment                                             */
    stat = libGAP_NewStat( libGAP_T_ASS_COMOBJ_EXPR, 3 * sizeof(libGAP_Stat) );

    /* enter the right hand side expression                                */
    rhsx = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[2] = (libGAP_Stat)rhsx;

    /* enter the name expression                                           */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the assignment                                                 */
    libGAP_PushStat( stat );
}

void            libGAP_CodeUnbComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Stat                stat;           /* unbind, result                  */
    libGAP_Expr                rec;            /* record expression               */

    /* allocate the unbind                                                 */
    stat = libGAP_NewStat( libGAP_T_UNB_COMOBJ_NAME, 2 * sizeof(libGAP_Stat) );

    /* enter the name                                                      */
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the unbind                                                     */
    libGAP_PushStat( stat );
}

void            libGAP_CodeUnbComObjExpr ( void )
{
    libGAP_Stat                stat;           /* unbind, result                  */
    libGAP_Expr                rec;            /* record expression               */
    libGAP_Expr                rnam;           /* name expression                 */

    /* allocate the unbind                                                 */
    stat = libGAP_NewStat( libGAP_T_UNB_COMOBJ_EXPR, 2 * sizeof(libGAP_Stat) );

    /* enter the name expression                                           */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[1] = (libGAP_Stat)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_STAT(stat)[0] = (libGAP_Stat)rec;

    /* push the unbind                                                     */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeElmComObjName( <rnam> ) . . . . . . . . .  code selection of a record
*F  CodeElmComObjExpr() . . . . . . . . . . . . .  code selection of a record
*/
void libGAP_CodeElmComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Expr                expr;           /* reference, result               */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the reference                                              */
    expr = libGAP_NewExpr( libGAP_T_ELM_COMOBJ_NAME, 2 * sizeof(libGAP_Expr) );

    /* enter the name                                                      */
    libGAP_ADDR_EXPR(expr)[1] = (libGAP_Expr)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the reference                                                  */
    libGAP_PushExpr( expr );
}

void libGAP_CodeElmComObjExpr ( void )
{
    libGAP_Expr                expr;           /* reference, result               */
    libGAP_Expr                rnam;           /* name expression                 */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the reference                                              */
    expr = libGAP_NewExpr( libGAP_T_ELM_COMOBJ_EXPR, 2 * sizeof(libGAP_Expr) );

    /* enter the expression                                                */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[1] = rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the reference                                                  */
    libGAP_PushExpr( expr );
}


/****************************************************************************
**
*F  CodeIsbComObjName( <rname> )  . . . . .  code bound com object name check
*/
void libGAP_CodeIsbComObjName (
    libGAP_UInt                rnam )
{
    libGAP_Expr                expr;           /* isbound, result                 */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the isbound                                                */
    expr = libGAP_NewExpr( libGAP_T_ISB_COMOBJ_NAME, 2 * sizeof(libGAP_Expr) );

    /* enter the name                                                      */
    libGAP_ADDR_EXPR(expr)[1] = (libGAP_Expr)rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the isbound                                                    */
    libGAP_PushExpr( expr );
}

/****************************************************************************
**
*F  CodeIsbComObjExpr() . . . . . . . . . .  code bound com object expr check
*/
void libGAP_CodeIsbComObjExpr ( void )
{
    libGAP_Expr                expr;           /* reference, result               */
    libGAP_Expr                rnam;           /* name expression                 */
    libGAP_Expr                rec;            /* record expresion                */

    /* allocate the isbound                                                */
    expr = libGAP_NewExpr( libGAP_T_ISB_COMOBJ_EXPR, 2 * sizeof(libGAP_Expr) );

    /* enter the expression                                                */
    rnam = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[1] = rnam;

    /* enter the record expression                                         */
    rec = libGAP_PopExpr();
    libGAP_ADDR_EXPR(expr)[0] = rec;

    /* push the isbound                                                    */
    libGAP_PushExpr( expr );
}


/****************************************************************************
**
*F  CodeEmpty()  . . . . code an empty statement
**
*/

extern void libGAP_CodeEmpty( void )
{
  libGAP_Stat stat;
  stat = libGAP_NewStat(libGAP_T_EMPTY, 0);
  libGAP_PushStat( stat );
}

/****************************************************************************
**
*F  CodeInfoBegin() . . . . . . . . . . . . .  start coding of Info statement
*F  CodeInfoMiddle()  . . . . . . . . .   shift to coding printable arguments
*F  CodeInfoEnd( <narg> ) . . Info statement complete, <narg> things to print
**
**  These  actions deal  with the  Info  statement, which is coded specially,
**  because not all of its arguments are always evaluated.
**
**  Only CodeInfoEnd actually does anything
*/
void libGAP_CodeInfoBegin ( void )
{
}

void libGAP_CodeInfoMiddle ( void )
{
}

void libGAP_CodeInfoEnd   (
    libGAP_UInt                narg )
{
    libGAP_Stat                stat;           /* we build the statement here     */
    libGAP_Expr                expr;           /* expression                      */
    libGAP_UInt                i;              /* loop variable                   */

    /* allocate the new statement                                          */
    stat = libGAP_NewStat( libGAP_T_INFO, libGAP_SIZE_NARG_INFO(2+narg) );

    /* narg only counts the printable arguments                            */
    for ( i = narg + 2; 0 < i; i-- ) {
        expr = libGAP_PopExpr();
        libGAP_ARGI_INFO( stat, i ) = expr;
    }

    /* push the statement                                                  */
    libGAP_PushStat( stat );
}


/****************************************************************************
**
*F  CodeAssertBegin() . . . . . . .  start interpretation of Assert statement
*F  CodeAsseerAfterLevel()  . . called after the first argument has been read
*F  CodeAssertAfterCondition() called after the second argument has been read
*F  CodeAssertEnd2Args() . . . . called after reading the closing parenthesis
*F  CodeAssertEnd3Args() . . . . called after reading the closing parenthesis
**
**  Only the End functions actually do anything
*/
void libGAP_CodeAssertBegin ( void )
{
}

void libGAP_CodeAssertAfterLevel ( void )
{
}

void libGAP_CodeAssertAfterCondition ( void )
{
}

void libGAP_CodeAssertEnd2Args ( void )
{
    libGAP_Stat                stat;           /* we build the statement here     */

    stat = libGAP_NewStat( libGAP_T_ASSERT_2ARGS, 2*sizeof(libGAP_Expr) );

    libGAP_ADDR_STAT(stat)[1] = libGAP_PopExpr(); /* condition */
    libGAP_ADDR_STAT(stat)[0] = libGAP_PopExpr(); /* level */

    libGAP_PushStat( stat );
}

void libGAP_CodeAssertEnd3Args ( void )
{
    libGAP_Stat                stat;           /* we build the statement here     */

    stat = libGAP_NewStat( libGAP_T_ASSERT_3ARGS, 3*sizeof(libGAP_Expr) );

    libGAP_ADDR_STAT(stat)[2] = libGAP_PopExpr(); /* message */
    libGAP_ADDR_STAT(stat)[1] = libGAP_PopExpr(); /* condition */
    libGAP_ADDR_STAT(stat)[0] = libGAP_PopExpr(); /* level */

    libGAP_PushStat( stat );
}

/****************************************************************************
**
*F  SaveBody( <body> ) . . . . . . . . . . . . . . .  workspace saving method
**
**  A body is made up of statements and expressions, and these are all
**  organised to regular boundaries based on the types Stat and Expr, which
**  are currently both UInt
**
**  String literals should really be saved byte-wise, to be safe across machines
**  of different endianness, but this would mean parsing the bag as we save it
**  which it would be nice to avoid just now.
*/
void libGAP_SaveBody ( libGAP_Obj body )
{
  libGAP_UInt i;
  libGAP_UInt *ptr;
  ptr = (libGAP_UInt *) libGAP_ADDR_OBJ(body);
  /* Save the new inforation in the body */
  for (i =0; i < libGAP_NUMBER_HEADER_ITEMS_BODY; i++)
    libGAP_SaveSubObj((libGAP_Obj)(*ptr++));
  /* and the rest */
  for (; i < (libGAP_SIZE_OBJ(body)+sizeof(libGAP_UInt)-1)/sizeof(libGAP_UInt); i++)
    libGAP_SaveUInt(*ptr++);
}

/****************************************************************************
**
*F  LoadBody( <body> ) . . . . . . . . . . . . . . . workspace loading method
**
**  A body is made up of statements and expressions, and these are all
**  organised to regular boundaries based on the types Stat and Expr, which
**  are currently both UInt
**
*/
void libGAP_LoadBody ( libGAP_Obj body )
{
  libGAP_UInt i;
  libGAP_UInt *ptr;
  ptr = (libGAP_UInt *) libGAP_ADDR_OBJ(body);
  for (i =0; i < libGAP_NUMBER_HEADER_ITEMS_BODY; i++)
    *(libGAP_Obj *)(ptr++) = libGAP_LoadSubObj();
  for (; i < (libGAP_SIZE_OBJ(body)+sizeof(libGAP_UInt)-1)/sizeof(libGAP_UInt); i++)
    *ptr++ = libGAP_LoadUInt();
}


/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* install the marking functions for function body bags                */
    libGAP_InfoBags[ libGAP_T_BODY ].name = "function body bag";
    libGAP_InitMarkFuncBags( libGAP_T_BODY, libGAP_MarkThreeSubBags );

    libGAP_SaveObjFuncs[ libGAP_T_BODY ] = libGAP_SaveBody;
    libGAP_LoadObjFuncs[ libGAP_T_BODY ] = libGAP_LoadBody;

    /* Allocate function bodies in the public data space */
    libGAP_MakeBagTypePublic(libGAP_T_BODY);

    /* make the result variable known to Gasman                            */
    libGAP_InitGlobalBag( &libGAP_CodeResult, "CodeResult" );
    
    libGAP_InitGlobalBag( &libGAP_FilenameCache, "FilenameCache" );

    /* allocate the statements and expressions stacks                      */
    libGAP_InitGlobalBag( &libGAP_TLS(libGAP_StackStat), "TLS(StackStat)" );
    libGAP_InitGlobalBag( &libGAP_TLS(libGAP_StackExpr), "TLS(StackExpr)" );

    /* some functions and globals needed for float conversion */
    libGAP_InitCopyGVar( "EAGER_FLOAT_LITERAL_CACHE", &libGAP_EAGER_FLOAT_LITERAL_CACHE);
    libGAP_InitFopyGVar( "CONVERT_FLOAT_LITERAL_EAGER", &libGAP_CONVERT_FLOAT_LITERAL_EAGER);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_UInt gv;
  libGAP_Obj cache;
    /* allocate the statements and expressions stacks                      */
    libGAP_TLS(libGAP_StackStat) = libGAP_NewBag( libGAP_T_BODY, 64*sizeof(libGAP_Stat) );
    libGAP_TLS(libGAP_StackExpr) = libGAP_NewBag( libGAP_T_BODY, 64*sizeof(libGAP_Expr) );
    libGAP_FilenameCache = libGAP_NEW_PLIST(libGAP_T_PLIST, 0);

    libGAP_GVAR_SAVED_FLOAT_INDEX = libGAP_GVarName("SavedFloatIndex");
    
    gv = libGAP_GVarName("EAGER_FLOAT_LITERAL_CACHE");
    cache = libGAP_NEW_PLIST(libGAP_T_PLIST+libGAP_IMMUTABLE, 1000L);
    libGAP_SET_LEN_PLIST(cache,0);
    libGAP_AssGVar(gv, cache);

    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . .  recover
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_GVAR_SAVED_FLOAT_INDEX = libGAP_GVarName("SavedFloatIndex");
  libGAP_NextFloatExprNumber = libGAP_INT_INTOBJ(libGAP_VAL_GVAR(libGAP_GVAR_SAVED_FLOAT_INDEX));
  return 0;
}


/****************************************************************************
**
*F  PreSave( <module> ) . . . . . . .  clean up before saving
*/
static libGAP_Int libGAP_PreSave (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_UInt i;

  /* Can't save in mid-parsing */
  if (libGAP_TLS(libGAP_CountExpr) || libGAP_TLS(libGAP_CountStat))
    return 1;

  /* push the FP cache index out into a GAP Variable */
  libGAP_AssGVar(libGAP_GVAR_SAVED_FLOAT_INDEX, libGAP_INTOBJ_INT(libGAP_NextFloatExprNumber));

  /* clean any old data out of the statement and expression stacks */
  for (i = 0; i < libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackStat))/sizeof(libGAP_UInt); i++)
    libGAP_ADDR_OBJ(libGAP_TLS(libGAP_StackStat))[i] = (libGAP_Obj)0;
  for (i = 0; i < libGAP_SIZE_BAG(libGAP_TLS(libGAP_StackExpr))/sizeof(libGAP_UInt); i++)
    libGAP_ADDR_OBJ(libGAP_TLS(libGAP_StackExpr))[i] = (libGAP_Obj)0;
  /* return success                                                      */
  return 0;
}



/****************************************************************************
**
*F  InitInfoCode()  . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "code",                             /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                        /* checkInit                      */
    libGAP_PreSave,                            /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                        /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoCode ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  code.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/



