/****************************************************************************
**
*W  blister.c                   GAP source                       Frank Celler
*W                                                         & Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This  file contains the functions  that mainly operate  on boolean lists.
**  Because boolean lists are  just a special case  of lists many  things are
**  done in the list package.
**
**  A *boolean list* is a list that has no holes and contains only 'true' and
**  'false'.  For  the full definition of  boolean list  see chapter "Boolean
**  Lists" in the {\GAP} Manual.  Read  also the section "More  about Boolean
**  Lists" about the different internal representations of such lists.
**
**  A list that is known to be a boolean list is represented by a bag of type
**  'T_BLIST', which has the following format:
**
**      +-------+-------+-------+-------+- - - -+-------+
**      |logical| block | block | block |       | last  |
**      |length |   0   |   1   |   2   |       | block |
**      +-------+-------+-------+-------+- - - -+-------+
**             /         \
**        .---'           `-----------.
**       /                             \
**      +---+---+---+---+- - - -+---+---+
**      |bit|bit|bit|bit|       |bit|bit|
**      | 0 | 1 | 2 | 3 |       |n-1| n |
**      +---+---+---+---+- - - -+---+---+
**
**  The  first  entry is  the logical  length of the list,  represented as  a
**  {\GAP} immediate integer.  The other entries are blocks, represented as C
**  unsigned  long integer.   Each  block corresponds  to  <n>  (usually  32)
**  elements of the list.  The <j>-th bit (the bit corresponding to '2\^<j>')
**  in  the <i>-th block  is 1 if  the element  '<list>[BIPEB*<i>+<j>+1]'  it
**  'true'  and '0' if  it  is 'false'.  If the logical length of the boolean
**  list is not a multiple of BIPEB the  last block will contain unused bits,
**  which are then zero.
**
**  Note that a list represented by a bag of type 'T_PLIST'  might still be a
**  boolean list.  It is just that the kernel does not known this.
**
**  This package consists of three parts.
**
**  The  first  part  consists  of  the  macros  'BIPEB',  'SIZE_PLEN_BLIST',
**  'PLEN_SIZE_BLIST',   'LEN_BLIST',   'SET_LEN_BLIST',   'ELM_BLIST',   and
**  'SET_ELM_BLIST'.   They  determine the  representation of boolean  lists.
**  The  rest  of the {\GAP} kernel  uses those macros  to access and  modify
**  boolean lists.
**
**  The  second  part  consists  of  the  functions  'LenBlist',  'ElmBlist',
**  'ElmsBlist',   'AssBlist',    'AsssBlist',   'PosBlist',    'PlainBlist',
**  'IsDenseBlist',  'IsPossBlist', 'EqBlist', and  'LtBlist'.  They  are the
**  functions required by the  generic lists  package.  Using these functions
**  the other parts of  the {\GAP} kernel can access and modify boolean lists
**  without actually being aware that they are dealing with a boolean list.
**
**  The third part consists   of the functions  'IsBlistConv', 'FuncIsBlist',
**  'FuncBLIST_LIST', 'FuncLIST_BLIST', 'FuncSIZE_BLIST', 'FuncIS_SUB_BLIST',
**  'FuncUNITE_BLIST', 'FuncINTER_BLIST',   and    'FuncSUBTR_BLIST'.   These
**  functions make it possible to make  boolean lists, either by converting a
**  list to  a boolean list, or by  computing the characteristic boolean list
**  of a sublist, or  by computing the  union, intersection or  difference of
**  two boolean lists.
**
*N  1992/12/16 martin should have 'LtBlist'
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "set.h"                 /* plain sets                      */
#include        "blister.h"             /* boolean lists                   */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */

#include	"code.h"		/* coder                           */
#include	"thread.h"		/* threads			   */
#include	"tls.h"			/* thread-local storage		   */


/****************************************************************************
**

*F  TypeBlist( <list> )  . . . . . . . . . . . . . . . type of a boolean list
**
**  'TypeBlist' returns the type of a boolean list.
**
**  'TypeBlist' is the function in 'TypeObjFuncs' for boolean lists.
*/

/* The following are imported from the GAP level, we have one type for
 * each blist TNUM. */
libGAP_Obj libGAP_TYPE_BLIST_MUT;
libGAP_Obj libGAP_TYPE_BLIST_IMM;
libGAP_Obj libGAP_TYPE_BLIST_NSORT_MUT;
libGAP_Obj libGAP_TYPE_BLIST_NSORT_IMM;
libGAP_Obj libGAP_TYPE_BLIST_SSORT_MUT;
libGAP_Obj libGAP_TYPE_BLIST_SSORT_IMM;
libGAP_Obj libGAP_TYPE_BLIST_EMPTY_MUT;
libGAP_Obj libGAP_TYPE_BLIST_EMPTY_IMM;

libGAP_Obj libGAP_TypeBlistMut (
    libGAP_Obj                 list )
{
    /* special case for the empty blist                                    */
    if ( libGAP_LEN_BLIST(list) == 0 ) {
        return libGAP_TYPE_BLIST_EMPTY_MUT;
    } else {
        return libGAP_TYPE_BLIST_MUT;
    }
}

libGAP_Obj libGAP_TypeBlistImm (
    libGAP_Obj                 list )
{
    /* special case for the empty blist                                    */
    if ( libGAP_LEN_BLIST(list) == 0 ) {
        return libGAP_TYPE_BLIST_EMPTY_IMM;
    } else {
        return libGAP_TYPE_BLIST_IMM;
    }
}

libGAP_Obj libGAP_TypeBlistNSortMut (
    libGAP_Obj                 list )
{
    /* special case for the empty blist                                    */
    if ( libGAP_LEN_BLIST(list) == 0 ) {
        return libGAP_TYPE_BLIST_EMPTY_MUT;
    } else {
        return libGAP_TYPE_BLIST_NSORT_MUT;
    }
}

libGAP_Obj libGAP_TypeBlistNSortImm (
    libGAP_Obj                 list )
{
    /* special case for the empty blist                                    */
    if ( libGAP_LEN_BLIST(list) == 0 ) {
        return libGAP_TYPE_BLIST_EMPTY_IMM;
    } else {
        return libGAP_TYPE_BLIST_NSORT_IMM;
    }
}

libGAP_Obj libGAP_TypeBlistSSortMut (
    libGAP_Obj                 list )
{
    /* special case for the empty blist                                    */
    if ( libGAP_LEN_BLIST(list) == 0 ) {
        return libGAP_TYPE_BLIST_EMPTY_MUT;
    } else {
        return libGAP_TYPE_BLIST_SSORT_MUT;
    }
}

libGAP_Obj libGAP_TypeBlistSSortImm (
    libGAP_Obj                 list )
{
    /* special case for the empty blist                                    */
    if ( libGAP_LEN_BLIST(list) == 0 ) {
        return libGAP_TYPE_BLIST_EMPTY_IMM;
    } else {
        return libGAP_TYPE_BLIST_SSORT_IMM;
    }
}

/****************************************************************************
**
*F  SaveBlist( <blist> ) . . . . . . . . . . . . . . . . . . . . save a blist
**
**   The saving method for the blist tnums
*/
void libGAP_SaveBlist (
    libGAP_Obj                 bl )
{
    libGAP_UInt                i;
    libGAP_UInt *              ptr;

    /* logical length                                                      */
    libGAP_SaveSubObj(libGAP_ADDR_OBJ(bl)[0]);
    ptr = libGAP_BLOCKS_BLIST(bl);
    for (i = 1; i <= libGAP_NUMBER_BLOCKS_BLIST( bl ); i++ )
        libGAP_SaveUInt(*ptr++);
    return;
}

/****************************************************************************
**
*F  LoadBlist( <blist> ) . . . . . . . . . . . . . . . . . . . . load a blist
**
**   The loading method for the blist tnums
*/
void libGAP_LoadBlist (
    libGAP_Obj                 bl )
{
    libGAP_UInt                i;
    libGAP_UInt *              ptr;
  
    /* get the length back, then NUMBER_BLOCKS_BLIST is OK                 */
    libGAP_ADDR_OBJ(bl)[0] = libGAP_LoadSubObj(); 
  
    /* Now load the real data                                              */
    ptr = (libGAP_UInt *)libGAP_BLOCKS_BLIST(bl);
    for (i = 1; i <= libGAP_NUMBER_BLOCKS_BLIST( bl ); i++ )
        *ptr++ = libGAP_LoadUInt();
    return;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * copy functions * * * * * * * * * * * * * * * *
*/

/****************************************************************************
**


*F  CopyBlist( <list>, <mut> )  . . . . . . . . . . . . . copy a boolean list
**
**  'CopyBlist' returns a structural (deep) copy  of the boolean list <list>,
**  i.e., a recursive copy that preserves the structure.
**
**  If <list>  has not yet been copied,   it makes a   copy, leaves a forward
**  pointer  to the copy in the  first entry of  the  boolean list, where the
**  size of the boolean list usually resides, and copies all the entries.  If
**  the boolean  list has already been  copied, it returns  the  value of the
**  forwarding pointer.
**
**  'CopyBlist' is the function in 'CopyObjFuncs' for boolean lists.
**
**  'CleanBlist' removes the mark and the forwarding pointer from the boolean
**  list <list>.
**
**  'CleanBlist' is the function in 'CleanObjFuncs' for boolean lists.
*/

libGAP_Obj libGAP_DoCopyBlist(libGAP_Obj list, libGAP_Int mut) {
  libGAP_Obj copy;
  libGAP_UInt *l;
  libGAP_UInt *c;
    /* make a copy                                                         */
    if ( mut ) {
      copy = libGAP_NewBag( libGAP_MUTABLE_TNUM(libGAP_TNUM_OBJ(list)), libGAP_SIZE_OBJ(list) );
    }
    else {
      copy = libGAP_NewBag( libGAP_IMMUTABLE_TNUM( libGAP_TNUM_OBJ(list) ), libGAP_SIZE_OBJ(list) );
    }


    /* copy the subvalues                                                  */
    l = (libGAP_UInt*)(libGAP_ADDR_OBJ(list));
    c = (libGAP_UInt*)(libGAP_ADDR_OBJ(copy));
    memcpy((void *)c, (void *)l, sizeof(libGAP_UInt)*(1+libGAP_NUMBER_BLOCKS_BLIST(list)));

    /* return the copy                                                     */
    return copy;
  
}

libGAP_Obj libGAP_CopyBlist (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{

    /* don't change immutable objects                                      */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        return list;
    }

    return libGAP_DoCopyBlist(list, mut);
}

libGAP_Obj libGAP_ShallowCopyBlist ( libGAP_Obj list)
{
  return libGAP_DoCopyBlist(list, 1);
}



/****************************************************************************
**
*F  CopyBlistCopy( <list>, <mut> )  . . . . . . . copy a already copied blist
*/
libGAP_Obj libGAP_CopyBlistCopy (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{
    return libGAP_ADDR_OBJ(list)[0];
}


/****************************************************************************
**
*F  CleanBlist( <list> )  . . . . . . . . . . . . . . clean up a boolean list
*/
void libGAP_CleanBlist (
    libGAP_Obj                 list )
{
}


/****************************************************************************
**
*F  CleanBlistCopy( <list> )  . . . . . . . . . . . . .  clean a copied blist
*/
void libGAP_CleanBlistCopy (
    libGAP_Obj                 list )
{
    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(list)[0] = libGAP_ADDR_OBJ( libGAP_ADDR_OBJ(list)[0] )[0];

    /* now it is cleaned                                                   */
    libGAP_UNMARK_LIST( list, libGAP_COPYING );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * list functions * * * * * * * * * * * * * * * *
*/

/****************************************************************************
**


*F  EqBlist( <listL>, <listR> ) . . . . . test if two boolean lists are equal
**
**  'EqBlist' returns 'true' if the two boolean lists <listL> and <listR> are
**  equal and 'false' otherwise.
*/
libGAP_Int libGAP_EqBlist (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    long                lenL;           /* length of the left operand      */
    long                lenR;           /* length of the right operand     */
    libGAP_UInt *              ptrL;           /* pointer to the left operand     */
    libGAP_UInt *              ptrR;           /* pointer to the right operand    */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the lengths of the lists and compare them                       */
    lenL = libGAP_LEN_BLIST( listL );
    lenR = libGAP_LEN_BLIST( listR );
    if ( lenL != lenR ) {
        return 0L;
    }

    /* test for equality blockwise                                         */
    ptrL = libGAP_BLOCKS_BLIST(listL);
    ptrR = libGAP_BLOCKS_BLIST(listR);
    for ( i = (lenL+libGAP_BIPEB-1)/libGAP_BIPEB; 0 < i; i-- ) {
        if ( *ptrL++ != *ptrR++ )
            return 0L;
    }

    /* no differences found, the lists are equal                           */
    return 1L;
}


/****************************************************************************
**
*F  LenBlist( <list> )  . . . . . . . . . . . . . .  length of a boolean list
**
**  'LenBlist' returns the length of the boolean list <list> as a C integer.
**
**  'LenBlist' is the function in 'LenListFuncs' for boolean lists.
*/
libGAP_Int libGAP_LenBlist (
    libGAP_Obj                 list )
{
    return libGAP_LEN_BLIST( list );
}


/****************************************************************************
**
*F  IsbBlist( <list>, <pos> ) . . . . . test for an element of a boolean list
**
**  'IsbBlist' returns  1 if the boolean list <list> contains an  element  at
**  the position <pos> and 0 otherwise.  It  is  the  responsibility  of  the
**  caller to ensure that <pos> is a positive integer.
**
**  'IsbBlist' is the function in 'IsbListFuncs' for boolean lists.
*/
libGAP_Int libGAP_IsbBlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return (pos <= libGAP_LEN_BLIST(list));
}


/****************************************************************************
**
*F  IsbvBlist( <list>, <pos> )  . . . . test for an element of a boolean list
*/
libGAP_Int libGAP_IsbvBlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return 1L;
}


/****************************************************************************
**

*F  Elm0Blist( <list>, <pos> )  . . . . . select an element of a boolean list
**
**  'Elm0Blist' returns the element at the position <pos> of the boolean list
**  <list>, or 0 if  <list>  has no  assigned object  at  <pos>.  It  is  the
**  responsibility of the caller to ensure that <pos> is a positive integer.
*/
libGAP_Obj libGAP_Elm0Blist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    if ( pos <= libGAP_LEN_BLIST( list ) ) {
        return libGAP_ELM_BLIST( list, pos );
    }
    else {
        return 0;
    }
}


/****************************************************************************
**
*F  Elm0vBlist( <list>, <pos> ) . . . . . select an element of a boolean list
**
**  'Elm0vPlist' does the same thing than 'Elm0List', but need not check that
**  <pos>  is less  than or  equal   to the length   of  <list>, this is  the
**  responsibility of the caller.
*/
libGAP_Obj libGAP_Elm0vBlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return libGAP_ELM_BLIST( list, pos );
}


/****************************************************************************
**
*F  ElmBlist( <list>, <pos> ) . . . . . . select an element of a boolean list
**
**  'ElmBlist' selects  the  element at  position <pos> of  the  boolean list
**  <list>.  It is the responsibility of the caller to ensure that <pos> is a
**  positive integer.   An  error is signalled   if <pos> is larger  than the
**  length of <list>.
**
**  'ElmBlist'   is  the  function  in    'ElmListFuncs'  for boolean  lists.
**  'ElmvBlist' is the function in 'ElmvListFuncs' for boolean lists.
*/
libGAP_Obj libGAP_ElmBlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{

    /* check the position                                                  */
    if ( libGAP_LEN_BLIST( list ) < pos ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            pos, 0L,
            "you can assign a value and 'return;'" );
        return libGAP_ELM_LIST( list, pos );
    }

    /* select and return the element                                       */
    return libGAP_ELM_BLIST( list, pos );
}

/****************************************************************************
**
*F  ElmvBlist( <list>, <pos> )  . . . . . select an element of a boolean list
**
**  'ElmvBlist' does the same thing than  'ElmBlist', but need not check that
**  <pos>   is less than   or equal  to the  length  of  <list>,  this is the
**  responsibility of the caller.
**
*/
libGAP_Obj libGAP_ElmvBlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    /* select and return the element                                       */
    return libGAP_ELM_BLIST( list, pos );
}


/****************************************************************************
**
*F  ElmsBlist( <list>, <poss> ) . . . .  select a sublist from a boolean list
**
**  'ElmsBlist' returns a new list  containing the elements at the  positions
**  given in the   list  <poss> from  the boolean   list <list>.   It  is the
**  responsibility of the caller to ensure  that <poss> is dense and contains
**  only positive integers.  An error is signalled if an element of <poss> is
**  larger than the length of <list>.
**
**  'ElmsBlist' is the function in 'ElmsListFuncs' for boolean lists.
*/
libGAP_Obj libGAP_ElmsBlist (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elms;           /* selected sublist, result        */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element from <list>         */
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_UInt                block;          /* one block of <elms>             */
    libGAP_UInt                bit;            /* one bit of a block              */
    libGAP_UInt                i;              /* loop variable                   */

    /* general code                                                        */
    if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_BLIST( list );

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST( poss );

        /* make the result list                                            */
        elms = libGAP_NewBag( libGAP_T_BLIST, libGAP_SIZE_PLEN_BLIST( lenPoss ) );
        libGAP_SET_LEN_BLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        block = 0;  bit = 1;
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
            pos = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, (libGAP_Int)i ) );
            if ( lenList < pos ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    pos, 0L,
                    "you can assign a value and 'return;'" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* select the element                                          */
            elm = libGAP_ELM_BLIST( list, pos );

            /* assign the element into <elms>                              */
            if ( elm == libGAP_True )
                block |= bit;
            bit <<= 1;
            if ( bit == 0 || i == lenPoss ) {
                libGAP_BLOCK_ELM_BLIST( elms, i) =  block;
                block = 0;
                bit = 1;
            }

        }

    }

    /* special code for ranges                                             */
    /*N 1992/12/15 martin special code for ranges with increment 1         */
    else {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_PLIST( list );

        /* get the length of <positions>, the first elements, and the inc. */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* check that no <position> is larger than 'LEN_LIST(<list>)'      */
        if ( lenList < pos ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                pos, 0L,
                "you can assign a value and 'return;'" );
            return libGAP_ELMS_LIST( list, poss );
        }
        if ( lenList < pos + (lenPoss-1) * inc ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                pos+(lenPoss-1)*inc, 0L,
                "you can assign a value and 'return;'" );
            return libGAP_ELMS_LIST( list, poss );
        }

        /* make the result list                                            */
        elms = libGAP_NewBag( libGAP_T_BLIST, libGAP_SIZE_PLEN_BLIST( lenPoss ) );
        libGAP_SET_LEN_BLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        block = 0;  bit = 1;
        for ( i = 1; i <= lenPoss; i++, pos += inc ) {

            /* select the element                                          */
            elm = libGAP_ELM_BLIST( list, pos );

            /* assign the element to <elms>                                */
            if ( elm == libGAP_True )
                block |= bit;
            bit <<= 1;
            if ( bit == 0 || i == lenPoss ) {
                libGAP_BLOCK_ELM_BLIST(elms, i) =  block;
                block = 0;
                bit = 1;
            }

        }

    }

    /* return the result                                                   */
    return elms;
}


/****************************************************************************
**

*F  AssBlist( <list>, <pos>, <val> )  . . . . . . .  assign to a boolean list
**
**  'AssBlist' assigns the   value <val> to  the  boolean list <list> at  the
**  position <pos>.   It is the responsibility  of the caller to  ensure that
**  <pos> is positive, and that <val> is not 0.
**
**  'AssBlist' is the function in 'AssListFuncs' for boolean lists.
**
**  If <pos> is less than or equal to the logical length  of the boolean list
**  and <val> is 'true' or   'false' the assignment  is  done by setting  the
**  corresponding bit.  If <pos>  is one more  than the logical length of the
**  boolean list  the assignment is  done by   resizing  the boolean list  if
**  necessary, setting the   corresponding bit and  incrementing  the logical
**  length  by one.  Otherwise  the boolean list is  converted to an ordinary
**  list and the assignment is performed the ordinary way.
*/
void libGAP_AssBlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    /* if <pos> is less than the logical length and <elm> is 'true'        */
    if      ( pos <= libGAP_LEN_BLIST(list) && val == libGAP_True ) {
        libGAP_SET_ELM_BLIST( list, pos, libGAP_True );
        libGAP_CLEAR_FILTS_LIST(list);
    }

    /* if <i> is less than the logical length and <elm> is 'false'         */
    else if ( pos <= libGAP_LEN_BLIST(list) && val == libGAP_False ) {
        libGAP_SET_ELM_BLIST( list, pos, libGAP_False );
        libGAP_CLEAR_FILTS_LIST(list);
    }

    /* if <i> is one more than the logical length and <elm> is 'true'      */
    else if ( pos == libGAP_LEN_BLIST(list)+1 && val == libGAP_True ) {
        if ( libGAP_SIZE_OBJ(list) < libGAP_SIZE_PLEN_BLIST(pos) )
            libGAP_ResizeBag( list, libGAP_SIZE_PLEN_BLIST(pos) );
        libGAP_SET_LEN_BLIST( list, pos );
        libGAP_SET_ELM_BLIST( list, pos, libGAP_True );
        libGAP_CLEAR_FILTS_LIST(list);
    }

    /* if <i> is one more than the logical length and <elm> is 'false'     */
    else if ( pos == libGAP_LEN_BLIST(list)+1 && val == libGAP_False ) {
        if ( libGAP_SIZE_OBJ(list) < libGAP_SIZE_PLEN_BLIST(pos) )
            libGAP_ResizeBag( list, libGAP_SIZE_PLEN_BLIST(pos) );
        libGAP_SET_LEN_BLIST( list, pos );
        libGAP_SET_ELM_BLIST( list, pos, libGAP_False );
        libGAP_CLEAR_FILTS_LIST(list);
    }

    /* otherwise convert to ordinary list and assign as in 'AssList'       */
    else {
        libGAP_PLAIN_LIST(list);
        libGAP_CLEAR_FILTS_LIST(list);
        if ( libGAP_LEN_PLIST(list) < pos ) {
            libGAP_GROW_PLIST( list, (libGAP_UInt)pos );
            libGAP_SET_LEN_PLIST( list, pos );
        }
        libGAP_SET_ELM_PLIST( list, pos, val );
        libGAP_CHANGED_BAG( list );
    }
}


/****************************************************************************
**
*F  AssBlistImm( <list>, <pos>, <val> ) . assign to an immutable boolean list
*/
void libGAP_AssBlistImm (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignment: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**
*F  AsssBlist( <list>, <poss>, <vals> ) .  assign several elements to a blist
**
**  'AsssBlist' assignes the values  from  the list  <vals> at the  positions
**  given  in the  list  <poss>  to the   boolean  list  <list>.  It is   the
**  responsibility of the caller to ensure that  <poss> is dense and contains
**  only positive integers, that <poss> and <vals>  have the same length, and
**  that <vals> is dense.
**
**  'AsssBlist' is intended as function in 'AsssListFuncs' for boolean lists.
**  Note that currently, we use AsssListDefault instead. This ensures 
**  automatically that <list> remains a blist if possible.
**
*/
void libGAP_AsssBlist (     /*  currently not used */
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 vals )
{
   libGAP_Int   i, len, pos;
   libGAP_Obj   val;

   len = libGAP_LEN_LIST(poss);
   for (i=1; i <= len; i++) {
      /* use generic macros because list might be unpacked */
      pos = libGAP_INT_INTOBJ(libGAP_ELMW_LIST(poss, i));
      val = libGAP_ELMW_LIST(vals, i);
      libGAP_ASS_LIST( list, pos, val);
    }
}


/****************************************************************************
**
*F  AsssBlistImm( <list>, <poss>, <vals> )  . .  assign to an immutable blist
*/
void libGAP_AsssBlistImm (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignments: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**

*F  PosBlist( <list>, <val>, <start> )   position of an elm in a boolean list
**
**  'PosBlist' returns the   position of  the first  occurrence  of the  value
**  <val>, which may be  an  object of  arbitrary  type, in the boolean  list
**  <list> after <start> as a C  integer.  If <val> does  not occur in <list>
**  after <start>, then 0 is returned.
**
**  'PosBlist' is the function in 'PosListFuncs' for boolean lists.
*/
libGAP_Obj libGAP_PosBlist (
    libGAP_Obj                 list,
    libGAP_Obj                 val,
    libGAP_Obj                 start )
{
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_UInt *              ptr;            /* pointer to the blocks           */
    libGAP_UInt                i,  j;          /* loop variables                  */
    libGAP_UInt                istart;
    libGAP_UInt                firstblock, lastblock;
    libGAP_UInt                firstoffset, lastoffset;
    libGAP_UInt                x;

    if (!libGAP_IS_INTOBJ(start))
      return libGAP_Fail;
    
    istart = libGAP_INT_INTOBJ(start);

    len = libGAP_LEN_BLIST(list);

    /* look just beyond end                                                */
    if ( len == istart ) {
      return libGAP_Fail;
    }

    ptr = libGAP_BLOCKS_BLIST(list);
    firstblock = istart/libGAP_BIPEB;
    lastblock = (len-1)/libGAP_BIPEB;
    firstoffset = istart%libGAP_BIPEB;
    lastoffset = (len-1)%libGAP_BIPEB;

    /* look for 'true'                                                     */
     if ( val == libGAP_True ) {

       x = ptr[firstblock];
       if (firstblock == lastblock) 
         {
           if (x != 0)
             for (j = firstoffset; j <= lastoffset; j++)
               if ((x & (1UL << j)) != 0)
                 return libGAP_INTOBJ_INT(libGAP_BIPEB*firstblock + j + 1);
           return libGAP_Fail;
         }
       if (x != 0)
         for (j = firstoffset; j < libGAP_BIPEB; j++)
           if ((x & (1UL << j)) != 0)
             return libGAP_INTOBJ_INT(libGAP_BIPEB*firstblock + j + 1);
       for (i  = firstblock + 1; i < lastblock; i++)
         {
           x = ptr[i];
           if (x != 0)
             for (j = 0; j < libGAP_BIPEB; j++)
               if ((x & (1UL << j)) != 0)
                 return libGAP_INTOBJ_INT(libGAP_BIPEB*i + j + 1);
         }
       x = ptr[lastblock];
       if (x != 0)
         for (j = 0; j <= lastoffset; j++)
           if ((x & (1UL << j)) != 0)
             return libGAP_INTOBJ_INT(libGAP_BIPEB*lastblock + j + 1);
       return libGAP_Fail;
    }

    /* look for 'false'                                                    */
    else if ( val == libGAP_False ) {
      x = ptr[firstblock];
      if (firstblock == lastblock) 
        {
          if (x != ~0UL)
            for (j = firstoffset; j <= lastoffset; j++)
              if ((x & (1UL << j)) == 0)
                return libGAP_INTOBJ_INT(libGAP_BIPEB*firstblock + j + 1);
           return libGAP_Fail;
         }
       if (x != ~0UL)
         for (j = firstoffset; j < libGAP_BIPEB; j++)
           if ((x & (1UL << j)) == 0)
             return libGAP_INTOBJ_INT(libGAP_BIPEB*firstblock + j + 1);
       for (i  = firstblock + 1; i < lastblock; i++)
         {
           x = ptr[i];
           if (x != ~0UL)
             for (j = 0; j < libGAP_BIPEB; j++)
               if ((x & (1UL << j)) == 0)
                 return libGAP_INTOBJ_INT(libGAP_BIPEB*i + j + 1);
         }
       x = ptr[lastblock];
       if (x != ~0UL)
         for (j = 0; j <= lastoffset; j++)
           if ((x & (1UL << j)) == 0)
             return libGAP_INTOBJ_INT(libGAP_BIPEB*lastblock + j + 1);
       return libGAP_Fail;
    }

    /* look for something else                                             */
    else {
      return libGAP_Fail;
    }

}


/****************************************************************************
**
*F  PlainBlist( <list> )  . . .  convert a boolean list into an ordinary list
**
**  'PlainBlist' converts the boolean list <list> to a plain list.
**
**  'PlainBlist' is the function in 'PlainListFuncs' for boolean lists.
*/
void libGAP_PlainBlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 len;            /* length of <list>                */
    libGAP_UInt                i;              /* loop variable                   */

    /* resize the list and retype it, in this order                        */
    len = libGAP_LEN_BLIST(list);
    libGAP_RetypeBag( list, libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE );
    libGAP_GROW_PLIST( list, (libGAP_UInt)len );
    libGAP_SET_LEN_PLIST( list, len );

    /* replace the bits by 'True' or 'False' as the case may be            */
    /* this must of course be done from the end of the list backwards      */
    for ( i = len; 0 < i; i-- )
        libGAP_SET_ELM_PLIST( list, i, libGAP_ELM_BLIST( list, i ) );

    /* 'CHANGED_BAG' not needed, 'True' and 'False' are safe           */
}



/****************************************************************************
**
*F  IsPossBlist( <list> ) . .  positions list test function for boolean lists
**
**  'IsPossBlist' returns  1 if  <list> is  empty, and 0 otherwise, since a
**  boolean list is a positions list if and only if it is empty.
*/
libGAP_Int libGAP_IsPossBlist (
    libGAP_Obj                 list )
{
    return libGAP_LEN_BLIST(list) == 0;
}


/****************************************************************************
**

*F  IsDenseBlist( <list> )  . . .  dense list test function for boolean lists
**
**  'IsDenseBlist' returns 1, since boolean lists are always dense.
**
**  'IsDenseBlist' is the function in 'IsDenseBlistFuncs' for boolean lists.
*/
libGAP_Int libGAP_IsDenseBlist (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**
*F  IsHomogBlist( <list> )  . . . . . . . . . . check if <list> is homogenous
*/
libGAP_Int libGAP_IsHomogBlist (
    libGAP_Obj                 list )
{
    return (0 < libGAP_LEN_BLIST(list));
}


/****************************************************************************
**
*F  IsSSortBlist( <list> )  . . . . . . .  check if <list> is strictly sorted
*/
libGAP_Int libGAP_IsSSortBlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 isSort;

    if ( libGAP_LEN_BLIST(list) <= 1 ) {
        isSort = 1;
    }
    else if ( libGAP_LEN_BLIST(list) == 2 ) {
        isSort = (libGAP_ELM_BLIST(list,1) == libGAP_True && libGAP_ELM_BLIST(list,2) == libGAP_False);
    }
    else {
        isSort = 0;
    }
    libGAP_SET_FILT_LIST( list, (isSort ? libGAP_FN_IS_SSORT : libGAP_FN_IS_NSORT) );

    return isSort;
}


/****************************************************************************
**
*F  IsSSortBlistNot( <list> ) . . . . . . . . . . . . . unsorted boolean list
*/
libGAP_Int libGAP_IsSSortBlistNot (
    libGAP_Obj                 list )
{
    return 0L;
}


/****************************************************************************
**
*F  IsSSortBlistYes( <list> ) . . . . . . . . . . . . . . sorted boolean list
*/
libGAP_Int libGAP_IsSSortBlistYes (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**

*F  ConvBlist( <list> ) . . . . . . . . .  convert a list into a boolean list
**
**  `ConvBlist' changes the representation of boolean  lists into the compact
**  representation of type 'T_BLIST' described above.
*/
void libGAP_ConvBlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_UInt                block;          /* one block of the boolean list   */
    libGAP_UInt                bit;            /* one bit of a block              */
    libGAP_UInt                i;              /* loop variable                   */

    /* if <list> is known to be a boolean list, it is very easy            */
    if ( libGAP_IS_BLIST_REP(list) ) {
        return;
    }

    /* change its representation                                           */
    block = 0;
    bit = 1;
    len = libGAP_LEN_LIST( list );
    for ( i = 1; i <= len; i++ ) {
        if ( libGAP_ELMW_LIST( list, (libGAP_Int)i ) == libGAP_True )
            block |= bit;
        bit = bit << 1;
        if ( bit == 0 || i == len ) {
            libGAP_BLOCK_ELM_BLIST(list,i) = block;
            block = 0;
            bit = 1;
        }
    }
    libGAP_RetypeBag( list, libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_BLIST : libGAP_T_BLIST+libGAP_IMMUTABLE );
    libGAP_ResizeBag( list, libGAP_SIZE_PLEN_BLIST(len) );
    libGAP_SET_LEN_BLIST( list, len );
}


/****************************************************************************
**
*F  IsBlist( <list> ) . . . . . . . . . test whether a list is a boolean list
**
**  'IsBlist' returns  1  if  the  list  <list> is  a  boolean list, i.e.,  a
**  list that   has no holes  and contains  only  'true' and  'false',  and 0
**  otherwise.
*/
libGAP_Int libGAP_IsBlist (
    libGAP_Obj                 list )
{
    libGAP_UInt                isBlist;        /* result of the test              */
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_UInt                i;              /* loop variable                   */

    /* if <list> is known to be a boolean list, it is very easy            */
    if ( libGAP_IS_BLIST_REP(list) ) {
        isBlist = 1;
    }

    /* if <list> is not a small list, its not a boolean list (convert to list)   */
    else if ( ! libGAP_IS_SMALL_LIST( list ) ) {
        isBlist = 0;
    }

    /* otherwise test if there are holes and if all elements are boolean   */
    else {

        /* test that all elements are bound and either 'true' or 'false'   */
        len = libGAP_LEN_LIST( list );
        for ( i = 1; i <= len; i++ ) {
            if ( libGAP_ELMV0_LIST( list, (libGAP_Int)i ) == 0
              || (libGAP_ELMW_LIST( list, (libGAP_Int)i ) != libGAP_True
               && libGAP_ELMW_LIST( list, (libGAP_Int)i ) != libGAP_False) ) {
                break;
            }
        }

        isBlist = (len < i);
    }

    /* return the result                                                   */
    return isBlist;
}


/****************************************************************************
**
*F  IsBlistConv( <list> ) . test whether a list is a boolean list and convert
**
**  'IsBlistConv' returns 1 if  the list <list> is  a  boolean list, i.e.,  a
**  list that   has no holes  and contains  only  'true' and  'false',  and 0
**  otherwise.  As a  side effect 'IsBlistConv' changes the representation  of
**  boolean lists into the compact representation of type 'T_BLIST' described
**  above.
*/
libGAP_Int libGAP_IsBlistConv (
    libGAP_Obj                 list )
{
    libGAP_UInt                isBlist;        /* result of the test              */
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_UInt                i;              /* loop variable                   */

    /* if <list> is known to be a boolean list, it is very easy            */
    if ( libGAP_IS_BLIST_REP(list) ) {
        isBlist = 1;
    }

    /* if <list> is not a list, its not a boolean list (convert to list)   */
    else if ( ! libGAP_IS_SMALL_LIST(list) ) {
        isBlist = 0;
    }

    /* otherwise test if there are holes and if all elements are boolean   */
    else {

        /* test that all elements are bound and either 'true' or 'false'   */
        len = libGAP_LEN_LIST( list );
        for ( i = 1;  i <= len;  i++ ) {
            if ( libGAP_ELMV0_LIST( list, (libGAP_Int)i ) == 0
              || (libGAP_ELMW_LIST( list, (libGAP_Int)i ) != libGAP_True
               && libGAP_ELMW_LIST( list, (libGAP_Int)i ) != libGAP_False) ) {
                break;
            }
        }

        /* if <list> is a boolean list, change its representation        */
        isBlist = (len < i);
        if ( isBlist ) {
            libGAP_ConvBlist(list);
        }
    }

    /* return the result                                                   */
    return isBlist;
}


/****************************************************************************
**
*F  SizeBlist( <blist> )  . . . .  number of 'true' entries in a boolean list
**
**  'SizeBlist' returns   the number of  entries of  the boolean list <blist>
**  that are 'true'.
**
**  The sequence to compute the  number of bits in  a block is quite  clever.
**  The idea is that after the <i>-th instruction each subblock of $2^i$ bits
**  holds the number of   bits of this  subblock  in the original block  <m>.
**  This is illustrated in the example below for a block of with 8 bits:
**
**       // a b c d e f g h
**      m = (m & 0x55)       +  ((m >> 1) & 0x55);
**       // . b . d . f . h  +  . a . c . e . g   =  a+b c+d e+f g+h
**      m = (m & 0x33)       +  ((m >> 2) & 0x33);
**       // . . c+d . . g+h  +  . . a+b . . e+f   =  a+b+c+d e+f+g+h
**      m = (m & 0x0f)       +  ((m >> 4) & 0x0f);
**       // . . . . e+f+g+h  +  . . . . a+b+c+d   =  a+b+c+d+e+f+g+h
**
**  In the actual  code  some unnecessary mask  have  been removed, improving
**  performance quite a bit,  because masks are 32  bit immediate values  for
**  which most RISC  processors need two  instructions to load them.  Talking
**  about performance.  The code is  close to optimal,  it should compile  to
**  only about  22 MIPS  or SPARC instructions.   Dividing the  block into  4
**  bytes and looking up the number of bits  of a byte in a  table may be 10%
**  faster, but only if the table lives in the data cache.
**
*N  1992/12/15 martin this depends on 'BIPEB' being 32 
*N  1996/11/12 Steve  altered to handle 64 bit also
**
**  Introduced the SizeBlist function for kernel use, and the 
**  COUNT_TRUES_BLOCK( <var> ) macro which replaces a block of bits in <var> 
**  by the number of ones it contains. It will fail horribly if <var> is not 
**  a variable.
*/
libGAP_UInt libGAP_SizeBlist (
    libGAP_Obj                 blist )
{
    libGAP_UInt *              ptr;            /* pointer to blist                */
    libGAP_UInt                nrb;            /* number of blocks in blist       */
    libGAP_UInt                m;              /* number of bits in a block       */
    libGAP_UInt                n;              /* number of bits in blist         */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the number of blocks and a pointer                              */
    nrb = libGAP_NUMBER_BLOCKS_BLIST(blist);
    ptr = libGAP_BLOCKS_BLIST( blist );

    /* loop over the blocks, adding the number of bits of each one         */
    n = 0;
    for ( i = 1; i <= nrb; i++ ) {
        m = *ptr++;
        libGAP_COUNT_TRUES_BLOCK(m);
        n += m;
    }

    /* return the number of bits                                           */
    return n;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * GAP level functions  * * * * * * * * * * * * *
*/

/****************************************************************************
**


*F  FuncIS_BLIST( <self>, <val> ) . . . . . test if a value is a boolean list
**
**  'FuncIS_BLIST' handles the internal function 'IsBlist'.
**
**  'IsBlist( <val> )'
**
**  'IsBlist' returns 'true' if the value <val> is a boolean list and 'false'
**  otherwise.  A value is a   boolean list if  it is  a lists without  holes
**  containing only  'true' and 'false'.
*/
libGAP_Obj libGAP_IsBlistFilt;

libGAP_Obj libGAP_FuncIS_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 val )
{
    /* let 'IsBlist' do the work                                           */
    return libGAP_IsBlist( val ) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncIS_BLIST_CONV( <self>, <val> )  . . test if a value is a boolean list
**
**  'FuncIS_BLIST_CONV' handles the internal function 'IsBlist'.
**
**  'IsBlistConv( <val> )'
**
**  'IsBlist' returns 'true' if the value <val> is a boolean list and 'false'
**  otherwise.  A value is a   boolean list if  it is  a lists without  holes
**  containing only  'true' and 'false'.
*/
libGAP_Obj libGAP_IsBlistFilt;

libGAP_Obj libGAP_FuncIS_BLIST_CONV (
    libGAP_Obj                 self,
    libGAP_Obj                 val )
{
    /* let 'IsBlist' do the work                                           */
    return libGAP_IsBlistConv( val ) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncCONV_BLIST( <self>, <blist> ) . . . . convert into a boolean list rep
*/
libGAP_Obj libGAP_FuncCONV_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 blist )
{
    /* check whether <blist> is a boolean list                             */
    while ( ! libGAP_IsBlistConv(blist) ) {
        blist = libGAP_ErrorReturnObj(
            "CONV_BLIST: <blist> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(blist), 0L,
            "you can replace <blist> via 'return <blist>;'" );
    }

    /* return nothing                                                      */
    return 0;
}

/****************************************************************************
**
**
*F  FuncIS_BLIST_REP( <self>, <obj> ) . . test if value is a boolean list rep
*/
libGAP_Obj libGAP_IsBlistRepFilt;

libGAP_Obj libGAP_FuncIS_BLIST_REP (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_BLIST_REP( obj ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**

*F  FuncSIZE_BLIST( <self>, <blist> ) . . number of 'true' entries in <blist>
**
**  'FuncSIZE_BLIST' implements the internal function 'SizeBlist'
*/
libGAP_Obj libGAP_FuncSIZE_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 blist )
{
    /* get and check the argument                                          */
    while ( ! libGAP_IsBlistConv(blist) ) {
        blist = libGAP_ErrorReturnObj(
            "SizeBlist: <blist> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(blist), 0L,
            "you can replace <blist> via 'return <blist>;'" );
    }
  
    return libGAP_INTOBJ_INT(libGAP_SizeBlist(blist));
}



/****************************************************************************
**
*F  FuncBLIST_LIST( <self>, <list>, <sub> )  make boolean list from a sublist
**
**  'FuncBLIST_LIST' implements the internal function 'BlistList'.
**
**  'BlistList( <list>, <sub> )'
**
**  'BlistList'  creates a boolean  list   that describes the  list <sub>  as
**  sublist  of the list  <list>.  The result  is a new boolean list <blist>,
**  which has the same length  as <list>, such  that '<blist>[<i>]' is 'true'
**  if '<list>[<i>]' is an element of <sub> and 'false' otherwise.
**
**  'BlistList' is most effective if <list> is a set, but can be used with an
**  arbitrary list that has no holes.
*/
libGAP_Obj libGAP_FuncBLIST_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 sub )
{
    libGAP_Obj                 blist;          /* boolean list, result            */
    libGAP_UInt  *             ptrBlist;       /* pointer to the boolean list     */
    libGAP_UInt                block;          /* one block of boolean list       */
    libGAP_UInt                bit;            /* one bit of block                */
    libGAP_Int                 lenList;        /* logical length of the list      */
    libGAP_Obj *               ptrSub;         /* pointer to the sublist          */
    libGAP_UInt                lenSub;         /* logical length of sublist       */
    libGAP_UInt                i, j, k = 0, l;     /* loop variables                  */
    long                s, t;           /* elements of a range             */

    /* get and check the arguments                                         */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "BlistList: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(sub) ) {
        sub = libGAP_ErrorReturnObj(
            "BlistList: <sub> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(sub), 0L,
            "you can replace <sub> via 'return <sub>;'" );
    }

    /* for a range as subset of a range, it is extremely easy               */
    if ( libGAP_IS_RANGE(list) && libGAP_IS_RANGE(sub) && libGAP_GET_INC_RANGE( list ) == 1
          && libGAP_GET_INC_RANGE( sub ) == 1) {

        /* allocate the boolean list and get pointer                       */
        lenList  = libGAP_GET_LEN_RANGE( list );
        lenSub   = libGAP_GET_LEN_RANGE( sub );
        blist = libGAP_NewBag( libGAP_T_BLIST, libGAP_SIZE_PLEN_BLIST( lenList ) );
        libGAP_ADDR_OBJ(blist)[0] = libGAP_INTOBJ_INT(lenList);
        ptrBlist = libGAP_BLOCKS_BLIST(blist);

        /* get the bounds of the subset with respect to the boolean list   */
        s = libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( list, 1 ) );
        t = libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( sub, 1 ) );
        if ( s <= t )  i = t - s + 1;
        else           i = 1;

        if ( i + lenSub - 1 <= lenList )  j = i + lenSub - 1;
        else                              j = lenList;

        /* set the corresponding entries to 'true'                         */
        for ( k = i; k <= j && (k-1)%libGAP_BIPEB != 0; k++ )
            ptrBlist[(k-1)/libGAP_BIPEB] |= (1UL << (k-1)%libGAP_BIPEB);
        for ( ; k+libGAP_BIPEB <= j; k += libGAP_BIPEB )
            ptrBlist[(k-1)/libGAP_BIPEB] = ~(libGAP_UInt)0;
        for ( ; k <= j; k++ )
            ptrBlist[(k-1)/libGAP_BIPEB] |= (1UL << (k-1)%libGAP_BIPEB);

    }

    /* for a list as subset of a range, we need basically no search        */
    else if ( libGAP_IS_RANGE(list) && libGAP_GET_INC_RANGE( list) == 1
          && libGAP_IS_PLIST(sub) ) {

        /* allocate the boolean list and get pointer                       */
        lenList  = libGAP_GET_LEN_RANGE( list );
        lenSub   = libGAP_LEN_LIST( sub );
        blist = libGAP_NewBag( libGAP_T_BLIST, libGAP_SIZE_PLEN_BLIST( lenList ) );
        libGAP_ADDR_OBJ(blist)[0] = libGAP_INTOBJ_INT(lenList);
        ptrBlist = libGAP_BLOCKS_BLIST(blist);
        ptrSub = libGAP_ADDR_OBJ(sub);

        /* loop over <sub> and set the corresponding entries to 'true'     */
        s = libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( list, 1 ) );
        for ( l = 1; l <= libGAP_LEN_LIST(sub); l++ ) {
            if ( ptrSub[l] != 0 ) {

                /* if <sub>[<l>] is an integer it is very easy             */
                if ( libGAP_TNUM_OBJ( ptrSub[l] ) == libGAP_T_INT ) {
                    t = libGAP_INT_INTOBJ( ptrSub[l] ) - s + 1;
                    if ( 0 < t && t <= lenList )
                        ptrBlist[(t-1)/libGAP_BIPEB] |= (1UL << (t-1)%libGAP_BIPEB);
                }

     /* Nobody seems to remember what the code below is good for,
      * we will now just assume that non-immediate integers are
      * never in a range. I'll leave the old code in a comment
      * for a while, the third arg for PosRange is wrong anyway.
      * FL  */
                /* otherwise it may be a record, let 'PosRange' handle it  */
              /*  else {
                    Obj pos;
                    pos = PosRange( list, ptrSub[l], 0L );
                    if (pos != Fail) {
                      k = INT_INTOBJ(pos);
                      ptrBlist[(k-1)/BIPEB] |= (1UL << (k-1)%BIPEB);
                    }  
                } */

            }
        }

    }

    /* if <list> is a set we have two possibilities                        */
    else if ( libGAP_IsSet( list ) ) {

        /* get the length of <list> and its logarithm                      */
        lenList = libGAP_LEN_PLIST( list );
        for ( i = lenList, l = 0; i != 0; i >>= 1, l++ ) ;
        libGAP_PLAIN_LIST( sub );
        lenSub = libGAP_LEN_LIST( sub );

        /* if <sub> is small, we loop over <sub> and use binary search     */
        if ( l * lenSub < 2 * lenList ) {

            /* allocate the boolean list and get pointer                   */
            blist = libGAP_NewBag( libGAP_T_BLIST, libGAP_SIZE_PLEN_BLIST( lenList ) );
            libGAP_ADDR_OBJ(blist)[0] = libGAP_INTOBJ_INT(lenList);

            /* run over the elements of <sub> and search for the elements  */
            for ( l = 1; l <= libGAP_LEN_LIST(sub); l++ ) {
                if ( libGAP_ADDR_OBJ(sub)[l] != 0 ) {

                    /* perform the binary search to find the position      */
                    i = 0;  k = lenList+1;
                    while ( i+1 < k ) {
                        j = (i + k) / 2;
                        if ( libGAP_LT(libGAP_ADDR_OBJ(list)[j],libGAP_ADDR_OBJ(sub)[l]) )
                            i = j;
                        else
                            k = j;
                    }

                    /* set bit if <sub>[<l>] was found at position k       */
                    if ( k <= lenList
                      && libGAP_EQ( libGAP_ADDR_OBJ(list)[k], libGAP_ADDR_OBJ(sub)[l] ) )
                      libGAP_SET_ELM_BLIST( blist, k, libGAP_True);
                }
            }

        }

        /* if <sub> is large, run over both list in parallel               */
        else {

            /* turn the <sub> into a set for faster searching              */
            if ( ! libGAP_IsSet( sub ) ) {
                sub = libGAP_SetList( sub );
                lenSub = libGAP_LEN_LIST( sub );
            }

            /* allocate the boolean list and get pointer                   */
            blist = libGAP_NewBag( libGAP_T_BLIST, libGAP_SIZE_PLEN_BLIST( lenList ) );
            libGAP_ADDR_OBJ(blist)[0] = libGAP_INTOBJ_INT(lenList);

            /* run over the elements of <list>                             */
            k = 1;
            block = 0;
            bit   = 1;
            for ( l = 1; l <= lenList; l++ ) {

                /* test if <list>[<l>] is in <sub>                         */
                while ( k <= lenSub
                     && libGAP_LT(libGAP_ADDR_OBJ(sub)[k],libGAP_ADDR_OBJ(list)[l]) )
                    k++;

                /* if <list>[<k>] is in <sub> set the current bit in block */
                if ( k <= lenSub
                  && libGAP_EQ(libGAP_ADDR_OBJ(sub)[k],libGAP_ADDR_OBJ(list)[l]) ) {
                    block |= bit;
                    k++;
                }

                /* if block is full add it to boolean list and start next  */
                bit = bit << 1;
                if ( bit == 0 || l == lenList ) {
                  libGAP_BLOCK_ELM_BLIST( blist, l) = block;
                  block = 0;
                  bit   = 1;
                }

            }
        }

    }

    /* if <list> is not a set, we have to use brute force                  */
    else {

        /* convert left argument to an ordinary list, ignore return value  */
        libGAP_PLAIN_LIST( list );

        /* turn <sub> into a set for faster searching                      */
        if ( ! libGAP_IsSet( sub ) )  sub = libGAP_SetList( sub );

        /* allocate the boolean list and get pointer                       */
        lenList  = libGAP_LEN_LIST( list );
        lenSub   = libGAP_LEN_PLIST( sub );
        blist = libGAP_NewBag( libGAP_T_BLIST, libGAP_SIZE_PLEN_BLIST( lenList ) );
        libGAP_ADDR_OBJ(blist)[0] = libGAP_INTOBJ_INT(lenList);

        /* run over the elements of <list>                                 */
        k = 1;
        block = 0;
        bit   = 1;
        for ( l = 1; l <= lenList; l++ ) {

            /* test if <list>[<l>] is in <sub>                             */
            if ( l == 1 || libGAP_LT(libGAP_ADDR_OBJ(list)[l-1],libGAP_ADDR_OBJ(list)[l]) ){
                while ( k <= lenSub
                     && libGAP_LT(libGAP_ADDR_OBJ(sub)[k],libGAP_ADDR_OBJ(list)[l]) )
                    k++;
            }
            else {
                i = 0;  k = libGAP_LEN_PLIST(sub) + 1;
                while ( i+1 < k ) {
                    j = (i + k) / 2;
                    if ( libGAP_LT( libGAP_ADDR_OBJ(sub)[j], libGAP_ADDR_OBJ(list)[l] ) )
                        i = j;
                    else
                        k = j;
                }
            }

            /* if <list>[<k>] is in <sub> set the current bit in the block */
            if ( k <= lenSub
              && libGAP_EQ( libGAP_ADDR_OBJ(sub)[k], libGAP_ADDR_OBJ(list)[l] ) ) {
                block |= bit;
                k++;
            }

            /* if block is full add it to the boolean list and start next  */
            bit = bit << 1;
            if ( bit == 0 || l == lenList ) {
                libGAP_BLOCK_ELM_BLIST( blist, l) =  block;
                block = 0;
                bit   = 1;
            }
        }

    }

    /* return the boolean list                                             */
    return blist;
}



/****************************************************************************
**
*F  FuncLIST_BLIST( <self>, <list>, <blist> ) . make a sublist from a <blist>
**
**  'FuncListBlist' implements the internal function 'ListBlist'.
**
**  'ListBlist( <list>, <blist> )'
**
**  'ListBlist' returns the  sublist of the  elements of the list  <list> for
**  which the boolean list   <blist>, which must   have  the same  length  as
**  <list>, contains 'true'.  The order of the elements in the result is  the
**  same as in <list>.
**
*/
libGAP_Obj libGAP_FuncLIST_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 blist )
{
    libGAP_Obj                 sub;            /* handle of the result            */
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_UInt                n;              /* number of bits in blist         */
    libGAP_UInt                nn;
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the first argument                                    */
    while ( ! libGAP_IS_SMALL_LIST( list ) ) {
        list = libGAP_ErrorReturnObj(
            "ListBlist: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    /* get and check the second argument                                   */
    while ( ! libGAP_IsBlistConv( blist ) ) {
        blist = libGAP_ErrorReturnObj(
            "ListBlist: <blist> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(blist), 0L,
            "you can replace <blist> via 'return <blist>;'" );
    }
    while ( libGAP_LEN_LIST( list ) != libGAP_LEN_BLIST( blist ) ) {
        blist = libGAP_ErrorReturnObj(
            "ListBlist: <blist> must have the same length as <list> (%d)",
            libGAP_LEN_PLIST( list ), 0L,
            "you can replace <blist> via 'return <blist>;'" );
    }

    /* compute the number of 'true'-s                                      */
    n = libGAP_SizeBlist(blist);

    /* make the sublist (we now know its size exactly)                    */
    sub = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE, n );
    libGAP_SET_LEN_PLIST( sub, n );

    /* loop over the boolean list and stuff elements into <sub>            */
    len = libGAP_LEN_LIST( list );
    nn  = 1;
    for ( i = 1;  nn <= n && i <= len;  i++  ) {
        if ( libGAP_ELM_BLIST( blist, i ) == libGAP_True ) {
            libGAP_SET_ELM_PLIST( sub, (libGAP_Int)nn, libGAP_ELMW_LIST( list, (libGAP_Int)i ) );
            libGAP_CHANGED_BAG( sub );
            nn++;
        }
    }

    /* return the sublist                                                  */
    return sub;
}


/****************************************************************************
**

*F  FuncPositionsTrueBlist( <self>, <blist> ) . . . true positions in a blist
**
*N  1992/12/15 martin this depends on 'BIPEB' being 32
*N  Fix up for 64 bit SL
*/
libGAP_Obj libGAP_FuncPositionsTrueBlist (
    libGAP_Obj                 self,
    libGAP_Obj                 blist )
{
    libGAP_Obj                 sub;            /* handle of the result            */
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_UInt  *             ptr;            /* pointer to blist                */
    libGAP_UInt                nrb;            /* number of blocks in blist       */
    libGAP_UInt                m;              /* number of bits in a block       */
    libGAP_UInt                n;              /* number of bits in blist         */
    libGAP_UInt                nn;
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the first argument                                    */
    while ( ! libGAP_IsBlistConv( blist ) ) {
        blist = libGAP_ErrorReturnObj(
            "ListBlist: <blist> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(blist), 0L,
            "you can replace <blist> via 'return <blist>;'" );
    }

    /* compute the number of 'true'-s just as in 'FuncSIZE_BLIST'            */
    nrb = libGAP_NUMBER_BLOCKS_BLIST( blist);
    ptr = libGAP_BLOCKS_BLIST( blist );
    n = 0;
    for ( i = 1; i <= nrb; i++ ) {
        m = *ptr++;
        libGAP_COUNT_TRUES_BLOCK(m);
        n += m;
    }

    /* make the sublist (we now know its size exactly)                    */
    sub = libGAP_NEW_PLIST( libGAP_T_PLIST, n );
    libGAP_SET_LEN_PLIST( sub, n );

    /* loop over the boolean list and stuff elements into <sub>            */
    /* This could be a bit quicker for sparse blists by skipping whole empty
       blocks as we go past                  SL  9/1/97                    */
    len = libGAP_LEN_BLIST( blist );
    nn  = 1;
    for ( i = 1; nn <= n && i <= len;  i++ ) {
        if ( libGAP_ELM_BLIST( blist, i ) == libGAP_True ) {
            libGAP_SET_ELM_PLIST( sub, nn, libGAP_INTOBJ_INT(i) );
            nn++;
        }
    }
    libGAP_CHANGED_BAG(sub);

    /* return the sublist                                                  */
    return sub;
}


/****************************************************************************
**
*F  FuncPositionNthTrueBlist( <self>, <blist>, <Nth> )  . . . find true value
**
*N  1992/12/15 martin this depends on 'BIPEB' being 32
*N  Fixed up for 64 SL
*/
libGAP_Obj libGAP_FuncPositionNthTrueBlist (

    libGAP_Obj                 self,
    libGAP_Obj                 blist,
    libGAP_Obj                 Nth )
{
    libGAP_UInt                nrb;
    libGAP_Int                 nth,  pos,  i;
    libGAP_UInt                m,  mask;
    libGAP_UInt  *             ptr;

    /* Check the arguments. */    
    while ( ! libGAP_IsBlistConv( blist ) ) {
        blist = libGAP_ErrorReturnObj(
            "ListBlist: <blist> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(blist), 0L,
            "you can replace <blist> via 'return <blist>;'" );
    }
    while ( ! libGAP_IS_INTOBJ(Nth) || libGAP_INT_INTOBJ(Nth) <= 0 ) {
        Nth = libGAP_ErrorReturnObj(
            "Position: <nth> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(Nth), 0L,
            "you can replace <nth> via 'return <nth>;'" );
    }
    
    nrb = libGAP_NUMBER_BLOCKS_BLIST(blist);
    if ( ! nrb )  return libGAP_Fail;
    nth = libGAP_INT_INTOBJ( Nth );
    pos = 0;
    ptr = libGAP_BLOCKS_BLIST( blist );
    i = 1;
    m = *ptr;
    libGAP_COUNT_TRUES_BLOCK(m);
    while ( nth > m ) {
        if ( ++i > nrb )  return libGAP_Fail;
        nth -= m;
        pos += libGAP_BIPEB;
        ptr++;
        m = *ptr;
        libGAP_COUNT_TRUES_BLOCK(m);
    }
    m = *ptr;
    mask = 0x1;
    while ( nth > 0 ) {
        pos++;
        if ( m & mask )  nth--;
        mask <<= 1;
    }
    return libGAP_INTOBJ_INT( pos );
}


/****************************************************************************
**
*F  FuncIsSubsetBlist( <self>, <list1>, <list2> ) . . . . . . . . subset test
**
**  'FuncIsSubsetBlist' implements the internal function 'IsSubsetBlist'.
**
**  'IsSubsetBlist( <list1>, <list2> )'
**
**  'IsSubsetBlist' returns 'true' if the boolean list <list2> is a subset of
**  the  boolean  list <list1>, which must  have  equal length.  <list2> is a
**  subset if <list1> if '<list2>[<i>] >= <list1>[<i>]' for all <i>.
*/
libGAP_Obj libGAP_FuncIS_SUB_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    libGAP_UInt *              ptr1;           /* pointer to the first argument   */
    libGAP_UInt *              ptr2;           /* pointer to the second argument  */
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the arguments                                         */
    while ( ! libGAP_IsBlistConv( list1 ) ) {
        list1 = libGAP_ErrorReturnObj(
            "IsSubsetBlist: <blist1> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list1), 0L,
            "you can replace <blist1> via 'return <blist1>;'" );
    }
    while ( ! libGAP_IsBlistConv( list2 ) ) {
        list2 = libGAP_ErrorReturnObj(
            "IsSubsetBlist: <blist2> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list2), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }
    while ( libGAP_LEN_BLIST(list1) != libGAP_LEN_BLIST(list2) ) {
        list2 = libGAP_ErrorReturnObj(
        "IsSubsetBlist: <blist2> must have the same length as <blist1> (%d)",
            libGAP_LEN_BLIST(list1), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }

    /* test for subset property blockwise                                  */
    ptr1 = libGAP_BLOCKS_BLIST(list1);
    ptr2 = libGAP_BLOCKS_BLIST(list2);

    for ( i = libGAP_NUMBER_BLOCKS_BLIST(list1); 0 < i; i-- ) {
        if ( *ptr1 != (*ptr1 | *ptr2) )
            break;
        ptr1++;  ptr2++;
    }

    /* if no counterexample was found, <blist2> is a subset of <blist1>    */
    return (i == 0) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncUNITE_BLIST( <self>, <list1>, <list2> ) . unite one list with another
**
**  'FuncUNITE_BLIST' implements the internal function 'UniteBlist'.
**
**  'UniteBlist( <blist1>, <blist2> )'
**
**  'UniteBlist'  unites  the  boolean list  <blist1>  with  the boolean list
**  <blist2>,  which  must  have the   same  length.  This  is  equivalent to
**  assigning '<blist1>[<i>] := <blist1>[<i>] or <blist2>[<i>]' for all <i>.
*/
libGAP_Obj libGAP_FuncUNITE_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    libGAP_UInt *              ptr1;           /* pointer to the first argument   */
    libGAP_UInt *              ptr2;           /* pointer to the second argument  */
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the arguments                                         */
    while ( ! libGAP_IsBlistConv( list1 ) ) {
        list1 = libGAP_ErrorReturnObj(
            "UniteBlist: <blist1> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list1), 0L,
            "you can replace <blist1> via 'return <blist1>;'" );
    }
    while ( ! libGAP_IsBlistConv( list2 ) ) {
        list2 = libGAP_ErrorReturnObj(
            "UniteBlist: <blist2> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list2), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }
    while ( libGAP_LEN_BLIST(list1) != libGAP_LEN_BLIST(list2) ) {
        list2 = libGAP_ErrorReturnObj(
           "UniteBlist: <blist2> must have the same length as <blist1> (%d)",
            libGAP_LEN_BLIST(list1), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }

    /* compute the union by *or*-ing blockwise                             */
    ptr1 = libGAP_BLOCKS_BLIST(list1);
    ptr2 = libGAP_BLOCKS_BLIST(list2);
    for ( i = (libGAP_LEN_BLIST(list1)+libGAP_BIPEB-1)/libGAP_BIPEB; 0 < i; i-- ) {
        *ptr1++ |= *ptr2++;
    }

    /* return nothing, this function is a procedure                        */
    return 0;
}


/****************************************************************************
**
*F  FuncUNITE_BLIST_LIST( <self>, <list>,<blist>, <sub> )
**
**  'FuncUNITE_BLIST_LIST' implements the internal function 'BlistList'.
**
**  'UniteBlistList( <list>,<blist>, <sub> )'
**
**  'UniteBlistList'  works like `BlistList', but adds the entries to the
**  existing <blist>.
*/
libGAP_Obj libGAP_FuncUNITE_BLIST_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 blist,
    libGAP_Obj                 sub )
{
    libGAP_UInt  *             ptrBlist;       /* pointer to the boolean list     */
    libGAP_UInt                block;          /* one block of boolean list       */
    libGAP_UInt                bit;            /* one bit of block                */
    libGAP_Int                 lenList;        /* logical length of the list      */
    libGAP_Obj *               ptrSub;         /* pointer to the sublist          */
    libGAP_UInt                lenSub;         /* logical length of sublist       */
    libGAP_UInt                i, j, k = 0, l;     /* loop variables                  */
    long                s, t;           /* elements of a range             */

    /* get and check the arguments                                         */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "UniteBlistList: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    while ( ! libGAP_IsBlistConv( blist ) ) {
        blist = libGAP_ErrorReturnObj(
            "UniteBlistList: <blist> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(blist), 0L,
            "you can replace <blist> via 'return <blist>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(sub) ) {
        sub = libGAP_ErrorReturnObj(
            "UniteBlistList: <sub> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(sub), 0L,
            "you can replace <sub> via 'return <sub>;'" );
    }

    /* for a range as subset of a range, it is extremely easy               */
    if ( libGAP_IS_RANGE(list) && libGAP_IS_RANGE(sub) && libGAP_GET_INC_RANGE( list ) == 1
          && libGAP_GET_INC_RANGE( sub ) == 1) {

        /* allocate the boolean list and get pointer                       */
        lenList  = libGAP_GET_LEN_RANGE( list );

        /* check length */
        while ( libGAP_LEN_BLIST(blist) != lenList ) {
            blist = libGAP_ErrorReturnObj(
              "UniteBlistList: <blist> must have the same length as <list> (%d)",
                lenList, 0L,
                "you can replace <blist> via 'return <blist>;'" );
        }

        lenSub   = libGAP_GET_LEN_RANGE( sub );
        ptrBlist = libGAP_BLOCKS_BLIST(blist);

        /* get the bounds of the subset with respect to the boolean list   */
        s = libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( list, 1 ) );
        t = libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( sub, 1 ) );
        if ( s <= t )  i = t - s + 1;
        else           i = 1;

        if ( i + lenSub - 1 <= lenList )  j = i + lenSub - 1;
        else                              j = lenList;

        /* set the corresponding entries to 'true'                         */
        for ( k = i; k <= j && (k-1)%libGAP_BIPEB != 0; k++ )
            ptrBlist[(k-1)/libGAP_BIPEB] |= (1UL << (k-1)%libGAP_BIPEB);
        for ( ; k+libGAP_BIPEB <= j; k += libGAP_BIPEB )
            ptrBlist[(k-1)/libGAP_BIPEB] = ~(libGAP_UInt)0;
        for ( ; k <= j; k++ )
            ptrBlist[(k-1)/libGAP_BIPEB] |= (1UL << (k-1)%libGAP_BIPEB);

    }

    /* for a list as subset of a range, we need basically no search        */
    else if ( libGAP_IS_RANGE(list) && libGAP_GET_INC_RANGE( list) == 1
          && libGAP_IS_PLIST(sub) ) {

        /* allocate the boolean list and get pointer                       */
        lenList  = libGAP_GET_LEN_RANGE( list );

        /* check length */
        while ( libGAP_LEN_BLIST(blist) != lenList ) {
            blist = libGAP_ErrorReturnObj(
              "UniteBlistList: <blist> must have the same length as <list> (%d)",
                lenList, 0L,
                "you can replace <blist> via 'return <blist>;'" );
        }

        lenSub   = libGAP_LEN_LIST( sub );
        ptrBlist = libGAP_BLOCKS_BLIST(blist);
        ptrSub = libGAP_ADDR_OBJ(sub);

        /* loop over <sub> and set the corresponding entries to 'true'     */
        s = libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( list, 1 ) );
        for ( l = 1; l <= libGAP_LEN_LIST(sub); l++ ) {
            if ( ptrSub[l] != 0 ) {

                /* if <sub>[<l>] is an integer it is very easy             */
                if ( libGAP_TNUM_OBJ( ptrSub[l] ) == libGAP_T_INT ) {
                    t = libGAP_INT_INTOBJ( ptrSub[l] ) - s + 1;
                    if ( 0 < t && t <= lenList )
                        ptrBlist[(t-1)/libGAP_BIPEB] |= (1UL << (t-1)%libGAP_BIPEB);
                }

       /* see comment where PosRange was used above   FL */         
                /* otherwise it may be a record, let 'PosRange' handle it  */
              /*  else {
                  Obj pos;
                    pos = PosRange( list, ptrSub[l], 0L );
                    if (pos != Fail)
                      k = INT_INTOBJ(pos);
                    ptrBlist[(k-1)/BIPEB] |= (1UL << (k-1)%BIPEB);
                }    */

            }
        }

    }

    /* if <list> is a set we have two possibilities                        */
    else if ( libGAP_IsSet( list ) ) {

        /* get the length of <list> and its logarithm                      */
        lenList = libGAP_LEN_PLIST( list );

        /* check length */
        while ( libGAP_LEN_BLIST(blist) != lenList ) {
            blist = libGAP_ErrorReturnObj(
              "UniteBlistList: <blist> must have the same length as <list> (%d)",
                lenList, 0L,
                "you can replace <blist> via 'return <blist>;'" );
        }

        for ( i = lenList, l = 0; i != 0; i >>= 1, l++ ) ;
        libGAP_PLAIN_LIST( sub );
        lenSub = libGAP_LEN_LIST( sub );

        /* if <sub> is small, we loop over <sub> and use binary search     */
        if ( l * lenSub < 2 * lenList ) {

            /* allocate the boolean list and get pointer                   */

            /* run over the elements of <sub> and search for the elements  */
            for ( l = 1; l <= libGAP_LEN_LIST(sub); l++ ) {
                if ( libGAP_ADDR_OBJ(sub)[l] != 0 ) {

                    /* perform the binary search to find the position      */
                    i = 0;  k = lenList+1;
                    while ( i+1 < k ) {
                        j = (i + k) / 2;
                        if ( libGAP_LT(libGAP_ADDR_OBJ(list)[j],libGAP_ADDR_OBJ(sub)[l]) )
                            i = j;
                        else
                            k = j;
                    }

                    /* set bit if <sub>[<l>] was found at position k       */
                    if ( k <= lenList
                      && libGAP_EQ( libGAP_ADDR_OBJ(list)[k], libGAP_ADDR_OBJ(sub)[l] ) )
                      libGAP_SET_ELM_BLIST( blist, k, libGAP_True);
                }
            }

        }

        /* if <sub> is large, run over both list in parallel               */
        else {

            /* turn the <sub> into a set for faster searching              */
            if ( ! libGAP_IsSet( sub ) ) {
                sub = libGAP_SetList( sub );
                lenSub = libGAP_LEN_LIST( sub );
            }

            /* run over the elements of <list>                             */
            k = 1;
            block = 0;
            bit   = 1;
            for ( l = 1; l <= lenList; l++ ) {

                /* test if <list>[<l>] is in <sub>                         */
                while ( k <= lenSub
                     && libGAP_LT(libGAP_ADDR_OBJ(sub)[k],libGAP_ADDR_OBJ(list)[l]) )
                    k++;

                /* if <list>[<k>] is in <sub> set the current bit in block */
                if ( k <= lenSub
                  && libGAP_EQ(libGAP_ADDR_OBJ(sub)[k],libGAP_ADDR_OBJ(list)[l]) ) {
                    block |= bit;
                    k++;
                }

                /* if block is full add it to boolean list and start next  */
                bit = bit << 1;
                if ( bit == 0 || l == lenList ) {
                  libGAP_BLOCK_ELM_BLIST( blist, l) |= block;
                  block = 0;
                  bit   = 1;
                }

            }
        }

    }

    /* if <list> is not a set, we have to use brute force                  */
    else {

        /* convert left argument to an ordinary list, ignore return value  */
        libGAP_PLAIN_LIST( list );

        /* turn <sub> into a set for faster searching                      */
        if ( ! libGAP_IsSet( sub ) )  sub = libGAP_SetList( sub );

        /* allocate the boolean list and get pointer                       */
        lenList  = libGAP_LEN_LIST( list );

        /* check length */
        while ( libGAP_LEN_BLIST(blist) != lenList ) {
            blist = libGAP_ErrorReturnObj(
              "UniteBlistList: <blist> must have the same length as <list> (%d)",
                lenList, 0L,
                "you can replace <blist> via 'return <blist>;'" );
        }

        lenSub   = libGAP_LEN_PLIST( sub );

        /* run over the elements of <list>                                 */
        k = 1;
        block = 0;
        bit   = 1;
        for ( l = 1; l <= lenList; l++ ) {

            /* test if <list>[<l>] is in <sub>                             */
            if ( l == 1 || libGAP_LT(libGAP_ADDR_OBJ(list)[l-1],libGAP_ADDR_OBJ(list)[l]) ){
                while ( k <= lenSub
                     && libGAP_LT(libGAP_ADDR_OBJ(sub)[k],libGAP_ADDR_OBJ(list)[l]) )
                    k++;
            }
            else {
                i = 0;  k = libGAP_LEN_PLIST(sub) + 1;
                while ( i+1 < k ) {
                    j = (i + k) / 2;
                    if ( libGAP_LT( libGAP_ADDR_OBJ(sub)[j], libGAP_ADDR_OBJ(list)[l] ) )
                        i = j;
                    else
                        k = j;
                }
            }

            /* if <list>[<k>] is in <sub> set the current bit in the block */
            if ( k <= lenSub
              && libGAP_EQ( libGAP_ADDR_OBJ(sub)[k], libGAP_ADDR_OBJ(list)[l] ) ) {
                block |= bit;
                k++;
            }

            /* if block is full add it to the boolean list and start next  */
            bit = bit << 1;
            if ( bit == 0 || l == lenList ) {
                libGAP_BLOCK_ELM_BLIST( blist,l) |= block;
                block = 0;
                bit   = 1;
            }
        }

    }

    /* return */
    return 0;
}


/****************************************************************************
**
*F  FuncINTER_BLIST( <self>, <list1>, <list2> ) .  <list1> intersection <list2>
**
**  'FuncINTER_BLIST' implements the function 'IntersectBlist'.
**
**  'IntersectBlist( <list1>, <list2> )'
**
**  'IntersectBlist' intersects the   boolean list <list1> with  the  boolean
**  list <list2>, which  must have the  same length.   This is equivalent  to
**  assigning '<list1>[<i>] := <list1>[<i>] and <list2>[<i>]' for all <i>.
*/
libGAP_Obj libGAP_FuncINTER_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    libGAP_UInt  *             ptr1;           /* pointer to the first argument   */
    libGAP_UInt  *             ptr2;           /* pointer to the second argument  */
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the arguments                                         */
    while ( ! libGAP_IsBlistConv( list1 ) ) {
        list1 = libGAP_ErrorReturnObj(
            "IntersectBlist: <blist1> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list1), 0L,
            "you can replace <blist1> via 'return <blist1>;'" );
    }
    while ( ! libGAP_IsBlistConv( list2 ) ) {
        list2 = libGAP_ErrorReturnObj(
            "IntersectBlist: <blist2> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list2), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }
    while ( libGAP_LEN_BLIST(list1) != libGAP_LEN_BLIST(list2) ) {
        list2 = libGAP_ErrorReturnObj(
       "IntersectBlist: <blist2> must have the same length as <blist1> (%d)",
            libGAP_LEN_BLIST(list1), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }

    /* compute the intersection by *and*-ing blockwise                     */
    ptr1 = libGAP_BLOCKS_BLIST(list1);
    ptr2 = libGAP_BLOCKS_BLIST(list2);
    for ( i = libGAP_NUMBER_BLOCKS_BLIST(list1); 0 < i; i-- ) {
        *ptr1++ &= *ptr2++;
    }

    /* return nothing, this function is a procedure                        */
    return 0;
}


/****************************************************************************
**
*F  FuncSUBTR_BLIST( <self>, <list1>, <list2> ) . . . . . . <list1> - <list2>
**
**  'FuncSUBTR_BLIST' implements the internal function 'SubtractBlist'.
**
**  'SubtractBlist( <list1>, <list2> )'
**
**  'SubtractBlist' subtracts the boolean  list <list2> from the boolean list
**  <list1>, which  must have the  same length.  This is equivalent assigning
**  '<list1>[<i>] := <list1>[<i>] and not <list2>[<i>]' for all <i>.
*/
libGAP_Obj libGAP_FuncSUBTR_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    libGAP_UInt *              ptr1;           /* pointer to the first argument   */
    libGAP_UInt *              ptr2;           /* pointer to the second argument  */
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the arguments                                         */
    while ( ! libGAP_IsBlistConv( list1 ) ) {
        list1 = libGAP_ErrorReturnObj(
            "SubtractBlist: <blist1> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list1), 0L,
            "you can replace <blist1> via 'return <blist1>;'" );
    }
    while ( ! libGAP_IsBlistConv( list2 ) ) {
        list2 = libGAP_ErrorReturnObj(
            "SubtractBlist: <blist2> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list2), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }
    while ( libGAP_LEN_BLIST(list1) != libGAP_LEN_BLIST(list2) ) {
        list2 = libGAP_ErrorReturnObj(
        "SubtractBlist: <blist2> must have the same length as <blist1> (%d)",
            libGAP_LEN_BLIST(list1), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }

    /* compute the difference by operating blockwise                       */
    ptr1 = libGAP_BLOCKS_BLIST(list1);
    ptr2 = libGAP_BLOCKS_BLIST(list2); 
    for ( i = libGAP_NUMBER_BLOCKS_BLIST(list1); 0 < i; i-- ) 
      { 
        *ptr1++ &= ~ *ptr2++; 
      }

    /* return nothing, this function is a procedure */ return 0; }

/****************************************************************************
**
*F  FuncMEET_BLIST( <self>, <list1>, <list2> ) . . . 
**
**  'FuncSUBTR_BLIST' implements the internal function 'MeetBlist'.
**
**  'MeetBlist( <list1>, <list2> )'
**
**  'MeetBlist' returns true if list1 and list2 have true in the same
**  position and false otherwise. It is equivalent to, but faster than
**  SizeBlist(IntersectionBlist(list1, list2)) <> 0
**  The lists must have the same length.
*/

libGAP_Obj libGAP_FuncMEET_BLIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    libGAP_UInt *              ptr1;           /* pointer to the first argument   */
    libGAP_UInt *              ptr2;           /* pointer to the second argument  */
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the arguments                                         */
    while ( ! libGAP_IsBlistConv( list1 ) ) {
        list1 = libGAP_ErrorReturnObj(
            "MeetBlist: <blist1> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list1), 0L,
            "you can replace <blist1> via 'return <blist1>;'" );
    }
    while ( ! libGAP_IsBlistConv( list2 ) ) {
        list2 = libGAP_ErrorReturnObj(
            "MeetBlist: <blist2> must be a boolean list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list2), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }
    while ( libGAP_LEN_BLIST(list1) != libGAP_LEN_BLIST(list2) ) {
        list2 = libGAP_ErrorReturnObj(
        "MeetBlist: <blist2> must have the same length as <blist1> (%d)",
            libGAP_LEN_BLIST(list1), 0L,
            "you can replace <blist2> via 'return <blist2>;'" );
    }

    /* compute the difference by operating blockwise                       */
    ptr1 = libGAP_BLOCKS_BLIST(list1);
    ptr2 = libGAP_BLOCKS_BLIST(list2); 
    for ( i = libGAP_NUMBER_BLOCKS_BLIST(list1); 0 < i; i-- ) 
      { 
	if (*ptr1++ & *ptr2++) return libGAP_True;
      }

    return libGAP_False;
}


/****************************************************************************
**
**
*F  MakeImmutableBlist( <blist> )
*/

void libGAP_MakeImmutableBlist( libGAP_Obj blist )
{
  libGAP_RetypeBag(blist, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(blist)));
}

/****************************************************************************
**
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  BagNames  . . . . . . . . . . . . . . . . . . . . . . . list of bag names
*/
static libGAP_StructBagNames libGAP_BagNames[] = {
  { libGAP_T_BLIST,                           "list (boolean)"                  },
  { libGAP_T_BLIST       +libGAP_IMMUTABLE,          "list (boolean,imm)"              },
  { libGAP_T_BLIST                  +libGAP_COPYING, "list (boolean,copied)"           },
  { libGAP_T_BLIST       +libGAP_IMMUTABLE +libGAP_COPYING, "list (boolean,imm,copied)"       },
  { libGAP_T_BLIST_NSORT,                     "list (boolean,nsort)"            },
  { libGAP_T_BLIST_NSORT +libGAP_IMMUTABLE,          "list (boolean,nsort,imm)"        },
  { libGAP_T_BLIST_NSORT            +libGAP_COPYING, "list (boolean,nsort,copied)"     },
  { libGAP_T_BLIST_NSORT +libGAP_IMMUTABLE +libGAP_COPYING, "list (boolean,nsort,imm,copied)" },
  { libGAP_T_BLIST_SSORT,                     "list (boolean,ssort)"            },
  { libGAP_T_BLIST_SSORT +libGAP_IMMUTABLE,          "list (boolean,ssort,imm)"        },
  { libGAP_T_BLIST_SSORT            +libGAP_COPYING, "list (boolean,ssort,copied)"     },
  { libGAP_T_BLIST_SSORT +libGAP_IMMUTABLE +libGAP_COPYING, "list (boolean,ssort,imm,copied)" },
  { -1,                                ""                                }
};


/****************************************************************************
**
*V  ClearFiltsTab . . . . . . . . . . . . . . . . . . . .  clear filter tnums
*/
static libGAP_Int libGAP_ClearFiltsTab [] = {
    libGAP_T_BLIST,                 libGAP_T_BLIST,
    libGAP_T_BLIST      +libGAP_IMMUTABLE, libGAP_T_BLIST+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT,           libGAP_T_BLIST,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE, libGAP_T_BLIST+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT,           libGAP_T_BLIST,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE, libGAP_T_BLIST+libGAP_IMMUTABLE,
    -1,                      -1
};


/****************************************************************************
**
*V  HasFiltTab  . . . . . . . . . . . . . . . . . . . . .  tester filter tnum
*/
static libGAP_Int libGAP_HasFiltTab [] = {

    /* mutable boolean list                                                */
    libGAP_T_BLIST,                    libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_BLIST,                    libGAP_FN_IS_EMPTY,    0,
    libGAP_T_BLIST,                    libGAP_FN_IS_DENSE,    1,
    libGAP_T_BLIST,                    libGAP_FN_IS_NDENSE,   0,
    libGAP_T_BLIST,                    libGAP_FN_IS_HOMOG,    1,
    libGAP_T_BLIST,                    libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_BLIST,                    libGAP_FN_IS_TABLE,    0,
    libGAP_T_BLIST,                    libGAP_FN_IS_SSORT,    0,
    libGAP_T_BLIST,                    libGAP_FN_IS_NSORT,    0,

    /* immutable boolean list                                               */
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    0,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    1,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   0,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    1,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    0,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    0,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    0,

    /* nsort mutable boolean list                                          */
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_EMPTY,    0,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_DENSE,    1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NDENSE,   0,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_HOMOG,    1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_TABLE,    0,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_SSORT,    0,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NSORT,    1,

    /* nsort immutable boolean list                                        */
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    0,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    1,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   0,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    1,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    0,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    0,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    1,

    /* ssort mutable boolean list                                          */
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_EMPTY,    0,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_DENSE,    1,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NDENSE,   0,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_HOMOG,    1,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_TABLE,    0,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_SSORT,    1,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NSORT,    0,

    /* ssort immutable boolean list                                        */
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    0,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    1,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   0,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    1,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    0,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    1,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    0,

    -1,                         -1,             -1
};


/****************************************************************************
**
*V  SetFiltTab  . . . . . . . . . . . . . . . . . . . . .  setter filter tnum
*/
static libGAP_Int libGAP_SetFiltTab [] = {

    /* mutable boolean list                                                */
    libGAP_T_BLIST,                    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST,                    libGAP_FN_IS_DENSE,    libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_BLIST,                    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_BLIST,                    libGAP_FN_IS_TABLE,    -1,
    libGAP_T_BLIST,                    libGAP_FN_IS_SSORT,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST,                    libGAP_FN_IS_NSORT,    libGAP_T_BLIST_NSORT,

    /* immutable boolean list                                              */
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    -1,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,

    /* nsort mutable boolean list                                          */
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_EMPTY,    -1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_DENSE,    libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_TABLE,    -1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_SSORT,    -1,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NSORT,    libGAP_T_BLIST_NSORT,

    /* nsort immutable boolean list                                        */
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    -1,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    -1,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    -1,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,

    /* ssort mutable boolean list                                          */
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_DENSE,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_TABLE,    -1,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_SSORT,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NSORT,    -1,

    /* ssort immutable boolean list                                        */
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    -1,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    -1,

    -1,                         -1,             -1

};


/****************************************************************************
**
*V  ResetFiltTab  . . . . . . . . . . . . . . . . . . .  unsetter filter tnum
*/
static libGAP_Int libGAP_ResetFiltTab [] = {

    /* mutable boolean list                                                */
    libGAP_T_BLIST,                    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST,                    libGAP_FN_IS_EMPTY,    libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_DENSE,    libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_NDENSE,   libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_NHOMOG,   libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_TABLE,    libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_SSORT,    libGAP_T_BLIST,
    libGAP_T_BLIST,                    libGAP_FN_IS_NSORT,    libGAP_T_BLIST,

    /* immutable boolean list                                               */
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST      +libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    libGAP_T_BLIST      +libGAP_IMMUTABLE,

    /* nsort mutable boolean list                                          */
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_DENSE,    libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NDENSE,   libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NHOMOG,   libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_TABLE,    libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_SSORT,    libGAP_T_BLIST_NSORT,
    libGAP_T_BLIST_NSORT,              libGAP_FN_IS_NSORT,    libGAP_T_BLIST,

    /* nsort immutable boolean list                                        */
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_BLIST      +libGAP_IMMUTABLE,

    /* ssort mutable boolean list                                          */
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_DENSE,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NDENSE,   libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NHOMOG,   libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_TABLE,    libGAP_T_BLIST_SSORT,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_SSORT,    libGAP_T_BLIST,
    libGAP_T_BLIST_SSORT,              libGAP_FN_IS_NSORT,    libGAP_T_BLIST_SSORT,

    /* ssort immutable boolean list                                        */
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_BLIST      +libGAP_IMMUTABLE,
    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,

    -1,                         -1,             -1

};


/****************************************************************************
**
*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_BLIST", "obj", &libGAP_IsBlistFilt, 
      libGAP_FuncIS_BLIST, "src/blister.c:IS_BLIST" },

    { "IS_BLIST_REP", "obj", &libGAP_IsBlistRepFilt, 
      libGAP_FuncIS_BLIST_REP, "src/blister.c:IS_BLIST_REP" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "IS_BLIST_CONV", 1, "obj", 
      libGAP_FuncIS_BLIST_CONV, "src/blister.c:IS_BLIST_CONV" },

    { "CONV_BLIST", 1, "blist",
      libGAP_FuncCONV_BLIST, "src/blister.c:CONV_BLIST" },

    { "BLIST_LIST", 2, "list, sub",
      libGAP_FuncBLIST_LIST, "src/blister.c:BLIST_LIST" },

    { "LIST_BLIST", 2, "list, blist",
      libGAP_FuncLIST_BLIST, "src/blister.c:LIST_BLIST" },

    { "SIZE_BLIST", 1, "blist",
      libGAP_FuncSIZE_BLIST, "src/blister.c:SIZE_BLIST" },

    { "IS_SUB_BLIST", 2, "blist1, blist2",
      libGAP_FuncIS_SUB_BLIST, "src/blister.c:IS_SUB_BLIST" },

    { "UNITE_BLIST", 2, "blist1, blist2",
      libGAP_FuncUNITE_BLIST, "src/blister.c:UNITE_BLIST" },

    { "UNITE_BLIST_LIST", 3, "list, blist, sub",
      libGAP_FuncUNITE_BLIST_LIST, "src/blister.c:UNITE_BLIST_LIST" },

    { "INTER_BLIST", 2, "blist1, blist2",
      libGAP_FuncINTER_BLIST, "src/blister.c:INTER_BLIST" },

    { "SUBTR_BLIST", 2, "blist1, blist2",
      libGAP_FuncSUBTR_BLIST, "src/blister.c:SUBTR_BLIST" },

    { "MEET_BLIST", 2, "blist1, blist2",
      libGAP_FuncMEET_BLIST, "src/blister.c:MEET_BLIST" },

    { "PositionNthTrueBlist", 2, "blist, nth",
      libGAP_FuncPositionNthTrueBlist, "src/blister.c:PositionNthTrueBlist" },

    { "PositionsTrueBlist", 1, "blist",
      libGAP_FuncPositionsTrueBlist, "src/blister.c:PositionsTrueBlist" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                t1;
    libGAP_UInt                t2;

    /* check dependencies                                                  */
    libGAP_RequireModule( libGAP_module, "lists", 403600000UL );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* GASMAN marking functions and GASMAN names                           */
    libGAP_InitBagNamesFromTable( libGAP_BagNames );

    for ( t1 = libGAP_T_BLIST;  t1 <= libGAP_T_BLIST_SSORT;  t1 += 2 ) {
        libGAP_InitMarkFuncBags( t1,                      libGAP_MarkNoSubBags  );
        libGAP_InitMarkFuncBags( t1 +libGAP_IMMUTABLE,           libGAP_MarkNoSubBags  );
        libGAP_InitMarkFuncBags( t1            +libGAP_COPYING , libGAP_MarkOneSubBags );
        libGAP_InitMarkFuncBags( t1 +libGAP_IMMUTABLE +libGAP_COPYING , libGAP_MarkOneSubBags );
    }

    /* Make immutable blists public					   */
    for ( t1 = libGAP_T_BLIST; t1 <= libGAP_T_BLIST_SSORT; t1 += 2 ) {
        libGAP_MakeBagTypePublic( t1 + libGAP_IMMUTABLE );
    }

    /* install the type methods                                            */
    libGAP_TypeObjFuncs[ libGAP_T_BLIST ] = libGAP_TypeBlistMut;
    libGAP_TypeObjFuncs[ libGAP_T_BLIST +libGAP_IMMUTABLE ] = libGAP_TypeBlistImm;
    libGAP_TypeObjFuncs[ libGAP_T_BLIST_NSORT ] = libGAP_TypeBlistNSortMut;
    libGAP_TypeObjFuncs[ libGAP_T_BLIST_NSORT +libGAP_IMMUTABLE ] = libGAP_TypeBlistNSortImm;
    libGAP_TypeObjFuncs[ libGAP_T_BLIST_SSORT ] = libGAP_TypeBlistSSortMut;
    libGAP_TypeObjFuncs[ libGAP_T_BLIST_SSORT +libGAP_IMMUTABLE ] = libGAP_TypeBlistSSortImm;

    /* initialise list tables                                              */
    libGAP_InitClearFiltsTNumsFromTable   ( libGAP_ClearFiltsTab );
    libGAP_InitHasFiltListTNumsFromTable  ( libGAP_HasFiltTab    );
    libGAP_InitSetFiltListTNumsFromTable  ( libGAP_SetFiltTab    );
    libGAP_InitResetFiltListTNumsFromTable( libGAP_ResetFiltTab  );

    /* Install the saving functions -- cannot save while copying           */
    for ( t1 = libGAP_T_BLIST;  t1 <= libGAP_T_BLIST_SSORT;  t1 += 2 ) {
        libGAP_SaveObjFuncs[ t1            ] = libGAP_SaveBlist;
        libGAP_SaveObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_SaveBlist;
        libGAP_LoadObjFuncs[ t1            ] = libGAP_LoadBlist;
        libGAP_LoadObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_LoadBlist;
    }

    /* install the copy functions                                          */
    for ( t1 = libGAP_T_BLIST; t1 <= libGAP_T_BLIST_SSORT; t1 += 2 ) {
        libGAP_CopyObjFuncs [ t1                     ] = libGAP_CopyBlist;
        libGAP_CopyObjFuncs [ t1 +libGAP_IMMUTABLE          ] = libGAP_CopyBlist;
        libGAP_CopyObjFuncs [ t1            +libGAP_COPYING ] = libGAP_CopyBlistCopy;
        libGAP_CopyObjFuncs [ t1 +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CopyBlistCopy;
        libGAP_CleanObjFuncs[ t1                     ] = libGAP_CleanBlist;
        libGAP_CleanObjFuncs[ t1 +libGAP_IMMUTABLE          ] = libGAP_CleanBlist;
        libGAP_CleanObjFuncs[ t1            +libGAP_COPYING ] = libGAP_CleanBlistCopy;
        libGAP_CleanObjFuncs[ t1 +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CleanBlistCopy;
	libGAP_ShallowCopyObjFuncs[ t1 ] = libGAP_ShallowCopyBlist;
	libGAP_ShallowCopyObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_ShallowCopyBlist;
    }

    /* install the comparison methods                                      */
    for ( t1 = libGAP_T_BLIST;  t1 <= libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE;  t1++ ) {
        for ( t2 = libGAP_T_BLIST;  t2 <= libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE;  t2++ ) {
            libGAP_EqFuncs[ t1 ][ t2 ] = libGAP_EqBlist;
        }
    }

    /* install the list functions in the tables                            */
    for ( t1 = libGAP_T_BLIST; t1 <= libGAP_T_BLIST_SSORT; t1 += 2 ) {
        libGAP_LenListFuncs    [ t1            ] = libGAP_LenBlist;
        libGAP_LenListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_LenBlist;
        libGAP_IsbListFuncs    [ t1            ] = libGAP_IsbBlist;
        libGAP_IsbListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbBlist;
        libGAP_IsbvListFuncs   [ t1            ] = libGAP_IsbvBlist;
        libGAP_IsbvListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbvBlist;
        libGAP_Elm0ListFuncs   [ t1            ] = libGAP_Elm0Blist;
        libGAP_Elm0ListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_Elm0Blist;
        libGAP_Elm0vListFuncs  [ t1            ] = libGAP_Elm0vBlist;
        libGAP_Elm0vListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_Elm0vBlist;
        libGAP_ElmListFuncs    [ t1            ] = libGAP_ElmBlist;
        libGAP_ElmListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmBlist;
        libGAP_ElmvListFuncs   [ t1            ] = libGAP_ElmvBlist;
        libGAP_ElmvListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmvBlist;
        libGAP_ElmwListFuncs   [ t1            ] = libGAP_ElmvBlist;
        libGAP_ElmwListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmvBlist;
        libGAP_ElmsListFuncs   [ t1            ] = libGAP_ElmsBlist;
        libGAP_ElmsListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmsBlist;
        libGAP_AssListFuncs    [ t1            ] = libGAP_AssBlist;
        libGAP_AssListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_AssBlistImm;
        libGAP_AsssListFuncs   [ t1            ] = libGAP_AsssListDefault;
        libGAP_AsssListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_AsssBlistImm;
        libGAP_IsDenseListFuncs[ t1            ] = libGAP_IsDenseBlist;
        libGAP_IsDenseListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsDenseBlist;
        libGAP_IsHomogListFuncs[ t1            ] = libGAP_IsHomogBlist;
        libGAP_IsHomogListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsHomogBlist;
        libGAP_IsSSortListFuncs[ t1            ] = libGAP_IsSSortBlist;
        libGAP_IsSSortListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsSSortBlist;
        libGAP_IsPossListFuncs [ t1            ] = libGAP_IsPossBlist;
        libGAP_IsPossListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_IsPossBlist;
        libGAP_PosListFuncs    [ t1            ] = libGAP_PosBlist;
        libGAP_PosListFuncs    [ t1 +libGAP_IMMUTABLE ] = libGAP_PosBlist;
        libGAP_PlainListFuncs  [ t1            ] = libGAP_PlainBlist;
        libGAP_PlainListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_PlainBlist;
        libGAP_MakeImmutableObjFuncs [ t1      ] = libGAP_MakeImmutableBlist;
    }
    libGAP_IsSSortListFuncs[ libGAP_T_BLIST_NSORT            ] = libGAP_IsSSortBlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_BLIST_NSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortBlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_BLIST_SSORT            ] = libGAP_IsSSortBlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_BLIST_SSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortBlistYes;

    /* Import the types of blists: */
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_MUT", &libGAP_TYPE_BLIST_MUT );
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_IMM", &libGAP_TYPE_BLIST_IMM );
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_NSORT_MUT", &libGAP_TYPE_BLIST_NSORT_MUT );
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_NSORT_IMM", &libGAP_TYPE_BLIST_NSORT_IMM );
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_SSORT_MUT", &libGAP_TYPE_BLIST_SSORT_MUT );
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_SSORT_IMM", &libGAP_TYPE_BLIST_SSORT_IMM );
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_EMPTY_MUT", &libGAP_TYPE_BLIST_EMPTY_MUT );
    libGAP_ImportGVarFromLibrary( "TYPE_BLIST_EMPTY_IMM", &libGAP_TYPE_BLIST_EMPTY_IMM );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoBlist() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "blister",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoBlist ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*E  blister.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
