/****************************************************************************
**
*W  range.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions that deal with ranges.
**
**  A *range* is  a list without  holes  consisting  of consecutive integers.
**  For the full definition of ranges see chapter "Ranges" in the GAP Manual.
**  Read  also   "More about Ranges"  about  the different  representation of
**  ranges.
**
**  A list that is  known to be  a  range is  represented  by a  bag of  type
**  'T_RANGE', which has the following format:
**
**      +-------+-------+-------+
**      |logical| first | incr- |
**      | length|element| ement |
**      +-------+-------+-------+
**
**  The first entry is the handle of the logical length.  The second entry is
**  the first element of the range.  The last  entry  is the  increment.  All
**  three are represented as immediate GAP integers.
**
**  The element at position <pos> is thus simply <first> + (<pos>-1) * <inc>.
**
**  Note  that  a list  represented by a   bag of type   'T_LIST', 'T_SET' or
**  'T_VECTOR' might still  be a range.  It is  just that the kernel does not
**  know this.
**
**  This package consists of three parts.
**
**  The  first part   consists   of  the  macros    'NEW_RANGE',  'IS_RANGE',
**  'SET_LEN_RANGE',   'GET_LEN_RANGE',    'SET_LOW_RANGE',  'GET_LOW_RANGE',
**  'SET_INC_RANGE', 'GET_INC_RANGE',    and 'GET_ELM_RANGE'.  They determine
**  the representation of ranges.  Everything else in  this file and the rest
**  of the {\GAP} kernel uses those macros to access and modify ranges.
**
**  The  second part  consists  of   the functions  'LenRange',   'ElmRange',
**  'ElmsRange',   'AssRange',      'AsssRange',   'PosRange',  'PlainRange',
**  'IsDenseRange',   'IsPossRange', 'PrintRange', 'EqRange', and  'LtRange'.
**  They  are the  functions required by  the generic   lists package.  Using
**  these functions the other parts of the {\GAP} kernel can access or modify
**  ranges without actually being aware that they are dealing with a range.
**
**  The  third part consists  ...
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */


/****************************************************************************
**

*F  NEW_RANGE() . . . . . . . . . . . . . . . . . . . . . .  make a new range
**
**  'NEW_RANGE' returns a new range.  Note that  you must set the length, the
**  low value, and the increment before you can use the range.
**
**  'NEW_RANGE' is defined in the declaration part of this package as follows
**
#define NEW_RANGE_NSORT() NewBag( T_RANGE_NSORT, 3 * sizeof(Obj) )
#define NEW_RANGE_SSORT() NewBag( T_RANGE_SSORT, 3 * sizeof(Obj) )
*/


/****************************************************************************
**
*F  IS_RANGE(<val>) . . . . . . . . . . . . . . .  test if a value is a range
**
**  'IS_RANGE' returns 1  if the value  <val> is known  to be a range,  and 0
**  otherwise.  Note that a list for which 'IS_RANGE' returns  0 may still be
**  a range, but  the kernel does not know  this yet.  Use  'IsRange' to test
**  whether a list is a range.
**
**  Note that  'IS_RANGE' is a  macro, so do not  call it with arguments that
**  have side effects.
**
**  'IS_RANGE' is defined in the declaration part of this package as follows
**
#define IS_RANGE(val)   (TNUM_OBJ(val)==T_RANGE_NSORT || TNUM_OBJ(val)==T_RANGE_SSORT)
*/


/****************************************************************************
**
*F  SET_LEN_RANGE(<list>,<len>) . . . . . . . . . . set the length of a range
**
**  'SET_LEN_RANGE' sets the length  of the range <list>  to the value <len>,
**  which must be a C integer larger than 1.
**
**  Note that 'SET_LEN_RANGE' is a macro,  so  do not  call it with arguments
**  that have side effects.
**
**  'SET_LEN_RANGE' is  defined in  the declaration part  of  this package as
**  follows
**
#define SET_LEN_RANGE(list,len)         (ADDR_OBJ(list)[0] = INTOBJ_INT(len))
*/


/****************************************************************************
**
*F  GET_LEN_RANGE(<list>) . . . . . . . . . . . . . . . . . length of a range
**
**  'GET_LEN_RANGE' returns the  logical length of  the range <list>, as  a C
**  integer.
**
**  Note that  'GET_LEN_RANGE' is a macro, so  do not call  it with arguments
**  that have side effects.
**
**  'GET_LEN_RANGE' is  defined in  the declaration part  of this  package as
**  follows
**
#define GET_LEN_RANGE(list)             INT_INTOBJ( ADDR_OBJ(list)[0] )
*/


/****************************************************************************
**
*F  SET_LOW_RANGE(<list>,<low>) . . . . . .  set the first element of a range
**
**  'SET_LOW_RANGE' sets the  first element of the range  <list> to the value
**  <low>, which must be a C integer.
**
**  Note  that 'SET_LOW_RANGE' is a macro, so do not call  it with  arguments
**  that have side effects.
**
**  'SET_LOW_RANGE' is defined  in the declaration  part  of this package  as
**  follows
**
#define SET_LOW_RANGE(list,low)         (ADDR_OBJ(list)[1] = INTOBJ_INT(low))
*/


/****************************************************************************
**
*F  GET_LOW_RANGE(<list>) . . . . . . . . . . . . .  first element of a range
**
**  'GET_LOW_RANGE' returns the first  element  of the  range  <list> as a  C
**  integer.
**
**  Note that 'GET_LOW_RANGE' is a  macro, so do not  call it with  arguments
**  that have side effects.
**
**  'GET_LOW_RANGE'  is defined in  the declaration  part  of this package as
**  follows
**
#define GET_LOW_RANGE(list)             INT_INTOBJ( ADDR_OBJ(list)[1] )
*/


/****************************************************************************
**
*F  SET_INC_RANGE(<list>,<inc>) . . . . . . . .  set the increment of a range
**
**  'SET_INC_RANGE' sets  the  increment of  the range  <list>   to the value
**  <inc>, which must be a C integer.
**
**  Note that  'SET_INC_RANGE' is a macro,  so do  not call it with arguments
**  that have side effects.
**
**  'SET_INC_RANGE' is  defined  in the  declaration part of  this package as
**  follows
**
#define SET_INC_RANGE(list,inc)         (ADDR_OBJ(list)[2] = INTOBJ_INT(inc))
*/


/****************************************************************************
**
*F  GET_INC_RANGE(<list>) . . . . . . . . . . . . . . .  increment of a range
**
**  'GET_INC_RANGE' returns the increment of the range <list> as a C integer.
**
**  Note  that 'GET_INC_RANGE' is  a macro, so  do not call it with arguments
**  that have side effects.
**
**  'GET_INC_RANGE' is  defined  in the  declaration part  of this package as
**  follows
**
#define GET_INC_RANGE(list)             INT_INTOBJ( ADDR_OBJ(list)[2] )
*/


/****************************************************************************
**
*F  GET_ELM_RANGE(<list>,<pos>) . . . . . . . . . . . . .  element of a range
**
**  'GET_ELM_RANGE' return  the <pos>-th element  of the range <list>.  <pos>
**  must be a positive integer less than or equal to the length of <list>.
**
**  Note that 'GET_ELM_RANGE'  is a macro, so do  not call  it with arguments
**  that have side effects.
**
**  'GET_ELM_RANGE'  is  defined in the  declaration part  of this package as
**  follows
**
#define GET_ELM_RANGE(list,pos)         INTOBJ_INT( GET_LOW_RANGE(list) \
                                          + ((pos)-1) * GET_INC_RANGE(list) )
*/


/****************************************************************************
**

*F  TypeRangeNSortImmutable( <range> )  . . . . . . . . . . . kind of a range
**
**  'TypeRangeNSortMutable' is the   function in 'TypeObjFuncs' for immutable
**  ranges which are not strictly sorted.
*/
libGAP_Obj libGAP_TYPE_RANGE_NSORT_IMMUTABLE;

libGAP_Obj libGAP_TypeRangeNSortImmutable (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_RANGE_NSORT_IMMUTABLE;
}
    
/****************************************************************************
**
*F  TypeRangeNSortMutable( <range> )  . . . . . . . . . . . . kind of a range
**
**  'TypeRangeNSortMutable' is the   function in 'TypeObjFuncs' for   mutable
**  ranges which are not strictly sorted.
*/
libGAP_Obj libGAP_TYPE_RANGE_NSORT_MUTABLE;

libGAP_Obj libGAP_TypeRangeNSortMutable (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_RANGE_NSORT_MUTABLE;
}
    
/****************************************************************************
**
*F  TypeRangeSSortImmutable( <range> )  . . . . . . . . . . . kind of a range
**
**  'TypeRangeNSortMutable' is the   function in 'TypeObjFuncs' for immutable
**  ranges which are strictly sorted.
*/
libGAP_Obj libGAP_TYPE_RANGE_SSORT_IMMUTABLE;

libGAP_Obj libGAP_TypeRangeSSortImmutable (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_RANGE_SSORT_IMMUTABLE;
}


/****************************************************************************
**
*F  TypeRangeSSortMutable( <range> )  . . . . . . . . . . . . kind of a range
**
**  'TypeRangeNSortMutable' is the   function in 'TypeObjFuncs' for   mutable
**  ranges which are strictly sorted.
*/
libGAP_Obj libGAP_TYPE_RANGE_SSORT_MUTABLE;

libGAP_Obj libGAP_TypeRangeSSortMutable (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_RANGE_SSORT_MUTABLE;
}
    
/****************************************************************************
**

*F  CopyRange( <list>, <mut> )  . . . . . . . . . . . . . . . .  copy a range
**
**  'CopyRange' returns a structural (deep) copy of the range <list>, i.e., a
**  recursive copy that preserves the structure.
**
**  If <list> has not  yet  been copied, it makes   a copy, leaves  a forward
**  pointer to the copy  in the first entry of  the range, where the size  of
**  the range usually  resides, and copies the other  entries.   If the range
**  has already been copied, it returns the value of the forwarding pointer.
**
**  'CopyRange' is the function in 'CopyObjFuncs' for ranges.
**
**  'CleanRange' removes the mark  and the forwarding  pointer from the range
**  <list>.
**
**  'CleanRange' is the function in 'CleanObjFuncs' for ranges.
*/
libGAP_Obj libGAP_CopyRange (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* copy, result                    */

    /* don't change immutable objects                                      */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        return list;
    }

    /* make a copy                                                         */
    if ( mut ) {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(list), libGAP_SIZE_OBJ(list) );
    }
    else {
        copy = libGAP_NewBag( libGAP_IMMUTABLE_TNUM( libGAP_TNUM_OBJ(list) ), libGAP_SIZE_OBJ(list) );
    }
    libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(list)[0];

    /* leave a forwarding pointer                                          */
    libGAP_ADDR_OBJ(list)[0] = copy;
    libGAP_CHANGED_BAG( list );

    /* now it is copied                                                    */
    libGAP_RetypeBag( list, libGAP_TNUM_OBJ(list) + libGAP_COPYING );

    /* copy the subvalues                                                  */
    libGAP_ADDR_OBJ(copy)[1] = libGAP_ADDR_OBJ(list)[1];
    libGAP_ADDR_OBJ(copy)[2] = libGAP_ADDR_OBJ(list)[2];

    /* return the copy                                                     */
    return copy;
}


/****************************************************************************
**
*F  CopyRangeCopy( <list>, <mut> )  . . . . . . . . . . . copy a copied range
*/
libGAP_Obj libGAP_CopyRangeCopy (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{
    return libGAP_ADDR_OBJ(list)[0];
}


/****************************************************************************
**
*F  CleanRange( <list> )  . . . . . . . . . . . . . . . . .  clean up a range
*/
void libGAP_CleanRange (
    libGAP_Obj                 list )
{
}


/****************************************************************************
**
*F  CleanRange( <list> )  . . . . . . . . . . . . . . clean up a copied range
*/
void libGAP_CleanRangeCopy (
    libGAP_Obj                 list )
{
    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(list)[0] = libGAP_ADDR_OBJ( libGAP_ADDR_OBJ(list)[0] )[0];

    /* now it is cleaned                                                   */
    libGAP_RetypeBag( list, libGAP_TNUM_OBJ(list) - libGAP_COPYING );
}


/****************************************************************************
**

*F  PrintRange(<list>)  . . . . . . . . . . . . . . . . . . . . print a range
**
**  'PrintRange' prints the range <list>.
**
**  'PrintRange' handles bags of type 'T_RANGE'.
*/
void            libGAP_PrintRange (
    libGAP_Obj                 list )
{
    libGAP_Pr( "%2>[ %2>%d",   
       libGAP_GET_LOW_RANGE(list), 0L );
    if ( libGAP_GET_INC_RANGE(list) != 1 ) {
        libGAP_Pr( "%<,%< %2>%d",
           libGAP_GET_LOW_RANGE(list)+libGAP_GET_INC_RANGE(list), 0L );
    }
    libGAP_Pr( "%2< .. %2>%d%4< ]",
       libGAP_GET_LOW_RANGE(list)+(libGAP_GET_LEN_RANGE(list)-1)*libGAP_GET_INC_RANGE(list), 0L );
}


/****************************************************************************
**
*F  EqRange(<listL>,<listR>)  . . . . . . . . .  test if two ranges are equal
**
**  'EqRange' returns 'true' if the two ranges <listL>  and <listR> are equal
**  and 'false' otherwise.
*/
libGAP_Int             libGAP_EqRange (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    return ( libGAP_GET_LEN_RANGE(listL) == libGAP_GET_LEN_RANGE(listR)
          && libGAP_GET_LOW_RANGE(listL) == libGAP_GET_LOW_RANGE(listR)
          && libGAP_GET_INC_RANGE(listL) == libGAP_GET_INC_RANGE(listR) );
}


/****************************************************************************
**
*F  LtRange(<listL>,<listR>)  . . . . . . . . .  test if two ranges are equal
**
**  'LtRange' returns 'true'  if  the range  <listL> is less  than the  range
**  <listR> and 'false' otherwise.
*/
libGAP_Int             libGAP_LtRange (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    /* first compare the first elements                                    */
    if ( libGAP_GET_LOW_RANGE(listL) < libGAP_GET_LOW_RANGE(listR) )
        return 1L;
    else if ( libGAP_GET_LOW_RANGE(listR) < libGAP_GET_LOW_RANGE(listL) )
        return 0L;

    /* next compare the increments (or the second elements)                */
    if ( libGAP_GET_INC_RANGE(listL) < libGAP_GET_INC_RANGE(listR) )
        return 1L;
    else if ( libGAP_GET_INC_RANGE(listR) < libGAP_GET_INC_RANGE(listL) )
        return 0L;

    /* finally compare the lengths                                         */
    if ( libGAP_GET_LEN_RANGE(listL) < libGAP_GET_LEN_RANGE(listR) )
        return 1L;
    else if ( libGAP_GET_LEN_RANGE(listR) < libGAP_GET_LEN_RANGE(listL) )
        return 0L;

    /* the two ranges are equal                                            */
    return 0L;
}


/****************************************************************************
**
*F  LenRange(<list>)  . . . . . . . . . . . . . . . . . . . length of a range
**
**  'LenRange' returns the length of the range <list> as a C integer.
**
**  'LenRange' is the function in 'LenListFuncs' for ranges.
*/
libGAP_Int             libGAP_LenRange (
    libGAP_Obj                 list )
{
    return libGAP_GET_LEN_RANGE( list );
}


/****************************************************************************
**
*F  IsbRange(<list>,<pos>)  . . . . . . . .  test for an element from a range
**
**  'IsbRange' returns 1 if the range has an element at  the  position  <pos>
**  and 0 otherwise.  It is the responsibility of the caller to  ensure  that
**  <pos> is a positive integer.
**
**  'IsbRange' is the function in 'IsbListFuncs' for ranges.
*/
libGAP_Int             libGAP_IsbRange (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return (pos <= libGAP_GET_LEN_RANGE(list));
}

libGAP_Int             libGAP_IsbvRange (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return 1L;
}


/****************************************************************************
**
*F  Elm0Range(<list>,<pos>) . . . . . . . . . .  select an element of a range
*F  Elm0vRange(<list>,<pos>)  . . . . . . . . .  select an element of a range
**
**  'Elm0Range' returns  the  element  at  the position   <pos> of  the range
**  <list>, or 0   if <list> has   no assigned object  at  <pos>.  It  is the
**  responsibility of the caller to ensure that <pos> is a positive integer.
**
**  'Elm0vRange' does  the same thing  than  'Elm0Range', but  need not check
**  that <pos> is  less than or  equal to the  length of <list>, this is  the
**  responsibility of the caller.
*/
libGAP_Obj             libGAP_Elm0Range (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    if ( pos <= libGAP_GET_LEN_RANGE( list ) ) {
        return libGAP_GET_ELM_RANGE( list, pos );
    }
    else {
        return 0;
    }
}

libGAP_Obj             libGAP_Elm0vRange (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return libGAP_GET_ELM_RANGE( list, pos );
}


/****************************************************************************
**
*F  ElmRange(<list>,<pos>)  . . . . . . . . . .  select an element of a range
**
**  'ElmRange' selects the element at position <pos> of the range <list>.  It
**  is the  responsibility of the  caller to ensure that  <pos> is a positive
**  integer.  An error is signaller  if <pos>  is  larger than the length  of
**  <list>.
**
**  'ElmvRange' does the same thing than 'ElmRange', but  need not check that
**  <pos>   is less than or   equal  to the  length  of  <list>,  this is the
**  responsibility of the caller.
**
**  'ElmRange' is the function in  'ElmListFuncs' for ranges.  'ElmvRange' is
**  the function in 'ElmvListFuncs' for ranges.
*/
libGAP_Obj             libGAP_ElmRange (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    /* check the position                                                  */
    if ( libGAP_GET_LEN_RANGE( list ) < pos ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            (libGAP_Int)pos, 0L,
            "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, pos );
    }

    /* return the selected element                                         */
    return libGAP_GET_ELM_RANGE( list, pos );
}

libGAP_Obj             libGAP_ElmvRange (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return libGAP_GET_ELM_RANGE( list, pos );
}


/****************************************************************************
**
*F  ElmsRange(<list>,<poss>)  . . . . . . . . . select a sublist from a range
**
**  'ElmsRange' returns a new list  containing the elements at the  positions
**  given in the list <poss> from the range <list>.  It is the responsibility
**  of the caller to ensure that  <poss> is dense  and contains only positive
**  integers.  An error  is signalled if an element  of <poss> is larger than
**  the length of <list>.
**
**  'ElmsRange' is the function in 'ElmsListFuncs' for ranges.
*/
libGAP_Obj             libGAP_ElmsRange (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elms;           /* selected sublist, result        */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element from <list>         */
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Int                 i;              /* loop variable                   */

    /* general code                                                        */
    if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <list>                                        */
        lenList = libGAP_GET_LEN_RANGE( list );

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST( poss );

        /* make the result list                                            */
        elms = libGAP_NEW_PLIST( libGAP_T_PLIST, lenPoss );
        libGAP_SET_LEN_PLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
            pos = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, i ) );
            if ( lenList < pos ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* select the element                                          */
            elm = libGAP_GET_ELM_RANGE( list, pos );

            /* assign the element into <elms>                              */
            libGAP_SET_ELM_PLIST( elms, i, elm );

        }

    }

    /* special code for ranges                                             */
    else {

        /* get the length of <list>                                        */
        lenList = libGAP_GET_LEN_RANGE( list );

        /* get the length of <positions>, the first elements, and the inc. */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* check that no <position> is larger than 'LEN_LIST(<list>)'      */
        if ( lenList < pos ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }
        if ( lenList < pos + (lenPoss-1) * inc ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)(pos + (lenPoss-1) * inc), 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }

        /* make the result range                                           */
        if ( 0 < inc * libGAP_GET_INC_RANGE(list) )
            elms = libGAP_NEW_RANGE_SSORT();
        else
            elms = libGAP_NEW_RANGE_NSORT();
        libGAP_SET_LEN_RANGE( elms, lenPoss );
        libGAP_SET_LOW_RANGE( elms, libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( list, pos ) ) );
        libGAP_SET_INC_RANGE( elms, inc * libGAP_GET_INC_RANGE( list ) );

    }

    /* return the result                                                   */
    return elms;
}


/****************************************************************************
**
*F  AssRange(<list>,<pos>,<val>)  . . . . . . . . . . . . . assign to a range
**
**  'AssRange' assigns the value  <val> to the range  <list> at the  position
**  <pos>.  It  is the responsibility of the  caller to ensure that  <pos> is
**  positive, and that <val> is not 0.
**
**  'AssRange' is the function in 'AssListFuncs' for ranges.
**
**  'AssRange' simply converts the range into a plain list, and then does the
**  same stuff as 'AssPlist'.  This is because a  range is not very likely to
**  stay a range after the assignment.
*/
void            libGAP_AssRange (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    /* convert the range into a plain list                                 */
    libGAP_PLAIN_LIST( list );
    libGAP_RetypeBag( list, libGAP_T_PLIST );

    /* resize the list if necessary                                        */
    if ( libGAP_LEN_PLIST( list ) < pos ) {
        libGAP_GROW_PLIST( list, pos );
        libGAP_SET_LEN_PLIST( list, pos );
    }

    /* now perform the assignment and return the assigned value            */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );
}

void            libGAP_AssRangeImm (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignment: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**
*F  AsssRange(<list>,<poss>,<vals>) . . .  assign several elements to a range
**
**  'AsssRange' assignes the  values  from the list  <vals>  at the positions
**  given in the  list <poss> to the range  <list>.  It is the responsibility
**  of the caller to  ensure that <poss> is dense  and contains only positive
**  integers, that <poss> and <vals> have the same length, and that <vals> is
**  dense.
**
**  'AsssRange' is the function in 'AsssListFuncs' for ranges.
**
**  'AsssRange' simply converts the range to a plain  list  and then does the
**  same stuff as 'AsssPlist'.  This is because a range is not very likely to
**  stay a range after the assignment.
*/
void            libGAP_AsssRange (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 vals )
{
    /* convert <list> to a plain list                                      */
    libGAP_PLAIN_LIST( list );
    libGAP_RetypeBag( list, libGAP_T_PLIST );

    /* and delegate                                                        */
    libGAP_ASSS_LIST( list, poss, vals );
}

void            libGAP_AsssRangeImm (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignments: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**
*F  IsDenseRange(<list>)  . . . . . . . . dense list test function for ranges
**
**  'IsDenseRange' returns 1, since ranges are always dense.
**
**  'IsDenseRange' is the function in 'IsDenseListFuncs' for ranges.
*/
libGAP_Int             libGAP_IsDenseRange (
    libGAP_Obj                 list )
{
    return 1;
}


/****************************************************************************
**
*F  IsHomogRange(<list>)
*/
libGAP_Int             libGAP_IsHomogRange (
    libGAP_Obj                 list )
{
    return 1;
}


/****************************************************************************
**
*F  IsSSortRange(<list>)
*/
libGAP_Int             libGAP_IsSSortRangeNot (
    libGAP_Obj                 list )
{
    return 0;
}

libGAP_Int             libGAP_IsSSortRangeYes (
    libGAP_Obj                 list )
{
    return 1;
}


/****************************************************************************
**
*F  IsPossRange(<list>) . . . . . . . positions list test function for ranges
**
**  'IsPossRange' returns 1  if the range <list>  is a dense list  containing
**  only positive integers, and 0 otherwise.
**
**  'IsPossRange' is the function in 'IsPossListFuncs' for ranges.
*/
libGAP_Int             libGAP_IsPossRange (
    libGAP_Obj                 list )
{
    /* test if the first element is positive                               */
    if ( libGAP_GET_LOW_RANGE( list ) <= 0 )
        return 0;

    /* test if the last element is positive                                */
    if ( libGAP_INT_INTOBJ( libGAP_GET_ELM_RANGE( list, libGAP_GET_LEN_RANGE(list) ) ) <= 0 )
        return 0;

    /* otherwise <list> is a positions list                                */
    return 1;
}


/****************************************************************************
**
*F  PosRange(<list>,<val>,<start>)  . . . . position of an element in a range
**
**  'PosRange' returns the position  of the value <val>  in the range  <list>
**  after the first position <start> as a GAP integer. Fail is returned if <val>
**  is not in the list.
**
**  'PosRange' is the function in 'PosListFuncs' for ranges.
*/
libGAP_Obj             libGAP_PosRange (
    libGAP_Obj                 list,
    libGAP_Obj                 val,
    libGAP_Obj                 start )
{
    libGAP_Int                 k;              /* position, result                */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Int                 low;            /* first element of <list>         */
    libGAP_Int                 inc;            /* increment of <list>             */
    libGAP_Int                 v;              /* numerical value of <val>        */
    libGAP_Int    istart;

    /* if the starting position is too big to be a small int
       then there can't be anything to find */
    if (!libGAP_IS_INTOBJ(start))
      return libGAP_Fail;
    
    istart = libGAP_INT_INTOBJ(start);
    /* get the length, the first element, and the increment of <list>      */
    lenList = libGAP_GET_LEN_RANGE(list);
    low     = libGAP_GET_LOW_RANGE(list);
    inc     = libGAP_GET_INC_RANGE(list);

    /* look just beyond the end                                            */
    if ( istart == lenList ) {
        k = 0;
    }

    /* look for an integer                                                 */
    else if ( libGAP_TNUM_OBJ(val) == libGAP_T_INT ) {
        v = libGAP_INT_INTOBJ(val);
        if ( 0 < inc
          && low + istart * inc <= v && v <= low + (lenList-1) * inc
          && (v - low) % inc == 0 ) {
            k = (v - low) / inc + 1;
        }
        else if ( inc < 0
          && low + (lenList-1) * inc <= v && v <= low + istart * inc
          && (v - low) % inc == 0 ) {
            k = (v - low) / inc + 1;
        }
        else {
            k = 0;
        }
    }

 /* I have no idea how a record can ever be equal to an integer. I'll leave
  * the code in comments for a while, in case someone digs out what this is
  * good for. FL
  * */
    /* for a record compare every entry                                    */
/*    else if ( TNUM_OBJ(val) == T_PREC ) {
        for ( k = istart+1; k <= lenList; k++ ) {
            if ( EQ( INTOBJ_INT( low + (k-1) * inc ), val ) )
                break;
        }
        if ( lenList < k ) {
            k = 0;
        }
    }  */

    /* otherwise it can not be an element of the range                     */
    else {
        k = 0;
    }

    /* return the position                                                 */
    return k == 0 ? libGAP_Fail : libGAP_INTOBJ_INT(k);
}


/****************************************************************************
**
*F  PlainRange(<list>)  . . . . . . . . . . . convert a range to a plain list
**
**  'PlainRange' converts the range <list> to a plain list.
**
**  'PlainRange' is the function in 'PlainListFuncs' for ranges.
*/
void            libGAP_PlainRange (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Int                 low;            /* first element of <list>         */
    libGAP_Int                 inc;            /* increment of <list>             */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length, the first element, and the increment of <list>      */
    lenList = libGAP_GET_LEN_RANGE( list );
    low     = libGAP_GET_LOW_RANGE( list );
    inc     = libGAP_GET_INC_RANGE( list );

    /* change the type of the list, and allocate enough space              */
    libGAP_RetypeBag( list, libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST : libGAP_T_PLIST + libGAP_IMMUTABLE );
    libGAP_GROW_PLIST( list, lenList );
    libGAP_SET_LEN_PLIST( list, lenList );

    /* enter the values in <list>                                          */
    for ( i = 1; i <= lenList; i++ ) {
        libGAP_SET_ELM_PLIST( list, i, libGAP_INTOBJ_INT( low + (i-1) * inc ) );
    }
}


/****************************************************************************
**
*F  IsRange(<list>) . . . . . . . . . . . . . . . . test if a list is a range
**
**  'IsRange' returns 1 if the list  with the handle <list>  is a range and 0
**  otherwise.  As a  side effect 'IsRange' converts proper ranges represented
**  the ordinary way to the compact representation.
*/
libGAP_Obj libGAP_IsRangeFilt;

libGAP_Int             libGAP_IsRange (
    libGAP_Obj                 list )
{
    libGAP_Int                 isRange;        /* result of the test              */
    libGAP_Int                 len;            /* logical length of list          */
    libGAP_Int                 low;            /* value of first element of range */
    libGAP_Int                 inc;            /* increment                       */
    libGAP_Int                 i;              /* loop variable                   */

    /* if <list> is represented as a range, it is of course a range      */
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_RANGE_NSORT
      || libGAP_TNUM_OBJ(list) == libGAP_T_RANGE_SSORT ) {
        isRange = 1;
    }

    /* if <list> is not a list, it is not a range at the moment        */
    else if ( ! libGAP_IS_SMALL_LIST( list ) ) {
       /* isRange = 0; */
       isRange = (libGAP_DoFilter(libGAP_IsRangeFilt, list) == libGAP_True);
    }

    /* if <list> is the empty list, it is a range by definition          */
    else if ( libGAP_LEN_LIST(list) == 0 ) {
        isRange = 1;
    }

    /* if <list> is a list with just one integer, it is also a range     */
    else if ( libGAP_LEN_LIST(list)==1 && libGAP_TNUM_OBJ(libGAP_ELMW_LIST(list,1))==libGAP_T_INT ) {
        isRange = 1;
    }

    /* if the first element is not an integer, it is not a range           */
    else if ( libGAP_ELMV0_LIST(list,1)==0 || libGAP_TNUM_OBJ(libGAP_ELMW_LIST(list,1))!=libGAP_T_INT ) {
        isRange = 0;
    }

    /* if the second element is not an integer, it is not a range          */
    else if ( libGAP_ELMV0_LIST(list,2)==0 || libGAP_TNUM_OBJ(libGAP_ELMW_LIST(list,2))!=libGAP_T_INT ) {
        isRange = 0;
    }

    /* if the first and the second element are equal it is also not a range*/
    else if ( libGAP_ELMW_LIST(list,1) == libGAP_ELMW_LIST(list,2) ) {
        isRange = 0;
    }

    /* otherwise, test if the elements are consecutive integers            */
    else {

        /* get the logical length of the list                              */
        len = libGAP_LEN_LIST(list);
        low = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( list, 1 ) );
        inc = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( list, 2 ) ) - low;

        /* test all entries against the first one                          */
        for ( i = 3;  i <= len;  i++ ) {
            if ( libGAP_ELMV0_LIST(list,i) != libGAP_INTOBJ_INT( low + (i-1) * inc ) )
                break;
        }

        /* if <list> is a range, convert to the compact representation   */
        isRange = (len < i);
        if ( isRange ) {
            if ( libGAP_IS_MUTABLE_OBJ(list) ) {
                libGAP_RetypeBag( list, (0 < inc ? libGAP_T_RANGE_SSORT : libGAP_T_RANGE_NSORT) );
            }
            else {
                libGAP_RetypeBag( list, (0 < inc ? libGAP_T_RANGE_SSORT : libGAP_T_RANGE_NSORT)
                                 + libGAP_IMMUTABLE );
            }
            libGAP_ResizeBag( list, 3 * sizeof(libGAP_Obj) );
            libGAP_SET_LEN_RANGE( list, len );
            libGAP_SET_LOW_RANGE( list, low );
            libGAP_SET_INC_RANGE( list, inc );
        }

    }

    /* return the result of the test                                       */
    return isRange;
}


/****************************************************************************
**
*F  FuncIsRange(<self>,<obj>) . . . . . . . . . . . . . . .  test for a range
**
**  'FuncIsRange' implements the internal function 'IsRange'.
**
**  'IsRange( <obj> )'
**
**  'IsRange' returns 'true' if <obj>, which may be an object of any type, is
**  a range and 'false' otherwise.  A range is a list without holes such that
**  the elements are  consecutive integers.
*/
libGAP_Obj libGAP_FuncIS_RANGE (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    /* let 'IsRange' do the work for lists                                 */
    return libGAP_IsRange(obj) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  SaveRange( <range> )
**
*/

void libGAP_SaveRange( libGAP_Obj range )
{
  libGAP_SaveSubObj(libGAP_ADDR_OBJ(range)[0]); /* length */
  libGAP_SaveSubObj(libGAP_ADDR_OBJ(range)[1]); /* base */
  libGAP_SaveSubObj(libGAP_ADDR_OBJ(range)[2]); /* increment */
}

/****************************************************************************
**
*F  LoadRange( <range> )
**
*/

void libGAP_LoadRange( libGAP_Obj range )
{
  libGAP_ADDR_OBJ(range)[0] = libGAP_LoadSubObj(); /* length */
  libGAP_ADDR_OBJ(range)[1] = libGAP_LoadSubObj(); /* base */
  libGAP_ADDR_OBJ(range)[2] = libGAP_LoadSubObj(); /* increment */
}


/****************************************************************************
**
*F  Range2Check( <first>, <last> )  . . . . . . . . . . . . . construct range
*/
libGAP_Obj libGAP_Range2Check (
    libGAP_Obj                 first,
    libGAP_Obj                 last )
{
    libGAP_Obj                 range;
    libGAP_Int                 f, l;
    if ( ! libGAP_IS_INTOBJ(first) ) {
        libGAP_ErrorQuit(
            "Range: <first> must be a positive small integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(first), 0L );
    }
    f = libGAP_INT_INTOBJ(first);
    if ( ! libGAP_IS_INTOBJ(last) ) {
        libGAP_ErrorQuit(
            "Range: <last> must be a positive small integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(last), 0L );
    }
    l = libGAP_INT_INTOBJ(last);
    if ( f > l ) {
        range = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
        libGAP_SET_LEN_PLIST( range, 0 );
    }
    else if ( f == l ) {
        range = libGAP_NEW_PLIST( libGAP_T_PLIST, 1 );
        libGAP_SET_LEN_PLIST( range, 1 );
        libGAP_SET_ELM_PLIST( range, 1, first );
    }
    else {
        range = libGAP_NEW_RANGE_SSORT();
        libGAP_SET_LEN_RANGE( range, (l-f) + 1 );
        libGAP_SET_LOW_RANGE( range, f );
        libGAP_SET_INC_RANGE( range, 1 );
    }
    return range;
}


/****************************************************************************
**
*F  Range3Check( <first>, <second>, <last> )  . . . . . . . . construct range
*/
libGAP_Obj libGAP_Range3Check (
    libGAP_Obj                 first,
    libGAP_Obj                 second,
    libGAP_Obj                 last )
{
    libGAP_Obj                 range;
    libGAP_Int                 f, i, l;
    if ( ! libGAP_IS_INTOBJ(first) ) {
        libGAP_ErrorQuit(
            "Range: <first> must be a positive small integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(first), 0L );
    }
    f = libGAP_INT_INTOBJ(first);
    if ( ! libGAP_IS_INTOBJ(second) ) {
        libGAP_ErrorQuit(
            "Range: <second> must be a positive small integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(second), 0L );
    }
    if ( first == second ) {
        libGAP_ErrorQuit(
            "Range: <second> must not be equal to <first> (%d)",
            (libGAP_Int)libGAP_INT_INTOBJ(first), 0L );
    }
    i = libGAP_INT_INTOBJ(second) - f;
    if ( ! libGAP_IS_INTOBJ(last) ) {
        libGAP_ErrorQuit(
            "Range: <last> must be a positive small integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(last), 0L );
    }
    l = libGAP_INT_INTOBJ(last);
    if ( (l - f) % i != 0 ) {
        libGAP_ErrorQuit(
            "Range: <last>-<first> (%d) must be divisible by <inc> (%d)",
            (libGAP_Int)(l - f), (libGAP_Int)i );
    }
    if ( (0 < i && f > l) || (i < 0 && f < l) ) {
        range = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
        libGAP_SET_LEN_PLIST( range, 0 );
    }
    else if ( f == l ) {
        range = libGAP_NEW_PLIST( libGAP_T_PLIST, 1 );
        libGAP_SET_LEN_PLIST( range, 1 );
        libGAP_SET_ELM_PLIST( range, 1, first );
    }
    else {
        if ( 0 < i )
            range = libGAP_NEW_RANGE_SSORT();
        else
            range = libGAP_NEW_RANGE_NSORT();
        libGAP_SET_LEN_RANGE( range, (l - f) / i + 1 );
        libGAP_SET_LOW_RANGE( range, f );
        libGAP_SET_INC_RANGE( range, i );
    }
    return range;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * GAP level functions  * * * * * * * * * * * * *
*/

/****************************************************************************
**


*F  FuncIS_RANGE_REP( <self>, <obj> ) . . . . . test if value is in range rep
*/
libGAP_Obj libGAP_IsRangeRepFilt;

libGAP_Obj libGAP_FuncIS_RANGE_REP (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_RANGE( obj ) ? libGAP_True : libGAP_False);
}

/****************************************************************************
**
*F  MakeImmutableRange( <range> )
**
*/

void libGAP_MakeImmutableRange( libGAP_Obj range )
{
  libGAP_RetypeBag( range, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(range)));
}

/****************************************************************************
**
*F  FuncINTER_RANGE( <range1>, <range2> )
**
*/

static libGAP_Int libGAP_egcd (libGAP_Int a, libGAP_Int b, libGAP_Int *lastx, libGAP_Int *lasty)
{
  libGAP_Int x = 0, y = 1;
  *lastx = 1; *lasty = 0;

  while (b != 0) {
    libGAP_Int t, q;
    t = b; q = a / b; b = a % b; a = t;
    if (lastx) { t = x; x = *lastx - q*x; *lastx = t; }
    if (lasty) { t = y; y = *lasty - q*y; *lasty = t; }
  }
  return a;
} /* returns g=gcd(a,b), with lastx*a+lasty*b = g */

libGAP_Obj libGAP_FuncINTER_RANGE( libGAP_Obj self, libGAP_Obj r1, libGAP_Obj r2)
{
  libGAP_Int low1, low2, inc1, inc2, lowi, inci, libGAP_g, x, y;
  libGAP_UInt len1, len2, leni;
  
  low1 = libGAP_GET_LOW_RANGE(r1);
  low2 = libGAP_GET_LOW_RANGE(r2);
  inc1 = libGAP_GET_INC_RANGE(r1);
  inc2 = libGAP_GET_INC_RANGE(r2);
  len1 = libGAP_GET_LEN_RANGE(r1);
  len2 = libGAP_GET_LEN_RANGE(r2);

  if (inc1 < 0)
    {
      low1 = low1 + (len1-1)*inc1;
      inc1 = -inc1;
    }
  if (inc2 < 0)
    {
      low2 = low2 + (len2-1)*inc2;
      inc2 = -inc2;
    }

  if (low1 > low2)
    {
      libGAP_Int t;
      libGAP_UInt ut;
      t = low1; low1 = low2; low2 = t;
      t = inc1; inc1 = inc2; inc2 = t;
      ut = len1; len1 = len2; len2 = ut;
    }

  libGAP_g = libGAP_egcd(inc1, inc2, &x, &y);

  inci = (inc1 / libGAP_g) * inc2;

  if (inci / inc2 != inc1 / libGAP_g) /* overflow */
    goto empty_range;

  if ((low2 - low1) % libGAP_g) /* ranges are disjoint */
    goto empty_range;

  x = (-y * ((low2 - low1) / libGAP_g)) % (inc1 / libGAP_g);
  if (x < 0)
    x += inc1/libGAP_g;
  lowi = low2 + inc2 * x;

  x = (low1 + (len1-1)*inc1 - lowi);
  y = (low2 + (len2-1)*inc2 - lowi);
  if (x < 0 || y < 0)
    goto empty_range;
  if (x > y)
    leni = 1 + y / inci;
  else
    leni = 1 + x / inci;
  
  libGAP_SET_LOW_RANGE(r1,lowi);
  libGAP_SET_LEN_RANGE(r1,leni);
  libGAP_SET_INC_RANGE(r1,inci);
  return (libGAP_Obj)0;

 empty_range:
  libGAP_RetypeBag(r1, libGAP_T_PLIST_EMPTY);
  libGAP_ResizeBag(r1,sizeof(libGAP_Obj));
  libGAP_SET_LEN_PLIST(r1, 0L);
  return (libGAP_Obj) 0;
}
/* torture:
a := [-10..10];; INTER_RANGE(a,[-10..10]); a;
a := [-10..10];; INTER_RANGE(a,[-11..10]); a;
a := [-10..10];; INTER_RANGE(a,[-10..11]); a;
a := [-10..10];; INTER_RANGE(a,[-11..9]); a;
a := [-10..10];; INTER_RANGE(a,[-9..11]); a;
a := [-10..10];; INTER_RANGE(a,[-21,-18..21]); a;
a := [-10..10];; INTER_RANGE(a,[-6,-3..21]); a;
a := [-10..10];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-10,-7..20];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-9,-6..21];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-12,-10..20];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-15,-12..3];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-12,-9..3];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-9,-6..3];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-9,-3..3];; INTER_RANGE(a,[-21,-18..6]); a;
a := [-9,-5..3];; INTER_RANGE(a,[-21,-18..6]); a;
*/    

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  BagNames  . . . . . . . . . . . . . . . . . . . . . . . list of bag names
*/
static libGAP_StructBagNames libGAP_BagNames[] = {
  { libGAP_T_RANGE_NSORT,                     "list (range,nsort)"            },
  { libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE,          "list (range,nsort,imm)"        },
  { libGAP_T_RANGE_SSORT,                     "list (range,ssort)"            },
  { libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE,          "list (range,ssort,imm)"        },
  { libGAP_T_RANGE_NSORT            +libGAP_COPYING, "list (range,nsort,copied)"     },
  { libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE +libGAP_COPYING, "list (range,nsort,imm,copied)" },
  { libGAP_T_RANGE_SSORT            +libGAP_COPYING, "list (range,ssort,copied)"     },
  { libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE +libGAP_COPYING, "list (range,ssort,imm,copied)" },
  { -1,                                ""                              }
};


/****************************************************************************
**
*V  ClearFiltsTab . . . . . . . . . . . . . . . . . . . .  clear filter tnums
*/
static libGAP_Int libGAP_ClearFiltsTab [] = {
    libGAP_T_RANGE_NSORT,           libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE, libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT,           libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE, libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    -1,                      -1
};


/****************************************************************************
**
*V  HasFiltTab  . . . . . . . . . . . . . . . . . . . . .  tester filter tnum
*/
static libGAP_Int libGAP_HasFiltTab [] = {

    /* nsort mutable range                                                 */
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_EMPTY,    0,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_DENSE,    1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NDENSE,   0,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_HOMOG,    1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_TABLE,    0,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_RECT,    0,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_SSORT,    0,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NSORT,    1,

    /* nsort immutable range                                               */
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    0,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   0,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    0,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_RECT,    0,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    0,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    1,

    /* ssort mutable range                                                 */
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_EMPTY,    0,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_DENSE,    1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NDENSE,   0,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_HOMOG,    1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_TABLE,    0,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_RECT,    0,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_SSORT,    1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NSORT,    0,

    /* ssort immutable range                                               */
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    0,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    1,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   0,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    1,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    0,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_RECT,    0,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    1,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    0,

    -1,                         -1,             -1
};


/****************************************************************************
**
*V  SetFiltTab  . . . . . . . . . . . . . . . . . . . . .  setter filter tnum
*/
static libGAP_Int libGAP_SetFiltTab [] = {

    /* nsort mutable range                                                 */
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_EMPTY,    -1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_DENSE,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_TABLE,    -1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_RECT,    -1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_SSORT,    -1,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NSORT,    libGAP_T_RANGE_NSORT,

    /* nsort immutable range                                               */
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    -1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    -1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_RECT,    -1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    -1,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,

    /* ssort mutable range                                                 */
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_EMPTY,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_DENSE,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_TABLE,    -1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_RECT,    -1,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_SSORT,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NSORT,    -1,

    /* ssort immutable range                                               */
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   -1,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   -1,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    -1,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_RECT,    -1,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    -1,


    -1,                         -1,             -1

};


/****************************************************************************
**
*V  ResetFiltTab  . . . . . . . . . . . . . . . . . . .  unsetter filter tnum
*/
static libGAP_Int libGAP_ResetFiltTab [] = {

    /* nsort mutable range                                                 */
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_EMPTY,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_DENSE,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NDENSE,   libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NHOMOG,   libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_TABLE,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_RECT,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_SSORT,    libGAP_T_RANGE_NSORT,
    libGAP_T_RANGE_NSORT,              libGAP_FN_IS_NSORT,    libGAP_T_RANGE_NSORT,

    /* nsort immutable range                                               */
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_RECT,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_RANGE_NSORT+libGAP_IMMUTABLE,

    /* ssort mutable range                                                 */
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_EMPTY,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_DENSE,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NDENSE,   libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NHOMOG,   libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_TABLE,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_RECT,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_SSORT,    libGAP_T_RANGE_SSORT,
    libGAP_T_RANGE_SSORT,              libGAP_FN_IS_NSORT,    libGAP_T_RANGE_SSORT,

    /* ssort immutable range                                               */
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_MUTABLE,  libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_EMPTY,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_DENSE,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NDENSE,   libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_HOMOG,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NHOMOG,   libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_TABLE,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_RECT,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_SSORT,    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,
    libGAP_T_RANGE_SSORT+libGAP_IMMUTABLE,    libGAP_FN_IS_NSORT,    libGAP_T_BLIST_SSORT+libGAP_IMMUTABLE,

    -1,                         -1,             -1

};


/****************************************************************************
**
*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_RANGE", "obj", &libGAP_IsRangeFilt,
      libGAP_FuncIS_RANGE, "src/range.c:IS_RANGE" },

    { "IS_RANGE_REP", "obj", &libGAP_IsRangeRepFilt,
      libGAP_FuncIS_RANGE_REP, "src/range.c:IS_RANGE_REP" },

    { 0 }

};

/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "INTER_RANGE", 2, "range1, range2",
      libGAP_FuncINTER_RANGE, "src/range.c:INTER_RANGE" },


    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* check dependencies                                                  */
    libGAP_RequireModule( libGAP_module, "lists", 403600000UL );

    /* GASMAN marking functions and GASMAN names                           */
    libGAP_InitBagNamesFromTable( libGAP_BagNames );

    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_NSORT                     , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE          , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_SSORT                     , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE          , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_NSORT            +libGAP_COPYING , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE +libGAP_COPYING , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_SSORT            +libGAP_COPYING , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags(   libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE +libGAP_COPYING , libGAP_MarkAllSubBags );

    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_RANGE_NSORT_MUTABLE",
                           &libGAP_TYPE_RANGE_NSORT_MUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_RANGE_SSORT_MUTABLE",
                           &libGAP_TYPE_RANGE_SSORT_MUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_RANGE_NSORT_IMMUTABLE",
                           &libGAP_TYPE_RANGE_NSORT_IMMUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_RANGE_SSORT_IMMUTABLE",
                           &libGAP_TYPE_RANGE_SSORT_IMMUTABLE );

    libGAP_TypeObjFuncs[ libGAP_T_RANGE_NSORT            ] = libGAP_TypeRangeNSortMutable;
    libGAP_TypeObjFuncs[ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_TypeRangeNSortImmutable;
    libGAP_TypeObjFuncs[ libGAP_T_RANGE_SSORT            ] = libGAP_TypeRangeSSortMutable;
    libGAP_TypeObjFuncs[ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_TypeRangeSSortImmutable;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* Saving functions */
    libGAP_SaveObjFuncs[libGAP_T_RANGE_NSORT            ] = libGAP_SaveRange;
    libGAP_SaveObjFuncs[libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_SaveRange;
    libGAP_SaveObjFuncs[libGAP_T_RANGE_SSORT            ] = libGAP_SaveRange;
    libGAP_SaveObjFuncs[libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_SaveRange;
    libGAP_LoadObjFuncs[libGAP_T_RANGE_NSORT            ] = libGAP_LoadRange;
    libGAP_LoadObjFuncs[libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_LoadRange;
    libGAP_LoadObjFuncs[libGAP_T_RANGE_SSORT            ] = libGAP_LoadRange;
    libGAP_LoadObjFuncs[libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_LoadRange;

    /* install the copy methods                                            */
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_NSORT                     ] = libGAP_CopyRange;
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_NSORT            +libGAP_COPYING ] = libGAP_CopyRangeCopy;
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE          ] = libGAP_CopyRange;
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CopyRangeCopy;
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_SSORT                     ] = libGAP_CopyRange;
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_SSORT            +libGAP_COPYING ] = libGAP_CopyRangeCopy;
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE          ] = libGAP_CopyRange;
    libGAP_CopyObjFuncs [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CopyRangeCopy;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_NSORT                     ] = libGAP_CleanRange;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_NSORT            +libGAP_COPYING ] = libGAP_CleanRangeCopy;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE          ] = libGAP_CleanRange;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CleanRangeCopy;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_SSORT                     ] = libGAP_CleanRange;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_SSORT            +libGAP_COPYING ] = libGAP_CleanRangeCopy;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE          ] = libGAP_CleanRange;
    libGAP_CleanObjFuncs[ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CleanRangeCopy;

    /* Make immutable methods */
    libGAP_MakeImmutableObjFuncs[ libGAP_T_RANGE_NSORT ] = libGAP_MakeImmutableRange;
    libGAP_MakeImmutableObjFuncs[ libGAP_T_RANGE_SSORT ] = libGAP_MakeImmutableRange;
    
    /* install the print method                                            */
    libGAP_PrintObjFuncs[ libGAP_T_RANGE_NSORT            ] = libGAP_PrintRange;
    libGAP_PrintObjFuncs[ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_PrintRange;
    libGAP_PrintObjFuncs[ libGAP_T_RANGE_SSORT            ] = libGAP_PrintRange;
    libGAP_PrintObjFuncs[ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_PrintRange;

    /* initialise list tables                                              */
    libGAP_InitClearFiltsTNumsFromTable   ( libGAP_ClearFiltsTab );
    libGAP_InitHasFiltListTNumsFromTable  ( libGAP_HasFiltTab    );
    libGAP_InitSetFiltListTNumsFromTable  ( libGAP_SetFiltTab    );
    libGAP_InitResetFiltListTNumsFromTable( libGAP_ResetFiltTab  );

    /* install the comparison methods                                      */
    libGAP_EqFuncs[ libGAP_T_RANGE_NSORT ][ libGAP_T_RANGE_NSORT ] = libGAP_EqRange;
    libGAP_EqFuncs[ libGAP_T_RANGE_NSORT ][ libGAP_T_RANGE_SSORT ] = libGAP_EqRange;
    libGAP_EqFuncs[ libGAP_T_RANGE_SSORT ][ libGAP_T_RANGE_NSORT ] = libGAP_EqRange;
    libGAP_EqFuncs[ libGAP_T_RANGE_SSORT ][ libGAP_T_RANGE_SSORT ] = libGAP_EqRange;
    libGAP_LtFuncs[ libGAP_T_RANGE_NSORT ][ libGAP_T_RANGE_NSORT ] = libGAP_LtRange;
    libGAP_LtFuncs[ libGAP_T_RANGE_NSORT ][ libGAP_T_RANGE_SSORT ] = libGAP_LtRange;
    libGAP_LtFuncs[ libGAP_T_RANGE_SSORT ][ libGAP_T_RANGE_NSORT ] = libGAP_LtRange;
    libGAP_LtFuncs[ libGAP_T_RANGE_SSORT ][ libGAP_T_RANGE_SSORT ] = libGAP_LtRange;

    /* install the list functions in the tables                            */
    libGAP_LenListFuncs    [ libGAP_T_RANGE_NSORT            ] = libGAP_LenRange;
    libGAP_LenListFuncs    [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_LenRange;
    libGAP_LenListFuncs    [ libGAP_T_RANGE_SSORT            ] = libGAP_LenRange;
    libGAP_LenListFuncs    [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_LenRange;
    libGAP_IsbListFuncs    [ libGAP_T_RANGE_NSORT            ] = libGAP_IsbRange;
    libGAP_IsbListFuncs    [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_IsbRange;
    libGAP_IsbListFuncs    [ libGAP_T_RANGE_SSORT            ] = libGAP_IsbRange;
    libGAP_IsbListFuncs    [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_IsbRange;
    libGAP_IsbvListFuncs   [ libGAP_T_RANGE_NSORT            ] = libGAP_IsbvRange;
    libGAP_IsbvListFuncs   [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_IsbvRange;
    libGAP_IsbvListFuncs   [ libGAP_T_RANGE_SSORT            ] = libGAP_IsbvRange;
    libGAP_IsbvListFuncs   [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_IsbvRange;
    libGAP_Elm0ListFuncs   [ libGAP_T_RANGE_NSORT            ] = libGAP_Elm0Range;
    libGAP_Elm0ListFuncs   [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_Elm0Range;
    libGAP_Elm0ListFuncs   [ libGAP_T_RANGE_SSORT            ] = libGAP_Elm0Range;
    libGAP_Elm0ListFuncs   [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_Elm0Range;
    libGAP_Elm0vListFuncs  [ libGAP_T_RANGE_NSORT            ] = libGAP_Elm0vRange;
    libGAP_Elm0vListFuncs  [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_Elm0vRange;
    libGAP_Elm0vListFuncs  [ libGAP_T_RANGE_SSORT            ] = libGAP_Elm0vRange;
    libGAP_Elm0vListFuncs  [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_Elm0vRange;
    libGAP_ElmListFuncs    [ libGAP_T_RANGE_NSORT            ] = libGAP_ElmRange;
    libGAP_ElmListFuncs    [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_ElmRange;
    libGAP_ElmListFuncs    [ libGAP_T_RANGE_SSORT            ] = libGAP_ElmRange;
    libGAP_ElmListFuncs    [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_ElmRange;
    libGAP_ElmvListFuncs   [ libGAP_T_RANGE_NSORT            ] = libGAP_ElmvRange;
    libGAP_ElmvListFuncs   [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_ElmvRange;
    libGAP_ElmvListFuncs   [ libGAP_T_RANGE_SSORT            ] = libGAP_ElmvRange;
    libGAP_ElmvListFuncs   [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_ElmvRange;
    libGAP_ElmwListFuncs   [ libGAP_T_RANGE_NSORT            ] = libGAP_ElmvRange;
    libGAP_ElmwListFuncs   [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_ElmvRange;
    libGAP_ElmwListFuncs   [ libGAP_T_RANGE_SSORT            ] = libGAP_ElmvRange;
    libGAP_ElmwListFuncs   [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_ElmvRange;
    libGAP_ElmsListFuncs   [ libGAP_T_RANGE_NSORT            ] = libGAP_ElmsRange;
    libGAP_ElmsListFuncs   [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_ElmsRange;
    libGAP_ElmsListFuncs   [ libGAP_T_RANGE_SSORT            ] = libGAP_ElmsRange;
    libGAP_ElmsListFuncs   [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_ElmsRange;
    libGAP_AssListFuncs    [ libGAP_T_RANGE_NSORT            ] = libGAP_AssRange;
    libGAP_AssListFuncs    [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_AssRangeImm;
    libGAP_AssListFuncs    [ libGAP_T_RANGE_SSORT            ] = libGAP_AssRange;
    libGAP_AssListFuncs    [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_AssRangeImm;
    libGAP_AsssListFuncs   [ libGAP_T_RANGE_NSORT            ] = libGAP_AsssRange;
    libGAP_AsssListFuncs   [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_AsssRangeImm;
    libGAP_AsssListFuncs   [ libGAP_T_RANGE_SSORT            ] = libGAP_AsssRange;
    libGAP_AsssListFuncs   [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_AsssRangeImm;
    libGAP_IsDenseListFuncs[ libGAP_T_RANGE_NSORT            ] = libGAP_IsDenseRange;
    libGAP_IsDenseListFuncs[ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_IsDenseRange;
    libGAP_IsDenseListFuncs[ libGAP_T_RANGE_SSORT            ] = libGAP_IsDenseRange;
    libGAP_IsDenseListFuncs[ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_IsDenseRange;
    libGAP_IsHomogListFuncs[ libGAP_T_RANGE_NSORT            ] = libGAP_IsHomogRange;
    libGAP_IsHomogListFuncs[ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_IsHomogRange;
    libGAP_IsHomogListFuncs[ libGAP_T_RANGE_SSORT            ] = libGAP_IsHomogRange;
    libGAP_IsHomogListFuncs[ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_IsHomogRange;
    libGAP_IsSSortListFuncs[ libGAP_T_RANGE_NSORT            ] = libGAP_IsSSortRangeNot;
    libGAP_IsSSortListFuncs[ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortRangeNot;
    libGAP_IsSSortListFuncs[ libGAP_T_RANGE_SSORT            ] = libGAP_IsSSortRangeYes;
    libGAP_IsSSortListFuncs[ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortRangeYes;
    libGAP_IsPossListFuncs [ libGAP_T_RANGE_NSORT            ] = libGAP_IsPossRange;
    libGAP_IsPossListFuncs [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_IsPossRange;
    libGAP_IsPossListFuncs [ libGAP_T_RANGE_SSORT            ] = libGAP_IsPossRange;
    libGAP_IsPossListFuncs [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_IsPossRange;
    libGAP_PosListFuncs    [ libGAP_T_RANGE_NSORT            ] = libGAP_PosRange;
    libGAP_PosListFuncs    [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_PosRange;
    libGAP_PosListFuncs    [ libGAP_T_RANGE_SSORT            ] = libGAP_PosRange;
    libGAP_PosListFuncs    [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_PosRange;
    libGAP_PlainListFuncs  [ libGAP_T_RANGE_NSORT            ] = libGAP_PlainRange;
    libGAP_PlainListFuncs  [ libGAP_T_RANGE_NSORT +libGAP_IMMUTABLE ] = libGAP_PlainRange;
    libGAP_PlainListFuncs  [ libGAP_T_RANGE_SSORT            ] = libGAP_PlainRange;
    libGAP_PlainListFuncs  [ libGAP_T_RANGE_SSORT +libGAP_IMMUTABLE ] = libGAP_PlainRange;

    /* initialise list tables                                              */
    libGAP_InitClearFiltsTNumsFromTable   ( libGAP_ClearFiltsTab );
    libGAP_InitHasFiltListTNumsFromTable  ( libGAP_HasFiltTab    );
    libGAP_InitSetFiltListTNumsFromTable  ( libGAP_SetFiltTab    );
    libGAP_InitResetFiltListTNumsFromTable( libGAP_ResetFiltTab  );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoRange() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "range",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoRange ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  range.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
