/****************************************************************************
**
*W  read.c                      GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This module contains the functions to read expressions and statements.
*/
#include        <string.h>              /* memcpy */
#include        "system.h"              /* system dependent part           */



#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "string.h"              /* strings                         */
#include        "calls.h"               /* generic call mechanism          */
#include        "code.h"                /* coder                           */
#include        "vars.h"                /* variables                       */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */


#include        "intrprtr.h"            /* interpreter                     */

#include        "read.h"                /* reader                          */

#include        "bool.h"

#include         <assert.h>

/****************************************************************************
**

*F  READ_ERROR()  . . . . . . . . . . . . . . . . . . . reader found an error
**
**  'READ_ERROR' returns a non-zero value if the reader found an error, or if
**  the interpretation of  an expression  or  statement lead to an  error (in
**  which case 'ReadEvalError' jumps back to 'READ_ERROR' via 'longjmp').
**
#define READ_ERROR()    (NrError || (NrError+=sySetjmp(ReadJmpError)))
*/
libGAP_syJmp_buf         libGAP_ReadJmpError;


/****************************************************************************
**
*V  StackNams . . . . . . . . . . . . .  stack of local variables names lists
*V  CountNams . . . . . . . . . number of local variables names list on stack
**
**  'StackNams' is a stack of local variables  names lists.  A new names list
**  is pushed onto this stack when the  reader begins to  read a new function
**  expression  (after  reading the argument   list  and the local  variables
**  list), and popped again when the reader has finished reading the function
**  expression (after reading the 'end').
**
**  'CountNams' is the number of local variables names lists currently on the
**  stack.
*/
libGAP_Obj             libGAP_StackNams;

libGAP_UInt            libGAP_CountNams;


/****************************************************************************
**
*V  ReadTop . . . . . . . . . . . . . . . . . . . . . .  top level expression
*V  ReadTilde . . . . . . . . . . . . . . . . . . . . . . . . . .  tilde read
**
**  'ReadTop' is 0  if the reader is   currently not reading  a  list or record
**  expression.  'ReadTop'  is 1 if the  reader is currently reading an outmost
**  list or record expression.   'ReadTop' is larger than   1 if the  reader is
**  currently reading a nested list or record expression.
**
**  'ReadTilde' is 1 if the reader has read a  reference to the global variable
**  '~' within the current outmost list or record expression.
*/
libGAP_UInt            libGAP_ReadTop;

libGAP_UInt            libGAP_ReadTilde;


/****************************************************************************
**
*V  CurrLHSGVar . . . . . . . . . . . .  current left hand side of assignment
**
**  'CurrLHSGVar' is the current left hand side of an assignment.  It is used
**  to prevent undefined global variable  warnings, when reading a  recursive
**  function.
*/
libGAP_UInt            libGAP_CurrLHSGVar;


/****************************************************************************
**
**  The constructs <Expr> and <Statments> may have themself as subpart, e.g.,
**  '<Var>( <Expr> )'  is  <Expr> and 'if   <Expr> then <Statments> fi;'   is
**  <Statments>.  The  functions 'ReadExpr' and  'ReadStats' must therefor be
**  declared forward.
*/
void            libGAP_ReadExpr (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode );

libGAP_UInt            libGAP_ReadStats (
    libGAP_TypSymbolSet        follow );

void            libGAP_ReadFuncExpr1 (
    libGAP_TypSymbolSet        follow );


static libGAP_UInt libGAP_CurrentGlobalForLoopVariables[100];
static libGAP_UInt libGAP_CurrentGlobalForLoopDepth;

void libGAP_PushGlobalForLoopVariable( libGAP_UInt var)
{
  if (libGAP_CurrentGlobalForLoopDepth < 100)
    libGAP_CurrentGlobalForLoopVariables[libGAP_CurrentGlobalForLoopDepth] = var;
  libGAP_CurrentGlobalForLoopDepth++;
}

void libGAP_PopGlobalForLoopVariable( void )
{
  assert(libGAP_CurrentGlobalForLoopDepth);
  libGAP_CurrentGlobalForLoopDepth--;
}

libGAP_UInt libGAP_GlobalComesFromEnclosingForLoop (libGAP_UInt var)
{
  libGAP_UInt i;
  for (i = 0; i < libGAP_CurrentGlobalForLoopDepth; i++)
    {
      if (i==100)
	return 0;
      if (libGAP_CurrentGlobalForLoopVariables[i] == var)
	return 1;
    }
  return 0;
}

/****************************************************************************
**
*F * * * * * * * * * * read symbols and call interpreter  * * * * * * * * * *
*/


/****************************************************************************
**

*F  ReadCallVarAss( <follow>, <mode> )  . . . . . . . . . . . read a variable
**
**  'ReadCallVarAss' reads  a variable.  In  case  of an  error it skips  all
**  symbols up to one contained in  <follow>.  The <mode>  must be one of the
**  following:
**
**  'i':        check if variable, record component, list entry is bound
**  'r':        reference to a variable
**  's':        assignment via ':='
**  'u':        unbind a variable
**  'x':        either 'r' or 's' depending on <Symbol>
**
**  <Ident> :=  a|b|..|z|A|B|..|Z { a|b|..|z|A|B|..|Z|0|..|9|_ }
**
**  <Var> := <Ident>
**        |  <Var> '[' <Expr> ']'
**        |  <Var> '{' <Expr> '}'
**        |  <Var> '.' <Ident>
**        |  <Var> '(' [ <Expr> { ',' <Expr> } ] [':' [ <options> ]] ')'
*/
extern libGAP_Obj libGAP_ExprGVars;
extern libGAP_Obj libGAP_ErrorLVars;
extern libGAP_Obj libGAP_BottomLVars;

/* This function reads the options part at the end of a function call
   The syntax is

   <options> := <option> [, <options> ]
   <option>  := <Ident> | '(' <Expr> ')' [ ':=' <Expr> ]

   empty options lists are handled further up
*/
void libGAP_ReadFuncCallOption( libGAP_TypSymbolSet follow )
{
  volatile libGAP_UInt       rnam;           /* record component name           */
  if ( libGAP_Symbol == libGAP_S_IDENT ) {
    rnam = libGAP_RNamName( libGAP_Value );
    libGAP_Match( libGAP_S_IDENT, "identifier", libGAP_S_COMMA | follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncCallOptionsBeginElmName( rnam ); }
  }
  else if ( libGAP_Symbol == libGAP_S_LPAREN ) {
    libGAP_Match( libGAP_S_LPAREN, "(", libGAP_S_COMMA | follow );
    libGAP_ReadExpr( follow, 'r' );
    libGAP_Match( libGAP_S_RPAREN, ")", libGAP_S_COMMA | follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncCallOptionsBeginElmExpr(); }
  }
  else {
    libGAP_SyntaxError("identifier expected");
  }
  if ( libGAP_Symbol == libGAP_S_ASSIGN )
    {
      libGAP_Match( libGAP_S_ASSIGN, ":=", libGAP_S_COMMA | follow );
      libGAP_ReadExpr( libGAP_S_COMMA | libGAP_S_RPAREN|follow, 'r' );
      if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncCallOptionsEndElm(); }
    }
  else
    {
      if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncCallOptionsEndElmEmpty(); }
    }
  return;
}

void libGAP_ReadFuncCallOptions( libGAP_TypSymbolSet follow )
{
  volatile libGAP_UInt nr;
  if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncCallOptionsBegin( ); }
  libGAP_ReadFuncCallOption( follow);
  nr = 1;
  while ( libGAP_Symbol == libGAP_S_COMMA )
    {
      libGAP_Match(libGAP_S_COMMA, ",", follow );
      libGAP_ReadFuncCallOption( follow );
      nr++;
    }
  if (!libGAP_READ_ERROR()) {
    libGAP_IntrFuncCallOptionsEnd( nr );
  }

  return;
}

static libGAP_Obj libGAP_GAPInfo;

static libGAP_UInt libGAP_WarnOnUnboundGlobalsRNam;

void libGAP_ReadCallVarAss (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    volatile libGAP_Char       type = ' ';     /* type of variable                */
    volatile libGAP_Obj        nams;           /* list of names of local vars.    */
    volatile libGAP_Obj        lvars;          /* environment                     */
    volatile libGAP_UInt       nest  = 0;      /* nesting level of a higher var.  */
    volatile libGAP_Obj        lvars0;          /* environment                     */
    volatile libGAP_UInt       nest0  = 0;      /* nesting level of a higher var.  */
    volatile libGAP_UInt       indx  = 0;      /* index of a local variable       */
    volatile libGAP_UInt       var   = 0;      /* variable                        */
    volatile libGAP_UInt       level = 0;      /* number of '{}' selectors        */
    volatile libGAP_UInt       rnam  = 0;      /* record component name           */
    volatile libGAP_UInt       narg  = 0;      /* number of arguments             */


    /* all variables must begin with an identifier                         */
    if ( libGAP_Symbol != libGAP_S_IDENT ) {
        libGAP_SyntaxError( "identifier expected" );
        return;
    }

    /* try to look up the variable on the stack of local variables         */
    nest = 0;
    while ( type == ' ' && nest < libGAP_CountNams ) {
        nams = libGAP_ELM_LIST( libGAP_StackNams, libGAP_CountNams-nest );
        for ( indx = libGAP_LEN_LIST( nams ); 1 <= indx; indx-- ) {
            if ( strcmp( libGAP_Value, libGAP_CSTR_STRING(libGAP_ELM_LIST(nams,indx)) ) == 0 ) {
                if ( nest == 0 ) {
                    type = 'l';
                    var = indx;
                }
                else {
                    type = 'h';

                    /* Ultrix 4.2 cc get's confused if the UInt is missing */
                    var = ((libGAP_UInt)nest << 16) + indx;
                }
                break;
            }
        }
        nest++;
    }

    /* try to look up the variable on the error stack                      */
    /* the outer loop runs up the calling stack, while the inner loop runs
       up the static definition stack for each call function */
    lvars0 = libGAP_ErrorLVars;
    nest0 = 0;
    while ( type == ' ' && lvars0 != 0 && lvars0 != libGAP_BottomLVars) {
      lvars = lvars0;
      nest = 0;
      while ( type == ' ' && lvars != 0 && lvars != libGAP_BottomLVars ) {
	nams = libGAP_NAMS_FUNC(libGAP_PTR_BAG(lvars)[0]);
	if (nams != (libGAP_Obj) 0)
	  {
	    indx = libGAP_LEN_LIST( nams );
	    if (indx >= 1024)
	      {
		libGAP_Pr("Warning; Ignoring local names after 1024th in search for %s\n",
		   (libGAP_Int) libGAP_Value,
		   0L);
		indx = 1023;
	      }
	    for ( ; 1 <= indx; indx-- ) {
	      if ( strcmp( libGAP_Value, libGAP_CSTR_STRING(libGAP_ELM_LIST(nams,indx)) ) == 0 ) {
		type = 'd';

		/* Ultrix 4.2 cc get's confused if the UInt is missing     */
		var = ((libGAP_UInt)nest << 16) + indx;
		break;
	      }
	    }
        }
        lvars = libGAP_ENVI_FUNC( libGAP_PTR_BAG( lvars )[0] );
        nest++;
	if (nest >= 65536)
	  {
	    libGAP_Pr("Warning: abandoning search for %s at 65536th higher frame\n",
	       (libGAP_Int)libGAP_Value,0L);
	    break;
	  }
      }
      lvars0 = libGAP_PTR_BAG( lvars0 )[2];
      nest0++;
	if (nest0 >= 65536)
	  {
	    libGAP_Pr("Warning: abandoning search for %s 65536 frames up stack\n",
	       (libGAP_Int)libGAP_Value,0L);
	    break;
	  }
    }

    /* get the variable as a global variable                               */
    if ( type == ' ' ) {
        type = 'g';
        var = libGAP_GVarName( libGAP_Value );
    }

    /* match away the identifier, now that we know the variable            */
    libGAP_Match( libGAP_S_IDENT, "identifier", follow );

    /* if this was actually the beginning of a function literal            */
    /* then we are in the wrong function                                   */
    if ( libGAP_Symbol == libGAP_S_MAPTO ) {
      if (mode == 'r' || mode == 'x')
	{
	  libGAP_ReadFuncExpr1( follow );
	  return;
	}
      else
	libGAP_SyntaxError("function literal in impossible context");
    }

    /* check whether this is an unbound global variable                    */

    if (libGAP_WarnOnUnboundGlobalsRNam == 0)
      libGAP_WarnOnUnboundGlobalsRNam = libGAP_RNamName("WarnOnUnboundGlobals");

    if ( type == 'g'
      && libGAP_CountNams != 0
      && var != libGAP_CurrLHSGVar
      && var != libGAP_Tilde
      && libGAP_VAL_GVAR(var) == 0
      && libGAP_ELM_PLIST(libGAP_ExprGVars,var) == 0
      && ! libGAP_IntrIgnoring
      && ! libGAP_GlobalComesFromEnclosingForLoop(var)
      && (libGAP_GAPInfo == 0 || !libGAP_IS_REC(libGAP_GAPInfo) || !libGAP_ISB_REC(libGAP_GAPInfo,libGAP_WarnOnUnboundGlobalsRNam) ||
             libGAP_ELM_REC(libGAP_GAPInfo,libGAP_WarnOnUnboundGlobalsRNam) != libGAP_False )
      && ! libGAP_SyCompilePlease )
    {
        libGAP_SyntaxError("warning: unbound global variable");
        libGAP_NrError--;
        libGAP_NrErrLine--;
    }

    /* check whether this is a reference to the global variable '~'        */
    if ( type == 'g' && var == libGAP_Tilde ) { libGAP_ReadTilde = 1; }

    /* followed by one or more selectors                                   */
    while ( libGAP_IS_IN( libGAP_Symbol, libGAP_S_LPAREN|libGAP_S_LBRACK|libGAP_S_LBRACE|libGAP_S_DOT ) ) {

        /* so the prefix was a reference                                   */
      if ( libGAP_READ_ERROR() ) {}
        else if ( type == 'l' ) { libGAP_IntrRefLVar( var );           level=0; }
        else if ( type == 'h' ) { libGAP_IntrRefHVar( var );           level=0; }
        else if ( type == 'd' ) { libGAP_IntrRefDVar( var, nest0 - 1 );           level=0; }
        else if ( type == 'g' ) { libGAP_IntrRefGVar( var );           level=0; }
        else if ( type == '[' ) { libGAP_IntrElmList();                         }
        else if ( type == ']' ) { libGAP_IntrElmListLevel( level );             }
        else if ( type == '{' ) { libGAP_IntrElmsList();               level++; }
        else if ( type == '}' ) { libGAP_IntrElmsListLevel( level );   level++; }
        else if ( type == '<' ) { libGAP_IntrElmPosObj();                       }
        else if ( type == '>' ) { libGAP_IntrElmPosObjLevel( level );           }
        else if ( type == '(' ) { libGAP_IntrElmsPosObj();             level++; }
        else if ( type == ')' ) { libGAP_IntrElmsPosObjLevel( level ); level++; }
        else if ( type == '.' ) { libGAP_IntrElmRecName( rnam );       level=0; }
        else if ( type == ':' ) { libGAP_IntrElmRecExpr();             level=0; }
        else if ( type == '!' ) { libGAP_IntrElmComObjName( rnam );    level=0; }
        else if ( type == '|' ) { libGAP_IntrElmComObjExpr();          level=0; }
        else if ( type == 'c' || type == 'C' )
	  { libGAP_IntrFuncCallEnd( 1UL, type == 'C', narg ); level=0; }

        /* <Var> '[' <Expr> ']'  list selector                             */
        if ( libGAP_Symbol == libGAP_S_LBRACK ) {
            libGAP_Match( libGAP_S_LBRACK, "[", follow );
            libGAP_ReadExpr( libGAP_S_RBRACK|follow, 'r' );
            libGAP_Match( libGAP_S_RBRACK, "]", follow );
            type = (level == 0 ? '[' : ']');
        }

        /* <Var> '{' <Expr> '}'  sublist selector                          */
        else if ( libGAP_Symbol == libGAP_S_LBRACE ) {
            libGAP_Match( libGAP_S_LBRACE, "{", follow );
            libGAP_ReadExpr( libGAP_S_RBRACE|follow, 'r' );
            libGAP_Match( libGAP_S_RBRACE, "}", follow );
            type = (level == 0 ? '{' : '}');
        }

        /* <Var> '![' <Expr> ']'  list selector                            */
        else if ( libGAP_Symbol == libGAP_S_BLBRACK ) {
            libGAP_Match( libGAP_S_BLBRACK, "![", follow );
            libGAP_ReadExpr( libGAP_S_RBRACK|follow, 'r' );
            libGAP_Match( libGAP_S_RBRACK, "]", follow );
            type = (level == 0 ? '<' : '>');
        }

        /* <Var> '!{' <Expr> '}'  sublist selector                         */
        else if ( libGAP_Symbol == libGAP_S_BLBRACE ) {
            libGAP_Match( libGAP_S_BLBRACE, "!{", follow );
            libGAP_ReadExpr( libGAP_S_RBRACE|follow, 'r' );
            libGAP_Match( libGAP_S_RBRACE, "}", follow );
            type = (level == 0 ? '(' : ')');
        }

        /* <Var> '.' <Ident>  record selector                              */
        else if ( libGAP_Symbol == libGAP_S_DOT ) {
            libGAP_Match( libGAP_S_DOT, ".", follow );
            if ( libGAP_Symbol == libGAP_S_IDENT || libGAP_Symbol == libGAP_S_INT ) {
                rnam = libGAP_RNamName( libGAP_Value );
                libGAP_Match( libGAP_Symbol, "identifier", follow );
                type = '.';
            }
            else if ( libGAP_Symbol == libGAP_S_LPAREN ) {
                libGAP_Match( libGAP_S_LPAREN, "(", follow );
                libGAP_ReadExpr( libGAP_S_RPAREN|follow, 'r' );
                libGAP_Match( libGAP_S_RPAREN, ")", follow );
                type = ':';
            }
            else {
                libGAP_SyntaxError("record component name expected");
            }
            level = 0;
        }

        /* <Var> '!.' <Ident>  record selector                             */
        else if ( libGAP_Symbol == libGAP_S_BDOT ) {
            libGAP_Match( libGAP_S_BDOT, "!.", follow );
            if ( libGAP_Symbol == libGAP_S_IDENT || libGAP_Symbol == libGAP_S_INT ) {
                rnam = libGAP_RNamName( libGAP_Value );
                libGAP_Match( libGAP_Symbol, "identifier", follow );
                type = '!';
            }
            else if ( libGAP_Symbol == libGAP_S_LPAREN ) {
                libGAP_Match( libGAP_S_LPAREN, "(", follow );
                libGAP_ReadExpr( libGAP_S_RPAREN|follow, 'r' );
                libGAP_Match( libGAP_S_RPAREN, ")", follow );
                type = '|';
            }
            else {
                libGAP_SyntaxError("record component name expected");
            }
            level = 0;
        }

        /* <Var> '(' [ <Expr> { ',' <Expr> } ] ')'  function call          */
        else if ( libGAP_Symbol == libGAP_S_LPAREN ) {
            libGAP_Match( libGAP_S_LPAREN, "(", follow );
            if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncCallBegin(); }
            narg = 0;
            if ( libGAP_Symbol != libGAP_S_RPAREN && libGAP_Symbol != libGAP_S_COLON) {
                libGAP_ReadExpr( libGAP_S_RPAREN|follow, 'r' );
                narg++;
            }
            while ( libGAP_Symbol == libGAP_S_COMMA ) {
                libGAP_Match( libGAP_S_COMMA, ",", follow );
                libGAP_ReadExpr( libGAP_S_RPAREN|follow, 'r' );
                narg++;
            }
	    type = 'c';
	    if (libGAP_Symbol == libGAP_S_COLON ) {
	      libGAP_Match( libGAP_S_COLON, ":", follow );
	      if ( libGAP_Symbol != libGAP_S_RPAREN ) /* save work for empty options */
		{
		  libGAP_ReadFuncCallOptions(libGAP_S_RPAREN | follow);
		  type = 'C';
		}
	    }
	    libGAP_Match( libGAP_S_RPAREN, ")", follow );
        }

    }

    /* if we need a reference                                              */
    if ( mode == 'r' || (mode == 'x' && libGAP_Symbol != libGAP_S_ASSIGN) ) {
        if ( libGAP_READ_ERROR() ) {}
        else if ( type == 'l' ) { libGAP_IntrRefLVar( var );           }
        else if ( type == 'h' ) { libGAP_IntrRefHVar( var );           }
        else if ( type == 'd' ) { libGAP_IntrRefDVar( var, nest0 - 1 );           }
        else if ( type == 'g' ) { libGAP_IntrRefGVar( var );           }
        else if ( type == '[' ) { libGAP_IntrElmList();                }
        else if ( type == ']' ) { libGAP_IntrElmListLevel( level );    }
        else if ( type == '{' ) { libGAP_IntrElmsList();               }
        else if ( type == '}' ) { libGAP_IntrElmsListLevel( level );   }
        else if ( type == '<' ) { libGAP_IntrElmPosObj();                }
        else if ( type == '>' ) { libGAP_IntrElmPosObjLevel( level );    }
        else if ( type == '(' ) { libGAP_IntrElmsPosObj();               }
        else if ( type == ')' ) { libGAP_IntrElmsPosObjLevel( level );   }
        else if ( type == '.' ) { libGAP_IntrElmRecName( rnam );       }
        else if ( type == ':' ) { libGAP_IntrElmRecExpr();             }
        else if ( type == '!' ) { libGAP_IntrElmComObjName( rnam );      }
        else if ( type == '|' ) { libGAP_IntrElmComObjExpr();            }
        else if ( type == 'c' || type == 'C') {
            if ( mode == 'x' && libGAP_Symbol == libGAP_S_SEMICOLON ) {
                libGAP_IntrFuncCallEnd( 0UL, type == 'C', narg );
            }
            else {
                libGAP_IntrFuncCallEnd( 1UL, type == 'C', narg );
            }
        }
    }

    /* if we need a statement                                              */
    else if ( mode == 's' || (mode == 'x' && libGAP_Symbol == libGAP_S_ASSIGN) ) {
        if ( type != 'c' && type != 'C') {
            libGAP_Match( libGAP_S_ASSIGN, ":=", follow );
            if ( libGAP_CountNams == 0 || !libGAP_IntrCoding ) { libGAP_CurrLHSGVar = (type == 'g' ? var : 0); }
            libGAP_ReadExpr( follow, 'r' );
        }
        if ( libGAP_READ_ERROR() ) {}
        else if ( type == 'l' ) { libGAP_IntrAssLVar( var );             }
        else if ( type == 'h' ) { libGAP_IntrAssHVar( var );             }
        else if ( type == 'd' ) { libGAP_IntrAssDVar( var, nest0 - 1 );  }
        else if ( type == 'g' ) { libGAP_IntrAssGVar( var );             }
        else if ( type == '[' ) { libGAP_IntrAssList();                  }
        else if ( type == ']' ) { libGAP_IntrAssListLevel( level );      }
        else if ( type == '{' ) { libGAP_IntrAsssList();                 }
        else if ( type == '}' ) { libGAP_IntrAsssListLevel( level );     }
        else if ( type == '<' ) { libGAP_IntrAssPosObj();                }
        else if ( type == '>' ) { libGAP_IntrAssPosObjLevel( level );    }
        else if ( type == '(' ) { libGAP_IntrAsssPosObj();               }
        else if ( type == ')' ) { libGAP_IntrAsssPosObjLevel( level );   }
        else if ( type == '.' ) { libGAP_IntrAssRecName( rnam );         }
        else if ( type == ':' ) { libGAP_IntrAssRecExpr();               }
        else if ( type == '!' ) { libGAP_IntrAssComObjName( rnam );      }
        else if ( type == '|' ) { libGAP_IntrAssComObjExpr();            }
        else if ( type == 'c' || type == 'C' )
	  { libGAP_IntrFuncCallEnd( 0UL, type == 'C', narg ); }
    }

    /*  if we need an unbind                                               */
    else if ( mode == 'u' ) {
      if (libGAP_Symbol != libGAP_S_RPAREN) {
	libGAP_SyntaxError("'Unbind': argument should be followed by ')'");
      }
        if ( libGAP_READ_ERROR() ) {}
        else if ( type == 'l' ) { libGAP_IntrUnbLVar( var );             }
        else if ( type == 'h' ) { libGAP_IntrUnbHVar( var );             }
        else if ( type == 'd' ) { libGAP_IntrUnbDVar( var, nest0 - 1 );  }
        else if ( type == 'g' ) { libGAP_IntrUnbGVar( var );             }
        else if ( type == '[' ) { libGAP_IntrUnbList();                  }
        else if ( type == '<' ) { libGAP_IntrUnbPosObj();                }
        else if ( type == '.' ) { libGAP_IntrUnbRecName( rnam );         }
        else if ( type == ':' ) { libGAP_IntrUnbRecExpr();               }
        else if ( type == '!' ) { libGAP_IntrUnbComObjName( rnam );      }
        else if ( type == '|' ) { libGAP_IntrUnbComObjExpr();            }
        else { libGAP_SyntaxError("illegal operand for 'Unbind'");       }
    }


    /* if we need an isbound                                               */
    else /* if ( mode == 'i' ) */ {
        if ( libGAP_READ_ERROR() ) {}
        else if ( type == 'l' ) { libGAP_IntrIsbLVar( var );             }
        else if ( type == 'h' ) { libGAP_IntrIsbHVar( var );             }
        else if ( type == 'd' ) { libGAP_IntrIsbDVar( var, nest0 - 1 );  }
        else if ( type == 'g' ) { libGAP_IntrIsbGVar( var );             }
        else if ( type == '[' ) { libGAP_IntrIsbList();                  }
        else if ( type == '<' ) { libGAP_IntrIsbPosObj();                }
        else if ( type == '.' ) { libGAP_IntrIsbRecName( rnam );         }
        else if ( type == ':' ) { libGAP_IntrIsbRecExpr();               }
        else if ( type == '!' ) { libGAP_IntrIsbComObjName( rnam );      }
        else if ( type == '|' ) { libGAP_IntrIsbComObjExpr();            }
        else { libGAP_SyntaxError("illegal operand for 'IsBound'");      }
    }

}


/****************************************************************************
**
*F  ReadIsBound( <follow> ) . . . . . . . . . . .  read an isbound expression
**
**  'ReadIsBound' reads an isbound expression.  In  case of an error it skips
**  all symbols up to one contained in <follow>.
**
**  <Atom> := 'IsBound' '(' <Var> ')'
*/
void            libGAP_ReadIsBound (
    libGAP_TypSymbolSet        follow )
{
    libGAP_Match( libGAP_S_ISBOUND, "IsBound", follow );
    libGAP_Match( libGAP_S_LPAREN, "(", follow );
    libGAP_ReadCallVarAss( libGAP_S_RPAREN|follow, 'i' );
    libGAP_Match( libGAP_S_RPAREN, ")", follow );
}


/****************************************************************************
**
*F  ReadPerm( <follow> )  . . . . . . . . . . . . . . . .  read a permutation
**
**  'ReadPerm' reads a permutation.  In case of an error it skips all symbols
**  up to one contained in <follow>.
**
**  Note that the first expression has already been read.  The reason is that
**  until the first  expression has been  read and a  comma is found it could
**  also be a parenthesized expression.
**
**  <Perm> :=  ( <Expr> {, <Expr>} ) { ( <Expr> {, <Expr>} ) }
**
*/
void libGAP_ReadPerm (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       nrc;            /* number of cycles                */
    volatile libGAP_UInt       nrx;            /* number of expressions in cycle  */

    /* read the first cycle (first expression has already been read)       */
    nrx = 1;
    while ( libGAP_Symbol == libGAP_S_COMMA ) {
        libGAP_Match( libGAP_S_COMMA, ",", follow );
        libGAP_ReadExpr( libGAP_S_COMMA|libGAP_S_RPAREN|follow, 'r' );
        nrx++;
    }
    libGAP_Match( libGAP_S_RPAREN, ")", follow );
    nrc = 1;
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrPermCycle( nrx, nrc ); }

    /* read the remaining cycles                                           */
    while ( libGAP_Symbol == libGAP_S_LPAREN ) {
        libGAP_Match( libGAP_S_LPAREN, "(", follow );
        libGAP_ReadExpr( libGAP_S_COMMA|libGAP_S_RPAREN|follow, 'r' );
        nrx = 1;
        while ( libGAP_Symbol == libGAP_S_COMMA ) {
            libGAP_Match( libGAP_S_COMMA, ",", follow );
            libGAP_ReadExpr( libGAP_S_COMMA|libGAP_S_RPAREN|follow, 'r' );
            nrx++;
        }
        libGAP_Match( libGAP_S_RPAREN, ")", follow );
        nrc++;
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrPermCycle( nrx, nrc ); }
    }

    /* that was the permutation                                            */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrPerm( nrc ); }
}

/****************************************************************************
**
*F  ReadLongNumber( <follow> )  . . . . . . . . . . . . . . . read a long integer
**
**  A `long integer' here means one whose digits don't fit into `Value',
**  see scanner.c.  This function copies repeatedly  digits from `Value'
**  into a GAP string until the full integer is read.
**
*/

static libGAP_UInt libGAP_appendToString(libGAP_Obj string, libGAP_UInt len)
{
       libGAP_UInt len1 = strlen(libGAP_Value);
       libGAP_GROW_STRING(string, len+len1+1);
       memcpy(libGAP_CHARS_STRING(string) + len, (void *)libGAP_Value, len1+1);
       libGAP_SET_LEN_STRING(string, len+len1);
       return len + len1;
}

void libGAP_ReadLongNumber(
      libGAP_TypSymbolSet        follow )
{
     libGAP_Obj  string;
     libGAP_UInt len;
     libGAP_UInt status;
     libGAP_UInt done;

     /* string in which to accumulate number */
     len = strlen(libGAP_Value);
     libGAP_C_NEW_STRING( string, len, libGAP_Value );
     done = 0;

     while (!done) {
       /* remember the current symbol and get the next one */
       status = libGAP_Symbol;
       libGAP_Match(libGAP_Symbol, "partial number", follow);

       /* Now there are just lots of cases */
       switch (status) {
       case libGAP_S_PARTIALINT:
	 switch (libGAP_Symbol) {
	 case libGAP_S_INT:
	   len = libGAP_appendToString(string, len);
	   libGAP_Match(libGAP_S_INT, "integer", follow);
	   libGAP_IntrLongIntExpr(string);
	   done = 1;
	   break;

	 case libGAP_S_PARTIALINT:
	   len = libGAP_appendToString(string, len);
	   /*	   Match(S_PARTIALINT, "integer", follow);*/
	   break;

	 case libGAP_S_PARTIALFLOAT1:
	   assert(0);
	   libGAP_Pr("Parsing error, this should never happen", 0L, 0L);
	   libGAP_SyExit(2);

	 case libGAP_S_PARTIALFLOAT2:
	 case libGAP_S_PARTIALFLOAT3:
	 case libGAP_S_PARTIALFLOAT4:
	   status = libGAP_Symbol;
	   len = libGAP_appendToString(string, len);
	   /* Match(Symbol, "float", follow); */
	   break;

	 case libGAP_S_FLOAT:
	   len = libGAP_appendToString(string, len);
	   libGAP_Match(libGAP_S_FLOAT, "float", follow);
	   libGAP_IntrLongFloatExpr(string);
	   done = 1;
	   break;

	 case libGAP_S_IDENT:
	   libGAP_SyntaxError("Identifier over 1024 characters");

	 default:
	   len = libGAP_appendToString(string, len);
	   libGAP_IntrLongIntExpr(string);
	   done = 1;
	 }
	 break;

       case libGAP_S_PARTIALFLOAT1:
	 switch (libGAP_Symbol) {
	 case libGAP_S_INT:
	 case libGAP_S_PARTIALINT:
	 case libGAP_S_PARTIALFLOAT1:
	   assert(0);
	   libGAP_Pr("Parsing error, this should never happen", 0L, 0L);
	   libGAP_SyExit(2);


	 case libGAP_S_PARTIALFLOAT2:
	 case libGAP_S_PARTIALFLOAT3:
	 case libGAP_S_PARTIALFLOAT4:
	   status = libGAP_Symbol;
	   len = libGAP_appendToString(string, len);
	   /* Match(Symbol, "float", follow); */
	   break;

	 case libGAP_S_FLOAT:
	   len = libGAP_appendToString(string, len);
	   libGAP_Match(libGAP_S_FLOAT, "float", follow);
	   libGAP_IntrLongFloatExpr(string);
	   done = 1;
	   break;

	 default:
	   libGAP_SyntaxError("Badly Formed Number");
	 }
	 break;

       case libGAP_S_PARTIALFLOAT2:
	 switch (libGAP_Symbol) {
	 case libGAP_S_INT:
	 case libGAP_S_PARTIALINT:
	 case libGAP_S_PARTIALFLOAT1:
	   assert(0);
	   libGAP_Pr("Parsing error, this should never happen", 0L, 0L);
	   libGAP_SyExit(2);


	 case libGAP_S_PARTIALFLOAT2:
	 case libGAP_S_PARTIALFLOAT3:
	 case libGAP_S_PARTIALFLOAT4:
	   status = libGAP_Symbol;
	   len = libGAP_appendToString(string, len);
	   /* Match(Symbol, "float", follow); */
	   break;

	 case libGAP_S_FLOAT:
	   len = libGAP_appendToString(string, len);
	   libGAP_Match(libGAP_S_FLOAT, "float", follow);
	   libGAP_IntrLongFloatExpr(string);
	   done = 1;
	   break;


	 case libGAP_S_IDENT:
	   libGAP_SyntaxError("Badly Formed Number");

	 default:
	   len = libGAP_appendToString(string, len);
	   libGAP_IntrLongFloatExpr(string);
	   done = 1;
	 }
	 break;

       case libGAP_S_PARTIALFLOAT3:
	 switch (libGAP_Symbol) {
	 case libGAP_S_INT:
	 case libGAP_S_PARTIALINT:
	 case libGAP_S_PARTIALFLOAT1:
	 case libGAP_S_PARTIALFLOAT2:
	 case libGAP_S_PARTIALFLOAT3:
	   assert(0);
	   libGAP_Pr("Parsing error, this should never happen", 0L, 0L);
	   libGAP_SyExit(2);


	 case libGAP_S_PARTIALFLOAT4:
	   status = libGAP_Symbol;
	   len = libGAP_appendToString(string, len);
	   /* Match(Symbol, "float", follow); */
	   break;

	 case libGAP_S_FLOAT:
	   len = libGAP_appendToString(string, len);
	   libGAP_Match(libGAP_S_FLOAT, "float", follow);
	   libGAP_IntrLongFloatExpr(string);
	   done = 1;
	   break;


	 default:
	   libGAP_SyntaxError("Badly Formed Number");

	 }
	 break;
       case libGAP_S_PARTIALFLOAT4:
	 switch (libGAP_Symbol) {
	 case libGAP_S_INT:
	 case libGAP_S_PARTIALINT:
	 case libGAP_S_PARTIALFLOAT1:
	 case libGAP_S_PARTIALFLOAT2:
	 case libGAP_S_PARTIALFLOAT3:
	   assert(0);
	   libGAP_Pr("Parsing error, this should never happen", 0L, 0L);
	   libGAP_SyExit(2);


	 case libGAP_S_PARTIALFLOAT4:
	   status = libGAP_Symbol;
	   len = libGAP_appendToString(string, len);
	   /* Match(Symbol, "float", follow); */
	   break;

	 case libGAP_S_FLOAT:
	   len = libGAP_appendToString(string, len);
	   libGAP_Match(libGAP_S_FLOAT, "float", follow);
	   libGAP_IntrLongFloatExpr(string);
	   done = 1;
	   break;

	 case libGAP_S_IDENT:
	   libGAP_SyntaxError("Badly Formed Number");

	 default:
	   len = libGAP_appendToString(string, len);
	   libGAP_IntrLongFloatExpr(string);
	   done = 1;

	 }
	 break;
       default:
	 assert(0);
	 libGAP_Pr("Parsing error, this should never happen", 0L, 0L);
	 libGAP_SyExit(2);
       }
     }
}

/****************************************************************************
**
*F  ReadString( <follow> )  . . . . . . . . . . . . . . read a (long) string
**
**  A string is  read by copying parts of `Value'  (see scanner.c) given
**  by `ValueLen' into  a string GAP object. This is  repeated until the
**  end of the string is reached.
**
*/
void libGAP_ReadString(
      libGAP_TypSymbolSet        follow )
{
     libGAP_Obj  string;
     libGAP_UInt len;

     libGAP_C_NEW_STRING( string, libGAP_ValueLen, libGAP_Value );
     len = libGAP_ValueLen;

     while (libGAP_Symbol == libGAP_S_PARTIALSTRING) {
         libGAP_Match(libGAP_S_PARTIALSTRING, "", follow);
         libGAP_GROW_STRING(string, len + libGAP_ValueLen);
         memcpy(libGAP_CHARS_STRING(string) + len, (void *)libGAP_Value,
                                        libGAP_ValueLen);
         len += libGAP_ValueLen;
     }

     libGAP_Match(libGAP_S_STRING, "", follow);
     libGAP_SET_LEN_STRING(string, len);
     /* ensure trailing zero for interpretation as C-string */
     *(libGAP_CHARS_STRING(string) + len) = 0;
     libGAP_IntrStringExpr( string );
}

/****************************************************************************
**
*F  ReadListExpr( <follow> )  . . . . . . . . . . . . . . . . . . read a list
**
**  'ReadListExpr'  reads a list literal expression.   In case of an error it
**  skips all symbols up to one contained in <follow>.
**
**  <List> := '[' [ <Expr> ] {',' [ <Expr> ] } ']'
**         |  '[' <Expr> [',' <Expr>] '..' <Expr> ']'
*/
void libGAP_ReadListExpr (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       pos;            /* actual position of element      */
    volatile libGAP_UInt       nr;             /* number of elements              */
    volatile libGAP_UInt       range;          /* is the list expression a range  */

    /* '['                                                                 */
    libGAP_Match( libGAP_S_LBRACK, "[", follow );
    libGAP_ReadTop++;
    if ( libGAP_ReadTop == 1 ) { libGAP_ReadTilde = 0; }
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrListExprBegin( (libGAP_ReadTop == 1) ); }
    pos   = 1;
    nr    = 0;
    range = 0;

    /* [ <Expr> ]                                                          */
    if ( libGAP_Symbol != libGAP_S_COMMA && libGAP_Symbol != libGAP_S_RBRACK ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrListExprBeginElm( pos ); }
        libGAP_ReadExpr( libGAP_S_RBRACK|follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrListExprEndElm(); }
        nr++;
    }

    /* {',' [ <Expr> ] }                                                   */
    while ( libGAP_Symbol == libGAP_S_COMMA ) {
        libGAP_Match( libGAP_S_COMMA, ",", follow );
        pos++;
        if ( libGAP_Symbol != libGAP_S_COMMA && libGAP_Symbol != libGAP_S_RBRACK ) {
            if ( ! libGAP_READ_ERROR() ) { libGAP_IntrListExprBeginElm( pos ); }
            libGAP_ReadExpr( libGAP_S_RBRACK|follow, 'r' );
            if ( ! libGAP_READ_ERROR() ) { libGAP_IntrListExprEndElm(); }
            nr++;
        }
    }

    /* '..' <Expr> ']'                                                     */
    if ( libGAP_Symbol == libGAP_S_DOTDOT ) {
        if ( pos != nr ) {
            libGAP_SyntaxError("must have no unbound entries in range");
        }
        if ( 2 < nr ) {
            libGAP_SyntaxError("must have at most 2 entries before '..'");
        }
        range = 1;
        libGAP_Match( libGAP_S_DOTDOT, "..", follow );
        pos++;
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrListExprBeginElm( pos ); }
        libGAP_ReadExpr( libGAP_S_RBRACK|follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrListExprEndElm(); }
        nr++;
        if ( libGAP_ReadTop == 1 && libGAP_ReadTilde == 1 ) {
            libGAP_SyntaxError("sorry, '~' not allowed in range");
        }
    }

    /* ']'                                                                 */
    libGAP_Match( libGAP_S_RBRACK, "]", follow );
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrListExprEnd( nr, range, (libGAP_ReadTop == 1), (libGAP_ReadTilde == 1) );
    }
    if ( libGAP_ReadTop == 1 ) { libGAP_ReadTilde = 0; }
    libGAP_ReadTop--;
}


/****************************************************************************
**
*F  ReadRecExpr( <follow> ) . . . . . . . . . . . . . . . . . . read a record
**
**  'ReadRecExpr' reads a record literal expression.  In  case of an error it
**  skips all symbols up to one contained in <follow>.
**
**  <Record> := 'rec( [ <Ident>:=<Expr> {, <Ident>:=<Expr> } ] )'
*/
void libGAP_ReadRecExpr (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       rnam;           /* record component name           */
    volatile libGAP_UInt       nr;             /* number of components            */

    /* 'rec('                                                              */
    libGAP_Match( libGAP_S_REC, "rec", follow );
    libGAP_Match( libGAP_S_LPAREN, "(", follow|libGAP_S_RPAREN|libGAP_S_COMMA );
    libGAP_ReadTop++;
    if ( libGAP_ReadTop == 1 ) { libGAP_ReadTilde = 0; }
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRecExprBegin( (libGAP_ReadTop == 1) ); }
    nr = 0;

    /* [ <Ident> | '(' <Expr> ')' ':=' <Expr>                              */
    do {
      if (nr || libGAP_Symbol == libGAP_S_COMMA) {
	libGAP_Match(libGAP_S_COMMA, ",", follow);
      }
      if ( libGAP_Symbol != libGAP_S_RPAREN ) {
        if ( libGAP_Symbol == libGAP_S_INT ) {
	  rnam = libGAP_RNamName( libGAP_Value );
	  libGAP_Match( libGAP_S_INT, "integer", follow );
	  if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRecExprBeginElmName( rnam ); }
        }
        else if ( libGAP_Symbol == libGAP_S_IDENT ) {
	  rnam = libGAP_RNamName( libGAP_Value );
	  libGAP_Match( libGAP_S_IDENT, "identifier", follow );
	  if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRecExprBeginElmName( rnam ); }
        }
        else if ( libGAP_Symbol == libGAP_S_LPAREN ) {
	  libGAP_Match( libGAP_S_LPAREN, "(", follow );
	  libGAP_ReadExpr( follow, 'r' );
	  libGAP_Match( libGAP_S_RPAREN, ")", follow );
	  if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRecExprBeginElmExpr(); }
        }
        else {
	  libGAP_SyntaxError("identifier expected");
        }
        libGAP_Match( libGAP_S_ASSIGN, ":=", follow );
        libGAP_ReadExpr( libGAP_S_RPAREN|follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRecExprEndElm(); }
        nr++;
      }

    }
  while ( libGAP_Symbol == libGAP_S_COMMA );

    /* ')'                                                                 */
    libGAP_Match( libGAP_S_RPAREN, ")", follow );
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrRecExprEnd( nr, (libGAP_ReadTop == 1), (libGAP_ReadTilde == 1) );
    }
    if ( libGAP_ReadTop == 1) { libGAP_ReadTilde = 0; }
    libGAP_ReadTop--;
}


/****************************************************************************
**
*F  ReadFuncExpr( <follow> )  . . . . . . . . . .  read a function definition
**
**  'ReadFuncExpr' reads a function literal expression.  In  case of an error
**  it skips all symbols up to one contained in <follow>.
**
**  <Function> := 'function (' [ <Ident> {',' <Ident>} ] ')'
**                             [ 'local'  <Ident> {',' <Ident>} ';' ]
**                             <Statments>
**                'end'
*/


void libGAP_ReadFuncExpr (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_Obj        nams;           /* list of local variables names   */
    volatile libGAP_Obj        name;           /* one local variable name         */
    volatile libGAP_UInt       narg;           /* number of arguments             */
    volatile libGAP_UInt       nloc;           /* number of locals                */
    volatile libGAP_UInt       nr;             /* number of statements            */
    volatile libGAP_UInt       i;              /* loop variable                   */
    volatile libGAP_UInt       nrError;        /* copy of <NrError>               */
    volatile libGAP_Bag        currLVars;      /* copy of <CurrLVars>             */
    volatile libGAP_Int        startLine;      /* line number of function keyword */

    /* begin the function               */
    startLine = libGAP_Input->number;
    libGAP_Match( libGAP_S_FUNCTION, "function", follow );
    libGAP_Match( libGAP_S_LPAREN, "(", libGAP_S_IDENT|libGAP_S_RPAREN|libGAP_S_LOCAL|libGAP_STATBEGIN|libGAP_S_END|follow );

    /* make and push the new local variables list (args and locals)        */
    narg = nloc = 0;
    nams = libGAP_NEW_PLIST( libGAP_T_PLIST, narg+nloc );
    libGAP_SET_LEN_PLIST( nams, narg+nloc );
    libGAP_CountNams += 1;
    libGAP_ASS_LIST( libGAP_StackNams, libGAP_CountNams, nams );
    if ( libGAP_Symbol != libGAP_S_RPAREN ) {
        libGAP_C_NEW_STRING_DYN( name, libGAP_Value );
        narg += 1;
        libGAP_ASS_LIST( nams, narg+nloc, name );
        libGAP_Match(libGAP_S_IDENT,"identifier",libGAP_S_RPAREN|libGAP_S_LOCAL|libGAP_STATBEGIN|libGAP_S_END|follow);
    }
    while ( libGAP_Symbol == libGAP_S_COMMA ) {
        libGAP_Match( libGAP_S_COMMA, ",", follow );
        for ( i = 1; i <= narg; i++ ) {
            if ( strcmp(libGAP_CSTR_STRING(libGAP_ELM_LIST(nams,i)),libGAP_Value) == 0 ) {
                libGAP_SyntaxError("name used for two arguments");
            }
        }
        libGAP_C_NEW_STRING_DYN( name, libGAP_Value );
        narg += 1;
        libGAP_ASS_LIST( nams, narg+nloc, name );
        libGAP_Match(libGAP_S_IDENT,"identifier",libGAP_S_RPAREN|libGAP_S_LOCAL|libGAP_STATBEGIN|libGAP_S_END|follow);
    }
    libGAP_Match( libGAP_S_RPAREN, ")", libGAP_S_LOCAL|libGAP_STATBEGIN|libGAP_S_END|follow );
    if ( libGAP_Symbol == libGAP_S_LOCAL ) {
        libGAP_Match( libGAP_S_LOCAL, "local", follow );
        for ( i = 1; i <= narg; i++ ) {
            if ( strcmp(libGAP_CSTR_STRING(libGAP_ELM_LIST(nams,i)),libGAP_Value) == 0 ) {
                libGAP_SyntaxError("name used for argument and local");
            }
        }
        libGAP_C_NEW_STRING_DYN( name, libGAP_Value );
        nloc += 1;
        libGAP_ASS_LIST( nams, narg+nloc, name );
        libGAP_Match( libGAP_S_IDENT, "identifier", libGAP_STATBEGIN|libGAP_S_END|follow );
        while ( libGAP_Symbol == libGAP_S_COMMA ) {
            /* init to avoid strange message in case of empty string */
            libGAP_Value[0] = '\0';
            libGAP_Match( libGAP_S_COMMA, ",", follow );
            for ( i = 1; i <= narg; i++ ) {
                if ( strcmp(libGAP_CSTR_STRING(libGAP_ELM_LIST(nams,i)),libGAP_Value) == 0 ) {
                    libGAP_SyntaxError("name used for argument and local");
                }
            }
            for ( i = narg+1; i <= narg+nloc; i++ ) {
                if ( strcmp(libGAP_CSTR_STRING(libGAP_ELM_LIST(nams,i)),libGAP_Value) == 0 ) {
                    libGAP_SyntaxError("name used for two locals");
                }
            }
            libGAP_C_NEW_STRING_DYN( name, libGAP_Value );
            nloc += 1;
            libGAP_ASS_LIST( nams, narg+nloc, name );
            libGAP_Match( libGAP_S_IDENT, "identifier", libGAP_STATBEGIN|libGAP_S_END|follow );
        }
        libGAP_Match( libGAP_S_SEMICOLON, ";", libGAP_STATBEGIN|libGAP_S_END|follow );
    }

    /* 'function( arg )' takes a variable number of arguments              */
    if ( narg == 1 && ! strcmp( "arg", libGAP_CSTR_STRING( libGAP_ELM_LIST(nams,1) ) ) )
        narg = -1;

    /* remember the current variables in case of an error                  */
    currLVars = libGAP_CurrLVars;
    nrError   = libGAP_NrError;

    /* now finally begin the function                                      */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncExprBegin( narg, nloc, nams, startLine ); }

    /* <Statments>                                                         */
    nr = libGAP_ReadStats( libGAP_S_END|follow );

    /* and end the function again                                          */
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrFuncExprEnd( nr, 0UL );
    }

    /* an error has occured *after* the 'IntrFuncExprEnd'                  */
    else if ( nrError == 0 && libGAP_IntrCoding ) {
        libGAP_CodeEnd(1);
        libGAP_IntrCoding--;
        libGAP_CurrLVars = currLVars;
        libGAP_PtrLVars  = libGAP_PTR_BAG( libGAP_CurrLVars );
        libGAP_PtrBody   = (libGAP_Stat*) libGAP_PTR_BAG( libGAP_BODY_FUNC( libGAP_CURR_FUNC ) );
    }

    /* pop the new local variables list                                    */
    assert(libGAP_CountNams > 0);
    libGAP_CountNams--;

    /* 'end'                                                               */
    libGAP_Match( libGAP_S_END, "end", follow );
}


/****************************************************************************
**
*F  ReadFuncExpr1(<follow>) . . . . . . . . . . .  read a function expression
**
**  'ReadFuncExpr1' reads  an abbreviated  function literal   expression.  In
**  case of an error it skips all symbols up to one contained in <follow>.
**
**      <Function>      := <Var> '->' <Expr>
*/
void libGAP_ReadFuncExpr1 (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_Obj        nams;           /* list of local variables names   */
    volatile libGAP_Obj        name;           /* one local variable name         */
    volatile libGAP_UInt       nrError;        /* copy of <NrError>               */
    volatile libGAP_Bag        currLVars;      /* copy of <CurrLVars>             */

    /* make and push the new local variables list                          */
    nams = libGAP_NEW_PLIST( libGAP_T_PLIST, 1 );
    libGAP_SET_LEN_PLIST( nams, 0 );
    libGAP_CountNams++;
    libGAP_ASS_LIST( libGAP_StackNams, libGAP_CountNams, nams );
    libGAP_C_NEW_STRING_DYN( name, libGAP_Value );
    libGAP_ASS_LIST( nams, 1, name );

    /* match away the '->'                                                 */
    libGAP_Match( libGAP_S_MAPTO, "->", follow );

    /* remember the current variables in case of an error                  */
    currLVars = libGAP_CurrLVars;
    nrError   = libGAP_NrError;

    /* begin interpreting the function expression (with 1 argument)        */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFuncExprBegin( 1L, 0L, nams, libGAP_Input->number ); }

    /* read the expression and turn it into a return-statement             */
    libGAP_ReadExpr( follow, 'r' );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrReturnObj(); }

    /* end interpreting the function expression (with 1 statement)         */
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrFuncExprEnd( 1UL, 1UL );
    }

    /* an error has occured *after* the 'IntrFuncExprEnd'                  */
    else if ( nrError == 0  && libGAP_IntrCoding ) {
        libGAP_CodeEnd(1);
        libGAP_IntrCoding--;
        libGAP_CurrLVars = currLVars;
        libGAP_PtrLVars  = libGAP_PTR_BAG( libGAP_CurrLVars );
        libGAP_PtrBody   = (libGAP_Stat*) libGAP_PTR_BAG( libGAP_BODY_FUNC( libGAP_CURR_FUNC ) );
    }

    /* pop the new local variables list                                    */
    assert(libGAP_CountNams > 0);
    libGAP_CountNams--;
}


/****************************************************************************
**
*F  ReadLiteral( <follow> ) . . . . . . . . . . . . . . . . . .  read an atom
**
**  'ReadLiteral' reads a  literal expression.  In  case of an error it skips
**  all symbols up to one contained in <follow>.
**
**  <Literal> := <Int>
**            |  'true'
**            |  'false'
**            |  <Char>
**            |  <Perm>
**            |  <String>
**            |  <List>
**            |  <Record>
**            |  <Function>
**
**  <Int>     := 0|1|..|9 { 0|1|..|9 }
**
**  <Char>    := ' <any character> '
**
**  <String>  := " { <any character> } "
*/
void libGAP_ReadLiteral (
    libGAP_TypSymbolSet        follow )
{
    /* <Int>                                                               */
    if ( libGAP_Symbol == libGAP_S_INT ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIntExpr( libGAP_Value ); }
        libGAP_Match( libGAP_S_INT, "integer", follow );
    }

    /* <Float> */
    else if ( libGAP_Symbol == libGAP_S_FLOAT ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrFloatExpr( libGAP_Value ); }
        libGAP_Match( libGAP_S_FLOAT, "float", follow );
    }


    /* partial Int */
    else if ( libGAP_Symbol == libGAP_S_PARTIALINT || libGAP_Symbol == libGAP_S_PARTIALFLOAT1 ||
	      libGAP_Symbol == libGAP_S_PARTIALFLOAT2 ) {
         libGAP_ReadLongNumber( follow );
    }

    /* 'true'                                                              */
    else if ( libGAP_Symbol == libGAP_S_TRUE ) {
        libGAP_Match( libGAP_S_TRUE, "true", follow );
        libGAP_IntrTrueExpr();
    }

    /* 'false'                                                             */
    else if ( libGAP_Symbol == libGAP_S_FALSE ) {
        libGAP_Match( libGAP_S_FALSE, "false", follow );
        libGAP_IntrFalseExpr();
    }

    /* <Char>                                                              */
    else if ( libGAP_Symbol == libGAP_S_CHAR ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrCharExpr( libGAP_Value[0] ); }
        libGAP_Match( libGAP_S_CHAR, "character", follow );
    }

    /* (partial) string */
    else if ( libGAP_Symbol == libGAP_S_STRING || libGAP_Symbol == libGAP_S_PARTIALSTRING ) {
        libGAP_ReadString( follow );
    }

    /* <List>                                                              */
    else if ( libGAP_Symbol == libGAP_S_LBRACK ) {
        libGAP_ReadListExpr( follow );
    }

    /* <Rec>                                                               */
    else if ( libGAP_Symbol == libGAP_S_REC ) {
        libGAP_ReadRecExpr( follow );
    }

    /* <Function>                                                          */
    else if ( libGAP_Symbol == libGAP_S_FUNCTION ) {
        libGAP_ReadFuncExpr( follow );
    }

    else if (libGAP_Symbol == libGAP_S_DOT ) {
      /* Hack The only way a dot could turn up here is in
       a floating point literal that starts with .. So, change the token
      to  a partial float of the right kind to end with a . and an
      associated value and dive into the long float literal handler in the parser*/
      libGAP_Symbol = libGAP_S_PARTIALFLOAT1;
      libGAP_Value[0] = '.';
      libGAP_Value[1] = '\0';
      libGAP_ReadLongNumber( follow );
    }

    /* signal an error, we want to see a literal                           */
    else {
        libGAP_Match( libGAP_S_INT, "literal", follow );
    }
}


/****************************************************************************
**
*F  ReadAtom( <follow>, <mode> )  . . . . . . . . . . . . . . .  read an atom
**
**  'ReadAtom' reads an atom.  In case  of an error it skips  all symbols up to
**  one contained in <follow>.
**
**   <Atom> := <Var>
**          |  'IsBound' '(' <Var> ')'
**          |  <Literal>
**          |  '(' <Expr> ')'
*/
void libGAP_ReadAtom (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    /* read a variable                                                     */
    if ( libGAP_Symbol == libGAP_S_IDENT ) {
        libGAP_ReadCallVarAss( follow, mode );
    }

    /* 'IsBound' '(' <Var> ')'                                             */
    else if ( libGAP_Symbol == libGAP_S_ISBOUND ) {
        libGAP_ReadIsBound( follow );
    }
    /* otherwise read a literal expression                                 */
    else if (libGAP_IS_IN(libGAP_Symbol,libGAP_S_INT|libGAP_S_TRUE|libGAP_S_FALSE|libGAP_S_CHAR|libGAP_S_STRING|libGAP_S_LBRACK|
                          libGAP_S_REC|libGAP_S_FUNCTION| libGAP_S_FLOAT | libGAP_S_DOT))
    {
        libGAP_ReadLiteral( follow );
    }

    /* '(' <Expr> ')'                                                      */
    else if ( libGAP_Symbol == libGAP_S_LPAREN ) {
        libGAP_Match( libGAP_S_LPAREN, "(", follow );
        if ( libGAP_Symbol == libGAP_S_RPAREN ) {
            libGAP_Match( libGAP_S_RPAREN, ")", follow );
            if ( ! libGAP_READ_ERROR() ) { libGAP_IntrPerm( 0UL ); }
            return;
        }
        libGAP_ReadExpr( libGAP_S_RPAREN|follow, 'r' );
        if ( libGAP_Symbol == libGAP_S_COMMA ) {
            libGAP_ReadPerm( follow );
            return;
        }
        libGAP_Match( libGAP_S_RPAREN, ")", follow );
    }

    /* otherwise signal an error                                           */
    else {
        libGAP_Match( libGAP_S_INT, "expression", follow );
    }
}


/****************************************************************************
**
*F  ReadFactor( <follow>, <mode> )  . . . . . . . . . . . . . . read a factor
**
**  'ReadFactor' reads a factor.  In case of an error it skips all symbols up
**  to one contained in <follow>.
**
**  <Factor> := {'+'|'-'} <Atom> [ '^' {'+'|'-'} <Atom> ]
*/
void libGAP_ReadFactor (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    volatile libGAP_Int        sign1;
    volatile libGAP_Int        sign2;

    /* { '+'|'-' }  leading sign                                           */
    sign1 = 0;
    if ( libGAP_Symbol == libGAP_S_MINUS  || libGAP_Symbol == libGAP_S_PLUS ) {
        if ( sign1 == 0 )  sign1 = 1;
        if ( libGAP_Symbol == libGAP_S_MINUS ) { sign1 = -sign1; }
        libGAP_Match( libGAP_Symbol, "unary + or -", follow );
    }

    /* <Atom>                                                              */
    libGAP_ReadAtom( follow, (sign1 == 0 ? mode : 'r') );

    /* ['^' <Atom> ] implemented as {'^' <Atom> } for better error message */
    while ( libGAP_Symbol == libGAP_S_POW ) {

        /* match the '^' away                                              */
        libGAP_Match( libGAP_S_POW, "^", follow );

        /* { '+'|'-' }  leading sign                                       */
        sign2 = 0;
        if ( libGAP_Symbol == libGAP_S_MINUS  || libGAP_Symbol == libGAP_S_PLUS ) {
            if ( sign2 == 0 )  sign2 = 1;
            if ( libGAP_Symbol == libGAP_S_MINUS ) { sign2 = -sign2; }
            libGAP_Match( libGAP_Symbol, "unary + or -", follow );
        }

        /* ['^' <Atom>]                                                    */
        libGAP_ReadAtom( follow, 'r' );

        /* interpret the unary minus                                       */
        if ( sign2 == -1 && ! libGAP_READ_ERROR() ) { libGAP_IntrAInv(); }

        /* interpret the power                                             */
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrPow(); }

        /* check for multiple '^'                                          */
        if ( libGAP_Symbol == libGAP_S_POW ) { libGAP_SyntaxError("'^' is not associative"); }

    }

    /* interpret the unary minus                                           */
    if ( sign1 == -1 && ! libGAP_READ_ERROR() ) { libGAP_IntrAInv(); }
}


/****************************************************************************
**
*F  ReadTerm( <follow>, <mode> )  . . . . . . . . . . . . . . . . read a term
**
**  'ReadTerm' reads a term.  In case of an error it  skips all symbols up to
**  one contained in <follow>.
**
**  <Term> := <Factor> { '*'|'/'|'mod' <Factor> }
*/
void libGAP_ReadTerm (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    volatile libGAP_UInt       symbol;

    /* <Factor>                                                            */
    libGAP_ReadFactor( follow, mode );

    /* { '*'|'/'|'mod' <Factor> }                                          */
    /* do not use 'IS_IN', since 'IS_IN(S_POW,S_MULT|S_DIV|S_MOD)' is true */
    while ( libGAP_Symbol == libGAP_S_MULT || libGAP_Symbol == libGAP_S_DIV || libGAP_Symbol == libGAP_S_MOD ) {
        symbol = libGAP_Symbol;
        libGAP_Match( libGAP_Symbol, "*, /, or mod", follow );
        libGAP_ReadFactor( follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) {
            if      ( symbol == libGAP_S_MULT ) { libGAP_IntrProd(); }
            else if ( symbol == libGAP_S_DIV  ) { libGAP_IntrQuo();  }
            else if ( symbol == libGAP_S_MOD  ) { libGAP_IntrMod();  }
        }
    }
}


/****************************************************************************
**
*F  ReadAri( <follow>, <mode> ) . . . . . . . . read an arithmetic expression
**
**  'ReadAri' reads an  arithmetic expression.  In  case of an error it skips
**  all symbols up to one contained in <follow>.
**
**  <Arith> := <Term> { '+'|'-' <Term> }
*/
void libGAP_ReadAri (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    libGAP_UInt                symbol;

    /* <Term>                                                              */
    libGAP_ReadTerm( follow, mode );

    /* { '+'|'-' <Term> }                                                  */
    while ( libGAP_IS_IN( libGAP_Symbol, libGAP_S_PLUS|libGAP_S_MINUS ) ) {
        symbol = libGAP_Symbol;
        libGAP_Match( libGAP_Symbol, "+ or -", follow );
        libGAP_ReadTerm( follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) {
            if      ( symbol == libGAP_S_PLUS  ) { libGAP_IntrSum();  }
            else if ( symbol == libGAP_S_MINUS ) { libGAP_IntrDiff(); }
        }
    }
}


/****************************************************************************
**
*F  ReadRel( <follow>, <mode> ) . . . . . . . .  read a relational expression
**
**  'ReadRel' reads a relational  expression.  In case  of an error it  skips
**  all symbols up to one contained in <follow>.
**
**  <Rel> := { 'not' } <Arith> { '=|<>|<|>|<=|>=|in' <Arith> }
*/
void libGAP_ReadRel (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    volatile libGAP_UInt       symbol;
    volatile libGAP_UInt       isNot;

    /* { 'not' }                                                           */
    isNot = 0;
    while ( libGAP_Symbol == libGAP_S_NOT ) {
        isNot++;
        libGAP_Match( libGAP_S_NOT, "not", follow );
    }

    /* <Arith>                                                             */
    libGAP_ReadAri( follow, (isNot == 0 ? mode : 'r') );

    /* { '=|<>|<|>|<=|>=|in' <Arith> }                                     */
    if ( libGAP_IS_IN( libGAP_Symbol, libGAP_S_EQ|libGAP_S_LT|libGAP_S_GT|libGAP_S_NE|libGAP_S_LE|libGAP_S_GE|libGAP_S_IN ) ) {
        symbol = libGAP_Symbol;
        libGAP_Match( libGAP_Symbol, "comparison operator", follow );
        libGAP_ReadAri( follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) {
            if      ( symbol == libGAP_S_EQ ) { libGAP_IntrEq(); }
            else if ( symbol == libGAP_S_NE ) { libGAP_IntrNe(); }
            else if ( symbol == libGAP_S_LT ) { libGAP_IntrLt(); }
            else if ( symbol == libGAP_S_GE ) { libGAP_IntrGe(); }
            else if ( symbol == libGAP_S_GT ) { libGAP_IntrGt(); }
            else if ( symbol == libGAP_S_LE ) { libGAP_IntrLe(); }
            else if ( symbol == libGAP_S_IN ) { libGAP_IntrIn(); }
        }
    }

    /* interpret the not                                                   */
    if ( (isNot % 2) != 0 ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrNot(); }
    }
}


/****************************************************************************
**
*F  ReadAnd( <follow>, <mode> ) . . . . . . . read a logical 'and' expression
**
**  'ReadAnd' reads an and   expression.  In case of  an  error it  skips all
**  symbols up to one contained in <follow>.
**
**  <And> := <Rel> { 'and' <Rel> }
*/
void libGAP_ReadAnd (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    /* <Rel>                                                               */
    libGAP_ReadRel( follow, mode );

    /* { 'and' <Rel> }                                                     */
    while ( libGAP_Symbol == libGAP_S_AND ) {
        libGAP_Match( libGAP_S_AND, "and", follow );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrAndL(); }
        libGAP_ReadRel( follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrAnd(); }
    }
}


/****************************************************************************
**
*F  ReadExpr( <follow>, <mode> )  . . . . . . . . . . . .  read an expression
**
**  'ReadExpr' reads an expression.  In case of an error it skips all symbols
**  up to one contained in <follow>.
**
**  <Expr> := <And> { 'or' <And> }
*/
void libGAP_ReadExpr (
    libGAP_TypSymbolSet        follow,
    libGAP_Char                mode )
{
    /* <And>                                                               */
    libGAP_ReadAnd( follow, mode );

    /* { 'or' <And> }                                                      */
    while ( libGAP_Symbol == libGAP_S_OR ) {
        libGAP_Match( libGAP_S_OR, "or", follow );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrOrL(); }
        libGAP_ReadAnd( follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrOr(); }
    }
}


/****************************************************************************
**
*F  ReadUnbind( <follow> )  . . . . . . . . . . . .  read an unbind statement
**
**  'ReadUnbind' reads an unbind statement.  In case of an error it skips all
**  symbols up to one contained in <follow>.
**
**  <Statment> := 'Unbind' '(' <Var> ')' ';'
*/
void libGAP_ReadUnbind (
    libGAP_TypSymbolSet        follow )
{
    libGAP_Match( libGAP_S_UNBIND, "Unbind", follow );
    libGAP_Match( libGAP_S_LPAREN, "(", follow );
    libGAP_ReadCallVarAss( libGAP_S_RPAREN|follow, 'u' );
    libGAP_Match( libGAP_S_RPAREN, ")", follow );
}


/****************************************************************************
**
*F  ReadEmpty( <follow> )  . . . . . . . . . . . . . .read an empty statement
**
**  'ReadEmpty' reads  an empty statement.  The argument is actually ignored
**
**  <Statment> :=  ';'
*/
void libGAP_ReadEmpty (
    libGAP_TypSymbolSet        follow )
{
  libGAP_IntrEmpty();
}

/****************************************************************************
**
*F  ReadInfo( <follow> )  . . . . . . . . . . . . . .  read an info statement
**
**  'ReadInfo' reads  an info statement.  In  case of an  error  it skips all
**  symbols up to one contained in <follow>.
**
**  <Statment> := 'Info' '(' <Expr> ',' <Expr> { ',' <Expr> } ')' ';'
*/
void libGAP_ReadInfo (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       narg;     /* numer of arguments to print (or not)  */

    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrInfoBegin(); }
    libGAP_Match( libGAP_S_INFO, "Info", follow );
    libGAP_Match( libGAP_S_LPAREN, "(", follow );
    libGAP_ReadExpr( libGAP_S_RPAREN | libGAP_S_COMMA | follow, 'r');
    libGAP_Match( libGAP_S_COMMA, ",", libGAP_S_RPAREN|follow);
    libGAP_ReadExpr( libGAP_S_RPAREN | libGAP_S_COMMA | follow, 'r');
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrInfoMiddle(); }
    narg = 0;
    while ( libGAP_Symbol == libGAP_S_COMMA ) {
        narg++;
        libGAP_Match( libGAP_S_COMMA, "", 0L);
        libGAP_ReadExpr( libGAP_S_RPAREN | libGAP_S_COMMA | follow, 'r');
    }
    libGAP_Match( libGAP_S_RPAREN, ")", follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrInfoEnd(narg); }
}


/****************************************************************************
**
*F  ReadAssert( <follow> )  . . . . . . . . . . . . .read an assert statement
**
**  'ReadAssert' reads an assert statement.  In case of an error it skips all
**  symbols up to one contained in <follow>.
**
**  <Statment> := 'Assert' '(' <Expr> ',' <Expr> [ ',' <Expr> ]  ')' ';'
*/
void libGAP_ReadAssert (
    libGAP_TypSymbolSet        follow )
{
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrAssertBegin(); }
    libGAP_Match( libGAP_S_ASSERT, "Assert", follow );
    libGAP_Match( libGAP_S_LPAREN, "(", follow );
    libGAP_ReadExpr( libGAP_S_RPAREN | libGAP_S_COMMA | follow, 'r' );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrAssertAfterLevel(); }
    libGAP_Match( libGAP_S_COMMA, ",", libGAP_S_RPAREN|follow );
    libGAP_ReadExpr( libGAP_S_RPAREN | libGAP_S_COMMA | follow, 'r' );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrAssertAfterCondition(); }
    if ( libGAP_Symbol == libGAP_S_COMMA )
      {
        libGAP_Match( libGAP_S_COMMA, "", 0L);
        libGAP_ReadExpr( libGAP_S_RPAREN |  follow, 'r' );
        libGAP_Match( libGAP_S_RPAREN, ")", follow );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrAssertEnd3Args(); }
      }
    else
      {
	libGAP_Match( libGAP_S_RPAREN, ")", follow );
	if ( ! libGAP_READ_ERROR() ) { libGAP_IntrAssertEnd2Args(); }
      }
}

/****************************************************************************
**
*F  ReadIf( <follow> )  . . . . . . . . . . . . . . . .  read an if statement
**
**  'ReadIf' reads an if-statement.  In case of an error it skips all symbols
**  up to one contained in <follow>.
**
**  <Statement> := 'if'   <Expr> 'then' <Statments>
**                 { 'elif' <Expr> 'then' <Statments> }
**                 [ 'else'               <Statments> ]
**                 'fi' ';'
*/
void libGAP_ReadIf (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       nrb;            /* number of branches              */
    volatile libGAP_UInt       nrs;            /* number of statements in a body  */

    /* 'if' <Expr>  'then' <Statments>                                     */
    nrb = 0;
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfBegin(); }
    libGAP_Match( libGAP_S_IF, "if", follow );
    libGAP_ReadExpr( libGAP_S_THEN|libGAP_S_ELIF|libGAP_S_ELSE|libGAP_S_FI|follow, 'r' );
    libGAP_Match( libGAP_S_THEN, "then", libGAP_STATBEGIN|libGAP_S_ELIF|libGAP_S_ELSE|libGAP_S_FI|follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfBeginBody(); }
    nrs = libGAP_ReadStats( libGAP_S_ELIF|libGAP_S_ELSE|libGAP_S_FI|follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfEndBody( nrs ); }
    nrb++;

    /* { 'elif' <Expr>  'then' <Statments> }                               */
    while ( libGAP_Symbol == libGAP_S_ELIF ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfElif(); }
        libGAP_Match( libGAP_S_ELIF, "elif", follow );
        libGAP_ReadExpr( libGAP_S_THEN|libGAP_S_ELIF|libGAP_S_ELSE|libGAP_S_FI|follow, 'r' );
        libGAP_Match( libGAP_S_THEN, "then", libGAP_STATBEGIN|libGAP_S_ELIF|libGAP_S_ELSE|libGAP_S_FI|follow );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfBeginBody(); }
        nrs = libGAP_ReadStats( libGAP_S_ELIF|libGAP_S_ELSE|libGAP_S_FI|follow );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfEndBody( nrs ); }
        nrb++;
    }

    /* [ 'else' <Statments> ]                                              */
    if ( libGAP_Symbol == libGAP_S_ELSE ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfElse(); }
        libGAP_Match( libGAP_S_ELSE, "else", follow );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfBeginBody(); }
        nrs = libGAP_ReadStats( libGAP_S_FI|follow );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfEndBody( nrs ); }
        nrb++;
    }

    /* 'fi'                                                                */
    libGAP_Match( libGAP_S_FI, "fi", follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrIfEnd( nrb ); }
}


/****************************************************************************
**
*F  ReadFor( <follow> ) . . . . . . . . . . . . . . . .  read a for statement
**
**  'ReadFor' reads a for-loop.  In case of an error it  skips all symbols up
**  to one contained in <follow>.
**
**  <Statement> := 'for' <Var>  'in' <Expr>  'do'
**                     <Statments>
**                 'od' ';'
*/


void libGAP_ReadFor (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       nrs;            /* number of statements in body    */
    volatile libGAP_UInt       nrError;        /* copy of <NrError>               */
    volatile libGAP_Bag        currLVars;      /* copy of <CurrLVars>             */

    /* remember the current variables in case of an error                  */
    currLVars = libGAP_CurrLVars;
    nrError   = libGAP_NrError;

    /* 'for'                                                               */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrForBegin(); }
    libGAP_Match( libGAP_S_FOR, "for", follow );

    /* <Var>                                                               */
    libGAP_ReadCallVarAss( follow, 'r' );

    /* 'in' <Expr>                                                         */
    libGAP_Match( libGAP_S_IN, "in", libGAP_S_DO|libGAP_S_OD|follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrForIn(); }
    libGAP_ReadExpr( libGAP_S_DO|libGAP_S_OD|follow, 'r' );

    /* 'do' <Statments>                                                    */
    libGAP_Match( libGAP_S_DO, "do", libGAP_STATBEGIN|libGAP_S_OD|follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrForBeginBody(); }
    nrs = libGAP_ReadStats( libGAP_S_OD|follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrForEndBody( nrs ); }

    /* 'od'                                                                */
    libGAP_Match( libGAP_S_OD, "od", follow );
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrForEnd();
    }

    /* an error has occured *after* the 'IntrForEndBody'                  */
    /* If we hadn't actually come out of coding the body, we need
       to recover. Otherwise it was probably an error in executing the body and
       we just return */
    else if ( nrError == 0  && libGAP_IntrCoding ) {
      libGAP_CodeEnd(1);
      libGAP_IntrCoding--;
      libGAP_CurrLVars = currLVars;
      libGAP_PtrLVars  = libGAP_PTR_BAG( libGAP_CurrLVars );
      libGAP_PtrBody   = (libGAP_Stat*) libGAP_PTR_BAG( libGAP_BODY_FUNC( libGAP_CURR_FUNC ) );
     }
}


/****************************************************************************
**
*F  ReadWhile( <follow> ) . . . . . . . . . . . . . .  read a while statement
**
**  'ReadWhile' reads a while-loop.  In case of an error it skips all symbols
**  up to one contained in <follow>.
**
**  <Statement> := 'while' <Expr>  'do'
**                     <Statments>
**                 'od' ';'
*/
void libGAP_ReadWhile (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       nrs;            /* number of statements in body    */
    volatile libGAP_UInt       nrError;        /* copy of <NrError>               */
    volatile libGAP_Bag        currLVars;      /* copy of <CurrLVars>             */

    /* remember the current variables in case of an error                  */
    currLVars = libGAP_CurrLVars;
    nrError   = libGAP_NrError;

    /* 'while' <Expr>  'do'                                                */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrWhileBegin(); }
    libGAP_Match( libGAP_S_WHILE, "while", follow );
    libGAP_ReadExpr( libGAP_S_DO|libGAP_S_OD|follow, 'r' );
    libGAP_Match( libGAP_S_DO, "do", libGAP_STATBEGIN|libGAP_S_DO|follow );

    /*     <Statments>                                                     */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrWhileBeginBody(); }
    nrs = libGAP_ReadStats( libGAP_S_OD|follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrWhileEndBody( nrs ); }

    /* 'od'                                                                */
    libGAP_Match( libGAP_S_OD, "od", follow );
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrWhileEnd();
    }

    /* an error has occured *after* the 'IntrWhileEndBody'                 */
    /* If we hadn't actually come out of coding the body, we need
       to recover. Otherwise it was probably an error in executing the body and
       we just return */
    else if ( nrError == 0 && libGAP_IntrCoding ) {
        libGAP_CodeEnd(1);
        libGAP_IntrCoding--;
        libGAP_CurrLVars = currLVars;
        libGAP_PtrLVars  = libGAP_PTR_BAG( libGAP_CurrLVars );
        libGAP_PtrBody   = (libGAP_Stat*) libGAP_PTR_BAG( libGAP_BODY_FUNC( libGAP_CURR_FUNC ) );
    }
}


/****************************************************************************
**
*F  ReadRepeat( <follow> )  . . . . . . . . . . . . . read a repeat statement
**
**  'ReadRepeat' reads a  repeat-loop.   In case  of an  error it skips   all
**  symbols up to one contained in <follow>.
**
** <Statement> := 'repeat'
**                    <Statments>
**                'until' <Expr> ';'
*/
void libGAP_ReadRepeat (
    libGAP_TypSymbolSet        follow )
{
    volatile libGAP_UInt       nrs;            /* number of statements in body    */
    volatile libGAP_UInt       nrError;        /* copy of <NrError>               */
    volatile libGAP_Bag        currLVars;      /* copy of <CurrLVars>             */

    /* remember the current variables in case of an error                  */
    currLVars = libGAP_CurrLVars;
    nrError   = libGAP_NrError;

    /* 'repeat'                                                            */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRepeatBegin(); }
    libGAP_Match( libGAP_S_REPEAT, "repeat", follow );

    /*  <Statments>                                                        */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRepeatBeginBody(); }
    nrs = libGAP_ReadStats( libGAP_S_UNTIL|follow );
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrRepeatEndBody( nrs ); }

    /* 'until' <Expr>                                                      */
    libGAP_Match( libGAP_S_UNTIL, "until", libGAP_EXPRBEGIN|follow );
    libGAP_ReadExpr( follow, 'r' );
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrRepeatEnd();
    }

    /* an error has occured *after* the 'IntrFuncExprEnd'                  */
    /* If we hadn't actually come out of coding the body, we need
       to recover. Otherwise it was probably an error in executing the body and
       we just return */
    else if ( nrError == 0 && libGAP_IntrCoding ) {
        libGAP_CodeEnd(1);
        libGAP_IntrCoding--;
        libGAP_CurrLVars = currLVars;
        libGAP_PtrLVars  = libGAP_PTR_BAG( libGAP_CurrLVars );
        libGAP_PtrBody   = (libGAP_Stat*) libGAP_PTR_BAG( libGAP_BODY_FUNC( libGAP_CURR_FUNC ) );
    }
}


/****************************************************************************
**
*F  ReadBreak(<follow>) . . . . . . . . . . . . . . .  read a break statement
**
**  'ReadBreak' reads a  break-statement.  In case  of an error  it skips all
**  symbols up to one contained in <follow>.
**
**  <Statement> := 'break' ';'
*/
void libGAP_ReadBreak (
    libGAP_TypSymbolSet        follow )
{
    /* skip the break symbol                                               */
    libGAP_Match( libGAP_S_BREAK, "break", follow );

    /* interpret the break statement                                       */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrBreak(); }
}

/****************************************************************************
**
*F  ReadContinue(<follow>) . . . . . . . . . . . . . . .  read a continue statement
**
**  'ReadContinue' reads a  continue-statement.  In case  of an error  it skips all
**  symbols up to one contained in <follow>.
**
**  <Statement> := 'continue' ';'
*/
void libGAP_ReadContinue (
    libGAP_TypSymbolSet        follow )
{
    /* skip the continue symbol                                               */
    libGAP_Match( libGAP_S_CONTINUE, "continue", follow );

    /* interpret the continue statement                                       */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrContinue(); }
}


/****************************************************************************
**
*F  ReadReturn( <follow> )  . . . . . . . . . . . . . read a return statement
**
**  'ReadReturn'   reads  a  return-statement.   Return  with   no expression
**  following is used  in functions to return void.   In case of an error  it
**  skips all symbols up to one contained in <follow>.
**
**  <Statement> := 'return' [ <Expr> ] ';'
**
**  It is still legal to use parenthesis but they  are  no  longer  required,
**  a return statememt is not a function call and should not look  like  one.
*/
void libGAP_ReadReturn (
    libGAP_TypSymbolSet        follow )
{
    /* skip the return symbol                                              */
    libGAP_Match( libGAP_S_RETURN, "return", follow );

    /* 'return' with no expression following                               */
    if ( libGAP_Symbol == libGAP_S_SEMICOLON ) {
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrReturnVoid(); }
    }

    /* 'return' with an expression following                               */
    else {
        libGAP_ReadExpr( follow, 'r' );
        if ( ! libGAP_READ_ERROR() ) { libGAP_IntrReturnObj(); }
    }
}


/****************************************************************************
**
*F  ReadTryNext(<follow>) . . . . . . . . .  read a try-next-method statement
**
**  'ReadTryNext' reads a try-next-method statement.  In case of an error  it
**  skips all symbols up to one contained in <follow>.
**
**  <Statment> := 'TryNextMethod' '(' ')' ';'
*/
void libGAP_ReadTryNext (
    libGAP_TypSymbolSet        follow )
{
    libGAP_Match( libGAP_S_TRYNEXT, "TryNextMethod", follow );
    libGAP_Match( libGAP_S_LPAREN, "(", follow );
    libGAP_Match( libGAP_S_RPAREN, ")", follow );
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrRefGVar( libGAP_GVarName( "TRY_NEXT_METHOD" ) );
        libGAP_IntrReturnObj();
    }
}


/****************************************************************************
**
*F  ReadQuit( <follow> )  . . . . . . . . . . . . . . . read a quit statement
**
**  'ReadQuit' reads a  quit  statement.  In case   of an error it skips  all
**  symbols up to one contained in <follow>.
**
**  <Statement> := 'quit' ';'
*/
void            libGAP_ReadQuit (
    libGAP_TypSymbolSet        follow )
{
    /* skip the quit symbol                                                */
    libGAP_Match( libGAP_S_QUIT, "quit", follow );

    /* interpret the quit                                                  */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrQuit(); }
}

/****************************************************************************
**
*F  ReadQUIT( <follow> )  . . . . . . . . . . . . . . . read a QUIT statement
**
**  'ReadQUIT' reads a  QUIT  statement.  In case   of an error it skips  all
**  symbols up to one contained in <follow>.
**
**  <Statement> := 'QUIT' ';'
*/
void            libGAP_ReadQUIT (
    libGAP_TypSymbolSet        follow )
{
    /* skip the quit symbol                                                */
    libGAP_Match( libGAP_S_QQUIT, "QUIT", follow );

    /* interpret the quit                                                  */
    if ( ! libGAP_READ_ERROR() ) { libGAP_IntrQUIT(); }
}


/****************************************************************************
**
*F  ReadStats(<follow>) . . . . . . . . . . . . . . read a statement sequence
**
**  'ReadStats' reads a statement sequence.  In case of an error it skips all
**  symbols up to one contained in <follow>.
**
**  <Statments> := { <Statment> }
**
**  <Statment>  := <Var> ':=' <Expr> ';'
**              |  <Var> '(' [ <Expr> { ',' <Expr> } ] ')' ';'
**              |  'Unbind' '(' <Var> ')' ';'
**              |  'if'   <Expr>  'then' <Statments>
**                 { 'elif' <Expr>  'then' <Statments> }
**                 [ 'else'                <Statments> ] 'fi' ';'
**              |  'for' <Var> 'in' <Expr> 'do' <Statments> 'od' ';'
**              |  'while' <Expr>  'do' <Statments>  'od' ';'
**              |  'repeat' <Statments>  'until' <Expr> ';'
**              |  'break' ';'
**              |  'return' [ <Expr> ] ';'
**              |  ';'
*/
libGAP_UInt libGAP_ReadStats (
    libGAP_TypSymbolSet        follow )
{
    libGAP_UInt               nr;            /* number of statements            */

    /* read the statements                                                 */
    nr = 0;
    while ( libGAP_IS_IN( libGAP_Symbol, libGAP_STATBEGIN|libGAP_S_SEMICOLON ) ) {

        /* read a statement                                                */
        if      ( libGAP_Symbol == libGAP_S_IDENT  ) libGAP_ReadCallVarAss(follow,'s');
        else if ( libGAP_Symbol == libGAP_S_UNBIND ) libGAP_ReadUnbind(    follow    );
        else if ( libGAP_Symbol == libGAP_S_INFO   ) libGAP_ReadInfo(      follow    );
        else if ( libGAP_Symbol == libGAP_S_ASSERT ) libGAP_ReadAssert(    follow    );
        else if ( libGAP_Symbol == libGAP_S_IF     ) libGAP_ReadIf(        follow    );
        else if ( libGAP_Symbol == libGAP_S_FOR    ) libGAP_ReadFor(       follow    );
        else if ( libGAP_Symbol == libGAP_S_WHILE  ) libGAP_ReadWhile(     follow    );
        else if ( libGAP_Symbol == libGAP_S_REPEAT ) libGAP_ReadRepeat(    follow    );
        else if ( libGAP_Symbol == libGAP_S_BREAK  ) libGAP_ReadBreak(     follow    );
        else if ( libGAP_Symbol == libGAP_S_CONTINUE) libGAP_ReadContinue(     follow    );
        else if ( libGAP_Symbol == libGAP_S_RETURN ) libGAP_ReadReturn(    follow    );
        else if ( libGAP_Symbol == libGAP_S_TRYNEXT) libGAP_ReadTryNext(   follow    );
	else if ( libGAP_Symbol == libGAP_S_QUIT   ) libGAP_ReadQuit(      follow    );
	else                           libGAP_ReadEmpty(     follow    );
	nr++;
        libGAP_Match( libGAP_S_SEMICOLON, ";", follow );

    }

    /* return the number of statements                                     */
    return nr;
}


/****************************************************************************
**

*F * * * * * * * * * * * * read and evaluate symbols  * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  ReadEvalResult  . . . . . . . . result of reading one command immediately
*/
libGAP_Obj libGAP_ReadEvalResult;


/****************************************************************************
**
*F  ReadEvalCommand() . . . . . . . . . . . . . . . . . . .  read one command
**
**  'ReadEvalCommand' reads one command and interprets it immediately.
**
**  It does not expect the  first symbol of its input  already read and  wont
**  read the  first symbol of the  next  input.
**
*/


void libGAP_RecreateStackNams( libGAP_Obj context )
{
  libGAP_Obj lvars = context;
  libGAP_Obj nams;
  libGAP_UInt i;
  while (lvars != libGAP_BottomLVars && lvars != (libGAP_Obj)0)
    {
      nams = libGAP_NAMS_FUNC(libGAP_PTR_BAG(lvars)[0]);
      if (nams != (libGAP_Obj) 0)
	{
	  libGAP_GROW_PLIST(libGAP_StackNams, ++libGAP_CountNams);
	  libGAP_SET_ELM_PLIST( libGAP_StackNams, libGAP_CountNams, nams);
	  libGAP_SET_LEN_PLIST( libGAP_StackNams, libGAP_CountNams);
	}
      lvars = libGAP_ENVI_FUNC(libGAP_PTR_BAG(lvars)[0]);
    }

  /* At this point we have the stack upside down, so invert it */
  for (i = 1; i <= libGAP_CountNams/2; i++)
    {
      nams = libGAP_ELM_PLIST(libGAP_StackNams, i);
      libGAP_SET_ELM_PLIST( libGAP_StackNams,
		     i,
		     libGAP_ELM_PLIST(libGAP_StackNams, libGAP_CountNams + 1 -i));
      libGAP_SET_ELM_PLIST( libGAP_StackNams,
		     libGAP_CountNams + 1 -i,
		     nams);
    }
}


libGAP_ExecStatus libGAP_ReadEvalCommand ( libGAP_Obj context )
{
    libGAP_ExecStatus          type;
    libGAP_Obj                 stackNams;
    libGAP_UInt                countNams;
    libGAP_UInt                readTop;
    libGAP_UInt                readTilde;
    libGAP_UInt                currLHSGVar;
    libGAP_Obj                 errorLVars;
    libGAP_Obj                 errorLVars0;
    libGAP_syJmp_buf           readJmpError;

    /* get the first symbol from the input                                 */
    libGAP_Match( libGAP_Symbol, "", 0UL );

    /* if we have hit <end-of-file>, then give up                          */
    if ( libGAP_Symbol == libGAP_S_EOF )  { return libGAP_STATUS_EOF; }

    /* print only a partial prompt from now on                             */
    if ( !libGAP_SyQuiet )
      libGAP_Prompt = "> ";
    else
      libGAP_Prompt = "";

    /* remember the old reader context                                     */
    stackNams   = libGAP_StackNams;
    countNams   = libGAP_CountNams;
    readTop     = libGAP_ReadTop;
    readTilde   = libGAP_ReadTilde;
    currLHSGVar = libGAP_CurrLHSGVar;
    memcpy( readJmpError, libGAP_ReadJmpError, sizeof(libGAP_syJmp_buf) );

    /* intialize everything and begin an interpreter                       */
    libGAP_StackNams   = libGAP_NEW_PLIST( libGAP_T_PLIST, 16 );
    libGAP_CountNams   = 0;
    libGAP_ReadTop     = 0;
    libGAP_ReadTilde   = 0;
    libGAP_CurrLHSGVar = 0;
    libGAP_RecreateStackNams(context);
    errorLVars = libGAP_ErrorLVars;
    errorLVars0 = libGAP_ErrorLVars0;
    libGAP_ErrorLVars = context;
    libGAP_ErrorLVars0 = libGAP_ErrorLVars;

    libGAP_IntrBegin( context );

    /* read an expression or an assignment or a procedure call             */
    if      ( libGAP_Symbol == libGAP_S_IDENT  ) { libGAP_ReadExpr(   libGAP_S_SEMICOLON|libGAP_S_EOF, 'x' ); }

    /* otherwise read a statement                                          */
    else if (libGAP_Symbol==libGAP_S_UNBIND    ) { libGAP_ReadUnbind( libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_INFO      ) { libGAP_ReadInfo(   libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_ASSERT    ) { libGAP_ReadAssert( libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_IF        ) { libGAP_ReadIf(     libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_FOR       ) { libGAP_ReadFor(    libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_WHILE     ) { libGAP_ReadWhile(  libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_REPEAT    ) { libGAP_ReadRepeat( libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_BREAK     ) { libGAP_ReadBreak(  libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_CONTINUE     ) { libGAP_ReadContinue(  libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_RETURN    ) { libGAP_ReadReturn( libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_TRYNEXT   ) { libGAP_ReadTryNext(libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_QUIT      ) { libGAP_ReadQuit(   libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_QQUIT     ) { libGAP_ReadQUIT(   libGAP_S_SEMICOLON|libGAP_S_EOF      ); }
    else if (libGAP_Symbol==libGAP_S_SEMICOLON ) { libGAP_ReadEmpty(  libGAP_S_SEMICOLON|libGAP_S_EOF      ); }

    /* otherwise try to read an expression                                 */
    /* Unless the statement is empty, in which case do nothing             */
    else                           { libGAP_ReadExpr(   libGAP_S_SEMICOLON|libGAP_S_EOF, 'r' ); }

    /* every statement must be terminated by a semicolon  \                 */
    if ( libGAP_Symbol != libGAP_S_SEMICOLON ) {
        libGAP_SyntaxError( "; expected");
    }

    /* check for dual semicolon                                            */
    if ( *libGAP_In == ';' ) {
        libGAP_GetSymbol();
        libGAP_DualSemicolon = 1;
    }
    else {
        libGAP_DualSemicolon = 0;
    }

    /* end the interpreter                                                 */
    if ( ! libGAP_READ_ERROR() ) {
        type = libGAP_IntrEnd( 0UL );
    }
    else {
        libGAP_IntrEnd( 1UL );
        type = libGAP_STATUS_ERROR;
    }

    /* switch back to the old reader context                               */
    memcpy( libGAP_ReadJmpError, readJmpError, sizeof(libGAP_syJmp_buf) );
    libGAP_StackNams   = stackNams;
    libGAP_CountNams   = countNams;
    libGAP_ReadTop     = readTop;
    libGAP_ReadTilde   = readTilde;
    libGAP_CurrLHSGVar = currLHSGVar;
    libGAP_ErrorLVars = errorLVars;
    libGAP_ErrorLVars0 = errorLVars0;

    /* copy the result (if any)                                            */
    libGAP_ReadEvalResult = libGAP_IntrResult;

    /* return whether a return-statement or a quit-statement were executed */
    return type;
}

/****************************************************************************
**
*F  ReadEvalFile()  . . . . . . . . . . . . . . . . . . . . . . . read a file
**
**  'ReadEvalFile' reads an entire file and returns (in 'ReadEvalResult') the
**  entire file as thunk, i.e., as function of no argument.
**
**  It does not expect the  first symbol of its input  already read and  wont
**  reads to the end of the input (unless an error happens).
*/
libGAP_UInt libGAP_ReadEvalFile ( void )
{
    volatile libGAP_ExecStatus type;
    volatile libGAP_Obj        stackNams;
    volatile libGAP_UInt       countNams;
    volatile libGAP_UInt       readTop;
    volatile libGAP_UInt       readTilde;
    volatile libGAP_UInt       currLHSGVar;
    libGAP_syJmp_buf             readJmpError;
    volatile libGAP_UInt       nr;
    volatile libGAP_Obj        name;
    volatile libGAP_Obj        nams;
    volatile libGAP_Int        nloc;
    volatile libGAP_Int        i;

    /* get the first symbol from the input                                 */
    libGAP_Match( libGAP_Symbol, "", 0UL );

    /* if we have hit <end-of-file>, then give up                          */
    if ( libGAP_Symbol == libGAP_S_EOF )  { return libGAP_STATUS_EOF; }

    /* print only a partial prompt from now on                             */
    if ( !libGAP_SyQuiet )
      libGAP_Prompt = "> ";
    else
      libGAP_Prompt = "";

    /* remember the old reader context                                     */
    stackNams   = libGAP_StackNams;
    countNams   = libGAP_CountNams;
    readTop     = libGAP_ReadTop;
    readTilde   = libGAP_ReadTilde;
    currLHSGVar = libGAP_CurrLHSGVar;
    memcpy( readJmpError, libGAP_ReadJmpError, sizeof(libGAP_syJmp_buf) );

    /* intialize everything and begin an interpreter                       */
    libGAP_StackNams   = libGAP_NEW_PLIST( libGAP_T_PLIST, 16 );
    libGAP_CountNams   = 0;
    libGAP_ReadTop     = 0;
    libGAP_ReadTilde   = 0;
    libGAP_CurrLHSGVar = 0;
    libGAP_IntrBegin(libGAP_BottomLVars);

    /* check for local variables                                           */
    nloc = 0;
    nams = libGAP_NEW_PLIST( libGAP_T_PLIST, nloc );
    libGAP_SET_LEN_PLIST( nams, nloc );
    libGAP_CountNams += 1;
    libGAP_ASS_LIST( libGAP_StackNams, libGAP_CountNams, nams );
    if ( libGAP_Symbol == libGAP_S_LOCAL ) {
        libGAP_Match( libGAP_S_LOCAL, "local", 0L );
        libGAP_C_NEW_STRING_DYN( name, libGAP_Value );
        nloc += 1;
        libGAP_ASS_LIST( nams, nloc, name );
        libGAP_Match( libGAP_S_IDENT, "identifier", libGAP_STATBEGIN|libGAP_S_END );
        while ( libGAP_Symbol == libGAP_S_COMMA ) {
            libGAP_Value[0] = '\0';
            libGAP_Match( libGAP_S_COMMA, ",", 0L );
            for ( i = 1; i <= nloc; i++ ) {
                if ( strcmp(libGAP_CSTR_STRING(libGAP_ELM_LIST(nams,i)),libGAP_Value) == 0 ) {
                    libGAP_SyntaxError("name used for two locals");
                }
            }
            libGAP_C_NEW_STRING_DYN( name, libGAP_Value );
            nloc += 1;
            libGAP_ASS_LIST( nams, nloc, name );
            libGAP_Match( libGAP_S_IDENT, "identifier", libGAP_STATBEGIN|libGAP_S_END );
        }
        libGAP_Match( libGAP_S_SEMICOLON, ";", libGAP_STATBEGIN|libGAP_S_END );
    }

    /* fake the 'function ()'                                              */
    libGAP_IntrFuncExprBegin( 0L, nloc, nams, libGAP_Input->number );

    /* read the statements                                                 */
    nr = libGAP_ReadStats( libGAP_S_SEMICOLON | libGAP_S_EOF );

    /* we now want to be at <end-of-file>                                  */
    if ( libGAP_Symbol != libGAP_S_EOF ) {
        libGAP_SyntaxError("<end-of-file> expected");
    }

    /* fake the 'end;'                                                     */
    if ( ! libGAP_READ_ERROR() ) {
        libGAP_IntrFuncExprEnd( nr, 0UL );
    }
    else {
        libGAP_Obj fexp;
        libGAP_CodeEnd(1);
        libGAP_IntrCoding--;
        fexp = libGAP_CURR_FUNC;
        if (fexp && libGAP_ENVI_FUNC(fexp))  libGAP_SWITCH_TO_OLD_LVARS(libGAP_ENVI_FUNC(fexp));
    }

    /* end the interpreter                                                 */
    if ( ! libGAP_READ_ERROR() ) {
        type = libGAP_IntrEnd( 0UL );
    }
    else {
        libGAP_IntrEnd( 1UL );
        type = libGAP_STATUS_ERROR;
    }

    /* switch back to the old reader context                               */
    memcpy( libGAP_ReadJmpError, readJmpError, sizeof(libGAP_syJmp_buf) );
    libGAP_StackNams   = stackNams;
    libGAP_CountNams   = countNams;
    libGAP_ReadTop     = readTop;
    libGAP_ReadTilde   = readTilde;
    libGAP_CurrLHSGVar = currLHSGVar;

    /* copy the result (if any)                                            */
    libGAP_ReadEvalResult = libGAP_IntrResult;

    /* return whether a return-statement or a quit-statement were executed */
    return type;
}


/****************************************************************************
**
*F  ReadEvalError() . . . . . . . . . . . . . . . . . .  return with an error
*/
void            libGAP_ReadEvalError ( void )
{
    libGAP_PtrBody  = (libGAP_Stat*)libGAP_PTR_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC));
    libGAP_PtrLVars = libGAP_PTR_BAG(libGAP_CurrLVars);
    libGAP_syLongjmp( libGAP_ReadJmpError, 1 );
}

/****************************************************************************
**
**   Reader state -- the next group of functions are used to "push" the current
**  interpreter state allowing GAP code to be interpreted in the middle of other
**  code. This is used, for instance, in the command-line editor.
*/


struct libGAP_SavedReaderState {
  libGAP_Obj                 stackNams;
  libGAP_UInt                countNams;
  libGAP_UInt                readTop;
  libGAP_UInt                readTilde;
  libGAP_UInt                currLHSGVar;
  libGAP_UInt                userHasQuit;
  libGAP_syJmp_buf             readJmpError;
  libGAP_UInt                intrCoding;
  libGAP_UInt                intrIgnoring;
  libGAP_UInt                intrReturning;
  libGAP_UInt                nrError;
};

static void libGAP_SaveReaderState( struct libGAP_SavedReaderState *s) {
  s->stackNams   = libGAP_StackNams;
  s->countNams   = libGAP_CountNams;
  s->readTop     = libGAP_ReadTop;
  s->readTilde   = libGAP_ReadTilde;
  s->currLHSGVar = libGAP_CurrLHSGVar;
  s->userHasQuit = libGAP_UserHasQuit;
  s->intrCoding = libGAP_IntrCoding;
  s->intrIgnoring = libGAP_IntrIgnoring;
  s->intrReturning = libGAP_IntrReturning;
  s->nrError = libGAP_NrError;
  memcpy( s->readJmpError, libGAP_ReadJmpError, sizeof(libGAP_syJmp_buf) );
}

static void libGAP_ClearReaderState( void ) {
  libGAP_StackNams   = libGAP_NEW_PLIST( libGAP_T_PLIST, 16 );
  libGAP_CountNams   = 0;
  libGAP_ReadTop     = 0;
  libGAP_ReadTilde   = 0;
  libGAP_CurrLHSGVar = 0;
  libGAP_UserHasQuit = 0;
  libGAP_IntrCoding = 0;
  libGAP_IntrIgnoring = 0;
  libGAP_IntrReturning = 0;
  libGAP_NrError = 0;
}

static void libGAP_RestoreReaderState( const struct libGAP_SavedReaderState *s) {
  memcpy( libGAP_ReadJmpError, s->readJmpError, sizeof(libGAP_syJmp_buf) );
  libGAP_UserHasQuit = s->userHasQuit;
  libGAP_StackNams   = s->stackNams;
  libGAP_CountNams   = s->countNams;
  libGAP_ReadTop     = s->readTop;
  libGAP_ReadTilde   = s->readTilde;
  libGAP_CurrLHSGVar = s->currLHSGVar;
  libGAP_IntrCoding = s->intrCoding;
  libGAP_IntrIgnoring = s->intrIgnoring;
  libGAP_IntrReturning = s->intrReturning;
  libGAP_NrError = s->nrError;
}


/****************************************************************************
**
*F  Call0ArgsInNewReader(Obj f)  . . . . . . . . . . . . call a GAP function
**
**  The current reader context is saved and a new one is started.
*/
libGAP_Obj libGAP_Call0ArgsInNewReader(libGAP_Obj f)

{
  /* for the new interpreter context: */
/*  ExecStatus          type; */
  struct libGAP_SavedReaderState s;
  libGAP_Obj result;

  /* remember the old reader context                                     */
  libGAP_SaveReaderState(&s);

  /* intialize everything and begin an interpreter                       */
  libGAP_ClearReaderState();
  libGAP_IntrBegin( libGAP_BottomLVars );

  if (!libGAP_READ_ERROR()) {
    result = libGAP_CALL_0ARGS(f);
    libGAP_PushVoidObj();
    /* end the interpreter                                                 */
    libGAP_IntrEnd( 0UL );
  } else {
    result = (libGAP_Obj) 0L;
    libGAP_IntrEnd( 1UL );
    libGAP_ClearError();
  }

  /* switch back to the old reader context                               */
  libGAP_RestoreReaderState(&s);
  return result;
}

/****************************************************************************
**
*F  Call1ArgsInNewReader(Obj f,Obj a) . . . . . . . . . . call a GAP function
**
**  The current reader context is saved and a new one is started.
*/
libGAP_Obj libGAP_Call1ArgsInNewReader(libGAP_Obj f,libGAP_Obj a)

{
  /* for the new interpreter context: */
/*ExecStatus          type; */
  struct libGAP_SavedReaderState s;
  libGAP_Obj result;

  /* remember the old reader context                                     */

  libGAP_SaveReaderState(&s);

  /* intialize everything and begin an interpreter                       */
  libGAP_ClearReaderState();
  libGAP_IntrBegin( libGAP_BottomLVars );

  if (!libGAP_READ_ERROR()) {
    result = libGAP_CALL_1ARGS(f,a);
    libGAP_PushVoidObj();
    /* end the interpreter                                                 */
    libGAP_IntrEnd( 0UL );
  } else {
    result = (libGAP_Obj) 0L;
    libGAP_IntrEnd( 1UL );
    libGAP_ClearError();
  }

  /* switch back to the old reader context                               */
  libGAP_RestoreReaderState(&s);
  return result;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_ErrorLVars = (libGAP_UInt **)0;
  libGAP_CurrentGlobalForLoopDepth = 0;
    libGAP_InitGlobalBag( &libGAP_ReadEvalResult, "src/read.c:ReadEvalResult" );
    libGAP_InitGlobalBag( &libGAP_StackNams,      "src/read.c:StackNams"      );
    libGAP_InitCopyGVar( "GAPInfo", &libGAP_GAPInfo);
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoRead()  . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "read",                             /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    0,                                  /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoRead ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  read.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/



