/****************************************************************************
**
*W  opers.c                     GAP source                       Frank Celler
*W                                                         & Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the  filters, operations, attributes,
**  and properties package.
*/
#include        <assert.h>
#include        "system.h"              /* Ints, UInts                     */




#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "calls.h"               /* generic call mechanism          */

#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "plist.h"               /* plain lists                     */
#include        "blister.h"             /* boolean lists                   */
#include        "string.h"              /* strings                         */
#include        "range.h"               /* ranges                          */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "saveload.h"            /* saving and loading              */

#include        "listfunc.h"   
#include        "integer.h"   

/****************************************************************************
**
*V  TRY_NEXT_METHOD . . . . . . . . . . . . . . . . . `TRY_NEXT_MESSAGE' flag
*/
libGAP_Obj libGAP_TRY_NEXT_METHOD;


#define libGAP_CACHE_SIZE 5


/****************************************************************************
**

*F * * * * * * * * * * * * internal flags functions * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  PrintFlags( <flags> ) . . . . . . . . . . . . . . . .  print a flags list
*/
void libGAP_PrintFlags (
    libGAP_Obj                 flags )
{
    libGAP_Pr( "<flag list>", 0L, 0L );
}


/****************************************************************************
**
*F  TypeFlags( <flags> )  . . . . . . . . . . . . . . .  type of a flags list
*/
libGAP_Obj libGAP_TYPE_FLAGS;

libGAP_Obj libGAP_TypeFlags (
    libGAP_Obj                 flags )
{
    return libGAP_TYPE_FLAGS;
}


/****************************************************************************
**
*F  SaveFlags( <flags> )  . . . . . . . . . . . . . . . . . save a flags list
**
*/
void libGAP_SaveFlags (
    libGAP_Obj         flags )
{
    libGAP_UInt        i, len, *ptr;

    libGAP_SaveSubObj(libGAP_TRUES_FLAGS(flags));
    libGAP_SaveSubObj(libGAP_HASH_FLAGS(flags));
    libGAP_SaveSubObj(libGAP_ADDR_OBJ(flags)[2]); /* length, as an object */
    libGAP_SaveSubObj(libGAP_AND_CACHE_FLAGS(flags));

    len = libGAP_NRB_FLAGS(flags);
    ptr = libGAP_BLOCKS_FLAGS(flags);
    for ( i = 1;  i <= len;  i++ )
        libGAP_SaveUInt(*ptr++);
    return;
}


/****************************************************************************
**
*F  LoadFlags( <flags> )  . . . . . . . . . . . . . . . . . load a flags list
**
*/
void libGAP_LoadFlags(
    libGAP_Obj         flags )
{
    libGAP_Obj         sub;
    libGAP_UInt        i, len, *ptr;

    sub = libGAP_LoadSubObj();  libGAP_SET_TRUES_FLAGS( flags, sub );
    sub = libGAP_LoadSubObj();  libGAP_SET_HASH_FLAGS( flags, sub );
    libGAP_ADDR_OBJ(flags)[2] = libGAP_LoadSubObj(); /* length, as an object */
    sub = libGAP_LoadSubObj();  libGAP_SET_AND_CACHE_FLAGS( flags, sub );
    
    len = libGAP_NRB_FLAGS(flags);
    ptr = libGAP_BLOCKS_FLAGS(flags);
    for ( i = 1;  i <= len;  i++ )
        *ptr++ = libGAP_LoadUInt();
    return;
}


/****************************************************************************
**

*F * * * * * * * * * * * * *  GAP flags functions * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  FuncLEN_FLAGS( <self>, <flags> )  . . . . . . . .  length of a flags list
**
*/
libGAP_Obj libGAP_FuncLEN_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags )
{
    /* do some trivial checks                                              */
    while ( libGAP_TNUM_OBJ(flags) != libGAP_T_FLAGS ) {
        flags = libGAP_ErrorReturnObj( "<flags> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags), 0L,
            "you can replace <flags> via 'return <flags>;'" );
    }

    return libGAP_INTOBJ_INT( libGAP_LEN_FLAGS(flags) );
}


/****************************************************************************
**
*F  FuncELM_FLAGS( <self>, <flags>, <pos> ) . . . . . element of a flags list
*/
libGAP_Obj libGAP_FuncELM_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags,
    libGAP_Obj                 pos )
{
    /* do some trivial checks                                              */
    while ( libGAP_TNUM_OBJ(flags) != libGAP_T_FLAGS ) {
        flags = libGAP_ErrorReturnObj( "<flags> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags), 0L,
            "you can replace <flags> via 'return <flags>;'" );
    }

    /* select and return the element                                       */
    return libGAP_ELM_FLAGS( flags, libGAP_INT_INTOBJ(pos) );
}


/****************************************************************************
**
*F  FuncHASH_FLAGS( <self>, <flags> ) . . . . . .  hash value of a flags list
**
**  The hash value is independent of the size of a machine word (32 or 64).
**
**  The rather peculiar cast in the definition of HASH_FLAGS_SIZE is needed
**  to get the calculation to work right on the alpha.
**
*T  The 64 bit version depends on the byte order -- it assumes that
**  the lower addressed half-word is the less significant
**
*/
#define libGAP_HASH_FLAGS_SIZE (libGAP_Int4)67108879L

libGAP_Obj libGAP_FuncHASH_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags )
{
    libGAP_Int4                 hash;
    libGAP_Int4                 x;
    libGAP_Int                  len;
    libGAP_UInt4 *              ptr;
    libGAP_Int                  i;

    /* do some trivial checks                                              */
    while ( libGAP_TNUM_OBJ(flags) != libGAP_T_FLAGS ) {
            flags = libGAP_ErrorReturnObj( "<flags> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags), 0L,
            "you can replace <flags> via 'return <flags>;'" );
    }
    if ( libGAP_HASH_FLAGS(flags) != 0 ) {
        return libGAP_HASH_FLAGS(flags);
    }

    /* do the real work*/
#ifndef libGAP_SYS_IS_64_BIT

    /* 32 bit case  -- this is the "defining" case, others are
     adjusted to comply with this */
    len = libGAP_NRB_FLAGS(flags);
    ptr = (libGAP_UInt4 *)libGAP_BLOCKS_FLAGS(flags);
    hash = 0;
    x    = 1;
    for ( i = len; i >= 1; i-- ) {
        hash = (hash + (*ptr % libGAP_HASH_FLAGS_SIZE) * x) % libGAP_HASH_FLAGS_SIZE;
        x    = ((8*sizeof(libGAP_UInt4)-1) * x) % libGAP_HASH_FLAGS_SIZE;
        ptr++;
    }
#else
#ifdef WORDS_BIGENDIAN

    /* This is the hardest case */
    len = libGAP_NRB_FLAGS(flags);
    ptr = (libGAP_UInt4 *)libGAP_BLOCKS_FLAGS(flags);
    hash = 0;
    x    = 1;
    for ( i = len; i >= 1; i-- ) {

        /* least significant 32 bits first */
        hash = (hash + (ptr[1] % libGAP_HASH_FLAGS_SIZE) * x) % libGAP_HASH_FLAGS_SIZE;
        x    = ((8*sizeof(libGAP_UInt4)-1) * x) % libGAP_HASH_FLAGS_SIZE;
        /* now the more significant */
        hash = (hash + (*ptr % libGAP_HASH_FLAGS_SIZE) * x) % libGAP_HASH_FLAGS_SIZE;
        x    = ((8*sizeof(libGAP_UInt4)-1) * x) % libGAP_HASH_FLAGS_SIZE;
        
        ptr+= 2;
    }
#else

    /* and the middle case -- for DEC alpha, the 32 bit chunks are
       in the right order, and we merely have to be sure to process them as
       32 bit chunks */
    len = libGAP_NRB_FLAGS(flags)*(sizeof(libGAP_UInt)/sizeof(libGAP_UInt4));
    ptr = (libGAP_UInt4 *)libGAP_BLOCKS_FLAGS(flags);
    hash = 0;
    x    = 1;
    for ( i = len; i >= 1; i-- ) {
        hash = (hash + (*ptr % libGAP_HASH_FLAGS_SIZE) * x) % libGAP_HASH_FLAGS_SIZE;
        x    = ((8*sizeof(libGAP_UInt4)-1) * x) % libGAP_HASH_FLAGS_SIZE;
        ptr++;
    }
#endif
#endif
    libGAP_SET_HASH_FLAGS( flags, libGAP_INTOBJ_INT((libGAP_UInt)hash+1) );
    libGAP_CHANGED_BAG(flags);
    return libGAP_HASH_FLAGS(flags);
}


/****************************************************************************
**
*F  FuncTRUES_FLAGS( <self>, <flags> )  . . .  true positions of a flags list
**
**  see 'FuncPositionsTruesBlist' in "blister.c" for information.
*/
libGAP_Obj libGAP_FuncTRUES_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags )
{
    libGAP_Obj                 sub;            /* handle of the result            */
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_UInt *              ptr;            /* pointer to flags                */
    libGAP_UInt                nrb;            /* number of blocks in flags       */
    libGAP_UInt                m;              /* number of bits in a block       */
    libGAP_UInt                n;              /* number of bits in flags         */
    libGAP_UInt                nn;
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the first argument                                    */
    while ( libGAP_TNUM_OBJ(flags) != libGAP_T_FLAGS ) {
        flags = libGAP_ErrorReturnObj( "<flags> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags), 0L,
            "you can replace <flags> via 'return <flags>;'" );
    }
    if ( libGAP_TRUES_FLAGS(flags) != 0 ) {
        return libGAP_TRUES_FLAGS(flags);
    }

    /* compute the number of 'true'-s just as in 'FuncSizeBlist'            */
    nrb = libGAP_NRB_FLAGS(flags);
    ptr = (libGAP_UInt*)libGAP_BLOCKS_FLAGS(flags);
    n = 0;
    for ( i = 1; i <= nrb; i++ ) {
        m = *ptr++;
        libGAP_COUNT_TRUES_BLOCK(m); 
        n += m;
    }

    /* make the sublist (we now know its size exactly)                    */
    sub = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, n );
    libGAP_SET_LEN_PLIST( sub, n );

    /* loop over the boolean list and stuff elements into <sub>            */
    len = libGAP_LEN_FLAGS( flags );
    nn  = 1;
    for ( i = 1; nn <= n && i <= len;  i++ ) {
        if ( libGAP_ELM_FLAGS( flags, i ) == libGAP_True ) {
            libGAP_SET_ELM_PLIST( sub, nn, libGAP_INTOBJ_INT(i) );
            nn++;
        }
    }
    libGAP_CHANGED_BAG(sub);

    /* return the sublist                                                  */
    libGAP_SET_TRUES_FLAGS( flags, sub );
    libGAP_CHANGED_BAG(flags);
    return sub;
}


/****************************************************************************
**
*F  FuncSIZE_FLAGS( <self>, <flags> ) . . . . number of trues of a flags list
**
**  see 'FuncSIZE_FLAGS'
*/
libGAP_Obj libGAP_FuncSIZE_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags )
{
    libGAP_UInt *              ptr;            /* pointer to flags                */
    libGAP_UInt                nrb;            /* number of blocks in flags       */
    libGAP_UInt                m;              /* number of bits in a block       */
    libGAP_UInt                n;              /* number of bits in flags         */
    libGAP_UInt                i;              /* loop variable                   */

    /* get and check the first argument                                    */
    while ( libGAP_TNUM_OBJ(flags) != libGAP_T_FLAGS ) {
        flags = libGAP_ErrorReturnObj( "<flags> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags), 0L,
            "you can replace <flags> via 'return <flags>;'" );
    }
    if ( libGAP_TRUES_FLAGS(flags) != 0 ) {
        return libGAP_INTOBJ_INT( libGAP_LEN_PLIST( libGAP_TRUES_FLAGS(flags) ) );
    }

    /* get the number of blocks and a pointer                              */
    nrb = libGAP_NRB_FLAGS(flags);
    ptr = libGAP_BLOCKS_FLAGS(flags);

    /* loop over the blocks, adding the number of bits of each one         */
    n = 0;
    for ( i = 1; i <= nrb; i++ ) {
        m = *ptr++;
        libGAP_COUNT_TRUES_BLOCK(m);
        n += m;
    }

    /* return the number of bits                                           */
    return libGAP_INTOBJ_INT( n );
}


/****************************************************************************
**

*F  FuncIS_EQUAL_FLAGS( <self>, <flags1>, <flags2> )  equality of flags lists
*/
libGAP_Obj libGAP_FuncIS_EQUAL_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags1,
    libGAP_Obj                 flags2 )
{
    libGAP_Int                 len1;
    libGAP_Int                 len2;
    libGAP_UInt  *             ptr1;
    libGAP_UInt  *             ptr2;
    libGAP_Int                 i;

    /* do some trivial checks                                              */
    while ( libGAP_TNUM_OBJ(flags1) != libGAP_T_FLAGS ) {
        flags1 = libGAP_ErrorReturnObj( "<flags1> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags1), 0L,
            "you can replace <flags1> via 'return <flags1>;'" );
    }
    while ( libGAP_TNUM_OBJ(flags2) != libGAP_T_FLAGS ) {
        flags2 = libGAP_ErrorReturnObj( "<flags2> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags2), 0L,
            "you can replace <flags2> via 'return <flags2>;'" );
    }
    if ( flags1 == flags2 ) {
        return libGAP_True;
    }

    /* do the real work                                                    */
    len1 = libGAP_NRB_FLAGS(flags1);
    len2 = libGAP_NRB_FLAGS(flags2);
    ptr1 = libGAP_BLOCKS_FLAGS(flags1);
    ptr2 = libGAP_BLOCKS_FLAGS(flags2);
    if ( len1 <= len2 ) {
        for ( i = 1; i <= len1; i++ ) {
            if ( *ptr1 != *ptr2 )
                return libGAP_False;
            ptr1++;  ptr2++;
        }
        for ( ; i <= len2; i++ ) {
            if ( 0 != *ptr2 )
                return libGAP_False;
            ptr2++;
        }
    }
    else {
        for ( i = 1; i <= len2; i++ ) {
            if ( *ptr1 != *ptr2 )
                return libGAP_False;
            ptr1++;  ptr2++;
        }
        for ( ; i <= len1; i++ ) {
            if ( *ptr1 != 0 )
                return libGAP_False;
            ptr1++;
        }
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncIS_SUBSET_FLAGS( <self>, <flags1>, <flags2> ) . . . . . . subset test
*/
libGAP_Int libGAP_IsSubsetFlagsCalls;
libGAP_Int libGAP_IsSubsetFlagsCalls1;
libGAP_Int libGAP_IsSubsetFlagsCalls2;

libGAP_Obj libGAP_FuncIS_SUBSET_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags1,
    libGAP_Obj                 flags2 )
{
    libGAP_Int                 len1;
    libGAP_Int                 len2;
    libGAP_UInt *              ptr1;
    libGAP_UInt *              ptr2;
    libGAP_Int                 i;
    libGAP_Obj                 trues;

    /* do some trivial checks                                              */
    while ( libGAP_TNUM_OBJ(flags1) != libGAP_T_FLAGS ) {
        flags1 = libGAP_ErrorReturnObj( "<flags1> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags1), 0L,
            "you can replace <flags1> via 'return <flags1>;'" );
    }
    while ( libGAP_TNUM_OBJ(flags2) != libGAP_T_FLAGS ) {
        flags2 = libGAP_ErrorReturnObj( "<flags2> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags2), 0L,
            "you can replace <flags2> via 'return <flags2>;'" );
    }
    if ( flags1 == flags2 ) {
        return libGAP_True;
    }

    /* do the real work                                                    */
#ifdef COUNT_OPERS
    libGAP_IsSubsetFlagsCalls++;
#endif

    /* first check the trues                                               */
    trues = libGAP_TRUES_FLAGS(flags2);
    if ( trues != 0 ) {
        len2 = libGAP_LEN_PLIST(trues);
        if ( libGAP_TRUES_FLAGS(flags1) != 0 ) {
            if ( libGAP_LEN_PLIST(libGAP_TRUES_FLAGS(flags1)) < len2 ) {
#ifdef COUNT_OPERS
                libGAP_IsSubsetFlagsCalls1++;
#endif
                return libGAP_False;
            }
        }
        if ( len2 < 3 ) {
#ifdef COUNT_OPERS
            libGAP_IsSubsetFlagsCalls2++;
#endif
            if ( libGAP_LEN_FLAGS(flags1) < libGAP_INT_INTOBJ(libGAP_ELM_PLIST(trues,len2)) ) {
                return libGAP_False;
            }
            for ( i = len2;  0 < i;  i-- ) {
               if (libGAP_ELM_FLAGS(flags1,libGAP_INT_INTOBJ(libGAP_ELM_PLIST(trues,i)))==libGAP_False) {
                   return libGAP_False;
               }
            }
            return libGAP_True;
        }
    }

    /* compare the bit lists                                               */
    len1 = libGAP_NRB_FLAGS(flags1);
    len2 = libGAP_NRB_FLAGS(flags2);
    ptr1 = libGAP_BLOCKS_FLAGS(flags1);
    ptr2 = libGAP_BLOCKS_FLAGS(flags2);
    if ( len1 <= len2 ) {
        for ( i = 1; i <= len1; i++ ) {
            if ( (*ptr1 & *ptr2) != *ptr2 ) {
                return libGAP_False;
            }
            ptr1++;  ptr2++;
        }
        for ( ; i <= len2; i++ ) {
            if ( 0 != *ptr2 ) {
                return libGAP_False;
            }
            ptr2++;
        }
    }
    else {
        for ( i = 1; i <= len2; i++ ) {
            if ( (*ptr1 & *ptr2) != *ptr2 ) {
                return libGAP_False;
            }
            ptr1++;  ptr2++;
        }
    }
    return libGAP_True;
}


/****************************************************************************
**

*F  FuncSUB_FLAGS( <self>, <flags1>, <flags2> ) . . .  substract a flags list
*/
libGAP_Obj libGAP_FuncSUB_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags1,
    libGAP_Obj                 flags2 )
{
    libGAP_Obj                 flags;
    libGAP_Int                 len1;
    libGAP_Int                 len2;
    libGAP_Int                 size1;
    libGAP_Int                 size2;
    libGAP_UInt *              ptr;
    libGAP_UInt *              ptr1;
    libGAP_UInt *              ptr2;
    libGAP_Int                 i;

    /* do some trivial checks                                              */
    while ( libGAP_TNUM_OBJ(flags1) != libGAP_T_FLAGS ) {
        flags1 = libGAP_ErrorReturnObj( "<flags1> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags1), 0L,
            "you can replace <flags1> via 'return <flags1>;'" );
    }
    while ( libGAP_TNUM_OBJ(flags2) != libGAP_T_FLAGS ) {
        flags2 = libGAP_ErrorReturnObj( "<flags2> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags2), 0L,
            "you can replace <flags2> via 'return <flags2>;'" );
    }

    /* do the real work                                                    */
    len1   = libGAP_LEN_FLAGS(flags1);
    size1  = libGAP_NRB_FLAGS(flags1);
    len2   = libGAP_LEN_FLAGS(flags2);
    size2  = libGAP_NRB_FLAGS(flags2);
    if ( len1 < len2 ) {
        libGAP_NEW_FLAGS( flags, len1 );
        libGAP_SET_LEN_FLAGS( flags, len1 );
        ptr1 = libGAP_BLOCKS_FLAGS(flags1);
        ptr2 = libGAP_BLOCKS_FLAGS(flags2);
        ptr  = libGAP_BLOCKS_FLAGS(flags);
        for ( i = 1; i <= size1; i++ )
            *ptr++ = *ptr1++ & ~ *ptr2++;
    }
    else {
        libGAP_NEW_FLAGS( flags, len1 );
        libGAP_SET_LEN_FLAGS( flags, len1 );
        ptr1 = libGAP_BLOCKS_FLAGS(flags1);
        ptr2 = libGAP_BLOCKS_FLAGS(flags2);
        ptr  = libGAP_BLOCKS_FLAGS(flags);
        for ( i = 1; i <= size2; i++ )
            *ptr++ = *ptr1++ & ~ *ptr2++;
        for (      ; i <= size1; i++ )
            *ptr++ = *ptr1++;
    }        

    return flags;
}


/****************************************************************************
**
*F  FuncAND_FLAGS( <self>, <flags1>, <flags2> ) . . . .  `and' of flags lists
*/
#define libGAP_AND_FLAGS_HASH_SIZE             50

libGAP_Int libGAP_AndFlagsCacheHit;
libGAP_Int libGAP_AndFlagsCacheMiss;
libGAP_Int libGAP_AndFlagsCacheLost;

libGAP_Obj libGAP_FuncAND_FLAGS (
    libGAP_Obj                 self,
    libGAP_Obj                 flags1,
    libGAP_Obj                 flags2 )
{
    libGAP_Obj                 flags;
    libGAP_Int                 len1;
    libGAP_Int                 len2;
    libGAP_Int                 size1;
    libGAP_Int                 size2;
    libGAP_UInt *              ptr;
    libGAP_UInt *              ptr1;
    libGAP_UInt *              ptr2;
    libGAP_Int                 i;

#ifdef libGAP_AND_FLAGS_HASH_SIZE
    libGAP_Obj                 flagsX;
    libGAP_Obj                 cache;
    libGAP_Obj                 entry;
    libGAP_UInt                hash;
    libGAP_UInt                hash2;
    static libGAP_UInt         next = 0;
#endif

    /* do some trivial checks                                              */
    while ( libGAP_TNUM_OBJ(flags1) != libGAP_T_FLAGS ) {
        flags1 = libGAP_ErrorReturnObj( "<flags1> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags1), 0L,
            "you can replace <flags1> via 'return <flags1>;'" );
    }
    while ( libGAP_TNUM_OBJ(flags2) != libGAP_T_FLAGS ) {
        flags2 = libGAP_ErrorReturnObj( "<flags2> must be a flags list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(flags2), 0L,
            "you can replace <flags2> via 'return <flags2>;'" );
    }

    /* check the cache                                                     */
#   ifdef libGAP_AND_FLAGS_HASH_SIZE
        if ( libGAP_INT_INTOBJ(flags1) < libGAP_INT_INTOBJ(flags2) ) {
            flagsX = flags2;
            cache  = libGAP_AND_CACHE_FLAGS(flags1);
            if ( cache == 0 ) {
                cache = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*libGAP_AND_FLAGS_HASH_SIZE );
                libGAP_SET_AND_CACHE_FLAGS( flags1, cache );
                libGAP_CHANGED_BAG(flags1);
            }
        }
        else {
            flagsX = flags1;
            cache  = libGAP_AND_CACHE_FLAGS(flags2);
            if ( cache == 0 ) {
                cache = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*libGAP_AND_FLAGS_HASH_SIZE );
                libGAP_SET_AND_CACHE_FLAGS( flags2, cache );
                libGAP_CHANGED_BAG(flags2);
            }
        }
        hash = (libGAP_UInt)libGAP_INT_INTOBJ(flagsX);
        for ( i = 0;  i < 24;  i++ ) {
            hash2 = (hash + 97*i) % libGAP_AND_FLAGS_HASH_SIZE;
            entry = libGAP_ELM_PLIST( cache, 2*hash2+1 );
            if ( entry == 0 ) {
                break;
            }
            if ( entry == flagsX ) {
#               ifdef COUNT_OPERS
                    libGAP_AndFlagsCacheHit++;
#               endif
                return libGAP_ELM_PLIST( cache, 2*hash2+2 );
            }
        }
        if ( entry == 0 ) {
            hash = hash2;
        }
        else {
            next = (next+1) % 24;
            hash = (hash + 97*next) % libGAP_AND_FLAGS_HASH_SIZE;
        }
#       ifdef COUNT_OPERS
            libGAP_AndFlagsCacheMiss++;
#       endif
#   endif


    /* do the real work                                                    */
    len1   = libGAP_LEN_FLAGS(flags1);
    size1  = libGAP_NRB_FLAGS(flags1);
    len2   = libGAP_LEN_FLAGS(flags2);
    size2  = libGAP_NRB_FLAGS(flags2);
    if ( len1 == 0 ) {
        return flags2;
    }
    if ( len2 == 0 ) {
        return flags1;
    }
    if ( len1 < len2 ) {
        libGAP_NEW_FLAGS( flags, len2 );
        libGAP_SET_LEN_FLAGS( flags, len2 );
        ptr1 = libGAP_BLOCKS_FLAGS(flags1);
        ptr2 = libGAP_BLOCKS_FLAGS(flags2);
        ptr  = libGAP_BLOCKS_FLAGS(flags);
        for ( i = 1; i <= size1; i++ )
            *ptr++ = *ptr1++ | *ptr2++;
        for (      ; i <= size2; i++ )
            *ptr++ =           *ptr2++;
    }
    else {
        libGAP_NEW_FLAGS( flags, len1 );
        libGAP_SET_LEN_FLAGS( flags, len1 );
        ptr1 = libGAP_BLOCKS_FLAGS(flags1);
        ptr2 = libGAP_BLOCKS_FLAGS(flags2);
        ptr  = libGAP_BLOCKS_FLAGS(flags);
        for ( i = 1; i <= size2; i++ )
            *ptr++ = *ptr1++ | *ptr2++;
        for (      ; i <= size1; i++ )
            *ptr++ = *ptr1++;
    }        

    /* store result in the cache                                           */
#   ifdef libGAP_AND_FLAGS_HASH_SIZE
#       ifdef COUNT_OPERS
            if ( libGAP_ELM_PLIST(cache,2*hash+1) != 0 ) {
                    libGAP_AndFlagsCacheLost++;
            }
#       endif
        libGAP_SET_ELM_PLIST( cache, 2*hash+1, flagsX );
        libGAP_SET_ELM_PLIST( cache, 2*hash+2, flags  );
        libGAP_CHANGED_BAG(cache);
#   endif

    /* and return the result                                               */
    return flags;
}


/****************************************************************************
**

*F * * * * * * * * * * *  internal filter functions * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  Countlags  . . . . . . . . . . . . . . . . . . . . next free flag number
*/
libGAP_Int libGAP_CountFlags;


/****************************************************************************
**

*F  SetterFilter( <oper> )  . . . . . . . . . . . . . . .  setter of a filter
*/
libGAP_Obj libGAP_SetterFilter (
    libGAP_Obj                 oper )
{
    libGAP_Obj                 setter;

    setter = libGAP_SETTR_FILT( oper );
    if ( setter == libGAP_INTOBJ_INT(0xBADBABE) )
        setter = libGAP_SetterAndFilter( oper );
    return setter;
}


/****************************************************************************
**
*F  SetterAndFilter( <getter> )  . . . . . .  setter of a concatenated filter
*/
libGAP_Obj libGAP_DoSetAndFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 val )
{
    libGAP_Obj                 op;

    while (val != libGAP_True)
      val = libGAP_ErrorReturnObj(
             "You cannot set an \"and-filter\" except to true", 0L, 0L,
             "you can type 'return true;' to set all components true\n"
             "(but you might really want to reset just one component)");
    
    /* call the first 'and'-ed function                                    */
    op = libGAP_FLAG1_FILT( self );
    libGAP_CALL_2ARGS( op, obj, val );
    
    /* call the second 'and'-ed function                                   */
    op = libGAP_FLAG2_FILT( self );
    libGAP_CALL_2ARGS( op, obj, val );
    
    /* return 'void'                                                       */
    return 0;
}


libGAP_Obj libGAP_SetterAndFilter (
    libGAP_Obj                 getter )
{
    libGAP_Obj                 setter;

    if ( libGAP_SETTR_FILT( getter ) == libGAP_INTOBJ_INT(0xBADBABE) ) {
        setter = libGAP_NewFunctionCT( libGAP_T_FUNCTION, libGAP_SIZE_OPER,
                                "<<setter-and-filter>>", 2L, "obj, val",
                                libGAP_DoSetAndFilter );
        libGAP_FLAG1_FILT(setter)  = libGAP_SetterFilter( libGAP_FLAG1_FILT(getter) );
        libGAP_FLAG2_FILT(setter)  = libGAP_SetterFilter( libGAP_FLAG2_FILT(getter) );
        libGAP_SETTR_FILT(getter)  = setter;
        libGAP_CHANGED_BAG(getter);
    }

    return libGAP_SETTR_FILT(getter);
}
        

/****************************************************************************
**
*F  TesterFilter( <oper> )  . . . . . . . . . . . . . . .  tester of a filter
*/
libGAP_Obj libGAP_TesterFilter (
    libGAP_Obj                 oper )
{
    libGAP_Obj                 tester;

    tester = libGAP_TESTR_FILT( oper );
    if ( tester == libGAP_INTOBJ_INT(0xBADBABE) )
        tester = libGAP_TesterAndFilter( oper );
    return tester;
}


/****************************************************************************
**
*F  TestAndFilter( <getter> )  . . . . . . . .tester of a concatenated filter
*/
libGAP_Obj libGAP_DoTestAndFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Obj                 op;
    
    /* call the first 'and'-ed function                                    */
    op = libGAP_FLAG1_FILT( self );
    val = libGAP_CALL_1ARGS( op, obj );
    if ( val != libGAP_True )  return libGAP_False;
    
    /* call the second 'and'-ed function                                   */
    op = libGAP_FLAG2_FILT( self );
    val = libGAP_CALL_1ARGS( op, obj );
    if ( val != libGAP_True )  return libGAP_False;
    
    /* return 'true'                                                       */
    return libGAP_True;
}


libGAP_Obj libGAP_TesterAndFilter (
    libGAP_Obj                 getter )
{
    libGAP_Obj                 tester;

    if ( libGAP_TESTR_FILT( getter ) == libGAP_INTOBJ_INT(0xBADBABE) ) {
        tester = libGAP_NewAndFilter( libGAP_TesterFilter( libGAP_FLAG1_FILT(getter) ),
                               libGAP_TesterFilter( libGAP_FLAG2_FILT(getter) ) );
        libGAP_TESTR_FILT(getter) = tester;
        libGAP_CHANGED_BAG(getter);

    }
    return libGAP_TESTR_FILT(getter);
}


/****************************************************************************
**
*F  NewFilter( <name>, <narg>, <nams>, <hdlr> )  . . . . .  make a new filter
*/
libGAP_Obj libGAP_DoTestFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_True;
}

libGAP_Obj libGAP_NewTesterFilter (
    libGAP_Obj                 getter )
{
    libGAP_Obj                 tester;
    tester = libGAP_ReturnTrueFilter;
    return tester;
}


libGAP_Obj libGAP_DoSetFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 val )
{
    libGAP_Int                 flag1;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;
    
    /* get the flag for the getter                                         */
    flag1 = libGAP_INT_INTOBJ( libGAP_FLAG1_FILT( self ) );
    
    /* get the kind of the object and its flags                            */
    kind  = libGAP_TYPE_OBJ( obj );
    flags = libGAP_FLAGS_TYPE( kind );
    
    /* return the value of the feature                                     */
    if ( flag1 <= libGAP_LEN_FLAGS( flags ) ) {
        if ( val != libGAP_ELM_FLAGS( flags, flag1 ) ) {
            libGAP_ErrorReturnVoid(
                "value feature is already set the other way",
                0L, 0L,
                "you can 'return;' and ignore it" );
        }
    }
    else {
        if ( val != libGAP_False ) {
            libGAP_ErrorReturnVoid(
                "value feature is already set the other way",
                0L, 0L,
                "you can 'return;' and ignore it" );
        }
    }

    /* return 'void'                                                       */
    return 0;
}

static libGAP_Obj libGAP_StringFilterSetter;
static libGAP_Obj libGAP_ArglistObjVal;

libGAP_Obj libGAP_NewSetterFilter (
    libGAP_Obj                 getter )
{
    libGAP_Obj                 setter;

    setter = libGAP_NewOperation( libGAP_StringFilterSetter, 2, libGAP_ArglistObjVal,
                           libGAP_DoSetFilter );
    libGAP_FLAG1_FILT(setter)  = libGAP_FLAG1_FILT(getter);
    libGAP_FLAG2_FILT(setter)  = libGAP_INTOBJ_INT( 0 );
    libGAP_CHANGED_BAG(setter);

    return setter;
}


libGAP_Obj libGAP_DoFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Int                 flag1;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;
    
    /* get the flag for the getter                                         */
    flag1 = libGAP_INT_INTOBJ( libGAP_FLAG1_FILT( self ) );
    
    /* get the kind of the object and its flags                            */
    kind  = libGAP_TYPE_OBJ( obj );
    flags = libGAP_FLAGS_TYPE( kind );
    
    /* return the value of the feature                                     */
    if ( flag1 <= libGAP_LEN_FLAGS( flags ) ) {
        val = libGAP_ELM_FLAGS( flags, flag1 );
    }
    else {
        val = libGAP_False;
    }
    
    /* return the value                                                    */
    return val;
}


libGAP_Obj libGAP_NewFilter (
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams,
    libGAP_ObjFunc             hdlr )
{
    libGAP_Obj                 getter;
    libGAP_Obj                 setter;
    libGAP_Obj                 tester;
    libGAP_Int                 flag1;
    libGAP_Obj                 flags;
    
    flag1 = ++libGAP_CountFlags;

    getter = libGAP_NewOperation( name, 1L, nams, (hdlr ? hdlr : libGAP_DoFilter) );
    libGAP_FLAG1_FILT(getter)  = libGAP_INTOBJ_INT( flag1 );
    libGAP_FLAG2_FILT(getter)  = libGAP_INTOBJ_INT( 0 );
    libGAP_NEW_FLAGS( flags, flag1 );
    libGAP_SET_LEN_FLAGS( flags, flag1 );
    libGAP_SET_ELM_FLAGS( flags, flag1, libGAP_True );
    libGAP_FLAGS_FILT(getter)  = flags;
    libGAP_CHANGED_BAG(getter);

    setter = libGAP_NewSetterFilter( getter );
    libGAP_SETTR_FILT(getter)  = setter;
    libGAP_CHANGED_BAG(getter);
    
    tester = libGAP_NewTesterFilter( getter );
    libGAP_TESTR_FILT(getter)  = tester;
    libGAP_CHANGED_BAG(getter);

    return getter;    
}


/****************************************************************************
**
*F  NewAndFilter( <filt1>, <filt2> ) . . . . . make a new concatenated filter
*/
libGAP_Obj libGAP_DoAndFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Obj                 op;
    
    /* call the first 'and'-ed function                                    */
    op = libGAP_FLAG1_FILT( self );
    val = libGAP_CALL_1ARGS( op, obj );
    if ( val != libGAP_True )  return libGAP_False;
    
    /* call the second 'and'-ed function                                   */
    op = libGAP_FLAG2_FILT( self );
    val = libGAP_CALL_1ARGS( op, obj );
    if ( val != libGAP_True )  return libGAP_False;
    
    /* return 'true'                                                       */
    return libGAP_True;
}

static libGAP_Obj libGAP_StringAndFilter;
static libGAP_Obj libGAP_ArglistObj;

libGAP_Obj libGAP_NewAndFilter (
    libGAP_Obj                 oper1,
    libGAP_Obj                 oper2 )
{
    libGAP_Obj                 getter;
    libGAP_Obj                 flags;

    if ( oper1 == libGAP_ReturnTrueFilter && oper2 == libGAP_ReturnTrueFilter )
        return libGAP_ReturnTrueFilter;

    getter = libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_OPER, libGAP_StringAndFilter, 1,
                           libGAP_ArglistObj, libGAP_DoAndFilter );
    libGAP_FLAG1_FILT(getter)  = oper1;
    libGAP_FLAG2_FILT(getter)  = oper2;
    flags = libGAP_FuncAND_FLAGS( 0, libGAP_FLAGS_FILT(oper1), libGAP_FLAGS_FILT(oper2) );
    libGAP_FLAGS_FILT(getter)  = flags;
    libGAP_SETTR_FILT(getter)  = libGAP_INTOBJ_INT(0xBADBABE);
    libGAP_TESTR_FILT(getter)  = libGAP_INTOBJ_INT(0xBADBABE);
    libGAP_CHANGED_BAG(getter);

    return getter;
}

libGAP_Obj libGAP_FuncIS_AND_FILTER( libGAP_Obj self, libGAP_Obj filt )
{
  return (libGAP_IS_FUNC(filt) && libGAP_HDLR_FUNC(filt, 1) == libGAP_DoAndFilter) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**

*V  ReturnTrueFilter . . . . . . . . . . . . . . . . the return 'true' filter
*/
libGAP_Obj libGAP_ReturnTrueFilter;


/****************************************************************************
**
*F  NewReturnTrueFilter() . . . . . . . . . . create a new return true filter
*/
libGAP_Obj libGAP_DoTestReturnTrueFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_True;
}

libGAP_Obj libGAP_TesterReturnTrueFilter (
    libGAP_Obj                 getter )
{
    return getter;
}

libGAP_Obj libGAP_DoSetReturnTrueFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 val )
{
    if ( val != libGAP_True ) {
         libGAP_ErrorReturnVoid( "you cannot set this flag to 'false'",
             0L, 0L,
             "you can 'return;' and ignore it" );
    }
    return 0;
}

libGAP_Obj libGAP_SetterReturnTrueFilter (
    libGAP_Obj                 getter )
{
    libGAP_Obj                 setter;

    setter = libGAP_NewFunctionCT( libGAP_T_FUNCTION, libGAP_SIZE_OPER,
        "<<setter-true-filter>>", 2L, "obj, val",
        libGAP_DoSetReturnTrueFilter );
    libGAP_FLAG1_FILT(setter)  = libGAP_INTOBJ_INT( 0 );
    libGAP_FLAG2_FILT(setter)  = libGAP_INTOBJ_INT( 0 );
    libGAP_CHANGED_BAG(setter);

    return setter;    
}

libGAP_Obj libGAP_DoReturnTrueFilter (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_True;
}

libGAP_Obj libGAP_NewReturnTrueFilter ( void )
{
    libGAP_Obj                 getter;
    libGAP_Obj                 setter;
    libGAP_Obj                 tester;
    libGAP_Obj                 flags;

    getter = libGAP_NewFunctionCT( libGAP_T_FUNCTION, libGAP_SIZE_OPER,
        "ReturnTrueFilter", 1L, "obj",
        libGAP_DoReturnTrueFilter );
    libGAP_FLAG1_FILT(getter)  = libGAP_INTOBJ_INT( 0 );
    libGAP_FLAG2_FILT(getter)  = libGAP_INTOBJ_INT( 0 );
    libGAP_NEW_FLAGS( flags, 0 );
    libGAP_SET_LEN_FLAGS( flags, 0 );
    libGAP_FLAGS_FILT(getter)  = flags;
    libGAP_CHANGED_BAG(getter);

    setter = libGAP_SetterReturnTrueFilter( getter );
    libGAP_SETTR_FILT(getter)  = setter;
    libGAP_CHANGED_BAG(getter);

    tester = libGAP_TesterReturnTrueFilter( getter );
    libGAP_TESTR_FILT(getter)  = tester;
    libGAP_CHANGED_BAG(getter);
        
    return getter;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * GAP filter functions * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  FuncNEW_FILTER( <self>, <name> )  . . . . . . . . . . . . .  new filter
*/
libGAP_Obj libGAP_FuncNEW_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    if ( ! libGAP_IsStringConv(name) ) {
        libGAP_ErrorQuit("usage: NewFilter( <name> )",0L,0L);
        return 0;
    }

    /* make the new operation                                              */
    return libGAP_NewFilter( name, 1L, (libGAP_Obj)0, (libGAP_ObjFunc)0 );
}


/****************************************************************************
**

*F  FuncFLAG1_FILTER( <self>, <oper> )  . . . . . . . . . . . .  `FLAG1_FILT'
*/
libGAP_Obj libGAP_FuncFLAG1_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    libGAP_Obj                 flag1;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    flag1 = libGAP_FLAG1_FILT( oper );
    if ( flag1 == 0 )
        flag1 = libGAP_INTOBJ_INT(0);
    return flag1;
}


/****************************************************************************
**
*F  FuncSET_FLAG1_FILTER( <self>, <oper>, <flag1> ) . . . .  set `FLAG1_FILT'
*/
libGAP_Obj libGAP_FuncSET_FLAG1_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 flag1 )
{
    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    libGAP_FLAG1_FILT( oper ) = flag1;
    return 0;
}


/****************************************************************************
**
*F  FuncFLAG2_FILTER( <self>, <oper> )  . . . . . . . . . . . .  `FLAG2_FILT'
*/
libGAP_Obj libGAP_FuncFLAG2_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    libGAP_Obj                 flag2;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    flag2 = libGAP_FLAG2_FILT( oper );
    if ( flag2 == 0 )
        flag2 = libGAP_INTOBJ_INT(0);
    return flag2;
}


/****************************************************************************
**
*F  FuncSET_FLAG2_FILTER( <self>, <oper>, <flag2> ) . . . .  set `FLAG2_FILT'
*/
libGAP_Obj libGAP_FuncSET_FLAG2_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 flag2 )
{
    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    libGAP_FLAG2_FILT( oper ) = flag2;
    return 0;
}


/****************************************************************************
**
*F  FuncFLAGS_FILTER( <self>, <oper> )  . . . . . . . . . . . .  `FLAGS_FILT'
*/
libGAP_Obj libGAP_FuncFLAGS_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    libGAP_Obj                 flags;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    flags = libGAP_FLAGS_FILT( oper );
    if ( flags == 0 )
        flags = libGAP_False;
    return flags;
}


/****************************************************************************
**
*F  FuncSET_FLAGS_FILTER( <self>, <oper>, <flags> ) . . . .  set `FLAGS_FILT'
*/
libGAP_Obj libGAP_FuncSET_FLAGS_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 flags )
{
    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    libGAP_FLAGS_FILT( oper ) = flags;
    return 0;
}


/****************************************************************************
**

*F  FuncSETTER_FILTER( <self>, <oper> ) . . . . . . . . .  setter of a filter
*/
libGAP_Obj libGAP_FuncSETTER_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    libGAP_Obj                 setter;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    setter = libGAP_SetterFilter( oper );
    if ( setter == 0 )  setter = libGAP_False;
    return setter;
}


/****************************************************************************
**
*F  FuncSET_SETTER_FILTER( <self>, <oper>, <setter> )  set setter of a filter
*/
libGAP_Obj libGAP_FuncSET_SETTER_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 setter )
{
    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    libGAP_SETTR_FILT( oper ) = setter;
    return 0;
}


/****************************************************************************
**
*F  FuncTESTER_FILTER( <self>, <oper> ) . . . . . . . . .  tester of a filter
*/
libGAP_Obj libGAP_FuncTESTER_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    libGAP_Obj                 tester;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    tester = libGAP_TesterFilter( oper );
    if ( tester == 0 )  tester = libGAP_False;
    return tester;
}


/****************************************************************************
**
*F  FuncSET_TESTER_FILTER( <self>, <oper>, <tester> )  set tester of a filter
*/
libGAP_Obj libGAP_FuncSET_TESTER_FILTER (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 tester )
{
    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    if ( libGAP_SIZE_OBJ(oper) != libGAP_SIZE_OPER ) {
        libGAP_ResizeBag( oper, libGAP_SIZE_OPER );
    }
    libGAP_TESTR_FILT( oper ) = tester;
    return 0;
}


/****************************************************************************
**

*F * * * * * * * * * *  internal operation functions  * * * * * * * * * * * *
*/


/****************************************************************************
**
*F  CallHandleMethodNotFound( <oper>, <nargs>, <args>, <verbose>, <constructor>)
**
*/

static libGAP_UInt libGAP_RNamOperation;
static libGAP_UInt libGAP_RNamArguments;
static libGAP_UInt libGAP_RNamIsVerbose;
static libGAP_UInt libGAP_RNamIsConstructor;
static libGAP_UInt libGAP_RNamPrecedence;
static libGAP_Obj libGAP_HandleMethodNotFound;

libGAP_Obj libGAP_CallHandleMethodNotFound( libGAP_Obj oper,
                              libGAP_Int nargs,
                              libGAP_Obj *args,
                              libGAP_UInt verbose,
                              libGAP_UInt constructor,
                              libGAP_Obj precedence)
{
  libGAP_Obj r;
  libGAP_Obj arglist;
  libGAP_UInt i;
  r = libGAP_NEW_PREC(5);
  if (libGAP_RNamOperation == 0)
    {
      /* we can't do this in initialization because opers
         is initialized BEFORE records */
      libGAP_RNamIsConstructor = libGAP_RNamName("isConstructor");
      libGAP_RNamIsVerbose = libGAP_RNamName("isVerbose");
      libGAP_RNamOperation = libGAP_RNamName("Operation");
      libGAP_RNamArguments = libGAP_RNamName("Arguments");
      libGAP_RNamPrecedence = libGAP_RNamName("Precedence");
    }
  libGAP_AssPRec(r,libGAP_RNamOperation,oper);
  arglist = libGAP_NEW_PLIST(nargs ? libGAP_T_PLIST_DENSE+libGAP_IMMUTABLE:
                      libGAP_T_PLIST_EMPTY+libGAP_IMMUTABLE, nargs);
  libGAP_SET_LEN_PLIST(arglist,nargs);
  for (i = 0; i < nargs; i++)
    libGAP_SET_ELM_PLIST( arglist, i+1, args[i]);
  libGAP_AssPRec(r,libGAP_RNamArguments,arglist);
  libGAP_AssPRec(r,libGAP_RNamIsVerbose,verbose ? libGAP_True : libGAP_False);
  libGAP_AssPRec(r,libGAP_RNamIsConstructor,constructor ? libGAP_True : libGAP_False);
  libGAP_AssPRec(r,libGAP_RNamPrecedence,precedence);
  libGAP_SortPRecRNam(r,0);
  return libGAP_CALL_1ARGS(libGAP_HandleMethodNotFound, r);
}

/****************************************************************************
**
*F  FuncCompactTypeIDs( <self> ) . . . garbage collect the type IDs
**
*/

static libGAP_Int libGAP_NextTypeID;
libGAP_Obj libGAP_IsType;

static void libGAP_FixTypeIDs( libGAP_Bag b ) {
  if ( (libGAP_TNUM_OBJ( b )  == libGAP_T_POSOBJ) &&
       (libGAP_DoFilter(libGAP_IsType, b ) == libGAP_True ))
    {
      libGAP_ID_TYPE(b) = libGAP_INTOBJ_INT(libGAP_NextTypeID);
      libGAP_NextTypeID++;
    } 
}


libGAP_Obj libGAP_FuncCompactTypeIDs( libGAP_Obj self )
{
  libGAP_NextTypeID = -(1L << libGAP_NR_SMALL_INT_BITS);
  libGAP_CallbackForAllBags( libGAP_FixTypeIDs );
  return libGAP_INTOBJ_INT(libGAP_NextTypeID);
}

/****************************************************************************
**
*F  DoOperation( <name> ) . . . . . . . . . . . . . . .  make a new operation
*/
libGAP_UInt libGAP_CacheIndex;

libGAP_Obj libGAP_Method0Args;
libGAP_Obj libGAP_NextMethod0Args;
libGAP_Obj libGAP_Method1Args;
libGAP_Obj libGAP_NextMethod1Args;
libGAP_Obj libGAP_Method2Args;
libGAP_Obj libGAP_NextMethod2Args;
libGAP_Obj libGAP_Method3Args;
libGAP_Obj libGAP_NextMethod3Args;
libGAP_Obj libGAP_Method4Args;
libGAP_Obj libGAP_NextMethod4Args;
libGAP_Obj libGAP_Method5Args;
libGAP_Obj libGAP_NextMethod5Args;
libGAP_Obj libGAP_Method6Args;
libGAP_Obj libGAP_NextMethod6Args;
libGAP_Obj libGAP_MethodXArgs;
libGAP_Obj libGAP_NextMethodXArgs;

libGAP_Obj libGAP_VMethod0Args;
libGAP_Obj libGAP_NextVMethod0Args;
libGAP_Obj libGAP_VMethod1Args;
libGAP_Obj libGAP_NextVMethod1Args;
libGAP_Obj libGAP_VMethod2Args;
libGAP_Obj libGAP_NextVMethod2Args;
libGAP_Obj libGAP_VMethod3Args;
libGAP_Obj libGAP_NextVMethod3Args;
libGAP_Obj libGAP_VMethod4Args;
libGAP_Obj libGAP_NextVMethod4Args;
libGAP_Obj libGAP_VMethod5Args;
libGAP_Obj libGAP_NextVMethod5Args;
libGAP_Obj libGAP_VMethod6Args;
libGAP_Obj libGAP_NextVMethod6Args;
libGAP_Obj libGAP_VMethodXArgs;
libGAP_Obj libGAP_NextVMethodXArgs;




/****************************************************************************
**
**  DoOperation0Args( <oper> )
*/
libGAP_Int libGAP_OperationHit;
libGAP_Int libGAP_OperationMiss;
libGAP_Int libGAP_OperationNext;


/* This avoids a function call in the case of external objects with a
   stored type */

static inline libGAP_Obj libGAP_TYPE_OBJ_FEO (
                libGAP_Obj obj
        )
{
  if ( libGAP_TNUM_OBJ(obj) >= libGAP_FIRST_EXTERNAL_TNUM &&
       libGAP_TNUM_OBJ(obj) <= libGAP_T_DATOBJ) /* avoid T_WPOBJ */
    return libGAP_TYPE_ANYOBJ(obj);
  else
    return libGAP_TYPE_OBJ(obj);
}

static inline libGAP_Obj libGAP_CacheOper (
    libGAP_Obj                 oper,
    libGAP_UInt                i )
{
    libGAP_Obj                 cache;
    libGAP_UInt len;
    cache = libGAP_CACHE_OPER( oper, i );
    if ( cache == 0 ) {
        len = (i < 7 ? libGAP_CACHE_SIZE * (i+2) : libGAP_CACHE_SIZE * (1+2));
        cache = libGAP_NEW_PLIST( libGAP_T_PLIST, len);
        libGAP_SET_LEN_PLIST( cache, len ); 
        libGAP_CACHE_OPER( oper, i ) = cache;
        libGAP_CHANGED_BAG( oper );
    }
    return cache;
}

libGAP_Obj libGAP_DoOperation0Args (
    libGAP_Obj                 oper )
{
    libGAP_Obj                 res;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                prec;

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 0 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 2*libGAP_CACHE_SIZE; i+= 2) {
            if (  cache[i] != 0  && cache[i+1] == prec) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_1ARGS( libGAP_Method0Args, oper );
          else
            method = libGAP_CALL_2ARGS( libGAP_NextMethod0Args, oper, prec );

          /* If there was no method found, then pass the information needed for
             the error reporting. This function rarely returns */
          
          while (method == libGAP_Fail)
            method = libGAP_CallHandleMethodNotFound( oper, 0, (libGAP_Obj *) 0, 0, 0, prec);
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 0 ) );
              cache[2*libGAP_CacheIndex] = method;
              cache[2*libGAP_CacheIndex+1] = prec;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,0));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_0ARGS( method );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoOperation1Args( <oper>, <a1> )
*/
libGAP_Obj libGAP_DoOperation1Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 id1;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );  id1 = libGAP_ID_TYPE( kind1 );

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 1 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 3*libGAP_CACHE_SIZE; i+= 3) {
            if (  cache[i+1] == prec && cache[i+2] == id1 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_2ARGS( libGAP_Method1Args, oper, kind1 );
          else
            method = libGAP_CALL_3ARGS( libGAP_NextMethod1Args, oper, prec, kind1 );
          
          /* If there was no method found, then pass the information needed for
             the error reporting. This function rarely returns */
          if (method == libGAP_Fail)
          {
            libGAP_Obj args[1];
            args[0] = arg1;
            while (method == libGAP_Fail)
              method = libGAP_CallHandleMethodNotFound( oper, 1, (libGAP_Obj *) args, 0, 0, prec);
          }
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 1 ) );
              cache[3*libGAP_CacheIndex] = method;
              cache[3*libGAP_CacheIndex+1] = prec;
              cache[3*libGAP_CacheIndex+2] = id1;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,1));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_1ARGS( method, arg1 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoOperation2Args( <oper>, <a1>, <a2> )
*/
libGAP_Obj libGAP_DoOperation2Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 id1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );  id1 = libGAP_ID_TYPE( kind1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 2 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 4*libGAP_CACHE_SIZE; i+= 4) {
            if (  cache[i+1] == prec && cache[i+2] == id1
                  && cache[i+3] == id2 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_3ARGS( libGAP_Method2Args, oper, kind1, kind2 );
          else
            method = libGAP_CALL_4ARGS( libGAP_NextMethod2Args, oper, prec, kind1, kind2 );
          
          /* If there was no method found, then pass the information needed for
             the error reporting. This function rarely returns */
          if (method == libGAP_Fail)
          {
            libGAP_Obj args[2];
            args[0] = arg1;
            args[1] = arg2;
            while (method == libGAP_Fail)
              method = libGAP_CallHandleMethodNotFound( oper, 2, (libGAP_Obj *) args, 0, 0, prec);
          }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 2 ) );
              cache[4*libGAP_CacheIndex] = method;
              cache[4*libGAP_CacheIndex+1] = prec;
              cache[4*libGAP_CacheIndex+2] = id1;
              cache[4*libGAP_CacheIndex+3] = id2;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,2));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_2ARGS( method, arg1, arg2 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}



/****************************************************************************
**
**  DoOperation3Args( <oper>, <a1>, <a2>, <a3> )
*/
libGAP_Obj libGAP_DoOperation3Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 id1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );  id1 = libGAP_ID_TYPE( kind1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 3 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 5*libGAP_CACHE_SIZE; i+= 5) {
            if (  cache[i+1] == prec && cache[i+2] == id1
                  && cache[i+3] == id2 && cache[i+4] == id3 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_4ARGS( libGAP_Method3Args, oper, kind1, kind2, kind3 );
          else
            method = libGAP_CALL_5ARGS( libGAP_NextMethod3Args, oper, prec, kind1, kind2, kind3 );
          /* If there was no method found, then pass the information needed for
             the error reporting. This function rarely returns */
          if (method == libGAP_Fail)
          {
            libGAP_Obj args[3];
            args[0] = arg1;
            args[1] = arg2;
            args[2] = arg3;
            while (method == libGAP_Fail)
              method = libGAP_CallHandleMethodNotFound( oper, 3, (libGAP_Obj *) args, 0, 0, prec);
          }

          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 3 ) );
              cache[5*libGAP_CacheIndex] = method;
              cache[5*libGAP_CacheIndex+1] = prec;
              cache[5*libGAP_CacheIndex+2] = id1;
              cache[5*libGAP_CacheIndex+3] = id2;
              cache[5*libGAP_CacheIndex+4] = id3;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,3));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_3ARGS( method, arg1, arg2, arg3 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoOperation4Args( <oper>, <a1>, <a2>, <a3>, <a4> )
*/
libGAP_Obj libGAP_DoOperation4Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 id1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 id4;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;


    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );  id1 = libGAP_ID_TYPE( kind1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );  id4 = libGAP_ID_TYPE( kind4 );

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 4 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 6*libGAP_CACHE_SIZE; i+= 6) {
            if (  cache[i+1] == prec && cache[i+2] == id1 &&
                  cache[i+3] == id2 && cache[i+4] == id3 &&
                  cache[i+5] == id4 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_5ARGS( libGAP_Method4Args, oper, kind1, kind2, kind3, kind4 );
          else
            method = libGAP_CALL_6ARGS( libGAP_NextMethod4Args, oper, prec, kind1, kind2, kind3, kind4 );
          
          /* If there was no method found, then pass the information needed for
             the error reporting. This function rarely returns */
          if (method == libGAP_Fail)
          {
            libGAP_Obj args[4];
            args[0] = arg1;
            args[1] = arg2;
            args[2] = arg3;
            args[3] = arg4;
            while (method == libGAP_Fail)
              method = libGAP_CallHandleMethodNotFound( oper, 4, (libGAP_Obj *) args, 0, 0, prec);
          }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 4 ) );
              cache[6*libGAP_CacheIndex] = method;
              cache[6*libGAP_CacheIndex+1] = prec;
              cache[6*libGAP_CacheIndex+2] = id1;
              cache[6*libGAP_CacheIndex+3] = id2;
              cache[6*libGAP_CacheIndex+4] = id3;
              cache[6*libGAP_CacheIndex+5] = id4;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,4));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_4ARGS( method, arg1, arg2, arg3, arg4 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}



/****************************************************************************
**
**  DoOperation5Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5> )
*/
libGAP_Obj libGAP_DoOperation5Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 id1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 id4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 id5;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;
    libGAP_Obj                 margs;


    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );  id1 = libGAP_ID_TYPE( kind1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );  id4 = libGAP_ID_TYPE( kind4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );  id5 = libGAP_ID_TYPE( kind5 );
    
    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 5 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 7*libGAP_CACHE_SIZE; i+= 7) {
            if (  cache[i+1] == prec && cache[i+2] == id1 &&
                  cache[i+3] == id2 && cache[i+4] == id3 &&
                  cache[i+5] == id4 && cache[i+6] == id5 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_6ARGS( libGAP_Method5Args, oper, kind1, kind2, kind3, kind4, kind5 );
          else
            {
              margs = libGAP_NEW_PLIST(libGAP_T_PLIST, 7);
              libGAP_SET_ELM_PLIST(margs, 1, oper );
              libGAP_SET_ELM_PLIST(margs, 2, prec );
              libGAP_SET_ELM_PLIST(margs, 3, kind1 );
              libGAP_SET_ELM_PLIST(margs, 4, kind2 );
              libGAP_SET_ELM_PLIST(margs, 5, kind3 );
              libGAP_SET_ELM_PLIST(margs, 6, kind4 );
              libGAP_SET_ELM_PLIST(margs, 7, kind5 );
              libGAP_SET_LEN_PLIST(margs, 7);
              method = libGAP_CALL_XARGS( libGAP_NextMethod5Args, margs );
            }
          
          
          /* If there was no method found, then pass the information needed for
             the error reporting. This function rarely returns */
          if (method == libGAP_Fail)
          {
            libGAP_Obj args[5];
            args[0] = arg1;
            args[1] = arg2;
            args[2] = arg3;
            args[3] = arg4;
            args[4] = arg5;
            while (method == libGAP_Fail)
              method = libGAP_CallHandleMethodNotFound( oper, 5, (libGAP_Obj *) args, 0, 0, prec);
          }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 5 ) );
              cache[7*libGAP_CacheIndex] = method;
              cache[7*libGAP_CacheIndex+1] = prec;
              cache[7*libGAP_CacheIndex+2] = id1;
              cache[7*libGAP_CacheIndex+3] = id2;
              cache[7*libGAP_CacheIndex+4] = id3;
              cache[7*libGAP_CacheIndex+5] = id4;
              cache[7*libGAP_CacheIndex+6] = id5;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,5));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_5ARGS( method, arg1, arg2, arg3, arg4, arg5 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}



/****************************************************************************
**
**  DoOperation6Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5>, <a6> )
*/
libGAP_Obj libGAP_DoOperation6Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 id1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 id4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 id5;
    libGAP_Obj                 kind6;
    libGAP_Obj                 id6;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Obj                 margs;
    libGAP_Int                 i;
    libGAP_Obj                 prec;


    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );  id1 = libGAP_ID_TYPE( kind1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );  id4 = libGAP_ID_TYPE( kind4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );  id5 = libGAP_ID_TYPE( kind5 );
    kind6 = libGAP_TYPE_OBJ_FEO( arg6 );  id6 = libGAP_ID_TYPE( kind6 );
    
    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 6 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 8*libGAP_CACHE_SIZE; i+= 8) {
            if (  cache[i+1] == prec && cache[i+2] == id1 &&
                  cache[i+3] == id2 && cache[i+4] == id3 &&
                  cache[i+5] == id4 && cache[i+6] == id5 &&
                  cache[i+7] == id6) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            {
              margs = libGAP_NEW_PLIST(libGAP_T_PLIST, 7);
              libGAP_SET_ELM_PLIST(margs, 1, oper );
              libGAP_SET_ELM_PLIST(margs, 2, kind1 );
              libGAP_SET_ELM_PLIST(margs, 3, kind2 );
              libGAP_SET_ELM_PLIST(margs, 4, kind3 );
              libGAP_SET_ELM_PLIST(margs, 5, kind4 );
              libGAP_SET_ELM_PLIST(margs, 6, kind5 );
              libGAP_SET_ELM_PLIST(margs, 7, kind6 );
              libGAP_SET_LEN_PLIST(margs, 7);
              method = libGAP_CALL_XARGS( libGAP_Method6Args, margs );

            }
          else
            {
              margs = libGAP_NEW_PLIST(libGAP_T_PLIST, 8);
              libGAP_SET_ELM_PLIST(margs, 1, oper );
              libGAP_SET_ELM_PLIST(margs, 2, prec );
              libGAP_SET_ELM_PLIST(margs, 3, kind1 );
              libGAP_SET_ELM_PLIST(margs, 4, kind2 );
              libGAP_SET_ELM_PLIST(margs, 5, kind3 );
              libGAP_SET_ELM_PLIST(margs, 6, kind4 );
              libGAP_SET_ELM_PLIST(margs, 7, kind5 );
              libGAP_SET_ELM_PLIST(margs, 8, kind6 );
              libGAP_SET_LEN_PLIST(margs, 8);
              method = libGAP_CALL_XARGS( libGAP_NextMethod6Args, margs );
            }
          
          
          /* If there was no method found, then pass the information needed for
             the error reporting. This function rarely returns */
          if (method == libGAP_Fail)
          {
            libGAP_Obj args[6];
            args[0] = arg1;
            args[1] = arg2;
            args[2] = arg3;
            args[3] = arg4;
            args[4] = arg5;
            args[5] = arg6;
            while (method == libGAP_Fail)
              method = libGAP_CallHandleMethodNotFound( oper, 6, (libGAP_Obj *) args, 0, 0, prec);
          }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 6 ) );
              cache[8*libGAP_CacheIndex] = method;
              cache[8*libGAP_CacheIndex+1] = prec;
              cache[8*libGAP_CacheIndex+2] = id1;
              cache[8*libGAP_CacheIndex+3] = id2;
              cache[8*libGAP_CacheIndex+4] = id3;
              cache[8*libGAP_CacheIndex+5] = id4;
              cache[8*libGAP_CacheIndex+6] = id5;
              cache[8*libGAP_CacheIndex+7] = id6;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,6));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_6ARGS( method, arg1, arg2, arg3, arg4, arg5, arg6 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}

/****************************************************************************
**
**  DoOperationXArgs( <oper>, ... )
*/
libGAP_Obj libGAP_DoOperationXArgs (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit("sorry: cannot yet have X argument operations",0L,0L);
    return 0;
}


/****************************************************************************
**
**  DoVerboseOperation0Args( <oper> )
*/
libGAP_Obj libGAP_DoVerboseOperation0Args (
    libGAP_Obj                 oper )
{
    libGAP_Obj                 res;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_1ARGS( libGAP_VMethod0Args, oper );
    while (method == libGAP_Fail)
      {
        method = libGAP_CallHandleMethodNotFound( oper, 0, (libGAP_Obj *) 0, 1, 0, libGAP_INTOBJ_INT(0));
      }
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_0ARGS( method );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_2ARGS( libGAP_NextVMethod0Args, oper, libGAP_INTOBJ_INT(i) );
            while (method == libGAP_Fail)
              {
                method = libGAP_CallHandleMethodNotFound( oper, 0, (libGAP_Obj *) 0, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_0ARGS( method );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseOperation1Args( <oper>, <a1> )
*/
libGAP_Obj libGAP_DoVerboseOperation1Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_2ARGS( libGAP_VMethod1Args, oper, kind1 );

    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[1];
        arglist[0] = arg1;
        method = libGAP_CallHandleMethodNotFound( oper, 1, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_1ARGS( method, arg1 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_3ARGS( libGAP_NextVMethod1Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[1];
                arglist[0] = arg1;
                method = libGAP_CallHandleMethodNotFound( oper, 1, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
    
            i++;
            res = libGAP_CALL_1ARGS( method, arg1 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseOperation2Args( <oper>, <a1>, <a2> )
*/
libGAP_Obj libGAP_DoVerboseOperation2Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_3ARGS( libGAP_VMethod2Args, oper, kind1, kind2 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[2];
        arglist[0] = arg1;
        arglist[1] = arg2;
        method = libGAP_CallHandleMethodNotFound( oper, 2, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_2ARGS( method, arg1, arg2 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_4ARGS( libGAP_NextVMethod2Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1, kind2 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[2];
                arglist[0] = arg1;
                arglist[1] = arg2;
                method = libGAP_CallHandleMethodNotFound( oper, 2, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
    
            i++;
            res = libGAP_CALL_2ARGS( method, arg1, arg2 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseOperation3Args( <oper>, <a1>, <a2>, <a3> )
*/
libGAP_Obj libGAP_DoVerboseOperation3Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_4ARGS( libGAP_VMethod3Args, oper, kind1, kind2, kind3 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[3];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        method = libGAP_CallHandleMethodNotFound( oper, 3, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_3ARGS( method, arg1, arg2, arg3 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_5ARGS( libGAP_NextVMethod3Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1, kind2, kind3 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[3];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                method = libGAP_CallHandleMethodNotFound( oper, 3, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
    
            i++;
            res = libGAP_CALL_3ARGS( method, arg1, arg2, arg3 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseOperation4Args( <oper>, <a1>, <a2>, <a3>, <a4> )
*/
libGAP_Obj libGAP_DoVerboseOperation4Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_5ARGS( libGAP_VMethod4Args, oper, kind1, kind2, kind3, kind4 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[4];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        arglist[3] = arg4;
        method = libGAP_CallHandleMethodNotFound( oper, 4, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_4ARGS( method, arg1, arg2, arg3, arg4 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_6ARGS( libGAP_NextVMethod4Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1, kind2, kind3, kind4 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[4];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                arglist[3] = arg4;
                method = libGAP_CallHandleMethodNotFound( oper, 4, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_4ARGS( method, arg1, arg2, arg3, arg4 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseOperation5Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5> )
*/
libGAP_Obj libGAP_DoVerboseOperation5Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 method;
    libGAP_Obj                 margs;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_6ARGS( libGAP_VMethod5Args, oper, kind1, kind2, kind3, kind4,
                         kind5 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[5];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        arglist[3] = arg4;
        arglist[4] = arg5;
        method = libGAP_CallHandleMethodNotFound( oper, 5, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_5ARGS( method, arg1, arg2, arg3, arg4, arg5 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            margs = libGAP_NEW_PLIST( libGAP_T_PLIST, 7 );
            libGAP_SET_LEN_PLIST( margs, 7 );
            libGAP_SET_ELM_PLIST( margs, 1, oper );
            libGAP_SET_ELM_PLIST( margs, 2, libGAP_INTOBJ_INT(i) );
            libGAP_SET_ELM_PLIST( margs, 3, kind1 );
            libGAP_SET_ELM_PLIST( margs, 4, kind2 );
            libGAP_SET_ELM_PLIST( margs, 5, kind3 );
            libGAP_SET_ELM_PLIST( margs, 6, kind4 );
            libGAP_SET_ELM_PLIST( margs, 7, kind5 );
            method = libGAP_CALL_XARGS( libGAP_NextVMethod5Args, margs );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[5];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                arglist[3] = arg4;
                arglist[4] = arg5;
                method = libGAP_CallHandleMethodNotFound( oper, 5, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_5ARGS( method, arg1, arg2, arg3, arg4, arg5 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseOperation6Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5>, <a6> )
*/
libGAP_Obj libGAP_DoVerboseOperation6Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 kind6;
    libGAP_Obj                 method;
    libGAP_Obj                 margs;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    kind1 = libGAP_TYPE_OBJ_FEO( arg1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );
    kind6 = libGAP_TYPE_OBJ_FEO( arg6 );

    /* try to find one in the list of methods                              */
    margs = libGAP_NEW_PLIST( libGAP_T_PLIST, 7 );
    libGAP_SET_LEN_PLIST( margs, 7 );
    libGAP_SET_ELM_PLIST( margs, 1, oper );
    libGAP_SET_ELM_PLIST( margs, 2, kind1 );
    libGAP_SET_ELM_PLIST( margs, 3, kind2 );
    libGAP_SET_ELM_PLIST( margs, 4, kind3 );
    libGAP_SET_ELM_PLIST( margs, 5, kind4 );
    libGAP_SET_ELM_PLIST( margs, 6, kind5 );
    libGAP_SET_ELM_PLIST( margs, 7, kind6 );
    method = libGAP_CALL_XARGS( libGAP_VMethod6Args, margs );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[6];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        arglist[3] = arg4;
        arglist[4] = arg5;
        arglist[5] = arg6;
        method = libGAP_CallHandleMethodNotFound( oper, 6, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_6ARGS( method, arg1, arg2, arg3, arg4, arg5, arg6 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            margs = libGAP_NEW_PLIST( libGAP_T_PLIST, 8 );
            libGAP_SET_LEN_PLIST( margs, 8 );
            libGAP_SET_ELM_PLIST( margs, 1, oper );
            libGAP_SET_ELM_PLIST( margs, 2, libGAP_INTOBJ_INT(i) );
            libGAP_SET_ELM_PLIST( margs, 3, kind1 );
            libGAP_SET_ELM_PLIST( margs, 4, kind2 );
            libGAP_SET_ELM_PLIST( margs, 5, kind3 );
            libGAP_SET_ELM_PLIST( margs, 6, kind4 );
            libGAP_SET_ELM_PLIST( margs, 7, kind5 );
            libGAP_SET_ELM_PLIST( margs, 8, kind6 );
            method = libGAP_CALL_XARGS( libGAP_NextVMethod6Args, margs );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[6];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                arglist[3] = arg4;
                arglist[4] = arg5;
                arglist[5] = arg6;
                method = libGAP_CallHandleMethodNotFound( oper, 6, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_6ARGS( method, arg1, arg2, arg3, arg4, arg5, arg6 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseOperationXArgs( <oper>, ... )
*/
libGAP_Obj libGAP_DoVerboseOperationXArgs (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit("sorry: cannot yet have X argument operations",0L,0L);
    return 0;
}


/****************************************************************************
**
*F  NewOperation( <name>, <narg>, <nams>, <hdlr> )
*/
libGAP_Obj libGAP_NewOperation (
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams,
    libGAP_ObjFunc             hdlr )
{
    libGAP_Obj                 oper;
#ifdef  PREALLOCATE_TABLES
    libGAP_Obj                 cache;
    libGAP_Obj                 methods;
    libGAP_UInt                i;
#endif

    /* create the function                                                 */
    oper = libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_OPER, name, narg, nams, hdlr );

    /* enter the handlers                                                  */
    libGAP_HDLR_FUNC(oper,0) = libGAP_DoOperation0Args;
    libGAP_HDLR_FUNC(oper,1) = libGAP_DoOperation1Args;
    libGAP_HDLR_FUNC(oper,2) = libGAP_DoOperation2Args;
    libGAP_HDLR_FUNC(oper,3) = libGAP_DoOperation3Args;
    libGAP_HDLR_FUNC(oper,4) = libGAP_DoOperation4Args;
    libGAP_HDLR_FUNC(oper,5) = libGAP_DoOperation5Args;
    libGAP_HDLR_FUNC(oper,6) = libGAP_DoOperation6Args;
    libGAP_HDLR_FUNC(oper,7) = libGAP_DoOperationXArgs;

    /* reenter the given handler */
    if (narg != -1)
      libGAP_HDLR_FUNC(oper,narg) = hdlr;

    /*N 1996/06/06 mschoene this should not be done here                   */
    libGAP_FLAG1_FILT(oper) = libGAP_INTOBJ_INT(0);
    libGAP_FLAG2_FILT(oper) = libGAP_INTOBJ_INT(0);
    libGAP_FLAGS_FILT(oper) = libGAP_False;
    libGAP_SETTR_FILT(oper) = libGAP_False;
    libGAP_TESTR_FILT(oper) = libGAP_False;
    
    /* create caches and methods lists                                     */
#ifdef  PREALLOCATE_TABLES
    for ( i = 0; i <= 7; i++ ) {
        methods = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
        libGAP_METHS_OPER( oper, i ) = methods;
        cache = libGAP_NEW_PLIST( libGAP_T_PLIST, (i < 7 ? libGAP_CACHE_SIZE * (i+2) : libGAP_CACHE_SIZE * (1+2)) );
        libGAP_CACHE_OPER( oper, i ) = cache;
        libGAP_CHANGED_BAG(oper);
    }
#endif

    /* This isn't an attribute (yet) */
    libGAP_SET_ENABLED_ATTR(oper, 0);

    /* return operation                                                    */
    return oper;
}


/****************************************************************************
**

*F  DoConstructor( <name> ) . . . . . . . . . . . . .  make a new constructor
*/
libGAP_UInt libGAP_CacheIndex;

libGAP_Obj libGAP_Constructor0Args;
libGAP_Obj libGAP_NextConstructor0Args;
libGAP_Obj libGAP_Constructor1Args;
libGAP_Obj libGAP_NextConstructor1Args;
libGAP_Obj libGAP_Constructor2Args;
libGAP_Obj libGAP_NextConstructor2Args;
libGAP_Obj libGAP_Constructor3Args;
libGAP_Obj libGAP_NextConstructor3Args;
libGAP_Obj libGAP_Constructor4Args;
libGAP_Obj libGAP_NextConstructor4Args;
libGAP_Obj libGAP_Constructor5Args;
libGAP_Obj libGAP_NextConstructor5Args;
libGAP_Obj libGAP_Constructor6Args;
libGAP_Obj libGAP_NextConstructor6Args;
libGAP_Obj libGAP_ConstructorXArgs;
libGAP_Obj libGAP_NextConstructorXArgs;

libGAP_Obj libGAP_VConstructor0Args;
libGAP_Obj libGAP_NextVConstructor0Args;
libGAP_Obj libGAP_VConstructor1Args;
libGAP_Obj libGAP_NextVConstructor1Args;
libGAP_Obj libGAP_VConstructor2Args;
libGAP_Obj libGAP_NextVConstructor2Args;
libGAP_Obj libGAP_VConstructor3Args;
libGAP_Obj libGAP_NextVConstructor3Args;
libGAP_Obj libGAP_VConstructor4Args;
libGAP_Obj libGAP_NextVConstructor4Args;
libGAP_Obj libGAP_VConstructor5Args;
libGAP_Obj libGAP_NextVConstructor5Args;
libGAP_Obj libGAP_VConstructor6Args;
libGAP_Obj libGAP_NextVConstructor6Args;
libGAP_Obj libGAP_VConstructorXArgs;
libGAP_Obj libGAP_NextVConstructorXArgs;

/****************************************************************************
**
**  DoConstructor0Args( <oper> )
**
** I'm not sure if this makes any sense at all
*/

libGAP_Obj libGAP_DoConstructor0Args (
    libGAP_Obj                 oper )
{
    libGAP_Obj                 res;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                prec;

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 0 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 2*libGAP_CACHE_SIZE; i+= 2) {
            if (  cache[i] != 0 && cache[i+1] == prec) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_1ARGS( libGAP_Constructor0Args, oper );
          else
            method = libGAP_CALL_2ARGS( libGAP_NextConstructor0Args, oper, prec );
          
          while (method == libGAP_Fail)
            method = libGAP_CallHandleMethodNotFound( oper, 0, (libGAP_Obj *)0, 0, 1, prec);
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 0 ) );
              cache[2*libGAP_CacheIndex] = method;
              cache[2*libGAP_CacheIndex+1] = prec;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,0));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_0ARGS( method );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoConstructor1Args( <oper>, <a1> )
*/
libGAP_Obj libGAP_DoConstructor1Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    
    kind1 = libGAP_FLAGS_FILT( arg1 );  

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 1 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 3*libGAP_CACHE_SIZE; i+= 3) {
            if (  cache[i+1] == prec && cache[i+2] == kind1 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              ConstructorHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_2ARGS( libGAP_Constructor1Args, oper, kind1 );
          else
            method = libGAP_CALL_3ARGS( libGAP_NextConstructor1Args, oper, prec, kind1 );

          while (method == libGAP_Fail)
            {
              libGAP_Obj arglist[1];
              arglist[0] = arg1;
              method = libGAP_CallHandleMethodNotFound(oper, 1, arglist, 0, 1, prec);
            }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 1 ) );
              cache[3*libGAP_CacheIndex] = method;
              cache[3*libGAP_CacheIndex+1] = prec;
              cache[3*libGAP_CacheIndex+2] = kind1;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,1));
            }
#ifdef COUNT_OPERS
          ConstructorMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_1ARGS( method, arg1 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoConstructor2Args( <oper>, <a1>, <a2> )
*/
libGAP_Obj libGAP_DoConstructor2Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    kind1 = libGAP_FLAGS_FILT( arg1 );  
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 2 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 4*libGAP_CACHE_SIZE; i+= 4) {
            if (  cache[i+1] == prec && cache[i+2] == kind1
                  && cache[i+3] == id2 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              OperationgHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_3ARGS( libGAP_Constructor2Args, oper, kind1, kind2 );
          else
            method = libGAP_CALL_4ARGS( libGAP_NextConstructor2Args, oper, prec, kind1, kind2 );
          
          while (method == libGAP_Fail)
            {
              libGAP_Obj arglist[2];
              arglist[0] = arg1;
              arglist[1] = arg2;
              method = libGAP_CallHandleMethodNotFound(oper, 2, arglist, 0, 1, prec);
            }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 2 ) );
              cache[4*libGAP_CacheIndex] = method;
              cache[4*libGAP_CacheIndex+1] = prec;
              cache[4*libGAP_CacheIndex+2] = kind1;
              cache[4*libGAP_CacheIndex+3] = id2;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,2));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_2ARGS( method, arg1, arg2 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}



/****************************************************************************
**
**  DoConstructor3Args( <oper>, <a1>, <a2>, <a3> )
*/
libGAP_Obj libGAP_DoConstructor3Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    kind1 = libGAP_FLAGS_FILT( arg1 ); 
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 3 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 5*libGAP_CACHE_SIZE; i+= 5) {
            if (  cache[i+1] == prec && cache[i+2] == kind1
                  && cache[i+3] == id2 && cache[i+4] == id3 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_4ARGS( libGAP_Constructor3Args, oper, kind1, kind2, kind3 );
          else
            method = libGAP_CALL_5ARGS( libGAP_NextConstructor3Args, oper, prec, kind1, kind2, kind3 );
          
          while (method == libGAP_Fail)
            {
              libGAP_Obj arglist[3];
              arglist[0] = arg1;
              arglist[1] = arg2;
              arglist[2] = arg3;
              method = libGAP_CallHandleMethodNotFound(oper, 3, arglist, 0, 1, prec);
            }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 3 ) );
              cache[5*libGAP_CacheIndex] = method;
              cache[5*libGAP_CacheIndex+1] = prec;
              cache[5*libGAP_CacheIndex+2] = kind1;
              cache[5*libGAP_CacheIndex+3] = id2;
              cache[5*libGAP_CacheIndex+4] = id3;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,3));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_3ARGS( method, arg1, arg2, arg3 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoConstructor4Args( <oper>, <a1>, <a2>, <a3>, <a4> )
*/
libGAP_Obj libGAP_DoConstructor4Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 id4;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;


    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    kind1 = libGAP_FLAGS_FILT( arg1 ); 
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );  id4 = libGAP_ID_TYPE( kind4 );

    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 4 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 6*libGAP_CACHE_SIZE; i+= 6) {
            if (  cache[i+1] == prec && cache[i+2] == kind1 &&
                  cache[i+3] == id2 && cache[i+4] == id3 &&
                  cache[i+5] == id4 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_5ARGS( libGAP_Constructor4Args, oper, kind1, kind2, kind3, kind4 );
          else
            method = libGAP_CALL_6ARGS( libGAP_NextConstructor4Args, oper, prec, kind1, kind2, kind3, kind4 );
          
          while (method == libGAP_Fail)
            {
              libGAP_Obj arglist[4];
              arglist[0] = arg1;
              arglist[1] = arg2;
              arglist[2] = arg3;
              arglist[3] = arg4;
              method = libGAP_CallHandleMethodNotFound(oper, 4, arglist, 0, 1, prec);
            }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 4 ) );
              cache[6*libGAP_CacheIndex] = method;
              cache[6*libGAP_CacheIndex+1] = prec;
              cache[6*libGAP_CacheIndex+2] = kind1;
              cache[6*libGAP_CacheIndex+3] = id2;
              cache[6*libGAP_CacheIndex+4] = id3;
              cache[6*libGAP_CacheIndex+5] = id4;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,4));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_4ARGS( method, arg1, arg2, arg3, arg4 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}



/****************************************************************************
**
**  DoConstructor5Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5> )
*/
libGAP_Obj libGAP_DoConstructor5Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 id4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 id5;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Int                 i;
    libGAP_Obj                 prec;
    libGAP_Obj                 margs;


    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    kind1 = libGAP_FLAGS_FILT( arg1 ); 
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );  id4 = libGAP_ID_TYPE( kind4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );  id5 = libGAP_ID_TYPE( kind5 );
    
    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 5 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 7*libGAP_CACHE_SIZE; i+= 7) {
            if (  cache[i+1] == prec && cache[i+2] == kind1 &&
                  cache[i+3] == id2 && cache[i+4] == id3 &&
                  cache[i+5] == id4 && cache[i+6] == id5 ) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            method = libGAP_CALL_6ARGS( libGAP_Constructor5Args, oper, kind1, kind2, kind3, kind4, kind5 );
          else
            {
              margs = libGAP_NEW_PLIST(libGAP_T_PLIST, 7);
              libGAP_SET_ELM_PLIST(margs, 1, oper );
              libGAP_SET_ELM_PLIST(margs, 2, prec );
              libGAP_SET_ELM_PLIST(margs, 3, kind1 );
              libGAP_SET_ELM_PLIST(margs, 4, kind2 );
              libGAP_SET_ELM_PLIST(margs, 5, kind3 );
              libGAP_SET_ELM_PLIST(margs, 6, kind4 );
              libGAP_SET_ELM_PLIST(margs, 7, kind5 );
              libGAP_SET_LEN_PLIST(margs, 7);
              method = libGAP_CALL_XARGS( libGAP_NextConstructor5Args, margs );
            }
          
          while (method == libGAP_Fail)
            {
              libGAP_Obj arglist[5];
              arglist[0] = arg1;
              arglist[1] = arg2;
              arglist[2] = arg3;
              arglist[3] = arg4;
              arglist[4] = arg5;
              method = libGAP_CallHandleMethodNotFound(oper, 5, arglist, 0, 1, prec);
            }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 5 ) );
              cache[7*libGAP_CacheIndex] = method;
              cache[7*libGAP_CacheIndex+1] = prec;
              cache[7*libGAP_CacheIndex+2] = kind1;
              cache[7*libGAP_CacheIndex+3] = id2;
              cache[7*libGAP_CacheIndex+4] = id3;
              cache[7*libGAP_CacheIndex+5] = id4;
              cache[7*libGAP_CacheIndex+6] = id5;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,5));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_5ARGS( method, arg1, arg2, arg3, arg4, arg5 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}



/****************************************************************************
**
**  DoConstructor6Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5>, <a6> )
*/
libGAP_Obj libGAP_DoConstructor6Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 id2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 id3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 id4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 id5;
    libGAP_Obj                 kind6;
    libGAP_Obj                 id6;
    libGAP_Obj *               cache;
    libGAP_Obj                 method;
    libGAP_Obj                 margs;
    libGAP_Int                 i;
    libGAP_Obj                 prec;


    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    kind1 = libGAP_FLAGS_FILT( arg1 ); 
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );  id2 = libGAP_ID_TYPE( kind2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );  id3 = libGAP_ID_TYPE( kind3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );  id4 = libGAP_ID_TYPE( kind4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );  id5 = libGAP_ID_TYPE( kind5 );
    kind6 = libGAP_TYPE_OBJ_FEO( arg6 );  id6 = libGAP_ID_TYPE( kind6 );
    
    /* try to find an applicable method in the cache                       */
    cache = 1+libGAP_ADDR_OBJ( libGAP_CacheOper( oper, 6 ) );
    prec = libGAP_INTOBJ_INT(-1);

    do {
      /* The next line depends on the implementation of INTOBJS */
      prec = (libGAP_Obj)(((libGAP_Int)prec) +4);
      method = 0;

      /* Up to CACHE_SIZE methods might be in the cache */
      if (prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
        {
          for (i = 0;  i < 8*libGAP_CACHE_SIZE; i+= 8) {
            if (  cache[i+1] == prec && cache[i+2] == kind1 &&
                  cache[i+3] == id2 && cache[i+4] == id3 &&
                  cache[i+5] == id4 && cache[i+6] == id5 &&
                  cache[i+7] == id6) {
              method = cache[i];
#ifdef COUNT_OPERS
              libGAP_OperationHit++;
#endif
              break;
            }
          }
        }
      
      /* otherwise try to find one in the list of methods                    */
      if (!method)
        {
          if (prec == libGAP_INTOBJ_INT(0))
            {
              margs = libGAP_NEW_PLIST(libGAP_T_PLIST, 7);
              libGAP_SET_ELM_PLIST(margs, 1, oper );
              libGAP_SET_ELM_PLIST(margs, 2, kind1 );
              libGAP_SET_ELM_PLIST(margs, 3, kind2 );
              libGAP_SET_ELM_PLIST(margs, 4, kind3 );
              libGAP_SET_ELM_PLIST(margs, 5, kind4 );
              libGAP_SET_ELM_PLIST(margs, 6, kind5 );
              libGAP_SET_ELM_PLIST(margs, 7, kind6 );
              libGAP_SET_LEN_PLIST(margs, 7);
              method = libGAP_CALL_XARGS( libGAP_Constructor6Args, margs );

            }
          else
            {
              margs = libGAP_NEW_PLIST(libGAP_T_PLIST, 8);
              libGAP_SET_ELM_PLIST(margs, 1, oper );
              libGAP_SET_ELM_PLIST(margs, 2, prec );
              libGAP_SET_ELM_PLIST(margs, 3, kind1 );
              libGAP_SET_ELM_PLIST(margs, 4, kind2 );
              libGAP_SET_ELM_PLIST(margs, 5, kind3 );
              libGAP_SET_ELM_PLIST(margs, 6, kind4 );
              libGAP_SET_ELM_PLIST(margs, 7, kind5 );
              libGAP_SET_ELM_PLIST(margs, 8, kind6 );
              libGAP_SET_LEN_PLIST(margs, 8);
              method = libGAP_CALL_XARGS( libGAP_NextConstructor6Args, margs );
            }
          
          while (method == libGAP_Fail)
            {
              libGAP_Obj arglist[6];
              arglist[0] = arg1;
              arglist[1] = arg2;
              arglist[2] = arg3;
              arglist[3] = arg4;
              arglist[4] = arg5;
              arglist[5] = arg6;
              method = libGAP_CallHandleMethodNotFound(oper, 6, arglist, 0, 1, prec);
            }
          
          /* update the cache */
          if (method && prec < libGAP_INTOBJ_INT(libGAP_CACHE_SIZE))
            {
              cache = 1+libGAP_ADDR_OBJ( libGAP_CACHE_OPER( oper, 6 ) );
              cache[8*libGAP_CacheIndex] = method;
              cache[8*libGAP_CacheIndex+1] = prec;
              cache[8*libGAP_CacheIndex+2] = kind1;
              cache[8*libGAP_CacheIndex+3] = id2;
              cache[8*libGAP_CacheIndex+4] = id3;
              cache[8*libGAP_CacheIndex+5] = id4;
              cache[8*libGAP_CacheIndex+6] = id5;
              cache[8*libGAP_CacheIndex+7] = id6;
              libGAP_CacheIndex = (libGAP_CacheIndex + 1) % libGAP_CACHE_SIZE;
              libGAP_CHANGED_BAG(libGAP_CACHE_OPER(oper,6));
            }
#ifdef COUNT_OPERS
          libGAP_OperationMiss++;
#endif
        }
      if ( !method )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
      }
      
      /* call this method                                                    */
      res = libGAP_CALL_6ARGS( method, arg1, arg2, arg3, arg4, arg5, arg6 );
    }
    while (res == libGAP_TRY_NEXT_METHOD );

    /* return the result                                                   */
    return res;
}




/****************************************************************************
**
**  DoConstructorXArgs( <oper>, ... )
*/
libGAP_Obj libGAP_DoConstructorXArgs (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit("sorry: cannot yet have X argument constructors",0L,0L);
    return 0;
}

/****************************************************************************
**
**  DoVerboseConstructor0Args( <oper> )
*/
libGAP_Obj libGAP_DoVerboseConstructor0Args (
    libGAP_Obj                 oper )
{
    libGAP_Obj                 res;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_1ARGS( libGAP_VConstructor0Args, oper );
    while (method == libGAP_Fail)
      {
        method = libGAP_CallHandleMethodNotFound( oper, 0, (libGAP_Obj *) 0, 1, 0, libGAP_INTOBJ_INT(0));
      }
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_0ARGS( method );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_2ARGS( libGAP_NextVConstructor0Args, oper, libGAP_INTOBJ_INT(i) );
            while (method == libGAP_Fail)
              {
                method = libGAP_CallHandleMethodNotFound( oper, 0, (libGAP_Obj *) 0, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_0ARGS( method );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseConstructor1Args( <oper>, <a1> )
*/
libGAP_Obj libGAP_DoVerboseConstructor1Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    
    kind1 = libGAP_FLAGS_FILT( arg1 );  


    /* try to find one in the list of methods                              */
    method = libGAP_CALL_2ARGS( libGAP_VConstructor1Args, oper, kind1 );

    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[1];
        arglist[0] = arg1;
        method = libGAP_CallHandleMethodNotFound( oper, 1, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_1ARGS( method, arg1 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_3ARGS( libGAP_NextVConstructor1Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[1];
                arglist[0] = arg1;
                method = libGAP_CallHandleMethodNotFound( oper, 1, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
    
            i++;
            res = libGAP_CALL_1ARGS( method, arg1 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseConstructor2Args( <oper>, <a1>, <a2> )
*/
libGAP_Obj libGAP_DoVerboseConstructor2Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    
    kind1 = libGAP_FLAGS_FILT( arg1 );  
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_3ARGS( libGAP_VConstructor2Args, oper, kind1, kind2 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[2];
        arglist[0] = arg1;
        arglist[1] = arg2;
        method = libGAP_CallHandleMethodNotFound( oper, 2, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_2ARGS( method, arg1, arg2 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_4ARGS( libGAP_NextVConstructor2Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1, kind2 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[2];
                arglist[0] = arg1;
                arglist[1] = arg2;
                method = libGAP_CallHandleMethodNotFound( oper, 2, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
    
            i++;
            res = libGAP_CALL_2ARGS( method, arg1, arg2 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseConstructor3Args( <oper>, <a1>, <a2>, <a3> )
*/
libGAP_Obj libGAP_DoVerboseConstructor3Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    
    kind1 = libGAP_FLAGS_FILT( arg1 );
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_4ARGS( libGAP_VConstructor3Args, oper, kind1, kind2, kind3 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[3];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        method = libGAP_CallHandleMethodNotFound( oper, 3, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_3ARGS( method, arg1, arg2, arg3 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_5ARGS( libGAP_NextVConstructor3Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1, kind2, kind3 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[3];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                method = libGAP_CallHandleMethodNotFound( oper, 3, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
    
            i++;
            res = libGAP_CALL_3ARGS( method, arg1, arg2, arg3 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseConstructor4Args( <oper>, <a1>, <a2>, <a3>, <a4> )
*/
libGAP_Obj libGAP_DoVerboseConstructor4Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 method;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    
    kind1 = libGAP_FLAGS_FILT( arg1 );  
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_5ARGS( libGAP_VConstructor4Args, oper, kind1, kind2, kind3, kind4 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[4];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        arglist[3] = arg4;
        method = libGAP_CallHandleMethodNotFound( oper, 4, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_4ARGS( method, arg1, arg2, arg3, arg4 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            method = libGAP_CALL_6ARGS( libGAP_NextVConstructor4Args, oper, libGAP_INTOBJ_INT(i),
                                 kind1, kind2, kind3, kind4 );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[4];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                arglist[3] = arg4;
                method = libGAP_CallHandleMethodNotFound( oper, 4, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_4ARGS( method, arg1, arg2, arg3, arg4 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseConstructor5Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5> )
*/
libGAP_Obj libGAP_DoVerboseConstructor5Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 method;
    libGAP_Obj                 margs;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    
    kind1 = libGAP_FLAGS_FILT( arg1 );  
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );

    /* try to find one in the list of methods                              */
    method = libGAP_CALL_6ARGS( libGAP_VConstructor5Args, oper, kind1, kind2, kind3, kind4,
                         kind5 );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[5];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        arglist[3] = arg4;
        arglist[4] = arg5;
        method = libGAP_CallHandleMethodNotFound( oper, 5, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_5ARGS( method, arg1, arg2, arg3, arg4, arg5 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            margs = libGAP_NEW_PLIST( libGAP_T_PLIST, 7 );
            libGAP_SET_LEN_PLIST( margs, 7 );
            libGAP_SET_ELM_PLIST( margs, 1, oper );
            libGAP_SET_ELM_PLIST( margs, 2, libGAP_INTOBJ_INT(i) );
            libGAP_SET_ELM_PLIST( margs, 3, kind1 );
            libGAP_SET_ELM_PLIST( margs, 4, kind2 );
            libGAP_SET_ELM_PLIST( margs, 5, kind3 );
            libGAP_SET_ELM_PLIST( margs, 6, kind4 );
            libGAP_SET_ELM_PLIST( margs, 7, kind5 );
            method = libGAP_CALL_XARGS( libGAP_NextVConstructor5Args, margs );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[5];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                arglist[3] = arg4;
                arglist[4] = arg5;
                method = libGAP_CallHandleMethodNotFound( oper, 5, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_5ARGS( method, arg1, arg2, arg3, arg4, arg5 );
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
**  DoVerboseConstructor6Args( <oper>, <a1>, <a2>, <a3>, <a4>, <a5>, <a6> )
*/
libGAP_Obj libGAP_DoVerboseConstructor6Args (
    libGAP_Obj                 oper,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Obj                 res;
    libGAP_Obj                 kind1;
    libGAP_Obj                 kind2;
    libGAP_Obj                 kind3;
    libGAP_Obj                 kind4;
    libGAP_Obj                 kind5;
    libGAP_Obj                 kind6;
    libGAP_Obj                 method;
    libGAP_Obj                 margs;
    libGAP_Int                 i;

    /* get the kinds of the arguments                                      */
    while (!libGAP_IS_OPERATION(arg1))
      {
        arg1 = libGAP_ErrorReturnObj(
                "Constructor: the first argument must be a filter not a %s",
                (libGAP_Int)libGAP_TNAM_OBJ(arg1), 0L, 
                "you can replace the first argument <arg1> via 'return <arg1>;'");
      }
    
    kind1 = libGAP_FLAGS_FILT( arg1 );  
    kind2 = libGAP_TYPE_OBJ_FEO( arg2 );
    kind3 = libGAP_TYPE_OBJ_FEO( arg3 );
    kind4 = libGAP_TYPE_OBJ_FEO( arg4 );
    kind5 = libGAP_TYPE_OBJ_FEO( arg5 );
    kind6 = libGAP_TYPE_OBJ_FEO( arg6 );

    /* try to find one in the list of methods                              */
    margs = libGAP_NEW_PLIST( libGAP_T_PLIST, 7 );
    libGAP_SET_LEN_PLIST( margs, 7 );
    libGAP_SET_ELM_PLIST( margs, 1, oper );
    libGAP_SET_ELM_PLIST( margs, 2, kind1 );
    libGAP_SET_ELM_PLIST( margs, 3, kind2 );
    libGAP_SET_ELM_PLIST( margs, 4, kind3 );
    libGAP_SET_ELM_PLIST( margs, 5, kind4 );
    libGAP_SET_ELM_PLIST( margs, 6, kind5 );
    libGAP_SET_ELM_PLIST( margs, 7, kind6 );
    method = libGAP_CALL_XARGS( libGAP_VConstructor6Args, margs );
    while (method == libGAP_Fail)
      {
        libGAP_Obj arglist[6];
        arglist[0] = arg1;
        arglist[1] = arg2;
        arglist[2] = arg3;
        arglist[3] = arg4;
        arglist[4] = arg5;
        arglist[5] = arg6;
        method = libGAP_CallHandleMethodNotFound( oper, 6, arglist, 1, 0, libGAP_INTOBJ_INT(0));
      }
    if ( method == 0 )  {
        libGAP_ErrorQuit( "no method returned", 0L, 0L );
    }

    /* call this method                                                    */
    res = libGAP_CALL_6ARGS( method, arg1, arg2, arg3, arg4, arg5, arg6 );

    /* try until a method doesn't give up                                  */
    if ( res == libGAP_TRY_NEXT_METHOD ) {
        i = 1;
        while ( res == libGAP_TRY_NEXT_METHOD ) {
#ifdef COUNT_OPERS
            libGAP_OperationNext++;
#endif
            margs = libGAP_NEW_PLIST( libGAP_T_PLIST, 8 );
            libGAP_SET_LEN_PLIST( margs, 8 );
            libGAP_SET_ELM_PLIST( margs, 1, oper );
            libGAP_SET_ELM_PLIST( margs, 2, libGAP_INTOBJ_INT(i) );
            libGAP_SET_ELM_PLIST( margs, 3, kind1 );
            libGAP_SET_ELM_PLIST( margs, 4, kind2 );
            libGAP_SET_ELM_PLIST( margs, 5, kind3 );
            libGAP_SET_ELM_PLIST( margs, 6, kind4 );
            libGAP_SET_ELM_PLIST( margs, 7, kind5 );
            libGAP_SET_ELM_PLIST( margs, 8, kind6 );
            method = libGAP_CALL_XARGS( libGAP_NextVConstructor6Args, margs );
            while (method == libGAP_Fail)
              {
                libGAP_Obj arglist[6];
                arglist[0] = arg1;
                arglist[1] = arg2;
                arglist[2] = arg3;
                arglist[3] = arg4;
                arglist[4] = arg5;
                arglist[5] = arg6;
                method = libGAP_CallHandleMethodNotFound( oper, 6, arglist, 1, 0, libGAP_INTOBJ_INT(i));
              }
            i++;
            res = libGAP_CALL_6ARGS( method, arg1, arg2, arg3, arg4, arg5, arg6 );
        }
    }

    /* return the result                                                   */
    return res;
}



/****************************************************************************
**
**  DoVerboseConstructorXArgs( <oper>, ... )
*/
libGAP_Obj libGAP_DoVerboseConstructorXArgs (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit("sorry: cannot yet have X argument constructors",0L,0L);
    return 0;
}


/****************************************************************************
**
*F  NewConstructor( <name>, <narg>, <nams>, <hdlr> )
*/
libGAP_Obj libGAP_NewConstructor (
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams,
    libGAP_ObjFunc             hdlr )
{
    libGAP_Obj                 oper;
#ifdef  PREALLOCATE_TABLES
    libGAP_Obj                 cache;
    libGAP_Obj                 methods;
    libGAP_UInt                i;
#endif

    /* create the function                                                 */
    oper = libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_OPER, name, narg, nams, hdlr );

    /* enter the handlers                                                  */
    if ( narg == -1 ) {
        libGAP_HDLR_FUNC(oper,0) = libGAP_DoConstructor0Args;
        libGAP_HDLR_FUNC(oper,1) = libGAP_DoConstructor1Args;
        libGAP_HDLR_FUNC(oper,2) = libGAP_DoConstructor2Args;
        libGAP_HDLR_FUNC(oper,3) = libGAP_DoConstructor3Args;
        libGAP_HDLR_FUNC(oper,4) = libGAP_DoConstructor4Args;
        libGAP_HDLR_FUNC(oper,5) = libGAP_DoConstructor5Args;
        libGAP_HDLR_FUNC(oper,6) = libGAP_DoConstructor6Args;
        libGAP_HDLR_FUNC(oper,7) = libGAP_DoConstructorXArgs;
    }

    /*N 1996/06/06 mschoene this should not be done here                   */
    libGAP_FLAG1_FILT(oper) = libGAP_INTOBJ_INT(0);
    libGAP_FLAG2_FILT(oper) = libGAP_INTOBJ_INT(0);
    libGAP_FLAGS_FILT(oper) = libGAP_False;
    libGAP_SETTR_FILT(oper) = libGAP_False;
    libGAP_TESTR_FILT(oper) = libGAP_False;
    
#ifdef  PREALLOCATE_TABLES
    /* create caches and methods lists                                     */
    for ( i = 0; i <= 7; i++ ) {
        methods = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
        libGAP_METHS_OPER( oper, i ) = methods;
        cache = libGAP_NEW_PLIST( libGAP_T_PLIST, (i < 7 ? libGAP_CACHE_SIZE * (i+1) : libGAP_CACHE_SIZE * (1+1)) );
        libGAP_CACHE_OPER( oper, i ) = cache;
        libGAP_CHANGED_BAG(oper);
    }
#endif

    /* return constructor                                                  */
    return oper;
}


/****************************************************************************
**
*F  DoAttribute( <name> ) . . . . . . . . . . . . . . .  make a new attribute
*/



/****************************************************************************
**
**  DoTestAttribute( <attr>, <obj> )
*/
libGAP_Obj libGAP_DoTestAttribute (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flag for the tester                                         */
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, return 'true'        */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_True;
    }
    
    /* return the 'false'                                                  */
    return libGAP_False;
}


/****************************************************************************
**
**  DoAttribute( <attr>, <obj> )
*/
#define libGAP_DoSetAttribute  libGAP_DoOperation2Args

libGAP_Obj libGAP_DoAttribute (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flag for the tester                                         */
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, simply return it     */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_DoOperation1Args( self, obj );
    }
    
    /* call the operation to compute the value                             */
    val = libGAP_DoOperation1Args( self, obj );
    while (val == (libGAP_Obj) 0) {
        val = libGAP_ErrorReturnObj("Method for an attribute must return a value",
                             0L, 0L, 
                             "you can supply a value <val> via 'return <val>;'");
    }
    val = libGAP_CopyObj( val, 0 );
    
    /* set the value (but not for internal objects)                        */
    if ( libGAP_ENABLED_ATTR( self ) == 1 ) {
        switch ( libGAP_TNUM_OBJ( obj ) ) {
        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            libGAP_DoSetAttribute( libGAP_SETTR_FILT(self), obj, val );
        }
    }
    
    /* return the value                                                    */
    return val;    
}


/****************************************************************************
**
**  DoVerboseAttribute( <attr>, <obj> )
*/
#define libGAP_DoVerboseSetAttribute  libGAP_DoVerboseOperation2Args

libGAP_Obj libGAP_DoVerboseAttribute (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flag for the tester                                         */
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, simply return it     */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_DoVerboseOperation1Args( self, obj );
    }
    
    /* call the operation to compute the value                             */
    val = libGAP_CopyObj( libGAP_DoVerboseOperation1Args( self, obj ), 0 );
    
    /* set the value (but not for internal objects)                        */
    if ( libGAP_ENABLED_ATTR( self ) == 1 ) {
        switch ( libGAP_TNUM_OBJ( obj ) ) {
        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            libGAP_DoVerboseSetAttribute( libGAP_SETTR_FILT(self), obj, val );
        }
    }
    
    /* return the value                                                    */
    return val;    
}


/****************************************************************************
**
**  DoMutableAttribute( <attr>, <obj> )
*/
libGAP_Obj libGAP_DoMutableAttribute (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flag for the tester                                         */
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, simply return it     */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_DoOperation1Args( self, obj );
    }
    
    /* call the operation to compute the value                             */
    val = libGAP_DoOperation1Args( self, obj );
    
    /* set the value (but not for internal objects)                        */
    if ( libGAP_ENABLED_ATTR( self ) == 1 ) {
        switch ( libGAP_TNUM_OBJ( obj ) ) {
        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            libGAP_DoSetAttribute( libGAP_SETTR_FILT(self), obj, val );
        }
    }
    
    /* return the value                                                    */
    return val;    
}


/****************************************************************************
**
**  DoVerboseMutableAttribute( <attr>, <obj> )
*/
libGAP_Obj libGAP_DoVerboseMutableAttribute (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flag for the tester                                         */
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, simply return it     */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_DoVerboseOperation1Args( self, obj );
    }
    
    /* call the operation to compute the value                             */
    val = libGAP_DoVerboseOperation1Args( self, obj );
    
    /* set the value (but not for internal objects)                        */
    if ( libGAP_ENABLED_ATTR( self ) == 1 ) {
        switch ( libGAP_TNUM_OBJ( obj ) ) {
        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            libGAP_DoVerboseSetAttribute( libGAP_SETTR_FILT(self), obj, val );
        }
    }
    
    /* return the value                                                    */
    return val;    
}


/****************************************************************************
**
*F  NewAttribute( <name>, <narg>, <nams>, <hdlr> )
**
** MakeSetter, MakeTester and SetupAttribute are support functions
*/

#define libGAP_WRAP_NAME(fname, name, addon) \
    do { \
        libGAP_UInt name_len = libGAP_GET_LEN_STRING(name); \
        libGAP_UInt addon_len = sizeof(addon) - 1; \
        char *tmp; \
        fname = libGAP_NEW_STRING( name_len + addon_len + 2 ); \
        tmp = libGAP_CSTR_STRING(fname); \
        memcpy( tmp, addon, addon_len ); tmp += addon_len; \
        *tmp++ = '('; \
        memcpy( tmp, libGAP_CSTR_STRING(name), name_len ); tmp += name_len; \
        *tmp++ = ')'; \
        *tmp = 0; \
        libGAP_RetypeBag( fname, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(fname)) ); \
    } while(0)

static libGAP_Obj libGAP_MakeSetter( libGAP_Obj name, libGAP_Int flag)
{
  libGAP_Obj fname;
  libGAP_Obj setter;
  libGAP_WRAP_NAME(fname, name, "Setter");
  setter = libGAP_NewOperation( fname, 2L, 0L, libGAP_DoSetAttribute );
  libGAP_FLAG1_FILT(setter)  = libGAP_INTOBJ_INT( 0 );
  libGAP_FLAG2_FILT(setter)  = libGAP_INTOBJ_INT( flag );
  libGAP_CHANGED_BAG(setter);
  return setter;
}

static libGAP_Obj libGAP_MakeTester( libGAP_Obj name, libGAP_Int flag)
{
    libGAP_Obj fname;
    libGAP_Obj tester;
    libGAP_Obj flags;
    libGAP_WRAP_NAME(fname, name, "Tester");
    tester = libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_OPER, fname, 1L, 0L,
                           libGAP_DoTestAttribute );    
    libGAP_FLAG1_FILT(tester)  = libGAP_INTOBJ_INT( 0 );
    libGAP_FLAG2_FILT(tester)  = libGAP_INTOBJ_INT( flag );
    libGAP_NEW_FLAGS( flags, flag );
    libGAP_SET_LEN_FLAGS( flags, flag );
    libGAP_SET_ELM_FLAGS( flags, flag, libGAP_True );
    libGAP_FLAGS_FILT(tester)  = flags;
    libGAP_SETTR_FILT(tester)  = 0;
    libGAP_TESTR_FILT(tester)  = libGAP_ReturnTrueFilter;
    libGAP_CHANGED_BAG(tester);
    return tester;
}


static void libGAP_SetupAttribute(libGAP_Obj attr, libGAP_Obj setter, libGAP_Obj tester, libGAP_Int flag2)
{
  libGAP_Obj flags;

    /* Install additional data */
  libGAP_FLAG1_FILT(attr)  = libGAP_INTOBJ_INT( 0 );
  libGAP_FLAG2_FILT(attr)  = libGAP_INTOBJ_INT( flag2 );
  libGAP_NEW_FLAGS( flags, flag2 );
  libGAP_SET_LEN_FLAGS( flags, flag2 );
  libGAP_SET_ELM_FLAGS( flags, flag2, libGAP_True );
  
  /*    FLAGS_FILT(tester)  = flags; */
  libGAP_FLAGS_FILT(attr) = libGAP_FLAGS_FILT(tester);
  
  libGAP_SETTR_FILT(attr)  = setter;
  libGAP_TESTR_FILT(attr)  = tester;
  libGAP_SET_ENABLED_ATTR(attr,1);
  libGAP_CHANGED_BAG(attr);
  return;
}

  

libGAP_Obj libGAP_NewAttribute (
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams,
    libGAP_ObjFunc             hdlr )
{
    libGAP_Obj                 getter;
    libGAP_Obj                 setter;
    libGAP_Obj                 tester;
    libGAP_Int                 flag2;
    
    flag2 = ++libGAP_CountFlags;
    setter = libGAP_MakeSetter(name, flag2);
    tester = libGAP_MakeTester(name, flag2);

    getter = libGAP_NewOperation( name, 1L, nams, (hdlr ? hdlr : libGAP_DoAttribute) ); 
    
    libGAP_SetupAttribute(getter, setter, tester, flag2);

    return getter;    
}

/****************************************************************************
**
*F  ConvertOperationIntoAttribute( <oper> )  transform an operation (which 
**  should not have any one-argument declarations) into an attribute
*/

void libGAP_ConvertOperationIntoAttribute( libGAP_Obj oper, libGAP_ObjFunc hdlr ) 
{
    libGAP_Obj                 setter;
    libGAP_Obj                 tester;
    libGAP_Int                 flag2;
    libGAP_Obj                  name;

    /* Need to get the name from oper */
    name = libGAP_NAME_FUNC(oper);

    flag2 = ++libGAP_CountFlags;

    /* Make the setter */
    setter = libGAP_MakeSetter(name, flag2);

    /* and the tester */
    tester = libGAP_MakeTester(name, flag2);

    /* Change the handlers */
    libGAP_HDLR_FUNC(oper, 1) = hdlr ? hdlr : libGAP_DoAttribute;

    libGAP_SetupAttribute( oper, setter, tester, flag2);

    return;
}


/****************************************************************************
**

*F  DoProperty( <name> )  . . . . . . . . . . . . . . . . make a new property
*/
libGAP_Obj libGAP_SET_FILTER_OBJ;

libGAP_Obj libGAP_RESET_FILTER_OBJ;


/****************************************************************************
**
**  DoTestProperty( <prop>, <obj> )
*/
libGAP_Obj libGAP_DoTestProperty (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flags for the getter and the tester                         */
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, return 'true'        */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_True;
    }
    
    /* otherwise return 'false'                                            */
    return libGAP_False;
}


/****************************************************************************
**
**  DoSetProperty( <prop>, <obj>, <val> )
*/
libGAP_Obj libGAP_DoSetProperty (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 val )
{
    libGAP_Int                 flag1;
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flags for the getter and the tester                         */
    flag1 = libGAP_INT_INTOBJ( libGAP_FLAG1_FILT( self ) );
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, compare it           */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        if ( val == libGAP_ELM_FLAGS( flags, flag1 ) ) {
            return 0;
        }
        else {
            libGAP_ErrorReturnVoid(
                "Value property is already set the other way",
                0L, 0L,
                "you can 'return;' to set it anyhow" );
        }
    }

    /* set the value                                                       */
    /*N 1996/06/28 mschoene <self> is the <setter> here, not the <getter>! */
    /*N 1996/06/28 mschoene see hack below                                 */
    if      ( libGAP_TNUM_OBJ( obj ) == libGAP_T_COMOBJ ) {
        flags = (val == libGAP_True ? self : libGAP_TESTR_FILT(self));
        libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, flags );
    }
    else if ( libGAP_TNUM_OBJ( obj ) == libGAP_T_POSOBJ ) {
        flags = (val == libGAP_True ? self : libGAP_TESTR_FILT(self));
        libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, flags );
    }
    else if ( libGAP_TNUM_OBJ( obj ) == libGAP_T_DATOBJ ) {
        flags = (val == libGAP_True ? self : libGAP_TESTR_FILT(self));
        libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, flags );
    }
    else if ( libGAP_IS_PLIST(obj) || libGAP_IS_RANGE(obj) || libGAP_IS_STRING_REP(obj)
           || libGAP_IS_BLIST_REP(obj) )  {
        if ( val == libGAP_True ) {
            libGAP_FuncSET_FILTER_LIST( 0, obj, self );
        }
    }
    else {
        libGAP_ErrorReturnVoid(
            "Value cannot be set for internal objects",
            0L, 0L,
            "you can 'return;' without setting it" );
    }

    /* return the value                                                    */
    return 0;
}


/****************************************************************************
**
**  DoProperty( <prop>, <obj> )
*/
libGAP_Obj libGAP_DoProperty (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Int                 flag1;
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flags for the getter and the tester                         */
    flag1 = libGAP_INT_INTOBJ( libGAP_FLAG1_FILT( self ) );
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, simply return it     */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_ELM_FLAGS( flags, flag1 );
    }

    /* call the operation to compute the value                             */
    val = libGAP_DoOperation1Args( self, obj );
    while ( val != libGAP_True && val != libGAP_False ) {
        val = libGAP_ErrorReturnObj( 
               "Method for a property did not return true or false",
               0L, 0L, 
               "you can 'return true;' or 'return false;'");
    }
    
    /* set the value (but not for internal objects)                        */
    if ( libGAP_ENABLED_ATTR(self) == 1 && ! libGAP_IS_MUTABLE_OBJ(obj) ) {
        switch ( libGAP_TNUM_OBJ( obj ) ) {
        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            flags = (val == libGAP_True ? self : libGAP_TESTR_FILT(self));
            libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, flags );
        }
    }

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
**  DoVerboseProperty( <prop>, <obj> )
*/
libGAP_Obj libGAP_DoVerboseProperty (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 val;
    libGAP_Int                 flag1;
    libGAP_Int                 flag2;
    libGAP_Obj                 kind;
    libGAP_Obj                 flags;

    /* get the flags for the getter and the tester                         */
    flag1 = libGAP_INT_INTOBJ( libGAP_FLAG1_FILT( self ) );
    flag2 = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT( self ) );

    /* get kind of the object and its flags                                */
    kind  = libGAP_TYPE_OBJ_FEO( obj );
    flags = libGAP_FLAGS_TYPE( kind );

    /* if the value of the property is already known, simply return it     */
    if ( flag2 <= libGAP_LEN_FLAGS( flags ) && libGAP_ELM_FLAGS( flags, flag2 ) == libGAP_True ) {
        return libGAP_ELM_FLAGS( flags, flag1 );
    }

    /* call the operation to compute the value                             */
    val = libGAP_DoVerboseOperation1Args( self, obj );
    
    /* set the value (but not for internal objects)                        */
    if ( libGAP_ENABLED_ATTR(self) == 1 && ! libGAP_IS_MUTABLE_OBJ(obj) ) {
      if      ( libGAP_TNUM_OBJ( obj ) == libGAP_T_COMOBJ ) {
        flags = (val == libGAP_True ? self : libGAP_TESTR_FILT(self));
        libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, flags );
      }
      else if ( libGAP_TNUM_OBJ( obj ) == libGAP_T_POSOBJ ) {
        flags = (val == libGAP_True ? self : libGAP_TESTR_FILT(self));
        libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, flags );
      }
      else if ( libGAP_TNUM_OBJ( obj ) == libGAP_T_DATOBJ ) {
        flags = (val == libGAP_True ? self : libGAP_TESTR_FILT(self));
        libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, flags );
      }
    }

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  NewProperty( <name>, <narg>, <nams>, <hdlr> )
*/
libGAP_Obj libGAP_NewProperty (
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams,
    libGAP_ObjFunc             hdlr )
{
    libGAP_Obj                 getter;
    libGAP_Obj                 setter;
    libGAP_Obj                 tester;
    libGAP_Int                 flag1;
    libGAP_Int                 flag2;
    libGAP_Obj                 flags;
    libGAP_Obj                 fname;
    
    flag1 = ++libGAP_CountFlags;
    flag2 = ++libGAP_CountFlags;

    libGAP_WRAP_NAME(fname, name, "Setter");
    setter = libGAP_NewOperation( fname, 2L, 0L, libGAP_DoSetProperty );
    libGAP_FLAG1_FILT(setter)  = libGAP_INTOBJ_INT( flag1 );
    libGAP_FLAG2_FILT(setter)  = libGAP_INTOBJ_INT( flag2 );
    libGAP_CHANGED_BAG(setter);

    libGAP_WRAP_NAME(fname, name, "Tester");
    tester = libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_OPER, fname, 1L, 0L,
                           libGAP_DoTestProperty );    
    libGAP_FLAG1_FILT(tester)  = libGAP_INTOBJ_INT( flag1 );
    libGAP_FLAG2_FILT(tester)  = libGAP_INTOBJ_INT( flag2 );
    libGAP_NEW_FLAGS( flags, flag2 );
    libGAP_SET_LEN_FLAGS( flags, flag2 );
    libGAP_SET_ELM_FLAGS( flags, flag2, libGAP_True );
    libGAP_FLAGS_FILT(tester)  = flags;
    libGAP_SETTR_FILT(tester)  = 0;
    libGAP_TESTR_FILT(tester)  = libGAP_ReturnTrueFilter;
    libGAP_CHANGED_BAG(tester);

    getter = libGAP_NewOperation( name, 1L, nams, (hdlr ? hdlr : libGAP_DoProperty) ); 

    libGAP_FLAG1_FILT(getter)  = libGAP_INTOBJ_INT( flag1 );
    libGAP_FLAG2_FILT(getter)  = libGAP_INTOBJ_INT( flag2 );
    libGAP_NEW_FLAGS( flags, flag2 );
    libGAP_SET_LEN_FLAGS( flags, flag2 );
    libGAP_SET_ELM_FLAGS( flags, flag2, libGAP_True );
    libGAP_SET_ELM_FLAGS( flags, flag1, libGAP_True );
    libGAP_FLAGS_FILT(getter)  = flags;
    libGAP_SETTR_FILT(getter)  = setter;
    libGAP_TESTR_FILT(getter)  = tester;
    libGAP_SET_ENABLED_ATTR(getter,1);
    libGAP_CHANGED_BAG(getter);

    /*N 1996/06/28 mschoene bad hack see comment in <setter>               */
    libGAP_FLAGS_FILT(setter)  = flags;
    libGAP_SETTR_FILT(setter)  = setter;
    libGAP_TESTR_FILT(setter)  = tester;

    /* return the getter                                                   */
    return getter;    
}


/****************************************************************************
**

*F  DoOperationArgs( <name> ) . . . . . . . . . . . make a new operation args
*/


/****************************************************************************
**
**  DoUninstalledOperationArgs( <oper>, <args> )
*/
libGAP_Obj libGAP_DoUninstalledOperationArgs (
    libGAP_Obj                 oper,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit( "%s: function is not yet defined",
               (libGAP_Int)libGAP_CSTR_STRING(libGAP_NAME_FUNC(oper)), 0L );
    return 0;
}


/****************************************************************************
**
*F  NewOperationArgs( <name>, <nargs>, <nams> )
*/
libGAP_Obj libGAP_NewOperationArgs (
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams )
{
    libGAP_Obj                 func;

    /* create the function                                                 */
    func = libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_FUNC, name, narg, nams, 
                         libGAP_DoUninstalledOperationArgs );

    /* check the number of args                                            */
    if ( narg == -1 ) {
        libGAP_HDLR_FUNC(func,0) = libGAP_DoUninstalledOperationArgs;
        libGAP_HDLR_FUNC(func,1) = libGAP_DoUninstalledOperationArgs;
        libGAP_HDLR_FUNC(func,2) = libGAP_DoUninstalledOperationArgs;
        libGAP_HDLR_FUNC(func,3) = libGAP_DoUninstalledOperationArgs;
        libGAP_HDLR_FUNC(func,4) = libGAP_DoUninstalledOperationArgs;
        libGAP_HDLR_FUNC(func,5) = libGAP_DoUninstalledOperationArgs;
        libGAP_HDLR_FUNC(func,6) = libGAP_DoUninstalledOperationArgs;
        libGAP_HDLR_FUNC(func,7) = libGAP_DoUninstalledOperationArgs;
    }
    else {
        libGAP_ErrorQuit("number of args must be -1 in `NewOperationArgs'",0L,0L);
        return 0;
    }

    /* added the name                                                      */
    libGAP_NAME_FUNC(func) = libGAP_CopyObj( name, 0 );

    /* and return                                                          */
    return func;
}


/****************************************************************************
**
*F  InstallMethodArgs( <oper>, <func> ) . . . . . . . . . . .  clone function
**
**  There is a problem  with uncompleted functions: if  they are  cloned then
**  only   the orignal and not  the  clone will be  completed.  Therefore the
**  clone must postpone the real cloning.
*/
void libGAP_InstallMethodArgs (
    libGAP_Obj                 oper,
    libGAP_Obj                 func )
{
    libGAP_Obj                 name;
    libGAP_Int                 i;

    /* get the name                                                        */
    name = libGAP_NAME_FUNC(oper);

    /* clone the function                                                  */
    if ( libGAP_SIZE_OBJ(oper) != libGAP_SIZE_OBJ(func) ) {
        libGAP_ErrorQuit( "size mismatch of function bags", 0L, 0L );
    }

    /* clone the functions                                                 */
    else {
        for ( i = 0;  i < libGAP_SIZE_OBJ(func)/sizeof(libGAP_Obj);  i++ ) {
            libGAP_ADDR_OBJ(oper)[i] = libGAP_ADDR_OBJ(func)[i];
        }
    }
    libGAP_NAME_FUNC(oper) = name;
    libGAP_CHANGED_BAG(oper);
}


/****************************************************************************
**

*F  SaveOperationExtras( <oper> ) . . . additional saving for functions which
**
**  This is called by SaveFunction when the function bag is too large to be
**  a simple function, and so must be an operation
**
*/
void libGAP_SaveOperationExtras (
    libGAP_Obj         oper )
{
    libGAP_UInt        i;

    libGAP_SaveSubObj(libGAP_FLAG1_FILT(oper));
    libGAP_SaveSubObj(libGAP_FLAG2_FILT(oper));
    libGAP_SaveSubObj(libGAP_FLAGS_FILT(oper));
    libGAP_SaveSubObj(libGAP_SETTR_FILT(oper));
    libGAP_SaveSubObj(libGAP_TESTR_FILT(oper));
    libGAP_SaveUInt(libGAP_ENABLED_ATTR(oper));
    for (i = 0; i <= 7; i++)
        libGAP_SaveSubObj(libGAP_METHS_OPER(oper,i));
    for (i = 0; i <= 7; i++)
        libGAP_SaveSubObj(libGAP_CACHE_OPER(oper,i));
    return;
}


/****************************************************************************
**
*F  LoadOperationExtras( <oper> ) . .  additional loading for functions which
**                                     are operations
**  This is called by LoadFunction when the function bag is too large to be
**  a simple function, and so must be an operation
**
*/
void libGAP_LoadOperationExtras (
    libGAP_Obj         oper )
{
    libGAP_UInt        i;

    libGAP_FLAG1_FILT(oper) = libGAP_LoadSubObj();
    libGAP_FLAG2_FILT(oper) = libGAP_LoadSubObj();
    libGAP_FLAGS_FILT(oper) = libGAP_LoadSubObj();
    libGAP_SETTR_FILT(oper) = libGAP_LoadSubObj();
    libGAP_TESTR_FILT(oper) = libGAP_LoadSubObj();
    i = libGAP_LoadUInt();
    libGAP_SET_ENABLED_ATTR(oper,i);
    for (i = 0; i <= 7; i++)
        libGAP_METHS_OPER(oper,i) = libGAP_LoadSubObj();
    for (i = 0; i <= 7; i++)
        libGAP_CACHE_OPER(oper,i) = libGAP_LoadSubObj();
    return;
}


/****************************************************************************
**
**

*F * * * * * * * * * * * * GAP operation functions  * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  FuncNEW_OPERATION( <self>, <name> ) . . . . . . . . . . . . new operation
*/
libGAP_Obj libGAP_FuncNEW_OPERATION (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    if ( ! libGAP_IsStringConv(name) ) {
        libGAP_ErrorQuit("usage: NewOperation( <name> )",0L,0L);
        return 0;
    }

    /* make the new operation                                              */
    return libGAP_NewOperation( name, -1L, (libGAP_Obj)0, libGAP_DoOperationXArgs );
}


/****************************************************************************
**
*F  FuncNEW_CONSTRUCTOR( <self>, <name> ) . . . . . . . . . . new constructor
*/
libGAP_Obj libGAP_FuncNEW_CONSTRUCTOR (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    if ( ! libGAP_IsStringConv(name) ) {
        libGAP_ErrorQuit("usage: NewConstructor( <name> )",0L,0L);
        return 0;
    }

    /* make the new constructor                                            */
    return libGAP_NewConstructor( name, -1L, (libGAP_Obj)0, libGAP_DoConstructorXArgs );
}


/****************************************************************************
**
*F  FuncNEW_ATTRIBUTE( <self>, <name> ) . . . . . . . . . . . . new attribute
*/
libGAP_Obj libGAP_FuncNEW_ATTRIBUTE (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    if ( ! libGAP_IsStringConv(name) ) {
        libGAP_ErrorQuit("usage: NewAttribute( <name> )",0L,0L);
        return 0;
    }

    /* make the new operation                                              */
    return libGAP_NewAttribute( name, -1L, (libGAP_Obj)0, libGAP_DoAttribute );
}
/****************************************************************************
**
*F  FuncOPER_TO_ATTRIBUTE( <self>, oper ) make existing operation into attribute
*/
libGAP_Obj libGAP_FuncOPER_TO_ATTRIBUTE (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    /* check the argument                                                  */
  if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("usage: OPER_TO_ATTRIBUTE( <oper> )",0L,0L);
        return 0;
    }

    /* make the new operation                                              */
  libGAP_ConvertOperationIntoAttribute( oper, (libGAP_ObjFunc) 0L );
    return (libGAP_Obj) 0L;
}

/****************************************************************************
**
*F  FuncOPER_TO_MUTABLE_ATTRIBUTE( <self>, oper ) make existing operation into attribute
*/
libGAP_Obj libGAP_FuncOPER_TO_MUTABLE_ATTRIBUTE (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    /* check the argument                                                  */
  if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("usage: OPER_TO_MUTABLE_ATTRIBUTE( <oper> )",0L,0L);
        return 0;
    }

    /* make the new operation                                              */
  libGAP_ConvertOperationIntoAttribute( oper, libGAP_DoMutableAttribute );
  return (libGAP_Obj) 0L;
}


/****************************************************************************
**
*F  FuncNEW_MUTABLE_ATTRIBUTE( <self>, <name> ) . . . . new mutable attribute
*/
libGAP_Obj libGAP_FuncNEW_MUTABLE_ATTRIBUTE (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    if ( ! libGAP_IsStringConv(name) ) {
        libGAP_ErrorQuit("usage: NewMutableAttribute( <name> )",0L,0L);
        return 0;
    }

    /* make the new operation                                              */
    return libGAP_NewAttribute( name, -1L, (libGAP_Obj)0, libGAP_DoMutableAttribute );
}


/****************************************************************************
**
*F  FuncNEW_PROPERTY( <self>, <name> )  . . . . . . . . . . . .  new property
*/
libGAP_Obj libGAP_FuncNEW_PROPERTY (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    /* check the argument                                                  */
    if ( ! libGAP_IsStringConv(name) ) {
        libGAP_ErrorQuit("usage: NewProperty( <name> )",0L,0L);
        return 0;
    }

    /* make the new operation                                              */
    return libGAP_NewProperty( name, -1L, (libGAP_Obj)0, libGAP_DoProperty );
}


/****************************************************************************
**
*F  FuncNEW_OPERATION_ARGS( <self>, <name> )  . . . . . .  new operation args
*/
libGAP_Obj libGAP_FuncNEW_OPERATION_ARGS (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    libGAP_Obj                 args;           
    libGAP_Obj                 list;

    /* check the argument                                                  */
    if ( ! libGAP_IsStringConv(name) ) {
        libGAP_ErrorQuit( "usage: NewOperationArgs( <name> )", 0L, 0L );
        return 0;
    }

    /* make the new operation                                              */
    libGAP_C_NEW_STRING_CONST( args, "args" )
    list = libGAP_NEW_PLIST( libGAP_T_PLIST, 1 );
    libGAP_SET_LEN_PLIST( list, 1 );
    libGAP_SET_ELM_PLIST( list, 1, args );
    return libGAP_NewOperationArgs( name, -1, list );
}


/****************************************************************************
**
*F  FuncINSTALL_METHOD_ARGS( <self>, <oper>, <func> ) . . install method args
*/
static libGAP_Obj libGAP_REREADING;

libGAP_Obj libGAP_FuncINSTALL_METHOD_ARGS (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 func )
{
    /* check the arguments                                                 */
    if ( ! libGAP_IS_FUNC(oper) ) {
        libGAP_ErrorQuit( "<oper> must be a function (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(oper), 0L );
    }
    if ( (libGAP_REREADING != libGAP_True) &&
         (libGAP_HDLR_FUNC(oper,0) != (libGAP_ObjFunc)libGAP_DoUninstalledOperationArgs) ) {
        libGAP_ErrorQuit( "operation already installed",
                   0L, 0L );
        return 0;
    }
    if ( ! libGAP_IS_FUNC(func) ) {
        libGAP_ErrorQuit( "<func> must be a function (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(func), 0L );
        return 0;
    }
    if ( libGAP_IS_OPERATION(func) ) {
        libGAP_ErrorQuit( "<func> must not be an operation", 0L, 0L );
        return 0;
    }

    /* install the new method                                              */
    libGAP_InstallMethodArgs( oper, func );
    return 0;
}


/****************************************************************************
**

*F  FuncIS_OPERATION( <self>, <obj> ) . . . . . . . . . is <obj> an operation
*/
libGAP_Obj libGAP_IsOperationFilt;

libGAP_Obj libGAP_FuncIS_OPERATION (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    if ( libGAP_TNUM_OBJ(obj) == libGAP_T_FUNCTION && libGAP_IS_OPERATION(obj) ) {
        return libGAP_True;
    }
    else if ( libGAP_TNUM_OBJ(obj) < libGAP_FIRST_EXTERNAL_TNUM ) {
        return libGAP_False;
    }
    else {
        return libGAP_DoFilter( self, obj );
    }
}


/****************************************************************************
**
*F  FuncMETHODS_OPERATION( <self>, <oper>, <narg> ) . . . . .  list of method
*/
libGAP_Obj libGAP_MethsOper (
    libGAP_Obj                 oper,
    libGAP_UInt                i )
{
    libGAP_Obj                 methods;
    methods = libGAP_METHS_OPER( oper, i );
    if ( methods == 0 ) {
        methods = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
        libGAP_METHS_OPER( oper, i ) = methods;
        libGAP_CHANGED_BAG( oper );
    }
    return methods;
}

libGAP_Obj libGAP_FuncMETHODS_OPERATION (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 narg )
{
    libGAP_Int                 n;
    libGAP_Obj                 meth;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    if ( libGAP_TNUM_OBJ(narg) != libGAP_T_INT || libGAP_INT_INTOBJ(narg) < 0 ) {
        libGAP_ErrorQuit("<narg> must be a nonnegative integer",0L,0L);
        return 0;
    }
    n = libGAP_INT_INTOBJ( narg );
    meth = libGAP_MethsOper( oper, (libGAP_UInt)n );
    return meth == 0 ? libGAP_Fail : meth;
}


/****************************************************************************
**
*F  FuncCHANGED_METHODS_OPERATION( <self>, <oper>, <narg> ) . . . clear cache
*/
libGAP_Obj libGAP_FuncCHANGED_METHODS_OPERATION (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 narg )
{
    libGAP_Obj *               cache;
    libGAP_Int                 n;
    libGAP_Int                 i;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    if ( libGAP_TNUM_OBJ(narg) != libGAP_T_INT || libGAP_INT_INTOBJ(narg) < 0 ) {
        libGAP_ErrorQuit("<narg> must be a nonnegative integer",0L,0L);
        return 0;
    }
    n = libGAP_INT_INTOBJ( narg );
    cache = libGAP_ADDR_OBJ( libGAP_CacheOper( oper, (libGAP_UInt) n ) );
    for ( i = 0;  i < libGAP_SIZE_OBJ(libGAP_CACHE_OPER(oper,n)) / sizeof(libGAP_Obj);  i++ ) {
        cache[i] = 0;
    }
    return 0;
}


/****************************************************************************
**
*F  FuncSET_METHODS_OPERATION( <self>, <oper>, <narg>, <list> ) . set methods
*/
libGAP_Obj libGAP_FuncSET_METHODS_OPERATION (
    libGAP_Obj                 self,
    libGAP_Obj                 oper,
    libGAP_Obj                 narg,
    libGAP_Obj                 meths )
{
    libGAP_Int                 n;

    if ( ! libGAP_IS_OPERATION(oper) ) {
        libGAP_ErrorQuit("<oper> must be an operation",0L,0L);
        return 0;
    }
    if ( libGAP_TNUM_OBJ(narg) != libGAP_T_INT || libGAP_INT_INTOBJ(narg) < 0 ) {
        libGAP_ErrorQuit("<narg> must be a nonnegative integer",0L,0L);
        return 0;
    }
    n = libGAP_INT_INTOBJ( narg );
    libGAP_METHS_OPER( oper, n ) = meths;
    return 0;
}


/****************************************************************************
**

*F  FuncSETTER_FUNCTION( <self>, <name> ) . . . . . . default attribut setter
*/
libGAP_Obj libGAP_DoSetterFunction (
    libGAP_Obj                 self,
    libGAP_Obj                 obj,
    libGAP_Obj                 value )
{
    libGAP_Obj                 tmp;
    libGAP_Obj                 tester;
    libGAP_Obj                 flags;
    libGAP_UInt                flag2;
    libGAP_Obj                 kind;

    if ( libGAP_TNUM_OBJ(obj) != libGAP_T_COMOBJ ) {
        libGAP_ErrorQuit( "<obj> must be an component object", 0L, 0L );
        return 0L;
    }

    /* if the attribute is already there *do not* chage it                 */
    tmp = libGAP_ENVI_FUNC(self);
    tester = libGAP_ELM_PLIST( tmp, 2 );
    flag2  = libGAP_INT_INTOBJ( libGAP_FLAG2_FILT(tester) );
    kind   = libGAP_TYPE_OBJ_FEO(obj);
    flags  = libGAP_FLAGS_TYPE(kind);
    if ( flag2 <= libGAP_LEN_FLAGS(flags) && libGAP_ELM_FLAGS(flags,flag2) == libGAP_True ) {
        return 0;
    }

    /* set the value                                                       */
    libGAP_AssPRec( obj, (libGAP_UInt)libGAP_INT_INTOBJ(libGAP_ELM_PLIST(tmp,1)), libGAP_CopyObj(value,0) );
    libGAP_CALL_2ARGS( libGAP_SET_FILTER_OBJ, obj, tester );
    return 0;
}


libGAP_Obj libGAP_FuncSETTER_FUNCTION (
    libGAP_Obj                 self,
    libGAP_Obj                 name,
    libGAP_Obj                 filter )
{
    libGAP_Obj                 func;
    libGAP_Obj                 fname;
    libGAP_Obj                 tmp;

    libGAP_WRAP_NAME(fname, name, "SetterFunc");
    func = libGAP_NewFunctionCT( libGAP_T_FUNCTION, libGAP_SIZE_FUNC, libGAP_CSTR_STRING(fname), 2,
                         "object, value", libGAP_DoSetterFunction );
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( tmp, 2 );
    libGAP_SET_ELM_PLIST( tmp, 1, libGAP_INTOBJ_INT( libGAP_RNamObj(name) ) );
    libGAP_SET_ELM_PLIST( tmp, 2, filter );
    libGAP_CHANGED_BAG(tmp);
    libGAP_ENVI_FUNC(func) = tmp;
    libGAP_CHANGED_BAG(func);
    return func;
}


/****************************************************************************
**
*F  FuncGETTER_FUNCTION( <self>, <name> ) . . . . . . default attribut getter
*/
libGAP_Obj libGAP_DoGetterFunction (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    if ( libGAP_TNUM_OBJ(obj) != libGAP_T_COMOBJ ) {
        libGAP_ErrorQuit( "<obj> must be an component object", 0L, 0L );
        return 0L;
    }
    return libGAP_ElmPRec( obj, (libGAP_UInt)libGAP_INT_INTOBJ(libGAP_ENVI_FUNC(self)) );
}


libGAP_Obj libGAP_FuncGETTER_FUNCTION (
    libGAP_Obj                 self,
    libGAP_Obj                 name )
{
    libGAP_Obj                 func;
    libGAP_Obj                 fname;

    libGAP_WRAP_NAME(fname, name, "GetterFunc");
    func = libGAP_NewFunctionCT( libGAP_T_FUNCTION, libGAP_SIZE_FUNC, libGAP_CSTR_STRING(fname), 1,
                         "object, value", libGAP_DoGetterFunction );
    libGAP_ENVI_FUNC(func) = libGAP_INTOBJ_INT( libGAP_RNamObj(name) );
    return func;
}


/****************************************************************************
**

*F  FuncOPERS_CACHE_INFO( <self> )  . . . . . . .  return cache stats as list
*/
libGAP_Obj libGAP_FuncOPERS_CACHE_INFO (
    libGAP_Obj                        self )
{
    libGAP_Obj                 list;

    list = libGAP_NEW_PLIST( libGAP_IMMUTABLE_TNUM(libGAP_T_PLIST), 9 );
    libGAP_SET_LEN_PLIST( list, 9 );
    libGAP_SET_ELM_PLIST( list, 1, libGAP_INTOBJ_INT(libGAP_AndFlagsCacheHit)    );
    libGAP_SET_ELM_PLIST( list, 2, libGAP_INTOBJ_INT(libGAP_AndFlagsCacheMiss)   );
    libGAP_SET_ELM_PLIST( list, 3, libGAP_INTOBJ_INT(libGAP_AndFlagsCacheLost)   );
    libGAP_SET_ELM_PLIST( list, 4, libGAP_INTOBJ_INT(libGAP_OperationHit)        );
    libGAP_SET_ELM_PLIST( list, 5, libGAP_INTOBJ_INT(libGAP_OperationMiss)       );
    libGAP_SET_ELM_PLIST( list, 6, libGAP_INTOBJ_INT(libGAP_IsSubsetFlagsCalls)  );
    libGAP_SET_ELM_PLIST( list, 7, libGAP_INTOBJ_INT(libGAP_IsSubsetFlagsCalls1) );
    libGAP_SET_ELM_PLIST( list, 8, libGAP_INTOBJ_INT(libGAP_IsSubsetFlagsCalls2) );
    libGAP_SET_ELM_PLIST( list, 9, libGAP_INTOBJ_INT(libGAP_OperationNext)       );

    return list;
}



/****************************************************************************
**
*F  FuncCLEAR_CACHE_INFO( <self> )  . . . . . . . . . . . . clear cache stats
*/
libGAP_Obj libGAP_FuncCLEAR_CACHE_INFO (
    libGAP_Obj                        self )
{
    libGAP_AndFlagsCacheHit = 0;
    libGAP_AndFlagsCacheMiss = 0;
    libGAP_AndFlagsCacheLost = 0;
    libGAP_OperationHit = 0;
    libGAP_OperationMiss = 0;
    libGAP_IsSubsetFlagsCalls = 0;
    libGAP_IsSubsetFlagsCalls1 = 0;
    libGAP_IsSubsetFlagsCalls2 = 0;
    libGAP_OperationNext = 0;

    return 0;
}

/****************************************************************************
**

*F  ChangeDoOperations( <oper>, <verb> )  . . .  verbose or silent operations
*/
static libGAP_ObjFunc libGAP_TabSilentVerboseOperations[] =
{
    (libGAP_ObjFunc) libGAP_DoOperation0Args,   (libGAP_ObjFunc) libGAP_DoVerboseOperation0Args,
    (libGAP_ObjFunc) libGAP_DoOperation1Args,   (libGAP_ObjFunc) libGAP_DoVerboseOperation1Args,
    (libGAP_ObjFunc) libGAP_DoOperation2Args,   (libGAP_ObjFunc) libGAP_DoVerboseOperation2Args,
    (libGAP_ObjFunc) libGAP_DoOperation3Args,   (libGAP_ObjFunc) libGAP_DoVerboseOperation3Args,
    (libGAP_ObjFunc) libGAP_DoOperation4Args,   (libGAP_ObjFunc) libGAP_DoVerboseOperation4Args,
    (libGAP_ObjFunc) libGAP_DoOperation5Args,   (libGAP_ObjFunc) libGAP_DoVerboseOperation5Args,
    (libGAP_ObjFunc) libGAP_DoOperation6Args,   (libGAP_ObjFunc) libGAP_DoVerboseOperation6Args,
    (libGAP_ObjFunc) libGAP_DoOperationXArgs,   (libGAP_ObjFunc) libGAP_DoVerboseOperationXArgs,
    (libGAP_ObjFunc) libGAP_DoConstructor0Args, (libGAP_ObjFunc) libGAP_DoVerboseConstructor0Args,
    (libGAP_ObjFunc) libGAP_DoConstructor1Args, (libGAP_ObjFunc) libGAP_DoVerboseConstructor1Args,
    (libGAP_ObjFunc) libGAP_DoConstructor2Args, (libGAP_ObjFunc) libGAP_DoVerboseConstructor2Args,
    (libGAP_ObjFunc) libGAP_DoConstructor3Args, (libGAP_ObjFunc) libGAP_DoVerboseConstructor3Args,
    (libGAP_ObjFunc) libGAP_DoConstructor4Args, (libGAP_ObjFunc) libGAP_DoVerboseConstructor4Args,
    (libGAP_ObjFunc) libGAP_DoConstructor5Args, (libGAP_ObjFunc) libGAP_DoVerboseConstructor5Args,
    (libGAP_ObjFunc) libGAP_DoConstructor6Args, (libGAP_ObjFunc) libGAP_DoVerboseConstructor6Args,
    (libGAP_ObjFunc) libGAP_DoConstructorXArgs, (libGAP_ObjFunc) libGAP_DoVerboseConstructorXArgs,
    (libGAP_ObjFunc) libGAP_DoAttribute,        (libGAP_ObjFunc) libGAP_DoVerboseAttribute,
    (libGAP_ObjFunc) libGAP_DoMutableAttribute, (libGAP_ObjFunc) libGAP_DoVerboseMutableAttribute,
    (libGAP_ObjFunc) libGAP_DoProperty,         (libGAP_ObjFunc) libGAP_DoVerboseProperty,
    0,                          0
};


void libGAP_ChangeDoOperations (
    libGAP_Obj                 oper,
    libGAP_Int                 verb )
{
    libGAP_Int                 i;
    libGAP_Int                 j;

    /* be verbose                                                          */
    if ( verb ) {

        /* catch infix operations                                          */
        if ( oper == libGAP_EqOper   )  { libGAP_InstallEqObject(1);   }
        if ( oper == libGAP_LtOper   )  { libGAP_InstallLtObject(1);   }
        if ( oper == libGAP_InOper   )  { libGAP_InstallInObject(1);   }
        if ( oper == libGAP_SumOper  )  { libGAP_InstallSumObject(1);  }
        if ( oper == libGAP_DiffOper )  { libGAP_InstallDiffObject(1); }
        if ( oper == libGAP_ProdOper )  { libGAP_InstallProdObject(1); }
        if ( oper == libGAP_QuoOper  )  { libGAP_InstallQuoObject(1);  }
        if ( oper == libGAP_LQuoOper )  { libGAP_InstallLQuoObject(1); }
        if ( oper == libGAP_PowOper  )  { libGAP_InstallPowObject(1);  }
        if ( oper == libGAP_CommOper )  { libGAP_InstallCommObject(1); }
        if ( oper == libGAP_ModOper  )  { libGAP_InstallModObject(1);  }
        if ( oper == libGAP_InvOp  )  { libGAP_InstallInvObject(1);  }
        if ( oper == libGAP_OneOp  )  { libGAP_InstallOneObject(1);  }
        if ( oper == libGAP_AInvOp )  { libGAP_InstallAinvObject(1); } 
        if ( oper == libGAP_ZeroOp )  { libGAP_InstallZeroObject(1); }

        /* switch do with do verbose                                       */
        for ( j = 0;  libGAP_TabSilentVerboseOperations[j];  j += 2 ) {
            for ( i = 0;  i <= 7;  i++ ) {
                if ( libGAP_HDLR_FUNC(oper,i) == libGAP_TabSilentVerboseOperations[j] ) {
                    libGAP_HDLR_FUNC(oper,i) = libGAP_TabSilentVerboseOperations[j+1];
                }
            }
        }
    }

    /* be silent                                                           */
    else {

        /* catch infix operations                                          */
        if ( oper == libGAP_EqOper   )  { libGAP_InstallEqObject(0);   }
        if ( oper == libGAP_LtOper   )  { libGAP_InstallLtObject(0);   }
        if ( oper == libGAP_InOper   )  { libGAP_InstallInObject(0);   }
        if ( oper == libGAP_SumOper  )  { libGAP_InstallSumObject(0);  }
        if ( oper == libGAP_DiffOper )  { libGAP_InstallDiffObject(0); }
        if ( oper == libGAP_ProdOper )  { libGAP_InstallProdObject(0); }
        if ( oper == libGAP_QuoOper  )  { libGAP_InstallQuoObject(0);  }
        if ( oper == libGAP_LQuoOper )  { libGAP_InstallLQuoObject(0); }
        if ( oper == libGAP_PowOper  )  { libGAP_InstallPowObject(0);  }
        if ( oper == libGAP_CommOper )  { libGAP_InstallCommObject(0); }
        if ( oper == libGAP_ModOper  )  { libGAP_InstallModObject(0); }
        if ( oper == libGAP_InvOp  )  { libGAP_InstallInvObject(0);  }
        if ( oper == libGAP_OneOp  )  { libGAP_InstallOneObject(0);  }
        if ( oper == libGAP_AInvOp )  { libGAP_InstallAinvObject(0); }
        if ( oper == libGAP_ZeroOp )  { libGAP_InstallZeroObject(0); }

        /* switch do verbose with do                                       */
        for ( j = 1;  libGAP_TabSilentVerboseOperations[j-1];  j += 2 ) {
            for ( i = 0;  i <= 7;  i++ ) {
                if ( libGAP_HDLR_FUNC(oper,i) == libGAP_TabSilentVerboseOperations[j] ) {
                    libGAP_HDLR_FUNC(oper,i) = libGAP_TabSilentVerboseOperations[j-1];
                }
            }
        }
    }
}


/****************************************************************************
**
*F  FuncTRACE_METHODS( <oper> ) . . . . . . . .  switch tracing of methods on
*/
libGAP_Obj libGAP_FuncTRACE_METHODS (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{
    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(oper) != libGAP_T_FUNCTION || libGAP_SIZE_OBJ(oper) != libGAP_SIZE_OPER ) {
        libGAP_ErrorQuit( "<oper> must be an operation", 0L, 0L );
        return 0;
    }

    /* install trace handler                                               */
    libGAP_ChangeDoOperations( oper, 1 );

    /* return nothing                                                      */
    return 0;
}


/****************************************************************************
**
*F  FuncUNTRACE_METHODS( <oper> ) . . . . . . . switch tracing of methods off
*/
libGAP_Obj libGAP_FuncUNTRACE_METHODS (
    libGAP_Obj                 self,
    libGAP_Obj                 oper )
{

    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(oper) != libGAP_T_FUNCTION || libGAP_SIZE_OBJ(oper) != libGAP_SIZE_OPER ) {
        libGAP_ErrorQuit( "<oper> must be an operation", 0L, 0L );
        return 0;
    }

    /* install trace handler                                               */
    libGAP_ChangeDoOperations( oper, 0 );

    /* return nothing                                                      */
    return 0;
}

/****************************************************************************
**
*F  FuncSET_ATTRIBUTE_STORING( <self>, <attr>, <val> )
**               switch off or on the setter call of an attribute
*/
libGAP_Obj libGAP_FuncSET_ATTRIBUTE_STORING (
    libGAP_Obj                 self,
    libGAP_Obj                 attr,
    libGAP_Obj                 val )
{
  libGAP_SET_ENABLED_ATTR(attr, (val == libGAP_True) ? 1L : 0L);
  return 0;
}

/****************************************************************************
**
*F  FuncDO_NOTHING_SETTER(<self> , <obj>, <val> )
**
*/
libGAP_Obj libGAP_FuncDO_NOTHING_SETTER( libGAP_Obj self, libGAP_Obj obj, libGAP_Obj val)
{
  return 0;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_OPERATION", "obj", &libGAP_IsOperationFilt,
      libGAP_FuncIS_OPERATION, "src/opers.c:IS_OPERATION" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "AND_FLAGS", 2, "oper1, oper2",
      libGAP_FuncAND_FLAGS, "src/opers.c:AND_FLAGS" },

    { "SUB_FLAGS", 2, "oper1, oper2",
      libGAP_FuncSUB_FLAGS, "src/opers.c:SUB_FLAGS" },

    { "HASH_FLAGS", 1, "flags",
      libGAP_FuncHASH_FLAGS, "src/opers.c:HASH_FLAGS" },

    { "IS_EQUAL_FLAGS", 2, "flags1, flags2",
      libGAP_FuncIS_EQUAL_FLAGS, "src/opers.c:IS_EQUAL_FLAGS" },

    { "IS_SUBSET_FLAGS", 2, "flags1, flags2",
      libGAP_FuncIS_SUBSET_FLAGS, "src/opers.c:IS_SUBSET_FLAGS" },

    { "TRUES_FLAGS", 1, "flags",
      libGAP_FuncTRUES_FLAGS, "src/opers.c:TRUES_FLAGS" },

    { "SIZE_FLAGS", 1, "flags",
      libGAP_FuncSIZE_FLAGS, "src/opers.c:SIZE_FLAGS" },

    { "LEN_FLAGS", 1, "flags",
      libGAP_FuncLEN_FLAGS, "src/opers.c:LEN_FLAGS" },

    { "ELM_FLAGS", 2, "flags, pos",
      libGAP_FuncELM_FLAGS, "src/opers.c:ELM_FLAGS" },

    { "FLAG1_FILTER", 1, "oper",
      libGAP_FuncFLAG1_FILTER, "src/opers.c:FLAG1_FILTER" },

    { "SET_FLAG1_FILTER", 2, "oper, flag1",
      libGAP_FuncSET_FLAG1_FILTER, "src/opers.c:SET_FLAG1_FILTER" },

    { "FLAG2_FILTER", 1, "oper",
      libGAP_FuncFLAG2_FILTER, "src/opers.c:FLAG2_FILTER" },

    { "SET_FLAG2_FILTER", 2, "oper, flag2",
      libGAP_FuncSET_FLAG2_FILTER, "src/opers.c:SET_FLAG2_FILTER" },

    { "FLAGS_FILTER", 1, "oper",
      libGAP_FuncFLAGS_FILTER, "src/opers.c:FLAGS_FILTER" },

    { "SET_FLAGS_FILTER", 2, "oper, flags",
      libGAP_FuncSET_FLAGS_FILTER, "src/opers.c:SET_FLAGS_FILTER" },

    { "SETTER_FILTER", 1, "oper",
      libGAP_FuncSETTER_FILTER, "src/opers.c:SETTER_FILTER" },

    { "SET_SETTER_FILTER", 2, "oper, other",
      libGAP_FuncSET_SETTER_FILTER, "src/opers.c:SET_SETTER_FILTER" },

    { "TESTER_FILTER", 1, "oper",
      libGAP_FuncTESTER_FILTER, "src/opers.c:TESTER_FILTER" },

    { "SET_TESTER_FILTER", 2, "oper, other",
      libGAP_FuncSET_TESTER_FILTER, "src/opers.c:SET_TESTER_FILTER" },

    { "METHODS_OPERATION", 2, "oper, narg",
      libGAP_FuncMETHODS_OPERATION, "src/opers.c:METHODS_OPERATION" },

    { "SET_METHODS_OPERATION", 3, "oper, narg, meths",
      libGAP_FuncSET_METHODS_OPERATION, "src/opers.c:SET_METHODS_OPERATION" },

    { "CHANGED_METHODS_OPERATION", 2, "oper, narg",
      libGAP_FuncCHANGED_METHODS_OPERATION, "src/opers.c:CHANGED_METHODS_OPERATION" },

    { "NEW_FILTER", 1, "name",
      libGAP_FuncNEW_FILTER, "src/opers.c:NEW_FILTER" },

    { "NEW_OPERATION", 1, "name",
      libGAP_FuncNEW_OPERATION, "src/opers.c:NEW_OPERATION" },

    { "NEW_CONSTRUCTOR", 1, "name",
      libGAP_FuncNEW_CONSTRUCTOR, "src/opers.c:NEW_CONSTRUCTOR" },

    { "NEW_ATTRIBUTE", 1, "name",
      libGAP_FuncNEW_ATTRIBUTE, "src/opers.c:NEW_ATTRIBUTE" },

    { "NEW_MUTABLE_ATTRIBUTE", 1, "name",
      libGAP_FuncNEW_MUTABLE_ATTRIBUTE, "src/opers.c:NEW_MUTABLE_ATTRIBUTE" },

    { "NEW_PROPERTY", 1, "name",
      libGAP_FuncNEW_PROPERTY, "src/opers.c:NEW_PROPERTY" },

    { "SETTER_FUNCTION", 2, "name, filter",
      libGAP_FuncSETTER_FUNCTION, "src/opers.c:SETTER_FUNCTION" },

    { "GETTER_FUNCTION", 1, "name",
      libGAP_FuncGETTER_FUNCTION, "src/opers.c:GETTER_FUNCTION" },

    { "NEW_OPERATION_ARGS", 1, "name",
      libGAP_FuncNEW_OPERATION_ARGS, "src/opers.c:NEW_OPERATION_ARGS" },

    { "INSTALL_METHOD_ARGS", 2, "oper, func",
      libGAP_FuncINSTALL_METHOD_ARGS, "src/opers.c:INSTALL_METHOD_ARGS" },

    { "TRACE_METHODS", 1, "oper",
      libGAP_FuncTRACE_METHODS, "src/opers.c:TRACE_METHODS" },

    { "UNTRACE_METHODS", 1, "oper",
      libGAP_FuncUNTRACE_METHODS, "src/opers.c:UNTRACE_METHODS" },

    { "OPERS_CACHE_INFO", 0, "",
      libGAP_FuncOPERS_CACHE_INFO, "src/opers.c:OPERS_CACHE_INFO" },

    { "CLEAR_CACHE_INFO", 0, "",
      libGAP_FuncCLEAR_CACHE_INFO, "src/opers.c:CLEAR_CACHE_INFO" },
    
    { "SET_ATTRIBUTE_STORING", 2, "attr, val",
      libGAP_FuncSET_ATTRIBUTE_STORING, "src/opers.c:SET_ATTRIBUTE_STORING" },

    { "DO_NOTHING_SETTER", 2, "obj, val",
      libGAP_FuncDO_NOTHING_SETTER, "src/opers.c:DO_NOTHING_SETTER" },

    { "IS_AND_FILTER", 1, "filter",
      libGAP_FuncIS_AND_FILTER, "src/opers.c:IS_AND_FILTER" },

    { "COMPACT_TYPE_IDS", 0, "",
      libGAP_FuncCompactTypeIDs, "src/opers.c:COMPACT_TYPE_IDS" },

    { "OPER_TO_ATTRIBUTE", 1, "oper",
      libGAP_FuncOPER_TO_ATTRIBUTE, "src/opers.c:OPER_TO_ATTRIBUTE" },

    { "OPER_TO_MUTABLE_ATTRIBUTE", 1, "oper",
      libGAP_FuncOPER_TO_MUTABLE_ATTRIBUTE, "src/opers.c:OPER_TO_MUTABLE_ATTRIBUTE" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{

  libGAP_NextTypeID = 0;
  libGAP_CountFlags = 0;
    libGAP_InitGlobalBag( &libGAP_StringAndFilter,    "src/opers.c:StringAndFilter"    );
    libGAP_InitGlobalBag( &libGAP_StringFilterSetter, "src/opers.c:StringFilterSetter" );
    libGAP_InitGlobalBag( &libGAP_ArglistObj,         "src/opers.c:ArglistObj"         );
    libGAP_InitGlobalBag( &libGAP_ArglistObjVal,      "src/opers.c:ArglistObjVal"      );

    /* Declare the handlers used in various places.  Some of the commonest */
    /* ones are abbreviated to save space in saved workspace.              */
    libGAP_InitHandlerFunc( libGAP_DoFilter,                  "df"                                    );
    libGAP_InitHandlerFunc( libGAP_DoSetFilter,               "dsf"                                   );
    libGAP_InitHandlerFunc( libGAP_DoAndFilter,               "daf"                                   );
    libGAP_InitHandlerFunc( libGAP_DoSetAndFilter,            "dsaf"                                  );
    libGAP_InitHandlerFunc( libGAP_DoReturnTrueFilter,        "src/opers.c:DoReturnTrueFilter"        );
    libGAP_InitHandlerFunc( libGAP_DoSetReturnTrueFilter,     "src/opers.c:DoSetReturnTrueFilter"     );
    
    libGAP_InitHandlerFunc( libGAP_DoAttribute,               "da"                                    );
    libGAP_InitHandlerFunc( libGAP_DoSetAttribute,            "dsa"                                   );
    libGAP_InitHandlerFunc( libGAP_DoTestAttribute,           "src/opers.c:DoTestAttribute"           );
    libGAP_InitHandlerFunc( libGAP_DoVerboseAttribute,        "src/opers.c:DoVerboseAttribute"        );
    libGAP_InitHandlerFunc( libGAP_DoMutableAttribute,        "src/opers.c:DoMutableAttribute"        );
    libGAP_InitHandlerFunc( libGAP_DoVerboseMutableAttribute, "src/opers.c:DoVerboseMutableAttribute" );

    libGAP_InitHandlerFunc( libGAP_DoProperty,                "src/opers.c:DoProperty"                );
    libGAP_InitHandlerFunc( libGAP_DoSetProperty,             "src/opers.c:DoSetProperty"             );
    libGAP_InitHandlerFunc( libGAP_DoTestProperty,            "src/opers.c:DoTestProperty"            );
    libGAP_InitHandlerFunc( libGAP_DoVerboseProperty,         "src/opers.c:DoVerboseProperty"         );

    libGAP_InitHandlerFunc( libGAP_DoSetterFunction,          "dtf"                                   );
    libGAP_InitHandlerFunc( libGAP_DoGetterFunction,          "dgf"                                   );
    
    libGAP_InitHandlerFunc( libGAP_DoOperation0Args,          "o0"                                    );
    libGAP_InitHandlerFunc( libGAP_DoOperation1Args,          "o1"                                    );
    libGAP_InitHandlerFunc( libGAP_DoOperation2Args,          "o2"                                    );
    libGAP_InitHandlerFunc( libGAP_DoOperation3Args,          "o3"                                    );
    libGAP_InitHandlerFunc( libGAP_DoOperation4Args,          "o4"                                    );
    libGAP_InitHandlerFunc( libGAP_DoOperation5Args,          "o5"                                    );
    libGAP_InitHandlerFunc( libGAP_DoOperation6Args,          "o6"                                    );
    libGAP_InitHandlerFunc( libGAP_DoOperationXArgs,          "o7"                                    );

    libGAP_InitHandlerFunc( libGAP_DoVerboseOperation0Args,   "src/opers.c:DoVerboseOperation0Args"   );
    libGAP_InitHandlerFunc( libGAP_DoVerboseOperation1Args,   "src/opers.c:DoVerboseOperation1Args"   );
    libGAP_InitHandlerFunc( libGAP_DoVerboseOperation2Args,   "src/opers.c:DoVerboseOperation2Args"   );
    libGAP_InitHandlerFunc( libGAP_DoVerboseOperation3Args,   "src/opers.c:DoVerboseOperation3Args"   );
    libGAP_InitHandlerFunc( libGAP_DoVerboseOperation4Args,   "src/opers.c:DoVerboseOperation4Args"   );
    libGAP_InitHandlerFunc( libGAP_DoVerboseOperation5Args,   "src/opers.c:DoVerboseOperation5Args"   );
    libGAP_InitHandlerFunc( libGAP_DoVerboseOperation6Args,   "src/opers.c:DoVerboseOperation6Args"   );
    libGAP_InitHandlerFunc( libGAP_DoVerboseOperationXArgs,   "src/opers.c:DoVerboseOperationXArgs"   );
    
    libGAP_InitHandlerFunc( libGAP_DoConstructor0Args,        "src/opers.c:DoConstructor0Args"        );
    libGAP_InitHandlerFunc( libGAP_DoConstructor1Args,        "src/opers.c:DoConstructor1Args"        );
    libGAP_InitHandlerFunc( libGAP_DoConstructor2Args,        "src/opers.c:DoConstructor2Args"        );
    libGAP_InitHandlerFunc( libGAP_DoConstructor3Args,        "src/opers.c:DoConstructor3Args"        );
    libGAP_InitHandlerFunc( libGAP_DoConstructor4Args,        "src/opers.c:DoConstructor4Args"        );
    libGAP_InitHandlerFunc( libGAP_DoConstructor5Args,        "src/opers.c:DoConstructor5Args"        );
    libGAP_InitHandlerFunc( libGAP_DoConstructor6Args,        "src/opers.c:DoConstructor6Args"        );
    libGAP_InitHandlerFunc( libGAP_DoConstructorXArgs,        "src/opers.c:DoConstructorXArgs"        );

    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructor0Args, "src/opers.c:DoVerboseConstructor0Args" );
    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructor1Args, "src/opers.c:DoVerboseConstructor1Args" );
    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructor2Args, "src/opers.c:DoVerboseConstructor2Args" );
    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructor3Args, "src/opers.c:DoVerboseConstructor3Args" );
    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructor4Args, "src/opers.c:DoVerboseConstructor4Args" );
    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructor5Args, "src/opers.c:DoVerboseConstructor5Args" );
    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructor6Args, "src/opers.c:DoVerboseConstructor6Args" );
    libGAP_InitHandlerFunc( libGAP_DoVerboseConstructorXArgs, "src/opers.c:DoVerboseConstructorXArgs" );

    libGAP_InitHandlerFunc( libGAP_DoUninstalledOperationArgs, "src/opers.c:DoUninstalledOperationArgs" );

    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_FLAGS", &libGAP_TYPE_FLAGS );
    libGAP_TypeObjFuncs[ libGAP_T_FLAGS ] = libGAP_TypeFlags;

    /* make the 'true' operation                                           */  
    libGAP_InitGlobalBag( &libGAP_ReturnTrueFilter, "src/opers.c:ReturnTrueFilter" );

    /* install the (function) copies of global variables                   */
    /* for the inside-out (kernel to library) interface                    */
    libGAP_InitGlobalBag( &libGAP_TRY_NEXT_METHOD, "src/opers.c:TRY_NEXT_METHOD" );

    libGAP_ImportFuncFromLibrary( "METHOD_0ARGS", &libGAP_Method0Args );
    libGAP_ImportFuncFromLibrary( "METHOD_1ARGS", &libGAP_Method1Args );
    libGAP_ImportFuncFromLibrary( "METHOD_2ARGS", &libGAP_Method2Args );
    libGAP_ImportFuncFromLibrary( "METHOD_3ARGS", &libGAP_Method3Args );
    libGAP_ImportFuncFromLibrary( "METHOD_4ARGS", &libGAP_Method4Args );
    libGAP_ImportFuncFromLibrary( "METHOD_5ARGS", &libGAP_Method5Args );
    libGAP_ImportFuncFromLibrary( "METHOD_6ARGS", &libGAP_Method6Args );
    libGAP_ImportFuncFromLibrary( "METHOD_XARGS", &libGAP_MethodXArgs );

    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_0ARGS", &libGAP_NextMethod0Args );
    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_1ARGS", &libGAP_NextMethod1Args );
    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_2ARGS", &libGAP_NextMethod2Args );
    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_3ARGS", &libGAP_NextMethod3Args );
    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_4ARGS", &libGAP_NextMethod4Args );
    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_5ARGS", &libGAP_NextMethod5Args );
    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_6ARGS", &libGAP_NextMethod6Args );
    libGAP_ImportFuncFromLibrary( "NEXT_METHOD_XARGS", &libGAP_NextMethodXArgs );

    libGAP_ImportFuncFromLibrary( "VMETHOD_0ARGS", &libGAP_VMethod0Args );
    libGAP_ImportFuncFromLibrary( "VMETHOD_1ARGS", &libGAP_VMethod1Args );
    libGAP_ImportFuncFromLibrary( "VMETHOD_2ARGS", &libGAP_VMethod2Args );
    libGAP_ImportFuncFromLibrary( "VMETHOD_3ARGS", &libGAP_VMethod3Args );
    libGAP_ImportFuncFromLibrary( "VMETHOD_4ARGS", &libGAP_VMethod4Args );
    libGAP_ImportFuncFromLibrary( "VMETHOD_5ARGS", &libGAP_VMethod5Args );
    libGAP_ImportFuncFromLibrary( "VMETHOD_6ARGS", &libGAP_VMethod6Args );
    libGAP_ImportFuncFromLibrary( "VMETHOD_XARGS", &libGAP_VMethodXArgs );

    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_0ARGS", &libGAP_NextVMethod0Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_1ARGS", &libGAP_NextVMethod1Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_2ARGS", &libGAP_NextVMethod2Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_3ARGS", &libGAP_NextVMethod3Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_4ARGS", &libGAP_NextVMethod4Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_5ARGS", &libGAP_NextVMethod5Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_6ARGS", &libGAP_NextVMethod6Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VMETHOD_XARGS", &libGAP_NextVMethodXArgs );

    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_0ARGS", &libGAP_Constructor0Args );
    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_1ARGS", &libGAP_Constructor1Args );
    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_2ARGS", &libGAP_Constructor2Args );
    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_3ARGS", &libGAP_Constructor3Args );
    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_4ARGS", &libGAP_Constructor4Args );
    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_5ARGS", &libGAP_Constructor5Args );
    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_6ARGS", &libGAP_Constructor6Args );
    libGAP_ImportFuncFromLibrary( "CONSTRUCTOR_XARGS", &libGAP_ConstructorXArgs );

    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_0ARGS", &libGAP_NextConstructor0Args );
    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_1ARGS", &libGAP_NextConstructor1Args );
    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_2ARGS", &libGAP_NextConstructor2Args );
    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_3ARGS", &libGAP_NextConstructor3Args );
    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_4ARGS", &libGAP_NextConstructor4Args );
    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_5ARGS", &libGAP_NextConstructor5Args );
    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_6ARGS", &libGAP_NextConstructor6Args );
    libGAP_ImportFuncFromLibrary( "NEXT_CONSTRUCTOR_XARGS", &libGAP_NextConstructorXArgs );

    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_0ARGS", &libGAP_VConstructor0Args );
    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_1ARGS", &libGAP_VConstructor1Args );
    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_2ARGS", &libGAP_VConstructor2Args );
    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_3ARGS", &libGAP_VConstructor3Args );
    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_4ARGS", &libGAP_VConstructor4Args );
    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_5ARGS", &libGAP_VConstructor5Args );
    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_6ARGS", &libGAP_VConstructor6Args );
    libGAP_ImportFuncFromLibrary( "VCONSTRUCTOR_XARGS", &libGAP_VConstructorXArgs );

    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_0ARGS", &libGAP_NextVConstructor0Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_1ARGS", &libGAP_NextVConstructor1Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_2ARGS", &libGAP_NextVConstructor2Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_3ARGS", &libGAP_NextVConstructor3Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_4ARGS", &libGAP_NextVConstructor4Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_5ARGS", &libGAP_NextVConstructor5Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_6ARGS", &libGAP_NextVConstructor6Args );
    libGAP_ImportFuncFromLibrary( "NEXT_VCONSTRUCTOR_XARGS", &libGAP_NextVConstructorXArgs );

    libGAP_ImportFuncFromLibrary( "SET_FILTER_OBJ",   &libGAP_SET_FILTER_OBJ );
    libGAP_ImportFuncFromLibrary( "RESET_FILTER_OBJ", &libGAP_RESET_FILTER_OBJ );
    
    libGAP_ImportFuncFromLibrary( "HANDLE_METHOD_NOT_FOUND", &libGAP_HandleMethodNotFound );
    libGAP_ImportGVarFromLibrary( "IsType", &libGAP_IsType );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* install the marking function                                        */
    libGAP_InfoBags[libGAP_T_FLAGS].name = "flags list";
    libGAP_InitMarkFuncBags( libGAP_T_FLAGS, libGAP_MarkFourSubBags );

    /* install the printing function                                       */
    libGAP_PrintObjFuncs[ libGAP_T_FLAGS ] = libGAP_PrintFlags;

    /* and the saving function */
    libGAP_SaveObjFuncs[ libGAP_T_FLAGS ] = libGAP_SaveFlags;
    libGAP_LoadObjFuncs[ libGAP_T_FLAGS ] = libGAP_LoadFlags;

    /* import copy of REREADING */
    libGAP_ImportGVarFromLibrary( "REREADING", &libGAP_REREADING );
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  postRestore( <module> ) . . . . . . .  initialise library data structures
**
*/


static libGAP_Int libGAP_postRestore (
    libGAP_StructInitInfo *    libGAP_module )
{

  libGAP_CountFlags = libGAP_LEN_LIST(libGAP_VAL_GVAR(libGAP_GVarName("FILTERS")))+1;
  return 0;
}

/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Obj                 str;

    /* share between uncompleted functions                                 */
    libGAP_C_NEW_STRING_CONST( libGAP_StringAndFilter, "<<and-filter>>" );
    libGAP_RESET_FILT_LIST( libGAP_StringAndFilter, libGAP_FN_IS_MUTABLE );

    libGAP_C_NEW_STRING_CONST( libGAP_StringFilterSetter, "<<filter-setter>>" );
    libGAP_RESET_FILT_LIST( libGAP_StringFilterSetter, libGAP_FN_IS_MUTABLE );

    libGAP_ArglistObj = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, 1 );
    libGAP_SET_LEN_PLIST( libGAP_ArglistObj, 1 );
    libGAP_C_NEW_STRING_CONST( str, "obj" );
    libGAP_RESET_FILT_LIST( str, libGAP_FN_IS_MUTABLE );
    libGAP_SET_ELM_PLIST( libGAP_ArglistObj, 1, str );

    libGAP_ArglistObjVal = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, 2 );
    libGAP_SET_LEN_PLIST( libGAP_ArglistObjVal, 2 );
    libGAP_C_NEW_STRING_CONST( str, "obj" );
    libGAP_RESET_FILT_LIST( str, libGAP_FN_IS_MUTABLE );
    libGAP_SET_ELM_PLIST( libGAP_ArglistObjVal, 1, str );
    libGAP_C_NEW_STRING_CONST( str, "val" );
    libGAP_RESET_FILT_LIST( str, libGAP_FN_IS_MUTABLE );
    libGAP_SET_ELM_PLIST( libGAP_ArglistObjVal, 2, str );

    /* make the 'true' operation                                           */  
    libGAP_ReturnTrueFilter = libGAP_NewReturnTrueFilter();
    libGAP_AssGVar( libGAP_GVarName( "IS_OBJECT" ), libGAP_ReturnTrueFilter );

    /* install the (function) copies of global variables                   */
    /* for the inside-out (kernel to library) interface                    */
    libGAP_C_NEW_STRING_CONST(libGAP_TRY_NEXT_METHOD, "TRY_NEXT_METHOD");
    libGAP_AssGVar( libGAP_GVarName("TRY_NEXT_METHOD"), libGAP_TRY_NEXT_METHOD );

    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoOpers() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "opers",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_postRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoOpers ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  opers.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/


