/****************************************************************************
**
*W  objscoll.c                  GAP source                       Frank Celler
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains a single collector for finite polycyclic groups.
**
**  Unfortunately, there  are quite a  lot of stacks  required  in the single
**  collector. The collector functions will adjust the lists to have physical
**  length equal  to the maximum defined  in 'maxStackSize'.  Therefore it is
**  possible to initialise all stacks with an empty list.
**
**  There  are  also    two temporary   collector    vectors  'cwVector'  and
**  'cw2Vector',  the functions   'CXBits_VectorWord' will  adjust the string
**  length to  match the number of rws  generators.  Therefore it is possible
**  to initialise these  vectors with an  empty string.  WARNING: if  you use
**  such  a  vector, you *must* clear   it afterwards, because  all functions
**  assume that the vectors are cleared.
*/
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */
#include        "gap.h"                 /* error handling, initialisation  */

#include        "calls.h"               /* generic call mechanism          */

#include        "records.h"             /* generic records                 */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */

#include        "objfgelm.h"            /* objects of free groups          */

#include        "objscoll.h"            /* single collector                */

#include        "objccoll.h"            /* combinatorial collector         */

/****************************************************************************
**

*F * * * * * * * * * * * * local defines and typedefs * * * * * * * * * * * *
*/

/****************************************************************************
**

*T  FinPowConjCol
**
**  'FinPowConjCol' is a structure containing  all the functions depending on
**  the number of bits used in the a finite power/conjugate collector.
*/
typedef libGAP_Int (*libGAP_FuncIOOO)  (libGAP_Obj,libGAP_Obj,libGAP_Obj);
typedef libGAP_Obj (*libGAP_FuncOOOI)  (libGAP_Obj,libGAP_Obj,libGAP_Int);
typedef libGAP_Int (*libGAP_FuncIOOI)  (libGAP_Obj,libGAP_Obj,libGAP_Int);
typedef libGAP_Obj (*libGAP_FuncOOOO)  (libGAP_Obj,libGAP_Obj,libGAP_Obj);
typedef libGAP_Int (*libGAP_FuncIOOOF) (libGAP_Obj,libGAP_Obj,libGAP_Obj,libGAP_FuncIOOO);

typedef struct {

    libGAP_FuncOOOI    wordVectorAndClear;
    libGAP_FuncIOOI    vectorWord;
    libGAP_FuncIOOO    collectWord;
    libGAP_FuncIOOOF   solution;

} libGAP_FinPowConjCol;


/****************************************************************************
**

*F * * * * * * * * * * * internal collector functions * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  C8Bits_WordVectorAndClear( <kind>, <vv>, <num> )
*/
libGAP_Obj libGAP_C8Bits_WordVectorAndClear ( libGAP_Obj kind, libGAP_Obj vv, libGAP_Int num )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         j;              /* loop variable for exponent vector       */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */
    libGAP_UInt1 *     ptr;            /* pointer into the data area of <obj>     */
    libGAP_Obj         obj;            /* result                                  */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* construct a new object                                              */
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt1 pointer for eight bits, clear <vv>                        */
    ptr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
    qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vv)+1);
    for ( i = 1, j = 0;  i <= num;  i++,  qtr++ ) {
        if ( *qtr != 0 ) {
            *ptr++ = ((i-1) << ebits) | (*qtr & expm);
            *qtr = 0;
            j++;
        }
    }

    /* correct the size of <obj>                                           */
    libGAP_RESIZE_WORD( obj, j );
    return obj;
}


/****************************************************************************
**
*F  C8Bits_VectorWord( <vv>, <v>, <num> )
**
**  WARNING: This function assumes that <vv> is cleared!
*/
libGAP_Int libGAP_C8Bits_VectorWord ( libGAP_Obj vv, libGAP_Obj v, libGAP_Int num )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         pos;            /* generator number                        */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */
    libGAP_UInt1 *     ptr;            /* pointer into the data area of <obj>     */

    /* <vv> must be a string                                               */
    if ( libGAP_TNUM_OBJ(vv) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(vv) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( vv, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(vv), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(vv) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        libGAP_ResizeBag( vv, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vv)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
    }

    /* if <v> is zero, return                                              */
    if ( v == 0 )
        return 0;

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(v);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* unfold <v> into <vv>                                                */
    ptr = (libGAP_UInt1*)libGAP_DATA_WORD(v);
    qtr = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    for ( i = libGAP_NPAIRS_WORD(v);  0 < i;  i--, ptr++ ) {
        pos = ((*ptr) >> ebits)+1;
        if ( pos > num ) {
           libGAP_ErrorQuit( "word contains illegal generators %d", (libGAP_Int)i, 0L );
           return 0;
        }
        if ( (*ptr) & exps )
            qtr[pos] = ((*ptr)&expm)-exps;
        else
            qtr[pos] = (*ptr)&expm;
    }
    return 0;
}

/****************************************************************************
**
*F  C8Bits_SingleCollectWord( <sc>, <vv>, <w> )
**
**  If a stack overflow occurs, we simply stop and return false.
**
**  SC_PUSH_WORD( word, exp )
**    push <word>  with global exponent <exp>  into the stack, the macro uses
**    <word> and <exp> only once.
**
**  SC_POP_WORD()
**    remove topmost word from stack
*/
#define libGAP_SC_PUSH_WORD( word, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(word); \
    *++lw = *nw + (libGAP_INT_INTOBJ((((libGAP_Obj*)(*nw))[-1])) - 1); \
    *++pw = *nw; \
    *++ew = (**pw) & expm; \
    *++ge = exp

#define libGAP_SC_POP_WORD() \
    sp--;  nw--;  lw--;  pw--;  ew--;  ge--


/****************************************************************************
**
**  The following functions are  used to add  a word into the exponent vector
**  without collection.  Two different cases occur:
**
**  Add   a word  into  the exponent  vector.   Here we   can  use the global
**  exponent.
**
**  Add part  of a word  into the  exponent vector.   Here  we cannot use the
**  global exponent because the beginning of  the word might not commute with
**  the rest.
**/
static libGAP_Int libGAP_C8Bits_SAddWordIntoExpVec( libGAP_Int *v, libGAP_UInt1 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Obj *ro, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt1 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;
    libGAP_Int        start = 0;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( libGAP_INT_INTOBJ(ro[i]) <= v[i] ) {
            ex = v[i] / libGAP_INT_INTOBJ(ro[i]);
            v[i] -= ex * libGAP_INT_INTOBJ(ro[i]);
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                start = libGAP_C8Bits_SAddWordIntoExpVec( 
                    v, (libGAP_UInt1*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, ro, pow, lpow  );
            }
        }
        if( start < i && v[i] ) start = i;
    }
    return start;
}

static libGAP_Int libGAP_C8Bits_SAddPartIntoExpVec( libGAP_Int *v, libGAP_UInt1 *w, libGAP_UInt1 *wend,
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Obj* ro, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_Int        i;
    libGAP_Int        ex;
    libGAP_Int        start = 0;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm);     /* overflow check necessary? */
        if ( libGAP_INT_INTOBJ(ro[i]) <= v[i] ) {
            ex = v[i] / libGAP_INT_INTOBJ(ro[i]);
            v[i] -= ex * libGAP_INT_INTOBJ(ro[i]);
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                start = libGAP_C8Bits_SAddWordIntoExpVec( 
                    v, (libGAP_UInt1*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, ro, pow, lpow  );
            }
        }
        if( start < i && v[i] ) start = i;
    }
    return start;
}

libGAP_Int libGAP_C8Bits_SingleCollectWord ( libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    libGAP_Int         ebits;      /* number of bits in the exponent              */
    libGAP_UInt        expm;       /* unsigned exponent mask                      */
    libGAP_UInt        exps;       /* sign exponent mask                          */

    libGAP_Obj         vnw;        /* word stack                                  */
    libGAP_UInt1 **    nw;         /* address of <vnw>                            */
    libGAP_Obj         vlw;        /* last syllable stack                         */
    libGAP_UInt1 **    lw;         /* address of <vlw>                            */
    libGAP_Obj         vpw;        /* current syllable stack                      */
    libGAP_UInt1 **    pw;         /* address of <vpw>                            */
    libGAP_Obj         vew;        /* unprocessed exponent stack                  */
    libGAP_UInt1 *     ew;         /* address of <vew>                            */
    libGAP_Obj         vge;        /* global exponent stack                       */
    libGAP_Int *       ge;         /* address of <vge>                            */

    libGAP_Obj         vpow;       /* rhs of power relations                      */
    libGAP_Int         lpow;       /* length of <vpow>                            */
    libGAP_Obj *       pow;        /* address of <vpow>                           */

    libGAP_Obj         vcnj;       /* rhs of conjugate relations                  */
    libGAP_Int         lcnj;       /* length of <vcnj>                            */
    libGAP_Obj *       cnj;        /* address of <vcnj>                           */

    libGAP_Obj *       avc;        /* address of the avector                      */
    libGAP_Obj *       gns;        /* address of the list of generators           */
    libGAP_Obj *       ro;         /* address of the list of relative orders      */
    libGAP_Obj *       inv;        /* address of the list of inverses             */

    libGAP_Int *       v;          /* address of <vv>                             */

    libGAP_Int         max;        /* maximal stack size                          */
    libGAP_Int         sp;         /* stack pointer                               */
    libGAP_Int         i, j;       /* loop variable                               */
    libGAP_Int         gn;         /* current generator number                    */
    libGAP_Int         ex;         /* current exponent                            */
    libGAP_Int         start;      /* last non-trivial entry                      */
    
    libGAP_Obj         tmp;        /* temporary obj for power                     */

    libGAP_Int         resized = 0;/* indicates whether a Resize() happened       */

    /* <start> is the first non-trivial entry in <v>                       */
    start = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* if <w> is the identity return now                                   */
    if ( libGAP_NPAIRS_WORD(w) == 0 ) {
        return start;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* get the exponent sign masks                                         */
    exps = 1UL << (ebits-1);

    /* <nw> contains the stack of words to insert                          */
    vnw = libGAP_SC_NW_STACK(sc);

    /* <lw> contains the word end of the word in <nw>                      */
    vlw = libGAP_SC_LW_STACK(sc);

    /* <pw> contains the position of the word in <nw> to look at           */
    vpw = libGAP_SC_PW_STACK(sc);

    /* <ew> contains the unprocessed exponents at position <pw>            */
    vew = libGAP_SC_EW_STACK(sc);

    /* <ge> contains the global exponent of the word                       */
    vge = libGAP_SC_GE_STACK(sc);

    /* get the maximal stack size                                          */
    max = libGAP_SC_MAX_STACK_SIZE(sc);

    /* ensure that the stacks are large enough                             */
    if ( libGAP_SIZE_OBJ(vnw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vnw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vnw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vlw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vlw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vlw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vpw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vpw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vpw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vew)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vew, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vew, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vge)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vge, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vge, libGAP_T_STRING );
        resized = 1;
    }
    if( resized ) return -1;

    /* from now on we use addresses instead of handles most of the time    */
    v  = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    nw = (libGAP_UInt1**)libGAP_ADDR_OBJ(vnw);
    lw = (libGAP_UInt1**)libGAP_ADDR_OBJ(vlw);
    pw = (libGAP_UInt1**)libGAP_ADDR_OBJ(vpw);
    ew = (libGAP_UInt1*)libGAP_ADDR_OBJ(vew);
    ge = (libGAP_Int*)libGAP_ADDR_OBJ(vge);

    /* conjujagtes, powers, order, generators, avector, inverses           */
    vpow = libGAP_SC_POWERS(sc);
    lpow = libGAP_LEN_PLIST(vpow);
    pow  = libGAP_ADDR_OBJ(vpow);

    vcnj = libGAP_SC_CONJUGATES(sc);
    lcnj = libGAP_LEN_PLIST(vcnj);
    (void) lcnj; /* please compiler -- lcnj not actually used */
    cnj  = libGAP_ADDR_OBJ(vcnj);

    avc = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR(sc) );
    gns = libGAP_ADDR_OBJ( libGAP_SC_RWS_GENERATORS(sc) );

    ro  = libGAP_ADDR_OBJ( libGAP_SC_RELATIVE_ORDERS(sc) );
    inv = libGAP_ADDR_OBJ( libGAP_SC_INVERSES(sc) );

    /* initialize the stack with <w>                                        */
    sp = 0;
    libGAP_SC_PUSH_WORD( w, 1 );

    /* run until the stack is empty                                        */
    while ( 0 < sp ) {

        /* if <ew> is negative use inverse                                 */
        if ( *ew & exps ) {
            gn = ((**pw) >> ebits) + 1;
            ex = ( *ew & (exps-1) ) - exps;
            *ew = 0;
            libGAP_SC_PUSH_WORD( inv[gn], -ex );
        }

        /* if <ew> is zero get next syllable                               */
        else if ( 0 == *ew ) {

            /* if <pw> has reached <lw> get next & reduce globale exponent */
            if ( *pw == *lw ) {

                /* if the globale exponent is greater one reduce it        */
                if ( 1 < *ge ) {
                    (*ge)--;
                    *pw = *nw;
                    *ew = (**pw) & expm;
                }

                /* otherwise get the next word from the stack              */
                else {
                    libGAP_SC_POP_WORD();
                }
            }

            /* otherwise set <ew> to exponent of next syllable             */
            else {
                (*pw)++;
                *ew = (**pw) & expm;
            }
        }

        /* now move the next generator to the correct position             */
        else {

            /* get generator number                                        */
            gn = ((**pw) >> ebits) + 1;

            /* we can move <gn> directly to the correct position           */
            if ( libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              /*
              *T  This if-statemant implies that the next two cases are never
              *T  executed.  This is intended for the time being because we 
              *T  need the single collector to work with pc-presentation
              *T  whose rhs are not reduced while the next two if-case need
              *T  reduced rhs.  This will be fixed at a later stage.
              */
                v[gn] += *ew;
                *ew = 0;
                if ( start <= gn )
                    start = gn;
            }

            /* collect a whole word exponent pair                          */
            else if( *pw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              gn = libGAP_C8Bits_SAddWordIntoExpVec( 
                   v, *pw, *ge, ebits, expm, ro, pow, lpow  );
              *pw = *lw;
              *ew = *ge = 0;

              if( start <= gn ) start = gn;
              continue;
            }

            /* move the rest of a word directly into the correct positions */
            else if( libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              gn = libGAP_C8Bits_SAddPartIntoExpVec( 
                   v, *pw, *lw, ebits, expm, ro, pow, lpow  );
              *pw = *lw;
              *ew = 0;

              if( start <= gn ) start = gn;
              continue;
            }

            /* we have to move <gn> step by step                           */
            else {
                (*ew)--; v[gn]++;

                i = libGAP_INT_INTOBJ(avc[gn]);
                if ( start < i )
                    i = start;

                /* Find the first position in v from where on ordinary
                   collection  has to be applied.                          */
                for( ; gn < i; i-- )
                    if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                            break;
                    }

                /* Stack up this part of v if we run through the next 
                   for-loop or if a power relation will be applied         */
                if( gn < i || (libGAP_INT_INTOBJ(ro[gn]) <= v[gn] &&
                    gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn])) ) {
                    j = libGAP_INT_INTOBJ(avc[gn]);
                    for( ; i < j; j-- )
                        if( v[j] ) {
                            libGAP_SC_PUSH_WORD( gns[j], v[j] );
                            v[j] = 0;
                        }
                }

                if( gn < i ) {
                  for ( ;  gn < i;  i-- ) {
                    if ( v[i] ) {
                        if ( libGAP_LEN_PLIST(cnj[i]) < gn )
                            tmp = gns[i];
                        else {
                            tmp = libGAP_ELM_PLIST( cnj[i], gn );
                            if ( tmp == 0 || libGAP_NPAIRS_WORD(tmp) == 0 )
                                tmp = gns[i];
                        }
                        libGAP_SC_PUSH_WORD( tmp, v[i] );
                        v[i] = 0;
                    }
                  }
                  if ( start <= libGAP_INT_INTOBJ(avc[gn]) )
                    start = gn;
                }
                if( start <= gn ) start = gn;
            }

            /* check that the exponent is not too big                      */
            if ( libGAP_INT_INTOBJ(ro[gn]) <= v[gn] ) {
                i = v[gn] / libGAP_INT_INTOBJ(ro[gn]);
                v[gn] -= i * libGAP_INT_INTOBJ(ro[gn]);
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_SC_PUSH_WORD( pow[gn], i );
                }
            }
        }
    }
    return start;
}
#undef libGAP_SC_PUSH_WORD
#undef libGAP_SC_POP_WORD


/****************************************************************************
**
*F  C8Bits_Solution( <sc>, <ww>, <uu>, <func> )
*/
libGAP_Int libGAP_C8Bits_Solution( 
    libGAP_Obj         sc,
    libGAP_Obj         ww,
    libGAP_Obj         uu,
    libGAP_FuncIOOO    func )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         ro;             /* relative order                          */
    libGAP_Obj         rod;            /* relative orders                         */
    libGAP_Obj         libGAP_g;              /* one generator word                      */
    libGAP_UInt1 *     gtr;            /* pointer into the data area of <g>       */
    libGAP_Int *       ptr;            /* pointer into the collect vector         */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */

    /* get the number of generators                                        */
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    rod = libGAP_SC_RELATIVE_ORDERS(sc);

    /* <ww> must be a string                                               */
    if ( libGAP_TNUM_OBJ(ww) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(ww) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( ww, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(ww), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(ww) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        i = (libGAP_SIZE_OBJ(ww)-sizeof(libGAP_Obj)-1) / sizeof(libGAP_Int);
        libGAP_ResizeBag( ww, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(ww)+1);
        for ( i = i+1;  i <= num;  i++ )
            qtr[i] = 0;
    }

    /* <uu> must be a string                                               */
    if ( libGAP_TNUM_OBJ(uu) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(uu) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( ww, libGAP_T_STRING );
        } else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(uu), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(uu) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        i = (libGAP_SIZE_OBJ(uu)-sizeof(libGAP_Obj)-1) / sizeof(libGAP_Int);
        libGAP_ResizeBag( uu, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(uu)+1);
        for ( i = i+1;  i <= num;  i++ )
            qtr[i] = 0;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* use <g> as right argument for the collector                         */
    libGAP_NEW_WORD( libGAP_g, libGAP_SC_DEFAULT_TYPE(sc), 1 );

    /* start clearing <ww>, storing the result in <uu>                     */
    ptr = (libGAP_Int*)(libGAP_ADDR_OBJ(ww)+1);
    qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(uu)+1);
    gtr = (libGAP_UInt1*)libGAP_DATA_WORD(libGAP_g);
    for ( i = num;  0 < i;  i--, ptr++, qtr++ ) {
        ro = libGAP_INT_INTOBJ(libGAP_ELMW_LIST(rod,num-i+1));
        *qtr = ( *qtr - *ptr ) % ro;
        if ( *qtr < 0 )  *qtr += ro;
        if ( *qtr != 0 ) {
            *gtr = ( (num-i) << ebits ) | ( *qtr & expm );
            if ( func(sc,ww,libGAP_g) == -1 )
                return -1;
        }
        *ptr = 0;
    }
    return 0;
}


/****************************************************************************
**
*V  C8Bits_SingleCollector
*/
libGAP_FinPowConjCol libGAP_C8Bits_SingleCollector = {
    libGAP_C8Bits_WordVectorAndClear,
    libGAP_C8Bits_VectorWord,
    libGAP_C8Bits_SingleCollectWord,
    libGAP_C8Bits_Solution
};


/****************************************************************************
**

*F  C16Bits_WordVectorAndClear( <kind>, <vv>, <num> )
*/
libGAP_Obj libGAP_C16Bits_WordVectorAndClear ( libGAP_Obj kind, libGAP_Obj vv, libGAP_Int num )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         j;              /* loop variable for exponent vector       */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */
    libGAP_UInt2 *     ptr;            /* pointer into the data area of <obj>     */
    libGAP_Obj         obj;            /* result                                  */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* construct a new object                                              */
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt2 pointer for sixteen bits, clear <vv>                      */
    ptr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vv)+1);
    for ( i = 1, j = 0;  i <= num;  i++,  qtr++ ) {
        if ( *qtr != 0 ) {
            *ptr++ = ((i-1) << ebits) | (*qtr & expm);
            *qtr = 0;
            j++;
        }
    }

    /* correct the size of <obj>                                           */
    libGAP_RESIZE_WORD( obj, j );
    return obj;
}


/****************************************************************************
**
*F  C16Bits_VectorWord( <vv>, <v>, <num> )
**
**  WARNING: This function assumes that <vv> is cleared!
*/
libGAP_Int libGAP_C16Bits_VectorWord ( libGAP_Obj vv, libGAP_Obj v, libGAP_Int num )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         pos;            /* generator number                        */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */
    libGAP_UInt2 *     ptr;            /* pointer into the data area of <obj>     */

    /* <vv> must be a string                                               */
    if ( libGAP_TNUM_OBJ(vv) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(vv) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( vv, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(vv), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(vv) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        libGAP_ResizeBag( vv, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vv)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
    }

    /* if <v> is zero, return                                              */
    if ( v == 0 )
        return 0;

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(v);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* unfold <v> into <vv>                                                */
    ptr = (libGAP_UInt2*)libGAP_DATA_WORD(v);
    qtr = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    for ( i = libGAP_NPAIRS_WORD(v);  0 < i;  i--, ptr++ ) {
        pos = ((*ptr) >> ebits)+1;
        if ( pos > num ) {
           libGAP_ErrorQuit( "word contains illegal generators %d", (libGAP_Int)i, 0L );
           return 0;
        }
        if ( (*ptr) & exps )
            qtr[pos] = ((*ptr)&expm)-exps;
        else
            qtr[pos] = (*ptr)&expm;
    }
    return 0;
}


/****************************************************************************
**
*F  C16Bits_SingleCollectWord( <sc>, <vv>, <w> )
**
**  If a stack overflow occurs, we simply stop and return false.
**
**  SC_PUSH_WORD( word, exp )
**    push <word>  with global exponent <exp>  into the stack, the macro uses
**    <word> and <exp> only once.
**
**  SC_POP_WORD()
**    remove topmost word from stack
*/
#define libGAP_SC_PUSH_WORD( word, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(word); \
    *++lw = *nw + (libGAP_INT_INTOBJ((((libGAP_Obj*)(*nw))[-1])) - 1); \
    *++pw = *nw; \
    *++ew = (**pw) & expm; \
    *++ge = exp

#define libGAP_SC_POP_WORD() \
    sp--;  nw--;  lw--;  pw--;  ew--;  ge--


/****************************************************************************
**
**  The following functions are  used to add  a word into the exponent vector
**  without collection.  Two different cases occur:
**
**  Add   a word  into  the exponent  vector.   Here we   can  use the global
**  exponent.
**
**  Add part  of a word  into the  exponent vector.   Here  we cannot use the
**  global exponent because the beginning of  the word might not commute with
**  the rest.
**/
static libGAP_Int libGAP_C16Bits_SAddWordIntoExpVec( libGAP_Int *v, libGAP_UInt2 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Obj *ro, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt2 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;
    libGAP_Int        start = 0;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( libGAP_INT_INTOBJ(ro[i]) <= v[i] ) {
            ex = v[i] / libGAP_INT_INTOBJ(ro[i]);
            v[i] -= ex * libGAP_INT_INTOBJ(ro[i]);
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                start = libGAP_C16Bits_SAddWordIntoExpVec( 
                    v, (libGAP_UInt2*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, ro, pow, lpow  );
            }
        }
        if( start < i && v[i] ) start = i;
    }
    return start;
}

static libGAP_Int libGAP_C16Bits_SAddPartIntoExpVec( libGAP_Int *v, libGAP_UInt2 *w, libGAP_UInt2 *wend,
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Obj* ro, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_Int        i;
    libGAP_Int        ex;
    libGAP_Int        start = 0;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm);     /* overflow check necessary? */
        if ( libGAP_INT_INTOBJ(ro[i]) <= v[i] ) {
            ex = v[i] / libGAP_INT_INTOBJ(ro[i]);
            v[i] -= ex * libGAP_INT_INTOBJ(ro[i]);
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                start = libGAP_C16Bits_SAddWordIntoExpVec( 
                    v, (libGAP_UInt2*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, ro, pow, lpow  );
            }
        }
        if( start < i && v[i] ) start = i;
    }
    return start;
}

libGAP_Int libGAP_C16Bits_SingleCollectWord ( libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    libGAP_Int         ebits;      /* number of bits in the exponent              */
    libGAP_UInt        expm;       /* unsigned exponent mask                      */
    libGAP_UInt        exps;       /* sign exponent mask                          */

    libGAP_Obj         vnw;        /* word stack                                  */
    libGAP_UInt2 **    nw;         /* address of <vnw>                            */
    libGAP_Obj         vlw;        /* last syllable stack                         */
    libGAP_UInt2 **    lw;         /* address of <vlw>                            */
    libGAP_Obj         vpw;        /* current syllable stack                      */
    libGAP_UInt2 **    pw;         /* address of <vpw>                            */
    libGAP_Obj         vew;        /* unprocessed exponent stack                  */
    libGAP_UInt2 *     ew;         /* address of <vew>                            */
    libGAP_Obj         vge;        /* global exponent stack                       */
    libGAP_Int *       ge;         /* address of <vge>                            */

    libGAP_Obj         vpow;       /* rhs of power relations                      */
    libGAP_Int         lpow;       /* length of <vpow>                            */
    libGAP_Obj *       pow;        /* address of <vpow>                           */

    libGAP_Obj         vcnj;       /* rhs of conjugate relations                  */
    libGAP_Int         lcnj;       /* length of <vcnj>                            */
    libGAP_Obj *       cnj;        /* address of <vcnj>                           */

    libGAP_Obj *       avc;        /* address of the avector                      */
    libGAP_Obj *       gns;        /* address of the list of generators           */
    libGAP_Obj *       ro;         /* address of the list of relative orders      */
    libGAP_Obj *       inv;        /* address of the list of inverses             */

    libGAP_Int *       v;          /* address of <vv>                             */

    libGAP_Int         max;        /* maximal stack size                          */
    libGAP_Int         sp;         /* stack pointer                               */
    libGAP_Int         i, j;       /* loop variable                               */
    libGAP_Int         gn;         /* current generator number                    */
    libGAP_Int         ex;         /* current exponent                            */
    libGAP_Int         start;      /* last non-trivial entry                      */

    libGAP_Obj         tmp;        /* temporary obj for power                     */

    libGAP_Int         resized = 0;/* indicates whether a Resize() happened       */

    /* <start> is the first non-trivial entry in <v>                       */
    start = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* if <w> is the identity return now                                   */
    if ( libGAP_NPAIRS_WORD(w) == 0 ) {
        return start;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* get the exponent sign masks                                         */
    exps = 1UL << (ebits-1);

    /* <nw> contains the stack of words to insert                          */
    vnw = libGAP_SC_NW_STACK(sc);

    /* <lw> contains the word end of the word in <nw>                      */
    vlw = libGAP_SC_LW_STACK(sc);

    /* <pw> contains the position of the word in <nw> to look at           */
    vpw = libGAP_SC_PW_STACK(sc);

    /* <ew> contains the unprocessed exponents at position <pw>            */
    vew = libGAP_SC_EW_STACK(sc);

    /* <ge> contains the global exponent of the word                       */
    vge = libGAP_SC_GE_STACK(sc);

    /* get the maximal stack size                                          */
    max = libGAP_SC_MAX_STACK_SIZE(sc);

    /* ensure that the stacks are large enough                             */
    if ( libGAP_SIZE_OBJ(vnw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vnw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vnw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vlw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vlw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vlw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vpw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vpw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vpw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vew)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vew, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vew, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vge)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vge, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vge, libGAP_T_STRING );
        resized = 1;
    }
    if( resized ) return -1;

    /* from now on we use addresses instead of handles most of the time    */
    v  = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    nw = (libGAP_UInt2**)libGAP_ADDR_OBJ(vnw);
    lw = (libGAP_UInt2**)libGAP_ADDR_OBJ(vlw);
    pw = (libGAP_UInt2**)libGAP_ADDR_OBJ(vpw);
    ew = (libGAP_UInt2*)libGAP_ADDR_OBJ(vew);
    ge = (libGAP_Int*)libGAP_ADDR_OBJ(vge);

    /* conjujagtes, powers, order, generators, avector, inverses           */
    vpow = libGAP_SC_POWERS(sc);
    lpow = libGAP_LEN_PLIST(vpow);
    pow  = libGAP_ADDR_OBJ(vpow);

    vcnj = libGAP_SC_CONJUGATES(sc);
    lcnj = libGAP_LEN_PLIST(vcnj);
    (void) lcnj; /* please compiler -- lcnj not actually used */
    cnj  = libGAP_ADDR_OBJ(vcnj);

    avc = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR(sc) );
    gns = libGAP_ADDR_OBJ( libGAP_SC_RWS_GENERATORS(sc) );

    ro  = libGAP_ADDR_OBJ( libGAP_SC_RELATIVE_ORDERS(sc) );
    inv = libGAP_ADDR_OBJ( libGAP_SC_INVERSES(sc) );

    /* initialize the stack with <w>                                        */
    sp = 0;
    libGAP_SC_PUSH_WORD( w, 1 );

    /* run until the stack is empty                                        */
    while ( 0 < sp ) {

        /* if <ew> is negative use inverse                                 */
        if ( *ew & exps ) {
            gn = ((**pw) >> ebits) + 1;
            ex = ( *ew & (exps-1) ) - exps;
            *ew = 0;
            libGAP_SC_PUSH_WORD( inv[gn], -ex );
        }

        /* if <ew> is zero get next syllable                               */
        else if ( 0 == *ew ) {

            /* if <pw> has reached <lw> get next & reduce globale exponent */
            if ( *pw == *lw ) {

                /* if the globale exponent is greater one reduce it        */
                if ( 1 < *ge ) {
                    (*ge)--;
                    *pw = *nw;
                    *ew = (**pw) & expm;
                }

                /* otherwise get the next word from the stack              */
                else {
                    libGAP_SC_POP_WORD();
                }
            }

            /* otherwise set <ew> to exponent of next syllable             */
            else {
                (*pw)++;
                *ew = (**pw) & expm;
            }
        }

        /* now move the next generator to the correct position             */
        else {

            /* get generator number                                        */
            gn = ((**pw) >> ebits) + 1;

            /* we can move <gn> directly to the correct position           */
            if ( libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              /*
              *T  This if-statemant implies that the next two cases are never
              *T  executed.  This is intended for the time being because we 
              *T  need the single collector to work with pc-presentation
              *T  whose rhs are not reduced while the next two if-case need
              *T  reduced rhs.  This will be fixed at a later stage.
              */
                v[gn] += *ew;
                *ew = 0;
                if ( start <= gn )
                    start = gn;
            }

            /* collect a whole word exponent pair                          */
            else if( *pw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              gn = libGAP_C16Bits_SAddWordIntoExpVec( 
                   v, *pw, *ge, ebits, expm, ro, pow, lpow  );
              *pw = *lw;
              *ew = *ge = 0;

              if( start <= gn ) start = gn;
              continue;
            }

            /* move the rest of a word directly into the correct positions */
            else if( libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              gn = libGAP_C16Bits_SAddPartIntoExpVec( 
                   v, *pw, *lw, ebits, expm, ro, pow, lpow  );
              *pw = *lw;
              *ew = 0;

              if( start <= gn ) start = gn;
              continue;
            }

            /* we have to move <gn> step by step                           */
            else {
                (*ew)--; v[gn]++;

                i = libGAP_INT_INTOBJ(avc[gn]);
                if ( start < i )
                    i = start;

                /* Find the first position in v from where on ordinary
                   collection  has to be applied.                          */
                for( ; gn < i; i-- )
                    if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                            break;
                    }

                /* Stack up this part of v if we run through the next 
                   for-loop or if a power relation will be applied         */
                if( gn < i || (libGAP_INT_INTOBJ(ro[gn]) <= v[gn] &&
                    gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn])) ) {
                    j = libGAP_INT_INTOBJ(avc[gn]);
                    for( ; i < j; j-- )
                        if( v[j] ) {
                            libGAP_SC_PUSH_WORD( gns[j], v[j] );
                            v[j] = 0;
                        }
                }

                if( gn < i ) {
                  for ( ;  gn < i;  i-- ) {
                    if ( v[i] ) {
                        if ( libGAP_LEN_PLIST(cnj[i]) < gn )
                            tmp = gns[i];
                        else {
                            tmp = libGAP_ELM_PLIST( cnj[i], gn );
                            if ( tmp == 0 || libGAP_NPAIRS_WORD(tmp) == 0 )
                                tmp = gns[i];
                        }
                        libGAP_SC_PUSH_WORD( tmp, v[i] );
                        v[i] = 0;
                    }
                  }
                  if ( start <= libGAP_INT_INTOBJ(avc[gn]) )
                    start = gn;
                }
                if( start <= gn ) start = gn;
            }

            /* check that the exponent is not too big                      */
            if ( libGAP_INT_INTOBJ(ro[gn]) <= v[gn] ) {
                i = v[gn] / libGAP_INT_INTOBJ(ro[gn]);
                v[gn] -= i * libGAP_INT_INTOBJ(ro[gn]);
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_SC_PUSH_WORD( pow[gn], i );
                }
            }
        }
    }
    return start;
}
#undef libGAP_SC_PUSH_WORD
#undef libGAP_SC_POP_WORD


/****************************************************************************
**
*F  C16Bits_Solution( <sc>, <ww>, <uu>, <func> )
*/
libGAP_Int libGAP_C16Bits_Solution( 
    libGAP_Obj         sc,
    libGAP_Obj         ww,
    libGAP_Obj         uu,
    libGAP_FuncIOOO    func )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         ro;             /* relative order                          */
    libGAP_Obj         rod;            /* relative orders                         */
    libGAP_Obj         libGAP_g;              /* one generator word                      */
    libGAP_UInt2 *     gtr;            /* pointer into the data area of <g>       */
    libGAP_Int *       ptr;            /* pointer into the collect vector         */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */

    /* get the number of generators                                        */
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    rod = libGAP_SC_RELATIVE_ORDERS(sc);

    /* <ww> must be a string                                               */
    if ( libGAP_TNUM_OBJ(ww) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(ww) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( ww, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(ww), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(ww) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        i = (libGAP_SIZE_OBJ(ww)-sizeof(libGAP_Obj)-1) / sizeof(libGAP_Int);
        libGAP_ResizeBag( ww, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(ww)+1);
        for ( i = i+1;  i <= num;  i++ )
            qtr[i] = 0;
    }

    /* <uu> must be a string                                               */
    if ( libGAP_TNUM_OBJ(uu) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(uu) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( uu, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(uu), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(uu) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        i = (libGAP_SIZE_OBJ(uu)-sizeof(libGAP_Obj)-1) / sizeof(libGAP_Int);
        libGAP_ResizeBag( uu, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(uu)+1);
        for ( i = i+1;  i <= num;  i++ )
            qtr[i] = 0;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* use <g> as right argument for the collector                         */
    libGAP_NEW_WORD( libGAP_g, libGAP_SC_DEFAULT_TYPE(sc), 1 );

    /* start clearing <ww>, storing the result in <uu>                     */
    ptr = (libGAP_Int*)(libGAP_ADDR_OBJ(ww)+1);
    qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(uu)+1);
    gtr = (libGAP_UInt2*)libGAP_DATA_WORD(libGAP_g);
    for ( i = num;  0 < i;  i--, ptr++, qtr++ ) {
        ro = libGAP_INT_INTOBJ(libGAP_ELMW_LIST(rod,num-i+1));
        *qtr = ( *qtr - *ptr ) % ro;
        if ( *qtr < 0 )  *qtr += ro;
        if ( *qtr != 0 ) {
            *gtr = ( (num-i) << ebits ) | ( *qtr & expm );
            if ( func(sc,ww,libGAP_g) == -1 )
                return -1;
        }
        *ptr = 0;
    }
    return 0;
}


/****************************************************************************
**
*V  C16Bits_SingleCollector
*/
libGAP_FinPowConjCol libGAP_C16Bits_SingleCollector = {
    libGAP_C16Bits_WordVectorAndClear,
    libGAP_C16Bits_VectorWord,
    libGAP_C16Bits_SingleCollectWord,
    libGAP_C16Bits_Solution
};


/****************************************************************************
**

*F  C32Bits_WordVectorAndClear( <kind>, <vv>, <num> )
*/
libGAP_Obj libGAP_C32Bits_WordVectorAndClear ( libGAP_Obj kind, libGAP_Obj vv, libGAP_Int num )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         j;              /* loop variable for exponent vector       */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */
    libGAP_UInt4 *     ptr;            /* pointer into the data area of <obj>     */
    libGAP_Obj         obj;            /* result                                  */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* construct a new object                                              */
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt4 pointer for 32 bits, clear <vv>                           */
    ptr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vv)+1);
    for ( i = 1, j = 0;  i <= num;  i++,  qtr++ ) {
        if ( *qtr != 0 ) {
            *ptr++ = ((i-1) << ebits) | (*qtr & expm);
            *qtr = 0;
            j++;
        }
    }

    /* correct the size of <obj>                                           */
    libGAP_RESIZE_WORD( obj, j );
    return obj;
}


/****************************************************************************
**
*F  C32Bits_VectorWord( <vv>, <v>, <num> )
**
**  WARNING: This function assumes that <vv> is cleared!
*/
libGAP_Int libGAP_C32Bits_VectorWord ( libGAP_Obj vv, libGAP_Obj v, libGAP_Int num )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         pos;            /* generator number                        */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */
    libGAP_UInt4 *     ptr;            /* pointer into the data area of <obj>     */

    /* <vv> must be a string                                               */
    if ( libGAP_TNUM_OBJ(vv) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(vv) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( vv, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(vv), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(vv) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        libGAP_ResizeBag( vv, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vv)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
    }

    /* if <v> is zero, return                                              */
    if ( v == 0 )
        return 0;

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(v);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* unfold <v> into <vv>                                                */
    ptr = (libGAP_UInt4*)libGAP_DATA_WORD(v);
    qtr = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    for ( i = libGAP_NPAIRS_WORD(v);  0 < i;  i--, ptr++ ) {
        pos = ((*ptr) >> ebits)+1;
        if ( pos > num ) {
           libGAP_ErrorQuit( "word contains illegal generators %d", (libGAP_Int)i, 0L );
           return 0;
        }
        if ( (*ptr) & exps )
            qtr[pos] = ((*ptr)&expm)-exps;
        else
            qtr[pos] = (*ptr)&expm;
    }
    return 0;
}

/****************************************************************************
**
*F  C32Bits_SingleCollectWord( <sc>, <vv>, <w> )
**
**  If a stack overflow occurs, we simply stop and return false.
**
**  SC_PUSH_WORD( word, exp )
**    push <word>  with global exponent <exp>  into the stack, the macro uses
**    <word> and <exp> only once.
**
**  SC_POP_WORD()
**    remove topmost word from stack
*/
#define libGAP_SC_PUSH_WORD( word, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(word); \
    *++lw = *nw + (libGAP_INT_INTOBJ((((libGAP_Obj*)(*nw))[-1])) - 1); \
    *++pw = *nw; \
    *++ew = (**pw) & expm; \
    *++ge = exp

#define libGAP_SC_POP_WORD() \
    sp--;  nw--;  lw--;  pw--;  ew--;  ge--


/****************************************************************************
**
**  The following functions are  used to add  a word into the exponent vector
**  without collection.  Two different cases occur:
**
**  Add   a word  into  the exponent  vector.   Here we   can  use the global
**  exponent.
**
**  Add part  of a word  into the  exponent vector.   Here  we cannot use the
**  global exponent because the beginning of  the word might not commute with
**  the rest.
**/
static libGAP_Int libGAP_C32Bits_SAddWordIntoExpVec( libGAP_Int *v, libGAP_UInt4 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Obj *ro, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt4 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;
    libGAP_Int        start = 0;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( libGAP_INT_INTOBJ(ro[i]) <= v[i] ) {
            ex = v[i] / libGAP_INT_INTOBJ(ro[i]);
            v[i] -= ex * libGAP_INT_INTOBJ(ro[i]);
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                start = libGAP_C32Bits_SAddWordIntoExpVec( 
                    v, (libGAP_UInt4*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, ro, pow, lpow  );
            }
        }
        if( start < i && v[i] ) start = i;
    }
    return start;
}

static libGAP_Int libGAP_C32Bits_SAddPartIntoExpVec( libGAP_Int *v, libGAP_UInt4 *w, libGAP_UInt4 *wend,
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Obj* ro, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_Int        i;
    libGAP_Int        ex;
    libGAP_Int        start = 0;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm);     /* overflow check necessary? */
        if ( libGAP_INT_INTOBJ(ro[i]) <= v[i] ) {
            ex = v[i] / libGAP_INT_INTOBJ(ro[i]);
            v[i] -= ex * libGAP_INT_INTOBJ(ro[i]);
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                start = libGAP_C32Bits_SAddWordIntoExpVec( 
                    v, (libGAP_UInt4*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, ro, pow, lpow  );
            }
        }
        if( start < i && v[i] ) start = i;
    }
    return start;
}

libGAP_Int libGAP_C32Bits_SingleCollectWord ( libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    libGAP_Int         ebits;      /* number of bits in the exponent              */
    libGAP_UInt        expm;       /* unsigned exponent mask                      */
    libGAP_UInt        exps;       /* sign exponent mask                          */

    libGAP_Obj         vnw;        /* word stack                                  */
    libGAP_UInt4 **    nw;         /* address of <vnw>                            */
    libGAP_Obj         vlw;        /* last syllable stack                         */
    libGAP_UInt4 **    lw;         /* address of <vlw>                            */
    libGAP_Obj         vpw;        /* current syllable stack                      */
    libGAP_UInt4 **    pw;         /* address of <vpw>                            */
    libGAP_Obj         vew;        /* unprocessed exponent stack                  */
    libGAP_UInt4 *     ew;         /* address of <vew>                            */
    libGAP_Obj         vge;        /* global exponent stack                       */
    libGAP_Int *       ge;         /* address of <vge>                            */

    libGAP_Obj         vpow;       /* rhs of power relations                      */
    libGAP_Int         lpow;       /* length of <vpow>                            */
    libGAP_Obj *       pow;        /* address of <vpow>                           */

    libGAP_Obj         vcnj;       /* rhs of conjugate relations                  */
    libGAP_Int         lcnj;       /* length of <vcnj>                            */
    libGAP_Obj *       cnj;        /* address of <vcnj>                           */

    libGAP_Obj *       avc;        /* address of the avector                      */
    libGAP_Obj *       gns;        /* address of the list of generators           */
    libGAP_Obj *       ro;         /* address of the list of relative orders      */
    libGAP_Obj *       inv;        /* address of the list of inverses             */

    libGAP_Int *       v;          /* address of <vv>                             */

    libGAP_Int         max;        /* maximal stack size                          */
    libGAP_Int         sp;         /* stack pointer                               */
    libGAP_Int         i, j;       /* loop variable                               */
    libGAP_Int         gn;         /* current generator number                    */
    libGAP_Int         ex;         /* current exponent                            */
    libGAP_Int         start;      /* last non-trivial entry                      */

    libGAP_Obj         tmp;        /* temporary obj for power                     */

    libGAP_Int         resized = 0;/* indicates whether a Resize() happened       */

    /* <start> is the first non-trivial entry in <v>                       */
    start = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* if <w> is the identity return now                                   */
    if ( libGAP_NPAIRS_WORD(w) == 0 ) {
        return start;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* get the exponent sign masks                                         */
    exps = 1UL << (ebits-1);

    /* <nw> contains the stack of words to insert                          */
    vnw = libGAP_SC_NW_STACK(sc);

    /* <lw> contains the word end of the word in <nw>                      */
    vlw = libGAP_SC_LW_STACK(sc);

    /* <pw> contains the position of the word in <nw> to look at           */
    vpw = libGAP_SC_PW_STACK(sc);

    /* <ew> contains the unprocessed exponents at position <pw>            */
    vew = libGAP_SC_EW_STACK(sc);

    /* <ge> contains the global exponent of the word                       */
    vge = libGAP_SC_GE_STACK(sc);

    /* get the maximal stack size                                          */
    max = libGAP_SC_MAX_STACK_SIZE(sc);

    /* ensure that the stacks are large enough                             */
    if ( libGAP_SIZE_OBJ(vnw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vnw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vnw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vlw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vlw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vlw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vpw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vpw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vpw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vew)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vew, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vew, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vge)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vge, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vge, libGAP_T_STRING );
        resized = 1;
    }
    if( resized ) return -1;

    /* from now on we use addresses instead of handles most of the time    */
    v  = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    nw = (libGAP_UInt4**)libGAP_ADDR_OBJ(vnw);
    lw = (libGAP_UInt4**)libGAP_ADDR_OBJ(vlw);
    pw = (libGAP_UInt4**)libGAP_ADDR_OBJ(vpw);
    ew = (libGAP_UInt4*)libGAP_ADDR_OBJ(vew);
    ge = (libGAP_Int*)libGAP_ADDR_OBJ(vge);

    /* conjujagtes, powers, order, generators, avector, inverses           */
    vpow = libGAP_SC_POWERS(sc);
    lpow = libGAP_LEN_PLIST(vpow);
    pow  = libGAP_ADDR_OBJ(vpow);

    vcnj = libGAP_SC_CONJUGATES(sc);
    lcnj = libGAP_LEN_PLIST(vcnj);
    (void) lcnj; /* please compiler -- lcnj not actually used */
    cnj  = libGAP_ADDR_OBJ(vcnj);

    avc = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR(sc) );
    gns = libGAP_ADDR_OBJ( libGAP_SC_RWS_GENERATORS(sc) );

    ro  = libGAP_ADDR_OBJ( libGAP_SC_RELATIVE_ORDERS(sc) );
    inv = libGAP_ADDR_OBJ( libGAP_SC_INVERSES(sc) );

    /* initialize the stack with <w>                                        */
    sp = 0;
    libGAP_SC_PUSH_WORD( w, 1 );

    /* run until the stack is empty                                        */
    while ( 0 < sp ) {

        /* if <ew> is negative use inverse                                 */
        if ( *ew & exps ) {
            gn = ((**pw) >> ebits) + 1;
            ex = ( *ew & (exps-1) ) - exps;
            *ew = 0;
            libGAP_SC_PUSH_WORD( inv[gn], -ex );
        }

        /* if <ew> is zero get next syllable                               */
        else if ( 0 == *ew ) {

            /* if <pw> has reached <lw> get next & reduce globale exponent */
            if ( *pw == *lw ) {

                /* if the globale exponent is greater one reduce it        */
                if ( 1 < *ge ) {
                    (*ge)--;
                    *pw = *nw;
                    *ew = (**pw) & expm;
                }

                /* otherwise get the next word from the stack              */
                else {
                    libGAP_SC_POP_WORD();
                }
            }

            /* otherwise set <ew> to exponent of next syllable             */
            else {
                (*pw)++;
                *ew = (**pw) & expm;
            }
        }

        /* now move the next generator to the correct position             */
        else {

            /* get generator number                                        */
            gn = ((**pw) >> ebits) + 1;

            /* we can move <gn> directly to the correct position           */
            if ( libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              /*
              *T  This if-statemant implies that the next two cases are never
              *T  executed.  This is intended for the time being because we 
              *T  need the single collector to work with pc-presentation
              *T  whose rhs are not reduced while the next two if-case need
              *T  reduced rhs.  This will be fixed at a later stage.
              */
                v[gn] += *ew;
                *ew = 0;
                if ( start <= gn )
                    start = gn;
            }

            /* collect a whole word exponent pair                          */
            else if( *pw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              gn = libGAP_C32Bits_SAddWordIntoExpVec( 
                   v, *pw, *ge, ebits, expm, ro, pow, lpow  );
              *pw = *lw;
              *ew = *ge = 0;

              if( start <= gn ) start = gn;
              continue;
            }

            /* move the rest of a word directly into the correct positions */
            else if( libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              gn = libGAP_C32Bits_SAddPartIntoExpVec( 
                   v, *pw, *lw, ebits, expm, ro, pow, lpow  );
              *pw = *lw;
              *ew = 0;

              if( start <= gn ) start = gn;
              continue;
            }

            /* we have to move <gn> step by step                           */
            else {
                (*ew)--; v[gn]++;

                i = libGAP_INT_INTOBJ(avc[gn]);
                if ( start < i )
                    i = start;

                /* Find the first position in v from where on ordinary
                   collection  has to be applied.                          */
                for( ; gn < i; i-- )
                    if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                            break;
                    }

                /* Stack up this part of v if we run through the next 
                   for-loop or if a power relation will be applied         */
                if( gn < i || (libGAP_INT_INTOBJ(ro[gn]) <= v[gn] &&
                    gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn])) ) {
                    j = libGAP_INT_INTOBJ(avc[gn]);
                    for( ; i < j; j-- )
                        if( v[j] ) {
                            libGAP_SC_PUSH_WORD( gns[j], v[j] );
                            v[j] = 0;
                        }
                }

                if( gn < i ) {
                  for ( ;  gn < i;  i-- ) {
                    if ( v[i] ) {
                        if ( libGAP_LEN_PLIST(cnj[i]) < gn )
                            tmp = gns[i];
                        else {
                            tmp = libGAP_ELM_PLIST( cnj[i], gn );
                            if ( tmp == 0 || libGAP_NPAIRS_WORD(tmp) == 0 )
                                tmp = gns[i];
                        }
                        libGAP_SC_PUSH_WORD( tmp, v[i] );
                        v[i] = 0;
                    }
                  }
                  if ( start <= libGAP_INT_INTOBJ(avc[gn]) )
                    start = gn;
                }
                if( start <= gn ) start = gn;
            }

            /* check that the exponent is not too big                      */
            if ( libGAP_INT_INTOBJ(ro[gn]) <= v[gn] ) {
                i = v[gn] / libGAP_INT_INTOBJ(ro[gn]);
                v[gn] -= i * libGAP_INT_INTOBJ(ro[gn]);
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_SC_PUSH_WORD( pow[gn], i );
                }
            }
        }
    }
    return start;
}
#undef libGAP_SC_PUSH_WORD
#undef libGAP_SC_POP_WORD


/****************************************************************************
**
*F  C32Bits_Solution( <sc>, <ww>, <uu>, <func> )
*/
libGAP_Int libGAP_C32Bits_Solution( 
    libGAP_Obj         sc,
    libGAP_Obj         ww,
    libGAP_Obj         uu,
    libGAP_FuncIOOO    func )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         ro;             /* relative order                          */
    libGAP_Obj         rod;            /* relative orders                         */
    libGAP_Obj         libGAP_g;              /* one generator word                      */
    libGAP_UInt4 *     gtr;            /* pointer into the data area of <g>       */
    libGAP_Int *       ptr;            /* pointer into the collect vector         */
    libGAP_Int *       qtr;            /* pointer into the collect vector         */

    /* get the number of generators                                        */
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    rod = libGAP_SC_RELATIVE_ORDERS(sc);

    /* <ww> must be a string                                               */
    if ( libGAP_TNUM_OBJ(ww) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(ww) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( ww, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(ww), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(ww) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        i = (libGAP_SIZE_OBJ(ww)-sizeof(libGAP_Obj)-1) / sizeof(libGAP_Int);
        libGAP_ResizeBag( ww, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(ww)+1);
        for ( i = i+1;  i <= num;  i++ )
            qtr[i] = 0;
    }

    /* <uu> must be a string                                               */
    if ( libGAP_TNUM_OBJ(uu) != libGAP_T_STRING ) {
        if ( libGAP_TNUM_OBJ(uu) == libGAP_IMMUTABLE_TNUM(libGAP_T_STRING) ) {
            libGAP_RetypeBag( uu, libGAP_T_STRING );
        }
        else {
            libGAP_ErrorQuit( "collect vector must be a string not a %s", 
                       (libGAP_Int)libGAP_TNAM_OBJ(uu), 0L );
            return -1;
        }
    }

    /* fix the length                                                      */
    if ( libGAP_SIZE_OBJ(uu) != num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 ) {
        i = (libGAP_SIZE_OBJ(uu)-sizeof(libGAP_Obj)-1) / sizeof(libGAP_Int);
        libGAP_ResizeBag( uu, num*sizeof(libGAP_Int)+sizeof(libGAP_Obj)+1 );
        qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(uu)+1);
        for ( i = i+1;  i <= num;  i++ )
            qtr[i] = 0;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* use <g> as right argument for the collector                         */
    libGAP_NEW_WORD( libGAP_g, libGAP_SC_DEFAULT_TYPE(sc), 1 );

    /* start clearing <ww>, storing the result in <uu>                     */
    ptr = (libGAP_Int*)(libGAP_ADDR_OBJ(ww)+1);
    qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(uu)+1);
    gtr = (libGAP_UInt4*)libGAP_DATA_WORD(libGAP_g);
    for ( i = num;  0 < i;  i--, ptr++, qtr++ ) {
        ro = libGAP_INT_INTOBJ(libGAP_ELMW_LIST(rod,num-i+1));
        *qtr = ( *qtr - *ptr ) % ro;
        if ( *qtr < 0 )  *qtr += ro;
        if ( *qtr != 0 ) {
            *gtr = ( (num-i) << ebits ) | ( *qtr & expm );
            if ( func(sc,ww,libGAP_g) == -1 )
                return -1;
        }
        *ptr = 0;
    }
    return 0;
}


/****************************************************************************
**
*V  C32Bits_SingleCollector
*/
libGAP_FinPowConjCol libGAP_C32Bits_SingleCollector = {
    libGAP_C32Bits_WordVectorAndClear,
    libGAP_C32Bits_VectorWord,
    libGAP_C32Bits_SingleCollectWord,
    libGAP_C32Bits_Solution
};

/****************************************************************************
**
*F * * * * * * * * * * *  combinatorial collectors  * * * * * * * * * * * * *
**
**  Here the combinatorial collectors are setup.  They behave like single
**  collectors and therefore can be used int the same way.
*/

/****************************************************************************
**

*V  C8Bits_CombiCollector
*/
libGAP_FinPowConjCol libGAP_C8Bits_CombiCollector = {
    libGAP_C8Bits_WordVectorAndClear,
    libGAP_C8Bits_VectorWord,
    libGAP_C8Bits_CombiCollectWord,
    libGAP_C8Bits_Solution
};

/****************************************************************************
**

*V  C16Bits_CombiCollector
*/
libGAP_FinPowConjCol libGAP_C16Bits_CombiCollector = {
    libGAP_C16Bits_WordVectorAndClear,
    libGAP_C16Bits_VectorWord,
    libGAP_C16Bits_CombiCollectWord,
    libGAP_C16Bits_Solution
};

/****************************************************************************
**

*V  C32Bits_CombiCollector
*/
libGAP_FinPowConjCol libGAP_C32Bits_CombiCollector = {
    libGAP_C32Bits_WordVectorAndClear,
    libGAP_C32Bits_VectorWord,
    libGAP_C32Bits_CombiCollectWord,
    libGAP_C32Bits_Solution
};

/****************************************************************************
**

*V  FinPowConjCollectors
*/
libGAP_FinPowConjCol * libGAP_FinPowConjCollectors [6] =
{
#define libGAP_C8Bits_SingleCollectorNo        0
       &libGAP_C8Bits_SingleCollector,
#define libGAP_C16Bits_SingleCollectorNo       1
       &libGAP_C16Bits_SingleCollector,
#define libGAP_C32Bits_SingleCollectorNo       2
       &libGAP_C32Bits_SingleCollector,
#define libGAP_C8Bits_CombiCollectorNo         3
       &libGAP_C8Bits_CombiCollector,
#define libGAP_C16Bits_CombiCollectorNo        4
       &libGAP_C16Bits_CombiCollector,
#define libGAP_C32Bits_CombiCollectorNo        5
       &libGAP_C32Bits_CombiCollector
};

/****************************************************************************
**

*F * * * * * * * * * * * * reduce something functions * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  CollectWordOrFail( <fc>, <sc>, <vv>, <w> )
*/
libGAP_Obj libGAP_CollectWordOrFail ( 
    libGAP_FinPowConjCol *     fc, 
    libGAP_Obj                 sc,
    libGAP_Obj                 vv,
    libGAP_Obj                 w )
{
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Obj *               ptr;            /* pointer into the array <vv>     */

    /* convert <vv> into a list of C integers                              */
    ptr = libGAP_ADDR_OBJ(vv)+1;
    for ( i = libGAP_LEN_PLIST(vv);  0 < i;  i--, ptr++ )
        *ptr = (libGAP_Obj)libGAP_INT_INTOBJ(*ptr);

    /* now collect <w> into <vv>                                           */
    if ( fc->collectWord( sc, vv, w ) == -1 ) {
         /* If the collector fails, we return the vector clean.            */
        ptr = libGAP_ADDR_OBJ(vv)+1;
        for ( i = libGAP_LEN_PLIST(vv);  0 < i;  i--, ptr++ )
            *ptr = libGAP_INTOBJ_INT(0);

        return libGAP_Fail;
    }

    /* and convert back                                                    */
    ptr = libGAP_ADDR_OBJ(vv)+1;
    for ( i = libGAP_LEN_PLIST(vv);  0 < i;  i--, ptr++ )
        *ptr = libGAP_INTOBJ_INT((libGAP_Int)*ptr);

    return libGAP_True;
}


/****************************************************************************
**
*F  ReducedComm( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedComm (
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 kind;       /* kind of the returned object         */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <u>*<w> to                           */
    vcw = libGAP_SC_CW_VECTOR(sc);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length, unpack <u> into it            */
    if ( fc->vectorWord( vcw, u, num ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <w> into it                                                 */
    if ( fc->collectWord( sc, vcw, w ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedComm( fc, sc, w, u );
    }

    /* use 'cw2Vector' to collect word <w>*<u> to                          */
    vc2 = libGAP_SC_CW2_VECTOR(sc);

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vc2, w, num ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <u> into it                                                 */
    if ( fc->collectWord( sc, vc2, u ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedComm( fc, sc, w, u );
    }

    /* now use 'Solution' to solve the equation, will clear <vcw>          */
    if ( fc->solution( sc, vcw, vc2, fc->collectWord ) == -1 )
    {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedComm( fc, sc, w, u );
    }

    /* convert the vector <vc2> into a word and clear <vc2>                */
    kind = libGAP_SC_DEFAULT_TYPE(sc);
    return fc->wordVectorAndClear( kind, vc2, num );
}


/****************************************************************************
**
*F  ReducedForm( <fc>, <sc>, <w> )
*/
libGAP_Obj libGAP_ReducedForm (
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w )
{
    libGAP_Int                 num;    /* number of gen/exp pairs in <data>       */
    libGAP_Int                 i;      /* loop variable for gen/exp pairs         */
    libGAP_Obj                 vcw;    /* collect vector                          */
    libGAP_Obj                 kind;   /* kind of the return objue                */
    libGAP_Int *               qtr;    /* pointer into the collect vector         */

    /* use 'cwVector' to collect word <w> to                               */
    vcw = libGAP_SC_CW_VECTOR(sc);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length                                */
    if ( fc->vectorWord( vcw, 0, num ) == -1 )
        return libGAP_Fail;

    /* and collect <w> into it                                             */
    while ( (i = fc->collectWord( sc, vcw, w )) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
    }
    num = i;

    /* get the default kind                                                */
    kind = libGAP_SC_DEFAULT_TYPE(sc);

    /* convert the vector <cvw> into a word and clear <vcw>                */
    return fc->wordVectorAndClear( kind, vcw, num );
}


/****************************************************************************
**
*F  ReducedLeftQuotient( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedLeftQuotient ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 kind;       /* kind of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <w> to                               */
    vcw = libGAP_SC_CW_VECTOR(sc);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vcw, w, num ) == -1 )  {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* use 'cw2Vector' to collect word <u> to                              */
    vc2 = libGAP_SC_CW2_VECTOR(sc);

    /* check that it has the correct length, unpack <u> into it            */
    if ( fc->vectorWord( vc2, u, num ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* now use 'Solution' to solve the equation, will clear <vcw>          */
    if ( fc->solution( sc, vcw, vc2, fc->collectWord ) == -1 )
    {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedLeftQuotient( fc, sc, w, u );
    }

    /* convert the vector <vc2> into a word and clear <vc2>                */
    kind = libGAP_SC_DEFAULT_TYPE(sc);
    return fc->wordVectorAndClear( kind, vc2, num );
}


/****************************************************************************
**
*F  ReducedProduct( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedProduct ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 kind;       /* kind of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <w> to                               */
    vcw = libGAP_SC_CW_VECTOR(sc);
    num = libGAP_SC_NUMBER_RWS_GENERATORS(sc);

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vcw, w, num ) == -1 )  {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <w> into it                                                 */
    if ( fc->collectWord( sc, vcw, u ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedProduct( fc, sc, w, u );
    }

    /* convert the vector <vcw> into a word and clear <vcw>                */
    kind = libGAP_SC_DEFAULT_TYPE(sc);
    return fc->wordVectorAndClear( kind, vcw, num );
}


/****************************************************************************
**
*F  ReducedPowerSmallInt( <fc>, <sc>, <w>, <pow> )
*/
libGAP_Obj libGAP_ReducedPowerSmallInt ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 vpow )
{
    libGAP_Obj                 kind;       /* kind of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int                 pow;        /* power to raise <w> to               */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */
    libGAP_Obj                 res;        /* the result                          */

    /* get the integer of <vpow>                                           */
    pow = libGAP_INT_INTOBJ(vpow);

    /* use 'cwVector' and 'cw2Vector to collect words to                   */
    vcw  = libGAP_SC_CW_VECTOR(sc);
    vc2  = libGAP_SC_CW2_VECTOR(sc);
    num  = libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    kind = libGAP_SC_DEFAULT_TYPE(sc);

    /* return the trivial word if <pow> is zero                            */
    if ( pow == 0 ) {
        libGAP_NEW_WORD( res, kind, 0 );
        return res;
    }

    /* invert <w> if <pow> is negative                                     */
    if ( pow < 0 ) {
        
        /* check that it has the correct length, unpack <w> into it        */
        if ( fc->vectorWord( vcw, w, num ) == -1 )  {
            for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--,qtr++ )
                *qtr = 0;
            return libGAP_Fail;
        }

        /* use 'Solution' to invert it, this will clear <vcw>              */
        if (fc->solution(sc,vcw,vc2,fc->collectWord) == -1) {
                for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0<i;  i--,qtr++ )
                    *qtr = 0;
                for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0<i;  i--,qtr++ )
                    *qtr = 0;
                return libGAP_ReducedPowerSmallInt(fc,sc,w,vpow);
        }

        /* and replace <pow> and <w> by its inverse                        */
        pow  = -pow;
        vpow = libGAP_INTOBJ_INT(pow);
        w    = fc->wordVectorAndClear( kind, vc2, num );

    }

    /* if <pow> is one, do nothing                                         */
    if ( pow == 1 ) {
        return w;
    }

    /* catch small cases                                                   */
    if ( pow < 6 ) {

        /* check that it has the correct length, unpack <w> into it        */
        if ( fc->vectorWord( vcw, w, num ) == -1 )  {
            for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--,qtr++ )
                *qtr = 0;
            return libGAP_Fail;
        }

        /* multiply <w> into <vcw>                                         */
        for ( i = pow;  1 < i;  i-- ) {
            if ( fc->collectWord( sc, vcw, w ) == -1 ) {
                for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0<i;  i--,qtr++ )
                    *qtr = 0;
                return libGAP_ReducedPowerSmallInt(fc,sc,w,vpow);
            }
        }

        /* convert it back, this will clear <vcw>                          */
        return fc->wordVectorAndClear( kind, vcw, num );

    }

    /* use "divide et impera" instead of repeated squaring r2l             */
    if ( pow % 2 ) {
        res = libGAP_ReducedPowerSmallInt( fc, sc, w, libGAP_INTOBJ_INT((pow-1)/2) );
        return libGAP_ReducedProduct( fc, sc, w,
            libGAP_ReducedProduct( fc, sc, res, res ) );
    }
    else {
        res = libGAP_ReducedPowerSmallInt( fc, sc, w, libGAP_INTOBJ_INT(pow/2) );
        return libGAP_ReducedProduct( fc, sc, res, res );
    }

}


/****************************************************************************
**
*F  ReducedQuotient( <fc>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_ReducedQuotient ( 
    libGAP_FinPowConjCol *     fc,
    libGAP_Obj                 sc,
    libGAP_Obj                 w,
    libGAP_Obj                 u )
{
    libGAP_Obj                 kind;       /* kind of the return objue            */
    libGAP_Int                 num;        /* number of gen/exp pairs in <data>   */
    libGAP_Int                 i;          /* loop variable for gen/exp pairs     */
    libGAP_Obj                 vcw;        /* collect vector                      */
    libGAP_Obj                 vc2;        /* collect vector                      */
    libGAP_Int *               qtr;        /* pointer into the collect vector     */

    /* use 'cwVector' to collect word <w> to                               */
    vcw  = libGAP_SC_CW_VECTOR(sc);
    vc2  = libGAP_SC_CW2_VECTOR(sc);
    num  = libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    kind = libGAP_SC_DEFAULT_TYPE(sc);

    /* check that it has the correct length, unpack <u> into it            */
    if ( fc->vectorWord( vcw, u, num ) == -1 )  {
        for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--,qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* use 'Solution' to invert it, this will clear <vcw>                  */
    if ( fc->solution( sc, vcw, vc2, fc->collectWord ) == -1 ) {
        for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0<i;  i--,qtr++ )
            *qtr = 0;
        for ( i=num, qtr=(libGAP_Int*)(libGAP_ADDR_OBJ(vc2)+1);  0<i;  i--,qtr++ )
            *qtr = 0;
        return libGAP_ReducedQuotient( fc, sc, w, u );
    }

    /* and replace <u> by its inverse                                      */
    u = fc->wordVectorAndClear( kind, vc2, num );

    /* check that it has the correct length, unpack <w> into it            */
    if ( fc->vectorWord( vcw, w, num ) == -1 )  {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_Fail;
    }

    /* collect <w> into it                                                 */
    if ( fc->collectWord( sc, vcw, u ) == -1 ) {
        for ( i = num, qtr = (libGAP_Int*)(libGAP_ADDR_OBJ(vcw)+1);  0 < i;  i--, qtr++ )
            *qtr = 0;
        return libGAP_ReducedQuotient( fc, sc, w, u );
    }

    /* convert the vector <vcw> into a word and clear <vcw>                */
    return fc->wordVectorAndClear( kind, vcw, num );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * exported GAP functions * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  FuncFinPowConjCol_CollectWordOrFail( <self>, <sc>, <vv>, <w> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_CollectWordOrFail ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    return libGAP_CollectWordOrFail( libGAP_SC_COLLECTOR(sc), sc, vv, w );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedComm( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedComm ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedComm( libGAP_SC_COLLECTOR(sc), sc, w, u );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedForm( <self>, <sc>, <w> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedForm ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w )
{
    return libGAP_ReducedForm( libGAP_SC_COLLECTOR(sc), sc, w );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedLeftQuotient( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedLeftQuotient ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedLeftQuotient( libGAP_SC_COLLECTOR(sc), sc, w, u );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedProduct( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedProduct ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedProduct( libGAP_SC_COLLECTOR(sc), sc, w, u );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedPowerSmallInt( <self>, <sc>, <w>, <pow> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedPowerSmallInt (libGAP_Obj self,libGAP_Obj sc,libGAP_Obj w,libGAP_Obj vpow)
{
    return libGAP_ReducedPowerSmallInt( libGAP_SC_COLLECTOR(sc), sc, w, vpow );
}


/****************************************************************************
**
*F  FuncFinPowConjCol_ReducedQuotient( <self>, <sc>, <w>, <u> )
*/
libGAP_Obj libGAP_FuncFinPowConjCol_ReducedQuotient ( libGAP_Obj self, libGAP_Obj sc, libGAP_Obj w, libGAP_Obj u )
{
    return libGAP_ReducedQuotient( libGAP_SC_COLLECTOR(sc), sc, w, u );
}




/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "FinPowConjCol_CollectWordOrFail", 3, "sc, list, word",
      libGAP_FuncFinPowConjCol_CollectWordOrFail, 
      "src/objscoll.c:FinPowConjCol_CollectWordOrFail" },

    { "FinPowConjCol_ReducedComm", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedComm, 
      "src/objscoll.c:FinPowConjCol_ReducedComm" },

    { "FinPowConjCol_ReducedForm", 2, "sc, word",
      libGAP_FuncFinPowConjCol_ReducedForm, 
      "src/objscoll.c:FinPowConjCol_ReducedForm" },

    { "FinPowConjCol_ReducedLeftQuotient", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedLeftQuotient, 
      "src/objscoll.c:FinPowConjCol_ReducedLeftQuotient" },

    { "FinPowConjCol_ReducedPowerSmallInt", 3, "sc, word, int",
      libGAP_FuncFinPowConjCol_ReducedPowerSmallInt,
      "src/objscoll.c:FinPowConjCol_ReducedPowerSmallInt" },

    { "FinPowConjCol_ReducedProduct", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedProduct,
      "src/objscoll.c:FinPowConjCol_ReducedProduct" },

    { "FinPowConjCol_ReducedQuotient", 3, "sc, word, word",
      libGAP_FuncFinPowConjCol_ReducedQuotient,
      "src/objscoll.c:FinPowConjCol_ReducedQuotient" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* export position numbers 'SCP_SOMETHING'                             */
    libGAP_AssGVar( libGAP_GVarName( "SCP_UNDERLYING_FAMILY" ),
             libGAP_INTOBJ_INT(libGAP_SCP_UNDERLYING_FAMILY) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_RWS_GENERATORS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_RWS_GENERATORS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_NUMBER_RWS_GENERATORS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_NUMBER_RWS_GENERATORS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_DEFAULT_TYPE" ),
             libGAP_INTOBJ_INT(libGAP_SCP_DEFAULT_TYPE) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_IS_DEFAULT_TYPE" ),
             libGAP_INTOBJ_INT(libGAP_SCP_IS_DEFAULT_TYPE) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_RELATIVE_ORDERS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_RELATIVE_ORDERS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_POWERS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_POWERS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_CONJUGATES" ),
             libGAP_INTOBJ_INT(libGAP_SCP_CONJUGATES) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_INVERSES" ),
             libGAP_INTOBJ_INT(libGAP_SCP_INVERSES) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_NW_STACK" ),
             libGAP_INTOBJ_INT(libGAP_SCP_NW_STACK) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_LW_STACK" ),
             libGAP_INTOBJ_INT(libGAP_SCP_LW_STACK) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_PW_STACK" ),
             libGAP_INTOBJ_INT(libGAP_SCP_PW_STACK) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_EW_STACK" ),
             libGAP_INTOBJ_INT(libGAP_SCP_EW_STACK) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_GE_STACK" ),
             libGAP_INTOBJ_INT(libGAP_SCP_GE_STACK) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_CW_VECTOR" ),
             libGAP_INTOBJ_INT(libGAP_SCP_CW_VECTOR) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_CW2_VECTOR" ),
             libGAP_INTOBJ_INT(libGAP_SCP_CW2_VECTOR) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_MAX_STACK_SIZE" ),
             libGAP_INTOBJ_INT(libGAP_SCP_MAX_STACK_SIZE) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_COLLECTOR" ),
             libGAP_INTOBJ_INT(libGAP_SCP_COLLECTOR) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_AVECTOR" ),
             libGAP_INTOBJ_INT(libGAP_SCP_AVECTOR) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_WEIGHTS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_WEIGHTS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_CLASS" ),
             libGAP_INTOBJ_INT(libGAP_SCP_CLASS) );
    libGAP_AssGVar( libGAP_GVarName( "SCP_AVECTOR2" ),
             libGAP_INTOBJ_INT(libGAP_SCP_AVECTOR2) );

    /* export collector number                                             */
    libGAP_AssGVar( libGAP_GVarName( "8Bits_SingleCollector" ),
             libGAP_INTOBJ_INT(libGAP_C8Bits_SingleCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "16Bits_SingleCollector" ),
             libGAP_INTOBJ_INT(libGAP_C16Bits_SingleCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "32Bits_SingleCollector" ),
             libGAP_INTOBJ_INT(libGAP_C32Bits_SingleCollectorNo) );

    libGAP_AssGVar( libGAP_GVarName( "8Bits_CombiCollector" ),
             libGAP_INTOBJ_INT(libGAP_C8Bits_CombiCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "16Bits_CombiCollector" ),
             libGAP_INTOBJ_INT(libGAP_C16Bits_CombiCollectorNo) );
    libGAP_AssGVar( libGAP_GVarName( "32Bits_CombiCollector" ),
             libGAP_INTOBJ_INT(libGAP_C32Bits_CombiCollectorNo) );

    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoSingleCollector() . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "objscoll",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoSingleCollector ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  objscoll.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
