/****************************************************************************
**
*W  gmpints.c                   GAP source                     John McDermott
**                                                           
**                                                           
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file implements the  functions  handling  GMP integers.
**
*/
#include        <sys/mman.h>
#include        "system.h"              /* Ints, UInts                     */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "gap.h"                 /* error handling, initialisation  */
#include        "code.h"                /* needed by stats.h */
#include        "stats.h"               /* for TakeInterrupt               */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */

#include        "intfuncs.h"

#include <stdio.h>

#ifdef HAVE_MATH_H
#include <math.h>
#endif

#include <stdlib.h>

#include <assert.h>
#include <string.h>
#include <ctype.h>


#ifdef USE_GMP

// GMP must be included outside of 'extern C'
#ifdef libGAP_GAP_IN_EXTERN_C
}
#endif
#include <gmp.h>
#ifdef libGAP_GAP_IN_EXTERN_C
extern "C" {
#endif

#include        "gmpints.h"             /* GMP integers                    */


/* macros to save typing later :)                                          */
#define libGAP_VAL_LIMB0(obj)         ( *(libGAP_TypLimb *)libGAP_ADDR_OBJ(obj)                  )
#define libGAP_SET_VAL_LIMB0(obj,val) ( *(libGAP_TypLimb *)libGAP_ADDR_OBJ(obj) = val            )
#define libGAP_IS_INTPOS(obj)         (  libGAP_TNUM_OBJ(obj) == libGAP_T_INTPOS                 )
#define libGAP_IS_INTNEG(obj)         (  libGAP_TNUM_OBJ(obj) == libGAP_T_INTNEG                 )
#define libGAP_IS_LARGEINT(obj)       (  ( libGAP_TNUM_OBJ(obj) == libGAP_T_INTPOS ) || \
                                  ( libGAP_TNUM_OBJ(obj) == libGAP_T_INTNEG )             )

/* for fallbacks to library */
libGAP_Obj libGAP_String;


/****************************************************************************
**
*F  TypeInt(<gmp>)  . . . . . . . . . . . . . . . . . . .  type of integer
**
**  'TypeInt' returns the type of the integer <gmp>.
**
**  'TypeInt' is the function in 'TypeObjFuncs' for integers.
*/
libGAP_Obj             libGAP_TYPE_INT_SMALL_ZERO;
libGAP_Obj             libGAP_TYPE_INT_SMALL_POS;
libGAP_Obj             libGAP_TYPE_INT_SMALL_NEG;
libGAP_Obj             libGAP_TYPE_INT_LARGE_POS;
libGAP_Obj             libGAP_TYPE_INT_LARGE_NEG;

libGAP_Obj             libGAP_TypeIntSmall (
    libGAP_Obj                 val )
{
    if ( 0 == libGAP_INT_INTOBJ(val) ) {
        return libGAP_TYPE_INT_SMALL_ZERO;
    }
    else if ( 0 < libGAP_INT_INTOBJ(val) ) {
        return libGAP_TYPE_INT_SMALL_POS;
    }
    else {
        return libGAP_TYPE_INT_SMALL_NEG;
    }
}

libGAP_Obj libGAP_TypeIntLargePos ( libGAP_Obj val )
{
    return libGAP_TYPE_INT_LARGE_POS;
}

libGAP_Obj libGAP_TypeIntLargeNeg ( libGAP_Obj val )
{
    return libGAP_TYPE_INT_LARGE_NEG;
}


/****************************************************************************
**
*F  FuncIS_INT( <self>, <val> ) . . . . . . . . . . internal function 'IsInt'
**
**  'FuncIS_INT' implements the internal filter 'IsInt'.
**
**  'IsInt( <val> )'
**
**  'IsInt'  returns 'true'  if the  value  <val>  is an small integer or a
**  large int, and 'false' otherwise.
*/
libGAP_Obj libGAP_IsIntFilt;

libGAP_Obj libGAP_FuncIS_INT ( libGAP_Obj self, libGAP_Obj val )
{
  if (    libGAP_TNUM_OBJ(val) == libGAP_T_INT 
       || libGAP_TNUM_OBJ(val) == libGAP_T_INTPOS
       || libGAP_TNUM_OBJ(val) == libGAP_T_INTNEG ) {
    return libGAP_True;
  }
  else if ( libGAP_TNUM_OBJ(val) <= libGAP_FIRST_EXTERNAL_TNUM ) {
    return libGAP_False;
  }
  else {
    return libGAP_DoFilter( self, val );
  }
}


/****************************************************************************
**
*F  SaveInt( <gmp> )
**
**  
*/
void libGAP_SaveInt( libGAP_Obj gmp )
{
  libGAP_TypLimb *ptr;
  libGAP_UInt i;
  ptr = (libGAP_TypLimb *)libGAP_ADDR_INT(gmp);
  for (i = 0; i < libGAP_SIZE_INT(gmp); i++)
    libGAP_SaveLimb(*ptr++);
  return;
}


/****************************************************************************
**
*F  LoadInt( <gmp> )
**
**
*/
void libGAP_LoadInt( libGAP_Obj gmp )
{
  libGAP_TypLimb *ptr;
  libGAP_UInt i;
  ptr = (libGAP_TypLimb *)libGAP_ADDR_INT(gmp);
  for (i = 0; i < libGAP_SIZE_INT(gmp); i++)
    *ptr++ = libGAP_LoadLimb();
  return;
}


/****************************************************************************
**
*F  NEW_INT( <gmp> )
**
**
*/
static inline libGAP_Obj libGAP_NEW_INT( libGAP_Obj gmp )
{
  libGAP_Obj new;

  new = libGAP_NewBag( libGAP_TNUM_OBJ(gmp), libGAP_SIZE_OBJ(gmp) );
  memcpy( libGAP_ADDR_INT(new), libGAP_ADDR_INT(gmp), libGAP_SIZE_OBJ(gmp) );

  return new;
}


/****************************************************************************
**
*F  NEW_INTPOS( <gmp> )
**
**
*/
static inline libGAP_Obj libGAP_NEW_INTPOS( libGAP_Obj gmp )
{
  libGAP_Obj new;

  new = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmp) );
  memcpy( libGAP_ADDR_INT(new), libGAP_ADDR_INT(gmp), libGAP_SIZE_OBJ(gmp) );

  return new;
}


/****************************************************************************
**
*F  NEW_INTNEG( <gmp> )
**
**
**
static inline Obj NEW_INTNEG( Obj gmp )
{
  Obj new;

  new = NewBag( T_INTNEG, SIZE_OBJ(gmp) );
  memcpy( ADDR_INT(new), ADDR_INT(gmp), SIZE_OBJ(gmp) );

  return new;
}
*/

/****************************************************************************
**
*F  GMP_NORMALIZE( <gmp> ) . . . . . . .  remove leading zeros from a GMP bag
**
**  'GMP_NORMALIZE' removes any leading zeros from a <GMP> and returns a
**  small int or resizes the bag if possible.
**  
*/
libGAP_Obj libGAP_FuncGMP_NORMALIZE( libGAP_Obj self, libGAP_Obj gmp )
{
  if ( !libGAP_IS_LARGEINT(gmp) ) return libGAP_Fail;
  return libGAP_GMP_NORMALIZE( gmp );
}

libGAP_Obj libGAP_GMP_NORMALIZE ( libGAP_Obj gmp )
{
  libGAP_TypGMPSize size;
  if libGAP_IS_INTOBJ( gmp ) {
    return gmp;
  }
  for ( size = libGAP_SIZE_INT(gmp); size != (libGAP_TypGMPSize)1; size-- ) {
    if ( libGAP_ADDR_INT(gmp)[(size - 1)] != (libGAP_TypLimb)0 ) {
      break;
    }
  }
  if ( size < libGAP_SIZE_INT(gmp) ) {
    libGAP_ResizeBag( gmp, size*sizeof(libGAP_TypLimb) );
  }
  return gmp;
}

libGAP_Obj libGAP_FuncGMP_REDUCE( libGAP_Obj self, libGAP_Obj gmp )
{
  if ( !libGAP_IS_LARGEINT(gmp) ) return libGAP_Fail;
  return libGAP_GMP_REDUCE( gmp );
}

libGAP_Obj libGAP_GMP_REDUCE( libGAP_Obj gmp )
{
  if libGAP_IS_INTOBJ( gmp ) {
    return gmp;
  }
  if ( libGAP_SIZE_INT(gmp) == 1){
    if ( ( libGAP_VAL_LIMB0(gmp) < (libGAP_TypLimb)((1L<<libGAP_NR_SMALL_INT_BITS)) ) ||
         ( libGAP_IS_INTNEG(gmp) && 
           ( libGAP_VAL_LIMB0(gmp) == (libGAP_TypLimb)(1L<<libGAP_NR_SMALL_INT_BITS) ) ) ) {
      if ( libGAP_IS_INTNEG(gmp) ) {
        return libGAP_INTOBJ_INT( -(libGAP_Int)libGAP_VAL_LIMB0(gmp) );
      }
      else {
        return libGAP_INTOBJ_INT(  (libGAP_Int)libGAP_VAL_LIMB0(gmp) );
      }
    }
  }
  return gmp;
}


/****************************************************************************
**
*F  FuncGMP_INTOBJ(<gmp>) . . . . . . . . . . . . . . . . . . . .  conversion
**
*/
libGAP_Obj libGAP_FuncGMP_INTOBJ( libGAP_Obj self, libGAP_Obj i )
{
  libGAP_Obj gmp;
  libGAP_Int   j;

  if ( !libGAP_IS_INTOBJ(i) ) {
    return libGAP_Fail;
  }
  else {
    j = libGAP_INT_INTOBJ( i );
    if ( j < 0 ) {
      gmp = libGAP_NewBag( libGAP_T_INTNEG, sizeof(libGAP_TypLimb) );
      j = -j;
    }
    else {
      gmp = libGAP_NewBag( libGAP_T_INTPOS, sizeof(libGAP_TypLimb) );
    }
  }
  memcpy( libGAP_ADDR_INT(gmp), &j, sizeof(libGAP_Int) );
  return gmp;
}

  
/****************************************************************************
**
*F  GMPorINTOBJ_INT( <cint> ) . . . . . . . .  convert c int to gmp or intobj
**
**  'GMPorINTOBJ_INT' takes the C integer <cint> and returns the equivalent
**  GMP obj or int obj, according to the value of <cint>.
**
*/
libGAP_Obj libGAP_GMPorINTOBJ_INT( libGAP_Int i )
{
  libGAP_Obj gmp;

  if ( (-(1L<<libGAP_NR_SMALL_INT_BITS) <= i) && (i < 1L<<libGAP_NR_SMALL_INT_BITS )) {
    return libGAP_INTOBJ_INT(i);
  }
    else if (i < 0 ) {
    gmp = libGAP_NewBag( libGAP_T_INTNEG, sizeof(libGAP_TypLimb) );
    }
  else {
    gmp = libGAP_NewBag( libGAP_T_INTPOS, sizeof(libGAP_TypLimb) );
  }
  libGAP_SET_VAL_LIMB0( gmp, i );
  return gmp;
}

libGAP_Obj libGAP_ObjInt_Int( libGAP_Int i )
{
  return libGAP_GMPorINTOBJ_INT( i );
}

libGAP_Obj libGAP_ObjInt_UInt( libGAP_UInt i )
{
  libGAP_Obj gmp;
  libGAP_UInt bound = 1UL << libGAP_NR_SMALL_INT_BITS;

  if (i < bound) {
    return libGAP_INTOBJ_INT(i);
  }
  else {
    gmp = libGAP_NewBag( libGAP_T_INTPOS, sizeof(libGAP_TypLimb) );
    libGAP_SET_VAL_LIMB0( gmp, i );
    return gmp;
  }
}


/****************************************************************************
**
*F  PrintInt( <gmp> ) . . . . . . . . . . . . . . . . print a GMP constant
**
**  'PrintInt' prints the GMP integer <gmp> in the usual decimal
**  notation.
**
**  cf PrintInt in integer.c
*/
void libGAP_PrintInt ( libGAP_Obj op )
{
  libGAP_Char buf[20000];
  libGAP_UInt signlength;
  libGAP_Obj str;
  /* print a small integer                                                 */
  if ( libGAP_IS_INTOBJ(op) ) {
    libGAP_Pr( "%>%d%<", libGAP_INT_INTOBJ(op), 0L );
  }
  
  /* print a large integer                                                 */
  else if ( libGAP_SIZE_INT(op) < 1000 ) {
    /* use gmp func to print int to buffer                                 */
    if (!libGAP_IS_INTPOS(op)) {
      buf[0] ='-';
      signlength = 1;
    } else {
      signlength = 0;
    }
    gmp_snprintf((char *)(buf+signlength),20000-signlength,
                 "%Nd", (libGAP_TypLimb *)libGAP_ADDR_INT(op),
                 (libGAP_TypGMPSize)libGAP_SIZE_INT(op));

    /* print the buffer, %> means insert '\' before a linebreak            */
    libGAP_Pr("%>%s%<",(libGAP_Int)buf, 0);
  }
  else {
    str = libGAP_CALL_1ARGS( libGAP_String, op );
    libGAP_Pr("%>%s%<",(libGAP_Int)(libGAP_CHARS_STRING(str)), 0);
    /* for a long time Print of large ints did not follow the general idea
     * that Print should produce something that can be read back into GAP:
       Pr("<<an integer too large to be printed>>",0L,0L); */
  }
}


/****************************************************************************
**
*F  FuncHexStringInt( <self>, <gmp> ) . . . . . . . .  hex string for gmp int
*F  FuncIntHexString( <self>, <string> ) . . . . . .  gmp int from hex string
**  
**  The  function  `FuncHexStringInt'  constructs from  a gmp integer  the
**  corresponding string in  hexadecimal notation. It has  a leading '-'
**  for negative numbers and the digits 10..15 are written as A..F.
**  
**  The  function `FuncIntHexString'  does  the converse,  but here  the
**  letters a..f are also allowed in <string> instead of A..F.
**  
*/
libGAP_Obj libGAP_FuncHexStringInt( libGAP_Obj self, libGAP_Obj integer )
{
  size_t alloc_size, str_size;
  libGAP_Int i, j, n; /* len */
  libGAP_UInt nf;
  /* TypLimb d, f; */
  libGAP_UInt1 *p, a, *s;
  libGAP_Obj res;
  
  /* immediate integers */
  if (libGAP_IS_INTOBJ(integer)) {
    n = libGAP_INT_INTOBJ(integer);
    /* 0 is special */
    if (n == 0) {
      res = libGAP_NEW_STRING(1);
      libGAP_CHARS_STRING(res)[0] = '0';
      return res;
    }
    
    /* else we create a string big enough for any immediate integer        */
    res = libGAP_NEW_STRING(2 * libGAP_NR_HEX_DIGITS + 1);
    p = libGAP_CHARS_STRING(res);
    /* handle sign */
    if (n<0) {
      p[0] = '-';
      n = -n;
      p++;
    }
    else 
      libGAP_SET_LEN_STRING(res, libGAP_GET_LEN_STRING(res)-1);
    /* collect digits, skipping leading zeros                              */
    j = 0;
    nf = ((libGAP_UInt)15) << (4*(2*libGAP_NR_HEX_DIGITS-1));
    for (i = 2*libGAP_NR_HEX_DIGITS; i; i-- ) {
      a = ((libGAP_UInt)n & nf) >> (4*(i-1));
      if (j==0 && a==0) libGAP_SET_LEN_STRING(res, libGAP_GET_LEN_STRING(res)-1);
      else if (a<10) p[j++] = a + '0';
      else p[j++] = a - 10 + 'A';
      nf = nf >> 4;
    }
    /* final null character                                                */
    p[j] = 0;
    return res;
  }

  else if ( libGAP_IS_LARGEINT(integer) ) {
    alloc_size = libGAP_SIZE_INT(integer)*sizeof(libGAP_TypLimb)*2+1;
    alloc_size += libGAP_IS_INTNEG(integer);

    res = libGAP_NEW_STRING( alloc_size );
    s = libGAP_CHARS_STRING( res );

    if ( libGAP_IS_INTNEG(integer) )
      *s++ = '-';

    str_size = mpn_get_str( s, 16, libGAP_ADDR_INT(integer), libGAP_SIZE_INT(integer) );
    assert ( str_size <= alloc_size - ( libGAP_IS_INTNEG(integer) ) );

    for (j = 0; j < str_size-1; j++)
      if (s[j] != 0)
        break;
    

    for ( i = 0; i < str_size-j; i++ )
      s[i] = "0123456789ABCDEF"[s[i+j]];

    assert ( str_size - j == 1 || *s != '0' );

    /* findme  - this fails: */
    /*    assert ( strlen( CSTR_STRING(res) ) == alloc_size ); */
    /* adjust length in case of trailing \0 characters */
    /* [Is there a way to get it right from the beginning? FL] */
    /*     while (s[alloc_size-1] == '\0') 
           alloc_size--; */
    libGAP_SET_LEN_STRING(res, str_size-j + (libGAP_IS_INTNEG(integer)));
    /*  assert ( strlen( CSTR_STRING(res) ) == GET_LEN_STRING(res) ); */
    return res;
  }

  else 
    libGAP_ErrorReturnObj("HexStringInt: argument must be a int, (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(integer), 0L,
                   "");
  /* please picky cc                                                       */
  return (libGAP_Obj) 0L; 

}


libGAP_Obj libGAP_FuncIntHexString( libGAP_Obj self,  libGAP_Obj str )
{
  libGAP_Obj res;
  libGAP_Int  i, j, len, sign, nd;
  libGAP_UInt n;
  libGAP_UInt1 *p, a;
  libGAP_UChar c;
  
  if (! libGAP_IsStringConv(str))
    libGAP_ErrorReturnObj("IntHexString: argument must be string (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(str), 0L,
                   "");

  len = libGAP_GET_LEN_STRING(str);
  if (len == 0) {
    res = libGAP_INTOBJ_INT(0);
    return res;
  }
  if (*(libGAP_CHARS_STRING(str)) == '-') {
    sign = -1;
    i = 1;
  }
  else {
    sign = 1;
    i = 0;
  }

  while ((libGAP_CHARS_STRING(str))[i] == '0' && i < len)
    i++;
    

  if ((len-i)*4 <= libGAP_NR_SMALL_INT_BITS) {
    n = 0;
    p = libGAP_CHARS_STRING(str);
    for (; i<len; i++) {
      a = p[i];
      if (a>96) 
        a -= 87;
      else if (a>64) 
        a -= 55;
      else 
        a -= 48;
      if (a > 15)
        libGAP_ErrorReturnObj("IntHexString: non-valid character in hex-string",
                       0L, 0L, "");
      n = (n << 4) + a;
    }
    res = libGAP_INTOBJ_INT(sign * n);
    return res;
  }

  else {
    nd = (len-i)/libGAP_NR_HEX_DIGITS;
    if (nd * libGAP_NR_HEX_DIGITS < (len-i)) nd++;
    /*   nd += ((3*nd) % 4); */
    if (sign == 1)
      res = libGAP_NewBag( libGAP_T_INTPOS, nd*sizeof(libGAP_TypLimb) );
    else
      res = libGAP_NewBag( libGAP_T_INTNEG, nd*sizeof(libGAP_TypLimb) );

    p = libGAP_CHARS_STRING(str)+i;

    /* findme */
    /* the following destroys the supplied string - document this          */
    for (j=0;j<len-i;j++){
      c=p[j];
      if (libGAP_IsDigit(c))
        p[j] = c - '0';
      else if (islower((unsigned int)c))
        p[j] = c - 'a' + 10;
      else if (isupper((unsigned int)c))
        p[j] = c - 'A' + 10;
      else
        libGAP_ErrorReturnObj("IntHexString: non-valid character in hex-string",
                       0L, 0L, "");
      if (p[j] >= 16)
        libGAP_ErrorReturnObj("IntHexString: non-valid character in hex-string",
                       0L, 0L, "");
    }

    mpn_set_str(libGAP_ADDR_INT(res),p,len-i,16);
    res = libGAP_GMP_NORMALIZE(res);
    return res;
  }
}


/****************************************************************************
**
*F  FuncLog2Int( <self>, <gmp> ) . . . . . . . . . .  nr of bits of a GMP - 1
**  
**  Given to GAP-Level as "Log2Int".
*/
libGAP_Obj libGAP_FuncLog2Int( libGAP_Obj self, libGAP_Obj integer)
{
  libGAP_Int res, d;
  libGAP_Int a, len;
  libGAP_Int mask;
  libGAP_TypLimb dmask;
  
  /* case of small ints                                                    */
  if (libGAP_IS_INTOBJ(integer)) {
    a = libGAP_INT_INTOBJ(integer);
    if (a < 0) a = -a;
    res = libGAP_NR_SMALL_INT_BITS;
    for(res = libGAP_NR_SMALL_INT_BITS - 1, mask = (libGAP_Int)1 << (libGAP_NR_SMALL_INT_BITS-1);
        (mask & a) == 0 && mask != (libGAP_Int)0;
        mask = mask >> 1, res--);
    return libGAP_INTOBJ_INT(res);
  }

  /* case of long ints                                                     */
  if ( libGAP_IS_LARGEINT(integer) ) {
    for (len = libGAP_SIZE_INT(integer); libGAP_ADDR_INT(integer)[len-1] == 0; len--);
    /* Instead of computing 
          res = len * GMP_LIMB_BITS - d;
       we keep len and d separate, because on 32 bit systems res may
       not fit into an Int (and not into an immediate integer).            */
    d = 1;
    a = (libGAP_TypLimb)(libGAP_ADDR_INT(integer)[len-1]); 
    for(dmask = (libGAP_TypLimb)1 << (GMP_LIMB_BITS - 1);
        (dmask & a) == 0 && dmask != (libGAP_TypLimb)0;
        dmask = dmask >> 1, d++);
    return libGAP_DiffInt(libGAP_ProdInt(libGAP_INTOBJ_INT(len), libGAP_INTOBJ_INT(GMP_LIMB_BITS)), 
                   libGAP_INTOBJ_INT(d));
  }
  else {
    libGAP_ErrorReturnObj("Log2Int: argument must be a int, (not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(integer), 0L,
           "");
    /* please picky cc                                                     */
    return (libGAP_Obj) 0L;
  }
}

/****************************************************************************
**
*F  FuncSTRING_INT( <self>, <gmp> ) . . . . . . . . convert a GMP to a string
**
**  `FuncSTRING_INT' returns an immutable string representing the integer
**  <gmp>
**
*/
libGAP_Obj libGAP_FuncSTRING_INT( libGAP_Obj self, libGAP_Obj integer )
{
  libGAP_Int   x;
  libGAP_Obj str;
  libGAP_Int len;
  libGAP_Int   i;
  libGAP_Char  c;
  libGAP_Int neg;

  
  /* handle a small integer                                                */
  if ( libGAP_IS_INTOBJ(integer) ) {
    x = libGAP_INT_INTOBJ(integer);
    str = libGAP_NEW_STRING( (libGAP_NR_SMALL_INT_BITS+5)/3 );
    libGAP_RetypeBag(str, libGAP_T_STRING+libGAP_IMMUTABLE);
    len = 0;
    /* Case of zero                                                        */
    if (x == 0)
      {
        libGAP_CHARS_STRING(str)[0] = '0';
        libGAP_CHARS_STRING(str)[1] = '\0';
        libGAP_ResizeBag(str, libGAP_SIZEBAG_STRINGLEN(1));
        libGAP_SET_LEN_STRING(str, 1);
        
        return str;
      }
    /* Negative numbers                                                    */
    if (x < 0)
      {
        libGAP_CHARS_STRING(str)[len++] = '-';
        x = -x;
        neg = 1;
      }
    else
      neg = 0;

    /* Now the main case                                                   */
    while (x != 0)
      {
        libGAP_CHARS_STRING(str)[len++] = '0'+ x % 10;
        x /= 10;
      }
    libGAP_CHARS_STRING(str)[len] = '\0';
    
    /* finally, reverse the digits in place                                */
    for (i = neg; i < (neg+len)/2; i++)
      {
        c = libGAP_CHARS_STRING(str)[neg+len-1-i];
        libGAP_CHARS_STRING(str)[neg+len-1-i] = libGAP_CHARS_STRING(str)[i];
        libGAP_CHARS_STRING(str)[i] = c;
      }
    
    libGAP_ResizeBag(str, libGAP_SIZEBAG_STRINGLEN(len));
    libGAP_SET_LEN_STRING(str, len);
    return str;
  }

  /* handle a large integer                                                */
  else if ( libGAP_SIZE_INT(integer) < 1000 ) {

    /* findme - enough space for a 1000 limb gmp int on a 64 bit machine     */
    /* change when 128 bit comes along!                                      */
    libGAP_Char buf[20000];

    if libGAP_IS_INTNEG(integer) {
    len = gmp_snprintf( buf, sizeof(buf)-1, "-%Ni", (libGAP_TypLimb *)libGAP_ADDR_INT(integer),
          (libGAP_TypGMPSize)libGAP_SIZE_INT(integer) );
    }
    else {
    len = gmp_snprintf( buf, sizeof(buf)-1,  "%Ni", (libGAP_TypLimb *)libGAP_ADDR_INT(integer),
          (libGAP_TypGMPSize)libGAP_SIZE_INT(integer) );
    }

    assert(len < sizeof(buf));
    libGAP_C_NEW_STRING( str, (libGAP_TypGMPSize)len, buf );

    return str;

  }

  else {

      /* Very large integer, fall back on the GAP function                 */
      return libGAP_CALL_1ARGS( libGAP_String, integer);
  }
}
  

/****************************************************************************
**
*F  EqInt( <gmpL>, <gmpR> ) . . . . . . . . .  test if two integers are equal
**
**  
**  'EqInt' returns 1  if  the two integer   arguments <intL> and  <intR> are
**  equal and 0 otherwise.
*/

  /* findme - For comparisons, do we first normalize and, if possible,
     reduce? Or (for one small, one gmp int) make the small int into a
     1-limb gmp to compare to the gmp. Or should we assume that gmp ints
     cannot be 'small'? */

libGAP_Int libGAP_EqInt ( libGAP_Obj gmpL, libGAP_Obj gmpR )
{
  libGAP_Obj opL;
  libGAP_Obj opR;

  /* compare two small integers                                          */
  if ( libGAP_ARE_INTOBJS( gmpL, gmpR ) ) {
    if ( libGAP_INT_INTOBJ(gmpL) == libGAP_INT_INTOBJ(gmpR) )  return 1L;
    else                                       return 0L;
  }

  /* small ints fit into one limb of a GMP                                 */
  if libGAP_IS_INTOBJ(gmpL) {
    if ( ( libGAP_INT_INTOBJ(gmpL) <  0 && libGAP_IS_INTPOS(gmpR) ) ||
         ( 0 <= libGAP_INT_INTOBJ(gmpL) && libGAP_IS_INTNEG(gmpR) ) ||
         ( libGAP_SIZE_INT(gmpR) > (libGAP_TypGMPSize)1 ) ) return 0L;
    opL = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, gmpL );
  }
  else {
    opL = gmpL;
  }

  if libGAP_IS_INTOBJ(gmpR) {
    if ( ( libGAP_INT_INTOBJ(gmpR) <  0 && libGAP_IS_INTPOS(gmpL) ) ||
         ( 0 <= libGAP_INT_INTOBJ(gmpR) && libGAP_IS_INTNEG(gmpL) ) ||
         ( libGAP_SIZE_INT(gmpL) > (libGAP_TypGMPSize)1 ) ) return 0L;
    opR = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, gmpR );
  }
  else {
    opR = gmpR;
  }

  /* compare the sign and size                                             */
  if ( libGAP_TNUM_OBJ(opL) != libGAP_TNUM_OBJ(opR)
       || libGAP_SIZE_INT(opL) != libGAP_SIZE_INT(opR) )
    return 0L;

  if ( mpn_cmp( libGAP_ADDR_INT(opL), libGAP_ADDR_INT(opR), libGAP_SIZE_INT(opL) ) == 0 ) 
    return 1L;
  else
    return 0L;
}

/****************************************************************************
**
*F  LtInt( <gmpL>, <gmpR> )  . . . . . . . . . . test whether <gmpL> < <gmpR>
**
*/
libGAP_Int libGAP_LtInt ( libGAP_Obj gmpL, libGAP_Obj gmpR )
{
  libGAP_Obj opL;
  libGAP_Obj opR;

  /* compare two small integers                                          */
  if ( libGAP_ARE_INTOBJS( gmpL, gmpR ) ) {
    if ( libGAP_INT_INTOBJ(gmpL) <  libGAP_INT_INTOBJ(gmpR) ) return 1L;
    else return 0L;
  }

  /* compare a small and a large integer                                   */
  if ( libGAP_IS_INTOBJ(gmpL) ) {
    if ( libGAP_SIZE_INT(gmpR) > (libGAP_TypGMPSize)1 ) {
      if ( libGAP_IS_INTPOS(gmpR) ) return 1L;
      else return 0L;
    }
    else opL = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, gmpL );
  }
  else {
    opL = gmpL;
  }

  if ( libGAP_IS_INTOBJ(gmpR) ) {
    if ( libGAP_SIZE_INT(gmpL) > (libGAP_TypGMPSize)1 ) {
      if ( libGAP_IS_INTNEG(gmpL) )  return 1L;
      else return 0L;
    }
    else opR = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, gmpR );
  }
  else {
    opR = gmpR;
  }

  /* compare two large integers                                            */
  if ( libGAP_IS_INTNEG(opL) && libGAP_IS_INTPOS(opR) )
    return 1L;
  else if (   libGAP_IS_INTPOS(opL) && libGAP_IS_INTNEG(opR) )
    return 0L;
  else if ( ( libGAP_IS_INTPOS(opR) && libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) ) ||
            ( libGAP_IS_INTNEG(opR) && libGAP_SIZE_INT(opL) > libGAP_SIZE_INT(opR) ) )
    return 1L;
  else if ( ( libGAP_IS_INTPOS(opL) && libGAP_SIZE_INT(opL) > libGAP_SIZE_INT(opR) ) ||
            ( libGAP_IS_INTNEG(opL) && libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) ) )
    return 0L;
  else if ( libGAP_IS_INTPOS(opL) ) {
    if ( mpn_cmp( libGAP_ADDR_INT(opL), libGAP_ADDR_INT(opR), libGAP_SIZE_INT(opL) ) < 0 ) 
      return 1L;
    else
      return 0L;
  }
  else {
    if ( mpn_cmp( libGAP_ADDR_INT(opL), libGAP_ADDR_INT(opR), libGAP_SIZE_INT(opL) ) > 0 ) 
      return 1L;
    else
      return 0L;
  }
}


/****************************************************************************
**
*F  SumInt( <gmpL>, <gmpR> ) . . . . . . . . . . . .  sum of two GMP integers
**
*/
libGAP_Obj libGAP_SumInt ( libGAP_Obj gmpL, libGAP_Obj gmpR )
{
  libGAP_Obj sum;

  sum = libGAP_SumOrDiffInt( gmpL, gmpR, +1 );

  return sum;

}


/****************************************************************************
**
*F  DiffInt( <gmpL>, <gmpR> ) . . . . . . . .  difference of two GMP integers
**
*/
libGAP_Obj libGAP_DiffInt ( libGAP_Obj gmpL, libGAP_Obj gmpR )
{
  libGAP_Obj dif;
  
  dif = libGAP_SumOrDiffInt( gmpL, gmpR, -1 );
  
  return dif;
}


/****************************************************************************
**
*F  SumOrDiffInt( <gmpL>, <gmpR> ) . . . . .  sum or diff of two Int integers
**
**  'SumOrDiffInt' returns the sum or difference of the two GMP int arguments
**  <gmpL> and  <gmpR>. 'SumOrDiffInt'  handles  operands  of  type  'T_INT',
**  'T_INTPOS' and 'T_INTNEG'.
**
**  'SumOrDiffInt'  is a little  bit  tricky since  there are  many different
**  cases to handle, each operand can be positive or negative, small or large
**  integer.
*/
libGAP_Obj libGAP_SumOrDiffInt ( libGAP_Obj gmpL, libGAP_Obj gmpR, libGAP_Int sign )
{
  libGAP_Obj       res; /* handle of result bag                                   */
  libGAP_Int  twosmall; /* sum of two smalls                                      */
  libGAP_Int  onesmall; /* set to 1 if one of args is a small int, 0 otherwise    */
  libGAP_Int   swapped; /* set to 1 if args were swapped, 0 otherwise             */
  libGAP_Int    resneg; /* set to 1 if result will be negative                    */
  libGAP_TypLimb carry; /* hold any carry or borrow                               */

  twosmall = 0;
  onesmall = 0;
  swapped  = 0;
  resneg   = 0;

  /* findme - later change to put the non-overflow versions of these small
int adds/subs into the caller funcs SumInt, DiffInt. Then remove check of
SUM or DIFF _INTOBJS and document (at least in code) that this should not
be called directly */

  if ( sign != 1 && sign != -1 ) {
    libGAP_ErrorReturnObj(
                   "SumOrDiffInt: <sign> must be +1 or -1. \nDo not call this function directly.",
                   0L, 0L,
                   "" );
  }

    /* adding two small integers                                           */
  if ( libGAP_ARE_INTOBJS( gmpL, gmpR ) ) {

    /* add or subtract two small integers with a small sum                 */
    if (sign == 1) {
      if ( libGAP_SUM_INTOBJS( res, gmpL, gmpR ) ) {
        return res;
      }
      else {
        twosmall = libGAP_INT_INTOBJ(gmpL) + libGAP_INT_INTOBJ(gmpR);
      }
    }
    else if (sign == -1) {
      if ( libGAP_DIFF_INTOBJS( res, gmpL, gmpR ) ) {
        return res;
      }
      else {
        twosmall = libGAP_INT_INTOBJ(gmpL) - libGAP_INT_INTOBJ(gmpR);
      }
    }

    /* if two small integers have a large sum or difference form the gmp int*/
    if ( 0 < twosmall ) {
      res = libGAP_NewBag( libGAP_T_INTPOS, sizeof(libGAP_TypLimb) );
      libGAP_SET_VAL_LIMB0( res, (libGAP_TypLimb)twosmall );
    }
    else {
      res = libGAP_NewBag( libGAP_T_INTNEG, sizeof(libGAP_TypLimb) );
      libGAP_SET_VAL_LIMB0( res, (libGAP_TypLimb)(-twosmall) );
    }

    return res;
  }

  /* findme - we repeat some of this work in the 'add' part later on.
     Can we recycle some code? */

  /* the case of one small integer and one large                           */
  else if ( libGAP_IS_INTOBJ( gmpL ) || libGAP_IS_INTOBJ( gmpR ) ) {
    onesmall = 1;
    if ( libGAP_IS_INTOBJ( gmpL ) ) {
      /* findme - do we need to normalize here? */
      gmpR = libGAP_GMP_NORMALIZE( gmpR );
      gmpR = libGAP_GMP_REDUCE( gmpR );
      if ( libGAP_IS_INTOBJ(gmpR) ) {
        return libGAP_INTOBJ_INT( libGAP_INT_INTOBJ(gmpL) + sign*libGAP_INT_INTOBJ(gmpR) );
      }
      res = gmpL; gmpL = gmpR; gmpR = res;
      swapped = 1;
    }
    else {
      gmpL = libGAP_GMP_NORMALIZE( gmpL );
      gmpL = libGAP_GMP_REDUCE( gmpL );
      if ( libGAP_IS_INTOBJ(gmpL) ) {
        return libGAP_INTOBJ_INT( libGAP_INT_INTOBJ(gmpL) + sign*libGAP_INT_INTOBJ(gmpR) );
      }
    }
  }

  /* two large ints                                                        */
  else if ( libGAP_SIZE_INT( gmpL ) < libGAP_SIZE_INT( gmpR ) ) {
    /* swap gmpL and gmpR                                                  */
    res = gmpL; gmpL = gmpR; gmpR = res;
    swapped = 1;
  }

  if      ( ( ( sign == +1 ) &&
              ( (  ( onesmall ) &&
                  ( (libGAP_IS_INTNEG(gmpL) && 0 <= libGAP_INT_INTOBJ(gmpR)) ||
                    (libGAP_IS_INTPOS(gmpL) && 0 >  libGAP_INT_INTOBJ(gmpR)) )  ) ||
                ( !( onesmall ) &&
                  ( (libGAP_IS_INTPOS(gmpL) &&       libGAP_IS_INTNEG(gmpR)) ||
                    (libGAP_IS_INTNEG(gmpL) &&       libGAP_IS_INTPOS(gmpR)) )  ) ) ) ||
            ( ( sign == -1 ) &&
              ( (  ( onesmall ) &&
                  ( (libGAP_IS_INTNEG(gmpL) && 0 >  libGAP_INT_INTOBJ(gmpR)) ||
                    (libGAP_IS_INTPOS(gmpL) && 0 <= libGAP_INT_INTOBJ(gmpR)) ) ) ||
                ( !( onesmall ) &&
                  ( (libGAP_IS_INTPOS(gmpL) &&       libGAP_IS_INTPOS(gmpR)) ||
                    (libGAP_IS_INTNEG(gmpL) &&       libGAP_IS_INTNEG(gmpR)) )  ) ) ) ) {

    /* the args have different sign (or same sign and this is a subtraction)
       - compare to see which to subtract                                  */
    if ( onesmall ) {
      if ( ( ( ( swapped == 1 && sign == +1 ) || swapped == 0 )
             && libGAP_IS_INTNEG(gmpL) ) || 
           ( swapped == 1 && sign == -1 && libGAP_IS_INTPOS(gmpL) ) ) {
        res = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmpL) );
      }
      else {
        res = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmpL) );
      }
      gmpR = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, gmpR );
      carry = mpn_sub_1( libGAP_ADDR_INT(res), 
                         libGAP_ADDR_INT(gmpL), libGAP_SIZE_INT(gmpL),
                        *libGAP_ADDR_INT(gmpR) );
    }
    /* this test correct since size(gmpL) >= size(gmpR)                    */
    else if ( libGAP_SIZE_INT(gmpL) != libGAP_SIZE_INT(gmpR) ) {
      if ( ( ( ( swapped == 1 && sign == +1 ) || swapped == 0 )
             && libGAP_IS_INTNEG(gmpL) ) || 
           ( swapped == 1 && sign == -1 && libGAP_IS_INTPOS(gmpL) ) ) {
        res = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmpL) );
      }
      else {
        res = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmpL) );
      }
      carry = mpn_sub( libGAP_ADDR_INT(res),
                       libGAP_ADDR_INT(gmpL), libGAP_SIZE_INT(gmpL),
                       libGAP_ADDR_INT(gmpR), libGAP_SIZE_INT(gmpR) );
    }
    /* ok, so they're the same size in limbs - which is the bigger number? */
    else if ( mpn_cmp( libGAP_ADDR_INT(gmpL),
                       libGAP_ADDR_INT(gmpR), libGAP_SIZE_INT(gmpL) ) < 0 ) {
      if ( libGAP_IS_INTPOS(gmpL) ) {
        res = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmpL) );
      }
      else {
        res = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmpL) );
      }
      carry = mpn_sub_n( libGAP_ADDR_INT(res),
                         libGAP_ADDR_INT(gmpR),
                         libGAP_ADDR_INT(gmpL), libGAP_SIZE_INT(gmpR) );
    }

    else {
      if ( libGAP_IS_INTNEG(gmpL) ) {
        res = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmpL) );
      }
      else {
        res = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmpL) );
      }
      carry = mpn_sub_n( libGAP_ADDR_INT(res),
                         libGAP_ADDR_INT(gmpL),
                         libGAP_ADDR_INT(gmpR), libGAP_SIZE_INT(gmpL) );
    }

    res = libGAP_GMP_NORMALIZE( res );
    res = libGAP_GMP_REDUCE( res );
    return res;
  }

  else {
    /* The args have the same sign (or opp sign and this is a subtraction) 
       - so add them. At this stage, we are dealing with a large and a 
       small, or two large integers                                        */

    /* Will the result be negative?                                        */
    if ( ( sign ==  1 && libGAP_IS_INTNEG(gmpL) ) ||
         ( sign == -1 &&
           ( ( swapped == 0 && libGAP_IS_INTNEG(gmpL) ) ||
             ( swapped == 1 && libGAP_IS_INTPOS(gmpL) ) ) ) ) {
      resneg = 1;
    }
         /*        ( ( onesmall        && IS_INTNEG(gmpL) ) ||
             ( IS_INTNEG(gmpL) && IS_INTNEG(gmpR) ) ||
             ( SIZE_INT(gmpL) > SIZE_INT(gmpR) && IS_INTNEG(gmpL) ) ) ) ||
    if ( resneg == 0 && sign == 1 && SIZE_INT(gmpL) == SIZE_INT(gmpR) ) {
      compare = mpn_cmp( ADDR_INT(gmpL), ADDR_INT(gmpR), SIZE_INT(gmpL) );
      if ( ( compare >= 0 && IS_INTNEG(gmpL) ) ||
           ( compare  < 0 && IS_INTNEG(gmpR) ) ) {
        resneg = 1;
      }
    }
         */
    if ( onesmall ) {
      if ( resneg == 0 ) {
        res = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmpL) + sizeof(libGAP_TypLimb) );
      }
      else {
        res = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmpL) + sizeof(libGAP_TypLimb) );
      }
      gmpR = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, gmpR );
      carry = mpn_add_1( libGAP_ADDR_INT(res),
                         libGAP_ADDR_INT(gmpL),libGAP_SIZE_INT(gmpL),
                        *libGAP_ADDR_INT(gmpR) );
      if ( carry == (libGAP_TypLimb)0 ) {
        libGAP_ResizeBag( res, libGAP_SIZE_OBJ(gmpL) );
      }
      else {
        ( libGAP_ADDR_INT(res) )[ libGAP_SIZE_INT(gmpL) ] = (libGAP_TypLimb)1; /* = carry ? */
      }
      /* findme - debugging 231107
      res = GMP_NORMALIZE( res );
      res = GMP_REDUCE( res ); */
    }

    else {
      /* put the smaller one (in limbs) to the right                       */
      if ( libGAP_SIZE_INT(gmpL) < libGAP_SIZE_INT(gmpR) ) { 
        res = gmpR; gmpR = gmpL; gmpL = res;
      }

      /* allocate result bag                                               */
      if ( resneg == 0 ) {
        res = libGAP_NewBag( libGAP_T_INTPOS, ( libGAP_SIZE_OBJ(gmpL) + sizeof(libGAP_TypLimb) ) );
      }
      else {
        res = libGAP_NewBag( libGAP_T_INTNEG, ( libGAP_SIZE_OBJ(gmpL) + sizeof(libGAP_TypLimb) ) );
      }
      
      /* mpn_lshift is faster still than mpn_add_n for adding a TypLimb
         number to itself                                                  */
      if ( gmpL == gmpR ) {
        carry = mpn_lshift( libGAP_ADDR_INT(res),
                            libGAP_ADDR_INT(gmpL), libGAP_SIZE_INT(gmpL),
                            1 );
      }
      else {
        carry =    mpn_add( libGAP_ADDR_INT(res),
                            libGAP_ADDR_INT(gmpL), libGAP_SIZE_INT(gmpL),
                            libGAP_ADDR_INT(gmpR), libGAP_SIZE_INT(gmpR) );
      }
      if ( carry == (libGAP_TypLimb)0 ){
        libGAP_ResizeBag( res, libGAP_SIZE_OBJ(gmpL) );
      }
      else{
        ( libGAP_ADDR_INT(res) )[ libGAP_SIZE_INT(gmpL) ] = (libGAP_TypLimb)1;
      }
      /* findme - don't need this after carry ? */
      /* res = GMP_NORMALIZE( res );
         res = GMP_REDUCE( res ); */
    }

    return res;
  }

}


/****************************************************************************
**
*F  ZeroInt(<gmp>)  . . . . . . . . . . . . . . . . . . . .  zero of integers
*/
libGAP_Obj libGAP_ZeroInt ( libGAP_Obj  op )
{
  return libGAP_INTOBJ_INT( (libGAP_Int)0 );
}


/****************************************************************************
**
*F  AInvInt(<gmp>) . . . . . . . . . . . . . . additive inverse of an integer
*/
libGAP_Obj libGAP_AInvInt ( libGAP_Obj gmp )
{
  libGAP_Obj inv;

  /* handle small integer                                                */
  if ( libGAP_IS_INTOBJ( gmp ) ) {
    
    /* special case (ugh)                                              */
    if ( gmp == libGAP_INTOBJ_INT( -(1L<<libGAP_NR_SMALL_INT_BITS) ) ) {
      inv = libGAP_NewBag( libGAP_T_INTPOS, sizeof(libGAP_TypLimb) );
      libGAP_SET_VAL_LIMB0( inv, 1L<<libGAP_NR_SMALL_INT_BITS );
    }
    
    /* general case                                                    */
    else {
      inv = libGAP_INTOBJ_INT( - libGAP_INT_INTOBJ( gmp ) );
    }
    
  }

  else {
    if ( libGAP_IS_INTPOS(gmp) ) {
      /* special case                                                        */
      if ( ( libGAP_SIZE_INT(gmp) == 1 ) 
           && ( libGAP_VAL_LIMB0(gmp) == (libGAP_TypLimb) (1L<<libGAP_NR_SMALL_INT_BITS) ) ) {
        return libGAP_INTOBJ_INT( -(libGAP_Int) (1L<<libGAP_NR_SMALL_INT_BITS) );
      }
      else {
        inv = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmp) );
      }
    }
    
    else {
      inv = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmp) );
    }

    memcpy( libGAP_ADDR_INT(inv), libGAP_ADDR_INT(gmp), libGAP_SIZE_OBJ(gmp) );
  }
  
  /* return the inverse                                                    */
  return inv;

}

libGAP_Obj libGAP_AbsInt( libGAP_Obj gmp )
{
  libGAP_Obj a;
  if (libGAP_IS_INTOBJ(gmp)) {
    if (((libGAP_Int)gmp) > 0) 
      return gmp;
    else if (gmp == libGAP_INTOBJ_INT(-(1L << libGAP_NR_SMALL_INT_BITS)))
      return libGAP_AInvInt(gmp);
    else
      return (libGAP_Obj)(2-(libGAP_Int)gmp);
  }
  if (libGAP_TNUM_OBJ(gmp) == libGAP_T_INTPOS)
    return gmp;
  a = libGAP_NewBag(libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmp));
  memcpy( libGAP_ADDR_INT(a), libGAP_ADDR_INT(gmp), libGAP_SIZE_OBJ(gmp) );
  return a;
}

libGAP_Obj libGAP_FuncABS_INT(libGAP_Obj self, libGAP_Obj gmp)
{
  return libGAP_AbsInt(gmp);
}


/****************************************************************************
**
*F  ProdInt( <intL>, <intR> ) . . . . . . . . . . . . product of two integers
**
**  'ProdInt' returns the product of the two  integer  arguments  <intL>  and
**  <intR>.  'ProdInt' handles  operands  of  type  'T_INT',  'T_INTPOS'  and
**  'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalProd' for a better efficiency.
**
**  Is called from the 'EvalProd' binop so both operands are already evaluated.
**
**  The only difficulty about this function is the fact that is has to handle
**  3 different situations, depending on how many arguments  are  small ints.
*/
libGAP_Obj libGAP_ProdInt ( libGAP_Obj gmpL, libGAP_Obj gmpR )
{
  libGAP_Obj                 prd;            /* handle of the result bag          */
  libGAP_Int                   i;            /* hold small int value              */
  libGAP_Int                   k;            /* hold small int value              */
  libGAP_TypLimb           carry;            /* most significant limb             */
  
  /* multiplying two small integers                                        */
  if ( libGAP_ARE_INTOBJS( gmpL, gmpR ) ) {
    
    /* multiply two small integers with a small product                    */
    /* multiply and divide back to check that no overflow occured          */
    if ( libGAP_PROD_INTOBJS( prd, gmpL, gmpR ) ) {
      return prd;
    }
    
    /* get the integer values                                              */
    i = libGAP_INT_INTOBJ(gmpL);
    k = libGAP_INT_INTOBJ(gmpR);
    
    /* allocate the product bag                                            */
    if ( (0 < i && 0 < k) || (i < 0 && k < 0) )
      prd = libGAP_NewBag( libGAP_T_INTPOS, 2*sizeof(libGAP_TypLimb) );
    else
      prd = libGAP_NewBag( libGAP_T_INTNEG, 2*sizeof(libGAP_TypLimb) );
    
    /* make both operands positive                                         */
    if ( i < 0 )  i = -i;
    if ( k < 0 )  k = -k;
    
    /* multiply                                                            */
    mpn_mul_n( libGAP_ADDR_INT( prd ), (libGAP_TypLimb*)( &i ),
               (libGAP_TypLimb*)( &k ), (libGAP_TypGMPSize)1 );
  }
  
  /* multiply a small and a large integer                                  */
  else if ( libGAP_IS_INTOBJ(gmpL) || libGAP_IS_INTOBJ(gmpR) ) {

    /* make the right operand the small one                                */
    if ( libGAP_IS_INTOBJ(gmpL) ) {
      k = libGAP_INT_INTOBJ(gmpL);  gmpL = gmpR;
    }
    else {
      k = libGAP_INT_INTOBJ(gmpR);
    }
    
    /* handle trivial cases first                                          */
    if ( k == 0 )
      return libGAP_INTOBJ_INT(0);
    if ( k == 1 )
      return gmpL;
    
    /* eg: for 32 bit systems, the large integer 1<<28 times -1 is the small
       integer -(1<<28)                                                    */
    if ( ( k == -1 ) && (libGAP_SIZE_INT(gmpL)==1) 
         && ( libGAP_VAL_LIMB0(gmpL) == (libGAP_TypLimb)(1L<<libGAP_NR_SMALL_INT_BITS) ) )
      return libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS));
    
    /* multiplication by -1 is easy, just switch the sign and copy         */
    if ( k == -1 ) {
      if ( libGAP_TNUM_OBJ(gmpL) == libGAP_T_INTPOS ) {
        prd = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmpL) );
      }
      else {
        prd = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmpL) );
      }
      memcpy( libGAP_ADDR_OBJ(prd), libGAP_ADDR_OBJ(gmpL), libGAP_SIZE_OBJ(gmpL) );
      return prd;
    }
    
    /* allocate a bag for the result                                       */
    if ( (0 < k && libGAP_TNUM_OBJ(gmpL) == libGAP_T_INTPOS)
         || (k < 0 && libGAP_TNUM_OBJ(gmpL) == libGAP_T_INTNEG) ) {
      prd = libGAP_NewBag( libGAP_T_INTPOS, (libGAP_SIZE_INT(gmpL)+1)*sizeof(libGAP_TypLimb) );
    }
    else {
      prd = libGAP_NewBag( libGAP_T_INTNEG, (libGAP_SIZE_INT(gmpL)+1)*sizeof(libGAP_TypLimb) );
    }
    
    if ( k < 0 )  k = -k;
    
    /* multiply                                                            */
    carry = mpn_mul_1( libGAP_ADDR_INT(prd), libGAP_ADDR_INT(gmpL),
               libGAP_SIZE_INT(gmpL), (libGAP_TypLimb)k );
    if ( carry == (libGAP_TypLimb)0 ) {
      libGAP_ResizeBag( prd, libGAP_SIZE_OBJ(gmpL) );
    }
    else {
      ( libGAP_ADDR_INT(prd) )[ libGAP_SIZE_INT(gmpL) ] = carry;
    }
  }
  
  /* multiply two large integers                                           */
  else {
    
    /* make the right operand the smaller one, for the mpn function        */
    if ( libGAP_SIZE_INT(gmpL) < libGAP_SIZE_INT(gmpR) ) {
      prd = gmpR;  gmpR = gmpL;  gmpL = prd;
    }
    
    /* allocate a bag for the result                                       */
    if ( libGAP_TNUM_OBJ(gmpL) == libGAP_TNUM_OBJ(gmpR) )
      prd = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(gmpL)+libGAP_SIZE_OBJ(gmpR) );
    else
      prd = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(gmpL)+libGAP_SIZE_OBJ(gmpR) );
    
    /* multiply                                                            */
    mpn_mul( libGAP_ADDR_INT(prd),
             libGAP_ADDR_INT(gmpL), libGAP_SIZE_INT(gmpL),
             libGAP_ADDR_INT(gmpR), libGAP_SIZE_INT(gmpR) );
  }
  
  /* normalize and return the product                                      */
  prd = libGAP_GMP_NORMALIZE( prd );
  prd = libGAP_GMP_REDUCE( prd );
  return prd;
}


/****************************************************************************
**
*F  ProdIntObj(<n>,<op>)  . . . . . . . . product of an integer and an object
*/
libGAP_Obj libGAP_ProdIntObj ( libGAP_Obj n, libGAP_Obj op )
{
  libGAP_Obj                 res = 0;        /* result                            */
  libGAP_UInt                i, k;           /* loop variables                    */
  libGAP_TypLimb             l;              /* loop variable                     */

  /* if the integer is zero, return the neutral element of the operand     */
  if      ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  0 ) {
    res = libGAP_ZERO( op );
  }
  
  /* if the integer is one, return the object if immutable -
     if mutable, add the object to its ZeroSameMutability to
     ensure correct mutability propagation                                 */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  1 ) {
    if (libGAP_IS_MUTABLE_OBJ(op))
      res = libGAP_SUM(libGAP_ZERO(op),op);
    else
      res = op;
  }
  
  /* if the integer is minus one, return the inverse of the operand        */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) == -1 ) {
    res = libGAP_AINV( op );
  }
  
  /* if the integer is negative, invert the operand and the integer        */
  else if ( ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) <  -1 )
            || libGAP_IS_INTNEG(n) ) {
    res = libGAP_AINV( op );
    if ( res == libGAP_Fail ) {
      return libGAP_ErrorReturnObj(
                            "Operations: <obj> must have an additive inverse",
                            0L, 0L,
                            "you can supply an inverse <inv> for <obj> via 'return <inv>;'" );
    }
    res = libGAP_PROD( libGAP_AINV( n ), res );
  }

  /* if the integer is small, compute the product by repeated doubling     */
  /* the loop invariant is <result> = <k>*<res> + <l>*<op>, <l> < <k>      */
  /* <res> = 0 means that <res> is the neutral element                     */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) >   1 ) {
    res = 0;
    k = 1L << (libGAP_NR_SMALL_INT_BITS+1);
    l = libGAP_INT_INTOBJ(n);
    while ( 1 < k ) {
      res = (res == 0 ? res : libGAP_SUM( res, res ));
      k = k / 2;
      if ( k <= l ) {
        res = (res == 0 ? op : libGAP_SUM( res, op ));
        l = l - k;
      }
    }
  }
  
  /* if the integer is large, compute the product by repeated doubling     */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INTPOS ) {
    res = 0;
    for ( i = libGAP_SIZE_INT(n); 0 < i; i-- ) {
      k = 8*sizeof(libGAP_TypLimb);
      l = ((libGAP_TypLimb*) libGAP_ADDR_INT(n))[i-1];
      while ( 0 < k ) {
        res = (res == 0 ? res : libGAP_SUM( res, res ));
        k--;
        if ( (l >> k) & 1 ) {
          res = (res == 0 ? op : libGAP_SUM( res, op ));
        }
      }
    }
  }
  
  /* return the result                                                     */
  return res;
}

libGAP_Obj libGAP_ProdIntObjFunc;

libGAP_Obj libGAP_FuncPROD_INT_OBJ ( libGAP_Obj self, libGAP_Obj opL, libGAP_Obj opR )
{
  return libGAP_ProdIntObj( opL, opR );
}


/****************************************************************************
**
*F  OneInt(<gmp>) . . . . . . . . . . . . . . . . . . . . . one of an integer
*/
static libGAP_Obj libGAP_OneAttr;

libGAP_Obj libGAP_OneInt ( libGAP_Obj op )
{
  return libGAP_INTOBJ_INT( 1 );
}


/****************************************************************************
**
*F  PowInt( <intL>, <intR> )  . . . . . . . . . . . . . . power of an integer
**
**  'PowInt' returns the <intR>-th (an integer) power of the integer  <intL>.
**  'PowInt' handles operands of type 'T_INT', 'T_INTPOS' and 'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalPow' for a better  efficiency.
**
**  Is called from the 'EvalPow'  binop so both operands are already evaluated.
*/
libGAP_Obj libGAP_PowInt ( libGAP_Obj gmpL, libGAP_Obj gmpR )
{
  libGAP_Int                 i;
  libGAP_Obj                 pow;
  
  /* power with a large exponent                                         */
  if ( ! libGAP_IS_INTOBJ(gmpR) ) {
    if ( gmpL == libGAP_INTOBJ_INT(0) )
      pow = libGAP_INTOBJ_INT(0);
    else if ( gmpL == libGAP_INTOBJ_INT(1) )
      pow = libGAP_INTOBJ_INT(1);
    else if ( gmpL == libGAP_INTOBJ_INT(-1) && libGAP_ADDR_INT(gmpR)[0] % 2 == 0 )
      pow = libGAP_INTOBJ_INT(1);
    else if ( gmpL == libGAP_INTOBJ_INT(-1) && libGAP_ADDR_INT(gmpR)[0] % 2 != 0 )
      pow = libGAP_INTOBJ_INT(-1);
    else {
      gmpR = libGAP_ErrorReturnObj(
                            "Integer operands: <exponent> is too large",
                            0L, 0L,
                            "you can replace the integer <exponent> via 'return <exponent>;'" );
      return libGAP_POW( gmpL, gmpR );
    }
  }
  
  /* power with a negative exponent                                      */
  else if ( libGAP_INT_INTOBJ(gmpR) < 0 ) {
    if ( gmpL == libGAP_INTOBJ_INT(0) ) {
      gmpL = libGAP_ErrorReturnObj(
                            "Integer operands: <base> must not be zero",
                            0L, 0L,
                            "you can replace the integer <base> via 'return <base>;'" );
      return libGAP_POW( gmpL, gmpR );
    }
    else if ( gmpL == libGAP_INTOBJ_INT(1) )
      pow = libGAP_INTOBJ_INT(1);
    else if ( gmpL == libGAP_INTOBJ_INT(-1) && libGAP_INT_INTOBJ(gmpR) % 2 == 0 )
      pow = libGAP_INTOBJ_INT(1);
    else if ( gmpL == libGAP_INTOBJ_INT(-1) && libGAP_INT_INTOBJ(gmpR) % 2 != 0 )
      pow = libGAP_INTOBJ_INT(-1);
    else
      pow = libGAP_QUO( libGAP_INTOBJ_INT(1),
                 libGAP_PowInt( gmpL, libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ(gmpR)) ) );
  }
  
  /* findme - can we use the gmp function mpz_n_pow_ui? */

  /* power with a small positive exponent, do it by a repeated squaring  */
  else {
    pow = libGAP_INTOBJ_INT(1);
    i = libGAP_INT_INTOBJ(gmpR);
    while ( i != 0 ) {
      if ( i % 2 == 1 )  pow = libGAP_ProdInt( pow, gmpL );
      if ( i     >  1 )  gmpL = libGAP_ProdInt( gmpL, gmpL );
      libGAP_TakeInterrupt();
      i = i / 2;
    }
  }
  
  /* return the power                                                    */
  return pow;
}


/****************************************************************************
**
*F  PowObjInt(<op>,<n>) . . . . . . . . . . power of an object and an integer
*/
libGAP_Obj             libGAP_PowObjInt ( libGAP_Obj op, libGAP_Obj n )
{
  libGAP_Obj                 res = 0;        /* result                          */
  libGAP_UInt                i, k;           /* loop variables                  */
  libGAP_TypLimb             l;              /* loop variable                   */
  
  /* if the integer is zero, return the neutral element of the operand   */
  if      ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  0 ) {
    return libGAP_ONE_MUT( op );
  }
  
  /* if the integer is one, return a copy of the operand                 */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  1 ) {
    res = libGAP_CopyObj( op, 1 );
  }
  
  /* if the integer is minus one, return the inverse of the operand      */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) == -1 ) {
    res = libGAP_INV_MUT( op );
  }
  
  /* if the integer is negative, invert the operand and the integer      */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) <   0 ) {
    res = libGAP_INV_MUT( op );
    if ( res == libGAP_Fail ) {
      return libGAP_ErrorReturnObj(
                            "Operations: <obj> must have an inverse",
                            0L, 0L,
                            "you can supply an inverse <inv> for <obj> via 'return <inv>;'" );
    }
    res = libGAP_POW( res, libGAP_AINV( n ) );
  }
  
  /* if the integer is negative, invert the operand and the integer      */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INTNEG ) {
    res = libGAP_INV_MUT( op );
    if ( res == libGAP_Fail ) {
      return libGAP_ErrorReturnObj(
                            "Operations: <obj> must have an inverse",
                            0L, 0L,
                            "you can supply an inverse <inv> for <obj> via 'return <inv>;'" );
    }
    res = libGAP_POW( res, libGAP_AINV( n ) );
  }
  
  /* if the integer is small, compute the power by repeated squaring     */
  /* the loop invariant is <result> = <res>^<k> * <op>^<l>, <l> < <k>    */
  /* <res> = 0 means that <res> is the neutral element                   */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) >   0 ) {
    res = 0;
    k = 1L << (libGAP_NR_SMALL_INT_BITS+1);
    l = libGAP_INT_INTOBJ(n);
    while ( 1 < k ) {
      res = (res == 0 ? res : libGAP_PROD( res, res ));
      k = k / 2;
      if ( k <= l ) {
        res = (res == 0 ? op : libGAP_PROD( res, op ));
        l = l - k;
      }
    }
  }
  
  /* if the integer is large, compute the power by repeated squaring     */
  else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INTPOS ) {
    res = 0;
    for ( i = libGAP_SIZE_INT(n); 0 < i; i-- ) {
      k = 8*sizeof(libGAP_TypLimb);
      l = ((libGAP_TypLimb*) libGAP_ADDR_INT(n))[i-1];
      while ( 0 < k ) {
        res = (res == 0 ? res : libGAP_PROD( res, res ));
        k--;
        if ( (l>>k) & 1 ) {
          res = (res == 0 ? op : libGAP_PROD( res, op ));
        }
      }
    }
  }

  /* return the result                                                   */
  return res;
}

libGAP_Obj libGAP_PowObjIntFunc;

libGAP_Obj libGAP_FuncPOW_OBJ_INT ( libGAP_Obj self, libGAP_Obj opL, libGAP_Obj opR )
{
  return libGAP_PowObjInt( opL, opR );
}


/****************************************************************************
**
*F  ModInt( <intL>, <intR> )  . representative of residue class of an integer
**
**  'ModInt' returns the smallest positive representant of the residue  class
**  of the  integer  <intL>  modulo  the  integer  <intR>.  'ModInt'  handles
**  operands of type 'T_INT', 'T_INTPOS', 'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalMod' for a better efficiency.
p**
**  Is called from the 'EvalMod'  binop so both operands are already evaluated.
*/
libGAP_Obj libGAP_ModInt ( libGAP_Obj opL, libGAP_Obj opR )
{
  libGAP_Int                    i;             /* loop count, value for small int */
  libGAP_Int                    k;             /* loop count, value for small int */
  libGAP_UInt                   c;             /* product of two digits           */
  libGAP_Obj                  mod;             /* handle of the remainder bag     */
  libGAP_Obj                  quo;             /* handle of the quotient bag      */
  
  /* compute the remainder of two small integers                           */
  if ( libGAP_ARE_INTOBJS( opL, opR ) ) {
    
    /* pathological case first                                             */
    if ( opR == libGAP_INTOBJ_INT(0) ) {
      opR = libGAP_ErrorReturnObj(
                           "Integer operations: <divisor> must be nonzero",
                           0L, 0L,
                           "you can replace the integer <divisor> via 'return <divisor>;'" );
      return libGAP_MOD( opL, opR );
    }
    
    /* get the integer values                                              */
    i = libGAP_INT_INTOBJ(opL);
    k = libGAP_INT_INTOBJ(opR);
    
    /* compute the remainder, make sure we divide only positive numbers    */
    if (      0 <= i && 0 <= k )  i =       (  i %  k );
    else if ( 0 <= i && k <  0 )  i =       (  i % -k );
    else if ( i < 0  && 0 <= k )  i = ( k - ( -i %  k )) % k;
    else if ( i < 0  && k <  0 )  i = (-k - ( -i % -k )) % k;
    mod = libGAP_INTOBJ_INT( i );
    
  }
  
  /* compute the remainder of a small integer by a large integer           */
  else if ( libGAP_IS_INTOBJ(opL) ) {
    
    /* the small int -(1<<28) mod the large int (1<<28) is 0               */
    if ( opL == libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS) )
         && ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
         && ( libGAP_SIZE_INT(opR) == 1 )
         && ( libGAP_VAL_LIMB0(opR) == (libGAP_TypLimb)(1L<<libGAP_NR_SMALL_INT_BITS) ) )
      mod = libGAP_INTOBJ_INT(0);
    
    /* in all other cases the remainder is equal the left operand          */
    else if ( 0 <= libGAP_INT_INTOBJ(opL) )
      mod = opL;
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
      mod = libGAP_SumOrDiffInt( opL, opR,  1 );
    else
      mod = libGAP_SumOrDiffInt( opL, opR, -1 );
  }
  
  /* compute the remainder of a large integer by a small integer           */
  else if ( libGAP_IS_INTOBJ(opR) ) {

    /* pathological case first                                             */
    if ( opR == libGAP_INTOBJ_INT(0) ) {
      opR = libGAP_ErrorReturnObj(
                           "Integer operations: <divisor> must be nonzero",
                           0L, 0L,
                           "you can replace the integer <divisor> via 'return <divisor>;'" );
      return libGAP_MOD( opL, opR );
    }
    
    /* get the integer value, make positive                                */
    i = libGAP_INT_INTOBJ(opR);  if ( i < 0 )  i = -i;
    
    /* maybe it's trivial                                                  */
    if ( libGAP_INTBASE % i == 0 ) {
      c = libGAP_ADDR_INT(opL)[0] % i;
    }
    
    /* otherwise use the gmp function to divide                            */
    else {
      c = mpn_mod_1( libGAP_ADDR_INT(opL), libGAP_SIZE_INT(opL), (libGAP_TypLimb)i );
    }
    
    /* now c is the result, it has the same sign as the left operand       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
      mod = libGAP_INTOBJ_INT( c );
    else if ( c == 0 )
      mod = libGAP_INTOBJ_INT( c );
    else if ( 0 <= libGAP_INT_INTOBJ(opR) )
      mod = libGAP_SumOrDiffInt( libGAP_INTOBJ_INT( -(libGAP_Int)c ), opR,  1 );
    else
      mod = libGAP_SumOrDiffInt( libGAP_INTOBJ_INT( -(libGAP_Int)c ), opR, -1 );
    
  }
  
  /* compute the remainder of a large integer modulo a large integer       */
  else {

    /* trivial case first                                                  */
    if ( libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) ) {
      if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
        return opL;
      else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
        mod = libGAP_SumOrDiffInt( opL, opR,  1 );
      else
        mod = libGAP_SumOrDiffInt( opL, opR, -1 );
      if libGAP_IS_INTNEG(mod) return libGAP_NEW_INTPOS(mod);
      else return mod;
    }
    
    mod = libGAP_NewBag( libGAP_TNUM_OBJ(opL), (libGAP_SIZE_INT(opL)+1)*sizeof(libGAP_TypLimb) );

    quo = libGAP_NewBag( libGAP_T_INTPOS,
                   (libGAP_SIZE_INT(opL)-libGAP_SIZE_INT(opR)+1)*sizeof(libGAP_TypLimb) );

    /* and let gmp do the work                                             */
    mpn_tdiv_qr( libGAP_ADDR_INT(quo), libGAP_ADDR_INT(mod), 0,
                 libGAP_ADDR_INT(opL), libGAP_SIZE_INT(opL),
                 libGAP_ADDR_INT(opR), libGAP_SIZE_INT(opR)    );
      
    /* reduce to small integer if possible, otherwise shrink bag           */
    mod = libGAP_GMP_NORMALIZE( mod );
    mod = libGAP_GMP_REDUCE( mod );
    
    /* make the representative positive                                    */
    if ( (libGAP_TNUM_OBJ(mod) == libGAP_T_INT && libGAP_INT_INTOBJ(mod) < 0)
         || libGAP_TNUM_OBJ(mod) == libGAP_T_INTNEG ) {
      if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
        mod = libGAP_SumOrDiffInt( mod, opR,  1 );
      else
        mod = libGAP_SumOrDiffInt( mod, opR, -1 );
    }
    
  }
  
  /* return the result                                                     */
  if libGAP_IS_INTNEG(mod)
    return libGAP_NEW_INTPOS(mod);
  else if ( libGAP_IS_INTOBJ(mod) && 0 > libGAP_INT_INTOBJ(mod) )
    return libGAP_INTOBJ_INT(-libGAP_INT_INTOBJ(mod));
  else
    return mod;
}


/****************************************************************************
**
*F  QuoInt( <intL>, <intR> )  . . . . . . . . . . . quotient of two integers
**
**  'QuoInt' returns the integer part of the two integers <intL> and  <intR>.
**  'QuoInt' handles operands of type  'T_INT',  'T_INTPOS'  and  'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**
**  Note that this routine is not called from 'EvalQuo', the  division  of  two
**  integers yields  a  rational  and  is  therefor  performed  in  'QuoRat'.
**  This operation is however available through the internal function 'Quo'.
*/
libGAP_Obj libGAP_QuoInt ( libGAP_Obj opL, libGAP_Obj opR )
{
  libGAP_Int                 i;              /* loop count, value for small int   */
  libGAP_Int                 k;              /* loop count, value for small int   */
  libGAP_Obj                 quo;            /* handle of the result bag          */
  libGAP_Obj                 rem;            /* handle of the remainder bag       */
  
  /* divide two small integers                                             */
  if ( libGAP_ARE_INTOBJS( opL, opR ) ) {
    
    /* pathological case first                                             */
    if ( opR == libGAP_INTOBJ_INT(0) ) {
      opR = libGAP_ErrorReturnObj(
                           "Integer operations: <divisor> must be nonzero",
                           0L, 0L,
                           "you can replace the integer <divisor> via 'return <divisor>;'" );
      return libGAP_QUO( opL, opR );
    }
    
    /* the small int -(1<<28) divided by -1 is the large int (1<<28)       */
    if ( opL == libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS)) 
         && opR == libGAP_INTOBJ_INT(-1) ) {
      quo = libGAP_NewBag( libGAP_T_INTPOS, sizeof(libGAP_TypLimb) );
      libGAP_SET_VAL_LIMB0( quo, 1L<<libGAP_NR_SMALL_INT_BITS );
      return quo;
    }
    
    /* get the integer values                                              */
    i = libGAP_INT_INTOBJ(opL);
    k = libGAP_INT_INTOBJ(opR);
    
    /* divide, make sure we divide only positive numbers                   */
    if (      0 <= i && 0 <= k )  i =    (  i /  k );
    else if ( 0 <= i && k <  0 )  i =  - (  i / -k );
    else if ( i < 0  && 0 <= k )  i =  - ( -i /  k );
    else if ( i < 0  && k <  0 )  i =    ( -i / -k );
    quo = libGAP_INTOBJ_INT( i );
    
  }
  
  /* divide a small integer by a large one                                 */
  else if ( libGAP_IS_INTOBJ(opL) ) {
    
    /* the small int -(1<<28) divided by the large int (1<<28) is -1       */
    
    if ( opL == libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS))
         && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS && libGAP_SIZE_INT(opR) == 1
         && libGAP_VAL_LIMB0(opR) == 1L<<libGAP_NR_SMALL_INT_BITS )
      quo = libGAP_INTOBJ_INT(-1);
    
    /* in all other cases the quotient is of course zero                   */
    else
      quo = libGAP_INTOBJ_INT(0);
    
  }
  
  /* divide a large integer by a small integer                             */
  else if ( libGAP_IS_INTOBJ(opR) ) {
    
    /* pathological case first                                             */
    if ( opR == libGAP_INTOBJ_INT(0) ) {
      opR = libGAP_ErrorReturnObj(
                           "Integer operations: <divisor> must be nonzero",
                           0L, 0L,
                           "you can replace the integer <divisor> via 'return <divisor>;'" );
      return libGAP_QUO( opL, opR );
    }
    
    /* allocate a bag for the result and set up the pointers               */
    if ( (libGAP_TNUM_OBJ(opL)==libGAP_T_INTPOS && 0 < libGAP_INT_INTOBJ(opR))
         || (libGAP_TNUM_OBJ(opL)==libGAP_T_INTNEG && libGAP_INT_INTOBJ(opR) < 0) )
      quo = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL) );
    else
      quo = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opL) );
    
    opR = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, opR );

    /* use gmp function for dividing by a 1-limb number                    */
    mpn_divrem_1( libGAP_ADDR_INT(quo), 0,
                  libGAP_ADDR_INT(opL), libGAP_SIZE_INT(opL),
                  *libGAP_ADDR_INT(opR) );
  }
  
  /* divide a large integer by a large integer                             */
  else {
    
    /* trivial case first                                                  */
    if ( libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) )
      return libGAP_INTOBJ_INT(0);
    
    /* create a new bag for the remainder                                  */
    rem = libGAP_NewBag( libGAP_TNUM_OBJ(opL), (libGAP_SIZE_INT(opL)+1)*sizeof(libGAP_TypLimb) );

    /* allocate a bag for the quotient                                     */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_TNUM_OBJ(opR) )
      quo = libGAP_NewBag( libGAP_T_INTPOS, 
                    (libGAP_SIZE_INT(opL)-libGAP_SIZE_INT(opR)+1)*sizeof(libGAP_TypLimb) );
    else
      quo = libGAP_NewBag( libGAP_T_INTNEG,
                    (libGAP_SIZE_INT(opL)-libGAP_SIZE_INT(opR)+1)*sizeof(libGAP_TypLimb) );

    mpn_tdiv_qr( libGAP_ADDR_INT(quo), libGAP_ADDR_INT(rem), 0,
                 libGAP_ADDR_INT(opL), libGAP_SIZE_INT(opL),
                 libGAP_ADDR_INT(opR), libGAP_SIZE_INT(opR) );
  }
  
  /* normalize and return the result                                       */
  quo = libGAP_GMP_NORMALIZE(quo);
  quo = libGAP_GMP_REDUCE( quo );
  return quo;
}


/****************************************************************************
**
*F  FuncQUO_INT(<self>,<opL>,<opR>) . . . . . . .  internal function 'QuoInt'
**
**  'FuncQUO_INT' implements the internal function 'QuoInt'.
**
**  'QuoInt( <i>, <k> )'
**
**  'Quo' returns the  integer part of the quotient  of its integer operands.
**  If <i>  and <k> are  positive 'Quo( <i>,  <k> )' is  the largest positive
**  integer <q>  such that '<q> * <k>  \<= <i>'.  If  <i> or  <k> or both are
**  negative we define 'Abs( Quo(<i>,<k>) ) = Quo( Abs(<i>), Abs(<k>) )'  and
**  'Sign( Quo(<i>,<k>) ) = Sign(<i>) * Sign(<k>)'.  Dividing by 0  causes an
**  error.  'Rem' (see "Rem") can be used to compute the remainder.
*/
libGAP_Obj libGAP_FuncQUO_INT ( libGAP_Obj self, libGAP_Obj opL, libGAP_Obj opR )
{
  /* check the arguments                                                   */
  while ( libGAP_TNUM_OBJ(opL) != libGAP_T_INT
          && libGAP_TNUM_OBJ(opL) != libGAP_T_INTPOS
          && libGAP_TNUM_OBJ(opL) != libGAP_T_INTNEG ) {
    opL = libGAP_ErrorReturnObj(
                         "QuoInt: <left> must be a int (not a %s)",
                         (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L,
                         "you can replace <left> via 'return <left>;'" );
  }
  while ( libGAP_TNUM_OBJ(opR) != libGAP_T_INT
          && libGAP_TNUM_OBJ(opR) != libGAP_T_INTPOS
          && libGAP_TNUM_OBJ(opR) != libGAP_T_INTNEG ) {
    opR = libGAP_ErrorReturnObj(
                         "QuoInt: <right> must be a int (not a %s)",
                         (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L,
                         "you can replace <right> via 'return <right>;'" );
  }
  
  /* return the quotient                                                   */
  return libGAP_QuoInt( opL, opR );
}


/****************************************************************************
**
*F  RemInt( <intL>, <intR> )  . . . . . . . . . . . remainder of two integers
**
**  'RemInt' returns the remainder of the quotient  of  the  integers  <intL>
**  and <intR>.  'RemInt' handles operands of type  'T_INT',  'T_INTPOS'  and
**  'T_INTNEG'.
**
**  Note that the remainder is different from the value returned by the 'mod'
**  operator which is always positive.
*/
libGAP_Obj libGAP_RemInt ( libGAP_Obj opL, libGAP_Obj opR )
{
  libGAP_Int                 i;              /* loop count, value for small int   */
  libGAP_Int                 k;              /* loop count, value for small int   */
  libGAP_UInt                c;              /* product of two digits             */
  libGAP_Obj                 rem;            /* handle of the remainder bag       */
  libGAP_Obj                 quo;            /* handle of the quotient bag        */

  /* compute the remainder of two small integers                           */
  if ( libGAP_ARE_INTOBJS( opL, opR ) ) {
    
    /* pathological case first                                             */
    if ( opR == libGAP_INTOBJ_INT(0) ) {
      opR = libGAP_ErrorReturnObj(
                           "Integer operations: <divisor> must be nonzero",
                           0L, 0L,
                           "you can replace the integer <divisor> via 'return <divisor>;'" );
      return libGAP_QUO( opL, opR );
    }
    
    /* get the integer values                                              */
    i = libGAP_INT_INTOBJ(opL);
    k = libGAP_INT_INTOBJ(opR);
    
    /* compute the remainder, make sure we divide only positive numbers    */
    if (      0 <= i && 0 <= k )  i =    (  i %  k );
    else if ( 0 <= i && k <  0 )  i =    (  i % -k );
    else if ( i < 0  && 0 <= k )  i =  - ( -i %  k );
    else if ( i < 0  && k <  0 )  i =  - ( -i % -k );
    rem = libGAP_INTOBJ_INT( i );
    
  }
  
  /* compute the remainder of a small integer by a large integer           */
  else if ( libGAP_IS_INTOBJ(opL) ) {
    
    /* the small int -(1<<28) rem the large int (1<<28) is 0               */
    if ( opL == libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS))
         && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS && libGAP_SIZE_INT(opR) == 1
         && libGAP_VAL_LIMB0(opR) == 1L<<libGAP_NR_SMALL_INT_BITS )
      rem = libGAP_INTOBJ_INT(0);
    
    /* in all other cases the remainder is equal the left operand          */
    else
      rem = opL;
  }
  
  /* compute the remainder of a large integer by a small integer           */
  else if ( libGAP_IS_INTOBJ(opR) ) {
    
    /* pathological case first                                             */
    if ( opR == libGAP_INTOBJ_INT(0) ) {
      opR = libGAP_ErrorReturnObj(
                           "Integer operations: <divisor> must be nonzero",
                           0L, 0L,
                           "you can replace the integer <divisor> via 'return <divisor>;'" );
      return libGAP_QUO( opL, opR );
    }
    
    /* maybe it's trivial                                                   */
    if ( libGAP_INTBASE % libGAP_INT_INTOBJ(libGAP_AbsInt(opR)) == 0 ) {
      c = libGAP_ADDR_INT(opL)[0] % libGAP_INT_INTOBJ(libGAP_AbsInt(opR));
    }
    
    /* otherwise run through the left operand and divide digitwise         */
    else {
      opR = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, opR );
      c = mpn_mod_1( libGAP_ADDR_INT(opL), libGAP_SIZE_INT(opL), *libGAP_ADDR_INT(opR) );
    }
    
    /* now c is the result, it has the same sign as the left operand       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
      rem = libGAP_INTOBJ_INT(  c );
    else
      rem = libGAP_INTOBJ_INT( -(libGAP_Int)c );
    
  }
  
  /* compute the remainder of a large integer modulo a large integer       */
  else {
    
    /* trivial case first                                                  */
    if ( libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) )
      return opL;
    
    rem = libGAP_NewBag( libGAP_TNUM_OBJ(opL), (libGAP_SIZE_INT(opL)+1)*sizeof(libGAP_TypLimb) );
    
    quo = libGAP_NewBag( libGAP_T_INTPOS,
                  (libGAP_SIZE_INT(opL)-libGAP_SIZE_INT(opR)+1)*sizeof(libGAP_TypLimb) );
    
    /* and let gmp do the work                                             */
    mpn_tdiv_qr( libGAP_ADDR_INT(quo), libGAP_ADDR_INT(rem), 0,
                 libGAP_ADDR_INT(opL), libGAP_SIZE_INT(opL),
                 libGAP_ADDR_INT(opR), libGAP_SIZE_INT(opR)    );
    
    /* reduce to small integer if possible, otherwise shrink bag           */
    rem = libGAP_GMP_NORMALIZE( rem );
    rem = libGAP_GMP_REDUCE( rem );
    
  }
  
  /* return the result                                                     */
  return rem;
}


/****************************************************************************
**
*F  FuncREM_INT(<self>,<opL>,<opR>)  . . . . . . .  internal function 'RemInt'
**
**  'FuncREM_INT' implements the internal function 'RemInt'.
**
**  'RemInt( <i>, <k> )'
**
**  'Rem' returns the remainder of its two integer operands,  i.e., if <k> is
**  not equal to zero 'Rem( <i>, <k> ) = <i> - <k> *  Quo( <i>, <k> )'.  Note
**  that the rules given  for 'Quo' (see "Quo") imply  that 'Rem( <i>, <k> )'
**  has the same sign as <i> and its absolute value is strictly less than the
**  absolute value of <k>.  Dividing by 0 causes an error.
*/
libGAP_Obj libGAP_FuncREM_INT ( libGAP_Obj self, libGAP_Obj opL, libGAP_Obj opR )
{
  /* check the arguments                                                   */
  while ( libGAP_TNUM_OBJ(opL) != libGAP_T_INT
          && libGAP_TNUM_OBJ(opL) != libGAP_T_INTPOS
          && libGAP_TNUM_OBJ(opL) != libGAP_T_INTNEG ) {
    opL = libGAP_ErrorReturnObj(
                         "RemInt: <left> must be an integer (not a %s)",
                         (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L,
                         "you can replace <left> via 'return <left>;'" );
  }
  while ( libGAP_TNUM_OBJ(opR) != libGAP_T_INT
          && libGAP_TNUM_OBJ(opR) != libGAP_T_INTPOS
          && libGAP_TNUM_OBJ(opR) != libGAP_T_INTNEG ) {
    opR = libGAP_ErrorReturnObj(
                         "RemInt: <right> must be an integer (not a %s)",
                         (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L,
                         "you can replace <right> via 'return <right>;'" );
  }

  /* return the remainder                                                  */
  return libGAP_RemInt( opL, opR );
}


/****************************************************************************
**
*F  GcdInt( <opL>, <opR> )  . . . . . . . . . . .  gcd of two GMP integers
**
**  'GcdInt' returns the gcd of the two integers <opL> and <opR>.
**
**  It is called from 'FuncGCD_INT' and from the rational package.
*/
libGAP_Obj libGAP_GcdInt ( libGAP_Obj opL, libGAP_Obj opR )
{
  libGAP_Int                 i;              /* loop count, value for small int   */
  libGAP_Int                 k;              /* loop count, value for small int   */
  libGAP_UInt                c;              /* product of two digits             */
  libGAP_Obj                 gmpL;           /* copy of the first arg             */
  libGAP_Obj                 gmpR;           /* copy of the second arg            */
  libGAP_Obj                 gcd;            /* handle of the result              */
  libGAP_TypLimb             gcdsize;        /* number of limbs mpn_gcd returns   */
  libGAP_Int                 p,q;            /* number of zero limbs per arg      */
  libGAP_Int                 r,s;            /* number of zero bits per arg       */
  libGAP_TypLimb             bmask;          /* bit mask                          */
  libGAP_UInt                plimbs;         /* zero limbs to shift by */
  libGAP_UInt                prest;          /* and remaining zero bits */
  libGAP_UInt                overflow;       /* possible overflow from most 
                                         significant word */

  /* compute the gcd of two small integers                                 */
  if ( libGAP_ARE_INTOBJS( opL, opR ) ) {
    
    /* get the integer values, make them positive                          */
    i = libGAP_INT_INTOBJ(opL);  if ( i < 0 )  i = -i;
    k = libGAP_INT_INTOBJ(opR);  if ( k < 0 )  k = -k;
    
    /* compute the gcd using Euclids algorithm                             */
    while ( k != 0 ) {
      c = k;
      k = i % k;
      i = c;
    }
    
    /* now i is the result                                                 */
    gcd = libGAP_GMPorINTOBJ_INT( (libGAP_Int)i );
    return gcd;    
  }
  
  /* compute the gcd of a small and a large integer                        */
  else if ( libGAP_IS_INTOBJ(opL) || libGAP_IS_INTOBJ(opR) ) {
    
    /* make the right operand the small one                                */
    if ( libGAP_IS_INTOBJ(opL) ) {
      gcd = opL;  opL = opR;  opR = gcd;
    }
    
    /* maybe it's trivial                                                  */
    if ( opR == libGAP_INTOBJ_INT(0) ) {
      if( libGAP_TNUM_OBJ( opL ) == libGAP_T_INTNEG ) {
        /* If opL is negative, change the sign.  We do this by
           copying opL into a bag of type T_INTPOS.  Note that
           opL is a large negative number, so it cannot be the
           the negative of 1 << NR_SMALL_INT_BITS.                     */
        gcd = libGAP_NEW_INTPOS( opL );
        return gcd;
      }
      else return opL;
    }
    
    /* compute the gcd                                                     */
    opR = libGAP_FuncGMP_INTOBJ( (libGAP_Obj)0, opR );
    i = mpn_gcd_1( libGAP_ADDR_INT(opL), libGAP_SIZE_INT(opL), *libGAP_ADDR_INT(opR) );
    gcd = libGAP_GMPorINTOBJ_INT( (libGAP_Int)i );
    return gcd;
  }
  
  /* compute the gcd of two large integers                                 */
  else {
    if ( libGAP_EqInt(opL,opR) ) {
      if libGAP_IS_INTNEG(opL) {
        return libGAP_NEW_INTPOS(opL);
      }
      else {
        return opL;
      }
    }
    gmpL = libGAP_NEW_INT(opL); gmpR = libGAP_NEW_INT(opR);

    /* find highest power of 2 dividing gmpL and divide out by this */
    for ( p = 0 ; libGAP_ADDR_INT(gmpL)[p] == (libGAP_TypLimb)0; p++ ) {
      for ( i = 0 ; i < mp_bits_per_limb ; i++ ) {
      }
    }
    for ( bmask = (libGAP_TypLimb)1, r = 0 ;
          ( (bmask & libGAP_ADDR_INT(gmpL)[p]) == 0 ) && bmask != (libGAP_TypLimb)0 ;
          bmask = bmask << 1, r++ ) {
    }
    p = p*mp_bits_per_limb+r;
    for ( i = 0 ; i < p ; i++ ){
      mpn_rshift( libGAP_ADDR_INT(gmpL), libGAP_ADDR_INT(gmpL),
                  libGAP_SIZE_INT(gmpL), (libGAP_UInt)1 );
    }
    gmpL = libGAP_GMP_NORMALIZE(gmpL);

    /* find highest power of 2 dividing gmpR and divide out by this */
    for ( q = 0 ; libGAP_ADDR_INT(gmpR)[q] == (libGAP_TypLimb)0; q++ ) {
      for ( i = 0 ; i < mp_bits_per_limb ; i++ ) {
      }
    }
    for ( bmask = (libGAP_TypLimb)1, s = 0 ;
          ( (bmask & libGAP_ADDR_INT(gmpR)[q]) == 0 ) && bmask != (libGAP_TypLimb)0 ;
          bmask=bmask << 1,  s++ ) {
    }
    q = q*mp_bits_per_limb+s;
    for (i=0;i<q;i++){
      mpn_rshift( libGAP_ADDR_INT(gmpR), libGAP_ADDR_INT(gmpR),
                  libGAP_SIZE_INT(gmpR), (libGAP_UInt)1 );
    }
    gmpR = libGAP_GMP_NORMALIZE(gmpR);

    /* put smaller object to right */
    if ( libGAP_SIZE_INT(gmpL) < libGAP_SIZE_INT(gmpR) ||
         ( libGAP_SIZE_INT(gmpL) == libGAP_SIZE_INT(gmpR) &&
           mpn_cmp( libGAP_ADDR_INT(gmpL), libGAP_ADDR_INT(gmpR), libGAP_SIZE_INT(gmpL) ) < 0 ) ) {
      gcd = gmpR; gmpR = gmpL; gmpL = gcd;
    }
    
    /* get gcd of odd numbers gmpL, gmpR - put it in a bag as big as one
       of the original args, which will be big enough for the gcd */
    gcd = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opR) );
    
    /* choose smaller of p,q and make it p 
       we are going to multiply back in by 2 to that power */
    if ( p > q ) p = q;
    plimbs = p/mp_bits_per_limb;
    prest = p - plimbs*mp_bits_per_limb;
    
    /* We deal with most of the power of two by placing some
       0 limbs below the least significant limb of the GCD */
    for (i = 0; i < plimbs; i++)
      libGAP_ADDR_INT(gcd)[i] = 0;

    /* Now we do the GCD -- gcdsize tells us the number of 
       limbs used for the result */

    gcdsize = mpn_gcd( plimbs + libGAP_ADDR_INT(gcd), 
             libGAP_ADDR_INT(gmpL), libGAP_SIZE_INT(gmpL),
             libGAP_ADDR_INT(gmpR), libGAP_SIZE_INT(gmpR) );
    
    /* Now we do the rest of the power of two */
    if (prest != 0)
      {
        overflow = mpn_lshift( plimbs + libGAP_ADDR_INT(gcd), plimbs + libGAP_ADDR_INT(gcd),
                  gcdsize, (libGAP_UInt)prest );
        
        /* which might extend the GCD to one more limb */
        if (overflow != 0)
          {
            libGAP_ADDR_INT(gcd)[gcdsize + plimbs] = overflow;
            gcdsize++;
          }
      }

    /* if the bag is too big, reduce it (stuff in extra space may not be 0) */
    if ( gcdsize+plimbs != libGAP_SIZE_INT(opR) ) {
      libGAP_ResizeBag( gcd, (gcdsize+plimbs)*sizeof(libGAP_TypLimb) );
    }

  }
  gcd = libGAP_GMP_NORMALIZE(gcd);
  gcd = libGAP_GMP_REDUCE(gcd);
  
  /* return the result                                                     */
  return gcd;
}

/****************************************************************************
**
*F  FuncGCD_INT(<self>,<opL>,<opR>)  . . . . . . .  internal function 'GcdInt'
**
**  'FuncGCD_INT' implements the internal function 'GcdInt'.
**
**  'GcdInt( <i>, <k> )'
**
**  'Gcd'  returns the greatest common divisor   of the two  integers <m> and
**  <n>, i.e.,  the  greatest integer that  divides  both <m>  and  <n>.  The
**  greatest common divisor is never negative, even if the arguments are.  We
**  define $gcd( m, 0 ) = gcd( 0, m ) = abs( m )$ and $gcd( 0, 0 ) = 0$.
*/
libGAP_Obj libGAP_FuncGCD_INT ( libGAP_Obj self, libGAP_Obj opL, libGAP_Obj opR )
{
  /* check the arguments                                                   */
  while ( libGAP_TNUM_OBJ(opL) != libGAP_T_INT
          && libGAP_TNUM_OBJ(opL) != libGAP_T_INTPOS
          && libGAP_TNUM_OBJ(opL) != libGAP_T_INTNEG ) {
    opL = libGAP_ErrorReturnObj(
                         "GcdInt: <left> must be an integer (not a %s)",
                         (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L,
                         "you can replace <left> via 'return <left>;'" );
  }
  while ( libGAP_TNUM_OBJ(opR) != libGAP_T_INT
          && libGAP_TNUM_OBJ(opR) != libGAP_T_INTPOS
          && libGAP_TNUM_OBJ(opR) != libGAP_T_INTNEG ) {
    opR = libGAP_ErrorReturnObj(
                         "GcdInt: <right> must be an integer (not a %s)",
                         (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L,
                         "you can replace <right> via 'return <right>;'" );
  }
  
  /* return the gcd                                                        */
  return libGAP_GcdInt( opL, opR );
}






/****************************************************************************
**
** * * * * * * * "Mersenne twister" random numbers  * * * * * * * * * * * * *
**
**  Part of this code for fast generation of 32 bit pseudo random numbers with 
**  a period of length 2^19937-1 and a 623-dimensional equidistribution is 
**  taken from:
**          http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html
**  (Also look in Wikipedia for "Mersenne twister".)
*/

/****************************************************************************
**
*F  RandomIntegerMT( <mtstr>, <nrbits> )
**  
**  Returns an integer with at most <nrbits> bits in uniform distribution. 
**  <nrbits> must be a small integer. <mtstr> is a string as returned by 
**  InitRandomMT.
**  
**  Implementation details are a bit tricky to obtain the same random 
**  integers on 32 bit and 64 bit machines (which have different long
**  integer digit lengths and different ranges of small integers).
**  
*/
/* for comparison in case result is small int */
libGAP_Obj libGAP_SMALLEST_INTPOS = NULL;

libGAP_Obj libGAP_FuncRandomIntegerMT(libGAP_Obj self, libGAP_Obj mtstr, libGAP_Obj nrbits)
{
  libGAP_Obj res;
  libGAP_Int i, n, q, r, qoff, len;
  libGAP_UInt4 *mt, rand;
  libGAP_TypLimb *pt;
  while (! libGAP_IsStringConv(mtstr)) {
     mtstr = libGAP_ErrorReturnObj(
         "<mtstr> must be a string, not a %s)",
         (libGAP_Int)libGAP_TNAM_OBJ(mtstr), 0L,
         "you can replace <mtstr> via 'return <mtstr>;'" );
  }
  while ((! libGAP_IsStringConv(mtstr)) || libGAP_GET_LEN_STRING(mtstr) < 2500) {
     mtstr = libGAP_ErrorReturnObj(
         "<mtstr> must be a string with at least 2500 characters, ",
         0L, 0L,
         "you can replace <mtstr> via 'return <mtstr>;'" );
  }
  while ((! libGAP_IS_INTOBJ(nrbits)) || libGAP_INT_INTOBJ(nrbits) < 0) {
     nrbits = libGAP_ErrorReturnObj(
         "<nrbits> must be a small non-negative integer, not a %s)",
         (libGAP_Int)libGAP_TNAM_OBJ(nrbits), 0L,
         "you can replace <mtstr> via 'return <mtstr>;'" );
  }
  n = libGAP_INT_INTOBJ(nrbits);

  /* small int case */
  if (n <= libGAP_NR_SMALL_INT_BITS) {
     mt = (libGAP_UInt4*) libGAP_CHARS_STRING(mtstr);
#ifdef libGAP_SYS_IS_64_BIT
     if (n <= 32) {
       res = libGAP_INTOBJ_INT((libGAP_Int)(libGAP_nextrandMT_int32(mt) & ((libGAP_UInt4) -1L >> (32-n))));
     }
     else {
       unsigned long  rd;
       rd = libGAP_nextrandMT_int32(mt);
       rd += (unsigned long) ((libGAP_UInt4) libGAP_nextrandMT_int32(mt) & 
                              ((libGAP_UInt4) -1L >> (64-n))) << 32;
       res = libGAP_INTOBJ_INT((libGAP_Int)rd);
     }  
#else
     res = libGAP_INTOBJ_INT((libGAP_Int)(libGAP_nextrandMT_int32(mt) & ((libGAP_UInt4) -1L >> (32-n))));
#endif
  }
  else {
     /* large int case - number of Limbs */
     q = n / GMP_LIMB_BITS;
     r = n - q*GMP_LIMB_BITS;
     qoff = q + (r==0 ? 0:1);
     len = qoff;
     res = libGAP_NewBag( libGAP_T_INTPOS, len*sizeof(libGAP_TypLimb) );
     pt = libGAP_ADDR_INT(res);
     mt = (libGAP_UInt4*) libGAP_CHARS_STRING(mtstr);
#ifdef libGAP_SYS_IS_64_BIT
     for (i = 0; i < qoff; i++, pt++) {
       rand = (libGAP_TypLimb) libGAP_nextrandMT_int32(mt);
       *pt = rand; 
       rand = (libGAP_TypLimb) libGAP_nextrandMT_int32(mt);
       *pt |= (libGAP_UInt)rand << 32; 
     }
#else
     for (i = 0; i < qoff; i++, pt++) {
       rand = libGAP_nextrandMT_int32(mt);
       *pt = (libGAP_TypLimb) rand;
     }
#endif
     if (r != 0) {
       libGAP_ADDR_INT(res)[qoff-1] = libGAP_ADDR_INT(res)[qoff-1] & ((libGAP_TypLimb)(-1)
                                                      >> (GMP_LIMB_BITS-r));
     }
     /* shrink bag if necessary */
     res = libGAP_GMP_NORMALIZE(res);
     /* convert result if small int */
     res = libGAP_GMP_REDUCE(res);
  }

  return res;
}

/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

  { "IS_INT", "obj", &libGAP_IsIntFilt,
    libGAP_FuncIS_INT, "src/gmpints.c:IS_INT" },

  { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

  { "QUO_INT", 2, "gmp1, gmp2",
    libGAP_FuncQUO_INT, "src/gmpints.c:QUO_INT" },

  { "ABS_INT", 1, "x",
    libGAP_FuncABS_INT, "src/gmpints.c:ABS_INT" },

  { "REM_INT", 2, "gmp1, gmp2",
    libGAP_FuncREM_INT, "src/gmpints.c:REM_INT" },

  { "GCD_INT", 2, "gmp1, gmp2",
    libGAP_FuncGCD_INT, "src/gmpints.c:GCD_INT" },
  
  { "PROD_INT_OBJ", 2, "gmp, obj",
    libGAP_FuncPROD_INT_OBJ, "src/gmpints.c:PROD_INT_OBJ" },
  
  { "POW_OBJ_INT", 2, "obj, gmp",
    libGAP_FuncPOW_OBJ_INT, "src/gmpints.c:POW_OBJ_INT" },
  
  { "GMP_REDUCE", 1, "obj",
    libGAP_FuncGMP_REDUCE, "src/gmpints.c:GMP_REDUCE" },

  { "GMP_NORMALIZE", 1, "obj",
    libGAP_FuncGMP_NORMALIZE, "src/gmpints.c:GMP_NORMALIZE" },

  { "HexStringInt", 1, "gmp",
    libGAP_FuncHexStringInt, "src/gmpints.c:HexStringInt" },
  
  { "IntHexString", 1, "string",
    libGAP_FuncIntHexString, "src/gmpints.c:IntHexString" },
  
  { "Log2Int", 1, "gmp",
    libGAP_FuncLog2Int, "src/gmpints.c:Log2Int" },

  { "STRING_INT", 1, "gmp",
    libGAP_FuncSTRING_INT, "src/gmpints.c:STRING_INT" },
  
  { "RandomIntegerMT", 2, "mtstr, nrbits",
    libGAP_FuncRandomIntegerMT, "src/gmpints.c:RandomIntegerMT" },
  
  
  { 0 }

};


/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/

static void *libGAP_allocForGmp(size_t size) {
  return mmap((void *)0, size, PROT_READ| PROT_WRITE, MAP_ANON | MAP_PRIVATE, -1, 0);
}

static void *libGAP_reallocForGmp (void *old, size_t old_size, size_t new_size) {
  void * newptr = libGAP_allocForGmp(new_size);
  size_t common_size = (new_size < old_size) ? new_size:old_size;
  memcpy(newptr, old, common_size);
  munmap(old, old_size);
  return newptr;
}

static  void libGAP_freeForGmp (void *ptr, size_t size) {
  munmap(ptr, size);
}

static libGAP_Int libGAP_InitKernel ( libGAP_StructInitInfo * libGAP_module )
{
  libGAP_UInt                t1,  t2;

  if (mp_bits_per_limb != GMP_LIMB_BITS) {
    libGAP_FPUTS_TO_STDERR("Panic, GMP limb size mismatch\n");
    libGAP_SyExit( 1 ); 
  }

   mp_set_memory_functions( libGAP_allocForGmp, libGAP_reallocForGmp, libGAP_freeForGmp); 

  /* init filters and functions                                            */
  libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
  libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );
  
  /* install the marking functions                                         */
  libGAP_InfoBags[         libGAP_T_INT    ].name = "integer";
#ifdef libGAP_SYS_IS_64_BIT
  libGAP_InfoBags[         libGAP_T_INTPOS ].name = "integer (>= 2^60)";
  libGAP_InfoBags[         libGAP_T_INTNEG ].name = "integer (< -2^60)";
#else
  libGAP_InfoBags[         libGAP_T_INTPOS ].name = "integer (>= 2^28)";
  libGAP_InfoBags[         libGAP_T_INTNEG ].name = "integer (< -2^28)";
#endif
  libGAP_InitMarkFuncBags( libGAP_T_INTPOS, libGAP_MarkNoSubBags );
  libGAP_InitMarkFuncBags( libGAP_T_INTNEG, libGAP_MarkNoSubBags );
  
  /* Install the saving methods */
  libGAP_SaveObjFuncs [ libGAP_T_INTPOS ] = libGAP_SaveInt;
  libGAP_SaveObjFuncs [ libGAP_T_INTNEG ] = libGAP_SaveInt;
  libGAP_LoadObjFuncs [ libGAP_T_INTPOS ] = libGAP_LoadInt;
  libGAP_LoadObjFuncs [ libGAP_T_INTNEG ] = libGAP_LoadInt;
  
  /* install the printing functions                                        */
  libGAP_PrintObjFuncs[ libGAP_T_INT    ] = libGAP_PrintInt;
  libGAP_PrintObjFuncs[ libGAP_T_INTPOS ] = libGAP_PrintInt;
  libGAP_PrintObjFuncs[ libGAP_T_INTNEG ] = libGAP_PrintInt;
  
  /* install the comparison methods                                        */
  for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
    for ( t2 = libGAP_T_INT; t2 <= libGAP_T_INTNEG; t2++ ) {
      libGAP_EqFuncs  [ t1 ][ t2 ] = libGAP_EqInt;
      libGAP_LtFuncs  [ t1 ][ t2 ] = libGAP_LtInt;
    }
  }
  
  /* install the unary arithmetic methods                                  */
  for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
    libGAP_ZeroFuncs[ t1 ] = libGAP_ZeroInt;
    libGAP_ZeroMutFuncs[ t1 ] = libGAP_ZeroInt;
    libGAP_AInvFuncs[ t1 ] = libGAP_AInvInt;
    libGAP_AInvMutFuncs[ t1 ] = libGAP_AInvInt;
    libGAP_OneFuncs [ t1 ] = libGAP_OneInt;
    libGAP_OneMutFuncs [ t1 ] = libGAP_OneInt;
  }    

    /* install the default product and power methods                       */
  for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
    for ( t2 = libGAP_FIRST_CONSTANT_TNUM;  t2 <= libGAP_LAST_CONSTANT_TNUM;  t2++ ) {
      libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdIntObj;
      libGAP_PowFuncs [ t2 ][ t1 ] = libGAP_PowObjInt;
    }
    for ( t2 = libGAP_FIRST_RECORD_TNUM;  t2 <= libGAP_LAST_RECORD_TNUM;  t2++ ) {
      libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdIntObj;
      libGAP_PowFuncs [ t2 ][ t1 ] = libGAP_PowObjInt;
    }
    for ( t2 = libGAP_FIRST_LIST_TNUM;    t2 <= libGAP_LAST_LIST_TNUM;    t2++ ) {
      libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdIntObj;
      libGAP_PowFuncs [ t2 ][ t1 ] = libGAP_PowObjInt;
    }
  }

  /* install the binary arithmetic methods                                 */
  for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
    for ( t2 = libGAP_T_INT; t2 <= libGAP_T_INTNEG; t2++ ) {
      libGAP_EqFuncs  [ t1 ][ t2 ] = libGAP_EqInt;
      libGAP_LtFuncs  [ t1 ][ t2 ] = libGAP_LtInt;
      libGAP_SumFuncs [ t1 ][ t2 ] = libGAP_SumInt;
      libGAP_DiffFuncs[ t1 ][ t2 ] = libGAP_DiffInt;
      libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdInt;
      libGAP_PowFuncs [ t1 ][ t2 ] = libGAP_PowInt;
      libGAP_ModFuncs [ t1 ][ t2 ] = libGAP_ModInt;
    }
  }

  /* gvars to import from the library                                      */
  libGAP_ImportGVarFromLibrary( "TYPE_INT_SMALL_ZERO", &libGAP_TYPE_INT_SMALL_ZERO );
  libGAP_ImportGVarFromLibrary( "TYPE_INT_SMALL_POS",  &libGAP_TYPE_INT_SMALL_POS );
  libGAP_ImportGVarFromLibrary( "TYPE_INT_SMALL_NEG",  &libGAP_TYPE_INT_SMALL_NEG );
  libGAP_ImportGVarFromLibrary( "TYPE_INT_LARGE_POS", &libGAP_TYPE_INT_LARGE_POS );
  libGAP_ImportGVarFromLibrary( "TYPE_INT_LARGE_NEG", &libGAP_TYPE_INT_LARGE_NEG );
  libGAP_ImportGVarFromLibrary( "SMALLEST_INTPOS", &libGAP_SMALLEST_INTPOS );

  libGAP_ImportFuncFromLibrary( "String", &libGAP_String );
  libGAP_ImportFuncFromLibrary( "One", &libGAP_OneAttr);

  /* install the kind functions                                          */
  libGAP_TypeObjFuncs[ libGAP_T_INT    ] = libGAP_TypeIntSmall;
  libGAP_TypeObjFuncs[ libGAP_T_INTPOS ] = libGAP_TypeIntLargePos;
  libGAP_TypeObjFuncs[ libGAP_T_INTNEG ] = libGAP_TypeIntLargeNeg;

  /* return success                                                        */
  return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary ( libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_UInt gvar; 

  /* init filters and functions                                            */
  libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
  libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );
  
  libGAP_SMALLEST_INTPOS = libGAP_NewBag( libGAP_T_INTPOS, sizeof(libGAP_TypLimb) );
  libGAP_SET_VAL_LIMB0(libGAP_SMALLEST_INTPOS, (1L<<libGAP_NR_SMALL_INT_BITS));
  gvar = libGAP_GVarName("SMALLEST_INTPOS");
  libGAP_MakeReadWriteGVar( gvar );
  libGAP_AssGVar( gvar, libGAP_SMALLEST_INTPOS );
  libGAP_MakeReadOnlyGVar(gvar);
  
  /* return success                                                        */
  return 0;
}


/****************************************************************************
**
*F  InitInfoInt() . . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
  libGAP_MODULE_BUILTIN,                        /* type                           */
  "gmpints",                             /* name                           */
  0,                                     /* revision entry of c file       */
  0,                                     /* revision entry of h file       */
  0,                                     /* version                        */
  0,                                     /* crc                            */
  libGAP_InitKernel,                            /* initKernel                     */
  libGAP_InitLibrary,                           /* initLibrary                    */
  0,                                     /* checkInit                      */
  0,                                     /* preSave                        */
  0,                                     /* postSave                       */
  0                                      /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoInt ( void )
{
  libGAP_FillInVersion( &libGAP_module );
  return &libGAP_module;
}

/* corresponds to USE_GMP test at start */
#endif

/****************************************************************************
**
*E  gmpints.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
