/****************************************************************************
**
*W  funcs.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the function interpreter package.
**
**  The function interpreter package   contains the executors  for  procedure
**  calls, the  evaluators  for function calls,  the   evaluator for function
**  expressions, and the handlers for the execution of function bodies.
**
**  It uses the function call mechanism defined by the calls package.
*/
#include        <stdio.h>               /* on SunOS, assert.h uses stderr
                                           but does not include stdio.h    */
#include        <assert.h>              /* assert                          */
#include        "system.h"              /* Ints, UInts                     */
#include        "bool.h"


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "string.h"              /* strings                         */
#include        "calls.h"               /* generic call mechanism          */

#include        "code.h"                /* coder                           */
#include        "vars.h"                /* variables                       */
#include        "exprs.h"               /* expressions                     */
#include        "stats.h"               /* statements                      */

#include        "funcs.h"               /* functions                       */

#include        "read.h"                /* read expressions                */
#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */


#include        "saveload.h"            /* saving and loading              */

#include        "opers.h"               /* generic operations              */
#include        "gvars.h"

/****************************************************************************
**
*F ExecProccallOpts( <call> ). . execute a procedure call with options
**
** Calls with options are wrapped in an outer statement, which is
** handled here
*/

static libGAP_Obj libGAP_PushOptions;
static libGAP_Obj libGAP_PopOptions;

libGAP_UInt libGAP_ExecProccallOpts(
    libGAP_Stat                call )
{
  libGAP_Obj opts;
  
  libGAP_SET_BRK_CURR_STAT( call );
  opts = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(call)[0] );
  libGAP_CALL_1ARGS(libGAP_PushOptions, opts);

  libGAP_EXEC_STAT( libGAP_ADDR_STAT( call )[1]);

  libGAP_CALL_0ARGS(libGAP_PopOptions);
  
  return 0;
}


/****************************************************************************
**
*F  ExecProccall0args(<call>)  .  execute a procedure call with 0    arguments
*F  ExecProccall1args(<call>)  .  execute a procedure call with 1    arguments
*F  ExecProccall2args(<call>)  .  execute a procedure call with 2    arguments
*F  ExecProccall3args(<call>)  .  execute a procedure call with 3    arguments
*F  ExecProccall4args(<call>)  .  execute a procedure call with 4    arguments
*F  ExecProccall5args(<call>)  .  execute a procedure call with 5    arguments
*F  ExecProccall6args(<call>)  .  execute a procedure call with 6    arguments
*F  ExecProccallXargs(<call>)  .  execute a procedure call with more arguments
**
**  'ExecProccall<i>args'  executes  a  procedure   call   to the    function
**  'FUNC_CALL(<call>)'   with   the   arguments   'ARGI_CALL(<call>,1)'   to
**  'ARGI_CALL(<call>,<i>)'.  It returns the value returned by the function.
*/

static libGAP_Obj libGAP_DispatchFuncCall( libGAP_Obj func, libGAP_Int nargs, libGAP_Obj arg1, libGAP_Obj arg2, libGAP_Obj arg3, libGAP_Obj arg4, libGAP_Obj arg5, libGAP_Obj arg6)
{ 
  libGAP_Obj arglist;
  if (nargs != -1) {
    arglist = libGAP_NEW_PLIST(libGAP_T_PLIST_DENSE, nargs);
    libGAP_SET_LEN_PLIST(arglist, nargs);
    switch(nargs) {
    case 6: 
      libGAP_SET_ELM_PLIST(arglist,6, arg6);
    case 5:
      libGAP_SET_ELM_PLIST(arglist,5, arg5);
    case 4: 
      libGAP_SET_ELM_PLIST(arglist,4, arg4);
    case 3:
      libGAP_SET_ELM_PLIST(arglist,3, arg3);
    case 2: 
      libGAP_SET_ELM_PLIST(arglist,2, arg2);
    case 1:
      libGAP_SET_ELM_PLIST(arglist,1, arg1);
    case 0:
      libGAP_CHANGED_BAG(arglist);
    }
  } else {
    arglist = arg1;
  }
  return libGAP_DoOperation2Args(libGAP_CallFuncListOper, func, arglist);
}


libGAP_UInt            libGAP_ExecProccall0args (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );

    /* call the function                                                   */
    libGAP_SET_BRK_CALL_TO( call );
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION)
      libGAP_DispatchFuncCall(func, 0, (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L);
    else {
      libGAP_CALL_0ARGS( func );
    }
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecProccall1args (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
  
    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
 
    /* call the function                                                   */
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION)
      libGAP_DispatchFuncCall(func, 1, (libGAP_Obj) arg1,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L);
    else {
      libGAP_SET_BRK_CALL_TO( call );
      libGAP_CALL_1ARGS( func, arg1 );
    } 
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecProccall2args (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
 
    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );

    /* call the function                                                   */
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION)
      libGAP_DispatchFuncCall(func, 2, (libGAP_Obj) arg1,  (libGAP_Obj) arg2,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L);
    else {
      libGAP_SET_BRK_CALL_TO( call );
      libGAP_CALL_2ARGS( func, arg1, arg2 );
    }
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecProccall3args (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
 
    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );

    /* call the function                                                   */
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION)
      libGAP_DispatchFuncCall(func, 3, (libGAP_Obj) arg1,  (libGAP_Obj) arg2,  (libGAP_Obj) arg3,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L,  (libGAP_Obj) 0L);
    else {
      libGAP_SET_BRK_CALL_TO( call );
      libGAP_CALL_3ARGS( func, arg1, arg2, arg3 );
    }
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecProccall4args (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */
    libGAP_Obj                 arg4;           /* fourth argument                 */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
 
    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );
    arg4 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 4 ) );

    /* call the function                                                   */
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION)
      libGAP_DispatchFuncCall(func, 4, (libGAP_Obj) arg1,  (libGAP_Obj) arg2,  (libGAP_Obj) arg3,  (libGAP_Obj) arg4,  (libGAP_Obj) 0,  (libGAP_Obj) 0);
    else {
      libGAP_SET_BRK_CALL_TO( call );
      libGAP_CALL_4ARGS( func, arg1, arg2, arg3, arg4 );
    }
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecProccall5args (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */
    libGAP_Obj                 arg4;           /* fourth argument                 */
    libGAP_Obj                 arg5;           /* fifth  argument                 */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
    while ( libGAP_TNUM_OBJ( func ) != libGAP_T_FUNCTION ) {
        func = libGAP_ErrorReturnObj(
            "Function Calls: <func> must be a function (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(func), 0L,
            "you can replace <func> via 'return <func>;'" );
    }

    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );
    arg4 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 4 ) );
    arg5 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 5 ) );

    /* call the function                                                   */
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION)
      libGAP_DispatchFuncCall(func, 5, (libGAP_Obj) arg1,  (libGAP_Obj) arg2,  (libGAP_Obj) arg3,  (libGAP_Obj) arg4,  (libGAP_Obj) arg5,  (libGAP_Obj) 0L);
    else {
      libGAP_SET_BRK_CALL_TO( call );
      libGAP_CALL_5ARGS( func, arg1, arg2, arg3, arg4, arg5 );
    }
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecProccall6args (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */
    libGAP_Obj                 arg4;           /* fourth argument                 */
    libGAP_Obj                 arg5;           /* fifth  argument                 */
    libGAP_Obj                 arg6;           /* sixth  argument                 */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
 
    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );
    arg4 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 4 ) );
    arg5 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 5 ) );
    arg6 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 6 ) );

    /* call the function                                                   */
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION)
      libGAP_DispatchFuncCall(func, 6, (libGAP_Obj) arg1,  (libGAP_Obj) arg2,  (libGAP_Obj) arg3,  (libGAP_Obj) arg4,  (libGAP_Obj) arg5,  (libGAP_Obj) arg6);
    else {
      libGAP_SET_BRK_CALL_TO( call );
      libGAP_CALL_6ARGS( func, arg1, arg2, arg3, arg4, arg5, arg6 );
    }
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

libGAP_UInt            libGAP_ExecProccallXargs (
    libGAP_Stat                call )
{
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 args;           /* argument list                   */
    libGAP_Obj                 argi;           /* <i>-th argument                 */
    libGAP_UInt                i;              /* loop variable                   */

    /* evaluate the function                                               */
    libGAP_SET_BRK_CURR_STAT( call );
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
 

    /* evaluate the arguments                                              */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_NARG_SIZE_CALL(libGAP_SIZE_STAT(call)) );
    libGAP_SET_LEN_PLIST( args, libGAP_NARG_SIZE_CALL(libGAP_SIZE_STAT(call)) );
    for ( i = 1; i <= libGAP_NARG_SIZE_CALL(libGAP_SIZE_STAT(call)); i++ ) {
        argi = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, i ) );
        libGAP_SET_ELM_PLIST( args, i, argi );
        libGAP_CHANGED_BAG( args );
    }

    /* call the function                                                   */
    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      libGAP_DoOperation2Args(libGAP_CallFuncListOper, func, args);
    } else {
      libGAP_SET_BRK_CALL_TO( call );
      libGAP_CALL_XARGS( func, args );
    }

    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    /* return 0 (to indicate that no leave-statement was executed)         */
    return 0;
}

/****************************************************************************
**
*F EvalFunccallOpts( <call> ). . evaluate a function call with options
**
** Calls with options are wrapped in an outer statement, which is
** handled here
*/

libGAP_Obj libGAP_EvalFunccallOpts(
    libGAP_Expr                call )
{
  libGAP_Obj opts;
  libGAP_Obj res;
  
  
  opts = libGAP_EVAL_EXPR( libGAP_ADDR_STAT(call)[0] );
  libGAP_CALL_1ARGS(libGAP_PushOptions, opts);

  res = libGAP_EVAL_EXPR( libGAP_ADDR_STAT( call )[1]);

  libGAP_CALL_0ARGS(libGAP_PopOptions);
  
  return res;
}


/****************************************************************************
**
*F  EvalFunccall0args(<call>)  . . execute a function call with 0    arguments
*F  EvalFunccall1args(<call>)  . . execute a function call with 1    arguments
*F  EvalFunccall2args(<call>)  . . execute a function call with 2    arguments
*F  EvalFunccall3args(<call>)  . . execute a function call with 3    arguments
*F  EvalFunccall4args(<call>)  . . execute a function call with 4    arguments
*F  EvalFunccall5args(<call>)  . . execute a function call with 5    arguments
*F  EvalFunccall6args(<call>)  . . execute a function call with 6    arguments
*F  EvalFunccallXargs(<call>)  . . execute a function call with more arguments
**
**  'EvalFunccall<i>args'  executes  a     function call   to   the  function
**  'FUNC_CALL(<call>)'    with  the   arguments    'ARGI_CALL(<call>,1)'  to
**  'ARGI_CALL(<call>,<i>)'.  It returns the value returned by the function.
*/

libGAP_Obj             libGAP_EvalFunccall0args (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_0ARGS( func );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}

libGAP_Obj             libGAP_EvalFunccall1args (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
      /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, 1, (libGAP_Obj) arg1, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_1ARGS( func, arg1 );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}

libGAP_Obj             libGAP_EvalFunccall2args (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );

    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, 2, (libGAP_Obj) arg1, (libGAP_Obj) arg2, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_2ARGS( func, arg1, arg2 );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}

libGAP_Obj             libGAP_EvalFunccall3args (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );

    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, 1, (libGAP_Obj) arg1, (libGAP_Obj) arg2, (libGAP_Obj) arg3, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_3ARGS( func, arg1, arg2, arg3 );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}

libGAP_Obj             libGAP_EvalFunccall4args (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */
    libGAP_Obj                 arg4;           /* fourth argument                 */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );
    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );
    arg4 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 4 ) );

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, 4, (libGAP_Obj) arg1, (libGAP_Obj) arg2, (libGAP_Obj) arg3, (libGAP_Obj) arg4, (libGAP_Obj) 0, (libGAP_Obj) 0 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_4ARGS( func, arg1, arg2, arg3, arg4 );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}

libGAP_Obj             libGAP_EvalFunccall5args (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */
    libGAP_Obj                 arg4;           /* fourth argument                 */
    libGAP_Obj                 arg5;           /* fifth  argument                 */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );

    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );
    arg4 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 4 ) );
    arg5 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 5 ) );

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, 5, (libGAP_Obj) arg1, (libGAP_Obj) arg2, (libGAP_Obj) arg3, (libGAP_Obj) arg4, (libGAP_Obj) arg5, (libGAP_Obj) 0 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_5ARGS( func, arg1, arg2, arg3, arg4, arg5 );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}

libGAP_Obj             libGAP_EvalFunccall6args (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 arg1;           /* first  argument                 */
    libGAP_Obj                 arg2;           /* second argument                 */
    libGAP_Obj                 arg3;           /* third  argument                 */
    libGAP_Obj                 arg4;           /* fourth argument                 */
    libGAP_Obj                 arg5;           /* fifth  argument                 */
    libGAP_Obj                 arg6;           /* sixth  argument                 */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );

    /* evaluate the arguments                                              */
    arg1 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 1 ) );
    arg2 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 2 ) );
    arg3 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 3 ) );
    arg4 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 4 ) );
    arg5 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 5 ) );
    arg6 = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, 6 ) );

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, 6, (libGAP_Obj) arg1, (libGAP_Obj) arg2, (libGAP_Obj) arg3, (libGAP_Obj) arg4, (libGAP_Obj) arg5, (libGAP_Obj) arg6 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_6ARGS( func, arg1, arg2, arg3, arg4, arg5, arg6 );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}

libGAP_Obj             libGAP_EvalFunccallXargs (
    libGAP_Expr                call )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 args;           /* argument list                   */
    libGAP_Obj                 argi;           /* <i>-th argument                 */
    libGAP_UInt                i;              /* loop variable                   */

    /* evaluate the function                                               */
    func = libGAP_EVAL_EXPR( libGAP_FUNC_CALL( call ) );

    /* evaluate the arguments                                              */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_NARG_SIZE_CALL(libGAP_SIZE_EXPR(call)) );
    libGAP_SET_LEN_PLIST( args, libGAP_NARG_SIZE_CALL(libGAP_SIZE_EXPR(call)) );
    for ( i = 1; i <= libGAP_NARG_SIZE_CALL(libGAP_SIZE_EXPR(call)); i++ ) {
        argi = libGAP_EVAL_EXPR( libGAP_ARGI_CALL( call, i ) );
        libGAP_SET_ELM_PLIST( args, i, argi );
        libGAP_CHANGED_BAG( args );
    }

    if (libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION) {
      return libGAP_DispatchFuncCall(func, -1, (libGAP_Obj) args, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0, (libGAP_Obj) 0 );
    }

    /* call the function and return the result                             */
    libGAP_SET_BRK_CALL_TO( call );
    result = libGAP_CALL_XARGS( func, args );
    if (libGAP_UserHasQuit || libGAP_UserHasQUIT) /* the procedure must have called
                                       READ() and the user quit from a break
                                       loop inside it */
      libGAP_ReadEvalError();
    while ( result == 0 ) {
        result = libGAP_ErrorReturnObj(
            "Function Calls: <func> must return a value",
            0L, 0L,
            "you can supply one by 'return <value>;'" );
    }
    return result;
}


/****************************************************************************
**
*F  DoExecFunc0args(<func>)  . . . .  interpret a function with 0    arguments
*F  DoExecFunc1args(<func>,<arg1>) .  interpret a function with 1    arguments
*F  DoExecFunc2args(<func>,<arg1>...) interpret a function with 2    arguments
*F  DoExecFunc3args(<func>,<arg1>...) interpret a function with 3    arguments
*F  DoExecFunc4args(<func>,<arg1>...) interpret a function with 4    arguments
*F  DoExecFunc5args(<func>,<arg1>...) interpret a function with 5    arguments
*F  DoExecFunc6args(<func>,<arg1>...) interpret a function with 6    arguments
*F  DoExecFuncXargs(<func>,<args>) .  interpret a function with more arguments
**
**  'DoExecFunc<i>args' interprets   the function  <func>  that  expects  <i>
**  arguments with the <i> actual argument <arg1>, <arg2>, and so on.  If the
**  function expects more than 4 arguments the actual arguments are passed in
**  the plain list <args>.
**
**  'DoExecFunc<i>args'  is the  handler  for interpreted functions expecting
**  <i> arguments.
**
**  'DoExecFunc<i>args' first switches  to a new  values bag.  Then it enters
**  the arguments <arg1>, <arg2>, and so on in this new  values bag.  Then it
**  executes  the function body.   After  that it  switches back  to  the old
**  values bag.  Finally it returns the result from 'ReturnObjStat'.
**
**  Note that these functions are never called directly, they are only called
**  through the function call mechanism.
**
**  The following functions implement the recursion depth control.
**
*/

libGAP_Int libGAP_RecursionDepth;
static libGAP_UInt libGAP_RecursionTrapInterval;

static void libGAP_RecursionDepthTrap( void )
{
    libGAP_Int recursionDepth;
    /* in interactive work the RecursionDepth could become slightly negative
     * when quit-ting a higher level brk-loop to a lower level one.
     * Therefore we don't do anything if  RecursionDepth <= 0
    */
    if (libGAP_RecursionDepth > 0) {
        recursionDepth = libGAP_RecursionDepth;
        libGAP_RecursionDepth = 0;
        libGAP_ErrorReturnVoid( "recursion depth trap (%d)\n",         
                         (libGAP_Int)recursionDepth, 0L,               
                         "you may 'return;'" );
        libGAP_RecursionDepth = recursionDepth;
    }
}
     
static inline void libGAP_CheckRecursionBefore( void )
{
    libGAP_RecursionDepth++;                                           
    if ( libGAP_RecursionTrapInterval &&                                
         0 == (libGAP_RecursionDepth % libGAP_RecursionTrapInterval) )
      libGAP_RecursionDepthTrap();
}


 libGAP_Obj libGAP_STEVES_TRACING;

#define libGAP_CHECK_RECURSION_BEFORE libGAP_CheckRecursionBefore();


#define libGAP_CHECK_RECURSION_AFTER     libGAP_RecursionDepth--;       


libGAP_Obj libGAP_DoExecFunc0args (
    libGAP_Obj                 func )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */

    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */

    libGAP_CHECK_RECURSION_BEFORE

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, 0, libGAP_NLOC_FUNC(func), oldLvars );

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* Switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}

libGAP_Obj             libGAP_DoExecFunc1args (
    libGAP_Obj                 func,
    libGAP_Obj                 arg1 )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */
    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */

    libGAP_CHECK_RECURSION_BEFORE

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, 1, libGAP_NLOC_FUNC(func), oldLvars );

    /* enter the arguments                                                 */
    libGAP_ASS_LVAR( 1, arg1 );

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}

libGAP_Obj             libGAP_DoExecFunc2args (
    libGAP_Obj                 func,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */
    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */

    libGAP_CHECK_RECURSION_BEFORE

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, 2, libGAP_NLOC_FUNC(func), oldLvars );

    /* enter the arguments                                                 */
    libGAP_ASS_LVAR( 1, arg1 );
    libGAP_ASS_LVAR( 2, arg2 );

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}

libGAP_Obj             libGAP_DoExecFunc3args (
    libGAP_Obj                 func,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */
    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */

    libGAP_CHECK_RECURSION_BEFORE

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, 3, libGAP_NLOC_FUNC(func), oldLvars );

    /* enter the arguments                                                 */
    libGAP_ASS_LVAR( 1, arg1 );
    libGAP_ASS_LVAR( 2, arg2 );
    libGAP_ASS_LVAR( 3, arg3 );

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}

libGAP_Obj             libGAP_DoExecFunc4args (
    libGAP_Obj                 func,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */
    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */

    libGAP_CHECK_RECURSION_BEFORE

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, 4, libGAP_NLOC_FUNC(func), oldLvars );

    /* enter the arguments                                                 */
    libGAP_ASS_LVAR( 1, arg1 );
    libGAP_ASS_LVAR( 2, arg2 );
    libGAP_ASS_LVAR( 3, arg3 );
    libGAP_ASS_LVAR( 4, arg4 );

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}

libGAP_Obj             libGAP_DoExecFunc5args (
    libGAP_Obj                 func,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */
    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */

    libGAP_CHECK_RECURSION_BEFORE

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, 5, libGAP_NLOC_FUNC(func), oldLvars );

    /* enter the arguments                                                 */
    libGAP_ASS_LVAR( 1, arg1 );
    libGAP_ASS_LVAR( 2, arg2 );
    libGAP_ASS_LVAR( 3, arg3 );
    libGAP_ASS_LVAR( 4, arg4 );
    libGAP_ASS_LVAR( 5, arg5 );

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}

libGAP_Obj             libGAP_DoExecFunc6args (
    libGAP_Obj                 func,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */
    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */

    libGAP_CHECK_RECURSION_BEFORE

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, 6, libGAP_NLOC_FUNC(func), oldLvars );

    /* enter the arguments                                                 */
    libGAP_ASS_LVAR( 1, arg1 );
    libGAP_ASS_LVAR( 2, arg2 );
    libGAP_ASS_LVAR( 3, arg3 );
    libGAP_ASS_LVAR( 4, arg4 );
    libGAP_ASS_LVAR( 5, arg5 );
    libGAP_ASS_LVAR( 6, arg6 );

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}

libGAP_Obj             libGAP_DoExecFuncXargs (
    libGAP_Obj                 func,
    libGAP_Obj                 args )
{
    libGAP_Bag                 oldLvars;       /* old values bag                  */
    libGAP_OLD_BRK_CURR_STAT                   /* old executing statement         */
    libGAP_UInt                len;            /* number of arguments             */
    libGAP_UInt                i;              /* loop variable                   */

    libGAP_CHECK_RECURSION_BEFORE

    /* check the number of arguments                                       */
    len = libGAP_NARG_FUNC( func );
    while ( len != libGAP_LEN_PLIST( args ) ) {
        args = libGAP_ErrorReturnObj(
            "Function Calls: number of arguments must be %d (not %d)",
            len, libGAP_LEN_PLIST( args ),
            "you can replace the <list> of arguments via 'return <list>;'" );
        libGAP_PLAIN_LIST( args );
    }

    /* switch to a new values bag                                          */
    libGAP_SWITCH_TO_NEW_LVARS( func, len, libGAP_NLOC_FUNC(func), oldLvars );

    /* enter the arguments                                                 */
    for ( i = 1; i <= len; i++ ) {
        libGAP_ASS_LVAR( i, libGAP_ELM_PLIST( args, i ) );
    }

    /* execute the statement sequence                                      */
    libGAP_REM_BRK_CURR_STAT();
    libGAP_EXEC_STAT( libGAP_FIRST_STAT_CURR_FUNC );
    libGAP_RES_BRK_CURR_STAT();

   /* remove the link to the calling function, in case this values bag
       stays alive due to higher variable reference */
    libGAP_SET_BRK_CALL_FROM( ((libGAP_Obj) 0));

    /* switch back to the old values bag                                   */
    libGAP_SWITCH_TO_OLD_LVARS( oldLvars );

    libGAP_CHECK_RECURSION_AFTER

    /* return the result                                                   */
      {
        libGAP_Obj                 returnObjStat;
        returnObjStat = libGAP_ReturnObjStat;
        libGAP_ReturnObjStat = (libGAP_Obj)0;
        return returnObjStat;
      }
}


/****************************************************************************
**
*F  MakeFunction(<fexp>)  . . . . . . . . . . . . . . . . . . make a function
**
**  'MakeFunction' makes a function from the function expression bag <fexp>.
*/
libGAP_Obj             libGAP_MakeFunction (
    libGAP_Obj                 fexp )
{
    libGAP_Obj                 func;           /* function, result                */
    libGAP_ObjFunc             hdlr;           /* handler                         */

    /* select the right handler                                            */
    if      ( libGAP_NARG_FUNC(fexp) ==  0 )  hdlr = libGAP_DoExecFunc0args;
    else if ( libGAP_NARG_FUNC(fexp) ==  1 )  hdlr = libGAP_DoExecFunc1args;
    else if ( libGAP_NARG_FUNC(fexp) ==  2 )  hdlr = libGAP_DoExecFunc2args;
    else if ( libGAP_NARG_FUNC(fexp) ==  3 )  hdlr = libGAP_DoExecFunc3args;
    else if ( libGAP_NARG_FUNC(fexp) ==  4 )  hdlr = libGAP_DoExecFunc4args;
    else if ( libGAP_NARG_FUNC(fexp) ==  5 )  hdlr = libGAP_DoExecFunc5args;
    else if ( libGAP_NARG_FUNC(fexp) ==  6 )  hdlr = libGAP_DoExecFunc6args;
    else if ( libGAP_NARG_FUNC(fexp) >=  7 )  hdlr = libGAP_DoExecFuncXargs;
    else   /* NARG_FUNC(fexp) == -1 */ hdlr = libGAP_DoExecFunc1args;

    /* make the function                                                   */
    func = libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_FUNC,
                         libGAP_NAME_FUNC( fexp ),
                         libGAP_NARG_FUNC( fexp ), libGAP_NAMS_FUNC( fexp ),
                         hdlr );

    /* install the things an interpreted function needs                    */
    libGAP_NLOC_FUNC( func ) = libGAP_NLOC_FUNC( fexp );
    libGAP_BODY_FUNC( func ) = libGAP_BODY_FUNC( fexp );
    libGAP_ENVI_FUNC( func ) = libGAP_CurrLVars;
    /* the 'CHANGED_BAG(CurrLVars)' is needed because it is delayed        */
    libGAP_CHANGED_BAG( libGAP_CurrLVars );
    libGAP_FEXS_FUNC( func ) = libGAP_FEXS_FUNC( fexp );

    /* return the function                                                 */
    return func;
}


/****************************************************************************
**
*F  EvalFuncExpr(<expr>)  . . .  evaluate a function expression to a function
**
**  'EvalFuncExpr' evaluates the function expression <expr> to a function.
*/
libGAP_Obj             libGAP_EvalFuncExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 fexs;           /* func. expr. list of curr. func. */
    libGAP_Obj                 fexp;           /* function expression bag         */

    /* get the function expression bag                                     */
    fexs = libGAP_FEXS_FUNC( libGAP_CURR_FUNC );
    fexp = libGAP_ELM_PLIST( fexs, (libGAP_Int)(libGAP_ADDR_EXPR(expr)[0]) );

    /* and make the function                                               */
    return libGAP_MakeFunction( fexp );
}


/****************************************************************************
**
*F  PrintFuncExpr(<expr>) . . . . . . . . . . . . print a function expression
**
**  'PrintFuncExpr' prints a function expression.
*/
void            libGAP_PrintFuncExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 fexs;           /* func. expr. list of curr. func. */
    libGAP_Obj                 fexp;           /* function expression bag         */

    /* get the function expression bag                                     */
    fexs = libGAP_FEXS_FUNC( libGAP_CURR_FUNC );
    fexp = libGAP_ELM_PLIST( fexs, (libGAP_Int)(libGAP_ADDR_EXPR(expr)[0]) );
    libGAP_PrintFunction( fexp );
    /* Pr("function ... end",0L,0L); */
}


/****************************************************************************
**
*F  PrintProccall(<call>) . . . . . . . . . . . . . .  print a procedure call
**
**  'PrintProccall' prints a procedure call.
*/
extern  void            libGAP_PrintFunccall (
            libGAP_Expr                call );

extern  void            libGAP_PrintFunccallOpts (
            libGAP_Expr                call );

void            libGAP_PrintProccall (
    libGAP_Stat                call )
{
    libGAP_PrintFunccall( call );
    libGAP_Pr( ";", 0L, 0L );
}

void            libGAP_PrintProccallOpts (
    libGAP_Stat                call )
{
    libGAP_PrintFunccallOpts( call );
    libGAP_Pr( ";", 0L, 0L );
}


/****************************************************************************
**
*F  PrintFunccall(<call>) . . . . . . . . . . . . . . . print a function call
**
**  'PrintFunccall' prints a function call.
*/
static void            libGAP_PrintFunccall1 (
    libGAP_Expr                call )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* print the expression that should evaluate to a function             */
    libGAP_Pr("%2>",0L,0L);
    libGAP_PrintExpr( libGAP_FUNC_CALL(call) );

    /* print the opening parenthesis                                       */
    libGAP_Pr("%<( %>",0L,0L);

    /* print the expressions that evaluate to the actual arguments         */
    for ( i = 1; i <= libGAP_NARG_SIZE_CALL( libGAP_SIZE_EXPR(call) ); i++ ) {
        libGAP_PrintExpr( libGAP_ARGI_CALL(call,i) );
        if ( i != libGAP_NARG_SIZE_CALL( libGAP_SIZE_EXPR(call) ) ) {
            libGAP_Pr("%<, %>",0L,0L);
        }
    }

    return;
    
}

void            libGAP_PrintFunccall (
    libGAP_Expr                call )
{
  libGAP_PrintFunccall1( call );
  
  /* print the closing parenthesis                                       */
  libGAP_Pr(" %2<)",0L,0L);
}


void             libGAP_PrintFunccallOpts (
    libGAP_Expr                call )
{
  libGAP_PrintFunccall1( libGAP_ADDR_STAT( call )[1]);
  libGAP_Pr(" :%2> ", 0L, 0L);
  libGAP_PrintRecExpr1 ( libGAP_ADDR_STAT( call )[0]);
  libGAP_Pr(" %4<)",0L,0L);
}

  

/****************************************************************************
**
*F  ExecBegin() . . . . . . . . . . . . . . . . . . . . .  begin an execution
*F  ExecEnd(<error>)  . . . . . . . . . . . . . . . . . . .  end an execution
*/
libGAP_Obj             libGAP_ExecState;

void            libGAP_ExecBegin ( libGAP_Obj frame )
{
    libGAP_Obj                 execState;      /* old execution state             */

    /* remember the old execution state                                    */
    execState = libGAP_NewBag( libGAP_T_PLIST, 4*sizeof(libGAP_Obj) );
    libGAP_ADDR_OBJ(execState)[0] = (libGAP_Obj)3;
    libGAP_ADDR_OBJ(execState)[1] = libGAP_ExecState;
    libGAP_ADDR_OBJ(execState)[2] = libGAP_CurrLVars;
    /* the 'CHANGED_BAG(CurrLVars)' is needed because it is delayed        */
    libGAP_CHANGED_BAG( libGAP_CurrLVars );
    libGAP_ADDR_OBJ(execState)[3] = libGAP_INTOBJ_INT((libGAP_Int)libGAP_CurrStat);
    libGAP_ExecState = execState;

    /* set up new state                                                    */
    libGAP_SWITCH_TO_OLD_LVARS( frame );
    libGAP_SET_BRK_CURR_STAT( 0 );
}

void            libGAP_ExecEnd (
    libGAP_UInt                error )
{
    /* if everything went fine                                             */
    if ( ! error ) {

        /* the state must be primal again                                  */
        assert( libGAP_CurrStat  == 0 );

        /* switch back to the old state                                    */
        libGAP_SET_BRK_CURR_STAT( (libGAP_Stat)libGAP_INT_INTOBJ((libGAP_ADDR_OBJ(libGAP_ExecState)[3]) ));
        libGAP_SWITCH_TO_OLD_LVARS( libGAP_ADDR_OBJ(libGAP_ExecState)[2] );
        libGAP_ExecState = libGAP_ADDR_OBJ(libGAP_ExecState)[1];

    }

    /* otherwise clean up the mess                                         */
    else {

        /* switch back to the old state                                    */
        libGAP_SET_BRK_CURR_STAT( (libGAP_Stat)libGAP_INT_INTOBJ((libGAP_ADDR_OBJ(libGAP_ExecState)[3]) ));
        libGAP_SWITCH_TO_OLD_LVARS( libGAP_ADDR_OBJ(libGAP_ExecState)[2] );
        libGAP_ExecState = libGAP_ADDR_OBJ(libGAP_ExecState)[1];

    }
}

/****************************************************************************
**
*F  FuncSetRecursionTrapInterval( <self>, <interval> )
**
*/

libGAP_Obj libGAP_FuncSetRecursionTrapInterval( libGAP_Obj self,  libGAP_Obj interval )
{
  while (!libGAP_IS_INTOBJ(interval) || libGAP_INT_INTOBJ(interval) < 0)
    interval = libGAP_ErrorReturnObj( "SetRecursionTrapInterval( <interval> ): "
                               "<interval> must be a non-negative small integer",
                               0L, 0L, 
                               "you can replace <interval> via 'return <interval>;'");
  libGAP_RecursionTrapInterval = libGAP_INT_INTOBJ( interval);
  return 0;
}


/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "SetRecursionTrapInterval", 1, "interval",
      libGAP_FuncSetRecursionTrapInterval, "src/funcs.c:SetRecursionTrapInterval" },

    { 0 }

};

/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );


    /* return success                                                      */
    return 0;
}


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_RecursionTrapInterval = 5000;
  libGAP_InitCopyGVar("STEVES_TRACING", &libGAP_STEVES_TRACING);
  
    /* make the global variable known to Gasman                            */
    libGAP_InitGlobalBag( &libGAP_ExecState, "src/funcs.c:ExecState" );

    /* Register the handler for our exported function                      */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* Import some functions from the library                              */
    libGAP_ImportFuncFromLibrary( "PushOptions", &libGAP_PushOptions );
    libGAP_ImportFuncFromLibrary( "PopOptions",  &libGAP_PopOptions  );

    /* use short cookies to save space in saved workspace                  */
    libGAP_InitHandlerFunc( libGAP_DoExecFunc0args, "i0");
    libGAP_InitHandlerFunc( libGAP_DoExecFunc1args, "i1");
    libGAP_InitHandlerFunc( libGAP_DoExecFunc2args, "i2");
    libGAP_InitHandlerFunc( libGAP_DoExecFunc3args, "i3");
    libGAP_InitHandlerFunc( libGAP_DoExecFunc4args, "i4");
    libGAP_InitHandlerFunc( libGAP_DoExecFunc5args, "i5");
    libGAP_InitHandlerFunc( libGAP_DoExecFunc6args, "i6");
    libGAP_InitHandlerFunc( libGAP_DoExecFuncXargs, "iX");

    /* install the evaluators and executors                                */
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_0ARGS ] = libGAP_ExecProccall0args;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_1ARGS ] = libGAP_ExecProccall1args;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_2ARGS ] = libGAP_ExecProccall2args;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_3ARGS ] = libGAP_ExecProccall3args;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_4ARGS ] = libGAP_ExecProccall4args;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_5ARGS ] = libGAP_ExecProccall5args;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_6ARGS ] = libGAP_ExecProccall6args;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_XARGS ] = libGAP_ExecProccallXargs;
    libGAP_ExecStatFuncs [ libGAP_T_PROCCALL_OPTS  ] = libGAP_ExecProccallOpts;

    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_0ARGS ] = libGAP_EvalFunccall0args;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_1ARGS ] = libGAP_EvalFunccall1args;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_2ARGS ] = libGAP_EvalFunccall2args;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_3ARGS ] = libGAP_EvalFunccall3args;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_4ARGS ] = libGAP_EvalFunccall4args;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_5ARGS ] = libGAP_EvalFunccall5args;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_6ARGS ] = libGAP_EvalFunccall6args;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_XARGS ] = libGAP_EvalFunccallXargs;
    libGAP_EvalExprFuncs [ libGAP_T_FUNCCALL_OPTS  ] = libGAP_EvalFunccallOpts;
    libGAP_EvalExprFuncs [ libGAP_T_FUNC_EXPR      ] = libGAP_EvalFuncExpr;

    /* install the printers                                                */
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_0ARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_1ARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_2ARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_3ARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_4ARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_5ARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_6ARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_XARGS ] = libGAP_PrintProccall;
    libGAP_PrintStatFuncs[ libGAP_T_PROCCALL_OPTS  ] = libGAP_PrintProccallOpts;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_0ARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_1ARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_2ARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_3ARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_4ARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_5ARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_6ARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_XARGS ] = libGAP_PrintFunccall;
    libGAP_PrintExprFuncs[ libGAP_T_FUNCCALL_OPTS  ] = libGAP_PrintFunccallOpts;
    libGAP_PrintExprFuncs[ libGAP_T_FUNC_EXPR      ] = libGAP_PrintFuncExpr;

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoFuncs() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "funcs",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoFuncs ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  funcs.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/



