/****************************************************************************
**
*W  exprs.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the expressions package.
**
**  The expressions  package is the  part  of the interpreter  that evaluates
**  expressions to their values and prints expressions.
*/
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "ariths.h"              /* basic arithmetic                */
#include        "records.h"             /* generic records                 */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */
#include        "integer.h"             /* integers                        */

#include        "permutat.h"            /* permutations                    */
#include        "trans.h"               /* transformations                 */
#include        "pperm.h"               /* partial perms                   */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */
#include        "calls.h"
#include        "vars.h"                /* variables                       */
#include        "stats.h"


#include        "exprs.h"               /* expressions                     */

#include <assert.h>

/****************************************************************************
**

*F  OBJ_REFLVAR(<expr>) . . . . . . . . . . . value of a reference to a local
**
**  'OBJ_REFLVAR'  returns  the value of  the reference  to a  local variable
**  <expr>.
**
**  'OBJ_REFLVAR'  is defined  in the  declaration  part of  this  package as
**  follows
**
#ifdef  NO_LVAR_CHECKS
#define OBJ_REFLVAR(expr)       \
                        OBJ_LVAR( LVAR_REFLVAR( (expr) ) )
#endif
#ifndef NO_LVAR_CHECKS
#define OBJ_REFLVAR(expr)       \
                        (*(Obj*)(((char*)PtrLVars)+(expr)+5) != 0 ? \
                         *(Obj*)(((char*)PtrLVars)+(expr)+5) : \
                         ObjLVar( LVAR_REFLVAR( expr ) ) )
#endif
*/


/****************************************************************************
**
*F  OBJ_INTEXPR(<expr>) . . . . . . . . . . .  value of an integer expression
**
**  'OBJ_INTEXPR' returns the (immediate)  integer  value of the  (immediate)
**  integer expression <expr>.
**
**  'OBJ_INTEXPR(<expr>)'  should  be 'OBJ_INT(INT_INTEXPR(<expr>))', but for
**  performance  reasons we implement  it   as '(Obj)(<expr>)'.  This is   of
**  course    highly  dependent  on    (immediate)  integer   expressions and
**  (immediate) integer values having the same representation.
**
**  'OBJ_INTEXPR' is  defined in  the declaration  part  of  this package  as
**  follow
**
#define OBJ_INTEXPR(expr)       \
                        ((Obj)(Int)(Int4)(expr))
*/


/****************************************************************************
**
*F  EVAL_EXPR(<expr>) . . . . . . . . . . . . . . . .  evaluate an expression
**
**  'EVAL_EXPR' evaluates the expression <expr>.
**
**  'EVAL_EXPR' returns the value of <expr>.
**
**  'EVAL_EXPR'  causes  the   evaluation of   <expr> by  dispatching  to the
**  evaluator, i.e., to  the function that evaluates  expressions of the type
**  of <expr>.
**
**  Note that 'EVAL_EXPR' does not use 'TNUM_EXPR', since it also handles the
**  two special cases that 'TNUM_EXPR' handles.
**
**  'EVAL_EXPR' is defined in the declaration part of this package as follows:
**
#define EVAL_EXPR(expr) \
                        (IS_REFLVAR(expr) ? OBJ_REFLVAR(expr) : \
                         (IS_INTEXPR(expr) ? OBJ_INTEXPR(expr) : \
                          (*EvalExprFuncs[ TNUM_STAT(expr) ])( expr ) ))
*/


/****************************************************************************
**
*V  EvalExprFuncs[<type>]  . . . . . evaluator for expressions of type <type>
**
**  'EvalExprFuncs'  is the dispatch table   that contains for  every type of
**  expressions a pointer  to the  evaluator  for expressions of this   type,
**  i.e., the function that should be  called to evaluate expressions of this
**  type.
*/
libGAP_Obj             (* libGAP_EvalExprFuncs [256]) ( libGAP_Expr expr );


/****************************************************************************
**
*F  EVAL_BOOL_EXPR(<expr>)  . . . . evaluate an expression to a boolean value
**
**  'EVAL_BOOL_EXPR' evaluates   the expression  <expr> and  checks  that the
**  value is either  'true' or 'false'.  If the  expression does not evaluate
**  to 'true' or 'false', then an error is signalled.
**
**  'EVAL_BOOL_EXPR' returns the  value of <expr> (which  is either 'true' or
**  'false').
**
**  'EVAL_BOOL_EXPR' is defined  in the declaration part  of this package  as
**  follows
**
#define EVAL_BOOL_EXPR(expr) \
                        ( (*EvalBoolFuncs[ TNUM_EXPR( expr ) ])( expr ) )
*/


/****************************************************************************
**
*V  EvalBoolFuncs[<type>] . . boolean evaluator for expression of type <type>
**
**  'EvalBoolFuncs'  is  the dispatch table that  contains  for every type of
**  expression a pointer to a boolean evaluator for expressions of this type,
**  i.e., a pointer to  a function which  is  guaranteed to return a  boolean
**  value that should be called to evaluate expressions of this type.
*/
libGAP_Obj             (* libGAP_EvalBoolFuncs [256]) ( libGAP_Expr expr );


/****************************************************************************
**
*F  EvalUnknownExpr(<expr>) . . . . . . . evaluate expression of unknown type
**
**  'EvalUnknownExpr' is the evaluator that  is called if  an attempt is made
**  to  evaluate an  expression  <expr> of  an  unknown type.   It signals an
**  error.  If this is ever called, then  GAP is in  serious trouble, such as
**  an overwritten type field of an expression.
*/
libGAP_Obj             libGAP_EvalUnknownExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr( "Panic: tried to evaluate an expression of unknown type '%d'\n",
        (libGAP_Int)libGAP_TNUM_EXPR(expr), 0L );
    return 0;
}


/****************************************************************************
**
*F  EvalUnknownBool(<expr>) . . . . boolean evaluator for general expressions
**
**  'EvalUnknownBool' evaluates   the expression <expr>  (using 'EVAL_EXPR'),
**  and checks that the value is either 'true' or 'false'.  If the expression
**  does not evaluate to 'true' or 'false', then an error is signalled.
**
**  This is the default function in 'EvalBoolFuncs' used for expressions that
**  are   not a priori    known  to evaluate  to a    boolean value  (such as
**  function calls).
*/
libGAP_Obj             libGAP_EvalUnknownBool (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */

    /* evaluate the expression                                             */
    val = libGAP_EVAL_EXPR( expr );

    /* check that the value is either 'true' or 'false'                    */
    while ( val != libGAP_True && val != libGAP_False ) {
        val = libGAP_ErrorReturnObj(
            "<expr> must be 'true' or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
            "you can replace <expr> via 'return <expr>;'" );
    }

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalOr(<expr>)  . . . . . . . . . . . . . evaluate a boolean or operation
**
**  'EvalOr' evaluates the or-expression <expr> and  returns its value, i.e.,
**  'true'  if  either of  the operands  is  'true',  and 'false'  otherwise.
**  'EvalOr'  is   called from  'EVAL_EXPR' to  evaluate  expressions of type
**  'T_OR'.
**
**  If '<expr>.left'  is   already  'true' 'EvalOr'  returns  'true'  without
**  evaluating '<expr>.right'.  This allows constructs like
**
**      if (index > max) or (list[index] = 0)  then ... fi;
*/
libGAP_Obj             libGAP_EvalOr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 opL;            /* evaluated left operand          */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* evaluate and test the left operand                                  */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_BOOL_EXPR( tmp );
    if ( opL != libGAP_False ) {
        return libGAP_True;
    }

    /* evaluate and test the right operand                                 */
    tmp = libGAP_ADDR_EXPR(expr)[1];
    return libGAP_EVAL_BOOL_EXPR( tmp );
}


/****************************************************************************
**
*F  EvalAnd(<expr>) . . . . . . . . . . . .  evaluate a boolean and operation
**
**  'EvalAnd'  evaluates  the and-expression <expr>   and  returns its value,
**  i.e.,   'true'  if both  operands  are   'true',  and  'false' otherwise.
**  'EvalAnd' is called from   'EVAL_EXPR' to  evaluate expressions  of  type
**  'T_AND'.
**
**  If '<expr>.left' is  already  'false' 'EvalAnd' returns 'false'   without
**  evaluating '<expr>.right'.  This allows constructs like
**
**      if (index <= max) and (list[index] = 0)  then ... fi;
*/
extern  libGAP_Obj             libGAP_NewAndFilter (
            libGAP_Obj                     oper1,
            libGAP_Obj                     oper2 );

libGAP_Obj             libGAP_EvalAnd (
    libGAP_Expr                expr )
{
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* if the left operand is 'false', this is the result                  */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    if      ( opL == libGAP_False ) {
        return opL;
    }

    /* if the left operand is 'true', the result is the right operand      */
    else if ( opL == libGAP_True  ) {
        tmp = libGAP_ADDR_EXPR(expr)[1];
        return libGAP_EVAL_BOOL_EXPR( tmp );
    }

    /* handle the 'and' of two filters                                    */
    else if ( libGAP_TNUM_OBJ(opL) == libGAP_T_FUNCTION ) {
        tmp = libGAP_ADDR_EXPR(expr)[1];
        opR = libGAP_EVAL_EXPR( tmp );
        if ( libGAP_TNUM_OBJ(opR) == libGAP_T_FUNCTION ) {
            return libGAP_NewAndFilter( opL, opR );
        }
        else {
            libGAP_ErrorQuit(
                "<expr> must be 'true' or 'false' (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L );
        }
    }
    
    /* signal an error                                                     */
    else {
        libGAP_ErrorQuit(
            "<expr> must be 'true' or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L );
    }
    
    /* please 'lint'                                                       */
    return 0;
}


/****************************************************************************
**
*F  EvalNot(<expr>) . . . . . . . . . . . . . . . . .  negate a boolean value
**
**  'EvalNot'  evaluates the  not-expression  <expr>  and returns its  value,
**  i.e., 'true' if the operand is 'false', and 'false' otherwise.  'EvalNot'
**  is called from 'EVAL_EXPR' to evaluate expressions of type 'T_NOT'.
*/
libGAP_Obj             libGAP_EvalNot (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 op;             /* evaluated operand               */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* evaluate the operand to a boolean                                   */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    op = libGAP_EVAL_BOOL_EXPR( tmp );

    /* compute the negation                                                */
    val = (op == libGAP_False ? libGAP_True : libGAP_False);

    /* return the negated value                                            */
    return val;
}


/****************************************************************************
**
*F  EvalEq(<expr>)  . . . . . . . . . . . . . . . . . . evaluate a comparison
**
**  'EvalEq' evaluates the  equality-expression <expr> and returns its value,
**  i.e.,  'true' if  the  operand '<expr>.left'   is equal  to  the  operand
**  '<expr>.right'   and   'false'  otherwise.   'EvalEq'  is   called   from
**  'EVAL_EXPR' to evaluate expressions of type 'T_EQ'.
**
**  'EvalEq' evaluates the operands and then calls the 'EQ' macro.
*/
libGAP_Obj             libGAP_EvalEq (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* compare the operands                                                */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = (libGAP_EQ( opL, opR ) ? libGAP_True : libGAP_False);

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalNe(<expr>)  . . . . . . . . . . . . . . . . . . evaluate a comparison
**
**  'EvalNe'   evaluates the  comparison-expression  <expr>  and  returns its
**  value, i.e.,  'true'  if the operand   '<expr>.left' is not equal  to the
**  operand  '<expr>.right' and  'false' otherwise.  'EvalNe'  is called from
**  'EVAL_EXPR' to evaluate expressions of type 'T_LT'.
**
**  'EvalNe' is simply implemented as 'not <objL> = <objR>'.
*/
libGAP_Obj             libGAP_EvalNe (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* compare the operands                                                */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = (libGAP_EQ( opL, opR ) ? libGAP_False : libGAP_True);

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalLt(<expr>)  . . . . . . . . . . . . . . . . . . evaluate a comparison
**
**  'EvalLt' evaluates  the  comparison-expression   <expr> and  returns  its
**  value, i.e., 'true' if the operand '<expr>.left' is less than the operand
**  '<expr>.right'  and  'false'   otherwise.    'EvalLt'  is   called   from
**  'EVAL_EXPR' to evaluate expressions of type 'T_LT'.
**
**  'EvalLt' evaluates the operands and then calls the 'LT' macro.
*/
libGAP_Obj             libGAP_EvalLt (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* compare the operands                                                */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = (libGAP_LT( opL, opR ) ? libGAP_True : libGAP_False);

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalGe(<expr>)  . . . . . . . . . . . . . . . . . . evaluate a comparison
**
**  'EvalGe'  evaluates  the comparison-expression   <expr>  and returns  its
**  value, i.e., 'true' if the operand '<expr>.left' is greater than or equal
**  to the operand '<expr>.right' and 'false'  otherwise.  'EvalGe' is called
**  from 'EVAL_EXPR' to evaluate expressions of type 'T_GE'.
**
**  'EvalGe' is simply implemented as 'not <objL> < <objR>'.
*/
libGAP_Obj             libGAP_EvalGe (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* compare the operands                                                */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = (libGAP_LT( opL, opR ) ? libGAP_False : libGAP_True);

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalGt(<expr>)  . . . . . . . . . . . . . . . . . . evaluate a comparison
**
**  'EvalGt'  evaluates  the  comparison-expression <expr>   and  returns its
**  value, i.e.,  'true' if the  operand  '<expr>.left' is  greater than  the
**  operand '<expr>.right' and 'false' otherwise.    'EvalGt' is called  from
**  'EVAL_EXPR' to evaluate expressions of type 'T_GT'.
**
**  'EvalGt' is simply implemented as '<objR> < <objL>'.
*/
libGAP_Obj             libGAP_EvalGt (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* compare the operands                                                */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = (libGAP_LT( opR, opL ) ? libGAP_True : libGAP_False);

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalLe(<expr>)  . . . . . . . . . . . . . . . . . . evaluate a comparison
**
**  'EvalLe' evaluates   the comparison-expression   <expr> and  returns  its
**  value, i.e., 'true' if the operand '<expr>.left' is  less or equal to the
**  operand '<expr>.right' and 'false'   otherwise.  'EvalLe' is  called from
**  'EVAL_EXPR' to evaluate expressions of type 'T_LE'.
**
**  'EvalLe' is simply implemented as 'not <objR> < <objR>'.
*/
libGAP_Obj             libGAP_EvalLe (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* compare the operands                                                */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = (libGAP_LT( opR, opL ) ? libGAP_False : libGAP_True);

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalIn(<in>)  . . . . . . . . . . . . . . . test for membership in a list
**
**  'EvalIn' evaluates the in-expression <expr>  and returns its value, i.e.,
**  'true' if  the  operand '<expr>.left'  is a  member of '<expr>.right' and
**  'false' otherwise.    'EvalIn' is  called  from  'EVAL_EXPR'  to evaluate
**  expressions of type 'T_IN'.
*/
libGAP_Obj             libGAP_EvalIn (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* evaluate <opL>                                                      */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );

    /* evaluate <opR>                                                      */
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* perform the test                                                    */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = (libGAP_IN( opL, opR ) ? libGAP_True : libGAP_False);

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalSum(<expr>) . . . . . . . . . . . . . . . . . . . . .  evaluate a sum
**
**  'EvalSum'  evaluates the  sum-expression  <expr> and  returns its  value,
**  i.e., the sum of   the  two operands '<expr>.left'   and  '<expr>.right'.
**  'EvalSum'   is called from 'EVAL_EXPR'   to  evaluate expressions of type
**  'T_SUM'.
**
**  'EvalSum' evaluates the operands and then calls the 'SUM' macro.
*/
libGAP_Obj             libGAP_EvalSum (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* first try to treat the operands as small integers with small result */
    if ( ! libGAP_ARE_INTOBJS( opL, opR ) || ! libGAP_SUM_INTOBJS( val, opL, opR ) ) {

        /* if that doesn't work, dispatch to the addition function         */
        libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
        val = libGAP_SUM( opL, opR );

    }

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalAInv(<expr>)  . . . . . . . . . . . . . . evaluate a additive inverse
**
**  'EvalAInv' evaluates  the additive  inverse-expression  and  returns  its
**  value, i.e., the  additive inverse of  the operand.  'EvalAInv' is called
**  from 'EVAL_EXPR' to evaluate expressions of type 'T_AINV'.
**
**  'EvalAInv' evaluates the operand and then calls the 'AINV' macro.
*/
libGAP_Obj             libGAP_EvalAInv (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );

    /* compute the additive inverse                                        */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = libGAP_AINV( opL );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalDiff(<expr>)  . . . . . . . . . . . . . . . . . evaluate a difference
**
**  'EvalDiff'  evaluates  the difference-expression <expr>   and returns its
**  value, i.e.,   the   difference of  the two  operands   '<expr>.left' and
**  '<expr>.right'.  'EvalDiff'    is  called from   'EVAL_EXPR'  to evaluate
**  expressions of type 'T_DIFF'.
**
**  'EvalDiff' evaluates the operands and then calls the 'DIFF' macro.
*/
libGAP_Obj             libGAP_EvalDiff (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* first try to treat the operands as small integers with small result */
    if ( ! libGAP_ARE_INTOBJS( opL, opR ) || ! libGAP_DIFF_INTOBJS( val, opL, opR ) ) {

        /* if that doesn't work, dispatch to the subtraction function      */
        libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
        val = libGAP_DIFF( opL, opR );

    }

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalProd(<expr>)  . . . . . . . . . . . . . . . . . .  evaluate a product
**
**  'EvalProd' evaluates the product-expression <expr>  and returns it value,
**  i.e., the product of  the two operands '<expr>.left'  and '<expr>.right'.
**  'EvalProd'  is called from   'EVAL_EXPR' to evaluate  expressions of type
**  'T_PROD'.
**
**  'EvalProd' evaluates the operands and then calls the 'PROD' macro.
*/
libGAP_Obj             libGAP_EvalProd (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* result                          */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* first try to treat the operands as small integers with small result */
    if ( ! libGAP_ARE_INTOBJS( opL, opR ) || ! libGAP_PROD_INTOBJS( val, opL, opR ) ) {

        /* if that doesn't work, dispatch to the multiplication function   */
        libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
        val = libGAP_PROD( opL, opR );

    }

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalInv(<expr>) . . . . . . . . . . . . evaluate a multiplicative inverse
**
**  'EvalInv' evaluates the multiplicative inverse-expression and returns its
**  value,  i.e., the multiplicative inverse  of  the operand.  'EvalInv' is
**  called from 'EVAL_EXPR' to evaluate expressions of type 'T_INV'.
**
**  'EvalInv' evaluates the operand and then calls the 'INV' macro.
*/
libGAP_Obj             libGAP_EvalInv (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );

    /* compute the multiplicative inverse                                  */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = libGAP_INV_MUT( opL );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalQuo(<expr>) . . . . . . . . . . . . . . . . . . . evaluate a quotient
**
**  'EvalQuo' evaluates the quotient-expression <expr> and returns its value,
**  i.e., the quotient of the  two operands '<expr>.left' and '<expr>.right'.
**  'EvalQuo' is  called  from 'EVAL_EXPR' to   evaluate expressions  of type
**  'T_QUO'.
**
**  'EvalQuo' evaluates the operands and then calls the 'QUO' macro.
*/
libGAP_Obj             libGAP_EvalQuo (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* dispatch to the division function                                   */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = libGAP_QUO( opL, opR );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalMod(<expr>) . . . . . . . . . . . . . . . . . .  evaluate a remainder
**
**  'EvalMod' evaluates the  remainder-expression   <expr> and returns    its
**  value, i.e.,  the  remainder  of   the two  operands   '<expr>.left'  and
**  '<expr>.right'.  'EvalMod'  is   called   from  'EVAL_EXPR'  to  evaluate
**  expressions of type 'T_MOD'.
**
**  'EvalMod' evaluates the operands and then calls the 'MOD' macro.
*/
libGAP_Obj             libGAP_EvalMod (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* dispatch to the remainder function                                  */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = libGAP_MOD( opL, opR );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalPow(<expr>) . . . . . . . . . . . . . . . . . . . .  evaluate a power
**
**  'EvalPow'  evaluates the power-expression  <expr>  and returns its value,
**  i.e.,   the power of the  two  operands '<expr>.left' and '<expr>.right'.
**  'EvalPow' is called  from  'EVAL_EXPR'  to evaluate expressions  of  type
**  'T_POW'.
**
**  'EvalPow' evaluates the operands and then calls the 'POW' macro.
*/
libGAP_Obj             libGAP_EvalPow (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* value, result                   */
    libGAP_Obj                 opL;            /* evaluated left  operand         */
    libGAP_Obj                 opR;            /* evaluated right operand         */
    libGAP_Expr                tmp;            /* temporary expression            */

    /* get the operands                                                    */
    tmp = libGAP_ADDR_EXPR(expr)[0];
    opL = libGAP_EVAL_EXPR( tmp );
    tmp = libGAP_ADDR_EXPR(expr)[1];
    opR = libGAP_EVAL_EXPR( tmp );

    /* dispatch to the powering function                                   */
    libGAP_SET_BRK_CALL_TO(expr);     /* Note possible call for FuncWhere */
    val = libGAP_POW( opL, opR );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalIntExpr(<expr>) . . . . . . . . . evaluate literal integer expression
**
**  'EvalIntExpr' evaluates the literal integer expression <expr> and returns
**  its value.
*/
#define libGAP_IDDR_EXPR(expr)         ((libGAP_UInt2*)libGAP_ADDR_EXPR(expr))

libGAP_Obj             libGAP_EvalIntExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 val;            /* integer, result                 */

    
    /* allocate the integer                                                */
    val = libGAP_NewBag( ((libGAP_UInt *)libGAP_ADDR_EXPR(expr))[0], libGAP_SIZE_EXPR(expr)-sizeof(libGAP_UInt));
    memcpy((void *)libGAP_ADDR_OBJ(val), (void *)(((libGAP_UInt *)libGAP_ADDR_EXPR(expr))+1), (size_t) (libGAP_SIZE_EXPR(expr)-sizeof(libGAP_UInt)));

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  EvalTrueExpr(<expr>)  . . . . . . . . .  evaluate literal true expression
**
**  'EvalTrueExpr' evaluates the  literal true expression <expr> and  returns
**  its value (True).
*/
libGAP_Obj             libGAP_EvalTrueExpr (
    libGAP_Expr                expr )
{
    return libGAP_True;
}


/****************************************************************************
**
*F  EvalFalseExpr(<expr>) . . . . . . . . . evaluate literal false expression
**
**  'EvalFalseExpr' evaluates the literal false expression <expr> and returns
**  its value (False).
*/
libGAP_Obj             libGAP_EvalFalseExpr (
    libGAP_Expr                expr )
{
    return libGAP_False;
}


/****************************************************************************
**
*F  EvalCharExpr(<expr>)  . . . . . . evaluate a literal character expression
**
**  'EvalCharExpr' evaluates  the   literal character expression <expr>   and
**  returns its value.
*/
libGAP_Obj             libGAP_EvalCharExpr (
    libGAP_Expr                expr )
{
    return libGAP_ObjsChar[ ((libGAP_UChar*)libGAP_ADDR_EXPR(expr))[0] ];
}


/****************************************************************************
**
*F  EvalPermExpr(<expr>)  . . . . . . . . . evaluate a permutation expression
**
**  'EvalPermExpr' evaluates the permutation expression <expr>.
*/
libGAP_Obj             libGAP_EvalPermExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 perm;           /* permutation, result             */
    libGAP_UInt4 *             ptr4;           /* pointer into perm               */
    libGAP_UInt2 *             ptr2;           /* pointer into perm               */
    libGAP_Obj                 val;            /* one entry as value              */
    libGAP_UInt                c, p, l;        /* entries in permutation          */
    libGAP_UInt                m;              /* maximal entry in permutation    */
    libGAP_Expr                cycle;          /* one cycle of permutation        */
    libGAP_UInt                i, j, k;        /* loop variable                   */

    /* special case for identity permutation                               */
    if ( libGAP_SIZE_EXPR(expr) == 0 ) {
        return libGAP_IdentityPerm;
    }

    /* allocate the new permutation                                        */
    m = 0;
    perm = libGAP_NEW_PERM4( 0 );

    /* loop over the cycles                                                */
    for ( i = 1; i <= libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr); i++ ) {
        cycle = libGAP_ADDR_EXPR(expr)[i-1];

        /* loop over the entries of the cycle                              */
        c = p = l = 0;
        for ( j = libGAP_SIZE_EXPR(cycle)/sizeof(libGAP_Expr); 1 <= j; j-- ) {

            /* get and check current entry for the cycle                   */
            val = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR( cycle )[j-1] );
            while ( ! libGAP_IS_INTOBJ(val) || libGAP_INT_INTOBJ(val) <= 0 ) {
                val = libGAP_ErrorReturnObj(
              "Permutation: <expr> must be a positive integer (not a %s)",
                    (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
                    "you can replace <expr> via 'return <expr>;'" );
            }
            c = libGAP_INT_INTOBJ(val);

            /* if necessary resize the permutation                         */
            if ( libGAP_SIZE_OBJ(perm)/sizeof(libGAP_UInt4) < c ) {
                libGAP_ResizeBag( perm, (c + 1023) / 1024 * 1024 * sizeof(libGAP_UInt4) );
                ptr4 = libGAP_ADDR_PERM4( perm );
                for ( k = m+1; k <= libGAP_SIZE_OBJ(perm)/sizeof(libGAP_UInt4); k++ ) {
                    ptr4[k-1] = k-1;
                }
            }
            if ( m < c ) {
                m = c;
            }

            /* check that the cycles are disjoint                          */
            ptr4 = libGAP_ADDR_PERM4( perm );
            if ( (p != 0 && p == c) || (ptr4[c-1] != c-1) ) {
                return libGAP_ErrorReturnObj(
                    "Permutation: cycles must be disjoint",
                    0L, 0L,
                    "you can replace permutation <perm> via 'return <perm>;'" );
            }

            /* enter the previous entry at current location                */
            ptr4 = libGAP_ADDR_PERM4( perm );
            if ( p != 0 ) { ptr4[c-1] = p-1; }
            else          { l = c;          }

            /* remember current entry for next round                       */
            p = c;
        }

        /* enter first (last popped) entry at last (first popped) location */
        ptr4 = libGAP_ADDR_PERM4( perm );
        ptr4[l-1] = p-1;

    }

    /* if possible represent the permutation with short entries            */
    if ( m <= 65536UL ) {
        ptr2 = libGAP_ADDR_PERM2( perm );
        ptr4 = libGAP_ADDR_PERM4( perm );
        for ( k = 1; k <= m; k++ ) {
            ptr2[k-1] = ptr4[k-1];
        };
        libGAP_RetypeBag( perm, libGAP_T_PERM2 );
        libGAP_ResizeBag( perm, m * sizeof(libGAP_UInt2) );
    }

    /* otherwise just shorten the permutation                              */
    else {
        libGAP_ResizeBag( perm, m * sizeof(libGAP_UInt4) );
    }

    /* return the permutation                                              */
    return perm;
}


/****************************************************************************
**
*F  EvalListExpr(<expr>)  . . . . .  evaluate list expression to a list value
**
**  'EvalListExpr'  evaluates the list   expression, i.e., not  yet evaluated
**  list, <expr> to a list value.
**
**  'EvalListExpr'  just  calls 'ListExpr1'  and  'ListExpr2' to evaluate the
**  list expression.
*/
libGAP_Obj             libGAP_ListExpr1 ( libGAP_Expr expr );
void            libGAP_ListExpr2 ( libGAP_Obj list, libGAP_Expr expr );
libGAP_Obj             libGAP_RecExpr1 ( libGAP_Expr expr );
void            libGAP_RecExpr2 ( libGAP_Obj rec, libGAP_Expr expr );

libGAP_Obj             libGAP_EvalListExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 list;         /* list value, result                */

    /* evalute the list expression                                         */
    list = libGAP_ListExpr1( expr );
    libGAP_ListExpr2( list, expr );

    /* return the result                                                   */
    return list;
}


/****************************************************************************
**
*F  EvalListTildeExpr(<expr>) . . . . evaluate a list expression with a tilde
**
**  'EvalListTildeExpr' evaluates the     list  expression, i.e., not     yet
**  evaluated list, <expr> to a list value.  The difference to 'EvalListExpr'
**  is that  in <expr> there are   occurrences of '~'  referring to  this list
**  value.
**
**  'EvalListTildeExpr' just  calls 'ListExpr1' to  create  the list, assigns
**  the list to  the variable '~', and  finally calls 'ListExpr2' to evaluate
**  the   subexpressions into  the  list.   Thus subexpressions  in  the list
**  expression can  refer to   this  variable and  its subobjects  to  create
**  objects that are not trees.
*/
libGAP_Obj             libGAP_EvalListTildeExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 list;           /* list value, result              */
    libGAP_Obj                 tilde;          /* old value of tilde              */

    /* remember the old value of '~'                                       */
    tilde = libGAP_VAL_GVAR( libGAP_Tilde );

    /* create the list value                                               */
    list = libGAP_ListExpr1( expr );

    /* assign the list to '~'                                              */
    libGAP_AssGVar( libGAP_Tilde, list );

    /* evaluate the subexpressions into the list value                     */
    libGAP_ListExpr2( list, expr );

    /* restore old value of '~'                                            */
    libGAP_AssGVar( libGAP_Tilde, tilde );

    /* return the list value                                               */
    return list;
}


/****************************************************************************
**
*F  ListExpr1(<expr>) . . . . . . . . . . . make a list for a list expression
*F  ListExpr2(<list>,<expr>)  . . .  enter the sublists for a list expression
**
**  'ListExpr1' and 'ListExpr2'  together evaluate the list expression <expr>
**  into the list <list>.
**
**  'ListExpr1'  allocates a new  plain  list of the  same  size as  the list
**  expression <expr> and returns this list.
**
**  'ListExpr2' evaluates  the  subexpression of <expr>   and puts the values
**  into the list  <list> (which should be a  plain list of  the same size as
**  the list expression <expr>, e.g., the one allocated by 'ListExpr1').
**
**  This two step allocation  is necessary, because  list expressions such as
**  '[ [1], ~[1] ]'  requires that the value of  one subexpression is entered
**  into the list value before the next subexpression is evaluated.
*/
libGAP_Obj libGAP_ListExpr1 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 list;           /* list value, result              */
    libGAP_Int                 len;            /* logical length of the list      */

    /* get the length of the list                                          */
    len = libGAP_SIZE_EXPR(expr) / sizeof(libGAP_Expr);

    /* allocate the list value                                             */
    if ( 0 == len ) {
        list = libGAP_NEW_PLIST( libGAP_T_PLIST_EMPTY, len );
    }
    else {
        list = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
    }
    libGAP_SET_LEN_PLIST( list, len );

    /* return the list                                                     */
    return list;
}

void libGAP_ListExpr2 (
    libGAP_Obj                 list,
    libGAP_Expr                expr )
{
    libGAP_Obj                 sub;            /* value of a subexpression        */
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Int                 posshole;       /* initially 0, set to 1 at
                                           first empty position, then
                                           next full position causes
                                           the list to be made
                                           non-dense */

    /* get the length of the list                                          */
    len = libGAP_SIZE_EXPR(expr) / sizeof(libGAP_Expr);

    /* initially we have not seen a hole                                   */
    posshole = 0;

    /* handle the subexpressions                                           */
    for ( i = 1; i <= len; i++ ) {

        /* if the subexpression is empty                                   */
        if ( libGAP_ADDR_EXPR(expr)[i-1] == 0 ) {
          if (!posshole)
            posshole = 1;
          continue;
        }
        else 
          {
            if (posshole == 1)
              {
                libGAP_SET_FILT_LIST(list, libGAP_FN_IS_NDENSE);
                posshole = 2;
              }

            /* special case if subexpression is a list expression              */
            if ( libGAP_TNUM_EXPR( libGAP_ADDR_EXPR(expr)[i-1] ) == libGAP_T_LIST_EXPR ) {
              sub = libGAP_ListExpr1( libGAP_ADDR_EXPR(expr)[i-1] );
              libGAP_SET_ELM_PLIST( list, i, sub );
              libGAP_CHANGED_BAG( list );
              libGAP_ListExpr2( sub, libGAP_ADDR_EXPR(expr)[i-1] );
            }
            
            /* special case if subexpression is a record expression            */
            else if ( libGAP_TNUM_EXPR( libGAP_ADDR_EXPR(expr)[i-1] ) == libGAP_T_REC_EXPR ) {
              sub = libGAP_RecExpr1( libGAP_ADDR_EXPR(expr)[i-1] );
              libGAP_SET_ELM_PLIST( list, i, sub );
              libGAP_CHANGED_BAG( list );
              libGAP_RecExpr2( sub, libGAP_ADDR_EXPR(expr)[i-1] );
            }
            
            /* general case                                                    */
            else {
              sub = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[i-1] );
              libGAP_SET_ELM_PLIST( list, i, sub );
              libGAP_CHANGED_BAG( list );
            }
          }

    }
    if (!posshole)
      libGAP_SET_FILT_LIST(list, libGAP_FN_IS_DENSE);

}


/****************************************************************************
**
*F  EvalRangeExpr(<expr>) . . . . .  eval a range expression to a range value
**
**  'EvalRangeExpr' evaluates the range expression <expr> to a range value.
*/
libGAP_Obj             libGAP_EvalRangeExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 range;          /* range, result                   */
    libGAP_Obj                 val;            /* subvalue of range               */
    libGAP_Int                 low;            /* low (as C integer)              */
    libGAP_Int                 inc;            /* increment (as C integer)        */
    libGAP_Int                 high;           /* high (as C integer)             */

    /* evaluate the low value                                              */
    val = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[0] );
    while ( ! libGAP_IS_INTOBJ(val) ) {
        val = libGAP_ErrorReturnObj(
            "Range: <first> must be an integer less than 2^%d (not a %s)",
            libGAP_NR_SMALL_INT_BITS, (libGAP_Int)libGAP_TNAM_OBJ(val),
            "you can replace <first> via 'return <first>;'" );
    }
    low = libGAP_INT_INTOBJ( val );

    /* evaluate the second value (if present)                              */
    if ( libGAP_SIZE_EXPR(expr) == 3*sizeof(libGAP_Expr) ) {
        val = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[1] );
        while ( ! libGAP_IS_INTOBJ(val) || libGAP_INT_INTOBJ(val) == low ) {
            if ( ! libGAP_IS_INTOBJ(val) ) {
                val = libGAP_ErrorReturnObj(
                    "Range: <second> must be an integer less than 2^%d (not a %s)",
                    libGAP_NR_SMALL_INT_BITS, (libGAP_Int)libGAP_TNAM_OBJ(val),
                    "you can replace <second> via 'return <second>;'" );
            }
            else {
                val = libGAP_ErrorReturnObj(
                    "Range: <second> must not be equal to <first> (%d)",
                    (libGAP_Int)low, 0L,
                    "you can replace the integer <second> via 'return <second>;'" );
            }
        }
        inc = libGAP_INT_INTOBJ(val) - low;
    }
    else {
        inc = 1;
    }

    /* evaluate and check the high value                                   */
    val = libGAP_EVAL_EXPR( libGAP_ADDR_EXPR(expr)[ libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr)-1 ] );
    while ( ! libGAP_IS_INTOBJ(val) || (libGAP_INT_INTOBJ(val) - low) % inc != 0 ) {
        if ( ! libGAP_IS_INTOBJ(val) ) {
            val = libGAP_ErrorReturnObj(
                "Range: <last> must be an integer less than 2^%d (not a %s)",
                libGAP_NR_SMALL_INT_BITS, (libGAP_Int)libGAP_TNAM_OBJ(val),
                "you can replace <last> via 'return <last>;'" );
        }
        else {
            val = libGAP_ErrorReturnObj(
                "Range: <last>-<first> (%d) must be divisible by <inc> (%d)",
                (libGAP_Int)(libGAP_INT_INTOBJ(val)-low), (libGAP_Int)inc,
                "you can replace the integer <last> via 'return <last>;'" );
        }
    }
    high = libGAP_INT_INTOBJ(val);

    /* if <low> is larger than <high> the range is empty                   */
    if ( (0 < inc && high < low) || (inc < 0 && low < high) ) {
        range = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
        libGAP_SET_LEN_PLIST( range, 0 );
    }

    /* if <low> is equal to <high> the range is a singleton list           */
    else if ( low == high ) {
        range = libGAP_NEW_PLIST( libGAP_T_PLIST, 1 );
        libGAP_SET_LEN_PLIST( range, 1 );
        libGAP_SET_ELM_PLIST( range, 1, libGAP_INTOBJ_INT(low) );
    }

    /* else make the range                                                 */
    else {
        /* the length must be a small integer as well */
        if ((high-low) / inc + 1 >= (1L<<libGAP_NR_SMALL_INT_BITS)) {
             libGAP_ErrorQuit("Range: the length of a range must be less than 2^%d.",
                        libGAP_NR_SMALL_INT_BITS, 0L);
        }
        if ( 0 < inc )
            range = libGAP_NEW_RANGE_SSORT();
        else
            range = libGAP_NEW_RANGE_NSORT();
        libGAP_SET_LEN_RANGE( range, (high-low) / inc + 1 );
        libGAP_SET_LOW_RANGE( range, low );
        libGAP_SET_INC_RANGE( range, inc );
    }

    /* return the range                                                    */
    return range;
}


/****************************************************************************
**
*F  EvalStringExpr(<expr>)  . . . . eval string expressions to a string value
**
**  'EvalStringExpr'   evaluates the  string  expression  <expr>  to a string
**  value.
*/
libGAP_Obj             libGAP_EvalStringExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 string;         /* string value, result            */
    libGAP_UInt                 len;           /* size of expression              */
    
    len = *((libGAP_UInt *)libGAP_ADDR_EXPR(expr));
    string = libGAP_NEW_STRING(len);
    memcpy((void *)libGAP_ADDR_OBJ(string), (void *)libGAP_ADDR_EXPR(expr), 
                      libGAP_SIZEBAG_STRINGLEN(len) );

    /* return the string                                                   */
    return string;
}

/****************************************************************************
**
*F  EvalFloatExprLazy(<expr>)  . . . . eval float expressions to a float value
**
**  'EvalFloatExpr'   evaluates the  float  expression  <expr>  to a float
**  value.
*/
static libGAP_Obj libGAP_CONVERT_FLOAT_LITERAL;
static libGAP_Obj libGAP_FLOAT_LITERAL_CACHE;
static libGAP_UInt libGAP_GVAR_FLOAT_LITERAL_CACHE;
static libGAP_Obj libGAP_MAX_FLOAT_LITERAL_CACHE_SIZE;

libGAP_Obj             libGAP_EvalFloatExprLazy (
    libGAP_Expr                expr )
{
    libGAP_Obj                 string;         /* string value            */
    libGAP_UInt                 len;           /* size of expression              */
    libGAP_UInt                 ix;
    libGAP_Obj cache= 0;
    libGAP_Obj fl;
    
    ix = ((libGAP_UInt *)libGAP_ADDR_EXPR(expr))[1];
    if (ix && (!libGAP_MAX_FLOAT_LITERAL_CACHE_SIZE || 
               libGAP_MAX_FLOAT_LITERAL_CACHE_SIZE == libGAP_INTOBJ_INT(0) ||
               ix <= libGAP_INT_INTOBJ(libGAP_MAX_FLOAT_LITERAL_CACHE_SIZE))) {
      cache = libGAP_FLOAT_LITERAL_CACHE;
      if (!cache)
        {
          cache = libGAP_NEW_PLIST(libGAP_T_PLIST,ix);
          libGAP_AssGVar(libGAP_GVAR_FLOAT_LITERAL_CACHE, cache);
        }
      else
        assert(libGAP_IS_PLIST(cache));
      libGAP_GROW_PLIST(cache,ix);
      fl = libGAP_ELM_PLIST(cache,ix);
      if (fl)
        return fl;
    }
    len = *((libGAP_UInt *)libGAP_ADDR_EXPR(expr));
    string = libGAP_NEW_STRING(len);
    memcpy((void *)libGAP_CHARS_STRING(string), 
           (void *)((char *)libGAP_ADDR_EXPR(expr) + 2*sizeof(libGAP_UInt)), 
           len );
    fl = libGAP_CALL_1ARGS(libGAP_CONVERT_FLOAT_LITERAL, string);
    if (cache) {
      libGAP_SET_ELM_PLIST(cache, ix, fl);
      libGAP_CHANGED_BAG(cache);
      if (libGAP_LEN_PLIST(cache) < ix)
        libGAP_SET_LEN_PLIST(cache, ix);
    }

    return fl;
}

/****************************************************************************
**
*F  EvalFloatExprEager(<expr>)  . . . . eval float expressions to a float value
**
**  'EvalFloatExpr'   evaluates the  float  expression  <expr>  to a float
**  value.
*/
static libGAP_Obj libGAP_EAGER_FLOAT_LITERAL_CACHE;

libGAP_Obj             libGAP_EvalFloatExprEager (
    libGAP_Expr                expr )
{
    libGAP_UInt                 ix;
    libGAP_Obj cache= 0;
    libGAP_Obj fl;
    
    ix = ((libGAP_UInt *)libGAP_ADDR_EXPR(expr))[0];
    cache = libGAP_EAGER_FLOAT_LITERAL_CACHE;
    assert(libGAP_IS_PLIST(cache));
    fl = libGAP_ELM_PLIST(cache,ix);
    assert(fl);
    return fl;
}


/****************************************************************************
**
*F  EvalRecExpr(<expr>) . . . . . .  eval record expression to a record value
**
**  'EvalRecExpr' evaluates the record expression,   i.e., not yet  evaluated
**  record, <expr> to a record value.
**
**  'EvalRecExpr' just calls 'RecExpr1' and 'RecExpr2' to evaluate the record
**  expression.
*/
libGAP_Obj             libGAP_EvalRecExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 rec;            /* record value, result            */

    /* evaluate the record expression                                      */
    rec = libGAP_RecExpr1( expr );
    libGAP_RecExpr2( rec, expr );

    /* return the result                                                   */
    return rec;
}


/****************************************************************************
**
*F  EvalRecTildeExpr(<expr>)  . . . evaluate a record expression with a tilde
**
**  'EvalRecTildeExpr'  evaluates  the    record expression,  i.e.,   not yet
**  evaluated   record, <expr>  to  a   record   value.  The   difference  to
**  'EvalRecExpr' is that in <expr> there are  occurrences of '~' referring to
**  this record value.
**
**  'EvalRecTildeExpr' just  calls 'RecExpr1'  to create teh  record, assigns
**  the record to the variable '~',  and finally calls 'RecExpr2' to evaluate
**  the subexpressions  into the record.  Thus  subexpressions  in the record
**  expression    can refer to this variable    and its  subobjects to create
**  objects that are not trees.
*/
libGAP_Obj             libGAP_EvalRecTildeExpr (
    libGAP_Expr                expr )
{
    libGAP_Obj                 rec;            /* record value, result            */
    libGAP_Obj                 tilde;          /* old value of tilde              */

    /* remember the old value of '~'                                       */
    tilde = libGAP_VAL_GVAR( libGAP_Tilde );

    /* create the record value                                             */
    rec = libGAP_RecExpr1( expr );

    /* assign the record value to the variable '~'                         */
    libGAP_AssGVar( libGAP_Tilde, rec );

    /* evaluate the subexpressions into the record value                   */
    libGAP_RecExpr2( rec, expr );

    /* restore the old value of '~'                                        */
    libGAP_AssGVar( libGAP_Tilde, tilde );

    /* return the record value                                             */
    return rec;
}


/****************************************************************************
**
*F  RecExpr1(<expr>)  . . . . . . . . . make a record for a record expression
*F  RecExpr2(<rec>,<expr>)  . .  enter the subobjects for a record expression
**
**  'RecExpr1' and 'RecExpr2' together  evaluate the record expression <expr>
**  into the record <rec>.
**
**  'RecExpr1' allocates   a new record  of the    same size as   the  record
**  expression <expr> and returns this record.
**
**  'RecExpr2' evaluates the subexpressions   of <expr> and puts the   values
**  into the record <rec>  (which should be a record  of the same size as the
**  record expression <expr>, e.g., the one allocated by 'RecExpr1').
**
**  This two step allocation is necessary, because record expressions such as
**  'rec(  a := 1,  ~.a  )' requires that the   value of one subexpression is
**  entered into the record value before the next subexpression is evaluated.
*/
libGAP_Obj             libGAP_RecExpr1 (
    libGAP_Expr                expr )
{
    libGAP_Obj                 rec;            /* record value, result            */
    libGAP_Int                 len;            /* number of components            */

    /* get the number of components                                        */
    len = libGAP_SIZE_EXPR( expr ) / (2*sizeof(libGAP_Expr));

    /* allocate the record value                                           */
    rec = libGAP_NEW_PREC( len );

    /* return the record                                                   */
    return rec;
}

void            libGAP_RecExpr2 (
    libGAP_Obj                 rec,
    libGAP_Expr                expr )
{
    libGAP_UInt                rnam;           /* name of component               */
    libGAP_Obj                 sub;            /* value of subexpression          */
    libGAP_Int                 len;            /* number of components            */
    libGAP_Expr                tmp;            /* temporary variable              */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the number of components                                        */
    len = libGAP_SIZE_EXPR( expr ) / (2*sizeof(libGAP_Expr));

    /* handle the subexpressions                                           */
    for ( i = 1; i <= len; i++ ) {

        /* handle the name                                                 */
        tmp = libGAP_ADDR_EXPR(expr)[2*i-2];
        if ( libGAP_IS_INTEXPR(tmp) ) {
            rnam = (libGAP_UInt)libGAP_INT_INTEXPR(tmp);
        }
        else {
            rnam = libGAP_RNamObj( libGAP_EVAL_EXPR(tmp) );
        }

        /* if the subexpression is empty (cannot happen for records)       */
        tmp = libGAP_ADDR_EXPR(expr)[2*i-1];
        if ( tmp == 0 ) {
            continue;
        }

        /* special case if subexpression is a list expression             */
        else if ( libGAP_TNUM_EXPR( tmp ) == libGAP_T_LIST_EXPR ) {
            sub = libGAP_ListExpr1( tmp );
            libGAP_AssPRec(rec,rnam,sub);
            libGAP_ListExpr2( sub, tmp );
        }

        /* special case if subexpression is a record expression            */
        else if ( libGAP_TNUM_EXPR( tmp ) == libGAP_T_REC_EXPR ) {
            sub = libGAP_RecExpr1( tmp );
            libGAP_AssPRec(rec,rnam,sub);
            libGAP_RecExpr2( sub, tmp );
        }

        /* general case                                                    */
        else {
            sub = libGAP_EVAL_EXPR( tmp );
            libGAP_AssPRec(rec,rnam,sub);
        }
    }
    libGAP_SortPRecRNam(rec,0);

}


/****************************************************************************
**
*F  PrintExpr(<expr>) . . . . . . . . . . . . . . . . . . print an expression
**
**  'PrintExpr' prints the expression <expr>.
**
**  'PrintExpr' simply dispatches  through  the table 'PrintExprFuncs' to the
**  appropriate printer.
*/
void            libGAP_PrintExpr (
    libGAP_Expr                expr )
{
    (*libGAP_PrintExprFuncs[ libGAP_TNUM_EXPR(expr) ])( expr );
}


/****************************************************************************
**
*V  PrintExprFuncs[<type>]  . .  printing function for objects of type <type>
**
**  'PrintExprFuncs' is the dispatching table that contains for every type of
**  expressions a pointer to the printer for expressions  of this type, i.e.,
**  the function that should be called to print expressions of this type.
*/
void            (* libGAP_PrintExprFuncs[256] ) ( libGAP_Expr expr );


/****************************************************************************
**
*F  PrintUnknownExpr(<expr>)  . . . . . . .  print expression of unknown type
**
**  'PrintUnknownExpr' is the printer that is called if an attempt is made to
**  print an expression <expr> of an unknown type.  It signals  an error.  If
**  this  is ever called,   then  GAP is  in  serious   trouble, such as   an
**  overwritten type field of an expression.
*/
void            libGAP_PrintUnknownExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr( "Panic: tried to print an expression of unknown type '%d'\n",
        (libGAP_Int)libGAP_TNUM_EXPR(expr), 0L );
}


/****************************************************************************
**
*V  PrintPreceedence  . . . . . . . . . . . . . . . current preceedence level
**
**  'PrintPreceedence' contains  the  current  preceedence   level,  i.e.  an
**  integer  indicating the binding power  of the currently printed operator.
**  If one of the operands is an operation that has lower binding power it is
**  printed in parenthesis.  If the right  operand has the same binding power
**  it is put in parenthesis, since  all the operations are left associative.
**  Preceedence: 14: ^; 12: mod,/,*; 10: -,+; 8: in,=; 6: not; 4: and; 2: or.
**  This sometimes puts in superflous parenthesis: 2 * f( (3 + 4) ), since it
**  doesn't know that a function call adds automatically parenthesis.
*/
libGAP_UInt            libGAP_PrintPreceedence;


/****************************************************************************
**
*F  PrintNot(<expr>)  . . . . . . . . . . . . .  print a boolean not operator
**
**  'PrintNot' print a not operation in the following form: 'not <expr>'.
*/
void            libGAP_PrintNot (
    libGAP_Expr                expr )
{
    libGAP_UInt                oldPrec;

    oldPrec = libGAP_PrintPreceedence;
    libGAP_PrintPreceedence = 6;
    
    /* if necessary print the opening parenthesis                          */
    if ( oldPrec >= libGAP_PrintPreceedence ) libGAP_Pr("%>(%>",0L,0L);
    else libGAP_Pr("%2>",0L,0L);
    
    libGAP_Pr("not%> ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<",0L,0L);
    
    /* if necessary print the closing parenthesis                          */
    if ( oldPrec >= libGAP_PrintPreceedence ) libGAP_Pr("%2<)",0L,0L);
    else libGAP_Pr("%2<",0L,0L);
    
    libGAP_PrintPreceedence = oldPrec;
}


/****************************************************************************
**
*F  PrintBinop(<expr>)  . . . . . . . . . . . . . .  prints a binary operator
**
**  'PrintBinop'  prints  the   binary operator    expression <expr>,   using
**  'PrintPreceedence' for parenthesising.
*/
void            libGAP_PrintAInv (
    libGAP_Expr                expr )
{
    libGAP_UInt                oldPrec;

    oldPrec = libGAP_PrintPreceedence;
    libGAP_PrintPreceedence = 11;
    
    /* if necessary print the opening parenthesis                          */
    if ( oldPrec >= libGAP_PrintPreceedence ) libGAP_Pr("%>(%>",0L,0L);
    else libGAP_Pr("%2>",0L,0L);
    
    libGAP_Pr("-%> ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<",0L,0L);
    
    /* if necessary print the closing parenthesis                          */
    if ( oldPrec >= libGAP_PrintPreceedence ) libGAP_Pr("%2<)",0L,0L);
    else libGAP_Pr("%2<",0L,0L);
    
    libGAP_PrintPreceedence = oldPrec;
}

void            libGAP_PrintInv (
    libGAP_Expr                expr )
{
    libGAP_UInt                oldPrec;

    oldPrec = libGAP_PrintPreceedence;
    libGAP_PrintPreceedence = 14;
    libGAP_Pr("%> ",0L,0L);
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Pr("%<^-1",0L,0L);
    libGAP_PrintPreceedence = oldPrec;
}

void            libGAP_PrintBinop (
    libGAP_Expr                expr )
{
    libGAP_UInt                oldPrec;        /* old preceedence level           */
    const libGAP_Char *        op;             /* operand                         */
    /* remember the current preceedence level                              */
    oldPrec = libGAP_PrintPreceedence;

    /* select the new preceedence level                                    */
    switch ( libGAP_TNUM_EXPR(expr) ) {
    case libGAP_T_OR:     op = "or";   libGAP_PrintPreceedence =  2;  break;
    case libGAP_T_AND:    op = "and";  libGAP_PrintPreceedence =  4;  break;
    case libGAP_T_EQ:     op = "=";    libGAP_PrintPreceedence =  8;  break;
    case libGAP_T_LT:     op = "<";    libGAP_PrintPreceedence =  8;  break;
    case libGAP_T_GT:     op = ">";    libGAP_PrintPreceedence =  8;  break;
    case libGAP_T_NE:     op = "<>";   libGAP_PrintPreceedence =  8;  break;
    case libGAP_T_LE:     op = "<=";   libGAP_PrintPreceedence =  8;  break;
    case libGAP_T_GE:     op = ">=";   libGAP_PrintPreceedence =  8;  break;
    case libGAP_T_IN:     op = "in";   libGAP_PrintPreceedence =  8;  break;
    case libGAP_T_SUM:    op = "+";    libGAP_PrintPreceedence = 10;  break;
    case libGAP_T_DIFF:   op = "-";    libGAP_PrintPreceedence = 10;  break;
    case libGAP_T_PROD:   op = "*";    libGAP_PrintPreceedence = 12;  break;
    case libGAP_T_QUO:    op = "/";    libGAP_PrintPreceedence = 12;  break;
    case libGAP_T_MOD:    op = "mod";  libGAP_PrintPreceedence = 12;  break;
    case libGAP_T_POW:    op = "^";    libGAP_PrintPreceedence = 16;  break;
    default:       op = "<bogus-operator>";   break;
    }

    /* if necessary print the opening parenthesis                          */
    if ( oldPrec > libGAP_PrintPreceedence ) libGAP_Pr("%>(%>",0L,0L);
    else libGAP_Pr("%2>",0L,0L);

    /* print the left operand                                              */
    if ( libGAP_TNUM_EXPR(expr) == libGAP_T_POW
         && ((  (libGAP_IS_INTEXPR(libGAP_ADDR_EXPR(expr)[0])
                 && libGAP_INT_INTEXPR(libGAP_ADDR_EXPR(expr)[0]) < 0)
                || libGAP_TNUM_EXPR(libGAP_ADDR_EXPR(expr)[0]) == libGAP_T_INTNEG)
             || libGAP_TNUM_EXPR(libGAP_ADDR_EXPR(expr)[0]) == libGAP_T_POW) ) {
        libGAP_Pr( "(", 0L, 0L );
        libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
        libGAP_Pr( ")", 0L, 0L );
    }
    else {
        libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
    }

    /* print the operator                                                  */
    libGAP_Pr("%2< %2>%s%> %<",(libGAP_Int)op,0L);

    /* print the right operand                                             */
    libGAP_PrintPreceedence++;
    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_PrintPreceedence--;

    /* if necessary print the closing parenthesis                          */
    if ( oldPrec > libGAP_PrintPreceedence ) libGAP_Pr("%2<)",0L,0L);
    else libGAP_Pr("%2<",0L,0L);

    /* restore the old preceedence level                                   */
    libGAP_PrintPreceedence = oldPrec;
}


/****************************************************************************
**
*F  PrintIntExpr(<expr>)  . . . . . . . . . . . . print an integer expression
**
**  'PrintIntExpr' prints the literal integer expression <expr>.
*/
void            libGAP_PrintIntExpr (
    libGAP_Expr                expr )
{
    if ( libGAP_IS_INTEXPR(expr) ) {
        libGAP_Pr( "%d", libGAP_INT_INTEXPR(expr), 0L );
    }
    else {
        libGAP_PrintInt(libGAP_EvalIntExpr(expr));
    }
}


/****************************************************************************
**
*F  PrintTrueExpr(<expr>) . . . . . . . . . . . print literal true expression
*/
void            libGAP_PrintTrueExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr( "true", 0L, 0L );
}


/****************************************************************************
**
*F  PrintFalseExpr(<expr>)  . . . . . . . . .  print literal false expression
*/
void            libGAP_PrintFalseExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr( "false", 0L, 0L );
}


/****************************************************************************
**
*F  PrintCharExpr(<expr>) . . . . . . . .  print literal character expression
*/
void            libGAP_PrintCharExpr (
    libGAP_Expr                expr )
{
    libGAP_UChar               chr;

    chr = *(libGAP_UChar*)libGAP_ADDR_EXPR(expr);
    if      ( chr == '\n'  )  libGAP_Pr("'\\n'",0L,0L);
    else if ( chr == '\t'  )  libGAP_Pr("'\\t'",0L,0L);
    else if ( chr == '\r'  )  libGAP_Pr("'\\r'",0L,0L);
    else if ( chr == '\b'  )  libGAP_Pr("'\\b'",0L,0L);
    else if ( chr == '\03' )  libGAP_Pr("'\\c'",0L,0L);
    else if ( chr == '\''  )  libGAP_Pr("'\\''",0L,0L);
    else if ( chr == '\\'  )  libGAP_Pr("'\\\\'",0L,0L);
    else                      libGAP_Pr("'%c'",(libGAP_Int)chr,0L);
}


/****************************************************************************
**
*F  PrintPermExpr(<expr>) . . . . . . . . . .  print a permutation expression
**
**  'PrintPermExpr' prints the permutation expression <expr>.
*/
void            libGAP_PrintPermExpr (
    libGAP_Expr                expr )
{
    libGAP_Expr                cycle;          /* one cycle of permutation expr.  */
    libGAP_UInt                i, j;           /* loop variables                  */

    /* if there are no cycles, print the identity permutation              */
    if ( libGAP_SIZE_EXPR(expr) == 0 ) {
        libGAP_Pr("()",0L,0L);
    }
    
    /* print all cycles                                                    */
    for ( i = 1; i <= libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr); i++ ) {
        cycle = libGAP_ADDR_EXPR(expr)[i-1];
        libGAP_Pr("%>(",0L,0L);

        /* print all entries of that cycle                                 */
        for ( j = 1; j <= libGAP_SIZE_EXPR(cycle)/sizeof(libGAP_Expr); j++ ) {
            libGAP_Pr("%>",0L,0L);
            libGAP_PrintExpr( libGAP_ADDR_EXPR(cycle)[j-1] );
            libGAP_Pr("%<",0L,0L);
            if ( j < libGAP_SIZE_EXPR(cycle)/sizeof(libGAP_Expr) )  libGAP_Pr(",",0L,0L);
        }

        libGAP_Pr("%<)",0L,0L);
    }
}


/****************************************************************************
**
*F  PrintListExpr(<expr>) . . . . . . . . . . . . . . print a list expression
**
**  'PrintListExpr' prints the list expression <expr>.
*/
void            libGAP_PrintListExpr (
    libGAP_Expr                expr )
{
    libGAP_Int                 len;            /* logical length of <list>        */
    libGAP_Expr                elm;            /* one element from <list>         */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the logical length of the list                                  */
    len = libGAP_SIZE_EXPR( expr ) / sizeof(libGAP_Expr);

    /* loop over the entries                                               */
    libGAP_Pr("%2>[ %2>",0L,0L);
    for ( i = 1;  i <= len;  i++ ) {
        elm = libGAP_ADDR_EXPR(expr)[i-1];
        if ( elm != 0 ) {
            if ( 1 < i )  libGAP_Pr("%<,%< %2>",0L,0L);
            libGAP_PrintExpr( elm );
        }
        else {
            if ( 1 < i )  libGAP_Pr("%2<,%2>",0L,0L);
        }
    }
    libGAP_Pr(" %4<]",0L,0L);
}


/****************************************************************************
**
*F  PrintRangeExpr(<expr>)  . . . . . . . . . . . . .  print range expression
**
**  'PrintRangeExpr' prints the record expression <expr>.
*/
void            libGAP_PrintRangeExpr (
    libGAP_Expr                expr )
{
    if ( libGAP_SIZE_EXPR( expr ) == 2*sizeof(libGAP_Expr) ) {
        libGAP_Pr("%2>[ %2>",0L,0L);    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
        libGAP_Pr("%2< .. %2>",0L,0L);  libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
        libGAP_Pr(" %4<]",0L,0L);
    }
    else {
        libGAP_Pr("%2>[ %2>",0L,0L);    libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[0] );
        libGAP_Pr("%<,%< %2>",0L,0L);   libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[1] );
        libGAP_Pr("%2< .. %2>",0L,0L);  libGAP_PrintExpr( libGAP_ADDR_EXPR(expr)[2] );
        libGAP_Pr(" %4<]",0L,0L);
    }
}


/****************************************************************************
**
*F  PrintStringExpr(<expr>) . . . . . . . . . . . . print a string expression
**
**  'PrintStringExpr' prints the string expression <expr>.
*/
void            libGAP_PrintStringExpr (
    libGAP_Expr                expr )
{
    libGAP_PrintString(libGAP_EvalStringExpr(expr));
    /*Pr( "\"%S\"", (Int)ADDR_EXPR(expr), 0L );*/
}

/****************************************************************************
**
*F  PrintFloatExpr(<expr>) . . . . . . . . . . . . print a float expression
**
**  'PrintFloatExpr' prints the float expression <expr>.
*/
void            libGAP_PrintFloatExprLazy (
    libGAP_Expr                expr )
{
  libGAP_Pr("%s", (libGAP_Int)(((char *)libGAP_ADDR_EXPR(expr) + 2*sizeof(libGAP_UInt))), 0L);
}

/****************************************************************************
**
*F  PrintFloatExprEager(<expr>) . . . . . . . . . . . . print a float expression
**
**  'PrintFloatExpr' prints the float expression <expr>.
*/
void            libGAP_PrintFloatExprEager (
    libGAP_Expr                expr )
{
  libGAP_Char mark;
  libGAP_Pr("%s", (libGAP_Int)(((char *)libGAP_ADDR_EXPR(expr) + 3*sizeof(libGAP_UInt))), 0L);
  libGAP_Pr("_",0L,0L);
  mark = (libGAP_Char)(((libGAP_UInt *)libGAP_ADDR_EXPR(expr))[2]);
  if (mark != '\0') {
    libGAP_Pr("%c",mark,0L);
  }
}


/****************************************************************************
**
*F  PrintRecExpr(<expr>)  . . . . . . . . . . . . . print a record expression
**
**  'PrintRecExpr' the record expression <expr>.
*/
void            libGAP_PrintRecExpr1 (
    libGAP_Expr                expr )
{
  libGAP_Expr                tmp;            /* temporary variable              */
  libGAP_UInt                i;              /* loop variable                   */
  
  for ( i = 1; i <= libGAP_SIZE_EXPR(expr)/(2*sizeof(libGAP_Expr)); i++ ) {
        /* print an ordinary record name                                   */
        tmp = libGAP_ADDR_EXPR(expr)[2*i-2];
        if ( libGAP_IS_INTEXPR(tmp) ) {
            libGAP_Pr( "%I", (libGAP_Int)libGAP_NAME_RNAM( libGAP_INT_INTEXPR(tmp) ), 0L );
        }

        /* print an evaluating record name                                 */
        else {
            libGAP_Pr(" (",0L,0L);
            libGAP_PrintExpr( tmp );
            libGAP_Pr(")",0L,0L);
        }

        /* print the component                                             */
        tmp = libGAP_ADDR_EXPR(expr)[2*i-1];
        libGAP_Pr("%< := %>",0L,0L);
        libGAP_PrintExpr( tmp );
        if ( i < libGAP_SIZE_EXPR(expr)/(2*sizeof(libGAP_Expr)) )
            libGAP_Pr("%2<,\n%2>",0L,0L);

    }
}

void            libGAP_PrintRecExpr (
    libGAP_Expr                expr )
{
    libGAP_Pr("%2>rec(\n%2>",0L,0L);
    libGAP_PrintRecExpr1(expr);
    libGAP_Pr(" %4<)",0L,0L);

}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                type;           /* loop variable                   */

    libGAP_InitFopyGVar("CONVERT_FLOAT_LITERAL",&libGAP_CONVERT_FLOAT_LITERAL);
    libGAP_InitCopyGVar("FLOAT_LITERAL_CACHE",&libGAP_FLOAT_LITERAL_CACHE);
    libGAP_InitCopyGVar("EAGER_FLOAT_LITERAL_CACHE",&libGAP_EAGER_FLOAT_LITERAL_CACHE);
    libGAP_InitCopyGVar("MAX_FLOAT_LITERAL_CACHE_SIZE",&libGAP_MAX_FLOAT_LITERAL_CACHE_SIZE);

    
    /* clear the evaluation dispatch table                                 */
    for ( type = 0; type < 256; type++ ) {
        libGAP_EvalExprFuncs[ type ] = libGAP_EvalUnknownExpr;
        libGAP_EvalBoolFuncs[ type ] = libGAP_EvalUnknownBool;
    }

    /* install the evaluators for logical operations                       */
    libGAP_EvalExprFuncs [ libGAP_T_OR             ] = libGAP_EvalOr;   
    libGAP_EvalExprFuncs [ libGAP_T_AND            ] = libGAP_EvalAnd;  
    libGAP_EvalExprFuncs [ libGAP_T_NOT            ] = libGAP_EvalNot;  

    /* the logical operations are guaranteed to return booleans            */
    libGAP_EvalBoolFuncs [ libGAP_T_OR             ] = libGAP_EvalOr;
    libGAP_EvalBoolFuncs [ libGAP_T_AND            ] = libGAP_EvalAnd;
    libGAP_EvalBoolFuncs [ libGAP_T_NOT            ] = libGAP_EvalNot;

    /* install the evaluators for comparison operations                    */
    libGAP_EvalExprFuncs [ libGAP_T_EQ             ] = libGAP_EvalEq;   
    libGAP_EvalExprFuncs [ libGAP_T_NE             ] = libGAP_EvalNe;   
    libGAP_EvalExprFuncs [ libGAP_T_LT             ] = libGAP_EvalLt;   
    libGAP_EvalExprFuncs [ libGAP_T_GE             ] = libGAP_EvalGe;   
    libGAP_EvalExprFuncs [ libGAP_T_GT             ] = libGAP_EvalGt;   
    libGAP_EvalExprFuncs [ libGAP_T_LE             ] = libGAP_EvalLe;   
    libGAP_EvalExprFuncs [ libGAP_T_IN             ] = libGAP_EvalIn;     

    /* the comparison operations are guaranteed to return booleans         */
    libGAP_EvalBoolFuncs [ libGAP_T_EQ             ] = libGAP_EvalEq;
    libGAP_EvalBoolFuncs [ libGAP_T_NE             ] = libGAP_EvalNe;
    libGAP_EvalBoolFuncs [ libGAP_T_LT             ] = libGAP_EvalLt;
    libGAP_EvalBoolFuncs [ libGAP_T_GE             ] = libGAP_EvalGe;
    libGAP_EvalBoolFuncs [ libGAP_T_GT             ] = libGAP_EvalGt;
    libGAP_EvalBoolFuncs [ libGAP_T_LE             ] = libGAP_EvalLe;
    libGAP_EvalBoolFuncs [ libGAP_T_IN             ] = libGAP_EvalIn;

    /* install the evaluators for binary operations                        */
    libGAP_EvalExprFuncs [ libGAP_T_SUM            ] = libGAP_EvalSum;
    libGAP_EvalExprFuncs [ libGAP_T_AINV           ] = libGAP_EvalAInv;
    libGAP_EvalExprFuncs [ libGAP_T_DIFF           ] = libGAP_EvalDiff;
    libGAP_EvalExprFuncs [ libGAP_T_PROD           ] = libGAP_EvalProd;
    libGAP_EvalExprFuncs [ libGAP_T_INV            ] = libGAP_EvalInv;
    libGAP_EvalExprFuncs [ libGAP_T_QUO            ] = libGAP_EvalQuo;
    libGAP_EvalExprFuncs [ libGAP_T_MOD            ] = libGAP_EvalMod;
    libGAP_EvalExprFuncs [ libGAP_T_POW            ] = libGAP_EvalPow;

    /* install the evaluators for literal expressions                      */
    libGAP_EvalExprFuncs [ libGAP_T_INT_EXPR       ] = libGAP_EvalIntExpr;
    libGAP_EvalExprFuncs [ libGAP_T_TRUE_EXPR      ] = libGAP_EvalTrueExpr;
    libGAP_EvalExprFuncs [ libGAP_T_FALSE_EXPR     ] = libGAP_EvalFalseExpr;
    libGAP_EvalExprFuncs [ libGAP_T_CHAR_EXPR      ] = libGAP_EvalCharExpr;
    libGAP_EvalExprFuncs [ libGAP_T_PERM_EXPR      ] = libGAP_EvalPermExpr;

    /* install the evaluators for list and record expressions              */
    libGAP_EvalExprFuncs [ libGAP_T_LIST_EXPR      ] = libGAP_EvalListExpr;
    libGAP_EvalExprFuncs [ libGAP_T_LIST_TILD_EXPR ] = libGAP_EvalListTildeExpr;
    libGAP_EvalExprFuncs [ libGAP_T_RANGE_EXPR     ] = libGAP_EvalRangeExpr;
    libGAP_EvalExprFuncs [ libGAP_T_STRING_EXPR    ] = libGAP_EvalStringExpr;
    libGAP_EvalExprFuncs [ libGAP_T_REC_EXPR       ] = libGAP_EvalRecExpr;
    libGAP_EvalExprFuncs [ libGAP_T_REC_TILD_EXPR  ] = libGAP_EvalRecTildeExpr;
    libGAP_EvalExprFuncs [ libGAP_T_FLOAT_EXPR_LAZY  ] = libGAP_EvalFloatExprLazy;
    libGAP_EvalExprFuncs [ libGAP_T_FLOAT_EXPR_EAGER  ] = libGAP_EvalFloatExprEager;

    /* clear the tables for the printing dispatching                       */
    for ( type = 0; type < 256; type++ ) {
        libGAP_PrintExprFuncs[ type ] = libGAP_PrintUnknownExpr;
    }

    /* install the printers for logical operations                         */
    libGAP_PrintExprFuncs[ libGAP_T_OR             ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_AND            ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_NOT            ] = libGAP_PrintNot;

    /* install the printers for comparison operations                      */
    libGAP_PrintExprFuncs[ libGAP_T_EQ             ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_LT             ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_NE             ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_GE             ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_GT             ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_LE             ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_IN             ] = libGAP_PrintBinop;

    /* install the printers for binary operations                          */
    libGAP_PrintExprFuncs[ libGAP_T_SUM            ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_AINV           ] = libGAP_PrintAInv;
    libGAP_PrintExprFuncs[ libGAP_T_DIFF           ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_PROD           ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_INV            ] = libGAP_PrintInv;
    libGAP_PrintExprFuncs[ libGAP_T_QUO            ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_MOD            ] = libGAP_PrintBinop;
    libGAP_PrintExprFuncs[ libGAP_T_POW            ] = libGAP_PrintBinop;

    /* install the printers for literal expressions                        */
    libGAP_PrintExprFuncs[ libGAP_T_INTEXPR        ] = libGAP_PrintIntExpr;
    libGAP_PrintExprFuncs[ libGAP_T_INT_EXPR       ] = libGAP_PrintIntExpr;
    libGAP_PrintExprFuncs[ libGAP_T_TRUE_EXPR      ] = libGAP_PrintTrueExpr;
    libGAP_PrintExprFuncs[ libGAP_T_FALSE_EXPR     ] = libGAP_PrintFalseExpr;
    libGAP_PrintExprFuncs[ libGAP_T_CHAR_EXPR      ] = libGAP_PrintCharExpr;
    libGAP_PrintExprFuncs[ libGAP_T_PERM_EXPR      ] = libGAP_PrintPermExpr;

    /* install the printers for list and record expressions                */
    libGAP_PrintExprFuncs[ libGAP_T_LIST_EXPR      ] = libGAP_PrintListExpr;
    libGAP_PrintExprFuncs[ libGAP_T_LIST_TILD_EXPR ] = libGAP_PrintListExpr;
    libGAP_PrintExprFuncs[ libGAP_T_RANGE_EXPR     ] = libGAP_PrintRangeExpr;
    libGAP_PrintExprFuncs[ libGAP_T_STRING_EXPR    ] = libGAP_PrintStringExpr;
    libGAP_PrintExprFuncs[ libGAP_T_FLOAT_EXPR_LAZY    ] = libGAP_PrintFloatExprLazy;
    libGAP_PrintExprFuncs[ libGAP_T_FLOAT_EXPR_EAGER    ] = libGAP_PrintFloatExprEager;
    libGAP_PrintExprFuncs[ libGAP_T_REC_EXPR       ] = libGAP_PrintRecExpr;
    libGAP_PrintExprFuncs[ libGAP_T_REC_TILD_EXPR  ] = libGAP_PrintRecExpr;

    /* return success                                                      */
    return 0;
}


static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_GVAR_FLOAT_LITERAL_CACHE = libGAP_GVarName("FLOAT_LITERAL_CACHE");
    return 0;
}

/****************************************************************************
**
*F  InitInfoExprs() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "exprs",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_InitLibrary                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoExprs ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  exprs.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
