/****************************************************************************
**
*W  vector.c                    GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions  that mainly  operate  on vectors  whose
**  elements are integers, rationals, or elements from cyclotomic fields.  As
**  vectors are special lists many things are done in the list package.
**
**  A *vector* is a list that has no holes,  and whose elements all come from
**  a common field.  For the full definition of vectors see chapter "Vectors"
**  in  the {\GAP} manual.   Read also about "More   about Vectors" about the
**  vector flag and the compact representation of vectors over finite fields.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "ariths.h"              /* basic arithmetic                */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "integer.h"             /* integers                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listoper.h"            /* operations for generic lists    */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "vector.h"              /* functions for plain vectors     */

#include        "range.h"               /* ranges                          */

#include <assert.h>

#define libGAP_IS_IMM_PLIST(list)  ((libGAP_TNUM_OBJ(list) - libGAP_T_PLIST) % 2)

/****************************************************************************
**
*F  SumIntVector(<elmL>,<vecR>) . . . . . . .  sum of an integer and a vector
**
**  'SumIntVector' returns the   sum of the   integer <elmL>  and  the vector
**  <vecR>.  The sum is a  list, where each element is  the sum of <elmL> and
**  the corresponding element of <vecR>.
**
**  'SumIntVector' is an improved version  of  'SumSclList', which  does  not
**  call 'SUM' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_SumIntVector (
    libGAP_Obj                 elmL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecS;           /* handle of the sum               */
    libGAP_Obj *               ptrS;           /* pointer into the sum            */
    libGAP_Obj                 elmS;           /* one element of sum list         */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecR);
    vecS = libGAP_NEW_PLIST(libGAP_TNUM_OBJ(vecR), len);
    libGAP_SET_LEN_PLIST(vecS, len);

    /* loop over the elements and add                                      */
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrS = libGAP_ADDR_OBJ(vecS);
    for (i = 1; i <= len; i++) {
        elmR = ptrR[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_SUM_INTOBJS(elmS, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecS);
            elmS = libGAP_SUM(elmL, elmR);
            ptrR = libGAP_ADDR_OBJ(vecR);
            ptrS = libGAP_ADDR_OBJ(vecS);
        }
        ptrS[i] = elmS;
    }

    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecS);
    return vecS;
}


/****************************************************************************
**
*F  SumVectorInt(<vecL>,<elmR>) . . . . . . .  sum of a vector and an integer
**
**  'SumVectorInt' returns  the sum of   the  vector <vecL> and  the  integer
**  <elmR>.  The sum is a  list, where each element  is the sum of <elmR> and
**  the corresponding element of <vecL>.
**
**  'SumVectorInt' is an improved version  of  'SumListScl', which  does  not
**  call 'SUM' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_SumVectorInt (
    libGAP_Obj                 vecL,
    libGAP_Obj                 elmR )
{
    libGAP_Obj                 vecS;           /* handle of the sum               */
    libGAP_Obj *               ptrS;           /* pointer into the sum            */
    libGAP_Obj                 elmS;           /* one element of sum list         */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecL);
    vecS = libGAP_NEW_PLIST(libGAP_TNUM_OBJ(vecL), len);
    libGAP_SET_LEN_PLIST(vecS, len);

    /* loop over the elements and add                                      */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrS = libGAP_ADDR_OBJ(vecS);
    for (i = 1; i <= len; i++) {
        elmL = ptrL[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_SUM_INTOBJS(elmS, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecS);
            elmS = libGAP_SUM(elmL, elmR);
            ptrL = libGAP_ADDR_OBJ(vecL);
            ptrS = libGAP_ADDR_OBJ(vecS);
        }
        ptrS[i] = elmS;
    }

    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecS);
    return vecS;
}


/****************************************************************************
**
*F  SumVectorVector(<vecL>,<vecR>)  . . . . . . . . . . .  sum of two vectors
**
**  'SumVectorVector' returns the sum  of the two  vectors <vecL> and <vecR>.
**  The sum is a new list, where each element is the sum of the corresponding
**  elements of <vecL> and <vecR>.
**
**  'SumVectorVector' is an improved version of 'SumListList', which does not
**  call 'SUM' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_SumVectorVector (
    libGAP_Obj                 vecL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecS;           /* handle of the sum               */
    libGAP_Obj *               ptrS;           /* pointer into the sum            */
    libGAP_Obj                 elmS;           /* one element of sum list         */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_UInt                lenL, lenR, len, lenmin; /* lengths                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    lenL = libGAP_LEN_PLIST(vecL);
    lenR = libGAP_LEN_PLIST(vecR);
    if (lenL < lenR) {
        lenmin = lenL;
        len = lenR;
    } else {
        lenmin = lenR;
        len = lenL;
    }
    vecS = libGAP_NEW_PLIST((libGAP_IS_IMM_PLIST(vecL) && libGAP_IS_IMM_PLIST(vecR)) ?
                      libGAP_T_PLIST_CYC + libGAP_IMMUTABLE : libGAP_T_PLIST_CYC, len);
    libGAP_SET_LEN_PLIST(vecS, len);

    /* loop over the elements and add                                      */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrS = libGAP_ADDR_OBJ(vecS);
    for (i = 1; i <= lenmin; i++) {
        elmL = ptrL[i];
        elmR = ptrR[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_SUM_INTOBJS(elmS, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecS);
            elmS = libGAP_SUM(elmL, elmR);
            ptrL = libGAP_ADDR_OBJ(vecL);
            ptrR = libGAP_ADDR_OBJ(vecR);
            ptrS = libGAP_ADDR_OBJ(vecS);
        }
        ptrS[i] = elmS;
    }
    if (lenL < lenR)
        for (; i <= lenR; i++) {
            ptrS[i] = ptrR[i];
        }
    else
        for (; i <= lenL; i++) {
            ptrS[i] = ptrL[i];
        }
    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecS);
    return vecS;
}


/****************************************************************************
**
*F  DiffIntVector(<elmL>,<vecR>)  . . . difference of an integer and a vector
**
**  'DiffIntVector' returns  the difference  of  the integer  <elmL> and  the
**  vector <vecR>.   The difference  is  a list,  where  each element is  the
**  difference of <elmL> and the corresponding element of <vecR>.
**
**  'DiffIntVector'  is an improved  version of 'DiffSclList', which does not
**  call 'DIFF' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_DiffIntVector (
    libGAP_Obj                 elmL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecD;           /* handle of the difference        */
    libGAP_Obj *               ptrD;           /* pointer into the difference     */
    libGAP_Obj                 elmD;           /* one element of difference list  */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecR);
    vecD = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecR) ?
                     libGAP_T_PLIST_CYC : libGAP_T_PLIST_CYC + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecD, len);

    /* loop over the elements and subtract                                 */
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrD = libGAP_ADDR_OBJ(vecD);
    for (i = 1; i <= len; i++) {
        elmR = ptrR[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_DIFF_INTOBJS(elmD, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecD);
            elmD = libGAP_DIFF(elmL, elmR);
            ptrR = libGAP_ADDR_OBJ(vecR);
            ptrD = libGAP_ADDR_OBJ(vecD);
        }
        ptrD[i] = elmD;
    }

    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecD);
    return vecD;
}


/****************************************************************************
**
*F  DiffVectorInt(<vecL>,<elmR>)  . . . difference of a vector and an integer
**
**  'DiffVectorInt' returns   the  difference of the  vector  <vecL>  and the
**  integer <elmR>.  The difference   is a list,   where each element  is the
**  difference of <elmR> and the corresponding element of <vecL>.
**
**  'DiffVectorInt' is an improved  version of 'DiffListScl', which  does not
**  call 'DIFF' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_DiffVectorInt (
    libGAP_Obj                 vecL,
    libGAP_Obj                 elmR )
{
    libGAP_Obj                 vecD;           /* handle of the difference        */
    libGAP_Obj *               ptrD;           /* pointer into the difference     */
    libGAP_Obj                 elmD;           /* one element of difference list  */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecL);
    vecD = libGAP_NEW_PLIST(libGAP_TNUM_OBJ(vecL), len);
    libGAP_SET_LEN_PLIST(vecD, len);

    /* loop over the elements and subtract                                 */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrD = libGAP_ADDR_OBJ(vecD);
    for (i = 1; i <= len; i++) {
        elmL = ptrL[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_DIFF_INTOBJS(elmD, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecD);
            elmD = libGAP_DIFF(elmL, elmR);
            ptrL = libGAP_ADDR_OBJ(vecL);
            ptrD = libGAP_ADDR_OBJ(vecD);
        }
        ptrD[i] = elmD;
    }

    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecD);
    return vecD;
}


/****************************************************************************
**
*F  DiffVectorVector(<vecL>,<vecR>) . . . . . . . . difference of two vectors
**
**  'DiffVectorVector'  returns the difference of the  two vectors <vecL> and
**  <vecR>.   The  difference is   a new   list, where  each  element  is the
**  difference of the corresponding elements of <vecL> and <vecR>.
**
**  'DiffVectorVector' is an improved  version of  'DiffListList', which does
**  not call 'DIFF' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_DiffVectorVector (
    libGAP_Obj                 vecL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecD;           /* handle of the sum               */
    libGAP_Obj *               ptrD;           /* pointer into the sum            */
    libGAP_Obj                 elmD;           /* one element of sum list         */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_UInt                lenL, lenR, len, lenmin; /* lengths                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    lenL = libGAP_LEN_PLIST(vecL);
    lenR = libGAP_LEN_PLIST(vecR);
    if (lenL < lenR) {
        lenmin = lenL;
        len = lenR;
    } else {
        lenmin = lenR;
        len = lenL;
    }
    vecD = libGAP_NEW_PLIST((libGAP_IS_IMM_PLIST(vecL) && libGAP_IS_IMM_PLIST(vecR)) ?
    libGAP_T_PLIST_CYC + libGAP_IMMUTABLE : libGAP_T_PLIST_CYC, len);
    libGAP_SET_LEN_PLIST(vecD, len);

    /* loop over the elements and subtract                                   */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrD = libGAP_ADDR_OBJ(vecD);
    for (i = 1; i <= lenmin; i++) {
        elmL = ptrL[i];
        elmR = ptrR[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_DIFF_INTOBJS(elmD, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecD);
            elmD = libGAP_DIFF(elmL, elmR);
            ptrL = libGAP_ADDR_OBJ(vecL);
            ptrR = libGAP_ADDR_OBJ(vecR);
            ptrD = libGAP_ADDR_OBJ(vecD);
        }
        ptrD[i] = elmD;
    }
    if (lenL < lenR)
        for (; i <= lenR; i++) {
            elmR = ptrR[i];
            if (! libGAP_IS_INTOBJ(elmR) || ! libGAP_DIFF_INTOBJS(elmD, libGAP_INTOBJ_INT(0), elmR)) {
                libGAP_CHANGED_BAG(vecD);
                elmD = libGAP_AINV(elmR);
                ptrR = libGAP_ADDR_OBJ(vecR);
                ptrD = libGAP_ADDR_OBJ(vecD);
            }
            ptrD[i] = elmD;
        }
    else
        for (; i <= lenL; i++) {
            ptrD[i] = ptrL[i];
        }
    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecD);
    return vecD;
}


/****************************************************************************
**
*F  ProdIntVector(<elmL>,<vecR>)  . . . .  product of an integer and a vector
**
**  'ProdIntVector' returns the product of the integer  <elmL> and the vector
**  <vecR>.  The product is  the list, where  each element is the product  of
**  <elmL> and the corresponding entry of <vecR>.
**
**  'ProdIntVector'  is an  improved version of 'ProdSclList', which does not
**  call 'PROD' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_ProdIntVector (
    libGAP_Obj                 elmL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 vecP;           /* handle of the product           */
    libGAP_Obj *               ptrP;           /* pointer into the product        */
    libGAP_Obj                 elmP;           /* one element of product list     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecR);
    vecP = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecR) ?
                     libGAP_T_PLIST_CYC : libGAP_T_PLIST_CYC + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecP, len);

    /* loop over the entries and multiply                                  */
    ptrR = libGAP_ADDR_OBJ(vecR);
    ptrP = libGAP_ADDR_OBJ(vecP);
    for (i = 1; i <= len; i++) {
        elmR = ptrR[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_PROD_INTOBJS(elmP, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecP);
            elmP = libGAP_PROD(elmL, elmR);
            ptrR = libGAP_ADDR_OBJ(vecR);
            ptrP = libGAP_ADDR_OBJ(vecP);
        }
        ptrP[i] = elmP;
    }

    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecP);
    return vecP;
}


/****************************************************************************
**
*F  ProdVectorInt(<vecL>,<elmR>)  . . . .  product of a scalar and an integer
**
**  'ProdVectorInt' returns the product of the integer  <elmR> and the vector
**  <vecL>.  The  product is the  list, where each element  is the product of
**  <elmR> and the corresponding element of <vecL>.
**
**  'ProdVectorInt'  is an  improved version of 'ProdSclList', which does not
**  call 'PROD' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_ProdVectorInt (
    libGAP_Obj                 vecL,
    libGAP_Obj                 elmR )
{
    libGAP_Obj                 vecP;           /* handle of the product           */
    libGAP_Obj *               ptrP;           /* pointer into the product        */
    libGAP_Obj                 elmP;           /* one element of product list     */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* make the result list                                                */
    len = libGAP_LEN_PLIST(vecL);
    vecP = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vecL) ?
                     libGAP_T_PLIST_CYC : libGAP_T_PLIST_CYC + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(vecP, len);

    /* loop over the entries and multiply                                  */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrP = libGAP_ADDR_OBJ(vecP);
    for (i = 1; i <= len; i++) {
        elmL = ptrL[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_PROD_INTOBJS(elmP, elmL, elmR)) {
            libGAP_CHANGED_BAG(vecP);
            elmP = libGAP_PROD(elmL, elmR);
            ptrL = libGAP_ADDR_OBJ(vecL);
            ptrP = libGAP_ADDR_OBJ(vecP);
        }
        ptrP[i] = elmP;
    }

    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecP);
    return vecP;
}


/****************************************************************************
**
*F  ProdVectorVector(<vecL>,<vecR>) . . . . . . . . .  product of two vectors
**
**  'ProdVectorVector'  returns the product  of   the two vectors <vecL>  and
**  <vecR>.  The  product  is the  sum of the   products of the corresponding
**  elements of the two lists.
**
**  'ProdVectorVector' is an improved version  of 'ProdListList',  which does
**  not call 'PROD' if the operands are immediate integers.
*/
libGAP_Obj             libGAP_ProdVectorVector (
    libGAP_Obj                 vecL,
    libGAP_Obj                 vecR )
{
    libGAP_Obj                 elmP;           /* product, result                 */
    libGAP_Obj                 elmS;           /* partial sum of result           */
    libGAP_Obj                 elmT;           /* one summand of result           */
    libGAP_Obj *               ptrL;           /* pointer into the left operand   */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_Obj *               ptrR;           /* pointer into the right operand  */
    libGAP_Obj                 elmR;           /* one element of right operand    */
    libGAP_UInt                lenL, lenR, len; /* length                          */
    libGAP_UInt                i;              /* loop variable                   */

    /* check that the lengths agree                                        */
    lenL = libGAP_LEN_PLIST(vecL);
    lenR = libGAP_LEN_PLIST(vecR);
    len = (lenL < lenR) ? lenL : lenR;

    /* loop over the entries and multiply                                  */
    ptrL = libGAP_ADDR_OBJ(vecL);
    ptrR = libGAP_ADDR_OBJ(vecR);
    elmL = ptrL[1];
    elmR = ptrR[1];
    if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_PROD_INTOBJS(elmT, elmL, elmR)) {
        elmT = libGAP_PROD(elmL, elmR);
        ptrL = libGAP_ADDR_OBJ(vecL);
        ptrR = libGAP_ADDR_OBJ(vecR);
    }
    elmP = elmT;
    for (i = 2; i <= len; i++) {
        elmL = ptrL[i];
        elmR = ptrR[i];
        if (! libGAP_ARE_INTOBJS(elmL, elmR) || ! libGAP_PROD_INTOBJS(elmT, elmL, elmR)) {
            elmT = libGAP_PROD(elmL, elmR);
            ptrL = libGAP_ADDR_OBJ(vecL);
            ptrR = libGAP_ADDR_OBJ(vecR);
        }
        if (! libGAP_ARE_INTOBJS(elmP, elmT) || ! libGAP_SUM_INTOBJS(elmS, elmP, elmT)) {
            elmS = libGAP_SUM(elmP, elmT);
            ptrL = libGAP_ADDR_OBJ(vecL);
            ptrR = libGAP_ADDR_OBJ(vecR);
        }
        elmP = elmS;
    }

    /* return the result                                                   */
    return elmP;
}


/****************************************************************************
**
*F  ProdVectorMatrix(<vecL>,<vecR>) . . . .  product of a vector and a matrix
**
**  'ProdVectorMatrix' returns the product of the vector <vecL> and the matrix
**  <vecR>.  The product is the sum of the  rows  of <vecR>, each multiplied by
**  the corresponding entry of <vecL>.
**
**  'ProdVectorMatrix'  is an improved version of 'ProdListList',  which does
**  not  call 'PROD' and  also accumulates  the sum into  one  fixed  vector
**  instead of allocating a new for each product and sum.
**
**  We now need to supply a handler for this and install it as a library method,
**  
*/
libGAP_Obj             libGAP_ProdVectorMatrix (
    libGAP_Obj                 vecL,
    libGAP_Obj                 matR )
{
    libGAP_Obj                 vecP;           /* handle of the product           */
    libGAP_Obj *               ptrP;           /* pointer into the product        */
    libGAP_Obj                 elmP;           /* one summand of product          */
    libGAP_Obj                 elmS;           /* temporary for sum               */
    libGAP_Obj                 elmT;           /* another temporary               */
    libGAP_Obj                 elmL;           /* one element of left operand     */
    libGAP_Obj                 vecR;           /* one vector of right operand     */
    libGAP_Obj *               ptrR;           /* pointer into the right vector   */
    libGAP_Obj                 elmR;           /* one element from right vector   */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                col;            /* length of the rows              */
    libGAP_UInt                i, k;           /* loop variables                  */

    /* check the lengths                                                   */
    len = libGAP_LEN_PLIST(vecL);
    if (len > libGAP_LEN_PLIST(matR))
        len = libGAP_LEN_PLIST(matR);
    col = libGAP_LEN_PLIST(libGAP_ELM_PLIST(matR, 1));

    /* make the result list */

    vecP = libGAP_NEW_PLIST((libGAP_IS_MUTABLE_OBJ(vecL) || libGAP_IS_MUTABLE_OBJ(libGAP_ELM_PLIST(matR, 1))) ?
                     libGAP_T_PLIST_CYC : libGAP_T_PLIST_CYC + libGAP_IMMUTABLE,
    col);
    libGAP_SET_LEN_PLIST(vecP, col);
    for (i = 1; i <= col; i++)
        libGAP_SET_ELM_PLIST(vecP, i, libGAP_INTOBJ_INT(0));


    /* loop over the entries and multiply                            */
    for (i = 1; i <= len; i++) {
        elmL = libGAP_ELM_PLIST(vecL, i);
        vecR = libGAP_ELM_PLIST(matR, i);
        ptrR = libGAP_ADDR_OBJ(vecR);
        ptrP = libGAP_ADDR_OBJ(vecP);
        if (elmL == libGAP_INTOBJ_INT(1L)) {
            for (k = 1; k <= col; k++) {
                elmT = ptrR[k];
                elmP = ptrP[k];
                if (! libGAP_ARE_INTOBJS(elmP, elmT)
                || ! libGAP_SUM_INTOBJS(elmS, elmP, elmT)) {
                    libGAP_CHANGED_BAG(vecP);
                    elmS = libGAP_SUM(elmP, elmT);
                    ptrR = libGAP_ADDR_OBJ(vecR);
                    ptrP = libGAP_ADDR_OBJ(vecP);
                }
                ptrP[k] = elmS;
            }
        } else if (elmL == libGAP_INTOBJ_INT(-1L)) {
            for (k = 1; k <= col; k++) {
                elmT = ptrR[k];
                elmP = ptrP[k];
                if (! libGAP_ARE_INTOBJS(elmP, elmT)
                        || ! libGAP_DIFF_INTOBJS(elmS, elmP, elmT)) {
                    libGAP_CHANGED_BAG(vecP);
                    elmS = libGAP_DIFF(elmP, elmT);
                    ptrR = libGAP_ADDR_OBJ(vecR);
                    ptrP = libGAP_ADDR_OBJ(vecP);
                }
                ptrP[k] = elmS;

            }
        } else if (elmL != libGAP_INTOBJ_INT(0L)) {
            for (k = 1; k <= col; k++) {
                elmR = ptrR[k];
                if (elmR != libGAP_INTOBJ_INT(0L)) {
                    if (! libGAP_ARE_INTOBJS(elmL, elmR)
                            || ! libGAP_PROD_INTOBJS(elmT, elmL, elmR)) {
                        libGAP_CHANGED_BAG(vecP);
                        elmT = libGAP_PROD(elmL, elmR);
                        ptrR = libGAP_ADDR_OBJ(vecR);
                        ptrP = libGAP_ADDR_OBJ(vecP);
                    }
                    elmP = ptrP[k];
                    if (! libGAP_ARE_INTOBJS(elmP, elmT)
                            || ! libGAP_SUM_INTOBJS(elmS, elmP, elmT)) {
                        libGAP_CHANGED_BAG(vecP);
                        elmS = libGAP_SUM(elmP, elmT);
                        ptrR = libGAP_ADDR_OBJ(vecR);
                        ptrP = libGAP_ADDR_OBJ(vecP);
                    }
                    ptrP[k] = elmS;
                }
            }
        }
    }

    /* return the result                                                   */
    libGAP_CHANGED_BAG(vecP);
    return vecP;
}

libGAP_Obj libGAP_FuncPROD_VECTOR_MATRIX(libGAP_Obj self, libGAP_Obj vec, libGAP_Obj mat)
{
    return libGAP_ProdVectorMatrix(vec, mat);
}

/****************************************************************************
**
*F  ZeroVector(<vec>) . . . .  zero of a cyclotomicVector
**
**  'ZeroVector' returns the zero of the vector <vec>.
**
**  It is a better version of ZeroListDefault for the case of cyclotomic
**  vectors, becuase it knows what the cyclotomic zero is.
*/

libGAP_Obj libGAP_ZeroVector( libGAP_Obj vec )
{
    libGAP_UInt i, len;
    libGAP_Obj res;
    assert(libGAP_TNUM_OBJ(vec) >= libGAP_T_PLIST_CYC && \
    libGAP_TNUM_OBJ(vec) <= libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE);
    len = libGAP_LEN_PLIST(vec);
    res = libGAP_NEW_PLIST(libGAP_IS_MUTABLE_OBJ(vec) ? libGAP_T_PLIST_CYC : libGAP_T_PLIST_CYC + libGAP_IMMUTABLE, len);
    libGAP_SET_LEN_PLIST(res, len);
    for (i = 1; i <= len; i++)
        libGAP_SET_ELM_PLIST(res, i, libGAP_INTOBJ_INT(0));
    return res;
}

libGAP_Obj libGAP_ZeroMutVector( libGAP_Obj vec )
{
    libGAP_UInt i, len;
    libGAP_Obj res;
    assert(libGAP_TNUM_OBJ(vec) >= libGAP_T_PLIST_CYC && \
    libGAP_TNUM_OBJ(vec) <= libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE);
    len = libGAP_LEN_PLIST(vec);
    res = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, len);
    libGAP_SET_LEN_PLIST(res, len);
    for (i = 1; i <= len; i++)
        libGAP_SET_ELM_PLIST(res, i, libGAP_INTOBJ_INT(0));
    return res;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

  { "PROD_VECTOR_MATRIX", 2, "vec, mat",
    libGAP_FuncPROD_VECTOR_MATRIX, "src/vector.c:PROD_VECTOR_MATRIX" },
  { 0 }
};


/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int                 t1;
    libGAP_Int                 t2;

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable(libGAP_GVarFuncs);

    /* install the arithmetic operation methods                            */
    for (t1 = libGAP_T_PLIST_CYC; t1 <= libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE; t1++) {
        libGAP_ZeroFuncs[ t1 ] = libGAP_ZeroVector;
        libGAP_ZeroMutFuncs[ t1 ] = libGAP_ZeroMutVector;

        for (t2 = libGAP_T_PLIST_CYC; t2 <= libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE; t2++) {
            libGAP_SumFuncs [ libGAP_T_INT     ][ t2        ] = libGAP_SumIntVector;
            libGAP_SumFuncs [ t1        ][ libGAP_T_INT     ] = libGAP_SumVectorInt;
            libGAP_SumFuncs [ t1        ][ t2        ] = libGAP_SumVectorVector;
            libGAP_DiffFuncs[ libGAP_T_INT     ][ t2        ] = libGAP_DiffIntVector;
            libGAP_DiffFuncs[ t1        ][ libGAP_T_INT     ] = libGAP_DiffVectorInt;
            libGAP_DiffFuncs[ t1        ][ t2        ] = libGAP_DiffVectorVector;
            libGAP_ProdFuncs[ libGAP_T_INT     ][ t2        ] = libGAP_ProdIntVector;
            libGAP_ProdFuncs[ t1        ][ libGAP_T_INT     ] = libGAP_ProdVectorInt;
            libGAP_ProdFuncs[ t1        ][ t2        ] = libGAP_ProdVectorVector;
        }
    }

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable(libGAP_GVarFuncs);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoVector()  . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "vector",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoVector ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  vector.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
