#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */
#include        "finfield.h"            /* finite fields and ff elements   */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "blister.h"             /* boolean lists                   */
#include        "string.h"              /* strings                         */

#include        "vecgf2.h"              /* GF2 vectors                     */

#include        "saveload.h"            /* saving and loading              */

#include        "integer.h"             /* integers                        */

#include        "vec8bit.h"             /* vectors over bigger small fields*/

#include        "code.h"                /* Needed for TakeInterrupt */
#include        "stats.h"

#include        <assert.h>

/****************************************************************************
**
*F * * * * * * * * * * * imported library variables * * * * * * * * * * * * *
*/


/****************************************************************************
**
*V  TYPE_LIST_GF2VEC  . . . . . . . . . . . . . . type of a GF2 vector object
*/
libGAP_Obj libGAP_TYPE_LIST_GF2VEC;


/****************************************************************************
**
*V  TYPE_LIST_GF2VEC_LOCKED. . . .  type of a mutable GF2 vector object
**                                          with locked representation
*/
libGAP_Obj libGAP_TYPE_LIST_GF2VEC_LOCKED;


/****************************************************************************
**
*V  TYPE_LIST_GF2VEC_IMM  . . . . . .  type of an immutable GF2 vector object
*/
libGAP_Obj libGAP_TYPE_LIST_GF2VEC_IMM;

/****************************************************************************
**
*V  TYPE_LIST_GF2VEC_IMM_LOCKED. . . .  type of an immutable GF2 vector object
**                                          with locked representation
*/
libGAP_Obj libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;


/****************************************************************************
**
*V  TYPE_LIST_GF2MAT  . . . . . . . . . . . . . . type of a GF2 matrix object
*/
libGAP_Obj libGAP_TYPE_LIST_GF2MAT;


/****************************************************************************
**
*V  TYPE_LIST_GF2MAT_IMM  . . . . . .  type of an immutable GF2 matrix object
*/
libGAP_Obj libGAP_TYPE_LIST_GF2MAT_IMM;


/****************************************************************************
**
*V  IsGF2VectorRep  . . . . . . . . . . . . . . . . . . . . . . . . .  filter
*/
libGAP_Obj libGAP_IsGF2VectorRep;


/****************************************************************************
**
*V  GF2One  . . . . . . . . . . . . . . . . . . . . . . . . . . .  one of GF2
*/
libGAP_Obj libGAP_GF2One;


/****************************************************************************
**
*V  GF2Zero . . . . . . . . . . . . . . . . . . . . . . . . . . . zero of GF2
*/
libGAP_Obj libGAP_GF2Zero;


/****************************************************************************
**
*F * * * * * * * * * * * * arithmetic operations  * * * * * * * * * * * * * *
*/

static inline void libGAP_AddGF2VecToGF2Vec(
  libGAP_UInt *	ptS,
  libGAP_UInt *	ptV,
  libGAP_UInt		len)
{
  register libGAP_UInt ct;
  ct = (len+libGAP_BIPEB-1)/libGAP_BIPEB;
  while ( ct-- ) {
    *ptS++ ^= *ptV++;
  }
}

/****************************************************************************
**
*F  AddCoeffsGF2VecGF2Vec( <sum>, <vec> ) . . . . . . . .  add <vec> to <sum>
**
**  `AddCoeffsGF2VecGF2Vec' adds  the   entries of <vec>  to <sum>.    If the
**  length  are not equal the  missing entries are  assumed  to be zero.  The
**  position of the rightmost Z(2) is returned.
*/

libGAP_UInt libGAP_RightMostOneGF2Vec (
    libGAP_Obj                 vec )
{
    libGAP_UInt                len;

    len = libGAP_LEN_GF2VEC(vec);
    while ( 0 < len ) {
        if ( libGAP_BLOCK_ELM_GF2VEC(vec,len) == 0 )
	    len = libGAP_BIPEB*((len-1)/libGAP_BIPEB);
        else if ( libGAP_BLOCK_ELM_GF2VEC(vec,len) & libGAP_MASK_POS_GF2VEC(len) )
            break;
	else
	  len--;
    }
    return len;
}


libGAP_Obj libGAP_AddCoeffsGF2VecGF2Vec (
    libGAP_Obj                 sum,
    libGAP_Obj                 vec )
{
    libGAP_UInt *              ptS;
    libGAP_UInt *              ptV;
    libGAP_UInt                len;

    /* get the length                                                      */
    len = libGAP_LEN_GF2VEC(vec);
    
    /* grow <sum> is necessary                                             */
    if ( libGAP_LEN_GF2VEC(sum) < len ) {
        libGAP_ResizeBag( sum, libGAP_SIZE_PLEN_GF2VEC(len) );
        libGAP_SET_LEN_GF2VEC( sum, len );
    }

    /* add <vec> to <sum>                                                  */
    ptS = libGAP_BLOCKS_GF2VEC(sum);
    ptV = libGAP_BLOCKS_GF2VEC(vec);
    libGAP_AddGF2VecToGF2Vec(ptS, ptV, len);
    return libGAP_INTOBJ_INT(libGAP_RightMostOneGF2Vec(sum));
}




static inline libGAP_UInt libGAP_highbits( libGAP_UInt word, libGAP_UInt howmany) 
{
  return (word  >> (libGAP_BIPEB-howmany));
}

static inline libGAP_UInt libGAP_lowbits(libGAP_UInt word, libGAP_UInt howmany)
{
  return word & (((libGAP_UInt)(-1L)) >> (libGAP_BIPEB - howmany));
}

static inline libGAP_UInt libGAP_midbits(libGAP_UInt word, libGAP_UInt from, libGAP_UInt howmany)
{
  return libGAP_lowbits(libGAP_highbits(word, libGAP_BIPEB-from), howmany);
}



static inline void libGAP_setlowbits(libGAP_UInt *dest, libGAP_UInt howmany, libGAP_UInt bits) 
{
  *dest = (libGAP_highbits(*dest, libGAP_BIPEB - howmany) << howmany) | bits;
}

static inline void libGAP_sethighbits(libGAP_UInt *dest, libGAP_UInt howmany, libGAP_UInt bits) 
{
  *dest = libGAP_lowbits(*dest, libGAP_BIPEB - howmany) | (bits << (libGAP_BIPEB - howmany));
}


static inline void libGAP_setmidbits(libGAP_UInt *dest, libGAP_UInt from, libGAP_UInt howmany, libGAP_UInt bits) 
{
  libGAP_UInt mask;
  if (from + howmany == libGAP_BIPEB)
    mask = 0;
  else
    mask = ((libGAP_UInt)(-1L)) << (from  + howmany);
  if (from != 0)
    mask |= ((libGAP_UInt)(-1L)) >> (libGAP_BIPEB - from);
  *dest = (*dest & mask) | (bits << from);
}


/* This is the time critical loop for the unaligned case
   we bring it out as an inline function to mark various things as const
   and allow us to include it once for each shift which saves about a factor of 2 */

static inline void libGAP_dothework( libGAP_UInt const *sptr, libGAP_UInt *dptr, const libGAP_UInt cbits, libGAP_UInt * const dend) {
  libGAP_UInt bits;
  libGAP_UInt x = *sptr++;
  while (dptr < dend) {
    bits = x >> (libGAP_BIPEB - cbits);
    x = *sptr++;
    *dptr++ =  bits | (x  << cbits);
    }
} 
    
void libGAP_CopySection_GF2Vecs(libGAP_Obj src, libGAP_Obj dest, libGAP_UInt smin, libGAP_UInt dmin, libGAP_UInt nelts)
{
  libGAP_UInt soff;
  libGAP_UInt doff;
  libGAP_UInt *sptr;
  libGAP_UInt *dptr;
  libGAP_UInt *dend;

  if (nelts == 0) {
    return;
  }
    
  /* switch to zero-based indices and find the first blocks and so on */
  soff = (--smin) %libGAP_BIPEB;
  doff = (--dmin) %libGAP_BIPEB;
  sptr = libGAP_BLOCKS_GF2VEC(src) + smin/libGAP_BIPEB;
  dptr = libGAP_BLOCKS_GF2VEC(dest) + dmin/libGAP_BIPEB;
  
  /* deal with some short section cases */
  libGAP_UInt bits;
  /* all the section is within the starting source block */
  if (nelts <= libGAP_BIPEB -soff) {
    /* get all the section in one go */
    bits = libGAP_midbits(*sptr, soff, nelts);
    /* they may or may not all hit one destination block */
    if (nelts <= libGAP_BIPEB - doff) 
      libGAP_setmidbits(dptr, doff, nelts, bits);
    else {
      libGAP_sethighbits(dptr++, libGAP_BIPEB- doff, libGAP_lowbits(bits, libGAP_BIPEB - doff));
      libGAP_setlowbits(dptr, nelts - libGAP_BIPEB + doff, (bits >> (libGAP_BIPEB - doff)));
    }
    return;
  }
  
  /* all the section is within the starting destination block */
  if (nelts <= libGAP_BIPEB - doff) {
    /* since we weren't in the last case, we need to collect the bits from two
       source blocks */

    bits = libGAP_highbits(*sptr++, libGAP_BIPEB-soff);
    bits |= (libGAP_lowbits(*sptr, nelts + soff  - libGAP_BIPEB) << (libGAP_BIPEB-soff));
    libGAP_setmidbits(dptr, doff, nelts, bits);
    return;
  }

  /* If we reach this point, we are reading from at least two source blocks
     and writing to at least two destination blocks */  

  /* Now, split according to relationship of soff and doff 
     easiest case first, when they are equal */
  if (soff == doff) {
    libGAP_UInt fullblocks;
    /* partial block at the start */
    if (soff != 0) {
      bits = libGAP_highbits(*sptr++, libGAP_BIPEB - soff);
      libGAP_sethighbits(dptr++, libGAP_BIPEB - soff, bits);
      fullblocks = (nelts + soff - libGAP_BIPEB)/libGAP_BIPEB;
    } else
      fullblocks = nelts/libGAP_BIPEB;
    /* Now zero or more full blocks */
    memmove(dptr, sptr, fullblocks*sizeof(libGAP_Obj));
    /* partial block at the end */
    libGAP_UInt eoff = (soff + nelts) % libGAP_BIPEB;
    if (eoff != 0) {
      bits = libGAP_lowbits(sptr[fullblocks],eoff);
      libGAP_setlowbits(dptr+fullblocks,eoff, bits);
    }
    return;
  } else {
    libGAP_UInt cbits, endbits;
    if (soff > doff) {
      libGAP_setmidbits(dptr, doff, libGAP_BIPEB - soff, libGAP_highbits(*sptr++, libGAP_BIPEB - soff));
      libGAP_sethighbits(dptr++, soff-doff, libGAP_lowbits(*sptr, soff-doff));
      cbits = libGAP_BIPEB + doff-soff;
    } else {
      libGAP_sethighbits(dptr++, libGAP_BIPEB -doff, libGAP_midbits(*sptr, soff, libGAP_BIPEB-doff));
      cbits = doff-soff;
    }

    /* At this point dptr points to a block that needs to be filled from the start
       with the cbits highbits of *sptr and the remaining bits from sptr[1] 
     except of course that it might be the final block and so need less than that */
    dend = libGAP_BLOCKS_GF2VEC(dest) + (dmin + nelts )/libGAP_BIPEB; /* first block we don't fill completely */
    /* We replicate the inner loop 31 or 63 times, so that the shifts are known at compile time*/


    switch(cbits) {
    case 1:     libGAP_dothework(sptr, dptr, 1, dend); break;
    case 2:     libGAP_dothework(sptr, dptr, 2, dend); break;
    case 3:     libGAP_dothework(sptr, dptr, 3, dend); break;
    case 4:     libGAP_dothework(sptr, dptr, 4, dend); break;
    case 5:     libGAP_dothework(sptr, dptr, 5, dend); break;
    case 6:     libGAP_dothework(sptr, dptr, 6, dend); break;
    case 7:     libGAP_dothework(sptr, dptr, 7, dend); break;
    case 8:     libGAP_dothework(sptr, dptr, 8, dend); break;
    case 9:     libGAP_dothework(sptr, dptr, 9, dend); break;
    case 10:     libGAP_dothework(sptr, dptr, 10, dend); break;
    case 11:     libGAP_dothework(sptr, dptr, 11, dend); break;
    case 12:     libGAP_dothework(sptr, dptr, 12, dend); break;
    case 13:     libGAP_dothework(sptr, dptr, 13, dend); break;
    case 14:     libGAP_dothework(sptr, dptr, 14, dend); break;
    case 15:     libGAP_dothework(sptr, dptr, 15, dend); break;
    case 16:     libGAP_dothework(sptr, dptr, 16, dend); break;
    case 17:     libGAP_dothework(sptr, dptr, 17, dend); break;
    case 18:     libGAP_dothework(sptr, dptr, 18, dend); break;
    case 19:     libGAP_dothework(sptr, dptr, 19, dend); break;
    case 20:     libGAP_dothework(sptr, dptr, 20, dend); break;
    case 21:     libGAP_dothework(sptr, dptr, 21, dend); break;
    case 22:     libGAP_dothework(sptr, dptr, 22, dend); break;
    case 23:     libGAP_dothework(sptr, dptr, 23, dend); break;
    case 24:     libGAP_dothework(sptr, dptr, 24, dend); break;
    case 25:     libGAP_dothework(sptr, dptr, 25, dend); break;
    case 26:     libGAP_dothework(sptr, dptr, 26, dend); break;
    case 27:     libGAP_dothework(sptr, dptr, 27, dend); break;
    case 28:     libGAP_dothework(sptr, dptr, 28, dend); break;
    case 29:     libGAP_dothework(sptr, dptr, 29, dend); break;
    case 30:     libGAP_dothework(sptr, dptr, 30, dend); break;
    case 31:     libGAP_dothework(sptr, dptr, 31, dend); break;
#ifdef libGAP_SYS_IS_64_BIT
    case 32:     libGAP_dothework(sptr, dptr, 32, dend); break;
    case 33:     libGAP_dothework(sptr, dptr, 33, dend); break;
    case 34:     libGAP_dothework(sptr, dptr, 34, dend); break;
    case 35:     libGAP_dothework(sptr, dptr, 35, dend); break;
    case 36:     libGAP_dothework(sptr, dptr, 36, dend); break;
    case 37:     libGAP_dothework(sptr, dptr, 37, dend); break;
    case 38:     libGAP_dothework(sptr, dptr, 38, dend); break;
    case 39:     libGAP_dothework(sptr, dptr, 39, dend); break;
    case 40:     libGAP_dothework(sptr, dptr, 40, dend); break;
    case 41:     libGAP_dothework(sptr, dptr, 41, dend); break;
    case 42:     libGAP_dothework(sptr, dptr, 42, dend); break;
    case 43:     libGAP_dothework(sptr, dptr, 43, dend); break;
    case 44:     libGAP_dothework(sptr, dptr, 44, dend); break;
    case 45:     libGAP_dothework(sptr, dptr, 45, dend); break;
    case 46:     libGAP_dothework(sptr, dptr, 46, dend); break;
    case 47:     libGAP_dothework(sptr, dptr, 47, dend); break;
    case 48:     libGAP_dothework(sptr, dptr, 48, dend); break;
    case 49:     libGAP_dothework(sptr, dptr, 49, dend); break;
    case 50:     libGAP_dothework(sptr, dptr, 50, dend); break;
    case 51:     libGAP_dothework(sptr, dptr, 51, dend); break;
    case 52:     libGAP_dothework(sptr, dptr, 52, dend); break;
    case 53:     libGAP_dothework(sptr, dptr, 53, dend); break;
    case 54:     libGAP_dothework(sptr, dptr, 54, dend); break;
    case 55:     libGAP_dothework(sptr, dptr, 55, dend); break;
    case 56:     libGAP_dothework(sptr, dptr, 56, dend); break;
    case 57:     libGAP_dothework(sptr, dptr, 57, dend); break;
    case 58:     libGAP_dothework(sptr, dptr, 58, dend); break;
    case 59:     libGAP_dothework(sptr, dptr, 59, dend); break;
    case 60:     libGAP_dothework(sptr, dptr, 60, dend); break;
    case 61:     libGAP_dothework(sptr, dptr, 61, dend); break;
    case 62:     libGAP_dothework(sptr, dptr, 62, dend); break;
    case 63:     libGAP_dothework(sptr, dptr, 63, dend); break;
#endif
    default:  libGAP_Pr("Illegal shift %i", cbits, 0);
      libGAP_SyExit(2);
    }

    /* fixup pointers */
    sptr += (dend - dptr);
    dptr = dend;
    /* OK, so now we may need to copy some more bits to fill the final block */
    endbits = (dmin + nelts) % libGAP_BIPEB;
    if (endbits) 
      {
	if (endbits <= cbits)
	  libGAP_setlowbits(dptr, endbits, libGAP_midbits(*sptr++,libGAP_BIPEB-cbits, endbits));
	else {
	  bits = libGAP_highbits(*sptr++,cbits);
	  libGAP_setlowbits(dptr, endbits, bits | (libGAP_lowbits(*sptr, endbits - cbits) << cbits));
	}
      }
    return;
  }
}

/****************************************************************************
**
*F  AddPartialGF2VecGF2Vec( <sum>, <vl>, <vr>, <n> )  . . . . . . partial sum
**
**  'AddPartialGF2VecGF2Vec' adds the entries  of <vl> and <vr> starting from
**  that block which is corresponding to the entry with number <n> and stores
**  the result in <sum>.
**
**  Note: The other entries are set to be zero. So use a higher value for <n>
**        only for vectors, which both have leading zero-entries. 
**
**  You  can use  the parameter  <n> for  example for  an  gauss-algorithm on 
**  gf2-matrices  can be  improved,  because when  using the gauss-algorithm,
**  you  know that  the leading entries of two vectors to be  added are equal
**  zero. If <n> = 1 all entries are added. 
**
**  Note that the caller has to ensure, that <sum> is a gf2-vector with the 
**  correct size.
*/
libGAP_Obj libGAP_AddPartialGF2VecGF2Vec (
    libGAP_Obj                 sum,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr,
    libGAP_UInt                n )
{
    libGAP_UInt *              ptL;            /* bit field of <vl>               */
    libGAP_UInt *              ptR;            /* bit field of <vr>               */
    libGAP_UInt *              ptS;            /* bit field of <sum>              */
    libGAP_UInt *              end;            /* end marker                      */
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                offset;         /* number of block to start adding */
    libGAP_UInt                x;
    

    /* both operands lie in the same field                                 */
    len = libGAP_LEN_GF2VEC(vl);
    if ( len != libGAP_LEN_GF2VEC(vr) ) {
        libGAP_ErrorMayQuit( "Vector +: vectors must have the same length",
                   0L, 0L );
        return 0;
    }


    /* calculate the offset for adding                                     */
    if ( n == 1 ) {  
        ptL = libGAP_BLOCKS_GF2VEC(vl);
        ptR = libGAP_BLOCKS_GF2VEC(vr);
        ptS = libGAP_BLOCKS_GF2VEC(sum);
        end = ptS + ((len+libGAP_BIPEB-1)/libGAP_BIPEB);
    } else {
        offset = ( n - 1 ) / libGAP_BIPEB;
        ptL = libGAP_BLOCKS_GF2VEC(vl) + offset ;
        ptR = libGAP_BLOCKS_GF2VEC(vr) + offset ;
        ptS = libGAP_BLOCKS_GF2VEC(sum) + offset ;
        end = ptS + ((len+libGAP_BIPEB-1)/libGAP_BIPEB) - offset;
    }
    
    /* loop over the entries and add                                       */
    if (vl == sum)
      while ( ptS < end )
	{
	  /* maybe remove this condition */
	  if ((x = *ptR)!= 0)
	    *ptS = *ptL ^ x;
	  ptL++; ptS++; ptR++;
	}
    else if (vr == sum)
      while ( ptS < end )
	{
	  /* maybe remove this condition */
	  if ((x = *ptL) != 0)
	    *ptS = *ptR ^ x;
	  ptL++; ptS++; ptR++;
	}
    else
      while (ptS < end )
	*ptS++ = *ptL++ ^ *ptR++;

    /* return the result                                                   */
    return sum;
}


/****************************************************************************
**
*F  ProdGF2VecGF2Vec( <vl>, <vr> )  . . . . . . .  product of two GF2 vectors
**
**  'ProdVecGF2VecGF2' returns  the product of  the two GF2 vectors <vl> and
**  <vr>.   The product is  the folded sum of   the corresponding entries of
**  <vl> and <vr>.
**
**  'ProdVecGF2VecGF2' is an improved  version of the general multiplication,
**  which  does not  call 'PROD'  but uses bit  operations instead.   It will
**  always return either 'GF2One' or 'GF2Zero'.
*/
#ifdef libGAP_SYS_IS_64_BIT

#define libGAP_PARITY_BLOCK(m) \
  do { m = m ^ (m>>32); \
       m = m ^ (m>>16); \
       m = m ^ (m>>8);  \
       m = m ^ (m>>4);  \
       m = m ^ (m>>2);  \
       m = m ^ (m>>1);  \
  } while(0)

#else

#define libGAP_PARITY_BLOCK(m) \
  do { m = m ^ (m>>16); \
       m = m ^ (m>>8);  \
       m = m ^ (m>>4);  \
       m = m ^ (m>>2);  \
       m = m ^ (m>>1);  \
  } while(0)

#endif

libGAP_Obj libGAP_ProdGF2VecGF2Vec ( libGAP_Obj vl, libGAP_Obj vr )
{
    libGAP_UInt *              ptL;            /* bit field of <vl>               */
    libGAP_UInt *              ptR;            /* bit field of <vr>               */
    libGAP_UInt                lenL;           /* length of the list              */
    libGAP_UInt                lenR;           /* length of the list              */
    libGAP_UInt                len;            /* minimum of the lengths          */
    libGAP_UInt                nrb;            /* number of whole blocks to use   */
    libGAP_UInt                m;              /* number of bits in a block       */
    libGAP_UInt                n;              /* number of bits in blist         */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_UInt                mask;           /* bit selecting mask              */

    /* both operands lie in the same field                                 */
    lenL = libGAP_LEN_GF2VEC(vl);
    lenR = libGAP_LEN_GF2VEC(vr);
    len = (lenL < lenR) ? lenL : lenR;
    
    if ( len == 0 ) {
      libGAP_ErrorMayQuit(
        "Vector *: both vectors must have at least one entry",
        (libGAP_Int)0, (libGAP_Int)0 );
      return 0;
    }

    /* loop over the entries and multiply                                  */
    ptL = libGAP_BLOCKS_GF2VEC(vl);
    ptR = libGAP_BLOCKS_GF2VEC(vr);
    nrb = len /libGAP_BIPEB;
    n   = 0;
    for ( i = nrb;  i > 0;  i-- ) {
        m = (*ptL++) & (*ptR++);
        libGAP_PARITY_BLOCK(m);
        n ^= m;
    }
    /* now process the remaining bits */

    mask = 1;
    for (i = 0; i < len % libGAP_BIPEB; i++)
      {
	n ^= (mask & *ptL & *ptR) >> i;
	mask <<= 1;
      }

    /* return the result                                                   */
    return (n & 1) ? libGAP_GF2One : libGAP_GF2Zero;
}


/****************************************************************************
**
*F  ProdGF2VecGF2Mat( <vl>, <vr> )  . .  product of GF2 vector and GF2 matrix
**
**  'ProdGF2VecGF2Mat'  returns the product  of the  GF2 vector <vl>  and the
**  GF2 matrix  <vr>.   The product is   the sum of  the  rows of <vr>,  each
**  multiplied by the corresponding entry of <vl>.  Note that the  caller has
**  to ensure, that <vl> is a gf2-vector and <vr> is a gf2-matrix.
*/
libGAP_Obj libGAP_ProdGF2VecGF2Mat ( libGAP_Obj vl, libGAP_Obj vr )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                stop;
    libGAP_UInt                col;            /* length of the rows              */
    libGAP_UInt                i;              /* loop variables                  */
    libGAP_Obj                 prod;           /* product, result                 */
    libGAP_Obj                 row1;           /* top row of matrix               */
    libGAP_UInt *              start;
    libGAP_UInt *              ptL;
    libGAP_UInt                mask;
    
    /* both operands lie in the same field                                 */
    len = libGAP_LEN_GF2VEC(vl);
    if (len > libGAP_LEN_GF2MAT(vr))
      len = libGAP_LEN_GF2MAT(vr);
    
    /* make the result vector                                              */
    row1 = libGAP_ELM_GF2MAT( vr, 1 );
    col = libGAP_LEN_GF2VEC( row1 );
    libGAP_NEW_GF2VEC( prod, (libGAP_IS_MUTABLE_OBJ(vl) || libGAP_IS_MUTABLE_OBJ(row1)) ? 
		libGAP_TYPE_LIST_GF2VEC : libGAP_TYPE_LIST_GF2VEC_IMM, col );
    libGAP_SET_LEN_GF2VEC( prod, col );
    
    /* get the start and end block                                         */
    start = libGAP_BLOCKS_GF2VEC(prod);
    ptL   = libGAP_BLOCKS_GF2VEC(vl);

    /* loop over the vector                                                */
    for ( i = 1;  i <= len;  ptL++ )  {

        /* if the whole block is zero, get the next entry                  */
        if (*ptL == 0) {
            i += libGAP_BIPEB;
            continue;
        }
        
        /* run through the block                                           */
        stop = i + libGAP_BIPEB - 1;
        if ( len < stop )
            stop = len;
        for ( mask = 1;  i <= stop;  i++, mask <<= 1 ) {

            /* if there is entry add the row to the result                 */
            if ( (*ptL & mask) != 0 ) {
                libGAP_UInt * ptRR = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(vr,i));
		libGAP_AddGF2VecToGF2Vec(start, ptRR, col);
            }
        }
    }

    /* return the result                                                   */
    return prod;
}


/****************************************************************************
**
*F  ProdGF2MatGF2Vec( <ml>, <vr> )  . .  product of GF2 matrix and GF2 vector
**
**  'ProdGF2MatGF2Vec'  returns the product  of the  GF2 matrix <ml>  and the
**  GF2 vector  <vr>.   The ith entry of the
**  product is the inner product of  the  ith row of <ml> with <vr>.
**  Note that the  caller has
**  to ensure, that <ml> is a GF2 matrix and <vr> is a GF2 vector.
*/
libGAP_Obj libGAP_ProdGF2MatGF2Vec ( libGAP_Obj ml, libGAP_Obj vr )
{
    libGAP_UInt                len;            /* length of the vector            */
    libGAP_UInt                ln1;            /* length of the rows of the mx    */
    libGAP_UInt                ln2;            /* length of the matrix            */
    libGAP_UInt *              ptL;            /* bit field of <ml>[j]            */
    libGAP_UInt *              ptR;            /* bit field of <vr>               */
    libGAP_UInt                nrb;            /* number of blocks in blist       */
    libGAP_UInt                m;              /* number of bits in a block       */
    libGAP_UInt                n;              /* number of bits in blist         */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_UInt                j;              /* loop variable                   */
    libGAP_Obj                 prod;           /* result                          */
    libGAP_UInt                mask;           /* a one bit mask */
    
    /* both operands lie in the same field                                 */
    len = libGAP_LEN_GF2VEC(vr);
    ln2 = libGAP_LEN_GF2MAT(ml);
    if ( 0 == ln2 ) {
      libGAP_ErrorMayQuit("PROD: empty GF2 matrix * GF2 vector not allowed",0,0);
    }

    ln1 = libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(ml,1));
    if ( len > ln1 ) {
      len = ln1;
    }

    /* make the result vector                                              */
    libGAP_NEW_GF2VEC( prod, (libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(ml,1)) || libGAP_IS_MUTABLE_OBJ(vr)) ? 
		libGAP_TYPE_LIST_GF2VEC :libGAP_TYPE_LIST_GF2VEC_IMM, ln2 );
    libGAP_SET_LEN_GF2VEC( prod, ln2 );

    /* loop over the entries and multiply                                  */
    nrb = len/libGAP_BIPEB;
    for ( j = 1;  j <= ln2;  j++ ) {
        ptL = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(ml,j));
        ptR = libGAP_BLOCKS_GF2VEC(vr);
        n   = 0;
        for ( i = 1;  i <= nrb;  i++ ) {
            m = (*ptL++) & (*ptR++);
            libGAP_PARITY_BLOCK(m);
            n ^= m;
        }

	mask = 1;
	for (i = 0; i < len % libGAP_BIPEB; i++)
	  {
	    n ^= (mask & *ptL & *ptR) >> i;
	    mask <<= 1;
	  }

	
        if ( n & 1 )
            libGAP_BLOCK_ELM_GF2VEC(prod,j) |= libGAP_MASK_POS_GF2VEC(j);
    }

    /* return the result                                                   */
    return prod;
}

/****************************************************************************
**
*F  ProdGF2MatGF2MatSimple( <ml>, <mr> )  . .  product of GF2 matrix and GF2 matrix
*F  ProdGF2MatGF2MatAdvanced( <ml>, <mr>, <greaselevel>, <blocksize> )
**                                     . .  product of GF2 matrix and GF2 matrix
**
**  'ProdGF2MatGF2MatSimple'  returns the product  of the  GF2 matrix <ml>  and the
**  GF2 matrix  <mr>.  This simply calls ProdGF2VecGF2Mat once on each row.
**
** ProdGF2MatGF2MatAdvanced uses the specified grease and blocking to accelerate
** larger matrix multiplies. In this case, the matrix dimensions must be compatible.
*/

libGAP_Obj libGAP_ProdGF2MatGF2MatSimple( libGAP_Obj ml, libGAP_Obj mr )
{
  libGAP_Obj prod;
  libGAP_UInt i;
  libGAP_UInt len;
  libGAP_Obj row;
  libGAP_Obj rtype;
  len = libGAP_LEN_GF2MAT(ml);
  prod = libGAP_NewBag(libGAP_T_POSOBJ, libGAP_SIZE_PLEN_GF2MAT(len));
  libGAP_SET_LEN_GF2MAT(prod,len);
  if (libGAP_IS_MUTABLE_OBJ(ml) || libGAP_IS_MUTABLE_OBJ(mr))
    {
      libGAP_TYPE_POSOBJ(prod) = libGAP_TYPE_LIST_GF2MAT;
      if (libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(ml,1)) || libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(mr,1)))
	rtype = libGAP_TYPE_LIST_GF2VEC_LOCKED;
      else
	rtype = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    }
  else
    {
      libGAP_TYPE_POSOBJ(prod) = libGAP_TYPE_LIST_GF2MAT_IMM;
      rtype = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    }
  for (i = 1; i <= len; i++)
    {
      row = libGAP_ProdGF2VecGF2Mat(libGAP_ELM_GF2MAT(ml,i),mr);

      /* Since I'm going to put this vector into a matrix, I must lock its
	 representation, so that it doesn't get rewritten over GF(2^k) */
      libGAP_TYPE_DATOBJ(row) = rtype;
      libGAP_SET_ELM_GF2MAT(prod,i,row);
      libGAP_CHANGED_BAG(prod);
      libGAP_TakeInterrupt();
    }
  return prod;
}


/* Utility functions for the advanced matrix multiply code below */


/* extract nbits bits starting from position from in vector vptr
   return them as the nbits least significant bits in a UInt.
   Bits are always numbered least-significant first */

static inline libGAP_UInt libGAP_getbits(libGAP_UInt * vptr, libGAP_UInt from, libGAP_UInt nbits)
{
  libGAP_UInt wno = (from  -1)/libGAP_BIPEB;
  libGAP_UInt word1 = vptr[wno];
  libGAP_UInt shift1 = (from -1)% libGAP_BIPEB;
  libGAP_UInt lbit = shift1 + nbits;
  libGAP_UInt word2;
  if (lbit <= libGAP_BIPEB)
    {
      /* range is all in one word */
      word1 <<= libGAP_BIPEB -lbit;
      word1 >>= libGAP_BIPEB - nbits;
    }
  else
    {
      /* range is split across two words */
      word1 >>= shift1;
      lbit -= libGAP_BIPEB;
      word2 = vptr[wno+1];
      word2 <<= libGAP_BIPEB-lbit;
      word2 >>= shift1-lbit;
      word1 |= word2;
    }
  return word1;
}

/* To avoid having a lot of arguments to the recursive getgreasedata function,
   we put the things that don't change in the recursive call into this structure */

struct libGAP_greaseinfo {
  libGAP_UInt *pgtags, *pgbuf,  nblocks, *pgrules;
  libGAP_UInt **prrows;
};

static struct libGAP_greaseinfo libGAP_g;

/* Make if necessary the grease row for bits
   controlled by the data in g. Recursive
   so can't be inlined */

static libGAP_UInt * libGAP_getgreasedata( libGAP_UInt bits)
{ 
  libGAP_UInt x,y;
  register libGAP_UInt *ps, *pd, *ps2,i ;
  libGAP_UInt *pd1;
  switch(libGAP_g.pgtags[bits])
    {
    case 0:
      /* Need to make the row */
      x = libGAP_g.pgrules[bits];
      y = bits ^ (1 << x);
      /* make it by adding row x to grease vector indexed y */
      ps =libGAP_g.prrows[x];
      ps2 = libGAP_getgreasedata(y);
      pd1 = libGAP_g.pgbuf + (bits-3)*libGAP_g.nblocks;
      pd = pd1;
      /* time critical inner loop */
      for (i = libGAP_g.nblocks; i > 0; i--)
	*pd++ = *ps++ ^ *ps2++;
      /* record that we made it */
      libGAP_g.pgtags[bits] = 1;
      return pd1;

    case 1:
      /* we've made this one already, so just return it */
      return  libGAP_g.pgbuf + (bits-3)*libGAP_g.nblocks;

    case 2:
      /* This one does not need making, bits actually
	 has just a single 1 bit in it */
      return libGAP_g.prrows[libGAP_g.pgrules[bits]];

    }
  return (libGAP_UInt *)0;		/* can't actually get here
				 include the return to pacify compiler */
}



libGAP_Obj libGAP_ProdGF2MatGF2MatAdvanced( libGAP_Obj ml, libGAP_Obj mr, libGAP_UInt greasesize , libGAP_UInt blocksize)
{
  libGAP_Obj prod;			/* Product Matrix */
  libGAP_UInt i,j, k, b;		/* Loop counters */
  libGAP_UInt gs;			/* Actual level of grease for current block */
  libGAP_UInt *rptr;			/* Pointer to current row of ml */
  libGAP_UInt bits;			/* current chunk of current row, for lookup in grease tables */
  libGAP_UInt *v;			/* pointer to computed grease vector */
  libGAP_UInt len, rlen, ilen;		/* len = length of ml, ilen = row length of ml = length of mr, rlen = row length of mr */
  libGAP_Obj row;			/* current row of ml, or row of prod when it is being built */
  libGAP_Obj rtype;			/* type of rows of prod */
  libGAP_Obj gbuf = (libGAP_Obj)0;		/* grease buffer */
  libGAP_Obj gtags = (libGAP_Obj) 0;		/* grease tags (whether that row is known yet */
  libGAP_Obj grules = (libGAP_Obj) 0;		/* rules for making new grease vectors */
  libGAP_UInt *pgrules;		/* pointer to contents of grules*/
  libGAP_UInt *pgtags = (libGAP_UInt *)0;	/* pointer to contents of gtags */
  libGAP_UInt *pgbuf = (libGAP_UInt *)0;	/* pointer to grease buffer */
  libGAP_UInt nwords;			/* number of words in a row of mr */
  libGAP_UInt glen;			/* 1 << greasesize */
  libGAP_UInt bs;			/* actual size of current block */
  libGAP_UInt *pprow;			/* pointer into current row of prod */
  libGAP_Obj lrowptrs;			/* cache of direct pointers to rows of ml */
  libGAP_UInt **plrows;		/* and a direct pointer to that cache */
  libGAP_Obj rrowptrs;			/* and for mr */
  libGAP_UInt **prrows;
  libGAP_Obj prowptrs;			/* and for prod */
  libGAP_UInt **pprows;
  
  len = libGAP_LEN_GF2MAT(ml);
  row = libGAP_ELM_GF2MAT(mr, 1);
  rlen = libGAP_LEN_GF2VEC(row);
  ilen = libGAP_LEN_GF2MAT(mr);
  nwords = libGAP_NUMBER_BLOCKS_GF2VEC(row);
  
  /* Make a zero product matrix */
  prod = libGAP_NewBag(libGAP_T_POSOBJ, libGAP_SIZE_PLEN_GF2MAT(len));
  libGAP_SET_LEN_GF2MAT(prod,len);
  if (libGAP_IS_MUTABLE_OBJ(ml) || libGAP_IS_MUTABLE_OBJ(mr))
    {
     libGAP_TYPE_POSOBJ(prod) = libGAP_TYPE_LIST_GF2MAT;
      if (libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(ml,1)) || libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(mr,1)))
	rtype = libGAP_TYPE_LIST_GF2VEC_LOCKED;
      else
	rtype = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    }
  else
    {
      libGAP_TYPE_POSOBJ(prod) = libGAP_TYPE_LIST_GF2MAT_IMM;
      rtype = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    }


  for (i = 1; i <= len; i++)
    {
      libGAP_NEW_GF2VEC(row, rtype, rlen);
      libGAP_SET_LEN_GF2VEC(row, rlen);
      libGAP_SET_ELM_GF2MAT(prod,i,row);
      libGAP_CHANGED_BAG(prod);
    }

  /* Cap greasesize and blocksize by the actual length  */
  if (ilen < greasesize)
    greasesize = ilen;
  if (ilen < greasesize*blocksize)
    blocksize = (ilen + greasesize-1)/greasesize;


  /* calculate glen*/
  glen = 1 << greasesize;
  
  /* Allocate memory */

  lrowptrs = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_UInt *)*len);
  rrowptrs = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_UInt *)*ilen);
  prowptrs = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_UInt *)*len);
  
  if (greasesize >= 2)
    {
      gbuf = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_UInt)*nwords*(glen-3)*blocksize);
      gtags = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_UInt)* glen*blocksize);
      grules = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_Int)* glen);
      
      
      /* From here no garbage collections */
      
      pgtags = (libGAP_UInt *)libGAP_ADDR_OBJ(gtags);
      pgrules = (libGAP_UInt *)libGAP_ADDR_OBJ(grules);
      pgbuf = (libGAP_UInt *)libGAP_ADDR_OBJ(gbuf);
      

      /* Calculate the greasing rules */
      for (j = 3; j < glen; j++)
	for (i = 0; i < greasesize; i++)
	  if ((j & (1 << i)) != 0)
	    {
	      pgrules[j] = i;
	      break;
	    }
      for (j = 0; j < greasesize; j++)
	pgrules[1<<j] = j;

      /* fill in some more bits of g */
      libGAP_g.pgrules = pgrules;
      libGAP_g.nblocks = nwords;
    }

  /* Take direct pointers to all the parts of all the matrices to avoid multiple
     indirection overheads */
  plrows = (libGAP_UInt **)libGAP_ADDR_OBJ(lrowptrs);
  prrows = (libGAP_UInt **)libGAP_ADDR_OBJ(rrowptrs);
  pprows = (libGAP_UInt **)libGAP_ADDR_OBJ(prowptrs);

  for (i = 0; i < len; i++)
    {
      plrows[i] = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(ml,i+1));
      pprows[i] = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(prod, i+1));
    }
  for (i = 0; i < ilen; i++)
    prrows[i] = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(mr, i+1));


  /* OK, finally ready to start work */
  /* loop over blocks */
  for (b = 1; b <= ilen; b += blocksize*greasesize)
    {
      /* last block may be a small one */
      bs = blocksize;
      if ((b + bs*greasesize) > ilen)
	bs = (ilen - b + greasesize)/greasesize;

      /* If we're greasing, start afresh */
      if (greasesize > 1)
	{
	  for (k = 0; k < bs; k++)
	    {
	      for (j = 0; j < 1 << greasesize; j++)
		pgtags[k*glen+j] = 0;
	      /* powers of 2 correspond to rows of mr */
	      for (j =0; j < greasesize; j++)
		pgtags[k*glen+ (1<<j)] = 2;
	    }
	}
      
      /* For each block, we run through rows of ml & prod */
      for (j = 1; j <= len; j++)
	{
	  /* get pointers */
	  rptr = plrows[j-1];
	  pprow = pprows[j-1];

	  /* Now within the block, we have multiple grease-units, run through them */
	  for (i = 0; i < bs; i++)
	    {
	      /* start of current grease unit */	     
	      k = b + i*greasesize;

	      /* last unit of last block may be short */
	      gs = greasesize;
	      if (k+gs > ilen)
		gs = ilen - k +1;

	      /* find the appropriate parts of grease tags
		 grease buffer and mr. Store in g */
	      if (gs > 1)
		{
		  libGAP_g.pgtags = pgtags + glen*i;
		  libGAP_g.pgbuf = pgbuf + (glen-3)*nwords*i;
		  libGAP_g.prrows = prrows + k -1;
		}

	      /* get a chunk from a row of ml */
	      bits = libGAP_getbits(rptr, k, gs);

	      /* 0 means nothing to do */
	      if (bits == 0)
		continue;
	      else if (bits == 1) /* handle this one specially to speed up the greaselevel 1 case */
		v = prrows[k-1]; /* -1 is because k is 1-based index */
	      else
		v = libGAP_getgreasedata(bits); /* The main case */
				/* This function should be inlined */
	      libGAP_AddGF2VecToGF2Vec(pprow, v,  rlen);  
	    }  
	  }
	
      /* Allow GAP to respond to Ctrl-C */
      if (libGAP_TakeInterrupt()) {
	/* Might have been a garbage collection, reload everything */
	if (greasesize >= 2) {
	  pgtags = (libGAP_UInt *)libGAP_ADDR_OBJ(gtags);
	  pgrules = (libGAP_UInt *)libGAP_ADDR_OBJ(grules);
	  pgbuf = (libGAP_UInt *)libGAP_ADDR_OBJ(gbuf);
	  /* fill in some more bits of g */
	  libGAP_g.pgrules = pgrules;
	  libGAP_g.nblocks = nwords;
	}
	plrows = (libGAP_UInt **)libGAP_ADDR_OBJ(lrowptrs);
	prrows = (libGAP_UInt **)libGAP_ADDR_OBJ(rrowptrs);
	pprows = (libGAP_UInt **)libGAP_ADDR_OBJ(prowptrs);
	for (i = 0; i < len; i++)
	  {
	    plrows[i] = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(ml,i+1));
	    pprows[i] = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(prod, i+1));
	  }
	for (i = 0; i < ilen; i++)
	  prrows[i] = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(mr, i+1));
      }
    }
  return prod;
}

/****************************************************************************
**
*F  FuncProdGF2VecAnyMat( <self>, <v>, <m>) . . . method to handle vector*plain list
**                                    of GF2Vectors reasonably efficiently
**
*/
libGAP_Obj libGAP_FuncProdGF2VecAnyMat ( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj mat )
{
  libGAP_Obj res;
  libGAP_UInt len;
  libGAP_UInt len1;
  libGAP_Obj row1;
  libGAP_UInt i;
  libGAP_UInt block = 0;
  
  len = libGAP_LEN_GF2VEC(vec);
  if (len > libGAP_LEN_PLIST(mat))
    len = libGAP_LEN_PLIST(mat);

  /* Get the first row, to establish the size of the result */
  row1 = libGAP_ELM_PLIST(mat,1);  
  if (! libGAP_IS_GF2VEC_REP(row1))
    return libGAP_TRY_NEXT_METHOD;
  len1 = libGAP_LEN_GF2VEC(row1);

  /* create the result space */
  libGAP_NEW_GF2VEC( res,
	      (libGAP_IS_MUTABLE_OBJ(vec) || libGAP_IS_MUTABLE_OBJ(mat)) ? libGAP_TYPE_LIST_GF2VEC : libGAP_TYPE_LIST_GF2VEC_IMM,
	      len1);
  libGAP_SET_LEN_GF2VEC(res,len1);

  /* Finally, we start work */
  for (i = 1; i <= len; i++)
    {
      if (i % libGAP_BIPEB == 1)
	block = libGAP_BLOCK_ELM_GF2VEC(vec,i);
      if (block & libGAP_MASK_POS_GF2VEC(i))
     	{
	  row1 = libGAP_ELM_PLIST(mat,i);  
	  if (! libGAP_IS_GF2VEC_REP(row1))
	    return libGAP_TRY_NEXT_METHOD;
	  libGAP_AddPartialGF2VecGF2Vec(res, res, row1, 1);
	}
    }
  return res;

}

/****************************************************************************
**
*F  InversePlistGF2VecsDesstructive( <list> )
**
**  This is intended to form the core of a method for InverseOp.
**  by this point it should be checked that list is a plain list of GF2 vectors
**  of equal lengths. 
*/
libGAP_Obj libGAP_InversePlistGF2VecsDesstructive( libGAP_Obj list )
{
    libGAP_UInt                len;            /* dimension                       */
    libGAP_Obj                 inv;            /* result                          */
    libGAP_Obj                 row;            /* row vector                      */
    libGAP_Obj                 old;            /* row from <mat>                  */
    libGAP_Obj                 tmp;            /* temporary                       */
    libGAP_UInt *              ptQ;            /* data block of <row>             */
    libGAP_UInt *              ptP;            /* data block of source row        */
    libGAP_UInt *              end;            /* end marker                      */
    libGAP_UInt *              end2;           /* end marker                      */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_UInt                k;              /* loop variable                   */

    len = libGAP_LEN_PLIST(list);
    
    /* create the identity matrix                                          */
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
    for ( i = len;  0 < i;  i-- ) {
      libGAP_NEW_GF2VEC( row, libGAP_TYPE_LIST_GF2VEC, len );
      libGAP_SET_LEN_GF2VEC( row, len );
      libGAP_BLOCK_ELM_GF2VEC(row,i) |= libGAP_MASK_POS_GF2VEC(i);
      libGAP_SET_ELM_PLIST( tmp, i, row );
      libGAP_CHANGED_BAG(tmp);
    }
    libGAP_SET_LEN_PLIST(tmp, len);
    inv = tmp;

    /* now start with ( id | mat ) towards ( inv | id )                    */
    for ( k = 1;  k <= len;  k++ ) {

        /* find a nonzero entry in column <k>                              */
        for ( i = k;  i <= len;  i++ ) {
            row = libGAP_ELM_PLIST( list, i );
            if ( libGAP_BLOCK_ELM_GF2VEC(row,k) & libGAP_MASK_POS_GF2VEC(k) )
                break;
        }
        if ( i > len )  {
            return libGAP_Fail;
        }
        if ( i != k )  {
            row = libGAP_ELM_PLIST( list, i );
            libGAP_SET_ELM_PLIST( list, i, libGAP_ELM_PLIST( list, k ) );
            libGAP_SET_ELM_PLIST( list, k, row );
            row = libGAP_ELM_PLIST( inv, i );
            libGAP_SET_ELM_PLIST( inv, i, libGAP_ELM_PLIST( inv, k ) );
            libGAP_SET_ELM_PLIST( inv, k, row );
        }
        
        /* clear entries                                                   */
        old = libGAP_ELM_PLIST( list, k );
        end = libGAP_BLOCKS_GF2VEC(old) + ((len+libGAP_BIPEB-1)/libGAP_BIPEB);
        for ( i = 1;  i <= len;  i++ ) {
            if ( i == k )
                continue;
            row = libGAP_ELM_PLIST( list, i );
            if ( libGAP_BLOCK_ELM_GF2VEC(row,k) & libGAP_MASK_POS_GF2VEC(k) ) {

                /* clear <mat>                                             */
                ptQ = &(libGAP_BLOCK_ELM_GF2VEC(row,k));
                ptP = &(libGAP_BLOCK_ELM_GF2VEC(old,k));
                while ( ptP < end ) {
                    *ptQ++ ^= *ptP++;
                }

                /* modify <inv>                                            */
                row  = libGAP_ELM_PLIST( inv, i );
                ptQ  = libGAP_BLOCKS_GF2VEC(row);
                row  = libGAP_ELM_PLIST( inv, k );
                ptP  = libGAP_BLOCKS_GF2VEC(row);
                end2 = ptP + ((len+libGAP_BIPEB-1)/libGAP_BIPEB);
                while ( ptP < end2 ) {
                    *ptQ++ ^= *ptP++;
                }
            }
        }
	libGAP_TakeInterrupt();
    }
    return inv;
}
    


/****************************************************************************
**
*F  InverseGF2Mat( <mat> )  . . . . . . . . . . . . . . inverse of GF2 matrix
**
**  This should be improved to work with mutable GF2 matrices
**
*/

libGAP_Obj libGAP_InverseGF2Mat (
    libGAP_Obj                 mat,
    libGAP_UInt                mut)
{
    libGAP_UInt                len;            /* dimension                       */
    libGAP_Obj                 inv;            /* result                          */
    libGAP_Obj                 row;            /* row vector                      */
    libGAP_Obj                 tmp;            /* temporary                       */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 old;            /* row from <mat>                  */
    libGAP_UInt *              ptQ;            /* data block of <row>             */
    libGAP_UInt *              ptP;            /* data block of source row        */
    libGAP_UInt *              end;            /* end marker                      */
    libGAP_Obj                 rtype;

    /* make a structural copy of <mat> as list of GF2 vectors              */
    len = libGAP_LEN_GF2MAT(mat);

    /* special routes for very small matrices */
    if ( len == 0 ) {
        return libGAP_CopyObj(mat,1);
    }
    if (len == 1 ) {
      row = libGAP_ELM_GF2MAT(mat,1);
      if (libGAP_BLOCKS_GF2VEC(row)[0] & 1)
	{
	  return libGAP_CopyObj(mat, 1);
	}
      else
	return libGAP_Fail;
    }
    
    tmp = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
    for ( i = len;  0 < i;  i-- ) {
        old = libGAP_ELM_GF2MAT( mat, i );
        libGAP_NEW_GF2VEC( row, libGAP_TYPE_LIST_GF2VEC_IMM, len );
        libGAP_SET_LEN_GF2VEC( row, len );
        ptQ = libGAP_BLOCKS_GF2VEC(old);
        ptP = libGAP_BLOCKS_GF2VEC(row);
        end = ptP + ((len+libGAP_BIPEB-1)/libGAP_BIPEB);
        while ( ptP < end )
            *ptP++ = *ptQ++;
        libGAP_SET_ELM_PLIST( tmp, i, row );
        libGAP_CHANGED_BAG(tmp);
    }
    libGAP_SET_LEN_PLIST(tmp,len);
    inv = libGAP_InversePlistGF2VecsDesstructive( tmp );
    if (inv == libGAP_Fail)
      return inv;
    
    /* convert list <inv> into a matrix                                    */
    libGAP_ResizeBag( inv, libGAP_SIZE_PLEN_GF2MAT(len) );
    if (mut == 2 ||
	(mut == 1 && libGAP_IS_MUTABLE_OBJ(mat) && libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(mat, 1))))
      rtype = libGAP_TYPE_LIST_GF2VEC_LOCKED;
    else
      rtype = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    for ( i = len;  0 < i;  i-- ) {
      row = libGAP_ELM_PLIST(inv,i);
      libGAP_SET_TYPE_POSOBJ(row, rtype);
      libGAP_SET_ELM_GF2MAT( inv, i, row );
    }
    libGAP_SET_LEN_GF2MAT( inv, len );
    libGAP_RetypeBag( inv, libGAP_T_POSOBJ );
    libGAP_TYPE_POSOBJ( inv ) = (mut == 2 || (mut == 1 && libGAP_IS_MUTABLE_OBJ(mat)))
      ? libGAP_TYPE_LIST_GF2MAT : libGAP_TYPE_LIST_GF2MAT_IMM;
    return inv;
}

/****************************************************************************
**
*F  ShallowCopyVecGF2( <vec> )
**
*/

libGAP_Obj libGAP_ShallowCopyVecGF2( libGAP_Obj vec )
{
  libGAP_Obj copy;
  libGAP_UInt len;
  libGAP_UInt *ptrS;
  libGAP_UInt *ptrD;
  len = libGAP_LEN_GF2VEC(vec);
  libGAP_NEW_GF2VEC( copy, libGAP_TYPE_LIST_GF2VEC, len);
  libGAP_SET_LEN_GF2VEC(copy,len);
  ptrS = libGAP_BLOCKS_GF2VEC(vec);
  ptrD = libGAP_BLOCKS_GF2VEC(copy);
  memcpy((void *) ptrD, (void *) ptrS, libGAP_NUMBER_BLOCKS_GF2VEC(vec)*sizeof(libGAP_UInt));
  return copy;
}

/****************************************************************************
**
*F  SemiEchelonPlistGF2Vecs( <mat>, <transformations-needed> )
**
**  The matrix needs to have mutable rows, so it can't be a GF2 mat
**
**  This has changed. There should now be a method for mutable GF2mats as well.
**
**  This function DOES NOT CHECK that the rows are all GF2 vectors
**
**  Does not copy the matrix, may destroy it, may include some
**  of the rows among the returned vectors
*/



static libGAP_UInt libGAP_RNheads, libGAP_RNvectors, libGAP_RNcoeffs, libGAP_RNrelns;


libGAP_Obj libGAP_SemiEchelonListGF2Vecs( libGAP_Obj mat, libGAP_UInt TransformationsNeeded )
{
  libGAP_UInt nrows, ncols;
  libGAP_UInt i,j,h;
  libGAP_Obj heads,vectors, coeffs = 0, relns = 0;
  libGAP_UInt nvecs, nrels = 0;
  libGAP_Obj coeffrow = 0;
  libGAP_Obj row;
  libGAP_UInt *rowp, *coeffrowp = 0;
  libGAP_Obj res;
  nrows = libGAP_LEN_PLIST(mat);
  ncols = libGAP_LEN_GF2VEC(libGAP_ELM_PLIST(mat,1));
  heads = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, ncols);
  libGAP_SET_LEN_PLIST(heads, ncols);
  vectors = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT, nrows);
  libGAP_SET_LEN_PLIST(vectors, 0);
  nvecs = 0;
  if (TransformationsNeeded)
    {
      coeffs = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT, nrows);
      libGAP_SET_LEN_PLIST(coeffs, 0);
      relns  = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT, nrows);
      libGAP_SET_LEN_PLIST(relns, 0);
      nrels = 0;
    }
  for (i = 1; i <= ncols; i++)
    libGAP_SET_ELM_PLIST(heads, i, libGAP_INTOBJ_INT(0));
  for (i = 1; i <= nrows; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (TransformationsNeeded)
	{
	  libGAP_NEW_GF2VEC(coeffrow, libGAP_TYPE_LIST_GF2VEC, nrows);
	  libGAP_SET_LEN_GF2VEC(coeffrow, nrows);
	  libGAP_BLOCK_ELM_GF2VEC( coeffrow, i) |= libGAP_MASK_POS_GF2VEC(i);
	}
      
      /* No garbage collection risk from here */
      rowp = libGAP_BLOCKS_GF2VEC(row);
      if (TransformationsNeeded)
	coeffrowp = libGAP_BLOCKS_GF2VEC(coeffrow);
      for (j = 1; j <= ncols; j++)
	{
	  h = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(heads, j));
	  if (h != 0)
	    {
	      if (rowp[(j-1)/libGAP_BIPEB] & libGAP_MASK_POS_GF2VEC(j))
		{
		  libGAP_AddGF2VecToGF2Vec(rowp, libGAP_BLOCKS_GF2VEC(libGAP_ELM_PLIST(vectors,h)),ncols);
		  if (TransformationsNeeded)
		    libGAP_AddGF2VecToGF2Vec(coeffrowp, libGAP_BLOCKS_GF2VEC(libGAP_ELM_PLIST(coeffs,h)),nrows);
		}
	    }
	}
      j = 1;
      while (j <= ncols && !*rowp)
	{
	  j += libGAP_BIPEB;
	  rowp++;
	}
      while ( j <= ncols && !(*rowp & libGAP_MASK_POS_GF2VEC(j)))
	j++;

      /* garbage collection OK again after here */
      if (j <= ncols)
	{
	  libGAP_SET_ELM_PLIST(vectors, ++nvecs, row);
          libGAP_CHANGED_BAG(vectors);    /* Could be an old bag by now. Max. */
	  libGAP_SET_LEN_PLIST(vectors, nvecs);
	  libGAP_SET_ELM_PLIST( heads, j, libGAP_INTOBJ_INT(nvecs));
	  if (TransformationsNeeded)
	    {
	      libGAP_SET_ELM_PLIST(coeffs, nvecs, coeffrow);
              libGAP_CHANGED_BAG(coeffs);    /* Could be an old bag by now. Max. */
	      libGAP_SET_LEN_PLIST(coeffs, nvecs);
	    }
	}
      else if (TransformationsNeeded)
	{
	  libGAP_SET_ELM_PLIST(relns, ++nrels, coeffrow);
          libGAP_CHANGED_BAG(relns);    /* Could be an old bag by now. Max. */
	  libGAP_SET_LEN_PLIST(relns, nrels);
	}
      libGAP_TakeInterrupt();
    }
  if (libGAP_RNheads == 0)
    {
      libGAP_RNheads = libGAP_RNamName("heads");
      libGAP_RNvectors = libGAP_RNamName("vectors");
    }
  res = libGAP_NEW_PREC( TransformationsNeeded ? 4 : 2);
  libGAP_AssPRec(res,libGAP_RNheads,heads);
  libGAP_AssPRec(res,libGAP_RNvectors,vectors);
  if (libGAP_LEN_PLIST(vectors) == 0)
    libGAP_RetypeBag(vectors, libGAP_T_PLIST_EMPTY);
  if (TransformationsNeeded)
    {
      if (libGAP_RNcoeffs == 0)
	{
	  libGAP_RNcoeffs = libGAP_RNamName("coeffs");
	  libGAP_RNrelns = libGAP_RNamName("relations");
	}
      libGAP_AssPRec(res,libGAP_RNcoeffs,coeffs);
      if (libGAP_LEN_PLIST(coeffs) == 0)
	libGAP_RetypeBag(coeffs, libGAP_T_PLIST_EMPTY);
      libGAP_AssPRec(res,libGAP_RNrelns,relns);
      if (libGAP_LEN_PLIST(relns) == 0)
	libGAP_RetypeBag(relns, libGAP_T_PLIST_EMPTY);
    }
  libGAP_SortPRecRNam(res,0);
  return res;
}

/****************************************************************************
**
*F  UInt TriangulizeListGF2Vecs( <mat>, <clearup> ) -- returns the rank
**
**  Again should add a method to work with mutable GF2 matrices
**
*/

libGAP_UInt libGAP_TriangulizeListGF2Vecs( libGAP_Obj mat, libGAP_UInt clearup)
{
  libGAP_UInt nrows;
  libGAP_UInt ncols;
  libGAP_UInt workcol;
  libGAP_UInt workrow;
  libGAP_UInt rank;
  libGAP_Obj row, row2;
  libGAP_UInt *rowp, *row2p;
  libGAP_UInt block;
  libGAP_UInt mask;
  libGAP_UInt j;
  nrows = libGAP_LEN_PLIST( mat );
  ncols = libGAP_LEN_GF2VEC( libGAP_ELM_PLIST(mat, 1));
  rank = 0;

  /* Nothing here can cause a garbage collection */
  
  for (workcol = 1; workcol <= ncols; workcol++)
    {
      block = (workcol-1)/libGAP_BIPEB;
      mask = libGAP_MASK_POS_GF2VEC(workcol);
      for (workrow = rank+1; workrow <= nrows &&
	     !(libGAP_BLOCKS_GF2VEC(libGAP_ELM_PLIST(mat, workrow))[block] & mask); workrow ++)
	;
      if (workrow <= nrows)
	{
	  rank++;
	  row = libGAP_ELM_PLIST(mat, workrow);
	  if (workrow != rank)
	    {
	      libGAP_SET_ELM_PLIST(mat, workrow, libGAP_ELM_PLIST(mat, rank));
	      libGAP_SET_ELM_PLIST(mat, rank, row);
	    }
	  rowp = libGAP_BLOCKS_GF2VEC(row);
	  if (clearup)
	    for (j = 1; j < rank; j ++)
	      {
		row2 = libGAP_ELM_PLIST(mat, j);
		row2p = libGAP_BLOCKS_GF2VEC(row2);
		if (row2p[block] & mask)
		  libGAP_AddGF2VecToGF2Vec(row2p,rowp, ncols);
	      }
	  for (j = workrow+1; j <= nrows; j++)
	    {
	      row2 = libGAP_ELM_PLIST(mat, j);
	      row2p = libGAP_BLOCKS_GF2VEC(row2);
	      if (row2p[block] & mask)
		libGAP_AddGF2VecToGF2Vec(row2p,rowp, ncols);
	    }
	  
	}
      libGAP_TakeInterrupt();
      
    }
  return rank;
}

/****************************************************************************
**

*F * * * * * * * * * * * *  conversion functions  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  PlainGF2Vec( <list> ) . . .  . convert a GF2 vector into an ordinary list
**
**  'PlainGF2Vec' converts the GF2 vector <list> to a plain list.
*/
static libGAP_Obj libGAP_IsLockedRepresentationVector;


void libGAP_PlainGF2Vec (
    libGAP_Obj                 list )
{
    libGAP_Int                 len;            /* length of <list>                */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 first = 0;          /* first entry                     */
    libGAP_UInt                tnum;


    /* check for representation lock */
    if (libGAP_True == libGAP_DoFilter( libGAP_IsLockedRepresentationVector, list))
      libGAP_ErrorMayQuit("Cannot convert a locked GF2 vector into a plain list", 0, 0);
    
    /* resize the list and retype it, in this order                        */
    len = libGAP_LEN_GF2VEC(list);

    if (len == 0)
      tnum = libGAP_T_PLIST_EMPTY;
    else
      tnum = libGAP_T_PLIST_FFE;
    if (!libGAP_IS_MUTABLE_OBJ(list))
      tnum += libGAP_IMMUTABLE;
    libGAP_RetypeBag( list, tnum);
    libGAP_GROW_PLIST( list, (libGAP_UInt)len );
    libGAP_SET_LEN_PLIST( list, len );

    /* keep the first entry because setting the second destroys the first  */
    if (len == 0)
      libGAP_SET_ELM_PLIST( list, 1 , 0);
    else
      first = libGAP_ELM_GF2VEC(list,1);

    /* wipe out the first entry of the GF2 vector (which becomes the  second */
    /* entry of the plain list, in case the list has length 1.             */
    if (len == 1)
      libGAP_SET_ELM_PLIST( list, 2, 0 );
    
    /* replace the bits by 'GF2One' or 'GF2Zero' as the case may be        */
    /* this must of course be done from the end of the list backwards      */
    for ( i = len;  1 < i;  i-- )
        libGAP_SET_ELM_PLIST( list, i, libGAP_ELM_GF2VEC( list, i ) );
    if (len != 0)
	libGAP_SET_ELM_PLIST( list, 1, first );

    libGAP_CHANGED_BAG(list);
}


/****************************************************************************
**
*F  PlainGF2Mat( <list> ) . . .  . convert a GF2 matrix into an ordinary list
**
**  'PlainGF2Mat' converts the GF2 matrix <list> to a plain list.
*/
void libGAP_PlainGF2Mat (
    libGAP_Obj                 list )
{
    libGAP_Int                 len;            /* length of <list>                */
    libGAP_UInt                i;              /* loop variable                   */

    /* resize the list and retype it, in this order                        */
    len = libGAP_LEN_GF2MAT(list);
    libGAP_RetypeBag( list, libGAP_IS_MUTABLE_OBJ(list) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE );
    libGAP_SET_LEN_PLIST( list, len );

    /* shift the entries to the left                                       */
    for ( i = 1;  i <= len;  i++ ) {
        libGAP_SET_ELM_PLIST( list, i, libGAP_ELM_GF2MAT( list, i ) );
    }
    libGAP_SHRINK_PLIST( list, len );
    libGAP_CHANGED_BAG(list);
}


/****************************************************************************
**
*F  ConvGF2Vec( <list> )  . . . . .  convert a list into a GF2 vector objects
*/
void libGAP_ConvGF2Vec (
    libGAP_Obj                 list )
{
    libGAP_Int                 len;            /* logical length of the vector    */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt                block;          /* one block of the boolean list   */
    libGAP_UInt                bit;            /* one bit of a block              */
    libGAP_Obj                 x;
        
    /* already in the correct representation                               */
    if ( libGAP_IS_GF2VEC_REP(list) ) {
        return;
    }

    /* Otherwise make it a plain list so that we will know where it keeps
       its data -- could do much better in the case of GF(2^n) ectors that actually
       lie over GF(2) */

    if (libGAP_IS_VEC8BIT_REP(list))
      libGAP_PlainVec8Bit(list);
    else
      libGAP_PLAIN_LIST( list );
    
    /* change its representation                                           */
    len   = libGAP_LEN_PLIST(list);

    /* We may have to resize the bag now because a length 1
       plain list is shorter than a length 1 VECGF2 */
    if (libGAP_SIZE_PLEN_GF2VEC(len) > libGAP_SIZE_OBJ(list))
      libGAP_ResizeBag( list, libGAP_SIZE_PLEN_GF2VEC(len) );

    /* now do the work */
    block = 0;
    bit   = 1;
    for ( i = 1;  i <= len;  i++ ) {
      x = libGAP_ELM_PLIST(list, i);
      if (x == libGAP_GF2One)
	block |= bit;
      else if (x != libGAP_GF2Zero)
	{
	  /* might be GF(2) elt written over bigger field */
	  if (libGAP_EQ(x, libGAP_GF2One))
	    block |= bit;
	  else
	    assert(libGAP_EQ(x, libGAP_GF2Zero));
	}
      
      bit = bit << 1;
      if ( bit == 0 || i == len ) {
	libGAP_BLOCK_ELM_GF2VEC(list,i) = block;
	block = 0;
	bit   = 1;
      }
    }

    /* retype and resize bag                                               */
    libGAP_ResizeBag( list, libGAP_SIZE_PLEN_GF2VEC(len) );
    libGAP_SET_LEN_GF2VEC( list, len );
    if ( libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_MUTABLE ) )
        libGAP_TYPE_DATOBJ( list ) = libGAP_TYPE_LIST_GF2VEC;
    else
        libGAP_TYPE_DATOBJ( list ) = libGAP_TYPE_LIST_GF2VEC_IMM;
    libGAP_RetypeBag( list, libGAP_T_DATOBJ );
}


/****************************************************************************
**
*F  FuncCONV_GF2VEC( <self>, <list> ) . . . . . convert into a GF2 vector rep
*/
libGAP_Obj libGAP_FuncCONV_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    /* check whether <list> is a GF2 vector                               */
    libGAP_ConvGF2Vec(list);

    /* return nothing                                                      */
    return 0;
}

/****************************************************************************
**
*F FuncCONV_GF2MAT (<self>, <list> ) . . . convert into a GF2 matrix rep
**
** <list> should be a a list of compressed GF2 vectors
**  
*/
libGAP_Obj libGAP_FuncCONV_GF2MAT( libGAP_Obj self, libGAP_Obj list)
{
  libGAP_UInt len, i;
  libGAP_Obj tmp;
  libGAP_UInt mut;
  len = libGAP_LEN_LIST(list);
  if (len == 0)
    return (libGAP_Obj)0;
  
  libGAP_PLAIN_LIST(list);
  libGAP_GROW_PLIST(list, len+1);
  for (i = len; i > 0 ; i--)
    {
      tmp = libGAP_ELM_PLIST(list, i);
      if (!libGAP_IS_GF2VEC_REP(tmp))
	{
	  int j;
	  for (j = i+1; j <= len; j++)
	    {
	      tmp = libGAP_ELM_PLIST(list, j+1);
	      libGAP_SET_ELM_PLIST(list,j,tmp);
	    }
	  libGAP_ErrorMayQuit("CONV_GF2MAT: argument must be a list of compressed GF2 vectors", 0L, 0L);
	}
      libGAP_TYPE_DATOBJ(tmp) = libGAP_IS_MUTABLE_OBJ(tmp) ? libGAP_TYPE_LIST_GF2VEC_LOCKED: libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
      libGAP_SET_ELM_PLIST(list, i+1, tmp);
    }
  libGAP_SET_ELM_PLIST(list,1,libGAP_INTOBJ_INT(len));
  mut = libGAP_IS_MUTABLE_OBJ(list);
  libGAP_RetypeBag(list, libGAP_T_POSOBJ);
  libGAP_SET_TYPE_POSOBJ(list, mut ? libGAP_TYPE_LIST_GF2MAT : libGAP_TYPE_LIST_GF2MAT_IMM);
  return (libGAP_Obj) 0;
}


/****************************************************************************
**
*F  FuncPLAIN_GF2VEC( <self>, <list> ) . . .  convert back into ordinary list
*/
libGAP_Obj libGAP_FuncPLAIN_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    /* check whether <list> is a GF2 vector                                */
    while ( ! libGAP_IS_GF2VEC_REP(list) ) {
        list = libGAP_ErrorReturnObj(
            "PLAIN_GF2VEC: <list> must be a GF2 vector (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    libGAP_PlainGF2Vec(list);

    /* return nothing                                                      */
    return 0;
}




/****************************************************************************
**
*F  revertbits -- utility function to reverse bit orders
*/


/*   A list of flip values for bytes (i.e. ..xyz -> zyx..) */

static const libGAP_UInt1 libGAP_revertlist [] ={
 0, 128, 64, 192, 32, 160, 96, 224, 16, 144, 80, 208, 48, 176, 112, 240, 8, 
  136, 72, 200, 40, 168, 104, 232, 24, 152, 88, 216, 56, 184, 120, 248, 4, 
  132, 68, 196, 36, 164, 100, 228, 20, 148, 84, 212, 52, 180, 116, 244, 12, 
  140, 76, 204, 44, 172, 108, 236, 28, 156, 92, 220, 60, 188, 124, 252, 2, 
  130, 66, 194, 34, 162, 98, 226, 18, 146, 82, 210, 50, 178, 114, 242, 10, 
  138, 74, 202, 42, 170, 106, 234, 26, 154, 90, 218, 58, 186, 122, 250, 6, 
  134, 70, 198, 38, 166, 102, 230, 22, 150, 86, 214, 54, 182, 118, 246, 14, 
  142, 78, 206, 46, 174, 110, 238, 30, 158, 94, 222, 62, 190, 126, 254, 1, 
  129, 65, 193, 33, 161, 97, 225, 17, 145, 81, 209, 49, 177, 113, 241, 9, 
  137, 73, 201, 41, 169, 105, 233, 25, 153, 89, 217, 57, 185, 121, 249, 5, 
  133, 69, 197, 37, 165, 101, 229, 21, 149, 85, 213, 53, 181, 117, 245, 13, 
  141, 77, 205, 45, 173, 109, 237, 29, 157, 93, 221, 61, 189, 125, 253, 3, 
  131, 67, 195, 35, 163, 99, 227, 19, 147, 83, 211, 51, 179, 115, 243, 11, 
  139, 75, 203, 43, 171, 107, 235, 27, 155, 91, 219, 59, 187, 123, 251, 7, 
  135, 71, 199, 39, 167, 103, 231, 23, 151, 87, 215, 55, 183, 119, 247, 15, 
  143, 79, 207, 47, 175, 111, 239, 31, 159, 95, 223, 63, 191, 127, 255
};

/* Takes an UInt a on n bits and returns the Uint obtained by reverting the
 * bits */
libGAP_UInt libGAP_revertbits(libGAP_UInt a, libGAP_Int n)
{
  libGAP_UInt b,c;
  b=0;
  while (n>8) {
    c=a&0xff; /* last byte */
    a = a>>8;
    b = b<<8;
    b += (libGAP_UInt) libGAP_revertlist[(libGAP_UInt1)c]; /* add flipped */
    n -=8;
  }
  /* cope with the last n bits */
  a &= 0xff;
  b= b<<n;
  c=(libGAP_UInt) libGAP_revertlist[(libGAP_UInt1)a];
  c = c>> (8-n);
  b+=c;
  return b;
}

/****************************************************************************
**
*F  Cmp_GF2Vecs( <vl>, <vr> )   compare GF2 vectors -- internal
**                                    returns -1, 0 or 1
*/
libGAP_Int libGAP_Cmp_GF2VEC_GF2VEC (
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_UInt *              bl;             /* block of <vl>                   */
    libGAP_UInt *              br;             /* block of <vr>                   */
    libGAP_UInt                len,lenl,lenr;  /* length of the list              */
    libGAP_UInt a,b,nb;

    /* get and check the length                                            */
    lenl = libGAP_LEN_GF2VEC(vl);
    lenr = libGAP_LEN_GF2VEC(vr);
    nb=libGAP_NUMBER_BLOCKS_GF2VEC(vl);
    a=libGAP_NUMBER_BLOCKS_GF2VEC(vr);
    if (a<nb) {
      nb = a;
    }

    /* check all blocks                                                    */
    bl = libGAP_BLOCKS_GF2VEC(vl);
    br = libGAP_BLOCKS_GF2VEC(vr);
    for ( i = nb;  1 < i;  i--, bl++, br++ ) {
	/* comparison is numeric of the reverted lists*/
      if (*bl != *br)
	{
	  a=libGAP_revertbits(*bl,libGAP_BIPEB);
	  b=libGAP_revertbits(*br,libGAP_BIPEB);
	  if ( a < b )
            return -1;
	  else return 1;
	}
    }
    

    /* The last block remains */
    len=lenl;
    if (len>lenr) {
      len=lenr;
    }

    /* are both vectors length 0? */
    if (len == 0 ) return 0;

    /* is there still a full block in common? */
    len = len % libGAP_BIPEB;
    if (len == 0) {
      a=libGAP_revertbits(*bl,libGAP_BIPEB);
      b=libGAP_revertbits(*br,libGAP_BIPEB);
    }
    else {
      a=libGAP_revertbits(*bl,len);
      b=libGAP_revertbits(*br,len);
    }

    if (a<b)
      return -1;
    if (a>b)
      return 1;
    
    /* blocks still the same --left length must be smaller to be true */
    if (lenr>lenl)
      return -1;
    if (lenl > lenr)
      return 1;
    
    return 0;

}


/****************************************************************************
**
*F  FuncEQ_GF2VEC_GF2VEC( <self>, <vl>, <vr> )   test equality of GF2 vectors
*/
libGAP_Obj libGAP_FuncEQ_GF2VEC_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
  /* we can do this case MUCH faster if we just want equality */
  if (libGAP_LEN_GF2VEC(vl)  != libGAP_LEN_GF2VEC(vr))
    return libGAP_False;
  return (libGAP_Cmp_GF2VEC_GF2VEC(vl,vr) == 0) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncLEN_GF2VEC( <self>, <list> )  . . . . . . . .  length of a GF2 vector
*/
libGAP_Obj libGAP_FuncLEN_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_INTOBJ_INT(libGAP_LEN_GF2VEC(list));
}


/****************************************************************************
**

*F  FuncELM0_GF2VEC( <self>, <list>, <pos> )  . select an elm of a GF2 vector
**
**  'ELM0_GF2VEC'  returns the element at the  position  <pos> of the boolean
**  list <list>, or `Fail' if <list> has no assigned  object at <pos>.  It is
**  the  responsibility of  the caller to   ensure  that <pos> is  a positive
**  integer.
*/
libGAP_Obj libGAP_FuncELM0_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UInt                p;

    
    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorMayQuit("ELM0_GF2VEC: position must be a small integer, not a %s",
		(libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);
    if ( libGAP_LEN_GF2VEC(list) < p ) {
        return libGAP_Fail;
    }
    else {
        return libGAP_ELM_GF2VEC( list, p );
    }
}


/****************************************************************************
**
*F  FuncELM_GF2VEC( <self>, <list>, <pos> ) . . select an elm of a GF2 vector
**
**  'ELM_GF2VEC' returns the element at the position <pos>  of the GF2 vector
**  <list>.   An  error  is signalled  if  <pos>  is  not bound.    It is the
**  responsibility of the caller to ensure that <pos> is a positive integer.
*/
libGAP_Obj libGAP_FuncELM_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UInt                p;

    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorMayQuit("ELM_GF2VEC: position must be a small integer, not a %s",
		(libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);
    if ( libGAP_LEN_GF2VEC(list) < p ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            p, 0L, "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, p );
    }
    else {
        return libGAP_ELM_GF2VEC( list, p );
    }
}


/****************************************************************************
**
*F  FuncELMS_GF2VEC( <self>, <list>, <poss> ) . . . sublist from a GF2 vector
**
**  'ELMS_GF2VEC' returns a new list containing  the elements at the position
**  given in    the   list  <poss> from  the   vector   <list>.   It  is  the
**  responsibility of the caller to ensure that <poss>  is dense and contains
**  only positive integers.  An error is signalled if an element of <poss> is
**  larger than the length of <list>.
*/
libGAP_Obj libGAP_FuncELMS_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elms;           /* selected sublist, result        */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Int                 lenPoss;        /* length of positions             */
    libGAP_Int                 pos;            /* position as integer             */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Obj                 apos;

    /* get the length of <list>                                            */
    lenList = libGAP_LEN_GF2VEC(list);

    /* general code for arbritrary lists, which are ranges                 */
    if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST(poss);

        /* make the result vector                                          */
        libGAP_NEW_GF2VEC( elms, libGAP_TYPE_LIST_GF2VEC, lenPoss );
        libGAP_SET_LEN_GF2VEC( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1;  i <= lenPoss;  i++ ) {

            /* get next position                                           */

	  apos = libGAP_ELM0_LIST( poss, i);
	  if (!apos || !libGAP_IS_INTOBJ(apos))
	    libGAP_ErrorMayQuit("ELMS_GF2VEC: error at position %d in positions list, entry must be bound to a small integer",
		      i, 0L);
	  pos = libGAP_INT_INTOBJ( apos );
	  if ( lenList < pos ) {
	    libGAP_ErrorMayQuit( "List Elements: <list>[%d] must have a value",
		       pos, 0L );
	    return 0;
	  }
	  
	  /* assign the element into <elms>                              */
	  if ( libGAP_ELM_GF2VEC( list, pos ) == libGAP_GF2One ) {
	    libGAP_BLOCK_ELM_GF2VEC(elms,i) |= libGAP_MASK_POS_GF2VEC(i);
	  }
        }
	
    }

    /* special code for ranges                                             */
    else {

        /* get the length of <positions>, the first elements, and the inc. */
        lenPoss = libGAP_GET_LEN_RANGE(poss);
        pos = libGAP_GET_LOW_RANGE(poss);
        inc = libGAP_GET_INC_RANGE(poss);

        /* check that no <position> is larger than <lenList>               */
        if ( lenList < pos ) {
            libGAP_ErrorMayQuit( "List Elements: <list>[%d] must have a value",
                       pos, 0L );
            return 0;
        }
        if ( lenList < pos + (lenPoss-1) * inc ) {
            libGAP_ErrorMayQuit( "List Elements: <list>[%d] must have a value",
                       pos + (lenPoss-1) * inc, 0L );
            return 0;
        }

        /* make the result vector                                          */
        libGAP_NEW_GF2VEC( elms, libGAP_TYPE_LIST_GF2VEC, lenPoss );
        libGAP_SET_LEN_GF2VEC( elms, lenPoss );
	
	/* increment 1 ranges is a block copy */
	if (inc == 1)
	  libGAP_CopySection_GF2Vecs(list, elms, pos, 1, lenPoss);

        /* loop over the entries of <positions> and select                 */
        else {
           for ( i = 1;  i <= lenPoss;  i++, pos += inc ) {
                if ( libGAP_ELM_GF2VEC(list,pos) == libGAP_GF2One ) {
                    libGAP_BLOCK_ELM_GF2VEC(elms,i) |= libGAP_MASK_POS_GF2VEC(i);
                }
            }
        }
    }

    /* return the result                                                   */
    return elms;
}


/****************************************************************************
**
*F  FuncASS_GF2VEC( <self>, <list>, <pos>, <elm> ) set an elm of a GF2 vector
**
**  'ASS_GF2VEC' assigns the element  <elm> at the position  <pos> to the GF2
**  vector <list>.
**
**  It is the responsibility of the caller  to ensure that <pos> is positive,
**  and that <elm> is not 0.
*/

static libGAP_Obj libGAP_ConvertToVectorRep;	/* BH: changed to static */

libGAP_Obj libGAP_FuncASS_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos,
    libGAP_Obj                 elm )
{
    libGAP_UInt                p;

    /* check that <list> is mutable                                        */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        libGAP_ErrorReturnVoid(
            "Lists Assignment: <list> must be a mutable list",
            0L, 0L,
            "you can 'return;' and ignore the assignment" );
        return 0;
    }

    /* get the position                                                    */
    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorMayQuit("ASS_VEC8BIT: position must be a small integer, not a %s",
		(libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);

    /* if <elm> is Z(2) or 0*Z(2) and the position is OK, keep rep         */
    if ( p <= libGAP_LEN_GF2VEC(list)+1 ) {
        if ( libGAP_LEN_GF2VEC(list)+1 == p ) {
	  if (libGAP_DoFilter(libGAP_IsLockedRepresentationVector, list) == libGAP_True)
	    libGAP_ErrorMayQuit("Assignment forbidden beyond the end of locked GF2 vector", 0, 0);
	  libGAP_ResizeBag( list, libGAP_SIZE_PLEN_GF2VEC(p) );
	  libGAP_SET_LEN_GF2VEC( list, p );
        }
        if ( libGAP_EQ(libGAP_GF2One,elm) ) {
            libGAP_BLOCK_ELM_GF2VEC(list,p) |= libGAP_MASK_POS_GF2VEC(p);
        }
        else if ( libGAP_EQ(libGAP_GF2Zero,elm) ) {
            libGAP_BLOCK_ELM_GF2VEC(list,p) &= ~libGAP_MASK_POS_GF2VEC(p);
        }
	else if (libGAP_IS_FFE(elm) && libGAP_CHAR_FF(libGAP_FLD_FFE(elm)) == 2 && libGAP_DEGR_FF(libGAP_FLD_FFE(elm)) <= 8)
	  {
	    /*	    Pr("Rewriting GF2 vector over larger field",0,0); */
	    libGAP_RewriteGF2Vec(list, libGAP_SIZE_FF(libGAP_FLD_FFE(elm)));
	    libGAP_FuncASS_VEC8BIT(self, list, pos, elm);
	  }
        else
	  {
	    /* 	    Pr("arbitrary assignment (GF2)",0,0); */
	    libGAP_PlainGF2Vec(list);
	    libGAP_ASS_LIST( list, p, elm );
	  }
    }
    else {
      /*       Pr("arbitrary assignment 2 (GF2)",0,0); */
        libGAP_PlainGF2Vec(list);
        libGAP_ASS_LIST( list, p, elm );
    }
    return 0;
}

/****************************************************************************
**
*F  FuncPLAIN_GF2MAT( <self>, <list> ) . . .  convert back into ordinary list
*/
libGAP_Obj libGAP_FuncPLAIN_GF2MAT (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    libGAP_PlainGF2Mat(list);

    /* return nothing                                                      */
    return 0;
}


/****************************************************************************
**
*F  FuncASS_GF2MAT( <self>, <list>, <pos>, <elm> ) set an elm of a GF2 matrix
**
**  'ASS_GF2MAT' assigns the element  <elm> at the position  <pos> to the GF2
**  matrix <list>.
**
**  It is the responsibility of the caller  to ensure that <pos> is positive,
**  and that <elm> is not 0.
*/
libGAP_Obj libGAP_FuncASS_GF2MAT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos,
    libGAP_Obj                 elm )
{
    libGAP_UInt                p;

    /* check that <list> is mutable                                        */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        libGAP_ErrorReturnVoid(
            "Lists Assignment: <list> must be a mutable list",
            0L, 0L,
            "you can 'return;' and ignore the assignment" );
        return 0;
    }

    /* get the position                                                    */
    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorMayQuit("ASS_GF2MAT: position must be a small integer, not a %s",
		(libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);

    /* if <elm> is a GF2 vector and the length is OK, keep the rep         */
    if ( ! libGAP_IS_GF2VEC_REP(elm)  ) {
        libGAP_PlainGF2Mat(list);
        libGAP_ASS_LIST( list, p, elm );
    }
    else if ( p == 1 && 1 >= libGAP_LEN_GF2MAT(list) ) {
        libGAP_ResizeBag( list, libGAP_SIZE_PLEN_GF2MAT(p) );
	libGAP_TYPE_DATOBJ(elm) = libGAP_IS_MUTABLE_OBJ(elm) ? libGAP_TYPE_LIST_GF2VEC_LOCKED : libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
        libGAP_SET_ELM_GF2MAT( list, p, elm );
	libGAP_CHANGED_BAG(list);
    }
    else if ( p > libGAP_LEN_GF2MAT(list)+1 ) {
        libGAP_PlainGF2Mat(list);
        libGAP_ASS_LIST( list, p, elm );
    }
    else if ( libGAP_LEN_GF2VEC(elm) == libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(list,1)) ) {
        if ( libGAP_LEN_GF2MAT(list)+1 == p ) {
            libGAP_ResizeBag( list, libGAP_SIZE_PLEN_GF2MAT(p) );
            libGAP_SET_LEN_GF2MAT( list, p );
        }
	libGAP_TYPE_DATOBJ(elm) = libGAP_IS_MUTABLE_OBJ(elm) ? libGAP_TYPE_LIST_GF2VEC_LOCKED : libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
        libGAP_SET_ELM_GF2MAT( list, p, elm );
        libGAP_CHANGED_BAG(list);
    }
    else {
        libGAP_PlainGF2Mat(list);
        libGAP_ASS_LIST( list, p, elm );
    }
    return 0;
}


/****************************************************************************
**

*F  FuncUNB_GF2VEC( <self>, <list>, <pos> ) . unbind position of a GF2 vector
**
**  'UNB_GF2VEC' unbind  the element at  the position  <pos> in  a GF2 vector
**  <list>.
**
**  It is the responsibility of the caller  to ensure that <pos> is positive.
*/
libGAP_Obj libGAP_FuncUNB_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UInt                p;

    /* check that <list> is mutable                                        */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        libGAP_ErrorReturnVoid(
            "Unbind: <list> must be a mutable list",
            0L, 0L,
            "you can 'return;' and ignore the operation" );
        return 0;
    }

    if (libGAP_DoFilter(libGAP_IsLockedRepresentationVector, list) == libGAP_True)
    {
      libGAP_ErrorReturnVoid( "Unbind forbidden on locked GF2 vector",
		          0L, 0L,
            "you can 'return;' and ignore the operation" );
        return 0;
    }

    /* get the position                                                    */
    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorMayQuit("UNB_GF2VEC: position must be a small integer, not a %s",
		(libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);

    /* if we unbind the last position keep the representation              */
    if ( libGAP_LEN_GF2VEC(list) < p ) {
        ;
    }
    else if ( libGAP_LEN_GF2VEC(list) == p ) {
        libGAP_ResizeBag( list, libGAP_SIZE_PLEN_GF2VEC(p-1) );
        libGAP_SET_LEN_GF2VEC( list, p-1 );
    }
    else {
        libGAP_PlainGF2Vec(list);
        libGAP_UNB_LIST( list, p );
    }
    return 0;
}


/****************************************************************************
**
*F  FuncUNB_GF2MAT( <self>, <list>, <pos> ) . unbind position of a GF2 matrix
**
**  'UNB_GF2VEC' unbind  the element at  the position  <pos> in  a GF2 matrix
**  <list>.
**
**  It is the responsibility of the caller  to ensure that <pos> is positive.
*/
libGAP_Obj libGAP_FuncUNB_GF2MAT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UInt                p;

    /* check that <list> is mutable                                        */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        libGAP_ErrorReturnVoid(
            "Lists Assignment: <list> must be a mutable list",
            0L, 0L,
            "you can 'return;' and ignore the assignment" );
        return 0;
    }

    /* get the position                                                    */
    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorMayQuit("UNB_GF2MAT: position must be a small integer, not a %s",
		(libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);

    /* if we unbind the last position keep the representation              */
    if ( p > 1 && libGAP_LEN_GF2MAT(list) < p ) {
        ;
    }
    else if ( libGAP_LEN_GF2MAT(list) == p ) {
        libGAP_ResizeBag( list, libGAP_SIZE_PLEN_GF2MAT(p-1) );
        libGAP_SET_LEN_GF2MAT( list, p-1 );
    }
    else {
        libGAP_PlainGF2Mat(list);
        libGAP_UNB_LIST( list, p );
    }
    return 0;
}


/****************************************************************************
**

*F * * * * * * * * * * * * arithmetic operations  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  FuncZERO_GF2VEC( <self>, <mat> )  . . . . . . . . . . . . zero GF2 vector
**
**  return the zero vector over GF2 of the same length as <mat>.
*/
libGAP_Obj libGAP_FuncZERO_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 mat )
{
    libGAP_Obj                 zero;
    libGAP_UInt                len;

    /* create a new GF2 vector                                             */
    len = libGAP_LEN_GF2VEC(mat);
    libGAP_NEW_GF2VEC( zero, libGAP_TYPE_LIST_GF2VEC, len );
    libGAP_SET_LEN_GF2VEC( zero, len );
    return zero;
}

/****************************************************************************
**
*F  FuncZERO_GF2VEC_2( <self>, <len>) . . . . . . . . . zero GF2 vector
**
**  return the zero vector over GF2 of length <len>
*/
libGAP_Obj libGAP_FuncZERO_GF2VEC_2 (
    libGAP_Obj                 self,
    libGAP_Obj                 len )
{
    libGAP_Obj                 zero;

    /* create a new GF2 vector*/
    if (!libGAP_IS_INTOBJ(len))
      libGAP_ErrorMayQuit("ZERO_GF2VEC2: length must be a small integer, not a %s",
		(libGAP_Int)libGAP_TNAM_OBJ(len),0L);
    
    libGAP_NEW_GF2VEC( zero, libGAP_TYPE_LIST_GF2VEC, libGAP_INT_INTOBJ(len) );
    libGAP_SET_LEN_GF2VEC( zero, libGAP_INT_INTOBJ(len) );
    return zero;
}


/****************************************************************************
**
*F  FuncINV_GF2MAT_MUTABLE( <self>, <mat> ) . .. . . . .  inverse GF2 matrix
**
** This might now be redundant, a library method using
**  INVERSE_PLIST_GF2VECS_DESTRUCTIVE
** might do just as good a job
*/
libGAP_Obj libGAP_FuncINV_GF2MAT_MUTABLE (
    libGAP_Obj                 self,
    libGAP_Obj                 mat )
{
    libGAP_UInt                len;

    len = libGAP_LEN_GF2MAT(mat);
    if ( len != 0 ) {
        if ( len != libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(mat,1)) ) {
            mat = libGAP_ErrorReturnObj( "<matrix> must be square", 0, 0,
                                  "you can replace <matrix> via 'return <matrix>;'" );
	    return libGAP_INV(mat);
        }
    }
    return libGAP_InverseGF2Mat(mat,2);
}

/****************************************************************************
**
*F  FuncINV_GF2MAT_SAME_MUTABILITY( <self>, <mat> ) . ...  inverse GF2 matrix
**
** This might now be redundant, a library method using
**  INVERSE_PLIST_GF2VECS_DESTRUCTIVE
** might do just as good a job
*/
libGAP_Obj libGAP_FuncINV_GF2MAT_SAME_MUTABILITY (
    libGAP_Obj                 self,
    libGAP_Obj                 mat )
{
    libGAP_UInt                len;

    len = libGAP_LEN_GF2MAT(mat);
    if ( len != 0 ) {
        if ( len != libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(mat,1)) ) {
            mat = libGAP_ErrorReturnObj( "<matrix> must be square", 0, 0,
                                  "you can replace <matrix> via 'return <matrix>;'" );
	    return libGAP_INV_MUT(mat);
        }
    }
    return libGAP_InverseGF2Mat(mat,1);
}

/****************************************************************************
**
*F  FuncINV_GF2MAT_IMMUTABLE( <self>, <mat> ) . .. . . .  inverse GF2 matrix
**
** This might now be redundant, a library method using
**  INVERSE_PLIST_GF2VECS_DESTRUCTIVE
** might do just as good a job
*/
libGAP_Obj libGAP_FuncINV_GF2MAT_IMMUTABLE (
    libGAP_Obj                 self,
    libGAP_Obj                 mat )
{
    libGAP_UInt                len;

    len = libGAP_LEN_GF2MAT(mat);
    if ( len != 0 ) {
        if ( len != libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(mat,1)) ) {
	  libGAP_Obj inv;
	  mat = libGAP_ErrorReturnObj( "<matrix> must be square", 0, 0,
				"you can replace <matrix> via 'return <matrix>;'" );
	  inv = libGAP_INV(mat);
	  libGAP_MakeImmutable(inv);
	  return inv;
        }
    }
    return libGAP_InverseGF2Mat(mat,0);
}


/****************************************************************************
**
*F  FuncINV_PLIST_GF2VECS_DESTRUCTIVE( <self>, <list> ) . . .invert possible GF2 matrix
*/
libGAP_Obj libGAP_FuncINV_PLIST_GF2VECS_DESTRUCTIVE( libGAP_Obj self, libGAP_Obj list )
{
  libGAP_UInt len,i;
  libGAP_Obj row;
  len = libGAP_LEN_PLIST(list);
  for (i = 1; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(list,i);
      if (!libGAP_IS_GF2VEC_REP(row) || libGAP_LEN_GF2VEC(row) != len)
	return libGAP_TRY_NEXT_METHOD;
    }
  if ( len == 0 ) {
    return libGAP_CopyObj(list,1);
  }
  if (len == 1 ) {
    row = libGAP_ELM_PLIST(list,1);
    if (libGAP_BLOCKS_GF2VEC(row)[0] & 1)
      {
	return libGAP_CopyObj(list, 1);
	}
    else
      return libGAP_Fail;
  }
  return libGAP_InversePlistGF2VecsDesstructive(list);
  
}


/****************************************************************************
**
*F  FuncSUM_GF2VEC_GF2VEC( <self>, <vl>, <vr> ) . . . . .  sum of GF2 vectors
**
**  'FuncSUM_GF2VEC_GF2VEC' returns the sum  of the two gf2-vectors <vl>  and
**  <vr>.  The sum is a new gf2-vector, where each element is  the sum of the
**  corresponding entries of <vl>  and <vr>.  The  major work is done  in the
**  routine   'AddPartialGF2VecGF2Vec'     which     is         called   from
**  'FuncSUM_GF2VEC_GF2VEC'.
**
**  'FuncSUM_GF2VEC_GF2VEC'  is  an improved  version of 'SumListList', which
**  does not call 'SUM' but uses bit operations instead.
*/
libGAP_Obj libGAP_FuncSUM_GF2VEC_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
    libGAP_Obj                 sum;            /* sum, result                     */
    libGAP_UInt ll,lr;

    ll = libGAP_LEN_GF2VEC(vl);
    lr = libGAP_LEN_GF2VEC(vr);
    
    
    if (ll < lr)
      {
	sum = libGAP_ShallowCopyVecGF2(vr);
	libGAP_AddGF2VecToGF2Vec(libGAP_BLOCKS_GF2VEC(sum),libGAP_BLOCKS_GF2VEC(vl),ll);
      }
    else
      {
	sum = libGAP_ShallowCopyVecGF2(vl);
	libGAP_AddGF2VecToGF2Vec(libGAP_BLOCKS_GF2VEC(sum),libGAP_BLOCKS_GF2VEC(vr),lr);
      }

    if (!libGAP_IS_MUTABLE_OBJ(vl) && !libGAP_IS_MUTABLE_OBJ(vr))
      libGAP_TYPE_POSOBJ(sum) = libGAP_TYPE_LIST_GF2VEC_IMM;
    
    return sum;
}

/****************************************************************************
**
*F  FuncMULT_ROW_VECTOR_GF2VECS_2( <self>, <vl>, <mul> )
**                                      . . . . .  sum of GF2 vectors
**
*/
libGAP_Obj libGAP_FuncMULT_ROW_VECTOR_GF2VECS_2 (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 mul )
{
    if (libGAP_EQ(mul,libGAP_GF2One))
      return (libGAP_Obj) 0;
    else if (libGAP_EQ(mul,libGAP_GF2Zero))
      {
	libGAP_AddCoeffsGF2VecGF2Vec(vl,vl);
	return (libGAP_Obj) 0;
      }
    else
      return libGAP_TRY_NEXT_METHOD;
}


/****************************************************************************
**
*F  FuncPROD_GF2VEC_GF2VEC( <self>, <vl>, <vr> )  . .  product of GF2 vectors
**
**  'FuncPROD_GF2VEC_GF2VEC' returns the product of  the two GF2 vectors <vl>
**  and <vr>.  The product is either `GF2One' or `GF2Zero'.
*/
libGAP_Obj libGAP_FuncPROD_GF2VEC_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
    return libGAP_ProdGF2VecGF2Vec( vl, vr );
}


/****************************************************************************
**
*F  FuncPROD_GF2VEC_GF2MAT( <self>, <vl>, <vr> ) product of GF2 vector/matrix
**
**  'FuncPROD_GF2VEC_GF2MAT' returns  the product of the GF2 vectors <vl> and
**  the GF2 matrix <vr>.
**
**  The  product is  again a  GF2 vector.  It  is  the  responsibility of the
**  caller to ensure that <vl> is a  GF2 vector, <vr>  a GF2 matrix.
*/
libGAP_Obj libGAP_FuncPROD_GF2VEC_GF2MAT (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
    return libGAP_ProdGF2VecGF2Mat( vl, vr );
}

/****************************************************************************
**
*F  FuncPROD_GF2MAT_GF2MAT( <self>, <ml>, <mr> ) product of GF2 vector/matrix
**
**  'FuncPROD_GF2MAT_GF2MAT' returns  the product of the GF2 matricess <ml> and
**  <mr>.
**
**  The  product is  again a  GF2 matrix.  It  is  the  responsibility of the
**  caller to ensure that <ml> and <mr> are  GF2 matrices
*/
libGAP_Obj libGAP_FuncPROD_GF2MAT_GF2MAT (
    libGAP_Obj                 self,
    libGAP_Obj                 ml,
    libGAP_Obj                 mr )
{
  libGAP_UInt lenl = libGAP_LEN_GF2MAT(ml);
  libGAP_UInt lenm;
  if (lenl >= 128)
    {
      lenm = libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(ml,1));
      if (lenm >= 128 && lenm == libGAP_LEN_GF2MAT(mr) && libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(mr,1)) >= 128)
	{
	  return libGAP_ProdGF2MatGF2MatAdvanced( ml, mr, 8, (lenm+255)/256);
	}
    }
    return libGAP_ProdGF2MatGF2MatSimple( ml, mr );
}

/****************************************************************************
**
*F  FuncPROD_GF2MAT_GF2MAT_SIMPLE( <self>, <ml>, <mr> ) product of GF2 vector/matrix
**
**  'FuncPROD_GF2MAT_GF2MAT' returns  the product of the GF2 matricess <ml> and
**  <mr>. It never uses grease or blocking.
**
**  The  product is  again a  GF2 matrix.  It  is  the  responsibility of the
**  caller to ensure that <ml> and <mr> are  GF2 matrices
*/
libGAP_Obj libGAP_FuncPROD_GF2MAT_GF2MAT_SIMPLE (
    libGAP_Obj                 self,
    libGAP_Obj                 ml,
    libGAP_Obj                 mr )
{
    return libGAP_ProdGF2MatGF2MatSimple( ml, mr );
}


/****************************************************************************
**
*F  FuncPROD_GF2MAT_GF2MAT_ADVANCED( <self>, <ml>, <mr>, <greaselevel>, <blocksize> ) 
**
**  'FuncPROD_GF2MAT_GF2MAT_ADVANCED' returns  the product of the GF2 matricess <ml> and
**  <mr> using grease level <greaselevel> and block size <blocksize>
**
**  The  product is  again a  GF2 matrix.  It  is  the  responsibility of the
**  caller to ensure that <ml> and <mr> are  GF2 matrices
*/
libGAP_Obj libGAP_FuncPROD_GF2MAT_GF2MAT_ADVANCED (
    libGAP_Obj                 self,
    libGAP_Obj                 ml,
    libGAP_Obj                 mr,
    libGAP_Obj                 greaselevel,
    libGAP_Obj                 blocksize)
{
    return libGAP_ProdGF2MatGF2MatAdvanced( ml, mr, libGAP_INT_INTOBJ(greaselevel), libGAP_INT_INTOBJ(blocksize) );
}


/****************************************************************************
**
*F  FuncPROD_GF2MAT_GF2VEC( <self>, <vl>, <vr> ) product of GF2 matrix/vector
**
**  'FuncPROD_GF2VEC_GF2MAT' returns  the product of the GF2 matrix  <vl> and
**  the GF2 vector <vr>.
**
**  The  product is  again a  GF2 vector.  It  is  the  responsibility of the
**  caller to ensure that <vr> is a  GF2 vector, <vl>  a GF2 matrix.
*/
libGAP_Obj libGAP_FuncPROD_GF2MAT_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
    return libGAP_ProdGF2MatGF2Vec( vl, vr );
}


/****************************************************************************
**
*F  FuncADDCOEFFS_GF2VEC_GF2VEC_MULT( <self>, <vl>, <vr>, <mul> ) GF2 vectors
*/
libGAP_Obj libGAP_FuncADDCOEFFS_GF2VEC_GF2VEC_MULT (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr,
    libGAP_Obj                 mul )
{
    /* do nothing if <mul> is zero                                         */
    if ( libGAP_EQ(mul,libGAP_GF2Zero) ) {
        return libGAP_INTOBJ_INT( libGAP_RightMostOneGF2Vec(vl) );
    }

    /* add if <mul> is one                                                 */
    if ( libGAP_EQ(mul,libGAP_GF2One) ) {
        return libGAP_AddCoeffsGF2VecGF2Vec( vl, vr );
    }

    /* try next method                                                     */
    return libGAP_TRY_NEXT_METHOD;
}

/****************************************************************************
**
*F  FuncADDCOEFFS_GF2VEC_GF2VEC_MULT( <self>, <vl>, <vr>, <mul>, <from>, <to> )
**  GF2 vectors
*/
libGAP_Obj libGAP_FuncADDCOEFFS_GF2VEC_GF2VEC_MULT_LIMS (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr,
    libGAP_Obj                 mul )
{
    /* do nothing if <mul> is zero                                         */
    if ( libGAP_EQ(mul,libGAP_GF2Zero) ) {
        return libGAP_INTOBJ_INT( libGAP_RightMostOneGF2Vec(vl) );
    }

    /* add if <mul> is one                                                 */
    if ( libGAP_EQ(mul,libGAP_GF2One) ) {
        return libGAP_AddCoeffsGF2VecGF2Vec( vl, vr );
    }

    /* try next method                                                     */
    return libGAP_TRY_NEXT_METHOD;
}


/****************************************************************************
**
*F  FuncADDCOEFFS_GF2VEC_GF2VEC( <self>, <vl>, <vr> ) . . . . . . GF2 vectors
*/
libGAP_Obj libGAP_FuncADDCOEFFS_GF2VEC_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
    return libGAP_AddCoeffsGF2VecGF2Vec( vl, vr );
}


/****************************************************************************
**
*F  FuncSHRINKCOEFFS_GF2VEC( <self>, <vec> )  . . . . . remove trailing zeros
*/
libGAP_Obj libGAP_FuncSHRINKCOEFFS_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vec )
{
    libGAP_UInt                len;
    libGAP_UInt                nbb;
    libGAP_UInt                onbb;
    libGAP_UInt *              ptr;
    libGAP_UInt		off;

    /* get length and number of blocks                                     */
    len = libGAP_LEN_GF2VEC(vec);
    if ( len == 0 ) {
        return libGAP_INTOBJ_INT(0);
    }

    nbb = ( len + libGAP_BIPEB - 1 ) / libGAP_BIPEB;
    onbb = nbb; 
    ptr = libGAP_BLOCKS_GF2VEC(vec) + (nbb-1);
    
    /* number of insignificant bit positions in last word */
    off = libGAP_BIPEB - ((len-1)%libGAP_BIPEB+1); 
    
    /* mask out the last bits */
#ifdef libGAP_SYS_IS_64_BIT
    *ptr &= 0xffffffffffffffff >> off;
#else
    *ptr &= 0xffffffff >> off;
#endif

    /* find last non-trivial block */
    while ( 0 < nbb && ! *ptr ) {
        nbb--;
        ptr--;
    }
    /* did the block number change? */
    if (nbb < onbb) {
      len = nbb * libGAP_BIPEB;
    }

    /* find position inside this block                   */
    /* we are guaranteed not to cross a block boundary ! */
    while ( 0 < len && ! ( *ptr & libGAP_MASK_POS_GF2VEC(len) ) ) {
        len--;
    }
    libGAP_ResizeBag( vec, libGAP_SIZE_PLEN_GF2VEC(len) );
    libGAP_SET_LEN_GF2VEC( vec, len );
    return libGAP_INTOBJ_INT(len);
}

/****************************************************************************
**
*F  FuncPOSITION_NONZERO_GF2VEC( <self>, <vec>, <zero>) ..find first non-zero
**
**  The pointless zero argument is because this is a method for PositionNot
**  It is *not* used in the code and can be replaced by a dummy argument.
*/

libGAP_UInt libGAP_PositionNonZeroGF2Vec ( libGAP_Obj vec, libGAP_UInt from)
{
    libGAP_UInt                len;
    libGAP_UInt                nbb;
    libGAP_UInt                nb;
    libGAP_UInt *              ptr;
    libGAP_UInt                pos;

    /* get length and number of blocks                                     */
    len = libGAP_LEN_GF2VEC(vec);
    if ( len == 0 ) {
      return 1;
    }


    nbb = from / libGAP_BIPEB;
    pos = from % libGAP_BIPEB;
    ptr = libGAP_BLOCKS_GF2VEC(vec)+nbb;
    if (pos) /* partial block to check */
      {
	pos = from+1;
	while ( (pos - 1)%libGAP_BIPEB  && pos <= len)
	  {
	    if ((*ptr) & libGAP_MASK_POS_GF2VEC(pos)) 
	      return (pos);
	    pos++;
	  }
	if (pos > len)
	  return len+1;
	nbb++;
	ptr++;
      }
    /* find first non-trivial block                                         */
    nb = libGAP_NUMBER_BLOCKS_GF2VEC(vec);
    while ( nbb < nb && ! *ptr ) {
        nbb++;
        ptr++;
    }

    /* find position inside this block                                     */
    pos = nbb * libGAP_BIPEB + 1;
    while ( pos <= len && ! ( *ptr & libGAP_MASK_POS_GF2VEC(pos) ) ) {
        pos++;
    }
    /* as the code is intended to run over, trailing 1's are innocent */
    if (pos <= len)
      return pos;
    else
      return len+1;
}


libGAP_Obj libGAP_FuncPOSITION_NONZERO_GF2VEC(
    libGAP_Obj                 self,
    libGAP_Obj                 vec,
    libGAP_Obj                 zero)
{
  return libGAP_INTOBJ_INT(libGAP_PositionNonZeroGF2Vec(vec, 0));
}

libGAP_Obj libGAP_FuncPOSITION_NONZERO_GF2VEC3(
    libGAP_Obj                 self,
    libGAP_Obj                 vec,
    libGAP_Obj                 zero,
    libGAP_Obj                 from)
{
  return libGAP_INTOBJ_INT(libGAP_PositionNonZeroGF2Vec(vec, libGAP_INT_INTOBJ(from)));
}



libGAP_Obj libGAP_FuncCOPY_SECTION_GF2VECS(libGAP_Obj self, libGAP_Obj src, libGAP_Obj dest, libGAP_Obj from, libGAP_Obj to, libGAP_Obj howmany) {
  if (!libGAP_IS_GF2VEC_REP(src) ||
      !libGAP_IS_GF2VEC_REP(dest) ||
      !libGAP_IS_INTOBJ(from) || 
      !libGAP_IS_INTOBJ(to) || 
      !libGAP_IS_INTOBJ(howmany)) 
    libGAP_ErrorMayQuit("Bad argument types", 0,0);
  libGAP_Int ifrom = libGAP_INT_INTOBJ(from);
  libGAP_Int ito = libGAP_INT_INTOBJ(to);
  libGAP_Int ihowmany = libGAP_INT_INTOBJ(howmany);
  libGAP_UInt lens = libGAP_LEN_GF2VEC(src);
  libGAP_UInt lend = libGAP_LEN_GF2VEC(dest);
  if (ifrom <= 0 || ito <= 0 ||
      ihowmany < 0 || ifrom + ihowmany -1 > lens || ito + ihowmany -1 > lend)
    libGAP_ErrorMayQuit("Bad argument values",0,0);
  if (!libGAP_IS_MUTABLE_OBJ(dest))
    libGAP_ErrorMayQuit("Immutable destination vector", 0,0);
  libGAP_CopySection_GF2Vecs(src, dest, (libGAP_UInt)ifrom, (libGAP_UInt)ito, (libGAP_UInt)ihowmany);
  return (libGAP_Obj) 0;
}


/****************************************************************************
**
*F  FuncAPPEND_VECGF2( <self>, <vecl>, <vecr> )
**
*/

libGAP_Obj libGAP_FuncAPPEND_VECGF2( libGAP_Obj self, libGAP_Obj vecl, libGAP_Obj vecr )
{
  libGAP_UInt lenl, lenr;
  lenl = libGAP_LEN_GF2VEC(vecl);
  lenr = libGAP_LEN_GF2VEC(vecr);
  if (libGAP_True == libGAP_DoFilter(libGAP_IsLockedRepresentationVector, vecl) && lenr > 0)
    {
      libGAP_ErrorMayQuit("Append to locked compressed vector is forbidden", 0, 0);
      return 0;
    }
  libGAP_ResizeBag(vecl, libGAP_SIZE_PLEN_GF2VEC(lenl+lenr));
  libGAP_CopySection_GF2Vecs(vecr, vecl, 1, lenl+1, lenr);
  libGAP_SET_LEN_GF2VEC(vecl,lenl+lenr);
  return (libGAP_Obj) 0;
}


/****************************************************************************
**
*F  FuncSHALLOWCOPY_VECGF2( <self>, <vec> )
**
*/

libGAP_Obj libGAP_FuncSHALLOWCOPY_VECGF2( libGAP_Obj self, libGAP_Obj vec)
{
  return libGAP_ShallowCopyVecGF2(vec);
}

/****************************************************************************
**
*F  FuncSUM_GF2MAT_GF2MAT( <self>, <matl>, <matr> )
**
*/


libGAP_Obj libGAP_FuncSUM_GF2MAT_GF2MAT( libGAP_Obj self, libGAP_Obj matl, libGAP_Obj matr)
{
  libGAP_UInt ll,lr,ls, lm,wl,wr,ws,wm;
  libGAP_Obj sum;
  libGAP_Obj vl,vr, sv;
  libGAP_UInt i;
  libGAP_Obj rtype;
  ll = libGAP_LEN_GF2MAT(matl);
  lr = libGAP_LEN_GF2MAT(matr);
  if (ll > lr)
    {
      ls = ll;
      lm = lr;
    }
  else
    {
      ls = lr;
      lm = ll;
    }
  wl = libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(matl, 1));
  wr = libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(matr, 1));
  if (wl > wr)
    {
      ws = wl;
      wm = wr;
    }
  else
    {
      ws = wr;
      wm = wl;
    }

  /* In this case, the result is not rectangular */
  
  if ((ll > lr && wr > wl) || (ll < lr && wr < wl)) 
    return libGAP_TRY_NEXT_METHOD;

  
  sum = libGAP_NewBag(libGAP_T_POSOBJ, libGAP_SIZE_PLEN_GF2MAT( ls ));
  if (libGAP_IS_MUTABLE_OBJ(matl) || libGAP_IS_MUTABLE_OBJ(matr))
    {
      libGAP_TYPE_POSOBJ(sum) = libGAP_TYPE_LIST_GF2MAT;
      if (libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(matl,1)) || libGAP_IS_MUTABLE_OBJ(libGAP_ELM_GF2MAT(matr,1)))
	rtype = libGAP_TYPE_LIST_GF2VEC_LOCKED;
      else
	rtype = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    }
  else
    {
      libGAP_TYPE_POSOBJ(sum) = libGAP_TYPE_LIST_GF2MAT_IMM;
      rtype = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    }
  
  libGAP_SET_LEN_GF2MAT(sum, ls);
  for (i = 1; i <= lm; i++)
    {

      /* copy the longer vector and add the shorter */
      if (wl == ws) 
	{
	  sv = libGAP_ShallowCopyVecGF2( libGAP_ELM_GF2MAT( matl, i));
	  vr = libGAP_ELM_GF2MAT( matr, i);
	  libGAP_AddGF2VecToGF2Vec(libGAP_BLOCKS_GF2VEC(sv), libGAP_BLOCKS_GF2VEC(vr), wm);
	}
      else
	{
	  sv = libGAP_ShallowCopyVecGF2( libGAP_ELM_GF2MAT(matr, i));
	  vl = libGAP_ELM_GF2MAT( matl, i);
	  libGAP_AddGF2VecToGF2Vec(libGAP_BLOCKS_GF2VEC(sv), libGAP_BLOCKS_GF2VEC(vl), wm);
	}
      
      libGAP_TYPE_DATOBJ(sv) = rtype;
      libGAP_SET_ELM_GF2MAT( sum, i, sv);
      libGAP_CHANGED_BAG(sum);
    }
  for (; i <= ls; i++)
    {
      if (ll > lr)
	sv = libGAP_ELM_GF2MAT(matl, i);
      else
	sv = libGAP_ELM_GF2MAT(matr, i);

      if (rtype == libGAP_TYPE_LIST_GF2VEC_LOCKED)
	sv = libGAP_ShallowCopyVecGF2( sv );
      
      libGAP_TYPE_DATOBJ(sv) = rtype;
      libGAP_SET_ELM_GF2MAT( sum, i, sv);
      libGAP_CHANGED_BAG(sum);      
    }
  return sum;
}


/****************************************************************************
**
*F  FuncTRANSPOSED_GF2MAT( <self>, <mat>)
**
*/
libGAP_Obj libGAP_FuncTRANSPOSED_GF2MAT( libGAP_Obj self, libGAP_Obj mat)
{
  libGAP_UInt l,w;
  libGAP_Obj tra,row;
  libGAP_Obj typ,r1;
  libGAP_UInt vals[libGAP_BIPEB];
  libGAP_UInt mask,val,bit;
  libGAP_UInt imod,nrb,nstart;
  libGAP_UInt i,j,k,n;
  libGAP_UInt * ptr;

  /* check argument */
  if (libGAP_TNUM_OBJ(mat) != libGAP_T_POSOBJ) {
     mat = libGAP_ErrorReturnObj(
            "TRANSPOSED_GF2MAT: Need compressed matrix over GF(2)\n",
            0, 0, 
            "You can return such matrix with 'return mat;'\n");
  }
  /* type for mat */
  typ = libGAP_TYPE_LIST_GF2MAT;


  /* we assume here that there is a first row */
  r1 = libGAP_ELM_GF2MAT(mat,1);
  
  l = libGAP_LEN_GF2MAT(mat);
  w = libGAP_LEN_GF2VEC(r1);
  nrb=libGAP_NUMBER_BLOCKS_GF2VEC(r1);

  tra = libGAP_NewBag(libGAP_T_POSOBJ, libGAP_SIZE_PLEN_GF2MAT( w ));
  libGAP_TYPE_POSOBJ(tra) = typ;
  
  /* type for rows */
  typ = libGAP_TYPE_LIST_GF2VEC_LOCKED;
  
  libGAP_SET_LEN_GF2MAT(tra, w);
  /* create new matrix */
  for (i = 1; i <= w; i++) {
    libGAP_NEW_GF2VEC( row,  typ, l );
    libGAP_SET_LEN_GF2VEC(row,l);

    ptr=libGAP_BLOCKS_GF2VEC(row);
    for (n=1;n<=nrb;n++){*ptr=0;}

    libGAP_SET_ELM_GF2MAT( tra, i, row);
    libGAP_CHANGED_BAG(tra);
  }
  /* set entries */
  /* run over BIPEB row chunks of the original matrix */
  for (i = 1; i <= l; i=i+libGAP_BIPEB) {
    imod=(i-1)/libGAP_BIPEB;
    /* run through these rows in block chunks */
    for (n=0;n<nrb;n++) {
      for (j=0;j<libGAP_BIPEB;j++) {
	if ((i+j)>l) {
	  vals[j]=0; /* outside matrix */ 
	}
	else {
	  ptr=libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(mat,i+j))+n;
	  vals[j]=*ptr;
	}
      }
      /* write transposed values in new matrix */
      mask=1;
      nstart=n*libGAP_BIPEB+1;
      for (j=0;j<libGAP_BIPEB;j++) { /* bit number = Row in transpose */
	if ((nstart+j)<=w) {
	  /* still within matrix */
	  val=0;
	  bit=1;
	  for (k=0;k<libGAP_BIPEB;k++) {
	    if (mask==(vals[k]&mask)) {
	      val|=bit; /* set bit */
	    }
	    bit=bit<<1;
	  }
	  /* set entry */
	  ptr=libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(tra,nstart+j))+imod;
	  *ptr=val;
	  /* next bit */
	  mask=mask<<1;
	}
      }
    }
  }
  return tra;
}


/****************************************************************************
**
*F  FuncNUMBER_VECGF2( <self>, <vect> )
**
*/

#ifdef USE_GMP

libGAP_Obj libGAP_FuncNUMBER_VECGF2( libGAP_Obj self, libGAP_Obj vec )
{
  libGAP_UInt len,nd,i;
  libGAP_UInt head,a;
  libGAP_UInt off,off2;		/* 0 based */
  libGAP_Obj zahl;  /* the long number */
  libGAP_UInt *num;
  libGAP_UInt *vp;
  len = libGAP_LEN_GF2VEC(vec);
  num = libGAP_BLOCKS_GF2VEC(vec) + (len-1)/libGAP_BIPEB;
  off = (len -1) % libGAP_BIPEB + 1; /* number of significant bits in last word */
  off2 = libGAP_BIPEB - off;         /* number of insignificant bits in last word */

  /* mask out the last bits */
#ifdef libGAP_SYS_IS_64_BIT
  *num &= 0xffffffffffffffff >> off2;
#else
  *num &= 0xffffffff >> off2;
#endif

  if (len <=libGAP_NR_SMALL_INT_BITS) 
    /* it still fits into a small integer */
    return libGAP_INTOBJ_INT(libGAP_revertbits(*num,len));
  else {
    /* we might have to build a long integer */

    /* the number of words (limbs) we need. */
    nd = ((len-1)/GMP_LIMB_BITS)+1;

    zahl = libGAP_NewBag( libGAP_T_INTPOS, nd*sizeof(libGAP_UInt) );
    /*    zahl = NewBag( T_INTPOS, (((nd+1)>>1)<<1)*sizeof(UInt) );*/
    /* +1)>>1)<<1: round up to next even number*/

    /* garbage collection might lose pointer */
    num = libGAP_BLOCKS_GF2VEC(vec) + (len-1)/libGAP_BIPEB;

    vp = (libGAP_TypLimb *)libGAP_ADDR_OBJ(zahl); /* the place we write to */
    i=1;

    if (off!=libGAP_BIPEB) {
      head = libGAP_revertbits(*num,off); /* the last 'off' bits, reverted */
      while (i<nd) {
	/* next word */
	num--;
	*vp = head; /* the bits left from last word */
	a = libGAP_revertbits(*num,libGAP_BIPEB); /* the full word reverted */
	head = a>>off2; /* next head: trailing `off' bits */
	a =a << off; /* the rest of the word */
	*vp |=a;
	vp++;
	i++;
      }
      *vp = head; /* last head bits */
      vp++;
    }
    else {
      while (i<=nd) {
        *vp=libGAP_revertbits(*num--,libGAP_BIPEB);
	vp++;
	i++;
      }
    }


    zahl = libGAP_GMP_NORMALIZE(zahl);
    zahl = libGAP_GMP_REDUCE(zahl);
    
    return zahl;
  }

}

#else

libGAP_Obj libGAP_FuncNUMBER_VECGF2( libGAP_Obj self, libGAP_Obj vec )
{
  libGAP_UInt len,nd,i,nonz;
  libGAP_UInt head,a;
  libGAP_UInt off,off2;		/* 0 based */
  libGAP_Obj zahl;  /* the long number */
  libGAP_UInt *num;
  libGAP_UInt *vp;
  len = libGAP_LEN_GF2VEC(vec);
  num = libGAP_BLOCKS_GF2VEC(vec) + (len-1)/libGAP_BIPEB;
  off = (len -1) % libGAP_BIPEB + 1; /* number of significant bits in last word */
  off2 = libGAP_BIPEB - off;         /* number of insignificant bits in last word */

  /* mask out the last bits */
#ifdef libGAP_SYS_IS_64_BIT
  *num &= 0xffffffffffffffff >> off2;
#else
  *num &= 0xffffffff >> off2;
#endif

  if (len <=libGAP_NR_SMALL_INT_BITS) 
    /* it still fits into a small integer */
    return libGAP_INTOBJ_INT(libGAP_revertbits(*num,len));
  else {
    /* we might have to build a long integer */

    /* the number of words we need. A word is two digits */
    nd = ((len-1)/libGAP_NR_DIGIT_BITS/2)+1;

    /* zahl = NewBag( T_INTPOS, nd*sizeof(UInt) ); */
       zahl = libGAP_NewBag( libGAP_T_INTPOS, (((nd+1)>>1)<<1)*sizeof(libGAP_UInt) );
    /* +1)>>1)<<1: round up to next even number 
     legacy long ints always have a multiple of 4 digits. */

    /* garbage collection might lose pointer */
    num = libGAP_BLOCKS_GF2VEC(vec) + (len-1)/libGAP_BIPEB;

    vp = (libGAP_UInt *)libGAP_ADDR_OBJ(zahl); /* the place we write to */
    nonz=0; /* last non-zero position */ 
    i=1;

    if (off!=libGAP_BIPEB) {
      head = libGAP_revertbits(*num,off); /* the last 'off' bits, reverted */
      while (i<nd) {
	/* next word */
	num--;
	*vp = head; /* the bits left from last word */
	a = libGAP_revertbits(*num,libGAP_BIPEB); /* the full word reverted */
	head = a>>off2; /* next head: trailing `off' bits */
	a =a << off; /* the rest of the word */
	*vp |=a;
#ifdef WORDS_BIGENDIAN
	*vp = (*vp >> libGAP_NR_DIGIT_BITS) | (*vp << libGAP_NR_DIGIT_BITS);
#endif
	if (*vp != 0) {nonz=i;} 
	vp++;
	i++;
      }
      *vp = head; /* last head bits */
#ifdef WORDS_BIGENDIAN
      *vp = (*vp >> libGAP_NR_DIGIT_BITS) | (*vp << libGAP_NR_DIGIT_BITS);
#endif
      vp++;
      if (head !=0) {
	nonz=i; 
      }
    }
    else {
      while (i<=nd) {
        *vp=libGAP_revertbits(*num--,libGAP_BIPEB);
	if (*vp != 0) {nonz=i;} 
#ifdef WORDS_BIGENDIAN
      *vp = (*vp >> libGAP_NR_DIGIT_BITS) | (*vp << libGAP_NR_DIGIT_BITS);
#endif
	vp++;
	i++;
      }
    }


    i=nd%2; 
    if (i==1) { 
      *vp=0; /* erase the trailing two digits if existing */
      nd++; /* point to the last position */
    }

    if ((nonz<=1)) {
      /* there were only entries in the first word. Can we make it a small
       * int? */
      num = (libGAP_UInt*)libGAP_ADDR_OBJ(zahl);
      head = *num;
#ifdef WORDS_BIGENDIAN
      head = (head >> libGAP_NR_DIGIT_BITS) | (head << libGAP_NR_DIGIT_BITS);
#endif
      if (0==(head & (((libGAP_UInt)15L)<<libGAP_NR_SMALL_INT_BITS))) {
	return libGAP_INTOBJ_INT(head);
      }
    }

    /* if there are mult. of 2 words of zero entries we have to remove them */
    if ((nd>2) && ((nd-1)>nonz)) {
      /* get the length */
      while ((nd>2) && ((nd-1)>nonz)) {
	nd -= 2;
      } 
      libGAP_ResizeBag(zahl,nd*sizeof(libGAP_UInt)); 

    }
    return zahl;
  }

}

#endif


/****************************************************************************
**
*F  FuncLT_GF2VEC_GF2VEC( <self>, <vl>, <vr> )   compare GF2 vectors
*/
libGAP_Obj libGAP_FuncLT_GF2VEC_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
  return (libGAP_Cmp_GF2VEC_GF2VEC(vl,vr) < 0) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  Cmp_GF2MAT_GF2MAT( <ml>, <mr> )   compare GF2 matrices
*/

libGAP_Int libGAP_Cmp_GF2MAT_GF2MAT( libGAP_Obj ml, libGAP_Obj mr)
{
  libGAP_UInt l1, l2,l, i;
  libGAP_Int c;
  l1 = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ml,1));
  l2 = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(mr,1));
  l = (l1 < l2) ? l1 : l2;
  for (i = 2; i <= l+1; i++)
    {
      c = libGAP_Cmp_GF2VEC_GF2VEC(libGAP_ELM_PLIST(ml,i), libGAP_ELM_PLIST(mr,i));
      if (c != 0)
	return c;
    }
  if (l1 < l2)
    return -1;
  if (l1 > l2)
    return 1;
  return 0;
  
}

/****************************************************************************
**
*F  FuncEQ_GF2MAT_GF2MAT( <ml>, <mr> )   compare GF2 matrices
*/

libGAP_Obj libGAP_FuncEQ_GF2MAT_GF2MAT( libGAP_Obj self, libGAP_Obj ml, libGAP_Obj mr)
{
  if (libGAP_ELM_PLIST(ml,1) != libGAP_ELM_PLIST(mr,1))
    return libGAP_False;
  return (0 == libGAP_Cmp_GF2MAT_GF2MAT(ml,mr)) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  FuncLT_GF2MAT_GF2MAT( <ml>, <mr> )   compare GF2 matrices
*/

libGAP_Obj libGAP_FuncLT_GF2MAT_GF2MAT( libGAP_Obj self, libGAP_Obj ml, libGAP_Obj mr)
{
  return (libGAP_Cmp_GF2MAT_GF2MAT(ml,mr) < 0) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  DistGF2Vecs( <ptL>, <ptR>, <len> )
**
**  computes the GF2-vector distance of two blocks in memory, pointed to by
**  ptL and ptR for a GF(2) vector of <len> entries.
**  
*/
libGAP_UInt libGAP_DistGF2Vecs(libGAP_UInt* ptL,libGAP_UInt* ptR,libGAP_UInt len)
{
  libGAP_UInt 			sum,m;
  libGAP_UInt *                end;            /* end marker                      */

  /*T this  function will not work if the vectors have more than 2^28
   * entries */

  end = ptL + ((len+libGAP_BIPEB-1)/libGAP_BIPEB);
  sum=0;
  /* loop over the entries */
  while ( ptL < end ) {
    m = *ptL++ ^ *ptR++; /* xor of bits, nr bits therein is difference */
    libGAP_COUNT_TRUES_BLOCK(m);
    sum += m;
  }
  return sum;
}

/****************************************************************************
**
*F  FuncDIST_GF2VEC_GF2VEC( <self>, <vl>, <vr> )
**
**  'FuncDIST_GF2VEC_GF2VEC' returns the number of position in which two
**  gf2-vectors <vl>  and  <vr> differ.                            
*/
libGAP_Obj libGAP_FuncDIST_GF2VEC_GF2VEC (
    libGAP_Obj                 self,
    libGAP_Obj                 vl,
    libGAP_Obj                 vr )
{
  libGAP_UInt                  len;            /* length of the list              */
  libGAP_UInt                  off;            /* bit offset at the end to clean out */
  libGAP_UInt *                ptL;            /* bit field of <vl>               */
  libGAP_UInt *                ptR;            /* bit field of <vr>               */
  libGAP_UInt *                end;            /* pointer used to zero out end bit */
  /* get and check the length                                            */
  len = libGAP_LEN_GF2VEC(vl);

  if ( len != libGAP_LEN_GF2VEC(vr) ) {
    libGAP_ErrorMayQuit(
      "DIST_GF2VEC_GF2VEC: vectors must have the same length",0L,0L);
    return 0;
  }

  /* calculate the offsets */
  ptL = libGAP_BLOCKS_GF2VEC(vl);
  ptR = libGAP_BLOCKS_GF2VEC(vr);

/* mask out the last bits */
  off = (len -1) % libGAP_BIPEB + 1; /* number of significant bits in last word */
  off = libGAP_BIPEB - off;          /* number of insignificant bits in last word */
  end = ptL + ((len-1)/libGAP_BIPEB);
#ifdef libGAP_SYS_IS_64_BIT
  *end &= 0xffffffffffffffff >> off;
#else
  *end &= 0xffffffff >> off;
#endif
  end = ptR + ((len-1)/libGAP_BIPEB);
#ifdef libGAP_SYS_IS_64_BIT
  *end &= 0xffffffffffffffff >> off;
#else
  *end &= 0xffffffff >> off;
#endif

  return libGAP_INTOBJ_INT(libGAP_DistGF2Vecs(ptL,ptR,len));
}


libGAP_Int libGAP_DistVecClosVec(
  libGAP_Obj		veclis, /* pointers to matrix vectors and their multiples */
  libGAP_Obj 	        ovec,    /* vector we compute distance to */
  libGAP_Obj		d,	/* distances list */
  libGAP_Obj  	        osum,	/* position of the sum vector */
  libGAP_UInt		pos,	/* recursion depth */
  libGAP_UInt		l,	/* length of basis */
  libGAP_UInt		len )	/* length of the involved vectors */
{
  libGAP_UInt 		i;
  libGAP_UInt		di;
  libGAP_Obj		cnt;
  libGAP_Obj		vp;
  libGAP_UInt *        vec;
  libGAP_UInt *        sum;
  libGAP_Obj           one;
  libGAP_Obj           tmp;
  libGAP_Int		chg;

  vec = libGAP_BLOCKS_GF2VEC(ovec);
  sum = libGAP_BLOCKS_GF2VEC(osum);
  vp = libGAP_ELM_PLIST(veclis,pos);
  one = libGAP_INTOBJ_INT(1);
  chg=0;
  
  for (i=0;i<=1;i++) {
    if (pos < l)
      {
	chg |= libGAP_DistVecClosVec(veclis,ovec,d,osum,pos+1,l,len);
	if (chg!=0) {
	  sum = libGAP_BLOCKS_GF2VEC(osum);
	  }
      }
    else
      {
	di=libGAP_DistGF2Vecs(sum,vec,len);

	cnt=libGAP_ELM_PLIST(d,di+1);
	if (libGAP_IS_INTOBJ(cnt) && libGAP_SUM_INTOBJS(tmp, cnt, one))
	  {
	    cnt = tmp;
	    libGAP_SET_ELM_PLIST(d,di+1,cnt);
	  }
	else
	  {
	    cnt=libGAP_SumInt(cnt,one);
	    vec = libGAP_BLOCKS_GF2VEC(ovec);
	    sum = libGAP_BLOCKS_GF2VEC(osum);
	    libGAP_SET_ELM_PLIST(d,di+1,cnt);
	    libGAP_CHANGED_BAG(d);
	    chg=1;
	  }
      }
    libGAP_AddGF2VecToGF2Vec(sum,libGAP_BLOCKS_GF2VEC(libGAP_ELM_PLIST(vp,i+1)),len);
  }
  return chg;
}

libGAP_Obj libGAP_FuncDistVecClosVec(
  libGAP_Obj		self,
  libGAP_Obj		veclis, /* pointers to matrix vectors and their multiples */
  libGAP_Obj		vec,    /* vector we compute distance to */
  libGAP_Obj		d )	/* distances list */

{
  libGAP_Obj		sum; /* sum vector */
  libGAP_UInt 		len;

  len = libGAP_LEN_GF2VEC(vec);

  /* get space for sum vector */
  libGAP_NEW_GF2VEC( sum, libGAP_TYPE_LIST_GF2VEC, len );
  libGAP_SET_LEN_GF2VEC( sum, len );

  /* do the recursive work */
  libGAP_DistVecClosVec(veclis,vec,d,sum,1,libGAP_LEN_PLIST(veclis),len);

  return (libGAP_Obj) 0;
}

libGAP_UInt libGAP_AClosVec(
  libGAP_Obj		veclis, /* pointers to matrix vectors and their multiples */
  libGAP_Obj 	        ovec,    /* vector we compute distance to */
  libGAP_Obj  	        osum,	/* position of the sum vector */
  libGAP_UInt		pos,	/* recursion depth */
  libGAP_UInt		l,	/* length of basis */
  libGAP_UInt		len,	/* length of the involved vectors */
  libGAP_UInt		cnt,	/* numbr of vectors used */
  libGAP_UInt		stop,	/* stop value */
  libGAP_UInt		bd,	/* best distance so far */
  libGAP_Obj		obv,    /* best vector so far */
  libGAP_Obj           coords,  /* coefficients to get current vector */
  libGAP_Obj           bcoords  /* coefficients to get best vector */
  )	
{
  libGAP_UInt		di;
  libGAP_Obj		vp;
  libGAP_UInt *        vec;
  libGAP_UInt *        sum;
  libGAP_UInt *        bv;
  libGAP_UInt *        end;
  libGAP_UInt *w;



  /* maybe we don't add this basis vector -- if this leaves us enough possibilitiies */
  if ( pos+cnt<l ) {
    bd = libGAP_AClosVec(veclis,ovec,osum,pos+1,l,len,cnt,stop,bd,obv,coords,bcoords);
      if (bd<=stop) {
        return bd;
      }
  }


  /* Otherwise we do */
  
  vec = libGAP_BLOCKS_GF2VEC(ovec);
  sum = libGAP_BLOCKS_GF2VEC(osum);
  vp = libGAP_ELM_PLIST(veclis,pos);
  w = libGAP_BLOCKS_GF2VEC(libGAP_ELM_PLIST(vp,1));
  libGAP_AddGF2VecToGF2Vec(sum,w,len);

  if (coords != (libGAP_Obj) 0)
    {
      libGAP_SET_ELM_PLIST(coords,pos,libGAP_INTOBJ_INT(1));
    }

  
  if (cnt == 0) /* this is a candidate */
    {
      di=libGAP_DistGF2Vecs(sum,vec,len);
      if (di<bd) {
	
	/* store new result */
	bd=di;
	bv = libGAP_BLOCKS_GF2VEC(obv);
	end = bv+((len+libGAP_BIPEB-1)/libGAP_BIPEB);
	while (bv<end) 
	  *bv++=*sum++;
	sum = libGAP_BLOCKS_GF2VEC(osum);
	if (coords != (libGAP_Obj) 0)
	  {
	    libGAP_UInt i;
	    for (i=1; i <= l; i++)
	      {
		libGAP_Obj x;
		x = libGAP_ELM_PLIST(coords,i);
		libGAP_SET_ELM_PLIST(bcoords,i,x);
	      }
	  }
      }
    }
  else /* need to add in some more */
    {
      bd=libGAP_AClosVec(veclis,ovec,osum,pos+1,l,len,cnt-1,stop,bd,obv,coords,bcoords);
      if (bd<=stop) {
	return bd;
      }
    }
    
  /* reset component  */
  libGAP_AddGF2VecToGF2Vec(sum,w,len);
  if (coords != (libGAP_Obj) 0)
    {
      libGAP_SET_ELM_PLIST(coords,pos,libGAP_INTOBJ_INT(0));
    }
  
  libGAP_TakeInterrupt();
  return bd;
}





libGAP_Obj libGAP_FuncAClosVec(
  libGAP_Obj		self,
  libGAP_Obj		veclis, /* pointers to matrix vectors and their multiples */
  libGAP_Obj		vec,    /* vector we compute distance to */
  libGAP_Obj		cnt,	/* distances list */
  libGAP_Obj		stop)	/* distances list */

{
  libGAP_Obj		sum; /* sum vector */
  libGAP_Obj		best; /* best vector */
  libGAP_UInt 		len;

  len = libGAP_LEN_GF2VEC(vec);

  if (!libGAP_ARE_INTOBJS(cnt,stop))
    libGAP_ErrorMayQuit("AClosVec: cnt and stop must be small integers, not a %s and a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(cnt), (libGAP_Int)libGAP_TNAM_OBJ(stop));
  

  /* get space for sum vector and zero out */
  libGAP_NEW_GF2VEC( sum, libGAP_TYPE_LIST_GF2VEC, len );
  libGAP_SET_LEN_GF2VEC( sum, len );

  libGAP_NEW_GF2VEC( best, libGAP_TYPE_LIST_GF2VEC, len );
  libGAP_SET_LEN_GF2VEC( best, len );


  /* do the recursive work */
  libGAP_AClosVec(veclis,vec,sum,1, libGAP_LEN_PLIST(veclis),len,
    libGAP_INT_INTOBJ(cnt),libGAP_INT_INTOBJ(stop),len+1, /* maximal value +1 */
	   best, (libGAP_Obj) 0, (libGAP_Obj) 0);

  return best;
}

libGAP_Obj libGAP_FuncAClosVecCoords(
  libGAP_Obj		self,
  libGAP_Obj		veclis, /* pointers to matrix vectors and their multiples */
  libGAP_Obj		vec,    /* vector we compute distance to */
  libGAP_Obj		cnt,	/* distances list */
  libGAP_Obj		stop)	/* distances list */

{
  libGAP_Obj		sum; /* sum vector */
  libGAP_Obj		best; /* best vector */
  libGAP_Obj         coords; /* coefficients of mat to get current */
  libGAP_Obj         bcoords; /* coefficients of mat to get best */
  libGAP_Obj           res; /* length 2 plist for results */
  libGAP_UInt 		len, len2,i;

  len = libGAP_LEN_GF2VEC(vec);
  len2 = libGAP_LEN_PLIST(veclis);

  if (!libGAP_ARE_INTOBJS(cnt,stop))
    libGAP_ErrorMayQuit("AClosVec: cnt and stop must be small integers, not a %s and a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(cnt), (libGAP_Int)libGAP_TNAM_OBJ(stop));
  

  /* get space for sum vector and zero out */
  libGAP_NEW_GF2VEC( sum, libGAP_TYPE_LIST_GF2VEC, len );
  libGAP_SET_LEN_GF2VEC( sum, len );

  libGAP_NEW_GF2VEC( best, libGAP_TYPE_LIST_GF2VEC, len );
  libGAP_SET_LEN_GF2VEC( best, len );

  coords = libGAP_NEW_PLIST(  libGAP_T_PLIST_CYC, len2 );
  libGAP_SET_LEN_PLIST( coords, len2 );

  bcoords = libGAP_NEW_PLIST(  libGAP_T_PLIST_CYC, len2 );
  libGAP_SET_LEN_PLIST( bcoords, len2 );
  
  for (i=1; i <= len2; i++)
    {
      libGAP_SET_ELM_PLIST(coords,i,libGAP_INTOBJ_INT(0));
      libGAP_SET_ELM_PLIST(bcoords,i,libGAP_INTOBJ_INT(0));
    }

  /* do the recursive work */
  libGAP_AClosVec(veclis,vec,sum,1, len2 ,len,
    libGAP_INT_INTOBJ(cnt),libGAP_INT_INTOBJ(stop),len+1, /* maximal value +1 */
	   best,coords,bcoords);

  res = libGAP_NEW_PLIST(libGAP_T_PLIST_DENSE_NHOM,2);
  libGAP_SET_LEN_PLIST(res,2);
  libGAP_SET_ELM_PLIST(res,1,best);
  libGAP_SET_ELM_PLIST(res,2,bcoords);
  libGAP_CHANGED_BAG(res);
  return res;
}

/****************************************************************************
**
*F FuncCOSET_LEADERS_INNER_GF2( <self>, <veclis>, <weight>, <tofind>, <leaders> )
**
** Search for new coset leaders of weight <weight>
*/

libGAP_UInt libGAP_CosetLeadersInnerGF2( libGAP_Obj veclis,
			   libGAP_Obj v,
			   libGAP_Obj w,
			   libGAP_UInt weight,
			   libGAP_UInt pos,
			   libGAP_Obj leaders,
			   libGAP_UInt tofind )
{
  libGAP_UInt found = 0;
  libGAP_UInt len = libGAP_LEN_GF2VEC(v);
  libGAP_UInt lenw = libGAP_LEN_GF2VEC(w);
  libGAP_UInt sy;
  libGAP_UInt u0;
  libGAP_Obj vc;
  libGAP_UInt i,j;
    
  /* we know that the length of w does not exceed BIPEB -4 here
     (or there would not be room in a PLIST for all the coset leaders)
    we use this to do a lot of GF2 vector operations for w "in-place"

    Even more in this direction could be done, but this no longer
  the rate-determining step for any feasible application*/
  
  if (weight == 1)
    {
      for (i = pos; i <= len; i++)
	{
	  u0 = libGAP_BLOCKS_GF2VEC(libGAP_ELM_PLIST(libGAP_ELM_PLIST(veclis, i),1))[0];
	  libGAP_BLOCKS_GF2VEC(w)[0] ^= u0;
	  libGAP_BLOCK_ELM_GF2VEC(v, i) |= libGAP_MASK_POS_GF2VEC(i);

	  sy = libGAP_revertbits(libGAP_BLOCKS_GF2VEC(w)[0], lenw);
	  if ((libGAP_Obj) 0 == libGAP_ELM_PLIST(leaders,sy+1))
	    {
	      libGAP_NEW_GF2VEC(vc, libGAP_TYPE_LIST_GF2VEC_IMM, len);
	      libGAP_SET_LEN_GF2VEC(vc, len);
	      for (j = 0; j < libGAP_NUMBER_BLOCKS_GF2VEC(v); j++)
		libGAP_BLOCKS_GF2VEC(vc)[j] = libGAP_BLOCKS_GF2VEC(v)[j];
	      libGAP_SET_ELM_PLIST(leaders,sy+1,vc);
	      libGAP_CHANGED_BAG(leaders);
	      if (++found == tofind)
		return found;
	    }
	  libGAP_BLOCKS_GF2VEC(w)[0] ^= u0;
	  libGAP_BLOCK_ELM_GF2VEC(v, i) &= ~libGAP_MASK_POS_GF2VEC(i);
	}
    }
  else
    {
      if (pos + weight <= len)
	{
	  found += libGAP_CosetLeadersInnerGF2(veclis, v, w, weight, pos+1, leaders, tofind);
	  if (found == tofind)
	    return found;
	}
      u0 = libGAP_BLOCKS_GF2VEC(libGAP_ELM_PLIST(libGAP_ELM_PLIST(veclis, pos),1))[0];
      libGAP_BLOCKS_GF2VEC(w)[0] ^= u0;
      libGAP_BLOCK_ELM_GF2VEC(v, pos) |= libGAP_MASK_POS_GF2VEC(pos);
      found += libGAP_CosetLeadersInnerGF2(veclis, v, w, weight -1, pos + 1, leaders, tofind - found);
      if (found == tofind)
	return found;
      libGAP_BLOCKS_GF2VEC(w)[0] ^= u0;
      libGAP_BLOCK_ELM_GF2VEC(v, pos) &= ~libGAP_MASK_POS_GF2VEC(pos);
    }
  libGAP_TakeInterrupt();
  return found;
}




libGAP_Obj libGAP_FuncCOSET_LEADERS_INNER_GF2( libGAP_Obj self, libGAP_Obj veclis, libGAP_Obj weight, libGAP_Obj tofind, libGAP_Obj leaders)
{
  libGAP_Obj v,w;
  libGAP_UInt lenv, lenw;

  if (!libGAP_ARE_INTOBJS(weight,tofind))
    libGAP_ErrorMayQuit("COSET_LEADERS_INNER_GF2: weight and tofind must be smal integers, not a %s and a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(weight), (libGAP_Int)libGAP_TNAM_OBJ(tofind));
  
  lenv = libGAP_LEN_PLIST(veclis);
  libGAP_NEW_GF2VEC(v, libGAP_TYPE_LIST_GF2VEC, lenv);
  libGAP_SET_LEN_GF2VEC(v, lenv);
  lenw = libGAP_LEN_GF2VEC(libGAP_ELM_PLIST(libGAP_ELM_PLIST(veclis,1),1));
  libGAP_NEW_GF2VEC(w, libGAP_TYPE_LIST_GF2VEC,lenw );
  libGAP_SET_LEN_GF2VEC(w,lenw);
  if (lenw > libGAP_BIPEB-4)
    libGAP_ErrorMayQuit("COSET_LEADERS_INNER_GF2: too many cosets to return the leaders in a plain list",0,0);
  return libGAP_INTOBJ_INT(libGAP_CosetLeadersInnerGF2( veclis, v, w, libGAP_INT_INTOBJ(weight), 1, leaders, libGAP_INT_INTOBJ(tofind)));
}

/****************************************************************************
**
*F  Polynomial Arithmetic Support
*/


/****************************************************************************
**
*F  FuncRIGHTMOST_NONZERO_GF2VEC (<self>, <vec> )
**
*/

libGAP_Obj libGAP_FuncRIGHTMOST_NONZERO_GF2VEC( libGAP_Obj self, libGAP_Obj vec )
{
  return libGAP_INTOBJ_INT(libGAP_RightMostOneGF2Vec(vec));
}

/****************************************************************************
**
*F  ResizeGF2Vec( <vec>, <newlen>, <clean> )
**
*/

void libGAP_ResizeGF2Vec( libGAP_Obj vec, libGAP_UInt newlen )
{
  libGAP_UInt len;
  libGAP_UInt *ptr;
  libGAP_UInt *nptr;
  libGAP_UInt off;
  len = libGAP_LEN_GF2VEC(vec);
  if (len == newlen)
    return;
  if (libGAP_True == libGAP_DoFilter(libGAP_IsLockedRepresentationVector, vec))
    {
      libGAP_ErrorReturnVoid("Resize of locked compressed vector is forbidden", 0, 0,
		      "You can `return;' to ignore the operation");
      return;
    }

  
  if (newlen > len)
    {
      libGAP_ResizeBag(vec,libGAP_SIZE_PLEN_GF2VEC(newlen));

      /* now clean remainder of last block */
      if (len == 0)
	ptr = libGAP_BLOCKS_GF2VEC(vec);
      else
	{
	  ptr = libGAP_BLOCKS_GF2VEC(vec) + (len -1)/libGAP_BIPEB;
	  off = libGAP_BIPEB - ((len -1)% libGAP_BIPEB + 1); /* number of insignificant bits in last word */
#ifdef libGAP_SYS_IS_64_BIT
	  *ptr &= 0xffffffffffffffff >> off;
#else
	  *ptr &= 0xffffffff >> off;
#endif
	  ptr++;
	}
      
      /* and clean new blocks -- shouldn't need to do this, but
	 it's very cheap */
      /* newlen can't be zero here, since it is bigger than len */
      nptr = libGAP_BLOCKS_GF2VEC(vec) + (newlen -1)/libGAP_BIPEB; 
      while (ptr <= nptr)
	*ptr++ = 0;

      libGAP_SET_LEN_GF2VEC(vec, newlen);
      return;
    }
  else
    {
      /* clean remainder of new last block, if any */
      if (newlen % libGAP_BIPEB)
	{
	  ptr = libGAP_BLOCKS_GF2VEC(vec) + (newlen -1)/libGAP_BIPEB;
	  off = libGAP_BIPEB - ((newlen-1) % libGAP_BIPEB + 1);
#ifdef libGAP_SYS_IS_64_BIT
	  *ptr &= 0xffffffffffffffff >> off;
#else
	  *ptr &= 0xffffffff >> off;
#endif
	}
      libGAP_SET_LEN_GF2VEC(vec, newlen);
      libGAP_ResizeBag(vec, libGAP_SIZE_PLEN_GF2VEC(newlen));
      return;
    }
}

/****************************************************************************
**
*F  FuncRESIZE_GF2VEC( <self>, <vec>, <newlen> )
**
*/

libGAP_Obj libGAP_FuncRESIZE_GF2VEC( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj newlen)
{
  libGAP_Int newlen1;
  if (!libGAP_IS_MUTABLE_OBJ(vec))
    {
      libGAP_ErrorReturnVoid("RESIZE_GF2VEC: the vector must be mutable", 0, 0,
		      "you may 'return;' to skip the operation");
      return (libGAP_Obj)0;
    }
  if (!libGAP_IS_INTOBJ(newlen))
    libGAP_ErrorMayQuit("RESIZE_GF2VEC: newlen must be a small integer, not a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(newlen), 0L);
  newlen1 = libGAP_INT_INTOBJ(newlen);
  if (newlen1 < 0) 
    libGAP_ErrorMayQuit("RESIZE_GF2VEC: the new size must be a non-negative integer, not %d", newlen1, 0);
  libGAP_ResizeGF2Vec(vec, newlen1);
  return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  ShiftLeftGF2Vec( <vec>, <amount> )
**
*/

void libGAP_ShiftLeftGF2Vec( libGAP_Obj vec, libGAP_UInt amount )
{
  libGAP_UInt len;
  libGAP_UInt *ptr1, *ptr2;
  libGAP_UInt i;
  libGAP_UInt block;
  libGAP_UInt off;
  if (amount == 0)
    return;
  len = libGAP_LEN_GF2VEC(vec);
  if (amount >= len)
    {
      libGAP_ResizeGF2Vec(vec, 0);
      return;
    }
  if (amount % libGAP_BIPEB == 0)
    {
      ptr1 = libGAP_BLOCKS_GF2VEC(vec);
      ptr2 = ptr1 + amount/libGAP_BIPEB;
      for (i = 0; i < (len - amount + libGAP_BIPEB - 1)/libGAP_BIPEB; i++)
	*ptr1++ = *ptr2++;
    }
  else
    {
      ptr1 = libGAP_BLOCKS_GF2VEC(vec);
      ptr2 = ptr1 + amount/libGAP_BIPEB;
      off = amount % libGAP_BIPEB;
      for (i = 0; i < (len - amount + libGAP_BIPEB - 1)/libGAP_BIPEB; i++)
	{
	  block = (*ptr2++) >> off;
	  block |= (*ptr2) << (libGAP_BIPEB - off);
	  *ptr1++ = block;
	}
    }
  libGAP_ResizeGF2Vec(vec, len-amount);
  return;
}

/****************************************************************************
**
*F  FuncSHIFT_LEFT_GF2VEC(<self>, <vec>, <amount> )
**
*/

libGAP_Obj libGAP_FuncSHIFT_LEFT_GF2VEC( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj amount)
{
  libGAP_Int amount1;
  if (!libGAP_IS_MUTABLE_OBJ(vec)) 
    {
      libGAP_ErrorReturnVoid("SHIFT_LEFT_GF2VEC: the vector must be mutable", 0, 0,
		      "you may 'return;' to skip the operation");
      return (libGAP_Obj)0;
    }
  if (!libGAP_IS_INTOBJ(amount))
    libGAP_ErrorMayQuit("SHIFT_LEFT_GF2VEC: the amnount to shift must be a small integer, not a %d",
	      (libGAP_Int)libGAP_TNAM_OBJ(amount), 0L);
  amount1 = libGAP_INT_INTOBJ(amount);
  if (amount1 < 0)
     libGAP_ErrorMayQuit("SHIFT_LEFT_GF2VEC: <amount> must be a non-negative integer, not %d", amount1, 0);
  libGAP_ShiftLeftGF2Vec(vec, amount1);
  return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  ShiftRightGF2Vec( <vec>, <amount> )
**
*/

void libGAP_ShiftRightGF2Vec( libGAP_Obj vec, libGAP_UInt amount )
{
  libGAP_UInt len;
  libGAP_UInt *ptr1, *ptr2, *ptr0;
  libGAP_UInt i;
  libGAP_UInt block;
  libGAP_UInt off;
  if (amount == 0)
    return;
  len = libGAP_LEN_GF2VEC(vec);
  libGAP_ResizeGF2Vec(vec, len+amount);
  if (amount % libGAP_BIPEB == 0)
    {
      /* move the blocks */
      ptr1 = libGAP_BLOCKS_GF2VEC(vec) + (len - 1 + amount)/libGAP_BIPEB;
      ptr2 = ptr1 - amount/libGAP_BIPEB;
      for (i = 0; i < (len + libGAP_BIPEB - 1)/libGAP_BIPEB; i++)
	*ptr1-- = *ptr2--;

      /* and fill with zeroes */
      ptr2 = libGAP_BLOCKS_GF2VEC(vec);
      while (ptr1 >= ptr2)
	*ptr1-- = 0;
    }
  else
    {
      ptr1 = libGAP_BLOCKS_GF2VEC(vec) + (len -1 + amount)/libGAP_BIPEB;
      ptr2 = ptr1 - amount/libGAP_BIPEB; /* this can sometimes be the block AFTER the old last block,
				     but this must be OK */
      off = amount % libGAP_BIPEB;
      ptr0 = libGAP_BLOCKS_GF2VEC(vec);
      while (1)
	{
	  block = (*ptr2--) << off;
	  if (ptr2 < ptr0)
	    break;
	  block |= (*ptr2) >> (libGAP_BIPEB - off);
	  *ptr1-- = block;
	}
      *ptr1-- = block;
      while (ptr1 >= ptr0)
	*ptr1-- = 0;
    }
  return;
}

/****************************************************************************
**
*F  FuncSHIFT_RIGHT_GF2VEC(<self>, <vec>, <amount> )
**
*/

libGAP_Obj libGAP_FuncSHIFT_RIGHT_GF2VEC( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj amount)
{
  libGAP_Int amount1;
  if (!libGAP_IS_MUTABLE_OBJ(vec))
    {
      libGAP_ErrorReturnVoid("SHIFT_RIGHT_GF2VEC: the vector must be mutable", 0, 0,
		      "you may 'return;' to skip the operation");
      return (libGAP_Obj)0;
    }
  if (!libGAP_IS_INTOBJ(amount))
    libGAP_ErrorMayQuit("SHIFT_RIGHT_GF2VEC: the amount to shift must be a small integer, not a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(amount), 0L);
  amount1 = libGAP_INT_INTOBJ(amount);
  if (amount1 < 0)
      libGAP_ErrorMayQuit("SHIFT_RIGHT_GF2VEC: <amount> must be a non-negative integer, not %d", amount1, 0);
  libGAP_ShiftRightGF2Vec(vec, amount1);
  return (libGAP_Obj)0;
}

/* ReduceCoeffs */

/****************************************************************************
**
*F  AddShiftedVecGF2VecGF2( <vec1>, <vec2>, <len2>, <off> )
**
*/

void libGAP_AddShiftedVecGF2VecGF2( libGAP_Obj vec1, libGAP_Obj vec2, libGAP_UInt len2, libGAP_UInt off )
{
  libGAP_UInt *ptr1, *ptr2;
  libGAP_UInt i;
  libGAP_UInt block;
  libGAP_UInt shift1, shift2;
  if (off % libGAP_BIPEB == 0)
    {
      ptr1 = libGAP_BLOCKS_GF2VEC(vec1) + off/libGAP_BIPEB;
      ptr2 = libGAP_BLOCKS_GF2VEC(vec2);
      for (i = 0; i < (len2 - 1)/libGAP_BIPEB; i++)
	*ptr1++ ^= *ptr2++;
      block = *ptr2;
#ifdef libGAP_SYS_IS_64_BIT
      block &= (0xFFFFFFFFFFFFFFFF >> (libGAP_BIPEB - (len2-1) % libGAP_BIPEB -1));
#else
      block &= (0xFFFFFFFF >> (libGAP_BIPEB - (len2-1) % libGAP_BIPEB-1));
#endif
      *ptr1 ^= block;
    }
  else
    {
      ptr1 = libGAP_BLOCKS_GF2VEC(vec1) + off/libGAP_BIPEB;
      ptr2 = libGAP_BLOCKS_GF2VEC(vec2);
      shift1 = off %libGAP_BIPEB;
      shift2 = libGAP_BIPEB - off%libGAP_BIPEB;
      for (i = 0; i < len2/libGAP_BIPEB; i++)
	{
	  *ptr1++ ^= *ptr2 << shift1;
	  *ptr1   ^= *ptr2++ >> shift2;
	}

      if (len2 % libGAP_BIPEB)
	{
	  block = *ptr2;
#ifdef libGAP_SYS_IS_64_BIT
	  block &= 0xFFFFFFFFFFFFFFFF >> (libGAP_BIPEB - (len2-1) % libGAP_BIPEB-1);
#else
	  block &= 0xFFFFFFFF >> (libGAP_BIPEB - (len2-1) % libGAP_BIPEB-1);
#endif
	  *ptr1++ ^= block << shift1;
	  if (len2 % libGAP_BIPEB + off % libGAP_BIPEB > libGAP_BIPEB)
	    {
	      assert(ptr1 < libGAP_BLOCKS_GF2VEC(vec1) + (libGAP_LEN_GF2VEC(vec1) + libGAP_BIPEB -1)/libGAP_BIPEB);
	      *ptr1 ^= block >> shift2;
	    }
	}
    }
}

/****************************************************************************
**
*F FuncADD_GF2VEC_GF2VEC_SHIFTED( <self>, <vec1>, <vec2>, <len2>, <off> )
**
*/

libGAP_Obj libGAP_FuncADD_GF2VEC_GF2VEC_SHIFTED( libGAP_Obj self, libGAP_Obj vec1, libGAP_Obj vec2, libGAP_Obj len2, libGAP_Obj off)
{
  libGAP_Int off1, len2a;
  if (!libGAP_IS_INTOBJ(off))
    libGAP_ErrorMayQuit("ADD_GF2VEC_GF2VEC_SHIFTED: offset should be a small integer not a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(off), 0L);
  off1 = libGAP_INT_INTOBJ(off);
  if (off1 < 0)
    {
       libGAP_ErrorMayQuit("ADD_GF2VEC_GF2VEC_SHIFTED: <offset> must be a non-negative integer",
			   0,0);
    }
  len2a = libGAP_INT_INTOBJ(len2);
  while (len2a < 0 && len2a <= libGAP_LEN_GF2VEC(vec2)) 
    {
      len2 = libGAP_ErrorReturnObj("ADD_GF2VEC_GF2VEC_SHIFTED: <len2> must be a non-negative integer\nand less than the actual length of the vector",
			   0,0,"you can replace <len2> via 'return <len2>;'");
      len2a = libGAP_INT_INTOBJ(len2);
    }
  if (len2a + off1 > libGAP_LEN_GF2VEC(vec1))
    libGAP_ResizeGF2Vec(vec1, len2a+ off1);
  libGAP_AddShiftedVecGF2VecGF2( vec1, vec2, len2a, off1);
  return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  ProductCoeffsGF2Vec( <vec1>, <len1>, <vec2>, <len2> )
**
*/

libGAP_Obj libGAP_ProductCoeffsGF2Vec( libGAP_Obj vec1, libGAP_UInt len1, libGAP_Obj vec2, libGAP_UInt len2)
{
  libGAP_Obj prod;
  libGAP_UInt i,e;
  libGAP_UInt *ptr;
  libGAP_UInt block = 0;
  libGAP_UInt len;
  if (len1 == 0 && len2 == 0)
    len = 0;
  else
    len = len1 + len2 -1;
  libGAP_NEW_GF2VEC(prod, libGAP_TYPE_LIST_GF2VEC, len);
  libGAP_SET_LEN_GF2VEC(prod, len);

  /* better to do the longer loop on the inside */
  if (len2 < len1)
    {
      libGAP_UInt tmp;
      libGAP_Obj tmpv;
      tmp = len1;
      len1 = len2;
      len2 = tmp;
      tmpv = vec1;
      vec1 = vec2;
      vec2 = tmpv;
    }

  ptr = libGAP_BLOCKS_GF2VEC(vec1);
  e = libGAP_BIPEB;
  for (i = 0; i < len1; i++)
    {
      if (e == libGAP_BIPEB)
	{
	  block = *ptr++;
	  e = 0;
	}
      if (block & ((libGAP_UInt)1 << e++))
	libGAP_AddShiftedVecGF2VecGF2( prod, vec2, len2, i);
    }
  return prod;
}

/****************************************************************************
**
*F  FuncPROD_COEFFS_GF2VEC( <self>, <vec1>, <len1>, <vec2>, <len2> )
**
*/

libGAP_Obj libGAP_FuncPROD_COEFFS_GF2VEC( libGAP_Obj self, libGAP_Obj vec1, libGAP_Obj len1, libGAP_Obj vec2, libGAP_Obj len2 )
{
  libGAP_UInt len1a, len2a;
  libGAP_Obj prod;
  libGAP_UInt last;
  if (!libGAP_ARE_INTOBJS(len1,len2))
    libGAP_ErrorMayQuit("PROD_COEFFS_GF2VEC: vector lengths must be small integers, not a %s and a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(len1), (libGAP_Int)libGAP_TNAM_OBJ(len2));
  len2a = libGAP_INT_INTOBJ(len2);
   if (len2a > libGAP_LEN_GF2VEC(vec2)) 
       libGAP_ErrorMayQuit("PROD_COEFFS_GF2VEC: <len2> must not be more than the actual\nlength of the vector",
		 0,0);
  len1a = libGAP_INT_INTOBJ(len1);
  if (len1a > libGAP_LEN_GF2VEC(vec1)) 
       libGAP_ErrorMayQuit("PROD_COEFFS_GF2VEC: <len1> must be not more than the actual\nlength of the vector",
		 0,0);
  prod = libGAP_ProductCoeffsGF2Vec( vec1, len1a, vec2, len2a );
  last = libGAP_RightMostOneGF2Vec(prod);
  if (last < libGAP_LEN_GF2VEC(prod))
    libGAP_ResizeGF2Vec(prod, last);
  return prod;
}

/****************************************************************************
**
*F  ReduceCoeffsGF2Vec(<vec1>, <vec2>, <len2>, <quotient> )
**
*/

void libGAP_ReduceCoeffsGF2Vec( libGAP_Obj vec1, libGAP_Obj vec2, libGAP_UInt len2, libGAP_Obj quotient )
{
  libGAP_UInt len1 = libGAP_LEN_GF2VEC(vec1);
  libGAP_UInt i,j,e;
  libGAP_UInt *ptr, *qptr = (libGAP_UInt *)0;
  if (len2 > len1)
    return;
  i = len1 -1;
  e = (i % libGAP_BIPEB);
  ptr = libGAP_BLOCKS_GF2VEC(vec1) + (i/libGAP_BIPEB);
  if (quotient != (libGAP_Obj) 0)
    qptr = libGAP_BLOCKS_GF2VEC(quotient);
  j = len1-len2+1;
  while (i+ 1 >= len2)
    {
      if (*ptr & ((libGAP_UInt)1 << e))
	{
	  libGAP_AddShiftedVecGF2VecGF2(vec1, vec2, len2, i - len2 + 1);
	  if (qptr)
	    qptr[(j-1)/libGAP_BIPEB] |= libGAP_MASK_POS_GF2VEC(j);
	}
      assert(!(*ptr & ((libGAP_UInt)1<<e)));
      if (e == 0)
	{
	  e = libGAP_BIPEB -1;
	  ptr--;
	}
      else
	e--;
      i--;
      j--;
    }
 return;
}

/****************************************************************************
**
*F  FuncREDUCE_COEFFS_GF2VEC( <self>, <vec1>, <len1>, <vec2>, <len2> )
**
*/
libGAP_Obj libGAP_FuncREDUCE_COEFFS_GF2VEC( libGAP_Obj self, libGAP_Obj vec1, libGAP_Obj len1, libGAP_Obj vec2, libGAP_Obj len2)
{
  libGAP_UInt last;
  libGAP_Int len2a;
  if (!libGAP_IS_INTOBJ(len1))
    libGAP_ErrorMayQuit("REDUCE_COEFFS_GF2VEC: given length <len1> of left argt must be a small integer, not a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(len1),0L);
  if (libGAP_INT_INTOBJ(len1) < 0 || libGAP_INT_INTOBJ(len1) > libGAP_LEN_GF2VEC(vec1))
    libGAP_ErrorMayQuit("ReduceCoeffs: given length <len1> of left argt (%d)\nis longer than the argt (%d)",
	      libGAP_INT_INTOBJ(len1), libGAP_LEN_GF2VEC(vec1));
  if (!libGAP_IS_INTOBJ(len2))
    libGAP_ErrorMayQuit("REDUCE_COEFFS_GF2VEC: given length <len2> of right argt must be a small integer, not a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(len2),0L);
  len2a = libGAP_INT_INTOBJ(len2);
  if ( len2a < 0 ||  len2a > libGAP_LEN_GF2VEC(vec2))
    libGAP_ErrorMayQuit("ReduceCoeffs: given length <len2> of right argt (%d)\nis longer than the argt (%d)",
		     len2a, libGAP_LEN_GF2VEC(vec2));
  libGAP_ResizeGF2Vec(vec1, libGAP_INT_INTOBJ(len1));
  
  while ( 0 < len2a ) {
    if ( libGAP_BLOCK_ELM_GF2VEC(vec2,len2a) == 0 )
      len2a = libGAP_BIPEB*((len2a-1)/libGAP_BIPEB);
    else if ( libGAP_BLOCK_ELM_GF2VEC(vec2,len2a) & libGAP_MASK_POS_GF2VEC(len2a) )
      break;
    else
      len2a--;
  }

  if (len2a == 0)
    {
      libGAP_ErrorReturnVoid("ReduceCoeffs: second argument must not be zero", 0, 0,
		      "you may 'return;' to skip the reduction");
      return 0;
    }
  
  libGAP_ReduceCoeffsGF2Vec( vec1, vec2, len2a, (libGAP_Obj)0);
  last = libGAP_RightMostOneGF2Vec(vec1);
  libGAP_ResizeGF2Vec(vec1, last);
  return libGAP_INTOBJ_INT(last);
}

/****************************************************************************
**
*F  FuncQUOTREM_COEFFS_GF2VEC( <self>, <vec1>, <len1>, <vec2>, <len2> )
**
*/
libGAP_Obj libGAP_FuncQUOTREM_COEFFS_GF2VEC( libGAP_Obj self, libGAP_Obj vec1, libGAP_Obj len1, libGAP_Obj vec2, libGAP_Obj len2)
{
     libGAP_Int len2a;
     libGAP_Int len1a = libGAP_INT_INTOBJ(len1);
     libGAP_Obj quotv, remv, ret;
     if (!libGAP_IS_INTOBJ(len1))
     libGAP_ErrorMayQuit("QUOTREM_COEFFS_GF2VEC: given length <len1> of left argt must be a small integer, not a %s",
		  (libGAP_Int)libGAP_TNAM_OBJ(len1),0L);
     if (libGAP_INT_INTOBJ(len1) < 0 || libGAP_INT_INTOBJ(len1) > libGAP_LEN_GF2VEC(vec1))
     libGAP_ErrorMayQuit("QuotremCoeffs: given length <len1> of left argt (%d)\nis longer than the argt (%d)",
		  libGAP_INT_INTOBJ(len1), libGAP_LEN_GF2VEC(vec1));
     if (!libGAP_IS_INTOBJ(len2))
     libGAP_ErrorMayQuit("QUOTREM_COEFFS_GF2VEC: given length <len2> of right argt must be a small integer, not a %s",
		  (libGAP_Int)libGAP_TNAM_OBJ(len2),0L);
     len2a = libGAP_INT_INTOBJ(len2);
     if ( len2a < 0 ||  len2a > libGAP_LEN_GF2VEC(vec2))
     libGAP_ErrorMayQuit("QuotremCoeffs: given length <len2> of right argt (%d)\nis longer than the argt (%d)",
		  len2a, libGAP_LEN_GF2VEC(vec2));
     
     while ( 0 < len2a ) {
       if ( libGAP_BLOCK_ELM_GF2VEC(vec2,len2a) == 0 )
	 len2a = libGAP_BIPEB*((len2a-1)/libGAP_BIPEB);
       else if ( libGAP_BLOCK_ELM_GF2VEC(vec2,len2a) & libGAP_MASK_POS_GF2VEC(len2a) )
	 break;
       else
	 len2a--;
     } 
     if (len2a == 0) {
       libGAP_ErrorReturnVoid("QuotremCoeffs: second argument must not be zero", 0, 0,
		       "you may 'return;' to skip the reduction");
       return 0;
     }

     libGAP_NEW_GF2VEC(remv, libGAP_TYPE_LIST_GF2VEC, len1a);
     libGAP_SET_LEN_GF2VEC(remv, len1a);
     memcpy((void *)libGAP_BLOCKS_GF2VEC(remv), (void *)libGAP_BLOCKS_GF2VEC(vec1),
	    ((len1a + libGAP_BIPEB-1)/libGAP_BIPEB)*sizeof(libGAP_UInt));
     
     libGAP_NEW_GF2VEC(quotv, libGAP_TYPE_LIST_GF2VEC, len1a-len2a+1);
     libGAP_SET_LEN_GF2VEC(quotv, len1a-len2a+1);
     libGAP_ReduceCoeffsGF2Vec( remv, vec2, len2a, quotv);
     
     ret = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB, 2);
     libGAP_SET_LEN_PLIST(ret, 2);
     
     libGAP_SET_ELM_PLIST(ret, 1, quotv);
     libGAP_SET_ELM_PLIST(ret, 2, remv);

     libGAP_CHANGED_BAG(ret);

return ret;
}


/****************************************************************************
**
*F  FuncSEMIECHELON_LIST_GF2VECS( <self>, <mat> )
**
**  Method for SemiEchelonMat for plain lists of GF2 vectors
**
** Method selection can guarantee us a plain list of characteristic 2 vectors 
*/

libGAP_Obj libGAP_FuncSEMIECHELON_LIST_GF2VECS( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_UInt i,len;
  libGAP_UInt width;
  libGAP_Obj row;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  width = libGAP_LEN_GF2VEC(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row) ||
	  libGAP_LEN_GF2VEC(row)!= width)
	{
	  return libGAP_TRY_NEXT_METHOD;
	}
    }
  return libGAP_SemiEchelonListGF2Vecs( mat, 0);
}

/****************************************************************************
**
*F  FuncSEMIECHELON_LIST_GF2VECS_TRANSFORMATIONS( <self>, <mat> )
**
**  Method for SemiEchelonMatTransformations for plain lists of GF2 vectors
**
** Method selection can guarantee us a plain list of characteristic 2 vectors 
*/

libGAP_Obj libGAP_FuncSEMIECHELON_LIST_GF2VECS_TRANSFORMATIONS( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_UInt i,len;
  libGAP_UInt width;
  libGAP_Obj row;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  width = libGAP_LEN_GF2VEC(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row) ||
	  libGAP_LEN_GF2VEC(row)!= width)
	{
	  return libGAP_TRY_NEXT_METHOD;
	}
    }
  return libGAP_SemiEchelonListGF2Vecs( mat, 1);
}

/****************************************************************************
**
*F  FuncTRIANGULIZE_LIST_GF2VECS( <self>, <mat> )
**
*/

libGAP_Obj libGAP_FuncTRIANGULIZE_LIST_GF2VECS( libGAP_Obj self, libGAP_Obj mat)
{
  libGAP_UInt i,len;
  libGAP_UInt width;
  libGAP_Obj row;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  width = libGAP_LEN_GF2VEC(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row) ||
	  libGAP_LEN_GF2VEC(row)!= width)
	{
	  return libGAP_TRY_NEXT_METHOD;
	}
    }
  libGAP_TriangulizeListGF2Vecs( mat, 1 );
  return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  FuncRANK_LIST_GF2VECS( <self>, <mat> )
**
*/

libGAP_Obj libGAP_FuncRANK_LIST_GF2VECS( libGAP_Obj self, libGAP_Obj mat)
{
  libGAP_UInt i,len;
  libGAP_UInt width;
  libGAP_Obj row;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  width = libGAP_LEN_GF2VEC(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row) ||
	  libGAP_LEN_GF2VEC(row)!= width)
	{
	  return libGAP_TRY_NEXT_METHOD;
	}
    }
  return libGAP_INTOBJ_INT(libGAP_TriangulizeListGF2Vecs( mat , 0));
}

/****************************************************************************
**
*F  FuncDETERMINANT_LIST_GF2VECS( <self>, <mat> )
**
*/

libGAP_Obj libGAP_FuncDETERMINANT_LIST_GF2VECS( libGAP_Obj self, libGAP_Obj mat)
{
  libGAP_UInt i,len;
  libGAP_UInt width;
  libGAP_Obj row;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  width = libGAP_LEN_GF2VEC(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_GF2VEC_REP(row) ||
	  libGAP_LEN_GF2VEC(row)!= width)
	{
	  return libGAP_TRY_NEXT_METHOD;
	}
    }
  return (len == libGAP_TriangulizeListGF2Vecs( mat , 0)) ? libGAP_GF2One : libGAP_GF2Zero;
}

/****************************************************************************
**
*F  FuncKRONECKERPRODUCT_GF2MAT_GF2MAT( <self>, <matl>, <matr>)
**
*/

libGAP_Obj libGAP_FuncKRONECKERPRODUCT_GF2MAT_GF2MAT( libGAP_Obj self, libGAP_Obj matl, libGAP_Obj matr)
{
  libGAP_UInt nrowl, nrowr, nrowp, ncoll, ncolr, ncolp, ncol,
    i, j, k, l, mutable;
  libGAP_Obj mat, type, row, shift[libGAP_BIPEB];
  libGAP_UInt *datar, *data;

  nrowl = libGAP_LEN_GF2MAT(matl);
  nrowr = libGAP_LEN_GF2MAT(matr);
  nrowp = nrowl*nrowr;
  ncoll = libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(matl,1));
  ncolr = libGAP_LEN_GF2VEC(libGAP_ELM_GF2MAT(matr,1));
  ncolp = ncoll*ncolr;

  mutable = libGAP_IS_MUTABLE_OBJ(matl) || libGAP_IS_MUTABLE_OBJ(matr);

  /* create a matrix */
  mat = libGAP_NewBag(libGAP_T_POSOBJ, libGAP_SIZE_PLEN_GF2MAT(nrowp));
  libGAP_SET_LEN_GF2MAT(mat,nrowp);
  if (mutable) {
    libGAP_TYPE_POSOBJ(mat) = libGAP_TYPE_LIST_GF2MAT;
    type = libGAP_TYPE_LIST_GF2VEC_LOCKED;
  } else {
    libGAP_TYPE_POSOBJ(mat) = libGAP_TYPE_LIST_GF2MAT_IMM;
    type = libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
  }

  /* allocate 0 matrix */

  for (i = 1; i <= nrowp; i++) {
    libGAP_NEW_GF2VEC(row, type, ncolp);
    libGAP_SET_LEN_GF2VEC(row, ncolp);
    libGAP_SET_ELM_GF2MAT(mat,i,row);
    libGAP_CHANGED_BAG(mat);
  }

  /* allocate data for shifts of rows of matr */
  for (i = 0; i < libGAP_BIPEB; i++) {
    shift[i] = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_PLEN_GF2VEC(ncolr+2*libGAP_BIPEB));
  }

  /* fill in matrix */
  for (j = 1; j <= nrowr; j++) {
    /* create shifts of rows of matr */
    data = (libGAP_UInt *) libGAP_ADDR_OBJ(shift[0]);
    datar = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(matr,j));
    for (k = 0; k < (ncolr+libGAP_BIPEB-1)/libGAP_BIPEB; k++)
      data[k] = datar[k];
    data[k] = 0;
    
    for (i = 1; i < libGAP_BIPEB; i++) { /* now shifts in [1..BIPEB-1] */
      data = (libGAP_UInt *) libGAP_ADDR_OBJ(shift[i]);
      datar = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(matr,j));
      data[0] = datar[0] << i;
      for (k = 1; k < (ncolr+libGAP_BIPEB-1)/libGAP_BIPEB; k++)
	data[k] = (datar[k] << i) | (datar[k-1] >> (libGAP_BIPEB-i));
      data[k] = datar[k-1] >> (libGAP_BIPEB-i);
    }
    for (i = 1; i <= nrowl; i++) {
      data = libGAP_BLOCKS_GF2VEC(libGAP_ELM_GF2MAT(mat,(i-1)*nrowr+j));
      ncol = 0;
      for (k = 1; k <= ncoll; k++) {
	l = 0;
	if (libGAP_BLOCK_ELM_GF2VEC(libGAP_ELM_GF2MAT(matl,i),k) & libGAP_MASK_POS_GF2VEC(k)) {
	  /* append shift[ncol%BIPEB] to data */
	  datar = (libGAP_UInt *) libGAP_ADDR_OBJ(shift[ncol%libGAP_BIPEB]);
	  if (ncol % libGAP_BIPEB) {
	    data[-1] ^= *datar++;
	    l = libGAP_BIPEB - ncol%libGAP_BIPEB;
	  }
	  for (; l < ncolr; l += libGAP_BIPEB)
	    *data++ = *datar++;
	} else {
	  if (ncol % libGAP_BIPEB)
	    l = libGAP_BIPEB - ncol%libGAP_BIPEB;
	  data += (ncolr+libGAP_BIPEB-1-l)/libGAP_BIPEB;
	}
	ncol += ncolr;
      }
    }
  }

  return mat;
}






/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "CONV_GF2VEC", 1, "list",
      libGAP_FuncCONV_GF2VEC, "src/vecgf2.c:CONV_GF2VEC" },

    { "PLAIN_GF2VEC", 1, "gf2vec",
      libGAP_FuncPLAIN_GF2VEC, "src/vecgf2.c:PLAIN_GF2VEC" },

    { "PLAIN_GF2MAT", 1, "gf2mat",
      libGAP_FuncPLAIN_GF2MAT, "src/vecgf2.c:PLAIN_GF2MAT" },

    { "EQ_GF2VEC_GF2VEC", 2, "gf2vec, gf2vec",
      libGAP_FuncEQ_GF2VEC_GF2VEC, "src/vecgf2.c:EQ_GF2VEC_GF2VEC" },

    { "LT_GF2VEC_GF2VEC", 2, "gf2vec, gf2vec",
      libGAP_FuncLT_GF2VEC_GF2VEC, "src/vecgf2.c:LT_GF2VEC_GF2VEC" },

    { "EQ_GF2MAT_GF2MAT", 2, "gf2mat, gf2mat",
      libGAP_FuncEQ_GF2MAT_GF2MAT, "src/vecgf2.c:EQ_GF2MAT_GF2MAT" },

    { "LT_GF2MAT_GF2MAT", 2, "gf2mat, gf2mat",
      libGAP_FuncLT_GF2MAT_GF2MAT, "src/vecgf2.c:LT_GF2MAT_GF2MAT" },

    { "LEN_GF2VEC", 1, "gf2vec",
      libGAP_FuncLEN_GF2VEC, "src/vecgf2.c:LEN_GF2VEC" },

    { "ELM0_GF2VEC", 2, "gf2vec, pos",
      libGAP_FuncELM0_GF2VEC, "src/vecgf2.c:ELM0_GF2VEC" },

    { "ELM_GF2VEC", 2, "gf2vec, pos",
      libGAP_FuncELM_GF2VEC, "src/vecgf2.c:ELM_GF2VEC" },

    { "ELMS_GF2VEC", 2, "gf2vec, poss",
      libGAP_FuncELMS_GF2VEC, "src/vecgf2.c:ELMS_GF2VEC" },

    { "ASS_GF2VEC", 3, "gf2vec, pos, elm",
      libGAP_FuncASS_GF2VEC, "src/vecgf2.c:ASS_GF2VEC" },

    { "ASS_GF2MAT", 3, "gf2mat, pos, elm",
      libGAP_FuncASS_GF2MAT, "src/vecgf2.c:ASS_GF2MAT" },

    { "UNB_GF2VEC", 2, "gf2vec, pos",
      libGAP_FuncUNB_GF2VEC, "src/vecgf2.c:UNB_GF2VEC" },

    { "UNB_GF2MAT", 2, "gf2mat, pos",
      libGAP_FuncUNB_GF2MAT, "src/vecgf2.c:UNB_GF2MAT" },

    { "ZERO_GF2VEC", 1, "gf2vec",
      libGAP_FuncZERO_GF2VEC, "src/vecgf2.c:ZERO_GF2VEC" },

    { "ZERO_GF2VEC_2", 1, "len",
      libGAP_FuncZERO_GF2VEC_2, "src/vecgf2.c:ZERO_GF2VEC_2" },

    { "INV_GF2MAT_MUTABLE", 1, "gf2mat",
      libGAP_FuncINV_GF2MAT_MUTABLE, "src/vecgf2.c:INV_GF2MAT_MUTABLE" },

    { "INV_GF2MAT_SAME_MUTABILITY", 1, "gf2mat",
      libGAP_FuncINV_GF2MAT_SAME_MUTABILITY, "src/vecgf2.c:INV_GF2MAT_SAME_MUTABILITY" },

    { "INV_GF2MAT", 1, "gf2mat",
      libGAP_FuncINV_GF2MAT_IMMUTABLE, "src/vecgf2.c:INV_GF2MAT_IMMUTABLE" },

    { "INV_PLIST_GF2VECS_DESTRUCTIVE", 1, "list",
      libGAP_FuncINV_PLIST_GF2VECS_DESTRUCTIVE, "src/vecgf2.c:INV_PLIST_GF2VECS_DESTRUCTIVE" },

    { "SUM_GF2VEC_GF2VEC", 2, "gf2vec, gf2vec",
      libGAP_FuncSUM_GF2VEC_GF2VEC, "src/vecgf2.c:SUM_GF2VEC_GF2VEC" },

    { "PROD_GF2VEC_GF2VEC", 2, "gf2vec, gf2vec",
      libGAP_FuncPROD_GF2VEC_GF2VEC, "src/vecgf2.c:PROD_GF2VEC_GF2VEC" },

    { "PROD_GF2VEC_GF2MAT", 2, "gf2vec, gf2mat",
      libGAP_FuncPROD_GF2VEC_GF2MAT, "src/vecgf2.c:PROD_GF2VEC_GF2MAT" },

    { "PROD_GF2MAT_GF2VEC", 2, "gf2mat, gf2vec",
      libGAP_FuncPROD_GF2MAT_GF2VEC, "src/vecgf2.c:PROD_GF2MAT_GF2VEC" },

    { "PROD_GF2MAT_GF2MAT", 2, "gf2matl, gf2matr",
      libGAP_FuncPROD_GF2MAT_GF2MAT, "src/vecgf2.c:PROD_GF2MAT_GF2MAT" },

    { "PROD_GF2MAT_GF2MAT_SIMPLE", 2, "gf2matl, gf2matr",
      libGAP_FuncPROD_GF2MAT_GF2MAT_SIMPLE, "src/vecgf2.c:PROD_GF2MAT_GF2MAT_SIMPLE" },

    { "PROD_GF2MAT_GF2MAT_ADVANCED", 4, "gf2matl, gf2matr, greaselevel, blocklevel",
      libGAP_FuncPROD_GF2MAT_GF2MAT_ADVANCED, "src/vecgf2.c:PROD_GF2MAT_GF2MAT_ADVANCED" },

    { "ADDCOEFFS_GF2VEC_GF2VEC_MULT", 3, "gf2vec, gf2vec, mul",
      libGAP_FuncADDCOEFFS_GF2VEC_GF2VEC_MULT, "src/vecgf2.c:ADDCOEFFS_GF2VEC_GF2VEC_MULT" },

    { "ADDCOEFFS_GF2VEC_GF2VEC", 2, "gf2vec, gf2vec",
      libGAP_FuncADDCOEFFS_GF2VEC_GF2VEC, "src/vecgf2.c:ADDCOEFFS_GF2VEC_GF2VEC" },

    { "SHRINKCOEFFS_GF2VEC", 1, "gf2vec",
      libGAP_FuncSHRINKCOEFFS_GF2VEC, "src/vecgf2.c:SHRINKCOEFFS_GF2VEC" },

    { "POSITION_NONZERO_GF2VEC", 2, "gf2vec, zero",
      libGAP_FuncPOSITION_NONZERO_GF2VEC, "src/vecgf2.c:POSITION_NONZERO_GF2VEC" },

    { "POSITION_NONZERO_GF2VEC3", 3, "gf2vec, zero, from",
      libGAP_FuncPOSITION_NONZERO_GF2VEC3, "src/vecgf2.c:POSITION_NONZERO_GF2VEC3" },

    { "MULT_ROW_VECTOR_GF2VECS_2", 2, "gf2vecl, mul",
      libGAP_FuncMULT_ROW_VECTOR_GF2VECS_2, "src/vecgf2.c:MULT_ROW_VECTOR_GF2VECS_2" },

    { "APPEND_GF2VEC", 2, "gf2vecl, gf2vecr",
      libGAP_FuncAPPEND_VECGF2, "src/vecgf2.c:APPEND_GF2VEC" },

    { "SHALLOWCOPY_GF2VEC", 1, "gf2vec",
      libGAP_FuncSHALLOWCOPY_VECGF2, "src/vecgf2.c:SHALLOWCOPY_GF2VEC" },

    { "NUMBER_GF2VEC", 1, "gf2vec",
      libGAP_FuncNUMBER_VECGF2, "src/vecgf2.c:NUMBER_GF2VEC" },

    { "TRANSPOSED_GF2MAT", 1, "gf2mat",
      libGAP_FuncTRANSPOSED_GF2MAT, "src/vecgf2.c:TRANSPOSED_GF2MAT" },

    { "DIST_GF2VEC_GF2VEC", 2, "gf2vec, gf2vec",
      libGAP_FuncDIST_GF2VEC_GF2VEC, "src/vecgf2.c:DIST_GF2VEC_GF2VEC" },

    { "DIST_VEC_CLOS_VEC", 3, "list, gf2vec, list",
      libGAP_FuncDistVecClosVec, "src/vecgf2.c:DIST_VEC_CLOS_VEC" },

    { "SUM_GF2MAT_GF2MAT", 2, "matl, matr",
      libGAP_FuncSUM_GF2MAT_GF2MAT, "src/vecgf2.c:SUM_GF2MAT_GF2MAT" },

    { "A_CLOS_VEC", 4, "list, gf2vec, int, int",
      libGAP_FuncAClosVec, "src/vecgf2.c:A_CLOS_VEC" },

    { "A_CLOS_VEC_COORDS", 4, "list, gf2vec, int, int",
      libGAP_FuncAClosVecCoords, "src/vecgf2.c:A_CLOS_VEC_COORDS" },

    { "COSET_LEADERS_INNER_GF2", 4, "veclis, weight, tofind, leaders",
      libGAP_FuncCOSET_LEADERS_INNER_GF2, "src/vecgf2.c:COSET_LEADERS_INNER_GF2" },

    { "CONV_GF2MAT", 1, "list",
      libGAP_FuncCONV_GF2MAT, "src/vecgf2.c:CONV_GF2MAT" },

    { "PROD_GF2VEC_ANYMAT", 2, "vec, mat",
      libGAP_FuncProdGF2VecAnyMat, "src/vecgf2.c:PROD_GF2VEC_ANYMAT" },
    
    { "RIGHTMOST_NONZERO_GF2VEC", 1, "vec",
      libGAP_FuncRIGHTMOST_NONZERO_GF2VEC, "src/vecgf2.c:RIGHTMOST_NONZERO_GF2VEC" },
    
    { "RESIZE_GF2VEC", 2, "vec, newlen",
      libGAP_FuncRESIZE_GF2VEC, "src/vecgf2.c:RESIZE_GF2VEC" },
    
    { "SHIFT_LEFT_GF2VEC", 2, "vec, amount",
      libGAP_FuncSHIFT_LEFT_GF2VEC, "src/vecgf2.c:SHIFT_LEFT_GF2VEC" },
    
    { "SHIFT_RIGHT_GF2VEC", 3, "vec, amount, zero",
      libGAP_FuncSHIFT_RIGHT_GF2VEC, "src/vecgf2.c:SHIFT_RIGHT_GF2VEC" },

    { "ADD_GF2VEC_GF2VEC_SHIFTED", 4, "vec1, vec2,len2, off",
      libGAP_FuncADD_GF2VEC_GF2VEC_SHIFTED, "src/vecgf2.c:ADD_GF2VEC_GF2VEC_SHIFTED" },
    
    { "PROD_COEFFS_GF2VEC", 4, "vec1, len1, vec2,len2",
      libGAP_FuncPROD_COEFFS_GF2VEC, "src/vecgf2.c:PROD_COEFFS_GF2VEC" },

    { "REDUCE_COEFFS_GF2VEC", 4, "vec1, len1, vec2,len2",
      libGAP_FuncREDUCE_COEFFS_GF2VEC, "src/vecgf2.c:REDUCE_COEFFS_GF2VEC" },

    { "QUOTREM_COEFFS_GF2VEC", 4, "vec1, len1, vec2,len2",
      libGAP_FuncQUOTREM_COEFFS_GF2VEC, "src/vecgf2.c:QUOTREM_COEFFS_GF2VEC" },

    { "SEMIECHELON_LIST_GF2VECS", 1, "mat",
      libGAP_FuncSEMIECHELON_LIST_GF2VECS, "src/vecgf2.c:SEMIECHELON_LIST_GF2VECS" },

    { "SEMIECHELON_LIST_GF2VECS_TRANSFORMATIONS", 1, "mat",
      libGAP_FuncSEMIECHELON_LIST_GF2VECS_TRANSFORMATIONS, "src/vecgf2.c:SEMIECHELON_LIST_GF2VECS_TRANSFORMATIONS" },

    { "TRIANGULIZE_LIST_GF2VECS", 1, "mat",
      libGAP_FuncTRIANGULIZE_LIST_GF2VECS, "src/vecgf2.c:TRIANGULIZE_LIST_GF2VECS" },

    { "DETERMINANT_LIST_GF2VECS", 1, "mat",
      libGAP_FuncDETERMINANT_LIST_GF2VECS, "src/vecgf2.c:DETERMINANT_LIST_GF2VECS" },

    { "RANK_LIST_GF2VECS", 1, "mat",
      libGAP_FuncRANK_LIST_GF2VECS, "src/vecgf2.c:RANK_LIST_GF2VECS" },
    
    { "KRONECKERPRODUCT_GF2MAT_GF2MAT", 2, "mat, mat",
      libGAP_FuncKRONECKERPRODUCT_GF2MAT_GF2MAT, "src/vecgf2.c:KRONECKERPRODUCT_GF2MAT_GF2MAT" },


    { "COPY_SECTION_GF2VECS", 5, "src, dest, from, to, howmany",
      libGAP_FuncCOPY_SECTION_GF2VECS, "src/vecgf2.c:COPY_SECTION_GF2VECS"},
    
    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_RNheads = 0;
  libGAP_RNvectors = 0;
  libGAP_RNcoeffs = 0;
  libGAP_RNrelns = 0;

    /* import kind functions                                               */
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_GF2VEC",     &libGAP_TYPE_LIST_GF2VEC     );
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_GF2VEC_IMM", &libGAP_TYPE_LIST_GF2VEC_IMM );
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_GF2VEC_IMM_LOCKED", &libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED );
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_GF2VEC_LOCKED", &libGAP_TYPE_LIST_GF2VEC_LOCKED );
    libGAP_ImportFuncFromLibrary( "IsGF2VectorRep",       &libGAP_IsGF2VectorRep       );
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_GF2MAT",     &libGAP_TYPE_LIST_GF2MAT     );
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_GF2MAT_IMM", &libGAP_TYPE_LIST_GF2MAT_IMM );

    /* initialize one and zero of GF2                                      */
    libGAP_ImportGVarFromLibrary( "GF2One",  &libGAP_GF2One  );
    libGAP_ImportGVarFromLibrary( "GF2Zero", &libGAP_GF2Zero );

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    libGAP_InitFopyGVar("ConvertToVectorRep", &libGAP_ConvertToVectorRep);
    libGAP_InitFopyGVar("IsLockedRepresentationVector", &libGAP_IsLockedRepresentationVector);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoGF2Vec()  . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "vecgf2",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoGF2Vec ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  vecgf2.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
