/****************************************************************************
**
*W  set.c                       GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions which mainly deal with proper sets.
**
**  A *proper set* is a list that has no holes, no duplicates, and is sorted.
**  For the full definition  of sets see chapter "Sets" in the {\GAP} Manual.
**  Read also section "More about Sets" about the internal flag for sets.
**
**  The second part consists  of the functions 'IsSet', 'SetList', 'SetList',
**  'IsEqualSet',  'IsSubsetSet',    'AddSet',    'RemoveSet',    'UniteSet',
**  'IntersectSet',  and 'SubtractSet'.  These  functions make it possible to
**  make sets, either  by converting a  list to  a  set, or  by computing the
**  union, intersection, or difference of two sets.
*/
#include        <assert.h>              /* assert                          */
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listfunc.h"            /* functions for generic lists     */
#include        "plist.h"               /* plain lists                     */
#include        "set.h"                 /* plain sets                      */
#include        "string.h"              /* strings                         */


/****************************************************************************
**
*F  IsSet( <list> ) . . . . . . . . . . . . . . . . . test if a list is a set
**
**  'IsSet' returns 1 if the list <list> is a proper set and 0
**  otherwise.  A proper set is a  list that has no holes,
**  no duplicates, and is sorted.  As a side effect 'IsSet' changes the
**  type of proper sets as appropriate.
**
**  A typical call in the set functions looks like this:
**
**  |    if ( ! IsSet(list) )  list = SetList(list); |
**
**  This tests if 'list' is a proper set and the type  is changed 
**  If it is not  then 'SetList' is  called to make  a copy of 'list', remove
**  the holes, sort the copy, and remove the duplicates.  
** 
*/

libGAP_Int libGAP_IsSet ( 
    libGAP_Obj                 list )
{
    libGAP_Int                 isSet;          /* result                          */

    /* if <list> is a plain list                                           */
    if ( libGAP_IS_PLIST( list ) ) {

        /* if <list> is the empty list, its a set (:-)                     */
        if ( libGAP_LEN_PLIST(list) == 0 ) {
            libGAP_SET_FILT_LIST( list, libGAP_FN_IS_EMPTY );
            isSet = 1;
        }

        /* if <list>  strictly sorted, its a set            */
        else if ( libGAP_IS_SSORT_LIST(list) ) {
            isSet = 1;
        }

        /* otherwise it is not a set                                       */
        else {
            isSet = 0;
        }

    }

    /* if it is another small list                                         */
    else if ( libGAP_IS_SMALL_LIST(list) ) {

        /* if <list> is the empty list, its a set (:-)                     */
        if ( libGAP_LEN_LIST(list) == 0 ) {
            libGAP_PLAIN_LIST( list );
            libGAP_SET_FILT_LIST( list, libGAP_FN_IS_EMPTY );
            isSet = 1;
        }

        /* if <list> strictly sorted, its a set            */
        else if (  libGAP_IS_SSORT_LIST(list) ) {
            libGAP_PLAIN_LIST( list );
            /* SET_FILT_LIST( list, FN_IS_HOMOG ); */
            libGAP_SET_FILT_LIST( list, libGAP_FN_IS_SSORT );
            isSet = 1;
        }

        /* otherwise it is not a set                                       */
        else {
            isSet = 0;
        }

    }

    /* otherwise it is certainly not a set                                 */
    else {
        isSet = 0;
    }

    /* return the result                                                   */
    return isSet;
}


/****************************************************************************
**
*F  SetList( <list> ) . . . . . . . . . . . . . . . .  make a set from a list
**
**  'SetList' returns  a new set that contains  the elements of <list>.  Note
**  that 'SetList' returns a new plain list even if <list> was already a set.
**
**  'SetList' makes a copy  of the list  <list>, removes the holes, sorts the
**  copy and finally removes duplicates, which must appear next to each other
**  now that the copy is sorted.
*/
libGAP_Obj libGAP_SetList (
    libGAP_Obj                 list )
{
    libGAP_Obj                 set;            /* result set                      */
    libGAP_Int                 lenSet;         /* length of <set>                 */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element of the list         */
    libGAP_UInt                status;        /* the elements are mutable        */
    libGAP_UInt                i;              /* loop variable                   */

    /* make a dense copy                                                   */
    lenList = libGAP_LEN_LIST( list );
    set = libGAP_NEW_PLIST( libGAP_T_PLIST, lenList );
    lenSet = 0;
    for ( i = 1; i <= lenList; i++ ) {
        elm = libGAP_ELMV0_LIST( list, i );
        if ( elm != 0 ) {
            lenSet += 1;
            libGAP_SET_ELM_PLIST( set, lenSet, elm );
	    libGAP_CHANGED_BAG(set);	/* in case elm had to be made, not just extracted  */
        }
    }
    libGAP_SET_LEN_PLIST( set, lenSet );
    libGAP_SET_FILT_LIST( set, libGAP_FN_IS_DENSE );

    /* sort the set (which is a dense plain list)                          */
    libGAP_SortDensePlist( set );

    /* remove duplicates                                                   */
    status = libGAP_RemoveDupsDensePlist( set );

    /* adjust flags where possible                                   */
    switch(status)
      {
      case 0:
	break;
	
      case 1:
	libGAP_SET_FILT_LIST(set, libGAP_FN_IS_NHOMOG);
	libGAP_SET_FILT_LIST(set, libGAP_FN_IS_SSORT);
	break;
	
      case 2:
	libGAP_SET_FILT_LIST( set, libGAP_FN_IS_HOMOG );
	libGAP_SET_FILT_LIST( set, libGAP_FN_IS_SSORT );
	break;
      }

    /* return set                                                          */
    return set;
}


/****************************************************************************
**
*F  FuncLIST_SORTED_LIST( <self>, <list> )  . . . . .  make a set from a list
**
**  'FuncLIST_SORTED_LIST' implements the internal function 'SetList'.
**
**  'SetList( <list> )'
**
**  'SetList' returns a new proper set, which is represented as a sorted list
**  without holes or duplicates, containing the elements of the list <list>.
**
**  'SetList' returns a new list even if the list <list> is already a  proper
**  set, in this case it is equivalent to 'ShallowCopy' (see  "ShallowCopy").
*/
libGAP_Obj libGAP_FuncLIST_SORTED_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    libGAP_Obj                 set;            /* result                          */

    /* check the argument                                                  */
    while ( ! libGAP_IS_SMALL_LIST( list ) ) {
        list = libGAP_ErrorReturnObj(
            "Set: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }

    /* if the list is empty create a new empty list                        */
    if ( libGAP_LEN_LIST(list) == 0 ) {
        set = libGAP_NEW_PLIST( libGAP_T_PLIST_EMPTY, 0 );
    }

    /* if <list> is a set just shallow copy it                             */
    else if ( /* IS_HOMOG_LIST(list) && */ libGAP_IS_SSORT_LIST(list) ) {
        set = libGAP_SHALLOW_COPY_OBJ( list );
    }

    /* otherwise let 'SetList' do the work                                 */
    else {
        set = libGAP_SetList( list );
    }

    /* return the set                                                      */
    return set;
}


/****************************************************************************
**
*F  FuncIS_EQUAL_SET(<self>,<l1>,<l2>) test if a two lists are equal as sets
**
**  'FuncIS_EQUAL_SET' implements the internal function 'IsEqualSet'.
**
**  'IsEqualSet( <list1>, <list2> )'
**
**  'IsEqualSet'  returns  'true' if the  two  lists <list1> and  <list2> are
**  equal *when viewed as sets*, and 'false'  otherwise.  <list1> and <list2>
**  are equal if every element of  <list1> is also  an element of <list2> and
**  if every element of <list2> is also an element of <list1>.
*/
libGAP_Int             libGAP_EqSet (
    libGAP_Obj                 listL,
    libGAP_Obj                 listR )
{
    libGAP_Int                 lenL;           /* length of the left operand      */
    libGAP_Int                 lenR;           /* length of the right operand     */
    libGAP_Obj                 elmL;           /* element of the left operand     */
    libGAP_Obj                 elmR;           /* element of the right operand    */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the lengths of the lists and compare them                       */
    lenL = libGAP_LEN_PLIST( listL );
    lenR = libGAP_LEN_PLIST( listR );
    if ( lenL != lenR ) {
        return 0L;
    }

    /* loop over the elements and compare them                             */
    for ( i = 1; i <= lenL; i++ ) {
        elmL = libGAP_ELM_PLIST( listL, i );
        elmR = libGAP_ELM_PLIST( listR, i );
        if ( ! libGAP_EQ( elmL, elmR ) ) {
            return 0L;
        }
    }

    /* no differences found, the lists are equal                           */
    return 1L;
}

libGAP_Obj             libGAP_FuncIS_EQUAL_SET (
    libGAP_Obj                 self,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    /* check the arguments, convert to sets if necessary                   */
    while ( ! libGAP_IS_SMALL_LIST(list1) ) {
        list1 = libGAP_ErrorReturnObj(
            "IsEqualSet: <list1> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list1), 0L,
            "you can replace <list1> via 'return <list1>;'" );
    }
    if ( ! libGAP_IsSet( list1 ) )  list1 = libGAP_SetList( list1 );
    while ( ! libGAP_IS_SMALL_LIST(list2) ) {
        list2 = libGAP_ErrorReturnObj(
            "IsEqualSet: <list2> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list2), 0L,
            "you can replace <list2> via 'return <list2>;'" );
    }
    if ( ! libGAP_IsSet( list2 ) )  list2 = libGAP_SetList( list2 );

    /* and now compare them                                                */
    return (libGAP_EqSet( list1, list2 ) ? libGAP_True : libGAP_False );
}


/****************************************************************************
**
*F  FuncIS_SUBSET_SET(<self>,<s1>,<s2>) test if a set is a subset of another
**
**  'FuncIS_SUBSET_SET' implements the internal function 'IsSubsetSet'.
**
**  'IsSubsetSet( <set1>, <set2> )'
**
**  'IsSubsetSet' returns 'true'  if the set  <set2> is a  subset of the  set
**  <set1>, that is if every element of <set2>  is also an element of <set1>.
**  Either  argument may also  be a list that is  not a proper  set, in which
**  case 'IsSubsetSet' silently applies 'Set' (see "Set") to it first.
*/
libGAP_Obj             libGAP_FuncIS_SUBSET_SET (
    libGAP_Obj                 self,
    libGAP_Obj                 set1,
    libGAP_Obj                 set2 )
{
    libGAP_UInt                len1;           /* length of  the left  set        */
    libGAP_UInt                len2;           /* length of  the right set        */
    libGAP_UInt                i1;             /* index into the left  set        */
    libGAP_UInt                i2;             /* index into the right set        */
    libGAP_Obj                 e1;             /* element of left  set            */
    libGAP_Obj                 e2;             /* element of right set            */
    libGAP_UInt                pos;            /* position                        */

    /* check the arguments, convert to sets if necessary                   */
    while ( ! libGAP_IS_SMALL_LIST(set1) ) {
        set1 = libGAP_ErrorReturnObj(
            "IsSubsetSet: <set1> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set1), 0L,
            "you can replace <set1> via 'return <set1>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(set2) ) {
        set2 = libGAP_ErrorReturnObj(
            "IsSubsetSet: <set2> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set2), 0L,
            "you can replace <set2> via 'return <set2>;'" );
    }
    if ( ! libGAP_IsSet( set1 ) )  set1 = libGAP_SetList( set1 );
    if ( ! libGAP_IsSet( set2 ) )  set2 = libGAP_SetList( set2 );

    /* special case if the second argument is a set                        */
    if ( libGAP_IsSet( set2 ) ) {

        /* get the logical lengths and get the pointer                     */
        len1 = libGAP_LEN_PLIST( set1 );
        len2 = libGAP_LEN_PLIST( set2 );
        i1 = 1;
        i2 = 1;

        /* now compare the two sets                                        */
        while ( i1 <= len1 && i2 <= len2 && len2 - i2 <= len1 - i1 ) {
            e1 = libGAP_ELM_PLIST( set1, i1 );
            e2 = libGAP_ELM_PLIST( set2, i2 );
            if ( libGAP_EQ( e1, e2 ) ) {
                i1++;  i2++;
            }
            else if ( libGAP_LT( e1, e2 ) ) {
                i1++;
            }
            else {
                break;
            }
        }

    }

    /* general case                                                        */
    else {

        /* first convert the other argument into a proper list             */
        libGAP_PLAIN_LIST( set2 );

        /* get the logical lengths                                         */
        len1 = libGAP_LEN_PLIST( set1 );
        len2 = libGAP_LEN_PLIST( set2 );

        /* loop over the second list and look for every element            */
        for ( i2 = 1; i2 <= len2; i2++ ) {

            /* ignore holes                                                */
            if ( libGAP_ELM_PLIST(set2,i2) == 0 )
                continue;

            /* perform the binary search to find the position              */
            pos = libGAP_PositionSortedDensePlist( set1, libGAP_ELM_PLIST(set2,i2) );

            /* test if the element was found at position k                 */
            if ( len1<pos || ! libGAP_EQ(libGAP_ELM_PLIST(set1,pos),libGAP_ELM_PLIST(set2,i2)) ) {
                break;
            }

        }

    }

    /* return 'true' if every element of <set2> appeared in <set1>         */
    return ((i2 == len2 + 1) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * GAP level functions  * * * * * * * * * * * * *
*/

/****************************************************************************
**


*F  FuncADD_SET( <self>, <set>, <obj> ) . . . . . . . add an element to a set
**
**  'FuncADD_SET' implements the internal function 'AddSet'.
**
**  'AddSet( <set>, <obj> )'
**
**  'AddSet' adds <obj>, which may be an object  of an arbitrary type, to the
**  set <set>, which must be a proper set.  If <obj> is already an element of
**  the set <set>, then <set> is not changed.  Otherwise <obj> is inserted at
**  the correct position such that <set> is again a set afterwards.
**
**  'AddSet' does not return  anything, it is only  called for the side effect
**  of changing <set>.
*/
libGAP_Obj libGAP_FuncADD_SET (
		 libGAP_Obj                 self,
		 libGAP_Obj                 set,
		 libGAP_Obj                 obj )
{
  libGAP_UInt                len;            /* logical length of the list      */
  libGAP_UInt                pos;            /* position                        */
  libGAP_UInt                isCyc;          /* True if the set being added to consists
					 of kernel cyclotomics           */
  libGAP_UInt                notpos;         /* position of an original element
					 (not the new one)               */
  libGAP_UInt                wasHom;
  libGAP_UInt                wasNHom;
  libGAP_UInt                wasTab;
    
  /* check the arguments                                                 */
  while ( ! libGAP_IsSet(set) || ! libGAP_IS_MUTABLE_OBJ(set) ) {
    set = libGAP_ErrorReturnObj(
			 "AddSet: <set> must be a mutable proper set (not a %s)",
			 (libGAP_Int)libGAP_TNAM_OBJ(set), 0L,
			 "you can replace <set> via 'return <set>;'" );
  }
  len = libGAP_LEN_LIST(set);

  /* perform the binary search to find the position                      */
  pos = libGAP_PositionSortedDensePlist( set, obj );

  /* add the element to the set if it is not already there               */
  if ( len < pos || ! libGAP_EQ( libGAP_ELM_PLIST(set,pos), obj ) ) {
    libGAP_GROW_PLIST( set, len+1 );
    libGAP_SET_LEN_PLIST( set, len+1 );
    {
      libGAP_Obj *ptr;
      ptr = libGAP_PTR_BAG(set);
      memmove((void *)(ptr + pos+1),(void*)(ptr+pos),(size_t)(sizeof(libGAP_Obj)*(len+1-pos)));
#if 0
      for ( i = len+1; pos < i; i-- ) {
	*ptr = *(ptr-1);
	ptr--;   */
	/* SET_ELM_PLIST( set, i, ELM_PLIST(set,i-1) ); */
      }
#endif
    }
    libGAP_SET_ELM_PLIST( set, pos, obj );
    libGAP_CHANGED_BAG( set );

    /* fix up the type of the result                                   */
    if ( libGAP_HAS_FILT_LIST( set, libGAP_FN_IS_SSORT ) ) {
      isCyc = (libGAP_TNUM_OBJ(set) == libGAP_T_PLIST_CYC_SSORT);
      wasHom = libGAP_HAS_FILT_LIST(set, libGAP_FN_IS_HOMOG);
      wasTab = libGAP_HAS_FILT_LIST(set, libGAP_FN_IS_TABLE);
      wasNHom = libGAP_HAS_FILT_LIST(set, libGAP_FN_IS_NHOMOG);
      libGAP_CLEAR_FILTS_LIST(set);
      /* the result of addset is always dense */
      libGAP_SET_FILT_LIST( set, libGAP_FN_IS_DENSE );

				/* if the object we added was not
                                   mutable then we might be able to
                                   conclude more */
      if ( ! libGAP_IS_MUTABLE_OBJ(obj) ) {
				/* a one element list is automatically
                                   homogenous  and ssorted */
	if (len == 0 )
	  {
	    if (libGAP_TNUM_OBJ(obj) <= libGAP_T_CYC)
	      libGAP_RetypeBag( set, libGAP_T_PLIST_CYC_SSORT);
	    else
	      {
		libGAP_SET_FILT_LIST( set, libGAP_FN_IS_HOMOG );
		libGAP_SET_FILT_LIST( set, libGAP_FN_IS_SSORT );
		if (libGAP_IS_HOMOG_LIST(obj))	/* it might be a table */
		  libGAP_SET_FILT_LIST( set, libGAP_FN_IS_TABLE );
	      }
	  }
	else
	  {
	    /* Now determine homogeneity */
	    if (isCyc)
	      if (libGAP_TNUM_OBJ(obj) <= libGAP_T_CYC)
		libGAP_RetypeBag( set, libGAP_T_PLIST_CYC_SSORT);
	      else
		{
		  libGAP_RESET_FILT_LIST(set, libGAP_FN_IS_HOMOG);
		  libGAP_SET_FILT_LIST(set, libGAP_FN_IS_NHOMOG);
		}
	    else if (wasHom)
	      {
		if (!libGAP_SyInitializing) {
		  notpos = (pos == 1) ? 2 : 1;
		  if (libGAP_FAMILY_OBJ(libGAP_ELM_PLIST(set,notpos)) == libGAP_FAMILY_OBJ(obj))
		    {
		      libGAP_SET_FILT_LIST(set, libGAP_FN_IS_HOMOG);
		      if (wasTab) {
			if (libGAP_IS_HOMOG_LIST( obj ))
			  libGAP_SET_FILT_LIST(set, libGAP_FN_IS_TABLE);
		      }
		    }

		  else
		    libGAP_SET_FILT_LIST(set, libGAP_FN_IS_NHOMOG);
		}
	      }
	    else if (wasNHom)
	      libGAP_SET_FILT_LIST(set, libGAP_FN_IS_NHOMOG);
	  }
      }
      libGAP_SET_FILT_LIST( set, libGAP_FN_IS_SSORT );
    }
    else {
      libGAP_CLEAR_FILTS_LIST(set);
      libGAP_SET_FILT_LIST( set, libGAP_FN_IS_DENSE );
    }
  }

  /* return void, this is a procedure                                    */
  return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncREM_SET( <self>, <set>, <obj> ) . . . .  remove an element from a set
**
**  'FuncREM_SET' implements the internal function 'RemoveSet'.
**
**  'RemoveSet( <set>, <obj> )'
**
**  'RemoveSet' removes <obj>, which may be an object of arbitrary type, from
**  the set <set>, which must be a  proper set.  If  <obj> is in  <set> it is
**  removed and all  entries of <set>  are shifted one position leftwards, so
**  that <set> has no  holes.  If <obj>  is not in  <set>, then <set>  is not
**  changed.  No error is signalled in this case.
**
**  'RemoveSet'   does   not return anything,  it   is  only called  for  the
**  side effect of changing <set>.
*/
libGAP_Obj libGAP_FuncREM_SET (
    libGAP_Obj                 self,
    libGAP_Obj                 set,
    libGAP_Obj                 obj )
{
    libGAP_UInt                len;            /* logical length of the list      */
    libGAP_UInt                pos;            /* position                        */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 *ptr;

    /* check the arguments                                                 */
    while ( ! libGAP_IsSet(set) || ! libGAP_IS_MUTABLE_OBJ(set) ) {
        set = libGAP_ErrorReturnObj(
            "RemoveSet: <set> must be a mutable proper set (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set), 0L,
            "you can replace <set> via 'return <set>;'" );
    }
    len = libGAP_LEN_LIST(set);

    /* perform the binary search to find the position                      */
    pos = libGAP_PositionSortedDensePlist( set, obj );

    /* remove the element from the set if it is there                      */
    if ( pos <= len && libGAP_EQ( libGAP_ELM_PLIST(set,pos), obj ) ) {

        ptr = libGAP_PTR_BAG(set) + pos;
        for ( i = pos; i < len; i++ ) {
	    *ptr = *(ptr+1);
	    ptr ++;
	    /*             SET_ELM_PLIST( set, i, ELM_PLIST(set,i+1) ); */
        }
        libGAP_SET_ELM_PLIST( set, len, 0 );
        libGAP_SET_LEN_PLIST( set, len-1 );

        /* fix up the type of the result                                   */
        if ( len-1 == 0 ) {
            libGAP_CLEAR_FILTS_LIST(set);
            libGAP_SET_FILT_LIST( set, libGAP_FN_IS_EMPTY );
        }
    }

    /* return void, this is a procedure                                    */
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*V  TmpUnion  . . . . . . . . . . . . . . . . . . buffer for the union, local
*F  FuncUNITE_SET( <self>, <set1>, <set2> ) . . .  unite one set with another
**
**  'FuncUNITE_SET' implements the internal function 'UniteSet'.
**
**  'UniteSet( <set1>, <set2> )'
**
**  'UniteSet' changes the set <set1> so that it becomes the  union of <set1>
**  and <set2>.  The union is the set of those elements  that are elements of
**  either set.  So 'UniteSet'  adds (see  "AddSet")  all elements to  <set1>
**  that are in <set2>.  <set2> may be a list that  is  not  a proper set, in
**  which case 'Set' is silently applied to it.
**
**  'FuncUNITE_SET' merges <set1> and <set2> into a  buffer that is allocated
**  at initialization time.
**
**  'TmpUnion' is the global  bag that serves as  temporary bag for the union.
**  It is created in 'InitSet' and is resized when necessary.
**
**   This doesn't work, because UniteSet calls EQ, which can result in a nested call to
** UniteSet, which must accordingly be re-entrant.
*/

libGAP_Obj libGAP_FuncUNITE_SET (
    libGAP_Obj                 self,
    libGAP_Obj                 set1,
    libGAP_Obj                 set2 )
{
    libGAP_UInt                lenr;           /* length  of result set           */
    libGAP_UInt                len1;           /* length  of left  set            */
    libGAP_UInt                len2;           /* length  of right set            */
    libGAP_UInt                i1;             /* index into left  set            */
    libGAP_UInt                i2;             /* index into right set            */
    libGAP_Obj                 e1;             /* element of left  set            */
    libGAP_Obj                 e2;             /* element of right set            */
    libGAP_Obj                 TmpUnion;

    /* check the arguments                                                 */
    while ( ! libGAP_IsSet(set1) || ! libGAP_IS_MUTABLE_OBJ(set1) ) {
        set1 = libGAP_ErrorReturnObj(
            "UniteSet: <set1> must be a mutable proper set (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set1), 0L,
            "you can replace <set1> via 'return <set1>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(set2) ) {
        set2 = libGAP_ErrorReturnObj(
            "UniteSet: <set2> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set2), 0L,
            "you can replace <set2> via 'return <set2>;'" );
    }
    if ( ! libGAP_IsSet(set2) )  set2 = libGAP_SetList(set2);

    /* get the logical lengths and the pointer                             */
    len1 = libGAP_LEN_PLIST( set1 );
    len2 = libGAP_LEN_PLIST( set2 );
    TmpUnion = libGAP_NEW_PLIST(libGAP_T_PLIST,len1+len2);
    /*     GROW_PLIST( TmpUnion, len1 + len2 );*/
    lenr = 0;
    i1 = 1;
    i2 = 1;

    /* now merge the two sets into the union                               */
    while ( i1 <= len1 && i2 <= len2 ) {
        e1 = libGAP_ELM_PLIST( set1, i1 );
        e2 = libGAP_ELM_PLIST( set2, i2 );
        if ( libGAP_EQ( e1, e2 ) ) {
            lenr++;
            libGAP_SET_ELM_PLIST( TmpUnion, lenr, e1 );
            libGAP_CHANGED_BAG( TmpUnion );
            i1++;  i2++;
        }
        else if ( libGAP_LT( e1, e2 ) ) {
            lenr++;
            libGAP_SET_ELM_PLIST( TmpUnion, lenr, e1 );
            libGAP_CHANGED_BAG( TmpUnion );
            i1++;
        }
        else {
            lenr++;
            libGAP_SET_ELM_PLIST( TmpUnion, lenr, e2 );
            libGAP_CHANGED_BAG( TmpUnion );
            i2++;
        }
    }
    while ( i1 <= len1 ) {
        e1 = libGAP_ELM_PLIST( set1, i1 );
        lenr++;
        libGAP_SET_ELM_PLIST( TmpUnion, lenr, e1 );
        libGAP_CHANGED_BAG( TmpUnion );
        i1++;
    }
    while ( i2 <= len2 ) {
        e2 = libGAP_ELM_PLIST( set2, i2 );
        lenr++;
        libGAP_SET_ELM_PLIST( TmpUnion, lenr, e2 );
        libGAP_CHANGED_BAG( TmpUnion );
        i2++;
    }

    /* fix up the type of the result                                       */
    if ( 0 == libGAP_LEN_PLIST(set1) ) {
        libGAP_RetypeBag( set1, libGAP_MUTABLE_TNUM(libGAP_TNUM_OBJ(set2)) );
    } else if ( 0 != libGAP_LEN_PLIST(set2)) {
      if (libGAP_HAS_FILT_LIST(set1, libGAP_FN_IS_HOMOG)) {
	if( !libGAP_HAS_FILT_LIST(set2, libGAP_FN_IS_HOMOG))
	  libGAP_RESET_FILT_LIST(set1, libGAP_FN_IS_HOMOG);
	else if (!libGAP_SyInitializing &&
		 libGAP_FAMILY_OBJ(libGAP_ELM_PLIST(set1,1)) != libGAP_FAMILY_OBJ(libGAP_ELM_PLIST(set2,1)))
	  {
	    libGAP_RetypeBag(set1, libGAP_T_PLIST_DENSE_NHOM);
	  }
      }
    }

    libGAP_SET_FILT_LIST(set1, libGAP_FN_IS_SSORT);

    /* resize the result and copy back from the union                      */
    libGAP_GROW_PLIST(    set1, lenr );
    libGAP_SET_LEN_PLIST( set1, lenr );
    for ( i1 = 1;  i1 <= lenr;  i1++ ) {
        libGAP_SET_ELM_PLIST( set1, i1, libGAP_ELM_PLIST( TmpUnion, i1 ) );
        libGAP_CHANGED_BAG( set1 );
        libGAP_SET_ELM_PLIST( TmpUnion, i1, (libGAP_Obj)0 );
    }

    /* return void, this is a procedure                                    */
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncINTER_SET( <self>, <set1>, <set2> ) .  intersect one set with another
**
**  'FuncINTER_SET' implements the internal function 'IntersectSet'.
**
**  'IntersectSet( <set1>, <set2> )'
**
**  'IntersectSet' changes the set <set1> so that it becomes the intersection
**  of <set1> and <set2>.  The intersection is the set of those elements that
**  are  elements in both sets.   So 'IntersectSet' removes (see "RemoveSet")
**  all elements from <set1> that are not  in  <set2>.  <set2> may be a  list
**  that is not a proper set, in which case 'Set' is silently applied to it.
*/

static libGAP_UInt libGAP_InterSetInner1( libGAP_Obj set1, libGAP_Obj set2, libGAP_UInt len1, libGAP_UInt len2) 
{
  libGAP_UInt lenr, i1,i2;
  libGAP_Obj e1,e2;
  lenr = 0;
  i1 = 1;
  i2 = 1;

  /* now merge the two sets into the intersection                        */
  while ( i1 <= len1 && i2 <= len2 ) {
    e1 = libGAP_ELM_PLIST( set1, i1 );
    e2 = libGAP_ELM_PLIST( set2, i2 );
    if ( libGAP_EQ( e1, e2 ) ) {
      lenr++;
      libGAP_SET_ELM_PLIST( set1, lenr, e1 );
      i1++;  i2++;
    }
    else if ( libGAP_LT( e1, e2 ) ) {
      i1++;
    }
    else {
      i2++;
    }
  }
  return lenr;
}

/* set1 should be the smaller set. setr should be the one
   in which to put the results; */
static libGAP_UInt libGAP_InterSetInner2( libGAP_Obj set1, libGAP_Obj set2, libGAP_Obj setr, libGAP_UInt len1, libGAP_UInt len2) 
{
  libGAP_UInt i1,i2=1,bottom,top,middle,lenr=0,found;
  libGAP_Obj e1,e2;
  for( i1 = 1; i1 <= len1; i1++)
    {
      e1 = libGAP_ELM_PLIST( set1, i1 );
      bottom = i2;
      top = len2;
      found = 0;
      while (bottom <= top)
	{
	  middle = (bottom + top)/2;
	  e2 = libGAP_ELM_PLIST(set2,middle);
	  if (libGAP_LT(e1,e2))
	    top = middle-1;
	  else if (libGAP_EQ(e1,e2)) {
	    lenr++;
	    libGAP_SET_ELM_PLIST(setr,lenr,e1);
	    i2 = middle+1;
	    found = 1;
	    break;
	  }
	  else
	    bottom = middle+1;
	}
      if (!found)
	i2 = bottom;
    }
  return lenr;
}


libGAP_Obj libGAP_FuncINTER_SET (
    libGAP_Obj                 self,
    libGAP_Obj                 set1,
    libGAP_Obj                 set2 )
{
    libGAP_UInt                len1;           /* length  of left  set            */
    libGAP_UInt                len2;           /* length  of right set            */
    libGAP_UInt                lenr;           /* length  of result set           */

    /* check the arguments                                                 */
    while ( ! libGAP_IsSet(set1) || ! libGAP_IS_MUTABLE_OBJ(set1) ) {
        set1 = libGAP_ErrorReturnObj(
            "IntersectSet: <set1> must be a mutable proper set (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set1), 0L,
            "you can replace <set1> via 'return <set1>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(set2) ) {
        set2 = libGAP_ErrorReturnObj(
            "IntersectSet: <set2> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set2), 0L,
            "you can replace <set2> via 'return <set2>;'" );
    }
    if ( ! libGAP_IsSet(set2) )  set2 = libGAP_SetList(set2);

    /* get the logical lengths and the pointer                             */
    len1 = libGAP_LEN_PLIST( set1 );
    len2 = libGAP_LEN_PLIST( set2 );

    /* decide how to do the calculation and do it */
    if (len1 < len2) 
      {
	libGAP_UInt x = len2;
	libGAP_UInt ll = 0;
	while (x > 0)
	  {
	    ll++;
	    x >>= 1;
	  }
	if (len1*ll < len2)
	  lenr = libGAP_InterSetInner2(set1,set2,set1,len1,len2);
	else
	  lenr = libGAP_InterSetInner1(set1,set2,len1,len2);
      }
    else
      {
	libGAP_UInt x = len1;
	libGAP_UInt ll = 0;
	while (x > 0)
	  {
	    ll++;
	    x >>= 1;
	  }
	if (len2*ll < len1)
	  lenr = libGAP_InterSetInner2(set2,set1,set1,len2,len1);
	else
	  lenr = libGAP_InterSetInner1(set1,set2,len1,len2);
      }

    /* resize the result or clear the rest of the bag                      */
    libGAP_SET_LEN_PLIST( set1, lenr );
    libGAP_SHRINK_PLIST(  set1, lenr );

    /* fix up the type of the result                                       */
    if ( lenr == 0 ) {
      libGAP_CLEAR_FILTS_LIST(set1);
      libGAP_SET_FILT_LIST( set1, libGAP_FN_IS_EMPTY );
    }
    else if ( lenr == 1) {
      if (libGAP_TNUM_OBJ(libGAP_ELM_PLIST(set1,1)) <= libGAP_T_CYC)
	libGAP_RetypeBag(set1, libGAP_T_PLIST_CYC_SSORT);
      else
	libGAP_RetypeBag(set1, libGAP_T_PLIST_HOM_SSORT);
    }
    else
      {
	if ( libGAP_TNUM_OBJ(set2) >= libGAP_T_PLIST_CYC )
	  libGAP_RetypeBag(set1, libGAP_MUTABLE_TNUM( libGAP_TNUM_OBJ(set2)));
	else
	  {
	    libGAP_RESET_FILT_LIST(set1, libGAP_FN_IS_NHOMOG);
	    if ( libGAP_HAS_FILT_LIST( set2, libGAP_FN_IS_HOMOG )) {
	      libGAP_SET_FILT_LIST(set1, libGAP_FN_IS_HOMOG );
	      libGAP_SET_FILT_LIST(set1, libGAP_FN_IS_SSORT );
	    }
	  }
      }

    /* return void, this is a procedure                                    */
    return (libGAP_Obj)0;
}



/****************************************************************************
**
*F  FuncSUBTR_SET( <self>, <set1>, <set2> ) . . subtract one set from another
**
**  'FuncSUBTR_SET' implements the internal function 'SubstractSet'.
**
**  'SubstractSet( <set1>, <set2> )'
**
**  'SubstractSet' changes the  set <set1> so  that it becomes the difference
**  of <set1> and <set2>.  The difference is the set of the elements that are
**  in <set1> but not in <set2>.  So 'SubtractSet' removes  (see "RemoveSet")
**  all elements from <set1> that are in <set2>.   <set2> may  be a list that
**  is not a proper set, in which case 'Set' is silently applied to it.
*/

static libGAP_UInt libGAP_SubtrSetInner1( libGAP_Obj set1, libGAP_Obj set2, libGAP_UInt len1, libGAP_UInt len2) 
{
  libGAP_UInt lenr, i1,i2;
  libGAP_Obj e1,e2;
  lenr = 0;
  i1 = 1;
  i2 = 1;

  /* now run through the two sets to find the difference  */
  while ( i1 <= len1 && i2 <= len2 ) {
    e1 = libGAP_ELM_PLIST( set1, i1 );
    e2 = libGAP_ELM_PLIST( set2, i2 );
    if ( libGAP_EQ( e1, e2 ) ) {
      i1++;  i2++;
    }
    else if ( libGAP_LT( e1, e2 ) ) {
      lenr++;
      libGAP_SET_ELM_PLIST( set1, lenr, e1 );
      i1++;
    }
    else {
      i2++;
    }
  }
  while (i1 <= len1)
    {
      e1 = libGAP_ELM_PLIST( set1, i1 );
      lenr++;
      libGAP_SET_ELM_PLIST( set1, lenr, e1 );
      i1++;
    }
  return lenr;
}

/* set1 should be smaller. */
static libGAP_UInt libGAP_SubtrSetInner2( libGAP_Obj set1, libGAP_Obj set2, libGAP_UInt len1, libGAP_UInt len2) 
{
  libGAP_UInt i1,i2=1,bottom,top,middle,lenr=0, found;
  libGAP_Obj e1,e2;
  for( i1 = 1; i1 <= len1; i1++)
    {
      e1 = libGAP_ELM_PLIST( set1, i1 );
      bottom = i2;
      top = len2;
      found = 0;
      while (bottom <= top)
	{
	  middle = (bottom + top)/2;
	  e2 = libGAP_ELM_PLIST(set2,middle);
	  if (libGAP_LT(e1,e2))
	    top = middle-1;
	  else if (libGAP_EQ(e1,e2)) {
	    found = 1;
	    i2 = middle+1;
	    break;
	  }
	  else
	    bottom = middle+1;
	}
      if (!found)
	{
	  lenr++;
	  libGAP_SET_ELM_PLIST(set1,lenr,e1);
	  i2 = bottom;
	}
    }
  return lenr;
}

libGAP_Obj libGAP_FuncSUBTR_SET (
    libGAP_Obj                 self,
    libGAP_Obj                 set1,
    libGAP_Obj                 set2 )
{
    libGAP_UInt                len1;           /* length  of left  set            */
    libGAP_UInt                len2;           /* length  of right set            */
    libGAP_UInt                lenr;           /* length  of result set           */
    libGAP_UInt                x;            
    libGAP_UInt                ll;           

    /* check the arguments                                                 */
    while ( ! libGAP_IsSet(set1) || ! libGAP_IS_MUTABLE_OBJ(set1) ) {
        set1 = libGAP_ErrorReturnObj(
            "SubtractSet: <set1> must be a mutable proper set (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set1), 0L,
            "you can replace <set1> via 'return <set1>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(set2) ) {
        set2 = libGAP_ErrorReturnObj(
            "SubtractSet: <set2> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set2), 0L,
            "you can replace <set2> via 'return <set2>;'" );
    }
    if ( ! libGAP_IsSet(set2) )  set2 = libGAP_SetList(set2);

    /* get the logical lengths and the pointer                             */
    len1 = libGAP_LEN_PLIST( set1 );
    len2 = libGAP_LEN_PLIST( set2 );
    /* decide how to do the calculation and do it */
    x = len2;
    ll = 0;
    while (x > 0)
      {
	ll++;
	x >>= 1;
      }
    if (len1*ll < len2)
      lenr = libGAP_SubtrSetInner2(set1,set2,len1,len2);
    else
      lenr = libGAP_SubtrSetInner1(set1,set2,len1,len2);

    /* resize the result or clear the rest of the bag                      */
    libGAP_SET_LEN_PLIST( set1, lenr );
    libGAP_SHRINK_PLIST(  set1, lenr );

    /* fix up the type of the result                                       */
    if ( lenr == 0 ) {
        libGAP_CLEAR_FILTS_LIST(set1);
        libGAP_SET_FILT_LIST( set1, libGAP_FN_IS_EMPTY );
    }
    else if ( lenr == 1) {
      if (libGAP_TNUM_OBJ(libGAP_ELM_PLIST(set1,1)) <= libGAP_T_CYC)
	libGAP_RetypeBag(set1, libGAP_T_PLIST_CYC_SSORT);
      else
	libGAP_RetypeBag(set1, libGAP_T_PLIST_HOM_SSORT);
    }
    else
      libGAP_RESET_FILT_LIST(set1, libGAP_FN_IS_NHOMOG);

    /* return void, this is a procedure                                    */
    return (libGAP_Obj)0;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "LIST_SORTED_LIST", 1, "list",
      libGAP_FuncLIST_SORTED_LIST, "src/set.c:LIST_SORTED_LIST" },

    { "IS_EQUAL_SET", 2, "set1, set2",
      libGAP_FuncIS_EQUAL_SET, "src/set.c:IS_EQUAL_SET" },

    { "IS_SUBSET_SET", 2, "set1, set2",
      libGAP_FuncIS_SUBSET_SET, "src/set.c:IS_SUBSET_SET" },

    { "ADD_SET", 2, "set, val",
      libGAP_FuncADD_SET, "src/set.c:ADD_SET" },

    { "REM_SET", 2, "set, val",
      libGAP_FuncREM_SET, "src/set.c:REM_SET" },

    { "UNITE_SET", 2, "set1, set2",
      libGAP_FuncUNITE_SET, "src/set.c:UNITE_SET" },

    { "INTER_SET", 2, "set1, set2",
      libGAP_FuncINTER_SET, "src/set.c:INTER_SET" },

    { "SUBTR_SET", 2, "set1, set2",
      libGAP_FuncSUBTR_SET, "src/set.c:SUBTR_SET" },


    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* create the temporary union bag                                      */
  /*    InitGlobalBag( &TmpUnion, "src/set.c:TmpUnion" ); */

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* create the temporary union bag                                      */
  /*     TmpUnion = NEW_PLIST( T_PLIST, 1024 );
	 SET_LEN_PLIST( TmpUnion, 1024 ); */

    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoSet() . . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "set",                              /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoSet ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  set.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
