/*******************************************************************************
**
** A partial perm is of the form: 
**
** [image set, domain, codegree, entries of image list]
**
** An element of the internal rep of a partial perm in T_PPERM2 must be at
** most 65535 and be of UInt2. The <codegree> is just the degree of the inverse
** or equivalently the maximum element of the image.
** 
*******************************************************************************/

#include        "pperm.h"               /* same header applies */

#define libGAP_MAX(a,b)          (a<b?b:a)
#define libGAP_MIN(a,b)          (a<b?a:b)

#define libGAP_IMG_PPERM(f)      (*(libGAP_Obj*)(libGAP_ADDR_OBJ(f)))
#define libGAP_DOM_PPERM(f)      (*((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+1))

#define libGAP_NEW_PPERM2(deg)   libGAP_NewBag(libGAP_T_PPERM2, (deg+1)*sizeof(libGAP_UInt2)+2*sizeof(libGAP_Obj))
#define libGAP_CODEG_PPERM2(f)   (*(libGAP_UInt2*)((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+2))
#define libGAP_ADDR_PPERM2(f)    ((libGAP_UInt2*)((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+2)+1)
#define libGAP_DEG_PPERM2(f)  ((libGAP_UInt)(libGAP_SIZE_OBJ(f)-sizeof(libGAP_UInt2)-2*sizeof(libGAP_Obj))/sizeof(libGAP_UInt2))
#define libGAP_RANK_PPERM2(f) (libGAP_IMG_PPERM(f)==NULL?libGAP_INIT_PPERM2(f):libGAP_LEN_PLIST(libGAP_IMG_PPERM(f)))

#define libGAP_NEW_PPERM4(deg)   libGAP_NewBag(libGAP_T_PPERM4, (deg+1)*sizeof(libGAP_UInt4)+2*sizeof(libGAP_Obj))
#define libGAP_CODEG_PPERM4(f)   (*(libGAP_UInt4*)((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+2))
#define libGAP_ADDR_PPERM4(f)    ((libGAP_UInt4*)((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+2)+1)
#define libGAP_DEG_PPERM4(f)     ((libGAP_UInt)(libGAP_SIZE_OBJ(f)-sizeof(libGAP_UInt4)-2*sizeof(libGAP_Obj))/sizeof(libGAP_UInt4))
#define libGAP_RANK_PPERM4(f) (libGAP_IMG_PPERM(f)==NULL?libGAP_INIT_PPERM4(f):libGAP_LEN_PLIST(libGAP_IMG_PPERM(f)))

#define libGAP_IMAGEPP(i, ptf, deg) (i<=deg?ptf[i-1]:0)
#define libGAP_IS_PPERM(f)   (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2||libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4)
#define libGAP_RANK_PPERM(f) (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2?libGAP_RANK_PPERM2(f):libGAP_RANK_PPERM4(f))
#define libGAP_DEG_PPERM(f)  (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2?libGAP_DEG_PPERM2(f):libGAP_DEG_PPERM4(f))
#define libGAP_CODEG_PPERM(f)(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2?libGAP_CODEG_PPERM2(f):libGAP_CODEG_PPERM4(f))

libGAP_Obj   libGAP_EmptyPartialPerm;

/****************************************************************************
**
*V  TmpPPerm . . . . . . . handle of the buffer bag of the pperm package
**
**  'TmpPPerm' is the handle of a bag of type 'T_PPERM4', 
**  which is created at initialization time of this package.  Functions in this
**  package can use this bag for  whatever purpose they want.  They have to
**  make sure of course that it is large enough.
**
**  The buffer is *not* guaranteed to have any particular value, routines
**  that require a zero-initialization need to do this at the start.
*/

libGAP_Obj libGAP_TmpPPerm;

static inline void libGAP_ResizeTmpPPerm( libGAP_UInt len ){
  if(libGAP_SIZE_OBJ(libGAP_TmpPPerm)<len*sizeof(libGAP_UInt4))
    libGAP_ResizeBag(libGAP_TmpPPerm,len*sizeof(libGAP_UInt4));
}

/*******************************************************************************
** Static functions for partial perms
*******************************************************************************/

// find domain and img set (unsorted) return the rank 

static libGAP_UInt libGAP_INIT_PPERM2(libGAP_Obj f){ 
  libGAP_UInt    deg, rank, i;
  libGAP_UInt2   *ptf;
  libGAP_Obj     img, dom;
  
  deg=libGAP_DEG_PPERM2(f);
  
  if(deg==0){
    dom=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(dom, 0);
    libGAP_DOM_PPERM(f)=dom;
    libGAP_IMG_PPERM(f)=dom;
    libGAP_CHANGED_BAG(f);
    return deg;
  }

  dom=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT+libGAP_IMMUTABLE, deg);
  img=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, deg);

  /* renew the ptr in case of garbage collection */
  ptf=libGAP_ADDR_PPERM2(f); 
 
  rank=0;
  for(i=0;i<deg;i++){        
    if(ptf[i]!=0){
      rank++;
      libGAP_SET_ELM_PLIST(dom, rank, libGAP_INTOBJ_INT(i+1));
      libGAP_SET_ELM_PLIST(img, rank, libGAP_INTOBJ_INT(ptf[i]));
    }
  }
  
  if(rank==0){
    libGAP_RetypeBag(img, libGAP_T_PLIST_EMPTY);
    libGAP_RetypeBag(dom, libGAP_T_PLIST_EMPTY);
  }

  libGAP_SHRINK_PLIST(img, (libGAP_Int) rank);
  libGAP_SET_LEN_PLIST(img, (libGAP_Int) rank);
  libGAP_SHRINK_PLIST(dom, (libGAP_Int) rank);
  libGAP_SET_LEN_PLIST(dom, (libGAP_Int) rank);
  
  libGAP_DOM_PPERM(f)=dom;
  libGAP_IMG_PPERM(f)=img;
  libGAP_CHANGED_BAG(f);
  return rank;
}

static libGAP_UInt libGAP_INIT_PPERM4(libGAP_Obj f){ 
  libGAP_UInt    deg, rank, i;
  libGAP_UInt4   *ptf;
  libGAP_Obj     img, dom;

  deg=libGAP_DEG_PPERM4(f);
  
  if(deg==0){
    dom=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(dom, 0);
    libGAP_DOM_PPERM(f)=dom;
    libGAP_IMG_PPERM(f)=dom;
    libGAP_CHANGED_BAG(f);
    return deg;
  }
  
  dom=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT+libGAP_IMMUTABLE, deg);
  img=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_NSORT+libGAP_IMMUTABLE, deg);

  ptf=libGAP_ADDR_PPERM4(f); 
 
  rank=0;
  for(i=0;i<deg;i++){        
    if(ptf[i]!=0){
      rank++;
      libGAP_SET_ELM_PLIST(dom, rank, libGAP_INTOBJ_INT(i+1));
      libGAP_SET_ELM_PLIST(img, rank, libGAP_INTOBJ_INT(ptf[i]));
    }
  }

  if(rank==0){
    libGAP_RetypeBag(img, libGAP_T_PLIST_EMPTY);
    libGAP_RetypeBag(dom, libGAP_T_PLIST_EMPTY);
  }
  
  libGAP_SHRINK_PLIST(img, (libGAP_Int) rank);
  libGAP_SET_LEN_PLIST(img, (libGAP_Int) rank);
  libGAP_SHRINK_PLIST(dom, (libGAP_Int) rank);
  libGAP_SET_LEN_PLIST(dom, (libGAP_Int) rank);
  
  libGAP_DOM_PPERM(f)=dom;
  libGAP_IMG_PPERM(f)=img;
  libGAP_CHANGED_BAG(f);
  return rank;
}

static libGAP_Obj libGAP_SORT_PLIST_CYC(libGAP_Obj res){ 
  libGAP_Obj     tmp;       
  libGAP_UInt    h, i, k, len; 
   
  len=libGAP_LEN_PLIST(res);  
  if(len==0) return res;

  h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1; 
  while ( 0 < h ) { 
    for ( i = h+1; i <= len; i++ ) { 
      tmp = libGAP_ADDR_OBJ(res)[i];  k = i; 
      while ( h < k && ((libGAP_Int)tmp < (libGAP_Int)(libGAP_ADDR_OBJ(res)[k-h])) ) { 
        libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h]; 
        k -= h; 
      } 
      libGAP_ADDR_OBJ(res)[k] = tmp; 
    } 
    h = h / 3; 
  } 
  libGAP_RetypeBag(res, libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE);  
  libGAP_CHANGED_BAG(res);
  return res;
}

static libGAP_Obj libGAP_PreImagePPermInt (libGAP_Obj pt, libGAP_Obj f){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  libGAP_UInt    i, cpt, deg;

  cpt=libGAP_INT_INTOBJ(pt);
  
  if(cpt>(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2?libGAP_CODEG_PPERM2(f):libGAP_CODEG_PPERM4(f)))
    return libGAP_Fail;

  i=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    ptf2=libGAP_ADDR_PPERM2(f);
    deg=libGAP_DEG_PPERM2(f);
    while(ptf2[i]!=cpt&&i<deg) i++;
    if(ptf2[i]!=cpt) return libGAP_Fail; 
  } else {
    ptf4=libGAP_ADDR_PPERM4(f);
    deg=libGAP_DEG_PPERM4(f);
    while(ptf4[i]!=cpt&&i<deg) i++;
    if(ptf4[i]!=cpt) return libGAP_Fail; 
  }
  return libGAP_INTOBJ_INT(i+1);
}

/*******************************************************************************
** GAP functions for partial perms
*******************************************************************************/


libGAP_Obj libGAP_FuncEmptyPartialPerm( libGAP_Obj self ){
  return libGAP_EmptyPartialPerm;
}

/* method for creating a partial perm */
libGAP_Obj libGAP_FuncDensePartialPermNC( libGAP_Obj self, libGAP_Obj img ){ 
  libGAP_UInt    deg, i, j, codeg; 
  libGAP_UInt2*  ptf2;
  libGAP_UInt4*  ptf4;
  libGAP_Obj     f; 

  if(libGAP_LEN_LIST(img)==0) return libGAP_EmptyPartialPerm;

  //remove trailing 0s
  deg=libGAP_LEN_LIST(img); 
  while(deg > 0 && libGAP_INT_INTOBJ(libGAP_ELM_LIST(img, deg))==0) deg--;
  
  if(deg==0) return libGAP_EmptyPartialPerm;

  //find if we are PPERM2 or PPERM4
  codeg=0; i=deg;
  while(codeg<65536&&i>0){
    j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(img, i--));
    if(j>codeg) codeg=j;
  }
  if(codeg<65536){
    f=libGAP_NEW_PPERM2(deg);
    ptf2=libGAP_ADDR_PPERM2(f);
    for(i=0;i<deg;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(img, i+1));
      *ptf2++=(libGAP_UInt2) j;
    }
    libGAP_CODEG_PPERM2(f)=(libGAP_UInt2) codeg; //codeg is already known
  } else {
    f=libGAP_NEW_PPERM4(deg);
    ptf4=libGAP_ADDR_PPERM4(f);
    for(i=0;i<deg;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(img, i+1));
      if(j>codeg) codeg=j;
      *ptf4++=(libGAP_UInt4) j;
    }
    libGAP_CODEG_PPERM4(f)=(libGAP_UInt4) codeg;
  }
  return f; 
}

/* assumes that dom is a set and that img is duplicatefree */
libGAP_Obj libGAP_FuncSparsePartialPermNC( libGAP_Obj self, libGAP_Obj dom, libGAP_Obj img ){
  libGAP_UInt  rank, deg, i, j, codeg;
  libGAP_Obj   f;
  libGAP_UInt2 *ptf2;
  libGAP_UInt4 *ptf4;

  rank=libGAP_LEN_LIST(dom);
  deg=libGAP_INT_INTOBJ(libGAP_ELM_LIST(dom, rank));
  
  // find if we are PPERM2 or PPERM4
  codeg=0; i=rank;
  while(codeg<65536&&i>0){
    j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(img, i--));
    if(j>codeg) codeg=j;
  }
  
  // make sure we have plain lists
  if(!libGAP_IS_PLIST(dom)) libGAP_PLAIN_LIST(dom);
  if(!libGAP_IS_PLIST(img)) libGAP_PLAIN_LIST(img);

  // make dom and img immutable
  if(libGAP_IS_MUTABLE_OBJ(dom)) libGAP_RetypeBag(dom, libGAP_TNUM_OBJ(dom)+libGAP_IMMUTABLE);
  if(libGAP_IS_MUTABLE_OBJ(img)) libGAP_RetypeBag(img, libGAP_TNUM_OBJ(img)+libGAP_IMMUTABLE);
  
  // create the pperm
  if(codeg<65536){ 
    f=libGAP_NEW_PPERM2(deg);
    ptf2=libGAP_ADDR_PPERM2(f);
    for(i=1;i<=rank;i++){ 
      ptf2[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1]=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i));
    }
    libGAP_DOM_PPERM(f)=dom;
    libGAP_IMG_PPERM(f)=img;
    libGAP_CODEG_PPERM2(f)=codeg;
  } else {
    f=libGAP_NEW_PPERM4(deg);
    ptf4=libGAP_ADDR_PPERM4(f);
    for(i=1;i<=rank;i++){ 
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i));
      if(j>codeg) codeg=j;
      ptf4[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1]=j;
    }
    libGAP_DOM_PPERM(f)=dom;
    libGAP_IMG_PPERM(f)=img;
    libGAP_CODEG_PPERM4(f)=codeg;
  }
  libGAP_CHANGED_BAG(f);
  return f;
}

/* the degree of pperm is the maximum point where it is defined */
libGAP_Obj libGAP_FuncDegreeOfPartialPerm(libGAP_Obj self, libGAP_Obj f){
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){ 
    return libGAP_INTOBJ_INT(libGAP_DEG_PPERM2(f));
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    return libGAP_INTOBJ_INT(libGAP_DEG_PPERM4(f));
  }
  libGAP_ErrorQuit("usage: the argument should be a partial perm,", 0L, 0L);
  return libGAP_Fail;
}

/* the codegree of pperm is the maximum point in its image */
libGAP_Obj libGAP_FuncCoDegreeOfPartialPerm(libGAP_Obj self, libGAP_Obj f){
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){ 
    return libGAP_INTOBJ_INT(libGAP_CODEG_PPERM2(f));
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    return libGAP_INTOBJ_INT(libGAP_CODEG_PPERM4(f));
  }
  libGAP_ErrorQuit("usage: the argument should be a partial perm,", 0L, 0L);
  return libGAP_Fail;
}

/* the rank is the number of points where it is defined */
libGAP_Obj libGAP_FuncRankOfPartialPerm (libGAP_Obj self, libGAP_Obj f){ 
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    return libGAP_INTOBJ_INT(libGAP_RANK_PPERM2(f));
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    return libGAP_INTOBJ_INT(libGAP_RANK_PPERM4(f));
  }
  libGAP_ErrorQuit("usage: the argument should be a partial perm,", 0L, 0L);
  return libGAP_Fail;
}

/* domain of a partial perm */
libGAP_Obj libGAP_FuncDOMAIN_PPERM(libGAP_Obj self, libGAP_Obj f){ 
  if(libGAP_DOM_PPERM(f)==NULL){
    if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
      libGAP_INIT_PPERM2(f);
    } else {
      libGAP_INIT_PPERM4(f);
    }
  }
  return libGAP_DOM_PPERM(f);
} 

/* image list of pperm */
libGAP_Obj libGAP_FuncIMAGE_PPERM(libGAP_Obj self, libGAP_Obj f ){ 
  libGAP_UInt2*    ptf2;
  libGAP_UInt4*    ptf4;
  libGAP_UInt      i, rank;
  libGAP_Obj       out, dom;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    if(libGAP_IMG_PPERM(f)==NULL){
      libGAP_INIT_PPERM2(f);
      return libGAP_IMG_PPERM(f);
    } else if(!libGAP_IS_SSORT_LIST(libGAP_IMG_PPERM(f))){
      return libGAP_IMG_PPERM(f);
    }
    rank=libGAP_RANK_PPERM2(f);
    if(rank==0){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
      libGAP_SET_LEN_PLIST(out, 0);
      return out;
    }
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_NSORT+libGAP_IMMUTABLE, rank);
    libGAP_SET_LEN_PLIST(out, rank);
    ptf2=libGAP_ADDR_PPERM2(f);
    dom=libGAP_DOM_PPERM(f);
    for(i=1;i<=rank;i++){ 
      libGAP_SET_ELM_PLIST(out,i,libGAP_INTOBJ_INT(ptf2[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1]));
    }
  } else {
    if(libGAP_IMG_PPERM(f)==NULL){
      libGAP_INIT_PPERM4(f);
      return libGAP_IMG_PPERM(f);
    } else if(!libGAP_IS_SSORT_LIST(libGAP_IMG_PPERM(f))){
      return libGAP_IMG_PPERM(f);
    }
    rank=libGAP_RANK_PPERM4(f);
    if(rank==0){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
      libGAP_SET_LEN_PLIST(out, 0);
      return out;
    }
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_NSORT+libGAP_IMMUTABLE, rank);
    libGAP_SET_LEN_PLIST(out, rank);
    ptf4=libGAP_ADDR_PPERM4(f);
    dom=libGAP_DOM_PPERM(f);
    for(i=1;i<=rank;i++){ 
      libGAP_SET_ELM_PLIST(out,i,libGAP_INTOBJ_INT(ptf4[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1]));
    }
  }
  return out;
} 

/* image set of partial perm */
libGAP_Obj libGAP_FuncIMAGE_SET_PPERM (libGAP_Obj self, libGAP_Obj f){ 
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    if(libGAP_IMG_PPERM(f)==NULL){
      libGAP_INIT_PPERM2(f);
      return libGAP_SORT_PLIST_CYC(libGAP_IMG_PPERM(f));
    } else if(!libGAP_IS_SSORT_LIST(libGAP_IMG_PPERM(f))){
      return libGAP_SORT_PLIST_CYC(libGAP_IMG_PPERM(f));
    }
    return libGAP_IMG_PPERM(f);  
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
   if(libGAP_IMG_PPERM(f)==NULL){
      libGAP_INIT_PPERM4(f);
      return libGAP_SORT_PLIST_CYC(libGAP_IMG_PPERM(f));
    } else if(!libGAP_IS_SSORT_LIST(libGAP_IMG_PPERM(f))){
      return libGAP_SORT_PLIST_CYC(libGAP_IMG_PPERM(f));
    }
    return libGAP_IMG_PPERM(f);
  } else {
    libGAP_ErrorQuit("usage: the argument must be a partial perm,", 0L, 0L);
  }
  return libGAP_Fail;
} 

/* preimage under a partial perm */
libGAP_Obj libGAP_FuncPREIMAGE_PPERM_INT (libGAP_Obj self, libGAP_Obj f, libGAP_Obj pt){
  return libGAP_PreImagePPermInt(pt, f);
}

// the least m, r such that f^m=f^m+r
libGAP_Obj libGAP_FuncINDEX_PERIOD_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, len, j, pow, gcd, rank, k, deg, n; 
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptf4;
  libGAP_Obj     dom, img, ord, out;
  
  pow=0;  ord=libGAP_INTOBJ_INT(1);  n=libGAP_MAX(libGAP_DEG_PPERM(f), libGAP_CODEG_PPERM(f));
  
  libGAP_ResizeTmpPPerm(n);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<n;i++) ptseen[i]=0; 

  rank=libGAP_RANK_PPERM(f);   img=libGAP_IMG_PPERM(f);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  
  //find img(f)
  for(i=1;i<=rank;i++) ptseen[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1]=1;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    dom=libGAP_DOM_PPERM(f);
    
    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==0){
        ptseen[j]=2;
        len=1;
        for(k=ptf2[j];(k<=deg&&ptf2[k-1]!=0);k=ptf2[k-1]){ 
          len++; ptseen[k-1]=2;
        }
        ptseen[k-1]=2;
        if(len>pow) pow=len;
      }
    }
    
    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==1){
        len=1;
        for(k=ptf2[j];k!=j+1;k=ptf2[k-1]){ 
          len++; ptseen[k-1]=0;
        }
        gcd=len;  j=libGAP_INT_INTOBJ(libGAP_ModInt(ord,libGAP_INTOBJ_INT(len)));
        while(j!=0){
          k=j;  j=gcd%j;  gcd=k;
        }
        ord=libGAP_ProdInt(ord,libGAP_INTOBJ_INT(len/gcd));
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    dom=libGAP_DOM_PPERM(f);

    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==0){
        ptseen[j]=2;
        len=1;
        for(k=ptf4[j];(k<=deg&&ptf4[k-1]!=0);k=ptf4[k-1]){ 
          len++; ptseen[k-1]=2;
        }
        ptseen[k-1]=2;
        if(len>pow) pow=len;
      }
    }

    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==1){
        len=1;
        for(k=ptf4[j];k!=j+1;k=ptf4[k-1]){ 
          len++; ptseen[k-1]=0;
        }
        gcd=len;  j=libGAP_INT_INTOBJ(libGAP_ModInt(ord,libGAP_INTOBJ_INT(len)));
        while(j!=0){
          k=j;  j=gcd%j;  gcd=k;
        }
        ord=libGAP_ProdInt(ord,libGAP_INTOBJ_INT(len/gcd));
      }
    }
  }
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 2);
  libGAP_SET_LEN_PLIST(out, 2);
  libGAP_SET_ELM_PLIST(out, 1, libGAP_INTOBJ_INT(pow+1));
  libGAP_SET_ELM_PLIST(out, 2, ord);
  return out;
}

// the least power of <f> which is an idempotent
libGAP_Obj libGAP_FuncSMALLEST_IDEM_POW_PPERM( libGAP_Obj self, libGAP_Obj f ){
  libGAP_Obj x, ind, per, pow;

  x=libGAP_FuncINDEX_PERIOD_PPERM(self, f);
  ind=libGAP_ELM_PLIST(x, 1);
  per=libGAP_ELM_PLIST(x, 2);
  pow=per;
  while(libGAP_LtInt(pow, ind)) pow=libGAP_SumInt(pow, per);
  return pow;
}

/* returns the least list <out> such that for all <i> in [1..degree(f)]
 * there exists <j> in <out> and a pos int <k> such that <j^(f^k)=i>. */
libGAP_Obj libGAP_FuncCOMPONENT_REPS_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, j, rank, k, deg, nr, n; 
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptf4;
  libGAP_Obj     dom, img, out;
 
  n=libGAP_MAX(libGAP_DEG_PPERM(f), libGAP_CODEG_PPERM(f));
  libGAP_ResizeTmpPPerm(n);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<n;i++) ptseen[i]=0; 
 
  rank=libGAP_RANK_PPERM(f);   img=libGAP_IMG_PPERM(f);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  
  //find img(f)
  for(i=1;i<=rank;i++) ptseen[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1]=1;
  
  deg=libGAP_DEG_PPERM(f);   nr=0;
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, deg);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){  
    dom=libGAP_DOM_PPERM(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    
    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(ptseen[j-1]==0){
        for(k=j;(k<=deg&&ptf2[k-1]!=0);k=ptf2[k-1]) ptseen[k-1]=2;
        ptseen[k-1]=2;  
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_ELM_PLIST(dom, i));
      }
    }
    
    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==1){
        ptseen[j]=0;
        for(k=ptf2[j];k!=j+1;k=ptf2[k-1]) ptseen[k-1]=0;
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_ELM_PLIST(dom, i));
      }
    }
  } else {
    dom=libGAP_DOM_PPERM(f);
    ptf4=libGAP_ADDR_PPERM4(f);      

    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(ptseen[j-1]==0){
        for(k=j;(k<=deg&&ptf4[k-1]!=0);k=ptf4[k-1]) ptseen[k-1]=2;
        ptseen[k-1]=2;  
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_ELM_PLIST(dom, i));
      }
    }
    
    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==1){
        ptseen[j]=0;
        for(k=ptf4[j];k!=j+1;k=ptf4[k-1]) ptseen[k-1]=0;
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_ELM_PLIST(dom, i));
      }
    }
  }
  
  libGAP_SHRINK_PLIST(out, (libGAP_Int) nr);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) nr);
  return out;
}

/* the number of components of a partial perm (as a functional digraph) */
libGAP_Obj libGAP_FuncNR_COMPONENTS_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, j, n, rank, k, deg, nr; 
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptf4;
  libGAP_Obj     dom, img;
 
  n=libGAP_MAX(libGAP_DEG_PPERM(f), libGAP_CODEG_PPERM(f));
  libGAP_ResizeTmpPPerm(n);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<n;i++) ptseen[i]=0; 
 
  rank=libGAP_RANK_PPERM(f);   img=libGAP_IMG_PPERM(f);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  
  //find img(f)
  for(i=1;i<=rank;i++) ptseen[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1]=1;
  
  nr=0;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    dom=libGAP_DOM_PPERM(f);

    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(ptseen[j-1]==0){
        nr++;
        for(k=j;(k<=deg&&ptf2[k-1]!=0);k=ptf2[k-1]) ptseen[k-1]=2;
        ptseen[k-1]=2;  //JDM really required?
      }
    }
    
    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==1){
        nr++;
        ptseen[j]=0;
        for(k=ptf2[j];k!=j+1;k=ptf2[k-1]) ptseen[k-1]=0;
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    dom=libGAP_DOM_PPERM(f);

    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(ptseen[j-1]==0){
        nr++;
        for(k=j;(k<=deg&&ptf4[k-1]!=0);k=ptf4[k-1]) ptseen[k-1]=2;
        ptseen[k-1]=2;  //REALLY REQUIRED?? JDM
      }
    }
    
    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptseen[j]==1){
        nr++;
        ptseen[j]=0;//REALLY REQUIRED??? JDM
        for(k=ptf4[j];k!=j+1;k=ptf4[k-1]) ptseen[k-1]=0;
      }
    }
  }
  return libGAP_INTOBJ_INT(nr);
}

/* the components of a partial perm (as a functional digraph) */
libGAP_Obj libGAP_FuncCOMPONENTS_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, j, n, rank, k, deg, nr, len; 
  libGAP_UInt4   *ptseen;
  libGAP_Obj     dom, img, out;

  // init the buffer
  n=libGAP_MAX(libGAP_DEG_PPERM(f), libGAP_CODEG_PPERM(f));
  libGAP_ResizeTmpPPerm(n);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<n;i++) ptseen[i]=0; 
 
  //find img(f)
  rank=libGAP_RANK_PPERM(f);   img=libGAP_IMG_PPERM(f);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=1;i<=rank;i++) ptseen[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1]=1;
  
  nr=0;
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, rank);
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){  
    deg=libGAP_DEG_PPERM2(f);
    dom=libGAP_DOM_PPERM(f);
    
    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[j-1]==0){
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 30));
        libGAP_CHANGED_BAG(out);
        len=0; k=j;
        do{
          libGAP_AssPlist(libGAP_ELM_PLIST(out, nr), ++len, libGAP_INTOBJ_INT(k));
          ((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[k-1]=2;
          k=libGAP_IMAGEPP(k, libGAP_ADDR_PPERM2(f), deg);
        }while(k!=0);
        libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(out, nr), len);
        libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, nr), (libGAP_Int) len);
        libGAP_CHANGED_BAG(out);
      }
    }
    
    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[j-1]==1){
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 30));
        libGAP_CHANGED_BAG(out);
        len=0; k=j;
        do{
          libGAP_AssPlist(libGAP_ELM_PLIST(out, nr), ++len, libGAP_INTOBJ_INT(k));
          ((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[k-1]=0; 
          k=libGAP_ADDR_PPERM2(f)[k-1];
        }while(k!=j);
        libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(out, nr), len);
        libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, nr), (libGAP_Int) len);
        libGAP_CHANGED_BAG(out);
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    dom=libGAP_DOM_PPERM(f);

    //find chains
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[j-1]==0){
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 30));
        libGAP_CHANGED_BAG(out);
        len=0; k=j;
        do{
          libGAP_AssPlist(libGAP_ELM_PLIST(out, nr), ++len, libGAP_INTOBJ_INT(k));
          ((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[k-1]=2;
          k=libGAP_IMAGEPP(k, libGAP_ADDR_PPERM4(f), deg);
        }while(k!=0);
        libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(out, nr), len);
        libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, nr), (libGAP_Int) len);
        libGAP_CHANGED_BAG(out);
      }
    }
    
    //find cycles 
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
      if(((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[j-1]==1){
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 30));
        libGAP_CHANGED_BAG(out);
        len=0; k=j;
        do{
          libGAP_AssPlist(libGAP_ELM_PLIST(out, nr), ++len, libGAP_INTOBJ_INT(k));
          ((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm)))[k-1]=0; 
          k=libGAP_ADDR_PPERM4(f)[k-1];
        }while(k!=j);
        libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(out, nr), len);
        libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, nr), (libGAP_Int) len);
        libGAP_CHANGED_BAG(out);
      }
    }
  }
  
  libGAP_SHRINK_PLIST(out, (libGAP_Int) nr);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) nr);
  return out;
}

// the points that can be obtained from <pt> by successively applying <f>. 
libGAP_Obj libGAP_FuncCOMPONENT_PPERM_INT(libGAP_Obj self, libGAP_Obj f, libGAP_Obj pt){
  libGAP_UInt    i, j, deg, len; 
  libGAP_Obj     out;

  i=libGAP_INT_INTOBJ(pt);
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    
    if(i>deg||(libGAP_ADDR_PPERM2(f))[i-1]==0){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
      libGAP_SET_LEN_PLIST(out, 0);
      return out;
    }
    
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 30);
    len=0; j=i;

    do{ libGAP_AssPlist(out, ++len, libGAP_INTOBJ_INT(j)); j=libGAP_IMAGEPP(j, libGAP_ADDR_PPERM2(f), deg);
    }while(j!=0&&j!=i);
  } else {
    deg=libGAP_DEG_PPERM4(f);
    
    if(i>deg||(libGAP_ADDR_PPERM4(f))[i-1]==0){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
      libGAP_SET_LEN_PLIST(out, 0);
      return out;
    }
    
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 30);
    len=0; j=i;

    do{ libGAP_AssPlist(out, ++len, libGAP_INTOBJ_INT(j)); j=libGAP_IMAGEPP(j, libGAP_ADDR_PPERM4(f), deg);
    }while(j!=0&&j!=i);

  }
  libGAP_SHRINK_PLIST(out, (libGAP_Int) len);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) len);
  return out;
}

//the fixed points of a partial perm
libGAP_Obj libGAP_FuncFIXED_PTS_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    len, i, j, deg, rank;
  libGAP_Obj     out, dom;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  
  len=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg);
      ptf2=libGAP_ADDR_PPERM2(f);
      for(i=0;i<deg;i++){
        if(ptf2[i]==i+1){ 
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
        }
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, rank);
      ptf2=libGAP_ADDR_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
        if(ptf2[j-1]==j){
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(j));
        }
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg);
      ptf4=libGAP_ADDR_PPERM4(f);
      for(i=0;i<deg;i++){
        if(ptf4[i]==i+1){ 
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
        }
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f); 
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, rank);
      ptf4=libGAP_ADDR_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
        if(ptf4[j-1]==j){
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(j));
        }
      }
    }
  }
  if(len==0) libGAP_RetypeBag(out, libGAP_T_PLIST_EMPTY);
  
  libGAP_SHRINK_PLIST(out, len);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) len);
  
  return out;
}

libGAP_Obj libGAP_FuncNR_FIXED_PTS_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    nr, i, j, deg, rank;
  libGAP_Obj     dom;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  
  nr=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf2[i]==i+1) nr++;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf2[j]==j+1) nr++;
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf4[i]==i+1) nr++;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf4[j]==j+1) nr++;
      }
    }
  }
  return libGAP_INTOBJ_INT(nr);
}

//the moved points of a partial perm
libGAP_Obj libGAP_FuncMOVED_PTS_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    len, i, j, deg, rank;
  libGAP_Obj     out, dom;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  
  len=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg);
      ptf2=libGAP_ADDR_PPERM2(f);
      for(i=0;i<deg;i++){
        if(ptf2[i]!=0&&ptf2[i]!=i+1){ 
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
        }
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, rank);
      ptf2=libGAP_ADDR_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf2[j]!=j+1){
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(j+1));
        }
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg);
      ptf4=libGAP_ADDR_PPERM4(f);
      for(i=0;i<deg;i++){
        if(ptf4[i]!=0&&ptf4[i]!=i+1){ 
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
        }
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, rank);
      ptf4=libGAP_ADDR_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf4[j]!=j+1){
          libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(j+1));
        }
      }
    }
  }
  if(len==0) libGAP_RetypeBag(out, libGAP_T_PLIST_EMPTY);
  libGAP_SHRINK_PLIST(out, len);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) len);
  return out;
}

libGAP_Obj libGAP_FuncNR_MOVED_PTS_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    nr, i, j, deg, rank;
  libGAP_Obj     dom;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  
  nr=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf2[i]!=0&&ptf2[i]!=i+1) nr++;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf2[j]!=j+1) nr++;
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf4[i]!=0&&ptf4[i]!=i+1) nr++;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf4[j]!=j+1) nr++;
      }
    }
  }
  return libGAP_INTOBJ_INT(nr);
}

libGAP_Obj libGAP_FuncLARGEST_MOVED_PT_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, j, deg;
  libGAP_Obj     dom;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=deg;i>0;i--){
        if(ptf2[i-1]!=0&&ptf2[i-1]!=i) return libGAP_INTOBJ_INT(i);
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      for(i=libGAP_RANK_PPERM2(f);i>=1;i--){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf2[j]!=j+1) return libGAP_INTOBJ_INT(j+1);
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=deg;i>0;i--){
        if(ptf4[i-1]!=0&&ptf4[i-1]!=i) return libGAP_INTOBJ_INT(i);
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      for(i=libGAP_RANK_PPERM4(f);i>=1;i--){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf4[j]!=j+1) return libGAP_INTOBJ_INT(j+1);
      }
    }
  }
  return libGAP_INTOBJ_INT(0);
}

libGAP_Obj libGAP_FuncSMALLEST_MOVED_PT_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, j, deg, rank;
  libGAP_Obj     dom;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++){
        if(ptf2[i]!=0&&ptf2[i]!=i+1) return libGAP_INTOBJ_INT(i+1);
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf2[j]!=j+1) return libGAP_INTOBJ_INT(j+1);
      }
    }
  } else {
    deg=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++){
        if(ptf4[i]!=0&&ptf4[i]!=i+1) return libGAP_INTOBJ_INT(i+1);
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf4[j]!=j+1) return libGAP_INTOBJ_INT(j+1);
      }
    }
  }
  return libGAP_Fail;
}

// convert a T_PPERM4 with codeg<65536 to a T_PPERM2
libGAP_Obj libGAP_FuncTRIM_PPERM (libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    deg, i;
  libGAP_UInt4   *ptf;

  if(libGAP_TNUM_OBJ(f)!=libGAP_T_PPERM4||libGAP_CODEG_PPERM4(f)>65535) return f;

  ptf=libGAP_ADDR_PPERM4(f)-1;
  deg=libGAP_DEG_PPERM4(f);
  for(i=0;i<deg+1;i++) ((libGAP_UInt2*)ptf)[i]=(libGAP_UInt2)ptf[i]; 

  libGAP_RetypeBag(f, libGAP_T_PPERM2);
  libGAP_ResizeBag(f, (deg+1)*sizeof(libGAP_UInt2)+2*sizeof(libGAP_Obj));
  return (libGAP_Obj)0;
}

libGAP_Obj libGAP_FuncHASH_FUNC_FOR_PPERM(libGAP_Obj self, libGAP_Obj f, libGAP_Obj data){
  libGAP_UInt codeg;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    codeg=libGAP_CODEG_PPERM4(f);
    if(codeg<65536){
      libGAP_FuncTRIM_PPERM(self, f);
    } else {
      return libGAP_INTOBJ_INT((libGAP_HASHKEY_BAG_NC(f, (libGAP_UInt4) 255, 
              2*sizeof(libGAP_Obj)+sizeof(libGAP_UInt4), (int) 4*libGAP_DEG_PPERM4(f))
              % (libGAP_INT_INTOBJ(data)))+1); 
    }
  }
  return libGAP_INTOBJ_INT((libGAP_HASHKEY_BAG_NC(f, (libGAP_UInt4) 255, 
              2*sizeof(libGAP_Obj)+sizeof(libGAP_UInt2), (int) 2*libGAP_DEG_PPERM2(f))
              % (libGAP_INT_INTOBJ(data)))+1); 
}

// test if a partial perm is an idempotent
libGAP_Obj libGAP_FuncIS_IDEM_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2*  ptf2;
  libGAP_UInt4*  ptf4;
  libGAP_UInt    deg, i, j, rank;
  libGAP_Obj     dom;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    ptf2=libGAP_ADDR_PPERM2(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      deg=libGAP_DEG_PPERM2(f);
      for(i=0;i<deg;i++){
        if(ptf2[i]!=0&&ptf2[i]!=i+1) return libGAP_False;
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf2[j]!=0&&ptf2[j]!=j+1) return libGAP_False;
      }
    }
  } else {
    ptf4=libGAP_ADDR_PPERM4(f);
    if(libGAP_DOM_PPERM(f)==NULL){
      deg=libGAP_DEG_PPERM4(f);
      for(i=0;i<deg;i++){
        if(ptf4[i]!=0&&ptf4[i]!=i+1) return libGAP_False;
      }
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf4[j]!=0&&ptf4[j]!=j+1) return libGAP_False;
      }
    }
  }
  return libGAP_True;
}

/* an idempotent partial perm <e> with ker(e)=ker(f) */
libGAP_Obj libGAP_FuncLEFT_ONE_PPERM( libGAP_Obj self, libGAP_Obj f){
  libGAP_Obj     dom, libGAP_g;
  libGAP_UInt    deg, i, j, rank;
  libGAP_UInt2   *ptg2;
  libGAP_UInt4   *ptg4;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    rank=libGAP_RANK_PPERM2(f);
    dom=libGAP_DOM_PPERM(f);
    deg=libGAP_DEG_PPERM2(f);
  } else {
    rank=libGAP_RANK_PPERM4(f);
    dom=libGAP_DOM_PPERM(f);
    deg=libGAP_DEG_PPERM4(f);
  }

  if(deg<65536){
    libGAP_g=libGAP_NEW_PPERM2(deg);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptg2[j]=j+1;
    }
    libGAP_CODEG_PPERM2(libGAP_g)=deg;
    libGAP_DOM_PPERM(libGAP_g)=dom;
    libGAP_IMG_PPERM(libGAP_g)=dom;
  } else {
    libGAP_g=libGAP_NEW_PPERM4(deg);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptg4[j]=j+1;
    }
    libGAP_CODEG_PPERM4(libGAP_g)=deg;
    libGAP_DOM_PPERM(libGAP_g)=dom;
    libGAP_IMG_PPERM(libGAP_g)=dom;
  }
  libGAP_CHANGED_BAG(libGAP_g);
  return libGAP_g;
}

// an idempotent partial perm <e> with im(e)=im(f) 
libGAP_Obj libGAP_FuncRIGHT_ONE_PPERM( libGAP_Obj self, libGAP_Obj f){
  libGAP_Obj     libGAP_g, img;
  libGAP_UInt    i, j, codeg, rank;
  libGAP_UInt2   *ptg2;
  libGAP_UInt4   *ptg4;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    codeg=libGAP_CODEG_PPERM2(f);
    rank=libGAP_RANK_PPERM2(f);
    img=libGAP_IMG_PPERM(f);
  }else{
    codeg=libGAP_CODEG_PPERM4(f);
    rank=libGAP_RANK_PPERM4(f);
    img=libGAP_IMG_PPERM(f);
  }

  if(codeg<65536){ 
    libGAP_g=libGAP_NEW_PPERM2(codeg);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1;
      ptg2[j]=j+1;
    }
    if(libGAP_IS_SSORT_LIST(img)){
      libGAP_DOM_PPERM(libGAP_g)=img;
      libGAP_IMG_PPERM(libGAP_g)=img;
    }
    libGAP_CODEG_PPERM2(libGAP_g)=codeg;
  } else {
    libGAP_g=libGAP_NEW_PPERM4(codeg);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1;
      ptg4[j]=j+1;
    }
    if(libGAP_IS_SSORT_LIST(img)){
      libGAP_DOM_PPERM(libGAP_g)=img;
      libGAP_IMG_PPERM(libGAP_g)=img;
    }
    libGAP_CODEG_PPERM4(libGAP_g)=codeg;
  }
  libGAP_CHANGED_BAG(libGAP_g);
  return libGAP_g;
}

// f<=g if and only if f is a restriction of g
libGAP_Obj libGAP_FuncNaturalLeqPartialPerm(libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    def, deg, i, j, rank;
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *ptf4, *ptg4;
  libGAP_Obj     dom;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    def=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    if(def==0) return libGAP_True;
    
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
      deg=libGAP_DEG_PPERM2(libGAP_g);
      ptg2=libGAP_ADDR_PPERM2(libGAP_g);
      if(libGAP_DOM_PPERM(f)==NULL){
        for(i=0;i<def;i++){
          if(ptf2[i]!=0&&ptf2[i]!=libGAP_IMAGEPP(i+1, ptg2, deg)) return libGAP_False;
        }
      } else {
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM2(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
          if(ptf2[j-1]!=libGAP_IMAGEPP(j, ptg2, deg)) return libGAP_False;
        }
      }
    } else if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM4){
      deg=libGAP_DEG_PPERM4(libGAP_g);
      ptg4=libGAP_ADDR_PPERM4(libGAP_g);
      if(libGAP_DOM_PPERM(f)==NULL){
        for(i=0;i<def;i++){
          if(ptf2[i]!=0&&ptf2[i]!=libGAP_IMAGEPP(i+1, ptg4, deg)) return libGAP_False;
        }
      } else {
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM2(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
          if(ptf2[j-1]!=libGAP_IMAGEPP(j, ptg4, deg)) return libGAP_False;
        }
      }
    } else {
      libGAP_ErrorQuit("usage: the arguments must be partial perms,", 0L, 0L);
    }
  }else if (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    def=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    if(def==0) return libGAP_True;
    
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
      deg=libGAP_DEG_PPERM2(libGAP_g);
      ptg2=libGAP_ADDR_PPERM2(libGAP_g);
      if(libGAP_DOM_PPERM(f)==NULL){
        for(i=0;i<def;i++){
          if(ptf4[i]!=0&&ptf4[i]!=libGAP_IMAGEPP(i+1, ptg2, deg)) return libGAP_False;
        }
      } else {
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM4(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
          if(ptf4[j-1]!=libGAP_IMAGEPP(j, ptg2, deg)) return libGAP_False;
        }
      }
    } else if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM4){
      deg=libGAP_DEG_PPERM4(libGAP_g);
      ptg4=libGAP_ADDR_PPERM4(libGAP_g);
      if(libGAP_DOM_PPERM(f)==NULL){
        for(i=0;i<def;i++){
          if(ptf4[i]!=0&&ptf4[i]!=libGAP_IMAGEPP(i+1, ptg4, deg)) return libGAP_False;
        }
      } else {
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM4(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i));
          if(ptf4[j-1]!=libGAP_IMAGEPP(j, ptg4, deg)) return libGAP_False;
        }
      }
    } else {
      libGAP_ErrorQuit("usage: the arguments must be partial perms,", 0L, 0L);
    }
  } else {
    libGAP_ErrorQuit("usage: the arguments must be partial perms,", 0L, 0L);
  }
  return libGAP_True;
}

// could add use of rank to improve things here. JDM
libGAP_Obj libGAP_FuncJOIN_IDEM_PPERMS(libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt  def, deg, dej, i; 
  libGAP_Obj   join;
  libGAP_UInt2 *ptjoin2, *ptf2, *ptg2;
  libGAP_UInt4 *ptjoin4, *ptf4, *ptg4;
  
  if(libGAP_EQ(f,libGAP_g)) return f;
  
  def=libGAP_DEG_PPERM(f);
  deg=libGAP_DEG_PPERM(libGAP_g);
  dej=libGAP_MAX(def, deg);
  if(dej<65536){
    join=libGAP_NEW_PPERM2(dej);
    libGAP_CODEG_PPERM2(join)=dej;
    ptjoin2=libGAP_ADDR_PPERM2(join);
    ptf2=libGAP_ADDR_PPERM2(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    if(def<deg){
      for(i=0;i<def;i++){
        if(ptf2[i]!=0){ 
          ptjoin2[i]=ptf2[i];
        } else if(ptg2[i]!=0) {
          ptjoin2[i]=ptg2[i];
        }
      }
      for(;i<deg;i++){ if(ptg2[i]!=0) ptjoin2[i]=ptg2[i]; }
    } else {
      for(i=0;i<deg;i++){
        if(ptg2[i]!=0){ 
          ptjoin2[i]=ptg2[i];
        } else if(ptf2[i]!=0) {
          ptjoin2[i]=ptf2[i];
        }
      }
      for(;i<def;i++){ if(ptf2[i]!=0) ptjoin2[i]=ptf2[i]; }
    }
  } else if (def>=65536&&deg>=65536){ //3 more cases required
    join=libGAP_NEW_PPERM4(dej);
    libGAP_CODEG_PPERM4(join)=dej;
    ptjoin4=libGAP_ADDR_PPERM4(join);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);
    if(def<deg){
      for(i=0;i<def;i++){
        if(ptf4[i]!=0){ 
          ptjoin4[i]=ptf4[i];
        } else if(ptg4[i]!=0) {
          ptjoin4[i]=ptg4[i];
        }
      }
      for(;i<deg;i++){ if(ptg4[i]!=0) ptjoin4[i]=ptg4[i]; }
    } else {
      for(i=0;i<deg;i++){
        if(ptg4[i]!=0){ 
          ptjoin4[i]=ptg4[i];
        } else if(ptf4[i]!=0) {
          ptjoin4[i]=ptf4[i];
        }
      }
      for(;i<def;i++){ if(ptf4[i]!=0) ptjoin4[i]=ptf4[i]; }
    }
  } else if (def>deg) {// def>=65536>deg
    join=libGAP_NEW_PPERM4(dej);
    libGAP_CODEG_PPERM4(join)=dej;
    ptjoin4=libGAP_ADDR_PPERM4(join);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    for(i=0;i<deg;i++){
      if(ptg2[i]!=0){ 
        ptjoin4[i]=ptg2[i];
      } else if(ptf4[i]!=0) {
        ptjoin4[i]=ptf4[i];
      }
    }
    for(;i<def;i++){ if(ptf4[i]!=0) ptjoin4[i]=ptf4[i]; }
  } else {
    return libGAP_FuncJOIN_IDEM_PPERMS(self, libGAP_g, f);
  }
  return join;
}

// the union of f and g where this defines an injective function
libGAP_Obj libGAP_FuncJOIN_PPERMS(libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, i, j, degf, degg, codeg, rank;
  libGAP_UInt2   *ptf2, *ptg2, *ptjoin2;
  libGAP_UInt4   *ptf4, *ptg4, *ptjoin4, *ptseen;
  libGAP_Obj     join, dom;
 
  if(libGAP_EQ(f,libGAP_g)) return f; 

  //init the buffer
  codeg=libGAP_MAX(libGAP_CODEG_PPERM(f), libGAP_CODEG_PPERM(libGAP_g));
  libGAP_ResizeTmpPPerm(codeg);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<codeg;i++) ptseen[i]=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM4){
    degf=libGAP_DEG_PPERM4(f);
    degg=libGAP_DEG_PPERM4(libGAP_g);
    deg=libGAP_MAX(degf, degg);
    join=libGAP_NEW_PPERM4(deg);
    libGAP_CODEG_PPERM4(join)=codeg;
    
    ptjoin4=libGAP_ADDR_PPERM4(join);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);
    ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
    
    if(libGAP_DOM_PPERM(f)!=NULL){
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptjoin4[j]=ptf4[j];
        ptseen[ptf4[j]-1]=1;
      }
    }
 
    if(libGAP_DOM_PPERM(libGAP_g)!=NULL){
      dom=libGAP_DOM_PPERM(libGAP_g);
      rank=libGAP_RANK_PPERM4(libGAP_g);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptjoin4[j]==0){
          if(ptseen[ptg4[j]-1]==0){
            ptjoin4[j]=ptg4[j];
            ptseen[ptg4[j]-1]=1;
          } else {
            return libGAP_Fail;//join is not injective
          }
        } else if(ptjoin4[j]!=ptg4[j]){
          return libGAP_Fail;
        }
      }
    }

    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<degf;i++){
        if(ptf4[i]!=0){
          if(ptjoin4[i]==0){
            if(ptseen[ptf4[i]-1]==0){
              ptjoin4[i]=ptf4[i];
              ptseen[ptf4[i]-1]=1;
            } else {
              return libGAP_Fail;
            }
          } else if(ptjoin4[i]!=ptf4[i]){ 
            return libGAP_Fail;
          }
        }
      }
    }
    
    if(libGAP_DOM_PPERM(libGAP_g)==NULL){
      for(i=0;i<degg;i++){
        if(ptg4[i]!=0){
          if(ptjoin4[i]==0){
            if(ptseen[ptg4[i]-1]==0){
              ptjoin4[i]=ptg4[i];
              ptseen[ptg4[i]-1]=1;
            } else {
              return libGAP_Fail;
            }
          } else if(ptjoin4[i]!=ptg4[i]){ 
            return libGAP_Fail;
          }
        }
      }
    }
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
    degf=libGAP_DEG_PPERM4(f);
    degg=libGAP_DEG_PPERM2(libGAP_g);
    deg=libGAP_MAX(degf, degg);
    join=libGAP_NEW_PPERM4(deg);
    libGAP_CODEG_PPERM4(join)=codeg;
    
    ptjoin4=libGAP_ADDR_PPERM4(join);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));

    if(libGAP_DOM_PPERM(f)!=NULL){
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptjoin4[j]=ptf4[j];
        ptseen[ptf4[j]-1]=1;
      }
    }
 
    if(libGAP_DOM_PPERM(libGAP_g)!=NULL){
      dom=libGAP_DOM_PPERM(libGAP_g);
      rank=libGAP_RANK_PPERM2(libGAP_g);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptjoin4[j]==0){
          if(ptseen[ptg2[j]-1]==0){
            ptjoin4[j]=ptg2[j];
            ptseen[ptg2[j]-1]=1;
          } else {
            return libGAP_Fail;//join is not injective
          }
        } else if(ptjoin4[j]!=ptg2[j]){
          return libGAP_Fail;
        }
      }
    }

    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<degf;i++){
        if(ptf4[i]!=0){
          if(ptjoin4[i]==0){
            if(ptseen[ptf4[i]-1]==0){
              ptjoin4[i]=ptf4[i];
              ptseen[ptf4[i]-1]=1;
            } else {
              return libGAP_Fail;
            }
          } else if(ptjoin4[i]!=ptf4[i]){ 
            return libGAP_Fail;
          }
        }
      }
    }
    
    if(libGAP_DOM_PPERM(libGAP_g)==NULL){
      for(i=0;i<degg;i++){
        if(ptg2[i]!=0){
          if(ptjoin4[i]==0){
            if(ptseen[ptg2[i]-1]==0){
              ptjoin4[i]=ptg2[i];
              ptseen[ptg2[i]-1]=1;
            } else {
              return libGAP_Fail;
            }
          } else if(ptjoin4[i]!=ptg2[i]){ 
            return libGAP_Fail;
          }
        }
      }
    }
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM4){
    return libGAP_FuncJOIN_PPERMS(self, libGAP_g, f);
  } else {
    degf=libGAP_DEG_PPERM2(f);
    degg=libGAP_DEG_PPERM2(libGAP_g);
    deg=libGAP_MAX(degf,degg);
    join=libGAP_NEW_PPERM2(deg);
    libGAP_CODEG_PPERM2(join)=codeg;
    
    ptjoin2=libGAP_ADDR_PPERM2(join);
    ptf2=libGAP_ADDR_PPERM2(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));

    if(libGAP_DOM_PPERM(f)!=NULL){
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptjoin2[j]=ptf2[j];
        ptseen[ptf2[j]-1]=1;
      }
    }
 
    if(libGAP_DOM_PPERM(libGAP_g)!=NULL){
      dom=libGAP_DOM_PPERM(libGAP_g);
      rank=libGAP_RANK_PPERM2(libGAP_g);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptjoin2[j]==0){
          if(ptseen[ptg2[j]-1]==0){
            ptjoin2[j]=ptg2[j];
            ptseen[ptg2[j]-1]=1;
          } else {
            return libGAP_Fail;//join is not injective
          }
        } else if(ptjoin2[j]!=ptg2[j]){
          return libGAP_Fail;
        }
      }
    }

    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<degf;i++){
        if(ptf2[i]!=0){
          if(ptjoin2[i]==0){
            if(ptseen[ptf2[i]-1]==0){
              ptjoin2[i]=ptf2[i];
              ptseen[ptf2[i]-1]=1;
            } else {
              return libGAP_Fail;
            }
          } else if(ptjoin2[i]!=ptf2[i]){ 
            return libGAP_Fail;
          }
        }
      }
    }
    
    if(libGAP_DOM_PPERM(libGAP_g)==NULL){
      for(i=0;i<degg;i++){
        if(ptg2[i]!=0){
          if(ptjoin2[i]==0){
            if(ptseen[ptg2[i]-1]==0){
              ptjoin2[i]=ptg2[i];
              ptseen[ptg2[i]-1]=1;
            } else {
              return libGAP_Fail;
            }
          } else if(ptjoin2[i]!=ptg2[i]){ 
            return libGAP_Fail;
          }
        }
      }
    }
  }
  return join;
}

libGAP_Obj libGAP_FuncMEET_PPERMS(libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, i, j, degf, degg, codeg;
  libGAP_UInt2   *ptf2, *ptg2, *ptmeet2;
  libGAP_UInt4   *ptf4, *ptg4, *ptmeet4;
  libGAP_Obj     meet;

  codeg=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
    degf=libGAP_DEG_PPERM2(f);
    degg=libGAP_DEG_PPERM2(libGAP_g);
    ptf2=libGAP_ADDR_PPERM2(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);

    //find degree
    for(deg=libGAP_MIN(degf,degg);deg>0;deg--){
      j=libGAP_IMAGEPP(deg, ptf2, degf);
      if(j!=0&&j==libGAP_IMAGEPP(deg, ptg2, degg)) break;
    }

    meet=libGAP_NEW_PPERM2(deg);
    ptmeet2=libGAP_ADDR_PPERM2(meet);
    ptf2=libGAP_ADDR_PPERM2(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    
    for(i=0;i<deg;i++){
      j=libGAP_IMAGEPP(i+1, ptf2, degf);
      if(libGAP_IMAGEPP(i+1, ptg2, degg)==j){ 
        ptmeet2[i]=j;
        if(j>codeg) codeg=j;
      }
    }
    libGAP_CODEG_PPERM2(meet)=codeg;
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
    degf=libGAP_DEG_PPERM4(f);
    degg=libGAP_DEG_PPERM2(libGAP_g);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);

    //find degree
    for(deg=(degf<degg?degf:degg);deg>0;deg--){
      j=libGAP_IMAGEPP(deg, ptf4, degf);
      if(j!=0&&j==libGAP_IMAGEPP(deg, ptg2, degg)) break;
    }

    meet=libGAP_NEW_PPERM2(deg);
    ptmeet2=libGAP_ADDR_PPERM2(meet);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    
    for(i=0;i<deg;i++){
      j=libGAP_IMAGEPP(i+1, ptf4, degf);
      if(libGAP_IMAGEPP(i+1, ptg2, degg)==j){ 
        ptmeet2[i]=j;
        if(j>codeg) codeg=j;
      }
    }
    libGAP_CODEG_PPERM2(meet)=codeg;
  }else if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM4){
    degf=libGAP_DEG_PPERM2(f);
    degg=libGAP_DEG_PPERM4(libGAP_g);
    ptf2=libGAP_ADDR_PPERM2(f);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);

    //find degree
    for(deg=libGAP_MIN(degf,degg);deg>0;deg--){
      j=libGAP_IMAGEPP(deg, ptf2, degf);
      if(j!=0&&j==libGAP_IMAGEPP(deg, ptg4, degg)) break;
    }

    meet=libGAP_NEW_PPERM2(deg);
    ptmeet2=libGAP_ADDR_PPERM2(meet);
    ptf2=libGAP_ADDR_PPERM2(f);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);
    
    for(i=0;i<deg;i++){
      j=libGAP_IMAGEPP(i+1, ptf2, degf);
      if(libGAP_IMAGEPP(i+1, ptg4, degg)==j){ 
        ptmeet2[i]=j;
        if(j>codeg) codeg=j;
      }
    }
    libGAP_CODEG_PPERM2(meet)=codeg;
  } else {
    degf=libGAP_DEG_PPERM4(f);
    degg=libGAP_DEG_PPERM4(libGAP_g);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);

    //find degree
    for(deg=libGAP_MIN(degf,degg);deg>0;deg--){
      j=libGAP_IMAGEPP(deg, ptf4, degf);
      if(j!=0&&j==libGAP_IMAGEPP(deg, ptg4, degg)) break;
    }

    meet=libGAP_NEW_PPERM4(deg);
    ptmeet4=libGAP_ADDR_PPERM4(meet);
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);
    
    for(i=0;i<deg;i++){
      j=libGAP_IMAGEPP(i+1, ptf4, degf);
      if(libGAP_IMAGEPP(i+1, ptg4, degg)==j){ 
        ptmeet4[i]=j;
        if(j>codeg) codeg=j;
      }
    }
    libGAP_CODEG_PPERM4(meet)=codeg;
  }
  return meet;
}

// restricted partial perm where set is assumed to be a set of positive ints
libGAP_Obj libGAP_FuncRESTRICTED_PPERM(libGAP_Obj self, libGAP_Obj f, libGAP_Obj set){
  libGAP_UInt    i, j, n, codeg, deg;
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *ptf4, *ptg4;
  libGAP_Obj     libGAP_g;

  n=libGAP_LEN_LIST(set);
  codeg=0;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_DEG_PPERM2(f);
    ptf2=libGAP_ADDR_PPERM2(f);
    
    // find pos in list corresponding to degree of new pperm
    while(n>0&&(libGAP_UInt) libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, n))>deg) n--;
    while(n>0&&ptf2[libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, n))-1]==0) n--;
    if(n==0) return libGAP_EmptyPartialPerm;

    libGAP_g=libGAP_NEW_PPERM2(libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, n)));
    ptf2=libGAP_ADDR_PPERM2(f);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);

    for(i=0;i<n;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i+1))-1;
      ptg2[j]=ptf2[j];
      if(ptg2[j]>codeg) codeg=ptg2[j];
    }
    libGAP_CODEG_PPERM2(libGAP_g)=codeg;
    return libGAP_g;
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    deg=libGAP_DEG_PPERM4(f);
    ptf4=libGAP_ADDR_PPERM4(f);
    
    while(n>0&&(libGAP_UInt) libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, n))>deg) n--;
    while(n>0&&ptf4[libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, n))-1]==0) n--;
    if(n==0) return libGAP_EmptyPartialPerm;
   
    libGAP_g=libGAP_NEW_PPERM4(libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, n)));
    ptf4=libGAP_ADDR_PPERM4(f);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);

    for(i=0;i<n;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i+1))-1;
      ptg4[j]=ptf4[j];
      if(ptg4[j]>codeg) codeg=ptg4[j];
    }
    libGAP_CODEG_PPERM4(libGAP_g)=codeg;
    return libGAP_g;
  }
  return libGAP_Fail;
}

// convert a permutation <p> to a partial perm on <set>, which is assumed to be
// a set of positive integers
libGAP_Obj libGAP_FuncAS_PPERM_PERM(libGAP_Obj self, libGAP_Obj p, libGAP_Obj set){
  libGAP_UInt    i, j, n, deg, codeg, dep;
  libGAP_UInt2   *ptf2, *ptp2;
  libGAP_UInt4   *ptf4, *ptp4;
  libGAP_Obj     f;

  n=libGAP_LEN_LIST(set);
  if(n==0) return libGAP_EmptyPartialPerm; 
  deg=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, n));
  codeg=0;

  if(libGAP_TNUM_OBJ(p)==libGAP_T_PERM2){
    dep=libGAP_DEG_PERM2(p);
    if(deg<65536){
      if(dep<deg){
        //Pr("Case 1\n", 0L, 0L);
        f=libGAP_NEW_PPERM2(deg);
        ptf2=libGAP_ADDR_PPERM2(f);
        ptp2=libGAP_ADDR_PERM2(p);
        for(i=1;i<=n;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i))-1;
          ptf2[j]=libGAP_IMAGE(j, ptp2, dep)+1;
        }
        libGAP_CODEG_PPERM2(f)=deg;
      } else { //deg(f)<=deg(p)<=65536
        //Pr("Case 2\n", 0L, 0L);
        f=libGAP_NEW_PPERM2(deg);
        ptf2=libGAP_ADDR_PPERM2(f);
        ptp2=libGAP_ADDR_PERM2(p);
        for(i=1;i<=n;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i))-1;
          ptf2[j]=ptp2[j]+1;
          if(ptf2[j]>codeg) codeg=ptf2[j];
        }
        libGAP_CODEG_PPERM2(f)=codeg;
      }
    } else { //deg(p)<=65536<=deg(f)
      //Pr("Case 3\n", 0L, 0L);
      f=libGAP_NEW_PPERM4(deg);
      ptf4=libGAP_ADDR_PPERM4(f);
      ptp2=libGAP_ADDR_PERM2(p);
      for(i=1;i<=n;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i))-1;
        ptf4[j]=libGAP_IMAGE(j, ptp2, dep)+1;
      }
      libGAP_CODEG_PPERM4(f)=deg;
    }
  } else { // p is PERM4
    dep=libGAP_DEG_PERM4(p);
    if(dep<deg){
      //Pr("Case 4\n", 0L, 0L);
      f=libGAP_NEW_PPERM4(deg);
      ptf4=libGAP_ADDR_PPERM4(f);
      ptp4=libGAP_ADDR_PERM4(p);
      for(i=1;i<=n;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i))-1;
        ptf4[j]=libGAP_IMAGE(j, ptp4, dep)+1;
      }
      libGAP_CODEG_PPERM4(f)=deg;
    } else { //deg<=dep 
      //find the codeg
      i=deg;
      ptp4=libGAP_ADDR_PERM4(p);
      while(codeg<65536&&i>0){
        j=ptp4[libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i--))-1]+1;
        if(j>codeg) codeg=j;
      }
      if(codeg<65536){

        //Pr("Case 5\n", 0L, 0L);
        f=libGAP_NEW_PPERM2(deg);
        ptf2=libGAP_ADDR_PPERM2(f);
        ptp4=libGAP_ADDR_PERM4(p);
        for(i=1;i<=n;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i))-1;
          ptf2[j]=ptp4[j]+1;
        }
        libGAP_CODEG_PPERM2(f)=codeg;
      } else {
        //Pr("Case 6\n", 0L, 0L);
        f=libGAP_NEW_PPERM4(deg);
        ptf4=libGAP_ADDR_PPERM4(f);
        ptp4=libGAP_ADDR_PERM4(p);
        for(i=1;i<=n;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, i))-1;
          ptf4[j]=ptp4[j]+1;
          if(ptf4[j]>codeg) codeg=ptf4[j];
        }
        libGAP_CODEG_PPERM4(f)=deg;
      }
    }
  }
  return f;
}

// for a partial perm with equal dom and img
libGAP_Obj libGAP_FuncAS_PERM_PPERM(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2, *ptp2;
  libGAP_UInt4   *ptf4, *ptp4;
  libGAP_UInt    deg, i, j, rank;
  libGAP_Obj     p, dom;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    if(!libGAP_EQ(libGAP_FuncIMAGE_SET_PPERM(self, f), libGAP_DOM_PPERM(f))){
      return libGAP_Fail;
    }
    deg=libGAP_DEG_PPERM2(f);
    p=libGAP_NEW_PERM2(deg);
    dom=libGAP_DOM_PPERM(f);
    ptp2=libGAP_ADDR_PERM2(p);
    ptf2=libGAP_ADDR_PPERM2(f);
    for(i=0;i<deg;i++) ptp2[i]=i;
    rank=libGAP_RANK_PPERM2(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptp2[j]=ptf2[j]-1;
    }
  } else {
   if(!libGAP_EQ(libGAP_FuncIMAGE_SET_PPERM(self, f), libGAP_DOM_PPERM(f))){
     return libGAP_Fail;
    }
    deg=libGAP_DEG_PPERM4(f);
    p=libGAP_NEW_PERM4(deg);
    dom=libGAP_DOM_PPERM(f);
    ptp4=libGAP_ADDR_PERM4(p);
    ptf4=libGAP_ADDR_PPERM4(f);
    for(i=0;i<deg;i++) ptp4[i]=i;
    rank=libGAP_RANK_PPERM4(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptp4[j]=ptf4[j]-1;
    }
  }
  return p;  
}

// the permutation induced on im(f) by f^-1*g when im(g)=im(f)
// and dom(f)=dom(g), no checking
libGAP_Obj libGAP_FuncPERM_LEFT_QUO_PPERM_NC(libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, i, j, rank;
  libGAP_Obj     perm, dom;
  libGAP_UInt2   *ptf2, *ptp2, *ptg2;
  libGAP_UInt4   *ptf4, *ptp4, *ptg4;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    deg=libGAP_CODEG_PPERM2(f);
    perm=libGAP_NEW_PERM2(deg);
    ptp2=libGAP_ADDR_PERM2(perm);
    for(i=0;i<deg;i++) ptp2[i]=i;
    rank=libGAP_RANK_PPERM2(f);
    dom=libGAP_DOM_PPERM(f);
    //renew pointers since RANK_PPERM can trigger garbage collection
    ptp2=libGAP_ADDR_PERM2(perm);
    ptf2=libGAP_ADDR_PPERM2(f);
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
      ptg2=libGAP_ADDR_PPERM2(libGAP_g);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptp2[ptf2[j]-1]=ptg2[j]-1;
      }
    } else {
      ptg4=libGAP_ADDR_PPERM4(libGAP_g);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptp2[ptf2[j]-1]=ptg4[j]-1;
      }
    }
  } else {
    deg=libGAP_CODEG_PPERM4(f);
    perm=libGAP_NEW_PERM4(deg);
    ptp4=libGAP_ADDR_PERM4(perm);
    for(i=0;i<deg;i++) ptp4[i]=i;
    rank=libGAP_RANK_PPERM4(f);
    dom=libGAP_DOM_PPERM(f);
    //renew pointers since RANK_PPERM can trigger garbage collection
    ptp4=libGAP_ADDR_PERM4(perm);
    ptf4=libGAP_ADDR_PPERM4(f);
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
      ptg2=libGAP_ADDR_PPERM2(libGAP_g);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptp4[ptf4[j]-1]=ptg2[j]-1;
      }
    } else {
      ptg4=libGAP_ADDR_PPERM4(libGAP_g);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptp4[ptf4[j]-1]=ptg4[j]-1;
      }
    }
  }
  return perm;
}

libGAP_Obj libGAP_FuncShortLexLeqPartialPerm( libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    rankf, rankg, i, j, k;
  libGAP_Obj     domf, domg;
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *ptf4, *ptg4;
 
  if(!libGAP_IS_PPERM(f)||!libGAP_IS_PPERM(libGAP_g)){
    libGAP_ErrorQuit("usage: the arguments must be partial perms,", 0L, 0L);
  }

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    if(libGAP_DEG_PPERM2(f)==0) return libGAP_True;
    rankf=libGAP_RANK_PPERM2(f);
    domf=libGAP_DOM_PPERM(f);
  } else {
    if(libGAP_DEG_PPERM4(f)==0) return libGAP_True;
    rankf=libGAP_RANK_PPERM4(f);
    domf=libGAP_DOM_PPERM(f);
  }

  if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
    if(libGAP_DEG_PPERM2(libGAP_g)==0) return libGAP_False;
    rankg=libGAP_RANK_PPERM2(libGAP_g);
    domg=libGAP_DOM_PPERM(libGAP_g);
  } else {
    if(libGAP_DEG_PPERM4(libGAP_g)==0) return libGAP_False;
    rankg=libGAP_RANK_PPERM4(libGAP_g);
    domg=libGAP_DOM_PPERM(f);
  }
  
  if(rankf!=rankg) return (rankf<rankg?libGAP_True:libGAP_False);
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    ptf2=libGAP_ADDR_PPERM2(f);
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
      ptg2=libGAP_ADDR_PPERM2(libGAP_g);
      for(i=1;i<=rankf;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domf, i))-1;
        k=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domg, i))-1;
        if(j!=k) return (j<k?libGAP_True:libGAP_False);
        if(ptf2[j]!=ptg2[j]) return (ptf2[j]<ptg2[j]?libGAP_True:libGAP_False);
      }
    } else {
      ptg4=libGAP_ADDR_PPERM4(libGAP_g);
      for(i=1;i<=rankf;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domf, i))-1;
        k=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domg, i))-1;
        if(j!=k) return (j<k?libGAP_True:libGAP_False);
        if(ptf2[j]!=ptg4[j]) return (ptf2[j]<ptg4[j]?libGAP_True:libGAP_False);
      }
    }
  } else {
    ptf4=libGAP_ADDR_PPERM4(f);
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PPERM2){
      ptg2=libGAP_ADDR_PPERM2(libGAP_g);
      for(i=1;i<=rankf;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domf, i))-1;
        k=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domg, i))-1;
        if(j!=k) return (j<k?libGAP_True:libGAP_False);
        if(ptf4[j]!=ptg2[j]) return (ptf4[j]<ptg2[j]?libGAP_True:libGAP_False);
      }
    } else {
      ptg4=libGAP_ADDR_PPERM4(libGAP_g);
      for(i=1;i<=rankf;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domf, i))-1;
        k=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(domg, i))-1;
        if(j!=k) return (j<k?libGAP_True:libGAP_False);
        if(ptf4[j]!=ptg4[j]) return (ptf4[j]<ptg4[j]?libGAP_True:libGAP_False);
      }
    }
  }

  return libGAP_False;
}

libGAP_Obj libGAP_FuncHAS_DOM_PPERM( libGAP_Obj self, libGAP_Obj f ){
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    return (libGAP_DOM_PPERM(f)==NULL?libGAP_False:libGAP_True);
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    return (libGAP_DOM_PPERM(f)==NULL?libGAP_False:libGAP_True);
  }
  return libGAP_Fail;
}

libGAP_Obj libGAP_FuncHAS_IMG_PPERM( libGAP_Obj self, libGAP_Obj f ){
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){
    return (libGAP_IMG_PPERM(f)==NULL?libGAP_False:libGAP_True);
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_PPERM4){
    return (libGAP_IMG_PPERM(f)==NULL?libGAP_False:libGAP_True);
  }
  return libGAP_Fail;
}

/****************************************************************************/

/* GAP kernel functions */

libGAP_Obj libGAP_ZeroPPerm(libGAP_Obj f){
  return libGAP_EmptyPartialPerm;
}

// an idempotent partial perm on the union of the domain and image 
libGAP_Obj libGAP_OnePPerm( libGAP_Obj f){
  libGAP_Obj     libGAP_g, img, dom;
  libGAP_UInt    i, j, deg, rank;
  libGAP_UInt2   *ptg2;
  libGAP_UInt4   *ptg4;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){//this could be shortened
    deg=libGAP_MAX(libGAP_DEG_PPERM2(f),libGAP_CODEG_PPERM2(f));
    rank=libGAP_RANK_PPERM2(f);
    dom=libGAP_DOM_PPERM(f);
    img=libGAP_IMG_PPERM(f);
  }else{
    deg=libGAP_MAX(libGAP_DEG_PPERM4(f),libGAP_CODEG_PPERM4(f));
    rank=libGAP_RANK_PPERM4(f);
    dom=libGAP_DOM_PPERM(f);
    img=libGAP_IMG_PPERM(f);
  }

  if(deg<65536){ 
    libGAP_g=libGAP_NEW_PPERM2(deg);
    ptg2=libGAP_ADDR_PPERM2(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1;
      ptg2[j]=j+1;
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptg2[j]=j+1;
    }
    libGAP_CODEG_PPERM2(libGAP_g)=deg;
  } else {
    libGAP_g=libGAP_NEW_PPERM4(deg);
    ptg4=libGAP_ADDR_PPERM4(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1;
      ptg4[j]=j+1;
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptg4[j]=j+1;
    }
    libGAP_CODEG_PPERM4(libGAP_g)=deg;
  }
  return libGAP_g;
}

// print a partial perm in disjoint cycle and chain notation
libGAP_Obj libGAP_PrintPPerm2(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, j, n, rank, k, deg; 
  libGAP_UInt4   *ptseen; 
  libGAP_UInt2   *ptf2;
  libGAP_Obj     dom, img;
  
  deg=libGAP_DEG_PPERM2(f);
  if(deg==0) libGAP_Pr("<empty partial perm>", 0L, 0L);
  
  n=libGAP_MAX(deg, libGAP_CODEG_PPERM2(f));
  libGAP_ResizeTmpPPerm(n);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<n;i++) ptseen[i]=0; 
  
  rank=libGAP_RANK_PPERM2(f);//finds dom and img too
  dom=libGAP_DOM_PPERM(f);
  img=libGAP_IMG_PPERM(f);

  ptf2=libGAP_ADDR_PPERM2(f);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  //find img(f)
  for(i=1;i<=rank;i++) ptseen[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1]=1;
  
  //find chains
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptseen[j]==0){
      libGAP_Pr("%>[%>%d%<", (libGAP_Int) j+1, 0L);
      ptseen[j]=2;
      for(k=ptf2[j];(k<=deg&&ptf2[k-1]!=0);k=ptf2[k-1]){ 
        libGAP_Pr(",%>%d%<", (libGAP_Int) k, 0L);
        ptseen[k-1]=2;
      }
      ptseen[k-1]=2;
      libGAP_Pr(",%>%d%<%<]", (libGAP_Int) k, 0L);
    }
  }
  
  //find cycles 
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptseen[j]==1){
      libGAP_Pr("%>(%>%d%<", (libGAP_Int) j+1, 0L);
      for(k=ptf2[j];k!=j+1;k=ptf2[k-1]){ 
        libGAP_Pr(",%>%d%<", (libGAP_Int) k, 0L);
        ptseen[k-1]=0;
      }
      libGAP_Pr("%<)", 0L, 0L);
    }
  }
  return 0L;
}

libGAP_Obj libGAP_PrintPPerm4(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    i, j, n, rank, k, deg; 
  libGAP_UInt4   *ptseen, *ptf4;
  libGAP_Obj     dom, img;
  
  deg=libGAP_DEG_PPERM4(f);
  if(deg==0) libGAP_Pr("<empty partial perm>", 0L, 0L);
  n=libGAP_MAX(deg, libGAP_CODEG_PPERM2(f));
  libGAP_ResizeTmpPPerm(n);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<n;i++) ptseen[i]=0; 
  
  rank=libGAP_RANK_PPERM4(f);//finds dom and img too
  dom=libGAP_DOM_PPERM(f);
  img=libGAP_IMG_PPERM(f);

  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptf4=libGAP_ADDR_PPERM4(f);
  //find img(f)
  for(i=1;i<=rank;i++) ptseen[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i))-1]=1;
  
  //find chains
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptseen[j]==0){
      libGAP_Pr("%>[%>%d%<", (libGAP_Int) j+1, 0L);
      ptseen[j]=2;
      for(k=ptf4[j];(k<=deg&&ptf4[k-1]!=0);k=ptf4[k-1]){ 
        libGAP_Pr(",%>%d%<", (libGAP_Int) k, 0L);
        ptseen[k-1]=2;
      }
      ptseen[k-1]=2;
      libGAP_Pr(",%>%d%<%<]", (libGAP_Int) k, 0L);
    }
  }
  
  //find cycles 
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptseen[j]==1){
      libGAP_Pr("%>(%>%d%<", (libGAP_Int) j+1, 0L);
      for(k=ptf4[j];k!=j+1;k=ptf4[k-1]){ 
        libGAP_Pr(",%>%d%<", (libGAP_Int) k, 0L);
        ptseen[k-1]=0;
      }
      libGAP_Pr("%<)", 0L, 0L);
    }
  }
  return 0L;
}

/* equality for partial perms */
libGAP_Int libGAP_EqPPerm22 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt2*  ptf=libGAP_ADDR_PPERM2(f);
  libGAP_UInt2*  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  libGAP_UInt    deg=libGAP_DEG_PPERM2(f);
  libGAP_UInt    i, j, rank;
  libGAP_Obj     dom;

  if(deg!=libGAP_DEG_PPERM2(libGAP_g)||libGAP_CODEG_PPERM2(f)!=libGAP_CODEG_PPERM2(libGAP_g)) return 0L;
     
  if(libGAP_DOM_PPERM(f)==NULL||libGAP_DOM_PPERM(libGAP_g)==NULL){ 
    for(i=0;i<deg;i++) if(*ptf++!=*ptg++) return 0L;
    return 1L;
  }

  if(libGAP_RANK_PPERM2(f)!=libGAP_RANK_PPERM2(libGAP_g)) return 0L;
  dom=libGAP_DOM_PPERM(f);
  rank=libGAP_RANK_PPERM2(f);
  
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptf[j]!=ptg[j]){
      return 0L;
    }
  }
  return 1L;
}

libGAP_Int libGAP_EqPPerm24 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt2*  ptf=libGAP_ADDR_PPERM2(f);
  libGAP_UInt4*  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  libGAP_UInt    deg=libGAP_DEG_PPERM2(f);
  libGAP_UInt    i, j, rank;
  libGAP_Obj     dom;

  if(deg!=libGAP_DEG_PPERM4(libGAP_g)||libGAP_CODEG_PPERM2(f)!=libGAP_CODEG_PPERM4(libGAP_g)) return 0L;
     
  if(libGAP_DOM_PPERM(f)==NULL||libGAP_DOM_PPERM(libGAP_g)==NULL){ 
    for(i=0;i<deg;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    return 1L;
  }

  if(libGAP_RANK_PPERM2(f)!=libGAP_RANK_PPERM4(libGAP_g)) return 0L;
  dom=libGAP_DOM_PPERM(f);
  rank=libGAP_RANK_PPERM2(f);
  
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptf[j]!=ptg[j]) return 0L;
  }
  return 1L;
}

libGAP_Int libGAP_EqPPerm42 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt4*  ptf=libGAP_ADDR_PPERM4(f);
  libGAP_UInt2*  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  libGAP_UInt    deg=libGAP_DEG_PPERM4(f);
  libGAP_UInt    i, j, rank;
  libGAP_Obj     dom;

  if(deg!=libGAP_DEG_PPERM2(libGAP_g)||libGAP_CODEG_PPERM4(f)!=libGAP_CODEG_PPERM2(libGAP_g)) return 0L;
     
  if(libGAP_DOM_PPERM(f)==NULL||libGAP_DOM_PPERM(libGAP_g)==NULL){ 
    for(i=0;i<deg;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    return 1L;
  }

  if(libGAP_RANK_PPERM4(f)!=libGAP_RANK_PPERM2(libGAP_g)) return 0L;
  dom=libGAP_DOM_PPERM(f);
  rank=libGAP_RANK_PPERM4(f);
  
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptf[j]!=ptg[j]) return 0L;
  }
  return 1L;
}

libGAP_Int libGAP_EqPPerm44 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt4*  ptf=libGAP_ADDR_PPERM4(f);
  libGAP_UInt4*  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  libGAP_UInt    i, j, rank;
  libGAP_UInt    deg=libGAP_DEG_PPERM4(f);
  libGAP_Obj     dom;

  if(deg!=libGAP_DEG_PPERM4(libGAP_g)||libGAP_CODEG_PPERM4(f)!=libGAP_CODEG_PPERM4(libGAP_g)) return 0L;
     
  if(libGAP_DOM_PPERM(f)==NULL||libGAP_DOM_PPERM(libGAP_g)==NULL){ 
    for(i=0;i<deg;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    return 1L;
  }

  if(libGAP_RANK_PPERM4(f)!=libGAP_RANK_PPERM4(libGAP_g)) return 0L;
  dom=libGAP_DOM_PPERM(f);
  rank=libGAP_RANK_PPERM4(f);
  
  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    if(ptf[j]!=ptg[j]) return 0L;
  }
  return 1L;
}

/* less than for partial perms */
// beware this is different than it used to be...
libGAP_Int libGAP_LtPPerm22(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt2*  ptf=libGAP_ADDR_PPERM2(f);
  libGAP_UInt2*  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  libGAP_UInt    deg, i;

  deg=libGAP_DEG_PPERM2(f);
  if(deg!=libGAP_DEG_PPERM2(libGAP_g)){
    if(deg<libGAP_DEG_PPERM2(libGAP_g)){ 
      return 1L; 
    } else { 
      return 0L; 
    }
  }

  for(i=0;i<deg;i++){
    if(*(ptf++)!=*(ptg++)){
      if (*(--ptf)< *(--ptg)) return 1L;
      else                    return 0L;
    }
  }
  return 0L;
}

libGAP_Int libGAP_LtPPerm24(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt2*  ptf=libGAP_ADDR_PPERM2(f);
  libGAP_UInt4*  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  libGAP_UInt    deg, i;

  deg=libGAP_DEG_PPERM2(f);
  if(deg!=libGAP_DEG_PPERM4(libGAP_g)){
    if(deg<libGAP_DEG_PPERM4(libGAP_g)){ 
      return 1L; 
    } else { 
      return 0L; 
    }
  }

  for(i=0;i<deg;i++){
    if(*(ptf++)!=*(ptg++)){
      if (*(--ptf)< *(--ptg)) return 1L;
      else                    return 0L;
    }
  }
  return 0L;
}

libGAP_Int libGAP_LtPPerm42(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt4*  ptf=libGAP_ADDR_PPERM4(f);
  libGAP_UInt2*  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  libGAP_UInt    deg, i;

  deg=libGAP_DEG_PPERM4(f);
  if(deg!=libGAP_DEG_PPERM2(libGAP_g)){
    if(deg<libGAP_DEG_PPERM2(libGAP_g)){ 
      return 1L; 
    } else { 
      return 0L; 
    }
  }

  for(i=0;i<deg;i++){
    if(*(ptf++)!=*(ptg++)){
      if (*(--ptf)< *(--ptg)) return 1L;
      else                    return 0L;
    }
  }
  return 0L;
}

libGAP_Int libGAP_LtPPerm44(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt4*  ptf=libGAP_ADDR_PPERM4(f);
  libGAP_UInt4*  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  libGAP_UInt    deg, i;

  deg=libGAP_DEG_PPERM4(f);
  if(deg!=libGAP_DEG_PPERM4(libGAP_g)){
    if(deg<libGAP_DEG_PPERM4(libGAP_g)){ 
      return 1L; 
    } else { 
      return 0L; 
    }
  }

  for(i=0;i<deg;i++){
    if(*(ptf++)!=*(ptg++)){
      if (*(--ptf)< *(--ptg)) return 1L;
      else                    return 0L;
    }
  }
  return 0L;
}

/* product of partial perm and partial perm */
libGAP_Obj libGAP_ProdPPerm22(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, degg, i, j, rank; 
  libGAP_UInt2   *ptf, *ptg, *ptfg, codeg;
  libGAP_Obj     fg, dom;

  if(libGAP_DEG_PPERM2(libGAP_g)==0) return libGAP_EmptyPartialPerm;

  // find the degree
  deg=libGAP_DEG_PPERM2(f);
  degg=libGAP_DEG_PPERM2(libGAP_g);
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  while(deg>0&&(ptf[deg-1]==0||libGAP_IMAGEPP(ptf[deg-1],ptg,degg)==0)) deg--;
  if(deg==0) return libGAP_EmptyPartialPerm;
  
  // create new pperm
  fg=libGAP_NEW_PPERM2(deg);
  ptfg=libGAP_ADDR_PPERM2(fg);
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  codeg=0;
 
  // compose in rank operations
  if(libGAP_DOM_PPERM(f)!=NULL){
    dom=libGAP_DOM_PPERM(f); 
    rank=libGAP_RANK_PPERM2(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(j<deg && ptf[j]<=degg){ 
        ptfg[j]=ptg[ptf[j]-1];
        if(ptfg[j]>codeg) codeg=ptfg[j];
      }
    }
  } else { 
  // compose in deg operations
    for(i=0;i<deg;i++){
      //JDM could have additional case so that we don't have to check
      //ptf[i]<=degg
      if(ptf[i]!=0&&ptf[i]<=degg){
        ptfg[i]=ptg[ptf[i]-1];
        if(ptfg[i]>codeg) codeg=ptfg[i];
      }
    }
  }
  libGAP_CODEG_PPERM2(fg)=codeg;
  return fg;
}

// the product is always pperm2
libGAP_Obj libGAP_ProdPPerm42(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, degg, i, j, rank;
  libGAP_UInt4   *ptf;
  libGAP_UInt2   *ptg, *ptfg, codeg;
  libGAP_Obj     fg, dom;
  
  if(libGAP_DEG_PPERM2(libGAP_g)==0) return libGAP_EmptyPartialPerm;

  // find the degree
  deg=libGAP_DEG_PPERM4(f);
  degg=libGAP_DEG_PPERM2(libGAP_g);
  ptf=libGAP_ADDR_PPERM4(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  while(deg>0&&(ptf[deg-1]==0||ptf[deg-1]>degg||ptg[ptf[deg-1]-1]==0)) deg--;
  if(deg==0) return libGAP_EmptyPartialPerm;
  
  // create new pperm
  fg=libGAP_NEW_PPERM2(deg);
  ptfg=libGAP_ADDR_PPERM2(fg);
  ptf=libGAP_ADDR_PPERM4(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  codeg=0;
 
  // compose in rank operations
  if(libGAP_DOM_PPERM(f)!=NULL){
    dom=libGAP_DOM_PPERM(f); 
    rank=libGAP_RANK_PPERM4(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(j<deg && ptf[j]<=degg){ 
        ptfg[j]=ptg[ptf[j]-1];
        if(ptfg[j]>codeg) codeg=ptfg[j];
      }
    }
  } else { 
  // compose in deg operations
    for(i=0;i<deg;i++){
      if(ptf[i]!=0&&ptf[i]<=degg){
        ptfg[i]=ptg[ptf[i]-1];
        if(ptfg[i]>codeg) codeg=ptfg[i];
      }
    }
  }
  libGAP_CODEG_PPERM2(fg)=codeg;
  return fg;
}

//it is possible that f*g could be represented as a PPERM2
libGAP_Obj libGAP_ProdPPerm44(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, degg, codeg, i, j, rank; 
  libGAP_UInt4   *ptf, *ptg, *ptfg;
  libGAP_Obj     fg, dom;

  if (libGAP_DEG_PPERM4(libGAP_g) == 0) { 
    return libGAP_EmptyPartialPerm;
  }

  // find the degree
  deg = libGAP_DEG_PPERM4(f);
  degg = libGAP_DEG_PPERM4(libGAP_g);
  ptf = libGAP_ADDR_PPERM4(f);
  ptg = libGAP_ADDR_PPERM4(libGAP_g);
  while (deg > 0 
      && (ptf[deg - 1] == 0 || ptf[deg - 1] > degg || ptg[ptf[deg - 1] - 1] == 0)){
    deg--;
  }
  
  if(deg == 0){
    return libGAP_EmptyPartialPerm;
  }
  
  // create new pperm
  fg = libGAP_NEW_PPERM4(deg);
  ptfg = libGAP_ADDR_PPERM4(fg);
  ptf = libGAP_ADDR_PPERM4(f);
  ptg = libGAP_ADDR_PPERM4(libGAP_g);
  codeg = 0;
 
  // compose in rank operations
  if (libGAP_DOM_PPERM(f) != NULL) {
    dom = libGAP_DOM_PPERM(f); 
    rank = libGAP_RANK_PPERM4(f);
    for (i = 1; i <= rank; i++) {
      j = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i)) - 1;
      if (j < deg && ptf[j] <= degg) { 
        ptfg[j] = ptg[ptf[j] - 1];
        if(ptfg[j] > codeg){
          codeg = ptfg[j];
        }
      }
    }
  } else { 
  // compose in deg operations
    for (i = 0; i < deg; i++) {
      if (ptf[i] != 0 && ptf[i] <= degg) {
        ptfg[i] = ptg[ptf[i] - 1];
        if (ptfg[i] > codeg) {
          codeg = ptfg[i];
        }
      }
    }
  }
  libGAP_CODEG_PPERM4(fg) = codeg;
  return fg;
}

//it is possible that f*g could be represented as a PPERM2
libGAP_Obj libGAP_ProdPPerm24(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, degg, i, j, codeg, rank;
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *ptg, *ptfg;
  libGAP_Obj     fg, dom;

  if(libGAP_DEG_PPERM4(libGAP_g)==0) return libGAP_EmptyPartialPerm;

  // find the degree
  deg=libGAP_DEG_PPERM2(f);
  degg=libGAP_DEG_PPERM4(libGAP_g);
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM4(libGAP_g);

  if(libGAP_CODEG_PPERM2(f)<=degg){
    while(deg>0&&(ptf[deg-1]==0||ptg[ptf[deg-1]-1]==0)) deg--;
  } else {
    while(deg>0&&(ptf[deg-1]==0||libGAP_IMAGEPP(ptf[deg-1], ptg, degg)==0)) deg--;
  }
  
  if(deg==0) return libGAP_EmptyPartialPerm;
  
  // create new pperm
  fg=libGAP_NEW_PPERM4(deg);
  ptfg=libGAP_ADDR_PPERM4(fg);
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  codeg=0;
 
  // compose in rank operations
  if(libGAP_DOM_PPERM(f)!=NULL){
    dom=libGAP_DOM_PPERM(f); 
    rank=libGAP_RANK_PPERM2(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(j<deg && ptf[j]<=degg){ 
        ptfg[j]=ptg[ptf[j]-1];
        if(ptfg[j]>codeg) codeg=ptfg[j];
      }
    }
  } else { 
  // compose in deg operations
    for(i=0;i<deg;i++){
      if(ptf[i]!=0&&ptf[i]<=degg){
        ptfg[i]=ptg[ptf[i]-1];
        if(ptfg[i]>codeg) codeg=ptfg[i];
      }
    }
  }
  libGAP_CODEG_PPERM4(fg)=codeg;
  return fg;
}

// compose partial perms and perms 
libGAP_Obj libGAP_ProdPPerm2Perm2( libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt2   *ptf, *ptp, *ptfp2;
  libGAP_UInt4   *ptfp4;
  libGAP_Obj     fp, dom;
  libGAP_UInt    codeg, dep, deg, i, j, rank;

  dep=libGAP_DEG_PERM2(p);
  deg=libGAP_DEG_PPERM2(f);
  
  if(dep<65536){
    fp=libGAP_NEW_PPERM2(deg);
  } else {// i.e. deg(p)=65536
    fp=libGAP_NEW_PPERM4(deg);
  }
  
  codeg=libGAP_CODEG_PPERM2(f);
  ptf=libGAP_ADDR_PPERM2(f);
  ptp=libGAP_ADDR_PERM2(p);
  
  if(dep<65536){
    ptfp2=libGAP_ADDR_PPERM2(fp);
    if(codeg<=dep){
      codeg=0;
      if(libGAP_DOM_PPERM(f)==NULL){
        //Pr("Case 2\n", 0L, 0L);
        for(i=0;i<deg;i++){ 
          if(ptf[i]!=0){
            ptfp2[i]=ptp[ptf[i]-1]+1;
            if(ptfp2[i]>codeg) codeg=ptfp2[i];
          }
        }
      } else {
        //Pr("Case 1\n", 0L, 0L);
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM2(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
          ptfp2[j]=ptp[ptf[j]-1]+1;
          if(ptfp2[j]>codeg) codeg=ptfp2[j];
        }
      }
    } else {
      if(libGAP_DOM_PPERM(f)==NULL){
       //Pr("Case 4\n", 0L, 0L);
        for(i=0;i<deg;i++){
          if(ptf[i]!=0){
            ptfp2[i]=libGAP_IMAGE(ptf[i]-1, ptp, dep)+1;
          }
        }
      }else{
        //Pr("Case 3\n", 0L, 0L);
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM2(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
          ptfp2[j]=libGAP_IMAGE(ptf[j]-1, ptp, dep)+1;
        }
      }
    }
    libGAP_CODEG_PPERM2(fp)=codeg;
  } else {
    ptfp4=libGAP_ADDR_PPERM4(fp);
    codeg=0;
    if(libGAP_DOM_PPERM(f)==NULL){
      //Pr("Case 6\n", 0L, 0L);
      for(i=0;i<deg;i++){
        if(ptf[i]!=0){
          ptfp4[i]=ptp[ptf[i]-1]+1;
          if(ptfp4[i]>codeg) codeg=ptfp4[i];
        }
      }
    } else {
      //Pr("Case 5\n", 0L, 0L);
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptfp4[j]=ptp[ptf[j]-1]+1;
        if(ptfp4[j]>codeg) codeg=ptfp4[j];
      }
    }
    libGAP_CODEG_PPERM4(fp)=codeg;
  }
  return fp;
}

libGAP_Obj libGAP_ProdPPerm4Perm4( libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt4   *ptf, *ptp, *ptfp;
  libGAP_Obj     fp, dom;
  libGAP_UInt    codeg, dep, deg, i, j, rank;

  deg=libGAP_DEG_PPERM4(f);
  fp=libGAP_NEW_PPERM4(deg);
  
  dep=libGAP_DEG_PERM4(p);
  codeg=libGAP_CODEG_PPERM4(f);
  
  ptf=libGAP_ADDR_PPERM4(f);
  ptp=libGAP_ADDR_PERM4(p);
  ptfp=libGAP_ADDR_PPERM4(fp);
  
  if(codeg<=dep){
    codeg=0;
    if(libGAP_DOM_PPERM(f)==NULL){
      //Pr("case 1\n", 0L, 0L);
      for(i=0;i<deg;i++){ 
        if(ptf[i]!=0){
          ptfp[i]=ptp[ptf[i]-1]+1;
          if(ptfp[i]>codeg) codeg=ptfp[i];
        }
      }
    } else {
      //Pr("case 2\n", 0L, 0L);
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptfp[j]=ptp[ptf[j]-1]+1;
        if(ptfp[j]>codeg) codeg=ptfp[j];
      }
    }
  } else {
    if(libGAP_DOM_PPERM(f)==NULL){
      //Pr("case 3\n", 0L, 0L);
      for(i=0;i<deg;i++){
        if(ptf[i]!=0){
          ptfp[i]=libGAP_IMAGE(ptf[i]-1, ptp, dep)+1;
        }
      }
    }else{
      //Pr("case 4\n", 0L, 0L);
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptfp[j]=libGAP_IMAGE(ptf[j]-1, ptp, dep)+1;
      }
    }
  }
  libGAP_CODEG_PPERM4(fp)=codeg;
  return fp;
}

libGAP_Obj libGAP_ProdPPerm2Perm4( libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt2   *ptf; 
  libGAP_UInt4   *ptp, *ptfp;
  libGAP_Obj     fp, dom;
  libGAP_UInt    deg, codeg, i, j, rank;

  fp=libGAP_NEW_PPERM4(libGAP_DEG_PPERM2(f));
  ptf=libGAP_ADDR_PPERM2(f);
  ptp=libGAP_ADDR_PERM4(p);
  ptfp=libGAP_ADDR_PPERM4(fp);
  codeg=0;
  
  if(libGAP_DOM_PPERM(f)==NULL){
    deg=libGAP_DEG_PPERM2(f);
    for(i=0;i<deg;i++){
      if(ptf[i]!=0){
        ptfp[i]=ptp[ptf[i]-1]+1;
        if(ptfp[i]>codeg) codeg=ptfp[i];
      }
    }
  } else {
    dom=libGAP_DOM_PPERM(f);
    rank=libGAP_RANK_PPERM2(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptfp[j]=ptp[ptf[j]-1]+1;
      if(ptfp[j]>codeg) codeg=ptfp[j];
    }
  }
  libGAP_CODEG_PPERM4(fp)=codeg;
  return fp;
}

libGAP_Obj libGAP_ProdPPerm4Perm2( libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt4   *ptf, *ptfp;
  libGAP_UInt2   *ptp, dep; 
  libGAP_Obj     fp, dom;
  libGAP_UInt    codeg, deg, i, j, rank;

  deg=libGAP_DEG_PPERM4(f);
  fp=libGAP_NEW_PPERM4(deg);
  
  dep=libGAP_DEG_PERM2(p);
  codeg=libGAP_CODEG_PPERM4(f);
  
  ptf=libGAP_ADDR_PPERM4(f);
  ptp=libGAP_ADDR_PERM2(p);
  ptfp=libGAP_ADDR_PPERM4(fp);
  
  if(libGAP_DOM_PPERM(f)==NULL){
    //Pr("case 1\n", 0L, 0L);
    for(i=0;i<deg;i++){
      if(ptf[i]!=0){
        ptfp[i]=libGAP_IMAGE(ptf[i]-1, ptp, dep)+1;
      }
    }
  }else{
    //Pr("case 2\n", 0L, 0L);
    dom=libGAP_DOM_PPERM(f);
    rank=libGAP_RANK_PPERM4(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptfp[j]=libGAP_IMAGE(ptf[j]-1, ptp, dep)+1;
    }
  }
  libGAP_CODEG_PPERM4(fp)=codeg;
  return fp;
}

// product of a perm and a partial perm
libGAP_Obj libGAP_ProdPerm2PPerm2( libGAP_Obj p, libGAP_Obj f){
  libGAP_UInt2   deg, *ptp, *ptf, *ptpf;
  libGAP_UInt    degf, i;
  libGAP_Obj     pf;

  if(libGAP_DEG_PPERM2(f)==0) return libGAP_EmptyPartialPerm;

  deg=libGAP_DEG_PERM2(p);
  degf=libGAP_DEG_PPERM2(f);
  
  if(deg<degf){
    //Pr("case 1\n", 0L, 0L);
    pf=libGAP_NEW_PPERM2(degf);
    ptpf=libGAP_ADDR_PPERM2(pf);
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM2(f);
    for(i=0;i<deg;i++) *ptpf++=ptf[*ptp++];
    for(;i<degf;i++) *ptpf++=ptf[i];
  } else { // deg(f)<=deg(p)
    //Pr("case 2\n", 0L, 0L);
    //find the degree
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM2(f);
    while(ptp[deg-1]>=degf||ptf[ptp[deg-1]]==0) deg--;
    pf=libGAP_NEW_PPERM2(deg);
    ptpf=libGAP_ADDR_PPERM2(pf);
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM2(f);
    for(i=0;i<deg;i++) if(ptp[i]<degf) ptpf[i]=ptf[ptp[i]];
  }
  libGAP_CODEG_PPERM2(pf)=libGAP_CODEG_PPERM2(f);
  return pf;
}

libGAP_Obj libGAP_ProdPerm4PPerm4( libGAP_Obj p, libGAP_Obj f){
  libGAP_UInt4   deg, *ptp, *ptf, *ptpf;
  libGAP_UInt    degf, i;
  libGAP_Obj     pf;

  if(libGAP_DEG_PPERM4(f)==0) return libGAP_EmptyPartialPerm;

  deg=libGAP_DEG_PERM4(p);
  degf=libGAP_DEG_PPERM4(f);
  
  if(deg<degf){
    //Pr("case 1\n", 0L, 0L);
    pf=libGAP_NEW_PPERM4(degf);
    ptpf=libGAP_ADDR_PPERM4(pf);
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM4(f);
    for(i=0;i<deg;i++) *ptpf++=ptf[*ptp++];
    for(;i<degf;i++) *ptpf++=ptf[i];
  } else { // deg(f)<deg(p)
    //Pr("case 2\n", 0L, 0L);
    //fin the degree
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM4(f);
    while(ptp[deg-1]>=degf||ptf[ptp[deg-1]]==0) deg--;
    pf=libGAP_NEW_PPERM4(deg);
    ptpf=libGAP_ADDR_PPERM4(pf);
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM4(f);
    for(i=0;i<deg;i++) if(ptp[i]<degf) ptpf[i]=ptf[ptp[i]];
  }
  libGAP_CODEG_PPERM4(pf)=libGAP_CODEG_PPERM4(f);
  return pf;
}

libGAP_Obj libGAP_ProdPerm4PPerm2( libGAP_Obj p, libGAP_Obj f){
  libGAP_UInt4   deg, *ptp;
  libGAP_UInt2   *ptf, *ptpf;
  libGAP_UInt    degf, i;
  libGAP_Obj     pf;
  
  if(libGAP_DEG_PPERM2(f)==0) return libGAP_EmptyPartialPerm;

  deg=libGAP_DEG_PERM4(p);
  degf=libGAP_DEG_PPERM2(f);
  if(deg<degf){
    //Pr("case 1\n", 0L, 0L);
    pf=libGAP_NEW_PPERM2(degf);
    ptpf=libGAP_ADDR_PPERM2(pf);
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM2(f);
    for(i=0;i<deg;i++) *ptpf++=ptf[*ptp++];
    for(;i<degf;i++) *ptpf++=ptf[i];
  } else { // deg(f)<=deg(p)
    //Pr("case 2\n", 0L, 0L);
    //find the degree
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM2(f);
    while(ptp[deg-1]>=degf||ptf[ptp[deg-1]]==0) deg--;
    pf=libGAP_NEW_PPERM2(deg);
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM2(f);
    ptpf=libGAP_ADDR_PPERM2(pf);
    for(i=0;i<deg;i++) if(ptp[i]<degf) ptpf[i]=ptf[ptp[i]];
  }
  
  libGAP_CODEG_PPERM2(pf)=libGAP_CODEG_PPERM2(f);
  return pf;
}

libGAP_Obj libGAP_ProdPerm2PPerm4( libGAP_Obj p, libGAP_Obj f){
  libGAP_UInt2   *ptp;
  libGAP_UInt4   *ptf, *ptpf;
  libGAP_UInt    deg, degf, i;
  libGAP_Obj     pf;
  
  if(libGAP_DEG_PPERM4(f)==0) return libGAP_EmptyPartialPerm;

  deg=libGAP_DEG_PERM2(p);
  degf=libGAP_DEG_PPERM4(f);
  if(deg<degf){
    //Pr("case 1\n", 0L, 0L);
    pf=libGAP_NEW_PPERM4(degf);
    ptpf=libGAP_ADDR_PPERM4(pf);
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM4(f);
    for(i=0;i<deg;i++) *ptpf++=ptf[*ptp++];
    for(;i<degf;i++) *ptpf++=ptf[i];
  } else { // deg(f)<deg(p)
    //Pr("case 2\n", 0L, 0L);
    //find the degree
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM4(f);
    while(ptp[deg-1]>=degf||ptf[ptp[deg-1]]==0) deg--;
    pf=libGAP_NEW_PPERM4(deg);
    ptpf=libGAP_ADDR_PPERM4(pf);
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM4(f);
    for(i=0;i<deg;i++) if(ptp[i]<degf) ptpf[i]=ptf[ptp[i]];
  }
  
  libGAP_CODEG_PPERM4(pf)=libGAP_CODEG_PPERM4(f);
  return pf;
}

//the inverse of a partial perm
libGAP_Obj libGAP_InvPPerm2 (libGAP_Obj f){
  libGAP_UInt    deg, codeg, i, j, rank;
  libGAP_UInt2   *ptf, *ptinv2;
  libGAP_UInt4   *ptinv4;
  libGAP_Obj     inv, dom;

  deg=libGAP_DEG_PPERM2(f);
  codeg=libGAP_CODEG_PPERM2(f);

  if(deg<65536){
    inv=libGAP_NEW_PPERM2(codeg);
    ptf=libGAP_ADDR_PPERM2(f);
    ptinv2=libGAP_ADDR_PPERM2(inv);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf[i]!=0) ptinv2[ptf[i]-1]=i+1;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptinv2[ptf[j]-1]=j+1;
      }
    }
    libGAP_CODEG_PPERM2(inv)=deg;
  } else { 
    inv=libGAP_NEW_PPERM4(codeg);
    ptf=libGAP_ADDR_PPERM2(f);
    ptinv4=libGAP_ADDR_PPERM4(inv);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf[i]!=0) ptinv4[ptf[i]-1]=i+1;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptinv4[ptf[j]-1]=j+1;
      }
    }
    libGAP_CODEG_PPERM4(inv)=deg;
  }
  return inv;
}      

libGAP_Obj libGAP_InvPPerm4 (libGAP_Obj f){
  libGAP_UInt    deg, codeg, i, j, rank;
  libGAP_UInt2   *ptinv2;
  libGAP_UInt4   *ptf, *ptinv4;
  libGAP_Obj     inv, dom;

  deg=libGAP_DEG_PPERM4(f);
  codeg=libGAP_CODEG_PPERM4(f);

  if(deg<65536){
    inv=libGAP_NEW_PPERM2(codeg);
    ptf=libGAP_ADDR_PPERM4(f);
    ptinv2=libGAP_ADDR_PPERM2(inv);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf[i]!=0) ptinv2[ptf[i]-1]=i+1;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptinv2[ptf[j]-1]=j+1;
      }
    }
    libGAP_CODEG_PPERM2(inv)=deg;
  } else { 
    inv=libGAP_NEW_PPERM4(codeg);
    ptf=libGAP_ADDR_PPERM4(f);
    ptinv4=libGAP_ADDR_PPERM4(inv);
    if(libGAP_DOM_PPERM(f)==NULL){
      for(i=0;i<deg;i++) if(ptf[i]!=0) ptinv4[ptf[i]-1]=i+1;
    } else {
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptinv4[ptf[j]-1]=j+1;
      }
    }
    libGAP_CODEG_PPERM4(inv)=deg;
  }
  return inv;
}      

//Conjugation: p^-1*f*p
// lose the assumption that dom is known and renovate as per LquoPermPPerm 
// JDM
libGAP_Obj libGAP_PowPPerm2Perm2(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt    deg, rank, degconj, i, j, k, codeg; 
  libGAP_UInt2   *ptf, *ptp, *ptconj, dep;
  libGAP_Obj     conj, dom;

  deg=libGAP_DEG_PPERM2(f);
  if(deg==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM2(p); 
  rank=libGAP_RANK_PPERM2(f);
  ptp=libGAP_ADDR_PERM2(p);
  dom=libGAP_DOM_PPERM(f); 
  
  //find deg of conjugate
  if(deg>dep){
    degconj=deg;
  } else {
    degconj=0;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptp[j]>=degconj) degconj=ptp[j]+1;
    }
  }
  
  conj=libGAP_NEW_PPERM2(degconj);
  ptconj=libGAP_ADDR_PPERM2(conj);
  ptp=libGAP_ADDR_PERM2(p);
  ptf=libGAP_ADDR_PPERM2(f);
  codeg=libGAP_CODEG_PPERM2(f);

  if(codeg>dep){
    libGAP_CODEG_PPERM2(conj)=codeg;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptconj[libGAP_IMAGE(j, ptp, dep)]=libGAP_IMAGE(ptf[j]-1, ptp, dep)+1;
    }
  } else {
    codeg=0;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      k=ptp[ptf[j]-1]+1;
      ptconj[libGAP_IMAGE(j, ptp, dep)]=k;
      if(k>codeg) codeg=k;
    }
    libGAP_CODEG_PPERM2(conj)=codeg;
  }

  return conj;
}

libGAP_Obj libGAP_PowPPerm2Perm4(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt    deg, rank, degconj, i, j, k, codeg; 
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *ptp, *ptconj, dep;
  libGAP_Obj     conj, dom;

  deg=libGAP_DEG_PPERM2(f);
  if(deg==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM4(p); 
  rank=libGAP_RANK_PPERM2(f);
  ptp=libGAP_ADDR_PERM4(p);
  dom=libGAP_DOM_PPERM(f); 
  //find deg of conjugate
  if(deg>dep){
    degconj=deg;
  } else {
    degconj=0;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptp[j]>=degconj) degconj=ptp[j]+1;
    }
  }
  
  conj=libGAP_NEW_PPERM4(degconj);
  ptconj=libGAP_ADDR_PPERM4(conj);
  ptp=libGAP_ADDR_PERM4(p);
  ptf=libGAP_ADDR_PPERM2(f);
  codeg=0;

  for(i=1;i<=rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
    k=ptp[ptf[j]-1]+1;
    ptconj[libGAP_IMAGE(j, ptp, dep)]=k;
    if(k>codeg) codeg=k;
  }
  libGAP_CODEG_PPERM4(conj)=codeg;

  return conj;
}

libGAP_Obj libGAP_PowPPerm4Perm2(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt    deg, rank, degconj, i, j, k, codeg; 
  libGAP_UInt4   *ptf, *ptconj, dep;
  libGAP_UInt2   *ptp;
  libGAP_Obj     conj, dom;

  deg=libGAP_DEG_PPERM4(f);
  if(deg==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM2(p); 
  rank=libGAP_RANK_PPERM4(f);
  ptp=libGAP_ADDR_PERM2(p);
  dom=libGAP_DOM_PPERM(f); 
  
  //find deg of conjugate
  if(deg>dep){
    degconj=deg;
  } else {
    degconj=0;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptp[j]>=degconj) degconj=ptp[j]+1;
    }
  }
  
  conj=libGAP_NEW_PPERM4(degconj);
  ptconj=libGAP_ADDR_PPERM4(conj);
  ptp=libGAP_ADDR_PERM2(p);
  ptf=libGAP_ADDR_PPERM4(f);
  codeg=libGAP_CODEG_PPERM4(f);
  
  if(codeg>dep){
    libGAP_CODEG_PPERM4(conj)=codeg;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptconj[libGAP_IMAGE(j, ptp, dep)]=libGAP_IMAGE(ptf[j]-1, ptp, dep)+1;
    }
  } else {
    codeg=0;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      k=ptp[ptf[j]-1]+1;
      ptconj[libGAP_IMAGE(j, ptp, dep)]=k;
      if(k>codeg) codeg=k;
    }
    libGAP_CODEG_PPERM4(conj)=codeg;
  }
  return conj;
}

libGAP_Obj libGAP_PowPPerm4Perm4(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt    deg, rank, degconj, i, j, k, codeg; 
  libGAP_UInt4   *ptf, *ptp, *ptconj, dep;
  libGAP_Obj     conj, dom;

  deg=libGAP_DEG_PPERM4(f);
  if(deg==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM4(p); 
  rank=libGAP_RANK_PPERM4(f);
  ptp=libGAP_ADDR_PERM4(p);
  dom=libGAP_DOM_PPERM(f); 
  
  //find deg of conjugate
  if(deg>dep){
    degconj=deg;
  } else {
    degconj=0;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptp[j]>=degconj) degconj=ptp[j]+1;
    }
  }
  
  conj=libGAP_NEW_PPERM4(degconj);
  ptconj=libGAP_ADDR_PPERM4(conj);
  ptp=libGAP_ADDR_PERM4(p);
  ptf=libGAP_ADDR_PPERM4(f);
  codeg=libGAP_CODEG_PPERM4(f);
  
  if(codeg>dep){
    //Pr("case 1\n", 0L, 0L);
    libGAP_CODEG_PPERM4(conj)=codeg;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptconj[libGAP_IMAGE(j, ptp, dep)]=libGAP_IMAGE(ptf[j]-1, ptp, dep)+1;
    }
  } else { //codeg(f)<=deg(p)
    //Pr("case 2"\n, 0L, 0L);
    codeg=0;
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      k=ptp[ptf[j]-1]+1;
      ptconj[libGAP_IMAGE(j, ptp, dep)]=k;
      if(k>codeg) codeg=k;
    }
    libGAP_CODEG_PPERM4(conj)=codeg;
  }
  return conj;
}

// g^-1*f*g
// JDM not sure this is worth it...
libGAP_Obj libGAP_PowPPerm22( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt2   *ptg, *ptf, *ptconj, img;
  libGAP_UInt    i, j, def, deg, dec, codeg, codec, min, len;
  libGAP_Obj     dom, conj;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM2(f);
  deg=libGAP_DEG_PPERM2(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  dom=libGAP_DOM_PPERM(f);
  codeg=libGAP_CODEG_PPERM2(libGAP_g); 
  dec=0;
  codec=0;
  
  if(dom==NULL){
    min=libGAP_MIN(def,deg);
    if(libGAP_CODEG_PPERM2(f)<=deg){
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&ptg[ptf[i]-1]!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM2(dec);
      ptconj=libGAP_ADDR_PPERM2(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=ptg[ptf[i]-1];
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&libGAP_IMAGEPP(ptf[i], ptg, deg)!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM2(dec);
      ptconj=libGAP_ADDR_PPERM2(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=libGAP_IMAGEPP(ptf[i], ptg, deg);
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else if(def>deg){ //dom(f) is known
    if(libGAP_CODEG_PPERM2(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM2(dec);
      ptconj=libGAP_ADDR_PPERM2(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      len=libGAP_LEN_PLIST(dom);
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM2(dec);
      ptconj=libGAP_ADDR_PPERM2(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else { // def<=deg and dom(f) is known
    if(libGAP_CODEG_PPERM2(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom); 
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM2(dec);
      ptconj=libGAP_ADDR_PPERM2(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else {//codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom); 
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM2(dec);
      ptconj=libGAP_ADDR_PPERM2(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  }
  libGAP_CODEG_PPERM2(conj)=codec;
  return conj;
}

libGAP_Obj libGAP_PowPPerm24( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt4   *ptg, *ptconj;
  libGAP_UInt2   *ptf;
  libGAP_UInt    i, j, def, deg, dec, codeg, codec, min, img, len;
  libGAP_Obj     dom, conj;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM2(f);
  deg=libGAP_DEG_PPERM4(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  dom=libGAP_DOM_PPERM(f);
  codeg=libGAP_CODEG_PPERM4(libGAP_g); 
  dec=0;
  codec=0;
  
  if(dom==NULL){
    min=libGAP_MIN(def,deg);
    if(libGAP_CODEG_PPERM2(f)<=deg){
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&ptg[ptf[i]-1]!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=ptg[ptf[i]-1];
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&libGAP_IMAGEPP(ptf[i], ptg, deg)!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=libGAP_IMAGEPP(ptf[i], ptg, deg);
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else if(def>deg){ //dom(f) is known
    if(libGAP_CODEG_PPERM2(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else { // def<=deg and dom(f) is known
    if(libGAP_CODEG_PPERM2(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else {//codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM2(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  }
  libGAP_CODEG_PPERM4(conj)=codec;
  return conj;
}

libGAP_Obj libGAP_PowPPerm42( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt4   *ptf, *ptconj;
  libGAP_UInt2   *ptg;
  libGAP_UInt    i, j, def, deg, dec, codeg, codec, min, img, len;
  libGAP_Obj     dom, conj;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM4(f);
  deg=libGAP_DEG_PPERM2(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM4(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  dom=libGAP_DOM_PPERM(f);
  codeg=libGAP_CODEG_PPERM2(libGAP_g); 
  dec=0;
  codec=0;
  
  if(dom==NULL){
    min=libGAP_MIN(def,deg);
    if(libGAP_CODEG_PPERM4(f)<=deg){
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&ptg[ptf[i]-1]!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=ptg[ptf[i]-1];
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&libGAP_IMAGEPP(ptf[i], ptg, deg)!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=libGAP_IMAGEPP(ptf[i], ptg, deg);
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else if(def>deg){ //dom(f) is known
    if(libGAP_CODEG_PPERM4(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);
      
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else { // def<=deg and dom(f) is known
    if(libGAP_CODEG_PPERM4(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else {//codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM2(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  }
  libGAP_CODEG_PPERM4(conj)=codec;
  return conj;
}

libGAP_Obj libGAP_PowPPerm44( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt4   *ptg, *ptf, *ptconj, img;
  libGAP_UInt    i, j, def, deg, dec, codeg, codec, min, len;
  libGAP_Obj     dom, conj;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM4(f);
  deg=libGAP_DEG_PPERM4(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM4(f);
  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  dom=libGAP_DOM_PPERM(f);
  codeg=libGAP_CODEG_PPERM4(libGAP_g); 
  dec=0;
  codec=0;
  
  if(dom==NULL){
    min=libGAP_MIN(def,deg);
    if(libGAP_CODEG_PPERM4(f)<=deg){
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&ptg[ptf[i]-1]!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=ptg[ptf[i]-1];
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]>dec&&libGAP_IMAGEPP(ptf[i], ptg, deg)!=0){
          dec=ptg[i];
          if(dec==codeg) break;
        }
      }

      if(dec==0) return libGAP_EmptyPartialPerm;
    
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=0;i<min;i++){
        if(ptf[i]!=0&&ptg[i]!=0){
          img=libGAP_IMAGEPP(ptf[i], ptg, deg);
          if(img!=0){
            ptconj[ptg[i]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else if(def>deg){ //dom(f) is known
    if(libGAP_CODEG_PPERM4(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else { //codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);
      
      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(libGAP_IMAGEPP(j+1, ptg, deg)!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  } else { // def<=deg and dom(f) is known
    if(libGAP_CODEG_PPERM4(f)<=deg){
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&ptg[ptf[j]-1]!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=ptg[ptf[j]-1];
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    } else {//codeg(f)>deg(g)
      //find the degree of conj
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]>dec&&libGAP_IMAGEPP(ptf[j], ptg, deg)!=0){
          dec=ptg[j];
          if(dec==codeg) break;
        }
      }
     
      //create new pperm
      conj=libGAP_NEW_PPERM4(dec);
      ptconj=libGAP_ADDR_PPERM4(conj);
      ptf=libGAP_ADDR_PPERM4(f);
      ptg=libGAP_ADDR_PPERM4(libGAP_g);

      //multiply
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptg[j]!=0){
          img=libGAP_IMAGEPP(ptf[j], ptg, deg);
          if(img!=0){
            ptconj[ptg[j]-1]=img;
            if(img>codec) codec=img;
          }
        }
      }
    }
  }
  libGAP_CODEG_PPERM4(conj)=codec;
  return conj;
}

// f*p^-1
libGAP_Obj libGAP_QuoPPerm2Perm2(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt2   *ptf, *ptp, *ptquo2;
  libGAP_UInt4   *ptquo4, *pttmp;
  libGAP_Obj     quo, dom;
  libGAP_UInt    codeg, lmp, deg, i, j, rank;

  if(libGAP_DEG_PPERM2(f)==0) return libGAP_EmptyPartialPerm;
  
  //find the largest moved point
  lmp=libGAP_DEG_PERM2(p);
  ptp=libGAP_ADDR_PERM2(p);
  while(lmp>0&&ptp[lmp-1]==lmp-1) lmp--;
  if(lmp==0) return f;

  //invert the permutation into the buffer bag
  libGAP_ResizeTmpPPerm(lmp); 
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptp=libGAP_ADDR_PERM2(p);
  for(i=0;i<lmp;i++) pttmp[*ptp++]=i; 

  //create new pperm
  deg=libGAP_DEG_PPERM2(f);
  codeg=libGAP_CODEG_PPERM2(f);
  
  //multiply the partial perm with the inverse
  if(lmp<65536){
    quo=libGAP_NEW_PPERM2(deg);
    ptf=libGAP_ADDR_PPERM2(f);
    pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
    ptquo2=libGAP_ADDR_PPERM2(quo);
    if(codeg<=lmp){
      codeg=0;
      if(libGAP_DOM_PPERM(f)==NULL){
        //Pr("Case 2\n", 0L, 0L);
        for(i=0;i<deg;i++){ 
          if(ptf[i]!=0){
            ptquo2[i]=pttmp[ptf[i]-1]+1;
            if(ptquo2[i]>codeg) codeg=ptquo2[i];
          }
        }
      } else {
        //Pr("Case 1\n", 0L, 0L);
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM2(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
          ptquo2[j]=pttmp[ptf[j]-1]+1;
          if(ptquo2[j]>codeg) codeg=ptquo2[j];
        }
      }
    } else {
      if(libGAP_DOM_PPERM(f)==NULL){
       //Pr("Case 4\n", 0L, 0L);
        for(i=0;i<deg;i++){
          if(ptf[i]!=0){
            ptquo2[i]=libGAP_IMAGE(ptf[i]-1, pttmp, lmp)+1;
          }
        }
      }else{
        //Pr("Case 3\n", 0L, 0L);
        dom=libGAP_DOM_PPERM(f);
        rank=libGAP_RANK_PPERM2(f);
        for(i=1;i<=rank;i++){
          j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
          ptquo2[j]=libGAP_IMAGE(ptf[j]-1, pttmp, lmp)+1;
        }
      }
    }
    libGAP_CODEG_PPERM2(quo)=codeg;
  } else {
    quo=libGAP_NEW_PPERM4(deg);
    ptquo4=libGAP_ADDR_PPERM4(quo);
    ptf=libGAP_ADDR_PPERM2(f);
    pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
    ptquo4=libGAP_ADDR_PPERM4(quo);
    codeg=0;
    if(libGAP_DOM_PPERM(f)==NULL){
      //Pr("Case 6\n", 0L, 0L);
      for(i=0;i<deg;i++){
        if(ptf[i]!=0){
          ptquo4[i]=pttmp[ptf[i]-1]+1;
          if(ptquo4[i]>codeg) codeg=ptquo4[i];
        }
      }
    } else {
      //Pr("Case 5\n", 0L, 0L);
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM2(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptquo4[j]=pttmp[ptf[j]-1]+1;
        if(ptquo4[j]>codeg) codeg=ptquo4[j];
      }
    }
    libGAP_CODEG_PPERM4(quo)=codeg;
  }
  return quo;
}

libGAP_Obj libGAP_QuoPPerm4Perm4(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt4   *ptf, *ptp, *ptquo, *pttmp;
  libGAP_Obj     quo, dom;
  libGAP_UInt    codeg, lmp, deg, i, j, rank;

  if(libGAP_DEG_PPERM4(f)==0) return libGAP_EmptyPartialPerm;
  
  //find the largest moved point
  lmp=libGAP_DEG_PERM4(p);
  ptp=libGAP_ADDR_PERM4(p);
  while(lmp>0&&ptp[lmp-1]==lmp-1) lmp--;
  if(lmp==0) return f;

  //invert the permutation into the buffer bag
  libGAP_ResizeTmpPPerm(lmp);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptp=libGAP_ADDR_PERM4(p);
  for(i=0;i<lmp;i++) pttmp[*ptp++]=i; 

  //create new pperm
  deg=libGAP_DEG_PPERM4(f);
  codeg=libGAP_CODEG_PPERM4(f);
  quo=libGAP_NEW_PPERM4(deg);
  
  //renew pointers  
  ptf=libGAP_ADDR_PPERM4(f);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptquo=libGAP_ADDR_PPERM4(quo);

  //multiply the partial perm with the inverse
  if(codeg<=lmp){
    codeg=0;
    if(libGAP_DOM_PPERM(f)==NULL){
      //Pr("case 1\n", 0L, 0L);
      for(i=0;i<deg;i++){ 
        if(ptf[i]!=0){
          ptquo[i]=pttmp[ptf[i]-1]+1;
          if(ptquo[i]>codeg) codeg=ptquo[i];
        }
      }
    } else {
      //Pr("case 2\n", 0L, 0L);
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptquo[j]=pttmp[ptf[j]-1]+1;
        if(ptquo[j]>codeg) codeg=ptquo[j];
      }
    }
  } else {
    if(libGAP_DOM_PPERM(f)==NULL){
      //Pr("case 3\n", 0L, 0L);
      for(i=0;i<deg;i++){
        if(ptf[i]!=0){
          ptquo[i]=libGAP_IMAGE(ptf[i]-1, pttmp, lmp)+1;
        }
      }
    }else{
      //Pr("case 4\n", 0L, 0L);
      dom=libGAP_DOM_PPERM(f);
      rank=libGAP_RANK_PPERM4(f);
      for(i=1;i<=rank;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptquo[j]=libGAP_IMAGE(ptf[j]-1, pttmp, lmp)+1;
      }
    }
  }
  libGAP_CODEG_PPERM4(quo)=codeg;
  return quo;
}

libGAP_Obj libGAP_QuoPPerm2Perm4(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt4   *ptp, *ptquo, *pttmp;
  libGAP_UInt2   *ptf;
  libGAP_Obj     quo, dom;
  libGAP_UInt    codeg, lmp, deg, i, j, rank;

  if(libGAP_DEG_PPERM2(f)==0) return libGAP_EmptyPartialPerm;
  
  //find the largest moved point
  lmp=libGAP_DEG_PERM4(p);
  ptp=libGAP_ADDR_PERM4(p);
  while(lmp>0&&ptp[lmp-1]==lmp-1) lmp--;
  if(lmp==0) return f;

  //invert the permutation into the buffer bag
  libGAP_ResizeTmpPPerm(lmp);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptp=libGAP_ADDR_PERM4(p);
  for(i=0;i<lmp;i++) pttmp[*ptp++]=i; 

  //create new pperm
  deg=libGAP_DEG_PPERM2(f);
  quo=libGAP_NEW_PPERM4(deg);
  
  //renew pointers  
  ptf=libGAP_ADDR_PPERM2(f);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptquo=libGAP_ADDR_PPERM4(quo);
  
  //multiply the partial perm with the inverse
  codeg=0;
  if(libGAP_DOM_PPERM(f)==NULL){
    for(i=0;i<deg;i++){
      if(ptf[i]!=0){
        ptquo[i]=pttmp[ptf[i]-1]+1;
        if(ptquo[i]>codeg) codeg=ptquo[i];
      }
    }
  } else {
    dom=libGAP_DOM_PPERM(f);
    rank=libGAP_RANK_PPERM2(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptquo[j]=pttmp[ptf[j]-1]+1;
      if(ptquo[j]>codeg) codeg=ptquo[j];
    }
  }
  libGAP_CODEG_PPERM4(quo)=codeg;
  return quo;
}

libGAP_Obj libGAP_QuoPPerm4Perm2( libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt4   *ptf, *ptquo, *pttmp;
  libGAP_UInt2   *ptp;
  libGAP_Obj     quo, dom;
  libGAP_UInt    lmp, deg, i, j, rank;

  if(libGAP_DEG_PPERM4(f)==0) return libGAP_EmptyPartialPerm;
  
  //find the largest moved point
  lmp=libGAP_DEG_PERM2(p);
  ptp=libGAP_ADDR_PERM2(p);
  while(lmp>0&&ptp[lmp-1]==lmp-1) lmp--;
  if(lmp==0) return f;

  //invert the permutation into the buffer bag
  libGAP_ResizeTmpPPerm(lmp);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptp=libGAP_ADDR_PERM2(p);
  for(i=0;i<lmp;i++) pttmp[*ptp++]=i; 

  //create new pperm
  deg=libGAP_DEG_PPERM4(f);
  quo=libGAP_NEW_PPERM4(deg);
  
  //renew pointers  
  ptf=libGAP_ADDR_PPERM4(f);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  ptquo=libGAP_ADDR_PPERM4(quo);
  
  //multiply the partial perm with the inverse
  if(libGAP_DOM_PPERM(f)==NULL){
    //Pr("case 1\n", 0L, 0L);
    for(i=0;i<deg;i++){
      if(ptf[i]!=0){
        ptquo[i]=libGAP_IMAGE(ptf[i]-1, pttmp, lmp)+1;
      }
    }
  }else{
    //Pr("case 2\n", 0L, 0L);
    dom=libGAP_DOM_PPERM(f);
    rank=libGAP_RANK_PPERM4(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      ptquo[j]=libGAP_IMAGE(ptf[j]-1, pttmp, lmp)+1;
    }
  }
  libGAP_CODEG_PPERM4(quo)=libGAP_CODEG_PPERM4(f);
  return quo;
}

// f*g^-1 for partial perms
libGAP_Obj libGAP_QuoPPerm22(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, i, j, deginv, codeg, rank; 
  libGAP_UInt2   *ptf, *ptg;
  libGAP_UInt4   *ptquo, *pttmp;
  libGAP_Obj     quo, dom;

  //do nothing in the trivial case 
  if(libGAP_DEG_PPERM2(libGAP_g)==0||libGAP_DEG_PPERM2(f)==0) return libGAP_EmptyPartialPerm;

  //init the buffer bag
  deginv=libGAP_CODEG_PPERM2(libGAP_g);
  libGAP_ResizeTmpPPerm(deginv);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<deginv;i++) pttmp[i]=0;

  //invert g into the buffer bag
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  if(libGAP_DOM_PPERM(libGAP_g)==NULL){
    deg=libGAP_DEG_PPERM2(libGAP_g);
    for(i=0;i<deg;i++) if(ptg[i]!=0) pttmp[ptg[i]-1]=i+1; 
  } else {
    dom=libGAP_DOM_PPERM(libGAP_g);
    rank=libGAP_RANK_PPERM2(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      pttmp[ptg[j]-1]=j+1;
    }
  }

  // find the degree of the quotient
  deg=libGAP_DEG_PPERM2(f); 
  ptf=libGAP_ADDR_PPERM2(f);
  while(deg>0&&(ptf[deg-1]==0||libGAP_IMAGEPP(ptf[deg-1],pttmp,deginv)==0)) deg--;
  if(deg==0) return libGAP_EmptyPartialPerm;

  // create new pperm
  quo=libGAP_NEW_PPERM4(deg);
  ptquo=libGAP_ADDR_PPERM4(quo);
  ptf=libGAP_ADDR_PPERM2(f);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  codeg=0;
 
  // compose f with g^-1 in rank operations
  if(libGAP_DOM_PPERM(f)!=NULL){
    dom=libGAP_DOM_PPERM(f);
    rank=libGAP_RANK_PPERM2(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]<=deginv){ 
        ptquo[j]=pttmp[ptf[j]-1];
        if(ptquo[j]>codeg) codeg=ptquo[j];
      }
    }
  } else { 
  // compose f with g^-1 in deg operations
    for(i=0;i<deg;i++){
      if(ptf[i]!=0&&ptf[i]<=deginv){
        ptquo[i]=pttmp[ptf[i]-1];
        if(ptquo[i]>codeg) codeg=ptquo[i];
      }
    }
  }
  libGAP_CODEG_PPERM2(quo)=codeg;
  return quo;
}

libGAP_Obj libGAP_QuoPPerm24(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, i, j, deginv, codeg, rank; 
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *ptg, *ptquo, *pttmp;
  libGAP_Obj     quo, dom;

  //do nothing in the trivial case 
  if(libGAP_DEG_PPERM4(libGAP_g)==0||libGAP_DEG_PPERM2(f)==0) return libGAP_EmptyPartialPerm;

  //init the buffer bag
  deginv=libGAP_CODEG_PPERM4(libGAP_g);
  libGAP_ResizeTmpPPerm(deginv);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<deginv;i++) pttmp[i]=0;

  //invert g into the buffer bag
  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  if(libGAP_DOM_PPERM(libGAP_g)==NULL){
    deg=libGAP_DEG_PPERM4(libGAP_g);
    for(i=0;i<deg;i++) if(ptg[i]!=0) pttmp[ptg[i]-1]=i+1; 
  } else {
    dom=libGAP_DOM_PPERM(libGAP_g);
    rank=libGAP_RANK_PPERM4(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      pttmp[ptg[j]-1]=j+1;
    }
  }

  // find the degree of the quotient
  deg=libGAP_DEG_PPERM2(f);
  ptf=libGAP_ADDR_PPERM2(f);
  if(libGAP_CODEG_PPERM2(f)<=deginv){
    while(deg>0&&(ptf[deg-1]==0||pttmp[ptf[deg-1]-1]==0)) deg--;
  } else {
    while(deg>0&&(ptf[deg-1]==0||libGAP_IMAGEPP(ptf[deg-1],pttmp,deginv)==0)) deg--;
  }

  if(deg==0) return libGAP_EmptyPartialPerm;

  // create new pperm
  quo=libGAP_NEW_PPERM4(deg);
  ptquo=libGAP_ADDR_PPERM4(quo);
  ptf=libGAP_ADDR_PPERM2(f);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  codeg=0;

  // compose f with g^-1 in rank operations
  if(libGAP_DOM_PPERM(f)!=NULL){
    dom=libGAP_DOM_PPERM(f); 
    rank=libGAP_RANK_PPERM2(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]<=deginv){ 
        ptquo[j]=pttmp[ptf[j]-1];
        if(ptquo[j]>codeg) codeg=ptquo[j];
      }
    }
  } else { 
  // compose f with g^-1 in deg operations
    for(i=0;i<deg;i++){
      if(ptf[i]!=0&&ptf[i]<=deginv){
        ptquo[i]=pttmp[ptf[i]-1];
        if(ptquo[i]>codeg) codeg=ptquo[i];
      }
    }
  }
  libGAP_CODEG_PPERM4(quo)=codeg;
  return quo;
}

libGAP_Obj libGAP_QuoPPerm42(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, i, j, deginv, codeg, rank; 
  libGAP_UInt2   *ptg;
  libGAP_UInt4   *ptf, *ptquo, *pttmp;
  libGAP_Obj     quo, dom;

  //do nothing in the trivial case 
  if(libGAP_DEG_PPERM2(libGAP_g)==0||libGAP_DEG_PPERM4(f)==0) return libGAP_EmptyPartialPerm;

  //init the buffer bag
  deginv=libGAP_CODEG_PPERM2(libGAP_g);
  libGAP_ResizeTmpPPerm(deginv);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<deginv;i++) pttmp[i]=0;

  //invert g into the buffer bag
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  if(libGAP_DOM_PPERM(libGAP_g)==NULL){
    deg=libGAP_DEG_PPERM2(libGAP_g);
    for(i=0;i<deg;i++) if(ptg[i]!=0) pttmp[ptg[i]-1]=i+1; 
  } else {
    dom=libGAP_DOM_PPERM(libGAP_g);
    rank=libGAP_RANK_PPERM2(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      pttmp[ptg[j]-1]=j+1;
    }
  }

  // find the degree of the quotient
  deg=libGAP_DEG_PPERM4(f);
  ptf=libGAP_ADDR_PPERM4(f);
  if(libGAP_CODEG_PPERM4(f)<=deginv){
    while(deg>0&&(ptf[deg-1]==0||pttmp[ptf[deg-1]-1]==0)) deg--;
  } else {
    while(deg>0&&(ptf[deg-1]==0||libGAP_IMAGEPP(ptf[deg-1],pttmp,deginv)==0)) deg--;
  }

  if(deg==0) return libGAP_EmptyPartialPerm;

  // create new pperm
  quo=libGAP_NEW_PPERM4(deg);
  ptquo=libGAP_ADDR_PPERM4(quo);
  ptf=libGAP_ADDR_PPERM4(f);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  codeg=0;

  // compose f with g^-1 in rank operations
  if(libGAP_DOM_PPERM(f)!=NULL){
    dom=libGAP_DOM_PPERM(f); 
    rank=libGAP_RANK_PPERM4(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]<=deginv){ 
        ptquo[j]=pttmp[ptf[j]-1];
        if(ptquo[j]>codeg) codeg=ptquo[j];
      }
    }
  } else { 
  // compose f with g^-1 in deg operations
    for(i=0;i<deg;i++){
      if(ptf[i]!=0&&ptf[i]<=deginv){
        ptquo[i]=pttmp[ptf[i]-1];
        if(ptquo[i]>codeg) codeg=ptquo[i];
      }
    }
  }
  libGAP_CODEG_PPERM4(quo)=codeg;
  return quo;
}

libGAP_Obj libGAP_QuoPPerm44(libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    deg, i, j, deginv, codeg, rank; 
  libGAP_UInt4   *ptf, *ptg, *ptquo, *pttmp;
  libGAP_Obj     quo, dom;

  //do nothing in the trivial case 
  if(libGAP_DEG_PPERM4(libGAP_g)==0||libGAP_DEG_PPERM4(f)==0) return libGAP_EmptyPartialPerm;

  //init the buffer bag
  deginv=libGAP_CODEG_PPERM4(libGAP_g);
  libGAP_ResizeTmpPPerm(deginv);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  for(i=0;i<deginv;i++) pttmp[i]=0;

  //invert g into the buffer bag
  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  if(libGAP_DOM_PPERM(libGAP_g)==NULL){
    deg=libGAP_DEG_PPERM4(libGAP_g);
    for(i=0;i<deg;i++) if(ptg[i]!=0) pttmp[ptg[i]-1]=i+1; 
  } else {
    dom=libGAP_DOM_PPERM(libGAP_g);
    rank=libGAP_RANK_PPERM4(libGAP_g);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      pttmp[ptg[j]-1]=j+1;
    }
  }

  // find the degree of the quotient
  deg=libGAP_DEG_PPERM4(f);
  ptf=libGAP_ADDR_PPERM4(f);
  while(deg>0&&(ptf[deg-1]==0||libGAP_IMAGEPP(ptf[deg-1],pttmp,deginv)==0)) deg--;
  if(deg==0) return libGAP_EmptyPartialPerm;

  // create new pperm
  quo=libGAP_NEW_PPERM4(deg);
  ptquo=libGAP_ADDR_PPERM4(quo);
  ptf=libGAP_ADDR_PPERM4(f);
  pttmp=((libGAP_UInt4*)libGAP_ADDR_OBJ(libGAP_TmpPPerm));
  codeg=0;
 
  // compose f with g^-1 in rank operations
  if(libGAP_DOM_PPERM(f)!=NULL){
    dom=libGAP_DOM_PPERM(f);
    rank=libGAP_RANK_PPERM4(f);
    for(i=1;i<=rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]<=deginv){ 
        ptquo[j]=pttmp[ptf[j]-1];
        if(ptquo[j]>codeg) codeg=ptquo[j];
      }
    }
  } else { 
  // compose f with g^-1 in deg operations
    for(i=0;i<deg;i++){
      if(ptf[i]!=0&&ptf[i]<=deginv){
        ptquo[i]=pttmp[ptf[i]-1];
        if(ptquo[i]>codeg) codeg=ptquo[i];
      }
    }
  }
  libGAP_CODEG_PPERM4(quo)=codeg;
  return quo;
}

// i^f 
libGAP_Obj libGAP_PowIntPPerm2(libGAP_Obj i, libGAP_Obj f){

  if(libGAP_TNUM_OBJ(i)!=libGAP_T_INT||libGAP_INT_INTOBJ(i)<=0){
    libGAP_ErrorQuit("usage: the first argument should be a positive integer,", 0L, 0L);
    return 0L;
  }
  return libGAP_INTOBJ_INT(libGAP_IMAGEPP((libGAP_UInt) libGAP_INT_INTOBJ(i), libGAP_ADDR_PPERM2(f), libGAP_DEG_PPERM2(f)));
}

libGAP_Obj libGAP_PowIntPPerm4(libGAP_Obj i, libGAP_Obj f){

  if(libGAP_TNUM_OBJ(i)!=libGAP_T_INT||libGAP_INT_INTOBJ(i)<=0){
    libGAP_ErrorQuit("usage: the first argument should be a positive integer,", 0L, 0L);
    return 0L;
  }
  return libGAP_INTOBJ_INT(libGAP_IMAGEPP((libGAP_UInt) libGAP_INT_INTOBJ(i), libGAP_ADDR_PPERM4(f), libGAP_DEG_PPERM4(f)));
}

// p^-1*f
libGAP_Obj libGAP_LQuoPerm2PPerm2( libGAP_Obj p, libGAP_Obj f ){
  libGAP_UInt2   *ptp, *ptf, *ptlquo, dep;
  libGAP_UInt    def, i, j, del, len;
  libGAP_Obj     dom, lquo;

  def=libGAP_DEG_PPERM2(f);
  if(def==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM2(p);
  dom=libGAP_DOM_PPERM(f);

  if(dep<def){
    lquo=libGAP_NEW_PPERM2(def);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM2(f);
    if(dom==NULL){
      for(i=0;i<dep;i++) ptlquo[ptp[i]]=ptf[i];
      for(;i<def;i++) ptlquo[i]=ptf[i];
    } else {
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[libGAP_IMAGE(j, ptp, dep)]=ptf[j];
      }
    }
  } else { //deg(p)>=deg(f)
    del=0;
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM2(f);
    if(dom==NULL){
      //find the degree
      for(i=0;i<def;i++){
        if(ptf[i]!=0&&ptp[i]>=del){
          del=ptp[i]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM2(del);
      ptlquo=libGAP_ADDR_PPERM2(lquo);
      ptp=libGAP_ADDR_PERM2(p);
      ptf=libGAP_ADDR_PPERM2(f);

      // if required below in case ptp[i]>del but ptf[i]=0
      for(i=0;i<def;i++) if(ptf[i]!=0) ptlquo[ptp[i]]=ptf[i];
    } else {//dom(f) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptp[j]>=del){
          del=ptp[j]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM2(del);
      ptlquo=libGAP_ADDR_PPERM2(lquo);
      ptp=libGAP_ADDR_PERM2(p);
      ptf=libGAP_ADDR_PPERM2(f);

      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[ptp[j]]=ptf[j];
      }
    }
  }

  libGAP_CODEG_PPERM2(lquo)=libGAP_CODEG_PPERM2(f);
  return lquo;
}

libGAP_Obj libGAP_LQuoPerm2PPerm4( libGAP_Obj p, libGAP_Obj f ){
  libGAP_UInt2   *ptp, dep;
  libGAP_UInt4   *ptf, *ptlquo;
  libGAP_UInt    def, i, j, del, len;
  libGAP_Obj     dom, lquo;

  def=libGAP_DEG_PPERM4(f);
  if(def==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM2(p);
  dom=libGAP_DOM_PPERM(f);

  if(dep<def){
    lquo=libGAP_NEW_PPERM4(def);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM4(f);
    if(dom==NULL){
      for(i=0;i<dep;i++) ptlquo[ptp[i]]=ptf[i];
      for(;i<def;i++) ptlquo[i]=ptf[i];
    } else {
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[libGAP_IMAGE(j, ptp, dep)]=ptf[j];
      }
    }
  } else { //deg(p)>=deg(f)
    del=0;
    ptp=libGAP_ADDR_PERM2(p);
    ptf=libGAP_ADDR_PPERM4(f);
    if(dom==NULL){
      //find the degree
      for(i=0;i<def;i++){
        if(ptf[i]!=0&&ptp[i]>=del){
          del=ptp[i]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM4(del);
      ptlquo=libGAP_ADDR_PPERM4(lquo);
      ptp=libGAP_ADDR_PERM2(p);
      ptf=libGAP_ADDR_PPERM4(f);

      // if required below in case ptp[i]>del but ptf[i]=0
      for(i=0;i<def;i++) if(ptf[i]!=0) ptlquo[ptp[i]]=ptf[i];
    } else {//dom(f) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptp[j]>=del){
          del=ptp[j]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM4(del);
      ptlquo=libGAP_ADDR_PPERM4(lquo);
      ptp=libGAP_ADDR_PERM2(p);
      ptf=libGAP_ADDR_PPERM4(f);

      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[ptp[j]]=ptf[j];
      }
    }
  }

  libGAP_CODEG_PPERM4(lquo)=libGAP_CODEG_PPERM4(f);
  return lquo;
}

libGAP_Obj libGAP_LQuoPerm4PPerm2( libGAP_Obj p, libGAP_Obj f ){
  libGAP_UInt4   *ptp, dep;
  libGAP_UInt2   *ptf, *ptlquo;
  libGAP_UInt    def, i, j, del, len;
  libGAP_Obj     dom, lquo;

  def=libGAP_DEG_PPERM2(f);
  if(def==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM4(p);
  dom=libGAP_DOM_PPERM(f);

  if(dep<def){
    lquo=libGAP_NEW_PPERM2(def);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM2(f);
    if(dom==NULL){
      for(i=0;i<dep;i++) ptlquo[ptp[i]]=ptf[i];
      for(;i<def;i++) ptlquo[i]=ptf[i];
    } else {
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[libGAP_IMAGE(j, ptp, dep)]=ptf[j];
      }
    }
  } else { //deg(p)>=deg(f)
    del=0;
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM2(f);
    if(dom==NULL){
      //find the degree
      for(i=0;i<def;i++){
        if(ptf[i]!=0&&ptp[i]>=del){
          del=ptp[i]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM2(del);
      ptlquo=libGAP_ADDR_PPERM2(lquo);
      ptp=libGAP_ADDR_PERM4(p);
      ptf=libGAP_ADDR_PPERM2(f);

      // if required below in case ptp[i]>del but ptf[i]=0
      for(i=0;i<def;i++) if(ptf[i]!=0) ptlquo[ptp[i]]=ptf[i];
    } else {//dom(f) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptp[j]>=del){
          del=ptp[j]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM2(del);
      ptlquo=libGAP_ADDR_PPERM2(lquo);
      ptp=libGAP_ADDR_PERM4(p);
      ptf=libGAP_ADDR_PPERM2(f);

      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[ptp[j]]=ptf[j];
      }
    }
  }

  libGAP_CODEG_PPERM2(lquo)=libGAP_CODEG_PPERM2(f);
  return lquo;
}

libGAP_Obj libGAP_LQuoPerm4PPerm4( libGAP_Obj p, libGAP_Obj f ){
  libGAP_UInt4   *ptp, *ptf, *ptlquo, dep;
  libGAP_UInt    def, i, j, del, len;
  libGAP_Obj     dom, lquo;

  def=libGAP_DEG_PPERM4(f);
  if(def==0) return libGAP_EmptyPartialPerm;
  
  dep=libGAP_DEG_PERM4(p);
  dom=libGAP_DOM_PPERM(f);

  if(dep<def){
    lquo=libGAP_NEW_PPERM4(def);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM4(f);
    if(dom==NULL){
      for(i=0;i<dep;i++) ptlquo[ptp[i]]=ptf[i];
      for(;i<def;i++) ptlquo[i]=ptf[i];
    } else {
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[libGAP_IMAGE(j, ptp, dep)]=ptf[j];
      }
    }
  } else { //deg(p)>=deg(f)
    del=0;
    ptp=libGAP_ADDR_PERM4(p);
    ptf=libGAP_ADDR_PPERM4(f);
    if(dom==NULL){
      //find the degree
      for(i=0;i<def;i++){
        if(ptf[i]!=0&&ptp[i]>=del){
          del=ptp[i]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM4(del);
      ptlquo=libGAP_ADDR_PPERM4(lquo);
      ptp=libGAP_ADDR_PERM4(p);
      ptf=libGAP_ADDR_PPERM4(f);

      // if required below in case ptp[i]>del but ptf[i]=0
      for(i=0;i<def;i++) if(ptf[i]!=0) ptlquo[ptp[i]]=ptf[i];
    } else {//dom(f) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptp[j]>=del){
          del=ptp[j]+1;
          if(del==dep) break;
        }
      }
      lquo=libGAP_NEW_PPERM4(del);
      ptlquo=libGAP_ADDR_PPERM4(lquo);
      ptp=libGAP_ADDR_PERM4(p);
      ptf=libGAP_ADDR_PPERM4(f);

      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        ptlquo[ptp[j]]=ptf[j];
      }
    }
  }

  libGAP_CODEG_PPERM4(lquo)=libGAP_CODEG_PPERM4(f);
  return lquo;
}

// f^-1*g
libGAP_Obj libGAP_LQuoPPerm22( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt2   *ptg, *ptf, *ptlquo;
  libGAP_UInt    i, j, def, deg, del, codef, codel, min, len;
  libGAP_Obj     dom, lquo;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM2(f);
  deg=libGAP_DEG_PPERM2(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  dom=libGAP_DOM_PPERM(libGAP_g);
  del=0;
  codef=libGAP_CODEG_PPERM2(f); 
  codel=0;
  
  if(dom==NULL){
    //find the degree of lquo
    min=libGAP_MIN(def,deg);
    for(i=0;i<min;i++){
      if(ptg[i]!=0&&ptf[i]>del){
        del=ptf[i];
        if(del==codef) break;
      }
    }
    if(del==0) return libGAP_EmptyPartialPerm;
    
    //create new pperm
    lquo=libGAP_NEW_PPERM2(del);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptf=libGAP_ADDR_PPERM2(f);
    ptg=libGAP_ADDR_PPERM2(libGAP_g);

    //multiply
    for(i=0;i<min;i++){
      if(ptf[i]!=0&&ptg[i]!=0){
        ptlquo[ptf[i]-1]=ptg[i];
        if(ptg[i]>codel) codel=ptg[i];
      }
    }
  } else if(deg>def){ //dom(g) is known
    //find the degree of lquo
    len=libGAP_LEN_PLIST(dom);
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)>del){
        del=ptf[j];
        if(del==codef) break;
      }
    }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM2(del);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptf=libGAP_ADDR_PPERM2(f);
    ptg=libGAP_ADDR_PPERM2(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  } else { // deg<=def and dom(g) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]>del){
        del=ptf[j];
        if(del==codef) break;
      }
    }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM2(del);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptf=libGAP_ADDR_PPERM2(f);
    ptg=libGAP_ADDR_PPERM2(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  }
  libGAP_CODEG_PPERM2(lquo)=codel;
  return lquo;
}

libGAP_Obj libGAP_LQuoPPerm24( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt4   *ptg, *ptlquo;
  libGAP_UInt2   *ptf;
  libGAP_UInt    i, j, def, deg, del, codef, codel, min, len;
  libGAP_Obj     dom, lquo;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM2(f);
  deg=libGAP_DEG_PPERM4(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM2(f);
  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  dom=libGAP_DOM_PPERM(libGAP_g);
  del=0;
  codef=libGAP_CODEG_PPERM2(f); 
  codel=0;
  
  if(dom==NULL){
    //find the degree of lquo
    min=libGAP_MIN(def,deg);
    for(i=0;i<min;i++){
      if(ptg[i]!=0&&ptf[i]>del){
        del=ptf[i];
        if(del==codef) break;
      }
    }
    if(del==0) return libGAP_EmptyPartialPerm;
    
    //create new pperm
    lquo=libGAP_NEW_PPERM4(del);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptf=libGAP_ADDR_PPERM2(f);
    ptg=libGAP_ADDR_PPERM4(libGAP_g);

    //multiply
    for(i=0;i<min;i++){
      if(ptf[i]!=0&&ptg[i]!=0){
        ptlquo[ptf[i]-1]=ptg[i];
        if(ptg[i]>codel) codel=ptg[i];
      }
    }
  } else if(deg>def){ //dom(g) is known
    //find the degree of lquo
    len=libGAP_LEN_PLIST(dom);
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)>del){
        del=ptf[j];
        if(del==codef) break;
      }
    }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM4(del);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptf=libGAP_ADDR_PPERM2(f);
    ptg=libGAP_ADDR_PPERM4(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  } else { // deg<=def and dom(g) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]>del){
        del=ptf[j];
        if(del==codef) break;
      }
    }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM4(del);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptf=libGAP_ADDR_PPERM2(f);
    ptg=libGAP_ADDR_PPERM4(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  }
  libGAP_CODEG_PPERM4(lquo)=codel;
  return lquo;
}

libGAP_Obj libGAP_LQuoPPerm42( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt2   *ptg, *ptlquo;
  libGAP_UInt4   *ptf;
  libGAP_UInt    i, j, def, deg, del, codef, codel, min, len;
  libGAP_Obj     dom, lquo;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM4(f);
  deg=libGAP_DEG_PPERM2(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM4(f);
  ptg=libGAP_ADDR_PPERM2(libGAP_g);
  dom=libGAP_DOM_PPERM(libGAP_g);
  del=0;
  codef=libGAP_CODEG_PPERM4(f); 
  codel=0;
  
  if(dom==NULL){
    //find the degree of lquo
    min=libGAP_MIN(def, deg);
    for(i=0;i<min;i++){
      if(ptg[i]!=0&&ptf[i]>del){
        del=ptf[i];
        if(del==codef) break;
      }
    }
    if(del==0) return libGAP_EmptyPartialPerm;
    
    //create new pperm
    lquo=libGAP_NEW_PPERM2(del);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptf=libGAP_ADDR_PPERM4(f);
    ptg=libGAP_ADDR_PPERM2(libGAP_g);

    //multiply
    for(i=0;i<min;i++){
      if(ptf[i]!=0&&ptg[i]!=0){
        ptlquo[ptf[i]-1]=ptg[i];
        if(ptg[i]>codel) codel=ptg[i];
      }
    }
  } else if(deg>def){ //dom(g) is known
    //find the degree of lquo
    len=libGAP_LEN_PLIST(dom);
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)>del){
        del=ptf[j];
        if(del==codef) break;
      }
    }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM2(del);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptf=libGAP_ADDR_PPERM4(f);
    ptg=libGAP_ADDR_PPERM2(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  } else { // deg<=def and dom(g) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]>del){
        del=ptf[j];
        if(del==codef) break;
      }
    }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM2(del);
    ptlquo=libGAP_ADDR_PPERM2(lquo);
    ptf=libGAP_ADDR_PPERM4(f);
    ptg=libGAP_ADDR_PPERM2(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  }
  libGAP_CODEG_PPERM2(lquo)=codel;
  return lquo;
}

libGAP_Obj libGAP_LQuoPPerm44( libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt4   *ptg, *ptf, *ptlquo;
  libGAP_UInt    i, j, def, deg, del, codef, codel, min, len;
  libGAP_Obj     dom, lquo;

  //check if we're in the trivial case
  def=libGAP_DEG_PPERM4(f);
  deg=libGAP_DEG_PPERM4(libGAP_g);
  if(def==0||deg==0) return libGAP_EmptyPartialPerm;
 
  ptf=libGAP_ADDR_PPERM4(f);
  ptg=libGAP_ADDR_PPERM4(libGAP_g);
  dom=libGAP_DOM_PPERM(libGAP_g);
  del=0;
  codef=libGAP_CODEG_PPERM4(f); 
  codel=0;
  
  if(dom==NULL){
    //find the degree of lquo
    min=libGAP_MIN(def, deg);
    for(i=0;i<min;i++){
      if(ptg[i]!=0&&ptf[i]>del){
        del=ptf[i];
        if(del==codef) break;
      }
    }
    if(del==0) return libGAP_EmptyPartialPerm;
    
    //create new pperm
    lquo=libGAP_NEW_PPERM4(del);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptf=libGAP_ADDR_PPERM4(f);
    ptg=libGAP_ADDR_PPERM4(libGAP_g);

    //multiply
    for(i=0;i<min;i++){
      if(ptf[i]!=0&&ptg[i]!=0){
        ptlquo[ptf[i]-1]=ptg[i];
        if(ptg[i]>codel) codel=ptg[i];
      }
    }
  } else if(deg>def){ //dom(g) is known
    //find the degree of lquo
    len=libGAP_LEN_PLIST(dom);
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)>del){
        del=ptf[j];
        if(del==codef) break;
      }
    }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM4(del);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptf=libGAP_ADDR_PPERM4(f);
    ptg=libGAP_ADDR_PPERM4(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(libGAP_IMAGEPP(j+1, ptf, def)!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  } else { // deg<=def and dom(g) is known
      len=libGAP_LEN_PLIST(dom);
      for(i=1;i<=len;i++){
        j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
        if(ptf[j]>del){
          del=ptf[j];
          if(del==codef) break;
        }
      }
   
    //create new pperm
    lquo=libGAP_NEW_PPERM4(del);
    ptlquo=libGAP_ADDR_PPERM4(lquo);
    ptf=libGAP_ADDR_PPERM4(f);
    ptg=libGAP_ADDR_PPERM4(libGAP_g);

    //multiply
    for(i=1;i<=len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(dom, i))-1;
      if(ptf[j]!=0){
        ptlquo[ptf[j]-1]=ptg[j];
        if(ptg[j]>codel) codel=ptg[j];
      }
    }
  }
  libGAP_CODEG_PPERM4(lquo)=codel;
  return lquo;
}

libGAP_Obj libGAP_OnSetsPPerm (libGAP_Obj set, libGAP_Obj f){
  libGAP_UInt2  *ptf2;
  libGAP_UInt4  *ptf4;
  libGAP_UInt   deg;
  libGAP_Obj    *ptset, *ptres, tmp, res;
  libGAP_UInt   i, isint, k, h, len;
  
  if(libGAP_LEN_LIST(set)==0) return set;

  res=libGAP_NEW_PLIST(libGAP_IS_MUTABLE_PLIST(set)?libGAP_T_PLIST:libGAP_T_PLIST+libGAP_IMMUTABLE,libGAP_LEN_LIST(set));

  /* get the pointer                                                 */
  ptset = libGAP_ADDR_OBJ(set) + libGAP_LEN_LIST(set);
  ptres = libGAP_ADDR_OBJ(res) + 1;
  len=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){   
    ptf2 = libGAP_ADDR_PPERM2(f);
    deg = libGAP_DEG_PPERM2(f);
    /* loop over the entries of the tuple                              */
    isint = 1;
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset-- ) {
      if ( libGAP_TNUM_OBJ( *ptset ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *ptset ) ) {
        k = libGAP_INT_INTOBJ( *ptset );
        if (k<=deg&&ptf2[k-1]!=0){
          tmp = libGAP_INTOBJ_INT( ptf2[k-1] );
          len++;
          *ptres++ = tmp;
        }
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L);
      }
    }
  } else {
    ptf4 = libGAP_ADDR_PPERM4(f);
    deg = libGAP_DEG_PPERM4(f);

    /* loop over the entries of the tuple                              */
    isint = 1;
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset-- ) {
      if ( libGAP_TNUM_OBJ( *ptset ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *ptset ) ) {
        k = libGAP_INT_INTOBJ( *ptset );
        if (k<=deg&&ptf4[k-1]!=0){
          tmp = libGAP_INTOBJ_INT( ptf4[k-1] );
          len++;
          *ptres++ = tmp;
        }
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L);
      }
    }
  }
  //maybe a problem here if the result <res> has length 0, this certainly
  //caused a problem in OnPosIntSetsPPerm...
  libGAP_ResizeBag(res, (len+1)*sizeof(libGAP_Obj));
  libGAP_SET_LEN_PLIST(res, len);

  /* sort the result */
  h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1;
  while ( 0 < h ) {
    for ( i = h+1; i <= len; i++ ) {
      tmp = libGAP_ADDR_OBJ(res)[i];  k = i;
      while ( h < k && ((libGAP_Int)tmp < (libGAP_Int)(libGAP_ADDR_OBJ(res)[k-h])) ) {
        libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h];
        k -= h;
      }
      libGAP_ADDR_OBJ(res)[k] = tmp;
    }
    h = h / 3;
  }

  /* retype if we only have integers */
  if(isint){
    libGAP_RetypeBag( res, libGAP_IS_MUTABLE_PLIST(set) ? libGAP_T_PLIST_CYC_SSORT :
     libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE ); 
  }

  return res;
}

libGAP_Obj libGAP_OnTuplesPPerm (libGAP_Obj tup, libGAP_Obj f){
  libGAP_UInt2  *ptf2;
  libGAP_UInt4  *ptf4;
  libGAP_UInt   deg;
  libGAP_Obj    *pttup, *ptres, res;
  libGAP_UInt   i, k, lentup, len;
  
  if(libGAP_LEN_LIST(tup)==0) return tup;

  res=libGAP_NEW_PLIST(libGAP_IS_MUTABLE_PLIST(tup)?libGAP_T_PLIST_CYC_NSORT:
     libGAP_T_PLIST_CYC_NSORT+libGAP_IMMUTABLE,libGAP_LEN_LIST(tup));
   
  /* get the pointer                                                 */
  pttup = libGAP_ADDR_OBJ(tup) + 1;
  ptres = libGAP_ADDR_OBJ(res) + 1;
  len=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){   
    ptf2 = libGAP_ADDR_PPERM2(f);
    deg = libGAP_DEG_PPERM2(f);
    /* loop over the entries of the tuple                              */
    lentup=libGAP_LEN_LIST(tup);
    for ( i=1 ; i <= lentup; i++, pttup++ ) {
      if ( libGAP_TNUM_OBJ( *pttup ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *pttup ) ) {
        k = libGAP_INT_INTOBJ( *pttup ) ;
        if (k<=deg&&ptf2[k-1]!=0){
          len++;
          *(ptres++) = libGAP_INTOBJ_INT( ptf2[k-1] );
        }
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L);
      }
    }
  } else {
    ptf4 = libGAP_ADDR_PPERM4(f);
    deg = libGAP_DEG_PPERM4(f);
    /* loop over the entries of the tuple                              */
    lentup=libGAP_LEN_LIST(tup);
    for ( i=1 ; i <= lentup; i++, pttup++ ) {
      if ( libGAP_TNUM_OBJ( *pttup ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *pttup ) ) {
        k = libGAP_INT_INTOBJ( *pttup ) ;
        if (k<=deg&&ptf4[k-1]!=0){
          len++;
          *(ptres++) = libGAP_INTOBJ_INT( ptf4[k-1] );
        }
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L);
      }
    }
  }
  libGAP_SET_LEN_PLIST(res, (libGAP_Int) len);
  libGAP_SHRINK_PLIST(res, (libGAP_Int) len);

  return res;
}

libGAP_Obj libGAP_FuncOnPosIntSetsPPerm (libGAP_Obj self, libGAP_Obj set, libGAP_Obj f){
  libGAP_UInt2  *ptf2;
  libGAP_UInt4  *ptf4;
  libGAP_UInt   deg;
  libGAP_Obj    *ptset, *ptres, tmp, res;
  libGAP_UInt   i, k, h, len;
  
  if(libGAP_LEN_LIST(set)==0) return set;
  
  if(libGAP_LEN_LIST(set)==1&&libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, 1))==0){
    return libGAP_FuncIMAGE_SET_PPERM(self, f);
  }

  libGAP_PLAIN_LIST(set);
  res=libGAP_NEW_PLIST(libGAP_IS_MUTABLE_PLIST(set)?libGAP_T_PLIST_CYC_SSORT:libGAP_T_PLIST_CYC_SSORT+libGAP_IMMUTABLE,libGAP_LEN_LIST(set));

  /* get the pointer                                                 */
  ptset = libGAP_ADDR_OBJ(set) + libGAP_LEN_LIST(set);
  ptres = libGAP_ADDR_OBJ(res) + 1;
  len=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_PPERM2){   
    ptf2 = libGAP_ADDR_PPERM2(f);
    deg = libGAP_DEG_PPERM2(f);
    /* loop over the entries of the tuple                              */
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset-- ) {
      k = libGAP_INT_INTOBJ( *ptset );
      if(k<=deg&&ptf2[k-1]!=0){
        tmp = libGAP_INTOBJ_INT( ptf2[k-1] );
        len++;
        *ptres++ = tmp;
      }
    }
  } else {
    ptf4 = libGAP_ADDR_PPERM4(f);
    deg = libGAP_DEG_PPERM4(f);
    /* loop over the entries of the tuple                              */
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset-- ) {
      k = libGAP_INT_INTOBJ( *ptset );
      if (k<=deg&&ptf4[k-1]!=0){
        tmp = libGAP_INTOBJ_INT( ptf4[k-1] );
        len++;
        *ptres++ = tmp;
      }
    }
  }
  libGAP_ResizeBag(res, (len+1)*sizeof(libGAP_Obj));
  libGAP_SET_LEN_PLIST(res, len);

  if(len==0){ 
    libGAP_RetypeBag(res, libGAP_T_PLIST_EMPTY);
    return res;
  }
  
  /* sort the result */
  h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1;
  while ( 0 < h ) {
    for ( i = h+1; i <= len; i++ ) {
      tmp = libGAP_ADDR_OBJ(res)[i];  k = i;
      while ( h < k && ((libGAP_Int)tmp < (libGAP_Int)(libGAP_ADDR_OBJ(res)[k-h])) ) {
        libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h];
        k -= h;
      }
      libGAP_ADDR_OBJ(res)[k] = tmp;
    }
    h = h / 3;
  }

  /* retype if we only have integers */
  return res;
}

/******************************************************************************/
/******************************************************************************/

/* other internal things */

/* so that domain and image set are preserved during garbage collection */
void libGAP_MarkPPermSubBags( libGAP_Obj f ){
  if(libGAP_IMG_PPERM(f)!=NULL){
    libGAP_MARK_BAG(libGAP_IMG_PPERM(f));
    libGAP_MARK_BAG(libGAP_DOM_PPERM(f));
  }
}

/* Save and load */
void libGAP_SavePPerm2( libGAP_Obj f){
  libGAP_UInt2 *ptr;
  libGAP_UInt  len, i;
  len=libGAP_DEG_PPERM2(f);
  ptr=libGAP_ADDR_PPERM2(f)-1;
  for (i = 0; i < len+1; i++) libGAP_SaveUInt2(*ptr++);
}

void libGAP_LoadPPerm2( libGAP_Obj f){
  libGAP_UInt2 *ptr;
  libGAP_UInt  len, i;
  len=libGAP_DEG_PPERM2(f);
  ptr=libGAP_ADDR_PPERM2(f)-1;
  for (i = 0; i < len+1; i++) *ptr++=libGAP_LoadUInt2();
}

void libGAP_SavePPerm4( libGAP_Obj f){
  libGAP_UInt4 *ptr;
  libGAP_UInt  len, i;
  len=libGAP_DEG_PPERM4(f);
  ptr=libGAP_ADDR_PPERM4(f)-1;
  for (i = 0; i < len+1; i++) libGAP_SaveUInt4(*ptr++);
}

void libGAP_LoadPPerm4( libGAP_Obj f){
  libGAP_UInt4 *ptr;
  libGAP_UInt  len, i;
  len=libGAP_DEG_PPERM4(f);
  ptr=libGAP_ADDR_PPERM4(f)-1;
  for (i = 0; i < len+1; i++) *ptr++=libGAP_LoadUInt4();
}

libGAP_Obj libGAP_TYPE_PPERM2;

libGAP_Obj libGAP_TypePPerm2(libGAP_Obj f){
  return libGAP_TYPE_PPERM2;
}

libGAP_Obj libGAP_TYPE_PPERM4;

libGAP_Obj libGAP_TypePPerm4(libGAP_Obj f){
  return libGAP_TYPE_PPERM4;
}

libGAP_Obj libGAP_IsPPermFilt;

libGAP_Obj libGAP_IsPPermHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 val )
{
    /* return 'true' if <val> is a partial perm and 'false' otherwise       */
    switch ( libGAP_TNUM_OBJ(val) ) {

        case libGAP_T_PPERM2:
        case libGAP_T_PPERM4:
            return libGAP_True;

        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            return libGAP_DoFilter( self, val );

        default:
            return libGAP_False;
    }
}

/*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * */

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_PPERM", "obj", &libGAP_IsPPermFilt,
      libGAP_IsPPermHandler, "src/pperm.c:IS_PPERM" },

    { 0 }

};

/******************************************************************************
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

  { "EmptyPartialPerm", 0, "",
     libGAP_FuncEmptyPartialPerm,
    "src/pperm.c:FuncEmptyPartialPerm" },

  { "DensePartialPermNC", 1, "img",
     libGAP_FuncDensePartialPermNC,
    "src/pperm.c:FuncDensePartialPermNC" },

  { "SparsePartialPermNC", 2, "dom, img",
     libGAP_FuncSparsePartialPermNC,
    "src/pperm.c:FuncSparsePartialPermNC" },

  { "DegreeOfPartialPerm", 1, "f",
     libGAP_FuncDegreeOfPartialPerm,
    "src/pperm.c:FuncDegreeOfPartialPerm" },

  { "CoDegreeOfPartialPerm", 1, "f",
     libGAP_FuncCoDegreeOfPartialPerm,
    "src/pperm.c:FuncCoDegreeOfPartialPerm" },
  
  { "RankOfPartialPerm", 1, "f",
     libGAP_FuncRankOfPartialPerm,
    "src/pperm.c:FuncRankOfPartialPerm" },

  { "IMAGE_PPERM", 1, "f",
     libGAP_FuncIMAGE_PPERM,
    "src/pperm.c:FuncIMAGE_PPERM" },

  { "DOMAIN_PPERM", 1, "f",
     libGAP_FuncDOMAIN_PPERM,
    "src/pperm.c:FuncDOMAIN_PPERM" },

  { "IMAGE_SET_PPERM", 1, "f",
     libGAP_FuncIMAGE_SET_PPERM,
    "src/pperm.c:FuncIMAGE_SET_PPERM" },

  { "PREIMAGE_PPERM_INT", 2, "f, i",
     libGAP_FuncPREIMAGE_PPERM_INT,
    "src/pperm.c:FuncPREIMAGE_PPERM_INT" },

  { "INDEX_PERIOD_PPERM", 1, "f",
     libGAP_FuncINDEX_PERIOD_PPERM,
    "src/pperm.c:FuncINDEX_PERIOD_PPERM" },

  { "SMALLEST_IDEM_POW_PPERM", 1, "f",
     libGAP_FuncSMALLEST_IDEM_POW_PPERM,
    "src/pperm.c:FuncSMALLEST_IDEM_POW_PPERM" },
  
  { "COMPONENT_REPS_PPERM", 1, "f",
     libGAP_FuncCOMPONENT_REPS_PPERM,
    "src/pperm.c:FuncCOMPONENT_REPS_PPERM" },
  
  { "NR_COMPONENTS_PPERM", 1, "f",
     libGAP_FuncNR_COMPONENTS_PPERM,
    "src/pperm.c:FuncNR_COMPONENTS_PPERM" },
  
  { "COMPONENTS_PPERM", 1, "f",
     libGAP_FuncCOMPONENTS_PPERM,
    "src/pperm.c:FuncCOMPONENTS_PPERM" },
  
  { "COMPONENT_PPERM_INT", 2, "f, pt",
     libGAP_FuncCOMPONENT_PPERM_INT,
    "src/pperm.c:FuncCOMPONENT_PPERM_INT" },
  
  { "FIXED_PTS_PPERM", 1, "f",
     libGAP_FuncFIXED_PTS_PPERM,
    "src/pperm.c:FuncFIXED_PTS_PPERM" },
  
  { "NR_FIXED_PTS_PPERM", 1, "f",
     libGAP_FuncNR_FIXED_PTS_PPERM,
    "src/pperm.c:FuncNR_FIXED_PTS_PPERM" },
  
  { "MOVED_PTS_PPERM", 1, "f",
     libGAP_FuncMOVED_PTS_PPERM,
    "src/pperm.c:FuncMOVED_PTS_PPERM" },
  
  { "NR_MOVED_PTS_PPERM", 1, "f",
     libGAP_FuncNR_MOVED_PTS_PPERM,
    "src/pperm.c:FuncNR_MOVED_PTS_PPERM" },
  
  { "LARGEST_MOVED_PT_PPERM", 1, "f",
     libGAP_FuncLARGEST_MOVED_PT_PPERM,
    "src/pperm.c:FuncLARGEST_MOVED_PT_PPERM" },
  
  { "SMALLEST_MOVED_PT_PPERM", 1, "f",
     libGAP_FuncSMALLEST_MOVED_PT_PPERM,
    "src/pperm.c:FuncSMALLEST_MOVED_PT_PPERM" },
  
  { "TRIM_PPERM", 1, "f",
     libGAP_FuncTRIM_PPERM,
    "src/pperm.c:FuncTRIM_PPERM" },

  { "HASH_FUNC_FOR_PPERM", 2, "f, data",
     libGAP_FuncHASH_FUNC_FOR_PPERM,
    "src/pperm.c:FuncHASH_FUNC_FOR_PPERM" },
  
  { "IS_IDEM_PPERM", 1, "f",
     libGAP_FuncIS_IDEM_PPERM,
    "src/pperm.c:FuncIS_IDEM_PPERM" },

  { "LEFT_ONE_PPERM", 1, "f",
     libGAP_FuncLEFT_ONE_PPERM,
    "src/pperm.c:FuncLEFT_ONE_PPERM" },

  { "RIGHT_ONE_PPERM", 1, "f",
     libGAP_FuncRIGHT_ONE_PPERM,
    "src/pperm.c:FuncRIGHT_ONE_PPERM" },

  { "NaturalLeqPartialPerm", 2, "f, g",
     libGAP_FuncNaturalLeqPartialPerm,
    "src/pperm.c:FuncNaturalLeqPartialPerm" },

  { "JOIN_PPERMS", 2, "f, g",
     libGAP_FuncJOIN_PPERMS,
    "src/pperm.c:FuncJOIN_PPERMS" },

  { "JOIN_IDEM_PPERMS", 2, "f, g",
     libGAP_FuncJOIN_IDEM_PPERMS,
    "src/pperm.c:FuncJOIN_IDEM_PPERMS" },

  { "MEET_PPERMS", 2, "f, g",
     libGAP_FuncMEET_PPERMS,
    "src/pperm.c:FuncMEET_PPERMS" },

  { "RESTRICTED_PPERM", 2, "f, g",
     libGAP_FuncRESTRICTED_PPERM,
    "src/pperm.c:FuncRESTRICTED_PPERM" },

  { "AS_PPERM_PERM", 2, "p, set",
     libGAP_FuncAS_PPERM_PERM,
    "src/pperm.c:FuncAS_PPERM_PERM" },

  { "AS_PERM_PPERM", 1, "f",
     libGAP_FuncAS_PERM_PPERM,
    "src/pperm.c:FuncAS_PERM_PPERM" },

  { "PERM_LEFT_QUO_PPERM_NC", 2, "f, g",
     libGAP_FuncPERM_LEFT_QUO_PPERM_NC,
    "src/pperm.c:FuncPERM_LEFT_QUO_PPERM_NC" },

  { "ShortLexLeqPartialPerm", 2, "f, g",
     libGAP_FuncShortLexLeqPartialPerm,
    "src/pperm.c:FuncShortLexLeqPartialPerm" },

  { "HAS_DOM_PPERM", 1, "f",
     libGAP_FuncHAS_DOM_PPERM,
    "src/pperm.c:FuncHAS_DOM_PPERM" },

  { "HAS_IMG_PPERM", 1, "f",
     libGAP_FuncHAS_IMG_PPERM,
    "src/pperm.c:FuncHAS_IMG_PPERM" },

  { "OnPosIntSetsPartialPerm", 2, "set, f", 
     libGAP_FuncOnPosIntSetsPPerm, 
     "src/pperm.c:FuncOnPosIntSetsPPerm" },

  { "PrintPPerm2", 1, "f", 
     libGAP_PrintPPerm2, 
     "src/pperm.c:PrintPPerm2" },

  { "PrintPPerm4", 1, "f", 
     libGAP_PrintPPerm4, 
     "src/pperm.c:PrintPPerm4" },

  { 0 }

};
/******************************************************************************
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel ( libGAP_StructInitInfo *libGAP_module )
{

    /* install the marking function                                        */
    libGAP_InfoBags[ libGAP_T_PPERM2 ].name = "partial perm (small)";
    libGAP_InfoBags[ libGAP_T_PPERM4 ].name = "partial perm (large)";
    libGAP_InitMarkFuncBags( libGAP_T_PPERM2, libGAP_MarkPPermSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_PPERM4, libGAP_MarkPPermSubBags );
    
    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_PPERM2", &libGAP_TYPE_PPERM2 );
    libGAP_ImportGVarFromLibrary( "TYPE_PPERM4", &libGAP_TYPE_PPERM4 );

    libGAP_TypeObjFuncs[ libGAP_T_PPERM2 ] = libGAP_TypePPerm2;
    libGAP_TypeObjFuncs[ libGAP_T_PPERM4 ] = libGAP_TypePPerm4;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* make the buffer bag                                                 */
    libGAP_InitGlobalBag( &libGAP_TmpPPerm, "src/pperm.c:TmpPPerm" );
    libGAP_InitGlobalBag( &libGAP_EmptyPartialPerm, "src/pperm.c:EmptyPartialPerm" );

    /* install the saving functions */
    libGAP_SaveObjFuncs[ libGAP_T_PPERM2 ] = libGAP_SavePPerm2;
    libGAP_LoadObjFuncs[ libGAP_T_PPERM2 ] = libGAP_LoadPPerm2;
    libGAP_SaveObjFuncs[ libGAP_T_PPERM4 ] = libGAP_SavePPerm4;
    libGAP_LoadObjFuncs[ libGAP_T_PPERM4 ] = libGAP_LoadPPerm4;
    
    /* install the comparison methods                                      */
    libGAP_EqFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM2  ] = libGAP_EqPPerm22;
    libGAP_EqFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM4  ] = libGAP_EqPPerm44;
    libGAP_EqFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM2  ] = libGAP_EqPPerm42;
    libGAP_EqFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM4  ] = libGAP_EqPPerm24;
    libGAP_LtFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM2  ] = libGAP_LtPPerm22;
    libGAP_LtFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM4  ] = libGAP_LtPPerm44;
    libGAP_LtFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM4  ] = libGAP_LtPPerm24;
    libGAP_LtFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM2  ] = libGAP_LtPPerm42;
    
    /* install the binary operations */
    libGAP_ProdFuncs [ libGAP_T_PPERM2  ][ libGAP_T_PPERM2 ] = libGAP_ProdPPerm22;
    libGAP_ProdFuncs [ libGAP_T_PPERM4  ][ libGAP_T_PPERM2 ] = libGAP_ProdPPerm42;
    libGAP_ProdFuncs [ libGAP_T_PPERM2  ][ libGAP_T_PPERM4 ] = libGAP_ProdPPerm24;
    libGAP_ProdFuncs [ libGAP_T_PPERM4  ][ libGAP_T_PPERM4 ] = libGAP_ProdPPerm44;
    libGAP_ProdFuncs [ libGAP_T_PPERM2  ][ libGAP_T_PERM2  ] = libGAP_ProdPPerm2Perm2;
    libGAP_ProdFuncs [ libGAP_T_PPERM4  ][ libGAP_T_PERM4  ] = libGAP_ProdPPerm4Perm4;
    libGAP_ProdFuncs [ libGAP_T_PPERM2  ][ libGAP_T_PERM4  ] = libGAP_ProdPPerm2Perm4;
    libGAP_ProdFuncs [ libGAP_T_PPERM4  ][ libGAP_T_PERM2  ] = libGAP_ProdPPerm4Perm2;
    libGAP_ProdFuncs [ libGAP_T_PERM2   ][ libGAP_T_PPERM2 ] = libGAP_ProdPerm2PPerm2;
    libGAP_ProdFuncs [ libGAP_T_PERM4   ][ libGAP_T_PPERM4 ] = libGAP_ProdPerm4PPerm4;
    libGAP_ProdFuncs [ libGAP_T_PERM4   ][ libGAP_T_PPERM2 ] = libGAP_ProdPerm4PPerm2;
    libGAP_ProdFuncs [ libGAP_T_PERM2   ][ libGAP_T_PPERM4 ] = libGAP_ProdPerm2PPerm4;
    libGAP_PowFuncs  [ libGAP_T_INT     ][ libGAP_T_PPERM2 ] = libGAP_PowIntPPerm2;
    libGAP_PowFuncs  [ libGAP_T_INT     ][ libGAP_T_PPERM4 ] = libGAP_PowIntPPerm4;
    libGAP_PowFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PERM2  ] = libGAP_PowPPerm2Perm2;
    libGAP_PowFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PERM4  ] = libGAP_PowPPerm2Perm4;
    libGAP_PowFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PERM2  ] = libGAP_PowPPerm4Perm2;
    libGAP_PowFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PERM4  ] = libGAP_PowPPerm4Perm4;
    libGAP_PowFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM2 ] = libGAP_PowPPerm22;
    libGAP_PowFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM4 ] = libGAP_PowPPerm24;
    libGAP_PowFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM2 ] = libGAP_PowPPerm42;
    libGAP_PowFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM4 ] = libGAP_PowPPerm44;
    libGAP_QuoFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PERM2  ] = libGAP_QuoPPerm2Perm2;
    libGAP_QuoFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PERM4  ] = libGAP_QuoPPerm4Perm4;
    libGAP_QuoFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PERM4  ] = libGAP_QuoPPerm2Perm4;
    libGAP_QuoFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PERM2  ] = libGAP_QuoPPerm4Perm2;
    libGAP_QuoFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM2 ] = libGAP_QuoPPerm22;
    libGAP_QuoFuncs  [ libGAP_T_PPERM2  ][ libGAP_T_PPERM4 ] = libGAP_QuoPPerm24;
    libGAP_QuoFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM2 ] = libGAP_QuoPPerm42;
    libGAP_QuoFuncs  [ libGAP_T_PPERM4  ][ libGAP_T_PPERM4 ] = libGAP_QuoPPerm44;
    libGAP_QuoFuncs  [ libGAP_T_INT     ][ libGAP_T_PPERM2 ] = libGAP_PreImagePPermInt;
    libGAP_QuoFuncs  [ libGAP_T_INT     ][ libGAP_T_PPERM4 ] = libGAP_PreImagePPermInt;
    libGAP_LQuoFuncs [ libGAP_T_PERM2   ][ libGAP_T_PPERM2 ] = libGAP_LQuoPerm2PPerm2;
    libGAP_LQuoFuncs [ libGAP_T_PERM2   ][ libGAP_T_PPERM4 ] = libGAP_LQuoPerm2PPerm4;
    libGAP_LQuoFuncs [ libGAP_T_PERM4   ][ libGAP_T_PPERM2 ] = libGAP_LQuoPerm4PPerm2;
    libGAP_LQuoFuncs [ libGAP_T_PERM4   ][ libGAP_T_PPERM4 ] = libGAP_LQuoPerm4PPerm4;
    libGAP_LQuoFuncs [ libGAP_T_PPERM2  ][ libGAP_T_PPERM2 ] = libGAP_LQuoPPerm22;
    libGAP_LQuoFuncs [ libGAP_T_PPERM2  ][ libGAP_T_PPERM4 ] = libGAP_LQuoPPerm24;
    libGAP_LQuoFuncs [ libGAP_T_PPERM4  ][ libGAP_T_PPERM2 ] = libGAP_LQuoPPerm42;
    libGAP_LQuoFuncs [ libGAP_T_PPERM4  ][ libGAP_T_PPERM4 ] = libGAP_LQuoPPerm44;
    
    /* install the zero function for partial perms */
    libGAP_ZeroFuncs [libGAP_T_PPERM2] = libGAP_ZeroPPerm;
    libGAP_ZeroFuncs [libGAP_T_PPERM4] = libGAP_ZeroPPerm;
    libGAP_ZeroMutFuncs [libGAP_T_PPERM2] = libGAP_ZeroPPerm;
    libGAP_ZeroMutFuncs [libGAP_T_PPERM4] = libGAP_ZeroPPerm;
    
    /* install the one function for partial perms */
    libGAP_OneFuncs [libGAP_T_PPERM2] = libGAP_OnePPerm;
    libGAP_OneFuncs [libGAP_T_PPERM4] = libGAP_OnePPerm;
    libGAP_OneMutFuncs [libGAP_T_PPERM2] = libGAP_OnePPerm;
    libGAP_OneMutFuncs [libGAP_T_PPERM4] = libGAP_OnePPerm;

    /* install the inverse functions for partial perms */
    libGAP_InvFuncs[ libGAP_T_PPERM2 ] = libGAP_InvPPerm2;
    libGAP_InvFuncs[ libGAP_T_PPERM4 ] = libGAP_InvPPerm4;
    libGAP_InvMutFuncs[ libGAP_T_PPERM2 ] = libGAP_InvPPerm2;
    libGAP_InvMutFuncs[ libGAP_T_PPERM4 ] = libGAP_InvPPerm4;
    
    /* return success                                                      */
    return 0;
}

/******************************************************************************
 *F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
 */
static libGAP_Int libGAP_InitLibrary ( libGAP_StructInitInfo *libGAP_module )
{
  /* init filters and functions                                          */
  libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );
  libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
  libGAP_TmpPPerm = libGAP_NEW_PPERM4(1000);

  libGAP_EmptyPartialPerm=libGAP_NEW_PPERM2(0);
  
  /* return success                                                      */
  return 0;
}

/****************************************************************************
 **
 *F  InitInfoPPerm()  . . . . . . . . . . . . . . . table of init functions
 */
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "pperm",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0,                                  /* postRestore                    */
    "src/pperm.c",                      /* filename                       */
    1                                   /* isGapRootRelative              */
};

libGAP_StructInitInfo * libGAP_InitInfoPPerm ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}

