/****************************************************************************
**
*W  plist.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions that deal with plain lists.
**
**  A plain list  is represented by  a bag  of type 'T_PLIST',  which has the
**  following format:
**
**      +-------+-------+-------+- - - -+-------+-------+- - - -+-------+
**      |logical| first |second |       |last   |       |       |       |
**      |length |element|element|       |element|   0   |       |   0   |
**      +-------+-------+-------+- - - -+-------+-------+- - - -+-------+
**
**  The first entry is the logical length of the list stored as GAP immediate
**  integer.  The second entry is the identifier of the  first element of the
**  list.  The third  entry is  the identifier  of the second  element of the
**  list, and so on.  If  the physical length  of a list  is greater than the
**  logical, there will be unused entries at the end  of the list, comtaining
**  0.  The physical length  might be greater than  the logical,  because the
**  physical size of a  list is increased by at  least 12.5\%, to avoid doing
**  this too often.
**
**  This representation  is encoded by  the macros 'NEW_PLIST', 'GROW_PLIST',
**  'SHRINK_PLIST', 'SET_LEN_PLIST',    'LEN_PLIST',     'SET_ELM_PLIST', and
**  'ELM_PLIST', which are used by the functions in this package and the rest
**  of the {\GAP} kernel to access plain lists.
**
**  This package also contains the list  functions for plain lists, i.e., the
**  functions called from the generic lists package.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */
#include        "finfield.h"            /* fintie fields                   */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */
#include        "blister.h"             /* boolean lists                   */

#include        "saveload.h"            /* saving and loading              */

#include        <assert.h>
/****************************************************************************
**

*F  GROW_PLIST(<list>,<plen>) . . . .  make sure a plain list is large enough
**
*/
libGAP_Int             libGAP_GrowPlist (
    libGAP_Obj                 list,
    libGAP_UInt                need )
{
    libGAP_UInt                plen;           /* new physical length             */
    libGAP_UInt                good;           /* good new physical length        */

    /* find out how large the plain list should become                     */
    good = 5 * (libGAP_SIZE_OBJ(list)/sizeof(libGAP_Obj)-1) / 4 + 4;

    /* but maybe we need more                                              */
    if ( need < good ) { plen = good; }
    else               { plen = need; }

    /* resize the plain list                                               */
    libGAP_ResizeBag( list, ((plen)+1)*sizeof(libGAP_Obj) );

    /* return something (to please some C compilers)                       */
    return 0L;
}


/****************************************************************************
**
*F  TypePlist(<list>) . . . . . . . . . . . . . . . . .  kind of a plain list
**
**  'TypePlist' returns the kind of the plain list <list>.
**
**  'TypePlist' is the function in 'TypeObjFuncs' for plain lists.
**
**  TypePlist works with KTnumPlist to determine the type of a plain list
**  Considerable care is needed to deal with self-referential lists. This is
**  basically achieved with the TESTING flag in the Tnum. This must be set in
**  the "current" list before triggering determination of the Type (or KTnum)
**  of any sublist.
**
**  KTnumPlist determined the "true" Tnum of the list, taking account of such
**  factors as denseness, homogeneity and so on. It modifies the stored Tnum
**  of the list to the most informative "safe" value, allowing for the
**  mutability of the list entries (and preserving TESTING).
**
**  Here begins a new attempt by Steve to describe how it all works:
**
**  We begin with the TNUMs attached to the objects. They are defined in
**  objects.h and consist of the following, each of which can be qualified by
**  adding the constant IMMUTABLE.
**
**   T_PLIST                    nothing is known
**   T_PLIST_NDENSE             known to have a hole
**   T_PLIST_DENSE              known only not to have a hole
**   T_PLIST_DENSE_NHOM         known to be dense but not homogenous *  **
**   T_PLIST_DENSE_NHOM_SSORT   dense, non-hom but strictly sorted
**   T_PLIST_DENSE_NHOM_NSORT   dense, non-hom, known not be be sorted
**   T_PLIST_EMPTY              the empty list
**   T_PLIST_HOM                known to be homogenous *
**   T_PLIST_HOM_NSORT           etc
**   T_PLIST_HOM_SSORT           etc
**   T_PLIST_TAB                known to be a table  *
**   T_PLIST_TAB_NSORT           etc
**   T_PLIST_TAB_SSORT           etc
**   T_PLIST_TAB_RECT           known to be a rectangular table  *
**   T_PLIST_TAB_RECT_NSORT      etc
**   T_PLIST_TAB_RECT_SSORT      etc
**   T_PLIST_CYC                known to be a list of constant kernel cyclotomics
**   T_PLIST_CYC_NSORT           etc
**   T_PLIST_CYC_SSORT           etc
**   T_PLIST_FFE                known to be a list of kernel FFEs over same field
**
**   * -- these tnums can only be safely given when none of the elements of the list
**        is mutable
**   ** -- dense recursive lists (have themselves as a (possibly nested) subobject)
**         appear here
**
**  There are 10 functions entered in TypeObjFuncs:
**      1. TypePlist
**      2. TypePlistNDenseMut/Imm
**      3. TypePlistDenseMut/Imm
**      4. TypePlistDenseNHomMut/Imm
**      5. TypePlistDenseNHomSSortMut/Imm
**      6. TypePlistDenseNHomNSortMut/Imm
**      7. TypePlistEmptyMut/Imm
**      8. TypePlistHom     -- also handles Tab and RectTab
**      9. TypePlistCyc
**      10.TypePlistFfe
**
**     Of these:
**         3 is actually an alias of 1
**         2,4, 5, 6 and 7  simply return a fixed type
**         Thus 1, 8, 9 and 10 have work to do.
**
**     9 and 10 look up the exact TNUM in a table associated with the element
**        family to find the type, calling out to a GAP function to make each type
**        for the first time.
**
**     1 and 8 now get really complicated. This is because they now have to
**     check properties of the list which may be currently true, but not yet
**     known, and possibly not storable due to the presence of mutable
**     elements in the list. If we didn't do this, a lot of matrix stuff
**     wouldn't work
**
**     8 is the simpler. It calls KTnumHomPlist, which checks whether we
**     should really be in T_PLIST_CYC, T_PLIST_FFE or T_PLIST_TAB and if so,
**     changes the TNUM appropriately and returns the new tnum.  The only
**     time this is slow is a homogenous list of lists which looks like a
**     table until the very last entry which has the wrong length. This
**     should be rare.
**     
**     1 is the real nightmare, because it has to handle recursive mutable
**     lists, lists with mutable subobjects, etc.  We now concentrate on this
**     case.
**
**     The entry point is the function TypePlistWithKTnum, which returns both
**     the type and the ktnum of the list. This must be done in one function
**     to avoid an exponential slowdown for deeply nested lists. This
**     function is mutually recursive with KTnumPlist, which also returns two
**     pieces of information: the ktnum of the list and, if it is homogenous,
**     the family of the elements.
**
**     recursive lists (ie lists which are there own subobjects are detected
**     using the TESTING tnums. Any list being examined must have TESTING added to
**     its tnum BEFORE any element of it is examined.
**
**     
**
*/

libGAP_Obj libGAP_TYPE_LIST_NDENSE_MUTABLE;
libGAP_Obj libGAP_TYPE_LIST_NDENSE_IMMUTABLE;
libGAP_Obj libGAP_TYPE_LIST_DENSE_NHOM_MUTABLE;
libGAP_Obj libGAP_TYPE_LIST_DENSE_NHOM_IMMUTABLE;
libGAP_Obj libGAP_TYPE_LIST_DENSE_NHOM_SSORT_MUTABLE;
libGAP_Obj libGAP_TYPE_LIST_DENSE_NHOM_SSORT_IMMUTABLE;
libGAP_Obj libGAP_TYPE_LIST_DENSE_NHOM_NSORT_MUTABLE;
libGAP_Obj libGAP_TYPE_LIST_DENSE_NHOM_NSORT_IMMUTABLE;
libGAP_Obj libGAP_TYPE_LIST_EMPTY_MUTABLE;
libGAP_Obj libGAP_TYPE_LIST_EMPTY_IMMUTABLE;
libGAP_Obj libGAP_TYPE_LIST_HOM;

#define libGAP_IS_TESTING_PLIST(list) \
    (libGAP_FIRST_TESTING_TNUM <= libGAP_TNUM_OBJ(list) \
  && libGAP_TNUM_OBJ(list) <= libGAP_LAST_TESTING_TNUM)


extern libGAP_Obj libGAP_TypePlistWithKTnum( libGAP_Obj list,
			       libGAP_UInt *ktnum );

libGAP_Int libGAP_KTNumPlist (
    libGAP_Obj                 list,
    libGAP_Obj                 *famfirst)
{
    libGAP_Int                 isHom   = 1;    /* is <list> homogeneous           */
    libGAP_Int                 isDense = 1;    /* is <list> dense                 */
    libGAP_Int                 isTable = 0;    /* are <list>s elms all lists      */
    libGAP_Int                 isRect  = 0;    /* are lists elms of equal length
                                     only test this one for PLIST elements */
    libGAP_Int                 areMut  = 0;    /* are <list>s elms mutable        */
    libGAP_Int                 len     = 0;    /* if so, this is the length       */
    libGAP_Obj                 family  = 0;    /* family of <list>s elements      */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm, x;         /* one element of <list>           */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Int                 testing;        /* to test or not to test type     */
    libGAP_Int                 res;            /* result                          */
    libGAP_Int                 knownDense;     /* set true if the list is already
					   known to be dense */
    libGAP_Int                 knownNDense;    /* set true if the list is already
					   known not to be dense */
    libGAP_UInt                ktnumFirst;

    /* if list has `TESTING' keep that                                     */
    testing = libGAP_IS_TESTING_PLIST(list) ? libGAP_TESTING : 0;

    libGAP_UNMARK_LIST(list, testing);
    knownDense = libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_DENSE );
    knownNDense = libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_NDENSE );
    libGAP_MARK_LIST(list, testing);

    /* get the length of the list                                          */
    lenList = libGAP_LEN_PLIST(list);

    /* special case for empty list                                         */
    if ( lenList == 0 ) {
        libGAP_UNMARK_LIST( list, testing );
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_EMPTY );
        res = libGAP_TNUM_OBJ(list);
        libGAP_MARK_LIST( list, testing );
	if (famfirst != (libGAP_Obj *) 0)
	  *famfirst = (libGAP_Obj) 0;
        return res;
    }

    /* look at the first element                                           */
    elm = libGAP_ELM_PLIST( list, 1 );
    if ( elm == 0 ) {
        isDense = 0;
    }
    else if ( libGAP_IS_TESTING_PLIST(elm) ) {
        isHom   = 0;
        areMut  = libGAP_IS_MUTABLE_PLIST(elm);
        isTable = 0;
    }
    else {
	if (!testing) libGAP_MARK_LIST(list, libGAP_TESTING);

	if (libGAP_IS_PLIST(elm))
	  family = libGAP_FAMILY_TYPE( libGAP_TypePlistWithKTnum(elm, &ktnumFirst));
	else
	  {
	    family  = libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ(elm) );
	    ktnumFirst = 0;
	  }
        isHom   = 1;
        areMut  = libGAP_IS_MUTABLE_OBJ(elm);
        if ( ktnumFirst >= libGAP_T_PLIST_HOM ||
	     ( ktnumFirst == 0 && libGAP_IS_HOMOG_LIST( elm) )) {

	  /* entry is a homogenous list, so this miught be a table */
	  isTable = 1;

	  /* also check for rectangularity, unless this would be expensive */
	  if (libGAP_IS_PLIST(elm))
	    {
	      isRect = 1;
	      len = libGAP_LEN_PLIST(elm);
	    }
	  
        }
	if (!testing) libGAP_UNMARK_LIST(list, libGAP_TESTING);
    }

    /* loop over the list                                                  */
    for ( i = 2; isDense && (isHom || ! areMut) && i <= lenList; i++ ) {
        elm = libGAP_ELM_PLIST( list, i );
        if ( elm == 0 ) {
            isDense = 0;
        }
        else if ( libGAP_IS_TESTING_PLIST(elm) ) {
            isHom   = 0;
            areMut  = (areMut || libGAP_IS_MUTABLE_PLIST(elm));
            isTable = 0;
	    isRect = 0;
        }
        else {
            isHom   = (isHom && family == libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ(elm) ));
            areMut  = (areMut || libGAP_IS_MUTABLE_OBJ(elm));
            isTable = isTable && libGAP_IS_LIST(elm); /*(isTable && IS_SMALL_LIST(elm) && LEN_LIST(elm) == len); */
	    isRect = isRect && libGAP_IS_PLIST(elm) && libGAP_LEN_PLIST(elm) == len;
        }
    }
    /* if we know it is not dense */
    if (knownNDense)
      isDense = 0;
    /* otherwise if we don't know that it IS dense */
    else if (!knownDense)
      for ( ;  isDense && i <= lenList;  i++ ) {
        elm = libGAP_ELM_PLIST( list, i );
        if ( elm == 0 ) {
	  isDense = 0;
        }
      }

    /* set the appropriate flags (not the hom. flag if elms are mutable)   */
    libGAP_UNMARK_LIST( list, testing );
    if      ( ! isDense ) {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NDENSE );
        res = libGAP_T_PLIST_NDENSE;
	if (famfirst != (libGAP_Obj *) 0)
	  *famfirst = (libGAP_Obj) 0;
    }
    else if ( isDense && ! isHom ) {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE );
        if ( ! areMut )
            libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NHOMOG );
        res = libGAP_T_PLIST_DENSE_NHOM;
	if (famfirst != (libGAP_Obj *) 0)
	  *famfirst = (libGAP_Obj) 0;
    }
    else if ( isDense &&   isHom && ! isTable ) {
        libGAP_SET_FILT_LIST( list, areMut ? libGAP_FN_IS_DENSE : libGAP_FN_IS_HOMOG );
	if (libGAP_TNUM_OBJ(libGAP_ELM_PLIST(list,1)) <= libGAP_T_CYC)
	  {
	    res = (lenList == 1) ? libGAP_T_PLIST_CYC_SSORT : libGAP_T_PLIST_CYC;
	    /* This is a hack */
	    libGAP_RetypeBag(list, res + ( libGAP_IS_MUTABLE_OBJ(list) ? 0 : libGAP_IMMUTABLE ));
	  }
	else if (libGAP_TNUM_OBJ(libGAP_ELM_PLIST(list,1)) == libGAP_T_FFE)
	  {
	    libGAP_FF fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(list,1));
	    libGAP_UInt isFFE = 1;
	    for (i = 2; i <= lenList; i++)
	      {
		x = libGAP_ELM_PLIST(list,i);
		if (!libGAP_IS_FFE(x) || libGAP_FLD_FFE(x) != fld)
		  {
		    isFFE = 0;
		    break;
		  }
	      }
	    if (isFFE)
	      {
		res = libGAP_T_PLIST_FFE;
		libGAP_RetypeBag(list, res + ( libGAP_IS_MUTABLE_OBJ(list) ? 0 : libGAP_IMMUTABLE ));
	      }
	    else
	      res = libGAP_T_PLIST_HOM;
	  }
	else
	  res = libGAP_T_PLIST_HOM;
	if (famfirst != (libGAP_Obj *) 0)
	  *famfirst = (libGAP_Obj) family;

    }
    else  if ( isDense &&   isHom &&   isTable && !isRect )  {
        libGAP_SET_FILT_LIST( list, areMut ? libGAP_FN_IS_DENSE : libGAP_FN_IS_TABLE );
        res = libGAP_T_PLIST_TAB;
	if (famfirst != (libGAP_Obj *) 0)
	  *famfirst = (libGAP_Obj) family;
    }
    else
      {
        libGAP_SET_FILT_LIST( list, areMut ? libGAP_FN_IS_DENSE : libGAP_FN_IS_RECT );
        res = libGAP_T_PLIST_TAB_RECT;
	if (famfirst != (libGAP_Obj *) 0)
	  *famfirst = (libGAP_Obj) family;
      }
    res = res + ( libGAP_IS_MUTABLE_OBJ(list) ? 0 : libGAP_IMMUTABLE );
    libGAP_MARK_LIST( list, testing );
    return res;
}


libGAP_Int libGAP_KTNumHomPlist (
    libGAP_Obj                 list)
{
    libGAP_Int                 isTable = 0;    /* are <list>s elms all lists   */
    libGAP_Int                 isRect  = 0;    /* are <list>s elms all equal length */
    libGAP_Int                 len     = 0;    /* if so, this is the length       */
    libGAP_Int                 lenList;        /* length of list                  */
    libGAP_Obj                 elm, x;         /* one element of <list>           */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Int                 testing;        /* to test or not to test type     */
    libGAP_Int                 res;            /* result                          */
    libGAP_Int                 isSSort;        /* list is (known to be) SSorted   */
    libGAP_Int                 isNSort;        /* list is (known to be) non-sorted*/

    /* if list has `TESTING' keep that                                     */
    testing = libGAP_IS_TESTING_PLIST(list) ? libGAP_TESTING : 0;

    /* get the length of the list                                          */
    lenList = libGAP_LEN_PLIST(list);

    /* special case for empty list                                         */
    assert(lenList);

    /* look at the first element                                           */
    elm = libGAP_ELM_PLIST( list, 1 );
    assert(elm);
    assert(!libGAP_IS_TESTING_PLIST(elm));

    isSSort = libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_SSORT );
    isNSort = libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_NSORT );

    /* if it's a kernel cyclotomic then we know where we are*/
    if (libGAP_TNUM_OBJ(elm) <= libGAP_T_CYC)
      {
	if (lenList == 1 || isSSort)
	  res = libGAP_T_PLIST_CYC_SSORT;
	else if (isNSort)
	  res = libGAP_T_PLIST_CYC_NSORT;
        else
	  res = libGAP_T_PLIST_CYC;
	
	/* This is a hack */
	libGAP_RetypeBag(list, res + ( libGAP_IS_MUTABLE_OBJ(list) ? 0 : libGAP_IMMUTABLE ));
	goto finish;
      }
    if (libGAP_TNUM_OBJ(elm) == libGAP_T_FFE)
      {
	libGAP_FF fld = libGAP_FLD_FFE(libGAP_ELM_PLIST(list,1));
	libGAP_UInt isFFE = 1;
	for (i = 2; i <= lenList; i++)
	  {
	    x = libGAP_ELM_PLIST(list,i);
	    if (!libGAP_IS_FFE(x) || libGAP_FLD_FFE(x) != fld)
	      {
		isFFE = 0;
		break;
	      }
	  }
	if (isFFE)
	  {
	    res = libGAP_T_PLIST_FFE;
	    libGAP_RetypeBag(list, res + ( libGAP_IS_MUTABLE_OBJ(list) ? 0 : libGAP_IMMUTABLE ));
	    goto finish;
	  }
      }
	

    /* Unless we already know it is, then check if the list is a table */
    if (!libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_TABLE ))
      {
	if ( libGAP_IS_HOMOG_LIST(elm) ) {
	  isTable = 1;
	  if (libGAP_IS_PLIST(elm))
	      {
		isRect = 1;
		len     = libGAP_LEN_PLIST(elm);
	      }
	}

	/* loop over the list */
	for ( i = 2; isTable && i <= lenList; i++ ) {
	  elm = libGAP_ELM_PLIST( list, i );
	  assert(elm);
	  assert(!libGAP_IS_TESTING_PLIST(elm));
	  isTable = isTable && libGAP_IS_LIST(elm); /* (isTable && IS_SMALL_LIST(elm) && LEN_LIST(elm) == len);*/
	  isRect = isRect && libGAP_IS_PLIST(elm) && libGAP_LEN_PLIST(elm) == len;
	}
      }
    else
      {
	isTable = 1;
	isRect = libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_RECT);
      }
    if (isTable && !isRect)
      {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_TABLE );
	if (isSSort)
	  res = libGAP_T_PLIST_TAB_SSORT;
	else if (isNSort)
	  res = libGAP_T_PLIST_TAB_NSORT;
	else
	  res = libGAP_T_PLIST_TAB;
      }
    else if (isRect)
      {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_RECT );
	if (isSSort)
	  res = libGAP_T_PLIST_TAB_RECT_SSORT;
	else if (isNSort)
	  res = libGAP_T_PLIST_TAB_RECT_NSORT;
	else
	  res = libGAP_T_PLIST_TAB_RECT;
	
      }
    else if (isSSort)
      res = libGAP_T_PLIST_HOM_SSORT;
    else if (isNSort)
      res = libGAP_T_PLIST_HOM_NSORT;
    else
      res = libGAP_T_PLIST_HOM;
    
 finish:
    res = res + ( libGAP_IS_MUTABLE_OBJ(list) ? 0 : libGAP_IMMUTABLE );
    libGAP_MARK_LIST( list, testing );
    return res;
}

libGAP_Obj libGAP_TypePlist( libGAP_Obj list)
{
  return libGAP_TypePlistWithKTnum( list, (libGAP_UInt *) 0);
}

libGAP_Obj libGAP_TypePlistWithKTnum (
    libGAP_Obj                 list,
    libGAP_UInt                *ktnum )
{
    libGAP_Obj                 kind;           /* kind, result                    */
    libGAP_Int                 ktype;          /* kind type of <list>             */
    libGAP_Obj                 family;         /* family of elements              */
    libGAP_Obj                 kinds;          /* kinds list of <family>          */

    /* recursion is possible for this type of list                         */
    libGAP_MARK_LIST( list, libGAP_TESTING );
    ktype = libGAP_KTNumPlist( list, &family);
    libGAP_UNMARK_LIST( list, libGAP_TESTING );
    if (ktnum != (libGAP_UInt *) 0)
      *ktnum = ktype;

    /* handle special cases                                                */
    switch (ktype)
      {
      case libGAP_T_PLIST_NDENSE:
        return libGAP_TYPE_LIST_NDENSE_MUTABLE;	
      case libGAP_T_PLIST_NDENSE+libGAP_IMMUTABLE:
        return libGAP_TYPE_LIST_NDENSE_IMMUTABLE;
      case libGAP_T_PLIST_DENSE_NHOM:
        return libGAP_TYPE_LIST_DENSE_NHOM_MUTABLE;
      case libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE:
        return libGAP_TYPE_LIST_DENSE_NHOM_IMMUTABLE;
      case libGAP_T_PLIST_DENSE_NHOM_SSORT:
        return libGAP_TYPE_LIST_DENSE_NHOM_SSORT_MUTABLE;
      case libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE:
        return libGAP_TYPE_LIST_DENSE_NHOM_SSORT_IMMUTABLE;
      case libGAP_T_PLIST_DENSE_NHOM_NSORT:
        return libGAP_TYPE_LIST_DENSE_NHOM_NSORT_MUTABLE;
      case libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE:
        return libGAP_TYPE_LIST_DENSE_NHOM_NSORT_IMMUTABLE;
      case libGAP_T_PLIST_EMPTY:
        return libGAP_TYPE_LIST_EMPTY_MUTABLE;
      case libGAP_T_PLIST_EMPTY+libGAP_IMMUTABLE:
        return libGAP_TYPE_LIST_EMPTY_IMMUTABLE;
      default: ; /* fall through into the rest of the function */
    }

    /* handle homogeneous list                                             */
    if ( libGAP_HasFiltListTNums[ktype][libGAP_FN_IS_HOMOG] ) {

        /* get the list kinds of the elements family */
        kinds  = libGAP_TYPES_LIST_FAM( family );

        /* if the kind is not yet known, compute it                        */
        kind = libGAP_ELM0_LIST( kinds, ktype-libGAP_T_PLIST_HOM+1 );
        if ( kind == 0 ) {
            kind = libGAP_CALL_2ARGS( libGAP_TYPE_LIST_HOM,
                family, libGAP_INTOBJ_INT(ktype-libGAP_T_PLIST_HOM+1) );
            libGAP_ASS_LIST( kinds, ktype-libGAP_T_PLIST_HOM+1, kind );
        }

        /* return the kind                                                 */
        return kind;
    }

    /* whats going on here?                                                */
    else {
        libGAP_ErrorQuit(
            "Panic: strange kind type '%s' ('%d')",
            (libGAP_Int)libGAP_TNAM_OBJ(list), (libGAP_Int)(libGAP_TNUM_OBJ(list)) );
        return 0;
    }

}

libGAP_Obj libGAP_TypePlistNDenseMut (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_NDENSE_MUTABLE;
}

libGAP_Obj libGAP_TypePlistNDenseImm (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_NDENSE_IMMUTABLE;
}

#define         libGAP_TypePlistDenseMut       libGAP_TypePlist
#define         libGAP_TypePlistDenseImm       libGAP_TypePlist

libGAP_Obj libGAP_TypePlistDenseNHomMut (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_DENSE_NHOM_MUTABLE;
}

libGAP_Obj libGAP_TypePlistDenseNHomImm (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_DENSE_NHOM_IMMUTABLE;
}
libGAP_Obj libGAP_TypePlistDenseNHomSSortMut (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_DENSE_NHOM_SSORT_MUTABLE;
}

libGAP_Obj libGAP_TypePlistDenseNHomSSortImm (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_DENSE_NHOM_SSORT_IMMUTABLE;
}
libGAP_Obj libGAP_TypePlistDenseNHomNSortMut (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_DENSE_NHOM_NSORT_MUTABLE;
}

libGAP_Obj libGAP_TypePlistDenseNHomNSortImm (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_DENSE_NHOM_NSORT_IMMUTABLE;
}

libGAP_Obj libGAP_TypePlistEmptyMut (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_EMPTY_MUTABLE;
}

libGAP_Obj libGAP_TypePlistEmptyImm (
    libGAP_Obj                 list )
{
    return libGAP_TYPE_LIST_EMPTY_IMMUTABLE;
}

libGAP_Obj libGAP_TypePlistHom (
    libGAP_Obj                 list )
{
    libGAP_Obj                 kind;           /* kind, result                    */
    libGAP_Int                 ktype;          /* kind type of <list>             */
    libGAP_Obj                 family;         /* family of elements              */
    libGAP_Obj                 kinds;          /* kinds list of <family>          */

    /* get the kind type and the family of the elements                    */
    ktype  = libGAP_KTNumHomPlist( list );
    family = libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ( libGAP_ELM_PLIST( list, 1 ) ) );

    /* get the list kinds of that family                                   */
    kinds  = libGAP_TYPES_LIST_FAM( family );

    /* if the kind is not yet known, compute it                            */
    kind = libGAP_ELM0_LIST( kinds, ktype-libGAP_T_PLIST_HOM+1 );
    if ( kind == 0 ) {
        kind = libGAP_CALL_2ARGS( libGAP_TYPE_LIST_HOM,
            family, libGAP_INTOBJ_INT(ktype-libGAP_T_PLIST_HOM+1) );
        libGAP_ASS_LIST( kinds, ktype-libGAP_T_PLIST_HOM+1, kind );
    }

    /* return the kind                                                     */
    return kind;
}

libGAP_Obj libGAP_TypePlistCyc (
    libGAP_Obj                 list )
{
    libGAP_Obj                 kind;           /* kind, result                    */
    libGAP_Int                 ktype;          /* kind type of <list>             */
    libGAP_Obj                 family;         /* family of elements              */
    libGAP_Obj                 kinds;          /* kinds list of <family>          */

    /* get the kind type and the family of the elements                    */
    ktype  = libGAP_TNUM_OBJ( list );

    /* This had better return the cyclotomics family, could be speeded up */
    family = libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ( libGAP_ELM_PLIST( list, 1 ) ) );

    /* get the list kinds of that family                                   */
    kinds  = libGAP_TYPES_LIST_FAM( family );

    /* if the kind is not yet known, compute it                            */
    kind = libGAP_ELM0_LIST( kinds, ktype-libGAP_T_PLIST_CYC+1 );
    if ( kind == 0 ) {
        kind = libGAP_CALL_2ARGS( libGAP_TYPE_LIST_HOM,
            family, libGAP_INTOBJ_INT(ktype-libGAP_T_PLIST_CYC+1) );
        libGAP_ASS_LIST( kinds, ktype-libGAP_T_PLIST_CYC+1, kind );
    }

    /* return the kind                                                     */
    return kind;
}

libGAP_Obj libGAP_TypePlistFfe (
    libGAP_Obj                 list )
{
    libGAP_Obj                 kind;           /* kind, result                    */
    libGAP_Int                 ktype;          /* kind type of <list>             */
    libGAP_Obj                 family;         /* family of elements              */
    libGAP_Obj                 kinds;          /* kinds list of <family>          */

    /* get the kind type and the family of the elements                    */
    ktype  = libGAP_TNUM_OBJ( list );
    family = libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ( libGAP_ELM_PLIST( list, 1 ) ) );

    /* get the list kinds of that family                                   */
    kinds  = libGAP_TYPES_LIST_FAM( family );

    /* if the kind is not yet known, compute it                            */
    kind = libGAP_ELM0_LIST( kinds, ktype-libGAP_T_PLIST_FFE+1 );
    if ( kind == 0 ) {
        kind = libGAP_CALL_2ARGS( libGAP_TYPE_LIST_HOM,
            family, libGAP_INTOBJ_INT(ktype-libGAP_T_PLIST_FFE+1) );
        libGAP_ASS_LIST( kinds, ktype-libGAP_T_PLIST_FFE+1, kind );
    }

    /* return the kind                                                     */
    return kind;
}


/****************************************************************************
**
*F  ShallowCopyPlist( <list> )  . . . . . . . .  shallow copy of a plain list
**
**  'ShallowCopyPlist' makes a copy of a plain list.
**
**  'ShallowCopyPlist'  only copies up to  the  logical length, the result is
**  always a mutable list.
*/
libGAP_Obj             libGAP_ShallowCopyPlist (
    libGAP_Obj                 list )
{
    libGAP_Obj                 new;
    libGAP_Obj *               o;
    libGAP_Obj *               n;
    libGAP_UInt                len;
    libGAP_UInt                i;

    /* make the new object and copy the contents                           */
    len = libGAP_LEN_PLIST(list);
    if ( ! libGAP_IS_MUTABLE_PLIST(list) ) {
        new = libGAP_NEW_PLIST( libGAP_TNUM_OBJ(list) - libGAP_IMMUTABLE, len );
    }
    else {
        new = libGAP_NEW_PLIST( libGAP_TNUM_OBJ(list), len );
    }
    o = libGAP_ADDR_OBJ(list);
    n = libGAP_ADDR_OBJ(new);
    for ( i = 0; i < len+1; i++ ) {
        *n++ = *o++;
    }
    /* 'CHANGED_BAG(new);' not needed, <new> is newest object              */
    return new;
}

/****************************************************************************
**
*F  FuncEmptyPlist( <self>, <len> ) . . . . . . . empty plist with space
*
* Returns an empty plain list, but with space for len entries preallocated.
*
*/
libGAP_Obj    libGAP_FuncEmptyPlist( libGAP_Obj self, libGAP_Obj len )
{
    libGAP_Obj                 new;
    while ( ! libGAP_IS_INTOBJ(len) ) {
        len = libGAP_ErrorReturnObj(
            "<len> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(len), 0L,
            "you can replace <len> via 'return <len>;'" );
    }

    new = libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, libGAP_INT_INTOBJ(len));
    libGAP_SET_LEN_PLIST(new, 0);
    return new;
}

/****************************************************************************
**
*F  FuncShrinkAllocationPlist( <self>, <list> ) . . . give back unneeded memory
*
*  Shrinks the bag of <list> to minimal possible size.
*
*/
libGAP_Obj   libGAP_FuncShrinkAllocationPlist( libGAP_Obj self, libGAP_Obj plist )
{
    while ( ! libGAP_IS_PLIST(plist) ) {
        plist = libGAP_ErrorReturnObj(
            "<plist> must be a plain list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(plist), 0,
            "you can replace <plist> via 'return <plist>;'" );
    }
    libGAP_SHRINK_PLIST(plist, libGAP_LEN_PLIST(plist));
    return (libGAP_Obj)0;
}

/****************************************************************************
**

*F  FuncIS_PLIST_REP( <self>, <obj> ) . . . . . . . .  handler for `IS_PLIST'
*/
libGAP_Obj libGAP_IsPListFilt;

libGAP_Obj libGAP_FuncIS_PLIST_REP (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_PLIST( obj ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**

*F  CopyPlist( <list>, <mut> )  . . . . . . . . . . . . . . copy a plain list
**
**  'CopyPlist' returns a  structural (deep) copy  of the plain  list <list>,
**  i.e., a recursive copy that preserves the structure.
**
**  If <list>  has not yet  been  copied, it makes  a copy,  leaves a forward
**  pointer to the copy in the first entry of the plain  list, where the size
**  of  the plain list usually resides,  and copies all  the entries.  If the
**  plain list  has  already   been  copied, it  returns  the   value  of the
**  forwarding pointer.
**
**  'CopyPlist' is the function in 'CopyObjFuncs' for plain lists.
**
**  'CleanPlist'  removes the  mark and the forwarding pointer from the plain
**  list <list>.
**
**  'CleanPlist' is the function in 'CleanObjFuncs' for plain lists.
*/
libGAP_Obj libGAP_CopyPlist (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* copy, result                    */
    libGAP_Obj                 tmp;            /* temporary variable              */
    libGAP_UInt                i;              /* loop variable                   */

    /* don't change immutable objects                                      */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        return list;
    }

    /* make a copy                                                         */
    if ( mut ) {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(list), libGAP_SIZE_OBJ(list) );
    }
    else {
        copy = libGAP_NewBag( libGAP_IMMUTABLE_TNUM( libGAP_TNUM_OBJ(list) ), libGAP_SIZE_OBJ(list) );
    }
    libGAP_ADDR_OBJ(copy)[0] = libGAP_ADDR_OBJ(list)[0];

    /* leave a forwarding pointer                                          */
    libGAP_ADDR_OBJ(list)[0] = copy;
    libGAP_CHANGED_BAG( list );

    /* now it is copied                                                    */
    libGAP_MARK_LIST( list, libGAP_COPYING );

    /* copy the subvalues                                                  */
    for ( i = 1; i <= libGAP_LEN_PLIST(copy); i++ ) {
        if ( libGAP_ADDR_OBJ(list)[i] != 0 ) {
            tmp = libGAP_COPY_OBJ( libGAP_ADDR_OBJ(list)[i], mut );
            libGAP_ADDR_OBJ(copy)[i] = tmp;
            libGAP_CHANGED_BAG( copy );
        }
    }

    /* return the copy                                                     */
    return copy;
}


/****************************************************************************
**
*F  CopyPlistCopy( <list>, <mut> )  . . . . . . . .  copy a copied plain list
*/
libGAP_Obj libGAP_CopyPlistCopy (
    libGAP_Obj                 list,
    libGAP_Int                 mut )
{
    return libGAP_ADDR_OBJ(list)[0];
}


/****************************************************************************
**
*F  CleanPlist( <list> )  . . . . . . . . . . . . . . . clean up a plain list
*/
void libGAP_CleanPlist (
    libGAP_Obj                 list )
{
}


/****************************************************************************
**
*F  CleanPlistCopy( <list> )  . . . . . . . . .  clean up a copied plain list
*/
void libGAP_CleanPlistCopy (
    libGAP_Obj                 list )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* remove the forwarding pointer                                       */
    libGAP_ADDR_OBJ(list)[0] = libGAP_ADDR_OBJ( libGAP_ADDR_OBJ(list)[0] )[0];

    /* now it is cleaned                                                   */
    libGAP_UNMARK_LIST( list, libGAP_COPYING );

    /* clean the subvalues                                                 */
    for ( i = 1; i <= libGAP_LEN_PLIST(list); i++ ) {
        if ( libGAP_ADDR_OBJ(list)[i] != 0 )
            libGAP_CLEAN_OBJ( libGAP_ADDR_OBJ(list)[i] );
    }

}


/****************************************************************************
**

*F  EqPlist(<left>,<right>) . . . . . . . . test if two plain lists are equal
**
**  'EqList'  returns 'true' if  the two plain lists   <left> and <right> are
**  equal and 'false' otherwise.
**
**  Is called from the 'EQ' binop so both  operands  are  already  evaluated.
*/
libGAP_Int             libGAP_EqPlist (
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Int                 lenL;           /* length of the left operand      */
    libGAP_Int                 lenR;           /* length of the right operand     */
    libGAP_Obj                 elmL;           /* element of the left operand     */
    libGAP_Obj                 elmR;           /* element of the right operand    */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the lengths of the lists and compare them                       */
    lenL = libGAP_LEN_PLIST( left );
    lenR = libGAP_LEN_PLIST( right );
    if ( lenL != lenR ) {
        return 0L;
    }

    /* loop over the elements and compare them                             */
    for ( i = 1; i <= lenL; i++ ) {
        elmL = libGAP_ELM_PLIST( left, i );
        elmR = libGAP_ELM_PLIST( right, i );
        if ( elmL == 0 && elmR != 0 ) {
            return 0L;
        }
        else if ( elmR == 0 && elmL != 0 ) {
            return 0L;
        }
        else if ( ! libGAP_EQ( elmL, elmR ) ) {
            return 0L;
        }
    }

    /* no differences found, the lists are equal                           */
    return 1L;
}


/****************************************************************************
**
*F  LtPlist(<left>,<right>) . . . . . . . . test if two plain lists are equal
**
**  'LtList' returns 'true' if the plain  list <left> is  less than the plain
**  list <right> and 'false' otherwise.
**
**  Is called from the 'LT' binop so both operands are already evaluated.
*/
libGAP_Int             libGAP_LtPlist (
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Int                 lenL;           /* length of the left operand      */
    libGAP_Int                 lenR;           /* length of the right operand     */
    libGAP_Obj                 elmL;           /* element of the left operand     */
    libGAP_Obj                 elmR;           /* element of the right operand    */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the lengths of the lists and compare them                       */
    lenL = libGAP_LEN_PLIST( left );
    lenR = libGAP_LEN_PLIST( right );

    /* loop over the elements and compare them                             */
    for ( i = 1; i <= lenL && i <= lenR; i++ ) {
        elmL = libGAP_ELM_PLIST( left, i );
        elmR = libGAP_ELM_PLIST( right, i );
        if ( elmL == 0 && elmR != 0 ) {
            return 1L;
        }
        else if ( elmR == 0 && elmL != 0 ) {
            return 0L;
        }
        else if ( ! libGAP_EQ( elmL, elmR ) ) {
            return libGAP_LT( elmL, elmR );
        }
    }

    /* reached the end of at least one list                                */
    return (lenL < lenR);
}


/****************************************************************************
**
*F  LenPlist(<list>)  . . . . . . . . . . . . . . . .  length of a plain list
**
**  'LenPlist' returns the length of the plain list <list> as a C integer.
**
**  'LenPlist' is the function in 'LenListFuncs' for plain lists.
*/
libGAP_Int             libGAP_LenPlist (
    libGAP_Obj                 list )
{
    return libGAP_LEN_PLIST( list );
}

libGAP_Int             libGAP_LenPlistEmpty (
    libGAP_Obj                 list )
{
    return 0L;
}


/****************************************************************************
**
*F  IsbPlist(<list>,<pos>)  . . . . . . test for an element from a plain list
*F  IsbvPlist(<list>,<pos>) . . . . . . test for an element from a plain list
**
**  'IsbPlist' returns 1 if the list <list> has an entry  at  position  <pos>
**  and 0 otherwise.  It is the responsibility of the caller to  ensure  that
**  <pos> is a positive integer.
**
**  'IsbvPlist' does the  same thing than  'IsbList', but need not check that
**  <pos>  is less  than or  equal  to the  length of   <list>,  this is  the
**  responsibility of the caller.
*/
libGAP_Int             libGAP_IsbPlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return (pos <= libGAP_LEN_PLIST( list ) && libGAP_ELM_PLIST( list, pos ) != 0);
}

libGAP_Int             libGAP_IsbPlistDense (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return (pos <= libGAP_LEN_PLIST( list ));
}

libGAP_Int             libGAP_IsbvPlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return (libGAP_ELM_PLIST( list, pos ) != 0);
}

libGAP_Int             libGAP_IsbvPlistDense (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return (1L);
}


/****************************************************************************
**
*F  Elm0Plist(<list>,<pos>) . . . . . . . . select an element of a plain list
*F  Elm0vPlist(<list>,<pos>)  . . . . . . . select an element of a plain list
**
**  'Elm0Plist' returns the element  at the position  <pos> of the plain list
**  <list>, or  0  if <list> has no  assigned  object at  <pos>.   It  is the
**  responsibility of the caller to ensure that <pos> is a positive integer.
**
**  'Elm0vPlist' does the same thing than 'Elm0List', but need not check that
**  <pos>  is less  than or  equal   to the length   of  <list>, this is  the
**  responsibility of the caller.
*/
libGAP_Obj             libGAP_Elm0Plist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    if ( pos <= libGAP_LEN_PLIST( list ) ) {
        return libGAP_ELM_PLIST( list, pos );
    }
    else {
        return 0;
    }
}

libGAP_Obj             libGAP_Elm0vPlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return libGAP_ELM_PLIST( list, pos );
}


/****************************************************************************
**
*F  ElmPlist(<list>,<pos>)  . . . . . . . . select an element of a plain list
*F  ElmvPlist(<list>,<pos>) . . . . . . . . select an element of a plain list
*F  ElmwPlist(<list>,<pos>) . . . . . . . . select an element of a plain list
**
**  'ElmPlist'  selects  the element at   position  <pos> of the  plain  list
**  <list>.  It is the responsibility of the caller to ensure that <pos> is a
**  positive   integer.  An error is signalled   if <pos> is  larger than the
**  length   of <list> or if  <list>  has no assigned  value  at the position
**  <pos>.
**
**  'ElmvPlist' does the same thing  than 'ElmList', but  need not check that
**  <pos> is  less  than or   equal  to the length   of  <list>, this  is the
**  responsibility of  the  caller.
**
**  'ElmPlist'   is the   function    in 'ElmListFuncs'   for  plain   lists.
**  'ElmvPlist' is the function in 'ElmvListFuncs' for plain lists.
*/
libGAP_Obj             libGAP_ElmPlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_Obj                 elm;            /* the selected element, result    */

    /* check the position                                                  */
    if ( libGAP_LEN_PLIST( list ) < pos ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            (libGAP_Int)pos, 0L,
            "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, pos );
    }

    /* select the element                                                  */
    elm = libGAP_ELM_PLIST( list, pos );

    /* check the element                                                   */
    if ( elm == 0 ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            (libGAP_Int)pos, 0L,
            "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, pos );
    }

    /* return the element                                                  */
    return elm;
}

libGAP_Obj             libGAP_ElmPlistDense (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_Obj                 elm;            /* the selected element, result    */

    /* check the position                                                  */
    if ( libGAP_LEN_PLIST( list ) < pos ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            (libGAP_Int)pos, 0L,
            "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, pos );
    }

    /* select the element                                                  */
    elm = libGAP_ELM_PLIST( list, pos );

    /* return the element                                                  */
    return elm;
}

libGAP_Obj             libGAP_ElmvPlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_Obj                 elm;            /* the selected element, result    */

    /* select the element                                                  */
    elm = libGAP_ELM_PLIST( list, pos );

    /* check the element                                                   */
    if ( elm == 0 ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            (libGAP_Int)pos, 0L,
            "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, pos );
    }

    /* return the element                                                  */
    return elm;
}

libGAP_Obj             libGAP_ElmvPlistDense (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_Obj                 elm;            /* the selected element, result    */

    /* select the element                                                  */
    elm = libGAP_ELM_PLIST( list, pos );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  ElmsPlist(<list>,<poss>)  . . . . . .  select a sublist from a plain list
**
**  'ElmsPlist' returns a  new list containing the  elements at  the position
**  given in the  list <poss>   from the  plain   list  <list>.  It is    the
**  responsibility of the caller to ensure that  <poss> is dense and contains
**  only positive integers.  An error is signalled if  <list> has no assigned
**  value at any of  the positions in <poss>, or  if an element of  <poss> is
**  larger than the length of <list>.
**
**  'ElmsPlist' is the function in 'ElmsListFuncs' for plain lists.
*/
libGAP_Obj             libGAP_ElmsPlist (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elms;           /* selected sublist, result        */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element from <list>         */
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Int                 i;              /* loop variable                   */

    /* select no element                                                   */
    if ( libGAP_LEN_LIST(poss) == 0 ) {
        elms = libGAP_NEW_PLIST( libGAP_T_PLIST_EMPTY, 0 );
    }

    /* general code                                                        */
    else if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_PLIST( list );

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST( poss );

        /* make the result list                                            */
        /* do not assert "dense", list might be homogeneous                */
        elms = libGAP_NEW_PLIST( libGAP_T_PLIST, lenPoss );
        libGAP_SET_LEN_PLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
            pos = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, i ) );
            if ( lenList < pos ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* select the element                                          */
            elm = libGAP_ELM_PLIST( list, pos );
            if ( elm == 0 ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* assign the element into <elms>                              */
            libGAP_SET_ELM_PLIST( elms, i, elm );

        }

        /* notify Gasman                                                   */
        libGAP_CHANGED_BAG( elms );

    }

    /* special code for ranges                                             */
    else {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_PLIST( list );

        /* get the length of <positions>, the first elements, and the inc. */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* check that no <position> is larger than 'LEN_LIST(<list>)'      */
        if ( lenList < pos ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }
        if ( lenList < pos + (lenPoss-1) * inc ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos + (lenPoss-1) * inc, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }

        /* make the result list                                            */
        /* do not assert "dense", list might be homogeneous                */
        elms = libGAP_NEW_PLIST( libGAP_T_PLIST, lenPoss );
        libGAP_SET_LEN_PLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++, pos += inc ) {

            /* select the element                                          */
            elm = libGAP_ELM_PLIST( list, pos );
            if ( elm == 0 ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* assign the element to <elms>                                */
            libGAP_SET_ELM_PLIST( elms, i, elm );

        }

        /* notify Gasman                                                   */
        libGAP_CHANGED_BAG( elms );

    }

    /* return the result                                                   */
    return elms;
}

libGAP_Obj             libGAP_ElmsPlistDense (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elms;           /* selected sublist, result        */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element from <list>         */
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Int                 i;              /* loop variable                   */

    /* select no element                                                   */
    if ( libGAP_LEN_LIST(poss) == 0 ) {
        elms = libGAP_NEW_PLIST( libGAP_T_PLIST_EMPTY, 0 );
    }

    /* general code                                                        */
    else if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_PLIST( list );

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST( poss );

        /* make the result list                                            */
        /* try to assert as many properties as possible                    */
	if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_SSORT) && libGAP_HAS_FILT_LIST(poss, libGAP_FN_IS_SSORT))
	  {
	    elms = libGAP_NEW_PLIST( libGAP_MUTABLE_TNUM(libGAP_TNUM_OBJ(list)), lenPoss);
	    libGAP_RESET_FILT_LIST( elms, libGAP_FN_IS_NHOMOG); /* can't deduce this one */
	  }
	else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_TABLE))
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_TAB, lenPoss );
	else if (libGAP_T_PLIST_CYC <= libGAP_TNUM_OBJ(list) && libGAP_TNUM_OBJ(list) <= 
                                                  libGAP_T_PLIST_CYC_SSORT+libGAP_IMMUTABLE)
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, lenPoss );
	else if (libGAP_T_PLIST_FFE <= libGAP_TNUM_OBJ(list) && libGAP_TNUM_OBJ(list) <= 
                                                  libGAP_T_PLIST_FFE+libGAP_IMMUTABLE)
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_FFE, lenPoss );
	else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_HOMOG))
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_HOM, lenPoss );
	else
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST, lenPoss);
	  
        libGAP_SET_LEN_PLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
            pos = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, i ) );
            if ( lenList < pos ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* select the element                                          */
            elm = libGAP_ELM_PLIST( list, pos );

            /* assign the element into <elms>                              */
            libGAP_SET_ELM_PLIST( elms, i, elm );

        }

        /* notify Gasman                                                   */
        libGAP_CHANGED_BAG( elms );

    }

    /* special code for ranges                                             */
    else {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_PLIST( list );

        /* get the length of <positions>, the first elements, and the inc. */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* check that no <position> is larger than 'LEN_LIST(<list>)'      */
        if ( pos < 1 || lenList < pos ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }
        if ( pos+(lenPoss-1) * inc < 1 || lenList < pos+(lenPoss-1) * inc ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos + (lenPoss-1) * inc, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }

        /* make the result list                                            */
        /* try to assert as many properties as possible                    */
        if      ( libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_SSORT) && inc > 0 )
	  {
	      elms = libGAP_NEW_PLIST( libGAP_MUTABLE_TNUM(libGAP_TNUM_OBJ(list)), lenPoss );
	/*      if (lenPoss > 1 && inc < 0)
		{
		  RESET_FILT_LIST(elms, FN_IS_SSORT);
		  SET_FILT_LIST(elms, FN_IS_NSORT);
		}  */
	  }
	else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_TABLE))
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_TAB, lenPoss );
	else if (libGAP_T_PLIST_CYC <= libGAP_TNUM_OBJ(list) && libGAP_TNUM_OBJ(list) <= 
                                                  libGAP_T_PLIST_CYC_SSORT+libGAP_IMMUTABLE)
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, lenPoss );
	else if (libGAP_T_PLIST_FFE <= libGAP_TNUM_OBJ(list) && libGAP_TNUM_OBJ(list) <= 
                                                  libGAP_T_PLIST_FFE+libGAP_IMMUTABLE)
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_FFE, lenPoss );
	else if (libGAP_HAS_FILT_LIST(list, libGAP_FN_IS_HOMOG))
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST_HOM, lenPoss );
	else
	  elms = libGAP_NEW_PLIST( libGAP_T_PLIST, lenPoss);
	
        libGAP_SET_LEN_PLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++, pos += inc ) {

            /* select the element                                          */
            elm = libGAP_ELM_PLIST( list, pos );

            /* assign the element to <elms>                                */
            libGAP_SET_ELM_PLIST( elms, i, elm );

        }

        /* notify Gasman                                                   */
        libGAP_CHANGED_BAG( elms );

    }

    /* return the result                                                   */
    return elms;
}


/****************************************************************************
**
*F  UnbPlist( <list>, <pos> ) . . . . . . unbind an element from a plain list
**
**  'UnbPlist' unbinds the element at position  <pos>  from  the  plain  list
**  <list>.  It is the responsibility of the caller to ensure that  <pos>  is
**  positive.
**
**  'UnbPlist' is the function in 'UnbListFuncs' for plain lists.
*/
void libGAP_UnbPlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    /* if <pos> is less than the length, convert to plain list and unbind  */
    if ( pos < libGAP_LEN_PLIST( list ) ) {
        libGAP_RESET_FILT_LIST( list, libGAP_FN_IS_DENSE );
        libGAP_SET_ELM_PLIST( list, pos, 0 );
    }

    /* if <pos> is equal to the length, unbind and compute new length      */
    else if ( pos == libGAP_LEN_PLIST( list ) ) {
        libGAP_CLEAR_FILTS_LIST(list);
        libGAP_SET_ELM_PLIST( list, pos, 0 );
        while ( 1 <= pos && libGAP_ELM_PLIST( list, pos ) == 0 ) { pos--; }
        libGAP_SET_LEN_PLIST( list, pos );
	if (libGAP_LEN_PLIST( list) == 0)
	  libGAP_SET_FILT_LIST(list, libGAP_FN_IS_EMPTY);
    }
}

void            libGAP_UnbPlistImm (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_ErrorReturnVoid(
        "List Unbind: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the unbind" );
}


/****************************************************************************
**
*F  AssPlist(<list>,<pos>,<val>)  . . . . . . . . . .  assign to a plain list
**
**  'AssPlist'  assigns  the value  <val>  to  the plain  list  <list> at the
**  position <pos>.  It  is the responsibility of the  caller to ensure  that
**  <pos> is positive, and that <val> is not 'Void'.
**
**  If the position is larger then the length of the list <list>, the list is
**  automatically  extended.  To avoid  making this too often, the bag of the
**  list is extended by at least '<length>/8 + 4' entries.  Thus in the loop
**
**      l := [];  for i in [1..1024]  do l[i] := i^2;  od;
**
**  the list 'l' is extended only 32 times not 1024 times.
**
**  'AssPlist' is the function in 'AssListFuncs' for plain lists.
*/
void            libGAP_AssPlist (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    /* resize the list if necessary                                        */
    if ( libGAP_LEN_PLIST( list ) < pos ) {
        libGAP_GROW_PLIST( list, pos );
        libGAP_SET_LEN_PLIST( list, pos );
    }

    /* now perform the assignment                                          */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );
}

void            libGAP_AssPlistXXX (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
  libGAP_Int len;
  
    /* the list will probably loose its flags/properties                   */
    libGAP_CLEAR_FILTS_LIST(list);

    /* resize the list if necessary                                        */
    len = libGAP_LEN_PLIST( list );
    if ( len < pos ) {
        libGAP_GROW_PLIST( list, pos );
        libGAP_SET_LEN_PLIST( list, pos );
    }

    /* now perform the assignment                                          */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );

    /* We may be able cheaply to tell that the list is non-dense          */
    if (len +1 < pos)
      libGAP_SET_FILT_LIST(list, libGAP_FN_IS_NDENSE);
}

void libGAP_AssPlistCyc   (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
  libGAP_Int len;
  
  
  /* resize the list if necessary                                        */
  len = libGAP_LEN_PLIST( list );
  if ( len < pos ) {
    libGAP_GROW_PLIST( list, pos );
    libGAP_SET_LEN_PLIST( list, pos );
  }

    /* now perform the assignment                                          */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );

    /* try and maintain maximum information about the list                */
    if (pos > len + 1)
      {
	libGAP_CLEAR_FILTS_LIST(list);
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NDENSE );
      }
    else if (libGAP_TNUM_OBJ(val) > libGAP_T_CYC)
      {
	libGAP_CLEAR_FILTS_LIST(list);
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE );
      }
    else
      {
      	libGAP_RESET_FILT_LIST( list, libGAP_FN_IS_NSORT );
	libGAP_RESET_FILT_LIST( list, libGAP_FN_IS_SSORT );
      }
	
}

void libGAP_AssPlistFfe   (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    libGAP_Int len;
  
    /* resize the list if necessary                                        */
    len = libGAP_LEN_PLIST( list );
    if ( len < pos ) {
        libGAP_GROW_PLIST( list, pos );
        libGAP_SET_LEN_PLIST( list, pos );
    }

    /* now perform the assignment                                          */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );

    /* try and maintain maximum information about the list                 */
    if( pos > len + 1 ) {
	libGAP_CLEAR_FILTS_LIST(list);
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NDENSE );
    }
    else if( libGAP_TNUM_OBJ(val) != libGAP_T_FFE ) {
	libGAP_CLEAR_FILTS_LIST(list);
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE );
    }
    else 
      { 
	libGAP_FF ffval;
	libGAP_Obj elm1;
	libGAP_FF ffelm1;
	libGAP_UInt otherpos;

	/* Here we select an other element to compare the field and
	   possibly characteristic of the assigned value to. This
	   code will never select pos, where the assignment has
	   already been done, unless we are replacing the only entry
	   of a length 1 list, in which case the result will always
	   still be a vecffe, so we are happy */
	
	if (pos == 1)
	  otherpos = len;
	else
	  otherpos = 1;
	elm1 = libGAP_ELM_PLIST( list, otherpos);
	ffval = libGAP_FLD_FFE(val);
	ffelm1 = libGAP_FLD_FFE(elm1);
	if( ffval != ffelm1 ) {
	  libGAP_CLEAR_FILTS_LIST(list);
	  libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE );
	  if (libGAP_CHAR_FF(ffval) == libGAP_CHAR_FF(ffelm1))
	    libGAP_SET_FILT_LIST( list, libGAP_FN_IS_HOMOG );
	}
	else 
	  {
	    libGAP_RESET_FILT_LIST( list, libGAP_FN_IS_NSORT );
	    libGAP_RESET_FILT_LIST( list, libGAP_FN_IS_SSORT );
	  }
      }
}

void libGAP_AssPlistDense (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
  libGAP_Int len;
  
  /* the list will probably loose its flags/properties                   */
  libGAP_CLEAR_FILTS_LIST(list);
  
  /* resize the list if necessary                                        */
  len = libGAP_LEN_PLIST( list );
  if ( len < pos ) {
    libGAP_GROW_PLIST( list, pos );
    libGAP_SET_LEN_PLIST( list, pos );
  }

    /* now perform the assignment                                          */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );

    /* restore denseness if we can */
    if (pos <= len+1)
      libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE );
    else
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NDENSE );
}

void libGAP_AssPlistHomog (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
  libGAP_Int len;
  libGAP_Obj fam;
  
  /* the list may loose its flags/properties                   */
  libGAP_CLEAR_FILTS_LIST(list);
  
  /* resize the list if necessary                                        */
  len = libGAP_LEN_PLIST( list );
  if ( len < pos ) {
    libGAP_GROW_PLIST( list, pos );
    libGAP_SET_LEN_PLIST( list, pos );
  }

    /* now perform the assignment                                          */
    libGAP_SET_ELM_PLIST( list, pos, val );
    libGAP_CHANGED_BAG( list );

    /* restore denseness if we can */
    if (pos <= len+1)
      {
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE );

	/* In this case, we may be able to restore homogeneity */
	if (len == 1 && pos == 1)
	  {

	    /* case of replacing the only list element */
	    if (libGAP_TNUM_OBJ( val ) <= libGAP_T_CYC)
	      {
		libGAP_RetypeBag( list, libGAP_T_PLIST_CYC_SSORT );
	      }
	    else
	      {
		libGAP_SET_FILT_LIST( list, libGAP_FN_IS_HOMOG );
		libGAP_SET_FILT_LIST( list, libGAP_FN_IS_SSORT );
	      }
	  }
	else if (!libGAP_SyInitializing && !libGAP_IS_MUTABLE_OBJ(val))
	  {
	    /* find the family of an original list element */
	    if (pos != 1)
	      fam = libGAP_FAMILY_OBJ(libGAP_ELM_PLIST(list, 1));
	    else
	      fam = libGAP_FAMILY_OBJ(libGAP_ELM_PLIST(list, 2));

	    /* restore homogeneity if we can */
	    if (fam == libGAP_FAMILY_OBJ( val ))
	      libGAP_SET_FILT_LIST(list, libGAP_FN_IS_HOMOG);
	    else
	      libGAP_SET_FILT_LIST(list, libGAP_FN_IS_NHOMOG);
	  }
      }
    else
      libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NDENSE );

}


void            libGAP_AssPlistImm (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignment: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**
*F  AssPlistEmpty( <list>, <pos>, <val> ) . . . . .  assignment to empty list
*/
void libGAP_AssPlistEmpty (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 val )
{
    /* if <pos> is large than one use `AssPlistDense'                        */
    if ( 1 != pos ) {
        libGAP_AssPlistDense( list, pos, val );
    }

    /* catch booleans                                                      */
    else if ( val == libGAP_True || val == libGAP_False ) {
        libGAP_ConvBlist(list);
        libGAP_AssBlist( list, pos, val );
    }

    /* catch FFE                                                           */
    else if ( libGAP_IS_FFE(val) ) {

      /* early in initialization, the type of the empty list may not be
	 available, in which case we must NOT call method selection */
      if (libGAP_TYPE_LIST_EMPTY_MUTABLE != 0)
        libGAP_AssListObject( list, pos, val );
      else
	libGAP_AssPlistXXX( list, pos, val );
    }

    /* catch constants                                                     */
    else if ( libGAP_TNUM_OBJ(val) < libGAP_FIRST_EXTERNAL_TNUM ) {
        libGAP_AssPlistXXX( list, pos, val );
	/* fix up type */
	libGAP_SET_FILT_LIST(list, libGAP_FN_IS_DENSE);
	if ( !libGAP_IS_MUTABLE_OBJ( val) )
	  {
	    libGAP_SET_FILT_LIST(list, libGAP_FN_IS_HOMOG);
	    if ( libGAP_TNUM_OBJ(val) <= libGAP_T_CYC ) 
	      libGAP_RetypeBag( list, libGAP_T_PLIST_CYC);
	  }
    }

    
    /* use method selection                                                */
    else {
      /* early in initialization, the type of the empty list may not be
	 available, in which case we must NOT call method selection */
      if (libGAP_TYPE_LIST_EMPTY_MUTABLE != 0)
        libGAP_AssListObject( list, pos, val );
      else
	libGAP_AssPlistXXX( list, pos, val );
    }

}


/****************************************************************************
**
*F  AsssPlist(<list>,<poss>,<vals>) . . . . assign several elements to a list
**
**  'AsssPlist' assignes the   values from the list  <vals>  at the positions
**  given in the list <poss> to the list <list>.  It is the responsibility of
**  the caller  to  ensure that <poss> is   dense and contains  only positive
**  integers, that <poss> and <vals> have the same length, and that <vals> is
**  dense.
**
**  'AsssPlist' is the function in 'AsssListFuncs' for plain lists.
*/
void            libGAP_AsssPlist (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 vals )
{
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 max;            /* larger position                 */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Obj                 val;            /* one element from <vals>         */
    libGAP_Int                 i;              /* loop variable                   */

    /* general code                                                        */
    if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST( poss );

        /* find the largest entry in <positions>                           */
        max = libGAP_LEN_PLIST( list );
        for ( i = 1; i <= lenPoss; i++ ) {
            if ( max < libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, i ) ) )
                max = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, i ) );
        }

        /* resize the list if necessary                                    */
        if ( libGAP_LEN_PLIST(list) < max ) {
            libGAP_GROW_PLIST( list, max );
            libGAP_SET_LEN_PLIST( list, max );
        }

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
            pos = libGAP_INT_INTOBJ( libGAP_ELMW_LIST( poss, i ) );

            /* select the element                                          */
            val = libGAP_ELMW_LIST( vals, i );

            /* assign the element into <elms>                              */
            libGAP_SET_ELM_PLIST( list, pos, val );

        }

        /* notify Gasman                                                   */
        libGAP_CHANGED_BAG( list );

    }

    /* special code for ranges                                             */
    else {

        /* get the length of <positions>                                   */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* find the largest entry in <positions>                           */
        max = libGAP_LEN_PLIST( list );
        if ( max < pos )
            max = pos;
        if ( max < pos + (lenPoss-1) * inc )
            max = pos + (lenPoss-1) * inc;

        /* resize the list if necessary                                    */
        if ( libGAP_LEN_PLIST(list) < max ) {
            libGAP_GROW_PLIST( list, max );
            libGAP_SET_LEN_PLIST( list, max );
        }

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++, pos += inc ) {

            /* select the element                                          */
            val = libGAP_ELMW_LIST( vals, i );

            /* assign the element to <elms>                                */
            libGAP_SET_ELM_PLIST( list, pos, val );

        }

        /* notify Gasman                                                   */
        libGAP_CHANGED_BAG( list );

    }
}

void            libGAP_AsssPlistXXX (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 vals )
{
    /* the list will probably loose its flags/properties                   */
    libGAP_CLEAR_FILTS_LIST(list);

    /* and delegate                                                        */
    libGAP_AsssPlist( list, poss, vals );
}

void            libGAP_AsssPlistImm (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Lists Assignments: <list> must be a mutable list",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}


/****************************************************************************
**
*F  IsDensePlist(<list>)  . . . . .  dense list test function for plain lists
**
**  'IsDensePlist' returns 1 if the plain list <list> is a dense list and 0
**  otherwise.
**
**  'IsDensePlist' is the function in 'IsDenseListFuncs' for plain lists.
*/
libGAP_Int             libGAP_IsDensePlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length of the list                                          */
    lenList = libGAP_LEN_PLIST( list );

    /* special case for empty list                                         */
    if ( lenList == 0 ) {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_EMPTY );
        return 1L;
    }

    /* loop over the entries of the list                                   */
    for ( i = 1; i <= lenList; i++ ) {
        if ( libGAP_ELM_PLIST( list, i ) == 0 )
            return 0L;
    }

    /* set the dense flag (even if the elements are mutable)               */
    libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE );

    /* no hole found                                                       */
    return 1L;
}

libGAP_Int             libGAP_IsDensePlistNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsDensePlistYes (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**
*F  IsHomogPlist(<list>)  . .  homogeneous list test function for plain lists
**
**  'IsHomogPlist' returns  1 if the plain  list <list> is a homogeneous list
**  and 0 otherwise.
**
**  'IsHomogPlist' is the function in 'IsHomogListFuncs' for plain lists.
*/
libGAP_Int             libGAP_IsHomogPlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 ktype;
    ktype = libGAP_KTNumPlist( list, (libGAP_Obj *)0 );
    return (libGAP_T_PLIST_HOM <= ktype);
}

libGAP_Int             libGAP_IsHomogPlistNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsHomogPlistYes (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**
*F  IsTablePlist(<list>)  . . . . . . . . table test function for plain lists
**
**  'IsTablePlist' returns  1 if the plain list  <list> is a homogeneous list
**  of lists and 0 otherwise.
**
**  'IsTablePlist' is the function in 'IsTableListFuncs' for plain lists.
*/
libGAP_Int             libGAP_IsTablePlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 ktype;
    ktype = libGAP_KTNumPlist( list, (libGAP_Obj *)0 );
    return (libGAP_T_PLIST_TAB <= ktype && ktype <= libGAP_T_PLIST_TAB_RECT_SSORT);
}

libGAP_Int             libGAP_IsTablePlistNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsTablePlistYes (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**
*F  IsSSortPlist(<list>)  . . . . . sorted list test function for plain lists
**
**  'IsSSortPlist'  returns 2  if the  plain  list <list>  is strictly sorted
**  (each element is strictly smaller than the next one), and 0 otherwise.
**
**  'IsSSortPlist' is the function in 'IsSSortListFuncs' for plain lists.
*/

libGAP_Int             libGAP_IsSSortPlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;
    libGAP_Obj elm1;
    libGAP_Obj elm2;
    libGAP_Int                 areMut;
    libGAP_Int                 i;
    libGAP_Obj                 fam=0;    /* initialize to help compiler */
    libGAP_Int                 isHom;
    
        
    /* get the length                                                      */
    lenList = libGAP_LEN_PLIST( list );

    /* special case for the empty list                                     */
    if ( lenList == 0 ) {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_EMPTY );
        return 2L;
    }

    /* get the first element                                               */
    elm1    = libGAP_ELM_PLIST( list, 1 );
    if (elm1 == 0)
      goto notDense;
    areMut   = libGAP_IS_MUTABLE_OBJ( elm1 );
    if (!libGAP_SyInitializing)
      {
	fam = libGAP_FAMILY_OBJ(elm1);
	isHom = 1;
      }
    else
      isHom = 0;
    
    /* loop over the other elements                                        */
    for ( i = 2; i <= lenList; i++ ) {
      elm2 = libGAP_ELM_PLIST( list, i );
      if (elm2 == 0)
	goto notDense;
      if ( ! libGAP_LT( elm1, elm2 ) )
	break;
      areMut = (areMut || libGAP_IS_MUTABLE_OBJ( elm2 ));
      isHom = (isHom && fam == libGAP_FAMILY_OBJ(elm2 ));
      elm1 = elm2;
      }
    /* set flags (unless the elements are mutable)                      */

    /* If we found inhomogeneity then it is real  */
    if (!areMut && !isHom) 
      {
	libGAP_SET_FILT_LIST(list,libGAP_FN_IS_NHOMOG); 
      } 
	
    if ( lenList < i ) { 
      /* we got to the end, so there were no holes */
      libGAP_SET_FILT_LIST( list, libGAP_FN_IS_DENSE);

      /* and we know about homoheneity */
      if ( ! areMut ) {
	if (isHom)
	  libGAP_SET_FILT_LIST( list, libGAP_FN_IS_HOMOG);
	else
	  libGAP_SET_FILT_LIST( list, libGAP_FN_IS_HOMOG);
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_SSORT );
      }
      return 2L;
    }
    else {
      if ( ! areMut ) {
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NSORT );
      }
      return 0L;
      
    }

 notDense:
    libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NDENSE );
    return 0L;
}

libGAP_Int             libGAP_IsSSortPlistDense (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;
    libGAP_Obj elm1;
    libGAP_Obj elm2;
    libGAP_Int                 areMut;
    libGAP_Int                 i;
    libGAP_Obj                 fam=0;     /* initialize to help compiler */
    libGAP_Int                 isHom;
        
    /* get the length                                                      */
    lenList = libGAP_LEN_PLIST( list );

    /* special case for the empty list                                     */
    if ( lenList == 0 ) {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_EMPTY );
        return 2L;
    }

    /* get the first element                                               */
    elm1    = libGAP_ELM_PLIST( list, 1 );
    areMut   = libGAP_IS_MUTABLE_OBJ( elm1 );
    if (!libGAP_SyInitializing)
      {
	fam = libGAP_FAMILY_OBJ(elm1);
	isHom = 1;
      }
    else
      isHom = 0;
    
    /* loop over the other elements                                        */
    for ( i = 2; i <= lenList; i++ ) {
      elm2 = libGAP_ELM_PLIST( list, i );
      if ( ! libGAP_LT( elm1, elm2 ) )
	break;
      areMut = (areMut || libGAP_IS_MUTABLE_OBJ( elm2 ));
      isHom = (isHom && fam == libGAP_FAMILY_OBJ(elm2 ));
      elm1 = elm2;
      }
    /* set flags (unless the elements are mutable)                      */

    if (!areMut && !isHom)
      libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NHOMOG);
    if ( lenList < i ) {
      if ( ! areMut ) {
	if (isHom)
	  libGAP_SET_FILT_LIST( list, libGAP_FN_IS_HOMOG);
	else
	  libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NHOMOG);
	libGAP_SET_FILT_LIST( list, libGAP_FN_IS_SSORT );
      }
      return 2L;
    }
    else {
        if ( ! areMut ) {
	  libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NSORT );
        }
        return 0L;
    }

}

libGAP_Int             libGAP_IsSSortPlistHom (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;
    libGAP_Obj elm1;
    libGAP_Obj elm2;
    libGAP_Int                 i;
        
    /* get the length                                                      */
    lenList = libGAP_LEN_PLIST( list );

    /* special case for the empty list                                     */
    if ( lenList == 0 ) {
        libGAP_SET_FILT_LIST( list, libGAP_FN_IS_EMPTY );
        return 2L;
    }

    /* get the first element                                               */
    elm1    = libGAP_ELM_PLIST( list, 1 );
    
    /* loop over the other elements                                        */
    for ( i = 2; i <= lenList; i++ ) {
      elm2 = libGAP_ELM_PLIST( list, i );
      if ( ! libGAP_LT( elm1, elm2 ) )
	break;
      elm1 = elm2;
      }
    /* set flags      */

    if ( lenList < i ) {
      libGAP_SET_FILT_LIST( list, libGAP_FN_IS_SSORT );
      return 2L;
    }
    else {
      libGAP_SET_FILT_LIST( list, libGAP_FN_IS_NSORT );
      return 0L;
    }

}


libGAP_Int             libGAP_IsSSortPlistNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsSSortPlistYes (
    libGAP_Obj                 list )
{
    return 2L;
}

libGAP_Obj libGAP_FuncSetIsSSortedPlist (libGAP_Obj self, libGAP_Obj list)
{
  libGAP_SET_FILT_LIST(list, libGAP_FN_IS_SSORT);
  return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  IsPossPlist(<list>) . . . .  positions list test function for plain lists
**
**  'IsPossPlist'  returns   1 if the plain    list <list> is  a   dense list
**  containing only positive integers, and 0 otherwise.
**
**  'IsPossPlist' is the function in 'IsPossListFuncs' for plain lists.
*/
libGAP_Int             libGAP_IsPossPlist (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element of <list>           */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length of the variable                                      */
    lenList = libGAP_LEN_PLIST( list );

    /* loop over the entries of the list                                   */
    for ( i = 1; i <= lenList; i++ ) {
        elm = libGAP_ELM_PLIST( list, i );
        if ( elm == 0)
	  return 0L;
	if( libGAP_IS_INTOBJ(elm))
	  {
	    if (libGAP_INT_INTOBJ(elm) <= 0 )
	      return 0L;
	  }
	else
	  if (libGAP_TNUM_OBJ(elm) != libGAP_T_INTPOS)
	    return 0L;
    }

    /* no problems found                                                   */
    return 1L;
}


/****************************************************************************
**
*F  PosPlist(<list>,<val>,<start>)  . . . .  position of an element in a list
**
**  'PosPlist'  returns the  position  of the value  <val>  in the plain list
**  <list> after the first position <start> as a C integer.  0 is returned if
**  <val> is not in the list.
**
**  'PosPlist' is the function in 'PosListFuncs' for plain lists.
*/
libGAP_Obj             libGAP_PosPlist (
    libGAP_Obj                 list,
    libGAP_Obj                 val,
    libGAP_Obj                 start )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element of <list>           */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt istart;

    /* if the starting position is too big to be a small int
       then there can't be anything to find */
    if (!libGAP_IS_INTOBJ(start))
      return libGAP_Fail;

    istart = libGAP_INT_INTOBJ(start);
    
    /* get the length of <list>                                            */
    lenList = libGAP_LEN_PLIST( list );

    /* loop over all entries in <list>                                     */
    for ( i = istart+1; i <= lenList; i++ ) {

        /* select one element from <list>                                  */
        elm = libGAP_ELM_PLIST( list, i );

        /* compare with <val>                                              */
        if ( elm != 0 && libGAP_EQ( elm, val ) )
            break;

    }

    /* return the position (0 if <val> was not found)                      */
    return (lenList < i ? libGAP_Fail : libGAP_INTOBJ_INT(i));
}

libGAP_Obj             libGAP_PosPlistDense (
    libGAP_Obj                 list,
    libGAP_Obj                 val,
    libGAP_Obj                 start )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element of <list>           */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt istart;

    /* if the starting position is too big to be a small int
       then there can't be anything to find */
    if (!libGAP_IS_INTOBJ(start))
      return libGAP_Fail;

    istart = libGAP_INT_INTOBJ(start);
    
    /* get the length of <list>                                            */
    lenList = libGAP_LEN_PLIST( list );

    /* loop over all entries in <list>                                     */
    for ( i = istart+1; i <= lenList; i++ ) {

        /* select one element from <list>                                  */
        elm = libGAP_ELM_PLIST( list, i );
	assert(elm);

        /* compare with <val>                                              */
        if ( libGAP_EQ( elm, val ) )
            break;

    }

    /* return the position (0 if <val> was not found)                      */
    return (lenList < i ? libGAP_Fail : libGAP_INTOBJ_INT(i));
}

libGAP_Obj             libGAP_PosPlistSort (
    libGAP_Obj                 list,
    libGAP_Obj                 val,
    libGAP_Obj                 start )
{
    libGAP_UInt                lenList;        /* logical length of the set       */
    libGAP_UInt                i, j, k;        /* loop variables                  */
    libGAP_UInt                istart;

    /* if the starting position is too big to be a small int
       then there can't be anything to find */
    if (!libGAP_IS_INTOBJ(start))
      return libGAP_Fail;

    istart = libGAP_INT_INTOBJ(start);

    /* get a pointer to the set and the logical length of the set          */
    lenList = libGAP_LEN_PLIST(list);

    /* perform the binary search to find the position                      */
    i = istart;  k = lenList + 1;
    while ( i+1 < k ) {                 /* set[i] < elm && elm <= set[k]   */
        j = (i + k) / 2;                /* i < j < k                       */
        if ( libGAP_LT( libGAP_ELM_PLIST(list,j), val ) )  i = j;
        else                                 k = j;
    }

    /* test if the element was found at position k                         */
    if ( lenList < k || ! libGAP_EQ( libGAP_ELM_PLIST(list,k), val ) )
        k = 0;

    /* return the position                                                 */
    return k == 0 ? libGAP_Fail : libGAP_INTOBJ_INT(k);
}


libGAP_Obj             libGAP_PosPlistHomSort (
    libGAP_Obj                 list,
    libGAP_Obj                 val,
    libGAP_Obj                 start )
{
    /* deal with the case which can be decided by the family relationship  */
    if (libGAP_FAMILY_OBJ(val) != libGAP_FAMILY_OBJ(libGAP_ELM_PLIST(list,1)))
      return libGAP_Fail;
    
    return libGAP_PosPlistSort(list, val, start);
}


/****************************************************************************
**
*F  PlainPlist(<list>)  . . . . . . . .  convert a plain list to a plain list
**
**  'PlainPlist' converts the plain  list <list> to a  plain list.  Not  much
**  work.
**
**  'PlainPlist' is the function in 'PlainListFuncs' for plain lists.
*/
void            libGAP_PlainPlist (
    libGAP_Obj                 list )
{
    return;
}

/****************************************************************************
**
*F  SavePlist( <list> )
**
*/

void libGAP_SavePlist( libGAP_Obj list )
{
  libGAP_UInt i;
  libGAP_SaveUInt(libGAP_LEN_PLIST(list));
  for (i = 1; i <= libGAP_LEN_PLIST(list); i++)
    libGAP_SaveSubObj(libGAP_ELM_PLIST(list,i));
  return;
}

/****************************************************************************
**
*F  LoadPlist( <list> )
**
*/

void libGAP_LoadPlist( libGAP_Obj list )
{
  libGAP_UInt i;
  libGAP_SET_LEN_PLIST(list, libGAP_LoadUInt());
  for (i = 1; i <= libGAP_LEN_PLIST(list); i++)
    libGAP_SET_ELM_PLIST(list,i, libGAP_LoadSubObj());
  return;
}


/****************************************************************************
**
*F  FuncASS_PLIST_DEFAULT( <self>, <plist>, <pos>, <val> )  . . `AssPlistXXX'
*/
libGAP_Obj libGAP_FuncASS_PLIST_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 plist,
    libGAP_Obj                 pos,
    libGAP_Obj                 val )
{
    libGAP_Int                 p;

    /* check the arguments                                                 */
    while ( ! libGAP_IS_INTOBJ(pos) ) {
        pos = libGAP_ErrorReturnObj(
            "<pos> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0,
            "you can replace <pos> via 'return <pos>;'" );
    }
    p = libGAP_INT_INTOBJ(pos);
    if ( p < 0 ) {
        pos = libGAP_ErrorReturnObj(
            "<pos> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pos), 0,
            "you can replace <pos> via 'return <pos>;'" );
        return libGAP_FuncASS_PLIST_DEFAULT( self, plist, pos, val );
    }
    while ( ! libGAP_IS_PLIST(plist) || ! libGAP_IS_MUTABLE_PLIST(plist) ) {
        plist = libGAP_ErrorReturnObj(
            "<list> must be a mutable plain list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(plist), 0,
            "you can replace <list> via 'return <list>;'" );
    }

    /* call `AssPlistXXX'                                                  */
    libGAP_AssPlistXXX( plist, p, val );
    return 0;
}

/****************************************************************************
**
*F  MakeImmutablePlistInHom( <plist> )
**
**  This is the function for Plists that might have mutable subobjects
**  which is currently exactly those that are not known to be homogenous
**  (or immutable, but MakeImmutable will have caught that case before we get h**  ere)
*/

void libGAP_MakeImmutablePlistInHom( libGAP_Obj list )
{
  libGAP_UInt i;
  libGAP_Obj elm;
  libGAP_RetypeBag( list, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(list)));
  for (i = 1; i <= libGAP_LEN_PLIST(list); i++)
    {
      elm = libGAP_ELM_PLIST( list, i);
      if (elm != 0)
	{
	  libGAP_MakeImmutable( elm );
	  libGAP_CHANGED_BAG(list);
	}
    }
}

/****************************************************************************
**
*F  MakeImmutablePlistNoMutElms( <plist> )
**
**  This is the function for Plists that cannot have mutable subobjects
**  which is currently  those that are  known to be homogenous or known to
**  be non-homogenous
**  (or immutable, but MakeImmutable will have caught that case before we get here)
*/

void libGAP_MakeImmutablePlistNoMutElms( libGAP_Obj list )
{
  libGAP_RetypeBag( list, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(list)));
}

/****************************************************************************
**
*F  FuncIsRectangularTablePlist( <plist> )
**
**  Determine whether a PLIST is a rectangulat table, set the TNUM 
**  appropriately. This is a mathod for IsRectangularTable, so the
**  method selection has alrewady checked that the list is a table
**  and should not bother to call us, unless we do not already know whether it
**  is rectangular
*/

libGAP_Obj libGAP_FuncIsRectangularTablePlist( libGAP_Obj self, libGAP_Obj plist)
{
  libGAP_Obj len;
  libGAP_UInt lenlist;
  libGAP_UInt i;
  libGAP_UInt hasMut = 0;
  libGAP_Obj elm;
  
  assert(!libGAP_HAS_FILT_LIST(plist, libGAP_FN_IS_RECT));
  lenlist = libGAP_LEN_PLIST(plist);
  assert(lenlist);
  if (lenlist == 1)
    {
      if (!libGAP_IS_MUTABLE_OBJ(libGAP_ELM_PLIST(plist,1)))
	libGAP_SET_FILT_LIST(plist, libGAP_FN_IS_RECT);
      return libGAP_True;
    }
  elm = libGAP_ELM_PLIST(plist,1);
  len = libGAP_LENGTH(elm);
  hasMut = libGAP_IS_MUTABLE_OBJ(elm);
  for (i = 2; i <= lenlist; i++)
    {
      elm = libGAP_ELM_PLIST(plist,i);
      if (!libGAP_EQ(len, libGAP_LENGTH(elm)))
	return libGAP_False;
      hasMut = hasMut || libGAP_IS_MUTABLE_OBJ(elm);
    }
  if (!hasMut)
    libGAP_SET_FILT_LIST(plist, libGAP_FN_IS_RECT);
  return libGAP_True;
  
}

/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  BagNames  . . . . . . . . . . . . . . . . . . . . . . . list of bag names
*/
static libGAP_StructBagNames libGAP_BagNames[] = {
  { libGAP_T_PLIST,                                "list (plain)" },
  { libGAP_T_PLIST            +libGAP_IMMUTABLE,          "list (plain,imm)" },
  { libGAP_T_PLIST                       +libGAP_COPYING, "list (plain,copied)" },
  { libGAP_T_PLIST            +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,imm,copied)" },

  { libGAP_T_PLIST_NDENSE,                         "list (plain,ndense)" },
  { libGAP_T_PLIST_NDENSE     +libGAP_IMMUTABLE,          "list (plain,ndense,imm)" },
  { libGAP_T_PLIST_NDENSE                +libGAP_COPYING, "list (plain,ndense,copied)" },
  { libGAP_T_PLIST_NDENSE     +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,ndense,imm,copied)" },

  { libGAP_T_PLIST_DENSE,                          "list (plain,dense)" },
  { libGAP_T_PLIST_DENSE      +libGAP_IMMUTABLE,          "list (plain,dense,imm)" },
  { libGAP_T_PLIST_DENSE                 +libGAP_COPYING, "list (plain,dense,copied)" },
  { libGAP_T_PLIST_DENSE      +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,dense,imm,copied)" },

  { libGAP_T_PLIST_DENSE_NHOM,                     "list (plain,dense,nhom)" },
  { libGAP_T_PLIST_DENSE_NHOM +libGAP_IMMUTABLE,          "list (plain,dense,nhom,imm)" },
  { libGAP_T_PLIST_DENSE_NHOM            +libGAP_COPYING, "list (plain,dense,nhom,copied)" },
  { libGAP_T_PLIST_DENSE_NHOM +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,dense,nhom,imm,copied)" },

  { libGAP_T_PLIST_DENSE_NHOM_SSORT,                     "list (plain,dense,nhom,ssort)" },
  { libGAP_T_PLIST_DENSE_NHOM_SSORT +libGAP_IMMUTABLE,          "list (plain,dense,nhom,ssort,imm)" },
  { libGAP_T_PLIST_DENSE_NHOM_SSORT            +libGAP_COPYING, "list (plain,dense,nhom,ssort,copied)" },
  { libGAP_T_PLIST_DENSE_NHOM_SSORT +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,dense,nhom,ssort,imm,copied)" },

  { libGAP_T_PLIST_DENSE_NHOM_NSORT,                     "list (plain,dense,nhom,nsort)" },
  { libGAP_T_PLIST_DENSE_NHOM_NSORT +libGAP_IMMUTABLE,          "list (plain,dense,nhom,nsort,imm)" },
  { libGAP_T_PLIST_DENSE_NHOM_NSORT            +libGAP_COPYING, "list (plain,dense,nhom,nsort,copied)" },
  { libGAP_T_PLIST_DENSE_NHOM_NSORT +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,dense,nhom,nsort,imm,copied)" },

  { libGAP_T_PLIST_EMPTY,                          "list (plain,empty)" },
  { libGAP_T_PLIST_EMPTY      +libGAP_IMMUTABLE,          "list (plain,empty,imm)" },
  { libGAP_T_PLIST_EMPTY                 +libGAP_COPYING, "list (plain,empty,copied)" },
  { libGAP_T_PLIST_EMPTY      +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,empty,imm,copied)" },

  { libGAP_T_PLIST_HOM,                            "list (plain,hom)" },
  { libGAP_T_PLIST_HOM        +libGAP_IMMUTABLE,          "list (plain,hom,imm)" },
  { libGAP_T_PLIST_HOM                   +libGAP_COPYING, "list (plain,hom,copied)" },
  { libGAP_T_PLIST_HOM        +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,hom,imm,copied)" },

  { libGAP_T_PLIST_HOM_NSORT,                      "list (plain,hom,nsort)" },
  { libGAP_T_PLIST_HOM_NSORT  +libGAP_IMMUTABLE,          "list (plain,hom,nsort,imm)" },
  { libGAP_T_PLIST_HOM_NSORT             +libGAP_COPYING, "list (plain,hom,nsort,copied)" },
  { libGAP_T_PLIST_HOM_NSORT  +libGAP_IMMUTABLE +libGAP_COPYING, "list (plain,hom,nsort,imm,copied)" },

  { libGAP_T_PLIST_HOM_SSORT,                      "list (plain,hom,ssort)" },
  { libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,           "list (plain,hom,ssort,imm)" },
  { libGAP_T_PLIST_HOM_SSORT            +libGAP_COPYING,  "list (plain,hom,ssort,copied)" },
  { libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,hom,ssort,imm,copied)" },

  { libGAP_T_PLIST_TAB,                            "list (plain,table)" },
  { libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,           "list (plain,table,imm)" },
  { libGAP_T_PLIST_TAB                  +libGAP_COPYING,  "list (plain,table,copied)" },
  { libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,table,imm,copied)" },

  { libGAP_T_PLIST_TAB_NSORT,                      "list (plain,table,nsort)" },
  { libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,           "list (plain,table,nsort,imm)" },
  { libGAP_T_PLIST_TAB_NSORT            +libGAP_COPYING,  "list (plain,table,nsort,copied)" },
  { libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,table,nsort,imm,copied)" },

  { libGAP_T_PLIST_TAB_SSORT,                      "list (plain,table,ssort)" },
  { libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,           "list (plain,table,ssort,imm)" },
  { libGAP_T_PLIST_TAB_SSORT            +libGAP_COPYING,  "list (plain,table,ssort,copied)" },
  { libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,table,ssort,imm,copied)" },

  { libGAP_T_PLIST_TAB_RECT,                            "list (plain,rect table)" },
  { libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,           "list (plain,rect table,imm)" },
  { libGAP_T_PLIST_TAB_RECT                  +libGAP_COPYING,  "list (plain,rect table,copied)" },
  { libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,rect table,imm,copied)" },

  { libGAP_T_PLIST_TAB_RECT_NSORT,                      "list (plain,rect table,nsort)" },
  { libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,           "list (plain,rect table,nsort,imm)" },
  { libGAP_T_PLIST_TAB_RECT_NSORT            +libGAP_COPYING,  "list (plain,rect table,nsort,copied)" },
  { libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,rect table,nsort,imm,copied)" },

  { libGAP_T_PLIST_TAB_RECT_SSORT,                      "list (plain,rect table,ssort)" },
  { libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,           "list (plain,rect table,ssort,imm)" },
  { libGAP_T_PLIST_TAB_RECT_SSORT            +libGAP_COPYING,  "list (plain,rect table,ssort,copied)" },
  { libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,rect table,ssort,imm,copied)" },

  { libGAP_T_PLIST_CYC,                            "list (plain,cyc)" },
  { libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,           "list (plain,cyc,imm)" },
  { libGAP_T_PLIST_CYC                  +libGAP_COPYING,  "list (plain,cyc,copied)" },
  { libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,cyc,imm,copied)" },

  { libGAP_T_PLIST_CYC_NSORT,                      "list (plain,cyc,nsort)" },
  { libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,           "list (plain,cyc,nsort,imm)" },
  { libGAP_T_PLIST_CYC_NSORT            +libGAP_COPYING,  "list (plain,cyc,nsort,copied)" },
  { libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,cyc,nsort,imm,copied)" },

  { libGAP_T_PLIST_CYC_SSORT,                      "list (plain,cyc,ssort)" },
  { libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,           "list (plain,cyc,ssort,imm)" },
  { libGAP_T_PLIST_CYC_SSORT            +libGAP_COPYING,  "list (plain,cyc,ssort,copied)" },
  { libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE +libGAP_COPYING,  "list (plain,cyc,ssort,imm,copied)" },

  { libGAP_T_PLIST_FFE,                     "list (sml fin fld elms)" },
  { libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,          "list (sml fin fld elms,imm)" },
  { libGAP_T_PLIST_FFE            +libGAP_COPYING, "list (sml fin fld elms,copied)" },
  { libGAP_T_PLIST_FFE +libGAP_IMMUTABLE +libGAP_COPYING, "list (sml fin fld elms,imm,copied)" },

  { -1,                                     "" }
};


/****************************************************************************
**
*V  ClearFiltsTab . . . . . . . . . . . . . . . . . . . .  clear filter tnums
*/
static libGAP_Int libGAP_ClearFiltsTab [] = {
    libGAP_T_PLIST,                          libGAP_T_PLIST,
    libGAP_T_PLIST           +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE,                   libGAP_T_PLIST,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE,                    libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM,               libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,         libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,         libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY,                    libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM,                      libGAP_T_PLIST,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB,                      libGAP_T_PLIST,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT,                      libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC,                      libGAP_T_PLIST,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT,                libGAP_T_PLIST,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE,                      libGAP_T_PLIST,
    libGAP_T_PLIST_FFE       +libGAP_IMMUTABLE,     libGAP_T_PLIST +libGAP_IMMUTABLE,

    -1,                               -1
};


/****************************************************************************
**
*V  HasFiltTab  . . . . . . . . . . . . . . . . . . . . .  tester filter tnum
*/
static libGAP_Int libGAP_HasFiltTab [] = {

    /* mutable plain lists                                                 */
    libGAP_T_PLIST,                      libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST,                      libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST,                      libGAP_FN_IS_DENSE,    0,
    libGAP_T_PLIST,                      libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST,                      libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST,                      libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST,                      libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST,                      libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST,                      libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST,                      libGAP_FN_IS_NSORT,    0,

    /* immutable plain lists                                               */
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* mutable empty list                                                  */
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_EMPTY,    1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NSORT,    0,

    /* immutable empty list                                                */
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* mutable dense list                                                  */
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NSORT,    0,

    /* immutable dense list                                                */
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* mutable dense list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NHOMOG,   1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NSORT,    0,

    /* immutable dense list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   1,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* mutable dense ssorted list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_NHOMOG,   1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,      libGAP_FN_IS_NSORT,    0,

    /* immutable dense ssorted list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* mutable dense nsorted list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_NHOMOG,   1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,           libGAP_FN_IS_NSORT,    1,

    /* immutable dense nsorted list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    1,

    /* a mutable list with holes                                           */
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_DENSE,    0,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NDENSE,   1,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NSORT,    0,

    /* an immutable list with holes                                        */
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   1,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* mutable dense list, which conts imms, is homogeneous, not a table   */
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_RECT,    0,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NSORT,    0,

    /* immutable dense list, which is immutable, homogeneous, not a table  */
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    0,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* ssort mutable dense list, which conts imms, is homog, not a table   */
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_RECT,    0,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NSORT,    0,

    /* ssort immutable dense list, which is immutable, homog, not a table  */
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    0,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* nsort mutable dense list, which conts imms, is homog, not a table   */
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_RECT,    0,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NSORT,    1,

    /* nsort immutable dense list, which is immutable, homog, not a table  */
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    1,

    /* mutable dense list, which is immutable, homog, non-empty, table     */
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NSORT,    0,

    /* immutable dense list, which is immutable, homog, non-empty, table   */
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* ssort, mutable dense list, which is imm, homog, non-empty, table    */
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NSORT,    0,

    /* ssort, immutable dense list, which is imm, homog, non-empty, table  */
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* nsort, mutable dense list, which is imm, homog, non-empty, table    */
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NSORT,    1,

    /* nsort, immutable dense list, which is imm, homog, non-empty, table  */
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    1,

    /* mutable dense list, which is immutable, homog, non-empty, rect table     */
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_RECT,     1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NSORT,    0,

    /* immutable dense list, which is immutable, homog, non-empty, rect table   */
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     1,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* ssort, mutable dense list, which is imm, homog, non-empty, rect table    */
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_RECT,     1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NSORT,    0,

    /* ssort, immutable dense list, which is imm, homog, non-empty, rect table  */
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* nsort, mutable dense list, which is imm, homog, non-empty, rect table    */
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_RECT,     1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NSORT,    1,

    /* nsort, immutable dense list, which is imm, homog, non-empty, rect table  */
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    1,

    /* mutable dense list, which only contains objects of type <= T_CYC    */
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NSORT,    0,

    /* immutable dense list, which only contains objects of type <= T_CYC  */
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* ssort mutable dense list, which only contains objs of type <= T_CYC */
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NSORT,    0,

    /* ssort immutable dense list, which contains objs of type <= T_CYC    */
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    /* nsort mutable dense list, which only contains objs of type <= T_CYC */
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NSORT,    1,

    /* nsort immutable dense list, which contains objs of type <= T_CYC    */
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    1,

    /* mutable dense list, which only contains objects of type T_FFE
       all written over the same field*/
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_MUTABLE,  1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NSORT,    0,

    /* immutable dense list, which only contains objects of type T_FFE
       all written over the same field */
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE,  0,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,    0,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,    1,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,   0,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,    1,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,   0,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,    0,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,     0,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,    0,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,    0,

    -1,                         -1,             -1
};


/****************************************************************************
**
*V  SetFiltTab  . . . . . . . . . . . . . . . . . . . . .  setter filter tnum
*/
static libGAP_Int libGAP_SetFiltTab [] = {

    /* mutable plain lists                                                 */
    libGAP_T_PLIST,                      libGAP_FN_IS_MUTABLE, libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST,                      libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST,                      libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST,                      libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST,                      libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST,                      libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST,                      libGAP_FN_IS_SSORT,   libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_NSORT,   libGAP_T_PLIST,

    /* immutable plain lists                                               */
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT   +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST           +libGAP_IMMUTABLE,

    /* mutable empty list                                                  */
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_DENSE,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_RECT,    -1,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_SSORT,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NSORT,   -1,


    /* immutable empty list                                                */
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   -1,

    /* mutable dense list                                                  */
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE,

    /* immutable dense list                                                */
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB_RECT  +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,

    /* mutable dense list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM_NSORT,

    /* immutable dense list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,

    /* mutable dense ssorted list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_NSORT,   -1,

    /* immutable dense ssorted list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   -1,

    /* mutable dense nsorted list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM_NSORT,

    /* immutable dense nsorted list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,

    /* a mutable list with holes                                           */
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_DENSE,   -1,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_SSORT,   libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NSORT,   libGAP_T_PLIST_NDENSE,

    /* an immutable list with holes                                        */
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   -1,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   -1,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,

    /* mutable dense list, which conts imms, is homogeneous, not a table   */
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM_NSORT,

    /* immutable dense list, which is immutable, homogeneous, not a table  */
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,

    /* ssort mutable dense list, which conts imms, is homog, not a table   */
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NSORT,   -1,

    /* ssort immutable dense list, which is immutable, homog, not a table  */
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   -1,

    /* nsort mutable dense list, which conts imms, is homog, not a table   */
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM_NSORT,

    /* nsort immutable dense list, which is immutable, homog, not a table  */
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,

    /* mutable dense list, which is immutable, homog, non-empty, table     */
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_NSORT,

    /* immutable dense list, which is immutable, homog, non-empty, table   */
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,

    /* ssort, mutable dense list, which is imm, homog, non-empty, table    */
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NSORT,   -1,

    /* ssort, immutable dense list, which is imm, homog, non-empty, table  */
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   -1,

    /* nsort, mutable dense list, which is imm, homog, non-empty, table    */
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_NSORT,

    /* nsort, immutable dense list, which is imm, homog, non-empty, table  */
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,


    /* mutable dense list, which is immutable, homog, non-empty, rect table     */
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT_NSORT,

    /* immutable dense list, which is immutable, homog, non-empty, rect table   */
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,

    /* ssort, mutable dense list, which is imm, homog, non-empty, rect table    */
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NSORT,   -1,

    /* ssort, immutable dense list, which is imm, homog, non-empty, rect table  */
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   -1,

    /* nsort, mutable dense list, which is imm, homog, non-empty, rect table    */
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT_NSORT,

    /* nsort, immutable dense list, which is imm, homog, non-empty, rect table  */
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,

    /* mutable dense list, which only contains objects of type <= T_CYC    */
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC_NSORT,

    /* immutable dense list, which only contains objects of type <= T_CYC  */
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,  -1,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,

    /* ssort mutable dense list, which only contains objs of type <= T_CYC */
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NSORT,   -1,

    /* ssort immutable dense list, which contains objs of type <= T_CYC    */
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   -1,

    /* nsort mutable dense list, which only contains objs of type <= T_CYC */
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC_NSORT,

    /* nsort immutable dense list, which contains objs of type <= T_CYC    */
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   -1,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,

    /* mutable dense list, which only contains objects of type T_FFE       */
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_FFE,

    /* immutable dense list, which only contains objects of type T_FFE     */
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   -1,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  -1,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  -1,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   -1,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   -1,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,

    -1,                           -1,            -1

};


/****************************************************************************
**
*V  ResetFiltTab  . . . . . . . . . . . . . . . . . . .  unsetter filter tnum
*/
static libGAP_Int libGAP_ResetFiltTab [] = {

    /* mutable plain lists                                                 */
    libGAP_T_PLIST,                      libGAP_FN_IS_MUTABLE, libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST,                      libGAP_FN_IS_EMPTY,   libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_NDENSE,  libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_HOMOG,   libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_TABLE,   libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_RECT,   libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_SSORT,   libGAP_T_PLIST,
    libGAP_T_PLIST,                      libGAP_FN_IS_NSORT,   libGAP_T_PLIST,

    /* immutable plain lists                                               */
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST           +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST           +libGAP_IMMUTABLE,

    /* mutable empty list                                                  */
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_EMPTY,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_HOMOG,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_TABLE,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_RECT,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_SSORT,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NSORT,   libGAP_T_PLIST_EMPTY,

    /* mutable empty list                                                  */
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_EMPTY,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_HOMOG,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_TABLE,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_RECT,   libGAP_T_PLIST_EMPTY,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_SSORT,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY,                libGAP_FN_IS_NSORT,   libGAP_T_PLIST_EMPTY,

    /* immutable empty list                                                */
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST,
    libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE,

    /* mutable dense list                                                  */
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE,                libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE,

    /* immutable dense list                                                */
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,

    /* mutable dense list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM,           libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM,

    /* immutable dense list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,

    /* mutable dense ssorted list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE_NHOM_SSORT,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM,
    libGAP_T_PLIST_DENSE_NHOM_SSORT,     libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM_SSORT,

    /* immutable dense ssorted list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE,

    /* mutable dense nsorted list, which contains immutables and is not homog      */
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM_NSORT,
    libGAP_T_PLIST_DENSE_NHOM_NSORT,     libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM,

    /* immutable dense nsorted list, which is immutable and not homogeneous        */
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE,

    /* a mutable list with holes                                           */
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_DENSE,   libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NDENSE,  libGAP_T_PLIST,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_TABLE,   libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_RECT,   libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_SSORT,   libGAP_T_PLIST_NDENSE,
    libGAP_T_PLIST_NDENSE,               libGAP_FN_IS_NSORT,   libGAP_T_PLIST_NDENSE,

    /* an immutable list with holes                                        */
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,
    libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE,

    /* mutable dense list, which conts imms, is homogeneous, not a table   */
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM      +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_RECT,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM,

    /* immutable dense list, which is immutable, homogeneous, not a table  */
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,

    /* ssort mutable dense list, which conts imms, is homog, not a table   */
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_SSORT+libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_HOM_SSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM_SSORT,

    /* ssort immutable dense list, which is immutable, homog, not a table  */
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,

    /* nsort mutable dense list, which conts imms, is homog, not a table   */
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_HOM_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM,

    /* nsort immutable dense list, which is immutable, homog, not a table  */
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,

    /* mutable dense list, which is immutable, homog, non-empty, table     */
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB,

    /* immutable dense list, which is immutable, homog, non-empty, table   */
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,

    /* ssort, mutable dense list, which is imm, homog, non-empty, table    */
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB_SSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_SSORT,

    /* ssort, immutable dense list, which is imm, homog, non-empty, table  */
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,

    /* nsort, mutable dense list, which is imm, homog, non-empty, table    */
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB,

    /* nsort, immutable dense list, which is imm, homog, non-empty, table  */
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    /* mutable dense list, which is immutable, homog, non-empty, rect table     */
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT,

    /* immutable dense list, which is immutable, homog, non-empty, rect table   */
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,

    /* ssort, mutable dense list, which is imm, homog, non-empty, rect table    */
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_RECT_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_TAB_SSORT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT,
    libGAP_T_PLIST_TAB_RECT_SSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT_SSORT,

    /* ssort, immutable dense list, which is imm, homog, non-empty, rect table  */
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE,

    /* nsort, mutable dense list, which is imm, homog, non-empty, rect table    */
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT_NSORT,
    libGAP_T_PLIST_TAB_RECT_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT,

    /* nsort, immutable dense list, which is imm, homog, non-empty, rect table  */
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,    libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE,

    /* mutable dense list, which only contains objects of type <= T_CYC    */
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_TABLE,   libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_RECT,   libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC,                  libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC,

    /* immutable dense list, which only contains objects of type <= T_CYC  */
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,

    /* ssort mutable dense list, which only contains objs of type <= T_CYC */
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_CYC_SSORT,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC,
    libGAP_T_PLIST_CYC_SSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC_SSORT,

    /* ssort immutable dense list, which contains objs of type <= T_CYC    */
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE,

    /* nsort mutable dense list, which only contains objs of type <= T_CYC */
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC_NSORT,
    libGAP_T_PLIST_CYC_NSORT,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC,

    /* nsort immutable dense list, which contains objs of type <= T_CYC    */
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST           +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE,
    libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE,

    /* mutable dense list, which only contains objects of type T_FFE       */
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_DENSE,   libGAP_T_PLIST,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_HOMOG,   libGAP_T_PLIST,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_TABLE,   libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_RECT,   libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_SSORT,   libGAP_T_PLIST_FFE,
    libGAP_T_PLIST_FFE,            libGAP_FN_IS_NSORT,   libGAP_T_PLIST_FFE,

    /* immutable dense list, which only contains objects of type T_FFE     */
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_MUTABLE, libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_EMPTY,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_DENSE,   libGAP_T_PLIST     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NDENSE,  libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_HOMOG,   libGAP_T_PLIST     +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NHOMOG,  libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_TABLE,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_RECT,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_SSORT,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,
    libGAP_T_PLIST_FFE +libGAP_IMMUTABLE, libGAP_FN_IS_NSORT,   libGAP_T_PLIST_FFE +libGAP_IMMUTABLE,

    -1,                         -1,             -1

};


/****************************************************************************
**
*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_PLIST_REP", "obj", &libGAP_IsPListFilt,
      libGAP_FuncIS_PLIST_REP, "src/lists.c:IS_PLIST_REP" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "ASS_PLIST_DEFAULT", 3, "list, pos, val",
      libGAP_FuncASS_PLIST_DEFAULT, "src/lists.c:ASS_PLIST_DEFAULT" },
    
    { "IsRectangularTablePlist", 1, "plist",
      libGAP_FuncIsRectangularTablePlist, "src/lists.c:IsRectangularTablePlist" },

    { "SET_IS_SSORTED_PLIST", 1, "list",
      libGAP_FuncSetIsSSortedPlist, "src/lists.c:SET_IS_SSORTED_PLIST" },
    
    { "EmptyPlist", 1, "len",
      libGAP_FuncEmptyPlist, "src/lists.c:FuncEmptyPlist" },
    
    { "ShrinkAllocationPlist", 1, "plist",
      libGAP_FuncShrinkAllocationPlist, "src/lists.c:FuncShrinkAllocationPlist" },
    
    { 0 }

};


/****************************************************************************
**
*V  SweepAndCheckNonDensePlist
*/

void libGAP_SweepAndCheckNonDensePlist ( libGAP_Bag * src,
				  libGAP_Bag *dest,
				  libGAP_UInt baglength )
{
  libGAP_UInt holeseen=0;
  libGAP_UInt listlength = (libGAP_UInt)(*src);
  libGAP_UInt i;
  if (listlength > baglength - 1)
    libGAP_Pr("#W Plain list with length %d in bag of size only %d\n", listlength, baglength);
  *(libGAP_UInt *)dest = listlength;
  if (listlength == 0)
    libGAP_Pr("#W Plain non-dense list length 0\n", 0, 0);
  for (i = 1; i < listlength; i++)
    if (!(dest[i] = src[i]))
      holeseen = 1;
  if (!(dest[listlength] = src[listlength]))
    libGAP_Pr("#W plain list length %d with final entry unbound\n", listlength, 0);
  if (!holeseen)
    libGAP_Pr("#W plain non-dense list length %d contains no hole\n", listlength, 0);
  for (i = listlength+1; i < baglength; i++)
    dest[i] = (libGAP_Bag)0;  
}
			  

/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                t1, t2;         /* loop variables                  */

    /* check dependencies                                                  */
    libGAP_RequireModule( libGAP_module, "lists", 403600000UL );

    /* GASMAN marking functions and GASMAN names                           */
    libGAP_InitBagNamesFromTable( libGAP_BagNames );

    for ( t1 = libGAP_T_PLIST;  t1 < libGAP_T_PLIST_FFE ;  t1 += 2 ) {
        libGAP_InitMarkFuncBags( t1                     , libGAP_MarkAllSubBags );
        libGAP_InitMarkFuncBags( t1 +libGAP_IMMUTABLE          , libGAP_MarkAllSubBags );
        libGAP_InitMarkFuncBags( t1            +libGAP_COPYING , libGAP_MarkAllSubBags );
        libGAP_InitMarkFuncBags( t1 +libGAP_IMMUTABLE +libGAP_COPYING , libGAP_MarkAllSubBags );
    }

    libGAP_InitMarkFuncBags( libGAP_T_PLIST_FFE                     , libGAP_MarkNoSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_PLIST_FFE +libGAP_IMMUTABLE          , libGAP_MarkNoSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_PLIST_FFE            +libGAP_COPYING , libGAP_MarkNoSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_PLIST_FFE +libGAP_IMMUTABLE +libGAP_COPYING , libGAP_MarkNoSubBags );

#ifdef CHECK_NDENSE_BAGS
    libGAP_InitSweepFuncBags( libGAP_T_PLIST_NDENSE, libGAP_SweepAndCheckNonDensePlist);
    libGAP_InitSweepFuncBags( libGAP_T_PLIST_NDENSE + libGAP_IMMUTABLE, libGAP_SweepAndCheckNonDensePlist);
#endif
    
    /* If T_PLIST_FFE is not the last PLIST type then some more
       work needs to be done here */

    for ( t1 = libGAP_T_PLIST;  t1 <= libGAP_LAST_PLIST_TNUM;  t1 += 2 ) {
        libGAP_SaveObjFuncs[ t1 ]             = libGAP_SavePlist;
        libGAP_SaveObjFuncs[ t1 + libGAP_IMMUTABLE ] = libGAP_SavePlist; 
        libGAP_LoadObjFuncs[ t1 ]             = libGAP_LoadPlist;
        libGAP_LoadObjFuncs[ t1 + libGAP_IMMUTABLE ] = libGAP_LoadPlist; 
    }

    /* get the kinds (resp. kind functions)                                */
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_NDENSE_MUTABLE", 
                           &libGAP_TYPE_LIST_NDENSE_MUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_LIST_NDENSE_IMMUTABLE", 
                           &libGAP_TYPE_LIST_NDENSE_IMMUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_LIST_DENSE_NHOM_MUTABLE", 
                           &libGAP_TYPE_LIST_DENSE_NHOM_MUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_LIST_DENSE_NHOM_IMMUTABLE", 
                           &libGAP_TYPE_LIST_DENSE_NHOM_IMMUTABLE );
    
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_DENSE_NHOM_SSORT_MUTABLE", 
                           &libGAP_TYPE_LIST_DENSE_NHOM_SSORT_MUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_LIST_DENSE_NHOM_SSORT_IMMUTABLE", 
                           &libGAP_TYPE_LIST_DENSE_NHOM_SSORT_IMMUTABLE );
    
    libGAP_ImportGVarFromLibrary( "TYPE_LIST_DENSE_NHOM_NSORT_MUTABLE", 
                           &libGAP_TYPE_LIST_DENSE_NHOM_NSORT_MUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_LIST_DENSE_NHOM_NSORT_IMMUTABLE", 
                           &libGAP_TYPE_LIST_DENSE_NHOM_NSORT_IMMUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_LIST_EMPTY_MUTABLE", 
                           &libGAP_TYPE_LIST_EMPTY_MUTABLE );

    libGAP_ImportGVarFromLibrary( "TYPE_LIST_EMPTY_IMMUTABLE", 
                           &libGAP_TYPE_LIST_EMPTY_IMMUTABLE );

    libGAP_ImportFuncFromLibrary( "TYPE_LIST_HOM",
                           &libGAP_TYPE_LIST_HOM );

    /* install the kind methods                                            */
    libGAP_TypeObjFuncs[ libGAP_T_PLIST                       ] = libGAP_TypePlist;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST            +libGAP_IMMUTABLE ] = libGAP_TypePlist;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_NDENSE                ] = libGAP_TypePlistNDenseMut;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_NDENSE     +libGAP_IMMUTABLE ] = libGAP_TypePlistNDenseImm;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE                 ] = libGAP_TypePlistDenseMut;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE      +libGAP_IMMUTABLE ] = libGAP_TypePlistDenseImm;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE_NHOM            ] = libGAP_TypePlistDenseNHomMut;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE_NHOM +libGAP_IMMUTABLE ] = libGAP_TypePlistDenseNHomImm;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT      ] = libGAP_TypePlistDenseNHomSSortMut;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE ] = libGAP_TypePlistDenseNHomSSortImm;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT      ] = libGAP_TypePlistDenseNHomNSortMut;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT +libGAP_IMMUTABLE ] = libGAP_TypePlistDenseNHomNSortImm;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_EMPTY                 ] = libGAP_TypePlistEmptyMut;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_EMPTY      +libGAP_IMMUTABLE ] = libGAP_TypePlistEmptyImm;
    
    for ( t1 = libGAP_T_PLIST_HOM; t1 <= libGAP_T_PLIST_TAB_RECT_SSORT; t1 += 2 ) {
        libGAP_TypeObjFuncs[ t1            ] = libGAP_TypePlistHom;
        libGAP_TypeObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_TypePlistHom;
    }
    
    for ( t1 = libGAP_T_PLIST_CYC; t1 <= libGAP_T_PLIST_CYC_SSORT; t1 += 2 ) {
        libGAP_TypeObjFuncs[ t1            ] = libGAP_TypePlistCyc;
        libGAP_TypeObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_TypePlistCyc;
    }

    libGAP_TypeObjFuncs[ libGAP_T_PLIST_FFE            ] = libGAP_TypePlistFfe;
    libGAP_TypeObjFuncs[ libGAP_T_PLIST_FFE +libGAP_IMMUTABLE ] = libGAP_TypePlistFfe;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* initialise list tables                                              */
    libGAP_InitClearFiltsTNumsFromTable   ( libGAP_ClearFiltsTab );
    libGAP_InitHasFiltListTNumsFromTable  ( libGAP_HasFiltTab    );
    libGAP_InitSetFiltListTNumsFromTable  ( libGAP_SetFiltTab    );
    libGAP_InitResetFiltListTNumsFromTable( libGAP_ResetFiltTab  );


    /* install the shallow copy methods                                    */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_ShallowCopyObjFuncs[ t1            ] = libGAP_ShallowCopyPlist;
        libGAP_ShallowCopyObjFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_ShallowCopyPlist;
    }


    /* install the copy list methods                                       */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_CopyObjFuncs [ t1                     ] = libGAP_CopyPlist;
        libGAP_CopyObjFuncs [ t1 +libGAP_IMMUTABLE          ] = libGAP_CopyPlist;
        libGAP_CopyObjFuncs [ t1            +libGAP_COPYING ] = libGAP_CopyPlistCopy;
        libGAP_CopyObjFuncs [ t1 +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CopyPlistCopy;
        libGAP_CleanObjFuncs[ t1                     ] = libGAP_CleanPlist;
        libGAP_CleanObjFuncs[ t1 +libGAP_IMMUTABLE          ] = libGAP_CleanPlist;
        libGAP_CleanObjFuncs[ t1            +libGAP_COPYING ] = libGAP_CleanPlistCopy;
        libGAP_CleanObjFuncs[ t1 +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CleanPlistCopy;
    }


    /* install the comparison methods                                      */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1++ ) {
        for ( t2 = libGAP_T_PLIST; t2 <= libGAP_LAST_PLIST_TNUM; t2++ ) {
            libGAP_EqFuncs[ t1 ][ t2 ] = libGAP_EqPlist;
            libGAP_LtFuncs[ t1 ][ t2 ] = libGAP_LtPlist;
        }
    }


    /* install the list length methods                                     */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_LenListFuncs[ t1            ] = libGAP_LenPlist;
        libGAP_LenListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_LenPlist;
    }
    libGAP_LenListFuncs[ libGAP_T_PLIST_EMPTY           ] = libGAP_LenPlistEmpty;
    libGAP_LenListFuncs[ libGAP_T_PLIST_EMPTY+libGAP_IMMUTABLE ] = libGAP_LenPlistEmpty;


    /* install the list element test methods                               */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_IsbListFuncs  [ t1            ] = libGAP_IsbPlist;
        libGAP_IsbListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbPlist;
    }
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_IsbListFuncs  [ t1            ] = libGAP_IsbPlistDense;
        libGAP_IsbListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbPlistDense;
    }
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_IsbvListFuncs [ t1            ] = libGAP_IsbvPlist;
        libGAP_IsbvListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbvPlist;
    }
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_IsbvListFuncs [ t1            ] = libGAP_IsbvPlistDense;
        libGAP_IsbvListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_IsbvPlistDense;
    }


    /* install the list element methods                                    */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_Elm0ListFuncs [ t1            ] = libGAP_Elm0Plist;
        libGAP_Elm0ListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_Elm0Plist;
    }
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_Elm0vListFuncs[ t1            ] = libGAP_Elm0vPlist;
        libGAP_Elm0vListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_Elm0vPlist;
    }
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_T_PLIST_NDENSE; t1 += 2 ) {
        libGAP_ElmListFuncs  [ t1            ] = libGAP_ElmPlist;
        libGAP_ElmListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmPlist;
    }
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_ElmListFuncs  [ t1            ] = libGAP_ElmPlistDense;
        libGAP_ElmListFuncs  [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmPlistDense;
    }
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_ElmvListFuncs [ t1            ] = libGAP_ElmvPlist;
        libGAP_ElmvListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmvPlist;
    }
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_ElmvListFuncs [ t1            ] = libGAP_ElmvPlistDense;
        libGAP_ElmvListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmvPlistDense;
    }
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_ElmwListFuncs [ t1            ] = libGAP_ElmvPlistDense;
        libGAP_ElmwListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmvPlistDense;
    }
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_ElmwListFuncs [ t1            ] = libGAP_ElmvPlistDense;
        libGAP_ElmwListFuncs [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmvPlistDense;
    }


    /* install the list elements methods                                   */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_T_PLIST_NDENSE; t1 += 2 ) {
        libGAP_ElmsListFuncs   [ t1            ] = libGAP_ElmsPlist;
        libGAP_ElmsListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmsPlist;
    }
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_ElmsListFuncs   [ t1            ] = libGAP_ElmsPlistDense;
        libGAP_ElmsListFuncs   [ t1 +libGAP_IMMUTABLE ] = libGAP_ElmsPlistDense;
    }


    /* install the list unbind methods                                     */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_UnbListFuncs    [ t1            ] = libGAP_UnbPlist;
        libGAP_UnbListFuncs    [ t1+libGAP_IMMUTABLE  ] = libGAP_UnbPlistImm;
    }


    /* install the list assignment methods                                 */
    libGAP_AssListFuncs    [ libGAP_T_PLIST           ] = libGAP_AssPlist;
    libGAP_AssListFuncs    [ libGAP_T_PLIST+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_NDENSE    ] = libGAP_AssPlistXXX;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_NDENSE+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE    ] = libGAP_AssPlistDense;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE_NHOM    ] = libGAP_AssPlistDense;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE_NHOM_SSORT    ] = libGAP_AssPlistDense;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE_NHOM_NSORT    ] = libGAP_AssPlistDense;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_EMPTY    ] = libGAP_AssPlistEmpty;
    libGAP_AssListFuncs    [ libGAP_T_PLIST_EMPTY+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    
    
    
    for ( t1 = libGAP_T_PLIST_HOM; t1 < libGAP_T_PLIST_CYC; t1 += 2 ) {
      libGAP_AssListFuncs[ t1+libGAP_IMMUTABLE      ] = libGAP_AssPlistImm;
      libGAP_AssListFuncs[ t1                ] = libGAP_AssPlistHomog;
    }

    for ( t1 = libGAP_T_PLIST_CYC; t1 <= libGAP_T_PLIST_CYC_SSORT; t1 += 2 ) {
      libGAP_AssListFuncs[ t1+libGAP_IMMUTABLE      ] = libGAP_AssPlistImm;
      libGAP_AssListFuncs[ t1                ] = libGAP_AssPlistCyc;
    }

    libGAP_AssListFuncs[ libGAP_T_PLIST_FFE+libGAP_IMMUTABLE ] = libGAP_AssPlistImm;
    libGAP_AssListFuncs[ libGAP_T_PLIST_FFE           ] = libGAP_AssPlistFfe;

    /* install the list assignments methods                                */
    libGAP_AsssListFuncs   [ libGAP_T_PLIST            ] = libGAP_AsssPlist;
    libGAP_AsssListFuncs   [ libGAP_T_PLIST +libGAP_IMMUTABLE ] = libGAP_AsssPlistImm;
    for ( t1 = libGAP_T_PLIST_NDENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_AsssListFuncs   [ t1             ] = libGAP_AsssPlistXXX;
        libGAP_AsssListFuncs   [ t1 +libGAP_IMMUTABLE  ] = libGAP_AsssPlistImm;
    }


    /* install the dense list test methods                                 */
    libGAP_IsDenseListFuncs[ libGAP_T_PLIST                   ] = libGAP_IsDensePlist;
    libGAP_IsDenseListFuncs[ libGAP_T_PLIST        +libGAP_IMMUTABLE ] = libGAP_IsDensePlist;
    libGAP_IsDenseListFuncs[ libGAP_T_PLIST_NDENSE            ] = libGAP_IsDensePlistNot;
    libGAP_IsDenseListFuncs[ libGAP_T_PLIST_NDENSE +libGAP_IMMUTABLE ] = libGAP_IsDensePlistNot;
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_IsDenseListFuncs[ t1            ] = libGAP_IsDensePlistYes;
        libGAP_IsDenseListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsDensePlistYes;
    }


    /* install the homogeneous list test methods                           */
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST                       ] = libGAP_IsHomogPlist;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST            +libGAP_IMMUTABLE ] = libGAP_IsHomogPlist;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_NDENSE                ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_NDENSE     +libGAP_IMMUTABLE ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE                 ] = libGAP_IsHomogPlist;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE      +libGAP_IMMUTABLE ] = libGAP_IsHomogPlist;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE_NHOM            ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE_NHOM +libGAP_IMMUTABLE ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT      ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT      ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT +libGAP_IMMUTABLE ] = libGAP_IsHomogPlistNot;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_EMPTY                 ] = libGAP_IsHomogPlistYes;
    libGAP_IsHomogListFuncs[ libGAP_T_PLIST_EMPTY      +libGAP_IMMUTABLE ] = libGAP_IsHomogPlistYes;
    for ( t1 = libGAP_T_PLIST_HOM; t1 <= libGAP_LAST_PLIST_TNUM; t1 += 2 ) {
        libGAP_IsHomogListFuncs[ t1            ] = libGAP_IsHomogPlistYes;
        libGAP_IsHomogListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsHomogPlistYes;
    }


    /* install the equal length list test methods                          */
    libGAP_IsTableListFuncs[ libGAP_T_PLIST                       ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST            +libGAP_IMMUTABLE ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_NDENSE                ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_NDENSE     +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE                 ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE      +libGAP_IMMUTABLE ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE_NHOM            ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE_NHOM +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT      ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT      ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_EMPTY                 ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_EMPTY      +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_HOM                   ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_HOM        +libGAP_IMMUTABLE ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_HOM_NSORT             ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_HOM_NSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_HOM_SSORT             ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_HOM_SSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlist;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB                   ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB        +libGAP_IMMUTABLE ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_NSORT             ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_NSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_SSORT             ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_SSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_RECT                   ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_RECT        +libGAP_IMMUTABLE ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_RECT_NSORT             ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_RECT_NSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_RECT_SSORT             ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_TAB_RECT_SSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlistYes;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_CYC                   ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_CYC        +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_CYC_NSORT             ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_CYC_NSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_CYC_SSORT             ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_CYC_SSORT  +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_FFE                   ] = libGAP_IsTablePlistNot;
    libGAP_IsTableListFuncs[ libGAP_T_PLIST_FFE        +libGAP_IMMUTABLE ] = libGAP_IsTablePlistNot;


    /* install the strictly sorted list test methods                       */
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST                      ] = libGAP_IsSSortPlist;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST           +libGAP_IMMUTABLE ] = libGAP_IsSSortPlist;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_NDENSE               ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_NDENSE    +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE                ] = libGAP_IsSSortPlistDense;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE     +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistDense;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE_NHOM           ] = libGAP_IsSSortPlistDense;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE_NHOM+libGAP_IMMUTABLE ] = libGAP_IsSSortPlistDense;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT     ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT+libGAP_IMMUTABLE ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT     ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_DENSE_NHOM_NSORT+libGAP_IMMUTABLE ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_EMPTY                ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_EMPTY     +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_HOM                  ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_HOM       +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_HOM_NSORT            ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_HOM_NSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_HOM_SSORT            ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB                  ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB       +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_NSORT            ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_NSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_SSORT            ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_RECT                  ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_RECT       +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_RECT_NSORT            ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_RECT_NSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_RECT_SSORT            ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_CYC                  ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_CYC       +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_CYC_NSORT            ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_CYC_NSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistNot;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_CYC_SSORT            ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistYes;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_FFE                  ] = libGAP_IsSSortPlistHom;
    libGAP_IsSSortListFuncs[ libGAP_T_PLIST_FFE       +libGAP_IMMUTABLE ] = libGAP_IsSSortPlistHom;


    /* install the position list test methods                              */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_T_PLIST_FFE; t1 += 2 ) {
        libGAP_IsPossListFuncs[ t1            ] = libGAP_IsPossPlist;
        libGAP_IsPossListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_IsPossPlist;
    }


    /* install the position list methods                                   */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_T_PLIST_NDENSE; t1 += 2 ) {
        libGAP_PosListFuncs[ t1            ] = libGAP_PosPlist;
        libGAP_PosListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_PosPlist;
    }
    for ( t1 = libGAP_T_PLIST_DENSE; t1 <= libGAP_T_PLIST_FFE; t1 += 2 ) {
        libGAP_PosListFuncs[ t1            ] = libGAP_PosPlistDense;
        libGAP_PosListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_PosPlistDense;
    }
    
    libGAP_PosListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT     ] = libGAP_PosPlistSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_DENSE_NHOM_SSORT +libGAP_IMMUTABLE ] = libGAP_PosPlistSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_HOM_SSORT            ] = libGAP_PosPlistHomSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_HOM_SSORT +libGAP_IMMUTABLE ] = libGAP_PosPlistHomSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_TAB_SSORT            ] = libGAP_PosPlistHomSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_TAB_SSORT +libGAP_IMMUTABLE ] = libGAP_PosPlistHomSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_TAB_RECT_SSORT            ] = libGAP_PosPlistHomSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_TAB_RECT_SSORT +libGAP_IMMUTABLE ] = libGAP_PosPlistHomSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_CYC_SSORT            ] = libGAP_PosPlistHomSort;
    libGAP_PosListFuncs[ libGAP_T_PLIST_CYC_SSORT +libGAP_IMMUTABLE ] = libGAP_PosPlistHomSort;


    /* install the plain list methods                                      */
    for ( t1 = libGAP_T_PLIST; t1 <= libGAP_T_PLIST_FFE; t1 += 2 ) {
        libGAP_PlainListFuncs[ t1            ] = libGAP_PlainPlist;
        libGAP_PlainListFuncs[ t1 +libGAP_IMMUTABLE ] = libGAP_PlainPlist;
    }

    for (t1 = libGAP_T_PLIST; t1 < libGAP_T_PLIST_DENSE_NHOM; t1 += 2 ) 
      libGAP_MakeImmutableObjFuncs[ t1 ] = libGAP_MakeImmutablePlistInHom;

    for (t1 = libGAP_T_PLIST_DENSE_NHOM; t1 <= libGAP_T_PLIST_FFE; t1 += 2 ) 
      libGAP_MakeImmutableObjFuncs[ t1 ] = libGAP_MakeImmutablePlistNoMutElms;

    /* mutable tables may have mutable rows */
      libGAP_MakeImmutableObjFuncs[libGAP_T_PLIST_TAB] = libGAP_MakeImmutablePlistInHom;
    


      
    
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoPlist() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "plist",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoPlist ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  plist.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
