/****************************************************************************
**
*W  macfloat.c                   GAP source                      Steve Linton
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions for the macfloat package.
**
** macfloats are stored as bags containing a 64 bit value
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */

#include        "gap.h"                 /* error handling, initialisation  */


#include        "plist.h"               /* lists */
#include        "ariths.h"              /* basic arithmetic                */
#include        "integer.h"             /* basic arithmetic                */

#include        "macfloat.h"                /* macfloateans                        */

#include        "bool.h"
#include        "scanner.h"
#include        "string.h"
#include        <assert.h>

/* the following two declarations would belong in `saveload.h', but then all
 * files get macfloat dependencies */
extern libGAP_Double libGAP_LoadDouble( void);
extern void libGAP_SaveDouble( libGAP_Double d);

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_MATH_H
#include <math.h>
#endif

#include <stdlib.h>

#define libGAP_SIZE_MACFLOAT   sizeof(libGAP_Double)

/****************************************************************************
**
*F  TypeMacfloat( <macfloat> )  . . . . . . . . . . . . . . . kind of a macfloat value
**
**  'TypeMacfloat' returns the kind of macfloatean values.
**
**  'TypeMacfloat' is the function in 'TypeObjFuncs' for macfloatean values.
*/
libGAP_Obj libGAP_TYPE_MACFLOAT;

libGAP_Obj libGAP_TypeMacfloat (
    libGAP_Obj                 val )
{  
    return libGAP_TYPE_MACFLOAT;
}


/****************************************************************************
**
*F  PrintMacfloat( <macfloat> ) . . . . . . . . . . . . . . . . print a macfloat value
**
**  'PrintMacfloat' prints the macfloating value <macfloat>.
*/
void libGAP_PrintMacfloat (
    libGAP_Obj                 x )
{
  libGAP_Char buf[32];
  snprintf(buf, sizeof(buf), "%.16" libGAP_PRINTFFORMAT, (libGAP_TOPRINTFFORMAT) libGAP_VAL_MACFLOAT(x));
  libGAP_Pr("%s",(libGAP_Int)buf, 0);
}


/****************************************************************************
**
*F  EqMacfloat( <macfloatL>, <macfloatR> )  . . . . . . . . .  test if <macfloatL> =  <macfloatR>
**
**  'EqMacfloat' returns 'True' if the two macfloatean values <macfloatL> and <macfloatR> are
**  equal, and 'False' otherwise.
*/
libGAP_Int libGAP_EqMacfloat (
    libGAP_Obj                 macfloatL,
    libGAP_Obj                 macfloatR )
{
  return libGAP_VAL_MACFLOAT(macfloatL) == libGAP_VAL_MACFLOAT(macfloatR);
}

libGAP_Obj libGAP_FuncEqMacfloat (
    libGAP_Obj                 macfloatL,
    libGAP_Obj                 macfloatR )
{
  return libGAP_EqMacfloat(macfloatL,macfloatR) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  LtMacfloat( <macfloatL>, <macfloatR> )  . . . . . . . . .  test if <macfloatL> <  <macfloatR>
**
*/
libGAP_Int libGAP_LtMacfloat (
    libGAP_Obj                 macfloatL,
    libGAP_Obj                 macfloatR )
{
  return libGAP_VAL_MACFLOAT(macfloatL) < libGAP_VAL_MACFLOAT(macfloatR);
}


/****************************************************************************
**
*F  IsMacfloatFilt( <self>, <obj> ) . . . . . . . . . .  test for a macfloatean value
**
**  'IsMacfloatFilt' implements the internal filter 'IsMacfloat'.
**
**  'IsMacfloat( <obj> )'
**
**  'IsMacfloat'  returns  'true'  if  <obj>  is   a macfloatean  value  and  'false'
**  otherwise.
*/
libGAP_Obj libGAP_IsMacfloatFilt;

libGAP_Obj libGAP_IsMacfloatHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
  return libGAP_IS_MACFLOAT(obj) ? libGAP_True : libGAP_False;
}



/****************************************************************************
**
*F  SaveMacfloat( <macfloat> ) . . . . . . . . . . . . . . . . . . . . save a Macfloatean 
**
*/

void libGAP_SaveMacfloat( libGAP_Obj obj )
{
  libGAP_SaveDouble(libGAP_VAL_MACFLOAT(obj));
  return;
}

/****************************************************************************
**
*F  LoadMacfloat( <macfloat> ) . . . . . . . . . . . . . . . . . . . . save a Macfloatean 
**
*/

void libGAP_LoadMacfloat( libGAP_Obj obj )
{
  libGAP_SET_VAL_MACFLOAT(obj, libGAP_LoadDouble());
}

libGAP_Obj libGAP_NEW_MACFLOAT( libGAP_Double val )
{
  libGAP_Obj f;
  f = libGAP_NewBag(libGAP_T_MACFLOAT,libGAP_SIZE_MACFLOAT);
  libGAP_SET_VAL_MACFLOAT(f,val);
  return f;
}

/****************************************************************************
**
*F  ZeroMacfloat(<macfloat> ) . . . . . . . . . . . . . . . . . . . return the zero 
**
*/


libGAP_Obj libGAP_ZeroMacfloat( libGAP_Obj f )
{
  return libGAP_NEW_MACFLOAT((libGAP_Double)0.0);
}

/****************************************************************************
**
*F  AinvMacfloat(<macfloat> ) . . . . . . . . . . . . . . . . . . . unary minus 
**
*/


libGAP_Obj libGAP_AInvMacfloat( libGAP_Obj f )
{
  return libGAP_NEW_MACFLOAT(-libGAP_VAL_MACFLOAT(f));
}

/****************************************************************************
**
*F  OneMacfloat(<macfloat> ) . . . . . . . . . . . . . . . . . . . return the one 
**
*/


libGAP_Obj libGAP_OneMacfloat( libGAP_Obj f )
{
  return libGAP_NEW_MACFLOAT((libGAP_Double)1.0);
}

/****************************************************************************
**
*F  InvMacfloat(<macfloat> ) . . . . . . . . . . . . . . . . . . . reciprocal
**
*/


libGAP_Obj libGAP_InvMacfloat( libGAP_Obj f )
{
  return libGAP_NEW_MACFLOAT((libGAP_Double)1.0/libGAP_VAL_MACFLOAT(f));
}

/****************************************************************************
**
*F  ProdMacfloat(<macfloatl>, <macfloatr> ) . . . . . . . . . . . . . . . product
**
*/


libGAP_Obj libGAP_ProdMacfloat( libGAP_Obj fl, libGAP_Obj fr )
{
  return libGAP_NEW_MACFLOAT(libGAP_VAL_MACFLOAT(fl)*libGAP_VAL_MACFLOAT(fr));
}

/****************************************************************************
**
*F  PowMacfloat(<macfloatl>, <macfloatr> ) . . . . . . . . . . . . . . exponentiation
**
*/


libGAP_Obj libGAP_PowMacfloat( libGAP_Obj fl, libGAP_Obj fr )
{
  return libGAP_NEW_MACFLOAT(libGAP_MATH(pow)(libGAP_VAL_MACFLOAT(fl),libGAP_VAL_MACFLOAT(fr)));
}

/****************************************************************************
**
*F  SumMacfloat(<macfloatl>, <macfloatr> ) . . . . . . . . . . . . . .  sum
**
*/


libGAP_Obj libGAP_SumMacfloat( libGAP_Obj fl, libGAP_Obj fr )
{
  return libGAP_NEW_MACFLOAT(libGAP_VAL_MACFLOAT(fl)+libGAP_VAL_MACFLOAT(fr));
}

/****************************************************************************
**
*F  DiffMacfloat(<macfloatl>, <macfloatr> ) . . . . . . . . . . . . . . difference
**
*/


libGAP_Obj libGAP_DiffMacfloat( libGAP_Obj fl, libGAP_Obj fr )
{
  return libGAP_NEW_MACFLOAT(libGAP_VAL_MACFLOAT(fl)-libGAP_VAL_MACFLOAT(fr));
}

/****************************************************************************
**
*F  QuoMacfloat(<macfloatl>, <macfloatr> ) . . . . . . . . . . . . . . quotient
**
*/


libGAP_Obj libGAP_QuoMacfloat( libGAP_Obj fl, libGAP_Obj fr )
{
  return libGAP_NEW_MACFLOAT(libGAP_VAL_MACFLOAT(fl)/libGAP_VAL_MACFLOAT(fr));
}

/****************************************************************************
**
*F  LQuoMacfloat(<macfloatl>, <macfloatr> ) . . . . . . . . . . . . . .left quotient
**
*/


libGAP_Obj libGAP_LQuoMacfloat( libGAP_Obj fl, libGAP_Obj fr )
{
  return libGAP_NEW_MACFLOAT(libGAP_VAL_MACFLOAT(fr)/libGAP_VAL_MACFLOAT(fl));
}

/****************************************************************************
**
*F  ModMacfloat(<macfloatl>, <macfloatr> ) . . . . . . . . . . . . . . .mod
**
*/


libGAP_Obj libGAP_ModMacfloat( libGAP_Obj fl, libGAP_Obj fr )
{
  return libGAP_NEW_MACFLOAT(libGAP_MATH(fmod)(libGAP_VAL_MACFLOAT(fl),libGAP_VAL_MACFLOAT(fr)));
}


/****************************************************************************
**
*F  FuncMACFLOAT_INT(<int>) . . . . . . . . . . . . . . . conversion
**
*/

libGAP_Obj libGAP_FuncMACFLOAT_INT( libGAP_Obj self, libGAP_Obj i )
{
  if (!libGAP_IS_INTOBJ(i))
    return libGAP_Fail;
  else
    return libGAP_NEW_MACFLOAT((libGAP_Double)libGAP_INT_INTOBJ(i));
}

/****************************************************************************
**
*F  FuncMACFLOAT_STRING(<string>) . . . . . . . . . . . . . . . conversion
**
*/

libGAP_Obj libGAP_FuncMACFLOAT_STRING( libGAP_Obj self, libGAP_Obj s )
{

  while (!libGAP_IsStringConv(s))
    {
      s = libGAP_ErrorReturnObj("MACFLOAT_STRING: object to be converted must be a string not a %s",
			 (libGAP_Int)(libGAP_InfoBags[libGAP_TNUM_OBJ(s)].name),0,"You can return a string to continue" );
    }
  char * endptr;
  libGAP_UChar *sp = libGAP_CHARS_STRING(s);
  libGAP_Obj res= libGAP_NEW_MACFLOAT((libGAP_Double) libGAP_STRTOD((char *)sp,&endptr));
  if ((libGAP_UChar *)endptr != sp + libGAP_GET_LEN_STRING(s)) 
    return libGAP_Fail;
  return res;
}

/****************************************************************************
**
*F SumIntMacfloat( <int>, <macfloat> )
**
*/

libGAP_Obj libGAP_SumIntMacfloat( libGAP_Obj i, libGAP_Obj f )
{
  return libGAP_NEW_MACFLOAT( (libGAP_Double)(libGAP_INT_INTOBJ(i)) + libGAP_VAL_MACFLOAT(f));
}


/****************************************************************************
**
*F FuncSIN_MACFLOAT( <self>, <macfloat> ) . .The sin function from the math library
**
*/

#define libGAP_MAKEMATHPRIMITIVE(NAME,name)			\
  libGAP_Obj Func##NAME##_MACFLOAT( libGAP_Obj self, libGAP_Obj f )		\
  {							\
    return libGAP_NEW_MACFLOAT(libGAP_MATH(name)(libGAP_VAL_MACFLOAT(f)));	\
  }

#define libGAP_MAKEMATHPRIMITIVE2(NAME,name)					\
  libGAP_Obj Func##NAME##_MACFLOAT( libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g)			\
  {									\
    return libGAP_NEW_MACFLOAT(libGAP_MATH(name)(libGAP_VAL_MACFLOAT(f),libGAP_VAL_MACFLOAT(libGAP_g)));	\
  }

libGAP_MAKEMATHPRIMITIVE(COS,cos)
libGAP_MAKEMATHPRIMITIVE(SIN,sin)
libGAP_MAKEMATHPRIMITIVE(TAN,tan)
libGAP_MAKEMATHPRIMITIVE(ACOS,acos)
libGAP_MAKEMATHPRIMITIVE(ASIN,asin)
libGAP_MAKEMATHPRIMITIVE(ATAN,atan)
libGAP_MAKEMATHPRIMITIVE(LOG,log)
libGAP_MAKEMATHPRIMITIVE(EXP,exp)
#if HAVE_LOG2
libGAP_MAKEMATHPRIMITIVE(LOG2,log2)
#endif
#if HAVE_LOG10
libGAP_MAKEMATHPRIMITIVE(LOG10,log10)
#endif
#if HAVE_LOG1P
libGAP_MAKEMATHPRIMITIVE(LOG1P,log1p)
#endif
#if HAVE_EXP2
libGAP_MAKEMATHPRIMITIVE(EXP2,exp2)
#endif
#if HAVE_EXPM1
libGAP_MAKEMATHPRIMITIVE(EXPM1,expm1)
#endif
#if HAVE_EXP10
libGAP_MAKEMATHPRIMITIVE(EXP10,exp10)
#endif
libGAP_MAKEMATHPRIMITIVE(SQRT,sqrt)
libGAP_MAKEMATHPRIMITIVE(RINT,rint)
libGAP_MAKEMATHPRIMITIVE(FLOOR,floor)
libGAP_MAKEMATHPRIMITIVE(CEIL,ceil)
libGAP_MAKEMATHPRIMITIVE(ABS,fabs)
libGAP_MAKEMATHPRIMITIVE2(ATAN2,atan2)
libGAP_MAKEMATHPRIMITIVE2(HYPOT,hypot)

extern libGAP_Obj libGAP_FuncIntHexString(libGAP_Obj,libGAP_Obj);

libGAP_Obj FuncINTFLOOR_MACFLOAT( libGAP_Obj self, libGAP_Obj obj )
{
#if HAVE_TRUNC
  libGAP_Double f = trunc(libGAP_VAL_MACFLOAT(obj));
#else
  libGAP_Double f = libGAP_VAL_MACFLOAT(obj);
  if (f >= 0.0)
    f = floor(f);
  else
    f = -floor(-f);
#endif


  if (fabs(f) < (libGAP_Double) (1L<<libGAP_NR_SMALL_INT_BITS))
    return libGAP_INTOBJ_INT((libGAP_Int)f);

  int str_len = (int) (log(fabs(f)) / log(16.0)) + 3;

  libGAP_Obj str = libGAP_NEW_STRING(str_len);
  char *s = libGAP_CSTR_STRING(str), *p = s+str_len-1;
  if (f < 0.0)
    f = -f, s[0] = '-';
  while (p > s || (p == s && s[0] != '-')) {
    int d = (int) fmod(f,16.0);
    *p-- = d < 10 ? '0'+d : 'a'+d-10;
    f /= 16.0;
  }
  return libGAP_FuncIntHexString(self,str);
}

libGAP_Obj libGAP_FuncSTRING_DIGITS_MACFLOAT( libGAP_Obj self, libGAP_Obj gapprec, libGAP_Obj f)
{
  libGAP_Char buf[50];
  libGAP_Obj str;
  int prec = libGAP_INT_INTOBJ(gapprec);
  if (prec > 40) /* too much anyways, and would risk buffer overrun */
    prec = 40;
  snprintf(buf, sizeof(buf), "%.*" libGAP_PRINTFFORMAT, prec, (libGAP_TOPRINTFFORMAT)libGAP_VAL_MACFLOAT(f));
  libGAP_C_NEW_STRING_DYN(str, buf);
  return str;
}

libGAP_Obj FuncSTRING_MACFLOAT( libGAP_Obj self, libGAP_Obj f) /* backwards compatibility */
{
  return libGAP_FuncSTRING_DIGITS_MACFLOAT(self,libGAP_INTOBJ_INT(libGAP_PRINTFDIGITS),f);
}

libGAP_Obj libGAP_FuncLDEXP_MACFLOAT( libGAP_Obj self, libGAP_Obj f, libGAP_Obj i)
{
  return libGAP_NEW_MACFLOAT(ldexp(libGAP_VAL_MACFLOAT(f),libGAP_INT_INTOBJ(i)));
}

libGAP_Obj FuncFREXP_MACFLOAT( libGAP_Obj self, libGAP_Obj f)
{
  int i;
  libGAP_Obj d = libGAP_NEW_MACFLOAT(frexp (libGAP_VAL_MACFLOAT(f), &i));
  libGAP_Obj l = libGAP_NEW_PLIST(libGAP_T_PLIST,2);
  libGAP_SET_ELM_PLIST(l,1,d);
  libGAP_SET_ELM_PLIST(l,2,libGAP_INTOBJ_INT(i));
  libGAP_SET_LEN_PLIST(l,2);
  return l;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_MACFLOAT", "obj", &libGAP_IsMacfloatFilt,
      libGAP_IsMacfloatHandler, "src/macfloat.c:IS_MACFLOAT" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
#define libGAP_GVARENTRY(NAME) { #NAME "_MACFLOAT", 1, "macfloat",		\
      Func##NAME##_MACFLOAT, "src/macfloat.c:" #NAME "_MACFLOAT" }

static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {
  { "MACFLOAT_INT", 1, "int",
    libGAP_FuncMACFLOAT_INT, "src/macfloat.c:MACFLOAT_INT" },

  { "MACFLOAT_STRING", 1, "string",
    libGAP_FuncMACFLOAT_STRING, "src/macfloat.c:MACFLOAT_STRING" },

  libGAP_GVARENTRY(SIN),
  libGAP_GVARENTRY(COS),
  libGAP_GVARENTRY(TAN),
  libGAP_GVARENTRY(ASIN),
  libGAP_GVARENTRY(ACOS),
  libGAP_GVARENTRY(ATAN),

  { "ATAN2_MACFLOAT", 2, "real, imag",
    FuncATAN2_MACFLOAT, "src/macfloat.c:ATAN2_MACFLOAT" },

  { "HYPOT_MACFLOAT", 2, "real, imag",
    FuncHYPOT_MACFLOAT, "src/macfloat.c:HYPOT_MACFLOAT" },

  libGAP_GVARENTRY(LOG),
  libGAP_GVARENTRY(EXP),
#if HAVE_LOG2
  libGAP_GVARENTRY(LOG2),
#endif
#if HAVE_LOG10
  libGAP_GVARENTRY(LOG10),
#endif  
#if HAVE_LOG1P
  libGAP_GVARENTRY(LOG1P),
#endif  
#if HAVE_EXP2
  libGAP_GVARENTRY(EXP2),
#endif  
#if HAVE_EXPM1
  libGAP_GVARENTRY(EXPM1),
#endif
#if HAVE_EXP10
  libGAP_GVARENTRY(EXP10),
#endif

  { "LDEXP_MACFLOAT", 2, "macfloat, int",
    libGAP_FuncLDEXP_MACFLOAT, "src/macfloat.c:LDEXP_MACFLOAT" },

  libGAP_GVARENTRY(FREXP),
  libGAP_GVARENTRY(SQRT),
  libGAP_GVARENTRY(RINT),
  libGAP_GVARENTRY(INTFLOOR),
  libGAP_GVARENTRY(FLOOR),
  libGAP_GVARENTRY(CEIL),
  libGAP_GVARENTRY(ABS),
  libGAP_GVARENTRY(STRING),

  { "STRING_DIGITS_MACFLOAT", 2, "digits, macfloat",
    libGAP_FuncSTRING_DIGITS_MACFLOAT, "src/macfloat.c:STRING_DIGITS_MACFLOAT" },

  { "EQ_MACFLOAT", 2, "x, y",
    libGAP_FuncEqMacfloat, "src/macfloat.c:FuncEqMacfloat" },

  {0}
};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int libGAP_EqObject (libGAP_Obj,libGAP_Obj);

    /* install the marking functions for macfloatean values                    */
    libGAP_InfoBags[ libGAP_T_MACFLOAT ].name = "macfloat";
    libGAP_InitMarkFuncBags( libGAP_T_MACFLOAT, libGAP_MarkNoSubBags );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_MACFLOAT", &libGAP_TYPE_MACFLOAT );
    libGAP_TypeObjFuncs[ libGAP_T_MACFLOAT ] = libGAP_TypeMacfloat;

    /* install the saving functions                                       */
    libGAP_SaveObjFuncs[ libGAP_T_MACFLOAT ] = libGAP_SaveMacfloat;

    /* install the loading functions                                       */
    libGAP_LoadObjFuncs[ libGAP_T_MACFLOAT ] = libGAP_LoadMacfloat;

    /* install the printer for macfloatean values                              */
    libGAP_PrintObjFuncs[ libGAP_T_MACFLOAT ] = libGAP_PrintMacfloat;

    /* install the comparison functions                                    */
    libGAP_EqFuncs[ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_EqMacfloat;
    libGAP_LtFuncs[ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_LtMacfloat;

    /* allow method selection to protest against comparisons of float and int */
    {
      int t;
      for (t = libGAP_T_INT; t <= libGAP_T_CYC; t++)
	libGAP_EqFuncs[libGAP_T_MACFLOAT][t] = libGAP_EqFuncs[t][libGAP_T_MACFLOAT] = libGAP_EqObject;
    }

    /* install the unary arithmetic methods                                */
    libGAP_ZeroFuncs[ libGAP_T_MACFLOAT ] = libGAP_ZeroMacfloat;
    libGAP_ZeroMutFuncs[ libGAP_T_MACFLOAT ] = libGAP_ZeroMacfloat;
    libGAP_AInvMutFuncs[ libGAP_T_MACFLOAT ] = libGAP_AInvMacfloat;
    libGAP_OneFuncs [ libGAP_T_MACFLOAT ] = libGAP_OneMacfloat;
    libGAP_OneMutFuncs [ libGAP_T_MACFLOAT ] = libGAP_OneMacfloat;
    libGAP_InvFuncs [ libGAP_T_MACFLOAT ] = libGAP_InvMacfloat;

    /* install binary arithmetic methods */
    libGAP_ProdFuncs[ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_ProdMacfloat;
    libGAP_PowFuncs [ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_PowMacfloat;
    libGAP_SumFuncs[ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_SumMacfloat;
    libGAP_DiffFuncs [ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_DiffMacfloat;
    libGAP_QuoFuncs [ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_QuoMacfloat;
    libGAP_LQuoFuncs [ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_LQuoMacfloat;
    libGAP_ModFuncs [ libGAP_T_MACFLOAT ][ libGAP_T_MACFLOAT ] = libGAP_ModMacfloat;
    libGAP_SumFuncs [ libGAP_T_INT ][ libGAP_T_MACFLOAT ] = libGAP_SumIntMacfloat;
    
    /* Probably support mixed ops with small ints in the kernel as well
       on any reasonable system, all small ints should have macfloat equivalents

       Anything else, like mixed ops with rationals, we can leave to the library
       at least for a while */
     
    
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
/*  UInt            gvar; */
/*  Obj             tmp;  */

    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoMacfloat()  . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "macfloat",                             /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoMacfloat ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  macfloat.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
