/****************************************************************************
**
*W  listfunc.c                  GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions for generic lists.
*/
#include        "system.h"              /* Ints, UInts                     */



#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "string.h"              /* strings                         */

#include        "bool.h"                /* booleans                        */

#include        "permutat.h"            /* permutations                    */
#include        "trans.h"               /* transformations                 */
#include        "pperm.h"               /* partial perms                   */

#include        "listfunc.h"            /* functions for generic lists     */

#include        "plist.h"               /* plain lists                     */
#include        "set.h"                 /* plain sets                      */
#include        "range.h"               /* ranges                          */

#include                <string.h>
#include                <stdlib.h> 

/****************************************************************************
**

*F  AddList(<list>,<obj>) . . . . . . . .  add an object to the end of a list
**
**  'AddList' adds the object <obj> to the end  of  the  list  <list>,  i.e.,
**  it is equivalent to the assignment '<list>[ Length(<list>)+1 ] := <obj>'.
**  The  list is  automatically extended to   make room for  the new element.
**  'AddList' returns nothing, it is called only for its side effect.
*/
void            libGAP_AddList (
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    libGAP_Int                 pos;            /* position to assign to           */
    pos = libGAP_LEN_LIST( list ) + 1;
    libGAP_ASS_LIST( list, pos, obj );
}

extern libGAP_Obj libGAP_FuncADD_LIST(
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj );

void            libGAP_AddPlist (
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    libGAP_Int                 pos;            /* position to assign to           */

    if ( ! libGAP_IS_MUTABLE_PLIST(list) ) {
        list = libGAP_ErrorReturnObj(
                "Lists Assignment: <list> must be a mutable list",
                0L, 0L,
                "you may replace <list> via 'return <list>;'" );
        libGAP_FuncADD_LIST( 0, list, obj );
        return;
    }
    /* in order to be optimistic when building list call assignment        */
    pos = libGAP_LEN_PLIST( list ) + 1;
    if ( pos == 1 ) {
        libGAP_AssPlistEmpty( list, pos, obj );
    }
    else {
      libGAP_ASS_LIST( list, pos, obj);
      /*  The code below is commented out and replaced by the line above, so
          as, at the cost of one extra dispatch, to take advantage of the
          special code in AssPlist<things> which maintain as much information
          about denseness homogeneity, etc as possible */
      /*        RetypeBag( list, T_PLIST );
                GROW_PLIST( list, pos );
                SET_LEN_PLIST( list, pos );
                SET_ELM_PLIST( list, pos, obj );
                CHANGED_BAG( list ); */
    }
}

libGAP_Obj libGAP_AddListOper;

libGAP_Obj libGAP_FuncADD_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    /* dispatch                                                            */
    if ( libGAP_IS_PLIST( list ) ) {
        libGAP_AddPlist( list, obj );
    }
    else if ( libGAP_TNUM_OBJ( list ) < libGAP_FIRST_EXTERNAL_TNUM ) {
        libGAP_AddList( list, obj );
    }
    else {
        libGAP_DoOperation2Args( self, list, obj );
    }

    /* return nothing                                                      */
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  RemList(<list>) . . . . . . . .  add an object to the end of a list
**
**  'RemList' removes the last object <obj> from the end  of  the  list  
** <list>,  and returns it.
*/
libGAP_Obj            libGAP_RemList (
    libGAP_Obj                 list)
{
    libGAP_Int                 pos; 
    libGAP_Obj result;
    pos = libGAP_LEN_LIST( list ) ;
    if ( pos == 0L ) {
        list = libGAP_ErrorReturnObj(
                "Remove: <list> must not be empty",
                0L, 0L,
                "you may replace <list> via 'return <list>;'" );
        return libGAP_RemList(list);
    }
    result = libGAP_ELM_LIST(list, pos);
    libGAP_UNB_LIST(list, pos);
    return result;
}

extern libGAP_Obj libGAP_FuncREM_LIST(
    libGAP_Obj                 self,
    libGAP_Obj                 list);

libGAP_Obj            libGAP_RemPlist (
                          libGAP_Obj                 list)
{
    libGAP_Int                 pos;           
    libGAP_Obj removed; 

    if ( ! libGAP_IS_MUTABLE_PLIST(list) ) {
        list = libGAP_ErrorReturnObj(
                "Remove: <list> must be a mutable list",
                0L, 0L,
                "you may replace <list> via 'return <list>;'" );
        return libGAP_FuncREM_LIST( 0, list);
    }
    pos = libGAP_LEN_PLIST( list );
    if ( pos == 0L ) {
        list = libGAP_ErrorReturnObj(
                "Remove: <list> must not be empty",
                0L, 0L,
                "you may replace <list> via 'return <list>;'" );
        return libGAP_FuncREM_LIST( 0, list);
    }
    removed = libGAP_ELM_PLIST(list, pos);
    libGAP_SET_ELM_PLIST(list, pos, (libGAP_Obj)0L);
    libGAP_SET_LEN_PLIST(list, pos-1);
    if ( pos == 1 ) {
      libGAP_RetypeBag(list, libGAP_T_PLIST_EMPTY);
    }
    if (4*pos*sizeof(libGAP_Obj) < 3*libGAP_SIZE_BAG(list))
      libGAP_SHRINK_PLIST(list, pos);
    return removed;
}

libGAP_Obj libGAP_RemListOper;

libGAP_Obj libGAP_FuncREM_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list)

{
    /* dispatch                                                            */
    if ( libGAP_IS_PLIST( list ) ) {
        return libGAP_RemPlist( list);
    }
    else if ( libGAP_TNUM_OBJ( list ) < libGAP_FIRST_EXTERNAL_TNUM ) {
        return libGAP_RemList( list);
    }
    else {
        return libGAP_DoOperation1Args( self, list);
    }

}


/****************************************************************************
**
*F  FuncAPPEND_LIST_INTR(<list1>,<list2>)  . . . . . append elements to a list
**
**  'FuncAPPEND_LIST_INTR' implements the function 'AppendList'.
**
**  'AppendList(<list1>,<list2>)'
**
**  'AppendList' adds (see "Add") the elements of the list <list2> to the end
**  of the list <list1>. It is allowed that <list2> contains empty positions,
**  in which case the corresponding positions  will be left empty in <list1>.
**  'AppendList' returns nothing, it is called only for its side effect.
*/
libGAP_Obj             libGAP_FuncAPPEND_LIST_INTR (
    libGAP_Obj                 self,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    libGAP_UInt                len1;           /* length of the first list        */
    libGAP_UInt                len2;           /* length of the second list       */
    libGAP_Obj *               ptr1;           /* pointer into the first list     */
    libGAP_Obj *               ptr2;           /* pointer into the second list    */
    libGAP_Obj                 elm;            /* one element of the second list  */
    libGAP_Int                 i;              /* loop variable                   */

    /* check the mutability of the first argument */
    while ( !libGAP_IS_MUTABLE_OBJ( list1) )
      list1 = libGAP_ErrorReturnObj (
                "Append: <list1> must be a mutable list",
                0L, 0L,
                "you can replace <list1> via 'return <list1>;'");
    

    /* handle the case of strings now */
    if ( libGAP_IS_STRING_REP(list1) && libGAP_IS_STRING_REP(list2))
      {
        len1 = libGAP_GET_LEN_STRING(list1);
        len2 = libGAP_GET_LEN_STRING(list2);
        libGAP_GROW_STRING(list1, len1 + len2);
        libGAP_SET_LEN_STRING(list1, len1 + len2);
        memmove( (void *)(libGAP_CHARS_STRING(list1) + len1), 
                (void *)libGAP_CHARS_STRING(list2), len2 + 1);
        /* ensure trailing zero */
        *(libGAP_CHARS_STRING(list1) + len1 + len2) = 0;    
        return (libGAP_Obj) 0;
      }
    
    /* check the type of the first argument                                */
    if ( libGAP_TNUM_OBJ( list1 ) != libGAP_T_PLIST ) {
        while ( ! libGAP_IS_SMALL_LIST( list1 ) ) {
            list1 = libGAP_ErrorReturnObj(
                "AppendList: <list1> must be a small list (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(list1), 0L,
                "you can replace <list1> via 'return <list1>;'" );
        }
        if ( ! libGAP_IS_PLIST( list1 ) ) {
            libGAP_PLAIN_LIST( list1 );
        }
        libGAP_RetypeBag( list1, libGAP_T_PLIST );
    }
    len1 = libGAP_LEN_PLIST( list1 );

    /* check the type of the second argument                               */
    if ( ! libGAP_IS_PLIST( list2 ) ) {
        while ( ! libGAP_IS_SMALL_LIST( list2 ) ) {
            list2 = libGAP_ErrorReturnObj(
                "AppendList: <list2> must be a small list (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(list2), 0L,
                "you can replace <list2> via 'return <list2>;'"  );
        }
        len2 = libGAP_LEN_LIST( list2 );
    }
    else {
        len2 = libGAP_LEN_PLIST( list2 );
    }

    /* if the list has no room at the end, enlarge it                      */
    if ( 0 < len2 ) {
        libGAP_GROW_PLIST( list1, len1+len2 );
        libGAP_SET_LEN_PLIST( list1, len1+len2 );
    }

    /* add the elements                                                    */
    if ( libGAP_IS_PLIST(list2) ) {
        ptr1 = libGAP_ADDR_OBJ(list1) + len1;
        ptr2 = libGAP_ADDR_OBJ(list2);
        for ( i = 1; i <= len2; i++ ) {
            ptr1[i] = ptr2[i];
            /* 'CHANGED_BAG(list1);' not needed, ELM_PLIST does not NewBag */
        }
        libGAP_CHANGED_BAG( list1 );
    }
    else {
        for ( i = 1; i <= len2; i++ ) {
            elm = libGAP_ELMV0_LIST( list2, i );
            libGAP_SET_ELM_PLIST( list1, i+len1, elm );
            libGAP_CHANGED_BAG( list1 );
        }
    }

    /* return void                                                         */
    return (libGAP_Obj)0;
}

libGAP_Obj             libGAP_AppendListOper;

libGAP_Obj             libGAP_FuncAPPEND_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    /* dispatch                                                            */
    if ( libGAP_TNUM_OBJ( list ) < libGAP_FIRST_EXTERNAL_TNUM ) {
        libGAP_FuncAPPEND_LIST_INTR( 0, list, obj );
    }
    else {
        libGAP_DoOperation2Args( self, list, obj );
    }

    /* return nothing                                                      */
    return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  POSITION_SORTED_LIST(<list>,<obj>)  . . . . find an object in a sorted list
*F  PositionSortedDensePlist(<list>,<obj>)  . find an object in a sorted list
**
**  'POSITION_SORTED_LIST' returns the position of the  object <obj>, which may
**  be an object of any type, with respect to the sorted list <list>.
**
**  'POSITION_SORTED_LIST' returns  <pos>  such that  '<list>[<pos>-1] < <obj>'
**  and '<obj> <= <list>[<pos>]'.  That means if <obj> appears once in <list>
**  its position is returned.  If <obj> appears several  times in <list>, the
**  position of the first occurrence is returned.  If <obj> is not an element
**  of <list>, the index where <obj> must be inserted to keep the list sorted
**  is returned.
*/
libGAP_UInt            libGAP_POSITION_SORTED_LIST (
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    libGAP_UInt                l;              /* low                             */
    libGAP_UInt                h;              /* high                            */
    libGAP_UInt                m;              /* mid                             */
    libGAP_Obj                 v;              /* one element of the list         */

    /* perform the binary search to find the position                      */
    l = 0;  h = libGAP_LEN_LIST( list ) + 1;
    while ( l+1 < h ) {                 /* list[l] < obj && obj <= list[h] */
        m = (l + h) / 2;                /* l < m < h                       */
        v = libGAP_ELMV_LIST( list, m );
        if ( libGAP_LT( v, obj ) ) { l = m; }
        else                { h = m; }
    }

    /* return the position                                                 */
    return h;
}

libGAP_UInt            libGAP_PositionSortedDensePlist (
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    libGAP_UInt                l;              /* low                             */
    libGAP_UInt                h;              /* high                            */
    libGAP_UInt                m;              /* mid                             */
    libGAP_Obj                 v;              /* one element of the list         */

    /* perform the binary search to find the position                      */
    l = 0;  h = libGAP_LEN_PLIST( list ) + 1;
    while ( l+1 < h ) {                 /* list[l] < obj && obj <= list[h] */
        m = (l + h) / 2;                /* l < m < h                       */
        v = libGAP_ELM_PLIST( list, m );
        if ( libGAP_LT( v, obj ) ) { l = m; }
        else                { h = m; }
    }

    /* return the position                                                 */
    return h;
}

libGAP_Obj             libGAP_FuncPOSITION_SORTED_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    libGAP_UInt                h;              /* position, result                */

    /* check the first argument                                            */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "POSITION_SORTED_LIST: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }

    /* dispatch                                                            */
    if ( libGAP_IS_DENSE_PLIST(list) ) {
        h = libGAP_PositionSortedDensePlist( list, obj );
    }
    else {
        h = libGAP_POSITION_SORTED_LIST( list, obj );
    }

    /* return the result                                                   */
    return libGAP_INTOBJ_INT( h );
}


/****************************************************************************
**
*F  POSITION_SORTED_LISTComp(<list>,<obj>,<func>)  . . find an object in a list
*F  PositionSortedDensePlistComp(<list>,<obj>,<func>)find an object in a list
**
**  'POSITION_SORTED_LISTComp' returns the position of the  object <obj>, which
**  may be an object of any type, with respect to the list <list>,  which  is
**  sorted with respect to the comparison function <func>.
**
**  'POSITION_SORTED_LISTComp' returns <pos> such that '<list>[<pos>-1] < <obj>'
**  and '<obj> <= <list>[<pos>]'.  That means if <obj> appears once in <list>
**  its position is returned.  If <obj> appears several  times in <list>, the
**  position of the first occurrence is returned.  If <obj> is not an element
**  of <list>, the index where <obj> must be inserted to keep the list sorted
**  is returned.
*/
libGAP_UInt            libGAP_POSITION_SORTED_LISTComp (
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 func )
{
    libGAP_UInt                l;              /* low                             */
    libGAP_UInt                h;              /* high                            */
    libGAP_UInt                m;              /* mid                             */
    libGAP_Obj                 v;              /* one element of the list         */

    /* perform the binary search to find the position                      */
    l = 0;  h = libGAP_LEN_LIST( list ) + 1;
    while ( l+1 < h ) {                 /* list[l] < obj && obj <= list[h] */
        m = (l + h) / 2;                /* l < m < h                       */
        v = libGAP_ELMV_LIST( list, m );
        if ( libGAP_CALL_2ARGS( func, v, obj ) == libGAP_True ) { l = m; }
        else                                      { h = m; }
    }

    /* return the position                                                 */
    return h;
}

libGAP_UInt            libGAP_PositionSortedDensePlistComp (
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 func )
{
    libGAP_UInt                l;              /* low                             */
    libGAP_UInt                h;              /* high                            */
    libGAP_UInt                m;              /* mid                             */
    libGAP_Obj                 v;              /* one element of the list         */

    /* perform the binary search to find the position                      */
    l = 0;  h = libGAP_LEN_PLIST( list ) + 1;
    while ( l+1 < h ) {                 /* list[l] < obj && obj <= list[h] */
        m = (l + h) / 2;                /* l < m < h                       */
        v = libGAP_ELM_PLIST( list, m );
        if ( libGAP_CALL_2ARGS( func, v, obj ) == libGAP_True ) { l = m; }
        else                                      { h = m; }
    }

    /* return the position                                                 */
    return h;
}

libGAP_Obj             libGAP_FuncPOSITION_SORTED_COMP (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 func )
{
    libGAP_UInt                h;              /* position, result                */

    /* check the first argument                                            */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "POSITION_SORTED_LISTComp: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }

    /* check the third argument                                            */
    while ( libGAP_TNUM_OBJ( func ) != libGAP_T_FUNCTION ) {
        func = libGAP_ErrorReturnObj(
            "POSITION_SORTED_LISTComp: <func> must be a function (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(func), 0L,
            "you can replace <func> via 'return <func>;'" );
    }

    /* dispatch                                                            */
    if ( libGAP_IS_DENSE_PLIST(list) ) {
        h = libGAP_PositionSortedDensePlistComp( list, obj, func );
    }
    else {
        h = libGAP_POSITION_SORTED_LISTComp( list, obj, func );
    }

    /* return the result                                                   */
    return libGAP_INTOBJ_INT( h );
}


libGAP_Obj             libGAP_FuncPOSITION_FIRST_COMPONENT_SORTED (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj)
{
  libGAP_UInt top, bottom,middle;
  libGAP_Obj x;
  libGAP_Obj l;
  bottom = 1;
  top = libGAP_LEN_PLIST(list);
  while (bottom <= top) {
    middle = (top + bottom)/2;
    l = libGAP_ELM_PLIST(list,middle);
    if (!libGAP_IS_PLIST(l))
      return libGAP_TRY_NEXT_METHOD;
    x = libGAP_ELM_PLIST(l,1);
    if (libGAP_LT(x,obj)) {
      bottom = middle+1;
    } else if (libGAP_LT(obj,x)) {
      top = middle -1;
    } else {
      return libGAP_INTOBJ_INT(middle);
    }
  }
  return libGAP_INTOBJ_INT(bottom);
}
      



/****************************************************************************
**
*F  SORT_LIST( <list> )  . . . . . . . . . . . . . . . . . . . .  sort a list
*F  SortDensePlist( <list> ) . . . . . . . . . . . . . . . . . .  sort a list
**
**  'SORT_LIST' sorts the list <list> in increasing  order.
**
**  'Sort' uses Shell's diminishing increment sort, which extends bubblesort.
**  The bubble sort works by  running  through  the  list  again  and  again,
**  each time exchanging pairs of adjacent elements which are out  of  order.
**  Thus large elements ``bubble'' to the top, hence the name of the  method.
**  However elements need many moves to come close to their  final  position.
**  In shellsort the first passes do not compare element j with its  neighbor
**  but with the element j+h, where h is larger than one.  Thus elements that
**  are not at their final position make large moves towards the destination.
**  This increment h is diminished, until during the last  pass  it  is  one.
**  A good sequence of incremements is given by Knuth:  (3^k-1)/2,... 13,4,1.
**  For this sequence shellsort uses on average  approximatly  N^1.25  moves.
**
**  Shellsort is the method of choice to  sort  lists  for  various  reasons:
**  Shellsort is quite easy to get right, much easier than,  say,  quicksort.
**  It runs as fast as quicksort for lists with  less  than  ~5000  elements.
**  It handles both  almost  sorted  and  reverse  sorted  lists  very  good.
**  It works well  in  the  presence  of  duplicate  elements  in  the  list.
**  Says Sedgewick: ``In short, if you have a sorting problem,  use the above
**  program, then determine whether the extra effort required to  replace  it
**  with a sophisticated method will be worthwile.''
**
**  Donald Knuth, The Art of Computer Programming, Vol.3, AddWes 1973, 84-95
**  Donald Shell, CACM 2, July 1959, 30-32
**  Robert Sedgewick, Algorithms 2nd ed., AddWes 1988, 107-123
*/

static void libGAP_BubbleDown(libGAP_Obj list, libGAP_UInt pos, libGAP_UInt len)
{
  libGAP_UInt lcpos, rcpos;
  libGAP_Obj lco, rco,x;

  lcpos = 2*pos;
  rcpos = 2*pos+1;
  if (lcpos > len)
    return;
  lco = libGAP_ELM_PLIST(list, lcpos);
  x = libGAP_ELM_PLIST(list,pos);
  if (rcpos > len)
    {
      if (libGAP_LT(x,lco))
        {
          libGAP_SET_ELM_PLIST(list, pos, lco);
          libGAP_SET_ELM_PLIST(list, lcpos, x);
        }
      return;
    }
  rco = libGAP_ELM_PLIST(list, rcpos);
  if (libGAP_LT(lco, rco)){
    if (libGAP_LT(x,rco)) {
      libGAP_SET_ELM_PLIST(list, pos, rco);
      libGAP_SET_ELM_PLIST(list, rcpos, x);
      libGAP_BubbleDown(list, rcpos, len);
    }
  }  else {
    if (libGAP_LT(x,lco)) {
      libGAP_SET_ELM_PLIST(list, pos, lco);
      libGAP_SET_ELM_PLIST(list, lcpos, x);
      libGAP_BubbleDown(list, lcpos, len);
    }
  }
  return;
}

libGAP_Obj libGAP_HEAP_SORT_PLIST ( libGAP_Obj self, libGAP_Obj list )
{
  libGAP_UInt len = libGAP_LEN_LIST(list);
  libGAP_UInt i;
  for (i = (len/2); i > 0 ; i--)
    libGAP_BubbleDown(list, i, len);
  for (i = len; i > 0; i--)
    {
      libGAP_Obj x = libGAP_ELM_PLIST(list, i);
      libGAP_Obj y = libGAP_ELM_PLIST(list, 1);
      libGAP_SET_ELM_PLIST(list, i, y);
      libGAP_SET_ELM_PLIST(list, 1, x);
      libGAP_BubbleDown(list, 1, i-1);
    }
  return (libGAP_Obj) 0;
}
  

void libGAP_SORT_LIST (
    libGAP_Obj                 list )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v, w;           /* two element of the list         */
    libGAP_UInt                i, k;           /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_LIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v = libGAP_ELMV_LIST( list, i );
            k = i;
            w = libGAP_ELMV_LIST( list, k-h );
            while ( h < k && libGAP_LT( v, w ) ) {
                libGAP_ASS_LIST( list, k, w );
                k -= h;
                if ( h < k )  w = libGAP_ELMV_LIST( list, k-h );
            }
            libGAP_ASS_LIST( list, k, v );
        }
        h = h / 3;
    }
    if (libGAP_IS_PLIST(list))
      libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
}

void libGAP_SortDensePlist (
    libGAP_Obj                 list )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v, w;           /* two element of the list         */
    libGAP_UInt                i, k;           /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_PLIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v = libGAP_ELM_PLIST( list, i );
            k = i;
            w = libGAP_ELM_PLIST( list, k-h );
            while ( h < k && libGAP_LT( v, w ) ) {
                libGAP_SET_ELM_PLIST( list, k, w );
                k -= h;
                if ( h < k )  w = libGAP_ELM_PLIST( list, k-h );
            }
            libGAP_SET_ELM_PLIST( list, k, v );
        }
        h = h / 3;
    }
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
}


/****************************************************************************
**
*F  SORT_LISTComp(<list>,<func>)  . . . . . . . . . . . . . . . . sort a list
*F  SortDensePlistComp(<list>,<func>) . . . . . . . . . . . . . . sort a list
**
**  'SORT_LISTComp' sorts the list <list> in increasing order, with respect to
**  comparison function <func>.
*/
void libGAP_SORT_LISTComp (
    libGAP_Obj                 list,
    libGAP_Obj                 func )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v, w;           /* two element of the list         */
    libGAP_UInt                i, k;           /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_LIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v = libGAP_ELMV_LIST( list, i );
            k = i;
            w = libGAP_ELMV_LIST( list, k-h );
            while ( h < k && libGAP_CALL_2ARGS( func, v, w ) == libGAP_True ) {
                libGAP_ASS_LIST( list, k, w );
                k -= h;
                if ( h < k )  w = libGAP_ELMV_LIST( list, k-h );
            }
            libGAP_ASS_LIST( list, k, v );
        }
        h = h / 3;
    }
    /* list is not necc. sorted wrt. \< (any longer) */
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_SSORT);
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
}






void libGAP_SortDensePlistComp (
    libGAP_Obj                 list,
    libGAP_Obj                 func )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v, w;           /* two element of the list         */
    libGAP_UInt                i, k;           /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_PLIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v = libGAP_ELM_PLIST( list, i );
            k = i;
            w = libGAP_ELM_PLIST( list, k-h );
            while ( h < k && libGAP_CALL_2ARGS( func, v, w ) == libGAP_True ) {
                libGAP_SET_ELM_PLIST( list, k, w );
                k -= h;
                if ( h < k )  w = libGAP_ELM_PLIST( list, k-h );
            }
            libGAP_SET_ELM_PLIST( list, k, v );
        }
        h = h / 3;
    }
    /* list is not necc. sorted wrt. \< (any longer) */
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_SSORT);
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
}

/****************************************************************************
**
*F  SORT_PARA_LIST( <list> )  . . . . . . . . . . .  sort a lists with shadow
*F  SortParaDensePlistPara( <list> )  . . . . . . .  sort a lists with shadow
*F  SORT_PARA_LISTComp(<list>,<func>) . . . . . . .  sort a lists with shadow
*F  SortParaDensePlistComp(<list>,<func>) . . . . .  sort a lists with shadow
**
**  The following suite of functions mirrors the sort functions above.  They
**  sort the first list given and perform the same operations on the second
**  list, the shadow list.  All functions assume that shadow list has (at
**  least) the length of the first list. 
**
**  The code here is a duplication of the code above with the operations on
**  the second list added in.
*/

void libGAP_SORT_PARA_LIST (
    libGAP_Obj                 list,
    libGAP_Obj               shadow )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v,  w;          /* two element of the list         */
    libGAP_Obj                 vs, ws;         /* two element of the shadow list  */
    libGAP_UInt                i,  k;          /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_LIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v  = libGAP_ELMV_LIST( list,   i ); 
            vs = libGAP_ELMV_LIST( shadow, i );
            k  = i;
            w  = libGAP_ELMV_LIST( list,   k-h );
            ws = libGAP_ELMV_LIST( shadow, k-h );
            while ( h < k && libGAP_LT( v, w ) ) {
              libGAP_ASS_LIST( list,   k, w  );
              libGAP_ASS_LIST( shadow, k, ws );
                k -= h;
                if ( h < k ) {
                    w  = libGAP_ELMV_LIST( list,   k-h );
                    ws = libGAP_ELMV_LIST( shadow, k-h );
                }
            }
            libGAP_ASS_LIST( list,   k, v  ); 
            libGAP_ASS_LIST( shadow, k, vs );
        }
        h = h / 3;
    }
    /* if list was ssorted, then it still will be,
       but, we don't know anything else any more */
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_SSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_NSORT);
}

void libGAP_SortParaDensePlist (
    libGAP_Obj                 list,
    libGAP_Obj               shadow )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v,  w;          /* two element of the list         */
    libGAP_Obj                 vs, ws;         /* two element of the shadow list  */
    libGAP_UInt                i,  k;          /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_PLIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v  = libGAP_ELM_PLIST( list,   i );
            vs = libGAP_ELM_PLIST( shadow, i );
            k  = i;
            w  = libGAP_ELM_PLIST( list,   k-h );
            ws = libGAP_ELM_PLIST( shadow, k-h );
            while ( h < k && libGAP_LT( v, w ) ) {
                libGAP_SET_ELM_PLIST( list,   k, w  );
                libGAP_SET_ELM_PLIST( shadow, k, ws );
                k -= h;
                if ( h < k ) {
                    w  = libGAP_ELM_PLIST( list,   k-h );
                    ws = libGAP_ELM_PLIST( shadow, k-h );
                }
            }
            libGAP_SET_ELM_PLIST( list,   k, v  );
            libGAP_SET_ELM_PLIST( shadow, k, vs );
        }
        h = h / 3;
    }

    /* if list was ssorted, then it still will be,
       but, we don't know anything else any more */
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_SSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_NSORT);
}

void libGAP_SORT_PARA_LISTComp (
    libGAP_Obj                 list,
    libGAP_Obj               shadow,
    libGAP_Obj                 func )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v,  w;          /* two element of the list         */
    libGAP_Obj                 vs, ws;         /* two element of the shadow list  */
    libGAP_UInt                i,  k;          /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_LIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v  = libGAP_ELMV_LIST( list,   i );    
            vs = libGAP_ELMV_LIST( shadow, i );
            k  = i;
            w  = libGAP_ELMV_LIST( list,   k-h );
            ws = libGAP_ELMV_LIST( shadow, k-h );
            while ( h < k && libGAP_CALL_2ARGS( func, v, w ) == libGAP_True ) {
                libGAP_ASS_LIST( list,   k, w );
                libGAP_ASS_LIST( shadow, k, ws );
                k -= h;
                if ( h < k ) {
                    w  = libGAP_ELMV_LIST( list,   k-h );
                    ws = libGAP_ELMV_LIST( shadow, k-h );
                }
            }
            libGAP_ASS_LIST( list,   k, v  );
            libGAP_ASS_LIST( shadow, k, vs );
        }
        h = h / 3;
    }
    /* list is not necc. sorted wrt. \< (any longer) */
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_SSORT);
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_NSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_SSORT);
}

void libGAP_SortParaDensePlistComp (
    libGAP_Obj                 list,
    libGAP_Obj               shadow,
    libGAP_Obj                 func )
{
    libGAP_UInt                len;            /* length of the list              */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_Obj                 v,  w;          /* two element of the list         */
    libGAP_Obj                 vs, ws;         /* two element of the shadow list  */
    libGAP_UInt                i,  k;          /* loop variables                  */

    /* sort the list with a shellsort                                      */
    len = libGAP_LEN_PLIST( list );
    h = 1;
    while ( 9*h + 4 < len ) { h = 3*h + 1; }
    while ( 0 < h ) {
        for ( i = h+1; i <= len; i++ ) {
            v  = libGAP_ELM_PLIST( list,   i );
            vs = libGAP_ELM_PLIST( shadow, i );
            k  = i;
            w  = libGAP_ELM_PLIST( list,   k-h );
            ws = libGAP_ELM_PLIST( shadow, k-h );
            while ( h < k && libGAP_CALL_2ARGS( func, v, w ) == libGAP_True ) {
                libGAP_SET_ELM_PLIST( list,   k, w  );
                libGAP_SET_ELM_PLIST( shadow, k, ws );
                k -= h;
                if ( h < k ) {
                    w  = libGAP_ELM_PLIST( list,   k-h );  
                    ws = libGAP_ELM_PLIST( shadow, k-h );
                }
            }
            libGAP_SET_ELM_PLIST( list,   k, v  );
            libGAP_SET_ELM_PLIST( shadow, k, vs );
        }
        h = h / 3;
    }
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_NSORT);
    libGAP_RESET_FILT_LIST(list, libGAP_FN_IS_SSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_NSORT);
    libGAP_RESET_FILT_LIST(shadow, libGAP_FN_IS_SSORT);
}



/****************************************************************************
**
*F  RemoveDupsDensePlist(<list>)  . . . . remove duplicates from a plain list
**
**  'RemoveDupsDensePlist' removes duplicate elements from the dense
**  plain list <list>.  <list> must be sorted.  'RemoveDupsDensePlist'
**  returns 0 if <list> contains mutable elements, 1 if not and 2 if
**  the list contains immutable elements all lying in the same family.
*/
libGAP_UInt            libGAP_RemoveDupsDensePlist (
    libGAP_Obj                 list )
{
    libGAP_UInt                mutable;        /* the elements are mutable        */
    libGAP_UInt                homog;          /* the elements all lie in the same family */
    libGAP_Int                 len;            /* length of the list              */
    libGAP_Obj                 v, w;           /* two elements of the list        */
    libGAP_UInt                l, i;           /* loop variables                  */
    libGAP_Obj                 fam;

    /* get the length, nothing to be done for empty lists                  */
    len = libGAP_LEN_PLIST( list );
    if ( len == 0 ) { return 0; }

    /* select the first element as the first representative                */
    l = 1;
    v = libGAP_ELM_PLIST( list, l );
    mutable = libGAP_IS_MUTABLE_OBJ(v);
    homog = 1;
    fam = libGAP_FAMILY_OBJ(v);

    /* loop over the other elements, compare them with the current rep.    */
    for ( i = 2; i <= len; i++ ) {
        w = libGAP_ELM_PLIST( list, i );
        mutable = (mutable || libGAP_IS_MUTABLE_OBJ(w));
        if ( ! libGAP_EQ( v, w ) ) {
            if ( l+1 != i ) {
                libGAP_SET_ELM_PLIST( list, l+1, w );
                libGAP_SET_ELM_PLIST( list, i, (libGAP_Obj)0 );
            }
            l += 1;
            v = w;
            homog = (!mutable && homog && fam == libGAP_FAMILY_OBJ(w));
        }
    }

    /* the list may be shorter now                                         */
    libGAP_SET_LEN_PLIST( list, l );
    libGAP_SHRINK_PLIST(  list, l );

    /* Set appropriate filters */
    if (!mutable)
      {
        if (!homog)
          libGAP_SET_FILT_LIST(list, libGAP_FN_IS_NHOMOG);
        else
          libGAP_SET_FILT_LIST(list, libGAP_FN_IS_HOMOG);
        libGAP_SET_FILT_LIST(list, libGAP_FN_IS_SSORT);
      }

    /* return whether the list contains mutable elements                   */
    if (mutable)
      return 0;
    if (!homog)
      return 1;
    else
      return 2;
}


/****************************************************************************
**
*F * * * * * * * * * * * * * * GAP level functions  * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  FuncSORT_LIST( <self>, <list> ) . . . . . . . . . . . . . . . sort a list
*/
libGAP_Obj libGAP_FuncSORT_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    /* check the first argument                                            */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "SORT_LIST: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }

    /* dispatch                                                            */
    if ( libGAP_IS_DENSE_PLIST(list) ) {
        libGAP_SortDensePlist( list );
    }
    else {
        libGAP_SORT_LIST( list );
    }
    libGAP_IS_SSORT_LIST(list);

    /* return nothing                                                      */
    return (libGAP_Obj)0;
}


/****************************************************************************
**


*F  FuncSORT_LIST_COMP( <self>, <list>, <func> )  . . . . . . . . sort a list
*/
libGAP_Obj libGAP_FuncSORT_LIST_COMP (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 func )
{
    /* check the first argument                                            */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "SORT_LISTComp: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }

    /* check the third argument                                            */
    while ( libGAP_TNUM_OBJ( func ) != libGAP_T_FUNCTION ) {
        func = libGAP_ErrorReturnObj(
            "SORT_LISTComp: <func> must be a function (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(func), 0L,
            "you can replace <func> via 'return <func>;'" );
    }

    /* dispatch                                                            */
    if ( libGAP_IS_DENSE_PLIST(list) ) {
        libGAP_SortDensePlistComp( list, func );
    }
    else {
        libGAP_SORT_LISTComp( list, func );
    }

    /* return nothing                                                      */
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncSORT_PARA_LIST( <self>, <list> )  . . . . . . sort a list with shadow
*/
libGAP_Obj libGAP_FuncSORT_PARA_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj               shadow )
{
    /* check the first two arguments                                       */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "SORT_PARA_LIST: first <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(shadow) ) {
        shadow = libGAP_ErrorReturnObj(
            "SORT_PARA_LIST: second <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(shadow), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    if( libGAP_LEN_LIST( list ) != libGAP_LEN_LIST( shadow ) ) {
        libGAP_ErrorReturnVoid( 
            "SORT_PARA_LIST: lists must have the same length (not %d and %d)",
            (libGAP_Int)libGAP_LEN_LIST( list ),
            (libGAP_Int)libGAP_LEN_LIST( shadow ),
            "you can 'return;'" );
    }

    /* dispatch                                                            */
    if ( libGAP_IS_DENSE_PLIST(list) && libGAP_IS_DENSE_PLIST(shadow) ) {
        libGAP_SortParaDensePlist( list, shadow );
    }
    else {
        libGAP_SORT_PARA_LIST( list, shadow );
    }
    libGAP_IS_SSORT_LIST(list);

    /* return nothing                                                      */
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncSORT_LIST_COMP( <self>, <list>, <func> )  . . . . . . . . sort a list
*/
libGAP_Obj libGAP_FuncSORT_PARA_LIST_COMP (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj               shadow,
    libGAP_Obj                 func )
{
    /* check the first two arguments                                       */
    while ( ! libGAP_IS_SMALL_LIST(list) ) {
        list = libGAP_ErrorReturnObj(
            "SORT_LISTComp: <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    while ( ! libGAP_IS_SMALL_LIST(shadow) ) {
        shadow = libGAP_ErrorReturnObj(
            "SORT_PARA_LIST: second <list> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(shadow), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    if( libGAP_LEN_LIST( list ) != libGAP_LEN_LIST( shadow ) ) {
        libGAP_ErrorReturnVoid( 
            "SORT_PARA_LIST: lists must have the same length (not %d and %d)",
            (libGAP_Int)libGAP_LEN_LIST( list ),
            (libGAP_Int)libGAP_LEN_LIST( shadow ),
            "you can 'return;'" );
    }

    /* check the third argument                                            */
    while ( libGAP_TNUM_OBJ( func ) != libGAP_T_FUNCTION ) {
        func = libGAP_ErrorReturnObj(
            "SORT_LISTComp: <func> must be a function (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(func), 0L,
            "you can replace <func> via 'return <func>;'" );
    }

    /* dispatch                                                            */
    if ( libGAP_IS_DENSE_PLIST(list) && libGAP_IS_DENSE_PLIST(shadow) ) {
        libGAP_SortParaDensePlistComp( list, shadow, func );
    }
    else {
        libGAP_SORT_PARA_LISTComp( list, shadow, func );
    }

    /* return nothing                                                      */
    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncOnPoints( <self>, <point>, <elm> )  . . . . . . . operation on points
**
**  'FuncOnPoints' implements the internal function 'OnPoints'.
**
**  'OnPoints( <point>, <elm> )'
**
**  specifies  the  canonical  default operation.   Passing  this function is
**  equivalent  to  specifying no operation.   This function  exists  because
**  there are places where the operation in not an option.
*/
libGAP_Obj             libGAP_FuncOnPoints (
    libGAP_Obj                 self,
    libGAP_Obj                 point,
    libGAP_Obj                 elm )
{
    return libGAP_POW( point, elm );
}


/****************************************************************************
**
*F  FuncOnPairs( <self>, <pair>, <elm> )  . . .  operation on pairs of points
**
**  'FuncOnPairs' implements the internal function 'OnPairs'.
**
**  'OnPairs( <pair>, <elm> )'
**
**  specifies  the componentwise operation    of group elements on  pairs  of
**  points, which are represented by lists of length 2.
*/
libGAP_Obj             libGAP_FuncOnPairs (
    libGAP_Obj                 self,
    libGAP_Obj                 pair,
    libGAP_Obj                 elm )
{
    libGAP_Obj                 img;            /* image, result                   */
    libGAP_Obj                 tmp;            /* temporary                       */

    /* check the type of the first argument                                */
    while ( ! libGAP_IS_SMALL_LIST( pair ) || libGAP_LEN_LIST( pair ) != 2 ) {
        pair = libGAP_ErrorReturnObj(
            "OnPairs: <pair> must be a list of length 2 (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(pair), 0L,
            "you can replace <pair> via 'return <pair>;'" );
    }

    /* create a new bag for the result                                     */
    img = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(pair) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE, 2 );
    libGAP_SET_LEN_PLIST( img, 2 );

    /* and enter the images of the points into the result bag              */
    tmp = libGAP_POW( libGAP_ELMV_LIST( pair, 1 ), elm );
    libGAP_SET_ELM_PLIST( img, 1, tmp );
    libGAP_CHANGED_BAG( img );
    tmp = libGAP_POW( libGAP_ELMV_LIST( pair, 2 ), elm );
    libGAP_SET_ELM_PLIST( img, 2, tmp );
    libGAP_CHANGED_BAG( img );

    /* return the result                                                   */
    return img;
}


/****************************************************************************
**
*F  FuncOnTuples( <self>, <tuple>, <elm> )  . . operation on tuples of points
**
**  'FuncOnTuples' implements the internal function 'OnTuples'.
**
**  'OnTuples( <tuple>, <elm> )'
**
**  specifies the componentwise  operation  of  group elements  on tuples  of
**  points, which are represented by lists.  'OnPairs' is the special case of
**  'OnTuples' for tuples with two elements.
*/
libGAP_Obj             libGAP_FuncOnTuples (
    libGAP_Obj                 self,
    libGAP_Obj                 tuple,
    libGAP_Obj                 elm )
{
    libGAP_Obj                 img;            /* image, result                   */
    libGAP_Obj                 tmp;            /* temporary                       */
    libGAP_UInt                i;              /* loop variable                   */

    /* check the type of the first argument                                */
    while ( ! libGAP_IS_SMALL_LIST( tuple ) ) {
        tuple = libGAP_ErrorReturnObj(
            "OnTuples: <tuple> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(tuple), 0L,
            "you can replace <tuple> via 'return <tuple>;'" );
    }

    /* special case for the empty list */
    if ( libGAP_HAS_FILT_LIST( tuple, libGAP_FN_IS_EMPTY )) {
      if (libGAP_IS_MUTABLE_OBJ(tuple)) {
        img = libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
        libGAP_SET_LEN_PLIST(img,0);
        return img;
      } else {
        return tuple;
      }
    }
    /* special case for permutations                                       */
    if ( libGAP_TNUM_OBJ(elm) == libGAP_T_PERM2 || libGAP_TNUM_OBJ(elm) == libGAP_T_PERM4 ) {
        libGAP_PLAIN_LIST( tuple );
        return libGAP_OnTuplesPerm( tuple, elm );
    }

    /* special case for transformations                                       */
    if ( libGAP_TNUM_OBJ(elm) == libGAP_T_TRANS2 || libGAP_TNUM_OBJ(elm) == libGAP_T_TRANS4 ) {
        libGAP_PLAIN_LIST( tuple );
        return libGAP_OnTuplesTrans( tuple, elm );
    }

    /* special case for partial perms */
    if ( libGAP_TNUM_OBJ(elm) == libGAP_T_PPERM2 || libGAP_TNUM_OBJ(elm) == libGAP_T_PPERM4 ) {
        libGAP_PLAIN_LIST( tuple );
        return libGAP_OnTuplesPPerm( tuple, elm );
    }

    /* create a new bag for the result                                     */
    img = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_OBJ(tuple) ? libGAP_T_PLIST : libGAP_T_PLIST+libGAP_IMMUTABLE, libGAP_LEN_LIST(tuple) );
    libGAP_SET_LEN_PLIST( img, libGAP_LEN_LIST(tuple) );

    /* and enter the images of the points into the result bag              */
    for ( i = libGAP_LEN_LIST(tuple); 1 <= i; i-- ) {
        tmp = libGAP_POW( libGAP_ELMV_LIST( tuple, i ), elm );
        libGAP_SET_ELM_PLIST( img, i, tmp );
        libGAP_CHANGED_BAG( img );
    }

    /* return the result (must be a dense plain list, see 'FuncOnSets')    */
    return img;
}


/****************************************************************************
**
*F  FuncOnSets( <self>, <tuple>, <elm> )  . . . . operation on sets of points
**
**  'FuncOnSets' implements the internal function 'OnSets'.
**
**  'OnSets( <tuple>, <elm> )'
**
**  specifies the operation  of group elements  on  sets of points, which are
**  represented by sorted lists of points without duplicates (see "Sets").
*/

libGAP_Obj             libGAP_FuncOnSets (
    libGAP_Obj                 self,
    libGAP_Obj                 set,
    libGAP_Obj                 elm )
{
    libGAP_Obj                 img;            /* handle of the image, result     */
    libGAP_UInt                status;        /* the elements are mutable        */

    /* check the type of the first argument                                */
    while ( !libGAP_HAS_FILT_LIST(set, libGAP_FN_IS_SSORT) && ! libGAP_IsSet( set ) ) {
        set = libGAP_ErrorReturnObj(
            "OnSets: <set> must be a set (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(set), 0L,
            "you can replace <set> via 'return <set>;'" );
    }

    /* special case for the empty list */
    if ( libGAP_HAS_FILT_LIST( set, libGAP_FN_IS_EMPTY )) {
      if (libGAP_IS_MUTABLE_OBJ(set)) {
        img = libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
        libGAP_SET_LEN_PLIST(img,0);
        return img;
      } else {
        return set;
      }
    }
        
    /* special case for permutations                                       */
    if ( libGAP_TNUM_OBJ(elm) == libGAP_T_PERM2 || libGAP_TNUM_OBJ(elm) == libGAP_T_PERM4 ) {
        libGAP_PLAIN_LIST( set );
        return libGAP_OnSetsPerm( set, elm );
    }

    /* special case for transformations */
    if ( libGAP_TNUM_OBJ(elm)== libGAP_T_TRANS2 || libGAP_TNUM_OBJ(elm) == libGAP_T_TRANS4 ){
      libGAP_PLAIN_LIST(set);
      return libGAP_OnSetsTrans( set, elm);
    }
    
    /* special case for partial perms */
    if ( libGAP_TNUM_OBJ(elm)== libGAP_T_PPERM2 || libGAP_TNUM_OBJ(elm) == libGAP_T_PPERM4 ){
      libGAP_PLAIN_LIST(set);
      return libGAP_OnSetsPPerm( set, elm);
    }

    /* compute the list of images                                          */
    img = libGAP_FuncOnTuples( self, set, elm );

    /* sort the images list (which is a dense plain list)                  */
    libGAP_SortDensePlist( img );

    /* remove duplicates, check for mutable elements                       */
    status = libGAP_RemoveDupsDensePlist( img );

    /* if possible, turn this into a set                                   */
    switch (status)
      {
      case 0:
        break;
        
      case 1:
        libGAP_RetypeBag( img, libGAP_T_PLIST_DENSE_NHOM_SSORT );

      case 2:
        libGAP_RetypeBag( img, libGAP_T_PLIST_HOM_SSORT );

      }


    /* return set                                                          */
    return img;
}


/****************************************************************************
**
*F  FuncOnRight( <self>, <point>, <elm> ) . operation by mult. from the right
**
**  'FuncOnRight' implements the internal function 'OnRight'.
**
**  'OnRight( <point>, <elm> )'
**
**  specifies that group elements operate by multiplication from the right.
*/
libGAP_Obj             libGAP_FuncOnRight (
    libGAP_Obj                 self,
    libGAP_Obj                 point,
    libGAP_Obj                 elm )
{
    return libGAP_PROD( point, elm );
}


/****************************************************************************
**
*F  FuncOnLeftAntiOperation( <self>, <point>, <elm> ) op. by mult. from the left
**
**  'FuncOnLeftAntiOperation' implements the internal function
**  'OnLeftAntiOperation'.
**
**  'OnLeftAntiOperation( <point>, <elm> )'
**
**  specifies that group elements operate by multiplication from the left.
*/
libGAP_Obj             libGAP_FuncOnLeftAntiOperation (
    libGAP_Obj                 self,
    libGAP_Obj                 point,
    libGAP_Obj                 elm )
{
    return libGAP_PROD( elm, point );
}


/****************************************************************************
**
*F  FuncOnLeftInverse( <self>, <point>, <elm> ) . . op by mult. from the left
**
**  'FuncOnLeftInverse' implements the internal function 'OnLeftInverse'.
**
**  'OnLeftInverse( <point>, <elm> )'
**
**  specifies that group elements operate by multiplication from the left
**  with the inverse.
*/
libGAP_Obj             libGAP_FuncOnLeftInverse (
    libGAP_Obj                 self,
    libGAP_Obj                 point,
    libGAP_Obj                 elm )
{
    elm = libGAP_INV(elm);
    return libGAP_PROD( elm, point );
}

/****************************************************************************
**
*F  FuncSTRONGLY_CONNECTED_COMPONENTS_DIGRAPH
**
**  `digraph' should be a list whose entries and the lists of out-neighbours
** of the vertices. So [[2,3],[1],[2]] represents the graph whose edges are
** 1->2, 1->3, 2->1 and 3->2.
**
**  returns a newly constructed list whose elements are lists representing the
** strongly connected components of the directed graph. Neither the components,
** nor their elements are in any particular order.
**
** The algorithm is that of Tarjan, based on the implementation in Sedgwick,
** with a bug fixed, and made non-recursive to avoid problems with stack limits
** under (for instance) Linux. This version is a bit slower than the recursive
** version, but much faster than any of the GAP implementations.
**
** A possible change is to allocate the internal arrays rather smaller, and
** grow them if needed. This might allow some computations to complete that would
** otherwise run out of memory, but would slow things down a bit.
*/


static libGAP_Obj libGAP_FuncSTRONGLY_CONNECTED_COMPONENTS_DIGRAPH(libGAP_Obj self, libGAP_Obj digraph)
{
  libGAP_UInt i,level,k,l,x,t,m;
  libGAP_UInt now = 0,n;
  libGAP_Obj val, stack, comps,comp;
  libGAP_Obj frames, adj;
  libGAP_UInt *fptr;

  n = libGAP_LEN_LIST(digraph);
  if (n == 0)
    {
      return libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY,0);
    }
  val = libGAP_NewBag(libGAP_T_DATOBJ, (n+1)*sizeof(libGAP_UInt));
  stack = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, n);
  libGAP_SET_LEN_PLIST(stack, 0);
  comps = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB, n);
  libGAP_SET_LEN_PLIST(comps, 0);
  frames = libGAP_NewBag(libGAP_T_DATOBJ, (4*n+1)*sizeof(libGAP_UInt));  
  for (k = 1; k <= n; k++)
    {
      if (((libGAP_UInt *)libGAP_ADDR_OBJ(val))[k] == 0)
        {
          level = 1;
          adj = libGAP_ELM_LIST(digraph, k);
          libGAP_PLAIN_LIST(adj);
          fptr = (libGAP_UInt *)libGAP_ADDR_OBJ(frames);
          fptr[0] = k;
          now++;
          ((libGAP_UInt *)libGAP_ADDR_OBJ(val))[k] = now;
          fptr[1] = now;
          l = libGAP_LEN_PLIST(stack);
          libGAP_SET_ELM_PLIST(stack, l+1, libGAP_INTOBJ_INT(k));
          libGAP_SET_LEN_PLIST(stack, l+1);
          fptr[2] = 1;
          fptr[3] = (libGAP_UInt)adj;
          while (level > 0 ) {
            if (fptr[2] > libGAP_LEN_PLIST(fptr[3]))
              {
                if (fptr[1] == ((libGAP_UInt *)libGAP_ADDR_OBJ(val))[fptr[0]])
                  {
                    l = libGAP_LEN_PLIST(stack);
                    i = l;
                    do {
                      x = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(stack, i));
                      libGAP_SET_ELM_PLIST(val, x, libGAP_INTOBJ_INT(n+1));
                      i--;
                    } while (x != fptr[0]);
                    comp = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, l-i);
                    libGAP_SET_LEN_PLIST(comp, l-i);
                    memcpy( (void *)((char *)(libGAP_ADDR_OBJ(comp)) + sizeof(libGAP_Obj)), 
                            (void *)((char *)(libGAP_ADDR_OBJ(stack)) + (i+1)*sizeof(libGAP_Obj)), 
                            (size_t)((l - i )*sizeof(libGAP_Obj)));
                    libGAP_SET_LEN_PLIST(stack, i);
                    l = libGAP_LEN_PLIST(comps);
                    libGAP_SET_ELM_PLIST(comps, l+1, comp);
                    libGAP_SET_LEN_PLIST(comps, l+1);
                    libGAP_CHANGED_BAG(comps);
                    fptr = (libGAP_UInt *)libGAP_ADDR_OBJ(frames)+(level-1)*4;
                  }
                level--;
                fptr -= 4;
                if (level > 0 && fptr[5]  < fptr[1])
                  fptr[1] = fptr[5];
              }
            else
              {
                adj = (libGAP_Obj)fptr[3];
                t = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(adj, (fptr[2])++));
                if (0 ==(m =  ((libGAP_UInt *)libGAP_ADDR_OBJ(val))[t]))
                  {
                    level++;
                    adj = libGAP_ELM_LIST(digraph, t);
                    libGAP_PLAIN_LIST(adj);
                    fptr = (libGAP_UInt *)libGAP_ADDR_OBJ(frames)+(level-1)*4;
                    fptr[0] = t;
                    now++;
                    ((libGAP_UInt *)libGAP_ADDR_OBJ(val))[t] = now;
                    fptr[1] = now;
                    l = libGAP_LEN_PLIST(stack);
                    libGAP_SET_ELM_PLIST(stack, l+1, libGAP_INTOBJ_INT(t));
                    libGAP_SET_LEN_PLIST(stack, l+1);
                    fptr[2] = 1;
                    fptr[3] = (libGAP_UInt)adj;
                  }
                else
                  {
                    if (m < fptr[1])
                      fptr[1] = m;
                  }
              }
          }
        }
      
    }
  libGAP_SHRINK_PLIST(comps, libGAP_LEN_PLIST(comps));
  return comps;
}


/****************************************************************************
**
*F  FuncCOPY_LIST_ENTRIES( <self>, <args> ) . . mass move of list entries
**
*/

static inline libGAP_Int libGAP_GetIntObj( libGAP_Obj list, libGAP_UInt pos)
{
  libGAP_Obj entry;
  entry = libGAP_ELM_PLIST(list, pos);
  if (!entry)
    {
      libGAP_Pr("panic: internal inconsistency", 0L, 0L);
      libGAP_SyExit(1);
    }
  while (!libGAP_IS_INTOBJ(entry))
    {
      entry = libGAP_ErrorReturnObj("COPY_LIST_ENTRIES: argument %d  must be a small integer, not a %s",
                             (libGAP_Int)pos, (libGAP_Int)libGAP_InfoBags[libGAP_TNUM_OBJ(entry)].name,
                             "you can return a small integer to continue");
    }
  return libGAP_INT_INTOBJ(entry);
}

libGAP_Obj libGAP_FuncCOPY_LIST_ENTRIES( libGAP_Obj self, libGAP_Obj args )
{  
  libGAP_Obj srclist;
  libGAP_UInt srcstart;
  libGAP_Int srcinc;
  libGAP_Obj dstlist;
  libGAP_UInt dststart;
  libGAP_Int dstinc;
  libGAP_UInt number;
  libGAP_UInt srcmax;
  libGAP_UInt dstmax;
  libGAP_Obj *sptr, *dptr;
  libGAP_UInt ct;

  if (!libGAP_IS_PLIST(args))
    {
      libGAP_Pr("panic: internal inconsistency",0L,0L);
      libGAP_SyExit(1);
    }
  if (libGAP_LEN_PLIST(args) != 7)
    {
      libGAP_ErrorMayQuit("COPY_LIST_ENTRIES: number of arguments must be 7, not %d",
                   (libGAP_Int)libGAP_LEN_PLIST(args), 0L);
    }
  srclist = libGAP_ELM_PLIST(args,1);
  if (!srclist)
    {
      libGAP_Pr("panic: internal inconsistency", 0L, 0L);
      libGAP_SyExit(1);
    }
  while (!libGAP_IS_PLIST(srclist))
    {
      srclist = libGAP_ErrorReturnObj("COPY_LIST_ENTRIES: source must be a plain list not a %s",
                               (libGAP_Int)libGAP_InfoBags[libGAP_TNUM_OBJ(srclist)].name, 0L,
                               "you can return a plain list to continue");
    }

  srcstart = (libGAP_UInt)libGAP_GetIntObj(args,2);
  srcinc = libGAP_GetIntObj(args,3);
  dstlist = libGAP_ELM_PLIST(args,4);
  if (!dstlist)
    {
      libGAP_Pr("panic: internal inconsistency", 0L, 0L);
      libGAP_SyExit(1);
    }
  while (!libGAP_IS_PLIST(dstlist) || !libGAP_IS_MUTABLE_OBJ(dstlist))
    {
      dstlist = libGAP_ErrorReturnObj("COPY_LIST_ENTRIES: destination must be a mutable plain list not a %s",
                               (libGAP_Int)libGAP_InfoBags[libGAP_TNUM_OBJ(dstlist)].name, 0L,
                               "you can return a plain list to continue");
    }
  dststart = (libGAP_UInt)libGAP_GetIntObj(args,5);
  dstinc = libGAP_GetIntObj(args,6);
  number = libGAP_GetIntObj(args,7);
  
  if (number == 0)
    return (libGAP_Obj) 0;
  
  srcmax = (srcinc > 0) ? srcstart + (number-1)*srcinc : srcstart;
  dstmax = (dstinc > 0) ? dststart + (number-1)*dstinc : dststart;
  
  libGAP_GROW_PLIST(dstlist, dstmax);
  libGAP_GROW_PLIST(srclist, srcmax);
  if (srcinc == 1 && dstinc == 1)
    {
      memmove((void *) (libGAP_ADDR_OBJ(dstlist) + dststart),
              (void *) (libGAP_ADDR_OBJ(srclist) + srcstart),
              (size_t) number*sizeof(libGAP_Obj));
    }
  else if (srclist != dstlist)
    {
      sptr = libGAP_ADDR_OBJ(srclist) + srcstart;
      dptr = libGAP_ADDR_OBJ(dstlist) + dststart;
      for (ct = 0; ct < number ; ct++)
        {
          *dptr = *sptr;
          sptr += srcinc;
          dptr += dstinc;
        }
    }
  else if (srcinc == dstinc)
    {
      if (srcstart == dststart)
        return (libGAP_Obj)0;
      else
        {
          if ((srcstart > dststart) == (srcinc > 0))
            {
              sptr = libGAP_ADDR_OBJ(srclist) + srcstart;
              dptr = libGAP_ADDR_OBJ(srclist) + dststart;
              for (ct = 0; ct < number ; ct++)
                {
                  *dptr = *sptr;
                  sptr += srcinc;
                  dptr += srcinc;
                }
            }
          else
            {
              sptr = libGAP_ADDR_OBJ(srclist) + srcstart + number*srcinc;
              dptr = libGAP_ADDR_OBJ(srclist) + dststart + number*srcinc;
              for (ct = 0; ct < number; ct++)
                {
                  sptr -= srcinc;
                  dptr -= srcinc;
                  *dptr = *sptr;
                }
              
            }
        }
              
    }
  else
    {
      libGAP_Obj tmplist = libGAP_NEW_PLIST(libGAP_T_PLIST,number);
      libGAP_Obj *tptr = libGAP_ADDR_OBJ(tmplist)+1;
      sptr = libGAP_ADDR_OBJ(srclist)+srcstart;
      for (ct = 0; ct < number; ct++)
        {
          *tptr = *sptr;
          tptr++;
          sptr += srcinc;
        }
      tptr = libGAP_ADDR_OBJ(tmplist)+1;
      dptr = libGAP_ADDR_OBJ(srclist)+dststart;
      for (ct = 0; ct < number; ct++)
        {
          *dptr = *tptr;
          tptr++;
          dptr += dstinc;
        }
    }

  if (dstmax > libGAP_LEN_PLIST(dstlist))
    {
      dptr = libGAP_ADDR_OBJ(dstlist)+dstmax;
      ct = dstmax;
      while (!*dptr)
        {
          ct--;
          dptr--;
        }
      libGAP_SET_LEN_PLIST(dstlist, ct);
    }
  if (libGAP_LEN_PLIST(dstlist) > 0)
    libGAP_RetypeBag(dstlist, libGAP_T_PLIST);
  else
    libGAP_RetypeBag(dstlist, libGAP_T_PLIST_EMPTY);
  return (libGAP_Obj) 0;

}

/****************************************************************************
**
*F  FuncBIMULT_MONOMIALS_ALGEBRA_ELEMENT
**
*/

libGAP_Obj libGAP_FuncBIMULT_MONOMIALS_ALGEBRA_ELEMENT(libGAP_Obj self, 
                                         libGAP_Obj mona, libGAP_Obj poly, libGAP_Obj monb)
{
  libGAP_UInt len,la,lb,lm,i,j;
  libGAP_Obj prd;
  libGAP_Obj mon;
  libGAP_Obj *po,*pn;

  /* TODO: test arguments */

  /* Make a new polynomial */
  len = libGAP_LEN_PLIST(poly);
  prd = libGAP_NEW_PLIST( libGAP_T_PLIST , len );
  libGAP_SET_LEN_PLIST( prd, len );

  if (mona == libGAP_False) {
    la=0;
  }
  else {
    la=libGAP_LEN_PLIST(mona);
  }

  if (monb == libGAP_False) {
    lb=0;
  }
  else {
    lb=libGAP_LEN_PLIST(monb);
  }

  /* fill its entries */
  for (i=1;i<=len;i=i+2) {
    lm = libGAP_LEN_PLIST(libGAP_ELM_PLIST(poly,i));
  /*Pr("i= %d, lm= %d \n",i,lm); */
    /* new monomial */
    mon = libGAP_NEW_PLIST(libGAP_T_PLIST,lm+la+lb);
    libGAP_SET_LEN_PLIST(mon,lm+la+lb);
    pn = libGAP_ADDR_OBJ(mon)+1;
    
    /* put in a */
    j=1;
    po = libGAP_ADDR_OBJ(mona)+1;
    while (j<=la) {
      *pn++=*po++;
      j++;
    }

    /* append monomial */
    j=1;
    po = libGAP_ADDR_OBJ(libGAP_ELM_PLIST(poly,i))+1;
    while (j<=lm) {
      *pn++=*po++;
      j++;
    }

    /* append b */
    j=1;
    po = libGAP_ADDR_OBJ(monb)+1;
    while (j<=lb) {
      *pn++=*po++;
      j++;
    }

    libGAP_SET_ELM_PLIST(prd,i,mon);
    /* copy coefficient */
    libGAP_SET_ELM_PLIST(prd,i+1,libGAP_ELM_PLIST(poly,i+1));
    libGAP_CHANGED_BAG(prd);
  }

  return prd;
}



/****************************************************************************
**
*F  FuncHORSPOOL_LISTS
**
*/

libGAP_Obj libGAP_FuncHORSPOOL_LISTS(libGAP_Obj self,libGAP_Obj wrep, libGAP_Obj subrep, libGAP_Obj pre)
{
  libGAP_UInt i,wsize,subsize,di;
  libGAP_Int j;
  libGAP_Obj pos;
  libGAP_Obj *pw,*ps,*pp;

  wsize = libGAP_LEN_PLIST(wrep);
  subsize = libGAP_LEN_PLIST(subrep);
  pw=libGAP_ADDR_OBJ(wrep);
  ps=libGAP_ADDR_OBJ(subrep);
  pp=libGAP_ADDR_OBJ(pre);
 
  pos = libGAP_Fail;

  if ( subsize <= wsize ) {
    i = 0;
    di = wsize-subsize;
    while (i <= di) {
      j=subsize;
      while (j>0) {
 /* Pr("i= %d j=%d \n",i,j);  */
        if (ps[j] != pw[i+j]) {
          j=-1;
        }
        else {
          j--;
        }
      }
      if (j==0) {
        pos=libGAP_INTOBJ_INT(i+1);
        i=wsize;
      }
      else {
 /* Pr("pw: %d \n",INT_INTOBJ(pw[i+subsize]),0L);  */
              i = i + libGAP_INT_INTOBJ(pp[libGAP_INT_INTOBJ(pw[i+subsize])]);
      }
    }
  }
  return pos;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarOpers . . . . . . . . . . . . . . . . .  list of operations to export
*/
static libGAP_StructGVarOper libGAP_GVarOpers [] = {

    { "ADD_LIST", 2, "list, val", &libGAP_AddListOper,
      libGAP_FuncADD_LIST, "src/listfunc.c:ADD_LIST" },

    { "REM_LIST", 1, "list", &libGAP_RemListOper,
      libGAP_FuncREM_LIST, "src/listfunc.c:REM_LIST" },

    { "APPEND_LIST", 2, "list, val", &libGAP_AppendListOper,
      libGAP_FuncAPPEND_LIST, "src/listfunc.c:APPEND_LIST" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "APPEND_LIST_INTR", 2, "list1, list2", 
      libGAP_FuncAPPEND_LIST_INTR, "src/listfunc.c:APPEND_LIST_INTR" },

    { "POSITION_SORTED_LIST", 2, "list, obj", 
      libGAP_FuncPOSITION_SORTED_LIST, "src/listfunc.c:POSITION_SORTED_LIST" },

    { "POSITION_SORTED_LIST_COMP", 3, "list, obj, func", 
      libGAP_FuncPOSITION_SORTED_COMP, "src/listfunc.c:POSITION_SORTED_LIST_COMP" },

    { "POSITION_FIRST_COMPONENT_SORTED", 2, "list, obj", 
      libGAP_FuncPOSITION_FIRST_COMPONENT_SORTED, "src/listfunc.c:POSITION_FIRST_COMPONENT_SORTED" },

    { "SORT_LIST", 1, "list",
      libGAP_FuncSORT_LIST, "src/listfunc.c:SORT_LIST" },

    { "SORT_LIST_COMP", 2, "list, func",
      libGAP_FuncSORT_LIST_COMP, "src/listfunc.c:SORT_LIST_COMP" },

    { "SORT_PARA_LIST", 2, "list, list",
      libGAP_FuncSORT_PARA_LIST, "src/listfunc.c:SORT_PARA_LIST" },

    { "SORT_PARA_LIST_COMP", 3, "list, list, func",
      libGAP_FuncSORT_PARA_LIST_COMP, "src/listfunc.c:SORT_PARA_LIST_COMP" },

    { "OnPoints", 2, "pnt, elm",
      libGAP_FuncOnPoints, "src/listfunc.c:OnPoints" },

    { "OnPairs", 2, "pair, elm",
      libGAP_FuncOnPairs, "src/listfunc.c:OnPairs" },

    { "OnTuples", 2, "tuple, elm",
      libGAP_FuncOnTuples, "src/listfunc.c:OnTuples" },

    { "OnSets", 2, "set, elm",
      libGAP_FuncOnSets, "src/listfunc.c:OnSets" },

    { "OnRight", 2, "pnt, elm",
      libGAP_FuncOnRight, "src/listfunc.c:OnRight" },

    { "OnLeftAntiOperation", 2, "pnt, elm",
      libGAP_FuncOnLeftAntiOperation, "src/listfunc.c:OnLeftAntiOperation" },

    { "OnLeftInverse", 2, "pnt, elm",
      libGAP_FuncOnLeftInverse, "src/listfunc.c:OnLeftInverse" },

    { "COPY_LIST_ENTRIES", -1, "srclist,srcstart,srcinc,dstlist,dststart,dstinc,number",
      libGAP_FuncCOPY_LIST_ENTRIES, "src/listfunc.c:COPY_LIST_ENTRIES" },

    { "STRONGLY_CONNECTED_COMPONENTS_DIGRAPH", 1, "digraph",
      libGAP_FuncSTRONGLY_CONNECTED_COMPONENTS_DIGRAPH, "src/listfunc.c:STRONGLY_CONNECTED_COMPONENTS_DIGRAPH" },

    { "BIMULT_MONOMIALS_ALGEBRA_ELEMENT",3,"mon, poly, mon",
      libGAP_FuncBIMULT_MONOMIALS_ALGEBRA_ELEMENT,"src/listfunc.c:BIMULT_MONOMIALS_ALGEBRA_ELEMENT" },

    { "HORSPOOL_LISTS",3,"list, sub, pre",
      libGAP_FuncHORSPOOL_LISTS,"src/listfunc.c:HORSPOOL_LISTS" },

    { "HEAP_SORT_PLIST",1,"list",
      libGAP_HEAP_SORT_PLIST,"src/listfunc.c:HEAP_SORT_PLIST" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrOpersFromTable( libGAP_GVarOpers );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarOpersFromTable( libGAP_GVarOpers );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoListFunc()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "listfunc",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoListFunc ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  listfunc.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
