/****************************************************************************
**
*W  integer.c                   GAP source                   Martin Schönert
**                                                           & Alice Niemeyer
**                                                           & Werner  Nickel
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file implements the  functions  handling  arbitrary  size  integers.
**
**  There are three integer types in GAP: 'T_INT', 'T_INTPOS' and 'T_INTNEG'.
**  Each integer has a unique representation, e.g., an integer  that  can  be
**  represented as 'T_INT' is never  represented as 'T_INTPOS' or 'T_INTNEG'.
**
**  'T_INT' is the type of those integers small enough to fit into  29  bits.
**  Therefore the value range of this small integers is $-2^{28}...2^{28}-1$.
**  This range contains about 99\% of all integers that usually occur in GAP.
**  (I just made up this number, obviously it depends on the application  :-)
**  Only these small integers can be used as index expression into sequences.
**
**  Small integers are represented by an immediate integer handle, containing
**  the value instead of pointing  to  it,  which  has  the  following  form:
**
**      +-------+-------+-------+-------+- - - -+-------+-------+-------+
**      | guard | sign  | bit   | bit   |       | bit   | tag   | tag   |
**      | bit   | bit   | 27    | 26    |       | 0     | 0     | 1     |
**      +-------+-------+-------+-------+- - - -+-------+-------+-------+
**
**  Immediate integers handles carry the tag 'T_INT', i.e. the last bit is 1.
**  This distinguishes immediate integers from other handles which  point  to
**  structures aligned on 4 byte boundaries and therefore have last bit zero.
**  (The second bit is reserved as tag to allow extensions of  this  scheme.)
**  Using immediates as pointers and dereferencing them gives address errors.
**
**  To aid overflow check the most significant two bits must always be equal,
**  that is to say that the sign bit of immediate integers has a  guard  bit.
**
**  The macros 'INTOBJ_INT' and 'INT_INTOBJ' should be used to convert between
**  a small integer value and its representation as immediate integer handle.
**
**  'T_INTPOS' and 'T_INTNEG' are the types of positive (respectively, negative)
**  integer values  that  can  not  be  represented  by  immediate  integers.
**
**  This large integers values are represented in signed base 65536 notation.
**  That means that the bag of  a  large  integer  has  the  following  form:
**
**      +-------+-------+-------+-------+- - - -+-------+-------+-------+
**      | digit | digit | digit | digit |       | digit | digit | digit |
**      | 0     | 1     | 2     | 3     |       | <n>-2 | <n>-1 | <n>   |
**      +-------+-------+-------+-------+- - - -+-------+-------+-------+
**
**  The value of this  is:  $d0 + d1 65536 + d2 65536^2 + ... + d_n 65536^n$,
**  respectively the negative of this if the type of this object is 'T_INTNEG'.
**
**  Each digit is  of  course  stored  as  a  16  bit  wide  unsigned  short.
**  Note that base 65536 allows us to multiply 2 digits and add a carry digit
**  without overflow in 32 bit long arithmetic, available on most processors.
**
**  The number of digits in every  large  integer  is  a  multiple  of  four.
**  Therefore the leading three digits of some values will actually be  zero.
**  Note that the uniqueness of representation implies that not four or more
**  leading digits may be zero, since |d0|d1|d2|d3| and |d0|d1|d2|d3|0|0|0|0|
**  have the same value only one, the first, can be a  legal  representation.
**
**  Because of this it is possible to do a  little  bit  of  loop  unrolling.
**  Thus instead of looping <n> times, handling one digit in each  iteration,
**  we can loop <n>/4 times, handling  four  digits  during  each  iteration.
**  This reduces the overhead of the loop by a factor of  approximately four.
**
**  Using base 65536 representation has advantages over  using  other  bases.
**  Integers in base 65536 representation can be packed  dense and therefore
**  use roughly 20\% less space than integers in base  10000  representation.
**  'SumInt' is 20\% and 'ProdInt' is 40\% faster for 65536 than  for  10000,
**  as their runtime is linear respectively quadratic in the number of digits.
**  Dividing by 65536 and computing the remainder mod 65536 can be done  fast
**  by shifting 16 bit to  the  right  and  by  taking  the  lower  16  bits.
**  Larger bases are difficult because the product of two digits will not fit
**  into 32 bit, which is the word size  of  most  modern  micro  processors.
**  Base 10000 would have the advantage that printing is  very  much  easier,
**  but 'PrInt' keeps a terminal at 9600 baud busy for almost  all  integers.
*/

#include        "system.h"              /* Ints, UInts                     */

#ifndef USE_GMP /* use this file, otherwise ignore what follows */

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "integer.h"             /* integers                        */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */

#include        "intfuncs.h"

#include <stdio.h>

/* for fallbacks to library */
libGAP_Obj libGAP_String;


/****************************************************************************
**
*F  TypeInt(<int>)  . . . . . . . . . . . . . . . . . . . . . kind of integer
**
**  'TypeInt' returns the kind of the integer <int>.
**
**  'TypeInt' is the function in 'TypeObjFuncs' for integers.
*/
libGAP_Obj             libGAP_TYPE_INT_SMALL_ZERO;
libGAP_Obj             libGAP_TYPE_INT_SMALL_POS;
libGAP_Obj             libGAP_TYPE_INT_SMALL_NEG;
libGAP_Obj             libGAP_TYPE_INT_LARGE_POS;
libGAP_Obj             libGAP_TYPE_INT_LARGE_NEG;

libGAP_Obj             libGAP_TypeIntSmall (
    libGAP_Obj                 val )
{
    if ( 0 == libGAP_INT_INTOBJ(val) ) {
        return libGAP_TYPE_INT_SMALL_ZERO;
    }
    else if ( 0 < libGAP_INT_INTOBJ(val) ) {
        return libGAP_TYPE_INT_SMALL_POS;
    }
    else /* if ( 0 > INT_INTOBJ(val) ) */ {
        return libGAP_TYPE_INT_SMALL_NEG;
    }
}

libGAP_Obj             libGAP_TypeIntLargePos (
    libGAP_Obj                 val )
{
    return libGAP_TYPE_INT_LARGE_POS;
}

libGAP_Obj             libGAP_TypeIntLargeNeg (
    libGAP_Obj                 val )
{
    return libGAP_TYPE_INT_LARGE_NEG;
}

/**************************************************************************
** The following two functions convert a C Int or UInt respectively into
** a GAP integer, either an immediate, small integer if possible or
** otherwise a new GAP bag with TNUM T_INTPOS or T_INTNEG.
**
*F ObjInt_Int(Int i)
*F ObjInt_UInt(UInt i)
**
****************************************************************************/

libGAP_Obj libGAP_ObjInt_Int(libGAP_Int i)
{
    libGAP_Obj n;
    libGAP_Int bound = 1L << libGAP_NR_SMALL_INT_BITS;
    if (i >= bound) {
        /* We have to make a big integer */
        n = libGAP_NewBag(libGAP_T_INTPOS,4*sizeof(libGAP_TypDigit));
        libGAP_ADDR_INT(n)[0] = (libGAP_TypDigit) (i & ((libGAP_Int) libGAP_INTBASE - 1L));
        libGAP_ADDR_INT(n)[1] = (libGAP_TypDigit) (i >> libGAP_NR_DIGIT_BITS);
        libGAP_ADDR_INT(n)[2] = 0;
        libGAP_ADDR_INT(n)[3] = 0;
        return n;
    } else if (-i > bound) {
        n = libGAP_NewBag(libGAP_T_INTNEG,4*sizeof(libGAP_TypDigit));
        libGAP_ADDR_INT(n)[0] = (libGAP_TypDigit) ((-i) & ((libGAP_Int) libGAP_INTBASE - 1L));
        libGAP_ADDR_INT(n)[1] = (libGAP_TypDigit) ((-i) >> libGAP_NR_DIGIT_BITS);
        libGAP_ADDR_INT(n)[2] = 0;
        libGAP_ADDR_INT(n)[3] = 0;
        return n;
    } else {
        return libGAP_INTOBJ_INT(i);
    }
}

libGAP_Obj libGAP_ObjInt_UInt(libGAP_UInt i)
{
    libGAP_Obj n;
    libGAP_UInt bound = 1UL << libGAP_NR_SMALL_INT_BITS;
    if (i >= bound) {
        /* We have to make a big integer */
        n = libGAP_NewBag(libGAP_T_INTPOS,4*sizeof(libGAP_TypDigit));
        libGAP_ADDR_INT(n)[0] = (libGAP_TypDigit) (i & ((libGAP_UInt) libGAP_INTBASE - 1L));
        libGAP_ADDR_INT(n)[1] = (libGAP_TypDigit) (i >> libGAP_NR_DIGIT_BITS);
        libGAP_ADDR_INT(n)[2] = 0;
        libGAP_ADDR_INT(n)[3] = 0;
        return n;
    } else {
        return libGAP_INTOBJ_INT(i);
    }
}



/****************************************************************************
**
*F  PrintInt( <int> ) . . . . . . . . . . . . . . . print an integer constant
**
**  'PrintInt'  prints  the integer  <int>   in the  usual  decimal notation.
**  'PrintInt' handles objects of type 'T_INT', 'T_INTPOS' and 'T_INTNEG'.
**
**  Large integers are first converted into  base  10000  and  then  printed.
**  The time for a conversion depends quadratically on the number of  digits.
**  For 2000 decimal digit integers, a screenfull,  it  is  reasonable  fast.
**
**  The number  of digits  needed in PrIntD[] is the ceiling of the logarithm
**  with respect to base PRINT_BASE of
**
**           ( (1<<NR_DIGIT_BITS) )^1000 - 1.
**
**  The latter is the largest number that can be represented with 1000 digits
**  of type TypDigit.
**
**  If NR_DIGIT_BITS is 16, we get 1205.
**  If NR_DIGIT_BITS is 32, we get 1071.
**
**  The subsidiary function IntToPrintBase converts an integer into base
**  PRINT_BASE, leaving the result in base PrIntD. It returns the index of the
**  most significant digits. It assumes that the argument is a large
**  integer small enough to fit.
*/

libGAP_TypDigit        libGAP_PrIntC [1000];          /* copy of integer to be printed   */

#ifdef libGAP_SYS_IS_64_BIT

#define libGAP_PRINT_BASE 1000000000L          /* 10^9                            */
#define libGAP_PRINT_FORMAT "%09d"             /* print 9 decimals at a time      */
#define libGAP_CHARS_PER_PRINT_BASE 9
libGAP_TypDigit        libGAP_PrIntD [1071];          /* integer converted to base 10^9  */
#define libGAP_NR_HEX_DIGITS 8

#else

#define libGAP_PRINT_BASE 10000
#define libGAP_PRINT_FORMAT "%04d"             /* print 4 decimals at a time      */
#define libGAP_CHARS_PER_PRINT_BASE 4
libGAP_TypDigit        libGAP_PrIntD [1205];          /* integer converted to base 10000 */
#define libGAP_NR_HEX_DIGITS 4

#endif
/****************************************************************************
**
*F  FuncHexStringInt( <self>, <int> ) . . . . . . . . hex string for integer
*F  FuncIntHexString( <self>, <string> ) . . . . . .  integer from hex string
**
**  The  function  `FuncHexStringInt'  constructs from  an  integer  the
**  corresponding string in  hexadecimal notation. It has  a leading '-'
**  for negative numbers and the digits 10..15 are written as A..F.
**
**  The  function `FuncIntHexString'  does  the converse,  but here  the
**  letters a..f are also allowed in <string> instead of A..F.
**
*/
libGAP_Obj libGAP_FuncHexStringInt( libGAP_Obj self, libGAP_Obj integer )
{
     libGAP_Int len, i, j, n;
     libGAP_UInt nf;
     libGAP_TypDigit d, f;
     libGAP_UInt1 *p, a;
     libGAP_Obj res;

     /* immediate integers */
     if (libGAP_IS_INTOBJ(integer)) {
         n = libGAP_INT_INTOBJ(integer);
         /* 0 is special */
         if (n == 0) {
             res = libGAP_NEW_STRING(1);
             libGAP_CHARS_STRING(res)[0] = '0';
             return res;
         }

         /* else we create a string big enough for any immediate integer */
         res = libGAP_NEW_STRING(2 * libGAP_NR_HEX_DIGITS + 1);
         p = libGAP_CHARS_STRING(res);
         /* handle sign */
         if (n<0) {
            p[0] = '-';
            n = -n;
            p++;
         }
         else
            libGAP_SET_LEN_STRING(res, libGAP_GET_LEN_STRING(res)-1);
         /* collect digits, skipping leading zeros */
         j = 0;
         nf = ((libGAP_UInt)15) << (4*(2*libGAP_NR_HEX_DIGITS-1));
         for (i = 2*libGAP_NR_HEX_DIGITS; i; i-- ) {
             a = ((libGAP_UInt)n & nf) >> (4*(i-1));
             if (j==0 && a==0) libGAP_SET_LEN_STRING(res, libGAP_GET_LEN_STRING(res)-1);
             else if (a<10) p[j++] = a + '0';
             else p[j++] = a - 10 + 'A';
             nf = nf >> 4;
         }
         /* final null character */
         p[j] = 0;
         return res;
     }
     else if (libGAP_TNUM_OBJ(integer) == libGAP_T_INTNEG || libGAP_TNUM_OBJ(integer) == libGAP_T_INTPOS) {
         /* nr of digits */
         len = libGAP_SIZE_INT(integer);
         for (; libGAP_ADDR_INT(integer)[len-1] == 0; len--);

         /* result string and sign */
         if (libGAP_TNUM_OBJ(integer) == libGAP_T_INTNEG) {
             res = libGAP_NEW_STRING(len * libGAP_NR_HEX_DIGITS + 1);
             p = libGAP_CHARS_STRING(res);
             p[0] = '-';
             p++;
         }
         else {
             res = libGAP_NEW_STRING(len * libGAP_NR_HEX_DIGITS);
             p = libGAP_CHARS_STRING(res);
         }
         /* collect digits */
         j = 0;
         for (; len; len--) {
             d = libGAP_ADDR_INT(integer)[len-1];
             f = 15L << (4*(libGAP_NR_HEX_DIGITS-1));
             for (i = libGAP_NR_HEX_DIGITS; i; i-- ) {
                 a = (d & f) >> (4*(i-1));
                 if (j==0 && a==0) libGAP_SET_LEN_STRING(res, libGAP_GET_LEN_STRING(res)-1);
                 else if (a<10) p[j++] = a + '0';
                 else p[j++] = a - 10 + 'A';
                 f = f >> 4;
             }
         }
         /* final null character */
         p[j] = 0;
         return res;
     }
     else
         libGAP_ErrorReturnObj("HexStringInt: argument must be integer, (not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(integer), 0L,
           "");
     return (libGAP_Obj) 0L; /* please picky cc */
}

libGAP_Obj  libGAP_FuncIntHexString( libGAP_Obj self,  libGAP_Obj str )
{
    libGAP_Obj res;
    libGAP_Int  i, j, s, ii, len, sign, nd;
    libGAP_UInt n;
    libGAP_UInt1 *p, a;
    libGAP_TypDigit d;

    if (! libGAP_IsStringConv(str))
        libGAP_ErrorReturnObj("IntHexString: argument must be string (not a %s)",
          (libGAP_Int)libGAP_TNAM_OBJ(str), 0L,
          "");

    /* number of hex digits and sign */
    len = libGAP_GET_LEN_STRING(str);
    if (len == 0) {
       res = libGAP_INTOBJ_INT(0);
       return res;
    }
    if (*(libGAP_CHARS_STRING(str)) == '-') {
       sign = -1;
       i = 1;
    }
    else {
       sign = 1;
       i = 0;
    }
    
    /* skip leading zeros */
    while ((libGAP_CHARS_STRING(str))[i] == '0' && i < len)
        i++;
    
    /* small int case */
    if ((len-i)*4 <= libGAP_NR_SMALL_INT_BITS) {
       n = 0;
       p = libGAP_CHARS_STRING(str);
       for (; i<len; i++) {
          a = p[i];
          if (a>96)
             a -= 87;
          else if (a>64)
             a -= 55;
          else
             a -= 48;
          if (a > 15)
              libGAP_ErrorReturnObj("IntHexString: non-valid character in hex-string",
                0L, 0L, "");
          n = (n << 4) + a;
       }
       res = libGAP_INTOBJ_INT(sign * n);
       return res;
    }
    else {
       /* number of Digits */
       nd = (len-i)/libGAP_NR_HEX_DIGITS;
       if (nd * libGAP_NR_HEX_DIGITS < (len-i)) nd++;
       nd += ((3*nd) % 4);
       if (sign == 1)
          res = libGAP_NewBag( libGAP_T_INTPOS, nd*sizeof(libGAP_TypDigit) );
       else
          res = libGAP_NewBag( libGAP_T_INTNEG, nd*sizeof(libGAP_TypDigit) );
       /* collect digits, easiest to start from the end */
       p = libGAP_CHARS_STRING(str);
       for (j=0; j < nd; j++) {
          d = 0;
          for (s=0, ii=len-j*libGAP_NR_HEX_DIGITS-1;
               ii>=i && ii>len-(j+1)*libGAP_NR_HEX_DIGITS-1;
               s+=4, ii--) {
             a = p[ii];
             if (a>96)
                a -= 87;
             else if (a>64)
                a -= 55;
             else
                a -= 48;
             if (a > 15)
               libGAP_ErrorReturnObj("IntHexString: non-valid character in hex-string",
                   0L, 0L, "");

             d += (a<<s);
          }
          libGAP_ADDR_INT(res)[j] = d;
       }
       return res;
    }
}



libGAP_Int libGAP_IntToPrintBase ( libGAP_Obj op )
{
    libGAP_UInt                 i, k;           /* loop counter                    */
    libGAP_TypDigit *          p;              /* loop pointer                    */
    libGAP_UInt                c;              /* carry in division step          */

    i = 0;
    for ( k = 0; k < libGAP_SIZE_INT(op); k++ )
      libGAP_PrIntC[k] = libGAP_ADDR_INT(op)[k];
    while ( k > 0 && libGAP_PrIntC[k-1] == 0 )  k--;
    while ( k > 0 ) {
      for ( c = 0, p = libGAP_PrIntC+k-1; p >= libGAP_PrIntC; p-- ) {
	c  = (c<<libGAP_NR_DIGIT_BITS) + *p;
	*p = (libGAP_TypDigit)(c / libGAP_PRINT_BASE);
	c  = c - libGAP_PRINT_BASE * *p;
      }
      libGAP_PrIntD[i++] = (libGAP_TypDigit)c;
      while ( k > 0 && libGAP_PrIntC[k-1] == 0 )  k--;
    }
    return i-1;

}

void            libGAP_PrintInt (
    libGAP_Obj                 op )
{
    libGAP_Int                 i;           /* loop counter                    */
    libGAP_Obj               str;           /* fallback to lib for large ints  */

    /* print a small integer                                               */
    if ( libGAP_IS_INTOBJ(op) ) {
        libGAP_Pr( "%>%d%<", libGAP_INT_INTOBJ(op), 0L );
    }

    /* print a large integer                                               */
    else if ( libGAP_SIZE_INT(op) < 1000 ) {

        /* start printing, %> means insert '\' before a linebreak          */
        libGAP_Pr("%>",0L,0L);

        if ( libGAP_TNUM_OBJ(op) == libGAP_T_INTNEG )
            libGAP_Pr("-",0L,0L);

        /* convert the integer into base PRINT_BASE                        */
	i = libGAP_IntToPrintBase(op);

        /* print the base PRINT_BASE digits                                 */
        libGAP_Pr( "%d", (libGAP_Int)libGAP_PrIntD[i], 0L );
        while ( i > 0 )
            libGAP_Pr( libGAP_PRINT_FORMAT, (libGAP_Int)libGAP_PrIntD[--i], 0L );
        libGAP_Pr("%<",0L,0L);

    }

    else {
        str = libGAP_CALL_1ARGS( libGAP_String, op );
        libGAP_Pr("%>%s%<",(libGAP_Int)(libGAP_CHARS_STRING(str)), 0);
        /* for a long time Print of large ints did not follow the general idea
         * that Print should produce something that can be read back into GAP:
           Pr("<<an integer too large to be printed>>",0L,0L); */
    }
}

/****************************************************************************
**
*F  FuncLog2Int( <self>, <int> ) . . . . . . . . . nr of bits of integer - 1
**
**  Given to GAP-Level as "Log2Int".
*/
libGAP_Obj libGAP_FuncLog2Int( libGAP_Obj self, libGAP_Obj integer)
{
  libGAP_Int res, d;
  libGAP_Int a, len;
  libGAP_Int mask;
  libGAP_TypDigit dmask;

  /* case of small ints */
  if (libGAP_IS_INTOBJ(integer)) {
    a = libGAP_INT_INTOBJ(integer);
    if (a < 0) a = -a;
    res = libGAP_NR_SMALL_INT_BITS;
    for(res = libGAP_NR_SMALL_INT_BITS - 1, mask = (libGAP_Int)1 << (libGAP_NR_SMALL_INT_BITS-1);
        (mask & a) == 0 && mask != (libGAP_Int)0;
        mask = mask >> 1, res--);
    return libGAP_INTOBJ_INT(res);
  }

  /* case of long ints */
  if (libGAP_TNUM_OBJ(integer) == libGAP_T_INTNEG || libGAP_TNUM_OBJ(integer) == libGAP_T_INTPOS) {
    for (len = libGAP_SIZE_INT(integer); libGAP_ADDR_INT(integer)[len-1] == 0; len--);
    /* Instead of computing 
          res = len * NR_DIGIT_BITS - d;
       we keep len and d separate, because on 32 bit systems res may
       not fit into an Int (and not into an immediate integer).            */
    d = 1;
    a = (libGAP_TypDigit)(libGAP_ADDR_INT(integer)[len-1]);
    for(dmask = (libGAP_TypDigit)1 << (libGAP_NR_DIGIT_BITS - 1);
        (dmask & a) == 0 && dmask != (libGAP_TypDigit)0;
        dmask = dmask >> 1, d++);
    return libGAP_DiffInt(libGAP_ProdInt(libGAP_INTOBJ_INT(len), libGAP_INTOBJ_INT(libGAP_NR_DIGIT_BITS)), 
                   libGAP_INTOBJ_INT(d));
  }
  else {
    libGAP_ErrorReturnObj("Log2Int: argument must be integer, (not a %s)",
           (libGAP_Int)libGAP_TNAM_OBJ(integer), 0L,
           "");
    return (libGAP_Obj) 0L; /* please picky cc */
  }
}

/****************************************************************************
**
*F  FuncSTRING_INT( <self>, <int> ) . . . . .  convert an integer to a string
**
**  `FuncSTRING_INT' returns an immutable string representing the integer
**  <int>
**
*/

libGAP_Obj libGAP_FuncSTRING_INT( libGAP_Obj self, libGAP_Obj integer )
{
  libGAP_Int x;
  libGAP_Obj str;
  libGAP_Int len;
  libGAP_Int i;
  libGAP_Char c;
  libGAP_Int j,top, chunk, neg;

  /* handle a small integer                                               */
  if ( libGAP_IS_INTOBJ(integer) ) {
    x = libGAP_INT_INTOBJ(integer);
    str = libGAP_NEW_STRING( (libGAP_NR_SMALL_INT_BITS+5)/3 );
    libGAP_RetypeBag(str, libGAP_T_STRING+libGAP_IMMUTABLE);
    len = 0;
    /* Case of zero */
    if (x == 0)
      {
	libGAP_CHARS_STRING(str)[0] = '0';
	libGAP_CHARS_STRING(str)[1] = '\0';
	libGAP_ResizeBag(str, libGAP_SIZEBAG_STRINGLEN(1));
	libGAP_SET_LEN_STRING(str, 1);

	return str;
      }
    /* Negative numbers */
    if (x < 0)
      {
	libGAP_CHARS_STRING(str)[len++] = '-';
	x = -x;
	neg = 1;
      }
    else
      neg = 0;

    /* Now the main case */
    while (x != 0)
      {
	libGAP_CHARS_STRING(str)[len++] = '0'+ x % 10;
	x /= 10;
      }
    libGAP_CHARS_STRING(str)[len] = '\0';

    /* finally, reverse the digits in place */
    for (i = neg; i < (neg+len)/2; i++)
      {
	c = libGAP_CHARS_STRING(str)[neg+len-1-i];
	libGAP_CHARS_STRING(str)[neg+len-1-i] = libGAP_CHARS_STRING(str)[i];
	libGAP_CHARS_STRING(str)[i] = c;
      }

    libGAP_ResizeBag(str, libGAP_SIZEBAG_STRINGLEN(len));
    libGAP_SET_LEN_STRING(str, len);
    return str;
  }

  /* handle a large integer                                               */
  else if ( libGAP_SIZE_INT(integer) < 1000 ) {

    /* convert the integer into base PRINT_BASE                        */
    len = libGAP_IntToPrintBase(integer);
    str = libGAP_NEW_STRING(libGAP_CHARS_PER_PRINT_BASE*(len+1)+2);
    libGAP_RetypeBag(str, libGAP_T_STRING+libGAP_IMMUTABLE);

    /* sort out the length of the top group */
    j = 1;
    top = (libGAP_Int)libGAP_PrIntD[len];
    while ( top >= j)
      {
	j *= 10;
      }

    /* Start filling in the string */
    i = 0;
    if ( libGAP_TNUM_OBJ(integer) == libGAP_T_INTNEG ) {
      libGAP_CHARS_STRING(str)[i++] = '-';
    }

    while (j > 1)
      {
	j /= 10;
        libGAP_CHARS_STRING(str)[i++] = '0' + (top / j) % 10;
      }

    /* Now the rest of the base PRINT_BASE digits are easy */
    while( len > 0)
      {
	chunk = (libGAP_Int)libGAP_PrIntD[--len];
	j = libGAP_PRINT_BASE/10;
	while (j > 0)
	  {
	    libGAP_CHARS_STRING(str)[i++] = '0' + (chunk / j) % 10;
	    j /= 10;
	  }
      }

    libGAP_CHARS_STRING(str)[i] = '\0';
    libGAP_ResizeBag(str, libGAP_SIZEBAG_STRINGLEN(i));
    libGAP_SET_LEN_STRING(str, i);
    return str;
  }
  else {

      /* Very large integer, fall back on the GAP function */
      return libGAP_CALL_1ARGS( libGAP_String, integer);
  }
}



/****************************************************************************
**
*F  EqInt( <intL>, <intR> ) . . . . . . . . .  test if two integers are equal
**
**  'EqInt' returns 1  if  the two integer   arguments <intL> and  <intR> are
**  equal and 0 otherwise.
*/
libGAP_Int             libGAP_EqInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 k;              /* loop counter                    */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */

    /* compare two small integers                                          */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {
        if ( libGAP_INT_INTOBJ(opL) == libGAP_INT_INTOBJ(opR) )  return 1L;
        else                                       return 0L;
    }

    /* compare a small and a large integer                                 */
    else if ( libGAP_IS_INTOBJ(opL) ) {
        return 0L;
    }
    else if ( libGAP_IS_INTOBJ(opR) ) {
        return 0L;
    }

    /* compare two large integers                                          */
    else {

        /* compare the sign and size                                       */
        if ( libGAP_TNUM_OBJ(opL) != libGAP_TNUM_OBJ(opR)
          || libGAP_SIZE_INT(opL) != libGAP_SIZE_INT(opR) )
            return 0L;

        /* set up the pointers                                             */
        l = libGAP_ADDR_INT(opL);
        r = libGAP_ADDR_INT(opR);

        /* run through the digits, four at a time                          */
        for ( k = libGAP_SIZE_INT(opL)/4-1; k >= 0; k-- ) {
            if ( *l++ != *r++ )  return 0L;
            if ( *l++ != *r++ )  return 0L;
            if ( *l++ != *r++ )  return 0L;
            if ( *l++ != *r++ )  return 0L;
        }

        /* no differences found, so they must be equal                     */
        return 1L;

    }
}


/****************************************************************************
**
*F  LtInt( <intL>, <intR> ) . . . . . test if an integer is less than another
**
**  'LtInt' returns 1 if the integer <intL> is strictly less than the integer
**  <intR> and 0 otherwise.
*/
libGAP_Int             libGAP_LtInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 k;              /* loop counter                    */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */

    /* compare two small integers                                          */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {
        if ( libGAP_INT_INTOBJ(opL) <  libGAP_INT_INTOBJ(opR) )  return 1L;
        else                                       return 0L;
    }

    /* compare a small and a large integer                                 */
    else if ( libGAP_IS_INTOBJ(opL) ) {
        if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )  return 1L;
        else                              return 0L;
    }
    else if ( libGAP_IS_INTOBJ(opR) ) {
        if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )  return 0L;
        else                              return 1L;
    }

    /* compare two large integers                                          */
    else {

        /* compare the sign and size                                       */
        if (      libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG
               && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
            return 1L;
        else if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS
               && libGAP_TNUM_OBJ(opR) == libGAP_T_INTNEG )
            return 0L;
        else if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS
                && libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR))
               || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG
                && libGAP_SIZE_INT(opL) > libGAP_SIZE_INT(opR)) )
            return 1L;
        else if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS
                && libGAP_SIZE_INT(opL) > libGAP_SIZE_INT(opR))
               || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG
                && libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR)) )
            return 0L;

        /* set up the pointers                                             */
        l = libGAP_ADDR_INT(opL);
        r = libGAP_ADDR_INT(opR);

        /* run through the digits, from the end downwards                  */
        for ( k = libGAP_SIZE_INT(opL)-1; k >= 0; k-- ) {
            if ( l[k] != r[k] ) {
                if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS
                   && l[k] < r[k])
                  || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG
                   && l[k] > r[k]) )
                    return 1L;
                else
                    return 0L;
            }
        }

        /* no differences found, so they must be equal                     */
        return 0L;

    }
}


/****************************************************************************
**
*F  SumInt( <intL>, <intR> )  . . . . . . . . . . . . . . sum of two integers
**
**  'SumInt' returns the sum of the two integer arguments <intL> and  <intR>.
**  'SumInt' handles operands of type 'T_INT', 'T_INTPOS' and 'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalSum' for a better  efficiency.
**
**  Is called from the 'EvalSum'  binop so both operands are already evaluated.
**
**  'SumInt' is a little bit difficult since there are 16  different cases to
**  handle, each operand can be positive or negative, small or large integer.
**  If the operands have opposite sign 'SumInt' calls 'DiffInt',  this  helps
**  reduce the total amount of code by a factor of two.
*/
libGAP_Obj             libGAP_SumInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Int                 k;              /* loop variable                   */
    libGAP_Int                 cs;             /* sum of two smalls */
    libGAP_UInt                 c;              /* sum of two digits               */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */
    libGAP_TypDigit *          s;              /* pointer into the sum            */
    libGAP_UInt *              l2;             /* pointer to get 2 digits at once */
    libGAP_UInt *              s2;             /* pointer to put 2 digits at once */
    libGAP_Obj                 sum;            /* handle of the result bag        */

    /* adding two small integers                                           */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {

        /* add two small integers with a small sum                         */
        /* add and compare top two bits to check that no overflow occured  */
        if ( libGAP_SUM_INTOBJS( sum, opL, opR ) ) {
            return sum;
        }

        /* add two small integers with a large sum                         */
        cs = libGAP_INT_INTOBJ(opL) + libGAP_INT_INTOBJ(opR);
        if ( 0 < cs ) {
            sum = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(sum)[0] = (libGAP_TypDigit)cs;
            libGAP_ADDR_INT(sum)[1] = (libGAP_TypDigit)(cs >> libGAP_NR_DIGIT_BITS);
        }
        else {
            sum = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(sum)[0] = (libGAP_TypDigit)(-cs);
            libGAP_ADDR_INT(sum)[1] = (libGAP_TypDigit)((-cs) >> libGAP_NR_DIGIT_BITS);
        }

    }

    /* adding one large integer and one small integer                      */
    else if ( libGAP_IS_INTOBJ(opL) || libGAP_IS_INTOBJ(opR) ) {

        /* make the right operand the small one                            */
        if ( libGAP_IS_INTOBJ(opL) ) {
            sum = opL;  opL = opR;  opR = sum;
        }

        /* if the integers have different sign, let 'DiffInt' do the work  */
        if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG && 0 <= libGAP_INT_INTOBJ(opR))
          || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS && libGAP_INT_INTOBJ(opR) <  0) ) {
            if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )  libGAP_RetypeBag( opL, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opL, libGAP_T_INTPOS );
            sum = libGAP_DiffInt( opR, opL );
            if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )  libGAP_RetypeBag( opL, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opL, libGAP_T_INTPOS );
            return sum;
        }

        /* allocate the result bag and set up the pointers                 */
        if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS ) {
            i   = libGAP_INT_INTOBJ(opR);
            sum = libGAP_NewBag( libGAP_T_INTPOS, (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        }
        else {
            i   = -libGAP_INT_INTOBJ(opR);
            sum = libGAP_NewBag( libGAP_T_INTNEG, (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        }
        l = libGAP_ADDR_INT(opL);
        s = libGAP_ADDR_INT(sum);

        /* add the first four digits,the right operand has only two digits */
        c = (libGAP_UInt)*l++ + (libGAP_TypDigit)i;                             *s++ = (libGAP_TypDigit)c;
        c = (libGAP_UInt)*l++ + (i>>libGAP_NR_DIGIT_BITS) + (c>>libGAP_NR_DIGIT_BITS); *s++ = (libGAP_TypDigit)c;
        c = (libGAP_UInt)*l++                      + (c>>libGAP_NR_DIGIT_BITS); *s++ = (libGAP_TypDigit)c;
        c = (libGAP_UInt)*l++                      + (c>>libGAP_NR_DIGIT_BITS); *s++ = (libGAP_TypDigit)c;

        /* propagate the carry, this loop is almost never executed         */
        for ( k = libGAP_SIZE_INT(opL)/4-1; k != 0 && (c>>libGAP_NR_DIGIT_BITS) != 0; k-- ) {
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
        }

        /* just copy the remaining digits, do it two digits at once        */
        for ( l2 = (libGAP_UInt*)l, s2 = (libGAP_UInt*)s; k != 0; k-- ) {
            *s2++ = *l2++;
            *s2++ = *l2++;
        }

        /* if there is a carry, enter it, otherwise shrink the sum         */
        if ( (c>>libGAP_NR_DIGIT_BITS) != 0 )
            *s++ = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
        else
            libGAP_ResizeBag( sum, (libGAP_SIZE_INT(sum)-4)*sizeof(libGAP_TypDigit) );

    }

    /* add two large integers                                              */
    else {

        /* if the integers have different sign, let 'DiffInt' do the work  */
        if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS && libGAP_TNUM_OBJ(opR) == libGAP_T_INTNEG)
          || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS) ) {
            if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )  libGAP_RetypeBag( opL, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opL, libGAP_T_INTPOS );
            sum = libGAP_DiffInt( opR, opL );
            if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )  libGAP_RetypeBag( opL, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opL, libGAP_T_INTPOS );
            return sum;
        }

        /* make the right operand the smaller one                          */
        if ( libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) ) {
            sum = opL;  opL = opR;  opR = sum;
        }

        /* allocate the result bag and set up the pointers                 */
        if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS ) {
            sum = libGAP_NewBag( libGAP_T_INTPOS, (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        }
        else {
            sum = libGAP_NewBag( libGAP_T_INTNEG, (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        }
        l = libGAP_ADDR_INT(opL);
        r = libGAP_ADDR_INT(opR);
        s = libGAP_ADDR_INT(sum);

        /* add the digits, convert to UInt to get maximum precision         */
        c = 0;
        for ( k = libGAP_SIZE_INT(opR)/4; k != 0; k-- ) {
            c = (libGAP_UInt)*l++ + (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
        }

        /* propagate the carry, this loop is almost never executed         */
        for ( k=(libGAP_SIZE_INT(opL)-libGAP_SIZE_INT(opR))/4;
             k!=0 && (c>>libGAP_NR_DIGIT_BITS)!=0; k-- ) {
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
            c = (libGAP_UInt)*l++ + (c>>libGAP_NR_DIGIT_BITS);  *s++ = (libGAP_TypDigit)c;
        }

        /* just copy the remaining digits, do it two digits at once        */
        for ( l2 = (libGAP_UInt*)l, s2 = (libGAP_UInt*)s; k != 0; k-- ) {
            *s2++ = *l2++;
            *s2++ = *l2++;
        }

        /* if there is a carry, enter it, otherwise shrink the sum         */
        if ( (c>>libGAP_NR_DIGIT_BITS) != 0 )
            *s++ = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
        else
            libGAP_ResizeBag( sum, (libGAP_SIZE_INT(sum)-4)*sizeof(libGAP_TypDigit) );

    }

    /* return the sum                                                      */
    return sum;
}


/****************************************************************************
**
*F  ZeroInt(<int>)  . . . . . . . . . . . . . . . . . . . .  zero of integers
*/
libGAP_Obj         libGAP_ZeroInt (
    libGAP_Obj                 op )
{
    return libGAP_INTOBJ_INT(0);
}


/****************************************************************************
**
*F  AInvInt(<int>)  . . . . . . . . . . . . .  additive inverse of an integer
*/
libGAP_Obj         libGAP_AInvInt (
    libGAP_Obj                 op )
{
    libGAP_Obj                 inv;
    libGAP_UInt                i;

    /* handle small integer                                                */
    if ( libGAP_IS_INTOBJ( op ) ) {

        /* special case (ugh)                                              */
        if ( op == libGAP_INTOBJ_INT( -(1L<<libGAP_NR_SMALL_INT_BITS) ) ) {
            inv = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(inv)[0] = 0;
            libGAP_ADDR_INT(inv)[1] = (libGAP_TypDigit)(1L<<(libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS));
        }

        /* general case                                                    */
        else {
            inv = libGAP_INTOBJ_INT( - libGAP_INT_INTOBJ( op ) );
        }

    }

    /* invert a large integer                                              */
    else {

        /* special case (ugh)                                              */
        if ( libGAP_TNUM_OBJ(op) == libGAP_T_INTPOS && libGAP_SIZE_INT(op) == 4
          && libGAP_ADDR_INT(op)[3] == 0
          && libGAP_ADDR_INT(op)[2] == 0
          && libGAP_ADDR_INT(op)[1] == (1L<<(libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS))
          && libGAP_ADDR_INT(op)[0] == 0 ) {
            inv = libGAP_INTOBJ_INT( -(1L<<libGAP_NR_SMALL_INT_BITS) );
        }

        /* general case                                                    */
        else {
            if ( libGAP_TNUM_OBJ(op) == libGAP_T_INTPOS ) {
                inv = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(op) );
            }
            else {
                inv = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(op) );
            }
            for ( i = 0; i < libGAP_SIZE_INT(op); i++ ) {
                libGAP_ADDR_INT(inv)[i] = libGAP_ADDR_INT(op)[i];
            }
        }

    }

    /* return the inverse                                                  */
    return inv;
}


/****************************************************************************
**
*F  DiffInt( <intL>, <intR> ) . . . . . . . . . .  difference of two integers
**
**  'DiffInt' returns the difference of the two integer arguments <intL>  and
**  <intR>.  'DiffInt' handles  operands  of  type  'T_INT',  'T_INTPOS'  and
**  'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalDiff' for a better efficiency.
**
**  Is called from the 'EvalDiff' binop so both operands are already evaluated.
**
**  'DiffInt' is a little bit difficult since there are 16 different cases to
**  handle, each operand can be positive or negative, small or large integer.
**  If the operands have opposite sign 'DiffInt' calls 'SumInt',  this  helps
**  reduce the total amount of code by a factor of two.
*/
libGAP_Obj             libGAP_DiffInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Int                 k;              /* loop variable                   */
    libGAP_Int                 c;              /* difference of two digits        */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */
    libGAP_TypDigit *          d;              /* pointer into the difference     */
    libGAP_UInt *              l2;             /* pointer to get 2 digits at once */
    libGAP_UInt *              d2;             /* pointer to put 2 digits at once */
    libGAP_Obj                 dif;            /* handle of the result bag        */

    /* subtracting two small integers                                      */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {

        /* subtract two small integers with a small difference             */
        /* sub and compare top two bits to check that no overflow occured  */
        if ( libGAP_DIFF_INTOBJS( dif, opL, opR ) ) {
            return dif;
        }

        /* subtract two small integers with a large difference             */
        c = libGAP_INT_INTOBJ(opL) - libGAP_INT_INTOBJ(opR);
        if ( 0 < c ) {
            dif = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(dif)[0] = (libGAP_TypDigit)c;
            libGAP_ADDR_INT(dif)[1] = (libGAP_TypDigit)(c >> libGAP_NR_DIGIT_BITS);
        }
        else {
            dif = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(dif)[0] = (libGAP_TypDigit)(-c);
            libGAP_ADDR_INT(dif)[1] = (libGAP_TypDigit)((-c) >> libGAP_NR_DIGIT_BITS);
        }

    }

    /* subtracting one small integer and one large integer                 */
    else if ( libGAP_IS_INTOBJ( opL ) || libGAP_IS_INTOBJ( opR ) ) {

        /* make the right operand the small one                            */
        if ( libGAP_IS_INTOBJ( opL ) ) {
            dif = opL;  opL = opR;  opR = dif;
            c = -1;
        }
        else {
            c =  1;
        }

        /* if the integers have different sign, let 'SumInt' do the work   */
        if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG && 0 <= libGAP_INT_INTOBJ(opR))
          || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS && libGAP_INT_INTOBJ(opR) < 0)  ) {
            if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )  libGAP_RetypeBag( opL, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opL, libGAP_T_INTPOS );
            dif = libGAP_SumInt( opL, opR );
            if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )  libGAP_RetypeBag( opL, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opL, libGAP_T_INTPOS );
            if ( c == 1 ) {
                if ( libGAP_TNUM_OBJ(dif) == libGAP_T_INTPOS )  libGAP_RetypeBag( dif, libGAP_T_INTNEG );
                else                              libGAP_RetypeBag( dif, libGAP_T_INTPOS );
            }
            return dif;
        }

        /* allocate the result bag and set up the pointers                 */
        if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS ) {
            i   = libGAP_INT_INTOBJ(opR);
            if ( c == 1 )  dif = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL) );
            else           dif = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opL) );
        }
        else {
            i   = - libGAP_INT_INTOBJ(opR);
            if ( c == 1 )  dif = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opL) );
            else           dif = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL) );
        }
        l = libGAP_ADDR_INT(opL);
        d = libGAP_ADDR_INT(dif);

        /* sub the first four digit, note the left operand has only two    */
        /*N (c>>16<) need not work, replace by (c<0?-1:0)                   */
        c = (libGAP_Int)*l++ - (libGAP_TypDigit)i;                              *d++ = (libGAP_TypDigit)c;
        c = (libGAP_Int)*l++ - (libGAP_TypDigit)(i/(1L<<libGAP_NR_DIGIT_BITS)) + (c<0?-1:0);  *d++ = (libGAP_TypDigit)c;
        c = (libGAP_Int)*l++                      + (c<0?-1:0);  *d++ = (libGAP_TypDigit)c;
        c = (libGAP_Int)*l++                      + (c<0?-1:0);  *d++ = (libGAP_TypDigit)c;

        /* propagate the carry, this loop is almost never executed         */
        for ( k = libGAP_SIZE_INT(opL)/4-1; k != 0 && c < 0; k-- ) {
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
        }

        /* just copy the remaining digits, do it two digits at once        */
        for ( l2 = (libGAP_UInt*)l, d2 = (libGAP_UInt*)d; k != 0; k-- ) {
            *d2++ = *l2++;
            *d2++ = *l2++;
        }

        /* no underflow since we subtracted a small int from a large one   */
        /* but there may be leading zeroes in the result, get rid of them  */
        /* occurs almost never, so it doesn't matter that it is expensive  */
        if ( ((libGAP_UInt*)d == d2
          && d[-4] == 0 && d[-3] == 0 && d[-2] == 0 && d[-1] == 0)
          || (libGAP_SIZE_INT(dif) == 4 && d[-2] == 0 && d[-1] == 0) ) {

            /* find the number of significant digits                       */
            d = libGAP_ADDR_INT(dif);
            for ( k = libGAP_SIZE_INT(dif); k != 0; k-- ) {
                if ( d[k-1] != 0 )
                    break;
            }

            /* reduce to small integer if possible, otherwise shrink bag   */
            if ( k <= 2 && libGAP_TNUM_OBJ(dif) == libGAP_T_INTPOS
              && (libGAP_UInt)(libGAP_INTBASE*d[1]+d[0])<(1L<<libGAP_NR_SMALL_INT_BITS) )
                dif = libGAP_INTOBJ_INT( libGAP_INTBASE*d[1]+d[0] );
            else if ( k <= 2 && libGAP_TNUM_OBJ(dif) == libGAP_T_INTNEG
              && (libGAP_UInt)(libGAP_INTBASE*d[1]+d[0])<=(1L<<libGAP_NR_SMALL_INT_BITS) )
                dif = libGAP_INTOBJ_INT( -(libGAP_Int)(libGAP_INTBASE*d[1]+d[0]) );
            else
                libGAP_ResizeBag( dif, (((k + 3) / 4) * 4) * sizeof(libGAP_TypDigit) );
        }

    }

    /* subtracting two large integers                                      */
    else {

        /* if the integers have different sign, let 'SumInt' do the work   */
        if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS && libGAP_TNUM_OBJ(opR) == libGAP_T_INTNEG)
          || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS) ) {
            if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )  libGAP_RetypeBag( opR, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opR, libGAP_T_INTPOS );
            dif = libGAP_SumInt( opL, opR );
            if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )  libGAP_RetypeBag( opR, libGAP_T_INTNEG );
            else                              libGAP_RetypeBag( opR, libGAP_T_INTPOS );
            return dif;
        }

        /* make the right operand the smaller one                          */
        if ( libGAP_SIZE_INT(opL) <  libGAP_SIZE_INT(opR)
          || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS && libGAP_LtInt(opL,opR) )
          || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG && libGAP_LtInt(opR,opL) ) ) {
            dif = opL;  opL = opR;  opR = dif;  c = -1;
        }
        else {
            c = 1;
        }

        /* allocate the result bag and set up the pointers                 */
        if ( (libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS && c ==  1)
          || (libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG && c == -1) )
            dif = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL) );
        else
            dif = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opL) );
        l = libGAP_ADDR_INT(opL);
        r = libGAP_ADDR_INT(opR);
        d = libGAP_ADDR_INT(dif);

        /* subtract the digits                                             */
        c = 0;
        for ( k = libGAP_SIZE_INT(opR)/4; k != 0; k-- ) {
            c = (libGAP_Int)*l++ - (libGAP_Int)*r++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ - (libGAP_Int)*r++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ - (libGAP_Int)*r++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ - (libGAP_Int)*r++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
        }

        /* propagate the carry, this loop is almost never executed         */
        for ( k=(libGAP_SIZE_INT(opL)-libGAP_SIZE_INT(opR))/4;
             k!=0 && c < 0; k-- ) {
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
            c = (libGAP_Int)*l++ + (c < 0 ? -1 : 0);  *d++ = (libGAP_TypDigit)c;
        }

        /* just copy the remaining digits, do it two digits at once        */
        for ( d2 = (libGAP_UInt*)d, l2 = (libGAP_UInt*)l; k != 0; k-- ) {
            *d2++ = *l2++;
            *d2++ = *l2++;
        }

        /* no underflow since we subtracted a small int from a large one   */
        /* but there may be leading zeroes in the result, get rid of them  */
        /* occurs almost never, so it doesn't matter that it is expensive  */
        if ( ((libGAP_UInt*)d == d2
          && d[-4] == 0 && d[-3] == 0 && d[-2] == 0 && d[-1] == 0)
          || (libGAP_SIZE_INT(dif) == 4 && d[-2] == 0 && d[-1] == 0) ) {

            /* find the number of significant digits                       */
            d = libGAP_ADDR_INT(dif);
            for ( k = libGAP_SIZE_INT(dif); k != 0; k-- ) {
                if ( d[k-1] != 0 )
                    break;
            }

            /* reduce to small integer if possible, otherwise shrink bag   */
            if ( k <= 2 && libGAP_TNUM_OBJ(dif) == libGAP_T_INTPOS
              && (libGAP_UInt)(libGAP_INTBASE*d[1]+d[0]) < (1L<<libGAP_NR_SMALL_INT_BITS) )
                dif = libGAP_INTOBJ_INT( libGAP_INTBASE*d[1]+d[0] );
            else if ( k <= 2 && libGAP_TNUM_OBJ(dif) == libGAP_T_INTNEG
              && (libGAP_UInt)(libGAP_INTBASE*d[1]+d[0])<=(1L<<libGAP_NR_SMALL_INT_BITS))
                dif = libGAP_INTOBJ_INT( -(libGAP_Int)(libGAP_INTBASE*d[1]+d[0]) );
            else
                libGAP_ResizeBag( dif, (((k + 3) / 4) * 4) * sizeof(libGAP_TypDigit) );

        }

    }

    /* return the difference                                               */
    return dif;
}


/****************************************************************************
**
*F  ProdInt( <intL>, <intR> ) . . . . . . . . . . . . product of two integers
**
**  'ProdInt' returns the product of the two  integer  arguments  <intL>  and
**  <intR>.  'ProdInt' handles  operands  of  type  'T_INT',  'T_INTPOS'  and
**  'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalProd' for a better efficiency.
**
**  Is called from the 'EvalProd' binop so both operands are already evaluated.
**
**  The only difficulty about this function is the fact that is has to handle
**  3 different situations, depending on how many arguments  are  small  ints.
*/
libGAP_Obj             libGAP_ProdInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;              /* loop count, value for small int */
    libGAP_Int                 k;              /* loop count, value for small int */
    libGAP_UInt                c;              /* product of two digits           */
    libGAP_TypDigit            l;              /* one digit of the left operand   */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */
    libGAP_TypDigit *          p;              /* pointer into the product        */
    libGAP_Obj                 prd;            /* handle of the result bag        */

    /* multiplying two small integers                                      */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {

        /* multiply two small integers with a small product                */
        /* multiply and divide back to check that no overflow occured      */
        if ( libGAP_PROD_INTOBJS( prd, opL, opR ) ) {
            return prd;
        }

        /* get the integer values                                          */
        i = libGAP_INT_INTOBJ(opL);
        k = libGAP_INT_INTOBJ(opR);

        /* allocate the product bag                                        */
        if ( (0 < i && 0 < k) || (i < 0 && k < 0) )
            prd = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
        else
            prd = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
        p = libGAP_ADDR_INT(prd);

        /* make both operands positive                                     */
        if ( i < 0 )  i = -i;
        if ( k < 0 )  k = -k;

        /* multiply digitwise                                              */
        c = (libGAP_UInt)(libGAP_TypDigit)i * (libGAP_TypDigit)k;            p[0] = (libGAP_TypDigit)c;
        c = (libGAP_UInt)(libGAP_TypDigit)i * (((libGAP_UInt)k)>>libGAP_NR_DIGIT_BITS)
          + (c>>libGAP_NR_DIGIT_BITS);                        p[1] = (libGAP_TypDigit)c;
        p[2] = c>>libGAP_NR_DIGIT_BITS;

        c = (libGAP_UInt)(libGAP_TypDigit)(((libGAP_UInt)i)>>libGAP_NR_DIGIT_BITS) * (libGAP_TypDigit)k
          + p[1];                                      p[1] = (libGAP_TypDigit)c;
        c = (libGAP_UInt)(libGAP_TypDigit)(((libGAP_UInt)i)>>libGAP_NR_DIGIT_BITS) * (libGAP_TypDigit)(((libGAP_UInt)k)>>libGAP_NR_DIGIT_BITS)
          + p[2] + (c>>libGAP_NR_DIGIT_BITS);                 p[2] = (libGAP_TypDigit)c;
        p[3] = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);

    }

    /* multiply a small and a large integer                                */
    else if ( libGAP_IS_INTOBJ(opL) || libGAP_IS_INTOBJ(opR) ) {

        /* make the left operand the small one                             */
        if ( libGAP_IS_INTOBJ(opR) ) {
            i = libGAP_INT_INTOBJ(opR);  opR = opL;
        }
        else {
            i = libGAP_INT_INTOBJ(opL);
        }

        /* handle trivial cases first                                      */
        if ( i == 0 )
            return libGAP_INTOBJ_INT(0);
        if ( i == 1 )
            return opR;

        /* the large integer 1<<28 times -1 is the small integer -(1<<28)  */
        if ( i == -1
          && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS && libGAP_SIZE_INT(opR) == 4
          && libGAP_ADDR_INT(opR)[3] == 0
          && libGAP_ADDR_INT(opR)[2] == 0
          && libGAP_ADDR_INT(opR)[1] == (1L<<(libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS))
          && libGAP_ADDR_INT(opR)[0] == 0 )
            return libGAP_INTOBJ_INT( -(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS) );

        /* multiplication by -1 is easy, just switch the sign and copy     */
        if ( i == -1 ) {
            if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
                prd = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opR) );
            else
                prd = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opR) );
            r = libGAP_ADDR_INT(opR);
            p = libGAP_ADDR_INT(prd);
            for ( k = libGAP_SIZE_INT(opR)/4; k != 0; k-- ) {
                /*N should be: *p2++=*r2++;  *p2++=*r2++;                  */
                *p++ = *r++;  *p++ = *r++;  *p++ = *r++;  *p++ = *r++;
            }
            return prd;
        }

        /* allocate a bag for the result                                   */
        if ( (0 < i && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS)
          || (i < 0 && libGAP_TNUM_OBJ(opR) == libGAP_T_INTNEG) )
            prd = libGAP_NewBag( libGAP_T_INTPOS, (libGAP_SIZE_INT(opR)+4)*sizeof(libGAP_TypDigit) );
        else
            prd = libGAP_NewBag( libGAP_T_INTNEG, (libGAP_SIZE_INT(opR)+4)*sizeof(libGAP_TypDigit) );
        if ( i < 0 )  i = -i;

        /* multiply with the lower digit of the left operand               */
        l = (libGAP_TypDigit)i;
        if ( l != 0 ) {

            r = libGAP_ADDR_INT(opR);
            p = libGAP_ADDR_INT(prd);
            c = 0;

            /* multiply the right with this digit and store in the product */
            for ( k = libGAP_SIZE_INT(opR)/4; k != 0; k-- ) {
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (c>>libGAP_NR_DIGIT_BITS);  *p++ = (libGAP_TypDigit)c;
            }
            *p = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
        }

        /* multiply with the larger digit of the left operand              */
        l = ((libGAP_UInt)i) >> libGAP_NR_DIGIT_BITS;
        if ( l != 0 ) {

            r = libGAP_ADDR_INT(opR);
            p = libGAP_ADDR_INT(prd) + 1;
            c = 0;

            /* multiply the right with this digit and add into the product */
            for ( k = libGAP_SIZE_INT(opR)/4; k != 0; k-- ) {
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
            }
            *p = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
        }

        /* remove the leading zeroes, note that there can't be more than 6 */
        p = libGAP_ADDR_INT(prd) + libGAP_SIZE_INT(prd);
        if ( p[-4] == 0 && p[-3] == 0 && p[-2] == 0 && p[-1] == 0 ) {
            libGAP_ResizeBag( prd, (libGAP_SIZE_INT(prd)-4)*sizeof(libGAP_TypDigit) );
        }

    }

    /* multiply two large integers                                         */
    else {

        /* make the left operand the smaller one, for performance          */
        if ( libGAP_SIZE_INT(opL) > libGAP_SIZE_INT(opR) ) {
            prd = opR;  opR = opL;  opL = prd;
        }

        /* allocate a bag for the result                                   */
        if ( libGAP_TNUM_OBJ(opL) == libGAP_TNUM_OBJ(opR) )
            prd = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL)+libGAP_SIZE_OBJ(opR) );
        else
            prd = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opL)+libGAP_SIZE_OBJ(opR) );

        /* run through the digits of the left operand                      */
        for ( i = 0; i < (libGAP_Int)libGAP_SIZE_INT(opL); i++ ) {

            /* set up pointer for one loop iteration                       */
            l = libGAP_ADDR_INT(opL)[i];
            if ( l == 0 )  continue;
            r = libGAP_ADDR_INT(opR);
            p = libGAP_ADDR_INT(prd) + i;
            c = 0;

            /* multiply the right with this digit and add into the product */
            for ( k = libGAP_SIZE_INT(opR)/4; k != 0; k-- ) {
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
                c = (libGAP_UInt)l * (libGAP_UInt)*r++ + (libGAP_UInt)*p + (c>>libGAP_NR_DIGIT_BITS); *p++ = (libGAP_TypDigit)c;
            }
            *p = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
        }

        /* remove the leading zeroes, note that there can't be more than 7 */
        p = libGAP_ADDR_INT(prd) + libGAP_SIZE_INT(prd);
        if ( p[-4] == 0 && p[-3] == 0 && p[-2] == 0 && p[-1] == 0 ) {
            libGAP_ResizeBag( prd, (libGAP_SIZE_INT(prd)-4)*sizeof(libGAP_TypDigit) );
        }

    }

    /* return the product                                                  */
    return prd;
}


/****************************************************************************
**
*F  ProdIntObj(<n>,<op>)  . . . . . . . . product of an integer and an object
*/
libGAP_Obj             libGAP_ProdIntObj (
    libGAP_Obj                 n,
    libGAP_Obj                 op )
{
    libGAP_Obj                 res = 0;        /* result                          */
    libGAP_UInt                i, k, l;        /* loop variables                  */

    /* if the integer is zero, return the neutral element of the operand   */
    if      ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  0 ) {
        res = libGAP_ZERO( op );
    }

    /* if the integer is one, return the object if immutable
       if mutable, add the object to its ZeroSameMutability to
       ensure correct mutability propagation */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  1 ) {
      if (libGAP_IS_MUTABLE_OBJ(op))
        res = libGAP_SUM(libGAP_ZERO(op),op);
      else
	res = op;
    }

    /* if the integer is minus one, return the inverse of the operand      */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) == -1 ) {
        res = libGAP_AINV( op );
    }

    /* if the integer is negative, invert the operand and the integer      */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) <  -1 ) {
        res = libGAP_AINV( op );
        if ( res == libGAP_Fail ) {
            return libGAP_ErrorReturnObj(
                "Operations: <obj> must have an additive inverse",
                0L, 0L,
                "you can supply an inverse <inv> for <obj> via 'return <inv>;'" );
        }
        res = libGAP_PROD( libGAP_AINV( n ), res );
    }

    /* if the integer is negative, invert the operand and the integer      */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INTNEG ) {
        res = libGAP_AINV( op );
        if ( res == libGAP_Fail ) {
            return libGAP_ErrorReturnObj(
                "Operations: <obj> must have an additive inverse",
                0L, 0L,
                "you can supply an inverse <inv> for <obj> via 'return <inv>;'" );
        }
        res = libGAP_PROD( libGAP_AINV( n ), res );
    }

    /* if the integer is small, compute the product by repeated doubling   */
    /* the loop invariant is <result> = <k>*<res> + <l>*<op>, <l> < <k>    */
    /* <res> = 0 means that <res> is the neutral element                   */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) >   1 ) {
        res = 0;
        k = 1L << (libGAP_NR_SMALL_INT_BITS+1);
        l = libGAP_INT_INTOBJ(n);
        while ( 1 < k ) {
            res = (res == 0 ? res : libGAP_SUM( res, res ));
            k = k / 2;
            if ( k <= l ) {
                res = (res == 0 ? op : libGAP_SUM( res, op ));
                l = l - k;
            }
        }
    }

    /* if the integer is large, compute the product by repeated doubling   */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INTPOS ) {
        res = 0;
        for ( i = libGAP_SIZE_OBJ(n)/sizeof(libGAP_TypDigit); 0 < i; i-- ) {
            k = 1L << (8*sizeof(libGAP_TypDigit));
            l = ((libGAP_TypDigit*) libGAP_ADDR_OBJ(n))[i-1];
            while ( 1 < k ) {
                res = (res == 0 ? res : libGAP_SUM( res, res ));
                k = k / 2;
                if ( k <= l ) {
                    res = (res == 0 ? op : libGAP_SUM( res, op ));
                    l = l - k;
                }
            }
        }
    }

    /* return the result                                                   */
    return res;
}

libGAP_Obj             libGAP_ProdIntObjFunc;

libGAP_Obj             libGAP_FuncPROD_INT_OBJ (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_ProdIntObj( opL, opR );
}


/****************************************************************************
**
*F  OneInt(<int>) . . . . . . . . . . . . . . . . . . . . . one of an integer
*/
libGAP_Obj             libGAP_OneInt (
    libGAP_Obj                 op )
{
    return libGAP_INTOBJ_INT( 1L );
}


/****************************************************************************
**
*F  PowInt( <intL>, <intR> )  . . . . . . . . . . . . . . power of an integer
**
**  'PowInt' returns the <intR>-th (an integer) power of the integer  <intL>.
**  'PowInt' handles operands of type 'T_INT', 'T_INTPOS' and 'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalPow' for a better  efficiency.
**
**  Is called from the 'EvalPow'  binop so both operands are already evaluated.
*/
libGAP_Obj             libGAP_PowInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;
    libGAP_Obj                 pow;

    /* power with a large exponent                                         */
    if ( ! libGAP_IS_INTOBJ(opR) ) {
        if ( opL == libGAP_INTOBJ_INT(0) )
            pow = libGAP_INTOBJ_INT(0);
        else if ( opL == libGAP_INTOBJ_INT(1) )
            pow = libGAP_INTOBJ_INT(1);
        else if ( opL == libGAP_INTOBJ_INT(-1) && libGAP_ADDR_INT(opR)[0] % 2 == 0 )
            pow = libGAP_INTOBJ_INT(1);
        else if ( opL == libGAP_INTOBJ_INT(-1) && libGAP_ADDR_INT(opR)[0] % 2 != 0 )
            pow = libGAP_INTOBJ_INT(-1);
        else {
            opR = libGAP_ErrorReturnObj(
                "Integer operands: <exponent> is too large",
                0L, 0L,
                "you can replace the integer <exponent> via 'return <exponent>;'" );
            return libGAP_POW( opL, opR );
        }
    }

    /* power with a negative exponent                                      */
    else if ( libGAP_INT_INTOBJ(opR) < 0 ) {
        if ( opL == libGAP_INTOBJ_INT(0) ) {
            opL = libGAP_ErrorReturnObj(
                "Integer operands: <base> must not be zero",
                0L, 0L,
                "you can replace the integer <base> via 'return <base>;'" );
            return libGAP_POW( opL, opR );
        }
        else if ( opL == libGAP_INTOBJ_INT(1) )
            pow = libGAP_INTOBJ_INT(1);
        else if ( opL == libGAP_INTOBJ_INT(-1) && libGAP_INT_INTOBJ(opR) % 2 == 0 )
            pow = libGAP_INTOBJ_INT(1);
        else if ( opL == libGAP_INTOBJ_INT(-1) && libGAP_INT_INTOBJ(opR) % 2 != 0 )
            pow = libGAP_INTOBJ_INT(-1);
        else
            pow = libGAP_QUO( libGAP_INTOBJ_INT(1),
                       libGAP_PowInt( opL, libGAP_INTOBJ_INT( -libGAP_INT_INTOBJ(opR)) ) );
    }

    /* power with a small positive exponent, do it by a repeated squaring  */
    else {
        pow = libGAP_INTOBJ_INT(1);
        i = libGAP_INT_INTOBJ(opR);
        while ( i != 0 ) {
            if ( i % 2 == 1 )  pow = libGAP_ProdInt( pow, opL );
            if ( i     >  1 )  opL = libGAP_ProdInt( opL, opL );
            i = i / 2;
        }
    }

    /* return the power                                                    */
    return pow;
}


/****************************************************************************
**
*F  PowObjInt(<op>,<n>) . . . . . . . . . . power of an object and an integer
*/
static libGAP_Obj libGAP_OneAttr;

libGAP_Obj             libGAP_PowObjInt (
    libGAP_Obj                 op,
    libGAP_Obj                 n )
{
    libGAP_Obj                 res = 0;        /* result                          */
    libGAP_UInt                i, k, l;        /* loop variables                  */

    /* if the integer is zero, return the neutral element of the operand   */
    if      ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  0 ) {
      return libGAP_ONE_MUT( op );
    }

    /* if the integer is one, return a copy of the operand                 */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) ==  1 ) {
        res = libGAP_CopyObj( op, 1 );
    }

    /* if the integer is minus one, return the inverse of the operand      */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) == -1 ) {
        res = libGAP_INV_MUT( op );
    }

    /* if the integer is negative, invert the operand and the integer      */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) <   0 ) {
        res = libGAP_INV_MUT( op );
        if ( res == libGAP_Fail ) {
            return libGAP_ErrorReturnObj(
                "Operations: <obj> must have an inverse",
                0L, 0L,
                "you can supply an inverse <inv> for <obj> via 'return <inv>;'" );
        }
        res = libGAP_POW( res, libGAP_AINV( n ) );
    }

    /* if the integer is negative, invert the operand and the integer      */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INTNEG ) {
        res = libGAP_INV_MUT( op );
        if ( res == libGAP_Fail ) {
            return libGAP_ErrorReturnObj(
                "Operations: <obj> must have an inverse",
                0L, 0L,
                "you can supply an inverse <inv> for <obj> via 'return <inv>;'" );
        }
        res = libGAP_POW( res, libGAP_AINV( n ) );
    }

    /* if the integer is small, compute the power by repeated squaring     */
    /* the loop invariant is <result> = <res>^<k> * <op>^<l>, <l> < <k>    */
    /* <res> = 0 means that <res> is the neutral element                   */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INT && libGAP_INT_INTOBJ(n) >   0 ) {
        res = 0;
        k = 1L << (libGAP_NR_SMALL_INT_BITS+1);
        l = libGAP_INT_INTOBJ(n);
        while ( 1 < k ) {
            res = (res == 0 ? res : libGAP_PROD( res, res ));
            k = k / 2;
            if ( k <= l ) {
                res = (res == 0 ? op : libGAP_PROD( res, op ));
                l = l - k;
            }
        }
    }

    /* if the integer is large, compute the power by repeated squaring     */
    else if ( libGAP_TNUM_OBJ(n) == libGAP_T_INTPOS ) {
        res = 0;
        for ( i = libGAP_SIZE_OBJ(n)/sizeof(libGAP_TypDigit); 0 < i; i-- ) {
            k = 1L << (8*sizeof(libGAP_TypDigit));
            l = ((libGAP_TypDigit*) libGAP_ADDR_OBJ(n))[i-1];
            while ( 1 < k ) {
                res = (res == 0 ? res : libGAP_PROD( res, res ));
                k = k / 2;
                if ( k <= l ) {
                    res = (res == 0 ? op : libGAP_PROD( res, op ));
                    l = l - k;
                }
            }
        }
    }

    /* return the result                                                   */
    return res;
}

libGAP_Obj             libGAP_PowObjIntFunc;

libGAP_Obj             libGAP_FuncPOW_OBJ_INT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_PowObjInt( opL, opR );
}


/****************************************************************************
**
*F  ModInt( <intL>, <intR> )  . representative of residue class of an integer
**
**  'ModInt' returns the smallest positive representant of the residue  class
**  of the  integer  <intL>  modulo  the  integer  <intR>.  'ModInt'  handles
**  operands of type 'T_INT', 'T_INTPOS', 'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**  This case is usually already handled in 'EvalMod' for a better efficiency.
p**
**  Is called from the 'EvalMod'  binop so both operands are already evaluated.
*/
libGAP_Obj             libGAP_ModInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;              /* loop count, value for small int */
    libGAP_Int                 k;              /* loop count, value for small int */
    libGAP_UInt                c;              /* product of two digits           */
    libGAP_TypDigit            d;              /* carry into the next digit       */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */
    libGAP_TypDigit            r1;             /* leading digit of the right oper */
    libGAP_TypDigit            r2;             /* next digit of the right operand */
    libGAP_UInt                rs;             /* size of the right operand       */
    libGAP_UInt                e;              /* we mult r by 2^e so r1 >= 32768 */
    libGAP_Obj                 mod;            /* handle of the remainder bag     */
    libGAP_TypDigit *          m;              /* pointer into the remainder      */
    libGAP_UInt                m01;            /* leading two digits of the rem.  */
    libGAP_TypDigit            m2;             /* next digit of the remainder     */
    libGAP_TypDigit            qi;             /* guessed digit of the quotient   */

    /* compute the remainder of two small integers                         */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {

        /* pathological case first                                         */
        if ( opR == libGAP_INTOBJ_INT(0) ) {
            opR = libGAP_ErrorReturnObj(
                "Integer operations: <divisor> must be nonzero",
                0L, 0L,
                "you can replace the integer <divisor> via 'return <divisor>;'" );
            return libGAP_MOD( opL, opR );
        }

        /* get the integer values                                          */
        i = libGAP_INT_INTOBJ(opL);
        k = libGAP_INT_INTOBJ(opR);

        /* compute the remainder, make sure we divide only positive numbers*/
        if (      0 <= i && 0 <= k )  i =       (  i %  k );
        else if ( 0 <= i && k <  0 )  i =       (  i % -k );
        else if ( i < 0  && 0 <= k )  i = ( k - ( -i %  k )) % k;
        else if ( i < 0  && k <  0 )  i = (-k - ( -i % -k )) % k;
        mod = libGAP_INTOBJ_INT( i );

    }

    /* compute the remainder of a small integer by a large integer         */
    else if ( libGAP_IS_INTOBJ(opL) ) {

        /* the small int -(1<<28) mod the large int (1<<28) is 0           */
        if ( opL == libGAP_INTOBJ_INT((libGAP_UInt)-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS))
          && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS && libGAP_SIZE_INT(opR) == 4
          && libGAP_ADDR_INT(opR)[3] == 0
          && libGAP_ADDR_INT(opR)[2] == 0
          && libGAP_ADDR_INT(opR)[1] == (libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS)
          && libGAP_ADDR_INT(opR)[0] == 0 )
            mod = libGAP_INTOBJ_INT(0);

        /* in all other cases the remainder is equal the left operand      */
        else if ( 0 <= libGAP_INT_INTOBJ(opL) )
            mod = opL;
        else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
            mod = libGAP_SumInt( opL, opR );
        else
            mod = libGAP_DiffInt( opL, opR );

    }

    /* compute the remainder of a large integer by a small integer         */
    else if ( libGAP_IS_INTOBJ(opR)
           && libGAP_INT_INTOBJ(opR) < libGAP_INTBASE
           && -(libGAP_Int)libGAP_INTBASE <= libGAP_INT_INTOBJ(opR) ) {

        /* pathological case first                                         */
        if ( opR == libGAP_INTOBJ_INT(0) ) {
            opR = libGAP_ErrorReturnObj(
                "Integer operations: <divisor> must be nonzero",
                0L, 0L,
                "you can replace the integer <divisor> via 'return <divisor>;'" );
            return libGAP_MOD( opL, opR );
        }

        /* get the integer value, make positive                            */
        i = libGAP_INT_INTOBJ(opR);  if ( i < 0 )  i = -i;

        /* maybe its trivial                                               */
        if ( libGAP_INTBASE % i == 0 ) {
            c = libGAP_ADDR_INT(opL)[0] % i;
        }

        /* otherwise run through the left operand and divide digitwise     */
        else {
            l = libGAP_ADDR_INT(opL) + libGAP_SIZE_INT(opL) - 1;
            c = 0;
            for ( ; l >= libGAP_ADDR_INT(opL); l-- ) {
                c  = (c<<libGAP_NR_DIGIT_BITS) + (libGAP_Int)*l;
                c  = c % i;
            }
        }

        /* now c is the result, it has the same sign as the left operand   */
        if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
            mod = libGAP_INTOBJ_INT( c );
        else if ( c == 0 )
            mod = libGAP_INTOBJ_INT( c );
        else if ( 0 <= libGAP_INT_INTOBJ(opR) )
            mod = libGAP_SumInt( libGAP_INTOBJ_INT( -(libGAP_Int)c ), opR );
        else
            mod = libGAP_DiffInt( libGAP_INTOBJ_INT( -(libGAP_Int)c ), opR );

    }

    /* compute the remainder of a large integer modulo a large integer     */
    else {
        /* a small divisor larger than one digit isn't handled above       */
        if ( libGAP_IS_INTOBJ(opR) ) {
            if ( 0 < libGAP_INT_INTOBJ(opR) ) {
                mod = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(mod)[0] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(mod)[1] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR)>>libGAP_NR_DIGIT_BITS);
                opR = mod;
            }
            else {
                mod = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(mod)[0] = (libGAP_TypDigit)(-libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(mod)[1] = (libGAP_TypDigit)((-libGAP_INT_INTOBJ(opR))>>libGAP_NR_DIGIT_BITS);
                opR = mod;
            }
        }

        /* trivial case first                                              */
        if ( libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) ) {
            if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
                return opL;
            else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
                return libGAP_SumInt( opL, opR );
            else
                return libGAP_DiffInt( opL, opR );
        }

        /* copy the left operand into a new bag, this holds the remainder  */
        mod = libGAP_NewBag( libGAP_TNUM_OBJ(opL), (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        l = libGAP_ADDR_INT(opL);
        m = libGAP_ADDR_INT(mod);
        for ( k = libGAP_SIZE_INT(opL)-1; k >= 0; k-- )
            *m++ = *l++;

        /* get the size of the right operand, and get the leading 2 digits */
        rs = libGAP_SIZE_INT(opR);
        r  = libGAP_ADDR_INT(opR);
        while ( r[rs-1] == 0 )  rs--;
        for ( e = 0;
             ((libGAP_Int)r[rs-1]<<e) + (e ?  r[rs-2]>>(libGAP_NR_DIGIT_BITS-e) : 0)
                               < libGAP_INTBASE/2; e++ ) ;

        r1 = ((libGAP_Int)r[rs-1]<<e) + (e ? r[rs-2]>>(libGAP_NR_DIGIT_BITS-e) : 0);
        r2 = ((libGAP_Int)r[rs-2]<<e) + ((rs>=3 && e)? r[rs-3]>>(libGAP_NR_DIGIT_BITS-e) : 0);

        /* run through the digits in the quotient                          */
        for ( i = libGAP_SIZE_INT(mod)-libGAP_SIZE_INT(opR)-1; i >= 0; i-- ) {

            /* guess the factor                                            */
            m = libGAP_ADDR_INT(mod) + rs + i;
            m01 = ((libGAP_INTBASE*m[0]+m[-1])<<e)
                           + (e ? m[-2]>>(libGAP_NR_DIGIT_BITS-e) : 0);
            if ( m01 == 0 )  continue;
            m2  = ((libGAP_Int)m[-2]<<e) + ((e && rs+i>=3) ? m[-3]>>(libGAP_NR_DIGIT_BITS-e) : 0);
            if ( ((libGAP_Int)m[0]<<e)+(e ? m[-1]>>(libGAP_NR_DIGIT_BITS-e) : 0) < r1 )
                    qi = m01 / r1;
            else    qi = libGAP_INTBASE - 1;
            while ( m01-(libGAP_Int)qi*r1 < (libGAP_Int)libGAP_INTBASE
                    && libGAP_INTBASE*(m01-(libGAP_Int)qi*r1)+m2 < (libGAP_Int)qi*r2 )
                    qi--;

            /* m = m - qi * r;                                             */
            d = 0;
            m = libGAP_ADDR_INT(mod) + i;
            r = libGAP_ADDR_INT(opR);
            for ( k = 0; k < (libGAP_Int)rs; ++k, ++m, ++r ) {
                c = (libGAP_Int)*m - (libGAP_Int)qi * *r - (libGAP_Int)d;
		*m = (libGAP_TypDigit)c;
		d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
            }
            c = (libGAP_Int)*m - d;  *m = (libGAP_TypDigit)c;  d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);

            /* if we have a borrow then add back                           */
            if ( d != 0 ) {
                d = 0;
                m = libGAP_ADDR_INT(mod) + i;
                r = libGAP_ADDR_INT(opR);
                for ( k = 0; k < (libGAP_Int)rs; ++k, ++m, ++r ) {
                    c = (libGAP_Int)*m + (libGAP_Int)*r + (libGAP_Int)d;
                    *m = (libGAP_TypDigit)c;
                    d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                }
                c = (libGAP_Int)*m + d;  *m = (libGAP_TypDigit)c;  d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                qi--;
            }

        }

        /* remove the leading zeroes                                       */
        m = libGAP_ADDR_INT(mod) + libGAP_SIZE_INT(mod);
        if ( (m[-4] == 0 && m[-3] == 0 && m[-2] == 0 && m[-1] == 0)
          || (libGAP_SIZE_INT(mod) == 4 && m[-2] == 0 && m[-1] == 0) ) {

            /* find the number of significant digits                       */
            m = libGAP_ADDR_INT(mod);
            for ( k = libGAP_SIZE_INT(mod); k != 0; k-- ) {
                if ( m[k-1] != 0 )
                    break;
            }

            /* reduce to small integer if possible, otherwise shrink bag   */

            if ( k <= 2 && libGAP_TNUM_OBJ(mod) == libGAP_T_INTPOS
              && (libGAP_UInt)(libGAP_INTBASE*m[1]+m[0])<(1L<<libGAP_NR_SMALL_INT_BITS) )
                mod = libGAP_INTOBJ_INT( libGAP_INTBASE*m[1]+m[0] );
            else if ( k <= 2 && libGAP_TNUM_OBJ(mod) == libGAP_T_INTNEG
              && (libGAP_UInt)(libGAP_INTBASE*m[1]+m[0])<=(1L<<libGAP_NR_SMALL_INT_BITS) )
                mod = libGAP_INTOBJ_INT( -(libGAP_Int)(libGAP_INTBASE*m[1]+m[0]) );
            else
                libGAP_ResizeBag( mod, (((k + 3) / 4) * 4) * sizeof(libGAP_TypDigit) );
        }

        /* make the representative positive                                  */
        if ( (libGAP_TNUM_OBJ(mod) == libGAP_T_INT && libGAP_INT_INTOBJ(mod) < 0)
          || libGAP_TNUM_OBJ(mod) == libGAP_T_INTNEG ) {
            if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS )
                mod = libGAP_SumInt( mod, opR );
            else
                mod = libGAP_DiffInt( mod, opR );
        }

    }

    /* return the result                                                   */
    return mod;
}


/****************************************************************************
**
*F  FuncIS_INT( <self>, <val> ) . . . . . . . . . . internal function 'IsInt'
**
**  'FuncIS_INT' implements the internal filter 'IsInt'.
**
**  'IsInt( <val> )'
**
**  'IsInt'  returns 'true'  if the  value  <val>  is an integer  and 'false'
**  otherwise.
*/
libGAP_Obj libGAP_IsIntFilt;

libGAP_Obj libGAP_FuncIS_INT (
    libGAP_Obj                 self,
    libGAP_Obj                 val )
{
    /* return 'true' if <obj> is an integer and 'false' otherwise          */
    if ( libGAP_TNUM_OBJ(val) == libGAP_T_INT
      || libGAP_TNUM_OBJ(val) == libGAP_T_INTPOS
      || libGAP_TNUM_OBJ(val) == libGAP_T_INTNEG ) {
        return libGAP_True;
    }
    else if ( libGAP_TNUM_OBJ(val) <= libGAP_FIRST_EXTERNAL_TNUM ) {
        return libGAP_False;
    }
    else {
        return libGAP_DoFilter( self, val );
    }
}


/****************************************************************************
**
*F  QuoInt( <intL>, <intR> )  . . . . . . . . . . . quotient of two integers
**
**  'QuoInt' returns the integer part of the two integers <intL> and  <intR>.
**  'QuoInt' handles operands of type  'T_INT',  'T_INTPOS'  and  'T_INTNEG'.
**
**  It can also be used in the cases that both operands  are  small  integers
**  and the result is a small integer too,  i.e., that  no  overflow  occurs.
**
**  Note that this routine is not called from 'EvalQuo', the  division  of  two
**  integers yields  a  rational  and  is  therefor  performed  in  'QuoRat'.
**  This operation is however available through the internal function 'Quo'.
*/
libGAP_Obj             libGAP_QuoInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;              /* loop count, value for small int */
    libGAP_Int                 k;              /* loop count, value for small int */
    libGAP_UInt                c;              /* product of two digits           */
    libGAP_TypDigit            d;              /* carry into the next digit       */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_UInt                l01;            /* leading two digits of the left  */
    libGAP_TypDigit            l2;             /* next digit of the left operand  */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */
    libGAP_TypDigit            r1;             /* leading digit of the right oper */
    libGAP_TypDigit            r2;             /* next digit of the right operand */
    libGAP_UInt                rs;             /* size of the right operand       */
    libGAP_UInt                e;              /* we mult r by 2^e so r1 >= 32768 */
    libGAP_TypDigit *          q;              /* pointer into the quotient       */
    libGAP_Obj                 quo;            /* handle of the result bag        */
    libGAP_TypDigit            qi;             /* guessed digit of the quotient   */

    /* divide to small integers                                            */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {

        /* pathological case first                                         */
        if ( opR == libGAP_INTOBJ_INT(0) ) {
            opR = libGAP_ErrorReturnObj(
                "Integer operations: <divisor> must be nonzero",
                0L, 0L,
                "you can replace the integer <divisor> via 'return <divisor>;'" );
            return libGAP_QUO( opL, opR );
        }

        /* the small int -(1<<28) divided by -1 is the large int (1<<28)   */
        if ( opL == libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS))
            && opR == libGAP_INTOBJ_INT(-1) ) {
            quo = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(quo)[1] = 1L<<(libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS);
            libGAP_ADDR_INT(quo)[0] = 0;
            return quo;
        }

        /* get the integer values                                          */
        i = libGAP_INT_INTOBJ(opL);
        k = libGAP_INT_INTOBJ(opR);

        /* divide, make sure we divide only positive numbers               */
        if (      0 <= i && 0 <= k )  i =    (  i /  k );
        else if ( 0 <= i && k <  0 )  i =  - (  i / -k );
        else if ( i < 0  && 0 <= k )  i =  - ( -i /  k );
        else if ( i < 0  && k <  0 )  i =    ( -i / -k );
        quo = libGAP_INTOBJ_INT( i );

    }

    /* divide a small integer by a large one                               */
    else if ( libGAP_IS_INTOBJ(opL) ) {

        /* the small int -(1<<28) divided by the large int (1<<28) is -1   */

        if ( opL == libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS))
          && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS && libGAP_SIZE_INT(opR) == 4
          && libGAP_ADDR_INT(opR)[3] == 0
          && libGAP_ADDR_INT(opR)[2] == 0
          && libGAP_ADDR_INT(opR)[1] == 1L<<(libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS)
          && libGAP_ADDR_INT(opR)[0] == 0 )
            quo = libGAP_INTOBJ_INT(-1);

        /* in all other cases the quotient is of course zero               */
        else
            quo = libGAP_INTOBJ_INT(0);

    }

    /* divide a large integer by a small integer                           */
    else if ( libGAP_IS_INTOBJ(opR)
           && libGAP_INT_INTOBJ(opR) < libGAP_INTBASE
           && -(libGAP_Int)libGAP_INTBASE  <= libGAP_INT_INTOBJ(opR) ) {

        /* pathological case first                                         */
        if ( opR == libGAP_INTOBJ_INT(0) ) {
            opR = libGAP_ErrorReturnObj(
                "Integer operations: <divisor> must be nonzero",
                0L, 0L,
                "you can replace the integer <divisor> via 'return <divisor>;'" );
            return libGAP_QUO( opL, opR );
        }

        /* get the integer value, make positive                            */
        i = libGAP_INT_INTOBJ(opR);  if ( i < 0 )  i = -i;

        /* allocate a bag for the result and set up the pointers           */
        if ( (libGAP_TNUM_OBJ(opL)==libGAP_T_INTPOS && 0 < libGAP_INT_INTOBJ(opR))
          || (libGAP_TNUM_OBJ(opL)==libGAP_T_INTNEG && libGAP_INT_INTOBJ(opR) < 0) )
            quo = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL) );
        else
            quo = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opL) );
        l = libGAP_ADDR_INT(opL) + libGAP_SIZE_INT(opL) - 1;
        q = libGAP_ADDR_INT(quo) + libGAP_SIZE_INT(quo) - 1;

        /* run through the left operand and divide digitwise               */
        c = 0;
        for ( ; l >= libGAP_ADDR_INT(opL); l--, q-- ) {
            c  = (c<<libGAP_NR_DIGIT_BITS) + (libGAP_Int)*l;
            *q = (libGAP_TypDigit)(c / i);
            c  = c - i * *q;
            /*N clever compilers may prefer:  c  = c % i;                  */
        }

        /* remove the leading zeroes, note that there can't be more than 5 */
        q = libGAP_ADDR_INT(quo) + libGAP_SIZE_INT(quo);
        if ( q[-4] == 0 && q[-3] == 0 && q[-2] == 0 && q[-1] == 0 ) {
            libGAP_ResizeBag( quo, (libGAP_SIZE_INT(quo)-4)*sizeof(libGAP_TypDigit) );
        }

        /* reduce to small integer if possible                             */
        q = libGAP_ADDR_INT(quo) + libGAP_SIZE_INT(quo);
        if ( libGAP_SIZE_INT(quo) == 4 && q[-2] == 0 && q[-1] == 0 ) {
            if ( libGAP_TNUM_OBJ(quo) == libGAP_T_INTPOS
              &&(libGAP_UInt)(libGAP_INTBASE*q[-3]+q[-4])
                < (1L<<libGAP_NR_SMALL_INT_BITS))
                quo = libGAP_INTOBJ_INT( libGAP_INTBASE*q[-3]+q[-4] );
            else if ( libGAP_TNUM_OBJ(quo) == libGAP_T_INTNEG
              && (libGAP_UInt)(libGAP_INTBASE*q[-3]+q[-4])
                     <= (1L<<libGAP_NR_SMALL_INT_BITS) )
                quo = libGAP_INTOBJ_INT( -(libGAP_Int)(libGAP_INTBASE*q[-3]+q[-4]) );
        }

    }

    /* divide a large integer by a large integer                           */
    else {

        /* a small divisor larger than one digit isn't handled above       */
        if ( libGAP_IS_INTOBJ(opR) ) {
            if ( 0 < libGAP_INT_INTOBJ(opR) ) {
                quo = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(quo)[0] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(quo)[1] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR)>>libGAP_NR_DIGIT_BITS);
                opR = quo;
            }
            else {
                quo = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(quo)[0] = (libGAP_TypDigit)(-libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(quo)[1] = (libGAP_TypDigit)((-libGAP_INT_INTOBJ(opR))>>libGAP_NR_DIGIT_BITS);
                opR = quo;
            }
        }

        /* trivial case first                                              */
        if ( libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) )
            return libGAP_INTOBJ_INT(0);

        /* copy the left operand into a new bag, this holds the remainder  */
        quo = libGAP_NewBag( libGAP_TNUM_OBJ(opL), (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        l = libGAP_ADDR_INT(opL);
        q = libGAP_ADDR_INT(quo);
        for ( k = libGAP_SIZE_INT(opL)-1; k >= 0; k-- )
            *q++ = *l++;
        opL = quo;

        /* get the size of the right operand, and get the leading 2 digits */
        rs = libGAP_SIZE_INT(opR);
        r  = libGAP_ADDR_INT(opR);
        while ( r[rs-1] == 0 )  rs--;
        for ( e = 0;
             ((libGAP_Int)r[rs-1]<<e) + (e ? r[rs-2]>>(libGAP_NR_DIGIT_BITS-e) : 0)
                               < libGAP_INTBASE/2 ; e++ ) ;

        r1 = ((libGAP_Int)r[rs-1]<<e) + (e ? r[rs-2]>>(libGAP_NR_DIGIT_BITS-e) : 0);
        r2 = ((libGAP_Int)r[rs-2]<<e) + ((e && rs>=3) ? r[rs-3]>>(libGAP_NR_DIGIT_BITS-e) : 0);

        /* allocate a bag for the quotient                                 */
        if ( libGAP_TNUM_OBJ(opL) == libGAP_TNUM_OBJ(opR) )
            quo = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL)-libGAP_SIZE_OBJ(opR) );
        else
            quo = libGAP_NewBag( libGAP_T_INTNEG, libGAP_SIZE_OBJ(opL)-libGAP_SIZE_OBJ(opR) );

        /* run through the digits in the quotient                          */
        for ( i = libGAP_SIZE_INT(opL)-libGAP_SIZE_INT(opR)-1; i >= 0; i-- ) {

            /* guess the factor                                            */
            l = libGAP_ADDR_INT(opL) + rs + i;
            l01 = ((libGAP_INTBASE*l[0]+l[-1])<<e)
              + (e ? l[-2]>>(libGAP_NR_DIGIT_BITS-e):0);

            if ( l01 == 0 )  continue;
            l2  = ((libGAP_Int)l[-2]<<e) + (( e && rs+i>=3) ? l[-3]>>(libGAP_NR_DIGIT_BITS-e) : 0);
            if ( ((libGAP_Int)l[0]<<e)+(e ? l[-1]>>(libGAP_NR_DIGIT_BITS-e):0) < r1 )
                     qi = l01 / r1;
            else     qi = libGAP_INTBASE - 1;
            while ( l01-(libGAP_Int)qi*r1 < (libGAP_Int)libGAP_INTBASE
                    && libGAP_INTBASE*(l01-(libGAP_UInt)qi*r1)+l2 < (libGAP_UInt)qi*r2 )
                qi--;

            /* l = l - qi * r;                                             */
            d = 0;
            l = libGAP_ADDR_INT(opL) + i;
            r = libGAP_ADDR_INT(opR);
            for ( k = 0; k < (libGAP_Int)rs; ++k, ++l, ++r ) {
                c = *l - (libGAP_Int)qi * *r - d;  *l = c;  d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
            }
            c = (libGAP_Int)*l - d; d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);

            /* if we have a borrow then add back                           */
            if ( d != 0 ) {
                d = 0;
                l = libGAP_ADDR_INT(opL) + i;
                r = libGAP_ADDR_INT(opR);
                for ( k = 0; k < (libGAP_Int)rs; ++k, ++l, ++r ) {
                    c = (libGAP_Int)*l + (libGAP_Int)*r + (libGAP_Int)d;
                    *l = (libGAP_TypDigit)c;
                    d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                }
                c = *l + d; d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                qi--;
            }

            /* store the digit in the quotient                             */
            libGAP_ADDR_INT(quo)[i] = qi;

        }

        /* remove the leading zeroes, note that there can't be more than 7 */
        q = libGAP_ADDR_INT(quo) + libGAP_SIZE_INT(quo);
        if ( libGAP_SIZE_INT(quo) > 4
          && q[-4] == 0 && q[-3] == 0 && q[-2] == 0 && q[-1] == 0 ) {
            libGAP_ResizeBag( quo, (libGAP_SIZE_INT(quo)-4)*sizeof(libGAP_TypDigit) );
        }

        /* reduce to small integer if possible                             */
        q = libGAP_ADDR_INT(quo) + libGAP_SIZE_INT(quo);
        if ( libGAP_SIZE_INT(quo) == 4 && q[-2] == 0 && q[-1] == 0 ) {
            if ( libGAP_TNUM_OBJ(quo) == libGAP_T_INTPOS
              && (libGAP_UInt)(libGAP_INTBASE*q[-3]+q[-4]) < (1L<<libGAP_NR_SMALL_INT_BITS) )
                quo = libGAP_INTOBJ_INT( libGAP_INTBASE*q[-3]+q[-4] );
            else if ( libGAP_TNUM_OBJ(quo) == libGAP_T_INTNEG
              && (libGAP_UInt)(libGAP_INTBASE*q[-3]+q[-4]) <= (1L<<libGAP_NR_SMALL_INT_BITS) )
                quo = libGAP_INTOBJ_INT( -(libGAP_Int)(libGAP_INTBASE*q[-3]+q[-4]) );
        }

    }

    /* return the result                                                   */
    return quo;
}


/****************************************************************************
**
*F  FuncQUO_INT(<self>,<opL>,<opR>) . . . . . . .  internal function 'QuoInt'
**
**  'FuncQUO_INT' implements the internal function 'QuoInt'.
**
**  'QuoInt( <i>, <k> )'
**
**  'Quo' returns the  integer part of the quotient  of its integer operands.
**  If <i>  and <k> are  positive 'Quo( <i>,  <k> )' is  the largest positive
**  integer <q>  such that '<q> * <k>  \<= <i>'.  If  <i> or  <k> or both are
**  negative we define 'Abs( Quo(<i>,<k>) ) = Quo( Abs(<i>), Abs(<k>) )'  and
**  'Sign( Quo(<i>,<k>) ) = Sign(<i>) * Sign(<k>)'.  Dividing by 0  causes an
**  error.  'Rem' (see "Rem") can be used to compute the remainder.
*/
libGAP_Obj libGAP_FuncQUO_INT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    /* check the arguments                                                 */
    while ( libGAP_TNUM_OBJ(opL) != libGAP_T_INT
         && libGAP_TNUM_OBJ(opL) != libGAP_T_INTPOS
         && libGAP_TNUM_OBJ(opL) != libGAP_T_INTNEG ) {
        opL = libGAP_ErrorReturnObj(
            "QuoInt: <left> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L,
            "you can replace <left> via 'return <left>;'" );
    }
    while ( libGAP_TNUM_OBJ(opR) != libGAP_T_INT
         && libGAP_TNUM_OBJ(opR) != libGAP_T_INTPOS
         && libGAP_TNUM_OBJ(opR) != libGAP_T_INTNEG ) {
        opR = libGAP_ErrorReturnObj(
            "QuoInt: <right> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L,
            "you can replace <right> via 'return <right>;'" );
    }

    /* return the quotient                                                 */
    return libGAP_QuoInt( opL, opR );
}


/****************************************************************************
**
*F  RemInt( <intL>, <intR> )  . . . . . . . . . . . remainder of two integers
**
**  'RemInt' returns the remainder of the quotient  of  the  integers  <intL>
**  and <intR>.  'RemInt' handles operands of type  'T_INT',  'T_INTPOS'  and
**  'T_INTNEG'.
**
**  Note that the remainder is different from the value returned by the 'mod'
**  operator which is always positive.
**
**  'RemInt' is called from 'FunRemInt'.
*/
libGAP_Obj             libGAP_RemInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;              /* loop count, value for small int */
    libGAP_Int                 k;              /* loop count, value for small int */
    libGAP_UInt                c;              /* product of two digits           */
    libGAP_TypDigit            d;              /* carry into the next digit       */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */
    libGAP_TypDigit            r1;             /* leading digit of the right oper */
    libGAP_TypDigit            r2;             /* next digit of the right operand */
    libGAP_UInt                rs;             /* size of the right operand       */
    libGAP_UInt                e;              /* we mult r by 2^e so r1 >= 32768 */
    libGAP_Obj                 rem;            /* handle of the remainder bag     */
    libGAP_TypDigit *          m;              /* pointer into the remainder      */
    libGAP_UInt                m01;            /* leading two digits of the rem.  */
    libGAP_TypDigit            m2;             /* next digit of the remainder     */
    libGAP_TypDigit            qi;             /* guessed digit of the quotient   */

    /* compute the remainder of two small integers                         */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {

        /* pathological case first                                         */
        if ( opR == libGAP_INTOBJ_INT(0) ) {
            opR = libGAP_ErrorReturnObj(
                "Integer operations: <divisor> must be nonzero",
                0L, 0L,
                "you can replace the integer <divisor> via 'return <divisor>;'" );
            return libGAP_QUO( opL, opR );
        }

        /* get the integer values                                          */
        i = libGAP_INT_INTOBJ(opL);
        k = libGAP_INT_INTOBJ(opR);

        /* compute the remainder, make sure we divide only positive numbers*/
        if (      0 <= i && 0 <= k )  i =    (  i %  k );
        else if ( 0 <= i && k <  0 )  i =    (  i % -k );
        else if ( i < 0  && 0 <= k )  i =  - ( -i %  k );
        else if ( i < 0  && k <  0 )  i =  - ( -i % -k );
        rem = libGAP_INTOBJ_INT( i );

    }

    /* compute the remainder of a small integer by a large integer         */
    else if ( libGAP_IS_INTOBJ(opL) ) {

        /* the small int -(1<<28) rem the large int (1<<28) is 0           */
        if ( opL == libGAP_INTOBJ_INT(-(libGAP_Int)(1L<<libGAP_NR_SMALL_INT_BITS))
          && libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS && libGAP_SIZE_INT(opR) == 4
          && libGAP_ADDR_INT(opR)[3] == 0
          && libGAP_ADDR_INT(opR)[2] == 0
          && libGAP_ADDR_INT(opR)[1] == 1L << (libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS)
          && libGAP_ADDR_INT(opR)[0] == 0 )
            rem = libGAP_INTOBJ_INT(0);

        /* in all other cases the remainder is equal the left operand      */
        else
            rem = opL;

    }

    /* compute the remainder of a large integer by a small integer         */
    else if ( libGAP_IS_INTOBJ(opR)
           && libGAP_INT_INTOBJ(opR) < libGAP_INTBASE
           && -(libGAP_Int)libGAP_INTBASE <= libGAP_INT_INTOBJ(opR) ) {

        /* pathological case first                                         */
        if ( opR == libGAP_INTOBJ_INT(0) ) {
            opR = libGAP_ErrorReturnObj(
                "Integer operations: <divisor> must be nonzero",
                0L, 0L,
                "you can replace the integer <divisor> via 'return <divisor>;'" );
            return libGAP_QUO( opL, opR );
        }

        /* get the integer value, make positive                            */
        i = libGAP_INT_INTOBJ(opR);  if ( i < 0 )  i = -i;

        /* maybe its trivial                                               */
        if ( libGAP_INTBASE % i == 0 ) {
            c = libGAP_ADDR_INT(opL)[0] % i;
        }

        /* otherwise run through the left operand and divide digitwise     */
        else {
            l = libGAP_ADDR_INT(opL) + libGAP_SIZE_INT(opL) - 1;
            c = 0;
            for ( ; l >= libGAP_ADDR_INT(opL); l-- ) {
                c  = (c<<libGAP_NR_DIGIT_BITS) + (libGAP_Int)*l;
                c  = c % i;
            }
        }

        /* now c is the result, it has the same sign as the left operand   */
        if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
            rem = libGAP_INTOBJ_INT(  c );
        else
            rem = libGAP_INTOBJ_INT( -(libGAP_Int)c );

    }

    /* compute the remainder of a large integer modulo a large integer     */
    else {

        /* a small divisor larger than one digit isn't handled above       */
        if ( libGAP_IS_INTOBJ(opR) ) {
            if ( 0 < libGAP_INT_INTOBJ(opR) ) {
                rem = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(rem)[0] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(rem)[1] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR)>>libGAP_NR_DIGIT_BITS);
                opR = rem;
            }
            else {
                rem = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(rem)[0] = (libGAP_TypDigit)(-libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(rem)[1] = (libGAP_TypDigit)((-libGAP_INT_INTOBJ(opR))>>libGAP_NR_DIGIT_BITS);
                opR = rem;
            }
        }

        /* trivial case first                                              */
        if ( libGAP_SIZE_INT(opL) < libGAP_SIZE_INT(opR) )
            return opL;

        /* copy the left operand into a new bag, this holds the remainder  */
        rem = libGAP_NewBag( libGAP_TNUM_OBJ(opL), (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        l = libGAP_ADDR_INT(opL);
        m = libGAP_ADDR_INT(rem);
        for ( k = libGAP_SIZE_INT(opL)-1; k >= 0; k-- )
            *m++ = *l++;

        /* get the size of the right operand, and get the leading 2 digits */
        rs = libGAP_SIZE_INT(opR);
        r  = libGAP_ADDR_INT(opR);
        while ( r[rs-1] == 0 )  rs--;
        for ( e = 0;
             ((libGAP_Int)r[rs-1]<<e) + (e ? r[rs-2]>>(libGAP_NR_DIGIT_BITS-e): 0)
                               < libGAP_INTBASE/2; e++ ) ;

        r1 = ((libGAP_Int)r[rs-1]<<e) + (e ? r[rs-2]>>(libGAP_NR_DIGIT_BITS-e):0);
        r2 = ((libGAP_Int)r[rs-2]<<e) + ((e && rs>=3) ? r[rs-3]>>(libGAP_NR_DIGIT_BITS-e) : 0);

        /* run through the digits in the quotient                          */
        for ( i = libGAP_SIZE_INT(rem)-libGAP_SIZE_INT(opR)-1; i >= 0; i-- ) {

            /* guess the factor                                            */
            m = libGAP_ADDR_INT(rem) + rs + i;
            m01 = ((libGAP_INTBASE*m[0]+m[-1])<<e) + (e ? m[-2]>>(libGAP_NR_DIGIT_BITS-e):0);
            if ( m01 == 0 )  continue;
            m2  = ((libGAP_Int)m[-2]<<e) + ((e && rs+i>=3) ? m[-3]>>(libGAP_NR_DIGIT_BITS-e) : 0);
            if ( ((libGAP_Int)m[0]<<e)+(e ? m[-1]>>(libGAP_NR_DIGIT_BITS-e): 0) < r1 )
                    qi = m01 / r1;
            else    qi = libGAP_INTBASE - 1;
            while ( m01-(libGAP_Int)qi*r1 < (libGAP_Int)libGAP_INTBASE
                   && libGAP_INTBASE*(m01-(libGAP_Int)qi*r1)+m2 < (libGAP_Int)qi*r2 )
                qi--;

            /* m = m - qi * r;                                             */
            d = 0;
            m = libGAP_ADDR_INT(rem) + i;
            r = libGAP_ADDR_INT(opR);
            for ( k = 0; k < (libGAP_Int)rs; ++k, ++m, ++r ) {
                c = (libGAP_Int)*m - (libGAP_Int)qi * *r - (libGAP_Int)d;
                *m = (libGAP_TypDigit)c;
                d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
            }
            c = (libGAP_Int)*m - d;  *m = (libGAP_TypDigit)c;  d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);

            /* if we have a borrow then add back                           */
            if ( d != 0 ) {
                d = 0;
                m = libGAP_ADDR_INT(rem) + i;
                r = libGAP_ADDR_INT(opR);
                for ( k = 0; k < (libGAP_Int)rs; ++k, ++m, ++r ) {
                    c = (libGAP_Int)*m + (libGAP_Int)*r + (libGAP_Int)d;
                    *m = (libGAP_TypDigit)c;
                    d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                }
                c = (libGAP_Int)*m + d;  *m = (libGAP_TypDigit)c;  d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                qi--;
            }

        }

        /* remove the leading zeroes                                       */
        m = libGAP_ADDR_INT(rem) + libGAP_SIZE_INT(rem);
        if ( (m[-4] == 0 && m[-3] == 0 && m[-2] == 0 && m[-1] == 0)
          || (libGAP_SIZE_INT(rem) == 4 && m[-2] == 0 && m[-1] == 0) ) {

            /* find the number of significant digits                       */
            m = libGAP_ADDR_INT(rem);
            for ( k = libGAP_SIZE_INT(rem); k != 0; k-- ) {
                if ( m[k-1] != 0 )
                    break;
            }

            /* reduce to small integer if possible, otherwise shrink bag   */
            if ( k <= 2 && libGAP_TNUM_OBJ(rem) == libGAP_T_INTPOS
              && (libGAP_UInt)(libGAP_INTBASE*m[1]+m[0]) < (1L<<libGAP_NR_SMALL_INT_BITS) )
                rem = libGAP_INTOBJ_INT( libGAP_INTBASE*m[1]+m[0] );
            else if ( k <= 2 && libGAP_TNUM_OBJ(rem) == libGAP_T_INTNEG
              && (libGAP_UInt)(libGAP_INTBASE*m[1]+m[0]) <= (1L<<libGAP_NR_SMALL_INT_BITS) )
                rem = libGAP_INTOBJ_INT( -(libGAP_Int)(libGAP_INTBASE*m[1]+m[0]) );
            else
                libGAP_ResizeBag( rem, (((k + 3) / 4) * 4) * sizeof(libGAP_TypDigit) );
        }

    }

    /* return the result                                                   */
    return rem;
}


/****************************************************************************
**
*F  FuncREM_INT(<self>,<opL>,<opR>)  . . . . . . .  internal function 'RemInt'
**
**  'FuncREM_INT' implements the internal function 'RemInt'.
**
**  'RemInt( <i>, <k> )'
**
**  'Rem' returns the remainder of its two integer operands,  i.e., if <k> is
**  not equal to zero 'Rem( <i>, <k> ) = <i> - <k> *  Quo( <i>, <k> )'.  Note
**  that the rules given  for 'Quo' (see "Quo") imply  that 'Rem( <i>, <k> )'
**  has the same sign as <i> and its absolute value is strictly less than the
**  absolute value of <k>.  Dividing by 0 causes an error.
*/
libGAP_Obj             libGAP_FuncREM_INT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    /* check the arguments                                                 */
    while ( libGAP_TNUM_OBJ(opL) != libGAP_T_INT
         && libGAP_TNUM_OBJ(opL) != libGAP_T_INTPOS
         && libGAP_TNUM_OBJ(opL) != libGAP_T_INTNEG ) {
        opL = libGAP_ErrorReturnObj(
            "RemInt: <left> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L,
            "you can replace <left> via 'return <left>;'" );
    }
    while ( libGAP_TNUM_OBJ(opR) != libGAP_T_INT
         && libGAP_TNUM_OBJ(opR) != libGAP_T_INTPOS
         && libGAP_TNUM_OBJ(opR) != libGAP_T_INTNEG ) {
        opR = libGAP_ErrorReturnObj(
            "RemInt: <right> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L,
            "you can replace <right> via 'return <right>;'" );
    }

    /* return the remainder                                                */
    return libGAP_RemInt( opL, opR );
}


/****************************************************************************
**
*F  GcdInt( <opL>, <opR> )  . . . . . . . . . . . . . . . gcd of two integers
**
**  'GcdInt' returns the gcd of the two integers <opL> and <opR>.
**
**  It is called from 'FunGcdInt' and the rational package.
*/
libGAP_Obj             libGAP_GcdInt (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 i;              /* loop count, value for small int */
    libGAP_Int                 k;              /* loop count, value for small int */
    libGAP_UInt                c;              /* product of two digits           */
    libGAP_TypDigit            d;              /* carry into the next digit       */
    libGAP_TypDigit *          r;              /* pointer into the right operand  */
    libGAP_TypDigit            r1;             /* leading digit of the right oper */
    libGAP_TypDigit            r2;             /* next digit of the right operand */
    libGAP_UInt                rs;             /* size of the right operand       */
    libGAP_UInt                e;              /* we mult r by 2^e so r1 >= 32768 */
    libGAP_TypDigit *          l;              /* pointer into the left operand   */
    libGAP_UInt                l01;            /* leading two digits of the rem.  */
    libGAP_TypDigit            l2;             /* next digit of the remainder     */
    libGAP_UInt                ls;             /* size of the left operand        */
    libGAP_TypDigit            qi;             /* guessed digit of the quotient   */
    libGAP_Obj                 gcd;            /* handle of the result            */

    /* compute the gcd of two small integers                               */
    if ( libGAP_ARE_INTOBJS( opL, opR ) ) {

        /* get the integer values, make them positive                      */
        i = libGAP_INT_INTOBJ(opL);  if ( i < 0 )  i = -i;
        k = libGAP_INT_INTOBJ(opR);  if ( k < 0 )  k = -k;

        /* compute the gcd using Euclids algorithm                         */
        while ( k != 0 ) {
            c = k;
            k = i % k;
            i = c;
        }

        /* now i is the result                                             */
        if ( i == (1L<<libGAP_NR_SMALL_INT_BITS) ) {
            gcd = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(gcd)[0] = (libGAP_TypDigit)i;
            libGAP_ADDR_INT(gcd)[1] = (libGAP_TypDigit)(i>>libGAP_NR_DIGIT_BITS);
        }
        else {
            gcd = libGAP_INTOBJ_INT( i );
        }

    }

    /* compute the gcd of a small and a large integer                      */
    else if ( (libGAP_IS_INTOBJ(opL)
               && libGAP_INT_INTOBJ(opL) < libGAP_INTBASE && -(libGAP_Int)libGAP_INTBASE <= libGAP_INT_INTOBJ(opL))
           || (libGAP_IS_INTOBJ(opR)
               && libGAP_INT_INTOBJ(opR) < libGAP_INTBASE && -(libGAP_Int)libGAP_INTBASE <= libGAP_INT_INTOBJ(opR)) ) {

        /* make the right operand the small one                            */
        if ( libGAP_IS_INTOBJ(opL) ) {
            gcd = opL;  opL = opR;  opR = gcd;
        }

        /* maybe it's trivial                                              */
        if ( opR == libGAP_INTOBJ_INT(0) ) {
            if( libGAP_TNUM_OBJ( opL ) == libGAP_T_INTNEG ) {
                /* If opL is negative, change the sign.  We do this by    */
                /* copying opL into a bag of type T_INTPOS.  Note that    */
                /* opL is a large negative number, so it cannot be the    */
                /* the negative of 1 << NR_SMALL_INT_BITS.                */
                gcd = libGAP_NewBag( libGAP_T_INTPOS, libGAP_SIZE_OBJ(opL) );
                l = libGAP_ADDR_INT(opL); r = libGAP_ADDR_INT(gcd);
                for ( k = libGAP_SIZE_INT(opL); k != 0; k-- ) *r++ = *l++;

                return gcd;
            }
            else return opL;
        }

        /* get the right operand value, make it positive                   */
        i = libGAP_INT_INTOBJ(opR);  if ( i < 0 )  i = -i;

        /* do one remainder operation                                      */
        l = libGAP_ADDR_INT(opL) + libGAP_SIZE_INT(opL) - 1;
        c = 0;
        for ( ; l >= libGAP_ADDR_INT(opL); l-- ) {
            c  = (c<<libGAP_NR_DIGIT_BITS) + (libGAP_Int)*l;
            c  = c % i;
        }
        k = c;

        /* compute the gcd using Euclids algorithm                         */
        while ( k != 0 ) {
            c = k;
            k = i % k;
            i = c;
        }

        /* now i is the result                                             */
        if ( i == (1L<<libGAP_NR_SMALL_INT_BITS) ) {
            gcd = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
            libGAP_ADDR_INT(gcd)[0] = 0;
            libGAP_ADDR_INT(gcd)[1] = 1L<<(libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS);
        }
        else {
            gcd = libGAP_INTOBJ_INT( i );
        }

    }

    /* compute the gcd of two large integers                               */
    else {

        /* a small divisor larger than one digit isn't handled above       */
        if ( libGAP_IS_INTOBJ(opL) ) {
            if ( 0 < libGAP_INT_INTOBJ(opL) ) {
                gcd = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(gcd)[0] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opL));
                libGAP_ADDR_INT(gcd)[1] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opL)>>libGAP_NR_DIGIT_BITS);
                opL = gcd;
            }
            else {
                gcd = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(gcd)[0] = (libGAP_TypDigit)(-libGAP_INT_INTOBJ(opL));
                libGAP_ADDR_INT(gcd)[1] = (libGAP_TypDigit)((-libGAP_INT_INTOBJ(opL))>>libGAP_NR_DIGIT_BITS);
                opL = gcd;
            }
        }

        /* a small dividend larger than one digit isn't handled above       */
        if ( libGAP_IS_INTOBJ(opR) ) {
            if ( 0 < libGAP_INT_INTOBJ(opR) ) {
                gcd = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(gcd)[0] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(gcd)[1] = (libGAP_TypDigit)(libGAP_INT_INTOBJ(opR)>>libGAP_NR_DIGIT_BITS);
                opR = gcd;
            }
            else {
                gcd = libGAP_NewBag( libGAP_T_INTNEG, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(gcd)[0] = (libGAP_TypDigit)(-libGAP_INT_INTOBJ(opR));
                libGAP_ADDR_INT(gcd)[1] = (libGAP_TypDigit)((-libGAP_INT_INTOBJ(opR))>>libGAP_NR_DIGIT_BITS);
                opR = gcd;
            }
        }

        /* copy the left operand into a new bag                            */
        gcd = libGAP_NewBag( libGAP_T_INTPOS, (libGAP_SIZE_INT(opL)+4)*sizeof(libGAP_TypDigit) );
        l = libGAP_ADDR_INT(opL);
        r = libGAP_ADDR_INT(gcd);
        for ( k = libGAP_SIZE_INT(opL)-1; k >= 0; k-- )
            *r++ = *l++;
        opL = gcd;

        /* get the size of the left operand                                */
        ls = libGAP_SIZE_INT(opL);
        l  = libGAP_ADDR_INT(opL);
        while ( ls >= 1 && l[ls-1] == 0 )  ls--;

        /* copy the right operand into a new bag                           */
        gcd = libGAP_NewBag( libGAP_T_INTPOS, (libGAP_SIZE_INT(opR)+4)*sizeof(libGAP_TypDigit) );
        r = libGAP_ADDR_INT(opR);
        l = libGAP_ADDR_INT(gcd);
        for ( k = libGAP_SIZE_INT(opR)-1; k >= 0; k-- )
            *l++ = *r++;
        opR = gcd;

        /* get the size of the right operand                               */
        rs = libGAP_SIZE_INT(opR);
        r  = libGAP_ADDR_INT(opR);
        while ( rs >= 1 && r[rs-1] == 0 )  rs--;

        /* repeat while the right operand is large                         */
        while ( rs >= 2 ) {

            /* get the leading two digits                                  */
            for ( e = 0;
                 ((libGAP_Int)r[rs-1]<<e) + (e ? r[rs-2]>>(libGAP_NR_DIGIT_BITS-e) : 0)
                                   < libGAP_INTBASE/2; e++ ) ;
            r1 = ((libGAP_Int)r[rs-1]<<e) + (e ? r[rs-2]>>(libGAP_NR_DIGIT_BITS-e): 0);
            r2 = ((libGAP_Int)r[rs-2]<<e) + ((e && rs>=3) ? r[rs-3]>>(libGAP_NR_DIGIT_BITS-e) : 0);

            /* run through the digits in the quotient                      */
            for ( i = ls - rs; i >= 0; i-- ) {

                /* guess the factor                                        */
                l = libGAP_ADDR_INT(opL) + rs + i;
                l01 = ((libGAP_INTBASE*l[0]+l[-1])<<e) + (e ? l[-2]>>(libGAP_NR_DIGIT_BITS-e):0);
                if ( l01 == 0 )  continue;
                l2  = ((libGAP_Int)l[-2]<<e) + ((e && rs+i>=3) ? l[-3]>>(libGAP_NR_DIGIT_BITS-e):0);
                if ( ((libGAP_Int)l[0]<<e)+(e ? l[-1]>>(libGAP_NR_DIGIT_BITS-e):0) < r1 )
                        qi = l01 / r1;
                else    qi = libGAP_INTBASE - 1;
                while ( l01-(libGAP_Int)qi*r1 < (libGAP_Int)libGAP_INTBASE
                        && (libGAP_INTBASE*(l01-(libGAP_Int)qi*r1)+l2) < ((libGAP_Int)qi*r2) )
                    qi--;

                /* l = l - qi * r;                                         */
                d = 0;
                l = libGAP_ADDR_INT(opL) + i;
                r = libGAP_ADDR_INT(opR);
                for ( k = 0; k < (libGAP_Int)rs; ++k, ++l, ++r ) {
                    c = (libGAP_Int)*l - (libGAP_Int)qi * *r - (libGAP_Int)d;
                    *l = (libGAP_TypDigit)c;
                    d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                }
                c = (libGAP_Int)*l - d;  *l = (libGAP_TypDigit)c;  d = -(libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);

                /* if we have a borrow then add back                       */
                if ( d != 0 ) {
                    d = 0;
                    l = libGAP_ADDR_INT(opL) + i;
                    r = libGAP_ADDR_INT(opR);
                    for ( k = 0; k < (libGAP_Int)rs; ++k, ++l, ++r ) {
                        c = (libGAP_Int)*l + (libGAP_Int)*r + (libGAP_Int)d;
                        *l = (libGAP_TypDigit)c;
                        d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                    }
                    c = *l + d;  *l = (libGAP_TypDigit)c;  d = (libGAP_TypDigit)(c>>libGAP_NR_DIGIT_BITS);
                    qi--;
                }
            }

            /* exchange the two operands                                   */
            gcd = opL;  opL = opR;  opR = gcd;
            ls = rs;

            /* get the size of the right operand                           */
            rs = libGAP_SIZE_INT(opR);
            r  = libGAP_ADDR_INT(opR);
            while ( rs >= 1 && r[rs-1] == 0 )  rs--;

        }

        /* if the right operand is zero now, the left is the gcd           */
        if ( rs == 0 ) {

            /* remove the leading zeroes                                   */
            l = libGAP_ADDR_INT(opL) + libGAP_SIZE_INT(opL);
            if ( (l[-4] == 0 && l[-3] == 0 && l[-2] == 0 && l[-1] == 0)
              || (libGAP_SIZE_INT(opL) == 4 && l[-2] == 0 && l[-1] == 0) ) {

                /* find the number of significant digits                   */
                l = libGAP_ADDR_INT(opL);
                for ( k = libGAP_SIZE_INT(opL); k != 0; k-- ) {
                    if ( l[k-1] != 0 )
                        break;
                }

                /* reduce to small integer if possible, otherwise shrink b */
                if ( k <= 2 && libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS
                  && (libGAP_UInt)(libGAP_INTBASE*l[1]+l[0]) < (1L<<libGAP_NR_SMALL_INT_BITS) )
                    opL = libGAP_INTOBJ_INT( libGAP_INTBASE*l[1]+l[0] );
                else if ( k <= 2 && libGAP_TNUM_OBJ(opL) == libGAP_T_INTNEG
                  && (libGAP_UInt)(libGAP_INTBASE*l[1]+l[0]) <= (1L<<libGAP_NR_SMALL_INT_BITS) )
                    opL = libGAP_INTOBJ_INT( -(libGAP_Int)(libGAP_INTBASE*l[1]+l[0]) );
                else
                    libGAP_ResizeBag( opL, (((k + 3) / 4) * 4) * sizeof(libGAP_TypDigit) );
            }

            gcd = opL;

        }

        /* otherwise handle one large and one small integer as above       */
        else {

            /* get the right operand value, make it positive               */
            i = r[0];

            /* do one remainder operation                                  */
            l = libGAP_ADDR_INT(opL) + libGAP_SIZE_INT(opL) - 1;
            c = 0;
            for ( ; l >= libGAP_ADDR_INT(opL); l-- ) {
                c  = (c<<libGAP_NR_DIGIT_BITS) + (libGAP_Int)*l;
                c  = c % i;
            }
            k = c;

            /* compute the gcd using Euclids algorithm                     */
            while ( k != 0 ) {
                c = k;
                k = i % k;
                i = c;
            }

            /* now i is the result                                         */
            if ( i == (1L<<libGAP_NR_SMALL_INT_BITS) ) {
                gcd = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
                libGAP_ADDR_INT(gcd)[0] = 0;
                libGAP_ADDR_INT(gcd)[1] = 1L<<(libGAP_NR_SMALL_INT_BITS-libGAP_NR_DIGIT_BITS);
            }
            else {
                gcd = libGAP_INTOBJ_INT( i );
            }

        }

    }

    /* return the result                                                   */
    return gcd;
}


/****************************************************************************
**
*F  FuncGCD_INT(<self>,<opL>,<opR>)  . . . . . . .  internal function 'GcdInt'
**
**  'FuncGCD_INT' implements the internal function 'GcdInt'.
**
**  'GcdInt( <i>, <k> )'
**
**  'Gcd'  returns the greatest common divisor   of the two  integers <m> and
**  <n>, i.e.,  the  greatest integer that  divides  both <m>  and  <n>.  The
**  greatest common divisor is never negative, even if the arguments are.  We
**  define $gcd( m, 0 ) = gcd( 0, m ) = abs( m )$ and $gcd( 0, 0 ) = 0$.
*/
libGAP_Obj             libGAP_FuncGCD_INT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    /* check the arguments                                                 */
    while ( libGAP_TNUM_OBJ(opL) != libGAP_T_INT
         && libGAP_TNUM_OBJ(opL) != libGAP_T_INTPOS
         && libGAP_TNUM_OBJ(opL) != libGAP_T_INTNEG ) {
        opL = libGAP_ErrorReturnObj(
            "GcdInt: <left> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opL), 0L,
            "you can replace <left> via 'return <left>;'" );
    }
    while ( libGAP_TNUM_OBJ(opR) != libGAP_T_INT
         && libGAP_TNUM_OBJ(opR) != libGAP_T_INTPOS
         && libGAP_TNUM_OBJ(opR) != libGAP_T_INTNEG ) {
        opR = libGAP_ErrorReturnObj(
            "GcdInt: <right> must be an integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(opR), 0L,
            "you can replace <right> via 'return <right>;'" );
    }

    /* return the gcd                                                      */
    return libGAP_GcdInt( opL, opR );
}






/****************************************************************************
**
*F  SaveInt( <int> )
**
**  Since the type is saved, we don't need to worry about sign
*/

void libGAP_SaveInt( libGAP_Obj bigint)
{
  libGAP_TypDigit *ptr;
  libGAP_UInt i;
  ptr = (libGAP_TypDigit *)libGAP_ADDR_OBJ(bigint);
  for (i = 0; i < libGAP_SIZE_INT(bigint); i++)
#ifdef libGAP_SYS_IS_64_BIT
    libGAP_SaveUInt4(*ptr++);
#else
    libGAP_SaveUInt2(*ptr++);
#endif
  return;
}

/****************************************************************************
**
*F  LoadInt( <int> )
**
**  Since the type is loaded, we don't need to worry about sign
*/

void libGAP_LoadInt( libGAP_Obj bigint)
{
  libGAP_TypDigit *ptr;
  libGAP_UInt i;
  ptr = (libGAP_TypDigit *)libGAP_ADDR_OBJ(bigint);
  for (i = 0; i < libGAP_SIZE_INT(bigint); i++)
#ifdef libGAP_SYS_IS_64_BIT
    *ptr++ = libGAP_LoadUInt4();
#else
    *ptr++ = libGAP_LoadUInt2();
#endif
  return;
}


/****************************************************************************
**
** * * * * * * * "Mersenne twister" random numbers  * * * * * * * * * * * * *
**
**  Part of this code for fast generation of 32 bit pseudo random numbers with
**  a period of length 2^19937-1 and a 623-dimensional equidistribution is
**  taken from:
**          http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html
**  (Also look in Wikipedia for "Mersenne twister".)
**  We use the file mt19937ar.c, version 2002/1/26.
*/



/****************************************************************************
**
*F  RandomIntegerMT( <mtstr>, <nrbits> )
**
**  Returns an integer with at most <nrbits> bits in uniform distribution.
**  <nrbits> must be a small integer. <mtstr> is a string as returned by
**  InitRandomMT.
**
**  Implementation details are a bit tricky to obtain the same random
**  integers on 32 bit and 64 bit machines (which have different long
**  integer digit lengths and different ranges of small integers).
**
*/
/* for comparison in case result is small int */
libGAP_Obj libGAP_SMALLEST_INTPOS = NULL;

libGAP_Obj libGAP_FuncRandomIntegerMT(libGAP_Obj self, libGAP_Obj mtstr, libGAP_Obj nrbits)
{
  libGAP_Obj res;
  libGAP_Int i, n, q, r, qoff, len, nlen;
  libGAP_UInt4 *mt, rand;
  libGAP_TypDigit *pt;
  while (! libGAP_IsStringConv(mtstr)) {
     mtstr = libGAP_ErrorReturnObj(
         "<mtstr> must be a string, not a %s)",
         (libGAP_Int)libGAP_TNAM_OBJ(mtstr), 0L,
         "you can replace <mtstr> via 'return <mtstr>;'" );
  }
  while ((! libGAP_IsStringConv(mtstr)) || libGAP_GET_LEN_STRING(mtstr) < 2500) {
     mtstr = libGAP_ErrorReturnObj(
         "<mtstr> must be a string with at least 2500 characters, ",
         0L, 0L,
         "you can replace <mtstr> via 'return <mtstr>;'" );
  }
  while ((! libGAP_IS_INTOBJ(nrbits)) || libGAP_INT_INTOBJ(nrbits) < 0) {
     nrbits = libGAP_ErrorReturnObj(
         "<nrbits> must be a small non-negative integer, not a %s)",
         (libGAP_Int)libGAP_TNAM_OBJ(nrbits), 0L,
         "you can replace <mtstr> via 'return <mtstr>;'" );
  }
  n = libGAP_INT_INTOBJ(nrbits);

  /* small int case */
  if (n <= libGAP_NR_SMALL_INT_BITS) {
     mt = (libGAP_UInt4*) libGAP_CHARS_STRING(mtstr);
#ifdef libGAP_SYS_IS_64_BIT
     if (n <= 32) {
       res = libGAP_INTOBJ_INT((libGAP_Int)(libGAP_nextrandMT_int32(mt) & ((libGAP_UInt4) -1L >> (32-n))));
     }
     else {
       unsigned long  rd;
       rd = libGAP_nextrandMT_int32(mt);
       rd += (unsigned long) ((libGAP_UInt4) libGAP_nextrandMT_int32(mt) &
                              ((libGAP_UInt4) -1L >> (64-n))) << 32;
       res = libGAP_INTOBJ_INT((libGAP_Int)rd);
     }
#else
     res = libGAP_INTOBJ_INT((libGAP_Int)(libGAP_nextrandMT_int32(mt) & ((libGAP_UInt4) -1L >> (32-n))));
#endif
  }
  else {
     /* number of Digits */
     q = n / libGAP_NR_DIGIT_BITS;
     r = n - q*libGAP_NR_DIGIT_BITS;
     qoff = q + (r==0 ? 0:1);
     len = qoff;
     len = 4*((len+3) / 4);
     res = libGAP_NewBag( libGAP_T_INTPOS, len*sizeof(libGAP_TypDigit) );
     pt = libGAP_ADDR_INT(res);
     mt = (libGAP_UInt4*) libGAP_CHARS_STRING(mtstr);
#ifdef libGAP_SYS_IS_64_BIT
     for (i = 0; i < qoff; i++, pt++) {
       rand = (libGAP_TypDigit) libGAP_nextrandMT_int32(mt);
       *pt = rand;
     }
#else
     for (i = 0; i < qoff/2; i++, pt++) {
       rand = libGAP_nextrandMT_int32(mt);
       *pt = (libGAP_TypDigit) (rand & 0xFFFFL);
       pt++;
       *pt = (libGAP_TypDigit) ((rand & 0xFFFF0000L) >> 16);
     }
     if (2*i != qoff) {
       rand = libGAP_nextrandMT_int32(mt);
       *pt = (libGAP_TypDigit) (rand & 0xFFFFL);
     }
#endif
     if (r != 0) {
       libGAP_ADDR_INT(res)[qoff-1] = libGAP_ADDR_INT(res)[qoff-1] & ((libGAP_TypDigit)(-1)
                                                      >> (libGAP_NR_DIGIT_BITS-r));
     }
     /* shrink bag if necessary */
     for (nlen = len, i=0; nlen >0 && libGAP_ADDR_INT(res)[nlen-1] == 0L;
                           nlen--, i++);
     if (i/4 != 0) {
        libGAP_ResizeBag(res, 4*((nlen+3) / 4)*sizeof(libGAP_TypDigit));
     }
     /* convert result if small int */
     if (libGAP_LtInt(res, libGAP_SMALLEST_INTPOS)) {
       res = libGAP_INTOBJ_INT((libGAP_Int)(libGAP_ADDR_INT(res)[0] +
                              ((libGAP_UInt)libGAP_ADDR_INT(res)[1] << libGAP_NR_DIGIT_BITS)));
     }
  }

  return res;
}



/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_INT", "obj", &libGAP_IsIntFilt,
      libGAP_FuncIS_INT, "src/integer.c:IS_INT" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "QUO_INT", 2, "int1, int2",
      libGAP_FuncQUO_INT, "src/integer.c:QUO_INT" },

    { "REM_INT", 2, "int1, int2",
      libGAP_FuncREM_INT, "src/integer.c:REM_INT" },

    { "GCD_INT", 2, "int1, int2",
      libGAP_FuncGCD_INT, "src/integer.c:GCD_INT" },

    { "PROD_INT_OBJ", 2, "int, obj",
      libGAP_FuncPROD_INT_OBJ, "src/integer.c:PROD_INT_OBJ" },

    { "POW_OBJ_INT", 2, "obj, int",
      libGAP_FuncPOW_OBJ_INT, "src/integer.c:POW_OBJ_INT" },

    { "HexStringInt", 1, "integer",
      libGAP_FuncHexStringInt, "src/integer.c:HexStringInt" },

    { "IntHexString", 1, "string",
      libGAP_FuncIntHexString, "src/integer.c:IntHexString" },

    { "Log2Int", 1, "int",
      libGAP_FuncLog2Int, "src/integer.c:Log2Int" },

    { "STRING_INT", 1, "int",
      libGAP_FuncSTRING_INT, "src/integer.c:STRING_INT" },


    { "RandomIntegerMT", 2, "mtstr, nrbits",
      libGAP_FuncRandomIntegerMT, "src/integer.c:RandomIntegerMT" },


    { 0 }

};


/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                t1,  t2;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* install the marking functions                                       */
    libGAP_InfoBags[         libGAP_T_INT    ].name = "integer";
#ifdef libGAP_SYS_IS_64_BIT
    libGAP_InfoBags[         libGAP_T_INTPOS ].name = "integer (>= 2^60)";
    libGAP_InfoBags[         libGAP_T_INTNEG ].name = "integer (< -2^60)";
#else
    libGAP_InfoBags[         libGAP_T_INTPOS ].name = "integer (>= 2^28)";
    libGAP_InfoBags[         libGAP_T_INTNEG ].name = "integer (< -2^28)";
#endif
    libGAP_InitMarkFuncBags( libGAP_T_INTPOS, libGAP_MarkNoSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_INTNEG, libGAP_MarkNoSubBags );

    /* Install the saving methods */
    libGAP_SaveObjFuncs [ libGAP_T_INTPOS ] = libGAP_SaveInt;
    libGAP_SaveObjFuncs [ libGAP_T_INTNEG ] = libGAP_SaveInt;
    libGAP_LoadObjFuncs [ libGAP_T_INTPOS ] = libGAP_LoadInt;
    libGAP_LoadObjFuncs [ libGAP_T_INTNEG ] = libGAP_LoadInt;

    /* install the printing function                                       */
    libGAP_PrintObjFuncs[ libGAP_T_INT    ] = libGAP_PrintInt;
    libGAP_PrintObjFuncs[ libGAP_T_INTPOS ] = libGAP_PrintInt;
    libGAP_PrintObjFuncs[ libGAP_T_INTNEG ] = libGAP_PrintInt;

    /* install the comparison methods                                      */
    for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
        for ( t2 = libGAP_T_INT; t2 <= libGAP_T_INTNEG; t2++ ) {
            libGAP_EqFuncs  [ t1 ][ t2 ] = libGAP_EqInt;
            libGAP_LtFuncs  [ t1 ][ t2 ] = libGAP_LtInt;
        }
    }

    /* install the unary arithmetic methods                                */
    for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
        libGAP_ZeroFuncs[ t1 ] = libGAP_ZeroInt;
        libGAP_ZeroMutFuncs[ t1 ] = libGAP_ZeroInt;
        libGAP_AInvFuncs[ t1 ] = libGAP_AInvInt;
        libGAP_AInvMutFuncs[ t1 ] = libGAP_AInvInt;
        libGAP_OneFuncs [ t1 ] = libGAP_OneInt;
        libGAP_OneMutFuncs [ t1 ] = libGAP_OneInt;
    }

    /* install the default product and power methods                       */
    for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
        for ( t2 = libGAP_FIRST_CONSTANT_TNUM;  t2 <= libGAP_LAST_CONSTANT_TNUM;  t2++ ) {
            libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdIntObj;
            libGAP_PowFuncs [ t2 ][ t1 ] = libGAP_PowObjInt;
        }
        for ( t2 = libGAP_FIRST_RECORD_TNUM;  t2 <= libGAP_LAST_RECORD_TNUM;  t2++ ) {
            libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdIntObj;
            libGAP_PowFuncs [ t2 ][ t1 ] = libGAP_PowObjInt;
        }
        for ( t2 = libGAP_FIRST_LIST_TNUM;    t2 <= libGAP_LAST_LIST_TNUM;    t2++ ) {
            libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdIntObj;
            libGAP_PowFuncs [ t2 ][ t1 ] = libGAP_PowObjInt;
        }
    }

    /* install the binary arithmetic methods                               */
    for ( t1 = libGAP_T_INT; t1 <= libGAP_T_INTNEG; t1++ ) {
        for ( t2 = libGAP_T_INT; t2 <= libGAP_T_INTNEG; t2++ ) {
            libGAP_EqFuncs  [ t1 ][ t2 ] = libGAP_EqInt;
            libGAP_LtFuncs  [ t1 ][ t2 ] = libGAP_LtInt;
            libGAP_SumFuncs [ t1 ][ t2 ] = libGAP_SumInt;
            libGAP_DiffFuncs[ t1 ][ t2 ] = libGAP_DiffInt;
            libGAP_ProdFuncs[ t1 ][ t2 ] = libGAP_ProdInt;
            libGAP_PowFuncs [ t1 ][ t2 ] = libGAP_PowInt;
            libGAP_ModFuncs [ t1 ][ t2 ] = libGAP_ModInt;
        }
    }

    /* gvars to import from the library                                    */
    libGAP_ImportGVarFromLibrary( "TYPE_INT_SMALL_ZERO", &libGAP_TYPE_INT_SMALL_ZERO );
    libGAP_ImportGVarFromLibrary( "TYPE_INT_SMALL_POS",  &libGAP_TYPE_INT_SMALL_POS );
    libGAP_ImportGVarFromLibrary( "TYPE_INT_SMALL_NEG",  &libGAP_TYPE_INT_SMALL_NEG );
    libGAP_ImportGVarFromLibrary( "TYPE_INT_LARGE_POS",  &libGAP_TYPE_INT_LARGE_POS );
    libGAP_ImportGVarFromLibrary( "TYPE_INT_LARGE_NEG",  &libGAP_TYPE_INT_LARGE_NEG );
    libGAP_ImportGVarFromLibrary( "SMALLEST_INTPOS", &libGAP_SMALLEST_INTPOS );

    libGAP_ImportFuncFromLibrary( "String", &libGAP_String );
    libGAP_ImportFuncFromLibrary( "One", &libGAP_OneAttr);

    /* install the kind functions                                          */
    libGAP_TypeObjFuncs[ libGAP_T_INT    ] = libGAP_TypeIntSmall;
    libGAP_TypeObjFuncs[ libGAP_T_INTPOS ] = libGAP_TypeIntLargePos;
    libGAP_TypeObjFuncs[ libGAP_T_INTNEG ] = libGAP_TypeIntLargeNeg;

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt gvar;
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* hold smallest large integer */
    libGAP_SMALLEST_INTPOS = libGAP_NewBag( libGAP_T_INTPOS, 4*sizeof(libGAP_TypDigit) );
    libGAP_ADDR_INT(libGAP_SMALLEST_INTPOS)[1] =
                 (libGAP_TypDigit) (1L << (libGAP_NR_SMALL_INT_BITS - libGAP_NR_DIGIT_BITS));
    gvar = libGAP_GVarName("SMALLEST_INTPOS");
    libGAP_MakeReadWriteGVar( gvar);
    libGAP_AssGVar( gvar, libGAP_SMALLEST_INTPOS );
    libGAP_MakeReadOnlyGVar(gvar);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoInt() . . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "integer",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoInt ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}

/* matches the USE_GMP test at the top of the file */
#endif

/****************************************************************************
**
*E  integer.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
