/****************************************************************************
**
*W  calls.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions for the function call mechanism package.
**
**  For a  description of what the function  call mechanism is  about see the
**  declaration part of this package.
**
**  Each function is  represented by a function  bag (of type  'T_FUNCTION'),
**  which has the following format.
**
**      +-------+-------+- - - -+-------+
**      |handler|handler|       |handler|   (for all functions)
**      |   0   |   1   |       |   7   |
**      +-------+-------+- - - -+-------+
**
**      +-------+-------+-------+-------+
**      | name  | number| args &| prof- |   (for all functions)
**      | func. |  args | locals| iling |
**      +-------+-------+-------+-------+
**
**      +-------+-------+-------+-------+
**      | number| body  | envir-| funcs.|   (only for interpreted functions)
**      | locals| func. | onment| exprs.|
**      +-------+-------+-------+-------+
**
**  ...what the handlers are..
**  ...what the other components are...
*/
#include        "system.h"              /* system dependent part           */



#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */

#include        "opers.h"               /* generic operations              */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */
#include        "vars.h"                /* variables                       */

#include        "stats.h"               /* statements                      */

#include        "saveload.h"            /* saving and loading              */


/****************************************************************************
**

*F * * * * wrapper for functions with variable number of arguments  * * * * *
*/

/****************************************************************************
**

*F  DoWrap0args( <self> ) . . . . . . . . . . . wrap up 0 arguments in a list
**
**  'DoWrap<i>args' accepts the  <i>  arguments  <arg1>, <arg2>, and   so on,
**  wraps them up in a list, and  then calls  <self>  again via 'CALL_XARGS',
**  passing this list.    'DoWrap<i>args' are the  handlers  for callees that
**  accept a   variable   number of   arguments.    Note that   there   is no
**  'DoWrapXargs' handler,  since in  this  case the function  call mechanism
**  already requires that the passed arguments are collected in a list.
*/
libGAP_Obj libGAP_DoWrap0args (
    libGAP_Obj                 self )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 args;           /* arguments list                  */

    /* make the arguments list                                             */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( args, 0 );

    /* call the variable number of arguments function                      */
    result = libGAP_CALL_XARGS( self, args );
    return result;
}


/****************************************************************************
**
*F  DoWrap1args( <self>, <arg1> ) . . . . . . . wrap up 1 arguments in a list
*/
libGAP_Obj libGAP_DoWrap1args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 args;           /* arguments list                  */

    /* make the arguments list                                             */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, 1 );
    libGAP_SET_LEN_PLIST( args, 1 );
    libGAP_SET_ELM_PLIST( args, 1, arg1 );

    /* call the variable number of arguments function                      */
    result = libGAP_CALL_XARGS( self, args );
    return result;
}


/****************************************************************************
**
*F  DoWrap2args( <self>, <arg1>, ... )  . . . . wrap up 2 arguments in a list
*/
libGAP_Obj libGAP_DoWrap2args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 args;           /* arguments list                  */

    /* make the arguments list                                             */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( args, 2 );
    libGAP_SET_ELM_PLIST( args, 1, arg1 );
    libGAP_SET_ELM_PLIST( args, 2, arg2 );

    /* call the variable number of arguments function                      */
    result = libGAP_CALL_XARGS( self, args );
    return result;
}


/****************************************************************************
**
*F  DoWrap3args( <self>, <arg1>, ... )  . . . . wrap up 3 arguments in a list
*/
libGAP_Obj libGAP_DoWrap3args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 args;           /* arguments list                  */

    /* make the arguments list                                             */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, 3 );
    libGAP_SET_LEN_PLIST( args, 3 );
    libGAP_SET_ELM_PLIST( args, 1, arg1 );
    libGAP_SET_ELM_PLIST( args, 2, arg2 );
    libGAP_SET_ELM_PLIST( args, 3, arg3 );

    /* call the variable number of arguments function                      */
    result = libGAP_CALL_XARGS( self, args );
    return result;
}


/****************************************************************************
**
*F  DoWrap4args( <self>, <arg1>, ... )  . . . . wrap up 4 arguments in a list
*/
libGAP_Obj libGAP_DoWrap4args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 args;           /* arguments list                  */

    /* make the arguments list                                             */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, 4 );
    libGAP_SET_LEN_PLIST( args, 4 );
    libGAP_SET_ELM_PLIST( args, 1, arg1 );
    libGAP_SET_ELM_PLIST( args, 2, arg2 );
    libGAP_SET_ELM_PLIST( args, 3, arg3 );
    libGAP_SET_ELM_PLIST( args, 4, arg4 );

    /* call the variable number of arguments function                      */
    result = libGAP_CALL_XARGS( self, args );
    return result;
}


/****************************************************************************
**
*F  DoWrap5args( <self>, <arg1>, ... )  . . . . wrap up 5 arguments in a list
*/
libGAP_Obj libGAP_DoWrap5args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 args;           /* arguments list                  */

    /* make the arguments list                                             */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, 5 );
    libGAP_SET_LEN_PLIST( args, 5 );
    libGAP_SET_ELM_PLIST( args, 1, arg1 );
    libGAP_SET_ELM_PLIST( args, 2, arg2 );
    libGAP_SET_ELM_PLIST( args, 3, arg3 );
    libGAP_SET_ELM_PLIST( args, 4, arg4 );
    libGAP_SET_ELM_PLIST( args, 5, arg5 );

    /* call the variable number of arguments function                      */
    result = libGAP_CALL_XARGS( self, args );
    return result;
}


/****************************************************************************
**
*F  DoWrap6args( <self>, <arg1>, ... )  . . . . wrap up 6 arguments in a list
*/
libGAP_Obj libGAP_DoWrap6args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 args;           /* arguments list                  */

    /* make the arguments list                                             */
    args = libGAP_NEW_PLIST( libGAP_T_PLIST, 6 );
    libGAP_SET_LEN_PLIST( args, 6 );
    libGAP_SET_ELM_PLIST( args, 1, arg1 );
    libGAP_SET_ELM_PLIST( args, 2, arg2 );
    libGAP_SET_ELM_PLIST( args, 3, arg3 );
    libGAP_SET_ELM_PLIST( args, 4, arg4 );
    libGAP_SET_ELM_PLIST( args, 5, arg5 );
    libGAP_SET_ELM_PLIST( args, 6, arg6 );

    /* call the variable number of arguments function                      */
    result = libGAP_CALL_XARGS( self, args );
    return result;
}


/****************************************************************************
**

*F * * wrapper for functions with do not support the number of arguments  * *
*/

/****************************************************************************
**

*F  DoFail0args( <self> )  . . . . . .  fail a function call with 0 arguments
**
**  'DoWrap<i>args' accepts the <i> arguments <arg1>, <arg2>,  and so on, and
**  signals an error,  because  the  function for  which  they  are installed
**  expects another number of arguments.  'DoFail<i>args' are the handlers in
**  the other slots of a function.
*/
libGAP_Obj libGAP_DoFail0args (
    libGAP_Obj                 self )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), 0L,
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**
*F  DoFail1args( <self>,<arg1> ) . . .  fail a function call with 1 arguments
*/
libGAP_Obj libGAP_DoFail1args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1 )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), 1L,
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**
*F  DoFail2args( <self>, <arg1>, ... )  fail a function call with 2 arguments
*/
libGAP_Obj libGAP_DoFail2args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), 2L,
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**
*F  DoFail3args( <self>, <arg1>, ... )  fail a function call with 3 arguments
*/
libGAP_Obj libGAP_DoFail3args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), 3L,
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**
*F  DoFail4args( <self>, <arg1>, ... )  fail a function call with 4 arguments
*/
libGAP_Obj libGAP_DoFail4args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), 4L,
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**
*F  DoFail5args( <self>, <arg1>, ... )  fail a function call with 5 arguments
*/
libGAP_Obj libGAP_DoFail5args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), 5L,
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**
*F  DoFail6args( <self>, <arg1>, ... )  fail a function call with 6 arguments
*/
libGAP_Obj libGAP_DoFail6args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), 6L,
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**
*F  DoFailXargs( <self>, <args> )  . .  fail a function call with X arguments
*/
libGAP_Obj libGAP_DoFailXargs (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 argx;           /* arguments list (to continue)    */
    argx = libGAP_ErrorReturnObj(
        "Function: number of arguments must be %d (not %d)",
        (libGAP_Int)libGAP_NARG_FUNC( self ), libGAP_LEN_LIST( args ),
        "you can replace the argument list <args> via 'return <args>;'" );
    return libGAP_FuncCALL_FUNC_LIST( (libGAP_Obj)0, self, argx );
}


/****************************************************************************
**

*F * * * * * * * * * * * * *  wrapper for profiling * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  TimeDone  . . . . . .   amount of time spent for completed function calls
**
**  'TimeDone' is  the amount of time spent  for all function calls that have
**  already been completed.
*/
libGAP_UInt libGAP_TimeDone;


/****************************************************************************
**
*V  StorDone  . . . . .  amount of storage spent for completed function calls
**
**  'StorDone' is the amount of storage spent for all function call that have
**  already been completed.
*/
libGAP_UInt libGAP_StorDone;


/****************************************************************************
**
*F  DoProf0args( <self> ) . . . . . . . . profile a function with 0 arguments
**
**  'DoProf<i>args' accepts the <i> arguments <arg1>, <arg2>,  and so on, and
**  calls  the function through the  secondary  handler.  It also updates the
**  profiling  information in  the profiling information   bag of  the called
**  function.  'DoProf<i>args' are  the primary  handlers  for all  functions
**  when profiling is requested.
*/
libGAP_Obj libGAP_DoProf0args (
    libGAP_Obj                 self )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_0ARGS_PROF( self );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children            */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function          */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**
*F  DoProf1args( <self>, <arg1>)  . . . . profile a function with 1 arguments
*/
libGAP_Obj libGAP_DoProf1args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_1ARGS_PROF( self, arg1 );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children            */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function          */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**
*F  DoProf2args( <self>, <arg1>, ... )  . profile a function with 2 arguments
*/
libGAP_Obj libGAP_DoProf2args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_2ARGS_PROF( self, arg1, arg2 );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children             */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function           */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**
*F  DoProf3args( <self>, <arg1>, ... )  . profile a function with 3 arguments
*/
libGAP_Obj libGAP_DoProf3args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_3ARGS_PROF( self, arg1, arg2, arg3 );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children            */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function          */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**
*F  DoProf4args( <self>, <arg1>, ... )  . profile a function with 4 arguments
*/
libGAP_Obj libGAP_DoProf4args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_4ARGS_PROF( self, arg1, arg2, arg3, arg4 );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children            */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function          */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**
*F  DoProf5args( <self>, <arg1>, ... )  . profile a function with 5 arguments
*/
libGAP_Obj libGAP_DoProf5args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_5ARGS_PROF( self, arg1, arg2, arg3, arg4, arg5 );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children            */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function          */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**
*F  DoProf6args( <self>, <arg1>, ... )  . profile a function with 6 arguments
*/
libGAP_Obj libGAP_DoProf6args (
    libGAP_Obj                 self,
    libGAP_Obj                 arg1,
    libGAP_Obj                 arg2,
    libGAP_Obj                 arg3,
    libGAP_Obj                 arg4,
    libGAP_Obj                 arg5,
    libGAP_Obj                 arg6 )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_6ARGS_PROF( self, arg1, arg2, arg3, arg4, arg5, arg6 );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children            */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function          */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**
*F  DoProfXargs( <self>, <args> ) . . . . profile a function with X arguments
*/
libGAP_Obj libGAP_DoProfXargs (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 result;         /* value of function call, result  */
    libGAP_Obj                 prof;           /* profiling bag                   */
    libGAP_UInt                timeElse;       /* time    spent elsewhere         */
    libGAP_UInt                timeCurr;       /* time    spent in current funcs. */
    libGAP_UInt                storElse;       /* storage spent elsewhere         */
    libGAP_UInt                storCurr;       /* storage spent in current funcs. */ 

    /* get the profiling bag                                               */
    prof = libGAP_PROF_FUNC( libGAP_PROF_FUNC( self ) );

    /* time and storage spent so far while this function what not active   */
    timeElse = libGAP_SyTime() - libGAP_TIME_WITH_PROF(prof);
    storElse = libGAP_SizeAllBags - libGAP_STOR_WITH_PROF(prof);

    /* time and storage spent so far by all currently suspended functions  */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone;

    /* call the real function                                              */
    result = libGAP_CALL_XARGS_PROF( self, args );

    /* number of invocation of this function                               */
    libGAP_SET_COUNT_PROF( prof, libGAP_COUNT_PROF(prof) + 1 );

    /* time and storage spent in this function and its children            */
    libGAP_SET_TIME_WITH_PROF( prof, libGAP_SyTime() - timeElse );
    libGAP_SET_STOR_WITH_PROF( prof, libGAP_SizeAllBags - storElse );

    /* time and storage spent by this invocation of this function          */
    timeCurr = libGAP_SyTime() - libGAP_TimeDone - timeCurr;
    libGAP_SET_TIME_WOUT_PROF( prof, libGAP_TIME_WOUT_PROF(prof) + timeCurr );
    libGAP_TimeDone += timeCurr;
    storCurr = libGAP_SizeAllBags - libGAP_StorDone - storCurr;
    libGAP_SET_STOR_WOUT_PROF( prof, libGAP_STOR_WOUT_PROF(prof) + storCurr );
    libGAP_StorDone += storCurr;

    /* return the result from the function                                 */
    return result;
}


/****************************************************************************
**

*F * * * * * * * * * * * * *  create a new function * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  InitHandlerFunc( <handler>, <cookie> ) . . . . . . . . register a handler
**
**  Every handler should  be registered (once) before  it is installed in any
**  function bag. This is needed so that it can be  identified when loading a
**  saved workspace.  <cookie> should be a  unique  C string, identifying the
**  handler
*/
#ifndef libGAP_MAX_HANDLERS
#define libGAP_MAX_HANDLERS 20000
#endif

typedef struct {
    libGAP_ObjFunc             hdlr;
    const libGAP_Char *        cookie;
}
libGAP_TypeHandlerInfo;

static libGAP_UInt libGAP_HandlerSortingStatus;

static libGAP_TypeHandlerInfo libGAP_HandlerFuncs[libGAP_MAX_HANDLERS];
static libGAP_UInt libGAP_NHandlerFuncs;
 
void libGAP_InitHandlerFunc (
    libGAP_ObjFunc             hdlr,
    const libGAP_Char *        cookie )
{
    if ( libGAP_NHandlerFuncs >= libGAP_MAX_HANDLERS ) {
        libGAP_Pr( "No room left for function handler\n", 0L, 0L );
        libGAP_SyExit(1);
    }
#ifdef DEBUG_HANDLER_REGISTRATION
    {
      libGAP_UInt i;
      for (i = 0; i < libGAP_NHandlerFuncs; i++)
        if (!strcmp(libGAP_HandlerFuncs[i].cookie, cookie))
          libGAP_Pr("Duplicate cookie %s\n", (libGAP_Int)cookie, 0L);
    }
#endif
    libGAP_HandlerFuncs[libGAP_NHandlerFuncs].hdlr   = hdlr;
    libGAP_HandlerFuncs[libGAP_NHandlerFuncs].cookie = cookie;
    libGAP_HandlerSortingStatus = 0; /* no longer sorted by handler or cookie */
    libGAP_NHandlerFuncs++;
}



/****************************************************************************
**
*f  CheckHandlersBag( <bag> ) . . . . . . check that handlers are initialised
*/

void libGAP_InitHandlerRegistration( void )
{
  /* initialize these here rather than statically to allow for restart */
  /* can't do them in InitKernel of this module because it's called too late
     so make it a function and call it from an earlier InitKernel */
  libGAP_HandlerSortingStatus = 0;
  libGAP_NHandlerFuncs = 0;

}

static void libGAP_CheckHandlersBag(
    libGAP_Bag         bag )
{
#ifdef DEBUG_HANDLER_REGISTRATION
    libGAP_UInt        i;
    libGAP_UInt        j;
    libGAP_ObjFunc     hdlr;

    if ( libGAP_TNUM_BAG(bag) == libGAP_T_FUNCTION ) {
        for ( j = 0;  j < 8;  j++ ) {
            hdlr = libGAP_HDLR_FUNC(bag,j);

            /* zero handlers are used in a few odd places                  */
            if ( hdlr != 0 ) {
                for ( i = 0;  i < libGAP_NHandlerFuncs;  i++ ) {
                    if ( hdlr == libGAP_HandlerFuncs[i].hdlr )
                        break;
                }
                if ( i == libGAP_NHandlerFuncs ) {
                    libGAP_Pr("Unregistered Handler %d args  ", j, 0L);
                    libGAP_PrintObj(libGAP_NAME_FUNC(bag));
                    libGAP_Pr("\n",0L,0L);
                }
            }
        }
    }
#endif
  return;
}

void libGAP_CheckAllHandlers(
       void )
{
  libGAP_CallbackForAllBags( libGAP_CheckHandlersBag);
    return;
}


static int libGAP_IsLessHandlerInfo (
    libGAP_TypeHandlerInfo *           h1, 
    libGAP_TypeHandlerInfo *           h2,
    libGAP_UInt                        byWhat )
{
    switch (byWhat) {
        case 1:
            /* cast to please Irix CC and HPUX CC */
            return (libGAP_UInt)(h1->hdlr) < (libGAP_UInt)(h2->hdlr);
        case 2:
            return strcmp(h1->cookie, h2->cookie) < 0;
        default:
            libGAP_ErrorQuit( "Invalid sort mode %u", (libGAP_Int)byWhat, 0L );
            return 0; /* please lint */
    }
}

void libGAP_SortHandlers( libGAP_UInt byWhat )
{
  libGAP_TypeHandlerInfo tmp;
  libGAP_UInt len, h, i, k;
  if (libGAP_HandlerSortingStatus == byWhat)
    return;
  len = libGAP_NHandlerFuncs;
  h = 1;
  while ( 9*h + 4 < len ) 
    { h = 3*h + 1; }
  while ( 0 < h ) {
    for ( i = h; i < len; i++ ) {
      tmp = libGAP_HandlerFuncs[i];
      k = i;
      while ( h <= k && libGAP_IsLessHandlerInfo(&tmp, libGAP_HandlerFuncs+(k-h), byWhat))
        {
          libGAP_HandlerFuncs[k] = libGAP_HandlerFuncs[k-h];
          k -= h;
        }
      libGAP_HandlerFuncs[k] = tmp;
    }
    h = h / 3;
  }
  libGAP_HandlerSortingStatus = byWhat;
  return;
}

const libGAP_Char * libGAP_CookieOfHandler (
    libGAP_ObjFunc             hdlr )
{
    libGAP_UInt                i, top, bottom, middle;

    if ( libGAP_HandlerSortingStatus != 1 ) {
        for ( i = 0; i < libGAP_NHandlerFuncs; i++ ) {
            if ( hdlr == libGAP_HandlerFuncs[i].hdlr )
                return libGAP_HandlerFuncs[i].cookie;
        }
        return (libGAP_Char *)0L;
    }
    else {
        top = libGAP_NHandlerFuncs;
        bottom = 0;
        while ( top >= bottom ) {
            middle = (top + bottom)/2;
            if ( (libGAP_UInt)(hdlr) < (libGAP_UInt)(libGAP_HandlerFuncs[middle].hdlr) )
                top = middle-1;
            else if ( (libGAP_UInt)(hdlr) > (libGAP_UInt)(libGAP_HandlerFuncs[middle].hdlr) )
                bottom = middle+1;
            else
                return libGAP_HandlerFuncs[middle].cookie;
        }
        return (libGAP_Char *)0L;
    }
}

libGAP_ObjFunc libGAP_HandlerOfCookie(
       const libGAP_Char * cookie )
{
  libGAP_Int i,top,bottom,middle;
  libGAP_Int res;
  if (libGAP_HandlerSortingStatus != 2) 
    {
      for (i = 0; i < libGAP_NHandlerFuncs; i++)
        {
          if (strcmp(cookie, libGAP_HandlerFuncs[i].cookie) == 0)
            return libGAP_HandlerFuncs[i].hdlr;
        }
      return (libGAP_ObjFunc)0L;
    }
  else
    {
      top = libGAP_NHandlerFuncs;
      bottom = 0;
      while (top >= bottom) {
        middle = (top + bottom)/2;
        res = strcmp(cookie,libGAP_HandlerFuncs[middle].cookie);
        if (res < 0)
          top = middle-1;
        else if (res > 0)
          bottom = middle+1;
        else
          return libGAP_HandlerFuncs[middle].hdlr;
      }
      return (libGAP_ObjFunc)0L;
    }
}
                                      
                       

/****************************************************************************
**

*F  NewFunction( <name>, <narg>, <nams>, <hdlr> ) . . . . make a new function
**
**  'NewFunction' creates and returns a new function.  <name> must be  a  GAP
**  string containing the name of the function.  <narg> must be the number of
**  arguments, where -1 means a variable number of arguments.  <nams> must be
**  a GAP list containg the names  of  the  arguments.  <hdlr>  must  be  the
**  C function (accepting <self> and  the  <narg>  arguments)  that  will  be
**  called to execute the function.
*/
libGAP_Obj libGAP_NewFunction (
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams,
    libGAP_ObjFunc             hdlr )
{
    return libGAP_NewFunctionT( libGAP_T_FUNCTION, libGAP_SIZE_FUNC, name, narg, nams, hdlr );
}
    

/****************************************************************************
**
*F  NewFunctionC( <name>, <narg>, <nams>, <hdlr> )  . . . make a new function
**
**  'NewFunctionC' does the same as 'NewFunction',  but  expects  <name>  and
**  <nams> as C strings.
*/
libGAP_Obj libGAP_NewFunctionC (
    const libGAP_Char *        name,
    libGAP_Int                 narg,
    const libGAP_Char *        nams,
    libGAP_ObjFunc             hdlr )
{
    return libGAP_NewFunctionCT( libGAP_T_FUNCTION, libGAP_SIZE_FUNC, name, narg, nams, hdlr );
}
    

/****************************************************************************
**
*F  NewFunctionT( <type>, <size>, <name>, <narg>, <nams>, <hdlr> )
**
**  'NewFunctionT' does the same as 'NewFunction', but allows to specify  the
**  <type> and <size> of the newly created bag.
*/
libGAP_Obj libGAP_NewFunctionT (
    libGAP_UInt                type,
    libGAP_UInt                size,
    libGAP_Obj                 name,
    libGAP_Int                 narg,
    libGAP_Obj                 nams,
    libGAP_ObjFunc             hdlr )
{
    libGAP_Obj                 func;           /* function, result                */
    libGAP_Obj                 prof;           /* profiling bag                   */


    /* make the function object                                            */
    func = libGAP_NewBag( type, size );

    /* create a function with a fixed number of arguments                  */
    if ( narg != -1 ) {
        libGAP_HDLR_FUNC(func,0) = libGAP_DoFail0args;
        libGAP_HDLR_FUNC(func,1) = libGAP_DoFail1args;
        libGAP_HDLR_FUNC(func,2) = libGAP_DoFail2args;
        libGAP_HDLR_FUNC(func,3) = libGAP_DoFail3args;
        libGAP_HDLR_FUNC(func,4) = libGAP_DoFail4args;
        libGAP_HDLR_FUNC(func,5) = libGAP_DoFail5args;
        libGAP_HDLR_FUNC(func,6) = libGAP_DoFail6args;
        libGAP_HDLR_FUNC(func,7) = libGAP_DoFailXargs;
        libGAP_HDLR_FUNC( func, (narg <= 6 ? narg : 7) ) = hdlr;
    }

    /* create a function with a variable number of arguments               */
    else {
        libGAP_HDLR_FUNC(func,0) = libGAP_DoWrap0args;
        libGAP_HDLR_FUNC(func,1) = libGAP_DoWrap1args;
        libGAP_HDLR_FUNC(func,2) = libGAP_DoWrap2args;
        libGAP_HDLR_FUNC(func,3) = libGAP_DoWrap3args;
        libGAP_HDLR_FUNC(func,4) = libGAP_DoWrap4args;
        libGAP_HDLR_FUNC(func,5) = libGAP_DoWrap5args;
        libGAP_HDLR_FUNC(func,6) = libGAP_DoWrap6args;
        libGAP_HDLR_FUNC(func,7) = hdlr;
    }

    /* enter the the arguments and the names                               */
    libGAP_NAME_FUNC(func) = name;
    libGAP_NARG_FUNC(func) = narg;
    libGAP_NAMS_FUNC(func) = nams;
    libGAP_CHANGED_BAG(func);

    /* enter the profiling bag                                             */
    prof = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_LEN_PROF );
    libGAP_SET_LEN_PLIST( prof, libGAP_LEN_PROF );
    libGAP_SET_COUNT_PROF( prof, 0 );
    libGAP_SET_TIME_WITH_PROF( prof, 0 );
    libGAP_SET_TIME_WOUT_PROF( prof, 0 );
    libGAP_SET_STOR_WITH_PROF( prof, 0 );
    libGAP_SET_STOR_WOUT_PROF( prof, 0 );
    libGAP_PROF_FUNC(func) = prof;
    libGAP_CHANGED_BAG(func);

    /* return the function bag                                             */
    return func;
}
    

/****************************************************************************
**
*F  NewFunctionCT( <type>, <size>, <name>, <narg>, <nams>, <hdlr> )
**
**  'NewFunctionCT' does the same as 'NewFunction', but  expects  <name>  and
**  <nams> as C strings, and allows to specify the <type> and <size>  of  the
**  newly created bag.
*/
libGAP_Obj libGAP_NewFunctionCT (
    libGAP_UInt                type,
    libGAP_UInt                size,
    const libGAP_Char *        name_c,
    libGAP_Int                 narg,
    const libGAP_Char *        nams_c,
    libGAP_ObjFunc             hdlr )
{
    libGAP_Obj                 name_o;         /* name as an object               */

    /* convert the name to an object                                       */
    libGAP_C_NEW_STRING_DYN(name_o, name_c);
    libGAP_RetypeBag(name_o, libGAP_T_STRING+libGAP_IMMUTABLE);

    /* make the function                                                   */
    return libGAP_NewFunctionT( type, size, name_o, narg, libGAP_ArgStringToList( nams_c ), hdlr );
}
    

/****************************************************************************
**
*F  ArgStringToList( <nams_c> )
**
** 'ArgStringToList' takes a C string <nams_c> containing a list of comma
** separated argument names, and turns it into a plist of strings, ready
** to be passed to 'NewFunction' as <nams>.
*/
libGAP_Obj libGAP_ArgStringToList(const libGAP_Char *nams_c) {
    libGAP_Obj                 tmp;            /* argument name as an object      */
    libGAP_Obj                 nams_o;         /* nams as an object               */
    libGAP_UInt                len;            /* length                          */
    libGAP_UInt                i, k, l;        /* loop variables                  */

    /* convert the arguments list to an object                             */
    len = 0;
    for ( k = 0; nams_c[k] != '\0'; k++ ) {
        if ( (0 == k || nams_c[k-1] == ' ' || nams_c[k-1] == ',')
          && (          nams_c[k  ] != ' ' && nams_c[k  ] != ',') ) {
            len++;
        }
    }
    nams_o = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
    libGAP_SET_LEN_PLIST( nams_o, len );
    k = 0;
    for ( i = 1; i <= len; i++ ) {
        while ( nams_c[k] == ' ' || nams_c[k] == ',' ) {
            k++;
        }
        l = k;
        while ( nams_c[l] != ' ' && nams_c[l] != ',' && nams_c[l] != '\0' ) {
            l++;
        }
        libGAP_C_NEW_STRING( tmp, l - k, nams_c + k );
        libGAP_RetypeBag( tmp, libGAP_T_STRING+libGAP_IMMUTABLE );
        libGAP_SET_ELM_PLIST( nams_o, i, tmp );
        k = l;
    }

    return nams_o;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * type and print function  * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  TypeFunction( <func> )  . . . . . . . . . . . . . . .  kind of a function
**
**  'TypeFunction' returns the kind of the function <func>.
**
**  'TypeFunction' is the function in 'TypeObjFuncs' for functions.
*/
libGAP_Obj libGAP_TYPE_FUNCTION;
libGAP_Obj libGAP_TYPE_OPERATION;

libGAP_Obj libGAP_TypeFunction (
    libGAP_Obj                 func )
{
    return ( libGAP_IS_OPERATION(func) ? libGAP_TYPE_OPERATION : libGAP_TYPE_FUNCTION );
}



/****************************************************************************
**
*F  PrintFunction( <func> )   . . . . . . . . . . . . . . .  print a function
**
*/

libGAP_Obj libGAP_PrintOperation;

void libGAP_PrintFunction (
    libGAP_Obj                 func )
{
    libGAP_Int                 narg;           /* number of arguments             */
    libGAP_Int                 nloc;           /* number of locals                */
    libGAP_Obj                 oldLVars;       /* terrible hack                   */
    libGAP_UInt                i;              /* loop variable                   */


    if ( libGAP_IS_OPERATION(func) ) {
      libGAP_CALL_1ARGS( libGAP_PrintOperation, func );
      return;
    }

    /* print 'function ('                                                  */
    libGAP_Pr("%5>function%< ( %>",0L,0L);

    /* print the arguments                                                 */
    narg = (libGAP_NARG_FUNC(func) == -1 ? 1 : libGAP_NARG_FUNC(func));
    for ( i = 1; i <= narg; i++ ) {
        if ( libGAP_NAMS_FUNC(func) != 0 )
            libGAP_Pr( "%I", (libGAP_Int)libGAP_NAMI_FUNC( func, (libGAP_Int)i ), 0L );
        else
            libGAP_Pr( "<<arg-%d>>", (libGAP_Int)i, 0L );
        if ( i != narg )  libGAP_Pr("%<, %>",0L,0L);
    }
    libGAP_Pr(" %<)",0L,0L);

        libGAP_Pr("\n",0L,0L);

        /* print the locals                                                */
        nloc = libGAP_NLOC_FUNC(func);
        if ( nloc >= 1 ) {
            libGAP_Pr("%>local  ",0L,0L);
            for ( i = 1; i <= nloc; i++ ) {
                if ( libGAP_NAMS_FUNC(func) != 0 )
                    libGAP_Pr( "%I", (libGAP_Int)libGAP_NAMI_FUNC( func, (libGAP_Int)(narg+i) ), 0L );
                else
                    libGAP_Pr( "<<loc-%d>>", (libGAP_Int)i, 0L );
                if ( i != nloc )  libGAP_Pr("%<, %>",0L,0L);
            }
            libGAP_Pr("%<;\n",0L,0L);
        }

        /* print the body                                                  */
        if ( libGAP_BODY_FUNC(func) == 0 || libGAP_SIZE_OBJ(libGAP_BODY_FUNC(func)) == libGAP_NUMBER_HEADER_ITEMS_BODY*sizeof(libGAP_Obj) ) {
            libGAP_Pr("<<kernel or compiled code>>",0L,0L);
        }
        else {
            libGAP_SWITCH_TO_NEW_LVARS( func, libGAP_NARG_FUNC(func), libGAP_NLOC_FUNC(func),
                                 oldLVars );
            libGAP_PrintStat( libGAP_FIRST_STAT_CURR_FUNC );
            libGAP_SWITCH_TO_OLD_LVARS( oldLVars );
        }
        libGAP_Pr("%4<\n",0L,0L);
    
    /* print 'end'                                                         */
    libGAP_Pr("end",0L,0L);
}


/****************************************************************************
**
*F  FuncIS_FUNCTION( <self>, <func> ) . . . . . . . . . . . test for function
**
**  'FuncIS_FUNCTION' implements the internal function 'IsFunction'.
**
**  'IsFunction( <func> )'
**
**  'IsFunction' returns   'true'  if  <func>   is a function    and  'false'
**  otherwise.
*/
libGAP_Obj libGAP_IsFunctionFilt;

libGAP_Obj libGAP_FuncIS_FUNCTION (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    if      ( libGAP_TNUM_OBJ(obj) == libGAP_T_FUNCTION ) {
        return libGAP_True;
    }
    else if ( libGAP_TNUM_OBJ(obj) < libGAP_FIRST_EXTERNAL_TNUM ) {
        return libGAP_False;
    }
    else {
        return libGAP_DoFilter( self, obj );
    }
}


/****************************************************************************
**
*F  FuncCALL_FUNC( <self>, <args> ) . . . . . . . . . . . . . call a function
**
**  'FuncCALL_FUNC' implements the internal function 'CallFunction'.
**
**  'CallFunction( <func>, <arg1>... )'
**
**  'CallFunction' calls the  function <func> with the  arguments  <arg1>...,
**  i.e., it is equivalent to '<func>( <arg1>, <arg2>... )'.
*/
libGAP_Obj libGAP_CallFunctionOper;



libGAP_Obj libGAP_FuncCALL_FUNC (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 result;         /* result                          */
    libGAP_Obj                 func;           /* function                        */
    libGAP_Obj                 list2;          /* list of arguments               */
    libGAP_Obj                 arg;            /* one argument                    */
    libGAP_UInt                i;              /* loop variable                   */

    /* the first argument is the function                                  */
    if ( libGAP_LEN_LIST( args ) == 0 ) {
        func = libGAP_ErrorReturnObj(
            "usage: CallFunction( <func>, <arg1>... )",
            0L, 0L,
            "you can replace function <func> via 'return <func>;'" );
    }
    else {
        func = libGAP_ELMV_LIST( args, 1 );
    }    

    /* check that the first argument is a function                         */
    /*N 1996/06/26 mschoene this should be done by 'CALL_<i>ARGS'          */
    while ( libGAP_TNUM_OBJ( func ) != libGAP_T_FUNCTION ) {
        func = libGAP_ErrorReturnObj(
            "CallFunction: <func> must be a function",
            0L, 0L,
            "you can replace function <func> via 'return <func>;'" );
    }

    /* call the function                                                   */
    if      ( libGAP_LEN_LIST(args) == 1 ) {
        result = libGAP_CALL_0ARGS( func );
    }
    else if ( libGAP_LEN_LIST(args) == 2 ) {
        result = libGAP_CALL_1ARGS( func, libGAP_ELMV_LIST(args,2) );
    }
    else if ( libGAP_LEN_LIST(args) == 3 ) {
        result = libGAP_CALL_2ARGS( func, libGAP_ELMV_LIST(args,2), libGAP_ELMV_LIST(args,3) );
    }
    else if ( libGAP_LEN_LIST(args) == 4 ) {
        result = libGAP_CALL_3ARGS( func, libGAP_ELMV_LIST(args,2), libGAP_ELMV_LIST(args,3),
                                   libGAP_ELMV_LIST(args,4) );
    }
    else if ( libGAP_LEN_LIST(args) == 5 ) {
        result = libGAP_CALL_4ARGS( func, libGAP_ELMV_LIST(args,2), libGAP_ELMV_LIST(args,3),
                                   libGAP_ELMV_LIST(args,4), libGAP_ELMV_LIST(args,5) );
    }
    else if ( libGAP_LEN_LIST(args) == 6 ) {
        result = libGAP_CALL_5ARGS( func, libGAP_ELMV_LIST(args,2), libGAP_ELMV_LIST(args,3),
                                   libGAP_ELMV_LIST(args,4), libGAP_ELMV_LIST(args,5),
                                   libGAP_ELMV_LIST(args,6) );
    }
    else if ( libGAP_LEN_LIST(args) == 7 ) {
        result = libGAP_CALL_6ARGS( func, libGAP_ELMV_LIST(args,2), libGAP_ELMV_LIST(args,3),
                                   libGAP_ELMV_LIST(args,4), libGAP_ELMV_LIST(args,5),
                                   libGAP_ELMV_LIST(args,6), libGAP_ELMV_LIST(args,7) );
    }
    else {
        list2 = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_LEN_LIST(args)-1 );
        libGAP_SET_LEN_PLIST( list2, libGAP_LEN_LIST(args)-1 );
        for ( i = 1; i <= libGAP_LEN_LIST(args)-1; i++ ) {
            arg = libGAP_ELMV_LIST( args, (libGAP_Int)(i+1) );
            libGAP_SET_ELM_PLIST( list2, i, arg );
        }
        result = libGAP_CALL_XARGS( func, list2 );
    }

    /* return the result                                                   */
    return result;
}


/****************************************************************************
**
*F  FuncCALL_FUNC_LIST( <self>, <func>, <list> )  . . . . . . call a function
**
**  'FuncCALL_FUNC_LIST' implements the internal function 'CallFuncList'.
**
**  'CallFuncList( <func>, <list> )'
**
**  'CallFuncList' calls the  function <func> with the arguments list <list>,
**  i.e., it is equivalent to '<func>( <list>[1], <list>[2]... )'.
*/
libGAP_Obj libGAP_CallFuncListOper;

libGAP_Obj libGAP_FuncCALL_FUNC_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 func,
    libGAP_Obj                 list )
{
    libGAP_Obj                 result;         /* result                          */
    libGAP_Obj                 list2;          /* list of arguments               */
    libGAP_Obj                 arg;            /* one argument                    */
    libGAP_UInt                i;              /* loop variable                   */

    /* check that the second argument is a list                            */
    while ( ! libGAP_IS_SMALL_LIST( list ) ) {
        list = libGAP_ErrorReturnObj(
            "CallFuncList: <list> must be a small list",
            0L, 0L,
            "you can replace <list> via 'return <list>;'" );
    }

    if (libGAP_TNUM_OBJ(func) == libGAP_T_FUNCTION) {

      /* call the function                                                   */
      if      ( libGAP_LEN_LIST(list) == 0 ) {
        result = libGAP_CALL_0ARGS( func );
      }
      else if ( libGAP_LEN_LIST(list) == 1 ) {
        result = libGAP_CALL_1ARGS( func, libGAP_ELMV_LIST(list,1) );
      }
      else if ( libGAP_LEN_LIST(list) == 2 ) {
        result = libGAP_CALL_2ARGS( func, libGAP_ELMV_LIST(list,1), libGAP_ELMV_LIST(list,2) );
      }
      else if ( libGAP_LEN_LIST(list) == 3 ) {
        result = libGAP_CALL_3ARGS( func, libGAP_ELMV_LIST(list,1), libGAP_ELMV_LIST(list,2),
                             libGAP_ELMV_LIST(list,3) );
      }
      else if ( libGAP_LEN_LIST(list) == 4 ) {
        result = libGAP_CALL_4ARGS( func, libGAP_ELMV_LIST(list,1), libGAP_ELMV_LIST(list,2),
                             libGAP_ELMV_LIST(list,3), libGAP_ELMV_LIST(list,4) );
      }
      else if ( libGAP_LEN_LIST(list) == 5 ) {
        result = libGAP_CALL_5ARGS( func, libGAP_ELMV_LIST(list,1), libGAP_ELMV_LIST(list,2),
                             libGAP_ELMV_LIST(list,3), libGAP_ELMV_LIST(list,4),
                             libGAP_ELMV_LIST(list,5) );
      }
      else if ( libGAP_LEN_LIST(list) == 6 ) {
        result = libGAP_CALL_6ARGS( func, libGAP_ELMV_LIST(list,1), libGAP_ELMV_LIST(list,2),
                             libGAP_ELMV_LIST(list,3), libGAP_ELMV_LIST(list,4),
                             libGAP_ELMV_LIST(list,5), libGAP_ELMV_LIST(list,6) );
      }
      else {
        list2 = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_LEN_LIST(list) );
        libGAP_SET_LEN_PLIST( list2, libGAP_LEN_LIST(list) );
        for ( i = 1; i <= libGAP_LEN_LIST(list); i++ ) {
          arg = libGAP_ELMV_LIST( list, (libGAP_Int)i );
          libGAP_SET_ELM_PLIST( list2, i, arg );
        }
        result = libGAP_CALL_XARGS( func, list2 );
      }
    } else {
      result = libGAP_DoOperation2Args(self, func, list);
    }
    /* return the result                                                   */
    return result;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * * utility functions  * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  FuncNAME_FUNC( <self>, <func> ) . . . . . . . . . . .  name of a function
*/
libGAP_Obj libGAP_NAME_FUNC_Oper;
libGAP_Obj libGAP_SET_NAME_FUNC_Oper;

libGAP_Obj libGAP_FuncNAME_FUNC (
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
    libGAP_Obj                 name;

    if ( libGAP_TNUM_OBJ(func) == libGAP_T_FUNCTION ) {
        name = libGAP_NAME_FUNC(func);
        if ( name == 0 ) {
            libGAP_C_NEW_STRING_CONST(name, "unknown");
            libGAP_RetypeBag(name, libGAP_T_STRING+libGAP_IMMUTABLE);
            libGAP_NAME_FUNC(func) = name;
            libGAP_CHANGED_BAG(func);
        }
        return name;
    }
    else {
        return libGAP_DoOperation1Args( self, func );
    }
}

libGAP_Obj libGAP_FuncSET_NAME_FUNC(
                      libGAP_Obj self,
                      libGAP_Obj func,
                      libGAP_Obj name )
{
  while (!libGAP_IsStringConv(name))
    name = libGAP_ErrorReturnObj("SET_NAME_FUNC( <func>, <name> ): <name> must be a string, not a %s",
                          (libGAP_Int)libGAP_TNAM_OBJ(name), 0, "YOu can return a new name to continue");
  if (libGAP_TNUM_OBJ(func) == libGAP_T_FUNCTION ) {
    libGAP_NAME_FUNC(func) = name;
    libGAP_CHANGED_BAG(func);
  } else
    libGAP_DoOperation2Args(libGAP_SET_NAME_FUNC_Oper, func, name);
  return (libGAP_Obj) 0;
}


/****************************************************************************
**
*F  FuncNARG_FUNC( <self>, <func> ) . . . . number of arguments of a function
*/
libGAP_Obj libGAP_NARG_FUNC_Oper;

libGAP_Obj libGAP_FuncNARG_FUNC (
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
    if ( libGAP_TNUM_OBJ(func) == libGAP_T_FUNCTION ) {
        return libGAP_INTOBJ_INT( libGAP_NARG_FUNC(func) );
    }
    else {
        return libGAP_DoOperation1Args( self, func );
    }
}


/****************************************************************************
**
*F  FuncNAMS_FUNC( <self>, <func> ) . . . . names of local vars of a function
*/
libGAP_Obj libGAP_NAMS_FUNC_Oper;

libGAP_Obj libGAP_FuncNAMS_FUNC (
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
  libGAP_Obj nams;
    if ( libGAP_TNUM_OBJ(func) == libGAP_T_FUNCTION ) {
        nams = libGAP_NAMS_FUNC(func);
        return (nams != (libGAP_Obj)0) ? nams : libGAP_Fail;
    }
    else {
        return libGAP_DoOperation1Args( self, func );
    }
}


/****************************************************************************
**
*F  FuncPROF_FUNC( <self>, <func> ) . . . . . .  profiling info of a function
*/
libGAP_Obj libGAP_PROF_FUNC_Oper;

libGAP_Obj libGAP_FuncPROF_FUNC (
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
    libGAP_Obj                 prof;

    if ( libGAP_TNUM_OBJ(func) == libGAP_T_FUNCTION ) {
        prof = libGAP_PROF_FUNC(func);
        if ( libGAP_TNUM_OBJ(prof) == libGAP_T_FUNCTION ) {
            return libGAP_PROF_FUNC(prof);
        } else {
            return prof;
        }
    }
    else {
        return libGAP_DoOperation1Args( self, func );
    }
}


/****************************************************************************
**

*F  FuncCLEAR_PROFILE_FUNC( <self>, <func> )  . . . . . . . . . clear profile
*/
libGAP_Obj libGAP_FuncCLEAR_PROFILE_FUNC(
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
    libGAP_Obj                 prof;

    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit( "<func> must be a function", 0L, 0L );
        return 0;
    }

    /* clear profile info                                                  */
    prof = libGAP_PROF_FUNC(func);
    if ( prof == 0 ) {
        libGAP_ErrorQuit( "<func> has corrupted profile info", 0L, 0L );
        return 0;
    }
    if ( libGAP_TNUM_OBJ(prof) == libGAP_T_FUNCTION ) {
        prof = libGAP_PROF_FUNC(prof);
    }
    if ( prof == 0 ) {
        libGAP_ErrorQuit( "<func> has corrupted profile info", 0L, 0L );
        return 0;
    }
    libGAP_SET_COUNT_PROF( prof, 0 );
    libGAP_SET_TIME_WITH_PROF( prof, 0 );
    libGAP_SET_TIME_WOUT_PROF( prof, 0 );
    libGAP_SET_STOR_WITH_PROF( prof, 0 );
    libGAP_SET_STOR_WOUT_PROF( prof, 0 );

    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncPROFILE_FUNC( <self>, <func> )  . . . . . . . . . . . . start profile
*/
libGAP_Obj libGAP_FuncPROFILE_FUNC(
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
    libGAP_Obj                 prof;
    libGAP_Obj                 copy;

    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit( "<func> must be a function", 0L, 0L );
        return 0;
    }
    /* uninstall trace handler                                             */
    libGAP_ChangeDoOperations( func, 0 );

    /* install profiling                                                   */
    prof = libGAP_PROF_FUNC(func);
    
    /* install new handlers                                                */
    if ( libGAP_TNUM_OBJ(prof) != libGAP_T_FUNCTION ) {
        copy = libGAP_NewBag( libGAP_TNUM_OBJ(func), libGAP_SIZE_OBJ(func) );
        libGAP_HDLR_FUNC(copy,0) = libGAP_HDLR_FUNC(func,0);
        libGAP_HDLR_FUNC(copy,1) = libGAP_HDLR_FUNC(func,1);
        libGAP_HDLR_FUNC(copy,2) = libGAP_HDLR_FUNC(func,2);
        libGAP_HDLR_FUNC(copy,3) = libGAP_HDLR_FUNC(func,3);
        libGAP_HDLR_FUNC(copy,4) = libGAP_HDLR_FUNC(func,4);
        libGAP_HDLR_FUNC(copy,5) = libGAP_HDLR_FUNC(func,5);
        libGAP_HDLR_FUNC(copy,6) = libGAP_HDLR_FUNC(func,6);
        libGAP_HDLR_FUNC(copy,7) = libGAP_HDLR_FUNC(func,7);
        libGAP_NAME_FUNC(copy)   = libGAP_NAME_FUNC(func);
        libGAP_NARG_FUNC(copy)   = libGAP_NARG_FUNC(func);
        libGAP_NAMS_FUNC(copy)   = libGAP_NAMS_FUNC(func);
        libGAP_PROF_FUNC(copy)   = libGAP_PROF_FUNC(func);
        libGAP_HDLR_FUNC(func,0) = libGAP_DoProf0args;
        libGAP_HDLR_FUNC(func,1) = libGAP_DoProf1args;
        libGAP_HDLR_FUNC(func,2) = libGAP_DoProf2args;
        libGAP_HDLR_FUNC(func,3) = libGAP_DoProf3args;
        libGAP_HDLR_FUNC(func,4) = libGAP_DoProf4args;
        libGAP_HDLR_FUNC(func,5) = libGAP_DoProf5args;
        libGAP_HDLR_FUNC(func,6) = libGAP_DoProf6args;
        libGAP_HDLR_FUNC(func,7) = libGAP_DoProfXargs;
        libGAP_PROF_FUNC(func)   = copy;
        libGAP_CHANGED_BAG(func);
    }

    return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  FuncIS_PROFILED_FUNC( <self>, <func> )  . . check if function is profiled
*/
libGAP_Obj libGAP_FuncIS_PROFILED_FUNC(
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit( "<func> must be a function", 0L, 0L );
        return 0;
    }
    return ( libGAP_TNUM_OBJ(libGAP_PROF_FUNC(func)) != libGAP_T_FUNCTION ) ? libGAP_False : libGAP_True;
}

libGAP_Obj libGAP_FuncFILENAME_FUNC(libGAP_Obj self, libGAP_Obj func) {

    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit( "<func> must be a function", 0L, 0L );
        return 0;
    }

    if (libGAP_BODY_FUNC(func)) {
        libGAP_Obj fn =  libGAP_FILENAME_BODY(libGAP_BODY_FUNC(func));
        if (fn)
            return fn;
    }
    return libGAP_Fail;
}

libGAP_Obj libGAP_FuncSTARTLINE_FUNC(libGAP_Obj self, libGAP_Obj func) {

    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit( "<func> must be a function", 0L, 0L );
        return 0;
    }

    if (libGAP_BODY_FUNC(func)) {
        libGAP_Obj sl = libGAP_STARTLINE_BODY(libGAP_BODY_FUNC(func));
        if (sl)
            return sl;
    }
    return libGAP_Fail;
}

libGAP_Obj libGAP_FuncENDLINE_FUNC(libGAP_Obj self, libGAP_Obj func) {

    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit( "<func> must be a function", 0L, 0L );
        return 0;
    }

    if (libGAP_BODY_FUNC(func)) {
        libGAP_Obj el = libGAP_ENDLINE_BODY(libGAP_BODY_FUNC(func));
        if (el)
            return el;
    }
    return libGAP_Fail;
}


/****************************************************************************
**
*F  FuncUNPROFILE_FUNC( <self>, <func> )  . . . . . . . . . . .  stop profile
*/
libGAP_Obj libGAP_FuncUNPROFILE_FUNC(
    libGAP_Obj                 self,
    libGAP_Obj                 func )
{
    libGAP_Obj                 prof;

    /* check the argument                                                  */
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit( "<func> must be a function", 0L, 0L );
        return 0;
    }

    /* uninstall trace handler                                             */
    libGAP_ChangeDoOperations( func, 0 );

    /* profiling is active, restore handlers                               */
    prof = libGAP_PROF_FUNC(func);
    if ( libGAP_TNUM_OBJ(prof) == libGAP_T_FUNCTION ) {
        libGAP_HDLR_FUNC(func,0) = libGAP_HDLR_FUNC(prof,0);
        libGAP_HDLR_FUNC(func,1) = libGAP_HDLR_FUNC(prof,1);
        libGAP_HDLR_FUNC(func,2) = libGAP_HDLR_FUNC(prof,2);
        libGAP_HDLR_FUNC(func,3) = libGAP_HDLR_FUNC(prof,3);
        libGAP_HDLR_FUNC(func,4) = libGAP_HDLR_FUNC(prof,4);
        libGAP_HDLR_FUNC(func,5) = libGAP_HDLR_FUNC(prof,5);
        libGAP_HDLR_FUNC(func,6) = libGAP_HDLR_FUNC(prof,6);
        libGAP_HDLR_FUNC(func,7) = libGAP_HDLR_FUNC(prof,7);
        libGAP_PROF_FUNC(func)   = libGAP_PROF_FUNC(prof);
        libGAP_CHANGED_BAG(func);
    }

    return (libGAP_Obj)0;
}

libGAP_Obj libGAP_FuncIsKernelFunction(libGAP_Obj self, libGAP_Obj func) {
  if (!libGAP_IS_FUNC(func))
    return libGAP_Fail;
  else return (libGAP_BODY_FUNC(func) == 0 || libGAP_SIZE_OBJ(libGAP_BODY_FUNC(func)) == 0) ? libGAP_True : libGAP_False;
}

libGAP_Obj libGAP_FuncHandlerCookieOfFunction(libGAP_Obj self, libGAP_Obj func)
{
  libGAP_Int narg;
  libGAP_ObjFunc hdlr;
  const libGAP_Char *cookie;
  libGAP_Obj cookieStr;
  if (!libGAP_IS_FUNC(func))
    return libGAP_Fail;
  narg = libGAP_NARG_FUNC(func);
  if (narg == -1)
    narg = 7;
  hdlr = libGAP_HDLR_FUNC(func, narg);
  cookie = libGAP_CookieOfHandler(hdlr);
  libGAP_C_NEW_STRING_DYN(cookieStr, cookie);
  return cookieStr;
}

/****************************************************************************
**

*F  SaveFunction( <func> )  . . . . . . . . . . . . . . . . . save a function
**
*/
void libGAP_SaveFunction ( libGAP_Obj func )
{
  libGAP_UInt i;
  for (i = 0; i <= 7; i++)
    libGAP_SaveHandler(libGAP_HDLR_FUNC(func,i));
  libGAP_SaveSubObj(libGAP_NAME_FUNC(func));
  libGAP_SaveUInt(libGAP_NARG_FUNC(func));
  libGAP_SaveSubObj(libGAP_NAMS_FUNC(func));
  libGAP_SaveSubObj(libGAP_PROF_FUNC(func));
  libGAP_SaveUInt(libGAP_NLOC_FUNC(func));
  libGAP_SaveSubObj(libGAP_BODY_FUNC(func));
  libGAP_SaveSubObj(libGAP_ENVI_FUNC(func));
  libGAP_SaveSubObj(libGAP_FEXS_FUNC(func));
  if (libGAP_SIZE_OBJ(func) != libGAP_SIZE_FUNC)
    libGAP_SaveOperationExtras( func );
}

/****************************************************************************
**
*F  LoadFunction( <func> )  . . . . . . . . . . . . . . . . . load a function
**
*/
void libGAP_LoadFunction ( libGAP_Obj func )
{
  libGAP_UInt i;
  for (i = 0; i <= 7; i++)
    libGAP_HDLR_FUNC(func,i) = libGAP_LoadHandler();
  libGAP_NAME_FUNC(func) = libGAP_LoadSubObj();
  libGAP_NARG_FUNC(func) = libGAP_LoadUInt();
  libGAP_NAMS_FUNC(func) = libGAP_LoadSubObj();
  libGAP_PROF_FUNC(func) = libGAP_LoadSubObj();
  libGAP_NLOC_FUNC(func) = libGAP_LoadUInt();
  libGAP_BODY_FUNC(func) = libGAP_LoadSubObj();
  libGAP_ENVI_FUNC(func) = libGAP_LoadSubObj();
  libGAP_FEXS_FUNC(func) = libGAP_LoadSubObj();
  if (libGAP_SIZE_OBJ(func) != libGAP_SIZE_FUNC)
    libGAP_LoadOperationExtras( func );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_FUNCTION", "obj", &libGAP_IsFunctionFilt, 
      libGAP_FuncIS_FUNCTION, "src/calls.c:IS_FUNCTION" },

    { 0 }

};


/****************************************************************************
**
*V  GVarOpers . . . . . . . . . . . . . . . . .  list of operations to export
*/
static libGAP_StructGVarOper libGAP_GVarOpers [] = {

    { "CALL_FUNC", -1, "args", &libGAP_CallFunctionOper,
      libGAP_FuncCALL_FUNC, "src/calls.c:CALL_FUNC" },

    { "CALL_FUNC_LIST", 2, "func, list", &libGAP_CallFuncListOper,
      libGAP_FuncCALL_FUNC_LIST, "src/calls.c:CALL_FUNC_LIST" },

    { "NAME_FUNC", 1, "func", &libGAP_NAME_FUNC_Oper,
      libGAP_FuncNAME_FUNC, "src/calls.c:NAME_FUNC" },

    { "SET_NAME_FUNC", 2, "func, name", &libGAP_SET_NAME_FUNC_Oper,
      libGAP_FuncSET_NAME_FUNC, "src/calls.c:SET_NAME_FUNC" },

    { "NARG_FUNC", 1, "func", &libGAP_NARG_FUNC_Oper,
      libGAP_FuncNARG_FUNC, "src/calls.c:NARG_FUNC" },

    { "NAMS_FUNC", 1, "func", &libGAP_NAMS_FUNC_Oper,
      libGAP_FuncNAMS_FUNC, "src/calls.c:NAMS_FUNC" },

    { "PROF_FUNC", 1, "func", &libGAP_PROF_FUNC_Oper,
      libGAP_FuncPROF_FUNC, "src/calls.c:PROF_FUNC" },


    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "CLEAR_PROFILE_FUNC", 1, "func",
      libGAP_FuncCLEAR_PROFILE_FUNC, "src/calls.c:CLEAR_PROFILE_FUNC" },

    { "IS_PROFILED_FUNC", 1, "func",
      libGAP_FuncIS_PROFILED_FUNC, "src/calls.c:IS_PROFILED_FUNC" },

    { "PROFILE_FUNC", 1, "func",
      libGAP_FuncPROFILE_FUNC, "src/calls.c:PROFILE_FUNC" },

    { "UNPROFILE_FUNC", 1, "func",
      libGAP_FuncUNPROFILE_FUNC, "src/calls.c:UNPROFILE_FUNC" },

    { "IsKernelFunction", 1, "func",
      libGAP_FuncIsKernelFunction, "src/calls.c:IsKernelFunction" },

    { "HandlerCookieOfFunction", 1, "func",
      libGAP_FuncHandlerCookieOfFunction, "src/calls.c:HandlerCookieOfFunction" },

    { "FILENAME_FUNC", 1, "func", 
      libGAP_FuncFILENAME_FUNC, "src/calls.c:FILENAME_FUNC" },

    { "STARTLINE_FUNC", 1, "func", 
      libGAP_FuncSTARTLINE_FUNC, "src/calls.c:STARTLINE_FUNC" },

    { "ENDLINE_FUNC", 1, "func", 
      libGAP_FuncENDLINE_FUNC, "src/calls.c:ENDLINE_FUNC" },
    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
  
    /* install the marking functions                                       */
    libGAP_InfoBags[ libGAP_T_FUNCTION ].name = "function";
    libGAP_InitMarkFuncBags( libGAP_T_FUNCTION , libGAP_MarkAllSubBags );

    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_FUNCTION",  &libGAP_TYPE_FUNCTION  );
    libGAP_ImportGVarFromLibrary( "TYPE_OPERATION", &libGAP_TYPE_OPERATION );
    libGAP_TypeObjFuncs[ libGAP_T_FUNCTION ] = libGAP_TypeFunction;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrOpersFromTable( libGAP_GVarOpers );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* and the saving function                                             */
    libGAP_SaveObjFuncs[ libGAP_T_FUNCTION ] = libGAP_SaveFunction;
    libGAP_LoadObjFuncs[ libGAP_T_FUNCTION ] = libGAP_LoadFunction;

    /* install the printer                                                 */
    libGAP_InitFopyGVar( "PRINT_OPERATION", &libGAP_PrintOperation );
    libGAP_PrintObjFuncs[ libGAP_T_FUNCTION ] = libGAP_PrintFunction;


    /* initialise all 'Do<Something><N>args' handlers, give the most       */
    /* common ones short cookies to save space in in the saved workspace   */
    libGAP_InitHandlerFunc( libGAP_DoFail0args, "f0" );
    libGAP_InitHandlerFunc( libGAP_DoFail1args, "f1" );
    libGAP_InitHandlerFunc( libGAP_DoFail2args, "f2" );
    libGAP_InitHandlerFunc( libGAP_DoFail3args, "f3" );
    libGAP_InitHandlerFunc( libGAP_DoFail4args, "f4" );
    libGAP_InitHandlerFunc( libGAP_DoFail5args, "f5" );
    libGAP_InitHandlerFunc( libGAP_DoFail6args, "f6" );
    libGAP_InitHandlerFunc( libGAP_DoFailXargs, "f7" );

    libGAP_InitHandlerFunc( libGAP_DoWrap0args, "w0" );
    libGAP_InitHandlerFunc( libGAP_DoWrap1args, "w1" );
    libGAP_InitHandlerFunc( libGAP_DoWrap2args, "w2" );
    libGAP_InitHandlerFunc( libGAP_DoWrap3args, "w3" );
    libGAP_InitHandlerFunc( libGAP_DoWrap4args, "w4" );
    libGAP_InitHandlerFunc( libGAP_DoWrap5args, "w5" );
    libGAP_InitHandlerFunc( libGAP_DoWrap6args, "w6" );

    libGAP_InitHandlerFunc( libGAP_DoProf0args, "p0" );
    libGAP_InitHandlerFunc( libGAP_DoProf1args, "p1" );
    libGAP_InitHandlerFunc( libGAP_DoProf2args, "p2" );
    libGAP_InitHandlerFunc( libGAP_DoProf3args, "p3" );
    libGAP_InitHandlerFunc( libGAP_DoProf4args, "p4" );
    libGAP_InitHandlerFunc( libGAP_DoProf5args, "p5" );
    libGAP_InitHandlerFunc( libGAP_DoProf6args, "p6" );
    libGAP_InitHandlerFunc( libGAP_DoProfXargs, "pX" );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module ){
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarOpersFromTable( libGAP_GVarOpers );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoCalls() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "calls",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoCalls ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  calls.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
